/*
 * Copyright 2013, 2015 - 2020, 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic ISO14443-4 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALI14443P4_H
#define PHPALI14443P4_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHPAL_I14443P4_SW

/** \defgroup phpalI14443p4_Sw Component : Software
* @{
*/

#define PHPAL_I14443P4_SW_ID        0x01    /**< ID for Software ISO14443-4 layer */



/**
* \name Custom Configs
*/
/** @{ */
#define PHPAL_I14443P4_SW_CONFIG_MAX_WTX_LOOP_TIME   (PH_CONFIG_CUSTOM_BEGIN)   /**< Set / Get Maximum Time used within a WTX loop (in seconds). */
#define PHPAL_I14443P4_SW_SUPPRESS_DESELECT          (PH_CONFIG_CUSTOM_BEGIN+1) /**< Enable/Diable the automatic send of a Deselect/Is the deselect disabled. */
#define PHPAL_I14443P4_SW_USED_RETRY_COUNT           (PH_CONFIG_CUSTOM_BEGIN+2) /**< Get the numbers of retransmits used. */
#define PHPAL_I14443P4_SW_FRAME_FORMAT_PCD_2_PICC    (PH_CONFIG_CUSTOM_BEGIN+3) /**< Set the frame format pcd 2 picc. */
#define PHPAL_I14443P4_SW_FRAME_FORMAT_PICC_2_PCD    (PH_CONFIG_CUSTOM_BEGIN+4) /**< Set the frame format picc 2 pcd. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PCD_2_PICC (PH_CONFIG_CUSTOM_BEGIN+5) /**< Set the frame options sync suppression pcd 2 picc. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PICC_2_PCD (PH_CONFIG_CUSTOM_BEGIN+6) /**< Set the frame options sync suppression  picc 2 pcd. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PCD_2_PICC (PH_CONFIG_CUSTOM_BEGIN+7) /**< Set the frame options SOF/EOF suppression pcd 2 picc. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD (PH_CONFIG_CUSTOM_BEGIN+8) /**< Set the frame options SOF/EOF suppression picc 2 pcd. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PCD_2_PICC (PH_CONFIG_CUSTOM_BEGIN+9) /**< Set the frame options start stop bit suppression pcd 2 picc. */
#define PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD (PH_CONFIG_CUSTOM_BEGIN+10) /**< Set the frame options stop bit suppression picc 2 pcd. */
/** @} */

/**
* \brief ISO14443-4 parameter structure
*/
typedef struct
{
    uint16_t  wId;              /**< Layer ID for this component, NEVER MODIFY! */
    void    * pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer. */
    uint8_t   bStateNow;        /**< Current Exchange-State. */
    uint8_t   bCidEnabled;      /**< Card Identifier Enabler; Unequal '0' if enabled. */
    uint8_t   bCid;             /**< Card Identifier; Ignored if *pCidSupported is equal '0'. */
    uint8_t   bNadEnabled;      /**< Node Address Enabler; Unequal '0' if enabled. */
    uint8_t   bNad;             /**< Node Address; Ignored if bNadEnabled is equal '0'. */
    uint8_t   bFwi;             /**< Frame Waiting Integer. */
    uint8_t   bFsdi;            /**< PCD Frame Size Integer; 0-0xC; */
    uint8_t   bFsci;            /**< PICC Frame Size Integer; 0-0xC; */
    uint8_t   bPcbBlockNum;     /**< Current Block-Number; 0/1; */
    uint8_t   bMaxRetryCount;   /**< Maximum Retry count for ISO/IEC 14443-4:2008(E) Rule 4 and 5. */
    uint8_t   bUsedRetryCount;  /**< How much retransmits are used durring the last exchange */
    uint8_t   bSuppressDeselect;/**< If this is PH_ON the deselect is suppressed even if retry count != 0 */
    uint32_t  dwWTXLoopTimeLimit;/**< Maximum Timing allowed to be on a WTX loop (if zero no limit). */
    uint8_t pbEnhancedBlockBufferTx[5000]; /**< Buffer used for storing the plain frame for FWEC. */
    uint8_t pbEnhancedBlockBufferRx[5000]; /**< Buffer used for storing the plain frame for FWEC. */
    uint8_t pbEnhancedBlockBufferTmp[5000]; /**< Buffer used for storing the plain frame for FWEC. */
    uint16_t  wTxBufferLength;      /**< Number of Data stored for transmission in pbEnhancedBlockBuffer */
    uint16_t  wRxBufferStartPos;    /**< Start position of the rx buffer in pbEnhancedBlockBuffer */
    uint8_t   bEnhancedBlockStateTx;  /**< Indicates if enhanced Block format is enabled or disabled. */
    uint8_t   bFwecSyncEnabledTx;     /**< Indicates if sync is enabled or disabled for Frame with error correction. */
    uint8_t   bEnhancedBlockStateRx;  /**< Indicates if enhanced Block format is enabled or disabled. */
    uint8_t   bFwecSyncEnabledRx;     /**< Indicates if sync is enabled or disabled for Frame with error correction. */
} phpalI14443p4_Sw_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Sw_Init(
                                 phpalI14443p4_Sw_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                    /**< [In] Specifies the size of the data parameter structure. */
                                 void * pHalDataParams                          /**< [In] Pointer to the parameter structure of the underlying layer. */
                                 );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4_SW */



#ifdef NXPBUILD__PHPAL_I14443P4_RD710

/** \defgroup phpalI14443p4_Rd710 Component : Rd710
* @{
*/

#define PHPAL_I14443P4_RD710_ID        0x03    /**< ID for RD710 ISO14443-4 layer */

/**
* \brief ISO14443-4 parameter structure
*/
typedef struct
{
    uint16_t  wId;              /**< Layer ID for this component, NEVER MODIFY!                 */
    phhalHw_Rd710_DataParams_t * pHalDataParams;   /**< pointer to the parameter structure of the underlying layer     */
} phpalI14443p4_Rd710_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Rd710_Init(
                                        phpalI14443p4_Rd710_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                        uint16_t wSizeOfDataParams,                        /**< [In] Specifies the size of the data parameter structure. */
                                        phhalHw_Rd710_DataParams_t * pHalDataParams        /**< [In] Pointer to the parameter structure of the underlying layer. */
                                        );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4_RD710 */

#ifdef NXPBUILD__PHPAL_I14443P4_ANDROID

/** \defgroup phpalI14443p4_Android Component : Android
* @{
*/

#define PHPAL_I14443P4_ANDROID_ID        0x06    /**< ID for ANDROID ISO14443-4 layer */


/**
* \brief Technologies available
*/
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE            0x00
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_ISO_DEP         0x01
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NFC_A           0x02
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NFC_B           0x03
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_MIFARE_CLASSIC  0x04
#define PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT             0x05

/**
* \brief Custom Configs
*/
#define PHPAL_I14443P4_ANDROID_CONFIG_SYS_INFO_SIZE     (PH_CONFIG_CUSTOM_BEGIN)    /**< Get the actual sysinfo size. */


/**
* \brief ISO14443-4 parameter structure
*/
typedef struct
{
    uint16_t  wId;                                    /**< Layer ID for this component, NEVER MODIFY!                 */
    phbalReg_SocketWin_DataParams_t * pBalDataParams; /**< pointer to the parameter structure of the underlying layer     */
    uint8_t bIsOpen;                                  /**< Inticated if the connection is opened */
    uint8_t abAvailableTechnologies[PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT]; /**< Stores all available technologies for the current present Tag */
    uint8_t bUsedTechnologies;                        /**< Stores the current used technologie */
    uint8_t * pExchangeBuffer;                        /**< Pointer to global Exchange buffer used by the Exchange() function. */
    uint16_t wExchangeBufSize;                        /**< Size of the global Exchange buffer. */
} phpalI14443p4_Android_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Android_Init(
                                        phpalI14443p4_Android_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                        uint16_t wSizeOfDataParams,                        /**< [In] Specifies the size of the data parameter structure. */
                                        phbalReg_SocketWin_DataParams_t * pBalDataParams,  /**< [In] Pointer to the parameter structure of the underlying layer. */
                                        uint8_t * pExchangeBuffer,                         /**< [In] Pointer to global Exchange buffer used by the Exchange() function. */
                                        uint16_t wExchangeBufSize                          /**< [In] Size of the global Exchange buffer. */
                                        );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4_ANDROID */

#ifdef NXPBUILD__PHPAL_I14443P4_PCSC

/** \defgroup phpalI14443p4_Pcsc Component : PCSC
* @{
*/

#define PHPAL_I14443P4_PCSC_ID      0x05          /**< ID for PCSC ISO14443-4 layer */

#define PHPAL_I14443P4_PCSC_CONFIG_EXCHANGE     (PH_CONFIG_CUSTOM_BEGIN)    /**< Set/Get PAL exchange.type; #PHPAL_I14443P4_PCSC_NORMAL (default) or #PHPAL_I14443P4_PCSC_TRANSPARENT. */

#define PHPAL_I14443P4_PCSC_TRANSPARENT    0U     /**< Exchange using HAL transparent exchange command (Only works with PCSC Readers that support transparent session). Chaining/error handling etc done by PAL. */
#define PHPAL_I14443P4_PCSC_NORMAL         1U     /**< Exchange the data as it is using normal HAL exchange (works with all PCSC Readers). Chaining/error handling etc controlled by the PCSC Reader. */

/**
* \brief PCSC ISO14443-4 parameter structure
*/
typedef struct
{
    uint16_t wId;                 /**< Layer ID for this component, NEVER MODIFY! */
    void * pHalDataParams;        /**< Pointer to the parameter structure of the underlying layer. */
    phbalReg_PcscWin_DataParams_t * pBalDataParams; /**< pointer to the parameter structure of the underlying pcsc bal. Is used to access the pcsc parameter */
    phhalHw_Pcsc_DataParams_t * pHalPcscDataParams; /**< pointer to the parameter structure of the underlying pcsc hal. Is used to access the pcsc parameter */
    uint8_t bStateNow;            /**< Current Exchange-State. */
    uint8_t bCidEnabled;          /**< Card Identifier Enabler; Unequal '0' if enabled. */
    uint8_t bCid;                 /**< Card Identifier; Ignored if *pCidSupported is equal '0'. */
    uint8_t bNadEnabled;          /**< Node Address Enabler; Unequal '0' if enabled. */
    uint8_t bNad;                 /**< Node Address; Ignored if bNadEnabled is equal '0'. */
    uint8_t bFwi;                 /**< Frame Waiting Integer. */
    uint8_t bFsdi;                /**< PCD Frame Size Integer; 0-8; */
    uint8_t bFsci;                /**< PICC Frame Size Integer; 0-8; */
    uint8_t bPcbBlockNum;         /**< Current Block-Number; 0/1; */
    uint8_t bMaxRetryCount;       /**< Maximum Retry count for ISO/IEC 14443-4:2008(E) Rule 4 and 5. */
    uint8_t bExchangeType;        /**< Type of exchange to use. #PHPAL_I14443P4_PCSC_NORMAL (default) or #PHPAL_I14443P4_PCSC_TRANSPARENT. */
} phpalI14443p4_Pcsc_DataParams_t;

/**
* \brief Initialise PCSC ISO14443-4 layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Pcsc_Init(
                                   phpalI14443p4_Pcsc_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wSizeOfDataParams,                       /**< [In] Specifies the size of the data parameter structure. */
                                   void * pHalDataParams                             /**< [In] Pointer to the parameter structure of HAL layer. */
                                   );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4_PCSC */



#ifdef NXPBUILD__PHPAL_I14443P4

/** \addtogroup ph_Error
* phpalI14443p4 Custom Errors
* @{
*/

/** \name phpalI14443p4 Error Codes
*/
/** @{ */
#define PHPAL_I14443P4_ERR_RECOVERY_FAILED  (PH_ERR_CUSTOM_BEGIN+0)     /**< ISO14443-4 error recovery failed. */
/** @} */

/** @} */

/** \defgroup phpalI14443p4 ISO/IEC14443-4
* \brief These Components implement the ISO/IEC 14443-4:2008(E) contactless protocol.
* @{
*/

#define PHPAL_I14443P4_CID_MAX          14  /**< The last valid CID. */
#define PHPAL_I14443P4_FWI_MAX          14  /**< The maximum allowed FWI value. */
#define PHPAL_I14443P4_FRAMESIZE_MAX    0x0F/**< The maximum allowed FSDI/FSCI value. */

/**
* \name ISO14443-4 Modes for SParam Execution
*/
/** @{ */
#define PHPAL_I14443P4_SPARAM_MODE_CHECK    0 /**< Default mode, sends a SParam request, checks if the requested parameter are supported from the card and return an error if the features are not supported. Otherwise send and activation of the required parameter */
#define PHPAL_I14443P4_SPARAM_MODE_FORCE    1 /**< Same as the default mode but did not check if the required parameters are supported just send an activation with the requied parameter */
#define PHPAL_I14443P4_SPARAM_MODE_AUTO     2 /**< Does not check the input of the SParam call, just request the supported data from the card and set the maximum possible datarate/best framing options */
#define PHPAL_I14443P4_SPARAM_MODE_GET      3 /**< Only send the sparam request and return the supported parameter */
#define PHPAL_I14443P4_SPARAM_MODE_ACTIVATE 4 /**< Perform the sparam activation without any request or check */
/** @} */

/**
* \name ISO14443-4 Parameters
*/
/** @{ */
/**
* \brief Set / Get Blocknumber.
*/
#define PHPAL_I14443P4_CONFIG_BLOCKNO           0x0000U
/**
* \brief Set / Get Card Identifier.
*
* \c wValue is parsed as follows:
* \verbatim
* CidEnabled = (wValue & 0xFF00) ? 1 : 0;
* Cid = (wValue & 0x00FF);
* \endverbatim
*/
#define PHPAL_I14443P4_CONFIG_CID               0x0001U
/**
* \brief Set / Get Node Address.
*
* \c wValue is parsed as follows:
* \verbatim
* NadEnabled = (wValue & 0xFF00) ? 1 : 0;
* Nad = (wValue & 0x00FF);
* \endverbatim
*/
#define PHPAL_I14443P4_CONFIG_NAD               0x0002U
/**
* \brief Set / Get Frame Waiting Integer.
*/
#define PHPAL_I14443P4_CONFIG_FWI               0x0003U
/**
* \brief Set / Get PCD & PICC Frame Size Integer.
*
* \c wValue is parsed as follows:
* \verbatim
* Fsdi = ((wValue & 0xFF00) >> 8);
* Fsci = (wValue & 0x00FF);
* \endverbatim
*/
#define PHPAL_I14443P4_CONFIG_FSI               0x0004U
/**
* \brief Set / Get Maximum Retry Count.
*/
#define PHPAL_I14443P4_CONFIG_MAXRETRYCOUNT     0x0005U
/**
* \brief Set / Get Remaining bytes to exchange.
*/
#define PHPAL_I14443P4_CONFIG_REMAINING_BYTES_TO_EXCHANGE     0x0006U
/** @} */

/**
* \brief Set the ISO14443-4 protocol parameters.
*
* This sets the protocol parameters for this layer.
* It is recommended to input this function with the parameters retrieved from either
* \ref phpalI14443p4a_GetProtocolParams or \ref phpalI14443p3b_GetProtocolParams respectively after card activation.
* \b Note: Refer to the ISO/IEC 14443-4:2008(E) specification for a detailed description of the parameters.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_SetProtocol(
                                     void * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t   bCidEnable,      /**< [In] Enable usage of Card Identifier; Unequal '0' if enabled. */
                                     uint8_t   bCid,            /**< [In] Card Identifier; Ignored if \c bCidEnable is equal '0'. */
                                     uint8_t   bNadEnable,      /**< [In] Enabler usage of Node Address; Unequal '0' if enabled. */
                                     uint8_t   bNad,            /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
                                     uint8_t   bFwi,            /**< [In] Frame Waiting Integer. */
                                     uint8_t   bFsdi,           /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
                                     uint8_t   bFsci            /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
                                     );

/**
* \brief Reset the ISO14443-4 protocol parameters.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_ResetProtocol(
                                       void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                                       );

/**
* \brief Deselect ISO14443-4 card.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Deselect(
                                  void * pDataParams    /**< [In] Pointer to this layer's parameter structure. */
                                  );

/**
* \brief Perform presence check for current card. (Perform R(NAK) polling as defined in ISO 14443-4:2008(E)).
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_PresCheck(
                                   void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                                   );

/**
* \brief Perform ISO14443-4 Data Exchange with Picc.
*
* \c wOption can be one of:\n
* \li #PH_EXCHANGE_DEFAULT
* \li #PH_EXCHANGE_TXCHAINING
* \li #PH_EXCHANGE_RXCHAINING
* \li #PH_EXCHANGE_RXCHAINING_BUFSIZE
*
* Additionally, the following options are also available:\n
* \li #PH_EXCHANGE_BUFFER_FIRST
* \li #PH_EXCHANGE_BUFFER_CONT
* \li #PH_EXCHANGE_BUFFER_LAST
*
* Alternatively to the FIRST/CONT/LAST options, the following bits can be combined:\n
* \li #PH_EXCHANGE_BUFFERED_BIT
* \li #PH_EXCHANGE_LEAVE_BUFFER_BIT
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Exchange(
                                  void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                  uint16_t wOption,         /**< [In] Option parameter. */
                                  uint8_t * pTxBuffer,      /**< [In] Data to transmit. */
                                  uint16_t wTxLength,       /**< [In] Length of data to transmit. */
                                  uint8_t ** ppRxBuffer,    /**< [Out] Pointer to received data. */
                                  uint16_t * pRxLength      /**< [Out] number of received data bytes. */
                                  );

/**
* \brief Set configuration parameter.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_SetConfig(
                                   void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wConfig,    /**< [In] Configuration Identifier */
                                   uint16_t wValue      /**< [In] Configuration Value */
                                   );

/**
* \brief Get configuration parameter.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_GetConfig(
                                   void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wConfig,    /**< [In] Configuration Identifier */
                                   uint16_t * pValue    /**< [Out] Configuration Value */
                                   );

/**
* \brief Perform an SParam for frame format.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_SParamFrameFormat(
                                   void * pDataParams,                  /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t bMode,                       /**< [In] Mode of the SParam Exchange */
                                   uint8_t  * pbFramePcd2Picc,          /**< [In/Out] Frame from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
                                   uint8_t  * pbFramePicc2Pcd,          /**< [In/Out] Frame from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
                                   uint8_t  * pbFramingOptionsPcd2Picc, /**< [In/Out] Framing Options from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
                                   uint8_t  * pbFramingOptionsPicc2Pcd  /**< [In/Out] Framing Options from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
                                   );


/**
* \brief Perform an SParam for datarate.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_SParamBitRates(
                                   void * pDataParams,           /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t bMode,                /**< [In] Mode of the SParam Exchange */
                                   uint16_t * pwBitRatePcd2Picc, /**< [In/Out] Datarate from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
                                   uint16_t * pwBitRatePicc2Pcd, /**< [In/Out] Datarate from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
                                   uint8_t  * pbFramingOptions   /**< [In/Out] Framing Options, supplied value is the required, returned value is the values supported by PICC */
                                   );
/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4 */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHPALI14443P4_H */
