/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;

using NxpRdLibNet;
using static NxpRdLibNet.Log;

namespace Example
{
    class Program
    {
        private static void CheckSuccess(Status_t status)
        {
            if (!status.Equals(Error_Gen.SUCCESS))
            {
                string message = "Call failed with status: " + status;
                System.Exception exec = new Exception(message);
                throw exec;
            }
        }

        private static void LogCallback( IntPtr pDataParams, eLogOption bOption, LogEntry_t[] pLogEntries )
        {
            int index;
            int status;
            string ConsoleOutput = null;
            bool FirstString = true;
            bool FirstParam = true;

            // Only print entries matching this criteria:
            if (!pLogEntries[0].pString.EndsWith("_OpenPort") &&
                !pLogEntries[0].pString.EndsWith("_ClosePort") &&
                !pLogEntries[0].pString.StartsWith("phKeyStore") &&
                !pLogEntries[0].pString.StartsWith("phpal") &&
                !pLogEntries[0].pString.StartsWith("phal"))
            {
                return;
            }

            foreach (Log.LogEntry_t logEntry in pLogEntries)
            {
                if (logEntry.bLogType == Log.eLogType.INFO && logEntry.pData.Length == 0)
                {
                    ConsoleOutput += logEntry.pString;
                    if (FirstString)
                    {
                        switch (bOption)
                        {
                            case Log.eLogOption.CATEGORY_ENTER:
                                ConsoleOutput += " [ENTRY] (";
                                break;
                            case Log.eLogOption.CATEGORY_GEN:
                                ConsoleOutput += " [GEN] (";
                                break;
                            case Log.eLogOption.CATEGORY_LEAVE:
                                ConsoleOutput += " [LEAVE] (";
                                break;
                        }
                        FirstString = false;
                    }
                }
                else
                {
                    if (!FirstParam)
                    {
                        if ((logEntry.pString != "status") && (logEntry.pData.Length > 0))
                        {
                            ConsoleOutput += ", ";
                        }
                    }
                    else
                    {
                        FirstParam = false;
                    }
                }

                if ((logEntry.pString == "wOption") && (logEntry.pData.Length == 2))
                {
                    int wOption = (logEntry.pData[0] << 8) | logEntry.pData[1];
                    ConsoleOutput += "wOption = ";
                    switch (wOption & ExchangeOptions.MODE_MASK)
                    {
                        case ExchangeOptions.DEFAULT:
                            ConsoleOutput += "DEFAULT";
                            break;
                        case ExchangeOptions.TXCHAINING:
                            ConsoleOutput += "TXCHAINING";
                            break;
                        case ExchangeOptions.RXCHAINING:
                            ConsoleOutput += "RXCHAINING";
                            break;
                        case ExchangeOptions.RXCHAINING_BUFSIZE:
                            ConsoleOutput += "RXCHAINING_BUFSIZE";
                            break;
                    }
                    if ((wOption & ExchangeOptions.BUFFERED_BIT) != 0)
                    {
                        ConsoleOutput += " | BUFFERED_BIT";
                    }
                    if ((wOption & ExchangeOptions.LEAVE_BUFFER_BIT) != 0)
                    {
                        ConsoleOutput += " | LEAVE_BUFFER_BIT";
                    }
                }
                else if ((logEntry.pString == "status") && (logEntry.pData.Length == 2))
                {
                    ConsoleOutput += ") = ";
                    status = (logEntry.pData[0] << 8) | logEntry.pData[1];
                    ConsoleOutput += new Status_t(status).ToString();
                }
                else if (logEntry.pData.Length > 0)
                {
                    ConsoleOutput += logEntry.pString + "(" + logEntry.pData.Length + ") = ";
                    for (index = 0; index < logEntry.pData.Length; index++)
                    {
                        ConsoleOutput += String.Format("{0:X2}", logEntry.pData[index]);
                        if ((index + 1) < logEntry.pData.Length)
                        {
                            ConsoleOutput += " ";
                        }
                    }
                }
            }

            if (bOption == Log.eLogOption.CATEGORY_LEAVE)
            {
                ConsoleOutput += "\n";
            }
            else
            {
                ConsoleOutput += ")";
            }
            Console.WriteLine(ConsoleOutput);
        }

        static void Main(string[] args)
        {
            // Init Logging
            Log log = new Log();
            CheckSuccess(log.Init(new Log.CallbackDelegate(LogCallback), 256));

            try
            {
                Tutorial.MifarePlusSL3();
            }
            catch (System.Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine("Stacktrace:");
                Console.WriteLine(ex.StackTrace);
            }

            Console.WriteLine("\nPress the Anykey to continue...");
            Console.ReadKey();
        }
    }
}
