/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Collections.Generic;
using System.Text;

using NxpRdLibNet;

using Bal = NxpRdLibNet.Bal;
using Hal = NxpRdLibNet.Hal;
using KeyStore = NxpRdLibNet.KeyStore;
using CryptoSym = NxpRdLibNet.CryptoSym;
using CryptoRng = NxpRdLibNet.CryptoRng;
using palI14443p3a = NxpRdLibNet.palI14443p3a;
using palI14443p4a = NxpRdLibNet.palI14443p4a;
using palI14443p4 = NxpRdLibNet.palI14443p4;
using palMifare = NxpRdLibNet.palMifare;
using alMfp = NxpRdLibNet.alMfp;

namespace Example
{
    class Tutorial
    {
        private static void CheckSuccess(Status_t status)
        {
            if (!status.Equals(Error_Gen.SUCCESS))
            {
                string message = "Call failed with status: " + status;
                System.Exception ex = new Exception(message);
                throw ex;
            }
        }

        private abstract class IReader
        {
            protected Bal.Generic m_bal;
            protected Hal.Generic m_hal;
            protected KeyStore.Generic m_KeyStore;

            public Bal.Generic Bal { get { return m_bal; } }
            public Hal.Generic Hal { get { return m_hal; } }
            public KeyStore.Generic KeyStore { get { return m_KeyStore; } }
        }

        private class Reader : IReader
        {
            private byte[] Atr;

            public enum ReaderType : int
            {
                RD70X = ConsoleKey.D0,
                RD710,
                RC523,
                RC663
            }

            public Reader()
            {
                m_bal = null;
                m_hal = null;
                m_KeyStore = null;

                #region KeyStore

                m_KeyStore = new NxpRdLibNet.KeyStore.Sw();
                CheckSuccess(((KeyStore.Sw)m_KeyStore).Init(10, 1, 1));

                #endregion
            }

            ~Reader()
            {
                try
                {
                    Close();
                }
                catch { }
            }

            public Status_t Open(ReaderType Reader)
            {
                string[] rd;

                switch (Reader)
                {
                    case ReaderType.RD70X:

                        #region BAL RD70X

                        // Instanciate BAL
                        Bal.Rd70xUsbWin balRd70x = new NxpRdLibNet.Bal.Rd70xUsbWin();
                        CheckSuccess(balRd70x.Init());

                        // Retrieve a list of connected readers
                        CheckSuccess(balRd70x.GetPortList(4, out rd));
                        if ((rd == null) || (rd.Length == 0))
                        {
                            return new Status_t(Error_CompCode.BAL, Error_Comm.INTERFACE_ERROR);
                        }

                        // Use the first found reader
                        CheckSuccess(balRd70x.SetPort(rd[0]));

                        // Open connection to the reader
                        CheckSuccess(balRd70x.OpenPort());

                        // Use as Generic BAL
                        m_bal = balRd70x;

                        #endregion

                        #region HAL Rd70X

                        // Instanciate HAL
                        Hal.Rd70x halRd70x = new NxpRdLibNet.Hal.Rd70x();
                        CheckSuccess(halRd70x.Init(balRd70x, 256, 256));

                        // Use as Generic HAL
                        m_hal = halRd70x;

                        break;

                        #endregion

                    case ReaderType.RD710:

                        #region BAL PCSC

                        // Instanciate BAL
                        Bal.PcscWin balPcsc = new NxpRdLibNet.Bal.PcscWin();
                        balPcsc.Init(out Atr);

                        // Configure DIRECT / RAW Mode
                        CheckSuccess(balPcsc.SetConfig(NxpRdLibNet.Bal.PcscWin.Config.SHARE, (int)NxpRdLibNet.Bal.PcscWin.Share.DIRECT));
                        CheckSuccess(balPcsc.SetConfig(NxpRdLibNet.Bal.PcscWin.Config.PROTOCOL, (int)NxpRdLibNet.Bal.PcscWin.Protocol.UNDEFINED));

                        // Retrieve a list of connected readers
                        CheckSuccess(balPcsc.GetPortList(20, out rd));
                        if ((rd == null) || (rd.Length == 0))
                        {
                            return new Status_t(Error_CompCode.BAL, Error_Comm.INTERFACE_ERROR);
                        }

                        // Search for Rd710 PCSC connection
                        bool found = false;
                        foreach (string Entry in rd)
                        {
                            if ((Entry.Contains("NXP Pegoda N CL")) ||
                                (Entry.Contains("NXP Pegoda S CL")))
                            {
                                CheckSuccess(balPcsc.SetPort(Entry));
                                found = true;
                                break;
                            }
                        }
                        if (found == false)
                        {
                            return new Status_t(Error_CompCode.BAL, Error_Comm.INTERFACE_ERROR);
                        }

                        // Open connection to the reader
                        CheckSuccess(balPcsc.OpenPort());

                        // Use as Generic BAL
                        m_bal = balPcsc;

                        #endregion

                        #region HAL RD710

                        // Instanciate HAL
                        Hal.Rd710 halRd710 = new NxpRdLibNet.Hal.Rd710();
                        CheckSuccess(halRd710.Init(balPcsc, 0, 256, 256));

                        // Initialize Reader
                        CheckSuccess(halRd710.Cmd_InitReader());

                        // Use as Generic HAL
                        m_hal = halRd710;

                        break;

                        #endregion

                    case ReaderType.RC523:
                    case ReaderType.RC663:

                        #region BAL SERIAL

                        // Instanciate Serial BAL
                        NxpRdLibNet.Bal.SerialWin balSerial = new NxpRdLibNet.Bal.SerialWin();
                        balSerial.Init(512);

                        // Retrieve a list of connected readers
                        CheckSuccess(balSerial.GetPortList(10, out rd));
                        if ((rd == null) || (rd.Length == 0))
                        {
                            return new Status_t(Error_CompCode.BAL, Error_Comm.INTERFACE_ERROR);
                        }

                        int i = 0;
                        Console.WriteLine("***********************");
                        Console.WriteLine("Choose Serial Port");
                        Console.WriteLine("***********************");
                        foreach (string Entry in rd)
                        {
                            Console.WriteLine("* " + i++ + ") " + Entry);
                        }
                        Console.WriteLine("***********************");
                        Console.Write("@:\\>");
                        ConsoleKeyInfo key = Console.ReadKey(false);
                        Console.Clear();
                        if ((key.Key < ConsoleKey.D0) || (key.Key > (ConsoleKey.D0 + (rd.Length - 1))))
                        {
                            return new Status_t(Error_CompCode.BAL, Error_Param.INVALID_PARAMETER);
                        }
                        i = (int)key.Key - (int)ConsoleKey.D0;

                        CheckSuccess(balSerial.SetPort(rd[i]));

                        // Open connection to the reader
                        CheckSuccess(balSerial.OpenPort());

                        // Use as Generic BAL
                        m_bal = balSerial;

                        #endregion

                        if (Reader == ReaderType.RC523)
                        {
                            #region HAL RC523

                            // Instanciate HAL
                            NxpRdLibNet.Hal.Rc523 halRc523 = new NxpRdLibNet.Hal.Rc523();
                            CheckSuccess(halRc523.Init(m_bal, m_KeyStore, 256, 256));

                            // Use as Generic HAL
                            m_hal = halRc523;

                            // Synchronize connection
                            m_hal.WriteRegister(0x37, 0xFF);

                            // Set Reader IC Baudrate to 115200
                            CheckSuccess(m_hal.SetConfig(
                                NxpRdLibNet.Hal.Config.SERIAL_BITRATE,
                                (int)NxpRdLibNet.Hal.Rs232BitRate.VALUE_115200));

                            // Set BAL Baudrate to 115200
                            CheckSuccess(m_bal.SetConfig(
                                NxpRdLibNet.Bal.SerialWin.Config.BITRATE,
                                (int)NxpRdLibNet.Bal.SerialWin.BitRate.VALUE_115200));

                            #endregion
                        }
                        else
                        {
                            #region HAL RC663

                            // Instanciate HAL
                            NxpRdLibNet.Hal.Rc663 halRc663 = new NxpRdLibNet.Hal.Rc663();
                            CheckSuccess(halRc663.Init(m_bal, 256, 256));

                            // Use as Generic HAL
                            m_hal = halRc663;

                            // Set BAL Baudrate to 115200
                            CheckSuccess(m_bal.SetConfig(
                                NxpRdLibNet.Bal.SerialWin.Config.BITRATE,
                                (int)NxpRdLibNet.Bal.SerialWin.BitRate.VALUE_115200));

                            // Synchronize connection
                            m_hal.WriteRegister(0x37, 0xFF);

                            #endregion
                        }

                        break;

                    default:
                        return new Status_t(Error_CompCode.HAL, Error_Param.UNSUPPORTED_PARAMETER);
                }

                return new Status_t();
            }

            public Status_t Close()
            {
                if (m_bal != null)
                {
                    // Reset Reader IC Baudrate to 9600
                    if (m_hal.GetType() == typeof(NxpRdLibNet.Hal.Rc523))
                    {
                        m_hal.SetConfig(
                            NxpRdLibNet.Hal.Config.SERIAL_BITRATE,
                            (int)NxpRdLibNet.Hal.Rs232BitRate.VALUE_9600);
                    }
                    return m_bal.ClosePort();
                }
                else
                {
                    return new Status_t(Error_CompCode.HAL, Error_Param.USE_CONDITION);
                }
            }
        }

        private class SamAV2 : IReader
        {
            private CryptoSym.Sw m_cryptoSymForSamEnc;
            private CryptoSym.Sw m_cryptoSymForSamMac;
            private CryptoSym.Sw m_cryptoSymForSamDiv;
            private CryptoSym.Sw m_cryptoSymForSamRng;
            private CryptoRng.Sw m_cryptoRngForSam;

            private byte[] Atr;
            private byte[] m_HostAuthKey;
            private IReader m_Reader_NonX;
            private Hal.SamAV2 m_halSamLC0;
            private KeyStore.Sw m_KeyStoreForSam;

            public SamAV2(byte[] HostAuthKey, IReader Reader)
            {
                m_bal = null;
                m_hal = null;
                m_KeyStore = null;

                m_HostAuthKey = HostAuthKey;
                m_Reader_NonX = Reader;

                #region KeyStore (for HostAuth)

                m_KeyStoreForSam = new NxpRdLibNet.KeyStore.Sw();
                CheckSuccess(((KeyStore.Sw)m_KeyStoreForSam).Init(1, 1, 1));

                #endregion

                #region Crypto

                m_cryptoSymForSamEnc = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForSamMac = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForSamDiv = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForSamRng = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoRngForSam = new NxpRdLibNet.CryptoRng.Sw();

                CheckSuccess(m_cryptoSymForSamEnc.Init(m_KeyStoreForSam));
                CheckSuccess(m_cryptoSymForSamMac.Init(m_KeyStoreForSam));
                CheckSuccess(m_cryptoSymForSamDiv.Init(m_KeyStoreForSam));
                CheckSuccess(m_cryptoSymForSamRng.Init(m_KeyStoreForSam));
                CheckSuccess(m_cryptoRngForSam.Init(m_cryptoSymForSamRng));

                // Set random number generator
                System.Random rand = new Random();
                byte[] seed = new byte[16];
                rand.NextBytes(seed);
                CheckSuccess(m_cryptoRngForSam.Seed(seed));

                #endregion
            }

            ~SamAV2()
            {
                try
                {
                    Close();
                }
                catch { }
            }

            public Status_t Open()
            {
                string[] rd;

                // Instanciate PCSC BAL
                Bal.PcscWin balPcsc = new NxpRdLibNet.Bal.PcscWin();
                balPcsc.Init(out Atr);

                // Retrieve a list of connected readers
                CheckSuccess(balPcsc.GetPortList(10, out rd));
                if ((rd == null) || (rd.Length == 0))
                {
                    return new Status_t(Error_CompCode.BAL, Error_Comm.INTERFACE_ERROR);
                }

                int i = 0;
                Console.WriteLine("***********************");
                Console.WriteLine("Choose SAM PC/SC reader");
                Console.WriteLine("***********************");
                foreach (string Entry in rd)
                {
                    Console.WriteLine("* " + i++ + ") " + Entry);
                }
                Console.WriteLine("***********************");
                Console.Write("@:\\>");
                ConsoleKeyInfo key = Console.ReadKey(false);
                Console.Clear();
                if ((key.Key < ConsoleKey.D0) || (key.Key > (ConsoleKey.D0 + (rd.Length - 1))))
                {
                    return new Status_t(Error_CompCode.BAL, Error_Param.INVALID_PARAMETER);
                }
                i = (int)key.Key - (int)ConsoleKey.D0;


                if ((m_Reader_NonX != null) &&
                    (rd[i].Contains("NXP Pegoda 2")) &&
                    (m_Reader_NonX.Hal.GetType().Equals(typeof(NxpRdLibNet.Hal.Rd710))))
                {
                    #region BAL RD710 SAM

                    // Instanciate BAL
                    Bal.Rd710Sam balRd710Sam = new NxpRdLibNet.Bal.Rd710Sam();
                    CheckSuccess(balRd710Sam.Init(m_Reader_NonX.Bal, 256, 256, out Atr));

                    // Reset and Activate SAM
                    CheckSuccess(balRd710Sam.OpenPort());
                    CheckSuccess(balRd710Sam.ClosePort());
                    CheckSuccess(balRd710Sam.OpenPort());

                    // Use as Generic BAL
                    m_bal = balRd710Sam;

                    #endregion
                }
                else
                {
                    #region BAL PCSC

                    // Use the second found reader
                    CheckSuccess(balPcsc.SetPort(rd[i]));

                    // Open connection to the reader
                    CheckSuccess(balPcsc.OpenPort());

                    // Use as Generic BAL
                    m_bal = balPcsc;

                    #endregion
                }

                #region HAL SamAV2

                // Instanciate HAL LC0
                m_halSamLC0 = new NxpRdLibNet.Hal.SamAV2();
                CheckSuccess(m_halSamLC0.Init(
                    m_bal,
                    (m_Reader_NonX != null) ? m_Reader_NonX.Hal : null,
                    m_KeyStoreForSam,
                    m_cryptoSymForSamEnc,
                    m_cryptoSymForSamMac,
                    m_cryptoRngForSam,
                    (m_Reader_NonX != null) ? NxpRdLibNet.Hal.SamAV2.OpMode.NON_X : NxpRdLibNet.Hal.SamAV2.OpMode.X_RC523,
                    NxpRdLibNet.Hal.SamAV2.LogicalChannel.LC0));

                // Detect SAM Mode
                int HostMode;
                CheckSuccess(m_halSamLC0.DetectMode());
                CheckSuccess(m_halSamLC0.GetConfig(NxpRdLibNet.Hal.SamAV2.Config.HOSTMODE, out HostMode));

                // AV1 Mode
                if (HostMode == 0x01)
                {
                    CheckSuccess(m_KeyStoreForSam.FormatKeyEntry(0, NxpRdLibNet.KeyStore.KeyType.DES));
                    CheckSuccess(m_KeyStoreForSam.SetKey(0, 0, NxpRdLibNet.KeyStore.KeyType.DES, m_HostAuthKey, 0));

                    // Instanciate additional LC1 HAL
                    Hal.SamAV2 halSamLC1 = new NxpRdLibNet.Hal.SamAV2();
                    CheckSuccess(halSamLC1.Init(
                        m_bal,
                        (m_Reader_NonX != null) ? m_Reader_NonX.Hal : null,
                        m_KeyStoreForSam,
                        m_cryptoSymForSamEnc,
                        m_cryptoSymForSamMac,
                        m_cryptoRngForSam,
                        (m_Reader_NonX != null) ? NxpRdLibNet.Hal.SamAV2.OpMode.NON_X : NxpRdLibNet.Hal.SamAV2.OpMode.X_RC523,
                        NxpRdLibNet.Hal.SamAV2.LogicalChannel.LC1));

                    // Init. LC1 HAL
                    CheckSuccess(halSamLC1.DetectMode());

                    // Use LC1 as Generic HAL
                    m_hal = halSamLC1;
                }
                // AV2 Mode
                else
                {
                    CheckSuccess(m_KeyStoreForSam.FormatKeyEntry(0, NxpRdLibNet.KeyStore.KeyType.AES128));
                    CheckSuccess(m_KeyStoreForSam.SetKey(0, 0, NxpRdLibNet.KeyStore.KeyType.AES128, m_HostAuthKey, 0));

                    // Use LC0 as Generic HAL
                    m_hal = m_halSamLC0;
                }

                // Perform Host Authentication in LC0
                CheckSuccess(m_halSamLC0.Cmd_SAM_AuthenticateHost(
                    (NxpRdLibNet.Hal.SamAV2.AuthenticateHost_AuthMode)0x00,
                    0,
                    0,
                    0,
                    0,
                    new byte[0]));

                #endregion

                #region KeyStore SamAV2

                // Create SAM keystore
                m_KeyStore = new NxpRdLibNet.KeyStore.SamAV2();
                CheckSuccess(((NxpRdLibNet.KeyStore.SamAV2 ) m_KeyStore).Init(m_halSamLC0));

                // Set SAM keystore to only create PICC keys
                CheckSuccess(m_KeyStore.SetConfig((int)NxpRdLibNet.KeyStore.SamAV2.Config.KEYCLASS, 0x01));

                #endregion

                return new Status_t();
            }

            public Status_t Close()
            {
                if (m_bal != null)
                {
                    return m_bal.ClosePort();
                }
                else
                {
                    return new Status_t(Error_CompCode.HAL, Error_Param.USE_CONDITION);
                }
            }
        }

        private abstract class II14443aCardStack
        {
            protected palI14443p3a.Generic m_palI14443p3a;
            protected palI14443p4a.Generic m_palI14443p4a;
            protected palI14443p4.Generic m_palI14443p4;
            protected palMifare.Generic m_palMifare;
            protected alMfp.Generic m_alMfp;

            public palI14443p3a.Generic palI14443p3a { get { return m_palI14443p3a; } }
            public palI14443p4a.Generic palI14443p4a { get { return m_palI14443p4a; } }
            public palI14443p4.Generic palI14443p4 { get { return m_palI14443p4; } }
            public palMifare.Generic palMifare { get { return m_palMifare; } }
            public alMfp.Generic alMfp { get { return m_alMfp; } }
        }

        private class II14443aCardStack_NoSam : II14443aCardStack
        {
            private IReader m_Reader;
            private CryptoSym.Sw m_cryptoSymForMfpEnc;
            private CryptoSym.Sw m_cryptoSymForMfpMac;
            private CryptoSym.Sw m_cryptoSymForMfpDiv;
            private CryptoSym.Sw m_cryptoSymForRng;
            private CryptoRng.Sw m_cryptoRngForMfp;

            public II14443aCardStack_NoSam(IReader Reader)
            {
                m_Reader = Reader;

                #region Crypto

                m_cryptoSymForMfpEnc = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForMfpMac = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForMfpDiv = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoSymForRng = new NxpRdLibNet.CryptoSym.Sw();
                m_cryptoRngForMfp = new NxpRdLibNet.CryptoRng.Sw();

                CheckSuccess(m_cryptoSymForMfpEnc.Init(Reader.KeyStore));
                CheckSuccess(m_cryptoSymForMfpMac.Init(Reader.KeyStore));
                CheckSuccess(m_cryptoSymForMfpDiv.Init(Reader.KeyStore));
                CheckSuccess(m_cryptoSymForRng.Init(Reader.KeyStore));
                CheckSuccess(m_cryptoRngForMfp.Init(m_cryptoSymForRng));

                // Set random number generator
                System.Random rand = new Random();
                byte[] seed = new byte[16];
                rand.NextBytes(seed);
                CheckSuccess(m_cryptoRngForMfp.Seed(seed));

                #endregion

                #region palI14443p3a

                m_palI14443p3a = new NxpRdLibNet.palI14443p3a.Sw();
                CheckSuccess(((palI14443p3a.Sw)m_palI14443p3a).Init(m_Reader.Hal));

                #endregion

                #region palI14443p4a

                m_palI14443p4a = new NxpRdLibNet.palI14443p4a.Sw();
                CheckSuccess(((palI14443p4a.Sw)m_palI14443p4a).Init(m_Reader.Hal));

                #endregion

                #region palI14443p4

                m_palI14443p4 = new NxpRdLibNet.palI14443p4.Sw();
                CheckSuccess(((palI14443p4.Sw)m_palI14443p4).Init(m_Reader.Hal));

                #endregion

                #region palMifareSw

                m_palMifare = new NxpRdLibNet.palMifare.Sw();
                CheckSuccess(((palMifare.Sw)m_palMifare).Init(m_Reader.Hal, m_palI14443p4));

                #endregion

                #region alMfp

                m_alMfp = new NxpRdLibNet.alMfp.Sw();
                CheckSuccess(((alMfp.Sw)m_alMfp).Init(
                    m_palMifare,
                    Reader.KeyStore,
                    m_cryptoSymForMfpEnc,
                    m_cryptoSymForMfpMac,
                    m_cryptoRngForMfp,
                    m_cryptoSymForMfpDiv));

                #endregion
            }
        }

        private class II14443aCardStack_Sam_NonX : II14443aCardStack
        {
            private IReader m_Reader;

            public II14443aCardStack_Sam_NonX(IReader Reader)
            {
                m_Reader = Reader;

                #region palI14443p3a

                m_palI14443p3a = new NxpRdLibNet.palI14443p3a.Sw();
                CheckSuccess(((palI14443p3a.Sw)m_palI14443p3a).Init(m_Reader.Hal));

                #endregion

                #region palI14443p4a

                m_palI14443p4a = new NxpRdLibNet.palI14443p4a.Sw();
                CheckSuccess(((palI14443p4a.Sw)m_palI14443p4a).Init(m_Reader.Hal));

                #endregion

                #region palI14443p4

                m_palI14443p4 = new NxpRdLibNet.palI14443p4.Sw();
                CheckSuccess(((palI14443p4.Sw)m_palI14443p4).Init(m_Reader.Hal));

                #endregion

                #region palMifareSw

                m_palMifare = new NxpRdLibNet.palMifare.Sw();
                CheckSuccess(((palMifare.Sw)m_palMifare).Init(m_Reader.Hal, m_palI14443p4));

                #endregion

                #region alMfp

                m_alMfp = new NxpRdLibNet.alMfp.SamAV2();
                CheckSuccess(((alMfp.SamAV2)m_alMfp).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal,
                    m_palMifare));

                #endregion
            }
        }

        private class II14443aCardStack_SamX : II14443aCardStack
        {
            private IReader m_Reader;

            public II14443aCardStack_SamX(IReader Reader)
            {
                m_Reader = Reader;

                #region palI14443p3a

                m_palI14443p3a = new NxpRdLibNet.palI14443p3a.SamAV2_X();
                CheckSuccess(((palI14443p3a.SamAV2_X)m_palI14443p3a).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal)
                    );

                #endregion

                #region palI14443p4a

                m_palI14443p4a = new NxpRdLibNet.palI14443p4a.SamAV2_X();
                CheckSuccess(((palI14443p4a.SamAV2_X)m_palI14443p4a).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal)
                    );

                #endregion

                #region palI14443p4

                m_palI14443p4 = new NxpRdLibNet.palI14443p4.SamAV2_X();
                CheckSuccess(((palI14443p4.SamAV2_X)m_palI14443p4).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal)
                    );

                #endregion

                #region palMifareSw

                m_palMifare = new NxpRdLibNet.palMifare.SamAV2_X();
                CheckSuccess(((palMifare.SamAV2_X)m_palMifare).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal,
                    m_palI14443p4));

                #endregion

                #region alMfp

                m_alMfp = new NxpRdLibNet.alMfp.SamAV2_X();
                CheckSuccess(((alMfp.SamAV2_X)m_alMfp).Init(
                    (NxpRdLibNet.Hal.SamAV2)m_Reader.Hal,
                    (NxpRdLibNet.palMifare.SamAV2_X)m_palMifare));

                #endregion
            }
        }

        public static void MifarePlusSL3()
        {
            byte[] Uid;
            byte Sak;
            byte MoreCardsAvailable;
            byte[] Ats;
            byte CidEnabled;
            byte Cid;
            byte NadSupported;
            byte Fwi;
            byte Fsdi;
            byte Fsci;
            byte[] PcdCap2Out;
            byte[] PdCap;
            byte[] MfpKey = new byte[] { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
            byte[] SamHostKey = new byte[] { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
            byte[] Blocks;

            IReader ReaderGen;
            Reader ReaderClassic = null;
            SamAV2 ReaderSam = null;
            II14443aCardStack CardStack;

            Console.WriteLine("***********************");
            Console.WriteLine("Choose SAM Mode");
            Console.WriteLine("***********************");
            Console.WriteLine("* 0) No Sam");
            Console.WriteLine("* 1) Sam in Non-X Mode");
            Console.WriteLine("* 2) Sam in X Mode");
            Console.WriteLine("***********************");
            Console.Write("@:\\>");
            ConsoleKeyInfo keySamMode = Console.ReadKey(false);
            Console.Clear();

            switch (keySamMode.Key)
            {
                case ConsoleKey.D0:
                case ConsoleKey.D1:

                    Console.WriteLine("***********************");
                    Console.WriteLine("Choose Reader");
                    Console.WriteLine("***********************");
                    Console.WriteLine("* 0) Pegoda 1 (Rd70x)");
                    Console.WriteLine("* 1) Pegoda 2 (Rd710)");
                    Console.WriteLine("* 2) Joiner (Rc523 / Pn512)");
                    Console.WriteLine("* 3) Rc663 (Eval. Board)");
                    Console.WriteLine("***********************");
                    Console.Write("@:\\>");
                    ConsoleKeyInfo keyReader = Console.ReadKey(false);
                    Console.Clear();

                    // Create Classic Reader
                    ReaderClassic = new Reader();
                    CheckSuccess(ReaderClassic.Open((Reader.ReaderType)keyReader.Key));

                    // Create SAM and link it to Reader
                    if (keySamMode.Key == ConsoleKey.D1)
                    {
                        ReaderSam = new SamAV2(SamHostKey, ReaderClassic);
                        CheckSuccess(ReaderSam.Open());

                        // Sam becomes the generic Reader object
                        ReaderGen = ReaderSam;

                        // Create Sam Non-X CardStack
                        CardStack = new II14443aCardStack_Sam_NonX(ReaderGen);
                    }
                    else
                    {
                        // Classic Reader becomes the generic Reader object
                        ReaderGen = ReaderClassic;

                        // Create NoSam CardStack
                        CardStack = new II14443aCardStack_NoSam(ReaderGen);
                    }

                    break;
                case ConsoleKey.D2:

                    // Create SAM without reader connection
                    ReaderSam = new SamAV2(SamHostKey, null);
                    CheckSuccess(ReaderSam.Open());

                    // Sam becomes the generic Reader object
                    ReaderGen = ReaderSam;

                    // Create SamX CardStack
                    CardStack = new II14443aCardStack_SamX(ReaderGen);

                    break;
                default:
                    CheckSuccess(new Status_t(Error_CompCode.BAL, Error_Param.INVALID_PARAMETER));
                    return;
            }

            // Prepare reader for ISO14443A
            CheckSuccess(ReaderGen.Hal.ApplyProtocolSettings(NxpRdLibNet.Hal.CardType.ISO14443A));

            // FieldReset
            CheckSuccess(ReaderGen.Hal.FieldReset());

            // Activate Layer3
            CheckSuccess(CardStack.palI14443p3a.ActivateCard(null, out Uid, out Sak, out MoreCardsAvailable));

            // Activate Layer4
            CheckSuccess(CardStack.palI14443p4a.ActivateCard(8, 1, 0, 0, out Ats));

            // Link protocol parameters from 4A layer
            CheckSuccess(CardStack.palI14443p4a.GetProtocolParams(out CidEnabled, out Cid, out NadSupported, out Fwi, out Fsdi, out Fsci));
            CheckSuccess(CardStack.palI14443p4.SetProtocol(CidEnabled, Cid, 0, 0, Fwi, Fsdi, Fsci));

            // Write MFP key into keystore
            CheckSuccess(ReaderGen.KeyStore.FormatKeyEntry(2, NxpRdLibNet.KeyStore.KeyType.AES128));
            CheckSuccess(ReaderGen.KeyStore.SetKey(2, 0, NxpRdLibNet.KeyStore.KeyType.AES128, MfpKey, 0));

            // Authenticate SL3 in Sector 2, KeyA
            Console.Write ( "Authentication with AES Sector 0 A key " );
            CheckSuccess (CardStack.alMfp.AuthenticateSL3(1, 0x4000, 2, 0, null, null, out PcdCap2Out, out PdCap));
            Console.WriteLine ( "Success" );

            // Read block 0 of Sector 2
            CheckSuccess(CardStack.alMfp.Read(1, 1, 1, 0x0004, 1, out Blocks));
            Console.WriteLine ( "Data from Block 0x04: " + BitConverter.ToString ( Blocks ).Replace ( "-", " " ) );

            if (ReaderClassic != null)
            {
                CheckSuccess(ReaderClassic.Close());
            }
            if (ReaderSam != null)
            {
                CheckSuccess(ReaderSam.Close());
            }
        }
    }
}
