/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alFelica
{
    #region DEFINES

    public enum Config : int
    {
        ADD_INFO = 0x0000
    }

    public enum Error : byte
    {
        /// <summary> Custom Errorcode for Felica application layer. </summary>
        FELICA = (CustomCodes.ERROR_BEGIN)
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DEFINES
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_RequestResponse(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ref byte pMode          /**< [OUT] Current Card Mode. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_RequestService(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte   bTxNumServices,      /**< [IN] */
            byte[] pTxServiceList,      /**< [IN] */
            ref byte pRxNumServices,    /**< [OUT] */
            byte[] pRxServiceList       /**< [OUT] */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_Read(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bNumServices,          /**< [IN] */
            byte[] pServiceList,        /**< [IN] */
            byte bTxNumBlocks,          /**< [IN] */
            byte[] pBlockList,          /**< [IN] */
            byte bBlockListLength,      /**< [IN] */
            ref byte pRxNumBlocks,      /**< [OUT] */
            byte[] pBlockData           /**< [OUT] */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_Write(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bNumServices,          /**< [IN] */
            byte[] pServiceList,        /**< [IN] */
            byte bNumBlocks,            /**< [IN] */
            byte[] pBlockList,          /**< [IN] */
            byte bBlockListLength,      /**< [IN] */
            byte[] pBlockData           /**< [IN] */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_GetConfig(
            IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,       /**< [In] Configuration Identifier */
            ref ushort pValue     /**< [Out] Configuration Value */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t RequestResponse(
            out byte bMode
            )
        {
            bMode = 0;
            return phalFelica_RequestResponse(m_pDataParams, ref bMode);
        }

        public Status_t RequestService(
            byte bTxNumServices,
            byte[] pTxServiceList,
            out byte pRxNumServices,
            out byte[] pRxServiceList
            )
        {
            Status_t status;
            pRxNumServices = 0;
            pRxServiceList = new byte[64];

            status = phalFelica_RequestService(m_pDataParams, bTxNumServices, pTxServiceList, ref pRxNumServices, pRxServiceList);
            Array.Resize<byte>(ref pRxServiceList, 2 * pRxNumServices);
            return status;
        }

        public Status_t Read(
            byte bNumServices,
            byte[] pServiceList,
            byte bTxNumBlocks,
            byte[] pBlockList,
            out byte[] pBlockData
            )
        {
            Status_t status;
            byte pRxNumBlocks = 0;
            pBlockData = new byte[16*12];

            status = phalFelica_Read(m_pDataParams, bNumServices, pServiceList, bTxNumBlocks, pBlockList, (byte)((pBlockList == null) ? 0 : pBlockList.Length), ref pRxNumBlocks, pBlockData);
            Array.Resize<byte>(ref pBlockData, 16 * pRxNumBlocks);
            return status;
        }

        public Status_t Write(
            byte bNumServices,
            byte[] pServiceList,
            byte bNumBlocks,
            byte[] pBlockList,
            byte[] pBlockData
            )
        {
            return phalFelica_Write(m_pDataParams, bNumServices, pServiceList, bNumBlocks, pBlockList, (byte)((pBlockList == null) ? 0 : pBlockList.Length), pBlockData);
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phalFelica_GetConfig(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = (int)wValue;
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }
    #endregion

    #region SW

    public class Sw : alFelica.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pPalDataParams;   /**< Pointer to the parameter structure of the underlying layer.  */
			public ushort wAdditionalInfo;     /**< Holds the last status flags sent by the Card. */
        };
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalFelica_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init(palFelica.Generic pPalFelicaDataParams)
        {
            return phalFelica_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pPalFelicaDataParams.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, palFelica.Generic pPalFelicaDataParams)
        {
            return phalFelica_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pPalFelicaDataParams.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion
}