﻿/*
 * Copyright 2017 - 2018, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alICode
{
    #region Enumerations
    #region Error
    /// <summary>
    /// Custom error codes for ISO15693 response code.
    /// </summary>
    public enum Error : byte
    {
        /// <summary> Custom commands Error codes. </summary>
        CUSTOM_COMMANDS_ERROR = ( CustomCodes.ERROR_BEGIN + 0 ),

        /// <summary> The command is not supported, i.e. the request code is not recognized. </summary>
        COMMAND_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 1 ),

        /// <summary> The command is not recognized, for example: a format error occurred. </summary>
        COMMAND_NOT_RECOGNIZED = ( CustomCodes.ERROR_BEGIN + 2 ),

        /// <summary> The command option is not supported. </summary>
        COMMAND_OPTION_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 3 ),

        /// <summary> Error with no information given or a specific error code is not supported. </summary>
        NO_INFORMATION = ( CustomCodes.ERROR_BEGIN + 4 ),

        /// <summary> The specified block is not available (doesn’t exist). </summary>
        BLOCK_NOT_AVAILABLE = ( CustomCodes.ERROR_BEGIN + 5 ),

        /// <summary> The specified block is already locked and thus cannot be locked again. </summary>
        BLOCK_LOCKED = ( CustomCodes.ERROR_BEGIN + 6 ),

        /// <summary> The specified block is locked and its content cannot be changed. </summary>
        CONTENT_CHANGE_FAILURE = ( CustomCodes.ERROR_BEGIN + 7 ),

        /// <summary> The specified block was not successfully programmed. </summary>
        BLOCK_PROGRAMMING_FAILURE = ( CustomCodes.ERROR_BEGIN + 8 ),

        /// <summary> The specified block was not successfully locked. </summary>
        BLOCK_NOT_LOCKED = ( CustomCodes.ERROR_BEGIN + 9 ),

        /// <summary> The specified block is protected. </summary>
        BLOCK_PROTECTED = ( CustomCodes.ERROR_BEGIN + 10 ),

        /// <summary> Generic cryptographic error. </summary>
        GENERIC_CRYPTO_ERROR = ( CustomCodes.ERROR_BEGIN + 11 ),
    }
    #endregion

    #region Flags
    /// <summary>
    /// Flags that are common and are part of bits 1 - 4 ( i.e. 0 - 3)\. These flags will be utilized
    /// for the below mentioned interfaces.
    ///     - <see cref="Generic.Inventory"/>
    ///     - <see cref="Generic.InventoryRead"/>
    ///     - <see cref="Generic.InventoryReadExtended"/>
    ///     - <see cref="Generic.FastInventoryRead"/>
    ///     - <see cref="Generic.InventoryPageRead"/>
    ///     - <see cref="Generic.FastInventoryPageRead"/>
    ///     - <see cref="Generic.GetConfig"/>
    ///     - <see cref="Generic.SetConfig"/>
    /// </summary>
    public enum Flags : byte
    {
        /// <summary>
        /// Sub-carrier_flag. If set two sub-carriers shall be used (VICC to VCD). Else a single sub-carrier
        /// frequency shall be used by the VICC.
        /// </summary>
        TWO_SUB_CARRIERS = 0x01,

        /// <summary>
        /// Data_rate_flag. If set high data rate shall be used (VICC to VCD). Else low data rate shall
        /// be used.
        /// </summary>
        DATA_RATE = 0x02,

        /// <summary>
        /// Inventory_flag. Shall be set for inventory request.
        /// If set flags 5 to 8 meaning is as per AFI, NBSLOTS and OPTION flags.
        /// If not set flags 5 to 8 meaning is as per SELECTED, ADDRESSED and OPTION flags.
        /// </summary>
        INVENTORY = 0x04,

        /// <summary>
        /// Protocol_Extension_flag. If set protocol format is extended. Reserved for future use. No supported
        /// in current version.
        /// </summary>
        PROTOCOL_EXTENSION = 0x08,

        /// <summary>
        /// Select_flag. If set only VICC in selected state shall respond. The Address_flag ADDRESSED shall be set to
        /// 0 and the UID field will not be included in the request. If not set Request shall be executed by any VICC
        /// according to the setting of Address_flag ADDRESSED.
        /// </summary>
        SELECTED = 0x10,

        /// <summary> AFI_flag. If set AFI field is present in request. If not set AFI field is not present in request. </summary>
        AFI = 0x10,

        /// <summary>
        /// Address_flag. If set request is addressed. UID field is included in request. It shall be executed only by the
        /// VICC whose UID matches the UID specified in the request. If not set request is not addressed. UID field is not
        /// included in request. It shall be executed by any VICC.
        /// </summary>
        ADDRESSED = 0x20,

        /// <summary> Nb_slots_flag. If set Use 1 slot instead of 16 slots for inventory. If not set use 16 slots. </summary>
        NBSLOTS = 0x20,

        /// <summary>
        /// Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by
        /// the command.
        /// </summary>
        OPTION = 0x40,

        /// <summary>
        /// RFU as per ISO15693 but this flag should be utilized for PLUTUS product only. This flag is used to set the
        /// hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps). Also to note that this flag should
        /// not be used for any Inventory commands. Currently the support is for
        /// <seealso cref="NxpRdLibNet.alICode.Generic.FastReadMultipleBlocks"/> and
        /// <seealso cref="NxpRdLibNet.alICode.Generic.ExtendedFastReadMultipleBlocks"/>.
        /// </summary>
        FAST_DATA_RATE = 0x80
    }
    #endregion

    #region Option
    /// <summary>
    /// Description for Option definitions that will be used in Mandatory, Optional and Custom set of
    /// commands.When these definitions are used it will update the Flags 7th bit (<see cref="Flags.OPTION">Option</see>)
    /// </summary>
    public enum Option
    {
        /// <summary> Option Bit is cleared. </summary>
        OFF = 0x00,

        /// <summary> Option bit is set </summary>
        ON = 0x01,

        /// <summary> Default Option. This flag represents OFF value. </summary>
        DEFAULT = OFF
    }
    #endregion

    #region Optional
    #region TAM Authentication Type
    /// <summary>
    /// Options to be used for <see cref="Generic.AuthenticateTAM"/> command.
    /// </summary>
    public enum AuthType_TAM : byte
    {
        /// <summary>
        /// Option to indicate Authentication type as TAM1.
        /// The command framing will be according to ISO/IEC 29167 specification.
        /// </summary>
        TAM1 = 0,

        /// <summary>
        /// Option to indicate Authentication type as AUTH1 or FAST_AUTH1 (TAM1.2).
        /// The command framing will be according to LETO product specification.
        /// </summary>
        TAM1_2 = 1,

        /// <summary>
        /// Option to indicate Authentication type as AUTH2 or FAST_AUTH2 (TAM1.3).
        /// The command framing will be according to LETO product specification.
        /// </summary>
        TAM1_3 = 2,

        /// <summary>
        /// Option to indicate Authentication type as TAM2.
        /// The command framing will be according to ISO/IEC 29167 specification.
        /// </summary>
        TAM2 = 4
    }
    #endregion

    #region Diversification
    /// <summary>
    /// Options to be used for <see cref="Generic.AuthenticateTAM"/> command.
    /// </summary>
    public enum Diversification : ushort
    {
        /// <summary> Option to indicate diversification is disabled. </summary>
        OFF = 0x0000,

        /// <summary> Option to indicate diversification is enabled. </summary>
        ON = 0x0010
    }
    #endregion

    #region GenerateRnd
    /// <summary>
    /// Options to be used for below mentioned commands.
    ///     - <see cref="Generic.AuthenticateTAM"/>
    ///     - <see cref="Generic.Challenge"/>
    /// </summary>
    public enum GenerateRnd : ushort
    {
        /// <summary> Option to generate the random number internally. </summary>
        INTERNAL = 0x0000,

        /// <summary> Option to use the random number provided by user. </summary>
        EXTERNAL = 0x0100
    }
    #endregion

    #region ChallengeType
    /// <summary>
    /// Options to be used for below mentioned commands.
    ///     - <see cref="Generic.Challenge"/>
    /// </summary>
    public enum ChallengeType : byte
    {
        /// <summary>
        /// Option to indicate Challenge type as TAM1.
        /// The command framing will be according to ISO/IEC 29167 specification.
        /// </summary>
        TAM1 = 0,

        /// <summary>
        /// Option to indicate Challenge type as CHALLENGE1 (TAM1.2).
        /// The command framing will be according to LETO product specification.
        /// </summary>
        TAM1_2 = 1,

        /// <summary>
        /// Option to indicate Challenge type as CHALLENGE2 (TAM1.3).
        /// The command framing will be according to LETO product specification.
        /// </summary>
        TAM1_3 = 2,
    }
    #endregion

    #region PurposeMAM2
    /// <summary>
    /// Options to be used for <see cref="Generic.AuthenticateMAM">AuthenticateMAM</see> command.
    /// </summary>
    public enum PurposeMAM2 : byte
    {
        /// <summary> Option to indicate no PurposeMAM2 value. </summary>
        AUTHENTICATE = 0x00,

        /// <summary> Option to indicate PurposeMAM2 value as Privacy disable until HF reset. </summary>
        DISABLE_PRIVACY_HF_RESET = 0x08,

        /// <summary> Option to indicate PurposeMAM2 value as Privacy enable. </summary>
        ENABLE_PRIVACY = 0x09,

        /// <summary> Option to indicate PurposeMAM2 value as Privacy disable. </summary>
        DISABLE_PRIVACY = 0x0A,

        /// <summary> Option to indicate PurposeMAM2 value as Destroy. </summary>
        DESTROY = 0x0B
    }
    #endregion

    #region Verify
    /// <summary>
    /// Options to be used for <see cref="Generic.ReadBuffer"/> command.
    /// </summary>
    public enum Verify : byte
    {
        /// <summary> Option to return the received response from VICC without verification and decryption. </summary>
        OFF = 0x00,

        /// <summary> Option to return the received response from VICC with verification and decryption. </summary>
        ON = 0x01
    }
    #endregion

    #region InfoParams
    /// <summary>
    /// Options to be used for <see cref="Generic.ExtendedGetSystemInformation"/> command.
    /// </summary>
    public enum InfoParams : byte
    {
        /// <summary> Option to indicate the default value of VICC response with memory addressing. </summary>
        DEFAULT = 0x10,

        /// <summary> Option to indicate the return of DSFID information from VICC. </summary>
        REQUEST_DSFID = 0x01,

        /// <summary> Option to indicate the return of AFI information from VICC. </summary>
        REQUEST_AFI = 0x02,

        /// <summary> Option to indicate the return of VICC memory size information from VICC. </summary>
        REQUEST_VICC_MEM_SIZE = 0x04,

        /// <summary> Option to indicate the return of IC Reference information from VICC. </summary>
        REQUEST_IC_REFERENCE = 0x08,

        /// <summary> Option to indicate the return of MOI (memory addressing) information from VICC. </summary>
        REQUEST_MOI = 0x10,

        /// <summary> Option to indicate the return of Supported Command List information from VICC. </summary>
        REQUEST_COMMAND_LIST = 0x20,

        /// <summary> Option to indicate the return of CSI information from VICC. </summary>
        REQUEST_CSI_INFORMATION = 0x40,

        /// <summary> Option to indicate the return of Two byte length of Ext Get System information from VICC. </summary>
        REQUEST_EXT_GET_SYS_INFO = 0x80
    }
    #endregion
    #endregion

    #region Custom
    #region InventoryReadExt
    /// <summary>
    /// Options to be used with <see cref="Generic.InventoryReadExtended"/> command.
    /// </summary>
    public enum InventoryReadExt : byte
    {
        /// <summary> Default option for InventoryReadExtended command. </summary>
        DEFAULT = 0x00,

        /// <summary> Option to indicate that the VICC with EAS enabled will respond. </summary>
        EAS_MODE = 0x01,

        /// <summary> Option to indicate complete UID transmission. </summary>
        UID_MODE = 0x02,

        /// <summary> Option to indicate transmission of 16 bit CID and only tags with the same CID will respond. </summary>
        CID_COMPARE = 0x04,

        /// <summary> Option to indicate transmission of Custom ID. </summary>
        CID_RESPONSE = 0x08,

        /// <summary> Option to indicate No user memory data are requested from the tag, first block number byte and number of
        /// blocks byte shall not be transmitted in the command. </summary>
        SKIP_DATA = 0x10,

        /// <summary> Option to indicate the state change to Quiet after response. </summary>
        QUIET = 0x20,

        /// <summary> Option to indicate the state change to Persistent Quiet after response. </summary>
        PERSIST_QUIET = 0x40,

        /// <summary> Option to indicate that tags in the PERSISTENT QUIET state will respond to the command. </summary>
        PERSIST_QUIET_RESPONSE = 0x60
    }
    #endregion

    #region Password
    /// <summary>
    /// Password option for the below interfaces.
    ///     - <see cref="Generic.SetPassword"/>
    ///     - <see cref="Generic.WritePassword"/>
    ///     - <see cref="Generic.LockPassword"/>
    /// </summary>
    public enum Password : byte
    {
        /// <summary> Option to indicate Password Identifier as Read. </summary>
        READ = 0x01,

        /// <summary> Option to indicate Password Identifier as Write. </summary>
        WRITE = 0x02,

        /// <summary> Option to indicate Password Identifier as Privacy. </summary>
        PRIVACY = 0x04,

        /// <summary> Option to indicate Password Identifier as Write access to Usage Counter. </summary>
        USAGECNT_WRITE = 0x06,

        /// <summary> Option to indicate Password Identifier as Destroy. </summary>
        DESTROY = 0x08,

        /// <summary> Option to indicate Password Identifier as EAS / AFI. </summary>
        EAS_AFI = 0x10
    }
    #endregion

    #region ProtectPage
    /// <summary>
    /// Describes the definitions for <see cref="Generic.ProtectPage"/> command.
    /// </summary>
    public enum ProtectPage : byte
    {
        #region Protection Status
        /// <summary> Option to indicate protection status as public. </summary>
        PUBLIC = 0x00,

        /// <summary>
        /// Option to indicate protection status as
        ///     - 32-bit Password Protection: Read and Write protection by Read password.
        ///     - 64-bit Password Protection: Read and Write protection by Read plus Write password.
        ///     - iCode DNA or other products that supports AES authentication, Mutual authentication
        ///       with a key with read privilege is required.
        ///     </summary>
        READ_WRITE_READ_PASSWORD = 0x01,

        /// <summary>
        /// Option to indicate protection status as
        ///     - 32-bit Password Protection: Write protection by Read password.
        ///     - 64-bit Password Protection: Write protection by Read plus Write password.
        ///     - iCode DNA or other products that supports AES authentication, Mutual authentication
        ///       with a key with write privilege is required.
        /// </summary>
        WRITE_PASSWORD = 0x10,

        /// <summary>
        /// Option to indicate protection status as
        ///     - 32-bit Password Protection: Read protected by Read and Write protection by Write password.
        ///     - 64-bit Password Protection: Read and Write protection by Read plus Write password.
        ///     - iCode DNA or other products that supports AES authentication, Mutual authentication
        ///       with a key with read and write privileges is required.
        /// </summary>
        READ_WRITE_PASSWORD_SEPERATE = 0x11,
        #endregion Protection Status

        #region Extended Protection Status
        /// <summary> Option to indicate Read protection for Lower pages. </summary>
        READ_LOW = 0x01,

        /// <summary> Option to indicate Write protection for Lower pages. </summary>
        WRITE_LOW = 0x02,

        /// <summary> Option to indicate Read protection for Higher pages. </summary>
        READ_HIGH = 0x10,

        /// <summary> Option to indicate Write protection for Higher pages. </summary>
        WRITE_HIGH = 0x20
        #endregion Extended Protection Status
    }
    #endregion

    #region Parameters
    /// <summary>
    /// BitRate and Timing Options to be used for below interfaces.
    ///     - <see cref="Generic.ParameterRequest"/>
    ///     - <see cref="Generic.ParameterSelect"/>
    /// </summary>
    public enum Parameters : byte
    {
        #region BitRate Options
        /// <summary> Option to indicate the 26kbps BitRate support in both directions.</summary>
        BITRATE_26KBPS_BOTH_DIRECTIONS = 0x00,

        /// <summary> Option to indicate the 53kbps BitRate support from VCD to VICC.</summary>
        BITRATE_53KBPS_VCD_VICC = 0x01,

        /// <summary> Option to indicate the 106kbps BitRate support from VCD to VICC.</summary>
        BITRATE_106KBPS_VCD_VICC = 0x02,

        /// <summary> Option to indicate the 212kbps BitRate support from VCD to VICC.</summary>
        BITRATE_212KBPS_VCD_VICC = 0x04,

        /// <summary> Option to indicate the 53kbps BitRate support from VICC to VCD.</summary>
        BITRATE_53KBPS_VICC_VCD = 0x10,

        /// <summary> Option to indicate the 106kbps BitRate support from VICC to VCD.</summary>
        BITRATE_106KBPS_VICC_VCD = 0x20,

        /// <summary> Option to indicate the 212kbps BitRate support from VICC to VCD.</summary>
        BITRATE_212KBPS_VICC_VCD = 0x40,

        /// <summary> Option to indicate the BitRate value as RFU(s).</summary>
        BITRATE_RFU = 0x88,

        /// <summary> Option to indicate the BitRate value as None in case of error or invalid value.</summary>
        BITRATE_NONE = 0xFF,
        #endregion BitRate Options

        #region Timing Options
        /// <summary> Option to indicate that VICC supports 320.9us timing only.</summary>
        TIMING_320_9_US = 0x00,

        /// <summary> Option to indicate that VICC supports 160.5us timing only.</summary>
        TIMING_160_5_US = 0x01,

        /// <summary> Option to indicate that VICC supports 80.2us timing only.</summary>
        TIMING_80_2_US = 0x02,

        /// <summary> Option to indicate that same timing on both the direction is compulsory.</summary>
        TIMING_SAME_BOTH_DIRECTIONS = 0x10,

        /// <summary> Option to indicate the timing value as RFU(s).</summary>
        TIMING_RFU = 0xEC,

        /// <summary> Option to indicate the timing value as None in case of error or invalid value.</summary>
        TIMING_NONE = 0xFF
        #endregion Timing Options
    }
    #endregion
    #endregion

    #region Config
    /// <summary>
    /// Describes the supported options for <see cref="Generic.SetConfig"/> and <see cref="Generic.GetConfig"/> interfaces.
    /// </summary>
    public enum Config : ushort
    {
        /// <summary> Option to Get / Set the Flags information. </summary>
        FLAGS = 0x00,

        /// <summary> Option to Get the additional information. </summary>
        ADD_INFO = 0x01,

        /// <summary> Option to Get the timeout value in milliseconds. </summary>
        TIMEOUT_US = 0x02,

        /// <summary> Option to Get the timeout value in microseconds. </summary>
        TIMEOUT_MS = 0x03,

        /// <summary>
        /// Option to Get / Set the buffering of response data to be enabled or disabled.
        /// By default this flag will be disabled.
        /// </summary>
        ENABLE_BUFFERING = 0x04,

        /// <summary>
        /// Option to Get / Set the timeout value in microsecond for Challenge command.
        /// </summary>
        CHALLENGE_TIMEOUT_US = 0x05,

        /// <summary>
        /// Option to Get / Set the timeout value in milliseconds for Challenge command
        /// </summary>
        CHALLENGE_TIMEOUT_MS = 0x06
    }
    #endregion

    #region TagType
    /// <summary>
    /// Describes the supported options for <see cref="Generic.GetTagType"/> interface.
    /// </summary>
    public enum TagType : ushort
    {
        /// <summary> Option to Get the tag type as Unknown. </summary>
        UNKNOWN = 0xFFFF,

        /// <summary> Option to Get the tag type as iCode SLI. </summary>
        ICODE_SLI = 0x0001,

        /// <summary> Option to Get the tag type as iCode SLI-S. </summary>
        ICODE_SLI_S = 0x0002,

        /// <summary> Option to Get the tag type as iCode SLI-L. </summary>
        ICODE_SLI_L = 0x0003,

        /// <summary> Option to Get the tag type as iCode SLIX. </summary>
        ICODE_SLIX = 0x5001,

        /// <summary> Option to Get the tag type as iCode SLIX-S. </summary>
        ICODE_SLIX_S = 0x5002,

        /// <summary> Option to Get the tag type as iCode SLIX-L. </summary>
        ICODE_SLIX_L = 0x5003,

        /// <summary> Option to Get the tag type as iCode SLIX2. </summary>
        ICODE_SLI_X2 = 0x0801,

        /// <summary> Option to Get the tag type as iCode SLIX3. </summary>
        ICODE_SLI_X3 = 0x2001,

        /// <summary> Option to Get the tag type as iCode DNA. </summary>
        ICODE_DNA = 0x1801,

        /// <summary> Option to Get the tag type as iCode NTAG5. </summary>
        ICODE_NTAG5 = 0x5801
    }
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Generic iCode application component of Reader Library .Net framework.
    /// </summary>
    public abstract class Generic
    {
        #region DLL Imports
        #region Mandatory Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_Inventory ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            ref byte pDsfid, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_StayQuiet ( IntPtr pDataParams );
        #endregion Mandatory Commands

        #region Optional Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadSingleBlock ( IntPtr pDataParams, byte bOption, byte bBlockNo, ref IntPtr ppData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteSingleBlock ( IntPtr pDataParams, byte bOption, byte bBlockNo, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockBlock ( IntPtr pDataParams, byte bOption, byte bBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadMultipleBlocks ( IntPtr pDataParams, byte bOption, byte bBlockNo, byte bNumBlocks, IntPtr pData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_Select ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ResetToReady ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteAFI ( IntPtr pDataParams, byte bOption, byte bAfi );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockAFI ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteDSFID ( IntPtr pDataParams, byte bOption, byte bDsfid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockDSFID ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetSystemInformation ( IntPtr pDataParams, ref IntPtr ppSystemInfo, ref ushort pSystemInfoLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetMultipleBlockSecurityStatus ( IntPtr pDataParams, byte bBlockNo, byte bNoOfBlocks, IntPtr pStatus,
            ref ushort pStatusLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastReadMultipleBlocks ( IntPtr pDataParams, byte bOption, byte bBlockNo, byte bNumBlocks, IntPtr pData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedReadSingleBlock ( IntPtr pDataParams, byte bOption, ushort wBlockNo, ref IntPtr ppData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedWriteSingleBlock ( IntPtr pDataParams, byte bOption, ushort wBlockNo, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedLockBlock ( IntPtr pDataParams, byte bOption, ushort wBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedReadMultipleBlocks ( IntPtr pDataParams, byte bOption, ushort wBlockNo, ushort wNumBlocks,
            IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_AuthenticateTAM ( IntPtr pDataParams, ushort wOption, byte bAuthType, byte bKeyNo, byte bKeyVer,
            byte bKeyID_CertNum, byte[] pDivInput, byte bDivLen, byte[] pChallenge, ushort wChallengeLen, byte[] pAddInfo, ushort wAddInfoLen,
            ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_AuthenticateTAM1 ( IntPtr pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard,
            byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_AuthenticateTAM2 ( IntPtr pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard,
            byte[] pDivInput, byte bDivLen, byte bBlockSize, byte bBlockCount, byte bProfile, byte bProtMode, ushort wOffset, byte[] pCustomData,
            ref ushort pCustomDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_AuthenticateMAM ( IntPtr pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard,
            byte bPurposeMAM2, byte[] pDivInput, byte bDivLen );

#if PACKAGE_INTERNAL
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteCertificate ( IntPtr pDataParams, byte bOption, byte bCertNo, ushort wWordPtr, byte[] pCertificate,
            byte bCertLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadCertificate ( IntPtr pDataParams, byte bCertNo, ushort wWordPtr, ushort wWordCount,
            ref IntPtr ppCertificate, ref ushort pCertLen );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_Challenge ( IntPtr pDataParams, ushort wOption, byte bChallengeType, byte bKeyID_CertNum, byte[] pChallenge,
            ushort wChallengeLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadBuffer ( IntPtr pDataParams, byte bVerify, byte bKeyNo, byte bKeyVer, ref IntPtr ppResponse,
            ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedGetSystemInformation ( IntPtr pDataParams, byte bInfoParams, ref IntPtr ppSystemInfo,
            ref ushort pSystemInfoLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedGetMultipleBlockSecurityStatus ( IntPtr pDataParams, ushort wBlockNo, ushort wNoOfBlocks,
            IntPtr pStatus, ref ushort pStatusLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ExtendedFastReadMultipleBlocks ( IntPtr pDataParams, byte bOption, ushort wBlockNo, ushort wNumBlocks,
            IntPtr pData, ref ushort pDataLen );
        #endregion Optional Commands

        #region Custom Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_InventoryRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            byte bBlockNo, byte bNoOfBlocks, byte[] pUid, ref byte pUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_InventoryReadExtended ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            byte bExtendedOptions, byte[] pCID, byte bBlockNo, byte bNoOfBlocks, byte[] pCDIDOut, byte[] pUid, ref byte pUidLen, byte[] pData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastInventoryRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            byte bBlockNo, byte bNoOfBlocks, byte[] pUid, ref byte pUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_SetEAS ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ResetEAS ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockEAS ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_EASAlarm ( IntPtr pDataParams, byte bOption, byte[] pEasIdValue, byte bEasIdMaskLen, ref IntPtr ppEas,
            ref ushort pEasLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_PasswordProtectEAS ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_PasswordProtectAFI ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteEASID ( IntPtr pDataParams, byte bOption, byte[] pEasIdValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadEPC ( IntPtr pDataParams, ref IntPtr ppEpc, ref ushort pEpcLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetNXPSystemInformation ( IntPtr pDataParams, ref IntPtr ppSystemInfo, ref ushort pSystemInfoLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_InventoryPageRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            byte bPageNo, byte bNoOfPages, ref IntPtr ppUid, ref byte pUidLen, ref IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastInventoryPageRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask, byte bMaskBitLen,
            byte bPageNo, byte bNoOfPages, ref IntPtr ppUid, ref byte pUidLen, ref IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetRandomNumber ( IntPtr pDataParams, ref IntPtr ppRnd, ref ushort pRndLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_SetPassword ( IntPtr pDataParams, byte bOption, byte bPwdIdentifier, byte[] pXorPwd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WritePassword ( IntPtr pDataParams, byte bOption, byte bPwdIdentifier, byte[] pPwd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockPassword ( IntPtr pDataParams, byte bOption, byte bPwdIdentifier );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ProtectPage ( IntPtr pDataParams, byte bOption, byte bPPAdd_PageNo, byte bProtectionStatus );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockPageProtectionCondition ( IntPtr pDataParams, byte bOption, byte bPageNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetMultipleBlockProtectionStatus ( IntPtr pDataParams, byte bBlockNo, byte bNoOfBlocks,
            IntPtr pProtectionStates, ref ushort pNumReceivedStates );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_Destroy ( IntPtr pDataParams, byte bOption, byte[] pXorPwd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_EnablePrivacy ( IntPtr pDataParams, byte bOption, byte[] pXorPwd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_64BitPasswordProtection ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_StayQuietPersistent ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadSignature ( IntPtr pDataParams, ref IntPtr ppSign, ref ushort pSignLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadConfig ( IntPtr pDataParams, byte bOption, byte bBlockAddr, byte bNoOfBlocks,
            ref IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteConfig ( IntPtr pDataParams, byte bOption, byte bBlockAddr, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_PickRandomID ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadTT ( IntPtr pDataParams, byte bOption, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ParameterRequest ( IntPtr pDataParams, ref byte pBitRate, ref byte pTiming );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ParameterSelect ( IntPtr pDataParams, byte bBitRate, byte bTiming );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadSRAM ( IntPtr pDataParams, byte bOption, byte bBlockNo, byte bNumBlocks,
            IntPtr pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteSRAM ( IntPtr pDataParams, byte bOption, byte bBlockNo, byte bNumBlocks,
            byte[] pData, ushort wDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_I2CMRead ( IntPtr pDataParams, byte bI2CParam, ushort wDataLen, IntPtr pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_I2CMWrite ( IntPtr pDataParams, byte bI2CParam, byte[] pData, ushort wDataLen );

#if PACKAGE_INTERNAL
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_ReadCnt ( IntPtr pDataParams, byte bCounter, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_WriteCnt ( IntPtr pDataParams, byte bOption, byte bCounter, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_IncrCnt ( IntPtr pDataParams, byte bOption, byte bCounter, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_LockCertificate ( IntPtr pDataParams, byte bOption, byte bCertNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastReadCertificate ( IntPtr pDataParams, byte bCertNo, ushort wWordPtr, ushort wWordCount,
            ref IntPtr ppCertificate, ref ushort pCertLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastAuthenticateTAM ( IntPtr pDataParams, ushort wOption, byte bAuthType, byte bKeyID_CertNum,
            byte[] pChallenge, ushort wChallengeLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_FastReadBuffer ( IntPtr pDataParams, ref IntPtr ppResponse, ref ushort pRespLen );
#endif
        #endregion Custom Commands

        #region Utilities
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetSerialNo ( IntPtr pDataParams, ref IntPtr ppUid, ref ushort pUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_SetSerialNo ( IntPtr pDataParams, byte[] pUid, byte bUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_GetTagType ( IntPtr pDataParams, ref ushort pTagType );
        #endregion Utilities
        #endregion

        #region Wrapper Functions
        #region Mandatory Commands
        /// <summary>
        /// Performs a Inventory command. This command performs the ISO15693 Anti-Collision sequence and detects one
        /// ISO15693 complaint VICC.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bDsfid">1 byte Data Storage Format Identifier.</param>
        /// <param name="aUid">8 bytes of Unique identifier of the card.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Inventory ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, out byte bDsfid, byte[] aUid )
        {
            Status_t oStatus;
            bDsfid = 0x00;
            aUid = new byte[8];

            oStatus = phalICode_Inventory ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, ref bDsfid, aUid );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aUid = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs a StayQuiet command. When receiving the Stay quiet command, the VICC shall enter the quiet state and shall
        /// NOT send back a response. There is NO response to the Stay quiet command.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t StayQuiet ()
        {
            return phalICode_StayQuiet ( m_pDataParams );
        }
        #endregion Mandatory Commands

        #region Optional Commands
        /// <summary>
        /// Performs a Single block read command. When receiving the Read Single Block command, the VICC shall read the requested
        /// block and send back its value in the response.
        ///
        /// Flag can be set by using command <see cref="SetConfig"/>
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockNo">Block number from where the data to be read</param>
        /// <param name="aData">Information received from VICC
        ///                         - If <see cref="Option.OFF"/>, Block Security Status information is not available. Only block data is available.
        ///                           Format will be 4 byte data
        ///                         - If <see cref="Option.ON"/>, both Block Security Status information and Block Data is available.
        ///                           Format of the response will be 1 byte Status + 4 byte data.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadSingleBlock ( byte bOption, byte bBlockNo, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr ppDataLen = IntPtr.Zero;

            aData = null;

            Status_t oStatus = phalICode_ReadSingleBlock ( m_pDataParams, bOption, bBlockNo, ref ppDataLen, ref wDataLen );
            if ( ( ppDataLen != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppDataLen, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs a Single block write command. When receiving the Write single block command, the VICC shall write the requested block with the
        /// data contained in the request and report the success of the operation in the response.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bBlockNo">Block number to which the data should be written.</param>
        /// <param name="aData">Information to be written to the specified block number.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteSingleBlock ( byte bOption, byte bBlockNo, byte[] aData )
        {
            return phalICode_WriteSingleBlock ( m_pDataParams, bOption, bBlockNo, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Performs a Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested block.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bBlockNo">Block number which should be locked.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockBlock ( byte bOption, byte bBlockNo )
        {
            return phalICode_LockBlock ( m_pDataParams, bOption, bBlockNo );
        }

        /// <summary>
        /// Performs a Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s) and send
        /// back its value in the response.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockNo">Block number from where the data to be read.</param>
        /// <param name="bNumBlocks">Total number of block to read.</param>
        /// <param name="aData">Information received from VICC
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, Block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
        ///
        ///                         If <see cref="Option.ON"/>, both Block Security Status information and Block Data is available.
        ///                         Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
        ///                         Status N + 4 byte data N, Where 1, 2 ... N is the block number.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadMultipleBlocks ( byte bOption, byte bBlockNo, byte bNumBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr pData = IntPtr.Zero;

            pData = Marshal.AllocHGlobal ( bOption.Equals ( 1 ) ? ( ( bNumBlocks * 4 ) + bNumBlocks ) : ( bNumBlocks * 4 ) );
            Status_t oStatus = phalICode_ReadMultipleBlocks ( m_pDataParams, (byte) bOption, bBlockNo, bNumBlocks, pData, ref wDataLen );

            aData = null;
            if ( ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( pData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( pData );

            return oStatus;
        }

        /// <summary>
        /// Performs a Select command.
        /// When receiving the Select command:
        ///     If the UID is equal to its own UID, the VICC shall enter the selected state and shall send a response.
        ///     If it is different, the VICC shall return to the Ready state and shall not send a response.The Select command
        ///     shall always be executed in Addressed mode. (The Select_flag is set to 0. The Address_flag is set to 1.)
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Select ()
        {
            return phalICode_Select ( m_pDataParams );
        }

        /// <summary>
        /// Performs a ResetToReady command. When receiving a Reset to ready command, the VICC shall return to the Ready state.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ResetToReady ()
        {
            return phalICode_ResetToReady ( m_pDataParams );
        }

        /// <summary>
        /// Performs a WriteAFI command. When receiving the Write AFI request, the VICC shall write the AFI value into its memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bAfi">Value of Application Family Identifier.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteAFI ( byte bOption, byte bAfi )
        {
            return phalICode_WriteAFI ( m_pDataParams, bOption, bAfi );
        }

        /// <summary>
        /// Performs a LockAFI command. When receiving the Lock AFI request, the VICC shall lock the AFI value permanently into its memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockAFI ( byte bOption )
        {
            return phalICode_LockAFI ( m_pDataParams, bOption );
        }

        /// <summary>
        /// Performs WriteDSFID command. When receiving the Write DSFID request, the VICC shall write the DSFID value into its memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bDsfid">Value of DSFID (data storage format identifier).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteDSFID ( byte bOption, byte bDsfid )
        {
            return phalICode_WriteDSFID ( m_pDataParams, bOption, bDsfid );
        }

        /// <summary>
        /// Performs LockDSFID command. When receiving the Lock DSFID request, the VICC shall lock the DSFID value permanently into its memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockDSFID ( byte bOption )
        {
            return phalICode_LockDSFID ( m_pDataParams, bOption );
        }

        /// <summary>
        /// Performs GetSystemInformation command. This command allows for retrieving the system information value from the VICC.
        ///
        /// Flag can be set using <seealso cref="SetConfig"/> "SetConfig" utility interface.
        /// Response will contain the below information.
        ///     | **Bit** |   **Flag Name**  | **Value** |                                     **Description**                                     |
        ///     |:-------:|:-----------------|:---------:|:----------------------------------------------------------------------------------------|
        ///     |    b1   | DSFID            |     0     | DSFID is not supported.DSFID field is not present                                       |
        ///     |    ^    | ^                |     1     | DSFID is supported.DSFID field is present                                               |
        ///     |    b2   | AFI              |     0     | AFI is not supported. AFI field is not present                                          |
        ///     |    ^    | ^                |     1     | AFI is supported.AFI field is present                                                   |
        ///     |    b3   | VICC Memory Size |     0     | Information on VICC memory size is not supported. <br>Memory size field is not present. |
        ///     |    ^    | ^                |     1     | Information on VICC memory size is supported.Memory<br> size field is present.          |
        ///     |    b4   | IC Reference     |     0     | Information on IC reference is not supported. IC<br> reference field is not present.    |
        ///     |    ^    | ^                |     1     | Information on IC reference is supported.IC reference<br>field is present.              |
        ///     |    b5   | RFU              |     0     |                                                                                         |
        ///     |    b6   | RFU              |     0     |                                                                                         |
        ///     |    b7   | RFU              |     0     |                                                                                         |
        ///     |    b8   | RFU              |     0     |                                                                                         |
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aSystemInfo">The system information of the VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetSystemInformation ( out byte[] aSystemInfo )
        {
            ushort wSysInfoLen = 0;
            IntPtr ppSysInfo = IntPtr.Zero;

            aSystemInfo = null;
            Status_t oStatus = phalICode_GetSystemInformation ( m_pDataParams, ref ppSysInfo, ref wSysInfoLen );
            if ( ( ppSysInfo != IntPtr.Zero ) && ( wSysInfoLen > 0 ) )
            {
                aSystemInfo = new byte[wSysInfoLen];
                Marshal.Copy ( ppSysInfo, aSystemInfo, 0, wSysInfoLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs GetMultipleBlockSecurityStatus. When receiving the Get multiple block security status command, the VICC
        /// shall send back the block security status.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bBlockNo">Block number for which the status should be returned.</param>
        /// <param name="bNoOfBlocks">Number of blocks to be used for returning the status.</param>
        /// <param name="aStatus">The status of the block number mentioned in bBlockNo until bNumBlocks.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetMultipleBlockSecurityStatus ( byte bBlockNo, byte bNoOfBlocks, out byte[] aStatus )
        {
            ushort wStatusLen = 0;
            IntPtr pStatus = IntPtr.Zero;

            pStatus = Marshal.AllocHGlobal ( bNoOfBlocks );
            Status_t status = phalICode_GetMultipleBlockSecurityStatus ( m_pDataParams, bBlockNo, bNoOfBlocks, pStatus, ref wStatusLen );

            aStatus = null;
            if ( ( pStatus != IntPtr.Zero ) && ( wStatusLen > 0 ) )
            {
                aStatus = new byte[wStatusLen];
                Marshal.Copy ( pStatus, aStatus, 0, wStatusLen );
            }

            Marshal.FreeHGlobal ( pStatus );

            return status;
        }

        /// <summary>
        /// Performs a Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s) and
        /// send back its value in the response.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockNo">Block number from where the data to be read.</param>
        /// <param name="bNumBlocks">Total number of block to read.</param>
        /// <param name="aData">Information received from VICC.
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, Block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
        ///
        ///                         If <see cref="Option.ON"/>, both Block Security Status information and Block Data is available.
        ///                         Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
        ///                         Status N + 4 byte data N, Where 1, 2 ... N is the block number.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastReadMultipleBlocks ( byte bOption, byte bBlockNo, byte bNumBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr pData = IntPtr.Zero;

            pData = Marshal.AllocHGlobal ( bOption.Equals ( 1 ) ? ( ( bNumBlocks * 4 ) + bNumBlocks ) : ( bNumBlocks * 4 ) );
            Status_t oStatus = phalICode_FastReadMultipleBlocks ( m_pDataParams, (byte) bOption, bBlockNo, bNumBlocks, pData, ref wDataLen );

            aData = null;
            if ( ( pData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( pData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( pData );

            return oStatus;
        }

        /// <summary>
        /// Performs a Extended Single block read command. When receiving the Extended Read Single Block command, the VICC shall read the
        /// requested block and send back its value in the response. If a VICC supports Extended read single block command, it shall also support
        /// Read single block command for the first 256 blocks of memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="wBlockNo">Block number from where the data to be read</param>
        /// <param name="aData">Information received from VICC
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data.
        ///
        ///                         If <see cref="Option.ON"/>, both block Security Status information and Block Data is available.
        ///                         Format of the response will be 1 byte Status + 4 byte data.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedReadSingleBlock ( byte bOption, ushort wBlockNo, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr ppDataLen = IntPtr.Zero;

            aData = null;

            Status_t oStatus = phalICode_ExtendedReadSingleBlock ( m_pDataParams, bOption, wBlockNo, ref ppDataLen, ref wDataLen );
            if ( ( ppDataLen != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppDataLen, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs a Extended Single block write command. When receiving the Write single block command, the VICC shall write the requested block
        /// with the data contained in the request and report the success of the operation in the response. If a VICC supports Extended rite single
        /// block command, it shall also support Write single block command for the first 256 blocks of memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="wBlockNo">Block number to which the data should be written.</param>
        /// <param name="pData">Information to be written to the specified block number.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedWriteSingleBlock ( byte bOption, ushort wBlockNo, byte[] aData )
        {
            return phalICode_ExtendedWriteSingleBlock ( m_pDataParams, bOption, wBlockNo, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Performs a Extended Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested block.
        /// If a VICC supports Extended lock block command, it shall also support Lock block command for the first 256 blocks of memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="wBlockNo">Block number which should be locked.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedLockBlock ( byte bOption, ushort wBlockNo )
        {
            return phalICode_ExtendedLockBlock ( m_pDataParams, bOption, wBlockNo );
        }

        /// <summary>
        /// Performs a Extended Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
        /// and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple blocks
        /// command for the first 256 blocks of memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="wBlockNo">Block number from where the data to be read.</param>
        /// <param name="wNumBlocks">Total number of block to read.</param>
        /// <param name="aData">Information received from VICC.
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, Block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
        ///
        ///                         If <see cref="Option.ON"/>, both Block Security Status information and Block Data is available.
        ///                         Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
        ///                         Status N + 4 byte data N, Where 1, 2 ... N is the block number.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedReadMultipleBlocks ( byte bOption, ushort wBlockNo, ushort wNumBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr ppData = IntPtr.Zero;

            ppData = Marshal.AllocHGlobal ( bOption.Equals ( 1 ) ? ( ( wNumBlocks * 4 ) + wNumBlocks ) : ( wNumBlocks * 4 ) );
            Status_t oStatus = phalICode_ExtendedReadMultipleBlocks ( m_pDataParams, (byte) bOption, wBlockNo, wNumBlocks, ppData, ref wDataLen );

            aData = null;
            if ( ( ppData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( ppData );

            return oStatus;
        }

        /// <summary>
        /// Performs TAM authentication with the tag. this command supports TAM1 authentication as specified in
        /// ISO 29167 specification.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="wOption">One of the below mentioned options.
        ///                         - Option flag
        ///                             - <see cref="Option.OFF"/>
        ///                             - <see cref="Option.ON"/>
        ///
        ///                         - Diversification option
        ///                             - <see cref="Diversification.OFF"/>
        ///                             - <see cref="Diversification.ON"/>
        ///
        ///                         - Challenge usage
        ///                             - <see cref="GenerateRnd.INTERNAL"/>
        ///                             - <see cref="GenerateRnd.EXTERNAL"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store or SAM hardware KeyStore.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="bKeyID">Block number of the AES key or Certificate number available in the tag.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key. The diversification input is available in SAM mode only.</param>
        /// <param name="aChallenge">Challenge information to be exchanged with tag. Will be one of the following.
        ///                             - If wOption has <see cref="GenerateRnd.EXTERNAL"/>
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM1"/>: 80 bit challenge (IChallenge_TAM1)
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM2"/>: 80 bit challenge (IChallenge_TAM2)
        ///                             - If \b wOption has <see cref="GenerateRnd.INTERNAL"/>, then NULL should be provided.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for difference in Random Challenge.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If wOption parameter is invalid.
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128.
        ///         For Invalid response format.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t AuthenticateTAM ( ushort wOption, byte bKeyNo, byte bKeyVer, byte bKeyID, byte[] aDivInput, byte[] aChallenge )
        {
            ushort wRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus =  phalICode_AuthenticateTAM ( m_pDataParams, wOption, ( byte ) AuthType_TAM.TAM1, bKeyNo, bKeyVer, bKeyID,
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aChallenge,
                ( ushort ) ( ( aDivInput == null ) ? 0 : aChallenge.Length ), null, 0,
                ref ppResponse, ref wRespLen );

            return oStatus;
        }

#if PACKAGE_INTERNAL
        /// <summary>
        /// Performs TAM authentication with the tag. Support various authentication schemes of TAM authentication.
        /// Refer description of \b bAuthType parameter for more information.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="wOption">One of the below mentioned options.
        ///                         - Option flag
        ///                             - <see cref="Option.OFF"/>
        ///                             - <see cref="Option.ON"/>
        ///
        ///                         - Diversification option
        ///                             - <see cref="Diversification.OFF"/>
        ///                             - <see cref="Diversification.ON"/>
        ///
        ///                         - Challenge usage
        ///                             - <see cref="GenerateRnd.INTERNAL"/>
        ///                             - <see cref="GenerateRnd.EXTERNAL"/>
        /// </param>
        /// <param name="bAuthType">Type of Authenticate. One of the below mentioned values,
        ///                             - <see cref="AuthType_TAM.TAM1"/>
        ///                             - <see cref="AuthType_TAM.TAM1_2"/>
        ///                             - <see cref="AuthType_TAM.TAM1_3"/>
        ///                             - <see cref="AuthType_TAM.TAM2"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store or SAM hardware KeyStore.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="bKeyID_CertNum">KeyID or Certificate number available in the tag.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key. The diversification input is available in SAM mode only.</param>
        /// <param name="aChallenge">Challenge information to be exchanged with tag. Will be one of the following.
        ///                             - If wOption has <see cref="GenerateRnd.EXTERNAL"/>
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM1"/>: 80 bit challenge (IChallenge_TAM1)
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM2"/>: 80 bit challenge (IChallenge_TAM2)
        ///
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM1_2"/>: 62 bit challenge (IChallenge_TAM1.2)
        ///                                 - bAuthType = <see cref="AuthType_TAM.TAM1_3"/>: 62 bit challenge (IChallenge_TAM1.3)
        ///                             - If \b wOption has <see cref="GenerateRnd.INTERNAL"/>, then NULL should be provided.
        /// </param>
        /// <param name="aAddInfo">The information to be exchanged based on \b bAuthType parameter data.
        ///                         - bAuthType = <see cref="AuthType_TAM.TAM1"/>: Will be NULL as there is no additional information to be exchanged.
        ///                         - bAuthType = <see cref="AuthType_TAM.TAM1_2"/>: Will be NULL as there is no additional information to be exchanged.
        ///                         - bAuthType = <see cref="AuthType_TAM.TAM1_3"/>: Will be NULL as there is no additional information to be exchanged.
        ///                         - bAuthType = <see cref="AuthType_TAM.TAM2"/>: Will be BlockSize, Profile, Offset, BlockCount and ProtMode
        ///                             - BlockSize: To select the size of custom data block to be used.The value should either be
        ///                               0x00 for 16 bit block size or 0x01 for 64 bit block size, as per ISO 29167
        ///                             - Profile: To select one of the memory profiles supported by the tag.
        ///                               The Profile range is from 0 - 15, as per ISO 29167
        ///
        ///                             - Offset: To set the offset for the specified profile.The Offset ranges form 0 - 4095, as per ISO 29167
        ///
        ///                             - BlockCount: To select the custom data block to be used from the offset specified.
        ///                               The BlockCount range is from 1 - 16.
        ///
        ///                             - ProtMode: To specify the mode of operation to be used for encryption/decryption.
        ///                               The ProtMode ranges form 0 - 3, as per ISO 29167
        /// </param>
        /// <param name="aResponse">Response received from the Tag. Will be based on \b bAuthType parameter information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for difference in Random Challenge.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If wOption parameter is invalid.
        ///         If bAuthType parameter is invalid.
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128.
        ///         For Invalid response format.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t AuthenticateTAM ( ushort wOption, byte bAuthType, byte bKeyNo, byte bKeyVer, byte bKeyID_CertNum, byte[] aDivInput,
            byte[] aChallenge, byte[] aAddInfo, out byte[] aResponse )
        {
            ushort wRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus =  phalICode_AuthenticateTAM ( m_pDataParams, wOption, bAuthType, bKeyNo, bKeyVer, bKeyID_CertNum, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aChallenge, ( ushort ) ( ( aChallenge == null ) ? 0 : aChallenge.Length ),
                aAddInfo, ( ushort ) ( ( aAddInfo == null ) ? 0 : aAddInfo.Length ), ref ppResponse, ref wRespLen );

            aResponse = null;
            if ( ( ppResponse != IntPtr.Zero ) && ( wRespLen > 0 ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, wRespLen );
            }

            return oStatus;
        }
#endif

        /// <summary>
        /// Performs TAM1 authentication with the card.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store or SAM hardware KeyStore.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="bKeyNoCard">Block number of the AES key available in the card.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key. The diversification input is available in SAM mode only</param>
        /// <param name="bDivLen">Length of diversification input used to diversify the key. If 0, no diversification is performed.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for difference in Random Challenge.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128.
        ///         For Invalid response format.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [Obsolete ( "AuthenticateTAM1 is obsolete and will be removed. Use AuthenticateTAM method as an alternative." )]
        public Status_t AuthenticateTAM1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard, byte[] aDivInput, byte bDivLen )
        {
            return phalICode_AuthenticateTAM1 ( m_pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, aDivInput, bDivLen );
        }

        /// <summary>
        /// Performs TAM-2 authentication with the card.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store or SAM hardware KeyStore.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="bKeyNoCard">Block number of the AES key available in the card</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key. The diversification input is available in SAM mode only</param>
        /// <param name="bDivLen">Length of diversification input used to diversify the key. If 0, no diversification is performed.</param>
        /// <param name="bBlockSize">To select the size of custom data block to be used.
        ///                          The value should either be 0x00 for 16 bit block size or 0x01 for 64 bit block size. As per ISO 29167</param>
        /// <param name="bBlockCount">To select the custom data block to be used from the offset specified. The BlockCount range is from 1 - 16.</param>
        /// <param name="bProfile">To select one of the memory profiles supported by the tag. The Profile range is from 0 - 15. As per ISO 29167.</param>
        /// <param name="bProtMode">To specify the mode of operation to be used for encryption/decryption. The ProtMode ranges form 0 - 3. As per ISO 29167.</param>
        /// <param name="wOffset">To set the offset for the specified profile. The Offset ranges form 0 - 4095. As per ISO 29167</param>
        /// <param name="aCustomData">The custom data returned by the card.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [Obsolete ( "AuthenticateTAM2 is obsolete and will be removed. Use AuthenticateTAM method as an alternative." )]
        public Status_t AuthenticateTAM2 ( byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard, byte[] aDivInput, byte bDivLen,
            byte bBlockSize, byte bBlockCount, byte bProfile, byte bProtMode, ushort wOffset, out byte[] aCustomData )
        {
            Status_t wStatus;
            ushort wCustomDataLen = 0;
            byte[] aCustData = new byte[256];

            aCustomData = null;

            wStatus = phalICode_AuthenticateTAM2 ( m_pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, aDivInput,
                bDivLen, bBlockSize, bBlockCount, bProfile, bProtMode, wOffset, aCustData,
                ref wCustomDataLen );

            if ( wStatus.Equals ( new Status_t () ) )
            {
                aCustomData = new byte[wCustomDataLen];
                Array.Copy ( aCustData, 0, aCustomData, 0, wCustomDataLen );
            }

            return wStatus;
        }

        /// <summary>
        /// Performs MAM authentication with the card. Both the MAM part 1 and MAM part 2 authentication are
        /// carried out internally by this interface.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store or SAM hardware KeyStore.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="bKeyNoCard">Block number of the AES key available in the card</param>
        /// <param name="bPurposeMAM2">The purpose MAM2 data to be used.
        ///                             <see cref="PurposeMAM2.DISABLE_PRIVACY_HF_RESET"/>
        ///                             <see cref="PurposeMAM2.ENABLE_PRIVACY"/>
        ///                             <see cref="PurposeMAM2.DISABLE_PRIVACY"/>
        ///                             <see cref="PurposeMAM2.DESTROY"/>
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key. The diversification input is available in SAM mode only</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key. If 0, no diversification is performed.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for difference in Random Challenge.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128
        ///         For invalid response format.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t AuthenticateMAM ( byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyNoCard, byte bPurposeMAM2, byte[] aDivInput, byte bDivInputLen )
        {
            return phalICode_AuthenticateMAM ( m_pDataParams, bOption, bKeyNo, bKeyVer, bKeyNoCard, bPurposeMAM2, aDivInput, bDivInputLen );
        }

#if PACKAGE_INTERNAL
        /// <summary>
        /// Performs Write Certificate with the card. The purpose of WRITE_CERT command is to write one or more
        /// block of selected ECC certificate
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method.
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bCertNo">Certificate record number to use for writing the certificate.</param>
        /// <param name="wWordPtr">Starting Block number to which the data should be written.</param>
        /// <param name="aCertificate">Certificate information to written to the specified <b>wWordPtr</b>.
        ///                            Supports the following.
        ///                                 - Can be maximum of 4 bytes of certificate data.In this case user must call this interface multiple
        ///                                   times with update of <b>wWordPtr</b> and data.
        ///                                 - Can be complete certificate data.In this case, library will iterate the certificate information
        ///                                   in chunks where each chunk is of 4 bytes each and <b>wWordPtr</b> will be incremented internally.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/> If <b>bCertLen</b> is not multiple of Block Size (4 byte(s)).
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteCertificate ( byte bOption, byte bCertNo, ushort wWordPtr, byte[] aCertificate )
        {
            return phalICode_WriteCertificate ( m_pDataParams, bOption, bCertNo, wWordPtr, aCertificate,
                ( byte ) ( ( aCertificate == null ) ? 0 : aCertificate.Length ) );
        }

        /// <summary>
        /// Performs Read Certificate with the card. The purpose of READ_CERT command is to read specified ECC certificate.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method.
        /// </summary>
        ///
        /// <param name="bCertNo">Certificate record number to use for writing the certificate.</param>
        /// <param name="wWordPtr">Starting Block number from which the data should be read. Currently 0000h should be used.</param>
        /// <param name="wWordCount">A 16-bit value indicating the number of double words to be read from the certificate record starting
        ///                          from the position indicated by WordPtr.The value of 0000h shall be used to indicate the full contents of
        ///                          the certificate record starting from the position indicated by WordPtr; i.e. memory shall be read from
        ///                          WordPtr until the end of the current certificate record.Currently 0000h should be used.
        /// </param>
        /// <param name="aCertificate">Certificate information read from the tag based on <b>wWordPtr</b> and <b>wWordCount</b> information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadCertificate ( byte bCertNo, ushort wWordPtr, ushort wWordCount, out byte[] aCertificate )
        {
            ushort wCertLen = 0;
            IntPtr ppCertificate = IntPtr.Zero;

            Status_t oStatus = phalICode_ReadCertificate ( m_pDataParams, bCertNo, wWordPtr, wWordCount, ref ppCertificate,
                ref wCertLen );

            aCertificate = null;
            if ( ( ppCertificate != null ) && ( wCertLen > 0 ) )
            {
                aCertificate = new byte[wCertLen];
                Marshal.Copy ( ppCertificate, aCertificate, 0, wCertLen );
            }

            return oStatus;
        }
#endif

        /// <summary>
        /// Performs tag authentication with the card. This is another method of authenticating with the card.
        /// Here the TAM1 challenge message is sent to the card. The card does not respond for this command. To verify
        /// if this command was success the command <seealso cref="ReadBuffer"/> should be called.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// - Timeout can be set using \ref phalICode_SetConfig "SetConfig" utility interface with below mentioned
        ///   configuration identifiers.
        ///     - <see cref="Config.CHALLENGE_TIMEOUT_US">Timeout in Microsecond</see>
        ///     - <see cref="Config.CHALLENGE_TIMEOUT_MS">Timeout in Millisecond</see>
        /// - By Default Timeout Long Timeout (20000us) is configured.
        /// </summary>
        ///
        /// <param name="bKeyID">Block number of the AES key available in the card.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Challenge ( byte bKeyID )
        {
            return phalICode_Challenge ( m_pDataParams, ( ushort ) GenerateRnd.INTERNAL, ( byte ) ChallengeType.TAM1, bKeyID, null,
                0 );
        }

#if PACKAGE_INTERNAL
        /// <summary>
        /// Performs tag authentication with the card. This is another method of authenticating with the card.
        /// Here the challenge message is sent to the card. The card does not respond for this command. To verify
        /// if this command was success the command <seealso cref="ReadBuffer"/> should be called.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        ///
        /// <param name="wOption">One of the below mentioned options.
        ///                             - <see cref="GenerateRnd.INTERNAL"/>
        ///                             - <see cref="GenerateRnd.EXTERNAL"/>
        /// </param>
        /// <param name="bChallengeType">Type of Challenge command to exchange. One of the below mentioned values,
        ///                             - <see cref="ChallengeType.TAM1"/>
        ///                             - <see cref="ChallengeType.TAM1_2"/>
        ///                             - <see cref="ChallengeType.TAM1_3"/>
        /// </param>
        /// <param name="bKeyID_CertNum">KeyID or Certificate number available in the tag.</param>
        /// <param name="aChallenge">Challenge information to be exchanged with tag. Will be one of the following.
        ///                             - If wOption has <see cref="GenerateRnd.EXTERNAL"/>
        ///                                 - bAuthType = <see cref="ChallengeType.TAM1"/>: 80 bit challenge (IChallenge_TAM1)
        ///
        ///                                 - bAuthType = <see cref="ChallengeType.TAM1_2"/>: 62 bit challenge (IChallenge_TAM1.2)
        ///                                 - bAuthType = <see cref="ChallengeType.TAM1_3"/>: 62 bit challenge (IChallenge_TAM1.3)
        ///                             - If \b wOption has <see cref="GenerateRnd.INTERNAL"/>, then NULL should be provided.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Challenge ( ushort wOption, byte bChallengeType, byte bKeyID_CertNum, byte[] aChallenge )
        {
            return phalICode_Challenge ( m_pDataParams, wOption, bChallengeType, bKeyID_CertNum, aChallenge,
                ( ushort ) ( ( aChallenge == null ) ? 0 : aChallenge.Length ) );
        }
#endif

        /// <summary>
        /// Reads the crypto calculation result of previous Challenge command. If the Challenge Command was success,
        /// Then the encrypted response will be returned. The response will be same as TAM1 response format. If verification
        /// is enabled (i.e. bVerify = <see cref="Verify.ON"/>), The encrypted response will be decrypted and the random number
        /// generated by the Challenge command will be compared against the received one. If fails AUTH_ERROR will be returned.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bVerify">To verify the received data with the random number generated by Challenge command.
        ///                         <see cref="Verify.OFF"/>
        ///                         <see cref="Verify.ON"/>
        /// </param>
        /// <param name="bKeyNo">AES key address in software key store.</param>
        /// <param name="bKeyVer">AES key version to be used.</param>
        /// <param name="aResponse">If verification is enabled the decrypted response data will be available. Also the response
        ///                         will be verified with the random number generated by <seealso cref="Generic.Challenge"/>
        ///                         command. If verification is disabled the encrypted response data will be available.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for difference in Random Challenge.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128
        ///         For invalid response format.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the verify values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadBuffer ( byte bVerify, byte bKeyNo, byte bKeyVer, out byte[] aResponse )
        {
            ushort wRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus = phalICode_ReadBuffer ( m_pDataParams, bVerify, bKeyNo, bKeyVer, ref ppResponse, ref wRespLen );

            aResponse = null;
            if ( ( ppResponse != IntPtr.Zero ) && ( wRespLen > 0 ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, wRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs ExtendedGetSystemInformation command. This command allows for retrieving the system information value
        /// from the VICC and shall be supported by the VICC if extended memory or security functionalities are supported by the VICC.
        ///
        /// Flag can be set using <seealso cref="SetConfig"/> "SetConfig" utility interface.
        /// Response will contain the below information.
        ///     | **Bit** |   **Flag Name**   | **Value** |                                     **Description**                                     |
        ///     |:-------:|:------------------|:---------:|:----------------------------------------------------------------------------------------|
        ///     |    b1   | DSFID             |     0     | DSFID is not supported.DSFID field is not present                                       |
        ///     |    ^    | ^                 |     1     | DSFID is supported.DSFID field is present                                               |
        ///     |    b2   | AFI               |     0     | AFI is not supported. AFI field is not present                                          |
        ///     |    ^    | ^                 |     1     | AFI is supported.AFI field is present                                                   |
        ///     |    b3   | VICC Memory Size  |     0     | Information on VICC memory size is not supported. Memory size field is not present.     |
        ///     |    ^    | ^                 |     1     | Information on VICC memory size is supported.Memory size field is present.              |
        ///     |    b4   | IC Reference      |     0     | Information on IC reference is not supported. IC reference field is not present.        |
        ///     |    ^    | ^                 |     1     | Information on IC reference is supported.IC reference field is present.                 |
        ///     |    b5   | MOI               |     0     | 1 byte memory addressing                                                                |
        ///     |    ^    | ^                 |     1     | 2 bytes memory addressing                                                               |
        ///     |    b6   | VICC Command List |     0     | Data field of all supported commands is not present                                     |
        ///     |    ^    | ^                 |     1     | Data field of all supported commands is present                                         |
        ///     |    b7   | CSI Information   |     0     | CSI list is not present                                                                 |
        ///     |    ^    | ^                 |     1     | CSI list is present                                                                     |
        ///     |    b8   | Info Flag Field   |     0     | One byte length of Info flag field                                                      |
        ///     |    ^    | ^                 |     1     | Two bytes length of Info flag field                                                     |
        /// </summary>
        ///
        /// <param name="bInfoParams">Extend Get System Information parameter request fields.
        ///                             <see cref="InfoParams.DEFAULT"/>
        ///                             <see cref="InfoParams.REQUEST_DSFID"/>
        ///                             <see cref="InfoParams.REQUEST_AFI"/>
        ///                             <see cref="InfoParams.REQUEST_VICC_MEM_SIZE"/>
        ///                             <see cref="InfoParams.REQUEST_IC_REFERENCE"/>
        ///                             <see cref="InfoParams.REQUEST_MOI"/>
        ///                             <see cref="InfoParams.REQUEST_COMMAND_LIST"/>
        ///                             <see cref="InfoParams.REQUEST_CSI_INFORMATION"/>
        ///                             <see cref="InfoParams.REQUEST_EXT_GET_SYS_INFO"/>
        /// </param>
        /// <param name="aSystemInfo">The system information of the VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedGetSystemInformation ( byte bInfoParams, out byte[] aSystemInfo )
        {
            ushort wSysInfoLen = 0;
            IntPtr ppSysInfo = IntPtr.Zero;

            aSystemInfo = null;
            Status_t oStatus = phalICode_ExtendedGetSystemInformation ( m_pDataParams, bInfoParams, ref ppSysInfo, ref wSysInfoLen );
            if ( ( ppSysInfo != IntPtr.Zero ) && ( wSysInfoLen > 0 ) )
            {
                aSystemInfo = new byte[wSysInfoLen];
                Marshal.Copy ( ppSysInfo, aSystemInfo, 0, wSysInfoLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs ExtendedGetMultipleBlockSecurityStatus. When receiving the Extended Get multiple block security status
        /// command, the VICC shall send back the block security status. The blocks are numbered from 0000 to FFFF (0 - 65535).
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="wBlockNo">Block number for which the status should be returned.</param>
        /// <param name="wNoOfBlocks">Number of blocks to be used for returning the status.</param>
        /// <param name="aStatus">The status of the block number mentioned in wBlockNo until wNumBlocks.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If wNoOfBlocks is zero.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedGetMultipleBlockSecurityStatus ( ushort wBlockNo, ushort wNoOfBlocks, out byte[] aStatus )
        {
            ushort wStatusLen = 0;
            IntPtr pStatus = IntPtr.Zero;

            pStatus = Marshal.AllocHGlobal ( wNoOfBlocks );
            Status_t status = phalICode_ExtendedGetMultipleBlockSecurityStatus ( m_pDataParams, wBlockNo, wNoOfBlocks, pStatus, ref wStatusLen );

            aStatus = null;
            if ( ( pStatus != IntPtr.Zero ) && ( wStatusLen > 0 ) )
            {
                aStatus = new byte[wStatusLen];
                Marshal.Copy ( pStatus, aStatus, 0, wStatusLen );
            }

            Marshal.FreeHGlobal ( pStatus );

            return status;
        }

        /// <summary>
        /// Performs a Extended Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
        /// and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple blocks
        /// command for the first 256 blocks of memory.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="wBlockNo">Block number from where the data to be read.</param>
        /// <param name="wNumBlocks">Total number of block to read.</param>
        /// <param name="aData">Information received from VICC
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, Block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If wNoOfBlocks is zero.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExtendedFastReadMultipleBlocks ( byte bOption, ushort wBlockNo, ushort wNumBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr pData = IntPtr.Zero;

            pData = Marshal.AllocHGlobal ( bOption.Equals ( 1 ) ? ( ( wNumBlocks * 4 ) + wNumBlocks ) : ( wNumBlocks * 4 ) );
            Status_t oStatus = phalICode_ExtendedFastReadMultipleBlocks ( m_pDataParams, (byte) bOption, wBlockNo, wNumBlocks, pData, ref wDataLen );

            aData = null;
            if ( ( pData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( pData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( pData );

            return oStatus;
        }
        #endregion Optional Commands

        #region Custom Commands
        /// <summary>
        /// Perform ISO15693 InventoryRead command. When receiving the INVENTORY READ request, the ICODE IC performs the same as
        /// the Anti-Collision sequence, with the difference that instead of the UID and the DSFID, the requested response is defined by
        /// additional options.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bBlockNo">Block number of first block to read.</param>
        /// <param name="bNoOfBlocks">Number of blocks to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t InventoryRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bBlockNo,
            byte bNoOfBlocks, byte[] aUid, out byte[] aData )
        {
            Status_t oStatus;
            byte bUidLen = 0;
            ushort wDataLen = 0;

            aUid = new byte[8];
            aData = new byte[512];

            oStatus = phalICode_InventoryRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bBlockNo, bNoOfBlocks, aUid, ref bUidLen, aData, ref wDataLen );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aUid = null;
                aData = null;
            }
            else
                Array.Resize ( ref aData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 InventoryReadExtended command.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bExtendedOptions">Supported Extended options are
        ///                                 <see cref="InventoryReadExt.DEFAULT"/>
        ///                                 <see cref="InventoryReadExt.EAS_MODE"/>
        ///                                 <see cref="InventoryReadExt.UID_MODE"/>
        ///                                 <see cref="InventoryReadExt.CID_COMPARE"/>
        ///                                 <see cref="InventoryReadExt.CID_RESPONSE"/>
        ///                                 <see cref="InventoryReadExt.SKIP_DATA"/>
        ///                                 <see cref="InventoryReadExt.QUIET"/>
        ///                                 <see cref="InventoryReadExt.PERSIST_QUIET"/>
        ///                                 <see cref="InventoryReadExt.PERSIST_QUIET_RESPONSE"/>
        /// </param>
        /// <param name="aCID">Two byte CID -> if marked in extended options.</param>
        /// <param name="bBlockNo">Block Number from where start reading.</param>
        /// <param name="bNoOfBlocks">Number of blocks to read.</param>
        /// <param name="aCDIDOut">Received CID.</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received Data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t InventoryReadExtended ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bExtendedOptions,
            byte[] aCID, byte bBlockNo, byte bNoOfBlocks, byte[] aCDIDOut, byte[] aUid, out byte[] aData )
        {
            Status_t oStatus;
            byte bUidLen = 0;
            ushort wDataLen = 0;

            aCDIDOut = new byte[2];
            aUid = new byte[8];
            aData = new byte[512];

            oStatus = phalICode_InventoryReadExtended ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bExtendedOptions, aCID, bBlockNo, bNoOfBlocks,
                aCDIDOut, aUid, ref bUidLen, aData, ref wDataLen );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aCDIDOut = null;
                aUid = null;
                aData = null;
            }
            else
                Array.Resize ( ref aData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 FastInventoryRead command. When receiving the FAST INVENTORY READ command the ICODE IC behaves the
        /// same as the INVENTORY READ command with the following exceptions:
        ///
        /// The data rate in the direction ICODE DNA IC to the interrogator is twice that defined in ISO/IEC 15693-3 depending on
        /// the Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
        ///
        /// The data rate from the interrogator to the ICODE DNA IC and the time between the rising edge of the EOF from the
        /// interrogator to the ICODE DNA IC remain unchanged (stay the same as defined in ISO/IEC 15693-3).
        ///
        /// In the ICODE DNA IC to the interrogator direction, only the single sub-carrier mode is supported.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bBlockNo">Block number of first block to read.</param>
        /// <param name="bNoOfBlocks">Number of blocks to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastInventoryRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bBlockNo, byte bNoOfBlocks,
            byte[] aUid, out byte[] aData )
        {
            Status_t oStatus;
            byte bUidLen = 0;
            ushort wDataLen = 0;

            aUid = new byte[8];
            aData = new byte[512];

            oStatus = phalICode_FastInventoryRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bBlockNo, bNoOfBlocks, aUid, ref bUidLen, aData, ref wDataLen );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aUid = null;
                aData = null;
            }
            else
                Array.Resize ( ref aData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// This command enables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
        /// the EAS password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetEAS ( byte bOption )
        {
            return phalICode_SetEAS ( m_pDataParams, bOption );
        }

        /// <summary>
        /// This command disables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
        /// the EAS password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ResetEAS ( byte bOption )
        {
            return phalICode_ResetEAS ( m_pDataParams, bOption );
        }

        /// <summary>
        /// This command locks the current state of the EAS mode and the EAS ID. If the EAS mode is password protected
        /// the EAS password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockEAS ( byte bOption )
        {
            return phalICode_LockEAS ( m_pDataParams, bOption );
        }

        /// <summary>
        /// This command returns the EAS sequence if the EAS mode is enabled.
        ///
        /// bOption disabled: bEasIdMaskLength and aEasIdValue are not transmitted, EAS Sequence is returned;
        /// bOption enabled and bEasIdMaskLength = 0: EAS ID is returned;
        /// bOption enabled and bEasIdMaskLength > 0: EAS Sequence is returned by ICs with matching pEasIdValue;
        ///
        /// If the EAS mode is disabled, the label remains silent.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>
        ///                             EAS ID mask length and EAS ID value shall not be transmitted.
        ///                             If the EAS mode is enabled, the EAS response is returned from the ICODE IC.
        ///                             This configuration is compliant with the EAS command of the ICODE IC
        ///
        ///                         <see cref="Option.ON"/>
        ///                             Within the command the EAS ID mask length has to be transmitted to identify how
        ///                             many bits of the following EAS ID value are valid (multiple of 8-bits). Only those
        ///                             ICODE ICs will respond with the EAS sequence which have stored the corresponding
        ///                             data in the EAS ID configuration (selective EAS) and if the EAS Mode is set.
        ///                             If the EAS ID mask length is set to 0, the ICODE IC will answer with its EAS ID
        /// </param>
        /// <param name="aEasIdValue">EAS ID; 0, 8 or 16 bits; optional.</param>
        /// <param name="bEasIdMaskLen">8 bits; optional.</param>
        /// <param name="aEas">EAS ID (16 bits) or EAS Sequence (256 bits).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t EASAlarm ( byte bOption, byte[] aEasIdValue, byte bEasIdMaskLen, out byte[] aEas )
        {
            ushort wEasLen = 0;
            IntPtr ppEas = IntPtr.Zero;

            aEas = null;

            Status_t oStatus = phalICode_EASAlarm ( m_pDataParams, bOption, aEasIdValue, bEasIdMaskLen, ref ppEas, ref wEasLen );
            if ( ( ppEas != IntPtr.Zero ) && ( wEasLen > 0 ) )
            {
                aEas = new byte[wEasLen];
                Marshal.Copy ( ppEas, aEas, 0, wEasLen );
            }

            return oStatus;
        }

        /// <summary>
        /// This command enables the password protection for EAS. The EAS password has to be transmitted before with
        /// <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t PasswordProtectEAS ()
        {
            return phalICode_PasswordProtectEAS ( m_pDataParams );
        }

        /// <summary>
        /// This command enables the password protection for AFI. The AFI password has to be transmitted before with
        /// <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t PasswordProtectAFI ()
        {
            return phalICode_PasswordProtectAFI ( m_pDataParams );
        }

        /// <summary>
        /// With this command, a new EAS identifier is stored in the corresponding configuration memory. If the EAS mode
        /// is password protected the EAS password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aEasIdValue">EAS ID; 16 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteEASID ( byte[] aEasIdValue )
        {
            return phalICode_WriteEASID ( m_pDataParams, ( byte ) Option.OFF, aEasIdValue );
        }

        /// <summary>
        /// With this command, a new EAS identifier is stored in the corresponding configuration memory. If the EAS mode
        /// is password protected the EAS password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="aEasIdValue">EAS ID; 16 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteEASID ( byte bOption, byte[] aEasIdValue )
        {
            return phalICode_WriteEASID ( m_pDataParams, bOption, aEasIdValue );
        }

        /// <summary>
        /// Performs ReadEPC command. On this command, the label will respond with it's EPC data.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aEpc">EPC data; 96 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadEPC ( out byte[] aEpc )
        {
            ushort wEpcLen = 0;
            IntPtr ppEpc = IntPtr.Zero;

            aEpc = null;

            Status_t oStatus = phalICode_ReadEPC ( m_pDataParams, ref ppEpc, ref wEpcLen );
            if ( ( ppEpc != IntPtr.Zero ) && ( wEpcLen > 0 ) )
            {
                aEpc = new byte[wEpcLen];
                Marshal.Copy ( ppEpc, aEpc, 0, wEpcLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Perform GetNXPSystemInformation command. This command allows for retrieving the NXP system information value from the VICC.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aSystemInfo">The NXP system information of the VICC. Refer specific product data-sheet for more information.
        ///                           Response will contain the below informations.
        ///                             Byte 1: 1 byte of Protection Pointer address
        ///                             Byte 2: 1 byte of Protection Pointer Conditions
        ///                             Byte 3: 1 byte of Lock Bit settings
        ///                             Byte 4: 4 bytes of Supported commands and Features
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetNXPSystemInformation ( out byte[] aSystemInfo )
        {
            ushort wSysInfoLen = 0;
            IntPtr ppSysInfo = IntPtr.Zero;

            aSystemInfo = null;
            Status_t oStatus = phalICode_GetNXPSystemInformation ( m_pDataParams, ref ppSysInfo, ref wSysInfoLen );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                aSystemInfo = new byte[wSysInfoLen];
                Marshal.Copy ( ppSysInfo, aSystemInfo, 0, wSysInfoLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Perform InventoryPageRead command. When receiving the Inventory Page Read request, the ICODE IC performs the same
        /// as in the anti-collision sequence, with the difference that instead of the UID and the DSFID the requested memory content
        /// is re-transmitted from the ICODE IC.
        ///
        /// If the Option flag is set to 0 N pages of data including page protection status (password protection condition) are
        /// re-transmitted. If the option flag is set to 1 N pages (4 blocks = 16 byte) of data including page protection status
        /// (password protection condition) and the part of the UID which is not part of the mask are re-transmitted.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within aMask.</param>
        /// <param name="bPageNo">Block number of first page to read.</param>
        /// <param name="bNoOfPages">Number of pages to read.</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t InventoryPageRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bPageNo, byte bNoOfPages,
            out byte[] aUid, out byte[] aData )
        {
            byte bUidLen = 0;
            ushort wDataLen = 0;
            IntPtr ppUid = IntPtr.Zero;
            IntPtr ppData = IntPtr.Zero;

            aUid = null;
            aData = null;

            Status_t oStatus = phalICode_InventoryPageRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bPageNo, bNoOfPages, ref ppUid, ref bUidLen, ref ppData, ref wDataLen );
            if ( ( ppUid != IntPtr.Zero ) && ( bUidLen > 0 ) )
            {
                aUid = new byte[bUidLen];
                Marshal.Copy ( ppUid, aUid, 0, bUidLen );
            }

            if ( ( ppData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppData, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Perform FastInventoryPageRead command. When receiving the FAST INVENTORY PAGE READ command the ICODE IC behaves the
        /// same as the INVENTORY PAGE READ command with the following exceptions:
        ///
        /// - The data rate in the direction ICODE Tag to the interrogator is twice that defined in ISO/IEC 15693-3 depending on
        ///   the Datarate_flag 53 kbit ( high data rate) or 13 kbit ( low data rate).
        ///
        /// - The data rate from the interrogator to the ICODE Tag and the time between the rising edge of the EOF from the
        ///   interrogator to the ICODE Tag remain unchanged ( stay the same as defined in ISO/IEC 15693-3).
        ///
        /// - In the ICODE Tag to the interrogator direction, only the single sub-carrier mode is supported.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte. Refer below set of flags
        ///                         Common Flags
        ///                             <see cref="Flags.TWO_SUB_CARRIERS"/>
        ///                             <see cref="Flags.DATA_RATE"/>
        ///                             <see cref="Flags.INVENTORY"/>
        ///                             <see cref="Flags.PROTOCOL_EXTENSION"/>
        ///                             <see cref="Flags.OPTION"/>
        ///
        ///                         Flags when Inventory is not set.
        ///                             <see cref="Flags.SELECTED"/>
        ///                             <see cref="Flags.ADDRESSED"/>
        ///
        ///                         Flags when Inventory is set.
        ///                             <see cref="Flags.AFI"/>
        ///                             <see cref="Flags.NBSLOTS"/>
        /// </param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bPageNo">Block number of first page to read.</param>
        /// <param name="bNoOfPages">Number of pages to read.</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastInventoryPageRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bPageNo, byte bNoOfPages,
            out byte[] aUid, out byte[] aData )
        {
            byte bUidLen = 0;
            ushort wDataLen = 0;
            IntPtr ppUid = IntPtr.Zero;
            IntPtr ppData = IntPtr.Zero;

            aUid = null;
            aData = null;

            Status_t oStatus = phalICode_FastInventoryPageRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bPageNo, bNoOfPages, ref ppUid, ref bUidLen, ref ppData, ref wDataLen );
            if ( ( ppUid != IntPtr.Zero ) && ( bUidLen > 0 ) )
            {
                aUid = new byte[bUidLen];
                Marshal.Copy ( ppUid, aUid, 0, bUidLen );
            }

            if ( ( ppData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppData, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs a GetRandomNumber command. On this command, the label will respond with a random number.
        /// The received random number shall be used to diversify the password for the <seealso cref="SetPassword"/>
        /// command.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aRnd">Random number; 16 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetRandomNumber ( out byte[] aRnd )
        {
            ushort wRndLen = 0;
            IntPtr ppRnd = IntPtr.Zero;

            aRnd = null;

            Status_t oStatus = phalICode_GetRandomNumber ( m_pDataParams, ref ppRnd, ref wRndLen );
            if ( ( ppRnd != IntPtr.Zero ) && ( wRndLen > 0 ) )
            {
                aRnd = new byte[wRndLen];
                Marshal.Copy ( ppRnd, aRnd, 0, wRndLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs SetPassword command. With this command the different passwords can be transmitted to the label.
        /// This command has to be executed just once for the related passwords if the label is powered.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        ///
        /// [XOR password calculation example]
        /// aXorPwd[0] = aPassword[0] ^ aRnd[0];
        /// aXorPwd[1] = aPassword[1] ^ aRnd[1];
        /// aXorPwd[2] = aPassword[2] ^ aRnd[0];
        /// aXorPwd[3] = aPassword[3] ^ pRnd[1];
        ///
        /// Remark: This command can only be executed in addressed or selected mode except of Privacy Password.
        /// </summary>
        ///
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        /// <param name="aXorPwd">XOR Password; 32 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetPassword ( byte bPwdIdentifier, byte[] aXorPwd )
        {
            return phalICode_SetPassword ( m_pDataParams, ( byte ) Option.OFF, bPwdIdentifier, aXorPwd );
        }

        /// <summary>
        /// Performs SetPassword command. With this command the different passwords can be transmitted to the label.
        /// This command has to be executed just once for the related passwords if the label is powered.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        ///
        /// [XOR password calculation example]
        /// aXorPwd[0] = aPassword[0] ^ aRnd[0];
        /// aXorPwd[1] = aPassword[1] ^ aRnd[1];
        /// aXorPwd[2] = aPassword[2] ^ aRnd[0];
        /// aXorPwd[3] = aPassword[3] ^ pRnd[1];
        ///
        /// Remark: This command can only be executed in addressed or selected mode except of Privacy Password.
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        /// <param name="aXorPwd">XOR Password; 32 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetPassword ( byte bOption, byte bPwdIdentifier, byte[] aXorPwd )
        {
            return phalICode_SetPassword ( m_pDataParams, bOption, bPwdIdentifier, aXorPwd );
        }

        /// <summary>
        /// Performs WritePassword command. With this command, a new password is written into the related memory. Note that the
        /// old password has to be transmitted before with <seealso cref="SetPassword"/>. The new password takes effect immediately which
        /// means that the new password has to be transmitted with <seealso cref="SetPassword"/> to get access to protected blocks/pages.
        ///
        /// Remark: This command can only be executed in addressed or selected mode.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        /// <param name="aPwd">Plain Password; 32 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WritePassword ( byte bPwdIdentifier, byte[] aPwd )
        {
            return phalICode_WritePassword ( m_pDataParams, ( byte ) Option.OFF, bPwdIdentifier, aPwd );
        }

        /// <summary>
        /// Performs WritePassword command. With this command, a new password is written into the related memory. Note that the
        /// old password has to be transmitted before with <seealso cref="SetPassword"/>. The new password takes effect immediately which
        /// means that the new password has to be transmitted with <seealso cref="SetPassword"/> to get access to protected blocks/pages.
        ///
        /// Remark: This command can only be executed in addressed or selected mode.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        /// <param name="aPwd">Plain Password; 32 bits.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WritePassword ( byte bOption, byte bPwdIdentifier, byte[] aPwd )
        {
            return phalICode_WritePassword ( m_pDataParams, bOption, bPwdIdentifier, aPwd );
        }

        /// <summary>
        /// Performs LockPassword command. This command locks the addressed password. Note that the addressed password
        /// has to be transmitted before with <seealso cref="SetPassword"/>. A locked password can not be changed any longer.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockPassword ( byte bPwdIdentifier )
        {
            return phalICode_LockPassword ( m_pDataParams, ( byte ) Option.OFF, bPwdIdentifier );
        }

        /// <summary>
        /// Performs LockPassword command. This command locks the addressed password. Note that the addressed password
        /// has to be transmitted before with <seealso cref="SetPassword"/>. A locked password can not be changed any longer.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bPwdIdentifier">Password Identifier;
        ///                                 <see cref="Password.READ"/>
        ///                                 <see cref="Password.WRITE"/>
        ///                                 <see cref="Password.PRIVACY"/>
        ///                                 <see cref="Password.DESTROY"/>
        ///                                 <see cref="Password.EAS_AFI"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockPassword ( byte bOption, byte bPwdIdentifier )
        {
            return phalICode_LockPassword ( m_pDataParams, bOption, bPwdIdentifier );
        }

        /// <summary>
        /// Performs Page protection command. This command changes the protection status of a page. Note that the related
        /// passwords have to be transmitted before with <seealso cref="SetPassword"/> if the page is not public.
        ///
        /// Flag can be set using <see cref="Generic.SetConfig"/> utility interface
        ///
        /// - Protection status options for the products that do not have pages characterized as high and Low (<b>Protection status bits definition</b>).
        ///     - Protection status bits definition in plain password mode (**where x denotes L (Low) or H ( High)**)
        ///         | **Wx** | **Rx** | **32-bit Protection**                                                                                                                              |
        ///         |:------:|:------:|:---------------------------------------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                       |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected by the Read password"                                              |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected by the Write password"                                                                |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read protected by the Read password and Write protected by the Read and Write password" |
        ///
        ///
        ///         | **Wx** | **Rx** | **64-bit Protection**                                                                                             |
        ///         |:------:|:------:|:------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                      |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protection by Read plus Write password"     |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protection by Read plus Write password"                        |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protection by Read plus Write password" |
        ///
        ///
        ///     - Protection status bits definition in AES mode (**where x denotes L (Low) or H ( High)**)
        ///         | **Wx** | **Rx** | **Protection**                                                                                                                                                   |
        ///         |:------:|:------:|:-----------------------------------------------------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                                     |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected: Mutual authentication with a key with read privilege is required"                |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected: Mutual authentication with a key with write privilege is required"                                  |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protected: Mutual authentication with a key with read and write privilege is required"  |
        ///
        ///
        /// - Extended Protection status options for the products that have pages characterized as high and Low (<b>Extended Protection status byte</b>)
        ///     | **Bit**   | **Name** | **Value** |        **Description**        |
        ///     |:---------:|:---------|:---------:|:------------------------------|
        ///     | b1 ( LSB) | RL       |     0     | Page L is not read protected  |
        ///     |    ^      | ^        |     1     | Page L is read protected      |
        ///     |    b2     | WL       |     0     | Page L is not write protected |
        ///     |    ^      | ^        |     1     | Page L is write protected     |
        ///     |    b3     |          |     0     | RFU                           |
        ///     |    b4     |          |     0     | RFU                           |
        ///     |    b5     | RH       |     0     | Page H is not read protected  |
        ///     |    ^      | ^        |     1     | Page H is read protected      |
        ///     |    b6     | WH       |     0     | Page H is not write protected |
        ///     |    ^      | ^        |     1     | Page H is write protected     |
        ///     |    b7     |          |     0     | RFU                           |
        ///     | b8 ( MSB) |          |     0     | RFU                           |
        /// </summary>
        ///
        /// <param name="bPPAdd_PageNo">Page number to be protected in case of products that do not have pages characterized
        ///                             as high and Low.
        ///                             Block number to be protected in case of products that have pages characterized as
        ///                             high and Low.</param>
        /// <param name="bProtectionStatus">Refer description for supported values</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ProtectPage ( byte bPPAdd_PageNo, byte bProtectionStatus )
        {
            return phalICode_ProtectPage ( m_pDataParams, ( byte ) Option.OFF, bPPAdd_PageNo, bProtectionStatus );
        }

        /// <summary>
        /// Performs Page protection command. This command changes the protection status of a page. Note that the related
        /// passwords have to be transmitted before with <seealso cref="SetPassword"/> if the page is not public.
        ///
        /// Flag can be set using <see cref="Generic.SetConfig"/> utility interface
        ///
        /// - Protection status options for the products that do not have pages characterized as high and Low (<b>Protection status bits definition</b>).
        ///     - Protection status bits definition in plain password mode (**where x denotes L (Low) or H ( High)**)
        ///         | **Wx** | **Rx** | **32-bit Protection**                                                                                                                              |
        ///         |:------:|:------:|:---------------------------------------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                       |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected by the Read password"                                              |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected by the Write password"                                                                |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read protected by the Read password and Write protected by the Read and Write password" |
        ///
        ///
        ///         | **Wx** | **Rx** | **64-bit Protection**                                                                                             |
        ///         |:------:|:------:|:------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                      |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protection by Read plus Write password"     |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protection by Read plus Write password"                        |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protection by Read plus Write password" |
        ///
        ///
        ///     - Protection status bits definition in AES mode (**where x denotes L (Low) or H ( High)**)
        ///         | **Wx** | **Rx** | **Protection**                                                                                                                                                   |
        ///         |:------:|:------:|:-----------------------------------------------------------------------------------------------------------------------------------------------------------------|
        ///         |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                                     |
        ///         |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected: Mutual authentication with a key with read privilege is required"                |
        ///         |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected: Mutual authentication with a key with write privilege is required"                                  |
        ///         |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protected: Mutual authentication with a key with read and write privilege is required"  |
        ///
        ///
        /// - Extended Protection status options for the products that have pages characterized as high and Low (<b>Extended Protection status byte</b>)
        ///     | **Bit**   | **Name** | **Value** |        **Description**        |
        ///     |:---------:|:---------|:---------:|:------------------------------|
        ///     | b1 ( LSB) | RL       |     0     | Page L is not read protected  |
        ///     |    ^      | ^        |     1     | Page L is read protected      |
        ///     |    b2     | WL       |     0     | Page L is not write protected |
        ///     |    ^      | ^        |     1     | Page L is write protected     |
        ///     |    b3     |          |     0     | RFU                           |
        ///     |    b4     |          |     0     | RFU                           |
        ///     |    b5     | RH       |     0     | Page H is not read protected  |
        ///     |    ^      | ^        |     1     | Page H is read protected      |
        ///     |    b6     | WH       |     0     | Page H is not write protected |
        ///     |    ^      | ^        |     1     | Page H is write protected     |
        ///     |    b7     |          |     0     | RFU                           |
        ///     | b8 ( MSB) |          |     0     | RFU                           |
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bPPAdd_PageNo">Page number to be protected in case of products that do not have pages characterized
        ///                             as high and Low.
        ///                             Block number to be protected in case of products that have pages characterized as
        ///                             high and Low.</param>
        /// <param name="bProtectionStatus">Refer description for supported values</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ProtectPage ( byte bOption, byte bPPAdd_PageNo, byte bProtectionStatus )
        {
            return phalICode_ProtectPage ( m_pDataParams, bOption, bPPAdd_PageNo, bProtectionStatus );
        }

        /// <summary>
        /// Perform LockPageProtectionCondition command. This command permanent locks the protection status of a page.
        /// Note that the related passwords have to be transmitted before with <seealso cref="SetPassword"/> if the page
        /// is not public.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bPageNo">Page number to be protected.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockPageProtectionCondition ( byte bPageNo )
        {
            return phalICode_LockPageProtectionCondition ( m_pDataParams, ( byte ) Option.OFF, bPageNo );
        }

        /// <summary>
        /// Perform LockPageProtectionCondition command. This command permanent locks the protection status of a page.
        /// Note that the related passwords have to be transmitted before with <seealso cref="SetPassword"/> if the page
        /// is not public.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bPageNo">Page number to be protected.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockPageProtectionCondition ( byte bOption, byte bPageNo )
        {
            return phalICode_LockPageProtectionCondition ( m_pDataParams, bOption, bPageNo );
        }

        /// <summary>
        /// Perform GetMultipleBlockProtectionStatus command. This instructs the label to return the block protection
        /// status of the requested blocks.
        ///
        /// Remark: If bBlockNo + bNoOfBlocks exceeds the total available number of user blocks, the number of received
        /// status bytes is less than the requested number. This means that the last returned status byte corresponds to the
        /// highest available user block.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        ///
        /// Each byte of the response will contain the below information.
        ///     |    **Bit**     |         **Name**         | **Value** |           **Description**             |
        ///     |:--------------:|:-------------------------|:---------:|:--------------------------------------|
        ///     | b1 ( LSB)      | Lock bit ( WAC)          |     0     | Block is not locked                   |
        ///     | ^              | ^                        |     1     | Block is locked ( Lock Block command) |
        ///     | b2             | Read password protected  |     0     | Disabled                              |
        ///     | ^              | ^                        |     1     | Enabled                               |
        ///     | b3             | Write password protected |     0     | Disabled                              |
        ///     | ^              | ^                        |     1     | Enabled                               |
        ///     | b4             | Page protection lock     |     0     | Not Locked                            |
        ///     | ^              | ^                        |     1     | Locked                                |
        ///     | b5 - b8 ( MSB) |                          |     0     |                                       |
        ///     | **WAC - Write AccessCondition** ||||
        /// </summary>
        ///
        /// <param name="bBlockNo">First Block number.</param>
        /// <param name="bNoOfBlocks">Number of block.</param>
        /// <param name="aProtectionStates">Protection states of requested blocks. Response will be
        ///                                 repeated based on bNoOfBlocks information.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If bNoOfBlocks is zero.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetMultipleBlockProtectionStatus ( byte bBlockNo, byte bNoOfBlocks, out byte[] aProtectionStates )
        {
            Status_t oStatus;
            ushort wNumReceivedStates = 0;
            IntPtr pProtectionStatus = IntPtr.Zero;

            pProtectionStatus = Marshal.AllocHGlobal ( bNoOfBlocks );
            oStatus = phalICode_GetMultipleBlockProtectionStatus ( m_pDataParams, bBlockNo, bNoOfBlocks, pProtectionStatus, ref wNumReceivedStates );

            aProtectionStates = null;
            if ( ( pProtectionStatus != IntPtr.Zero ) && ( wNumReceivedStates > 0 ) )
            {
                aProtectionStates = new byte[wNumReceivedStates];
                Marshal.Copy ( pProtectionStatus, aProtectionStates, 0, wNumReceivedStates );
            }

            Marshal.FreeHGlobal ( pProtectionStatus );

            return oStatus;
        }

        /// <summary>
        /// Performs Destroy command. This command permanently destroys the label.
        ///
        /// The Destroy password has to be transmitted before with <seealso cref="SetPassword"/>.
        /// Remark:
        ///     - This command is irreversible and the label will never respond to any command again.
        ///     - This command can only be executed in addressed or selected mode.
        ///
        /// Note: This command is not valid for iCode DNA product as the Destroy feature is part of Mutual
        /// Authentication command ( <see cref="Generic.AuthenticateMAM"/>).
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aXorPwd">XOR Password; 32 bits. Pass the password for the ICODE products that supports and NULL
        ///                       for the products that do not support.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Destroy ( byte[] aXorPwd )
        {
            return phalICode_Destroy ( m_pDataParams, ( byte ) Option.OFF, aXorPwd );
        }

        /// <summary>
        /// Performs Destroy command. This command permanently destroys the label.
        ///
        /// The Destroy password has to be transmitted before with <seealso cref="SetPassword"/>.
        /// Remark:
        ///     - This command is irreversible and the label will never respond to any command again.
        ///     - This command can only be executed in addressed or selected mode.
        ///
        /// Note: This command is not valid for iCode DNA product as the Destroy feature is part of Mutual
        /// Authentication command ( <see cref="Generic.AuthenticateMAM"/>).
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="aXorPwd">XOR Password; 32 bits. Pass the password for the ICODE products that supports and NULL
        ///                       for the products that do not support.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Destroy ( byte bOption, byte[] aXorPwd )
        {
            return phalICode_Destroy ( m_pDataParams, bOption, aXorPwd );
        }

        /// <summary>
        /// Performs EnablePrivacy command. This command instructs the label to enter privacy mode.
        ///
        /// In privacy mode, the label will only respond to <seealso cref="GetRandomNumber"/> and <seealso cref="SetPassword"/> commands.
        /// To get out of the privacy mode, the Privacy password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Note: This command is not valid for iCode DNA product as the Destroy feature is part of Mutual
        /// Authentication command ( <see cref="Generic.AuthenticateMAM"/>).
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aXorPwd">XOR Password; 32 bits. Pass the password for the ICODE products that supports and NULL
        ///                       for the products that do not support.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t EnablePrivacy ( byte[] aXorPwd )
        {
            return phalICode_EnablePrivacy ( m_pDataParams, ( byte ) Option.OFF, aXorPwd );
        }

        /// <summary>
        /// Performs EnablePrivacy command. This command instructs the label to enter privacy mode.
        ///
        /// In privacy mode, the label will only respond to <seealso cref="GetRandomNumber"/> and <seealso cref="SetPassword"/> commands.
        /// To get out of the privacy mode, the Privacy password has to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Note: This command is not valid for iCode DNA product as the Destroy feature is part of Mutual
        /// Authentication command ( <see cref="Generic.AuthenticateMAM"/>).
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="aXorPwd">XOR Password; 32 bits. Pass the password for the ICODE products that supports and NULL
        ///                       for the products that do not support.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t EnablePrivacy ( byte bOption, byte[] aXorPwd )
        {
            return phalICode_EnablePrivacy ( m_pDataParams, bOption, aXorPwd );
        }

        /// <summary>
        /// Perform 64-BitPasswordProtection command. This instructs the label that both of the Read and Write passwords
        /// are required for protected access.
        ///
        /// Note that both the Read and Write passwords have to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t PasswordProtection_64Bit ()
        {
            return phalICode_64BitPasswordProtection ( m_pDataParams, ( byte ) Option.OFF );
        }

        /// <summary>
        /// Perform 64-BitPasswordProtection command. This instructs the label that both of the Read and Write passwords
        /// are required for protected access.
        ///
        /// Note that both the Read and Write passwords have to be transmitted before with <seealso cref="SetPassword"/>.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t PasswordProtection_64Bit ( byte bOption )
        {
            return phalICode_64BitPasswordProtection ( m_pDataParams, bOption );
        }

        /// <summary>
        /// When receiving the STAY QUIET PERSISTENT command, the label IC enters the persistent quiet state and
        /// will not send back a response.
        ///
        /// Remark: The STAY QUIET PERSISTENT command provides the same behavior as the mandatory STAY QUIET command
        /// with the only difference at a reset (power off). The label IC will turn to the ready state, if the power off
        /// time is exceeding the persistent time.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t StayQuietPersistent ()
        {
            return phalICode_StayQuietPersistent ( m_pDataParams );
        }

        /// <summary>
        /// Performs ReadSignature command. On this command, the label will respond with the signature value.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="pSign">The originality signature returned by the VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadSignature ( out byte[] aSign )
        {
            Status_t oStatus;
            ushort wSignLen = 0;
            IntPtr ppSign = IntPtr.Zero;

            aSign = null;

            oStatus = phalICode_ReadSignature ( m_pDataParams, ref ppSign, ref wSignLen );
            if ( ( ppSign != IntPtr.Zero ) && ( wSignLen > 0 ) )
            {
                aSign = new byte[wSignLen];
                Marshal.Copy ( ppSign, aSign, 0, wSignLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Reads a multiple 4 byte(s) data from the mentioned configuration block address. Here the starting address of the
        /// configuration block should be given in the parameter wBlockAddr and the number of blocks to read from the starting
        /// block should be given in the parameter bNoOfBlocks.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bBlockAddr">Configuration block address.</param>
        /// <param name="bNoOfBlocks">The n block(s) to read the configuration data.</param>
        /// <param name="aData">Multiple of 4 (4 * No Of Blocks) byte(s) of data read from the mentioned
        ///                     configuration block address.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadConfig ( byte bBlockAddr, byte bNoOfBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr ppData = IntPtr.Zero;

            Status_t oStatus = phalICode_ReadConfig ( m_pDataParams, ( byte ) Option.OFF, bBlockAddr, bNoOfBlocks, ref ppData,
                ref wDataLen );

            aData = null;
            if ( ( ppData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppData, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Reads a multiple 4 byte(s) data from the mentioned configuration block address. Here the starting address of the
        /// configuration block should be given in the parameter wBlockAddr and the number of blocks to read from the starting
        /// block should be given in the parameter bNoOfBlocks.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockAddr">Configuration block address.</param>
        /// <param name="bNoOfBlocks">The n block(s) to read the configuration data.</param>
        /// <param name="aData">Multiple of 4 (4 * No Of Blocks) byte(s) of data read from the mentioned
        ///                     configuration block address.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadConfig ( byte bOption, byte bBlockAddr, byte bNoOfBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr ppData = IntPtr.Zero;

            Status_t oStatus = phalICode_ReadConfig ( m_pDataParams, bOption, bBlockAddr, bNoOfBlocks, ref ppData,
                ref wDataLen );

            aData = null;
            if ( ( ppData != IntPtr.Zero ) && ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( ppData, aData, 0, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Writes a 4 byte data to the mentioned configuration block address.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockAddr">Configuration block address.</param>
        /// <param name="aData">A 4 byte data to be written to the mentioned configuration block address.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteConfig ( byte bOption, byte bBlockAddr, byte[] aData )
        {
            byte[] aTxData = new byte[4];
            Array.Copy ( aData, 0, aTxData, 0, 4 );

            return phalICode_WriteConfig ( m_pDataParams, bOption, bBlockAddr, aData );
        }

        /// <summary>
        /// Enables the random ID generation in the tag. This interfaces is used to instruct the tag to generate
        /// a random number in privacy mode.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t PickRandomID ()
        {
            return phalICode_PickRandomID ( m_pDataParams );
        }

        /// <summary>
        /// Provides the tag tamper status.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method.
        ///
        /// Response will contain the below information.
        ///     - Byte1: TagTamper Current State
        ///     - Byte2: TagTamper Stored State
        ///     | **TT Current State**                       | **TT Stored State**                        |
        ///     |:-------------------------------------------|:-------------------------------------------|
        ///     | O: Open                                    | O: Open                                    |
        ///     | C: Closed                                  | C: Closed                                  |
        ///     | I: Invalid                                 | E: Electrical Error                        |
        ///     | E: Electrical Error                        | Z: Backup Management Error - Invalid Data  |
        ///     | 0: Feature not active or info not provided | 0: Feature not active or info not provided |
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="aResponse">Two bytes of Tag Tamper status.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadTT ( byte bOption, out byte[] aResponse )
        {
            ushort wRspLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus = phalICode_ReadTT ( m_pDataParams, bOption, ref ppResponse, ref wRspLen );

            aResponse = null;
            if ( ( ppResponse != IntPtr.Zero ) && ( wRspLen > 0 ) )
            {
                aResponse = new byte[wRspLen];
                Marshal.Copy ( ppResponse, aResponse, 0, wRspLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs Parameter Request command. When receiving VICC PARAMETER REQUEST, PLUTUS returns all supported bit rates
        /// and timing information.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bBitRate">One byte buffer containing the supported BitRates.
        ///                         <see cref="Parameters.BITRATE_26KBPS_BOTH_DIRECTIONS"/>
        ///                         <see cref="Parameters.BITRATE_53KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_106KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_212KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_53KBPS_VICC_VCD"/>
        ///                         <see cref="Parameters.BITRATE_106KBPS_VICC_VCD"/>
        ///                         <see cref="Parameters.BITRATE_212KBPS_VICC_VCD"/>
        /// </param>
        /// <param name="bTiming">One byte buffer containing the supported timing information.
        ///                         <see cref="Parameters.TIMING_320_9_US"/>
        ///                         <see cref="Parameters.TIMING_160_5_US"/>
        ///                         <see cref="Parameters.TIMING_80_2_US"/>
        ///                         <see cref="Parameters.TIMING_SAME_BOTH_DIRECTIONS"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ParameterRequest ( out byte bBitRate, out byte bTiming )
        {
            bBitRate = ( byte ) Parameters.BITRATE_NONE;
            bTiming = ( byte ) Parameters.TIMING_NONE;
            return phalICode_ParameterRequest ( m_pDataParams, ref bBitRate, ref bTiming );
        }

        /// <summary>
        /// Performs Parameter Select command. PARAMETER SELECT command is used to activate one bit rate combination and the T1
        /// timing indicated in PARAMETER REQUEST response. Only one option in each direction shall be chosen. After the response to PARAMETER
        /// SELECT command, new parameters are valid.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bBitRate">One byte buffer containing the supported BitRates.
        ///                         <see cref="Parameters.BITRATE_26KBPS_BOTH_DIRECTIONS"/>
        ///                         <see cref="Parameters.BITRATE_53KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_106KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_212KBPS_VCD_VICC"/>
        ///                         <see cref="Parameters.BITRATE_53KBPS_VICC_VCD"/>
        ///                         <see cref="Parameters.BITRATE_106KBPS_VICC_VCD"/>
        ///                         <see cref="Parameters.BITRATE_212KBPS_VICC_VCD"/>
        /// </param>
        /// <param name="bTiming">One byte buffer containing the supported timing information.
        ///                         <see cref="Parameters.TIMING_320_9_US"/>
        ///                         <see cref="Parameters.TIMING_160_5_US"/>
        ///                         <see cref="Parameters.TIMING_80_2_US"/>
        ///                         <see cref="Parameters.TIMING_SAME_BOTH_DIRECTIONS"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ParameterSelect ( byte bBitRate, byte bTiming )
        {
            return phalICode_ParameterSelect ( m_pDataParams, bBitRate, bTiming );
        }

        /// <summary>
        /// Performs a SRAM Read command.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                         <see cref="Option.OFF"/>
        ///                         <see cref="Option.ON"/>
        ///                         <see cref="Option.DEFAULT"/>
        /// </param>
        /// <param name="bBlockNo">Block number from where the data to be read.</param>
        /// <param name="bNumBlocks">Total number of block to read.</param>
        /// <param name="aData">Information received from VICC
        ///                         If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, Block Security Status information is not available.
        ///                         Only block data is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
        ///
        ///                         If <see cref="Option.ON"/>, both Block Security Status information and Block Data is available.
        ///                         Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
        ///                         Status N + 4 byte data N, Where 1, 2 ... N is the block number.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If bNoOfBlocks is zero.
        ///         If bNoOfBlocks + bBlockNo is greater than 3Fh.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadSRAM ( byte bOption, byte bBlockNo, byte bNumBlocks, out byte[] aData )
        {
            ushort wDataLen = 0;
            IntPtr pData = IntPtr.Zero;

            pData = Marshal.AllocHGlobal ( bOption.Equals ( 1 ) ? ( ( bNumBlocks * 4 ) + bNumBlocks ) : ( bNumBlocks * 4 ) );
            Status_t oStatus = phalICode_ReadSRAM ( m_pDataParams, bOption, bBlockNo, bNumBlocks, pData, ref wDataLen );

            aData = null;
            if ( ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( pData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( pData );

            return oStatus;
        }

        /// <summary>
        /// Performs a SRAM Write command.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bBlockNo">Block number from where the data should be written.</param>
        /// <param name="bNumBlocks">Total number of block to be written.</param>
        /// <param name="aData">Information to be written to VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If bNoOfBlocks is zero.
        ///         If bNoOfBlocks + bBlockNo is greater than 3Fh.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteSRAM ( byte bOption, byte bBlockNo, byte bNumBlocks, byte[] aData )
        {
            return phalICode_WriteSRAM ( m_pDataParams, bOption, bBlockNo, bNumBlocks, aData,
                ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Performs a I2CM Read command. This command is used to read from any I2C slave connected to NTAG I2C Host.
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bI2CParam">I2C Slave address from which the data should be read and the information
        ///                         to set the Stop bit.
        ///                             Bits 0 – 6: Is for slave address. Its 7 bit address.
        ///                             Bit 7     : Configuration Bit
        ///                                         0b: Generate stop condition
        ///                                         1b: Don’t generate stop condition</param>
        /// <param name="bDataLen">Total Number of data bytes to be read. If 1 byte has to be read then the
        ///                        length will be 1.</param>
        /// <param name="aData">Information to be read from the VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>If the value of \b wDataLen is higher than 256.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t I2CMRead ( byte bI2CParam, ushort wDataLen, out byte[] aData )
        {
            IntPtr pData = IntPtr.Zero;

            pData = Marshal.AllocHGlobal ( wDataLen );
            Status_t oStatus = phalICode_I2CMRead ( m_pDataParams, bI2CParam, wDataLen, pData );

            aData = null;
            if ( ( wDataLen > 0 ) )
            {
                aData = new byte[wDataLen];
                Marshal.Copy ( pData, aData, 0, wDataLen );
            }

            Marshal.FreeHGlobal ( pData );

            return oStatus;
        }

        /// <summary>
        /// Performs a I2CM Write command. This command is used to write to any I2C slave connected to NTAG I2C Host.
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bI2CParam">I2C Slave address to which the data should be written and the information
        ///                         to set the Stop bit.
        ///                             Bits 0 – 6: Is for slave address. Its 7 bit address.
        ///                             Bit 7     : Configuration Bit
        ///                                         0b: Generate stop condition
        ///                                         1b: Don’t generate stop condition</param>
        /// <param name="aData">Information to be written to the VICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>If the value of \b wDataLen is higher than 256.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t I2CMWrite ( byte bI2CParam, byte[] aData )
        {
            return phalICode_I2CMWrite ( m_pDataParams, bI2CParam, aData, ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

#if PACKAGE_INTERNAL
        /// <summary>
        /// Performs a reading of value from the specified counter.
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bCounter">The counter to be used for incrementing the value.</param>
        /// <param name="wValue">The value read from the specified counter.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the response pointer is null.
        ///         - For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadCnt ( byte bCounter, out ushort wValue )
        {
            wValue = 0;
            return phalICode_ReadCnt ( m_pDataParams, bCounter, ref wValue );
        }

        /// <summary>
        /// Performs an value write to the usage counter.
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the write operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bCounter">The counter to be used for writing the value.</param>
        /// <param name="wValue">The value to be written.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported..
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteCnt ( byte bOption, byte bCounter, ushort wValue )
        {
            return phalICode_WriteCnt ( m_pDataParams, bOption, bCounter, wValue );
        }

        /// <summary>
        /// Performs an value increment to the usage counter.
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the increment operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bCounter">The counter to be used for incrementing the value.</param>
        /// <param name="wValue">The value to be incremented.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported..
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t IncrCnt ( byte bOption, byte bCounter, ushort wValue )
        {
            return phalICode_IncrCnt ( m_pDataParams, bOption, bCounter, wValue );
        }

        /// <summary>
        /// Performs locking of specified certificate stored in the certificate memory segment
        /// against erasing or overwriting
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="bOption">Option flag. Supported values are,
        ///                             If <see cref="Option.OFF"/> or <see cref="Option.DEFAULT"/>, the VICC shall return its response when it has completed
        ///                             the lock operation starting after t1nom [4352/fc (320,9 us), see 9.1.1] + a multiple of 4096/fc (302 us) with a total
        ///                             tolerance of ±32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
        ///
        ///                             If <see cref="Option.ON"/>, The VICC shall wait for the reception of an EOF from the VCD and upon such reception shall
        ///                             return its response.
        /// </param>
        /// <param name="bCertNo">The value to be incremented.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For the option values that are not supported..
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LockCertificate ( byte bOption, byte bCertNo )
        {
            return phalICode_LockCertificate ( m_pDataParams, bOption, bCertNo );
        }

        /// <summary>
        /// Performs Read Certificate with the tag. The purpose of FAST_READ_CERT command is to read specified ECC certificate using one of
        /// supported fast response speeds.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method.
        /// </summary>
        ///
        /// <param name="bCertNo">Certificate record number to use for writing the certificate.</param>
        /// <param name="wWordPtr">Starting Block number from which the data should be read. Currently 0000h should be used.</param>
        /// <param name="wWordCount">A 16-bit value indicating the number of double words to be read from the certificate record starting
        ///                          from the position indicated by WordPtr.The value of 0000h shall be used to indicate the full contents of
        ///                          the certificate record starting from the position indicated by WordPtr; i.e. memory shall be read from
        ///                          WordPtr until the end of the current certificate record.Currently 0000h should be used.
        /// </param>
        /// <param name="aCertificate">Certificate information read from the tag based on <b>wWordPtr</b> and <b>wWordCount</b> information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastReadCertificate ( byte bCertNo, ushort wWordPtr, ushort wWordCount, out byte[] aCertificate )
        {
            ushort wCertLen = 0;
            IntPtr ppCertificate = IntPtr.Zero;

            Status_t oStatus = phalICode_FastReadCertificate ( m_pDataParams, bCertNo, wWordPtr, wWordCount, ref ppCertificate,
                ref wCertLen );

            aCertificate = null;
            if ( ( ppCertificate != null ) && ( wCertLen > 0 ) )
            {
                aCertificate = new byte[wCertLen];
                Marshal.Copy ( ppCertificate, aCertificate, 0, wCertLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs TAM authentication with the tag. Support various authentication schemes of TAM authentication.
        /// Refer description of \b bAuthType parameter for more information.The purpose of FAST_AUTH command is,
        ///     - To provide interrogator challenge to trigger generation of cryptographic response (authentication result )
        ///       using the private key corresponding to the public key within the default certificate ( C0) specified.
        ///
        ///     - Responding in one of supported fast response speeds
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="wOption">For Future use </param>
        /// <param name="bAuthType">Type of Authenticate. One of the below mentioned values,
        ///                             - <see cref="AuthType_TAM.TAM1_2"/>
        ///                             - <see cref="AuthType_TAM.TAM1_3"/>
        /// </param>
        /// <param name="bKeyID_CertNum">KeyID or Certificate number available in the tag.</param>
        /// <param name="aChallenge">Challenge information to be exchanged with tag. Will be one of the following.
        ///                             - bAuthType = <see cref="AuthType_TAM.TAM1_2"/>: 62 bit challenge (IChallenge_TAM1.2)
        ///                             - bAuthType = <see cref="AuthType_TAM.TAM1_3"/>: 62 bit challenge (IChallenge_TAM1.3)
        /// </param>
        /// <param name="aResponse">Response received from the Tag. Will be based on \b bAuthType parameter information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If bAuthType parameter is invalid.
        ///         If the buffers are null.
        ///         For the option values that are not supported.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If KeyType is not AES128.
        ///         For Invalid response format.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastAuthenticateTAM ( ushort wOption, byte bAuthType, byte bKeyID_CertNum, byte[] aChallenge, out byte[] aResponse )
        {
            ushort wRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus =  phalICode_FastAuthenticateTAM ( m_pDataParams, wOption, bAuthType, bKeyID_CertNum, aChallenge,
                ( ushort ) ( ( aChallenge == null ) ? 0 : aChallenge.Length ), ref ppResponse, ref wRespLen );

            aResponse = null;
            if ( ( ppResponse != IntPtr.Zero ) && ( wRespLen > 0 ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, wRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// The purpose of command is to read, using one of supported fast response speeds, the content of the ResponseBuffer
        /// under the condition that this content is declared as the valid one by the ResponseBuffer validity response flag set.
        ///
        /// Flag can be set by using <seealso cref="SetConfig"/> utility method
        /// </summary>
        ///
        /// <param name="aResponse">Response received from the tag </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For the verify values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FastReadBuffer ( out byte[] aResponse )
        {
            ushort wRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus = phalICode_FastReadBuffer ( m_pDataParams, ref ppResponse, ref wRespLen );

            aResponse = null;
            if ( ( ppResponse != IntPtr.Zero ) && ( wRespLen > 0 ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, wRespLen );
            }

            return oStatus;
        }
#endif
        #endregion Custom Commands

        #region Utilities
        /// <summary>
        /// Gets the UID of the tag.
        /// </summary>
        ///
        /// <param name="aUid">8 byte UID of the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetSerialNo ( out byte[] aUid )
        {
            IntPtr ppUid = IntPtr.Zero;
            ushort pUidLen = 0;

            ppUid = Marshal.AllocHGlobal ( 8 );
            Status_t oStatus = phalICode_GetSerialNo ( m_pDataParams, ref ppUid, ref pUidLen );

            aUid = null;
            if ( ( ppUid != IntPtr.Zero ) && ( pUidLen > 0 ) )
            {
                aUid = new byte[pUidLen];
                Marshal.Copy ( ppUid, aUid, 0, pUidLen );
            }

            Marshal.FreeHGlobal ( ppUid );
            return oStatus;
        }

        /// <summary>
        /// Sets the UID of the tag.
        /// </summary>
        ///
        /// <param name="aUid">8 byte UID of the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetSerialNo ( byte[] aUid )
        {
            return phalICode_SetSerialNo ( m_pDataParams, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ) );
        }

        /// <summary>
        /// Get the configuration settings.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read.
        ///                         <see cref="Config.FLAGS"/>
        ///                         <see cref="Config.ADD_INFO"/>
        ///                         <see cref="Config.TIMEOUT_US"/>
        ///                         <see cref="Config.TIMEOUT_MS"/>
        ///                         <see cref="Config.ENABLE_BUFFERING"/>
        ///                         <see cref="Config.CHALLENGE_TIMEOUT_US"/>
        ///                         <see cref="Config.CHALLENGE_TIMEOUT_MS"/>
        /// <param name="wValue">The value for the mentioned configuration information in wConfig parameter.
        ///                         For <see cref="Config.FLAGS"/> refer <see cref="Flags"/>
        ///                         For <see cref="Config.ADD_INFO"/> any additional information like error code will be available.
        ///                         For <see cref="Config.ENABLE_BUFFERING"/>, 0 or 1 should be passed.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For configuration values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetConfig ( ushort wConfig, out ushort wValue )
        {
            wValue = 0;
            return phalICode_GetConfig ( m_pDataParams, wConfig, ref wValue );
        }

        /// <summary>
        /// Set the configuration settings.
        ///
        /// NOTE: Both the flags <seealso cref="Flags.DATA_RATE"/> and
        /// <seealso cref="NxpRdLibNet.alICode.Flags.FAST_DATA_RATE"/> should not be combined,
        /// it should be passed separately along with other flag.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read.
        ///                         <see cref="Config.FLAGS"/>
        ///                         <see cref="Config.ENABLE_BUFFERING"/>
        ///                         <see cref="Config.CHALLENGE_TIMEOUT_US"/>
        ///                         <see cref="Config.CHALLENGE_TIMEOUT_MS"/>
        /// <param name="wValue">The value for the mentioned configuration information in wConfig parameter.
        ///                         For <see cref="Config.FLAGS"/> refer <see cref="Flags"/>
        ///                         For <see cref="Config.ENABLE_BUFFERING"/>, 0 or 1 should be passed.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For configuration values that are not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalICode_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Get the type of Tag.
        /// </summary>
        ///
        /// <param name="wTagType">The type of iCode tag.
        ///                         <see cref="TagType.UNKNOWN"/>
        ///                         <see cref="TagType.ICODE_SLI"/>
        ///                         <see cref="TagType.ICODE_SLIX_S"/>
        ///                         <see cref="TagType.ICODE_SLIX_L"/>
        ///                         <see cref="TagType.ICODE_SLIX"/>
        ///                         <see cref="TagType.ICODE_SLIX_S"/>
        ///                         <see cref="TagType.ICODE_SLIX_L"/>
        ///                         <see cref="TagType.ICODE_SLI_X2"/>
        ///                         <see cref="TagType.ICODE_SLI_X3"/>
        ///                         <see cref="TagType.ICODE_DNA"/>
        ///                         <see cref="TagType.ICODE_NTAG5"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="TagType.UNKNOWN"/> For Tag types not supported.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetTagType ( out ushort wTagType )
        {
            wTagType = 0;
            return phalICode_GetTagType ( m_pDataParams, ref wTagType );
        }
        #endregion Utilities
        #endregion

        #region Memory Mapping
        /// <summary>
        /// Pointer for this layers structure.
        /// </summary>
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion

        #region Parameter Access
        /// <summary>
        /// Gets / Sets the information for Challenge.
        /// </summary>
        public virtual byte[] RndChallenge { get; set; }

        /// <summary>
        /// Gets / Sets the information to perform Read operation with Buffering Enabled / Disabled.
        /// </summary>
        public virtual byte Buffering { get; set; }
        #endregion
    }
    #endregion

    #region Software
    /// <summary>
    /// Software iCode application component of Reader Library .Net framework.
    /// </summary>
    public class Sw : Generic
    {
        #region Constants
        private const byte CHALLENGE_LENGTH = 70;
        #endregion

        #region Data Structure
        /// <summary>
        /// Parameter structure for iCode Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to Pal parameter structure. </summary>
            public IntPtr pPalSli15693DataParams;

            /// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParams;

            /// <summary> Pointer to a CryptoRng component context. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to Key Store data parameters. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> 10 byte buffer to store the random number generated by Challenge command. </summary>
            public fixed byte aRnd_Challenge[CHALLENGE_LENGTH];

            /// <summary> Enable or disable the data buffering. </summary>
            public byte bBuffering;

            /// <summary>
            /// Type of Challenge processed for <see cref="Challenge"/> command.
            /// This will help to update the timing information while reading the TResponse using
            /// <see cref="Read Buffer"/>.
            /// </summary>
            public byte bChallengeType;

            /// <summary>
            /// Timeout value to be configured for Challenge command.
            /// This will be the timing to wait for response.
            /// </summary>
            public ushort wChallange_Timeout;

            /// <summary> Timeout unit to be configured for Challenge command. </summary>
            public ushort wChallange_Timeout_Unit;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalSli15693DataParams,
            IntPtr pCryptoDataParams, IntPtr pCryptoRngDataParams, IntPtr pKeyStoreDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the iCode application layer.
        /// </summary>
        ///
        /// <param name="pPalSli15693DataParams">Pointer to palSli15693 parameter structure.</param>
        /// <param name="pCryptoDataParams">Pointer to the parameter structure of the Crypto layer for encryption.</param>
        /// <param name="pCryptoRngDataParams">Pointer to a CryptoRng component context.</param>
        /// <param name="pKeyStoreDataParams">Pointer to Key Store data parameters.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( palSli15693.Generic pPalSli15693DataParams, CryptoSym.Generic pCryptoDataParams,
            CryptoRng.Generic pCryptoRngDataParams, KeyStore.Generic pKeyStoreDataParams )
        {
            return phalICode_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalSli15693DataParams == null ) ? IntPtr.Zero : pPalSli15693DataParams.m_pDataParams,
                ( pCryptoDataParams == null ) ? IntPtr.Zero : pCryptoDataParams.m_pDataParams,
                ( pCryptoRngDataParams == null ) ? IntPtr.Zero : pCryptoRngDataParams.m_pDataParams,
                ( pKeyStoreDataParams == null ) ? IntPtr.Zero : pKeyStoreDataParams.m_pDataParams );
        }

        /// <summary>
        /// Initializes the iCode application layer.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of the data params structure.</param>
        /// <param name="pPalSli15693DataParams">Pointer to palSli15693 parameter structure.</param>
        /// <param name="pCryptoDataParams">Pointer to the parameter structure of the Crypto layer for encryption.</param>
        /// <param name="pCryptoRngDataParams">Pointer to a CryptoRng component context.</param>
        /// <param name="pKeyStoreDataParams">Pointer to Key Store data parameters.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( int wDataParamSize, palSli15693.Generic pPalSli15693DataParams, CryptoSym.Generic pCryptoDataParams,
            CryptoRng.Generic pCryptoRngDataParams, KeyStore.Generic pKeyStoreDataParams )
        {
            return phalICode_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
                ( pPalSli15693DataParams == null ) ? IntPtr.Zero : pPalSli15693DataParams.m_pDataParams,
                ( pCryptoDataParams == null ) ? IntPtr.Zero : pCryptoDataParams.m_pDataParams,
                ( pCryptoRngDataParams == null ) ? IntPtr.Zero : pCryptoRngDataParams.m_pDataParams,
                ( pKeyStoreDataParams == null ) ? IntPtr.Zero : pKeyStoreDataParams.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        /// <summary>
        /// Layers structure instance.
        /// </summary>
        private DataParams_t[] m_DataParamsInt;

        #region Constructor
        /// <summary>
        /// Constructor of this class.
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }
        #endregion

        #region Destructor
        /// <summary>
        /// Destructor of this class.
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }
        #endregion

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        /// <summary>
        /// Gets / Sets the information for Challenge.
        /// </summary>
        public override byte[] RndChallenge
        {
            get
            {
                byte[] bValue = new byte[CHALLENGE_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < CHALLENGE_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aRnd_Challenge[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > CHALLENGE_LENGTH )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aRnd_Challenge[i] = value[i];
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Gets / Sets the information to perform Read operation with Buffering Enabled / Disabled.
        /// </summary>
        public override byte Buffering
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->bBuffering;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->bBuffering = value;
                    }
                }
            }
        }
        #endregion
    }
    #endregion Software

#if !PACKAGE_PUBLIC
    #region Sam_NonX
    /// <summary>
    /// SAM AV3 NonX component for ICODE product in Reader Library .Net framework.
    /// </summary>
    public class SamAV3_NonX : Generic
    {
        #region Constants
        private const byte CHALLENGE_LENGTH = 70;
        #endregion

        #region Data Structure
        /// <summary>
        /// Parameter structure for iCode Sam NonX layer implementation
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the Hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to Pal parameter structure. </summary>
            public IntPtr pPalSli15693DataParams;

            /// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParams;

            /// <summary> Pointer to a CryptoRng component context. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to Key Store data parameters. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> 10 byte buffer to store the random number generated by Challenge command. </summary>
            public fixed byte aRnd_Challenge[70];

            /// <summary> Enable or disable the data buffering. </summary>
            public byte bBuffering;

            /// <summary>
            /// Type of Challenge processed for <see cref="Challenge"/> command.
            /// This will help to update the timing information while reading the TResponse using
            /// <see cref="Read Buffer"/>.
            /// </summary>
            public byte bChallengeType;

            /// <summary>
            /// Timeout value to be configured for Challenge command.
            /// This will be the timing to wait for response.
            /// </summary>
            public ushort wChallange_Timeout;

            /// <summary> Timeout unit to be configured for Challenge command. </summary>
            public ushort wChallange_Timeout_Unit;
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalICode_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pPalSli15693DataParams, IntPtr pCryptoDataParams, IntPtr pCryptoRngDataParams, IntPtr pKeyStoreDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the iCode DNA SamAV3 layer in NonX mode.
        /// </summary>
        ///
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of the SAM.</param>
        /// <param name="pPalSli15693DataParams">Pointer to palSli15693 parameter structure.</param>
        /// <param name="pCryptoDataParams">Pointer to the parameter structure of the Crypto layer for encryption.</param>
        /// <param name="pCryptoRngDataParams">Pointer to a CryptoRng component context.</param>
        /// <param name="pKeyStoreDataParams">Pointer to Key Store data parameters.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( Hal.SamAV3 pHalSamDataParams, palSli15693.Generic pPalSli15693DataParams, CryptoSym.Generic pCryptoDataParams,
            CryptoRng.Generic pCryptoRngDataParams, KeyStore.Generic pKeyStoreDataParams )
        {
            return phalICode_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalSli15693DataParams == null ) ? IntPtr.Zero : pPalSli15693DataParams.m_pDataParams,
                ( pCryptoDataParams == null ) ? IntPtr.Zero : pCryptoDataParams.m_pDataParams,
                ( pCryptoRngDataParams == null ) ? IntPtr.Zero : pCryptoRngDataParams.m_pDataParams,
                ( pKeyStoreDataParams == null ) ? IntPtr.Zero : pKeyStoreDataParams.m_pDataParams );
        }

        /// <summary>
        /// Initializes the iCode DNA SamAV3 layer in NonX mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of the data params structure.</param>
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of the SAM.</param>
        /// <param name="pPalSli15693DataParams">Pointer to palSli15693 parameter structure.</param>
        /// <param name="pCryptoDataParams">Pointer to the parameter structure of the Crypto layer for encryption.</param>
        /// <param name="pCryptoRngDataParams">Pointer to a CryptoRng component context.</param>
        /// <param name="pKeyStoreDataParams">Pointer to Key Store data parameters.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHalSamDataParams, palSli15693.Generic pPalSli15693DataParams, CryptoSym.Generic pCryptoDataParams,
            CryptoRng.Generic pCryptoRngDataParams, KeyStore.Generic pKeyStoreDataParams )
        {
            return phalICode_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalSli15693DataParams == null ) ? IntPtr.Zero : pPalSli15693DataParams.m_pDataParams,
                ( pCryptoDataParams == null ) ? IntPtr.Zero : pCryptoDataParams.m_pDataParams,
                ( pCryptoRngDataParams == null ) ? IntPtr.Zero : pCryptoRngDataParams.m_pDataParams,
                ( pKeyStoreDataParams == null ) ? IntPtr.Zero : pKeyStoreDataParams.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        #region Constructor
        /// <summary>
        /// Constructor of this class.
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }
        #endregion

        #region Destructor
        /// <summary>
        /// Destructor of this class.
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }
        #endregion

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        /// <summary>
        /// Gets / Sets the information for Challenge.
        /// </summary>
        public override byte[] RndChallenge
        {
            get
            {
                byte[] bValue = new byte[CHALLENGE_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < CHALLENGE_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aRnd_Challenge[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > CHALLENGE_LENGTH )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aRnd_Challenge[i] = value[i];
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Gets / Sets the information to perform Read operation with Buffering Enabled / Disabled.
        /// </summary>
        public override byte Buffering
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->bBuffering;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->bBuffering = value;
                    }
                }
            }
        }
        #endregion
    }
    #endregion Sam_NonX
#endif
}
