/*
 * Copyright 2017 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfNtag42XDna
{
    #region Enumerations
    #region Error Codes
    #region Generic Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary> MF NTAG 42xDNA Response - No changes done to backup files. </summary>
        PHAL_MFNTAG42XDNA_ERR_FORMAT = 0x80,

        /// <summary> MF NTAG 42xDNA Response - Insufficient NV-Memory. </summary>
        PHAL_MFNTAG42XDNA_ERR_OUT_OF_EEPROM_ERROR,

        /// <summary> MF NTAG 42xDNA Invalid key number specified. </summary>
        PHAL_MFNTAG42XDNA_ERR_NO_SUCH_KEY,

        /// <summary> MF NTAG 42xDNA Current configuration/status does not allow the requested command. </summary>
        PHAL_MFNTAG42XDNA_ERR_PERMISSION_DENIED,

        /// <summary> MF NTAG 42xDNA Requested AID not found on PICC. </summary>
        PHAL_MFNTAG42XDNA_ERR_APPLICATION_NOT_FOUND,

        /// <summary> MF NTAG 42xDNA Attempt to read/write data from/to beyond the files/record's limits. </summary>
        PHAL_MFNTAG42XDNA_ERR_BOUNDARY_ERROR,

        /// <summary> MF NTAG 42xDNA Previous cmd not fully completed. Not all frames were requested or provided by the PCD. </summary>
        PHAL_MFNTAG42XDNA_ERR_COMMAND_ABORTED,

        /// <summary> MF NTAG 42xDNA Num. of applns limited to 28. No additional applications possible. </summary>
        PHAL_MFNTAG42XDNA_ERR_COUNT,

        /// <summary> MF NTAG 42xDNA File/Application with same number already exists. </summary>
        PHAL_MFNTAG42XDNA_ERR_DUPLICATE,

        /// <summary> MF NTAG 42xDNA Specified file number does not exist. </summary>
        PHAL_MFNTAG42XDNA_ERR_FILE_NOT_FOUND,

        /// <summary> MF NTAG 42xDNA Crypto error returned by PICC. </summary>
        PHAL_MFNTAG42XDNA_ERR_PICC_CRYPTO,

        /// <summary> MF NTAG 42xDNA Parameter value error returned by PICC. </summary>
        PHAL_MFNTAG42XDNA_ERR_PARAMETER_ERROR,

        /// <summary> MF NTAG 42xDNA DesFire Generic error. Check additional Info. </summary>
        PHAL_MFNTAG42XDNA_ERR_DF_GEN_ERROR,

        /// <summary> MF NTAG 42xDNA ISO 7816 Generic error. Check Additional Info. </summary>
        PHAL_MFNTAG42XDNA_ERR_DF_7816_GEN_ERROR,

        /// <summary> MF NTAG 42xDNA Invalid Command Error. </summary>
        PHAL_MFNTAG42XDNA_ERR_CMD_INVALID
    };
    #endregion Generic Error Codes

    #region ISO7816 Error Codes
    /// <summary>
    /// ISO7816 error codes.
    /// </summary>
    public enum ERROR_ISO7816 : uint
    {
        /// <summary> Correct execution. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_SUCCESS = 0x9000U,

        /// <summary> Wrong length. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_WRONG_LENGTH = 0x6700U,

        /// <summary> Application / file not found. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_INVALID_APPLN = 0x6A82U,

        /// <summary> Wrong parameters P1 and/or P2. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_WRONG_PARAMS = 0x6A86U,

        /// <summary> Lc inconsistent with P1/p2. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_WRONG_LC = 0x6A87U,

        /// <summary> Wrong Le. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_WRONG_LE = 0x6C00U,

        /// <summary> No precise diagnostics. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_NO_PRECISE_DIAGNOSTICS = 0x6F00U,

        /// <summary> End of File reached. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_EOF_REACHED = 0x6282U,

        /// <summary> Limited Functionality. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS = 0x6283,

        /// <summary> File access not allowed. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_FILE_ACCESS = 0x6982U,

        /// <summary> File empty or access conditions not satisfied. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_FILE_EMPTY = 0x6985U,

        /// <summary> File not found. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_FILE_NOT_FOUND = 0x6A82U,

        /// <summary> Memory failure (unsuccessful update). </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_MEMORY_FAILURE = 0x6581U,

        /// <summary> Wrong parameter p1 or p2. READ RECORDS. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_INCORRECT_PARAMS = 0x6B00U,

        /// <summary> Wrong Class byte. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_WRONG_CLA = 0x6E00U,

        /// <summary> Instruction not supported. </summary>
        PHAL_MFNTAG42XDNA_ISO7816_ERR_UNSUPPORTED_INS = 0x6D00U
    };
    #endregion ISO7816 Error Codes
    #endregion Error Codes

    #region Auth Types
    /// <summary>
    /// Type of EV2 Authentication to perform.
    /// </summary>
    public enum AuthType : byte
    {
        /// <summary> MIFARE NTAG 42xDNA authentication type as EV2 NonFirst. </summary>
        EV2_NON_FIRST_AUTH,

        /// <summary> MIFARE NTAG 42xDNA authentication type as EV2 First. </summary>
        EV2_FIRST_AUTH,

        /// <summary> MIFARE NTAG 42xDNA authentication type as LRP NonFirst. </summary>
        LRP_NON_FIRST_AUTH,

        /// <summary> MIFARE NTAG 42xDNA authentication type as LRP First. </summary>
        LRP_FIRST_AUTH
    }
    #endregion Auth Types

    #region Diversification Options
    /// <summary>
    /// Diversification method to be used for key diversification.
    /// </summary>
    public enum DivOption : ushort
    {
        /// <summary> MIFARE NTAG 42xDNA No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> MIFARE NTAG 42xDNA Encryption based method of diversification. </summary>
        AUTH_DIV_METHOD_ENCR = 0x0000,

        /// <summary> MIFARE NTAG 42xDNA CMAC based method of diversification. </summary>
        AUTH_DIV_METHOD_CMAC = 0x0001,

        /// <summary> Bit 1. Indicating diversification of new key requred. </summary>
        CHANGE_KEY_DIV_NEW_KEY = 0x0002,

        /// <summary> Bit 2 indicating old key was diversified. </summary>
        CHANGE_KEY_DIV_OLD_KEY = 0x0004,

        /// <summary> Bit 3 indicating new key diversification using one rnd. </summary>
        CHANGE_KEY_DIV_NEW_KEY_ONERND = 0x0008,

        /// <summary> Bit 4 indicating old key diversification using one rnd.. </summary>
        CHANGE_KEY_DIV_OLD_KEY_ONERND = 0x0010,

        /// <summary> Bit 5 indicating key diversification method based on CMAC. </summary>
        CHANGE_KEY_DIV_METHOD_CMAC = 0x0020,
    }
    #endregion Auth Types

    #region SetConfiguration Options
    /// <summary>
    /// Options for SetConfiguration command
    /// </summary>
    public enum SetConfig : byte
    {
        /// <summary> Option 0 for PICC configuration. </summary>
        OPTION_0 = 0x00,

        /// <summary> Option 4 for Secure Messaging Configuration. </summary>
        OPTION_4 = 0x04,

        /// <summary> Option 5 for Capibility Data. </summary>
        OPTION_5 = 0x05,

        /// <summary> Option 7 for Tag Tamper configuration. </summary>
        OPTION_7 = 0x07,

        /// <summary> Option 10 for Failed Authentication Counter Configuration. </summary>
        OPTION_10 = 0x0A,

        /// <summary> Option 11 for Hardware Configuration. </summary>
        OPTION_11 = 0x0B,

        /// <summary> Option 13 for Silent Mode. </summary>
        OPTION_13 = 0x0D,

        /// <summary> Option 14 for Set the key(s) for GetCardUID command. </summary>
        OPTION_14 = 0x0E,
    }
    #endregion SetConfiguration Options

    #region Communication Options
    /// <summary>
    /// The communication mode to be used.
    /// </summary>
    public enum ComOption : byte
    {
        /// <summary> Plain mode of communication. </summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x20,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x10,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x30,
    }
    #endregion Communication Options

    #region ChangeFileSettings Options

    /// <summary>
    /// The File option flags for ChangeFileSettings command.
    /// </summary>
    public enum FileOption : byte
    {
        /// <summary> Plain mode of communication. </summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x02,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x01,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x03,

        /// <summary> 6th Bit of FileOption indicating Secure Dynamin Messaging and Mirroring. </summary>
        SDM_ENABLED = 0x40,

        /// <summary> MF NTAG 42xDNA Specific feature to  be used. </summary>
        SPECIFICS_ENABLED = 0x01,
    }

    /// <summary>
    /// The File option and other flags for ChangeFileSettings command.
    /// </summary>
    public enum SDMOption : byte
    {
        /// <summary> Only VCUID is considred for SDM MAC calculation. </summary>
        VCUID_PRESENT = 0x80,

        /// <summary> Only RDCTR  is considred for SDM MAC calculation. </summary>
        RDCTR_PRESENT = 0x40,

        /// <summary> Indicates the presence of SDM Read Counter Limit. </summary>
        RDCTR_LIMIT_PRESENT = 0x20,

        /// <summary> Indicates the presence of SDM ENC File data. </summary>
        SDM_ENC_FILE_DATA_PRESENT = 0x10,

        /// <summary> Indicates the presence of SDM TT Status. </summary>
        TT_STATUS_PRESENT = 0x08,

        /// <summary> Indicates the encoding as ASCII. </summary>
        ENCODING_ASCII = 0x01,
    }
    #endregion ChangeFileSettings Options

    #region ISOSelect Options
    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum Selector : byte
    {
        /// <summary> Option to indicate Selection by 2 byte file Id. </summary>
        SELECT_MF_DF_EF_FILE_IDENTIFIER = 0x00,

        /// <summary> Option to indicate Selection by child DF. </summary>
        SELECT_CHILD_DF = 0x01,

        /// <summary> Option to indicate Select EF under current DF. Fid = EF id. </summary>
        SELECT_EF_CURRENT_DF = 0x02,

        /// <summary> Option to indicate Select parent DF of the current DF. </summary>
        SELECT_PARENT_DF_CURRENT_DF = 0x03,

        /// <summary> Option to indicate Selection by DF Name. DFName and len is then valid. </summary>
        SELECT_BY_DF_NAME = 0x04
    }

    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum FCI : byte
    {
        /// <summary> Option to indicate the return of FCI. </summary>
        RETURNED = 0x00,

        /// <summary> Option to indicate the no return of FCI. </summary>
        NOT_RETURNED = 0x0C
    }
    #endregion ISO Select Options

    #region Configuration
    /// <summary>
    /// The configuration to be used for SetConfig / GetConfig.
    /// </summary>
    public enum CONFIG : uint
    {
        /// <summary> Option for getconfig to get additional info of a generic error. </summary>
        PHAL_MFNTAG42XDNA_ADDITIONAL_INFO = 0x00A1,

        /// <summary> Option to get / set current status of command wrapping in ISO 7816-4 APDUs. </summary>
        PHAL_MFNTAG42XDNA_WRAPPED_MODE = 0x00A2,

        /// <summary> Option to get / set Short Length APDU wrapping in ISO 7816-4 APDUs. </summary>
        PHAL_MFNTAG42XDNA_SHORT_LENGTH_APDU = 0x00A3U,

        /// <summary>
        /// Distinguish between the LRP_AES or AES key.
        /// The values to be used for this configuration are
        /// AES128		: 0x00
        /// LRP_AES128	: 0x01
        /// </summary>
        PHAL_MFNTAG42XDNA_SDM_KEY_TYPE = 0x00A4U,
    }
    #endregion
    #endregion Enumerations

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    public abstract class Generic
    {
        #region Constants
        public const byte READ_BLOCK_LENGTH = 16;
        public const byte WRITE_BLOCK_LENGTH = 4;
        public const byte COMPWRITE_BLOCK_LENGTH = 16;
        public const byte SIG_LENGTH = 56;
        public const byte DES_BLOCK_SIZE = 8;
        public const byte DES_KEY_LENGTH = 16;
        #endregion

        #region DLL Imports
        #region Secure Messaging
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_AuthenticateEv2 ( IntPtr pDataParams, byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard,
            byte[] pDivInput, byte bDivLen, byte bLenPcdCapsIn, byte[] bPcdCapsIn, byte[] bPcdCapsOut, byte[] bPdCapsOut );
        #endregion Secure Messaging

        #region Memory and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_SetConfiguration ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetVersion ( IntPtr pDataParams, byte[] pVerInfo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetCardUID ( IntPtr pDataParams, byte[] pUid );
        #endregion Memory and Configuration

        #region Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ChangeKey ( IntPtr pDataParams, ushort wOption, ushort wOldKeyNo, ushort wOldKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetKeyVersion ( IntPtr pDataParams, byte bKeyNo, byte bKeySetNo, byte[] pKeyVersion, ref byte bRxLen );
        #endregion Key Management

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetFileSettings ( IntPtr pDataParams, byte bFileNo, byte[] pFSBuffer, ref byte bBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetFileCounters ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pFileCounters, ref byte pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ChangeFileSettings ( IntPtr pDataParams, byte bCommMode, byte bFileNo, byte bFileOption, byte[] pAccessRights,
            byte bAdditionalInfoLen, byte[] bAdditionalInfo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ChangeFileSettingsSDM ( IntPtr pDataParams, byte bOption, byte bFileNo, byte bFileOption, byte[] pAccessRights,
            byte bSdmOptions, byte[] pSdmAccessRights, byte[] pVCUIDOffset, byte[] pSDMReadCtrOffset, byte[] pPICCDataOffset, byte[] pTTPermStatusOffset,
            byte[] pSDMMACInputOffset, byte[] pSDMENCOffset, byte[] pSDMENCLen, byte[] pSDMMACOffset, byte[] pSDMReadCtrLimit );
        #endregion File Management

        #region Data Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ReadData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pLength,
            ref IntPtr ppRxdata, ref ushort pRxdataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_WriteData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pTxData,
            byte[] pTxDataLen );
        #endregion data Management

        #region ISO7816
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_IsoSelectFile ( IntPtr pDataParams, byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, byte bDFnameLen,
            byte bExtendedLenApdu, ref IntPtr ppFCI, ref ushort pwFCILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_IsoReadBinary ( IntPtr pDataParams, ushort wOption, byte bOffset, byte bSfid, UInt32 dwBytesToRead, byte bExtendedLenApdu,
            ref IntPtr pRxBuffer, ref UInt32 pBytesRead );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_IsoUpdateBinary ( IntPtr pDataParams, byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData, UInt32 dwDataLen );
        #endregion ISO7816

        #region Originality Check
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ReadSign ( IntPtr pDataParams, byte bAddr, ref IntPtr pSignature );
        #endregion Originality Check

        #region Tag Tamper
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetTagTamperStatus ( IntPtr pDataParams, byte[] pRxBuffer, ref byte pRxLen );
        #endregion Tag Tamper

        #region Miscellaneous
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_ResetAuthentication ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_CalculateMACSDM ( IntPtr pDataParams, byte bSdmOption, ushort wSDMMacKeyNo, ushort wSDMMacKeyVer,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pIndata, ushort wInDataLen, byte[] pRespMac );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_DecryptSDMENCFileData ( IntPtr pDataParams, ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pEncdata, ushort wInDataLen, byte[] pPlainData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_DecryptSDMPICCData ( IntPtr pDataParams, ushort wEncKeyNo, ushort wEncKeyVer, byte[] pEncdata, ushort wEncDataLen,
            byte[] pPlainData );
        #endregion Miscellaneous
        #endregion

        #region Wrapper Functions
        #region Secure Messaging
        /// <summary>
        /// Performs an First or Non First Authentication depending upon bFirstAuth Parameter. This will be using the AES128 keys and will
        /// generate and verify the contents based on generic AES algorithm.
        /// </summary>
        ///
        /// <param name="bFirstAuth">One of the below options.
        ///								NxpRdLibNet.alMfNtag42XDna.AuthType.EV2_NON_FIRST_AUTH
        ///								NxpRdLibNet.alMfNtag42XDna.AuthType.EV2__FIRST_AUTH
        ///								NxpRdLibNet.alMfNtag42XDna.AuthType.LRP_NON_FIRST_AUTH
        ///								NxpRdLibNet.alMfNtag42XDna.AuthType.LRP_NON_FIRST_AUTH</param>
        /// <param name="wOption">Diversification option can be one of
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.NO_DIVERSIFICATION
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.AUTH_DIV_METHOD_ENCR
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.AUTH_DIV_METHOD_CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in keystore of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        /// <param name="bPcdCapsIn">PCD Capabilities. Upto 6 bytes.</param>
        /// <param name="bPcdCapsOut">PCD Capabilities. 6 bytes.</param>
        /// <param name="bPdCapsOut">PD Capabilities. 6 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t AuthenticateEv2 ( byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput, byte[] bPcdCapsIn,
            out byte[] bPcdCapsOut, out byte[] bPdCapsOut )
        {
            bPcdCapsOut = new byte[6];
            bPdCapsOut = new byte[6];

            return phalMfNtag42XDna_AuthenticateEv2 ( m_pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), ( byte ) ( ( bPcdCapsIn == null ) ? 0 : bPcdCapsIn.Length ),
                bPcdCapsIn, bPcdCapsOut, bPdCapsOut );
        }
        #endregion Secure Messaging

        #region Memory and Configuration
        /// <summary>
        /// Configures the card and pre personalizes the card with a key, defines if the UID or the random ID is sent back during
        /// communication setup and configures the ATS string.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option. Define length and content of the Data parameter.
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_0
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_4
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_5
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_7
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_10
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_11
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_13
        ///							NxpRdLibNet.alMfNtag42XDna.SetConfig.OPTION_14</param>
        /// <param name="pData">Data for the option specified</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfiguration ( byte bOption, byte[] pData )
        {
            if ( pData != null )
            {
                byte[] pDataTmp = new byte[pData.Length];

                pData.CopyTo ( pDataTmp, 0 );
                return phalMfNtag42XDna_SetConfiguration ( m_pDataParams, bOption, pDataTmp, ( byte ) pDataTmp.Length );
            }
            else
            {
                return phalMfNtag42XDna_SetConfiguration ( m_pDataParams, bOption, null, 0 );
            }
        }

        /// <summary>
        /// Returns manufacturing related data of the PICC
        /// </summary>
        ///
        /// <param name="pVerInfo">The version information of the product.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetVersion ( out byte[] pVerInfo )
        {
            Status_t oStatus, oStatusTemp;
            byte[] aVersion = new byte[33];
            ushort wLength = 0, wVal = 0;

            pVerInfo = null;

            /*Do a Get Config of  ADDITIONAL_INFO to have the backup of current value*/
            oStatusTemp = GetConfig ( 0x00A1, ref wVal );

            oStatus = phalMfNtag42XDna_GetVersion ( m_pDataParams, aVersion );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                /* Do a Get Config of  ADDITIONAL_INFO to read the length(wLength) of the Version string */
                oStatusTemp = GetConfig ( 0x00A1, ref wLength );
                pVerInfo = new byte[wLength];
                /* Copy the version string only to the length retreived */
                Array.Copy ( aVersion, pVerInfo, wLength );
            }

            /*Do a Set Config of  ADDITIONAL_INFO to retain the backup value*/
            oStatusTemp = SetConfig ( 0x00A1, wVal );
            return oStatus;
        }

        /// <summary>
        /// Returns the Unique ID of the PICC
        /// </summary>
        ///
        /// <param name="pUid">The complete UID of the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetCardUID ( out byte[] pUid )
        {
            pUid = new byte[10];
            ushort bLength = 0;

            Status_t oStatus = phalMfNtag42XDna_GetCardUID ( m_pDataParams, pUid );

            /* Get the length of the UID */
            if ( oStatus == new Status_t () )
            {
                oStatus = GetConfig ( 0x00A1, ref bLength );
            }

            Array.Resize<byte> ( ref pUid, bLength );
            return oStatus;
        }
        #endregion Memory and Configuration

        #region Key Management
        /// <summary>
        /// Changes any key on the PICC
        ///
        ///		<remarks>
        ///			The key on the PICC is changed to the new key.
        ///			The key type of the application keys cannot be changed. The key type of only the PICC master key can be changed.
        ///			The keys changeable are PICCDAMAuthKey, PICCDAMMACKey, PICCDAMEncKey, VCConfigurationKey, SelectVCKey, VCProximityKey,
        ///			VCPollingEncKey, VCPollingMACKey.
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="wOption">Diversification option can be one of
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.NO_DIVERSIFICATION
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY_ONERND
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY_ONERND
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY |
        ///									NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY |
        ///									NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_NEW_KEY_ONERND | NxpRdLibNet.alMfNtag42XDna.DivOption.CHANGE_KEY_DIV_OLD_KEY_ONERND</param>
        /// <param name="wKeyNo">Old key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Old key version in keystore of software or SAM.</param>
        /// <param name="wNewKeyNo">New key number in keystore of software or SAM.</param>
        /// <param name="wNewKeyVer">New key version in keystore of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeKey ( ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo, ushort wNewKeyVer, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfNtag42XDna_ChangeKey ( m_pDataParams, wOption, wKeyNo, wKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Reads out the current key version of any key stored on the PICC
        /// </summary>
        /// <param name="bKeyNo">Key number on card.</param>
        /// <param name="bKeySetNo">1 byte Key set number. Optional as it is passed only when bit[6] of bKeyNo is set.</param>
        /// <param name="pResponse">The version of the specified key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKeyVersion ( byte bKeyNo, byte bKeySetNo, out byte[] pResponse )
        {
            pResponse = new byte[16];
            byte bRxLen = 0;

            Status_t oStatus = phalMfNtag42XDna_GetKeyVersion ( m_pDataParams, bKeyNo, bKeySetNo, pResponse, ref bRxLen );

            Array.Resize<byte> ( ref pResponse, bRxLen );

            return oStatus;
        }
        #endregion Key Management

        #region File Management
        /// <summary>
        /// Get informtion on the properties of a specific file
        /// </summary>
        ///
        /// <param name="bFileNo">The file number for which the setting to be retrieved.</param>
        /// <param name="pFSBuffer">The buffer containing the settings. The buffer should be 17 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetFileSettings ( byte bFileNo, out byte[] pFSBuffer )
        {
            byte bBufferLen = 0;

            pFSBuffer = new byte[35];

            Status_t oStatus = phalMfNtag42XDna_GetFileSettings ( m_pDataParams, bFileNo, pFSBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pFSBuffer, bBufferLen );

            return oStatus;
        }

        /// <summary>
        /// Returns manufacturing related data of the PICC
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.FULL</param>
        /// <param name="bFileNo">File number for which the Counter information need to be received.</param>
        /// <param name="pFSBuffer">The SDMReadCounter information returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetFileCounters ( byte bOption, byte bFileNo, out byte[] pFSBuffer )
        {
            byte bBufferLen = 0;

            pFSBuffer = new byte[10];

            Status_t oStatus = phalMfNtag42XDna_GetFileCounters ( m_pDataParams, bOption, bFileNo, pFSBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pFSBuffer, bBufferLen );

            return oStatus;
        }

        /// <summary>
        /// Changes the access parameters of an existing file
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.FULL</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.</param>
        /// <param name="bFileOption">New communication settings for the file.
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN_1
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.FULL</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bNumAddArs">Length of bytes available in pTmcLimit buffer. </param>
        /// <param name="pAddArs">One of the below inforamtion.
        ///								If Standard AES: 4 byte TMC Limit value.
        ///								If LRP         : 2 byte TMC Limit value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeFileSettings ( byte bCommMode, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bNumAddArs, byte[] pAddArs )
        {
            if ( pAccessRights == null )
            {
                byte[] pAddArsTmp = new byte[pAddArs.Length];
                pAddArs.CopyTo ( pAddArsTmp, 0 );
                return phalMfNtag42XDna_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, null, bNumAddArs, pAddArsTmp );
            }
            else if ( ( pAddArs == null ) )
            {
                byte[] pAccessRightsTmp = new byte[pAccessRights.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );

                return phalMfNtag42XDna_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, pAccessRightsTmp, bNumAddArs, null );
            }
            else
            {
                byte[] pAccessRightsTmp = new byte[pAccessRights.Length];
                byte[] pAddArsTmp = new byte[pAddArs.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );
                pAddArs.CopyTo ( pAddArsTmp, 0 );
                return phalMfNtag42XDna_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, pAccessRightsTmp, bNumAddArs, pAddArsTmp );
            }
        }

        /// <summary>
        /// Changes the access parameters of an existing file
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.FULL</param>
        /// <param name="bMfNtag42XDna">When set to 0x01, Indicates that API should be called to extract NTAG 42xDNA specific features</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.</param>
        /// <param name="bFileOption">New communication settings for the file.
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN_1
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.FULL</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bNumAddArs">Length of bytes available in pTmcLimit buffer. </param>
        /// <param name="pAddArs">One of the below inforamtion.
        ///								If Standard AES: 4 byte TMC Limit value.
        ///								If LRP         : 2 byte TMC Limit value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeFileSettings ( byte bCommMode, byte bMfNtag42XDna, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bNumAddArs, byte[] pAddArs )
        {
            /* bMfm is ORed with bCommMode to ensure that the C interface handles it either in Desfire Ev2 way or in MFMatch way
			 * The bit-0 of bCommMode is cleared in C Interface.
			 */
            if ( bMfNtag42XDna == 1 )
            {
                bCommMode |= 0x01;
            }

            return ChangeFileSettings ( bCommMode, bFileNo, bFileOption, pAccessRights, bNumAddArs, pAddArs );
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.ComOption.FULL</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.</param>
        /// <param name="bFileOption">New communication settings for the file.
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.PLAIN_1
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.MAC
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.FULL
        ///
        ///							  Ored with one of the below flags if required.
        ///								NxpRdLibNet.alMfNtag42XDna.FileOption.SDM_ENABLED</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bSdmOptions">One of the below values to be used. Can be ORed.
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.RDCTR_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.RDCTR_LIMIT_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.SDM_ENC_FILE_DATA_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.TT_STATUS_PRESENT
        ///
        ///							  Must be ored with the above values.
        ///								NxpRdLibNet.alMfNtag42XDna.SDMOption.ENCODING_ASCII</param>
        /// <param name="pSdmAccessRights">The SDM access rights to be applied. Should be 2 bytes.</param>
        /// <param name="pVCUIDOffset">VCUID Offset information. Should be 3 bytes.</param>
        /// <param name="pSDMReadCtrOffset">SDMReadCtrLimit value (LSB first). Should be 3 bytes.</param>
        /// <param name="pPICCDataOffset">Mirror position (LSB first) for encrypted PICCData. Should be 3 bytes.</param>
        /// <param name="pTTPermStatusOffset">Tag Tamper Permanent Status Offset value. Should be 3 bytes.</param>
        /// <param name="pSDMMACInputOffset">Offset in the file where the SDM MAC computation starts (LSB first). Should be 3 bytes.</param>
        /// <param name="pSDMENCOffset">SDMENCFileData mirror position (LSB first). Should be 3 bytes.</param>
        /// <param name="pSDMENCLen">Length of the SDMENCFileData (LSB first). Should be 3 bytes.</param>
        /// <param name="pSDMMACOffset">SDMMAC mirror position (LSB first). Should be 3 bytes.</param>
        /// <param name="pSDMReadCtrLimit">SDM Read Counter Limit value. Should be 3 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeFileSettingsSDM ( byte bOption, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bSdmOptions, byte[] pSdmAccessRights,
            byte[] pVCUIDOffset, byte[] pSDMReadCtrOffset, byte[] pPICCDataOffset, byte[] pTTPermStatusOffset, byte[] pSDMMACInputOffset, byte[] pSDMENCOffset,
            byte[] pSDMENCLen, byte[] pSDMMACOffset, byte[] pSDMReadCtrLimit )
        {
            byte[] pAccessRightsTmp = null;
            byte[] pSdmAccessRightsTmp = null;
            byte[] pVCUIDOffsetTmp = null;
            byte[] pSDMReadCtrOffsetTmp = null;
            byte[] pPICCDataOffsetTmp = null;
            byte[] pTTPermStatusOffsetTmp = null;
            byte[] pSDMMACInputOffsetTmp = null;
            byte[] pSDMENCOffsetTmp = null;
            byte[] pSDMENCLenTmp = null;
            byte[] pSDMMACOffsetTmp = null;
            byte[] pSDMReadCtrLimitTmp = null;

            if ( pAccessRights != null )
            {
                pAccessRightsTmp = new byte[pAccessRights.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );
            }

            if ( pSdmAccessRights != null )
            {
                pSdmAccessRightsTmp = new byte[pSdmAccessRights.Length];
                pSdmAccessRights.CopyTo ( pSdmAccessRightsTmp, 0 );
            }

            if ( pVCUIDOffset != null )
            {
                pVCUIDOffsetTmp = new byte[pVCUIDOffset.Length];
                pVCUIDOffset.CopyTo ( pVCUIDOffsetTmp, 0 );
            }

            if ( pSDMReadCtrOffset != null )
            {
                pSDMReadCtrOffsetTmp = new byte[pSDMReadCtrOffset.Length];
                pSDMReadCtrOffset.CopyTo ( pSDMReadCtrOffsetTmp, 0 );
            }

            if ( pPICCDataOffset != null )
            {
                pPICCDataOffsetTmp = new byte[pPICCDataOffset.Length];
                pPICCDataOffset.CopyTo ( pPICCDataOffsetTmp, 0 );
            }

            if ( pTTPermStatusOffset != null )
            {
                pTTPermStatusOffsetTmp = new byte[pTTPermStatusOffset.Length];
                pTTPermStatusOffset.CopyTo ( pTTPermStatusOffsetTmp, 0 );
            }

            if ( pSDMMACInputOffset != null )
            {
                pSDMMACInputOffsetTmp = new byte[pSDMMACInputOffset.Length];
                pSDMMACInputOffset.CopyTo ( pSDMMACInputOffsetTmp, 0 );
            }
            if ( pSDMENCOffset != null )
            {
                pSDMENCOffsetTmp = new byte[pSDMENCOffset.Length];
                pSDMENCOffset.CopyTo ( pSDMENCOffsetTmp, 0 );
            }
            if ( pSDMENCLen != null )
            {
                pSDMENCLenTmp = new byte[pSDMENCLen.Length];
                pSDMENCLen.CopyTo ( pSDMENCLenTmp, 0 );
            }
            if ( pSDMMACOffset != null )
            {
                pSDMMACOffsetTmp = new byte[pSDMMACOffset.Length];
                pSDMMACOffset.CopyTo ( pSDMMACOffsetTmp, 0 );
            }
            if ( pSDMReadCtrLimit != null )
            {
                pSDMReadCtrLimitTmp = new byte[pSDMReadCtrLimit.Length];
                pSDMReadCtrLimit.CopyTo ( pSDMReadCtrLimitTmp, 0 );
            }

            return phalMfNtag42XDna_ChangeFileSettingsSDM ( m_pDataParams, bOption, bFileNo, bFileOption, pAccessRightsTmp, bSdmOptions, pSdmAccessRightsTmp,
                pVCUIDOffsetTmp, pSDMReadCtrOffsetTmp, pPICCDataOffsetTmp, pTTPermStatusOffsetTmp, pSDMMACInputOffsetTmp, pSDMENCOffsetTmp, pSDMENCLenTmp,
                pSDMMACOffset, pSDMReadCtrLimitTmp );
        }
        #endregion File Management

        #region Data Management
        /// <summary>
        /// Reads data from standard data files or backup data files
        ///
        ///		<remarks></remarks>
        ///			Chaining upto the size of the HAL Rx buffer is handled within this function. If more data is to be read, the user has to call
        ///			this function again with bOption = #PH_EXCHANGE_RXCHAINING | [one of the communication options]
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.MAC
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.FULL
        ///
        ///						  NxpRdLibNet.ExchangeOptions.RXCHAINING: To be Or'd with the above option
        ///																  flag if Chaining status is returned. </param>
        /// <param name="bIns">If set, uses ISO 14443-4 chaining instead of DESFire application chaining.</param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="pOffset">The offset from where the data should be read. Will be of 3 bytes with LSB first.
        ///							If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///							If 0x10 bytes need to be read then it will be 10 00 00.
        ///							If complete file need to be read then it will be 00 00 00.</param>
        /// <param name="pRxdata">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pLength, out byte[] pRxdata )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            byte[] pRxdataLen;
            ushort rxDataLen = 0;

            pRxdataLen = new byte[3];

            Status_t oStatus = phalMfNtag42XDna_ReadData ( m_pDataParams, bOption, bIns, bFileNo, pOffset, pLength, ref ppRxdata, ref rxDataLen );
            if ( rxDataLen != 0 )
            {
                pRxdata = new byte[rxDataLen];
                Marshal.Copy ( ppRxdata, pRxdata, 0, rxDataLen );
            }
            else
            {
                pRxdata = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Writes data to standard data files or backup data files
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.PLAIN
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.MAC
        ///							NxpRdLibNet.alMfNtag42XDna.ComOption.FULL</param>
        /// <param name="bIns">If set, uses ISO 14443-4 chaining instead of DESFire application chaining.</param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="pOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///							If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pTxData">The data to be written to the PICC.</param>
        /// <param name="pTxDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///								If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WriteData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pTxData, byte[] pTxDataLen )
        {
            byte[] pTxDataTmp = null;
            byte[] pOffsetTmp = null;
            byte[] pTxDataLenTmp = null;
            if ( pTxData != null )
            {
                pTxDataTmp = new byte[pTxData.Length];
                pTxData.CopyTo ( pTxDataTmp, 0 );
            }
            if ( pOffset != null )
            {
                pOffsetTmp = new byte[pOffset.Length];
                pOffset.CopyTo ( pOffsetTmp, 0 );
            }
            if ( pTxDataLen != null )
            {
                pTxDataLenTmp = new byte[pTxDataLen.Length];
                pTxDataLen.CopyTo ( pTxDataLenTmp, 0 );
            }

            return phalMfNtag42XDna_WriteData ( m_pDataParams, bOption, bIns, bFileNo, pOffsetTmp, pTxDataTmp, pTxDataLenTmp );
        }
        #endregion data Management

        #region ISO7816
        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfNtag42XDna.FCI.RETURNED
        ///							NxpRdLibNet.alMfNtag42XDna.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_CHILD_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_EF_CURRENT_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when bOption = 0x04.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, out byte[] pFCI )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            ushort wBytesRead = 0;
            byte bDfNameLen = 0x00;

            /*
             * Modified API is provided immediately after this API under the commented section "ISOSELECTFILE".
             * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
             * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
             * to this modified API.
             */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc/Le is passed as short APDU */

            if ( pDFname != null )
            {
                bDfNameLen = ( byte ) pDFname.Length;
            }

            Status_t oStatus = phalMfNtag42XDna_IsoSelectFile ( m_pDataParams, bOption, bSelector, pFid, pDFname, bDfNameLen, bExtendedLenApdu,
                ref ppRxBuffer, ref wBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
            {
                pFCI = new byte[wBytesRead];
                Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
            }
            else
            {
                pFCI = null;
            }

            return oStatus;
        }

        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfNtag42XDna.FCI.RETURNED
        ///							NxpRdLibNet.alMfNtag42XDna.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_CHILD_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_EF_CURRENT_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///								NxpRdLibNet.alMfNtag42XDna.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when bOption = 0x04.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
        ///								   any other value for Short APDUs.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, byte bExtendedLenApdu, out byte[] pFCI )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            ushort wBytesRead = 0;
            byte bDfNameLen = 0x00;

            if ( pDFname != null )
            {
                bDfNameLen = ( byte ) pDFname.Length;
            }

            Status_t oStatus= phalMfNtag42XDna_IsoSelectFile ( m_pDataParams, bOption, bSelector, pFid, pDFname, bDfNameLen, bExtendedLenApdu,
                ref ppRxBuffer, ref wBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
            {
                pFCI = new byte[wBytesRead];
                Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
            }
            else
            {
                pFCI = null;
            }
            return oStatus;
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT    : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RX_CHAINING: To exchange the chaining command to PICC in
        ///																	 case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied. Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, byte bBytesToRead, out byte[] pRxBuffer )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            /*
             * Modified API is provided immediately after this API under the commented section "ISOREADBINARY".
             * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
             * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
             * to this modified API
             */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Le is passed as short APDU */

            Status_t oStatus= phalMfNtag42XDna_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, bBytesToRead, bExtendedLenApdu, ref ppRxBuffer,
                ref dwBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT    : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RX_CHAINING: To exchange the chaining command to PICC in
        ///																	 case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied. Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or any other value for Short APDUs.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, UInt32 bBytesToRead, byte bExtendedLenApdu, out byte[] pRxBuffer )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            Status_t oStatus= phalMfNtag42XDna_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, bBytesToRead, bExtendedLenApdu, ref ppRxBuffer,
                ref dwBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] pData )
        {
            /*
             * Modified API is provided immediately after this API under the commented section "ISOUPDATEBINARY".
             * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
             * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
             * to this modified API
             */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            return phalMfNtag42XDna_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, pData,
                ( byte ) ( ( pData == null ) ? 0 : pData.Length ) );
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
        ///								   any other value for Short APDUs.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData )
        {
            return phalMfNtag42XDna_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, pData,
                ( byte ) ( ( pData == null ) ? 0 : pData.Length ) );
        }
        #endregion ISO7816

        #region Originality Check
        /// <summary>
        /// Performs the originality check to verify the genuineness of chip.
        /// </summary>
        ///
        /// <param name="bAddr">Value is always 00. Present for forward compatibility reasons.</param>
        /// <param name="pSignature">The plain 56 bytes originality signature of the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadSign ( byte bAddr, out byte[] pSignature )
        {
            IntPtr ppRxdata = IntPtr.Zero;

            Status_t oStatus = phalMfNtag42XDna_ReadSign ( m_pDataParams, bAddr, ref ppRxdata );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                pSignature = new byte[SIG_LENGTH];
                Marshal.Copy ( ppRxdata, pSignature, 0, SIG_LENGTH );
            }
            else
            {
                pSignature = null;
            }

            return oStatus;
        }
        #endregion Originality Check

        #region GetTagTamper
        /// <summary>
        /// Returns Tag Tamper Status data of the PICC
        /// </summary>
        ///
        /// <param name="pOutBuffer">Read Tag Tamper Protection status</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetTagTamperStatus ( out byte[] pOutBuffer )
        {
            byte bBufferLen = 0;

            pOutBuffer = new byte[2];

            Status_t oStatus = phalMfNtag42XDna_GetTagTamperStatus ( m_pDataParams, pOutBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pOutBuffer, bBufferLen );

            return oStatus;
        }
        #endregion

        #region Miscellaneous
        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_ADDITIONAL_INFO
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_SDM_KEY_TYPE
        ///
        ///						  Support for Software mode only including the above ones.
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_WRAPPED_MODE
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_SHORT_LENGTH_APDU</param>
        /// <param name="pValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
        {
            return phalMfNtag42XDna_GetConfig ( m_pDataParams, wConfig, ref pValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_ADDITIONAL_INFO
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_SDM_KEY_TYPE
        ///
        ///						  Support for Software mode only including the above ones.
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_WRAPPED_MODE
        ///							NxpRdLibNet.alMfNtag42XDna.CONFIG.PHAL_MFNTAG42XDNA_SHORT_LENGTH_APDU</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalMfNtag42XDna_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ResetAuthentication ()
        {
            return phalMfNtag42XDna_ResetAuthentication ( m_pDataParams );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        /// <param name="bval">If set the PICC level is reset.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ResetAuthentication ( byte bval )
        {
            Status_t oStatusTemp, oStatus;
            ushort wVal = 0,wResetVal = 0xFFFF;

            if ( bval == 0x01 )
            {
                /*Do a Get Config of  ADDITIONAL_INFO to have the backup of current value*/
                oStatusTemp = GetConfig ( 0x00A1, ref wVal );

                /*Do a Set Config to set value 0xFFFF for PICC level Reset*/
                oStatusTemp = SetConfig ( 0x00A1, wResetVal );
            }

            oStatus = ResetAuthentication ();

            /*Do a Set Config of  ADDITIONAL_INFO to retain the backup value*/
            oStatusTemp = SetConfig ( 0x00A1, wVal );

            return oStatus;
        }

        /// <summary>
        /// Calculate MAC for SDM information.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option. 0xFFFF is for No Diversification.</param>
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.RDCTR_PRESENT</param>
        /// <param name="wSDMFileReadKeyNo">Key number in SW key store key.</param>
        /// <param name="wSDMFileReadKeyVer">Key version in SW key store key.</param>
        /// <param name="pDivInput">Diversification input to diversify the SDMMACKey.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pIndata">Data read out of PICC from SDMMacInputoffset for ReadLength bytes</param>
        /// <param name="wInDataLen">Length of bytes available in InData buffer.</param>
        /// <param name="pRespMac">The computed MAC information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CalculateMACSDM ( ushort wOption, byte bSdmOption, ushort wSDMFileReadKeyNo, ushort wSDMFileReadKeyVer, byte[] pDivInput,
            byte[] pUid, byte[] pSDMReadCtr, byte[] pIndata, ushort wInDataLen, out byte[] pRespMac )
        {
            pRespMac = new byte[8];
            return phalMfNtag42XDna_CalculateMACSDM ( m_pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer,
                0, 0, pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pIndata, wInDataLen, pRespMac );
        }

        /// <summary>
        /// Calculate MAC for SDM information.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfIdentity.SDMOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfIdentity.SDMOption.RDCTR_PRESENT</param>
        /// <param name="wSDMFileReadKeyNo">Key number in SW key store key.</param>
        /// <param name="wSDMFileReadKeyVer">Key version in SW key store key.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pIndata">Data read out of PICC from SDMMacInputoffset for ReadLength bytes</param>
        /// <param name="wInDataLen">Length of bytes available in InData buffer.</param>
        /// <param name="pRespMac">The computed MAC information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wSDMFileReadKeyNo, ushort wSDMFileReadKeyVer, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] pUid, byte[] pSDMReadCtr, byte[] pIndata, ushort wInDataLen, out byte[] pRespMac )
        {
            pRespMac = new byte[8];
            return phalMfNtag42XDna_CalculateMACSDM ( m_pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer, wRamKeyNo, wRamKeyVer,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pIndata, wInDataLen, pRespMac );
        }

        /// <summary>
        /// Decrypt the SDM File data information.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.RDCTR_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.SDM_ENC_FILE_DATA_PRESENT</param>
        /// <param name="wEncKeyNo">Key number in keystore of Software.</param>
        /// <param name="wEncKeyVer">Key version in keystore of Software.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pEncdata">Input Enciphered data. Caller has to pass only the Enciphered data which is between
        ///						   SDMEncOffset for SDMEncLength bytes.</param>
        /// <param name="pPlainData">The decrypted SDM ENC file data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptSDMENCFileData ( ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer, byte[] pUid, byte[] pSDMReadCtr, byte[] pEncdata,
            out byte[] pPlainData )
        {
            pPlainData = new byte[( pEncdata == null ) ? 0 : pEncdata.Length];

            Status_t oStatus = phalMfNtag42XDna_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wEncKeyNo, wEncKeyVer, 0, 0,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pEncdata, ( byte ) ( ( pEncdata == null ) ? 0 : pEncdata.Length ),
                pPlainData );

            if ( !oStatus.Equals ( new Status_t () ) )
                pPlainData = null;

            return oStatus;
        }

        /// <summary>
        /// Decrypt the SDM File data information.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.RDCTR_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.SDM_ENC_FILE_DATA_PRESENT</param>
        /// <param name="wEncKeyNo">Key number in keystore of Software.</param>
        /// <param name="wEncKeyVer">Key version in keystore of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pEncdata">Input Enciphered data. Caller has to pass only the Enciphered data which is between
        ///						   SDMEncOffset for SDMEncLength bytes.</param>
        /// <param name="pPlainData">The decrypted SDM ENC file data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptSDMENCFileData ( ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer, ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid,
            byte[] pSDMReadCtr, byte[] pEncdata, out byte[] pPlainData )
        {
            pPlainData = new byte[( pEncdata == null ) ? 0 : pEncdata.Length];

            Status_t oStatus = phalMfNtag42XDna_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wEncKeyNo, wEncKeyVer, wRamKeyNo, wRamKeyVer,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pEncdata, ( byte ) ( ( pEncdata == null ) ? 0 : pEncdata.Length ),
                pPlainData );

            if ( !oStatus.Equals ( new Status_t () ) )
                pPlainData = null;

            return oStatus;
        }

        /// <summary>
        /// Decrypt SDM PICC Data
        /// </summary>
        ///
        /// <param name="wEncKeyNo">Key number in SW key store key.</param>
        /// <param name="wEncKeyVer">Key version in SW key store key.</param>
        /// <param name="pEncdata">The Encrypted PICC data.</param>
        /// <param name="pPlainData">The decrypted SDM PICC data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptSDMPICCData ( ushort wEncKeyNo, ushort wEncKeyVer, byte[] pEncdata, out byte[] pPlainData )
        {
            byte[] aPlainDataTmp = new byte[16];

            pPlainData = null;
            Status_t oStatus = phalMfNtag42XDna_DecryptSDMPICCData ( m_pDataParams, wEncKeyNo, wEncKeyVer, pEncdata,
                ( byte ) ( ( pEncdata == null ) ? 0 : pEncdata.Length ), aPlainDataTmp );

            pPlainData = ( byte[] ) aPlainDataTmp.Clone ();

            return oStatus;
        }
        #endregion Miscellaneous
        #endregion

        #region Memory Maping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion Generic

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    [ClassInterface ( ClassInterfaceType.AutoDual )]
    public class Sw : alMfNtag42XDna.Generic
    {
        #region Constants
        private const byte AID_LENGTH = 3;
        private const byte IV_LENGTH = 16;
        private const byte SESSION_KEY_LENGTH = 24;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE NTAG 42xDNA Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>Layer ID for this component, NEVER MODIFY!.</summary>
            public ushort wId;

            /// <summary>Pointer to the parameter structure of the palMifare component.</summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the parameter structure of the KeyStore layer.</summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary>Pointer to the parameter structure of the Crypto layer for encryption.</summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary>Pointer to the parameter structure of the CryptoRng layer.</summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary>Pointer to the parameter structure of the CryptoMAC.</summary>
            public IntPtr pCryptoMACDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pHalDataParams;

            /// <summary>Session key for this authentication</summary>
            public fixed byte bSessionKey[SESSION_KEY_LENGTH];

            /// <summary>key number against which this authentication is done</summary>
            public byte bKeyNo;

            /// <summary>Max size of IV can be 16 bytes</summary>
            public fixed byte bIv[IV_LENGTH];

            /// <summary>Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA)</summary>
            public byte bAuthMode;

            /// <summary>Aid of the currently selected application</summary>
            public fixed byte aAid[AID_LENGTH];

            /// <summary>DES,3DES, 3K3DES or AES</summary>
            public byte bCryptoMethod;

            /// <summary>Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus.</summary>
            public byte bWrappedMode;

            /// <summary>2 Byte CRC initial value in Authenticate mode.</summary>
            public ushort wCrc;

            /// <summary>4 Byte CRC initial value in 0x1A, 0xAA mode.</summary>
            public uint dwCrc;

            /// <summary>Additional info.</summary>
            public ushort wAdditionalInfo;

            /// <summary>Amount of data to be read. Required for Enc read to verify CRC.</summary>
            public uint dwPayLoadLen;

            /// <summary>Command count within transaction.</summary>
            public ushort wCmdCtr;

            /// <summary>Transaction Identifier.</summary>
            public fixed byte bTI[4];

            /// <summary>Authentication MAC key for the session.</summary>
            public fixed byte bSesAuthMACKey[16];

            /// <summary>Session Auth master key.</summary>
            public fixed byte bKeySessionAuthMaster[16];

            /// <summary>Buffer containing unprocessed bytes for read mac answer stream.</summary>
            public fixed byte pUnprocByteBuff[16];

            /// <summary>Amount of data in the pUnprocByteBuff.</summary>
            public byte bNoUnprocBytes;

            /// <summary>Buffer to store last Block of encrypted data in case of chaining.</summary>
            public fixed byte bLastBlockBuffer[16];

            /// <summary>Last Block Buffer Index.</summary>
            public byte bLastBlockIndex;

            /// <summary>Parameter for force set Short Length APDU in case of BIG ISO read.</summary>
            public byte bShortLenApdu;

            /// <summary>Distinguish between the LRP_AES or AES key.</summary>
            public byte bKeyType;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams, IntPtr pKeyStoreDataParams,
            IntPtr pCryptoDataParamsEnc, IntPtr pCryptoDataParamsMac, IntPtr pCryptoRngDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE NTAG 42XDNA software component. This interface is for legacy support.
        /// </summary>
        ///
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to TMIUtils parameter sturcture.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [Obsolete ( "The intialization method has a parameter \"pTMIUtils\" which is not required. " +
            "Use the another method.", false )]
        public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMAC,
            CryptoRng.Generic pCryptoRng, NxpRdLibNet.TMIUtils pTMIUtils, Hal.Generic pHalparams )
        {
            return phalMfNtag42XDna_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
                ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams, ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams,
                ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams, ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }

        /// <summary>
        /// Initialization API for MIFARE NTAG 42XDNA software component.
        /// </summary>
        ///
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMAC,
            CryptoRng.Generic pCryptoRng, Hal.Generic pHalparams )
        {
            return phalMfNtag42XDna_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
                ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams, ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams,
                ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams, ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }
#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE NTAG 42XDNA software component. This interface is for legacy support.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to TMIUtils parameter sturcture.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [Obsolete ( "The intialization method has a parameter \"pTMIUtils\" which is not required. " +
            "Use the another method.", false )]
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc,
            CryptoSym.Generic pCryptoMAC, CryptoRng.Generic pCryptoRng, NxpRdLibNet.TMIUtils pTMIUtils, Hal.Generic pHalparams )
        {
            return phalMfNtag42XDna_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
                ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams, ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams,
                ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams, ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
                ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }

        /// <summary>
        /// Initialization API for MIFARE NTAG 42XDNA software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc,
            CryptoSym.Generic pCryptoMAC, CryptoRng.Generic pCryptoRng, Hal.Generic pHalparams )
        {
            return phalMfNtag42XDna_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
                ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams, ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams,
                ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams, ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
                ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }
#endif
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping

        #region Parameter Access
        public byte[] aAid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aAid[i] = value[i];
                        }
                    }
                }
            }
        }
        public byte[] bIv
        {
            get
            {
                byte[] bValue = new byte[IV_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < IV_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bIv[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > IV_LENGTH )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->bIv[i] = value[i];
                        }
                    }
                }
            }
        }
        public byte[] bSessionKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bSessionKey[i];
                        }
                    }
                }
                return bValue;
            }
        }
        public ushort wCmdCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->wCmdCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->wCmdCtr = value;
                    }
                }
            }
        }
        #endregion Parameter Access
    }
    #endregion Software

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam_NonX
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 NonX Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_NonX : alMfNtag42XDna.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE NTAG 42XDNA SamAV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
            public byte bAuthMode;

            /// <summary> AuthType. Whether its EV2 or LRP. </summary>
            public byte bAuthType;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte aAid[3];

            /// <summary> Specific error codes for Desfire generic errors. </summary>
            public ushort wAdditionalInfo;

            /// <summary>Distinguish between the LRP_AES or AES key.</summary>
            public byte bKeyType;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pPalMifareDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialization API for AL MIFARE NTAG 42XDNA to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="pPalMifare">Pointer to Pal Mifare parameter structure</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare )
        {
            return phalMfNtag42XDna_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL MIFARE NTAG 42XDNA to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="pPalMifare">Pointer to Pal Mifare parameter structure</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare )
        {
            return phalMfNtag42XDna_SamAV3_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams,
                ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
            }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte AuthType
        {
            get { return m_DataParamsInt[0].bAuthType; }
            set { m_DataParamsInt[0].bAuthType = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->aAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion SamAV3
    #endregion Sam_NonX

    #region Sam_X
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 X Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : alMfNtag42XDna.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE NTAG 42XDNA SamAV3 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
            public byte bAuthMode;

            /// <summary> AuthType. Whether its EV2 or LRP. </summary>
            public byte bAuthType;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte aAid[3];

            /// <summary> Specific error codes for Desfire generic errors. </summary>
            public ushort wAdditionalInfo;

            /// <summary>Distinguish between the LRP_AES or AES key.</summary>
            public byte bKeyType;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfNtag42XDna_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialization API for AL MIFARE NTAG 42XDNA to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSam )
        {
            ushort wStatus = phalMfNtag42XDna_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
            return wStatus;
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL MIFARE NTAG 42XDNA to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam )
        {
            return phalMfNtag42XDna_SamAV3_X_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
        }
#endif

        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte AuthType
        {
            get { return m_DataParamsInt[0].bAuthType; }
            set { m_DataParamsInt[0].bAuthType = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->aAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion SamAV3
    #endregion Sam_X
#endif
}
