/*
 * Copyright 2013 - 2014, 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfc
{
    #region Parameter Options
    #region Mifare Key Type
    /// <summary>
    /// Key section to be used for Authentication form Sectop Trailer.
    /// </summary>
    public enum KeyType : byte
    {
        /// <summary>
        /// Key A used be used for Authentication.
        /// </summary>
        A = 0x0A,

        /// <summary>
        /// Key B used be used for Authentication.
        /// </summary>
        B = 0x0B
    }
    #endregion Mifare Key Type

    #region PersonalizeUID Options
    /// <summary>
    /// Enumeration options for PersoanlizeUID command.
    /// </summary>
    public enum PersonalizeUID : byte
    {
        /// <summary>
        /// MIFARE(R) Classic UID type UIDF0. Anti-collision and selection with the double size
        /// UID according to ISO/IEC 14443-3.
        /// </summary>
        UIDF0 = 0x00,

        /// <summary>
        /// MIFARE(R) Classic UID type UIDF1. Anti-collision and selection with the double size UID
        /// according to ISO/IEC 14443-3 and optional usage of a selection process shortcut.
        /// </summary>
        UIDF1 = 0x40,

        /// <summary>
        /// MIFARE(R) Classic UID type UIDF2. anti-collision and selection with a single size random
        /// ID according to ISO/IEC 14443-3.
        /// </summary>
        UIDF2 = 0x20,

        /// <summary>
        /// MIFARE(R) Classic UID type UIDF3. anti-collision and selection with a single size NUID
        /// according to ISO/IEC 14443-3 where the NUID is calculated out of the 7-byte UID.
        /// </summary>
        UIDF3 = 0x60,
    }
    #endregion PersoanlizeUID
    #endregion Parameter Options

    #region BASE
    /// <summary>
    /// Generic class for Mifare Classic Application layer. Here all the interfaces related to Mifare
    /// Classic commands are available.
    /// </summary>
    public abstract class Generic
    {
        #region DLLIMPORTS
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Authenticate ( IntPtr pDataParams, byte bBlockNo, byte bKeyType, ushort wKeyNumber, ushort wKeyVersion,
            byte[] pUid, byte bUidLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Read ( IntPtr pDataParams, byte bBlockNo, byte[] pBlockData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_ReadValue ( IntPtr pDataParams, byte bBlockNo, byte[] pValue, ref byte bAddrData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Write ( IntPtr pDataParams, byte bBlockNo, byte[] pBlockData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_WriteValue ( IntPtr pDataParams, byte bBlockNo, byte[] pValue, byte bAddrData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Increment ( IntPtr pDataParams, byte bBlockNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Decrement ( IntPtr pDataParams, byte bBlockNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Transfer ( IntPtr pDataParams, byte bBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Restore ( IntPtr pDataParams, byte bBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_IncrementTransfer ( IntPtr pDataParams, byte bSrcBlockNo, byte bDstBlockNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_DecrementTransfer ( IntPtr pDataParams, byte bSrcBlockNo, byte bDstBlockNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_RestoreTransfer ( IntPtr pDataParams, byte bSrcBlockNo, byte bDstBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_PersonalizeUid ( IntPtr pDataParams, byte bUidType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_GetVersion ( IntPtr pDataParams, byte[] pVersion );
        #endregion

        #region DLL_WRAPPED_FUNCTIONS
        /// <summary>
        /// Authenticates with the card using MIFARE keys provided.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for Authentication.</param>
        /// <param name="bKeyType">The type of key to be used.
        ///								NxpRdLibNet.alMfc.KeyType.A
        ///								NxpRdLibNet.alMfc.KeyType.A
        ///							</param>
        /// <param name="wKeyNumber">Key number to be used in authentication.</param>
        /// <param name="wKeyVersion">Key version to be used in authentication.</param>
        /// <param name="pUid">Complete serial number. [4 / 7 / 10 bytes in lenght].</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Authenticate ( byte bBlockNo, byte bKeyType, ushort wKeyNumber, ushort wKeyVersion, byte[] pUid )
        {
            byte bUid_Len = 0;
            if ( pUid != null )
            {
                bUid_Len = ( byte ) pUid.Length;
            }

            return phalMfc_Authenticate ( m_pDataParams, bBlockNo, bKeyType, wKeyNumber, wKeyVersion, pUid, bUid_Len );
        }

        /// <summary>
        /// Authenticates with the card using MIFARE keys provided.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for Authentication.</param>
        /// <param name="eKeyType">The type of key to be used.
        ///								NxpRdLibNet.alMfc.KeyType.A
        ///								NxpRdLibNet.alMfc.KeyType.A
        ///							</param>
        /// <param name="wKeyNumber">Key number to be used in authentication.</param>
        /// <param name="wKeyVersion">Key version to be used in authentication.</param>
        /// <param name="pUid">Complete serial number. [4 / 7 / 10 bytes in lenght].</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Authenticate ( byte bBlockNo, KeyType eKeyType, ushort wKeyNumber, ushort wKeyVersion, byte[] pUid )
        {
            return Authenticate ( bBlockNo, ( byte ) eKeyType, wKeyNumber, wKeyVersion, pUid );
        }

        /// <summary>
        /// Perform MIFARE(R) Read command with MIFARE Picc.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for reading the data.</param>
        /// <param name="pBlockData">The data read from MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Read ( byte bBlockNo, out byte[] pBlockData )
        {
            pBlockData = new byte[16];
            Status_t status = phalMfc_Read ( m_pDataParams, bBlockNo, pBlockData );
            if ( !status.Equals ( Error_Gen.SUCCESS ) )
            {
                pBlockData = new byte[0];
            }
            return status;
        }

        /// <summary>
        /// Reads the data from from MIFARE PICC. This read data will be in value format.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for reading the data.</param>
        /// <param name="dwValue">The value read from MIFARE PICC.</param>
        /// <param name="bAddrData">The address read from MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadValueDw ( byte bBlockNo, out int dwValue, out byte bAddrData )
        {
            byte[] pValue = new byte[4];
            dwValue = 0;
            bAddrData = 0;
            Status_t status = phalMfc_ReadValue ( m_pDataParams, bBlockNo, pValue, ref bAddrData );
            if ( status.Equals ( Error_Gen.SUCCESS ) )
            {
                dwValue = ConvertByteArrayToInt ( pValue );
            }
            return status;
        }

        /// <summary>
        /// Reads the data from from MIFARE PICC. This read data will be in value format.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for reading the data.</param>
        /// <param name="pValue">The value read from MIFARE PICC.</param>
        /// <param name="bAddrData">The address read from MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadValue ( byte bBlockNo, out byte[] pValue, out byte bAddrData )
        {
            pValue = new byte[4];
            bAddrData = 0;
            Status_t status = phalMfc_ReadValue ( m_pDataParams, bBlockNo, pValue, ref bAddrData );
            return status;
        }

        /// <summary>
        /// Write the data to MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for writing the data.</param>
        /// <param name="pBlockData">The data to be written to MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Write ( byte bBlockNo, byte[] pBlockData )
        {
            if ( pBlockData.Length != 16 )
            {
                return new Status_t ( Error_CompCode.AL_MFC, Error_Param.INVALID_PARAMETER );
            }
            return phalMfc_Write ( m_pDataParams, bBlockNo, pBlockData );
        }

        /// <summary>
        /// Write the value to MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for writing the value.</param>
        /// <param name="dwValue">The value to be written to MIFARE PICC.</param>
        /// <param name="bAddrData">The address to written to MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WriteValueDw ( byte bBlockNo, int dwValue, byte bAddrData )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfc_WriteValue ( m_pDataParams, bBlockNo, pValue, bAddrData );
        }

        /// <summary>
        /// Write the value to MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for writing the value.</param>
        /// <param name="pValue">The value to be written to MIFARE PICC.</param>
        /// <param name="bAddrData">The address to written to MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WriteValue ( byte bBlockNo, byte[] pValue, byte bAddrData )
        {
            return phalMfc_WriteValue ( m_pDataParams, bBlockNo, pValue, bAddrData );
        }

        /// <summary>
        /// Increments the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for incrementing the value.</param>
        /// <param name="dwValue">The value to be incremented in a MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IncrementDw ( byte bBlockNo, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfc_Increment ( m_pDataParams, bBlockNo, pValue );
        }

        /// <summary>
        /// Increments the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for incrementing the value.</param>
        /// <param name="pValue">The value to be incremented in a MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Increment ( byte bBlockNo, byte[] pValue )
        {
            return phalMfc_Increment ( m_pDataParams, bBlockNo, pValue );
        }

        /// <summary>
        /// Decrements the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for decrementing the value.</param>
        /// <param name="dwValue">The value to be Decremented in a MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecrementDw ( byte bBlockNo, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfc_Decrement ( m_pDataParams, bBlockNo, pValue );
        }

        /// <summary>
        /// Decrements the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for decrementing the value.</param>
        /// <param name="pValue">The value to be Decremented in a MIFARE PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Decrement ( byte bBlockNo, byte[] pValue )
        {
            return phalMfc_Decrement ( m_pDataParams, bBlockNo, pValue );
        }

        /// <summary>
        /// Transfers the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for transferring the value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Transfer ( byte bBlockNo )
        {
            return phalMfc_Transfer ( m_pDataParams, bBlockNo );
        }

        /// <summary>
        /// Restores the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">The block number to be used for restoring the value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Restore ( byte bBlockNo )
        {
            return phalMfc_Restore ( m_pDataParams, bBlockNo );
        }

        /// <summary>
        /// Increments and Transfers the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bSrcBlockNo">The source block number used for incrementing the value.</param>
        /// <param name="bDstBlockNo">The destination block number used for transferring the value.</param>
        /// <param name="dwValue">The value to be incremented and transferred.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IncrementTransferDw ( byte bSrcBlockNo, byte bDstBlockNo, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfc_IncrementTransfer ( m_pDataParams, bSrcBlockNo, bDstBlockNo, pValue );
        }

        /// <summary>
        /// Increments and Transfers the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bSrcBlockNo">The source block number used for incrementing the value.</param>
        /// <param name="bDstBlockNo">The destination block number used for transferring the value.</param>
        /// <param name="pValue">The value to be incremented and transferred.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IncrementTransfer ( byte bSrcBlockNo, byte bDstBlockNo, byte[] pValue )
        {
            return phalMfc_IncrementTransfer ( m_pDataParams, bSrcBlockNo, bDstBlockNo, pValue );
        }

        /// <summary>
        /// Decrements and Transfers the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bSrcBlockNo">The source block number used for Decrementing the value.</param>
        /// <param name="bDstBlockNo">The destination block number used for transferring the value.</param>
        /// <param name="dwValue">The value to be decremented and transferred.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecrementTransferDw ( byte bSrcBlockNo, byte bDstBlockNo, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfc_DecrementTransfer ( m_pDataParams, bSrcBlockNo, bDstBlockNo, pValue );
        }

        /// <summary>
        /// Decrements and Transfers the value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bSrcBlockNo">The source block number used for Decrementing the value.</param>
        /// <param name="bDstBlockNo">The destination block number used for transferring the value.</param>
        /// <param name="pValue">The value to be decremented and transferred.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecrementTransfer ( byte bSrcBlockNo, byte bDstBlockNo, byte[] pValue )
        {
            return phalMfc_DecrementTransfer ( m_pDataParams, bSrcBlockNo, bDstBlockNo, pValue );
        }

        /// <summary>
        /// Restores the transferred value in a MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bSrcBlockNo">The source block number used for restoring the value.</param>
        /// <param name="bDstBlockNo">The destination block number used for restoring the value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t RestoreTransfer ( byte bSrcBlockNo, byte bDstBlockNo )
        {
            return phalMfc_RestoreTransfer ( m_pDataParams, bSrcBlockNo, bDstBlockNo );
        }

        /// <summary>
        /// Personalizes the UID based om the UID type provided.
        /// </summary>
        ///
        /// <param name="bUidType">One of the below types for personalizing the UID.
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF0
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF1
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF2
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF3
        ///							</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t PersonalizeUid ( byte bUidType )
        {
            return phalMfc_PersonalizeUid ( m_pDataParams, bUidType );
        }

        /// <summary>
        /// Personalizes the UID based om the UID type provided.
        /// </summary>
        ///
        /// <param name="eUidType">One of the below types for personalizing the UID.
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF0
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF1
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF2
        ///								NxpRdLibNet.alMfc.PersonalizeUID.UIDF3
        ///							</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t PersonalizeUid ( PersonalizeUID eUidType )
        {
            return PersonalizeUid ( ( byte ) eUidType );
        }

        /// <summary>
        /// Perform MIFARE(R) GET VERSION usage command sequence with MIFARE Picc. GET_VERSION command
        /// is used to retrieve information about the NTAG family, the product version, storage size
        /// and other product data required to identify the specific NTAG-I2C.
        /// </summary>
        ///
        /// <param name="pVersion">The version returned by PICC</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetVersion ( out byte[] pVersion )
        {
            pVersion = new byte[8];
            Status_t status = phalMfc_GetVersion ( m_pDataParams, pVersion );
            if ( !status.Equals ( Error_Gen.SUCCESS ) )
            {
                pVersion = new byte[0];
            }
            return status;
        }
        #endregion

        protected byte[] ConvertIntToByteArray ( int dwValue )
        {
            byte[] pValue = new byte[4];
            pValue[0] = ( byte ) dwValue;
            pValue[1] = ( byte ) ( dwValue >> 8 );
            pValue[2] = ( byte ) ( dwValue >> 16 );
            pValue[3] = ( byte ) ( dwValue >> 24 );
            return pValue;
        }
        protected int ConvertByteArrayToInt ( byte[] pValue )
        {
            int dwValue = 0;
            dwValue = pValue[0];
            dwValue += pValue[1] << 8;
            dwValue += pValue[2] << 16;
            dwValue += pValue[3] << 24;
            return dwValue;
        }

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        #endregion
    }
    #endregion

    #region Sw
    /// <summary>
    /// Software component class for Mifare Classic Application Layer. Here the software component
    /// initialization interface is available and also the interface related to Mifare Classic commands
    /// are available.
    /// </summary>
    public class Sw : alMfc.Generic
    {
        #region DATA_STRUCTURE
        /// <summary>
        /// Software component's structure.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this HAL component, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the parameter structure of the palMifare layer.
            /// </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>
            /// Pointer to the parameter structure of the KeyStore component.
            /// </summary>
            public IntPtr pKeyStoreDataParams;
        };
        #endregion

        #region DLLIMPORTS
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pIMifareDataParams,
            IntPtr pKeyStoreDataParams );
        #endregion

        #region INIT
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in software mode.
        /// </summary>
        ///
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare component.</param>
        /// <param name="pKeyStore">Pointer to the parameter structure of the KeyStore component.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( palMifare.Generic pMifare, KeyStore.Generic pKeyStore )
        {
            return phalMfc_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in software mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare component.</param>
        /// <param name="pKeyStore">Pointer to the parameter structure of the KeyStore component.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, palMifare.Generic pMifare, KeyStore.Generic pKeyStore )
        {
            return phalMfc_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams,
                ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams );
        }
#endif

        #endregion

        #region MEMORY_MAPPING
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region SamAV2_X
    public class SamAV2_X : alMfc.Generic
    {
        #region DATA_STRUCTURE
        /// <summary>
        /// SamAV2 X component's structure.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this HAL component, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the parameter structure of the underlying HAL layer.
            /// </summary>
            public IntPtr pHalDataParams;

            /// <summary>
            /// Pointer to the parameter structure of the palMifare layer.
            /// </summary>
            public IntPtr pPalMifareDataParams;
        };
        #endregion

        #region DLLIMPORTS
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams,
            IntPtr pIMifareDataParams );
        #endregion

        #region INIT
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in SamAV2 X mode.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( Hal.SamAV2 pHal, palMifare.SamAV2_X pMifare )
        {
            return phalMfc_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams, ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in SamAV2 X mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( int wDataParamSize, Hal.SamAV2 pHal, palMifare.SamAV2_X pMifare )
        {
            return phalMfc_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
                ( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams, ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams );
        }
#endif
        #endregion

        #region MEMORY_MAPPING
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X
    public class SamAV3_X : alMfc.Generic
    {
        #region DATA_STRUCTURE
        /// <summary>
        /// SamAV3 X component's structure.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this HAL component, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the parameter structure of the underlying HAL layer.
            /// </summary>
            public IntPtr pHalDataParams;

            /// <summary>
            /// Pointer to the parameter structure of the palMifare layer.
            /// </summary>
            public IntPtr pPalMifareDataParams;
        };
        #endregion

        #region DLLIMPORTS
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfc_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams,
            IntPtr pIMifareDataParams );
        #endregion

        #region INIT
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in SamAV3 X mode.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( Hal.SamAV3 pHal, palMifare.SamAV3_X pMifare )
        {
            return phalMfc_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams, ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Interface to initialize the Mifare Classic AL in SamAV3 X mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHal, palMifare.SamAV3_X pMifare )
        {
            return phalMfc_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
                ( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams, ( pMifare == null ) ? IntPtr.Zero : pMifare.m_pDataParams );
        }
#endif
        #endregion

        #region MEMORY_MAPPING
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif
}
