/*
 * Copyright 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfdfEVx
{
    #region Enumerations
    #region Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary>
        /// Field to indicate No Changes error.
        /// No changes done to backup files, CommitTransaction / AbortTransaction not necessary. or
        /// Format of the command is not correct (e.g. too many or too few bytes).
        /// </summary>
        PHAL_MFDFEVX_NO_CHANGES = CustomCodes.ERROR_BEGIN,

        /// <summary>
        /// Field to indicate Out Of Memory error.
        /// Insufficient NV-Memory to complete command.
        /// </summary>
        PHAL_MFDFEVX_ERR_OUT_OF_EEPROM_ERROR,

        /// <summary>
        /// Field to indicate No Suck Key error.
        /// Invalid key number specified.
        /// </summary>
        PHAL_MFDFEVX_ERR_NO_SUCH_KEY,

        /// <summary>
        /// Field to indicate Permission Denied error.
        /// Current configuration / oStatus does not allow the requested command.
        /// </summary>
        PHAL_MFDFEVX_ERR_PERMISSION_DENIED,

        /// <summary>
        /// Field to indicate Application Not Found error.
        /// Requested AID not present on PICC.
        /// </summary>
        PHAL_MFDFEVX_ERR_APPLICATION_NOT_FOUND,

        /// <summary>
        /// Field to indicate Boundary Error error.
        /// Attempt to read/write data from/to beyond the files/records limits. Attempt to
        /// exceed the limits of a value file.
        /// </summary>
        PHAL_MFDFEVX_ERR_BOUNDARY_ERROR,

        /// <summary>
        /// Field to indicate Command Aborted error.
        /// Previous Command was not fully completed. Not all Frames were requested or provided by the PCD.
        /// </summary>
        PHAL_MFDFEVX_ERR_COMMAND_ABORTED,

        /// <summary>
        /// Field to indicate Count error.
        /// Number of application limited to 28, no additional CreateApplication possible.
        /// </summary>
        PHAL_MFDFEVX_ERR_COUNT,

        /// <summary>
        /// Field to indicate Duplicate error.
        /// Creation of file/application failed because file/application with same number already exists.
        /// </summary>
        PHAL_MFDFEVX_ERR_DUPLICATE,

        /// <summary>
        /// Field to indicate File Not Found error.
        /// Specified file number does not exist.
        /// </summary>
        PHAL_MFDFEVX_ERR_FILE_NOT_FOUND,

        /// <summary>
        /// Field to indicate Integrity error.
        /// CRC or MAC does not match data. Padding bytes not valid.
        /// </summary>
        PHAL_MFDFEVX_ERR_PICC_CRYPTO,

        /// <summary>
        /// Field to indicate Parameter error.
        /// Value of the parameter(s) invalid.
        /// </summary>
        PHAL_MFDFEVX_ERR_PARAMETER_ERROR,

        /// <summary>
        /// Field to indicate DESFire General error. Call GetConfig interface
        /// with option as <see cref="NxpRdLibNet.alMfdfEVx.Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///		<item><description>Application Integrity Error (0xA1).</description></item>
        ///		<item><description>PICC Integrity Error (0xC1).</description></item>
        ///		<item><description>EEPROM Error (0xEE).</description></item>
        ///		<item><description>PICC Disabled Error (0xCD).</description></item>
        ///		<item><description>Illegal Command Code (0x1C).</description></item>
        ///	</list>
        /// </summary>
        PHAL_MFDFEVX_ERR_DF_GEN_ERROR,

        /// <summary>
        /// Field to indicate DESFire ISO7816 error(s).Call GetConfig interface
        /// with option as <see cref="NxpRdLibNet.alMfdfEVx.Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///		<item><description>Wrong Length (0x6700).</description></item>
        ///		<item><description>Wrong Le (0x6C00).</description></item>
        ///		<item><description>File not found (0x6A82).</description></item>
        ///		<item><description>Wrong Params (0x6A86).</description></item>
        ///		<item><description>Wrong LC (0x6A87).</description></item>
        ///		<item><description>No Precise Diagnostics (0x6F00).</description></item>
        ///		<item><description>EOF Reached (0x6282).</description></item>
        ///		<item><description>File Access (0x6982).</description></item>
        ///		<item><description>File Empty (0x6985).</description></item>
        ///		<item><description>Memory Failure (0x6581).</description></item>
        ///		<item><description>Incorrect Params (0x6B00).</description></item>
        ///		<item><description>Wrong CLA (0x6E00).</description></item>
        ///		<item><description>Unsupported INS (0x6D00).</description></item>
        ///		<item><description>Limited Functionality INS (0x6283).</description></item>
        ///	</list>
        /// </summary>
        PHAL_MFDFEVX_ERR_DF_7816_GEN_ERROR,

        /// <summary>
        /// Field to indicate Command Invalid error.
        /// Command is received in a state where this command is not supported, or a totally
        /// unknown command is received.
        /// </summary>
        PHAL_MFDFEVX_ERR_CMD_INVALID,

        /// <summary>
        /// Field to indicate Not Supported error.
        /// </summary>
        PHAL_MFDFEVX_ERR_NOT_SUPPORTED,

        /// <summary>
        /// Field to indicate Limited Operation error.
        /// Successful operation with limited functionality.
        /// </summary>
        PHAL_MFDFEVX_ERR_OPERATION_OK_LIM = ( CustomCodes.ERROR_BEGIN + 27 ),

        /// <summary>
        /// Field to indicate Command Overflow error.
        /// Too many commands in the session or transaction.
        /// </summary>
        PHAL_MFDFEVX_ERR_CMD_OVERFLOW = ( CustomCodes.ERROR_BEGIN + 28 ),

        /// <summary>
        /// Field to indicate General Failure error.
        /// Failure in the operation of the PD.
        /// </summary>
        PHAL_MFDFEVX_ERR_GEN_FAILURE = ( CustomCodes.ERROR_BEGIN + 29 ),

        /// <summary>
        /// Field to indicate Invalid Block number error.
        /// Invalid Block number: not existing in the implementation or not valid to target with this command.
        /// </summary>
        PHAL_MFDFEVX_ERR_BNR = ( CustomCodes.ERROR_BEGIN + 30 ),

        /// <summary>
        /// Field to indicate Format error.
        /// Format of the command is not correct (e.g. too many or too few bytes).
        /// </summary>
        PHAL_MFDFEVX_ERR_FORMAT = ( CustomCodes.ERROR_BEGIN + 31 )
    };
    #endregion Error Codes

    #region Communication Options
    /// <summary>
    /// The communication mode to be used.
    /// </summary>
    public enum CommOption : byte
    {
        /// <summary> Plain mode of communication.</summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x20,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x10,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x30,
    }
    #endregion Communication Options

    #region Auth Types
    /// <summary>
    /// Type of EV2 Authentication to perform.
    /// </summary>
    public enum AuthType : byte
    {
        /// <summary> Authentication type as EV2 NonFirst. </summary>
        EV2_NON_FIRST_AUTH,

        /// <summary> Authentication type as EV2 First. </summary>
        EV2_FIRST_AUTH
    }
    #endregion Auth Types

    #region Diversification Options
    /// <summary>
    /// Diversification method to be used for key diversification.
    /// </summary>
    public enum DivOption_Auth : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> Encryption based method of diversification. </summary>
        ENCR = 0x0000,

        /// <summary> CMAC based method of diversification. </summary>
        CMAC = 0x0001,

        /// <summary> Encryption based method, full key diversification.</summary>
        DES2K3_FULL = 0x0000,

        /// <summary> Encryption based method, half key diversification.</summary>
        DES2K3_HALF = 0x8000
    }

    /// <summary>
    /// Diversification method to be used for key diversification.
    /// </summary>
    public enum DivOption_ChangeKey : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> Bit 1. Indicating diversification of new key required. </summary>
        NEW_KEY = 0x0002,

        /// <summary> Bit 2 indicating old key was diversified. </summary>
        OLD_KEY = 0x0004,

        /// <summary> Bit 3 indicating new key diversification using one rnd. Default is two rounds. </summary>
        NEW_KEY_ONERND = 0x0008,

        /// <summary> Bit 4 indicating old key diversification using one rnd. Default is two rounds. </summary>
        OLD_KEY_ONERND = 0x0010,

        /// <summary> Bit 5 indicating key diversification method based on CMAC. Default is Encryption method. </summary>
        CMAC = 0x0020
    }

    /// <summary>
    /// Options to diversify the key.
    /// </summary>
    public enum DivOption : ushort
    {
        /// <summary> Option to disable key diversification. </summary>
        DIVERSIFICATION_OFF = 0xFFFF,

        /// <summary> Option to enable key diversification. </summary>
        DIVERSIFICATION_ON = 0x0000,

        /// <summary> CMAC based method of diversification. </summary>
        CMAC = 0x0001,
    }
    #endregion Auth Types

    #region Secondary Application
    /// <summary>
    /// Option to set the Seconday Application Indicator bit for most of the commands.
    /// </summary>
    public enum TargetApp : byte
    {
        /// <summary> Target primary application.</summary>
        PRIMARY = 0x00,

        /// <summary> Target secondary application.</summary>
        SECONDARY = 0x80
    }
    #endregion

    #region SetConfiguration Options
    /// <summary>
    /// Options for SetConfiguration command
    /// </summary>
    public enum SetConfig : byte
    {
        /// <summary> Option 0 for updating the PICC Configuration. </summary>
        OPTION_0 = 0x00,

        /// <summary> Option 1 for updating the Default Keys. </summary>
        OPTION_1 = 0x01,

        /// <summary> Option 2 for updating the ATS. </summary>
        OPTION_2 = 0x02,

        /// <summary> Option 3 for updating the SAK. </summary>
        OPTION_3 = 0x03,

        /// <summary> Option 4 for updating the Secure Messaging. </summary>
        OPTION_4 = 0x04,

        /// <summary> Option 5 for updating the Capibility Data. </summary>
        OPTION_5 = 0x05,

        /// <summary> Option 6 for updating the VC Installation Identifier. </summary>
        OPTION_6 = 0x06,

        /// <summary> Option C for updating the ATQA information. </summary>
        OPTION_12 = 0x0C,
    }
    #endregion SetConfiguration Options

    #region GetCardUID Options
    /// <summary>
    /// Options for GetCardUID command.
    /// </summary>
    public enum NUID : byte
    {
        /// <summary> Option for no retun of 4 byte NUID from PICC. </summary>
        NOT_RETURNED = 0x00,

        /// <summary> Option for retun of 4 byte NUID from PICC. </summary>
        RETURNED = 0x01
    }
    #endregion GetCardUID Options

    #region Key
    /// <summary>
    /// Option to be used for InitializeKeySet and CreateTransactionMAC File command.
    /// </summary>
    public enum Key : byte
    {
        /// <summary> Option for 2 Key Triple Des key type. </summary>
        DES2K3 = 0x00,

        /// <summary> Option for 3 Key Triple Des key type. </summary>
        DES3K3 = 0x01,

        /// <summary> Option for AES 128 key type. </summary>
        AES128 = 0x02
    }
    #endregion Key Type

    #region GetKeyVersion
    /// <summary>
    /// Option to be used for GetKeyVersion command.
    /// </summary>
    public enum GetKeyVersion : byte
    {
        /// <summary> Option for KeySetNo available in the command. </summary>
        KEYSETNO_PRESENT = 0x01,

        /// <summary> Option for retrieving the key set version. </summary>
        RETRIVE_KEY_SET_VERSION = 0x02
    }
    #endregion GetKeyVersion

    #region ISO
    /// <summary>
    /// Options for application / file creation commands command.
    /// </summary>
    public enum ISO : byte
    {
        /// <summary> Option to indicate no ISO File ID or ISODFName are present. </summary>
        FILE_INFO_NOT_AVAILABLE = 0x00,

        /// <summary> Option to indicate the presence of ISO FileID. </summary>
        FILE_ID_AVAILABLE = 0x01,

        /// <summary> Option to indicate the presence of ISO DFName. </summary>
        DF_NAME_AVAILABLE = 0x02,

        /// <summary> Option to indicate the presence of both ISO FileID and ISO DFName. </summary>
        FILE_ID_DF_NAME_AVAILABLE = 0x03
    }
    #endregion ISO

    #region Select Application
    /// <summary>
    /// Option to represent the selection of secondary application.
    /// </summary>
    public enum SelectApplication : byte
    {
        /// <summary> Option for Primary application selection. </summary>
        PRIMARY = 0x00,

        /// <summary> Option for Secondary application selection. </summary>
        SECONDARY = 0x01
    }
    #endregion Select Application

    #region File Options
    /// <summary>
    /// The File Option to be used.
    /// </summary>
    public enum FileOption : byte
    {
        /// <summary> Plain mode of communication.</summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x02,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x01,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x03,

        /// <summary> MIFARE Classic mapping support enabled. </summary>
        MFC_MAPPING_ENABLED = 0x20,

        /// <summary> Secure Dynamic Messaging and Mirroring support enabled. </summary>
        SDM_MIRRORING_ENABLED = 0x40,

        /// <summary> Transaction MAC Counter Limit present. </summary>
        TMCLIMIT_PRESENT = 0x20,

        /// <summary> Additional Access Rigths enabled. </summary>
        ADDITIONAL_AR_PRESENT = 0x80
    }
    #endregion File Options

    #region Limited Credit Options
    /// <summary>
    /// Option for CreateValueFile file command.
    /// </summary>
    public enum ValueFile : byte
    {
        /// <summary> Bit 0 set to 1 to enable Limited credit.</summary>
        ENABLE_LIMITEDCREDIT = 0x01,

        /// <summary> Bit 1 set to 1 to enable free GetValue. </summary>
        ENABLE_FREE_GETVALUE = 0x02
    }
    #endregion Limited Credit Options

    #region CreateTMACFile Options
    /// <summary>
    /// Options to diversify the key.
    /// </summary>
    public enum CreateTMACFile : byte
    {
        /// <summary> Option to disable key diversification. </summary>
        DIV_OFF = 0x00,

        /// <summary> Option to enable key diversification. </summary>
        DIV_ON = 0x01,
    }
    #endregion

    #region Chaining Options
    /// <summary>
    /// Chaining format to be used for DataManagement commands.
    /// </summary>
    public enum Chaining : byte
    {
        /// <summary>
        /// Option to represent the application chaining format.
        /// Here AF will be available in command and response.
        /// </summary>
        APPLICATION_CHAINING = 0x00,

        /// <summary>
        /// Option to represent the ISO14443-4 chaining format.
        /// Here AF will not be available in command and response
        /// rather the chaining will be done using the ISO14443-4 protocol. </summary>
        ISO_CHAINING = 0x01,
    }
    #endregion

    #region CommitTransaction Options
    /// <summary>
    /// Options for CommitTransaction command.
    /// </summary>
    public enum TmcTmv : byte
    {
        /// <summary> Option byte is not exchanged to the PICC. </summary>
        TMC_TMV_NOT_RETURNED = 0x00,

        /// <summary> Option byte exchanged to PICC and represent return of TMC and TMV. </summary>
        TMC_TMV_RETURNED = 0x01,

        /// <summary> Option byte exchanged to PICC. This is to support the exchange of Option byte to PICC. </summary>
        INCLUDE_OPTION = 0x80,
    }
    #endregion CommitTransaction Options

    #region ISOSelect Options
    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum Selector : byte
    {
        /// <summary> Option to indicate Selection by 2 byte file Id. </summary>
        SELECT_MF_DF_EF_FILE_IDENTIFIER = 0x00,

        /// <summary> Option to indicate Selection by child DF. </summary>
        SELECT_CHILD_DF = 0x01,

        /// <summary> Option to indicate Select EF under current DF. Fid = EF id. </summary>
        SELECT_EF_CURRENT_DF = 0x02,

        /// <summary> Option to indicate Select parent DF of the current DF. </summary>
        SELECT_PARENT_DF_CURRENT_DF = 0x03,

        /// <summary> Option to indicate Selection by DF Name. DFName and len is then valid. </summary>
        SELECT_BY_DF_NAME = 0x04
    }

    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum FCI : byte
    {
        /// <summary> Option to indicate the return of FCI. </summary>
        RETURNED = 0x00,

        /// <summary> Option to indicate the no return of FCI. </summary>
        NOT_RETURNED = 0x0C
    }
    #endregion ISOSelect Options

    #region Configuration options
    /// <summary>
    /// Option to configure some special operations.
    /// </summary>
    public enum Config : ushort
    {
        /// <summary>
        /// Option for Getconfig/SetConfig to get/set additional info of a generic error or some some length exposed by interfaces.
        /// </summary>
        ADDITIONAL_INFO = 0x00A1,

        /// <summary>
        /// Option for GetConfig/SetConfig to get/set current status of command wrapping in ISO 7816-4 APDUs.
        /// </summary>
        WRAPPED_MODE = 0x00A2,

        /// <summary>
        /// Option for Getconfig/SetConfig to get/set current status of Short Length APDU wrapping in ISO 7816-4 APDUs
        /// </summary>
        SHORT_LENGTH_APDU = 0x00A3,

        /// <summary>
		/// Option for Getconfig/SetConfig to get/set current status of Short Length APDU wrapping in ISO 7816-4 APDUs
		/// </summary>
        RETURN_FABID = 0x00A4
    };
    #endregion Configuration options

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SDM options
    /// <summary>
    /// Options for SDM information.
    /// </summary>
    public enum SDMOption : ushort
    {
        /// <summary>
        /// Only VCUID is considred for SDM Session key calculation.
        /// </summary>
        VCUID_PRESENT = 0x00A1,

        /// <summary>
        /// Only RDCTR  is considred for SDM Session key calculation.
        /// </summary>
        RDCTR_PRESENT = 0x00A2
    };
    #endregion Configuration options

    #region DAMMAC options
    /// <summary>
    /// Options for DAMMAC information.
    /// </summary>
    public enum DAMMAC : byte
    {
        /// <summary>
        /// Option to indicate the DAMMAC generation is for <see cref="NxpRdLibNet.alMfdfEVx.Generic.CreateDelegatedApplication"/> command.
        /// </summary>
        CREATE_APPLICATION = 0x00,

        /// <summary>
        /// Option to indicate the DAMMAC generation is for <see cref="NxpRdLibNet.alMfdfEVx.Generic.DeleteApplication"/> command.
        /// </summary>
        DELETE_APPLICATION = 0x80
    };
    #endregion DAMMAC options
#endif
    #endregion Enumerations

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    [NxpRdLibAttribute ( MIFARE_Products.DESFIRE, Message = "DESFire Products that support only DES Authentication." )]
    [NxpRdLibAttribute ( MIFARE_Products.DESFIRE_EV1, Message = "DESFire Products that support DES, AES and ISO Authentication." )]
    [NxpRdLibAttribute ( MIFARE_Products.DESFIRE_EV2, Message = "DESFire Products that support DES, AES, ISO and EV2 Authentication." )]
    [NxpRdLibAttribute ( MIFARE_Products.DESFIRE_EV3, Message = "DESFire Products that support DES, AES, ISO and EV2 Authentication. Also Support MIFARE Classic Mapping and Secure Dynamic Messaging." )]
    public abstract class Generic
    {
        #region Constants

        public const byte READ_BLOCK_LENGTH = 16;
        public const byte WRITE_BLOCK_LENGTH = 4;
        public const byte COMPWRITE_BLOCK_LENGTH = 16;

        /// <summary> Length of Originality Signature. </summary>
        public const byte SIG_LENGTH = 56;

        /// <summary> Length of DES Key Block. </summary>
        public const byte DES_BLOCK_SIZE = 8;

        /// <summary> Length of DES Key type. </summary>
        public const byte DES_KEY_LENGTH = 16;
        #endregion

        #region DLL Imports
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Secure Messaging
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Authenticate ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_AuthenticateISO ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_AuthenticateAES ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_AuthenticateEv2 ( IntPtr pDataParams, byte bFirstAuth, ushort wOption, ushort wKeyNo,
            ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput, byte bDivLen, byte bLenPcdCapsIn, byte[] bPcdCapsIn, byte[] bPcdCapsOut,
            byte[] bPdCapsOut );
        #endregion
#endif

        #region Memory and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_FreeMem ( IntPtr pDataParams, byte[] pMemInfo );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Format ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SetConfiguration ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataSize );
#endif
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetVersion ( IntPtr pDataParams, byte[] pVerInfo );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetCardUID ( IntPtr pDataParams, byte bExchangeOption, byte bOption, byte[] pUid );
#endif
        #endregion

        #region Key Management
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ChangeKey ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo,
            ushort wNewKeyVer, byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ChangeKeyEv2 ( IntPtr pDataParams, ushort wOption, ushort wOldKeyNo, ushort wOldKeyVer,
            ushort wNewKeyNo, ushort wNewKeyVer, byte bKeySetNo, byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_InitializeKeySet ( IntPtr pDataParams, byte bKeySetNo, byte bKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_FinalizeKeySet ( IntPtr pDataParams, byte bKeySetNo, byte bKeySetVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_RollKeySet ( IntPtr pDataParams, byte bKeySetNumber );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetKeySettings ( IntPtr pDataParams, byte[] pResponse, ref byte pRespLen );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ChangeKeySettings ( IntPtr pDataParams, byte bKeySettings );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetKeyVersion ( IntPtr pDataParams, byte bKeyNo, byte bKeySetNo, byte[] pResponse,
            ref byte bRxLen );
#endif
        #endregion

        #region Application Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateApplication ( IntPtr pDataParams, byte bOption, byte[] pAid, byte bKeySettings1,
            byte bKeySettings2, byte bKeySettings3, byte[] pKeySetValues, byte[] pISOFileId, byte[] pISODFName, byte bISODFNameLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_DeleteApplication ( IntPtr pDataParams, byte[] pAid, byte[] pDAMMAC, byte bDAMMAC_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateDelegatedApplication ( IntPtr pDataParams, byte bOption, byte[] pAid, byte[] pDamParams,
            byte bKeySettings1, byte bKeySettings2, byte bKeySettings3, byte[] bKeySetValues, byte[] pISOFileId, byte[] pISODFName,
            byte bISODFNameLen, byte[] pEncK, byte[] pDAMMAC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SelectApplication ( IntPtr pDataParams, byte bOption, byte[] pAid, byte[] pAid2 );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetApplicationIDs ( IntPtr pDataParams, byte bOption, ref IntPtr ppAidBuff, ref byte pNumAIDs );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetDFNames ( IntPtr pDataParams, byte bOption, byte[] pDFBuffer, ref byte bSize );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetDelegatedInfo ( IntPtr pDataParams, byte[] pDAMSlot, byte[] bDamSlotVer, byte[] pQuotaLimit,
            byte[] pFreeBlocks, byte[] pAid );
        #endregion

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateStdDataFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pISOFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pFileSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateBackupDataFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pISOFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pFileSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateValueFile ( IntPtr pDataParams, byte bFileNo, byte bCommSett, byte[] bAccessRights,
            byte[] bLowerLmit, byte[] bUpperLmit, byte[] bValue, byte bLimitedCredit );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateLinearRecordFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pIsoFileId,
            byte bCommSett, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateCyclicRecordFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pIsoFileId,
            byte bCommSett, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateTransactionMacFile ( IntPtr pDataParams, ushort wOption, byte bFileNo, byte bCommSett,
            byte[] pAccessRights, ushort wKeyNo, byte bKeyType, byte[] bTMKey, byte bTMKeyVer, byte[] pDivInput, byte bDivInputLength );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_DeleteFile ( IntPtr pDataParams, byte bFileNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetFileIDs ( IntPtr pDataParams, byte[] pFid, ref byte bNumFIDs );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetISOFileIDs ( IntPtr pDataParams, byte[] pFidBuffer, ref byte bNumFIDs );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetFileSettings ( IntPtr pDataParams, byte bFileNo, byte[] pFSBuffer, ref byte bBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetFileCounters ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pFSBuffer,
            ref byte bBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ChangeFileSettings ( IntPtr pDataParams, byte bOption, byte bFileNo, byte bFileOption,
            byte[] pAccessRights, byte bAddInfoLen, byte[] pAddInfo );
        #endregion

        #region Data Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ReadData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset,
            byte[] pLength, ref IntPtr ppRxdata, ref ushort pRxdataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_WriteData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset,
            byte[] pTxData, byte[] pTxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetValue ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Credit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Debit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_LimitedCredit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ReadRecords ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileId, byte[] pRecNo,
            byte[] pRecCount, byte[] pRecSize, ref IntPtr ppRxdata, ref ushort pRxdataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_WriteRecord ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileNo, byte[] pOffset,
            byte[] pData, byte[] pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_UpdateRecord ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileNo, byte[] pRecNo,
            byte[] pOffset, byte[] pData, byte[] pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ClearRecordFile ( IntPtr pDataParams, byte bFileNo );
        #endregion

        #region Transaction Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CommitTransaction ( IntPtr pDataParams, byte bOption, byte[] bTMC, byte[] bTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_AbortTransaction ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CommitReaderID ( IntPtr pDataParams, byte[] bTMRI, byte[] bEncTMRI );
        #endregion

        #region ISO7816-4
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoSelectFile ( IntPtr pDataParams, byte bOption, byte bSelector, byte[] pFid, byte[] pDFname,
            byte bDFnameLen, byte bExtendedLenApdu, ref IntPtr ppFCI, ref ushort pwFCILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoReadBinary ( IntPtr pDataParams, ushort wOption, byte bOffset, byte bSfid, UInt32 dwBytesToRead,
            byte bExtendedLenApdu, ref IntPtr pRxBuffer, ref UInt32 pBytesRead );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoUpdateBinary ( IntPtr pDataParams, byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData,
            UInt32 dwDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoReadRecords ( IntPtr pDataParams, ushort wOption, byte bRecNo, byte bReadAllFromP1, byte bSfid,
            UInt32 dwBytesToRead, byte bExtendedLenApdu, ref IntPtr ppRxBuffer, ref UInt32 pBytesRead );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoAppendRecord ( IntPtr pDataParams, byte bSfid, byte[] pData, UInt32 dwDataLen, byte bExtendedLenApdu );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoUpdateRecord ( IntPtr pDataParams, byte bIns, byte bRecNo, byte bSfid, byte bRefCtrl, byte[] pData,
            byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoGetChallenge ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVer, byte bExtendedLenApdu,
            UInt32 dwLe, byte[] pRPICC1 );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoExternalAuthenticate ( IntPtr pDataParams, byte[] pInput, byte bInputLen, byte bExtendedLenApdu,
            byte[] pDataOut, ref byte pOutLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoInternalAuthenticate ( IntPtr pDataParams, byte[] pInput, byte bInputLen, byte bExtendedLenApdu,
            byte[] pDataOut, ref byte pOutLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_IsoAuthenticate ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte bIsPICCkey );
#endif
        #endregion

        #region Originality Check
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ReadSign ( IntPtr pDataParams, byte bAddr, ref IntPtr pSignature );
        #endregion

        #region MIFARE Classic
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CreateMFCMapping ( IntPtr pDataParams, byte bComOption, byte bFileNo, byte bFileOption,
            byte[] pMFCBlockList, byte bMFCBlocksLen, byte bRestoreSource, byte[] pMFCLicense, byte bMFCLicenseLen, byte[] pMFCLicenseMAC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_RestoreTransfer ( IntPtr pDataParams, byte bCommOption, byte bTargetFileNo, byte bSourceFileNo );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_RestrictMFCUpdate ( IntPtr pDataParams, byte bOption,
            byte[] pMFCConfig, byte bMFCConfigLen, byte[] pMFCLicense, byte bMFCLicenseLen, byte[] pMFCLicenseMAC );
#endif
        #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Post Delivery Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_AuthenticatePDC ( IntPtr pDataParams, byte bRfu, byte bKeyNoCard, ushort wKeyNo,
            ushort wKeyVer, byte bUpgradeInfo );
        #endregion
#endif

        #region Miscellaneous
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ResetAuthentication ( IntPtr pDataParams );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CalculateTMV ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pTMI,
            uint dwTMILen, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_DecryptReaderID ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pEncTMRI,
            byte[] pTMRIPrev );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GenerateDAMEncKey ( IntPtr pDataParams, ushort wKeyNoDAMEnc, ushort wKeyVerDAMEnc,
            ushort wKeyNoAppDAMDefault, ushort wKeyVerAppDAMDefault, byte bAppDAMDefaultKeyVer, byte[] pDAMEncKey );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GenerateDAMMAC ( IntPtr pDataParams, byte bOption, ushort wKeyNoDAMMAC, ushort wKeyVerDAMMAC,
            byte[] pAid, byte[] pDamParams, byte bKeySettings1, byte bKeySettings2, byte bKeySettings3, byte[] pKeySetValues, byte[] pISOFileId,
            byte[] pISODFName, byte bISODFNameLen, byte[] pEncK, byte[] pDAMMAC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_GenerateDAMMACSetConfig ( IntPtr pDataParams, ushort wKeyNoDAMMAC, ushort wKeyVerDAMMAC,
            ushort wOldDFNameLen, byte[] pOldISODFName, ushort wNewDFNameLen, byte[] pNewISODFName, byte[] pDAMMAC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_ComputeMFCLicenseMAC ( IntPtr pDataParams, ushort wOption, ushort wMFCLicenseMACKeyNo, ushort wMFCLicenseMACKeyVer,
            byte[] pInput, ushort wInputLen, byte[] pDivInput, byte bDivInputLen, byte[] pMFCLicenseMAC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_DecryptSDMENCFileData ( IntPtr pDataParams, ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pEncdata, ushort wInDataLen, byte[] pPlainData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_DecryptSDMPICCData ( IntPtr pDataParams, ushort wEncKeyNo, ushort wEncKeyVer, byte[] pEncdata, ushort wInDataLen,
            byte[] pPlainData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_CalculateMACSDM ( IntPtr pDataParams, byte bSdmOption, ushort wSDMMacKeyNo, ushort wSDMMacKeyVer,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pIndata, ushort wInDataLen, byte[] pRespMac );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SetVCAParams ( IntPtr pDataParams, IntPtr pAlVCADataParams );
        #endregion
        #endregion

        #region Wrapper Functions
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Secure Messaging
        /// <summary>
        /// Performs an Authentication with the specified key number. This command supports the backward
        /// compatiable D40 authentication. The command can be used with DES and 2K3DES keys and performs
        /// DESFire4 native authentication.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR | NxpRdLibNet.alMfdfEVx.DivOption_Auth.DES2K3_FULL
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR | NxpRdLibNet.alMfdfEVx.DivOption_Auth.DES2K3_HALF
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in the key store of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Only DES (56 bits) / 2TDEA (112 bits) keys should be used for Authentication. After Authentication, D40 Backward compatible Secure Messaging is used.",
                Message = "D40 Backward compatibility Authentication."
            )
        ]
        public Status_t Authenticate ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfEVx_Authenticate ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Performs an DES Authentication in ISO CBS send mode. The command can be used with DES,3DES and 3K3DES keys.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR | NxpRdLibNet.alMfdfEVx.DivOption_Auth.DES2K3_FULL
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR | NxpRdLibNet.alMfdfEVx.DivOption_Auth.DES2K3_HALF
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in the key store of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Only 2TDEA (112 bits) / 3TDEA (168 bit) keys should be used for Authentication. After Authentication, EV1 Backward compatible Secure Messaging is used.",
                Message = "EV1 Backward compatibility Authentication."
            )
        ]
        public Status_t AuthenticateISO ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfEVx_AuthenticateISO ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Performs an AES Authentication. The command should be used with AES128 keys.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in the key store of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Only AES (128 bits) key should be used for Authentication. After Authentication, EV1 Backward compatible Secure Messaging is used.",
                Message = "EV1 Backward compatibility Authentication."
            )
        ]
        public Status_t AuthenticateAES ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfEVx_AuthenticateAES ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Performs an EV2 First or Non First Authentication depending upon bFirstAuth Parameter. This will
        /// be using the AES128 keys and will generate and verify the contents based on generic AES algorithm.
        /// </summary>
        ///
        /// <param name="bFirstAuth">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.AuthType.EV2_FIRST_AUTH
        ///							NxpRdLibNet.alMfdfEVx.AuthType.EV2_NON_FIRST_AUTH</param>
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.ENCR
        ///							NxpRdLibNet.alMfdfEVx.DivOption_Auth.CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in the key store of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        /// <param name="bPcdCapsIn">PCD Capabilities. Upto 6 bytes.</param>
        /// <param name="bPcdCapsOut">PCD Capabilities. 6 bytes.</param>
        /// <param name="bPdCapsOut">PD Capabilities. 6 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Only AES (128 bits) key should be used for Authentication. After Authentication, EV2 Secure Messaging is used.",
                Message = "The Secure Messaging depends on FirstAuth parameter. " +
                          "EV2_FIRST_AUTH: The Authentication is intended to be first in transaction. " +
                          "EV2_NON_FIRST_AUTH: The Authentication is intended for any subsequent authentication after Authenticate First."
            )
        ]
        public Status_t AuthenticateEv2 ( byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard,
            byte[] pDivInput, byte[] bPcdCapsIn, out byte[] bPcdCapsOut, out byte[] bPdCapsOut )
        {
            bPcdCapsOut = new byte[6];
            bPdCapsOut = new byte[6];

            return phalMfdfEVx_AuthenticateEv2 ( m_pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), ( byte ) ( ( bPcdCapsIn == null ) ? 0 : bPcdCapsIn.Length ),
                bPcdCapsIn, bPcdCapsOut, bPdCapsOut );
        }
        #endregion
#endif

        #region Memory and Configuration
        /// <summary>
        /// Returns free memory available on the PICC
        /// </summary>
        ///
        /// <param name="pMemInfo">Current free memory available. Will be of 3 bytes with LSB first.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the free memory available on the card."
            )
        ]
        public Status_t FreeMem ( out byte[] pMemInfo )
        {
            pMemInfo = new byte[3];
            return phalMfdfEVx_FreeMem ( m_pDataParams, pMemInfo );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Releases the PICC user memory
        /// </summary>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Deletes the Files / Applications and releases the memory for reuse."
            )
        ]
        public Status_t Format ()
        {
            return phalMfdfEVx_Format ( m_pDataParams );
        }

        /// <summary>
        /// Options for SetConfiguration command.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option.
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_0
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_1
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_2
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_3
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_4
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_5
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_6
        ///							NxpRdLibNet.alMfdfEVx.AuthType.SetConfig.OPTION_C</param>
        /// <param name="pData">Data for the option specified.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the free memory available on the card."
            )
        ]
        public Status_t SetConfiguration ( byte bOption, byte[] pData )
        {
            byte[] pDataTmp = new byte[pData.Length];
            pData.CopyTo ( pDataTmp, 0 );
            return phalMfdfEVx_SetConfiguration ( m_pDataParams, bOption, pDataTmp, ( byte ) pDataTmp.Length );
        }
#endif

        /// <summary>
        /// Returns manufacturing related data of the PICC.
        ///
        /// Note
        /// To get the Fabrication Identifier (FabID) as part of Part-3 response, perform <see cref="SetConfig">Return of FabID</see>
        /// configuration settings prior calling GetVersion method call.
        /// </summary>
        ///
        /// <param name="pVerInfo">The version information of the product.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "First 7 bytes will be Hardware related Information. Second 7 bytes will be Software related Information and last N bytes will be Product related information.\n" +
                            "To get the Fabrication Identifier (FabID) as part of Part-3 response, perform SetConfig with RETURN_FABID as configuration settings prior calling GetVersion method call.",
                Message = "Returns the Version information of the card. The information will contain Hardware, Software and Production related information."
            )
        ]
        public Status_t GetVersion ( out byte[] pVerInfo )
        {
            Status_t oStatus = null, oStatus_Tmp;
            byte[] aVersion = new byte[48];
            ushort wLength = 0, wVal = 0;

            pVerInfo = null;

            oStatus_Tmp = GetConfig ( 0x00A1, ref wVal );

            if ( ( oStatus = phalMfdfEVx_GetVersion ( m_pDataParams, aVersion ) ).Equals ( new Status_t () ) )
            {
                /* Do a Get Config of  ADDITIONAL_INFO to read the length(wLength) of the Version string */
                oStatus_Tmp = GetConfig ( 0x00A1, ref wLength );
                pVerInfo = new byte[wLength];

                /* Copy the version string only to the length retrieved */
                Array.Copy ( aVersion, pVerInfo, wLength );
            }

            oStatus_Tmp = SetConfig ( 0x00A1, wVal );

            return oStatus;
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Returns the Unique ID of the PICC.
        /// </summary>
        ///
        /// <param name="pUid">The complete UID of the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the Unique Identifier of the card."
            )
        ]
        public Status_t GetCardUID ( out byte[] pUid )
        {
            pUid = new byte[20];
            ushort bLength = 0;
            Status_t oStatus;
            oStatus = phalMfdfEVx_GetCardUID ( m_pDataParams, 0x00, 0x00, pUid );

            /* Get the Length of the UID */
            if ( oStatus == new Status_t () )
            {
                oStatus = GetConfig ( 0x00A1, ref bLength );
            }

            Array.Resize<byte> ( ref pUid, bLength );
            return oStatus;
        }

        /// <summary>
        /// Returns the Unique ID of the PICC.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.NUID.NOT_RETURNED
        ///							NxpRdLibNet.alMfdfEVx.NUID.RETURNED</param>
        /// <param name="pUid">The complete UID of the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the Unique Identifier of the card. Also returns the NonUnique UID if requested."
            )
        ]
        public Status_t GetCardUID ( byte bOption, out byte[] pUid )
        {
            pUid = new byte[20];
            ushort bLength = 0;
            Status_t oStatus;
            oStatus = phalMfdfEVx_GetCardUID ( m_pDataParams, 0x01, bOption, pUid );

            /* Get the length of the UID */
            if ( oStatus == new Status_t () )
            {
                oStatus = GetConfig ( 0x00A1, ref bLength );
            }

            Array.Resize<byte> ( ref pUid, bLength );
            return oStatus;
        }
#endif
        #endregion

        #region Key Management
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Changes any key on the PICC
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY_ONERND
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY_ONERND
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY |
        ///								NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY |
        ///								NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY_ONERND | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY_ONERND</param>
        /// <param name="wKeyNo">Old key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Old key version in keystore of software or SAM.</param>
        /// <param name="wNewKeyNo">New key number in keystore of software or SAM.</param>
        /// <param name="wNewKeyVer">New key version in keystore of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected AID, this command updates a key of the PICC or of an application Active Key Set (AKS)."
            )
        ]
        public Status_t ChangeKey ( ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfEVx_ChangeKey ( m_pDataParams, wOption, wKeyNo, wKeyVer, wNewKeyNo, wNewKeyVer,
                bKeyNoCard, pDivInput, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Changes any key present in keyset on the PICC
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NO_DIVERSIFICATION
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY_ONERND
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY_ONERND
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY |
        ///								NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.CMAC
        ///							NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY |
        ///								NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.NEW_KEY_ONERND | NxpRdLibNet.alMfdfEVx.DivOption_ChangeKey.OLD_KEY_ONERND</param>
        /// <param name="wKeyNo">Old key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Old key version in keystore of software or SAM.</param>
        /// <param name="wNewKeyNo">New key number in keystore of software or SAM.</param>
        /// <param name="wNewKeyVer">New key version in keystore of software or SAM.</param>
        /// <param name="bKeySetNo">Key set number within targeted application.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected AID, this command updates a key of the PICC or of one specified application keyset."
            )
        ]
        public Status_t ChangeKeyEv2 ( ushort wOption, ushort wOldKeyNo, ushort wOldKeyVer, ushort wNewKeyNo,
            ushort wNewKeyVer, byte bKeySetNo, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfEVx_ChangeKeyEv2 ( m_pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer,
                bKeySetNo, bKeyNoCard, pDivInput, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Initializes KeySet with bKeySetNo specified with type of the keyset bKeyType
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/>
        ///							to indicate secondary application indicator.</param>
        /// <param name="bKeyType">One of the below mentioned options.
        ///							NxpRdLibNet.alMfdfEVx.Key.DES2K3
        ///							NxpRdLibNet.alMfdfEVx.Key.DES3K3
        ///							NxpRdLibNet.alMfdfEVx.Key.AES128</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected application, initilizes the key set with specific index."
            )
        ]
        public Status_t InitializeKeySet ( byte bKeySetNo, byte bKeyType )
        {
            return phalMfdfEVx_InitializeKeySet ( m_pDataParams, bKeySetNo, bKeyType );
        }

        /// <summary>
        /// Finalizes KeySet targeted by bKeySetNo by setting the key set version to the bKeySetVersion
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/>
        ///							to indicate secondary application indicator.</param>
        /// <param name="bKeySetVersion">KeySet version of the key set to be finalized.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected application, finilize the key set with specified number."
            )
        ]
        public Status_t FinalizeKeySet ( byte bKeySetNo, byte bKeySetVersion )
        {
            return phalMfdfEVx_FinalizeKeySet ( m_pDataParams, bKeySetNo, bKeySetVersion );
        }

        /// <summary>
        /// RollKeySet changes the ActiveKeySet to key set currently targeted with bKeySetNumber
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/>
        ///							to indicate secondary application indicator.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected application, roll to the key set with specified number."
            )
        ]
        public Status_t RollKeySet ( byte bIndex )
        {
            return phalMfdfEVx_RollKeySet ( m_pDataParams, bIndex );
        }
#endif

        /// <summary>
        /// Gets information on the PICC and application master key settings.
        /// <remarks>In addition it returns the maximum number of keys which are configured for the selected application.</remarks>
        /// </summary>
        ///
        /// <param name="pResponse">Returns the key settings. Can be 2 or 3 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected AID, PICC Keysettings of PICC or the AppKeySettings of the (primary) application is returned. " +
                          "In addition it returns the number of keys which are configured for the selected application and if applicable the AppKeySetSettings."
            )
        ]
        public Status_t GetKeySettings ( out byte[] pResponse )
        {
            Status_t oStatus;
            pResponse = new byte[6];
            byte bRxLen = 0;

            oStatus = phalMfdfEVx_GetKeySettings ( m_pDataParams, pResponse, ref bRxLen );
            Array.Resize<byte> ( ref pResponse, bRxLen );

            return oStatus;
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Changes the master key settings on PICC and application level.
        /// </summary>
        ///
        /// <param name="bKeySettings">New key settings to be updated.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected AID, the command changes the PICCKeySetting of the PICC or the AppKeySettings of the application."
            )
        ]
        public Status_t ChangeKeySettings ( byte bKeySettings )
        {
            return phalMfdfEVx_ChangeKeySettings ( m_pDataParams, bKeySettings );
        }

        /// <summary>
        /// Reads out the current key version of any key stored on the PICC
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number of the targeted key. ORed with one of the below options
        ///							NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY
        ///							NxpRdLibNet.alMfdfEVx.GetKeyVersion.KEYSETNO_PRESENT
        ///							</param>
        /// <param name="bKeySetNo">Key set number. ORed with one of the below options
        ///							NxpRdLibNet.alMfdfEVx.GetKeyVersion.RETRIVE_KEY_SET_VERSION</param>
        /// <param name="pResponse">Key set versions of the selected application ordered by ascending
        ///							key set number, i.e. starting with the AKS.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Depending on the currently selected AID and given key number, key version of the target key and all key set versions of the selected application are returned."
            )
        ]
        public Status_t GetKeyVersion ( byte bKeyNo, byte bKeySetNo, out byte[] pResponse )
        {
            Status_t oStatus;
            pResponse = new byte[16];
            byte bRxLen = 0;

            oStatus = phalMfdfEVx_GetKeyVersion ( m_pDataParams, bKeyNo, bKeySetNo, pResponse, ref bRxLen );
            Array.Resize<byte> ( ref pResponse, bRxLen );

            return oStatus;
        }
#endif
        #endregion

        #region Application Management
        /// <summary>
        /// Creates new applications on the PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.DF_NAME_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_DF_NAME_AVAILABLE</param>
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        /// <param name="bKeySettings1">Application Key settings.</param>
        /// <param name="bKeySettings2">Additional key settings.</param>
        /// <param name="bKeySettings3">Additional key settings.</param>
        /// <param name="pKeySetValues">The Key set values for the application. Should as mentioned below.
        ///									Byte0 = AKS ver
        ///									Byte1 = Number of Keysets
        ///									Byte2 = MaxKeysize
        ///									Byte3 = Application KeySet Settings.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="pISODFName">ISO DF Name to be used. Will be 16 bytes and can also be NULL based on the option.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates a New Application on the PICC. The application is initialized according to the given settings. " +
                "The application key of the active key set are initialized with the Default Application Key."
            )
        ]
        public Status_t CreateApplication ( byte bOption, byte[] pAid, byte bKeySettings1, byte bKeySettings2, byte bKeySettings3,
            byte[] pKeySetValues, byte[] pISOFileId, byte[] pISODFName )
        {
            return phalMfdfEVx_CreateApplication ( m_pDataParams, bOption, pAid, bKeySettings1, bKeySettings2, bKeySettings3,
                pKeySetValues, pISOFileId, pISODFName, ( byte ) ( ( pISODFName == null ) ? 0 : pISODFName.Length ) );
        }

        /// <summary>
        /// Permanently deletes the applications on the PICC.
        /// </summary>
        ///
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Permanently deletes application on the PICC. The AID is released and new pplication can be creted."
            )
        ]
        public Status_t DeleteApplication ( byte[] pAid )
        {
            return phalMfdfEVx_DeleteApplication ( m_pDataParams, pAid, null, 0 );
        }

        /// <summary>
        /// Permanently deletes the applications on the PICC.
        /// </summary>
        ///
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        ///	<param name="aDAMMAC">[Optional, present if PICCDAMAuthKey or NXPDAMAuthKey is used for authentication]
        ///						  The MAC calculated by the card issuer to allow delegated application deletion.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Deletes the Delegated APllication. After deletion, the memory is release for creation of new Delegated Applicaiton of Same memory size and targeting the same memory slot."
            )
        ]
        public Status_t DeleteApplication ( byte[] aAid, byte[] aDAMMAC )
        {
            return phalMfdfEVx_DeleteApplication ( m_pDataParams, aAid, aDAMMAC,
                ( byte ) ( ( aDAMMAC == null ) ? 0 : aDAMMAC.Length ) );
        }

        /// <summary>
        /// Creates new Delegated application
        /// For delegated application creation, the PICC level (AID 0x000000) must have been selected.
        /// An active authentication with the PICCDAMAuthKey is required.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.DF_NAME_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_DF_NAME_AVAILABLE</param>
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        /// <param name="pDamParams">The DAM information. Should as mentioned below.
        ///								Byte0 = DAM Slot No [LSB]
        ///								Byte1 = DAM Slot No [MSB]
        ///								Byte2 = DAM Slot Version
        ///								Byte3 = Quota Limit [LSB]
        ///								Byte4 = Quota Limit [MSB]</param>
        /// <param name="bKeySettings1">Application Key settings.</param>
        /// <param name="bKeySettings2">Additional key settings.</param>
        /// <param name="bKeySettings3">Additional key settings.</param>
        /// <param name="pKeySetValues">The Key set values for the application. Should as mentioned below.
        ///									Byte0 = AKS ver
        ///									Byte1 = Number of Keysets
        ///									Byte2 = MaxKeysize
        ///									Byte3 = Application KeySet Settings.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="pISODFName">ISO DF Name to be used. Will be 16 bytes and can also be NULL based on the option.</param>
        /// <param name="pEncK">Encrypted initial application key. 32 bytes long.</param>
        /// <param name="pDAMMAC">8 byte DAM MAC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates delegated applications on the PICC with limited memory consumption. The application is initialized according to the given settings. The application keys" +
                          "of the active key set are initialized with the provided AppDAMDefaultKey."
            )
        ]
        public Status_t CreateDelegatedApplication ( byte wOption, byte[] pAid, byte[] pDamParams, byte bKeySettings1, byte bKeySettings2,
            byte bKeySettings3, byte[] bKeySetValues, byte[] pISOFileId, byte[] pISODFName, byte[] pEncK, byte[] pDAMMAC )
        {
            return phalMfdfEVx_CreateDelegatedApplication ( m_pDataParams, wOption, pAid, pDamParams, bKeySettings1, bKeySettings2,
                bKeySettings3, bKeySetValues, pISOFileId, pISODFName, ( byte ) ( ( pISODFName == null ) ? 0 : pISODFName.Length ),
                pEncK, pDAMMAC );
        }

        /// <summary>
        /// Selects one particular application on the PICC for further access.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE</param>
        /// <param name="pAid">The primary application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be selected then the Aid will be 01 00 00.</param>
        /// <param name="pAid2">The secondary application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be selected then the Aid will be 01 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Select 1 or 2 applications or the PICC level specified by their application identifier."
            )
        ]
        public Status_t SelectApplication ( byte bOption, byte[] pAid, byte[] pAid2 )
        {
            return phalMfdfEVx_SelectApplication ( m_pDataParams, bOption, pAid, pAid2 );
        }

        /// <summary>
        /// Returns application identifiers of all applications on the PICC
        ///
        /// <remarks>
        /// This response will be #PH_ERR_SUCCESS if all the application ids can be obtained in one call.
        /// If not, then #PH_ERR_SUCCESS_CHAINING is returned. The user has to call this function with
        /// bOption = #PH_EXCHANGE_RXCHAINING to get the remaining AIDs.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING</param>
        /// <param name="pAidBuff"></param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the Application IDentifiers of all active applications on a PICC."
            )
        ]
        public Status_t GetApplicationIDs ( byte bOption, out byte[] pAidBuff )
        {
            Status_t oStatus;
            IntPtr ppRxdata = IntPtr.Zero;
            byte NumAIDs = 0;

            oStatus = phalMfdfEVx_GetApplicationIDs ( m_pDataParams, bOption, ref ppRxdata, ref NumAIDs );

            if ( NumAIDs != 0x00 )
            {
                pAidBuff = new byte[NumAIDs * 3];
                Marshal.Copy ( ppRxdata, pAidBuff, 0, ( NumAIDs * 3 ) );
            }
            else
            {
                pAidBuff = null;
            }

            return oStatus;
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Returns the Dedicated File(DF) names
        ///
        /// <remarks>
        /// The pDFBuffer will be filled with 3 byte AID + 2 byte ISO Fid + 1 - 16 bytes DF Name at a time.
        /// If there are more DFs, then oStatus PH_ERR_SUCCESS_CHAINING is returned.
        /// The caller should call this again with bOption = PH_EXCHANGE_RXCHAINING.
        /// </remarks>
        ///
        /// CAUTION: This should not be called with AES or ISO authentication DOING SO WILL DAMAGE THE DESFIRE Card
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING</param>
        /// <param name="pDFBuffer">The ISO information about the application. At a time the information of one application
        ///							will be returned. The buffer should be 21 (3 + 2 + 16) bytes long.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the the Application IDentifiers together with a File ID and (optionally) a DF Name of all active applications with ISO/IEC 7816-4 support."
            )
        ]
        public Status_t GetDFNames ( byte bOption, out byte[] pDFBuffer )
        {
            Status_t oStatus;
            pDFBuffer = new byte[21];
            byte bSize = 0;

            oStatus = phalMfdfEVx_GetDFNames ( m_pDataParams, bOption, pDFBuffer, ref bSize );
            Array.Resize<byte> ( ref pDFBuffer, bSize );

            return oStatus;
        }
#endif

        /// <summary>
        /// Parameters associated with the delegated application can be retrieved using this command.
        /// </summary>
        ///
        /// <param name="pDAMSlot">Slot number associated with the memory space of the targeted
        ///						   delegated application. LSB first.</param>
        /// <param name="bDamSlotVer">Slot version associated with the memory space of the targeted
        ///							  delegated application.</param>
        /// <param name="pQuotaLimit">Maximal memory consumption of the targeted delegated application
        ///							  (in 32-byte blocks).</param>
        /// <param name="pFreeBlocks">Amount of unallocated memory of the targeted delegated application
        ///							  (in 32-byte blocks).</param>
        /// <param name="pAid">Application Identifier of the targeted DAM slot. This parameter is set to
        ///					   0x000000 if the DAM slot targeted by DAMSlotNo exists on the card, but is
        ///					   currently not occupied with an application, i.e. the latest application has
        ///					   been deleted.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the DAMSlotVersion and QuotaLimit of a target DAM slot on the card."
            )
        ]
        public Status_t GetDelegatedInfo ( byte[] pDAMSlot, out byte bDamSlotVer, out byte[] pQuotaLimit, out byte[] pFreeBlocks,
            out byte[] pAid )
        {
            bDamSlotVer = 0;
            pQuotaLimit = new byte[2];
            pFreeBlocks = new byte[2];
            pAid = new byte[3];
            byte[] pDamSlotVerTmp = new byte[1];

            Status_t oStatus = phalMfdfEVx_GetDelegatedInfo ( m_pDataParams, pDAMSlot, pDamSlotVerTmp, pQuotaLimit, pFreeBlocks,
                pAid );
            bDamSlotVer = pDamSlotVerTmp[0];

            return oStatus;
        }
        #endregion

        #region File Management
        /// <summary>
        /// Creates files for the storage of plain unformatted user data within an existing application
        /// on the PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MFC_MAPPING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.SDM_MIRRORING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="pFileSize">The size of the file. Will be of 3 bytes with LSB first.
        ///							If size 0x10 need to be created then the FileSize will be 10 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates files for the storage of plain unformatted user data within an existing application on the PICC."
            )
        ]
        public Status_t CreateStdDataFile ( byte bOption, byte bFileNo, byte[] pISOFileId, byte bFileOption, byte[] pAccessRights,
            byte[] pFileSize )
        {
            return phalMfdfEVx_CreateStdDataFile ( m_pDataParams, bOption, bFileNo, pISOFileId, bFileOption, pAccessRights,
                pFileSize );
        }

        /// <summary>
        /// Creates files for the storage of plain unformatted user data within an existing application
        /// on the PICC, additionally supporting the feature of an integrated backup
        /// mechanism.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        /// <param name="bFileNo">The file number to be created.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MFC_MAPPING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="pFileSize">The size of the file. Will be of 3 bytes with LSB first.
        ///							If size 0x10 need to be created then the FileSize will be 10 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates files for the storage of plain unformatted user data within an existing application on the PICC, additionally supporting the feature of an integrated backup mechanism."
            )
        ]
        public Status_t CreateBackupDataFile ( byte bOption, byte bFileNo, byte[] pISOFileId, byte bFileOption, byte[] pAccessRights,
            byte[] pFileSize )
        {
            return phalMfdfEVx_CreateBackupDataFile ( m_pDataParams, bOption, bFileNo, pISOFileId, bFileOption,
                pAccessRights, pFileSize );
        }

        /// <summary>
        /// Creates files for the storage and manipulation of 32bit signed integer values within
        /// an existing application on the PICC.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="bCommSett">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MFC_MAPPING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param></param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bLowerLmit">The lower limit for the file. Will be of 4 bytes with LSB first.
        ///							 If size 0x10 need to be created then the value will be 10 00 00 00.</param>
        /// <param name="bUpperLmit">The upper limit for the file. Will be of 4 bytes with LSB first.
        ///							 If size 0x20 need to be created then the value will be 20 00 00 00.</param>
        /// <param name="bValue">The initial value. Will be of 4 bytes with LSB first.
        ///						 If size 0x10 need to be created then the value will be 10 00 00 00.</param>
        /// <param name="bLimitedCredit">One of the below values.
        ///									NxpRdLibNet.alMfdfEVx.ValueFile.ENABLE_LIMITEDCREDIT
        ///									NxpRdLibNet.alMfdfEVx.ValueFile.ENABLE_FREE_GETVALUE</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates files for the storage and manipulation of 32bit signed integer values within an existing application on the PICC."
            )
        ]
        public Status_t CreateValueFile ( byte bFileNo, byte bCommSett, byte[] bAccessRights, byte[] bLowerLmit, byte[] bUpperLmit,
            byte[] bValue, byte bLimitedCredit )
        {
            return phalMfdfEVx_CreateValueFile ( m_pDataParams, bFileNo, bCommSett, bAccessRights, bLowerLmit,
                bUpperLmit, bValue, bLimitedCredit );
        }

        /// <summary>
        /// Creates files for multiple storage of structural similar data, for example for layalty programs
        /// within an existing application. Once the file is filled, further writing is not possible
        /// unless it is cleared.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        /// <param name="bFileNo">The file number to be created.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bCommSett">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="pRecordSize">The size of the file. Will be of 3 bytes with LSB first.
        ///							  If size 0x10 need to be created then the RecordSize will be 10 00 00.</param>
        /// <param name="pMaxNoOfRec">The maximum number of record in the file. Will be of 3 bytes with LSB first.
        ///							  If size 0x04 need to be created then the value will be 04 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates files for multiple storage of structural similar data, for example for loyalty programs, within an existing application on the PICC. Once the file is filled " +
                          "completely with data records, further writing to the file is not possible unless it is cleared."
            )
        ]
        public Status_t CreateLinearRecordFile ( byte bOption, byte bFileNo, byte[] pIsoFileId, byte bCommSett, byte[] pAccessRights,
            byte[] pRecordSize, byte[] pMaxNoOfRec )
        {
            return phalMfdfEVx_CreateLinearRecordFile ( m_pDataParams, bOption, bFileNo, pIsoFileId, bCommSett, pAccessRights,
                pRecordSize, pMaxNoOfRec );
        }

        /// <summary>
        /// Creates files for multiple storage of structural similar data, for example for logging transactions,
        /// within an existing application. Once the file is filled, the PICC automatically overwrites the
        /// oldest record with the latest written one.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        /// <param name="bFileNo">The file number to be created.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bCommSett">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="pRecordSize">The size of the file. Will be of 3 bytes with LSB first.
        ///							  If size 0x10 need to be created then the RecordSize will be 10 00 00.</param>
        /// <param name="pMaxNoOfRec">The maximum number of record in the file. Will be of 3 bytes with LSB first.
        ///							  If size 0x04 need to be created then the value will be 04 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates files for multiple storage of structural similar data, for example for logging transactions, within an existing application on the PICC. Once the file is filled completely " +
                          "with data records, the PICC automatically overwrites the oldest record with the latest written one. This wrap is fully transparent for the PCD."
            )
        ]
        public Status_t CreateCyclicRecordFile ( byte bOption, byte bFileNo, byte[] pIsoFileId, byte bCommSett, byte[] pAccessRights,
            byte[] pRecordSize, byte[] pMaxNoOfRec )
        {
            return phalMfdfEVx_CreateCyclicRecordFile ( m_pDataParams, bOption, bFileNo, pIsoFileId, bCommSett, pAccessRights,
                pRecordSize, pMaxNoOfRec );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Creates a Transaction MAC File and enables the Transaction MAC feature for the targeted application.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created.ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="bCommSett">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bTMKeyOption">The TMKeyOption and should be always <see cref="NxpRdLibNet.alMfdfEVx.Key.AES128"/>.</param>
        /// <param name="bTMKey">The 16 byte key value to be stored in the PICC. This is applicable for Software only.</param>
        /// <param name="bTMKeyVer">The version for the key to be stored in the PICC. This is applicable for Software and SAM.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates a Transaction MAC File and enables the Transaction MAC feature for the targeted application."
            )
        ]
        public Status_t CreateTransactionMacFile ( byte bFileNo, byte bCommSett, byte[] pAccessRights, byte bTMKeyOption, byte[] bTMKey,
            byte bTMKeyVer )
        {
            byte[] bTMKeyTmp = new byte[bTMKey.Length];
            bTMKey.CopyTo ( bTMKeyTmp, 0 );

            return phalMfdfEVx_CreateTransactionMacFile ( m_pDataParams, 0x00, bFileNo, bCommSett, pAccessRights,
                0x00, bTMKeyOption, bTMKeyTmp, bTMKeyVer, null, 0x00 );
        }

        /// <summary>
        /// Creates a Transaction MAC File and enables the Transaction MAC feature for the targeted application.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options. Only applicable for SAM S / X component.
        ///							NxpRdLibNet.alMfdfEVx.CreateTMACFile.DIV_OFF
        ///							NxpRdLibNet.alMfdfEVx.CreateTMACFile.DIV_ON</param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							 secondary application indicator.</param>
        /// <param name="bCommSett">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="wKeyNo">Key number in SAM keystore. For software is not applicable.</param>
        /// <param name="bTMKeyOption">The TMKeyOption and should be always <see cref="NxpRdLibNet.alMfdfEVx.Key.AES128"/>.</param>
        /// <param name="bTMKey">The 16 byte key value to be stored in the PICC. This is applicable for Software only.</param>
        /// <param name="bTMKeyVer">The version for the key to be stored in the PICC. This is applicable for Software and SAM.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Creates a Transaction MAC File and enables the Transaction MAC feature for the targeted application."
            )
        ]
        public Status_t CreateTransactionMacFile ( ushort wOption, byte bFileNo, byte bCommSett, byte[] pAccessRights, ushort wKeyNo,
            byte bTMKeyOption, byte[] bTMKey, byte bTMKeyVer, byte[] pDivInput )
        {

            byte[] bTMKeyTmp = new byte[bTMKey.Length];
            bTMKey.CopyTo ( bTMKeyTmp, 0 );

            return phalMfdfEVx_CreateTransactionMacFile ( m_pDataParams, wOption, bFileNo, bCommSett, pAccessRights, wKeyNo,
                bTMKeyOption, bTMKeyTmp, bTMKeyVer, pDivInput, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }
#endif

        /// <summary>
        /// Permanently deactivates a file within the file directory of the currently selected application.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be deleted. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param></param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Permanently deactivates a file within the file directory of the currently selected application."
            )
        ]
        public Status_t DeleteFile ( byte bFileNo )
        {
            return phalMfdfEVx_DeleteFile ( m_pDataParams, bFileNo );
        }

        /// <summary>
        /// Returns the file IDs of all active files within the currently selected application.
        /// </summary>
        ///
        /// <param name="pFid">The buffer containing the available file ID(s). Buffer should be 32 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetFileIDs ( out byte[] pFid )
        {
            byte bNumFIDs = 0;
            Status_t oStatus;
            pFid = new byte[32];

            oStatus = phalMfdfEVx_GetFileIDs ( m_pDataParams, pFid, ref bNumFIDs );
            Array.Resize<byte> ( ref pFid, bNumFIDs );

            return oStatus;
        }

        /// <summary>
        /// Get the ISO File IDs.
        /// </summary>
        ///
        /// <param name="pFidBuffer">The buffer containing the available ISO file ID(s). Buffer should be 64 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Returns the ISO File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetISOFileIDs ( out byte[] pFidBuffer )
        {
            Status_t oStatus;
            byte bNumFIDs = 0;

            pFidBuffer = new byte[64];

            oStatus = phalMfdfEVx_GetISOFileIDs ( m_pDataParams, pFidBuffer, ref bNumFIDs );

            Array.Resize<byte> ( ref pFidBuffer, bNumFIDs * 2 );
            return oStatus;
        }

        /// <summary>
        /// Get informtion on the properties of a specific file
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be deleted. ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param></param>
        /// <param name="pFSBuffer">The buffer containing the settings.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Get information on the properties of a specific file."
            )
        ]
        public Status_t GetFileSettings ( byte bFileNo, out byte[] pFSBuffer )
        {
            Status_t oStatus;
            byte bBufferLen = 0;

            pFSBuffer = new byte[100];

            oStatus = phalMfdfEVx_GetFileSettings ( m_pDataParams, bFileNo, pFSBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pFSBuffer, bBufferLen );

            return oStatus;
        }

        /// <summary>
        /// Get file related counters used for Secure Dynamic Messaging.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">File number for which the Counter information need to be received.</param>
        /// <param name="pFSBuffer">The SDMReadCounter information returned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Get file related counters used for Secure Dynamic Messaging."
            )
        ]
        public Status_t GetFileCounters ( byte bOption, byte bFileNo, out byte[] pFSBuffer )
        {
            Status_t oStatus;
            byte bBufferLen = 0;

            pFSBuffer = new byte[10];

            oStatus = phalMfdfEVx_GetFileCounters ( m_pDataParams, bOption, bFileNo, pFSBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pFSBuffer, bBufferLen );

            return oStatus;
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.
        ///						  ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="bFileOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.TMCLIMIT_PRESENT
        ///								NxpRdLibNet.alMfdfEVx.FileOption.SDM_MIRRORING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bAddInfoLen">Complete length consisting of
        ///								Additional Access Rights if bFileOption & <see cref="NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT"/>
        ///								TMC Limit                if bFileOption & <see cref="NxpRdLibNet.alMfdfEVx.FileOption.TMCLIMIT_PRESENT"/>
        ///									If TMCLimit and AddARs is not available, bAddInfoLen = 0
        ///									If AddARs is not available,              bAddInfoLen = 4
        ///									If TMCLimit is not available,            bAddInfoLen = 0 - 7
        ///									If TMCLimit and AddARs is available,     bAddInfoLen = ( ( 0 - 7 ) + 4 )</param>
        /// <param name="pAddInfo">Buffer should contain the following information.
        ///								Additional access rights || TMCLimit</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Changes the access parameters of an existing file."
            )
        ]
        public Status_t ChangeFileSettings ( byte bOption, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bAddInfoLen,
            byte[] pAddInfo )
        {
            byte[] pAccessRightsTmp = null;
            byte[] pAddInfoTmp = null;

            if ( pAccessRights != null )
            {
                pAccessRightsTmp = new byte[pAccessRights.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );
            }

            if ( pAddInfo != null )
            {
                pAddInfoTmp = new byte[pAddInfo.Length];
                pAddInfo.CopyTo ( pAddInfoTmp, 0 );
            }

            return phalMfdfEVx_ChangeFileSettings ( m_pDataParams, bOption, bFileNo, bFileOption, pAccessRightsTmp, bAddInfoLen,
                pAddInfoTmp );
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.
        ///						  ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="bFileOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.FileOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.FileOption.FULL
        ///
        ///							Ored with one of the below options.
        ///								NxpRdLibNet.alMfdfEVx.FileOption.TMCLIMIT_PRESENT
        ///								NxpRdLibNet.alMfdfEVx.FileOption.SDM_MIRRORING_ENABLED
        ///								NxpRdLibNet.alMfdfEVx.FileOption.ADDITIONAL_AR_PRESENT</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="pAddARs">Additional access rights. Should be ( 2 * bAddInfoLen )</param>
        /// <param name="pSDMInfo">Complete SDM information. Should be N (based on the SDMOptions being used).
        ///							Should be [SDMOption || SDM AccessRights || VCUIDOffset || SDMReadCtrOffset || PICCDataOffset ||
        ///							SDMACInputOffset || SDMENCOffset || SDMENCLength || SDMMACOffset]</param>
        /// <param name="pTMCLimit">TMCLimit information.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Can also be used for DESFire, DESFire EV1 and DESFire EV2 proudcts by providing NULL for the paramters that are not supported by respective products.",
                Message = "Changes the access parameters of an existing file."
            )
        ]
        public Status_t ChangeFileSettings ( byte bOption, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte[] pAddARs,
            byte[] pSDMInfo, byte[] pTMCLimit )
        {
            byte[] aAddARsTmp = new byte[( pAddARs == null ) ? 0 : pAddARs.Length];
            byte[] pSDMInfoTmp = new byte[( pSDMInfo == null ) ? 0 : pSDMInfo.Length];
            byte[] pTMCLimitTmp = new byte[( pTMCLimit == null ) ? 0 : pTMCLimit.Length];
            byte bOffset = 0;

            if ( !aAddARsTmp.Length.Equals ( 0 ) ) Array.Copy ( pAddARs, aAddARsTmp, aAddARsTmp.Length );
            if ( !pSDMInfoTmp.Length.Equals ( 0 ) ) Array.Copy ( pSDMInfo, pSDMInfoTmp, pSDMInfoTmp.Length );
            if ( !pTMCLimitTmp.Length.Equals ( 0 ) ) Array.Copy ( pTMCLimit, pTMCLimitTmp, pTMCLimitTmp.Length );

            byte[] pAddInfo = new byte[aAddARsTmp.Length + pSDMInfoTmp.Length + pTMCLimitTmp.Length];

            if ( !aAddARsTmp.Length.Equals ( 0 ) || ( ( bFileOption & 0x80U ) == 0x80U ) )
            {
                Array.Resize ( ref pAddInfo, pAddInfo.Length + 1 );
                pAddInfo[bOffset++] = ( byte ) ( aAddARsTmp.Length / 2 );
            }

            aAddARsTmp.CopyTo ( pAddInfo, bOffset );
            bOffset = ( byte ) ( bOffset + aAddARsTmp.Length );

            pSDMInfoTmp.CopyTo ( pAddInfo, bOffset );
            bOffset = ( byte ) ( bOffset + pSDMInfoTmp.Length );

            pTMCLimitTmp.CopyTo ( pAddInfo, bOffset );


            return ChangeFileSettings ( ( byte ) ( bOption | 0x80U ), bFileNo, bFileOption, pAccessRights, ( byte ) pAddInfo.Length,
                pAddInfo );
        }
        #endregion

        #region Data Management
        /// <summary>
        /// Reads data from standard data files or backup data files
        ///
        /// <remarks>
        /// Chaining upto the size of the HAL Rx buffer is handled within this function. If more data is to be read, the user has to call
        /// this function again with bOption = <see cref="NxpRdLibNet.ExchangeOptions.RXCHAINING"/> | [one of the communication options]
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL
        ///
        ///							<see cref="NxpRdLibNet.ExchangeOptions.RXCHAINING"/> Or'd with
        ///							the above option flag if Chaining oStatus is returned.</param>
        /// <param name="bIns">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.Chaining.APPLICATION_CHAINING
        ///							NxpRdLibNet.alMfdfEVx.Chaining.ISO_CHAINING</param>
        /// <param name="bFileNo">The file number from where the data to be read. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pOffset">The offset from where the data should be read. Will be of 3 bytes with LSB first.
        ///						  If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///						  If 0x10 bytes need to be read then it will be 10 00 00.
        ///						  If complete file need to be read then it will be 00 00 00.</param>
        /// <param name="pRxdata">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Both Native Chaining and Application chaining can be performed using the INS parameter information.",
                Message = "Reads data from FileType.StandardData, FileType.BackupData or FileType.TransactionMAC files."
            )
        ]
        public Status_t ReadData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pLength, out byte[] pRxdata )
        {
            Status_t oStatus;
            IntPtr ppRxdata = IntPtr.Zero;
            ushort rxDataLen = 0;

            oStatus = phalMfdfEVx_ReadData ( m_pDataParams, bOption, bIns, bFileNo, pOffset, pLength, ref ppRxdata, ref rxDataLen );
            if ( rxDataLen != 0 )
            {
                pRxdata = new byte[rxDataLen];
                Marshal.Copy ( ppRxdata, pRxdata, 0, rxDataLen );
            }
            else
            {
                pRxdata = null;
            }
            return oStatus;
        }

        /// <summary>
        /// Writes data to standard data files or backup data files
        /// <remarks> Implements chaining to the card. </remarks>
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bIns">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.Chaining.APPLICATION_CHAINING
        ///							NxpRdLibNet.alMfdfEVx.Chaining.ISO_CHAINING</param>
        /// <param name="bFileNo">The file number to which the data to be written. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///						  If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pTxData">The data to be written to the PICC.</param>
        /// <param name="pTxDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///							 If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Both Native Chaining and Application chaining can be performed using the INS parameter information.",
                Message = "Writes data to FileType.StandardData and FileType.BackupData files."
            )
        ]
        public Status_t WriteData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pTxData, byte[] pTxDataLen )
        {
            byte[] pTxDataTmp = new byte[pTxData.Length];

            pTxData.CopyTo ( pTxDataTmp, 0 );

            return phalMfdfEVx_WriteData ( m_pDataParams, bOption, bIns, bFileNo, pOffset, pTxDataTmp, pTxDataLen );
        }

        /// <summary>
        /// Reads the currently stored value from value files.
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">The file number for which the value to the retrieved. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pValue">The value returned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Reads the currently stored value from a FileType.Value file."
            )
        ]
        public Status_t GetValue ( byte bCommOption, byte bFileNo, out byte[] pValue )
        {
            pValue = new byte[4];

            return phalMfdfEVx_GetValue ( m_pDataParams, bCommOption, bFileNo, pValue );
        }

        /// <summary>
        /// Increases a value stored in a Value File
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">The file number to which the value should be credited. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pValue">The value to be credited. Will be of 4 bytes with LSB first.
        ///						 If 0x10 bytes need to be credited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Increases a value stored in a FileType.Value file."
            )
        ]
        public Status_t Credit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            byte[] pValueTmp = new byte[pValue.Length];

            pValue.CopyTo ( pValueTmp, 0 );

            return phalMfdfEVx_Credit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
        }

        /// <summary>
        /// Decreases a value stored in a Value File
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">The file number for which the value to the retrieved. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pValue">The value to be debited. Will be of 4 bytes with LSB first.
        ///						 If 0x10 bytes need to be debited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Decreases a value stored in a FileType.Value file."
            )
        ]
        public Status_t Debit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            byte[] pValueTmp = new byte[pValue.Length];

            pValue.CopyTo ( pValueTmp, 0 );

            return phalMfdfEVx_Debit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
        }

        /// <summary>
        /// Allows a limited increase of a value stored in a Value File without having full credit permissions to the file.
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">The file number for which the value to the retrieved. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pValue">The value to be credited. Will be of 4 bytes with LSB first.
        ///						 If 0x10 bytes need to be credited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Allows a limited increase of a value stored in a FileType.Value file without having full Cmd.Credit permissions to the file."
            )
        ]
        public Status_t LimitedCredit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            byte[] pValueTmp = new byte[pValue.Length];

            pValue.CopyTo ( pValueTmp, 0 );

            return phalMfdfEVx_LimitedCredit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
        }

        /// <summary>
        /// Reads out a set of complete records from a Cyclic or Linear Record File.
        ///
        /// <remarks>
        /// The readrecords command reads and stores data in the rxbuffer upto the rxbuffer size before returning
        /// to the user. The rxbuffer is configured during the HAL init and this is specified by the user.
        ///
        /// Chaining upto the size of the HAL Rx buffer is handled within this function.
        /// If more data is to be read, the user has to call this function again with
        /// bCommOption = NxpRdLibNet.ExchangeOptions.RXCHAINING | [one of the communication options]
        ///
        /// If TMI collection is ON, if pRecCount is zero then pRecSize is madatory parameter.
        /// If pRecSize and RecCount are zero and TMI collection is ON, then  PH_ERR_INVALID_PARAMETER error returned.
        /// If TMI collection is ON; and if wrong pRecSize is provided, then wrong RecCount value will be calculated
        /// and updated for TMI collection.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///								NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///								NxpRdLibNet.alMfdfEVx.CommOption.FULL
        ///
        ///							<see cref="NxpRdLibNet.ExchangeOptions.RXCHAINING"/> Or'd with
        ///							the above option flag if Chaining oStatus is returned.</param>
        /// <param name="bIns">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.Chaining.APPLICATION_CHAINING
        ///							NxpRdLibNet.alMfdfEVx.Chaining.ISO_CHAINING</param>
        /// <param name="bFileNo">The file number from where the data to be read. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pRecNo">3 bytes LSB first. Record number of the newest record targeted, starting to count
        ///						 from the latest record written.</param>
        /// <param name="pRecCount">3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read.</param>
        /// <param name="pRecSize">The number of bytes to be read. Will be of 3 bytes with LSB first.</param>
        /// <param name="pRxdata">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Both Native Chaining and Application chaining can be performed using the INS parameter information.",
                Message = "Reads out a set of complete records from a FileType.LinearRecord or FileType.CyclicRecord File."
            )
        ]
        public Status_t ReadRecords ( byte bCommOption, byte bIns, byte bFileId, byte[] pRecNo, byte[] pRecCount, byte[] pRecSize,
            out byte[] pRxdata )
        {
            Status_t oStatus;
            IntPtr ppRxdata = IntPtr.Zero;
            ushort pRxDataLen = 0;

            oStatus = phalMfdfEVx_ReadRecords ( m_pDataParams, bCommOption, bIns, bFileId, pRecNo, pRecCount, pRecSize,
                ref ppRxdata, ref pRxDataLen );

            if ( pRxDataLen != 0 )
            {
                pRxdata = new byte[pRxDataLen];
                Marshal.Copy ( ppRxdata, pRxdata, 0, pRxDataLen );
            }
            else
            {
                pRxdata = null;
            }
            return oStatus;

        }

        /// <summary>
        /// Writes data to a record in a Cyclic or Linear Record File.
        ///
        /// <remarks>
        /// Implements chaining to the card.
        /// The data provided on pData will be chained to the card by sending data
        /// upto the frame size of the DESFire PICC, at a time.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bIns">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.Chaining.APPLICATION_CHAINING
        ///							NxpRdLibNet.alMfdfEVx.Chaining.ISO_CHAINING</param>
        /// <param name="bFileNo">The file number to which the data should be written. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///						  If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pData">The data to be written to the PICC.</param>
        /// <param name="pDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///						   If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Both Native Chaining and Application chaining can be performed using the INS parameter information.",
                Message = "Writes data to a record in a FileType.LinearRecord or FileType.CyclicRecord file."
            )
        ]
        public Status_t WriteRecord ( byte bCommOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pData, byte[] pDataLen )
        {
            byte[] pDataTmp = new byte[pData.Length];

            pData.CopyTo ( pDataTmp, 0 );

            return phalMfdfEVx_WriteRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pOffset, pDataTmp, pDataLen );
        }

        /// <summary>
        /// Updates data to a record in a Cyclic or Linear Record File.
        ///
        /// <remarks>
        /// Implements chaining to the card.
        /// The data provided on pData will be chained to the card by sending
        /// data upto the frame size of the DESFire PICC, at a time.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bIns">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.Chaining.APPLICATION_CHAINING
        ///							NxpRdLibNet.alMfdfEVx.Chaining.ISO_CHAINING</param>
        /// <param name="bFileNo">The file number to which the data should be written. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        /// <param name="pRecNo">Record number. Records are numbered starting with the latest record written.</param>
        /// <param name="pOffset">Starting position for the update operationwithin the targeted record.</param>
        /// <param name="pData">The data to be written to the PICC.</param>
        /// <param name="pDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///						   If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Both Native Chaining and Application chaining can be performed using the INS parameter information.",
                Message = "Updates data of an existing record in a FileType.LinearRecord or FileType.CyclicRecord file."
            )
        ]
        public Status_t UpdateRecord ( byte bCommOption, byte bIns, byte bFileNo, byte[] pRecNo, byte[] pOffset, byte[] pData,
            byte[] pDataLen )
        {
            byte[] pDataTmp = new byte[pData.Length];

            pData.CopyTo ( pDataTmp, 0 );

            return phalMfdfEVx_UpdateRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pRecNo, pOffset, pDataTmp,
                pDataLen );
        }

        /// <summary>
        /// Resets a Cyclic or Linear Record File.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be cleared. ORed with
        ///						  <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///						  secondary application indicator.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Clear all records in a FileType.LinearRecord or FileType.CyclicRecord file."
            )
        ]
        public Status_t ClearRecordFile ( byte bFileNo )
        {
            return phalMfdfEVx_ClearRecordFile ( m_pDataParams, bFileNo );
        }
        #endregion

        #region Transaction Management
        /// <summary>
        /// Validates all previous write access' on Backup Data files, value files and record files within one application.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.alMfdfEVx.TmcTmv.TMC_TMV_NOT_RETURNED
        ///							NxpRdLibNet.alMfdfEVx.TmcTmv.TMC_TMV_RETURNED
        ///							NxpRdLibNet.alMfdfEVx.TmcTmv.INCLUDE_OPTION</param>
        /// <param name="bTMC">The increased value as stored in FileType.TransactionMAC.
        ///					   4 byte Transaction MAC counter value.</param>
        /// <param name="bTMAC">8 bytes Transaction MAC value.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Validates all previous write accesses on FileType.BackupData, FileType.Value, FileType.LinearRecord and FileType.CyclicRecord files within the selected application(s). " +
                          "If applicable, the FileType.TransactionMAC file is updated with the calculated Transaction MAC."
            )
        ]
        public Status_t CommitTransaction ( byte bOption, out byte[] bTMC, out byte[] bTMAC )
        {
            bTMC = new byte[4];
            bTMAC = new byte[8];
            Status_t oStatus;

            oStatus = phalMfdfEVx_CommitTransaction ( m_pDataParams, bOption, bTMC, bTMAC );

            /* User needs to check for null before dereferencing these variable*/
            if ( ( bOption & 0x01 ) == 0x00 )
            {
                bTMC = null;
                bTMAC = null;
            }

            return oStatus;

        }

        /// <summary>
        /// Aborts all previous write accesses on FileType.BackupData, FileType.Value, File-Type.LinearRecord and
        /// FileType.CyclicRecord files within the selected application(s). If applicable, the Transaction MAC calculation is aborted.
        /// </summary>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Aborts all previous write accesses on FileType.BackupData, FileType.Value, FileType.LinearRecord and FileType.CyclicRecord files within the selected application(s). " +
                          "If applicable, the Transaction MAC calculation is aborted."
            )
        ]
        public Status_t AbortTransaction ()
        {
            return phalMfdfEVx_AbortTransaction ( m_pDataParams );
        }

        /// <summary>
        /// Secures the transaction by commiting the application to ReaderID specified
        /// </summary>
        ///
        /// <param name="bTMRI">16 bytes of Transaction MAC reader ID.</param>
        /// <param name="bEncTMRI">16 bytes of Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Commits a ReaderID for the ongoing transaction. This will allow a backend to identify the attacking merchant in case of fraud detected."
            )
        ]
        public Status_t CommitReaderID ( byte[] bTMRI, out byte[] bEncTMRI )
        {
            Status_t oStatus = null;
            Status_t oStatus1 = null;
            ushort wLength = 0, wVal = 0;

            /* Do a GetConfig of ADDITIONAL_INFO to have the backup of current value. */
            oStatus1 = GetConfig ( 0x00A1, ref wVal );

            byte[] aTMRIenc = new byte[16];
            oStatus = phalMfdfEVx_CommitReaderID ( m_pDataParams, bTMRI, aTMRIenc );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                /* Do a Get Config of  ADDITIONAL_INFO to read the length(wLength) of the TMRI */
                oStatus1 = GetConfig ( 0x00A1, ref wLength );

                if ( wLength.Equals ( 0x00 ) )
                {
                    bEncTMRI = null;
                }
                else
                {
                    bEncTMRI = new byte[wLength];

                    /* Copy the version string only to the length retreived */
                    Array.Copy ( aTMRIenc, bEncTMRI, wLength );
                }
            }
            else
            {
                /* In Error case, set the enTMRI to NULL. */
                bEncTMRI = null;
            }

            /* Do a Set Config of  ADDITIONAL_INFO to retain the backup value. */
            oStatus1 = SetConfig ( 0x00A1, wVal );
            return oStatus;
        }
        #endregion

        #region ISO7816-4
        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfdfEVx.FCI.RETURNED
        ///							NxpRdLibNet.alMfdfEVx.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_CHILD_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_EF_CURRENT_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when
        ///						  bOption = <see cref="NxpRdLibNet.alMfdfEVx.Selector.SELECT_BY_DF_NAME"/>.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, out byte[] pFCI )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            ushort wBytesRead = 0;
            byte bDfNameLen = 0x00;

            /*
			 * Modified API is provided immediately after this API under the commented section "ISOSELECTFILE".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API.
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc/Le is passed as short APDU */

            if ( pDFname != null )
            {
                bDfNameLen = ( byte ) pDFname.Length;
            }

            oStatus = phalMfdfEVx_IsoSelectFile ( m_pDataParams, bOption, bSelector, pFid, pDFname, bDfNameLen,
                bExtendedLenApdu, ref ppRxBuffer, ref wBytesRead );
            if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
            {
                pFCI = new byte[wBytesRead];
                Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
            }
            else
            {
                pFCI = null;
            }
            return oStatus;
        }

        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfdfEVx.FCI.RETURNED
        ///							NxpRdLibNet.alMfdfEVx.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_CHILD_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_EF_CURRENT_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///							NxpRdLibNet.alMfdfEVx.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when
        ///						  bOption = <see cref="NxpRdLibNet.alMfdfEVx.Selector.SELECT_BY_DF_NAME"/>.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, byte bExtendedLenApdu,
            out byte[] pFCI )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            ushort wBytesRead = 0;
            byte bDfNameLen = 0x00;

            if ( pDFname != null )
            {
                bDfNameLen = ( byte ) pDFname.Length;
            }

            oStatus = phalMfdfEVx_IsoSelectFile ( m_pDataParams, bOption, bSelector, pFid, pDFname, bDfNameLen,
                bExtendedLenApdu, ref ppRxBuffer, ref wBytesRead );
            if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
            {
                pFCI = new byte[wBytesRead];
                Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
            }
            else
            {
                pFCI = null;
            }
            return oStatus;
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT   : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING: To exchange the chaining command to PICC in
        ///																	case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, byte bBytesToRead, out byte[] pRxBuffer )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            /*
			 * Modified API is provided immediately after this API under the commented section "ISOREADBINARY".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Le is passed as short APDU */

            oStatus = phalMfdfEVx_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, bBytesToRead, bExtendedLenApdu,
                ref ppRxBuffer, ref dwBytesRead );
            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }
            return oStatus;
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT   : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING: To exchange the chaining command to PICC in
        ///																	case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, UInt32 bBytesToRead, byte bExtendedLenApdu,
            out byte[] pRxBuffer )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            oStatus = phalMfdfEVx_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, bBytesToRead,
                bExtendedLenApdu, ref ppRxBuffer, ref dwBytesRead );
            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }
            return oStatus;
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] pData )
        {
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOUPDATEBINARY".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            return phalMfdfEVx_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, pData,
                 ( uint ) pData.Length );
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData )
        {
            return phalMfdfEVx_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, pData,
                 ( uint ) pData.Length );
        }

        /// <summary>
        /// Iso Read Records. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT   : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING: To exchange the chaining command to PICC in
        ///																	case PICC returns AF as the status.</param>
        /// <param name="bRecNo">Record to read / from where to read.</param>
        /// <param name="bReadAllFromP1">Whether to read all records from P1 or just one.</param>
        /// <param name="bSfid">Short ISO File Id.</param>
        /// <param name="bBytesToRead">Number of bytes to read.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Read from a record file."
            )
        ]
        public Status_t IsoReadRecords ( ushort wOption, byte bRecNo, byte bReadAllFromP1, byte bSfid, byte bBytesToRead,
            out byte[] pRxBuffer )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            /*
			 * Modified API is provided immediately after this API under the commented section "ISOREADRECORDS".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Le is passed as short APDU */

            oStatus = phalMfdfEVx_IsoReadRecords ( m_pDataParams, wOption, bRecNo, bReadAllFromP1, bSfid,
                bBytesToRead, bExtendedLenApdu, ref ppRxBuffer, ref dwBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }
            return oStatus;
        }

        /// <summary>
        ///
        /// Iso Read Records. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT   : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING: To exchange the chaining command to PICC in
        ///																	case PICC returns AF as the status.</param>
        /// <param name="bRecNo">Record to read / from where to read.</param>
        /// <param name="bReadAllFromP1">Whether to read all records from P1 or just one.</param>
        /// <param name="bSfid">Short ISO File Id.</param>
        /// <param name="bBytesToRead">Number of bytes to read.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Read from a record file."
            )
        ]
        public Status_t IsoReadRecords ( ushort wOption, byte bRecNo, byte bReadAllFromP1, byte bSfid, UInt32 bBytesToRead,
            byte bExtendedLenApdu, out byte[] pRxBuffer )
        {
            Status_t oStatus;
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            oStatus = phalMfdfEVx_IsoReadRecords ( m_pDataParams, wOption, bRecNo, bReadAllFromP1, bSfid, bBytesToRead,
                bExtendedLenApdu, ref ppRxBuffer, ref dwBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }
            return oStatus;
        }

        /// <summary>
        /// Iso Append record. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        /// <param name="bSfid">Short ISO File Id.</param>
        /// <param name="pData">Data to be appended.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Write a new record to a record file."
            )
        ]
        public Status_t IsoAppendRecord ( byte bSfid, byte[] pData )
        {
            /* Modified API is provided immediately after this API under the commented section "ISOAPPENDRECORD".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            return phalMfdfEVx_IsoAppendRecord ( m_pDataParams, bSfid, pData, ( UInt32 ) pData.Length,
                bExtendedLenApdu );
        }

        /// <summary>
        /// Iso Append record. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        /// <param name="bSfid">Short ISO File Id.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU</param>
        /// <param name="pData">Data to be appended.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Write a new record to a record file."
            )
        ]
        public Status_t IsoAppendRecord ( byte bSfid, byte bExtendedLenApdu, byte[] pData )
        {
            return phalMfdfEVx_IsoAppendRecord ( m_pDataParams, bSfid, pData, ( UInt32 ) pData.Length,
                bExtendedLenApdu );
        }

        /// <summary>
        /// Iso Update record
        /// </summary>
        ///
        /// <param name="bIns">Either 0xDC or 0xDD.</param>
        /// <param name="bRecNo">Record number. 00 indicates current record.</param>
        /// <param name="bSfid">Short Iso File Id bits 0..4 only code this value. Either 0 or sfid.</param>
        /// <param name="bRefCtrl">Bit 0 to bit 3 code the value based on bIns as defined in ISO 7816-4 Update Record.</param>
        /// <param name="pData">Data to write.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoUpdateRecord ( byte bIns, byte bRecNo, byte bSfid, byte bRefCtrl, byte[] pData )
        {
            return phalMfdfEVx_IsoUpdateRecord ( m_pDataParams, bIns, bRecNo, bSfid, bRefCtrl, pData,
                ( byte ) ( ( pData == null ) ? 0 : pData.Length ) );
        }

        /// <summary>
        /// ISOGetChallenge. This command is implemented in compliance with ISO/IEC 7816-4.
        /// Returns the random number from the PICC. Size depends on the key type referred by wKeyNo and wKeyVer.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </remarks>
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used form keystore.</param>
        /// <param name="wKeyVer">Key version to be used form keystore.</param>
        /// <param name="bLe">Length of expected challenge RPICC1.</param>
        /// <param name="pRPICC1">RPICC1 returned from PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Get a challenge (first part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoGetChallenge ( ushort wKeyNo, ushort wKeyVer, byte bLe, out byte[] pRPICC1 )
        {
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOGETCHALLENGE".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			  */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Le is passed as short APDU */

            pRPICC1 = new byte[16];
            return phalMfdfEVx_IsoGetChallenge ( m_pDataParams, wKeyNo, wKeyVer, bExtendedLenApdu, bLe, pRPICC1 );
        }

        /// <summary>
        /// ISOGetChallenge. This command is implemented in compliance with ISO/IEC 7816-4.
        /// Returns the random number from the PICC. Size depends on the key type referred by wKeyNo and wKeyVer.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </remarks>
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used form keystore.</param>
        /// <param name="wKeyVer">Key version to be used form keystore.</param>
        /// <param name="bLe">Length of expected challenge RPICC1.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU</param>
        /// <param name="pRPICC1">RPICC1 returned from PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Get a challenge (first part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoGetChallenge ( ushort wKeyNo, ushort wKeyVer, UInt32 bLe, byte bExtendedLenApdu, out byte[] pRPICC1 )
        {
            pRPICC1 = new byte[16];
            return phalMfdfEVx_IsoGetChallenge ( m_pDataParams, wKeyNo, wKeyVer, bExtendedLenApdu, bLe, pRPICC1 );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Iso External Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </remarks>
        /// </summary>
        ///
        /// <param name="pInput">Input data should be as mentioned below.
        ///							Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES, 04 => 3k3DES, 09 => AES128
        ///							Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise.
        ///							Key number on card - 1 Byte: 0x0 to 0xD
        ///							Length of random number: 1 Byte
        ///							Random number generated by PCD : 8 or 16 bytes. Not required for Sam non X mode.
        ///							Random number returned by GetChallenge command : 8 Bytes or 16 Bytes
        ///							Key number : 2 bytes
        ///							key Version: 2 bytes</param>
        /// <param name="bInputLen">Length of bytes available in Input buffer.</param>
        /// <param name="pDataOut">Returns Rnd number PCD2 in sam non x mode. Nothing in S/W mode.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Authenticate the PCD (second part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoExternalAuthenticate ( byte[] pInput, byte bInputLen, out byte[] pDataOut )
        {
            byte OutLen = 0;
            Status_t oStatus;
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOEXTERNALAUTHENTICATE".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            pDataOut = new byte[255];
            oStatus = phalMfdfEVx_IsoExternalAuthenticate ( m_pDataParams, pInput, bInputLen, bExtendedLenApdu,
                pDataOut, ref OutLen );
            Array.Resize<byte> ( ref pDataOut, OutLen );

            return oStatus;
        }

        /// <summary>
        /// Iso External Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </remarks>
        /// </summary>
        ///
        /// <param name="pInput">Input data should be as mentioned below.
        ///							Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES, 04 => 3k3DES, 09 => AES128
        ///							Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise.
        ///							Key number on card - 1 Byte: 0x0 to 0xD
        ///							Length of random number: 1 Byte
        ///							Random number generated by PCD : 8 or 16 bytes. Not required for Sam non X mode.
        ///							Random number returned by GetChallenge command : 8 Bytes or 16 Bytes
        ///							Key number : 2 bytes
        ///							key Version: 2 bytes</param>
        /// <param name="bInputLen">Length of bytes available in Input buffer.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU</param>
        /// <param name="pDataOut">Returns Rnd number PCD2 in sam non x mode. Nothing in S/W mode.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Authenticate the PCD (second part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoExternalAuthenticate ( byte[] pInput, byte bInputLen, byte bExtendedLenApdu, out byte[] pDataOut )
        {
            byte OutLen = 0;
            Status_t oStatus;

            pDataOut = new byte[255];
            oStatus = phalMfdfEVx_IsoExternalAuthenticate ( m_pDataParams, pInput, bInputLen, bExtendedLenApdu,
                pDataOut, ref OutLen );
            Array.Resize<byte> ( ref pDataOut, OutLen );

            return oStatus;
        }

        /// <summary>
        /// Iso Internal Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </summary>
        ///
        /// <param name="pInput">Input data should be as mentioned below.
        ///							Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES, 04 => 3k3DES, 09 => AES128
        ///							Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise.
        ///							Key number on card - 1 Byte: 0x0 to 0xD
        ///							Length of random number: 1 Byte
        ///							Random number Rpcd2 : 8 Bytes or 16 Bytes.
        ///							Key number : 2 bytes
        ///							key Version: 2 bytes</param>
        /// <param name="bInputLen">Length of bytes available in Input buffer.</param>
        /// <param name="pDataOut">RRPICC2 || RPCD2 after decryption in S/W mode. Nothing in Sam non x mode.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is not supported",
                Message = "Authenticate the PICC (third part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoInternalAuthenticate ( byte[] pInput, byte bInputLen, out byte[] pDataOut )
        {
            byte OutLen = 0;
            Status_t oStatus;
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOINTERNALAUTHENTICATE".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            pDataOut = new byte[255];
            oStatus = phalMfdfEVx_IsoInternalAuthenticate ( m_pDataParams, pInput, bInputLen, bExtendedLenApdu,
                pDataOut, ref OutLen );
            Array.Resize<byte> ( ref pDataOut, OutLen );

            return oStatus;
        }

        /// <summary>
        /// Iso Internal Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// <remarks>
        /// <b>THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.</b>
        /// </summary>
        ///
        /// <param name="pInput">Input data should be as mentioned below.
        ///							Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES, 04 => 3k3DES, 09 => AES128
        ///							Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise.
        ///							Key number on card - 1 Byte: 0x0 to 0xD
        ///							Length of random number: 1 Byte
        ///							Random number Rpcd2 : 8 Bytes or 16 Bytes.
        ///							Key number : 2 bytes
        ///							key Version: 2 bytes</param>
        /// <param name="bInputLen">Length of bytes available in Input buffer.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU</param>
        /// <param name="pDataOut">RRPICC2 || RPCD2 after decryption in S/W mode. Nothing in Sam non x mode.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Extended Lendth APDU is supported",
                Message = "Authenticate the PICC (third part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoInternalAuthenticate ( byte[] pInput, byte bInputLen, byte bExtendedLenApdu, out byte[] pDataOut )
        {
            byte OutLen = 0;
            Status_t oStatus;

            pDataOut = new byte[255];
            oStatus = phalMfdfEVx_IsoInternalAuthenticate ( m_pDataParams, pInput, bInputLen, bExtendedLenApdu, pDataOut,
                ref OutLen );
            Array.Resize<byte> ( ref pDataOut, OutLen );

            return oStatus;
        }

        /// <summary>
        /// Perform Iso authentication GetChallenge, External Authenticate & Internal Authenticate of a DESFire PICC
        ///
        /// For Software mode and Sam NonX mode, internally performs the three pass Iso authentication by calling
        /// GetChallenge
        /// External Authenticate
        /// Internal Authenticate
        /// Generates and stores the session key
        /// </summary>
        ///
        /// <param name="wKeyNo">DESFire key number or SAM Key entry number.</param>
        /// <param name="wKeyVer">Key version.</param>
        /// <param name="bKeyNoCard">Key number on card. 0x0 to 0xD.</param>
        /// <param name="bIsPICCkey">Is it PICC Master key ? 1 = YES.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an support command provided by Reader Library to perform all there commands in one go.",
                Message = "Perform Iso authentication GetChallenge, External Authenticate & Internal Authenticate of a DESFire PICC."
            )
        ]
        public Status_t IsoAuthenticate ( ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte bIsPICCkey )
        {
            return phalMfdfEVx_IsoAuthenticate ( m_pDataParams, wKeyNo, wKeyVer, bKeyNoCard, bIsPICCkey );
        }
#endif
        #endregion

        #region Originality Check
        /// <summary>
        /// Performs the originality check to verify the genuineness of PICC
        /// </summary>
        ///
        /// <param name="bAddr">Value is always 00. Present for forward compatibility reasons</param>
        /// <param name="pSignature">The plain 56 bytes originality signature of the PICC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                Message = "Retrieve the ECC originality check signature."
            )
        ]
        public Status_t ReadSign ( byte bAddr, out byte[] pSignature )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            Status_t oStatus = phalMfdfEVx_ReadSign ( m_pDataParams, bAddr, ref ppRxdata );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                pSignature = new byte[SIG_LENGTH];
                Marshal.Copy ( ppRxdata, pSignature, 0, SIG_LENGTH );
            }
            else
            {
                pSignature = null;
            }

            return oStatus;
        }
        #endregion

        #region MIFARE Classic
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Configure a mapping between one or more MIFARE Classic blocks and a DESFire FileType.StandardData,
        /// FileType.BackupData and FileType.Value file.
        /// </summary>
        ///
        /// <param name="bComOption">The communication mode that should be applied.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="wMFCLicenseMACKeyNo">Key number in key store for MFCLicenseMAC calculation.</param>
        /// <param name="wMFCLicenseMACKeyVer">Key version in key store for MFCLicenseMAC calculation.</param>
        /// <param name="bFileNo">The file number to be created.
        ///							ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							secondary application indicator.</param>
        /// <param name="bFileOption">The option to be enabled / disabled for the file.</param>
        /// <param name="aMFCBlockList">List of block numbers of the targeted blocks.</param>
        /// <param name="bRestoreSource">File number of the allowed restore source.</param>
        /// <param name="aMFCLicense">MIFARE Classic License. Should be as follows
        ///								BlockCount || BlockNr1 || BlockOption1 [|| BlockNr2 || BlockOption2 || ...
        ///								|| BlockNrN || BlockOptionN]</param>
        /// <param name="aMFCSectorSecrets">MIFARE Classic Sector Secrets. Should be as follows
        ///										MFCSectorSectrets1 [|| MFCSectorSectrets1 || ... || MFCSectorSectretsN]</param>
        /// <param name="aDivInput">Diversification input to diversify the Key.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Configure a mapping between one or more MIFARE Classic blocks and a DESFire FileType.StandardData, FileType.BackupData and FileType.Value file."
            )
        ]
        public Status_t CreateMFCMapping ( byte bComOption, ushort wMFCLicenseMACKeyNo, ushort wMFCLicenseMACKeyVer, byte bFileNo,
            byte bFileOption, byte[] aMFCBlockList, byte bRestoreSource, byte[] aMFCLicense, byte[] aMFCSectorSecrets, byte[] aDivInput )
        {
            /* Compute the MFCLicenseMAC */
            byte[] aInput = new byte[1 /* constant */];
            int dwInputLen = 0;

            /* Add the constant value to the Input buffer. */
            aInput[dwInputLen++] = 0x01;

            /* Add MFCLicense to Input buffer. */
            if ( aMFCLicense != null )
            {
                Array.Resize ( ref aInput, aMFCLicense.Length + dwInputLen );
                Array.Copy ( aMFCLicense, 0, aInput, dwInputLen, aMFCLicense.Length );
                dwInputLen += aMFCLicense.Length;
            }

            /* Add MFCSectorSecrets to Input buffer. */
            if ( aMFCSectorSecrets != null )
            {
                Array.Resize ( ref aInput, aMFCSectorSecrets.Length + dwInputLen );
                Array.Copy ( aMFCSectorSecrets, 0, aInput, dwInputLen, aMFCSectorSecrets.Length );
                dwInputLen += aMFCSectorSecrets.Length;
            }

            /* Frame the diversification option */
            ushort wOption = ( ushort ) ( ( aDivInput != null ) ? 0x01 /* CMAC diversification */ : 0xFFFF /* No Diversification. */);

            /* Compute the MAC. */
            byte[] aMFCLicenseMAC = new byte[8];
            Status_t oStatus = ComputeMFCLicenseMAC ( wOption, wMFCLicenseMACKeyNo, wMFCLicenseMACKeyVer, aInput, aDivInput, out aMFCLicenseMAC );
            if ( !oStatus.Equals ( new Status_t () ) )
                return oStatus;

            /* Call CreateMFCMapping command. */
            return phalMfdfEVx_CreateMFCMapping ( m_pDataParams, bComOption, bFileNo, bFileOption, aMFCBlockList,
                ( byte ) ( ( aMFCBlockList == null ) ? 0 : aMFCBlockList.Length ), bRestoreSource, aMFCLicense,
                ( byte ) ( ( aMFCLicense == null ) ? 0 : aMFCLicense.Length ), aMFCLicenseMAC );
        }
#endif

        /// <summary>
        /// Configure a mapping between one or more MIFARE Classic blocks and a DESFire FileType.StandardData,
        /// FileType.BackupData and FileType.Value file.
        /// </summary>
        ///
        /// <param name="bComOption">The communication mode that should be applied.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bFileNo">The file number to be created.
        ///							ORed with <see cref="NxpRdLibNet.alMfdfEVx.TargetApp.SECONDARY"/> to indicate
        ///							secondary application indicator.</param>
        /// <param name="bFileOption">The option to be enabled / disabled of the file.</param>
        /// <param name="aMFCBlockList">List of block numbers of the targeted blocks.</param>
        /// <param name="bRestoreSource">File number of the allowed restore source.</param>
        /// <param name="aMFCLicense">MIFARE Classic License. Should be as follows
        ///								BlockCount || BlockNr1 || BlockOption1 [|| BlockNr2 || BlockOption2 || ...
        ///								|| BlockNrN || BlockOptionN]</param>
        /// <param name="aMFCLicenseMAC">The computed MFCLicenseMAC. Should be 8 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "To Compute MFCLicenseMAC, ComputeMFCLicenseMAC method can be used.",
                Message = "Configure a mapping between one or more MIFARE Classic blocks and a DESFire FileType.StandardData, FileType.BackupData and FileType.Value file."
            )
        ]
        public Status_t CreateMFCMapping ( byte bComOption, byte bFileNo, byte bFileOption, byte[] aMFCBlockList, byte bRestoreSource,
            byte[] aMFCLicense, byte[] aMFCLicenseMAC )
        {
            /* Call CreateMFCMapping command. */
            return phalMfdfEVx_CreateMFCMapping ( m_pDataParams, bComOption, bFileNo, bFileOption, aMFCBlockList,
                ( byte ) ( ( aMFCBlockList == null ) ? 0 : aMFCBlockList.Length ), bRestoreSource, aMFCLicense,
                ( byte ) ( ( aMFCLicense == null ) ? 0 : aMFCLicense.Length ), aMFCLicenseMAC );
        }

        /// <summary>
        /// Restores the value from one FileType.Value file by transfering it into another File-Type.Value file.
        /// </summary>
        ///
        /// <param name="bCommOption">The communication mode that should be applied.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.MAC</param>
        /// <param name="bTargetFileNo">File number of the TargetFile to be updated.</param>
        /// <param name="bSourceFileNo">File number of the SourceFile to be restored.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Restores the value from one FileType.Value file by transfering it into another FileType.Value file."
            )
        ]
        public Status_t RestoreTransfer ( byte bCommOption, byte bTargetFileNo, byte bSourceFileNo )
        {
            return phalMfdfEVx_RestoreTransfer ( m_pDataParams, bCommOption, bTargetFileNo, bSourceFileNo );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Configure restriction of certain update operations over the MIFARE Classic interface.
        /// </summary>
        ///
        /// <param name="bComOption">The communication mode that should be applied.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="wMFCLicenseMACKeyNo">Key number in key store for MFCLicenseMAC calculation.</param>
        /// <param name="wMFCLicenseMACKeyVer">Key version in key store for MFCLicenseMAC calculation.</param>
        /// <param name="bOption">Configuration Option. Define length and content of the Data parameter.</param>
        /// <param name="aMFCConfig">Data content depends on option values.</param>
        /// <param name="aMFCLicense">MIFARE Classic License. Should be as follows
        ///								BlockCount || BlockNr1 || BlockOption1 [|| BlockNr2 || BlockOption2 || ...
        ///								|| BlockNrN || BlockOptionN]</param>
        /// <param name="aMFCSectorSecrets">MIFARE Classic Sector Secrets. Should be as follows
        ///										MFCSectorSectrets1 [|| MFCSectorSectrets1 || ... || MFCSectorSectretsN]</param>
        /// <param name="aDivInput">Diversification input to diversify the Key.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                Message = "Configure restriction of certain update operations over the MIFARE Classic interface."
            )
        ]
        public Status_t RestrictMFCUpdate ( ushort wMFCLicenseMACKeyNo, ushort wMFCLicenseMACKeyVer, byte bOption,
            byte[] aMFCConfig, byte[] aMFCLicense, byte[] aMFCSectorSecrets, byte[] aDivInput )
        {
            /* Compute the MFCLicenseMAC */
            byte[] aInput = new byte[1 /* constant */];
            int dwInputLen = 0;

            /* Add the constant value to the Input buffer. */
            aInput[dwInputLen++] = 0x01;

            /* Add MFCLicense to Input buffer. */
            if ( aMFCLicense != null )
            {
                Array.Resize ( ref aInput, aMFCLicense.Length + dwInputLen );
                Array.Copy ( aMFCLicense, 0, aInput, dwInputLen, aMFCLicense.Length );
                dwInputLen += aMFCLicense.Length;
            }

            /* Add MFCSectorSecrets to Input buffer. */
            if ( aMFCSectorSecrets != null )
            {
                Array.Resize ( ref aInput, aMFCSectorSecrets.Length + dwInputLen );
                Array.Copy ( aMFCSectorSecrets, 0, aInput, dwInputLen, aMFCSectorSecrets.Length );
                dwInputLen += aMFCSectorSecrets.Length;
            }

            /* Frame the diversification option */
            ushort wOption = ( ushort ) ( ( aDivInput != null ) ? 0x01 /* CMAC diversification */ : 0xFFFF /* No Diversification. */);

            /* Compute the MAC. */
            byte[] aMFCLicenseMAC = new byte[8];
            Status_t oStatus = ComputeMFCLicenseMAC ( wOption, wMFCLicenseMACKeyNo, wMFCLicenseMACKeyVer, aInput, aDivInput, out aMFCLicenseMAC );
            if ( !oStatus.Equals ( new Status_t () ) )
                return oStatus;

            return phalMfdfEVx_RestrictMFCUpdate ( m_pDataParams, bOption, aMFCConfig, ( byte ) ( ( aMFCConfig == null ) ? 0 : aMFCConfig.Length ),
                aMFCLicense, ( byte ) ( ( aMFCLicense == null ) ? 0 : aMFCLicense.Length ), aMFCLicenseMAC );
        }

        /// <summary>
        /// Configure restriction of certain update operations over the MIFARE Classic interface.
        /// </summary>
        ///
        /// <param name="bComOption">The communication mode that should be applied.
        ///							NxpRdLibNet.alMfdfEVx.CommOption.PLAIN
        ///							NxpRdLibNet.alMfdfEVx.CommOption.FULL</param>
        /// <param name="bOption">Configuration Option. Define length and content of the Data parameter.</param>
        /// <param name="aMFCConfig">Data content depends on option values.</param>
        /// <param name="aMFCLicense">MIFARE Classic License. Should be as follows
        ///								BlockCount || BlockNr1 || BlockOption1 [|| BlockNr2 || BlockOption2 || ...
        ///								|| BlockNrN || BlockOptionN]</param>
        /// <param name="aMFCLicenseMAC">The computed MFCLicenseMAC. Should be 8 bytes.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "To Compute MFCLicenseMAC, ComputeMFCLicenseMAC method can be used.",
                Message = "Configure restriction of certain update operations over the MIFARE Classic interface."
            )
        ]
        public Status_t RestrictMFCUpdate ( byte bOption, byte[] aMFCConfig, byte[] aMFCLicense, byte[] aMFCLicenseMAC )
        {
            return phalMfdfEVx_RestrictMFCUpdate ( m_pDataParams, bOption, aMFCConfig, ( byte ) ( ( aMFCConfig == null ) ? 0 : aMFCConfig.Length ),
                aMFCLicense, ( byte ) ( ( aMFCLicense == null ) ? 0 : aMFCLicense.Length ), aMFCLicenseMAC );
        }
#endif
        #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Post Delivery Configuration
        /// <summary>
        /// Perform Post Delivery Authentication.
        /// </summary>
        ///
        /// <param name="bRfu">The RFU information. Its zero by default.</param>
        /// <param name="bKeyNoCard">PICC Key number to be used for authentication.</param>
        /// <param name="wKeyNo">Key number to used from software or hardware keystore.</param>
        /// <param name="wKeyVer">Key version to used from software or hardware keystore.</param>
        /// <param name="bUpgradeInfo">The upgrade info input.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "Activating MFC Feature and Extending the memory are supported.",
                Message = "Mutual authentication for upgrading the card via post-delivery configuration."
            )
        ]
        public Status_t AuthenticatePDC ( byte bRfu, byte bKeyNoCard, byte wKeyNo, ushort wKeyVer, byte bUpgradeInfo )
        {
            return phalMfdfEVx_AuthenticatePDC ( m_pDataParams, bRfu, bKeyNoCard, wKeyNo, wKeyVer, bUpgradeInfo );
        }
        #endregion
#endif

        #region Miscellaneous
        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///							NxpRdLibNet.alMfdfEVx.Config.ADDITIONAL_INFO
        ///							NxpRdLibNet.alMfdfEVx.Config.WRAPPED_MODE
        ///							NxpRdLibNet.alMfdfEVx.Config.SHORT_LENGTH_APDU</param>
        /// <param name="pValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Get the configuration like WrappedMode, ShortLengthAPDU."
            )
        ]
        public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
        {
            return phalMfdfEVx_GetConfig ( m_pDataParams, wConfig, ref pValue ); ;
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///							NxpRdLibNet.alMfdfEVx.Config.ADDITIONAL_INFO
        ///							NxpRdLibNet.alMfdfEVx.Config.WRAPPED_MODE
        ///							NxpRdLibNet.alMfdfEVx.Config.SHORT_LENGTH_APDU</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Set the configuration like WrappedMode, ShortLengthAPDU."
            )
        ]
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalMfdfEVx_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE, MIFARE_Products.DESFIRE_EV1, MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Reset the authentication state. Clears existing session keys and set the state to not authenticated."
            )
        ]
        public Status_t ResetAuthentication ()
        {
            return phalMfdfEVx_ResetAuthentication ( m_pDataParams );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		KeyNoTMACKey should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option.
        ///							NxpRdLibNet.alMfdfEVx.DivOption.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.alMfdfEVx.DivOption.DIVERSIFICATION_ON</param>
        /// <param name="wKeyNoTMACKey">Key number to be used from software or hardware keystore.</param>
        /// <param name="wKeyVerTMACKey">Key version to be used from software or hardware keystore.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and should be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pTMI">Transaction MAC Input.</param>
        /// <param name="pTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Computed the Transaction MAC input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput,
            byte[] pTMC, byte[] pUid, byte[] pTMI, out byte[] pTMV )
        {
            pTMV = new byte[8];
            return phalMfdfEVx_CalculateTMV ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pTMI,
                ( uint ) ( ( pTMI == null ) ? 0 : pTMI.Length ), pTMV );
        }

        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		KeyNoTMACKey and RamKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Computed the Transaction MAC input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] aTMC, byte[] aUid, byte[] aTMI, out byte[] aTMV )
        {
            aTMV = new byte[8];
            return phalMfdfEVx_CalculateTMV ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null,
                0, aTMC, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aTMI, ( uint ) ( ( aTMI == null ) ? 0 : aTMI.Length ), aTMV );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		KeyNoTMACKey should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option.
        ///							NxpRdLibNet.alMfdfEVx.DivOption.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.alMfdfEVx.DivOption.DIVERSIFICATION_ON</param>
        /// <param name="wKeyNoTMACKey">Key number to be used from software or hardware keystore.</param>
        /// <param name="wKeyVerTMACKey">Key version to be used from software or hardware keystore.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and should be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Decrypted the reader ID as performed on the PICC."
            )
        ]
        public Status_t DecryptReaderID ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput,
            byte[] pTMC, byte[] pUid, byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfdfEVx_DecryptReaderID ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pEncTMRI, pTMRIPrev );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		KeyNoTMACKey and RamKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Decrypted the reader ID as performed on the PICC."
            )
        ]
        public Status_t DecryptReaderID ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer, byte[] pTMC, byte[] pUid,
            byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfdfEVx_DecryptReaderID ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null, 0, pTMC, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pEncTMRI, pTMRIPrev );
        }

        /// <summary>
        /// Generates Encrypted Key for Delegated application management
        ///	While using this command in SAM mode make sure the below points are taken care
        ///		KeyNoAppDAMDefault should have AllowDumpSecretKey enabled.
        ///		KeyNoDAMEnc should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wKeyNoDAMEnc">Key number in key store of DAM Encryption key.</param>
        /// <param name="wKeyVerDAMEnc">Key version in key store of DAM Encryption key.</param>
        /// <param name="wKeyNoAppDAMDefault">Key number in key store of DAM default key.</param>
        /// <param name="wKeyVerAppDAMDefault">Key version in key store of DAM default key.</param>
        /// <param name="bAppDAMDefaultKeyVer">DAM Default Key version. Value should be valid for AES key. Ignored for other key types.</param>
        /// <param name="pDAMEncKey">Encrypted DAM Default key data.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generates Encrypted Key for Delegated application management."
            )
        ]
        public Status_t GenerateDAMEncKey ( ushort wKeyNoDAMEnc, ushort wKeyVerDAMEnc, ushort wKeyNoAppDAMDefault, ushort wKeyVerAppDAMDefault,
            byte bAppDAMDefaultKeyVer, out byte[] pDAMEncKey )
        {
            pDAMEncKey = new byte[32];
            return phalMfdfEVx_GenerateDAMEncKey ( m_pDataParams, wKeyNoDAMEnc, wKeyVerDAMEnc, wKeyNoAppDAMDefault,
                wKeyVerAppDAMDefault, bAppDAMDefaultKeyVer, pDAMEncKey );
        }

        /// <summary>
        /// Generate MAC for Delegated application or Delete Application.
        /// While using this command in SAM mode make sure the below points are taken care.
        ///		KeyNoDAMMAC should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_INFO_NOT_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.FILE_ID_AVAILABLE
        ///							NxpRdLibNet.alMfdfEVx.ISO.DF_NAME_AVAILABLE
        ///
        ///						  To be ored with the above ones.
        ///							NxpRdLibNet.alMfdfEVx.DAMMAC.CREATE_APPLICATION
        ///							NxpRdLibNet.alMfdfEVx.DAMMAC.DELETE_APPLICATION</param>
        /// <param name="wKeyNoDAMMAC">Key number in key store of DAM MAC key.</param>
        /// <param name="wKeyVerDAMMAC">Key version in key store of DAM MAC key.</param>
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        /// <param name="pDamParams">The DAM information. Should as mentioned below.
        ///								Byte0 = DAM Slot No [LSB]
        ///								Byte1 = DAM Slot No [MSB]
        ///								Byte2 = DAM Slot Version
        ///								Byte3 = Quota Limit [LSB]
        ///								Byte4 = Quota Limit [MSB]</param>
        /// <param name="bKeySettings1">Application Key settings.</param>
        /// <param name="bKeySettings2">Additional key settings.</param>
        /// <param name="bKeySettings3">Additional key settings.</param>
        /// <param name="pKeySetValues">The Key set values for the application. Should as mentioned below.
        ///									Byte0 = AKS ver
        ///									Byte1 = Number of Keysets
        ///									Byte2 = MaxKeysize
        ///									Byte3 = Application KeySet Settings.</param>
        /// <param name="pISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="pISODFName">ISO DF Name to be used. Will be 16 bytes and can also be NULL based on the option.</param>
        /// <param name="pEncK">Encrypted initial application key. 32 bytes long.</param>
        /// <param name="pDAMMAC">8 byte DAMMAC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method can be used for DAMMAC generation for DeleteApplication by passing NULL for the not supported parameters.",
                Message = "Generate DAMMAC for Delegated applicatiom management and Delete Application."
            )
        ]
        public Status_t GenerateDAMMAC ( byte bOption, ushort wKeyNoDAMMAC, ushort wKeyVerDAMMAC, byte[] pAid, byte[] pDamParams, byte bKeySettings1,
            byte bKeySettings2, byte bKeySettings3, byte[] pKeySetValues, byte[] pISOFileId, byte[] pISODFName, byte[] pEncK, out byte[] pDAMMAC )
        {
            pDAMMAC = new byte[8];
            return phalMfdfEVx_GenerateDAMMAC ( m_pDataParams, bOption, wKeyNoDAMMAC, wKeyVerDAMMAC, pAid,
                pDamParams, bKeySettings1, bKeySettings2, bKeySettings3, pKeySetValues, pISOFileId,
                pISODFName, ( byte ) ( ( pISODFName == null ) ? 0 : pISODFName.Length ),
                pEncK, pDAMMAC );
        }

        /// <summary>
        /// Generate MAC for Delete application.
        /// While using this command in SAM mode make sure the below points are taken care.
        ///		KeyNoDAMMAC should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wKeyNoDAMMAC">Key number in key store of DAM MAC key.</param>
        /// <param name="wKeyVerDAMMAC">Key version in key store of DAM MAC key.</param>
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///					   If application 01 need to be created then the Aid will be 01 00 00.</param>
        /// <param name="pDAMMAC">8 byte DAMMAC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generate DAMMAC for Delete Application."
            )
        ]
        public Status_t GenerateDAMMAC ( ushort wKeyNoDAMMAC, ushort wKeyVerDAMMAC, byte[] pAid, out byte[] pDAMMAC )
        {
            return GenerateDAMMAC ( 0x80, wKeyNoDAMMAC, wKeyVerDAMMAC, pAid, null, 0, 0, 0, null,
                null, null, null, out pDAMMAC );
        }

        /// <summary>
        /// Generates DAMMAC for Set Configuration with option 0x06 for Delegated Application
        /// </summary>
        ///
        /// <param name="wKeyNoDAMMAC">Key number in key store of DAM MAC Key</param>
        /// <param name="wKeyVerDAMMAC">Key version in key store of DAM MAC Key</param>
        /// <param name="pOldISODFName">This means already created delegated app ISO DF Name. Maximum 16 bytes</param>
        /// <param name="pNewISODFName">This means new delegated app ISO DF Name which will replace the existing one. Maximum 16 bytes</param>
        /// <param name="pDAMMAC">Generated 8 bytes DAM MAC for setconfig option 0x06</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Generates DAMMAC for Set Configuration with option 0x06 for Delegated Application."
            )
        ]
        public Status_t GenerateDAMMACSetConfig ( ushort wKeyNoDAMMAC, ushort wKeyVerDAMMAC, byte[] pOldISODFName,
            byte[] pNewISODFName, out byte[] pDAMMAC )
        {
            pDAMMAC = new byte[8];
            return phalMfdfEVx_GenerateDAMMACSetConfig ( m_pDataParams, wKeyNoDAMMAC, wKeyVerDAMMAC,
                ( byte ) ( ( pOldISODFName == null ) ? 0 : pOldISODFName.Length ), pOldISODFName,
                ( byte ) ( ( pNewISODFName == null ) ? 0 : pNewISODFName.Length ), pNewISODFName,
                pDAMMAC );
        }

        /// <summary>
        /// Compute the MFCLicenseMAC for <see cref="NxpRdLibNet.alMfdfEVx.Generic.CreateMFCMapping"/> and
        /// <see cref="NxpRdLibNet.alMfdfEVx.Generic.RestrictMFCUpdate"/>command.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		MFCLicenseMACKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wOption">Diversification Option. One of the below values.
        ///							NxpRdLibNet.alMfdfEVx.DivOption.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.alMfdfEVx.DivOption.CMAC</param>
        /// <param name="wMFCLicenseMACKeyNo">Key number in key store for MFCLicenseMAC calculation.</param>
        /// <param name="wMFCLicenseMACKeyVer">Key version in key store for MFCLicenseMAC calculation.</param>
        /// <param name="aInput">The input for which the MAC should be computed. Input should be as follows.
        ///							0x01 || MFCLicense || MFCSectorSecrets</param>
        /// <param name="aDivInput">Diversification input to diversify the Key.</param>
        /// <param name="aMFCLicenseMAC">The computed MFCLicense MAC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Compute the MFCLicenseMAC."
            )
        ]
        public Status_t ComputeMFCLicenseMAC ( ushort wOption, ushort wMFCLicenseMACKeyNo, ushort wMFCLicenseMACKeyVer,
            byte[] aInput, byte[] aDivInput, out byte[] aMFCLicenseMAC )
        {
            aMFCLicenseMAC = new byte[8];
            return phalMfdfEVx_ComputeMFCLicenseMAC ( m_pDataParams, wOption, wMFCLicenseMACKeyNo, wMFCLicenseMACKeyVer, aInput,
                ( ushort ) ( ( aInput != null ) ? aInput.Length : 0 ), aDivInput, ( byte ) ( ( aDivInput != null ) ? aDivInput.Length : 0 ),
                aMFCLicenseMAC );
        }

        /// <summary>
        /// Calculate MAC SDM
        /// While using this command in SAM mode make sure the below points are taken care
        ///		SDMMacKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///								NxpRdLibNet.alMfdfEVx.SDMOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfdfEVx.SDMOption.RDCTR_PRESENT</param>
        /// <param name="wSDMFileReadKeyNo">Key number to be used from software or hardware keystore.</param>
        /// <param name="wSDMFileReadKeyVer">Key version to be used from software or hardware keystore.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pIndata">Data read out of PICC from SDMMacInputoffset until SDMMACOffset.
        ///							NOTE: If the data is having SDM Encrypted File data any where inside
        ///								  the start of SDMMacInputoffset, caller no need to decrypt the
        ///								  information. The data received from PICC should as is passed
        ///								  starting from SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="pRespMac">Computed SDM MAC.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Calculate Secure Dynamic MAC."
            )
        ]
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wSDMFileReadKeyNo, ushort wSDMFileReadKeyVer, byte[] pUid,
            byte[] pSDMReadCtr, byte[] pIndata, out byte[] pRespMac )
        {
            pRespMac = new byte[8];
            return phalMfdfEVx_CalculateMACSDM ( m_pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer, 0, 0, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pIndata,
                ( ushort ) ( pIndata != null ? pIndata.Length : 0 ), pRespMac );
        }

        /// <summary>
        /// Calculate MAC for SDM information.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		SDMMacKeyNo and RamKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfIdentity.SDMOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfIdentity.SDMOption.RDCTR_PRESENT</param>
        /// <param name="wSDMFileReadKeyNo">Key number in SW key store key.</param>
        /// <param name="wSDMFileReadKeyVer">Key version in SW key store key.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pIndata">Data read out of PICC from SDMMacInputoffset for ReadLength bytes</param>
        /// <param name="wInDataLen">Length of bytes available in InData buffer.</param>
        /// <param name="pRespMac">The computed MAC information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Calculate Secure Dynamic MAC."
            )
        ]
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wSDMFileReadKeyNo, ushort wSDMFileReadKeyVer, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] pUid, byte[] pSDMReadCtr, byte[] pIndata, ushort wInDataLen, out byte[] pRespMac )
        {
            pRespMac = new byte[8];
            return phalMfdfEVx_CalculateMACSDM ( m_pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer, wRamKeyNo, wRamKeyVer,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pIndata, wInDataLen, pRespMac );
        }

        /// <summary>
        /// Decrypt SDM enciphered data.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		EncKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///								NxpRdLibNet.alMfdfEVx.SDMOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfdfEVx.SDMOption.RDCTR_PRESENT</param>
        /// <param name="wEncKeyNo">Key number to be used from software or hardware keystore.</param>
        /// <param name="wEncKeyVer">Key version to be used from software or hardware keystore.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pEncdata">Input Enciphered data.
        ///						   NOTE:
        ///								- Caller has to pass only the Enciphered data
        ///								  which is between SDMEncOffset for SDMEncLength bytes.
        ///								- Caller has remove the ASCII formated data before passing to this interface.
        ///									For example, the Enciphered data is 32 bytes in length which
        ///									is in ASCII format, caller has to convert it to non ASCII format.
        ///									Apon converting, the length will be 16 bytes.</param>
        /// <param name="pPlainData">The decrypted SDMFile data.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Decrypts SDM Encrypted File data."
            )
        ]
        public Status_t DecryptSDMENCFileData ( ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer, byte[] pUid,
            byte[] pSDMReadCtr, byte[] pEncdata, out byte[] pPlainData )
        {
            pPlainData = new byte[( pEncdata == null ) ? 0 : pEncdata.Length];

            Status_t oStatus = phalMfdfEVx_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wEncKeyNo, wEncKeyVer, 0 , 0 ,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pEncdata, ( byte ) ( ( pEncdata == null ) ? 0 : pEncdata.Length ),
                pPlainData );

            if ( !oStatus.Equals ( new Status_t () ) )
                pPlainData = null;

            return oStatus;
        }

        /// <summary>
        /// Decrypt the SDM File data information.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		EncKeyNo and RamKeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered. Can be ORed.
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.VCUID_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.RDCTR_PRESENT
        ///								NxpRdLibNet.alMfNtag42XDna.ChangeFileSettingsOption.SDM_ENC_FILE_DATA_PRESENT</param>
        /// <param name="wEncKeyNo">Key number in keystore of Software.</param>
        /// <param name="wEncKeyVer">Key version in keystore of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="pEncdata">Input Enciphered data. Caller has to pass only the Enciphered data which is between
        ///						   SDMEncOffset for SDMEncLength bytes.</param>
        /// <param name="pPlainData">The decrypted SDM ENC file data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Decrypts SDM Encrypted File data."
            )
        ]
        public Status_t DecryptSDMENCFileData ( ushort bSdmOption, ushort wEncKeyNo, ushort wEncKeyVer, ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid,
            byte[] pSDMReadCtr, byte[] pEncdata, out byte[] pPlainData )
        {
            pPlainData = new byte[( pEncdata == null ) ? 0 : pEncdata.Length];

            Status_t oStatus = phalMfdfEVx_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wEncKeyNo, wEncKeyVer, wRamKeyNo, wRamKeyVer,
                pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pSDMReadCtr, pEncdata, ( byte ) ( ( pEncdata == null ) ? 0 : pEncdata.Length ),
                pPlainData );

            if ( !oStatus.Equals ( new Status_t () ) )
                pPlainData = null;

            return oStatus;
        }

        /// <summary>
        /// Decrypt SDM PICC Data.
        /// While using this command in SAM mode make sure the below points are taken care
        ///		KeyNo should be off Offline_Crypto Keyclass.
        /// </summary>
        ///
        /// <param name="wEncKeyNo">Key number to be used from software or hardware keystore.</param>
        /// <param name="wEncKeyVer">Key version to be used from software or hardware keystore.</param>
        /// <param name="pEncdata">Input Enciphered data.
        ///						   NOTE: Caller has remove the ASCII formated data before passing to this interface.
        ///									For example, the Enciphered data is 32 bytes in length which
        ///									is in ASCII format, caller has to convert it to non ASCII format.
        ///									Apon converting, the length will be 16 bytes.</param>
        /// <param name="pPlainData">The decrypted SDM PICC Data.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Decrypts SDM Encrypted PICC data."
            )
        ]
        public Status_t DecryptSDMPICCData ( ushort wEncKeyNo, ushort wEncKeyVer, byte[] pEncdata, out byte[] pPlainData )
        {
            byte[] aPlainDataTmp = null;
            if ( pEncdata != null )
                aPlainDataTmp = new byte[pEncdata.Length];

            pPlainData = null;
            Status_t oStatus = phalMfdfEVx_DecryptSDMPICCData(m_pDataParams, wEncKeyNo, wEncKeyVer, pEncdata,
                (byte)((pEncdata == null) ? 0 : pEncdata.Length), aPlainDataTmp);

            if ( aPlainDataTmp != null )
                pPlainData = ( byte[] ) aPlainDataTmp.Clone ();
            return oStatus;
        }
#endif

        /// <summary>
        /// This is a utility API which sets the VCA structure in DESFire EVx structure params
        /// </summary>
        ///
        /// <param name="pAlVCADataParams">Pointer to VCA AL parameter structure.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "This is a utility API which sets the VCA structure in DESFire EVx structure params."
            )
        ]
        public Status_t SetVCAParams ( IntPtr pAlVCADataParams )
        {
            return phalMfdfEVx_SetVCAParams ( m_pDataParams, pAlVCADataParams );
        }
        #endregion
        #endregion

        #region Memory Maping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    [ClassInterface ( ClassInterfaceType.AutoDual )]
    public class Sw : alMfdfEVx.Generic
    {
        #region Constants
        private const byte AID_LENGTH = 3;
        private const byte IV_LENGTH = 16;
        private const byte SESSION_KEY_LENGTH = 24;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DESFire EVx Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary> Pointer to the parameter structure of the CryptoMAC. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoMACDataParams;

            /// <summary> Pointer to the HAL parameters structure. </summary>
            public IntPtr pHalDataParams;

            /// <summary> Session key for this authentication. </summary>
            public fixed byte bSessionKey[SESSION_KEY_LENGTH];

            /// <summary> Key number against which this authentication is done. </summary>
            public byte bKeyNo;

            /// <summary> Max size of IV can be 16 bytes. </summary>
            public fixed byte bIv[IV_LENGTH];

            /// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA), etc.... </summary>
            public byte bAuthMode;

            /// <summary> Command code. This will be used for differentiating the comman error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Aid of the currently selected application. </summary>
            public fixed byte pAid[AID_LENGTH];

            /// <summary> DES,3DES, 3K3DES or AES. </summary>
            public byte bCryptoMethod;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. </summary>
            public byte bWrappedMode;

            /// <summary> 2 Byte CRC initial value in Authenticate mode. </summary>
            public ushort wCrc;

            /// <summary> 4 Byte CRC initial value in 0x1A, 0xAA mode. </summary>
            public uint dwCrc;

            /// <summary> Specific error codes for DESFire generic errors or To get the response length of some commands. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Amount of data to be read. Required for Enc read to verify CRC. </summary>
            public uint dwPayLoadLen;

            /// <summary> Command count within transaction. </summary>
            public ushort wCmdCtr;

            /// <summary> Transaction Identifier. </summary>
            public fixed byte bTI[4];

            /// <summary> Authentication MAC key for the session. </summary>
            public fixed byte bSesAuthMACKey[16];

            /// <summary> Buffer containing unprocessed bytes for read mac answer stream. </summary>
            public fixed byte pUnprocByteBuff[16];

            /// <summary> Amount of data in the pUnprocByteBuff. </summary>
            public byte bNoUnprocBytes;

            /// <summary> Buffer to store last Block of encrypted data in case of chaining. </summary>
            public fixed byte bLastBlockBuffer[16];

            /// <summary> Last Block Buffer Index. </summary>
            public byte bLastBlockIndex;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Parameter for force set Short Length APDU in case of BIG ISO read. </summary>
            public byte bShortLenApdu;

            /// <summary> Pointer to the parameter structure for Virtual Card. </summary>
            public IntPtr pVCADataParams;

            /// <summary> Complete UID of the PICC. This will be updated when <see cref="NxpRdLibNet.alMfdfEVx.Generic.GetCardUID"/> interface is called. </summary>
            public fixed byte bUid[15];

            /// <summary> Length of bytes available in bUid buffer. </summary>
            public byte bUidLength;

            /// <summary> Get / Set the SDM availability in the file. </summary>
            public byte bSDMStatus;

            /// <summary> Get / Set the return of Fabrication Identifier. When Set will be utilized during GetVersion command exchange. </summary>
            public byte bReturn_FabID;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams,
            IntPtr pKeyStoreDataParams, IntPtr pCryptoDataParamsEnc, IntPtr pCryptoRngDataParams, IntPtr pCryptoMACDataParams,
            IntPtr pTMIDataParams, IntPtr pVCADataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE DESFire EVx software component.
        /// </summary>
        ///
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a CryptoMAC component context.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to a TMI component.</param>
        /// <param name="pAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="pHalparams">Pointer to the HAL parameters structure.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc,
            CryptoSym.Generic pCryptoMAC, CryptoRng.Generic pCryptoRng, NxpRdLibNet.TMIUtils pTMIUtils,
            alVca.Generic pAlVcaParams, Hal.Generic pHalparams )
        {
            return phalMfdfEVx_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pCryptoEnc != null ) ? pCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( pCryptoMAC != null ) ? pCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( pCryptoRng != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ( pTMIUtils != null ) ? pTMIUtils.m_pDataParams : IntPtr.Zero,
                ( pAlVcaParams != null ) ? pAlVcaParams.m_pDataParams : IntPtr.Zero,
                ( pHalparams != null ) ? pHalparams.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE DESFire EVx software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a CryptoMAC component context.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to a TMI component.</param>
        /// <param name="pAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="pHalparams">Pointer to the HAL parameters structure.</param>
        ///
        /// <returns></returns>
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore,
            CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMAC, CryptoRng.Generic pCryptoRng,
            NxpRdLibNet.TMIUtils pTMIUtils, alVca.Generic pAlVcaParams, Hal.Generic pHalparams )
        {
            return phalMfdfEVx_Sw_Init (
                ref m_DataParamsInt[0],
                ( ushort ) wDataParamSize,
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pCryptoEnc != null ) ? pCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( pCryptoMAC != null ) ? pCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( pCryptoRng != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ( pTMIUtils != null ) ? pTMIUtils.m_pDataParams : IntPtr.Zero,
                ( pAlVcaParams != null ) ? pAlVcaParams.m_pDataParams : IntPtr.Zero,
                ( pHalparams != null ) ? pHalparams.m_pDataParams : IntPtr.Zero );
        }
#endif

        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public byte[] pAid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }

        public byte[] bIv
        {
            get
            {
                byte[] bValue = new byte[IV_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < IV_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bIv[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > IV_LENGTH )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->bIv[i] = value[i];
                        }
                    }
                }
            }
        }

        public byte[] bSessionKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bSessionKey[i];
                        }
                    }
                }
                return bValue;
            }
        }

        public ushort wCmdCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->wCmdCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->wCmdCtr = value;
                    }
                }
            }
        }

        public byte[] TI
        {
            get
            {
                byte[] bValue = new byte[4];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 4; i++ )
                        {
                            bValue[i] = pDataParams->bTI[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 4 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->bTI[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam_NonX
    #region SAM AV3
    /// <summary>
    /// Class for SAM AV3 S(NonX) Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_NonX : alMfdfEVx.Generic
    {
        #region Constants
        /// <summary>
        /// DESFire EVx Native, Native wrapped and Iso7816 custom error code values.
        /// </summary>
        public enum Error : byte
        {
            #region DESFire EVx native and Iso7816 wrapped error codes
            /// <summary>MF DF Response - No changes done to backup files.</summary>
            NO_CHANGES = ( CustomCodes.ERROR_BEGIN ),

            /// <summary>MF DF Response - Insufficient NV-Memory.</summary>
            OUT_OF_EEPROM_ERROR = ( CustomCodes.ERROR_BEGIN + 1 ),

            /// <summary>MF DF Invalid key number specified.</summary>
            NO_SUCH_KEY = ( CustomCodes.ERROR_BEGIN + 2 ),

            /// <summary>MF DF Current configuration/status does not allow the requested command.</summary>
            PERMISSION_DENIED = ( CustomCodes.ERROR_BEGIN + 3 ),

            /// <summary>MF DF Requested AID not found on PICC.</summary>
            APPLICATION_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 4 ),

            /// <summary>MF DF Attempt to read/write data from/to beyond the files/record's limits.</summary>
            BOUNDARY_ERROR = ( CustomCodes.ERROR_BEGIN + 5 ),

            /// <summary>MF DF Previous cmd not fully completed. Not all frames were requested or provided by the PCD.</summary>
            COMMAND_ABORTED = ( CustomCodes.ERROR_BEGIN + 6 ),

            /// <summary>MF DF File/Application with same number already exists.</summary>
            DUPLICATE = ( CustomCodes.ERROR_BEGIN + 8 ),

            /// <summary>MF DF Specified file number does not exist.</summary>
            FILE_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 9 ),

            /// <summary>MF DF Parameter value error returned by PICC.</summary>
            PARAMETER_ERROR = ( CustomCodes.ERROR_BEGIN + 11 ),

            /// <summary>MF DF ISO 7816 Generic error. Check Additional Info.</summary>
            CMD_INVALID = ( CustomCodes.ERROR_BEGIN + 14 ),

            /// <summary>MF DF illegal command code.</summary>
            ILLEGAL_COMMAND_CODE = ( CustomCodes.ERROR_BEGIN + 15 ),

            /// <summary>MF DF integrity error.</summary>
            INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 16 ),

            /// <summary>MF DF memory error.</summary>
            MEMORY_ERROR = ( CustomCodes.ERROR_BEGIN + 17 ),
            #endregion

            #region DESFire EVx Iso7816 error codes
            /// <summary>ISO7816 custom response code for memory failure.</summary>
            ISO7816_MEMORY_FAILURE = ( CustomCodes.ERROR_BEGIN + 18 ),

            /// <summary>ISO7816 custom response code for wrong length, no further indication.</summary>
            ISO7816_WRONG_LENGTH = ( CustomCodes.ERROR_BEGIN + 19 ),

            /// <summary>ISO7816 custom response code for security status not satisfied.</summary>
            ISO7816_SECURITY_STAT_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 20 ),

            /// <summary>ISO7816 custom response code for conditions of use not satisfied.</summary>
            ISO7816_CONDITIONS_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 21 ),

            /// <summary>ISO7816 custom response code for file or application not found.</summary>
            ISO7816_FILE_APPLICATION_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 22 ),

            /// <summary>ISO7816 custom response code for incorrect parameters P1-P2.</summary>
            ISO7816_INCORRECT_PARAM_P1_P2 = ( CustomCodes.ERROR_BEGIN + 23 ),

            /// <summary>ISO7816 custom response code for Lc inconsistent with parameter P1-P2.</summary>
            ISO7816_LC_INCONSISTENT = ( CustomCodes.ERROR_BEGIN + 24 ),

            /// <summary>ISO7816 custom response code for wrong LE field.</summary>
            ISO7816_WRONG_LE = ( CustomCodes.ERROR_BEGIN + 25 ),

            /// <summary>ISO7816 custom response code for instruction code not supported or invalid.</summary>
            ISO7816_INS_CODE_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 26 )
            #endregion
        }

        /// <summary>
        /// DESFire EVx AuthType, AuthMode, Key Derivation, Key Selection and Key Diversification options
        /// for Cmd.SAM_AuthenticatePICC and Cmd.SAM_IsoAuthenticatePICC
        /// </summary>
        public enum AuthenticateOption : byte
        {
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC, Bit 7 and Bit 6,
            /// Key diversification & authentication mode selection, EV2 authentication mode
            /// </summary>
            AUTH_MODE_D40 = 0x00,
            AUTH_MODE_EV1 = 0x00,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC, Bit 5,
            /// Key derivation info
            /// </summary>
            KDF_DISABLED = 0x20,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 4 and Bit 3,
            /// Key Diversification Mode
            /// </summary>
            KDF_AV1 = 0x00,
            KDF_AV1_DOUBLE_ENCRYPTION = 0x00,
            KDF_AV1_SINGLE_ENCRYPTION = 0x08,
            KDF_AV2 = 0x10,
            KDF_RFU = 0x18,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 1,
            /// Key selection
            /// </summary>
            KEY_SEL_KEY_ENTRY_NUMBER = 0x00,
            KEY_SEL_DESFIRE_KEY_ENTRY_NUMBER = 0x02,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 0,
            /// Use Key Diversification
            /// </summary>
            KEY_DIVERSIFICATION_OFF = 0x00,
            KEY_DIVERSIFICATION_ON = 0x01
        }

        /// <summary>
        /// MIFARE DESFire EVx Sam Non X options for Cmd.SAM_ChangeKeyPICC command.
        /// </summary>
        public enum ChangeKeyOption : byte
        {
            #region P1 information byte bit fields
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 5
            /// </summary>
            DIV_METHOD_AV1 = 0x00,
            DIV_METHOD_AV2 = 0x20,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 4 to Bit 1
            /// </summary>
            KDF_RFU = 0x00,
            KDF_CURRENT_KEY_DOUBLE_ENCRYPTION = 0x00,
            KDF_CURRENT_KEY_SINGLE_ENCRYPTION = 0x10,
            KDF_NEW_KEY_DOUBLE_ENCRYPTION = 0x00,
            KDF_NEW_KEY_SINGLE_ENCRYPTION = 0x08,
            KDF_CURRENT_KEY_OFF = 0x00,
            KDF_CURRENT_KEY_ON = 0x04,
            KDF_NEW_KEY_OFF = 0x00,
            KDF_NEW_KEY_ON = 0x02,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 0
            /// </summary>
            CRYPTO_MODE_TARGET_AUTH_KEY_DIFFERENT = 0x00,
            CRYPTO_MODE_TARGET_AUTH_KEY_SAME = 0x01,
            #endregion

            #region P2 information byte bit fields
            /// <summary>
            /// Enumeration fields for P2 information of Cmd.SAM_ChangeKeyPICC, Bit 5.
            /// </summary>
            CMD_TYPE_CHANGE_KEY = 0x00,
            CMD_TYPE_CHANGE_KEY_EV2 = 0x20,

            /// <summary>
            /// Enumeration fields for P2 information of Cmd.SAM_ChangeKeyPICC, Bit 4.
            /// This byte will be right shifted 2 time to update the bit 4 of P2 byte.
            /// </summary>
            MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE = 0x00,
            MASTER_KEY_UPDATE_INCLUDE_KEYTYPE = 0x40
            #endregion
        }

        public enum CreateTMFileOption : byte
        {
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_CreateTMFilePICC, Bit 0,
            /// Use Key Diversification
            /// </summary>
            KEY_DIVERSIFICATION_OFF = 0x00,
            KEY_DIVERSIFICATION_ON = 0x01,

            /// <summary>
            /// Enumeration fields for file options of Cmd.SAM_CreateTMFilePICC
            /// </summary>
            FILE_OPTION_PLAIN = 0x00,
            FILE_OPTION_MAC = 0x01,
            FILE_OPTION_FULL = 0x03
        }
        #endregion

        #region Data Structure
        /// <summary>
        /// Dataparams structure of DESFire EVx component for Sam Non X mode.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Temporary buffer for framing the command buffer. </summary>
            public IntPtr pTmpBuffer;

            /// <summary> Size of Temporary buffer. </summary>
            public ushort wTmpBufSize;

            /// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA), etc.... </summary>
            public byte bAuthMode;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus </summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format </summary>
            public byte bExtendedLenApdu;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for DESFire generic errors or error codes returned by the card. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Command code. This will be used for differentiating the comman error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Complete UID of the PICC. This will be updated when <see cref="NxpRdLibNet.alMfdfEVx.Generic.GetCardUID"/> interface is called. </summary>
            public fixed byte bUid[15];

            /// <summary> Length of bytes available in bUid buffer. </summary>
            public byte bUidLength;

            /// <summary> Get / Set the return of Fabrication Identifier. When Set will be utilized during GetVersion command exchange. </summary>
            public byte bReturn_FabID;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams,
            IntPtr pPalMifareDataParams, IntPtr pTMIDataParams, IntPtr pTmpBuffer, ushort wTmpBufSize );
        #endregion

        #region Initialization
        private byte[] aTmpBuffer;
        private GCHandle pTmpBuffer;

        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in Non X mode (S mode).
        /// This interface should be used by the developers or testers.
        /// </summary>
        ///
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="pPalMifare">Pointer to Pal Mifare parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams, pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }

#if DEBUG
        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in Non X mode (S mode). This interface should not be used by the developers
        /// or testers because this method will not be available in Release builds. This interface can be used for debugging purposes
        /// i.e. for increasing the reader library coverage.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="pPalMifare">Pointer to Pal Mifare parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_SamAV3_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams, pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            if ( this.pTmpBuffer.IsAllocated )
            {
                this.pTmpBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                if ( ( Handle.Target as NxpRdLibNet.Hal.SamAV2 ).DataParams.wId.Equals ( 0x0203 ) )
                    return ( Handle.Target as NxpRdLibNet.Hal.SamAV2 );
                else
                    return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
            }
        }

        public NxpRdLibNet.palMifare.Generic PalMifareDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pPalMifareDataParams;
                return ( Handle.Target as NxpRdLibNet.palMifare.Sw );
            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public byte WrappedMode
        {
            get { return m_DataParamsInt[0].bWrappedMode; }
            set { m_DataParamsInt[0].bWrappedMode = value; }
        }

        public byte ExtendedLenApdu
        {
            get { return m_DataParamsInt[0].bExtendedLenApdu; }
            set { m_DataParamsInt[0].bExtendedLenApdu = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region SAM AV4
    /// <summary>
    /// Class for SAM AV4 and future SAM S (NonX) Mode layer initialization interface and data params.
    /// </summary>
    public class Sam_NonX : alMfdfEVx.Generic
    {
        #region Constants
        /// <summary>
        /// DESFire EVx Native, Native wrapped and Iso7816 custom error code values.
        /// </summary>
        public enum Error : byte
        {
            #region DESFire EVx native and Iso7816 wrapped error codes
            /// <summary>MF DF Response - No changes done to backup files.</summary>
            NO_CHANGES = ( CustomCodes.ERROR_BEGIN ),

            /// <summary>MF DF Response - Insufficient NV-Memory.</summary>
            OUT_OF_EEPROM_ERROR = ( CustomCodes.ERROR_BEGIN + 1 ),

            /// <summary>MF DF Invalid key number specified.</summary>
            NO_SUCH_KEY = ( CustomCodes.ERROR_BEGIN + 2 ),

            /// <summary>MF DF Current configuration/status does not allow the requested command.</summary>
            PERMISSION_DENIED = ( CustomCodes.ERROR_BEGIN + 3 ),

            /// <summary>MF DF Requested AID not found on PICC.</summary>
            APPLICATION_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 4 ),

            /// <summary>MF DF Attempt to read/write data from/to beyond the files/record's limits.</summary>
            BOUNDARY_ERROR = ( CustomCodes.ERROR_BEGIN + 5 ),

            /// <summary>MF DF Previous cmd not fully completed. Not all frames were requested or provided by the PCD.</summary>
            COMMAND_ABORTED = ( CustomCodes.ERROR_BEGIN + 6 ),

            /// <summary>MF DF File/Application with same number already exists.</summary>
            DUPLICATE = ( CustomCodes.ERROR_BEGIN + 8 ),

            /// <summary>MF DF Specified file number does not exist.</summary>
            FILE_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 9 ),

            /// <summary>MF DF Parameter value error returned by PICC.</summary>
            PARAMETER_ERROR = ( CustomCodes.ERROR_BEGIN + 11 ),

            /// <summary>MF DF ISO 7816 Generic error. Check Additional Info.</summary>
            CMD_INVALID = ( CustomCodes.ERROR_BEGIN + 14 ),

            /// <summary>MF DF illegal command code.</summary>
            ILLEGAL_COMMAND_CODE = ( CustomCodes.ERROR_BEGIN + 15 ),

            /// <summary>MF DF integrity error.</summary>
            INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 16 ),

            /// <summary>MF DF memory error.</summary>
            MEMORY_ERROR = ( CustomCodes.ERROR_BEGIN + 17 ),
            #endregion

            #region DESFire EVx Iso7816 error codes
            /// <summary>ISO7816 custom response code for memory failure.</summary>
            ISO7816_MEMORY_FAILURE = ( CustomCodes.ERROR_BEGIN + 18 ),

            /// <summary>ISO7816 custom response code for wrong length, no further indication.</summary>
            ISO7816_WRONG_LENGTH = ( CustomCodes.ERROR_BEGIN + 19 ),

            /// <summary>ISO7816 custom response code for security status not satisfied.</summary>
            ISO7816_SECURITY_STAT_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 20 ),

            /// <summary>ISO7816 custom response code for conditions of use not satisfied.</summary>
            ISO7816_CONDITIONS_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 21 ),

            /// <summary>ISO7816 custom response code for file or application not found.</summary>
            ISO7816_FILE_APPLICATION_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 22 ),

            /// <summary>ISO7816 custom response code for incorrect parameters P1-P2.</summary>
            ISO7816_INCORRECT_PARAM_P1_P2 = ( CustomCodes.ERROR_BEGIN + 23 ),

            /// <summary>ISO7816 custom response code for Lc inconsistent with parameter P1-P2.</summary>
            ISO7816_LC_INCONSISTENT = ( CustomCodes.ERROR_BEGIN + 24 ),

            /// <summary>ISO7816 custom response code for wrong LE field.</summary>
            ISO7816_WRONG_LE = ( CustomCodes.ERROR_BEGIN + 25 ),

            /// <summary>ISO7816 custom response code for instruction code not supported or invalid.</summary>
            ISO7816_INS_CODE_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 26 )
            #endregion
        }

        /// <summary>
        /// DESFire EVx AuthType, AuthMode, Key Derivation, Key Selection and Key Diversification options
        /// for Cmd.SAM_AuthenticatePICC and Cmd.SAM_IsoAuthenticatePICC
        /// </summary>
        public enum AuthenticateOption : byte
        {
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC, Bit 7 and Bit 6,
            /// Key diversification & authentication mode selection, EV2 authentication mode
            /// </summary>
            AUTH_MODE_D40 = 0x00,
            AUTH_MODE_EV1 = 0x00,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC, Bit 5,
            /// Key derivation info
            /// </summary>
            KDF_DISABLED = 0x20,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 4 and Bit 3,
            /// Key Diversification Mode
            /// </summary>
            KDF_AV1 = 0x00,
            KDF_AV1_DOUBLE_ENCRYPTION = 0x00,
            KDF_AV1_SINGLE_ENCRYPTION = 0x08,
            KDF_AV2 = 0x10,
            KDF_RFU = 0x18,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 1,
            /// Key selection
            /// </summary>
            KEY_SEL_KEY_ENTRY_NUMBER = 0x00,
            KEY_SEL_DESFIRE_KEY_ENTRY_NUMBER = 0x02,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_AuthenticatePICC and SAM_IsoAuthenticatePICC, Bit 0,
            /// Use Key Diversification
            /// </summary>
            KEY_DIVERSIFICATION_OFF = 0x00,
            KEY_DIVERSIFICATION_ON = 0x01
        }

        /// <summary>
        /// MIFARE DESFire EVx Sam Non X options for Cmd.SAM_ChangeKeyPICC command.
        /// </summary>
        public enum ChangeKeyOption : byte
        {
            #region P1 information byte bit fields
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 5
            /// </summary>
            DIV_METHOD_AV1 = 0x00,
            DIV_METHOD_AV2 = 0x20,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 4 to Bit 1
            /// </summary>
            KDF_RFU = 0x00,
            KDF_CURRENT_KEY_DOUBLE_ENCRYPTION = 0x00,
            KDF_CURRENT_KEY_SINGLE_ENCRYPTION = 0x10,
            KDF_NEW_KEY_DOUBLE_ENCRYPTION = 0x00,
            KDF_NEW_KEY_SINGLE_ENCRYPTION = 0x08,
            KDF_CURRENT_KEY_OFF = 0x00,
            KDF_CURRENT_KEY_ON = 0x04,
            KDF_NEW_KEY_OFF = 0x00,
            KDF_NEW_KEY_ON = 0x02,

            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_ChangeKeyPICC, Bit 0
            /// </summary>
            CRYPTO_MODE_TARGET_AUTH_KEY_DIFFERENT = 0x00,
            CRYPTO_MODE_TARGET_AUTH_KEY_SAME = 0x01,
            #endregion

            #region P2 information byte bit fields
            /// <summary>
            /// Enumeration fields for P2 information of Cmd.SAM_ChangeKeyPICC, Bit 5.
            /// </summary>
            CMD_TYPE_CHANGE_KEY = 0x00,
            CMD_TYPE_CHANGE_KEY_EV2 = 0x20,

            /// <summary>
            /// Enumeration fields for P2 information of Cmd.SAM_ChangeKeyPICC, Bit 4.
            /// This byte will be right shifted 2 time to update the bit 4 of P2 byte.
            /// </summary>
            MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE = 0x00,
            MASTER_KEY_UPDATE_INCLUDE_KEYTYPE = 0x40
            #endregion
        }

        public enum CreateTMFileOption : byte
        {
            /// <summary>
            /// Enumeration fields for P1 information of Cmd.SAM_CreateTMFilePICC, Bit 0,
            /// Use Key Diversification
            /// </summary>
            KEY_DIVERSIFICATION_OFF = 0x00,
            KEY_DIVERSIFICATION_ON = 0x01,

            /// <summary>
            /// Enumeration fields for file options of Cmd.SAM_CreateTMFilePICC
            /// </summary>
            FILE_OPTION_PLAIN = 0x00,
            FILE_OPTION_MAC = 0x01,
            FILE_OPTION_FULL = 0x03
        }
        #endregion

        #region Data Structure
        /// <summary>
        /// Dataparams structure of DESFire EVx component for Sam Non X mode.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Temporary buffer for framing the command buffer. </summary>
            public IntPtr pTmpBuffer;

            /// <summary> Size of Temporary buffer. </summary>
            public ushort wTmpBufSize;

            /// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA), etc.... </summary>
            public byte bAuthMode;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus </summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format </summary>
            public byte bExtendedLenApdu;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for DESFire generic errors or error codes returned by the card. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Command code. This will be used for differentiating the comman error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Complete UID of the PICC. This will be updated when <see cref="NxpRdLibNet.alMfdfEVx.Generic.GetCardUID"/> interface is called. </summary>
            public fixed byte bUid[15];

            /// <summary> Length of bytes available in bUid buffer. </summary>
            public byte bUidLength;

            /// <summary> Get / Set the return of Fabrication Identifier. When Set will be utilized during GetVersion command exchange. </summary>
            public byte bReturn_FabID;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Sam_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams,
            IntPtr pPalMifareDataParams, IntPtr pTMIDataParams, IntPtr pTmpBuffer, ushort wTmpBufSize );
        #endregion

        #region Initialization
        private byte[] aTmpBuffer;
        private GCHandle pTmpBuffer;

        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in Non X mode (S mode).
        /// This interface should be used by the developers or testers.
        /// </summary>
        ///
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to Pal Mifare parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam oHalSam, palMifare.Generic oPalMifare, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_Sam_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams, ( oPalMifare == null ) ? IntPtr.Zero : oPalMifare.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams, pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }

#if DEBUG
        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in Non X mode (S mode). This interface should not be used by the developers
        /// or testers because this method will not be available in Release builds. This interface can be used for debugging purposes
        /// i.e. for increasing the reader library coverage.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to Pal Mifare parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.Sam oHalSam, palMifare.Generic oPalMifare, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_Sam_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams, ( oPalMifare == null ) ? IntPtr.Zero : oPalMifare.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams, pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            if ( this.pTmpBuffer.IsAllocated )
            {
                this.pTmpBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                if ( ( Handle.Target as NxpRdLibNet.Hal.SamAV3 ).DataParams.wId.Equals ( 0x0210 ) )
                    return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
                else
                    return ( Handle.Target as NxpRdLibNet.Hal.Sam );
            }
        }

        public NxpRdLibNet.palMifare.Generic PalMifareDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pPalMifareDataParams;
                return ( Handle.Target as NxpRdLibNet.palMifare.Sw );
            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public byte WrappedMode
        {
            get { return m_DataParamsInt[0].bWrappedMode; }
            set { m_DataParamsInt[0].bWrappedMode = value; }
        }

        public byte ExtendedLenApdu
        {
            get { return m_DataParamsInt[0].bExtendedLenApdu; }
            set { m_DataParamsInt[0].bExtendedLenApdu = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion
#endif
    #endregion

    #region Sam_X
    #region SAM AV3
    /// <summary>
    /// Class for SAM AV3 X Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : alMfdfEVx.Generic
    {
        #region Data Structure
        /// <summary>
        /// Dataparams structure of DESFire EVx component for Sam X mode.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Temporary buffer for framing the command buffer. </summary>
            public IntPtr pTmpBuffer;

            /// <summary> Size of Temporary buffer. </summary>
            public ushort wTmpBufSize;

            /// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA), etc.... </summary>
            public byte bAuthMode;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus </summary>
            public byte bWrappedMode;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for DESFire generic errors or error codes returned by the card. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Command code. This will be used for differentiating the comman error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Complete UID of the PICC. This will be updated when <see cref="NxpRdLibNet.alMfdfEVx.Generic.GetCardUID"/> interface is called. </summary>
            public fixed byte bUid[15];

            /// <summary> Length of bytes available in bUid buffer. </summary>
            public byte bUidLength;

            /// <summary> Return code from PICC. </summary>
            public fixed byte aPiccRetCode[2];

            /// <summary> Number of status bytes returned. </summary>
            public byte bPiccRetCodeLen;

            /// <summary> Get / Set the return of Fabrication Identifier. When Set will be utilized during GetVersion command exchange. </summary>
            public byte bReturn_FabID;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pTMIDataParams, IntPtr pTmpBuffer, ushort wTmpBufSize );
        #endregion

        #region Initialization
        private byte[] aTmpBuffer;
        private GCHandle pTmpBuffer;

        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in X.
        /// This interface should be used by the developers or testers.
        /// </summary>
        ///
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSam, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams,
                pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }

#if DEBUG
        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in X mode. This interface should not be used by the developers
        /// or testers because this method will not be available in Release builds. This interface can be used for debugging purposes
        /// i.e. for increasing the reader library coverage.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_SamAV3_X_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams,
                pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            if ( this.pTmpBuffer.IsAllocated )
            {
                this.pTmpBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );

            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public byte WrappedMode
        {
            get { return m_DataParamsInt[0].bWrappedMode; }
            set { m_DataParamsInt[0].bWrappedMode = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region SAM AV4
    /// <summary>
    /// Class for SAM AV4 and Future SAM's X Mode layer initialization interface and data params.
    /// </summary>
    public class Sam_X : alMfdfEVx.Generic
    {
        #region Data Structure
        /// <summary>
        /// Dataparams structure of DESFire EVx component for Sam X mode.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Temporary buffer for framing the command buffer. </summary>
            public IntPtr pTmpBuffer;

            /// <summary> Size of Temporary buffer. </summary>
            public ushort wTmpBufSize;

            /// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA), etc.... </summary>
            public byte bAuthMode;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus </summary>
            public byte bWrappedMode;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for DESFire generic errors or error codes returned by the card. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Command code. This will be used for differentiating the comman error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Complete UID of the PICC. This will be updated when <see cref="NxpRdLibNet.alMfdfEVx.Generic.GetCardUID"/> interface is called. </summary>
            public fixed byte bUid[15];

            /// <summary> Length of bytes available in bUid buffer. </summary>
            public byte bUidLength;

            /// <summary> Return code from PICC. </summary>
            public fixed byte aPiccRetCode[2];

            /// <summary> Number of status bytes returned. </summary>
            public byte bPiccRetCodeLen;

            /// <summary> Get / Set the return of Fabrication Identifier. When Set will be utilized during GetVersion command exchange. </summary>
            public byte bReturn_FabID;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfEVx_Sam_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pTMIDataParams, IntPtr pTmpBuffer, ushort wTmpBufSize );
        #endregion

        #region Initialization
        private byte[] aTmpBuffer;
        private GCHandle pTmpBuffer;

        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in X.
        /// This interface should be used by the developers or testers.
        /// </summary>
        ///
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam oHalSam, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams,
                pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }

#if DEBUG
        /// <summary>
        /// Initializes the DESFire EVx Sam AV3 component in X mode. This interface should not be used by the developers
        /// or testers because this method will not be available in Release builds. This interface can be used for debugging purposes
        /// i.e. for increasing the reader library coverage.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.Sam oHalSam, TMIUtils oTMIUtils )
        {
            aTmpBuffer = new byte[512];
            pTmpBuffer = GCHandle.Alloc ( aTmpBuffer, GCHandleType.Pinned );

            return phalMfdfEVx_Sam_X_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams,
                pTmpBuffer.AddrOfPinnedObject (), ( ushort ) aTmpBuffer.Length );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            if ( this.pTmpBuffer.IsAllocated )
            {
                this.pTmpBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.Sam );

            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public byte WrappedMode
        {
            get { return m_DataParamsInt[0].bWrappedMode; }
            set { m_DataParamsInt[0].bWrappedMode = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion
#endif
    #endregion
#endif
}
