/*
 * Copyright 2013, 2016 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfp
{
	#region Enumerations
	#region Error Codes
	/// <summary>
	/// Custom error codes equivalent to C library error codes.
	/// </summary>
	public enum Error : byte
	{
		/// <summary> MFP EV1 Authentication Error.
		/// 1. Access condition not fulfilled (the current command is not allowed on the requested block,
		///    or any of the requested blocks, when authorization is done with the key that was used in the
		///    latest authentication step) or
		///
		/// 2. The block that is to be accessed is not a value block and an operation is performed that only
		/// can work on value blocks
		///
		/// 3. Invalid padding of the value.
		/// 4. Trying to read invalid SectorTrailer.
		/// 5. The comparison of the random number in the authentication process failed.
		/// 6. EV0 secure messaging mode requested while not allowed by the PD configuration.
		/// </summary>
		AUTH = ( CustomCodes.ERROR_BEGIN ),

		/// <summary> Field to indicate Command Overflow error.
		/// Too many read or write commands in the session or in the transaction.
		/// Note that this error code is also used in the Proximity Checks.
		/// </summary>
		CMD_OVERFLOW,

		/// <summary> Field to indicate MAC PCD error.
		/// Invalid or missing MAC on the command sent by the PCD.
		/// </summary>
		MAC_PCD,

		/// <summary> Field to indicate Invalid Block Number error.
		/// Invalid Block number: not existing in the implementation or not valid to target with this command.
		/// </summary>
		BNR,

		/// <summary> Field to indicate Number of blocks not supported error.
		/// Invalid Amount of Blocks requested to be written or read.
		/// </summary>
		EXT,

		/// <summary> Field to indicate Command Invalid error.
		/// Command is received in a state where this command is not supported, or a totally unknown
		/// command is received. E.g. a Read or Write command is received while VCState SL3MFPAuthenticated
		/// </summary>
		CMD_INVALID,

		/// <summary> Field to indicate Format error.
		/// Format of the command is not correct (e.g. too many or too few bytes).
		///
		/// Note: The ErrBNotExist (not to be confused with ErrBnr) used in MFP SL0 is also mapped on
		///       ErrFormat.
		/// </summary>
		FORMAT,

		/// <summary> Field to indicate General error.
		/// 1. Failure in the operation of the PD, e.g. cannot write the data block.
		/// 2. Increment or IncrementTransfer command causes overflow.
		/// 3. Decrement command or DecrementTransfer command causes underflow.
		/// </summary>
		GEN_FAILURE
	}
	#endregion Error Codes

	#region ISO14443 options
	/// <summary>
	/// Options to indicate the ISO14443 protocol to be used for Authentication and other commands.
	/// </summary>
	public enum Iso144443 : byte
	{
		/// <summary> Field to indicate the Layer 3 protocol to be used. </summary>
		L3 = 0x00,

		/// <summary> Field to indicate the Layer 4 protocol to be used. </summary>
		L4 = 0x01
	}
	#endregion ISO14443 options

	#region Mifare Classic key types
	/// <summary>
	/// Key type options for Mifare Classic Authentication.
	/// </summary>
	public enum KeyType : byte
	{
		/// <summary> MIFARE(R) Key A. </summary>
		A = 0x0A,

		/// <summary> MIFARE(R) Key B. </summary>
		B = 0x0B
	}
	#endregion Mifare Classic key types

	#region Authentication options
	/// <summary>
	/// Options to indicate the Authentication type to be performed.
	/// </summary>
	public enum Auth : byte
	{
		/// <summary> Option to indicate the authenticate type as non-first or following. </summary>
		NON_FIRST = 0x00,

		/// <summary> Option to indicate the authenticate type as first. </summary>
		FIRST = 0x01
	}
	#endregion Authentication options

	#region Data Encryption options
	/// <summary>
	/// Options to indicate the communication mode.
	/// </summary>
	public enum Encryption : byte
	{
		/// <summary> Option to indicate the communication between PCD and PICC is plain. </summary>
		OFF = 0x00,

		/// <summary> Option to indicate the communication between PCD and PICC is encrypted. </summary>
		ON = 0x01
	}
	#endregion Data Encryption options

	#region Mac On Command options
	/// <summary>
	/// Options to indicate the communication mode as maced for PCD to PICC transfer.
	/// </summary>
	public enum MacOnCmd : byte
	{
		/// <summary> Option to indicate the communication is not maced for PCD to PICC transfer. </summary>
		OFF = 0x00,

		/// <summary> Option to indicate the communication is maced for PCD to PICC transfer. </summary>
		ON = 0x01
	}
	#endregion Mac On Command options

	#region Mac On Response options
	/// <summary>
	/// Options to indicate the communication mode as maced for PICC to PCD transfer.
	/// </summary>
	public enum MacOnRsp : byte
	{
		/// <summary> Option to indicate the communication is not maced for PICC to PCD transfer. </summary>
		OFF = 0x00,

		/// <summary> Option to indicate the communication is maced for PICC to PCD transfer. </summary>
		ON = 0x01
	}
	#endregion Mac On Command options

	#region Diversification options
	/// <summary>
	/// Options to diversify the key.
	/// </summary>
	public enum Diversify : ushort
	{
		/// <summary> Option to disable key diversification.</summary>
		OFF = 0xFFFF,

		/// <summary> Option to enable key diversification.</summary>
		ON = 0x0000
	}
	#endregion Diversification options
	#endregion Enumerations

	#region Generic
	/// <summary>
	/// Class having the wrapper for C command.
	/// </summary>
	public abstract class Generic
	{
		#region DLL Imports
		#region Personalization commands
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_WritePerso ( IntPtr pDataParams, byte bLayer4Comm, ushort wBlockNr, byte[] pValue );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_CommitPerso ( IntPtr pDataParams, byte bLayer4Comm );
		#endregion

		#region Authentication commands
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_AuthenticateClassicSL2 ( IntPtr pDataParams, byte bBlockNo, byte bKeyType, ushort wKeyNumber, ushort wKeyVersion,
			byte[] pUid, byte bUidLength );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_AuthenticateSL0 ( IntPtr pDataParams, byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber,
			ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In, byte[] pPcdCap2Out, byte[] pPdCap2 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_AuthenticateSL1 ( IntPtr pDataParams, byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber,
			ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In, byte[] pPcdCap2Out, byte[] pPdCap2 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_AuthenticateSL2 ( IntPtr pDataParams, byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber,
			ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In, byte[] pPcdCap2Out, byte[] pPdCap2, byte[] pKmf );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_AuthenticateSL3 ( IntPtr pDataParams, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
			byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In, byte[] pPcdCap2Out, byte[] pPdCap2 );
		#endregion

		#region Data Operation commands
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_MultiBlockWrite ( IntPtr pDataParams, byte bBlockNr, byte bNumBlocks, byte[] pBlocks );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_MultiBlockRead ( IntPtr pDataParams, byte bBlockNr, byte bNumBlocks, byte[] pBlocks );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Write ( IntPtr pDataParams, byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte bNumBlocks,
			byte[] pBlocks );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Read ( IntPtr pDataParams, byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr,
			byte bNumBlocks, byte[] pBlocks );
		#endregion

		#region Value Operation commands
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_WriteValue ( IntPtr pDataParams, byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte[] pValue,
			byte bAddrData );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_ReadValue ( IntPtr pDataParams, byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, byte[] pValue,
			ref byte pAddrData );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Increment ( IntPtr pDataParams, byte bIncrementMaced, ushort wBlockNr, byte[] pValue );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Decrement ( IntPtr pDataParams, byte bDecrementMaced, ushort wBlockNr, byte[] pValue );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_IncrementTransfer ( IntPtr pDataParams, byte bIncrementTransferMaced, ushort wSourceBlockNr,
			ushort wDestinationBlockNr, byte[] pValue );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_DecrementTransfer ( IntPtr pDataParams, byte bDecrementTransferMaced, ushort wSourceBlockNr,
			ushort wDestinationBlockNr, byte[] pValue );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Transfer ( IntPtr pDataParams, byte bTransferMaced, ushort wBlockNr );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Restore ( IntPtr pDataParams, byte bRestoreMaced, ushort wBlockNr );
		#endregion

		#region Special commands
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_ResetAuth ( IntPtr pDataParams );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_ResetSecMsgState ( IntPtr pDataParams );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_ProximityCheck ( IntPtr pDataParams, byte bGenerateRndC, byte[] pRndC, byte bPps1, byte bNumSteps, byte[] pUsedRndC );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_ChangeKey ( IntPtr pDataParams, byte bChangeKeyMaced, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
			byte bLenDivInput, byte[] pDivInput );
		#endregion
		#endregion

		#region Wrapper Functions
		#region Personalization commands
		/// <summary>
		/// Performs a Write Perso command. The Write Perso command can be executed using the ISO14443-3 communication
		/// protocol (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation).
		/// </summary>
		///
		/// <param name="bLayer4Comm">ISO14443 protocol to be used.
		///								NxpRdLibNet.alMfp.Iso144443.L3
		///								NxpRdLibNet.alMfp.Iso144443.L4</param>
		/// <param name="wBlockNr">Block number to be personalized.</param>
		/// <param name="pValue">The value for the block mentioned in BlockNr parameter. </param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t WritePerso ( byte bLayer4Comm, int wBlockNr, byte[] pValue )
		{
			return phalMfp_WritePerso ( m_pDataParams, bLayer4Comm, ( ushort ) wBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Commit Perso command. The Commit Perso command can be executed using the ISO14443-3 communication protocol
		/// (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation). This command commits the written
		/// data during WritePerso command and switches the SecurityLevel to 1 or 3 based on the option provided.
		/// </summary>
		///
		/// <param name="bLayer4Comm">ISO14443 protocol to be used.
		///								NxpRdLibNet.alMfp.Iso144443.L3
		///								NxpRdLibNet.alMfp.Iso144443.L4</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t CommitPerso ( byte bLayer4Comm )
		{
			return phalMfp_CommitPerso ( m_pDataParams, bLayer4Comm );
		}
		#endregion

		#region Authentication commands
		/// <summary>
		/// Perform MIFARE(R) Authenticate command in Security Level 2 with MIFARE Picc.
		/// </summary>
		///
		/// <param name="bBlockNo">PICC block number to be used for authentication.
		///								NxpRdLibNet.alMfp.KeyType.A
		///								NxpRdLibNet.alMfp.KeyType.B</param>
		/// <param name="bKeyType">Authentication key type to be used.
		///							</param>
		/// <param name="wKeyNumber">Key number to used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version to used from software or hardware keystore.</param>
		/// <param name="pUid">UID of the PICC received during anti-collision sequence.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t AuthenticateClassicSL2 ( byte bBlockNo, byte bKeyType, ushort wKeyNumber, ushort wKeyVersion, byte[] pUid )
		{
			return phalMfp_AuthenticateClassicSL2 ( m_pDataParams, bBlockNo, ( byte ) bKeyType, wKeyNumber, wKeyVersion, pUid, ( byte ) pUid.Length );
		}

		/// <summary>
		/// Performs a MIFARE Plus Authentication for Security Level 0.This command performs basic Authenticate First / Non-First
		/// command execution and also performs the AuthenticateContinue command internally.
		/// </summary>
		///
		/// <param name="bLayer4Comm">ISO14443 protocol to be used.
		///								NxpRdLibNet.alMfp.Iso144443.L3
		///								NxpRdLibNet.alMfp.Iso144443.L4</param>
		/// <param name="bFirstAuth">Type of authentication to be performed.
		///								NxpRdLibNet.alMfp.Auth.FIRST
		///								NxpRdLibNet.alMfp.Auth.NON_FIRST</param>
		/// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
		/// <param name="wKeyNumber">Key number be to used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version be to used from software or hardware keystore.</param>
		/// <param name="pDivInput">Diversification Input used to diversify the key.</param>
		/// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
		///							 If length is non zero, PCDCapabilites should be available.
		///							 If length is zero, PCDCapabilites buffer should be NULL.</param>
		/// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
		/// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t AuthenticateSL0 ( byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput,
			byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2 )
		{
			pPcdCap2Out = new byte[6];
			pPdCap2 = new byte[6];

			return phalMfp_AuthenticateSL0 ( m_pDataParams, bLayer4Comm, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput, ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
				pPcdCap2In, pPcdCap2Out, pPdCap2 );
		}

		/// <summary>
		/// Performs a MIFARE Plus Authentication for Security Level 1. This command performs basic Authenticate First / Non-First
		/// command execution and also performs the AuthenticateContinue command internally.
		/// </summary>
		///
		/// <param name="bLayer4Comm">ISO14443 protocol to be used.
		///								NxpRdLibNet.alMfp.Iso144443.L3
		///								NxpRdLibNet.alMfp.Iso144443.L4</param>
		/// <param name="bFirstAuth">Type of authentication to be performed.
		///								NxpRdLibNet.alMfp.Auth.FIRST
		///								NxpRdLibNet.alMfp.Auth.NON_FIRST</param>
		/// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
		/// <param name="wKeyNumber">Key number be to used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version be to used from software or hardware keystore.</param>
		/// <param name="pDivInput">Diversification Input used to diversify the key.</param>
		/// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
		///							 If length is non zero, PCDCapabilites should be available.
		///							 If length is zero, PCDCapabilites buffer should be NULL.</param>
		/// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
		/// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t AuthenticateSL1 ( byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput,
			byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2 )
		{
			pPcdCap2Out = new byte[6];
			pPdCap2 = new byte[6];

			return phalMfp_AuthenticateSL1 ( m_pDataParams, bLayer4Comm, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput, ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
				pPcdCap2In, pPcdCap2Out, pPdCap2 );
		}

		/// <summary>
		/// Performs a MIFARE Plus Authentication for Security Level 1. This command performs basic Authenticate First / Non-First
		/// command execution and also performs the AuthenticateContinue command internally.
		/// </summary>
		///
		/// <param name="bLayer4Comm">ISO14443 protocol to be used.
		///								NxpRdLibNet.alMfp.Iso144443.L3
		///								NxpRdLibNet.alMfp.Iso144443.L4</param>
		/// <param name="bFirstAuth">Type of authentication to be performed.
		///								NxpRdLibNet.alMfp.Auth.FIRST
		///								NxpRdLibNet.alMfp.Auth.NON_FIRST</param>
		/// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
		/// <param name="wKeyNumber">Key number be to used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version be to used from software or hardware keystore.</param>
		/// <param name="pDivInput">Diversification Input used to diversify the key.</param>
		/// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
		///							 If length is non zero, PCDCapabilites should be available.
		///							 If length is zero, PCDCapabilites buffer should be NULL.</param>
		/// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
		/// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
		/// <param name="pKmf">MIFARE Sector Key Modifier (6 bytes).</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t AuthenticateSL2 ( byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput,
			byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2, out byte[] pKmf )
		{
			pPcdCap2Out = new byte[6];
			pPdCap2 = new byte[6];
			pKmf = new byte[6];

			return phalMfp_AuthenticateSL2 ( m_pDataParams, bLayer4Comm, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput, ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
				pPcdCap2In, pPcdCap2Out, pPdCap2, pKmf );
		}

		/// <summary>
		/// Performs a MIFARE Plus Authentication for Security Level 3. This command performs basic Authenticate First / Non-First
		/// command execution and also performs the AuthenticateContinue command internally.
		/// </summary>
		///
		/// <param name="bFirstAuth">Type of authentication to be performed.
		///								NxpRdLibNet.alMfp.Auth.FIRST
		///								NxpRdLibNet.alMfp.Auth.NON_FIRST</param>
		/// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
		/// <param name="wKeyNumber">Key number be to used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version be to used from software or hardware keystore.</param>
		/// <param name="pDivInput">Diversification Input used to diversify the key.</param>
		/// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
		///							 If length is non zero, PCDCapabilites should be available.
		///							 If length is zero, PCDCapabilites buffer should be NULL.</param>
		/// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
		/// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t AuthenticateSL3 ( byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput, byte[] pPcdCap2In,
			out byte[] pPcdCap2Out, out byte[] pPdCap2 )
		{
			pPcdCap2Out = new byte[6];
			pPdCap2 = new byte[6];

			return phalMfp_AuthenticateSL3 ( m_pDataParams, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ),
				pDivInput, ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ), pPcdCap2In, pPcdCap2Out, pPdCap2 );
		}
		#endregion

		#region Data operation commands
		/// <summary>
		/// Performs a Multi Block Write command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
		/// </summary>
		///
		/// <param name="bBlockNr">PICC block number to which the data should be written.</param>
		/// <param name="bNumBlocks">Number of blocks to write (must not be more than 3).</param>
		/// <param name="pBlocks">Block(s) (16 * bNumBlocks bytes).</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t MultiBlockWrite ( byte bBlockNr, byte bNumBlocks, byte[] pBlocks )
		{
			return phalMfp_MultiBlockWrite ( m_pDataParams, bBlockNr, bNumBlocks, pBlocks );

		}

		/// <summary>
		/// Performs a Multi Block Read command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
		/// </summary>
		///
		/// <param name="bBlockNr">PICC block number from which the data should be read.</param>
		/// <param name="bNumBlocks">Number of blocks to read (must not be more than 3).</param>
		/// <param name="pBlocks">Block(s) (16 * bNumBlocks bytes).</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t MultiBlockRead ( byte bBlockNr, byte bNumBlocks, out byte[] pBlocks )
		{
			pBlocks = new byte[16 * bNumBlocks];
			return phalMfp_MultiBlockRead ( m_pDataParams, bBlockNr, bNumBlocks, pBlocks );

		}

		/// <summary>
		/// Performs a Write / Write MACed command. This command writes a 16 byte data to the PICC.
		/// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
		///
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to which the data should be written.</param>
		/// <param name="bNumBlocks">Number of blocks to write (must not be more than 3).</param>
		/// <param name="pBlocks">The data to be written. This buffer should have data equal to NumBlocks * 16,
		///						  where 16 is one block size.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Write ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte bNumBlocks, byte[] pBlocks )
		{
			return phalMfp_Write ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, bNumBlocks, pBlocks );
		}

		/// <summary>
		/// Performs a Read / Read MACed command.
		/// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
		/// not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnCmd.OFF
		///								NxpRdLibNet.alMfp.MacOnCmd.ON</param>
		/// <param name="wBlockNr">PICC block number from which the data should be read.</param>
		/// <param name="bNumBlocks">Number of blocks to be read (must not be more than 3).</param>
		/// <param name="pBlocks">The data to be read. This buffer should have data equal to NumBlocks * 16,
		///						  where 16 is one block size.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Read ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, byte bNumBlocks, out byte[] pBlocks )
		{
			pBlocks = new byte[16 * bNumBlocks + 8];

			Status_t oStatus = phalMfp_Read ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, bNumBlocks, pBlocks );
			Array.Resize ( ref pBlocks, 16 * bNumBlocks );

			return oStatus;
		}
		#endregion

		#region Value operation commands
		/// <summary>
		/// Performs a Write / Write MACed command of a value.
		/// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to which the value should be written.</param>
		/// <param name="pValue">The value to be written. This buffer should have value equal to 4 bytes.</param>
		/// <param name="bAddrData">The address to be written.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t WriteValue ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte[] pValue, byte bAddrData )
		{
			return phalMfp_WriteValue ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, pValue, bAddrData );
		}

		/// <summary>
		/// Performs a Write / Write MACed command of a value.
		/// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to which the value should be written.</param>
		/// <param name="dwValue">The value to be written. This buffer should have value equal to 4 bytes.</param>
		/// <param name="bAddrData">The address to be written.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t WriteValueDw ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, int dwValue, byte bAddrData )
		{
			byte[] pValue = ConvertIntToByteArray ( dwValue );

			return phalMfp_WriteValue ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, pValue, bAddrData );
		}

		/// <summary>
		/// Performs a Read / Read MACed Value command.
		/// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
		/// not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnCmd.OFF
		///								NxpRdLibNet.alMfp.MacOnCmd.ON</param>
		/// <param name="wBlockNr">PICC block number from which the value should be read.</param>
		/// <param name="pValue">The value read from the specified block number. The buffer will have 4 bytes
		///						 of value information.</param>
		/// <param name="pAddrData">The address from the read value information.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ReadValue ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, out byte[] pValue, out byte pAddrData )
		{
			pValue = new byte[4];
			pAddrData = 0;
			return phalMfp_ReadValue ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, pValue, ref pAddrData );
		}

		/// <summary>
		/// Performs a Read / Read MACed Value command.
		/// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
		/// not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
		///							 will be updated.
		///								NxpRdLibNet.alMfp.Encryption.OFF
		///								NxpRdLibNet.alMfp.Encryption.ON</param>
		/// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnRsp.OFF
		///								NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
		///							  command code will be updated.
		///								NxpRdLibNet.alMfp.MacOnCmd.OFF
		///								NxpRdLibNet.alMfp.MacOnCmd.ON</param>
		/// <param name="wBlockNr">PICC block number from which the value should be read.</param>
		/// <param name="dwValue">The value read from the specified block number. The buffer will have 4 bytes
		///						  of value information.</param>
		/// <param name="pAddrData">The address from the read value information.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ReadValueDw ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, out int dwValue, out byte pAddrData )
		{
			dwValue = 0;
			byte [] pValue = new byte[4];
			pAddrData = 0;

			Status_t oStatus =  phalMfp_ReadValue ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, pValue, ref pAddrData );
			if ( oStatus.Equals ( new Status_t () ) )
			{
				dwValue = ConvertByteArrayToInt ( pValue );
			}

			return oStatus;
		}

		/// <summary>
		/// Performs an Increment / Increment MACed command.
		/// The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bIncrementMaced">Indicate whether the response should be maced. Based on this flag the
		///								  command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for incrementing the value.</param>
		/// <param name="pValue">The value to be incremented. This buffer should have 4 bytes value information.
		///						 The value to be incremented should be LSB first order.
		///						 For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
		///						 0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Increment ( byte bIncrementMaced, ushort wBlockNr, byte[] pValue )
		{
			return phalMfp_Increment ( m_pDataParams, bIncrementMaced, wBlockNr, pValue );
		}

		/// <summary>
		/// Performs an Increment / Increment MACed command.
		/// The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bIncrementMaced">Indicate whether the response should be maced. Based on this flag the
		///								  command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for incrementing the value.</param>
		/// <param name="dwValue">The value to be incremented. This buffer should have 4 bytes value information.
		///						 The value to be incremented should be LSB first order.
		///						 For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
		///						 0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t IncrementDw ( byte bIncrementMaced, ushort wBlockNr, int dwValue )
		{
			byte[] pValue = ConvertIntToByteArray ( dwValue );
			return phalMfp_Increment ( m_pDataParams, bIncrementMaced, wBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Decrement / Decrement MACed command.
		/// The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bDecrementMaced">Indicate whether the response should be maced. Based on this flag the
		///								  command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for decrementing the value.</param>
		/// <param name="pValue">The value to be decremented. This buffer should have 4 bytes value information.
		///						 The value to be decremented should be LSB first order.
		///						 For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///						 0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Decrement ( byte bDecrementMaced, ushort wBlockNr, byte[] pValue )
		{
			return phalMfp_Decrement ( m_pDataParams, bDecrementMaced, wBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Decrement / Decrement MACed command.
		/// The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bDecrementMaced">Indicate whether the response should be maced. Based on this flag the
		///								  command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for decrementing the value.</param>
		/// <param name="dwValue">The value to be decremented. This buffer should have 4 bytes value information.
		///						 The value to be decremented should be LSB first order.
		///						 For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///						 0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t DecrementDw ( byte bDecrementMaced, ushort wBlockNr, int dwValue )
		{
			byte[] pValue = ConvertIntToByteArray ( dwValue );
			return phalMfp_Decrement ( m_pDataParams, bDecrementMaced, wBlockNr, pValue );
		}

		/// <summary>
		/// Performs an Increment Transfer / Increment Transfer MACed command.
		/// The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bIncrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
		///										  command code will be updated.
		///											NxpRdLibNet.alMfp.MacOnRsp.OFF
		///											NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wSourceBlockNr">PICC block number to be used for incrementing the value.</param>
		/// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
		/// <param name="pValue">The value to be incremented and transferred. This buffer should have 4 bytes
		///						  value information. The value to be incremented and transferred should be LSB first order.
		///							For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///							0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t IncrementTransfer ( byte bIncrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, byte[] pValue )
		{
			return phalMfp_IncrementTransfer ( m_pDataParams, bIncrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue );
		}

		/// <summary>
		/// Performs an Increment Transfer / Increment Transfer MACed command.
		/// The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bIncrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
		///										  command code will be updated.
		///											NxpRdLibNet.alMfp.MacOnRsp.OFF
		///											NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wSourceBlockNr">PICC block number to be used for incrementing the value.</param>
		/// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
		/// <param name="dwValue">The value to be incremented and transferred. This buffer should have 4 bytes
		///						  value information. The value to be incremented and transferred should be LSB first order.
		///							For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///							0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t IncrementTransferDw ( byte bIncrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, int dwValue )
		{
			byte[] pValue = ConvertIntToByteArray ( dwValue );
			return phalMfp_IncrementTransfer ( m_pDataParams, bIncrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Decrement Transfer / Decrement Transfer MACed command.
		/// The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bDecrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
		///										  command code will be updated.
		///											NxpRdLibNet.alMfp.MacOnRsp.OFF
		///											NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wSourceBlockNr">PICC block number to be used for decrementing the value.</param>
		/// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
		/// <param name="pValue">The value to be decremented and transferred. This buffer should have 4 bytes
		///						  value information. The value to be decremented and transferred should be LSB first order.
		///							For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///							0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t DecrementTransfer ( byte bDecrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, byte[] pValue )
		{
			return phalMfp_DecrementTransfer ( m_pDataParams, bDecrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Decrement Transfer / Decrement Transfer MACed command.
		/// The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bDecrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
		///										  command code will be updated.
		///											NxpRdLibNet.alMfp.MacOnRsp.OFF
		///											NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wSourceBlockNr">PICC block number to be used for decrementing the value.</param>
		/// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
		/// <param name="pValue">The value to be decremented and transferred. This buffer should have 4 bytes
		///						  value information. The value to be decremented and transferred should be LSB first order.
		///							For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
		///							0x01, 0x00, 0x00, 0x00.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t DecrementTransferDw ( byte bDecrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, int dwValue )
		{
			byte[] pValue = ConvertIntToByteArray ( dwValue );
			return phalMfp_DecrementTransfer ( m_pDataParams, bDecrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue );
		}

		/// <summary>
		/// Performs a Transfer / Transfer MACed command.
		/// The parameter TransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bTransferMaced">Indicate whether the response should be maced. Based on this flag the
		///								 command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for transferring the value.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Transfer ( byte bTransferMaced, ushort wBlockNr )
		{
			return phalMfp_Transfer ( m_pDataParams, bTransferMaced, wBlockNr );
		}

		/// <summary>
		/// Performs a Restore / Restore MACed command.
		/// The parameter RestoreMaced is valid only for MFP authenticated state and not for MFC authenticate state.
		/// </summary>
		///
		/// <param name="bRestoreMaced">Indicate whether the response should be maced. Based on this flag the
		///								command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to be used for restoring the value.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Restore ( byte bRestoreMaced, ushort wBlockNr )
		{
			return phalMfp_Restore ( m_pDataParams, bRestoreMaced, wBlockNr );
		}
		#endregion

		#region Special commands
		/// <summary>
		/// Performs a Reset Auth command.
		/// </summary>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ResetAuth ()
		{
			return phalMfp_ResetAuth ( m_pDataParams );
		}

		/// <summary>
		/// Reset the libraries internal secure messaging state.
		///
		/// This function must be called before interacting with the PICC to set the libraries internal card-state back to default.
		/// E.g. when an error occurred or after a reset of the field.
		/// </summary>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ResetSecMsgState ()
		{
			return phalMfp_ResetSecMsgState ( m_pDataParams );
		}

		/// <summary>
		/// Performs a Key Change of a MIFARE Plus key. Same as phalMfp_Write, but diversification input can be provided
		/// </summary>
		///
		/// <param name="bChangeKeyMaced">Indicate whether the response should be maced. Based on this flag the
		///								  command code will be updated.
		///									NxpRdLibNet.alMfp.MacOnRsp.OFF
		///									NxpRdLibNet.alMfp.MacOnRsp.ON</param>
		/// <param name="wBlockNr">PICC block number to which the key should be changed.</param>
		/// <param name="wKeyNumber">Key number to be used from software or hardware keystore.</param>
		/// <param name="wKeyVersion">Key version to be used from software or hardware keystore.</param>
		/// <param name="pDivInput">Diversification Input used to diversify the key.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ChangeKey ( byte bChangeKeyMaced, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput )
		{
			return phalMfp_ChangeKey ( m_pDataParams, bChangeKeyMaced, wBlockNr, wKeyNumber, wKeyVersion, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ),
				pDivInput );
		}

		/// <summary>
		/// Performs the whole Proximity Check command chain.
		/// </summary>
		///
		/// <param name="bGenerateRndC">Indicate the generation of RndC.</param>
		/// <param name="pRndC">Provided RndC (7 bytes), ignored if bGenerateRndC == 1.</param>
		/// <param name="bPps1">Communication Speed byte (PPS1).</param>
		/// <param name="bNumSteps">Number of ProximityCheck cycles; RFU, must be 1</param>
		/// <param name="pUsedRndRC">Used RndC (7 bytes), can be NULL.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t ProximityCheck ( byte bGenerateRndC, byte[] pRndC, byte bPps1, byte bNumSteps, out byte[] pUsedRndRC )
		{
			pUsedRndRC = new byte[14];
			return phalMfp_ProximityCheck ( m_pDataParams, bGenerateRndC, pRndC, bPps1, bNumSteps, pUsedRndRC );
		}
		#endregion
		#endregion

		#region Private functins
		protected byte[] ConvertIntToByteArray ( int dwValue )
		{
			byte[] pValue = new byte[4];
			pValue[0] = ( byte ) dwValue;
			pValue[1] = ( byte ) ( dwValue >> 8 );
			pValue[2] = ( byte ) ( dwValue >> 16 );
			pValue[3] = ( byte ) ( dwValue >> 24 );
			return pValue;
		}
		protected int ConvertByteArrayToInt ( byte[] pValue )
		{
			int dwValue = 0;
			dwValue = pValue[0];
			dwValue += pValue[1] << 8;
			dwValue += pValue[2] << 16;
			dwValue += pValue[3] << 24;
			return dwValue;
		}
		#endregion

		#region Memory Maping
		protected GCHandle m_pDataParamsInt;

		/// <summary>
		/// Retrieve private data storage of underlying C Object.
		/// </summary>
		public IntPtr m_pDataParams
		{
			get
			{
				return this.m_pDataParamsInt.AddrOfPinnedObject ();
			}
		}
		#endregion
	}
	#endregion

	#region Software
	/// <summary>
	/// Class for software layer initialization interface and data params.
	/// </summary>
	public class Sw : alMfp.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE Plus Software layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
			public IntPtr pKeyStoreDataParams;

			/// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
			public IntPtr pCryptoDataParamsEnc;

			/// <summary> Pointer to the parameter structure of the Crypto layer for macing. </summary>
			public IntPtr pCryptoDataParamsMac;

			/// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
			public IntPtr pCryptoRngDataParams;

			/// <summary> Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). </summary>
			public IntPtr pCryptoDiversifyDataParams;

			/// <summary> Key Modifier for MIFARE Plus SL2 authentication. </summary>
			public fixed byte bKeyModifier[6];

			/// <summary> R_CTR (read counter); the PICC's read counter is used for a following authentication. </summary>
			public ushort wRCtr;

			/// <summary> W_CTR (write counter); the PICC's write counter is used for a following authentication. </summary>
			public ushort wWCtr;

			/// <summary> Transaction Identifier; unused if 'bFirstAuth' = 1; uint8_t[4]. </summary>
			public fixed byte bTi[4];

			/// <summary> Amount of data in the pUnprocessedReadMacBuffer. </summary>
			public byte bNumUnprocessedReadMacBytes;

			/// <summary> Buffer containing unprocessed bytes for read mac answer stream. </summary>
			public fixed byte bUnprocessedReadMacBuffer[16];

			/// <summary> Intermediate MAC for Read Calculation. </summary>
			public fixed byte bIntermediateMac[16];

			/// <summary> Indicates whether this is a first read in a read MAC sequence or not. </summary>
			public byte bFirstRead;
		};
		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifare, IntPtr pKeyStore,
			IntPtr pCryptoEnc, IntPtr pCryptoMac, IntPtr pCryptoRngDataParams, IntPtr pCryptoDiversifyDataParams );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Sw_Cmd_PrepareProximityCheck ( IntPtr pDataParams );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Sw_Cmd_ProximityCheck ( IntPtr pDataParams, byte bGenerateRndC, byte[] pRndC, byte bNumSteps, byte[] pUsedRndRC );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_Sw_Cmd_VerifyProximityCheck ( IntPtr pDataParams, byte[] pRndRC, byte bPps1 );
		#endregion

		#region Wrapper Functions
		/// <summary>
		///
		/// </summary>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Cmd_PrepareProximityCheck ()
		{
			return phalMfp_Sw_Cmd_PrepareProximityCheck ( m_pDataParams );
		}

		/// <summary>
		///
		/// </summary>
		///
		/// <param name="bGenerateRndC"></param>
		/// <param name="pRndC"></param>
		/// <param name="bNumSteps"></param>
		/// <param name="pUsedRndRC"></param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Cmd_ProximityCheck ( byte bGenerateRndC, byte[] pRndC, byte bNumSteps, out byte[] pUsedRndRC )
		{
			pUsedRndRC = new byte[14];
			return phalMfp_Sw_Cmd_ProximityCheck ( m_pDataParams, bGenerateRndC, pRndC, bNumSteps, pUsedRndRC );
		}

		/// <summary>
		///
		/// </summary>
		///
		/// <param name="pRndRC"></param>
		/// <param name="bPps1"></param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Cmd_VerifyProximityCheck ( byte[] pRndRC, byte bPps1 )
		{
			return phalMfp_Sw_Cmd_VerifyProximityCheck ( m_pDataParams, pRndRC, bPps1 );
		}
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for MIFARE Plus software component.
		/// </summary>
		///
		/// <param name="pPalMifare">Pointer to a palMifare component context</param>
		/// <param name="pKeyStore">Pointer to a KeyStore component context</param>
		/// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption</param>
		/// <param name="pCryptoMac">Pointer to a Crypto component context for Macing</param>
		/// <param name="pCryptoRng">Pointer to a CryptoRng component context</param>
		/// <param name="pCryptoDiversify">Pointer to the parameter structure of the CryptoDiversify layer (can be NULL)</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMac,
			CryptoRng.Generic pCryptoRng, CryptoSym.Generic pCryptoDiversify )
		{
			return phalMfp_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
				( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
				( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams,
				( pCryptoMac == null ) ? IntPtr.Zero : pCryptoMac.m_pDataParams,
				( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
				( pCryptoDiversify == null ) ? IntPtr.Zero : pCryptoDiversify.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for MIFARE Plus software component.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure</param>
		/// <param name="pPalMifare">Pointer to a palMifare component context</param>
		/// <param name="pKeyStore">Pointer to a KeyStore component context</param>
		/// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption</param>
		/// <param name="pCryptoMac">Pointer to a Crypto component context for Macing</param>
		/// <param name="pCryptoRng">Pointer to a CryptoRng component context</param>
		/// <param name="pCryptoDiversify">Pointer to the parameter structure of the CryptoDiversify layer (can be NULL)</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( int wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc,
			CryptoSym.Generic pCryptoMac, CryptoRng.Generic pCryptoRng, CryptoSym.Generic pCryptoDiversify )
		{
			return phalMfp_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
				( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
				( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams,
				( pCryptoMac == null ) ? IntPtr.Zero : pCryptoMac.m_pDataParams,
				( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
				( pCryptoDiversify == null ) ? IntPtr.Zero : pCryptoDiversify.m_pDataParams );
		}
#endif
		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public Sw ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~Sw ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion
	}

	#endregion

	#region Sam_NonX
	#region SamAV2
	/// <summary>
	/// Class for Sam AV2 Non X (S Mode) layer initialization interface and data params.
	/// </summary>
	public class SamAV2 : alMfp.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE Plus SAM AV2 NonX layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Key Modifier for MIFARE Plus SL2 authentication. </summary>
			public fixed byte bKeyModifier[6];
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_SamAV2_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for ALL MIFARE Puls to communicate with SAM AV2 in S Mode.
		/// </summary>
		///
		/// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV2 pHalSam, palMifare.Generic pPalMifare )
		{
			return phalMfp_SamAV2_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for ALL MIFARE Puls to communicate with SAM AV2 in S Mode.
		/// </summary>
		///
		/// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
		/// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///
		public Status_t Init ( int wDataParamSize, Hal.SamAV2 pHal, palMifare.Generic pPalMifare )
		{
			return phalMfp_SamAV2_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif
		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor.
		/// </summary>
		public SamAV2 ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV2 ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion
	}
	#endregion SamAV2

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
	#region SamAV3
	/// <summary>
	/// Class for Sam AV3 Non X (S Mode) layer initialization interface and data params.
	/// </summary>
	public class SamAV3_NonX : alMfp.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE Plus SAM AV3 NonX layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Key Modifier for MIFARE Plus SL2 authentication. </summary>
			public fixed byte bKeyModifier[6];
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV3 in S Mode.
		/// </summary>
		///
		/// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV3 pHal, palMifare.Generic pPalMifare )
		{
			return phalMfp_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV3 in S Mode.
		/// </summary>
		///
		/// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
		/// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHal, palMifare.Generic pPalMifare )
		{
			return phalMfp_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif
		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV3_NonX ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV3_NonX ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion
	}
	#endregion
#endif
#endregion Sam_NonX

	#region Sam_X
	#region SamAV2_X
	/// <summary>
	/// Class for Sam AV2 X layer initialization interface and data params.
	/// </summary>
	public class SamAV2_X : alMfp.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE Plus SAM AV2 X layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public struct DataParams_t
		{
			/// <summary> Layer ID for this component. </summary>
			public ushort wId;

			/// <summary> Pointer to SamAV2 parameter structure. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Pointer to palMifare parameter structure. </summary>
			public IntPtr pPalMifareDataParams;
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams, IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV2 in X Mode.
		/// </summary>
		///
		/// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV2 pHal, palMifare.SamAV2_X pPalMifare )
		{
			return phalMfp_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV2 in X Mode.
		/// </summary>
		///
		/// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
		/// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( int wDataParamSize, Hal.SamAV2 pHal, palMifare.SamAV2_X pPalMifare )
		{
			return phalMfp_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif
		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV2_X ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV2_X ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion
	}
	#endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
	#region SamAV3_X
	/// <summary>
	/// Class for Sam AV3 X layer initialization interface and data params.
	/// </summary>
	public class SamAV3_X : alMfp.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE Plus SAM AV3 X layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public struct DataParams_t
		{
			/// <summary> Layer ID for this component. </summary>
			public ushort wId;

			/// <summary> Pointer to SamAV3 parameter structure. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Pointer to palMifare parameter structure. </summary>
			public IntPtr pPalMifareDataParams;
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfp_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams, IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV3 in X Mode.
		/// </summary>
		///
		/// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init (Hal.SamAV3 pHal, palMifare.SamAV3_X pPalMifare )
		{
			return phalMfp_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for ALL MIFARE Plus to communicate with SAM AV3 in X Mode.
		/// </summary>
		///
		/// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
		/// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHal, palMifare.SamAV3_X pPalMifare )
		{
			return phalMfp_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV3_X ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV3_X ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}
		#endregion
	}
	#endregion
#endif
#endregion
}
