/*
 * Copyright 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfpEVx
{
    #region Enumerations
    #region Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary> MFP EV1 Authentication Error.
        /// 1. Access condition not fulfilled (the current command is not allowed on the requested block,
        ///    or any of the requested blocks, when authorization is done with the key that was used in the
        ///    latest authentication step) or
        ///
        /// 2. The block that is to be accessed is not a value block and an operation is performed that only
        /// can work on value blocks
        ///
        /// 3. Invalid padding of the value.
        /// 4. Trying to read invalid SectorTrailer.
        /// 5. The comparison of the random number in the authentication process failed.
        /// 6. EV0 secure messaging mode requested while not allowed by the PD configuration.
        /// </summary>
        AUTH = ( CustomCodes.ERROR_BEGIN ),

        /// <summary> Field to indicate Command Overflow error.
        /// Too many read or write commands in the session or in the transaction.
        /// Note that this error code is also used in the Proximity Checks.
        /// </summary>
        CMD_OVERFLOW,

        /// <summary> Field to indicate MAC PCD error.
        /// Invalid or missing MAC on the command sent by the PCD.
        /// </summary>
        MAC_PCD,

        /// <summary> Field to indicate Invalid Block Number error.
        /// Invalid Block number: not existing in the implementation or not valid to target with this command.
        /// </summary>
        BNR,

        /// <summary> Field to indicate Number of blocks not supported error.
        /// Invalid Amount of Blocks requested to be written or read.
        /// </summary>
        EXT,

        /// <summary> Field to indicate Command Invalid error.
        /// Command is received in a state where this command is not supported, or a totally unknown
        /// command is received. E.g. a Read or Write command is received while VCState SL3MFPAuthenticated
        /// </summary>
        CMD_INVALID,

        /// <summary> Field to indicate Format error.
        /// Format of the command is not correct (e.g. too many or too few bytes).
        ///
        /// Note: The ErrBNotExist (not to be confused with ErrBnr) used in MFP SL0 is also mapped on
        ///       ErrFormat.
        /// </summary>
        FORMAT,

        /// <summary> Field to indicate General error.
        /// 1. Failure in the operation of the PD, e.g. cannot write the data block.
        /// 2. Increment or IncrementTransfer command causes overflow.
        /// 3. Decrement command or DecrementTransfer command causes underflow.
        /// </summary>
        GEN_FAILURE,

        /// <summary> Field to indicate Transaction MAC related error.
        /// 1. Trying to read/write blocks from sectors protected by different KeyID.TransactionMACKey.
        /// 2. Trying to update blocks from sectors where the related KeyID.TransactionMACKey is configured
        ///    to require Cmd.CommitReaderID while no ReaderID has been committed yet.
        /// 3. Cmd.CommitReaderID while a ReaderID has been committed already.
        /// </summary>
        TM,

        /// <summary> Field to indicate Not Supported error.
        /// The targeted functionality is not supported in the current implementation.
        /// E.g. update of activation parameters on mobile implementations.
        /// </summary>
        NOT_SUP,

        /// <summary> Field to indicate Format error.
        /// </summary>
        ISO7816_WRONG_LENGTH,

        /// <summary> Field to indicate Format error.
        /// </summary>
        ISO7816_WRONG_PARAMS,

        /// <summary> Field to indicate Format error.
        /// </summary>
        ISO7816_WRONG_LC,

        /// <summary> Field to indicate Format error.
        ///  </summary>
        ISO7816_WRONG_LE,

        /// <summary> Field to indicate Format error.
        /// </summary>
        ISO7816_WRONG_CLA
    }
    #endregion Error Codes

    #region ISO14443 options
    /// <summary>
    /// Options to indicate the ISO14443 protocol to be used for Authentication and other commands.
    /// Options to indicate the ISO14443 activation for SetConfigSL1 command.
    /// </summary>
    public enum Iso144443 : byte
    {
        #region For Authentication and other commands
        /// <summary> Field to indicate the Layer 3 protocol to be used. </summary>
        L3 = 0x00,

        /// <summary> Field to indicate the Layer 4 protocol to be used. </summary>
        L4 = 0x01,
        #endregion

        #region For SetConfigSL1 command
        /// <summary> Field to indicate the Layer 4 protocol to be used. </summary>
        ENABLE = 0x00,

        /// <summary> Field to indicate the Layer 4 protocol to be used. </summary>
        DISABLE = 0x01
        #endregion
    }
    #endregion ISO14443 options

    #region CommitPerso options
    /// <summary>
    /// Options to switch the Security Level to 1 or 3.
    /// </summary>
    public enum Commit : byte
    {
        /// <summary> Option to maintain the backward compatibility with MIFARE Plus PICC. </summary>
        MAINTAIN_BACKWARD_COMPATIBLE = 0x00,

        /// <summary> Option to switch the Security Level to 1. </summary>
        TO_SL1 = 0x01,

        /// <summary> Option to switch the Security Level to 3. </summary>
        TO_SL3 = 0x03
    }
    #endregion CommitPerso options

    #region MIFARE Classic key types
    /// <summary>
    /// Key type options for MIFARE Classic Authentication.
    /// </summary>
    public enum KeyType : byte
    {
        /// <summary> MIFARE(R) Key A. </summary>
        A = 0x0A,

        /// <summary> MIFARE(R) Key B. </summary>
        B = 0x0B
    }
    #endregion MIFARE Classic key types

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Authentication options
    /// <summary>
    /// Options to indicate the Authentication type to be performed.
    /// </summary>
    public enum Auth : byte
    {
        /// <summary> Option to indicate the authenticate type as non-first or following. </summary>
        NON_FIRST = 0x00,

        /// <summary> Option to indicate the authenticate type as first. </summary>
        FIRST = 0x01,

        /// <summary> Option to indicate the authenticate type as first Alternate. </summary>
        FIRST_ALTERNATE = 0x02
    }

    /// <summary>
    /// Options to indicate the type of diversification to be performed for Sector Switch Authenticate command.
    /// All these macros are applicable for Sam X and S mode only. For Software mode, these macros has no meaning even
    /// though they are passed.
    /// </summary>
    public enum SS_Auth : byte
    {
        /// <summary> Option to indicate that the diversification is disabled. </summary>
        NO_DIVERSIFICATION = 0x00,

        /// <summary> Option to indicate that the diversification is enabled for Sector Switch keys (9006 or 9007). </summary>
        SECTOR_SWITCH_DIVERSIFICATION = 0x01,

        /// <summary> Option to indicate that the diversification is enabled for AES Sector keys (4001, 4003, etc...). </summary>
        SECTOR_DIVERSIFICATION = 0x02,

        /// <summary> Option to indicate that the diversification is enabled for AES Sector keys (4001, 4003, etc...).  using a Master key. </summary>
        MASTER_SECTOR_DIVERSIFICATION = 0x04
    }
    #endregion Authentication options

    #region Data Encryption options
    /// <summary>
    /// Options to indicate the communication mode.
    /// </summary>
    public enum Encryption : byte
    {
        /// <summary> Option to indicate the communication between PCD and PICC is plain. </summary>
        OFF = 0x00,

        /// <summary> Option to indicate the communication between PCD and PICC is encrypted. </summary>
        ON = 0x01
    }
    #endregion Data Encryption options

    #region Mac On Command options
    /// <summary>
    /// Options to indicate the communication mode as maced for PCD to PICC transfer.
    /// </summary>
    public enum MacOnCmd : byte
    {
        /// <summary> Option to indicate the communication is not maced for PCD to PICC transfer. </summary>
        OFF = 0x00,

        /// <summary> Option to indicate the communication is maced for PCD to PICC transfer. </summary>
        ON = 0x01
    }
    #endregion Mac On Command options

    #region Mac On Response options
    /// <summary>
    /// Options to indicate the communication mode as maced for PICC to PCD transfer.
    /// </summary>
    public enum MacOnRsp : byte
    {
        /// <summary> Option to indicate the communication is not maced for PICC to PCD transfer. </summary>
        OFF = 0x00,

        /// <summary> Option to indicate the communication is maced for PICC to PCD transfer. </summary>
        ON = 0x01
    }
    #endregion Mac On Command options
#endif

    #region Personalize UID options
    /// <summary>
    /// Options to personalize the UID.
    /// </summary>
    public enum PersonalizeUID : byte
    {
        /// <summary> MIFARE(R) Plus EV1 UID type UIDF0. </summary>
        UIDF0 = 0x00,

        /// <summary> MIFARE(R) Plus EV1 UID type UIDF1. </summary>
        UIDF1 = 0x40,

        /// <summary> MIFARE(R) Plus EV1 UID type UIDF2. </summary>
        UIDF2 = 0x20,

        /// <summary> MIFARE(R) Plus EV1 UID type UIDF3. </summary>
        UIDF3 = 0x60
    }
    #endregion Personalize UID options

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Diversification options
    /// <summary>
    /// Options to diversify the key.
    /// </summary>
    public enum Diversify : ushort
    {
        /// <summary> Option to disable key diversification.</summary>
        OFF = 0xFFFF,

        /// <summary> Option to enable key diversification.</summary>
        ON = 0x0000
    }
    #endregion Diversification options
#endif

    #region Configuration options
    /// <summary>
    /// Option to configure some special operations.
    /// </summary>
    public enum Config : ushort
    {
        /// <summary> Option for GetConfig/SetConfig to get/set current status of command wrapping in ISO 7816-4 APDUs.</summary>
        WRAPPED_MODE = 0xA1,

        /// <summary> Option for GetConfig/SetConfig to get/set current status of extended wrapping in ISO 7816-4 APDUs.</summary>
        EXTENDED_APDU = 0xA2,

        /// <summary> Option to set the Authentication mode to perform negative testing.</summary>
        AUTH_MODE = 0xA3
    }

    /// <summary>
    /// Option to enable or disable the Wrapped or Extended APDU options.
    /// Option to configure the Authentication state.
    /// </summary>
    public enum Value : ushort
    {
        #region For Wrapped, Extended mode configuration
        /// <summary> Option to disable Wrapping or Extended Length APDU feature for ISO7816 support.</summary>
        DISABLE = 0x00,

        /// <summary> Option to enable Wrapping or Extended Length APDU feature for ISO7816 support.</summary>
        ENABLE = 0x01,

        /// <summary> Default Option value. This is equal to Disable.</summary>
        DEFAULT = DISABLE,
        #endregion

        #region For Authentication mode configuration
        /// <summary> Option to indicate the Authentication mode as MFP EV1 not authenticated.</summary>
        NOTAUTHENTICATED = 0x00,

        /// <summary> Option to indicate the Authentication mode as MFP EV1 SL1 MIFARE Authentication mode.</summary>
        SL1_MIFARE_AUTHENTICATED = 0x01,

        /// <summary> Option to indicate the Authentication mode as MFP EV1 SL1 Authentication mode.</summary>
        SL1_MFP_AUTHENTICATED = 0x02,

        /// <summary> Option to indicate the Authentication mode as MFP EV1 SL3 Authentication mode.</summary>
        SL3_MFP_AUTHENTICATED = 0x03,

        /// <summary> Option to indicate the Authentication mode as MFP EV1 not authenticated in ISO Layer 3.</summary>
        NOT_AUTHENTICATED_L3 = 0x04,

        /// <summary> Option to indicate the Authentication mode as MFP EV1 not authenticated in ISO Layer 4.</summary>
        NOT_AUTHENTICATED_L4 = 0x05
        #endregion
    }
    #endregion Configuration options
    #endregion Enumerations

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    [NxpRdLibAttribute
        (
            MIFARE_Products.PLUS_EV1,
            Message = "The next evolution of MIFARE Plus, the field-proven MIFARE Classic AES-128 security upgrade product, includes an expanded feature set that enables " +
                      "over-the-air services and selective upgrades of security-relevant applications."
        )
    ]
    [NxpRdLibAttribute
        (
            MIFARE_Products.PLUS_EV2,
            Message = "This second-generation MIFARE Plus IC, equipped with enhanced performance and security features, along with support for mobile services and over-the-air updates, is a " +
                       "quick, cost-effective way to upgrade existing contactless infrastructures to AES security, and also delivers better experiences, with greater convenience, in Smart City services."
        )
    ]
    public abstract class Generic
    {
        #region Constants
        public const byte SIG_LENGTH = 56;
        #endregion

        #region DLL Imports
        #region Personalization commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_WritePerso ( IntPtr pDataParams, byte bLayer4Comm, ushort wBlockNr, byte bNumBlocks,
            byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_CommitPerso ( IntPtr pDataParams, byte bOption, byte bLayer4Comm );
        #endregion Personalization commands

        #region Authentication commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_AuthenticateMfc ( IntPtr pDataParams, byte bBlockNo, byte bKeyType, ushort wKeyNumber,
            ushort wKeyVersion, byte[] pUid, byte bUidLength );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_AuthenticateSL0 ( IntPtr pDataParams, byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr,
            ushort wKeyNumber, ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In,
            byte[] pPcdCap2Out, byte[] pPdCap2 );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_AuthenticateSL1 ( IntPtr pDataParams, byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr,
            ushort wKeyNumber, ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In,
            byte[] pPcdCap2Out, byte[] pPdCap2 );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_AuthenticateSL3 ( IntPtr pDataParams, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber,
            ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput, byte bLenPcdCap2, byte[] pPcdCap2In, byte[] pPcdCap2Out, byte[] pPdCap2 );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SSAuthenticate ( IntPtr pDataParams, byte bOption, ushort wSSKeyBNr, ushort wSSKeyNr,
            ushort wSSKeyVer, byte bLenDivInputSSKey, byte[] pDivInputSSKey, byte bSecCount, ushort[] pSectorNos, ushort[] pKeyNos,
            ushort[] pKeyVers, byte bLenDivInputSectorKeyBs, byte[] pDivInputSectorKeyBs );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_AuthenticatePDC ( IntPtr pDataParams, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
            byte bLenDivInput, byte[] pDivInput, byte bUpgradeInfo );
#endif
        #endregion Authentication commands

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Data operation commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Write ( IntPtr pDataParams, byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte bNumBlocks,
            byte[] pBlocks, byte[] pTMC, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Read ( IntPtr pDataParams, byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr,
            byte bNumBlocks, byte[] pBlocks );
        #endregion Data operation commands

        #region Value operation commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_WriteValue ( IntPtr pDataParams, byte bEncrypted, byte bWriteMaced, ushort wBlockNr,
            byte[] pValue, byte bAddrData, byte[] pTMC, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ReadValue ( IntPtr pDataParams, byte bEncrypted, byte bReadMaced, byte bMacOnCmd,
            ushort wBlockNr, byte[] pValue, ref byte pAddrData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Increment ( IntPtr pDataParams, byte bIncrementMaced, ushort wBlockNr, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Decrement ( IntPtr pDataParams, byte bDecrementMaced, ushort wBlockNr, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_IncrementTransfer ( IntPtr pDataParams, byte bIncrementTransferMaced, ushort wSourceBlockNr,
            ushort wDestinationBlockNr, byte[] pValue, byte[] pTMC, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_DecrementTransfer ( IntPtr pDataParams, byte bDecrementTransferMaced, ushort wSourceBlockNr,
            ushort wDestinationBlockNr, byte[] pValue, byte[] pTMC, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Transfer ( IntPtr pDataParams, byte bTransferMaced, ushort wBlockNr, byte[] pTMC, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Restore ( IntPtr pDataParams, byte bRestoreMaced, ushort wBlockNr );
        #endregion Value operation commands
#endif

        #region Special commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_GetVersion ( IntPtr pDataParams, byte[] pVerInfo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ReadSign ( IntPtr pDataParams, byte bLayer4Comm, byte bAddr, ref IntPtr pSignature );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ResetAuth ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_PersonalizeUid ( IntPtr pDataParams, byte bUidType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SetConfigSL1 ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ReadSL1TMBlock ( IntPtr pDataParams, ushort wBlockNr, byte[] pBlocks );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ChangeKey ( IntPtr pDataParams, byte bChangeKeyMaced, ushort wBlockNr, ushort wKeyNumber,
            ushort wKeyVersion, byte bLenDivInput, byte[] pDivInput );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_CommitReaderID ( IntPtr pDataParams, ushort wBlockNr, byte[] pTMRI, byte[] pEncTMRI );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_VCSupportLastISOL3 ( IntPtr pDataParams, byte[] pIid, byte[] pPcdCapabilities,
            ref byte info );
        #endregion Special commands

        #region Utilities
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_ResetSecMsgState ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SetConfig ( IntPtr pDataParams, ushort wOption, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_GetConfig ( IntPtr pDataParams, ushort wOption, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SetVCAParams ( IntPtr pDataParams, IntPtr pAlVCADataParams );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_CalculateTMV ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pTMI,
            uint dwTMILen, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_DecryptReaderID ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pEncTMRI,
            byte[] pTMRIPrev );
#endif
        #endregion Utilities
        #endregion

        #region Wrapper Functions
        #region Personalization commands
        /// <summary>
        /// Performs a Write Perso command. The Write Perso command can be executed using the ISO14443-3 communication
        /// protocol (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation).
        /// </summary>
        ///
        /// <param name="bLayer4Comm">ISO14443 protocol to be used.
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L3
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L4</param>
        /// <param name="wBlockNr">Block number to be personalized.</param>
        /// <param name="bNumBlocks">Number of blocks to be personalized.
        ///                             15 block for Native communication if configured as Software component.
        ///                             15 block for Native communication if configured as Sam NonX component.
        ///                             13 blocks if configured as Sam X component.</param>
        /// <param name="pValue">The value for the block mentioned in BlockNr parameter.
        ///                         If number of blocks is 1, the length should be 16 bytes.
        ///                         If number of blocks more than 1, the length should be (NoBlocks * 16) bytes.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Personalize Sector Traliers, AES Keys, DaVa Blocks and Special block(MFPConfiguration, Field Configuration, etc...) from the initial delivery configuration to a customer specific value."
            )
        ]
        public Status_t WritePerso ( byte bLayer4Comm, int wBlockNr, byte bNumBlocks, byte[] pValue )
        {
            return phalMfpEVx_WritePerso ( m_pDataParams, bLayer4Comm, ( ushort ) wBlockNr, bNumBlocks, pValue );
        }

        /// <summary>
        /// Performs a Commit Perso command. The Commit Perso command can be executed using the ISO14443-3 communication protocol
        /// (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation). This command commits the written
        /// data during WritePerso command and switches the SecurityLevel to 1 or 3 based on the option provided.
        ///
        /// If the Option parameter is 0, only the command code will be exchanges to PICC. This is to maintain the backward
        /// compatibility with MIFARE Plus PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option to be used for Security Level switching.
        ///                             NxpRdLibNet.alMfpEVx.Commit.MAINTAIN_BACKWARD_COMPATIBLE
        ///                             NxpRdLibNet.alMfpEVx.Commit.TO_SL1
        ///                             NxpRdLibNet.alMfpEVx.Commit.TO_SL3</param>
        /// <param name="bLayer4Comm">ISO14443 protocol to be used.
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L3
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L4</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Finalize the personalization and switch up to SL1 or SL3."
            )
        ]
        public Status_t CommitPerso ( byte bOption, byte bLayer4Comm )
        {
            return phalMfpEVx_CommitPerso ( m_pDataParams, bOption, bLayer4Comm );
        }
        #endregion Personalization commands

        #region Authentication commands
        /// <summary>
        /// Perform MIFARE Authenticate command in Security Level 1 with MIFARE CLASSIC PICC.
        /// </summary>
        ///
        /// <param name="bBlockNo">PICC block number to be used for authentication.
        ///                             NxpRdLibNet.alMfpEVx.KeyType.A
        ///                             NxpRdLibNet.alMfpEVx.KeyType.B</param>
        /// <param name="bKeyType">Authentication key type to be used.
        ///                         </param>
        /// <param name="wKeyNumber">Key number to used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version to used from software or hardware KeyStore.</param>
        /// <param name="pUid">UID of the PICC received during anti-collision sequence.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Performs Authentication using MIFARE Keys.",
                OtherInfo = "This command is supported in Security Level 1 only."
            )
        ]
        public Status_t AuthenticateMfc ( byte bBlockNo, byte bKeyType, ushort wKeyNumber, ushort wKeyVersion, byte[] pUid )
        {
            return phalMfpEVx_AuthenticateMfc ( m_pDataParams, bBlockNo, bKeyType, wKeyNumber, wKeyVersion, pUid, ( byte ) pUid.Length );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Performs a MIFARE Plus Authentication for Security Level 0.This command performs basic Authenticate First / Non-First
        /// command execution and also performs the AuthenticateContinue command internally.
        /// </summary>
        ///
        /// <param name="bLayer4Comm">ISO14443 protocol to be used.
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L3
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L4</param>
        /// <param name="bFirstAuth">Type of authentication to be performed.
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.NON_FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST_ALTERNATE</param>
        /// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
        /// <param name="wKeyNumber">Key number be to used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version be to used from software or hardware KeyStore.</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
        ///                          If length is non zero, PCDCapabilites should be available.
        ///                          If length is zero, PCDCapabilites buffer should be NULL.</param>
        /// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
        /// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Mutual authentication to setup a secure authenticated channel.",
                OtherInfo = "Authentication to Originality keys only."
            )
        ]
        public Status_t AuthenticateSL0 ( byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
            byte[] pDivInput, byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2 )
        {
            pPcdCap2Out = new byte[6];
            pPdCap2 = new byte[6];
            return phalMfpEVx_AuthenticateSL0 ( m_pDataParams, bLayer4Comm, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput,
                ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
                pPcdCap2In, pPcdCap2Out, pPdCap2 );
        }

        /// <summary>
        /// Performs a MIFARE Plus Authentication for Security Level 1. This command performs basic Authenticate First / Non-First
        /// command execution and also performs the AuthenticateContinue command internally.
        /// </summary>
        ///
        /// <param name="eLayer4Comm">ISO14443 protocol to be used.
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L3
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L4</param>
        /// <param name="eFirstAuth">Type of authentication to be performed.
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.NON_FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST_ALTERNATE</param>
        /// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
        /// <param name="wKeyNumber">Key number be to used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version be to used from software or hardware KeyStore.</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
        ///                          If length is non zero, PCDCapabilites should be available.
        ///                          If length is zero, PCDCapabilites buffer should be NULL.</param>
        /// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
        /// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Mutual authentication to setup a secure authenticated channel.",
                OtherInfo = "Authentication using all the supported AES key that are for Authentication purpose only."
            )
        ]
        public Status_t AuthenticateSL1 ( byte bLayer4Comm, byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
            byte[] pDivInput, byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2 )
        {
            pPcdCap2Out = new byte[6];
            pPdCap2 = new byte[6];
            return phalMfpEVx_AuthenticateSL1 ( m_pDataParams, bLayer4Comm, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput,
                ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
                pPcdCap2In, pPcdCap2Out, pPdCap2 );
        }

        /// <summary>
        /// Performs a MIFARE Plus Authentication for Security Level 3. This command performs basic Authenticate First / Non-First
        /// command execution and also performs the AuthenticateContinue command internally.
        /// </summary>
        ///
        /// <param name="eFirstAuth">Type of authentication to be performed.
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.NON_FIRST
        ///                             NxpRdLibNet.alMfpEVx.Authentication.FIRST_ALTERNATE</param>
        /// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
        /// <param name="wKeyNumber">Key number be to used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version be to used from software or hardware KeyStore.</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="pPcdCap2In">Buffer containing the Input PCD Capabilities.
        ///                          If length is non zero, PCDCapabilites should be available.
        ///                          If length is zero, PCDCapabilites buffer should be NULL.</param>
        /// <param name="pPcdCap2Out">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
        /// <param name="pPdCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Mutual authentication to setup a secure authenticated channel.",
                OtherInfo = "Authentication using all the supported AES key that are for Authentication purpose only."
            )
        ]
        public Status_t AuthenticateSL3 ( byte bFirstAuth, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion,
            byte[] pDivInput, byte[] pPcdCap2In, out byte[] pPcdCap2Out, out byte[] pPdCap2 )
        {
            pPcdCap2Out = new byte[6];
            pPdCap2 = new byte[6];
            return phalMfpEVx_AuthenticateSL3 ( m_pDataParams, bFirstAuth, wBlockNr, wKeyNumber, wKeyVersion,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput,
                ( byte ) ( ( pPcdCap2In == null ) ? 0 : pPcdCap2In.Length ),
                pPcdCap2In, pPcdCap2Out, pPdCap2 );
        }

        /// <summary>
        /// Perform MIFARE(R) Sector switch authentication command. This command is valid in Security Level 1 only. This command
        /// performs basic Authenticate First / Non-First command execution and also performs the AuthenticateContinue command internally.
        /// </summary>
        ///
        /// <param name="bOption">Options for key diversification. Only applicable for SAM component,
        ///                       ignored for Software component.
        ///                       For Sam,
        ///                         NxpRdLibNet.alMfpEVx.SS_Auth.NO_DIVERSIFICATION
        ///                         NxpRdLibNet.alMfpEVx.SS_Auth.SECTOR_SWITCH_DIVERSIFICATION
        ///                         NxpRdLibNet.alMfpEVx.SS_Auth.SECTOR_DIVERSIFICATION
        ///                         NxpRdLibNet.alMfpEVx.SS_Auth.MASTER_SECTOR_DIVERSIFICATION</param>
        /// <param name="wSSKeyBNr">PICC block number to be used for Sector Switch authentication.</param>
        /// <param name="wSSKeyNr">Key number to be used from software or hardware KeyStore for sector switch key(SSKey)</param>
        /// <param name="wSSKeyVer">Key number to be used from software or hardware KeyStore for sector switch key(SSKey)</param>
        /// <param name="bLenDivInputSSKey">Length of diversification input used to diversify the Sector Switch key.
        ///                                     In Software, if 0, no diversification is performed.
        ///                                     If Sam, only if <seealso cref="NxpRdLibNet.alMfpEVx.SS_Auth.SECTOR_SWITCH_DIVERSIFICATION"/> is set, diversification
        ///                                     of Sector Switch key will be performed. In this case, there should be proper length available.</param>
        /// <param name="pDivInputSSKey">Diversification Input used to diversify the sector switch key.</param>
        /// <param name="bSecCount">Number of sectors available in SectorNos buffer.</param>
        /// <param name="pSectorNos">The list of AES sector B key numbers for switching the sectors.</param>
        /// <param name="pKeyNos">If Option is set to use the Master sector key, then the master sector key number
        ///                       to be used from software or hardware KeyStore should be passed, else individual
        ///                       Sector B key number to be used from software or hardware KeyStore should be passed.</param>
        /// <param name="pKeyVers">If Option is set to use the Master sector key, then the master sector key version
        ///                        to be used from software or hardware KeyStore should be passed, else individual
        ///                        Sector B key version to be used from software or hardware KeyStore should be passed.</param>
        /// <param name="bLenDivInputSectorKeyBs">Length of diversification input used to diversify the AES Sector B key.
        ///                                       For SAM if length is 0, the diversification input passed for Sector Switch key will be used.
        ///                                       For SW if length is 0, no diversification is performed.</param>
        /// <param name="pDivInputSectorKeyBs">Diversification Input used to diversify the AES Sector B key.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Mutual authentication to switch one or more sectors to a higher security level.",
                OtherInfo = "L3 Sector Switch and L1L3 Mix Sector Switch Keys are valid ones."
            )
        ]
        public Status_t SSAuthenticate ( byte bOption, ushort wSSKeyBNr, ushort wSSKeyNr, ushort wSSKeyVer, byte bLenDivInputSSKey,
            byte[] pDivInputSSKey, byte bSecCount, ushort[] pSectorNos, ushort[] pKeyNos, ushort[] pKeyVers, byte bLenDivInputSectorKeyBs,
            byte[] pDivInputSectorKeyBs )
        {
            return phalMfpEVx_SSAuthenticate ( m_pDataParams, bOption, wSSKeyBNr, wSSKeyNr, wSSKeyVer, bLenDivInputSSKey, pDivInputSSKey, bSecCount,
                pSectorNos, pKeyNos, pKeyVers, bLenDivInputSectorKeyBs, pDivInputSectorKeyBs );
        }

        /// <summary>
        /// Perform MIFARE(R) Sector switch authentication command. This command performs the AuthenticateContinue command internally.
        /// +-----------------+---------------+
        /// | CardSize/SubType| bUpgrade Info |
        /// +-----------------+---------------+
        /// |      0.5k       |      0xX0     |
        /// |      1k         |      0xX1     |
        /// |      2k         |      0xX2     |
        /// |      4k         |      0xX4     |
        /// |      8k         |      0xX8     |
        /// |      RFU        |   Other data  |
        /// +-----------------+---------------+
        /// </summary>
        ///
        /// <param name="wBlockNr">PICC Block number to be used for authentication.</param>
        /// <param name="wKeyNumber">Key number to used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version to used from software or hardware KeyStore.</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bUpgradeInfo">The upgrade info input.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Mutual authentication for upgrading the card via post-delivery configuration."
            )
        ]
        public Status_t AuthenticatePDC ( ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput, byte bUpgradeInfo )
        {
            return phalMfpEVx_AuthenticatePDC ( m_pDataParams, wBlockNr, wKeyNumber, wKeyVersion,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput, bUpgradeInfo );
        }
#endif
        #endregion Authentication commands

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Data operation commands
        /// <summary>
        /// Performs a Write / Write MACed command. This command writes a 16 byte data to the PICC.
        /// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
        ///
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to which the data should be written.</param>
        /// <param name="bNumBlocks">Number of blocks to write.
        ///                             15 block for Native communication if configured as Software component.
        ///                             13 blocks if configured as Sam X or Sam NonX component.</param>
        /// <param name="pBlocks">The data to be written. This buffer should have data equal to NumBlocks * 16,
        ///                       where 16 is one block size.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted and WriteMaced parameters. Returns TMC / TMV if the block is a TMProtected block.",
                Message = "Write the data from 1 up to 15 blocks. The size of the data should be NoOfBlocks * 16 bytes."
            )
        ]
        public Status_t Write ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte bNumBlocks, byte[] pBlocks,
            out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            return phalMfpEVx_Write ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, bNumBlocks, pBlocks, pTMC, pTMV );
        }

        /// <summary>
        /// Performs a Read / Read MACed command.
        /// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
        /// not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.ON</param>
        /// <param name="wBlockNr">PICC block number from which the data should be read.</param>
        /// <param name="bNumBlocks">Number of blocks to read.
        ///                             15 block for Native communication if configured as Software component.
        ///                             13 blocks if configured as Sam X or Sam NonX component.</param>
        /// <param name="pBlocks">The data to be read. This buffer should have data equal to NumBlocks * 16,
        ///                       where 16 is one block size.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Read the data from one or multiple blocks. The size of the data will be NoOfBlocks * 16 bytes."
            )
        ]
        public Status_t Read ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, byte bNumBlocks, out byte[] pBlocks )
        {
            pBlocks = new byte[16 * bNumBlocks + 8];
            Status_t oStatus = phalMfpEVx_Read ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, bNumBlocks, pBlocks );
            Array.Resize ( ref pBlocks, 16 * bNumBlocks );

            return oStatus;
        }
        #endregion Data operation commands

        #region Value operation commands
        #region Write Value
        /// <summary>
        /// Performs a Write / Write MACed command of a value.
        /// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to which the value should be written.</param>
        /// <param name="pValue">The value to be written. This buffer should have value equal to 4 bytes.</param>
        /// <param name="bAddrData">The address to be written.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted and WriteMaced parameters. Returns TMC / TMV if the block is a TMProtected block.",
                Message = "Write the 4 byte value and address."
            )
        ]
        public Status_t WriteValue ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, byte[] pValue, byte bAddrData, out byte[] pTMC,
            out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            return phalMfpEVx_WriteValue ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, pValue, bAddrData, pTMC, pTMV );
        }

        /// <summary>
        /// Performs a Write / Write MACed command of a value.
        /// The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bWriteMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to which the value should be written.</param>
        /// <param name="dwValue">The value to be written. This buffer should have value equal to 4 bytes.</param>
        /// <param name="bAddrData">The address to be written.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted and WriteMaced parameters. Returns TMC / TMV if the block is a TMProtected block.",
                Message = "Write the 4 byte value and address."
            )
        ]
        public Status_t WriteValueDw ( byte bEncrypted, byte bWriteMaced, ushort wBlockNr, int dwValue, byte bAddrData, out byte[] pTMC,
            out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfpEVx_WriteValue ( m_pDataParams, bEncrypted, bWriteMaced, wBlockNr, pValue, bAddrData, pTMC, pTMV );
        }
        #endregion Write Value

        #region Read Value
        /// <summary>
        /// Performs a Read / Read MACed Value command.
        /// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
        /// not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.ON</param>
        /// <param name="wBlockNr">PICC block number from which the value should be read.</param>
        /// <param name="pValue">The value read from the specified block number. The buffer will have 4 bytes
        ///                      of value information.</param>
        /// <param name="pAddrData">The address from the read value information.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Reads the 4 byte value and address."
            )
        ]
        public Status_t ReadValue ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, out byte[] pValue, out byte pAddrData )
        {
            pValue = new byte[4];
            pAddrData = 0;
            return phalMfpEVx_ReadValue ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, pValue, ref pAddrData );
        }

        /// <summary>
        /// Performs a Read / Read MACed Value command.
        /// The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
        /// not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bEncrypted">Type of communication to be used. Based on this flag the command code
        ///                          will be updated.
        ///                             NxpRdLibNet.alMfpEVx.Encryption.OFF
        ///                             NxpRdLibNet.alMfpEVx.Encryption.ON</param>
        /// <param name="bReadMaced">Indicate whether the response should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="bMacOnCmd">Indicate whether the command should be maced. Based on this flag the
        ///                           command code will be updated.
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.OFF
        ///                             NxpRdLibNet.alMfpEVx.MacOnCmd.ON</param>
        /// <param name="wBlockNr">PICC block number from which the value should be read.</param>
        /// <param name="dwValue">The value read from the specified block number. The buffer will have 4 bytes
        ///                       of value information.</param>
        /// <param name="pAddrData">The address from the read value information.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Encryption / Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Reads the 4 byte value and address."
            )
        ]
        public Status_t ReadValueDw ( byte bEncrypted, byte bReadMaced, byte bMacOnCmd, ushort wBlockNr, out int dwValue, out byte pAddrData )
        {
            dwValue = 0;
            byte[] pValue = new byte[4];
            pAddrData = 0;
            Status_t status = phalMfpEVx_ReadValue ( m_pDataParams, bEncrypted, bReadMaced, bMacOnCmd, wBlockNr, pValue, ref pAddrData );
            if ( status.Equals ( new Status_t () ) )
            {
                dwValue = ConvertByteArrayToInt ( pValue );
            }
            return status;
        }
        #endregion Read Value

        #region Increment
        /// <summary>
        /// Performs an Increment / Increment MACed command.
        /// The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bIncrementMaced">Indicate whether the response should be maced. Based on this flag the
        ///                               command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for incrementing the value.</param>
        /// <param name="pValue">The value to be incremented. This buffer should have 4 bytes value information.
        ///                      The value to be incremented should be LSB first order.
        ///                      For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
        ///                      0x01, 0x00, 0x00, 0x00.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Increments the value."
            )
        ]
        public Status_t Increment ( byte bIncrementMaced, ushort wBlockNr, byte[] pValue )
        {
            return phalMfpEVx_Increment ( m_pDataParams, bIncrementMaced, wBlockNr, pValue );
        }

        /// <summary>
        /// Performs an Increment / Increment MACed command.
        /// The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bIncrementMaced">Indicate whether the response should be maced. Based on this flag the
        ///                               command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for incrementing the value.</param>
        /// <param name="dwValue">The value to be incremented. This buffer should have 4 bytes value information.
        ///                      The value to be incremented should be LSB first order.
        ///                      For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
        ///                      0x01, 0x00, 0x00, 0x00.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Increments the value."
            )
        ]
        public Status_t IncrementDw ( byte bIncrementMaced, ushort wBlockNr, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfpEVx_Increment ( m_pDataParams, bIncrementMaced, wBlockNr, pValue );
        }
        #endregion Increment

        #region Decrement
        /// <summary>
        /// Performs a Decrement / Decrement MACed command.
        /// The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bDecrementMaced">Indicate whether the response should be maced. Based on this flag the
        ///                               command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for decrementing the value.</param>
        /// <param name="pValue">The value to be decremented. This buffer should have 4 bytes value information.
        ///                      The value to be decremented should be LSB first order.
        ///                      For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                      0x01, 0x00, 0x00, 0x00.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Decrements the value."
            )
        ]
        public Status_t Decrement ( byte bDecrementMaced, ushort wBlockNr, byte[] pValue )
        {
            return phalMfpEVx_Decrement ( m_pDataParams, bDecrementMaced, wBlockNr, pValue );
        }

        /// <summary>
        /// Performs a Decrement / Decrement MACed command.
        /// The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bDecrementMaced">Indicate whether the response should be maced. Based on this flag the
        ///                               command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for decrementing the value.</param>
        /// <param name="dwValue">The value to be decremented. This buffer should have 4 bytes value information.
        ///                      The value to be decremented should be LSB first order.
        ///                      For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                      0x01, 0x00, 0x00, 0x00.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Decrements the value."
            )
        ]
        public Status_t DecrementDw ( byte bDecrementMaced, ushort wBlockNr, int dwValue )
        {
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfpEVx_Decrement ( m_pDataParams, bDecrementMaced, wBlockNr, pValue );
        }
        #endregion Decrement

        #region IncrementTransfer
        /// <summary>
        /// Performs an Increment Transfer / Increment Transfer MACed command.
        /// The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bIncrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
        ///                                       command code will be updated.
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wSourceBlockNr">PICC block number to be used for incrementing the value.</param>
        /// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
        /// <param name="pValue">The value to be incremented and transferred. This buffer should have 4 bytes
        ///                       value information. The value to be incremented and transferred should be LSB first order.
        ///                         For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                         0x01, 0x00, 0x00, 0x00.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Increments the value and saves the value to the specified address."
            )
        ]
        public Status_t IncrementTransfer ( byte bIncrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, byte[] pValue,
            out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            return phalMfpEVx_IncrementTransfer ( m_pDataParams, bIncrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue, pTMC, pTMV );
        }

        /// <summary>
        /// Performs an Increment Transfer / Increment Transfer MACed command.
        /// The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bIncrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
        ///                                       command code will be updated.
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wSourceBlockNr">PICC block number to be used for incrementing the value.</param>
        /// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
        /// <param name="dwValue">The value to be incremented and transferred. This buffer should have 4 bytes
        ///                       value information. The value to be incremented and transferred should be LSB first order.
        ///                         For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                         0x01, 0x00, 0x00, 0x00.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Increments the value and saves the value to the specified address."
            )
        ]
        public Status_t IncrementTransferDw ( byte bIncrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, int dwValue,
            out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfpEVx_IncrementTransfer ( m_pDataParams, bIncrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue, pTMC, pTMV );
        }
        #endregion IncrementTransfer

        #region DecrementTransfer
        /// <summary>
        /// Performs a Decrement Transfer / Decrement Transfer MACed command.
        /// The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bDecrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
        ///                                       command code will be updated.
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wSourceBlockNr">PICC block number to be used for decrementing the value.</param>
        /// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
        /// <param name="pValue">The value to be decremented and transferred. This buffer should have 4 bytes
        ///                       value information. The value to be decremented and transferred should be LSB first order.
        ///                         For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                         0x01, 0x00, 0x00, 0x00.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Decrements the value and saves the value to the specified address."
            )
        ]
        public Status_t DecrementTransfer ( byte bDecrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, byte[] pValue,
            out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            return phalMfpEVx_DecrementTransfer ( m_pDataParams, bDecrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue, pTMC, pTMV );
        }

        /// <summary>
        /// Performs a Decrement Transfer / Decrement Transfer MACed command.
        /// The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bDecrementTransferMaced">Indicate whether the response should be maced. Based on this flag the
        ///                                       command code will be updated.
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                         NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wSourceBlockNr">PICC block number to be used for decrementing the value.</param>
        /// <param name="wDestinationBlockNr">PICC block number to be used for transferring the value.</param>
        /// <param name="pValue">The value to be decremented and transferred. This buffer should have 4 bytes
        ///                       value information. The value to be decremented and transferred should be LSB first order.
        ///                         For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
        ///                         0x01, 0x00, 0x00, 0x00.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Decrements the value and saves the value to the specified address."
            )
        ]
        public Status_t DecrementTransferDw ( byte bDecrementTransferMaced, ushort wSourceBlockNr, ushort wDestinationBlockNr, int dwValue,
            out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            byte[] pValue = ConvertIntToByteArray ( dwValue );
            return phalMfpEVx_DecrementTransfer ( m_pDataParams, bDecrementTransferMaced, wSourceBlockNr, wDestinationBlockNr, pValue, pTMC, pTMV );
        }
        #endregion DecrementTransfer

        #region Transfer
        /// <summary>
        /// Performs a Transfer / Transfer MACed command.
        /// The parameter TransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bTransferMaced">Indicate whether the response should be maced. Based on this flag the
        ///                              command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for transferring the value.</param>
        /// <param name="pTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///                    bytes of Transaction MAC counter information.</param>
        /// <param name="pTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///                    bytes of Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters. Can only be executed after an Increment, Decrement, IncrementTransfer, DecrementTransfer or Restore command.",
                Message = "Transfer the content of the TransferBuffer to the specified address."
            )
        ]
        public Status_t Transfer ( byte bTransferMaced, ushort wBlockNr, out byte[] pTMC, out byte[] pTMV )
        {
            pTMC = new byte[4];
            pTMV = new byte[8];
            return phalMfpEVx_Transfer ( m_pDataParams, bTransferMaced, wBlockNr, pTMC, pTMV );
        }
        #endregion Transfer

        #region Restore
        /// <summary>
        /// Performs a Restore / Restore MACed command.
        /// The parameter RestoreMaced is valid only for MFP authenticated state and not for MFC authenticate state.
        /// </summary>
        ///
        /// <param name="bRestoreMaced">Indicate whether the response should be maced. Based on this flag the
        ///                             command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to be used for restoring the value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "The Macing to be applied depends on Encrypted, ReadMaced and MacOnCmd parameters.",
                Message = "Copies the content found in the Value block at the given address to the TransferBuffer."
            )
        ]
        public Status_t Restore ( byte bRestoreMaced, ushort wBlockNr )
        {
            return phalMfpEVx_Restore ( m_pDataParams, bRestoreMaced, wBlockNr );
        }
        #endregion Restore
        #endregion Value operation commands
#endif

        #region Special commands
        /// <summary>
        /// Returns manufacturing related data of the PICC
        /// </summary>
        ///
        /// <param name="pVerInfo">The version information of the PICC.
        ///                         If UID is 4 bytes, the buffer will have 27 bytes of version information.
        ///                         If UID is 7 bytes, the buffer will have 28 bytes of version information.
        ///                         If UID is 10 bytes, the buffer will have 33 bytes of version information.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "First 7 bytes will be Hardware related Information. Second 7 bytes will be Software related Information and last N bytes will be Product related information.",
                Message = "Returns the Version information of the card. The information will contain Hardware, Software and Production related information."
            )
        ]
        public Status_t GetVersion ( out byte[] pVerInfo )
        {
            Status_t oStatus = null;
            byte[] aVersion = new byte[33];

            pVerInfo = null;

            oStatus = phalMfpEVx_GetVersion ( m_pDataParams, aVersion );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                byte bLen = 28; /* For 7 byte UID. */
                if ( aVersion[14].Equals ( 0 ) && aVersion[15].Equals ( 4 ) )
                    bLen = 27;  /* For 4 byte UID. */

                if ( aVersion[14].Equals ( 0 ) && aVersion[15].Equals ( 10 ) )
                    bLen = 33;  /* For 10 byte UID. */

                pVerInfo = new byte[bLen];
                Array.Copy ( aVersion, pVerInfo, bLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Read originality Signature from the PICC.
        /// </summary>
        ///
        /// <param name="bLayer4Comm">ISO14443 protocol to be used.
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L3
        ///                             NxpRdLibNet.alMfpEVx.Iso144443.L4</param>
        /// <param name="bAddr">Targeted ECC originality check signature.</param>
        /// <param name="pSignature">PICC's originality signature. The buffer will have 56 bytes of
        ///                          signature information.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Retrieve the ECC originality check signature."
            )
        ]
        public Status_t ReadSign ( byte bLayer4Comm, byte bAddr, out byte[] pSignature )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            Status_t status = phalMfpEVx_ReadSign ( m_pDataParams, bLayer4Comm, bAddr, ref ppRxdata );

            pSignature = null;
            if ( status.Equals ( Error_Gen.SUCCESS ) )
            {
                pSignature = new byte[SIG_LENGTH];
                Marshal.Copy ( ppRxdata, pSignature, 0, SIG_LENGTH );
            }

            return status;
        }

        /// <summary>
        /// Performs a Reset Authentication command.
        /// </summary>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Reset the authentication state. Clears existing sessions keys and set the state to not authenticated."
            )
        ]
        public Status_t ResetAuth ()
        {
            return phalMfpEVx_ResetAuth ( m_pDataParams );
        }

        /// <summary>
        /// Perform MIFARE(R) Personalize UID usage command sequence with MIFARE PICC.
        /// </summary>
        ///
        /// <param name="bUidType">UID type to be used for personalization.
        ///                         NxpRdLibNet.alMfpEVx.PersonalizeUID.UIDF0
        ///                         NxpRdLibNet.alMfpEVx.PersonalizeUID.UIDF1
        ///                         NxpRdLibNet.alMfpEVx.PersonalizeUID.UIDF2
        ///                         NxpRdLibNet.alMfpEVx.PersonalizeUID.UIDF3</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "Supported in Security Level 1 only. The UID Personalization depends on the information passed to UidType parameter. Once its personalized to a UIDType it cannot be reversed.",
                Message = "Personalizes the UID as Complete UID return after ISO14443 - 3 Activation, Reading of block 0 data without authentication, return of Random UID and Return of Non-Unique UID."
            )
        ]
        public Status_t PersonalizeUid ( byte bUidType )
        {
            return phalMfpEVx_PersonalizeUid ( m_pDataParams, bUidType );
        }

        /// <summary>
        /// Performs a configuration for ISO1443-4 enabling in Security Level 1.
        /// </summary>
        ///
        /// <param name="bOption">Option byte encoding the configuration to be set.
        ///                         NxpRdLibNet.alMfpEVx.Iso144443.ENABLE
        ///                         NxpRdLibNet.alMfpEVx.Iso144443.DISABLE</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                Message = "Supported in Security Level 1 only. Disable / Enable ISO14443-4 Activation."
            )
        ]
        public Status_t SetConfigSL1 ( byte bOption )
        {
            return phalMfpEVx_SetConfigSL1 ( m_pDataParams, bOption );
        }

        /// <summary>
        /// Performs read of the TMAC block in security layer 1 with ISO14443 Layer 3 activated.
        /// </summary>
        ///
        /// <param name="wBlockNr">PICC TM  block number from which the data should be read.</param>
        /// <param name="pBlocks">The data read from the specified block number. The buffer will
        ///                       contain 16 bytes information. </param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "Supported in Security Level 1 only. Command is exchanged as Plain with No MAC on Command and Data. This command is specially to read the TransactionMAC block data.",
                Message = "Reads the data from TransactionMAC block. The return length will always be 1 block length i.e. 16 bytes."
            )
        ]
        public Status_t ReadSL1TMBlock ( ushort wBlockNr, out byte[] pBlocks )
        {
            pBlocks = new byte[16];
            return phalMfpEVx_ReadSL1TMBlock ( m_pDataParams, wBlockNr, pBlocks );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Performs a Key Change of a MIFARE Plus key. Same as phalMfpEVx_Write, but diversification input can be provided
        /// </summary>
        ///
        /// <param name="bChangeKeyMaced">Indicate whether the response should be maced. Based on this flag the
        ///                               command code will be updated.
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.OFF
        ///                                 NxpRdLibNet.alMfpEVx.MacOnRsp.ON</param>
        /// <param name="wBlockNr">PICC block number to which the key should be changed.</param>
        /// <param name="wKeyNumber">Key number to be used from software or hardware KeyStore.</param>
        /// <param name="wKeyVersion">Key version to be used from software or hardware KeyStore.</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "Not a command. This is a utility method of reader library that performs Write command internally.",
                Message = "Changes the key of the specified block number."
            )
        ]
        public Status_t ChangeKey ( byte bChangeKeyMaced, ushort wBlockNr, ushort wKeyNumber, ushort wKeyVersion, byte[] pDivInput )
        {
            return phalMfpEVx_ChangeKey ( m_pDataParams, bChangeKeyMaced, wBlockNr, wKeyNumber, wKeyVersion,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDivInput );
        }

        /// <summary>
        /// Secures the transaction by committing the application to ReaderID specified.
        /// The encrypted Transaction MAC Reader Id of the previous transaction is used by the
        /// back-end that will decrypt this and check if the transaction was reported to be back-end or not.
        /// This command is always sent with MAC protection.
        /// </summary>
        ///
        /// <param name="wBlockNr">PICC's TM protected block number.</param>
        /// <param name="pTMRI">The reader ID information to be committed. The buffer should have
        ///                     have 16 bytes of ID information.</param>
        /// <param name="pEncTMRI">Encrypted Reader ID of the previous transaction. Buffer will contain
        ///                        16 bytes of information.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "To be used in Security Level 3 and the targeting block number should be TMProtected block.",
                Message = "Commits a ReaderID for the ongoing transaction. This will allow a back-end to identify the attacking merchant in case of fraud detected."
            )
        ]
        public Status_t CommitReaderID ( ushort wBlockNr, byte[] pTMRI, out byte[] pEncTMRI )
        {
            pEncTMRI = new byte[16];
            return phalMfpEVx_CommitReaderID ( m_pDataParams, wBlockNr, pTMRI, pEncTMRI );
        }
#endif

        /// <summary>
        /// Performs a VCSupportLastISOL3 command operation. This command is a special variant of virtual card operation.
        /// This command can be sent after a ISO14443-3 activation.
        /// </summary>
        ///
        /// <param name="pIid">The (last) IID that the PCD presents to the PD. The buffer
        ///                    should have 16 bytes of information.</param>
        /// <param name="pPcdCapabilities">Capability vector of the PCD.  The buffer should have 4 bytes
        ///                                of information.</param>
        /// <param name="info">One byte Information returned by PICC.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "Supported in Security Level 1 only.",
                Message = "The PCD communicates the last or only IID of a sequence of IIDs to the PD, as well as the PCD capabilities and a random challenge."
            )
        ]
        public Status_t VcSupportLastISOL3 ( byte[] pIid, byte[] pPcdCapabilities, ref byte info )
        {
            return phalMfpEVx_VCSupportLastISOL3 ( m_pDataParams, pIid, pPcdCapabilities, ref info );
        }
        #endregion Special commands

        #region Utilities commands
        /// <summary>
        /// Reset the libraries internal secure messaging state.
        ///
        /// This function must be called before interacting with the PICC to set the libraries internal card-state back to default.
        /// E.g. when an error occurred or after a reset of the field.
        /// </summary>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Reset the authentication state. Clears existing session keys and set the state to not authenticated."
            )
        ]
        public Status_t ResetSecMsgState ()
        {
            return phalMfpEVx_ResetSecMsgState ( m_pDataParams );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wOption">Option to set.
        ///                             NxpRdLibNet.alMfpEVx.Config.WRAPPED_MODE
        ///                             NxpRdLibNet.alMfpEVx.Config.EXTENDED_APDU
        ///                             NxpRdLibNet.alMfpEVx.Config.AUTH_MODE</param>
        /// <param name="wValue">Value for the selected option.
        ///                             NxpRdLibNet.alMfpEVx.Config.WRAPPED_MODE
        ///                                 NxpRdLibNet.alMfpEVx.Value.DEAULT
        ///                                 NxpRdLibNet.alMfpEVx.Value.DISABLE
        ///                                 NxpRdLibNet.alMfpEVx.Value.ENABLE
        ///                             NxpRdLibNet.alMfpEVx.Config.EXTENDED_APDU
        ///                                 NxpRdLibNet.alMfpEVx.Value.DEAULT
        ///                                 NxpRdLibNet.alMfpEVx.Value.DISABLE
        ///                                 NxpRdLibNet.alMfpEVx.Value.ENABLE
        ///                             NxpRdLibNet.alMfpEVx.Config.AUTH_MODE
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOTAUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL1_MIFARE_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL1_MFP_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL3_MFP_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOT_AUTHENTICATED_L3
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOT_AUTHENTICATED_L4</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Set the configuration like WrappedMode, ISO7816 ExtendedAPDU and AuthMode."
            )
        ]
        public Status_t SetConfig ( ushort wOption, ushort wValue )
        {
            return phalMfpEVx_SetConfig ( m_pDataParams, wOption, wValue );
        }

        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wOption">Option to get.
        ///                             NxpRdLibNet.alMfpEVx.Config.WRAPPED_MODE
        ///                             NxpRdLibNet.alMfpEVx.Config.EXTENDED_APDU
        ///                             NxpRdLibNet.alMfpEVx.Config.AUTH_MODE</param>
        /// <param name="wValue">Value of the selected option.
        ///                             NxpRdLibNet.alMfpEVx.Config.WRAPPED_MODE
        ///                                 NxpRdLibNet.alMfpEVx.Value.DEAULT
        ///                                 NxpRdLibNet.alMfpEVx.Value.DISABLE
        ///                                 NxpRdLibNet.alMfpEVx.Value.ENABLE
        ///                             NxpRdLibNet.alMfpEVx.Config.EXTENDED_APDU
        ///                                 NxpRdLibNet.alMfpEVx.Value.DEAULT
        ///                                 NxpRdLibNet.alMfpEVx.Value.DISABLE
        ///                                 NxpRdLibNet.alMfpEVx.Value.ENABLE
        ///                             NxpRdLibNet.alMfpEVx.Config.AUTH_MODE
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOTAUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL1_MIFARE_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL1_MFP_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.SL3_MFP_AUTHENTICATED
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOT_AUTHENTICATED_L3
        ///                                 NxpRdLibNet.alMfpEVx.Value.NOT_AUTHENTICATED_L4</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Get the configuration like WrappedMode, ISO7816 ExtendedAPDU and AuthMode."
            )
        ]
        public Status_t GetConfig ( ushort wOption, out ushort wValue )
        {
            wValue = 0x00;
            return phalMfpEVx_GetConfig ( m_pDataParams, wOption, ref wValue );
        }

        /// <summary>
        /// This is a utility API which sets the VCA structure in MFP Ev1 structure params.
        /// This interface is mandatory to be called if the Virtual Card and Proximity Check features
        /// are required.
        /// </summary>
        ///
        /// <param name="pAlVCADataParams">The VCA application layer's DataParams.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "This is a utility API which sets the VCA structure in Plus EVx structure params."
            )
        ]
        public Status_t SetVCAParams ( IntPtr pAlVCADataParams )
        {
            return phalMfpEVx_SetVCAParams ( m_pDataParams, pAlVCADataParams );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option. 0xFFFF is for No Diversification.</param>
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pTMI">Transaction MAC Input.</param>
        /// <param name="pTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Computed the Transaction MAC input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput, byte[] pTMC, byte[] pUid,
            byte[] pTMI, out byte[] pTMV )
        {
            pTMV = new byte[8];
            return phalMfpEVx_CalculateTMV ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ),
                pTMI, ( uint ) ( ( pTMI == null ) ? 0 : pTMI.Length ), pTMV );
        }

        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///                         To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///                         To be used for SAM AV3 only.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Computed the Transaction MAC input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] aTMC, byte[] aUid, byte[] aTMI, out byte[] aTMV )
        {
            aTMV = new byte[8];
            return phalMfpEVx_CalculateTMV ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null,
                0, aTMC, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aTMI, ( uint ) ( ( aTMI == null ) ? 0 : aTMI.Length ), aTMV );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option. 0xFFFF is for No Diversification.</param>
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Decrypted the reader ID as performed on the PICC."
            )
        ]
        public Status_t DecryptReaderID ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput, byte[] pTMC, byte[] pUid,
            byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfpEVx_DecryptReaderID ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ),
                pEncTMRI, pTMRIPrev );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///                         To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///                         To be used for SAM AV3 only.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.PLUS_EV1, MIFARE_Products.PLUS_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Decrypted the reader ID as performed on the PICC."
            )
        ]
        public Status_t DecryptReaderID ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer, byte[] pTMC, byte[] pUid,
            byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfpEVx_DecryptReaderID ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null, 0, pTMC, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pEncTMRI, pTMRIPrev );
        }
#endif
        #endregion Utilities commands
        #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Private functions
        protected byte[] ConvertIntToByteArray ( int dwValue )
        {
            byte[] pValue = new byte[4];
            pValue[0] = ( byte ) dwValue;
            pValue[1] = ( byte ) ( dwValue >> 8 );
            pValue[2] = ( byte ) ( dwValue >> 16 );
            pValue[3] = ( byte ) ( dwValue >> 24 );
            return pValue;
        }
        protected int ConvertByteArrayToInt ( byte[] pValue )
        {
            int dwValue = 0;
            dwValue = pValue[0];
            dwValue += pValue[1] << 8;
            dwValue += pValue[2] << 16;
            dwValue += pValue[3] << 24;
            return dwValue;
        }
        #endregion
#endif

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }

    #endregion

    #region Software
    /// <summary>
    /// Class for Software Mode layer initialization interface and DataParams.
    /// </summary>
    public class Sw : alMfpEVx.Generic
    {
        #region Data Structure
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary> Pointer to the parameter structure of the Crypto layer for macing. </summary>
            public IntPtr pCryptoDataParamsMac;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). </summary>
            public IntPtr pCryptoDiversifyDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Pointer to the parameter structure for Virtual Card. </summary>
            public IntPtr pVCADataParams;

            /// <summary> R_CTR (read counter); the PICC's read counter is used for a following authentication. </summary>
            public ushort wRCtr;

            /// <summary> W_CTR (write counter); the PICC's write counter is used for a following authentication. </summary>
            public ushort wWCtr;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs..</summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format.</summary>
            public byte bExtendedLenApdu;

            /// <summary> Transaction Identifier; unused if 'bFirstAuth' = 1; uint8_t[4]. </summary>
            public fixed byte bTi[4];

            /// <summary> Amount of data in the pUnprocessedReadMacBuffer. </summary>
            public byte bNumUnprocessedReadMacBytes;

            /// <summary> Buffer containing unprocessed bytes for read mac answer stream. </summary>
            public fixed byte bUnprocessedReadMacBuffer[16];

            /// <summary> Intermediate MAC for Read Calculation. </summary>
            public fixed byte bIntermediateMac[16];

            /// <summary> Indicates whether this is a first read in a read MAC sequence or not. </summary>
            public byte bFirstRead;

            /// <summary> Initialization vector. Max size of IV can be 16 bytes. </summary>
            public fixed byte bIv[16];

            /// <summary> Authentication ENC key for the session. </summary>
            public fixed byte bSesAuthENCKey[16];

            /// <summary> Authentication MAC key for the session. </summary>
            public fixed byte bSesAuthMACKey[16];

            /// <summary> Security level authenticate.</summary>
            public byte bAuthMode;

            /// <summary> Secure messaging mode. \c 0: EV0 Mode; \c 1: EV1 mode.</summary>
            public byte bSMMode;

        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifare,
            IntPtr pKeyStore, IntPtr pCryptoEnc, IntPtr pCryptoMac, IntPtr pCryptoRngDataParams, IntPtr pCryptoDiversifyDataParams,
            IntPtr pTMIDataParams, IntPtr pVCADataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initializes the AL component as software component.
        /// </summary>
        ///
        /// <param name="pPalMifare">Pointer to a palMifare data parameter structure.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore data parameter structure.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto data parameter structure for encryption.</param>
        /// <param name="pCryptoMac">Pointer to a Crypto data parameter structure for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng data parameter structure.</param>
        /// <param name="pCryptoDiversify">Pointer to the parameter structure of the CryptoDiversify layer (can be NULL).</param>
        /// <param name="pTMIUtils">Pointer to a TMI data parameter structure.</param>
        /// <param name="pAlVcaParams">Pointer to the parameter structure for Virtual Card.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMac,
            CryptoRng.Generic pCryptoRng, CryptoSym.Generic pCryptoDiversify, NxpRdLibNet.TMIUtils pTMIUtils, alVca.Generic pAlVcaParams )
        {
            return phalMfpEVx_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pCryptoEnc != null ) ? pCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( pCryptoMac != null ) ? pCryptoMac.m_pDataParams : IntPtr.Zero,
                ( pCryptoRng != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ( pCryptoDiversify != null ) ? pCryptoDiversify.m_pDataParams : IntPtr.Zero,
                ( pTMIUtils != null ) ? pTMIUtils.m_pDataParams : IntPtr.Zero,
                ( pAlVcaParams != null ) ? pAlVcaParams.m_pDataParams : IntPtr.Zero );
        }
#if DEBUG
        /// <summary>
        /// Initializes the AL component as software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare data parameter structure.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore data parameter structure.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto data parameter structure for encryption.</param>
        /// <param name="pCryptoMac">Pointer to a Crypto data parameter structure for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng data parameter structure.</param>
        /// <param name="pCryptoDiversify">Pointer to the parameter structure of the CryptoDiversify layer (can be NULL).</param>
        /// <param name="pTMIUtils">Pointer to a TMI data parameter structure.</param>
        /// <param name="pAlVcaParams">Pointer to the parameter structure for Virtual Card.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc,
            CryptoSym.Generic pCryptoMac, CryptoRng.Generic pCryptoRng, CryptoSym.Generic pCryptoDiversify, NxpRdLibNet.TMIUtils pTMIUtils,
            alVca.Generic pAlVcaParams )
        {
            return phalMfpEVx_Sw_Init (ref m_DataParamsInt[0], (ushort) wDataParamSize,
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pCryptoEnc != null ) ? pCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( pCryptoMac != null ) ? pCryptoMac.m_pDataParams : IntPtr.Zero,
                ( pCryptoRng != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ( pCryptoDiversify != null ) ? pCryptoDiversify.m_pDataParams : IntPtr.Zero,
                ( pTMIUtils != null ) ? pTMIUtils.m_pDataParams : IntPtr.Zero,
                ( pAlVcaParams != null ) ? pAlVcaParams.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to DataParams
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam_NonX
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 NonX Mode layer initialization interface and DataParams.
    /// </summary>
    public class SamAV3_NonX : alMfpEVx.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE PLUS EV1 SamAV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Security level authenticate.</summary>
            public byte bAuthMode;

            /// <summary> Secure messaging mode. \c 0: EV0 Mode; \c 1: EV1 mode.</summary>
            public byte bSMMode;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs..</summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format.</summary>
            public byte bExtendedLenApdu;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
                IntPtr pPalMifareDataParams, IntPtr pTMIDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initializes the MIFARE Plus EV1 SamAV3 layer in NonX mode.
        /// </summary>
        ///
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of SAM AV3.</param>
        /// <param name="pPalMifareDataParams">Pointer to Pal MIFARE parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSamDataParams, palMifare.Generic pPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalMifareDataParams == null ) ? IntPtr.Zero : pPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initializes the MIFARE Plus EV1 SamAV3 layer in NonX mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of the DataParams structure.</param>
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of SAM AV3.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHalSamDataParams, palMifare.Generic pPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_SamAV3_NonX_Init ( ref m_DataParamsInt[0], (ushort) wDataParamSize,
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalMifareDataParams == null ) ? IntPtr.Zero : pPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to DataParams
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion SamAV3

#if PACKAGE_INTERNAL
    #region Sam
    /// <summary>
    /// Class for Sam (AV4 and Future Sam's) NonX Mode layer initialization interface and DataParams.
    /// </summary>
    public class Sam_NonX : alMfpEVx.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE PLUS EV1 SamAV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Security level authenticate.</summary>
            public byte bAuthMode;

            /// <summary> Secure messaging mode. \c 0: EV0 Mode; \c 1: EV1 mode.</summary>
            public byte bSMMode;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs..</summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format.</summary>
            public byte bExtendedLenApdu;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_Sam_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
                IntPtr pPalMifareDataParams, IntPtr pTMIDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initializes the MIFARE Plus EV1 SamAV3 layer in NonX mode.
        /// </summary>
        ///
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of SAM AV3.</param>
        /// <param name="oPalMifareDataParams">Pointer to Pal MIFARE parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam pHalSamDataParams, palMifare.Generic oPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_Sam_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( oPalMifareDataParams == null ) ? IntPtr.Zero : oPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initializes the MIFARE Plus EV1 Sam layer in NonX mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of the DataParams structure.</param>
        /// <param name="oHalSamDataParams">Pointer to the HAL parameter structure of SAM.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.Generic oHalSamDataParams, palMifare.Generic oPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_Sam_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
                ( oHalSamDataParams == null ) ? IntPtr.Zero : oHalSamDataParams.m_pDataParams,
                ( oPalMifareDataParams == null ) ? IntPtr.Zero : oPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_NonX ()
        {
            // Free allocated pointer to DataParams
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Sam
#endif
    #endregion Sam_NonX

    #region Sam_X
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 X Mode layer initialization interface and DataParams.
    /// </summary>
    public class SamAV3_X : alMfpEVx.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE PLUS EV1 SamAV3 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Security level authenticate.</summary>
            public byte bAuthMode;

            /// <summary> Secure messaging mode. \c 0: EV0 Mode; \c 1: EV1 mode.</summary>
            public byte bSMMode;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfpEVx_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
                IntPtr pPalMifareDataParams, IntPtr pTMIDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initializes the MIFARE Plus EV1 SamAV3 layer in X mode.
        /// </summary>
        ///
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of SAM AV3.</param>
        /// <param name="pPalMifareDataParams">Pointer to Pal MIFARE parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSamDataParams, palMifare.Generic pPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalMifareDataParams == null ) ? IntPtr.Zero : pPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initializes the MIFARE Plus EV1 SamAV3 layer in X mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of the DataParams structure.</param>
        /// <param name="pHalSamDataParams">Pointer to the HAL parameter structure of SAM AV3.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.SamAV3 pHalSamDataParams, palMifare.Generic pPalMifareDataParams, TMIUtils oTMIUtils )
        {
            return phalMfpEVx_SamAV3_X_Init ( ref m_DataParamsInt[0], (ushort) wDataParamSize,
                ( pHalSamDataParams == null ) ? IntPtr.Zero : pHalSamDataParams.m_pDataParams,
                ( pPalMifareDataParams == null ) ? IntPtr.Zero : pPalMifareDataParams.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to DataParams
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion SamAV3
    #endregion Sam_X
#endif
}
