/*
 * Copyright 2013 - 2020, 2022, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMful
{
    #region Enumeration
    #region Authenticate
    /// <summary>
    /// Diversification option for Ultralight C Authenticate command
    /// </summary>
    public enum Authenticate : byte
    {
        /// <summary> Option to indicate the diversification is disabled. </summary>
        DIVERSIFICATION_OFF = 0,

        /// <summary> Option to indicate the diversification is enabled. </summary>
        DIVERSIFICATION_ON = 1
    }

    /// <summary>
    /// Diversification option for Ultralight EV1 Authenticate command.
    /// </summary>
    public enum PwdAuthOption : byte
    {
        /// <summary> Option to indicate the diversification is disabled. </summary>
        DIV_OFF = 0x00,

        /// <summary>
        /// Option to indicate the diversification is disabled but the diversification should be performed
        /// without DivInput. In this cases the Div algorithm will be CMAC(Kx, Const | Padding)
        /// </summary>
        DIV_OFF_INPUT_NOT_AVAILABLE = 0x01,

        /// <summary>
        /// Option to indicate the diversification is enabled and the diversification should be performed
        /// with DivInput. In this cases the Div algorithm will be CMAC(Kx, Const | M | Padding), where
        /// M is diversification input (1 - 31 bytes)
        /// </summary>
        DIV_ON = 0x02
    }
    #endregion

    #region LockSign
    /// <summary>
    /// Lock option for Ultralight LockSign command
    /// </summary>
    public enum LockSign : byte
    {
        /// <summary> Option to unlock the signature. </summary>
        UNLOCK = 0,

        /// <summary> Option to temporary lock the signature. </summary>
        LOCK = 1,

        /// <summary> Option to permanently lock the signature. </summary>
        PERMENT_LOCK = 2
    }
    #endregion LockSign

    #region Config
    /// <summary>
    /// The configuration to be used for SetConfig / GetConfig.
    /// </summary>
    public enum Config : byte
    {
        /// <summary> Option to Get/Set for inclusion of CMAC data for command and response. </summary>
        CMAC_STATUS = 0xA1,

        /// <summary> Option to Get/Set additional info of a generic error or length of response data. </summary>
        ADDITINAL_INFO = 0xA2
    }
    #endregion Config
    #endregion Enumeration

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    ///
    /// Note:
    /// - Below are the initializations supported by this component.
    ///    - <see cref="Sw.Init">Software Initialization</see>
    ///    - <see cref="Sw.Init">SAMAV2 NonX Initialization</see>
    ///    - <see cref="Sw.Init">SAMAV3 NonX Initialization</see>
    ///    - <see cref="Sw.Init">SAMAV2 X Initialization</see>
    ///    - <see cref="Sw.Init">SAMAV3 X Initialization</see>
    /// - MIFARE Ultralight AES product supports exchanging CMAC for commands post successful Authentication.
    ///    - If SEC_MSG_ACT bit is set, enable CMAC feature by configuring <see cref="Config.CMAC_STATUS">Configure CMAC</see>
    ///      identifier using <see cref="SetConfig">Set Configuration</see> interface and value as <see cref="Value.ON">Enable</see>.
    ///      Post configuring this identifier in library, library will append CMAC for each command exchange and will verify the
    ///      CMAC received form PICC also.
    /// - If SEC_MSG_ACT bit is not set, disable CMAC feature by configuring  <see cref="Config.CMAC_STATUS">Configure CMAC</see>
    ///   identifier using <see cref="SetConfig">Set Configuration</see> interface and value as <see cref="Value.ON">Disable</see>.
    ///   Post configuring this identifier in library, library will not append CMAC for each command exchange.
    /// </summary>
    [NxpRdLibAttribute
        (
            MIFARE_Products.ULTRALIGHT, MIFARE_Products.ULTRALIGHT_C, MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
            NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213,
            NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
            NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
            Message = "Commands to communicate with MIFARE Ultralight or NTAG product families." +
                      "Products that support Password Authentication. " +
                      "Products that support 16 byte DES Authentication. " +
                      "Products that support 16 byte AES Authentication. " +
                      "Products that do not support authentication. ",
            OtherInfo = "Backward compatible to Previous version of MIFARE Ultralight / NTAG Products."
        )
    ]
    public abstract class Generic
    {
        #region Defines
        public const byte READ_BLOCK_LENGTH = 16;
        public const byte WRITE_BLOCK_LENGTH = 4;
        public const byte COMPWRITE_BLOCK_LENGTH = 16;
        public const byte COUNTER_VALUE_LENGTH = 3;
        public const byte PACK_LENGTH = 2;
        public const byte VERSION_LENGTH = 8;
        public const byte TT_STATUS_LENGTH = 5;
        public const byte SIG_LENGTH = 32;

        public const byte DES_BLOCK_SIZE = 8;
        public const byte DES_KEY_LENGTH = 16;
        #endregion Defines

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_UlcAuthenticate ( IntPtr pDataParams, byte bOption, ushort wKeyNo, ushort wKeyVer, byte[] pDivInput, byte bDivInputLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_AuthenticateAES ( IntPtr pDataParams, byte bOption, ushort wKeyNo, ushort wKeyVer, byte bRamKeyNo, byte bRamKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivInputLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_Read ( IntPtr pDataParams, byte bAddress, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_Write ( IntPtr pDataParams, byte bAddress, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_FastWrite ( IntPtr pDataParams, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_CompatibilityWrite ( IntPtr pDataParams, byte bAddress, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_IncrCnt ( IntPtr pDataParams, byte bCntNum, byte[] pCnt );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_ReadCnt ( IntPtr pDataParams, byte bCntNum, byte[] pCntValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_PwdAuth ( IntPtr pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte[] pDivInput,
            byte bDivInputLen, byte[] pPwd, byte[] pPack );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_GetVersion ( IntPtr pDataParams, byte[] pPwd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_FastRead ( IntPtr pDataParams, byte bStartAddr, byte bEndAddr, ref IntPtr pData, ref ushort wNumBytes );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SectorSelect ( IntPtr pDataParams, byte bSecNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_ReadSign ( IntPtr pDataParams, byte bAddr, ref IntPtr pSignature );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_ChkTearingEvent ( IntPtr pDataParams, byte bCntNum, byte[] pValidFlag );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_WriteSign ( IntPtr pDataParams, byte bAddress, byte[] pSignature );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_LockSign ( IntPtr pDataParams, byte bLockMode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_VirtualCardSelect ( IntPtr pDataParams, byte[] pVCIID, byte bVCIIDLen, ref byte pVCTID );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_ReadTTStatus ( IntPtr pDataParams, byte bAddr, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_CalculateSunCMAC ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVer,
        byte[] pIndata, ushort wInDataLen, byte[] pRespMac );
        #endregion DLL Imports

        #region Wrapper Functions
        /// <summary>
        /// Perform MIFARE(R) Ultralight-C Authenticate command with PICC.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used in authentication.</param>
        /// <param name="wKeyVer">Key version to be used in authentication.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If DataParams is null
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>  Unexpected PICC response length or unexpected first byte received from PICC.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_C,
                Message = "Method to perform Authentication for MIFARE Ultralight C product without key diversification.",
                OtherInfo = "Support Authentication using 16 byte DES Key."
            )
        ]
        public Status_t UlcAuthenticate ( int wKeyNo, int wKeyVer )
        {
            return phalMful_UlcAuthenticate ( m_pDataParams, 0x00, ( ushort ) wKeyNo, ( ushort ) wKeyVer, null, 0 );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight-C Authenticate command with PICC.
        /// Only valid when component is initialized for SAM.
        /// </summary>
        ///
        /// <param name="bOption">Option for specifying the diversification input. Only valid for Sam AV3.
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_OFF
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_ON</param>
        /// <param name="wKeyNo">Key number to be used in authentication.</param>
        /// <param name="wKeyVer">Key version to be used in authentication.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If DataParams is null
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>  Unexpected PICC response length or unexpected first byte received from PICC.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_C,
                Message = "Method to perform Authentication for MIFARE Ultralight C product with key diversification.",
                OtherInfo = "Support Authentication using 16 byte DES Key."
            )
        ]
        public Status_t UlcAuthenticate ( byte bOption, ushort wKeyNo, ushort wKeyVer, byte[] aDivInput )
        {
            return phalMful_UlcAuthenticate ( m_pDataParams, bOption, wKeyNo, wKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NTAG AES Authentication with PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option for specifying the diversification input.
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_OFF
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_ON</param>
        /// <param name="wKeyNo">Key number to be used in authentication.</param>
        /// <param name="wKeyVer">Key version to be used in authentication.</param>
        /// <param name="bKeyNoCard"> Key number on card.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If DataParams is null
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES, NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Method to perform AES Authentication for MIFARE(R) Ultralight / NTAG product with key diversification.",
                OtherInfo = "Support Authentication using 16 byte AES Key."
            )
        ]
        public Status_t AuthenticateAES ( byte bOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] aDivInput )
        {
            return phalMful_AuthenticateAES ( m_pDataParams, bOption, wKeyNo, wKeyVer, 0, 0, bKeyNoCard, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NTAG AES Authentication with PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option for specifying the diversification input.
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_OFF
        ///                         NxpRdLibNet.alMful.Authenticate.DIVERSIFICATION_ON</param>
        /// <param name="wKeyNo">Key number to be used in authentication.</param>
        /// <param name="wKeyVer">Key version to be used in authentication.</param>
        /// <param name="bRamKeyNo">Key number of Destination Key where the computed session key will be stored.
        ///                         To be used for SAM AV3 only.
        /// </param>
        /// <param name="bRamKeyVer">Key version of Destination Key where the computed session key will be stored.
        ///                         To be used for SAM AV3 only.
        /// </param>
        /// <param name="bKeyNoCard"> Key number on card.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/> If DataParams is null
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES, NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Method to perform AES Authentication for MIFARE(R) Ultralight / NTAG product with key diversification.",
                OtherInfo = "Support Authentication using 16 byte AES Key. This method supports Authentication via SAM module."
            )
        ]
        public Status_t AuthenticateAES ( byte bOption, ushort wKeyNo, ushort wKeyVer, byte bRamKeyNo, byte bRamKeyVer, byte bKeyNoCard,
            byte[] aDivInput )
        {
            return phalMful_AuthenticateAES ( m_pDataParams, bOption, wKeyNo, wKeyVer, bRamKeyNo, bRamKeyVer, bKeyNoCard,
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NTAG Read command with PICC.
        /// </summary>
        ///
        /// <param name="bAddress">Address on PICC to read from.</param>
        /// <param name="aData">Buffer containing data returned from the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT, MIFARE_Products.ULTRALIGHT_C, MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213,
                NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NTAG Read command with PICC.",
                OtherInfo = "Returns 16 byte of data starting from the address provided."
            )
        ]
        public Status_t Read ( byte bAddress, out byte[] aData )
        {
            aData = new byte[READ_BLOCK_LENGTH];
            Status_t oStatus = phalMful_Read ( m_pDataParams, bAddress, aData );
            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Write command with PICC.
        /// </summary>
        ///
        /// <param name="bAddress">Address on PICC to write to.</param>
        /// <param name="aData">Buffer containing the data to be written to the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT, MIFARE_Products.ULTRALIGHT_C, MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213,
                NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NTAG Write command with PICC.",
                OtherInfo = "Data to be written should be 4 bytes in length."
            )
        ]
        public Status_t Write ( byte bAddress, byte[] aData )
        {
            return phalMful_Write ( m_pDataParams, bAddress, aData );
        }

        /// <summary>
        /// Perform NATG Fast Write command which writes 64 bytes from 0xF0 to 0xFF address blocks.
        /// </summary>
        ///
        /// <param name="aData">Buffer containing the block data to be written to the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_I2C,
                Message = "Perform MIFARE(R) Ultralight / NATG Fast Write command which writes 64 bytes from 0xF0 to 0xFF address blocks.",
                OtherInfo = @"The FAST_WRITE allows writing data in ACTIVE state to the complete SRAM (64 bytes) in pass-through mode.
                              It requires the start block address (F0h), end address (FFh) and writes 64 bytes of data into the NTAG I2C
                              plus SRAM "
            )
        ]
        public Status_t FastWrite ( byte[] aData )
        {
            return phalMful_FastWrite ( m_pDataParams, aData );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Compatibility-Write command with PICC.
        /// </summary>
        ///
        /// <param name="bAddress">Address on PICC to write to.</param>
        /// <param name="aData">Buffer containing the data to be written to the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT, MIFARE_Products.ULTRALIGHT_C, MIFARE_Products.ULTRALIGHT_EV1,
                NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213, NTAG_Products.NTAG213_TT,
                NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD, NTAG_Products.NTAG224_DNA,
                NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Compatibility-Write command with PICC.",
                OtherInfo = @"The COMPATIBILITY WRITE command was implemented to accommodate the established MIFARE PCD infrastructure.
                              It requires 16 bytes of data to be transmitted to be PICC but only 4 bytes will be written."
            )
        ]
        public Status_t CompatiblityWrite ( byte bAddress, byte[] aData )
        {
            return phalMful_CompatibilityWrite ( m_pDataParams, bAddress, aData );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight Increment count command with PICC.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                Message = "Perform MIFARE(R) Ultralight Increment count command with PICC.",
                OtherInfo = @"The INCR_CNT command is used to increment one of the 3x one-way counters."
            )
        ]
        public Status_t IncrCnt ( byte bCntNum, byte[] aCnt )
        {
            return phalMful_IncrCnt ( m_pDataParams, bCntNum, aCnt );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Read count command with PICC.
        /// </summary>'
        ///
        /// <param name="bCntNum">One byte counter number 00 to 02</param>
        /// <param name="aCntValue">Buffer containing the counter value. LSB first. Three bytes counter value returned from PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG213, NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA,
                NTAG_Products.NTAG223_DNA_SD, NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Read count command with PICC.",
                OtherInfo = @"The READ_CNT command is used to read out the current value of one of the 3 one-way counters."
            )
        ]
        public Status_t ReadCnt ( byte bCntNum, out byte[] aCntValue )
        {
            aCntValue = new byte[COUNTER_VALUE_LENGTH];
            Status_t oStatus = phalMful_ReadCnt ( m_pDataParams, bCntNum, aCntValue );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aCntValue = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Password auth command with PICC.
        /// </summary>
        ///
        /// <param name="aPwd">Four byte array of type uint8_t containing four password bytes. User has to pass the password information.</param>
        /// <param name="aPack">Two byte array of type uint8_t containing two password acknowledge bytes
        ///                     For software and Sam AV3 NonX layer, the user will be provided with the Pack information returned by the card.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG212, NTAG_Products.NTAG213, NTAG_Products.NTAG213_TT,
                NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Password auth command with PICC.",
                OtherInfo = @"To access a protected memory area configured in AUTH0 configuration. Do not support diversification."
            )
        ]
        public Status_t PwdAuth ( byte[] aPwd, out byte[] aPack )
        {
            aPack = new byte[PACK_LENGTH];
            Status_t oStatus = phalMful_PwdAuth ( m_pDataParams, 0x00, 0x00, 0x00, null, 0x00, aPwd, aPack );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aPack = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Password auth command with PICC.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options should be used. This option byte is supported for Sam component only.
        ///                         NxpRdLibNet.alMful.PwdAuthOption.DIV_OFF
        ///                         NxpRdLibNet.alMful.PwdAuthOption.DIV_OFF_INPUT_NOT_AVAILABLE
        ///                         NxpRdLibNet.alMful.PwdAuthOption.DIV_ON</param>
        /// <param name="bKeyNo">Key number to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode.</param>
        /// <param name="bKeyVer">Key version to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode.</param>
        /// <param name="aDivInput">Diversification input. Only valid for Sam AV3 NonX and X mode.</param>
        /// <param name="aPwd">Four byte array of type uint8_t containing four password bytes.
        ///                    For Sam AV3 NonX layer, the parameter will be updated with the password generated by Sam hardware.</param>
        /// <param name="aPack">Two byte array of type uint8_t containing two password acknowledge bytes
        ///                     For software and Sam AV3 NonX layer, the user will be provided with the Pack information returned by the card.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG212, NTAG_Products.NTAG213, NTAG_Products.NTAG213_TT,
                NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Password auth command with PICC.",
                OtherInfo = @"To access a protected memory area configured in AUTH0 configuration. Support diversification."
            )
        ]
        public Status_t PwdAuth ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte[] aPwd, out byte[] aPack )
        {
            aPack = new byte[PACK_LENGTH];
            Status_t oStatus = phalMful_PwdAuth ( m_pDataParams, bOption, bKeyNo, bKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPwd, aPack );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aPack = new byte[0];
            }

            return oStatus;

        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG get version command with PICC.
        /// </summary>
        ///
        /// <param name="aVersion">An eight byte array containing version information bytes</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213,
                NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG get version command with PICC."
            )
        ]
        public Status_t GetVersion ( out byte[] aVersion )
        {
            aVersion = new byte[VERSION_LENGTH];
            Status_t oStatus = phalMful_GetVersion ( m_pDataParams, aVersion );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aVersion = new byte[0];
            }
            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Fast read command with PICC.
        /// </summary>
        ///
        /// <param name="bStartAddr">One byte start address</param>
        /// <param name="bEndAddr">One byte end address</param>
        /// <param name="aData">Pointer to the data read from the card. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG212, NTAG_Products.NTAG213, NTAG_Products.NTAG213_TT,
                NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD, NTAG_Products.NTAG224_DNA,
                NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Fast read command with PICC.",
                OtherInfo = @"Reads data starting from start address until end address. This is next version of Read command where only 16 bytes are read."
            )
        ]
        public Status_t FastRead ( byte bStartAddr, byte bEndAddr, out byte[] aData )
        {
            ushort wNumBytes = 0;
            IntPtr ppRxdata = IntPtr.Zero;

            Status_t oStatus = phalMful_FastRead ( m_pDataParams, bStartAddr, bEndAddr, ref ppRxdata, ref wNumBytes );

            if ( wNumBytes != 0 )
            {
                aData = new byte[wNumBytes];
                Marshal.Copy ( ppRxdata, aData, 0, wNumBytes );
            }
            else
            {
                aData = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform NTAG Sector Select with PICC.
        /// </summary>
        ///
        /// <param name="bSecNo">The number of the sector to be selected.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_I2C,
                Message = "Perform NTAG Sector Select with PICC.",
                OtherInfo = @"Selects the block / sector of memory to access."
            )
        ]
        public Status_t SectorSelect ( byte bSecNo )
        {
            return phalMful_SectorSelect ( m_pDataParams, bSecNo );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG read signature command with PICC.
        /// </summary>
        /// <param name="bAddr">Value is always 00. Present for forward compatibility reasons</param>
        /// <param name="aSignature">Pointer to a 32 byte signature read from the card</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG_I2C, NTAG_Products.NTAG210, NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG212, NTAG_Products.NTAG213,
                NTAG_Products.NTAG213_TT, NTAG_Products.NTAG215, NTAG_Products.NTAG216, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG read signature command with PICC.",
                OtherInfo = @"The READ_SIG retrieves the originality signature based on Elliptic Curve Cryptography Digital Signature Algorithm (ECDSA).
                              The purpose of originality check signature is to protect the product from mass copying"
            )
        ]
        public Status_t ReadSign ( byte bAddr, out byte[] aSignature )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            ushort wDataLen = SIG_LENGTH;
            Status_t oStatus = phalMful_ReadSign ( m_pDataParams, bAddr, ref ppRxdata );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                GetConfig ( ( ushort ) Config.ADDITINAL_INFO, out wDataLen );

                aSignature = new byte[wDataLen];
                Marshal.Copy ( ppRxdata, aSignature, 0, wDataLen );
            }
            else
            {
                aSignature = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight EV1 check tearing event command with PICC.
        /// </summary>
        ///
        /// <param name="bCntNum">Value specifying the counter number 00 to 02</param>
        /// <param name="aValidFlag">Address of one byte containing the valid flag byte</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1,
                Message = "Perform MIFARE(R) Ultralight EV1 check tearing event command with PICC.",
                OtherInfo = @"The CHECK_TEARING_EVENT command enables the application to identify if a tearing event happened on a specified counter element.
                              It takes the counter number as single argument and returns a specified valid flag for this counter. If the returned valid flag is
                              not equal to the predefined value, a tearing event happened."
            )
        ]
        public Status_t ChkTearingEvent ( byte bCntNum, out byte[] aValidFlag )
        {
            aValidFlag = new byte[1];
            Status_t atatus = phalMful_ChkTearingEvent ( m_pDataParams, bCntNum, aValidFlag );

            if ( !atatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aValidFlag = new byte[0];
            }

            return atatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Write Signature command with PICC.
        /// This command writes the signature information to the PICC.
        /// </summary>
        ///
        /// <param name="bAddress">Relative page location of the signature part to be written.</param>
        /// <param name="aSignature">4 bytes of signature value to be written to the specified relative page address.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG213_TT, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Write Signature command with PICC.",
                OtherInfo = @"The WRITE_SIG command allows the writing of a customized originality signature into the dedicated originality signature memory."
            )
        ]
        public Status_t WriteSign ( byte bAddress, byte[] aSignature )
        {
            return phalMful_WriteSign ( m_pDataParams, bAddress, aSignature );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight / NATG Lock Signature command with PICC.
        /// This command locks the signature temporarily or permanently based on the information provided in the API. The locking and unlocking of the
        /// signature can be performed using this command if the signature is not locked or temporary locked. If the signature is permanently locked,
        /// then the unlocking cannot be done.
        /// </summary>
        ///
        /// <param name="bLockMode">Configuration to Lock, Unlock and Permanently Lock the signature data.
        ///                         NxpRdLibNet.alMful.LockSign.UNLOCK
        ///                         NxpRdLibNet.alMful.LockSign.LOCK
        ///                         NxpRdLibNet.alMful.LockSign.PERMENT_LOCK</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES,
                NTAG_Products.NTAG210_MICRO, NTAG_Products.NTAG213_TT, NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD,
                NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform MIFARE(R) Ultralight / NATG Lock Signature command with PICC.",
                OtherInfo = @"The LOCK_SIG command allows the user to unlock, lock or permanently lock the dedicated originality signature memory.
                              The originality signature can only be unlocked, if the originality signature is not permanently locked."
            )
        ]
        public Status_t LockSign ( byte bLockMode )
        {
            return phalMful_LockSign ( m_pDataParams, bLockMode );
        }

        /// <summary>
        /// Perform MIFARE(R) Ultralight Virtual Card Select command with PICC.
        /// </summary>
        ///
        /// <param name="aVCIID">The Virtual Card Installation Identifier data for selecting the VC.</param>
        /// <param name="aVCTID">1 byte of Virtual card type identifier data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_EV1, MIFARE_Products.ULTRALIGHT_AES,
                Message = "Perform MIFARE(R) Ultralight Virtual Card Select command with PICC.",
                OtherInfo = @"The VCSL command is used to enable a unique identification and selection process across different physical 
                              MIFARE product-based cards and virtual MIFARE implementations. The command requires a 16-byte installation 
                              identifier IID and a 4-byte PCD capability value as parameters"
            )
        ]
        public Status_t VirtualCardSelect ( byte[] aVCIID, out byte aVCTID )
        {
            aVCTID = 0;
            Status_t aStatus = phalMful_VirtualCardSelect ( m_pDataParams, aVCIID, ( byte ) aVCIID.Length, ref aVCTID );

            return aStatus;
        }

        /// <summary>
        /// Perform NTAG Tag Temper Status Read.
        /// </summary>
        ///
        /// <param name="bAddr">Value is always 00. Present for forward compatibility response</param>
        /// <param name="aData">5 bytes of Tag Temper Status.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/>CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG213_TT, NTAG_Products.NTAG223_DNA_SD, NTAG_Products.NTAG224_DNA_SD,
                Message = "Perform NTAG Tag Temper Status Read.",
                OtherInfo = @"The READ_TT_Status command provides the information about the tag tamper status which is detected when
                              the product is powered by an RF field."
            )
        ]
        public Status_t ReadTTStatus ( byte bAddr, out byte[] aData )
        {
            aData = new byte[TT_STATUS_LENGTH];
            Status_t oStatus = phalMful_ReadTTStatus ( m_pDataParams, bAddr, aData );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aData = new byte[0];
            }
            return oStatus;
        }

        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig"> Configuration to get the status of CMAC inclusion in command / response.
        ///                         NxpRdLibNet.alMful.Config.CMAC_STATUS
        ///                         NxpRdLibNet.alMful.Config.ADDITINAL_INFO</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> If the configuration identifier is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES,
                Message = "Get the configuration like CMAC Status and to retrieve additional information.",
                OtherInfo = "This is an utility method for internal purpose of the library."
            )
        ]
        public Status_t GetConfig ( ushort wConfig, out ushort wValue )
        {
            wValue = 0;
            return phalMful_GetConfig ( m_pDataParams, wConfig, ref wValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig"> Configuration to set the status of CMAC inclusion in command / response.
        ///                         NxpRdLibNet.alMful.Config.CMAC_STATUS
        ///                         NxpRdLibNet.alMful.Config.ADDITINAL_INFO</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> If the configuration identifier is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.ULTRALIGHT_AES,
                Message = "Set the configuration like CMAC Status and to retrieve additional information.",
                OtherInfo = "This is an utility method for internal purpose of the library."
            )
        ]
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalMful_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Calculate SUN CMAC.
        /// Input for SUNCAMC computation will be the data in hex values (not mirrored ASCII values) of the UID,
        /// NFC counter and Status Detect (if applicable).
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used from software or hardware KeyStore.</param>
        /// <param name="wKeyVer">Key version to be used from software or hardware KeyStore.</param>
        /// <param name="pIndata">Data to calculate Sun CMAC.</param>
        /// <param name="pRespMac">Computed SUN CMAC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffer is null.
        ///     Returns <see cref="Error_Comm.INTEGRITY_ERROR"/> CMAC verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG223_DNA, NTAG_Products.NTAG223_DNA_SD, NTAG_Products.NTAG224_DNA, NTAG_Products.NTAG224_DNA_SD,
                Message = "Compute SUNCMAC.",
                OtherInfo = @"Input for SUNCAMC computation will be the data in hex values (not mirrored ASCII values) of the UID, 
                              NFC counter and Status Detect (if applicable)."
            )
        ]
        public Status_t CalculateSUNCMAC ( ushort wKeyNo, ushort wKeyVer, byte[] pIndata, out byte[] pRespMac )
        {
            pRespMac = new byte[8];
            return phalMful_CalculateSunCMAC ( m_pDataParams, wKeyNo, wKeyVer, pIndata,
            ( ushort ) ( pIndata != null ? pIndata.Length : 0 ), pRespMac );
        }
        #endregion Wrapper Functions

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion Memory Mapping
    }
    #endregion Generic

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    public class Sw : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE Ultralight / NTAG Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to palMifare parameter structure. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to phKeystore parameter structure. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to phCrypto data parameters structure. </summary>
            public IntPtr pCryptoDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Get/Set Config to include CMAC for required Commands. </summary>
            public byte bCMACReq;

            /// <summary> Command count within transaction. </summary>
            public ushort wCmdctr;

            /// <summary> Authentication PWD [0x1B] or AES [0x1A] </summary>
            public byte   bAuthMode;

            /// <summary>
            /// Specific error codes for MFUL generic errors or To get the response length of some commands.
            /// </summary>
            public byte bAdditionalInfo;
        };
        #endregion

        #region DLL Import
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams,
            IntPtr pKeyStoreDataParams, IntPtr pCryptoDataParams, IntPtr pCryptoRngDataParams );
        #endregion DLL Import

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG software component.
        /// </summary>
        ///
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( palMifare.Generic oPalMifare )
        {
            return phalMful_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), oPalMifare.m_pDataParams,
                IntPtr.Zero, IntPtr.Zero, IntPtr.Zero );
        }

        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG products.
        /// </summary>
        ///
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        /// <param name="oKeyStore">Pointer to phKeystore parameter structure.</param>
        /// <param name="oCrypto">Pointer to phCrypto data parameters structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( palMifare.Generic oPalMifare, KeyStore.Generic oKeyStore, CryptoSym.Generic oCrypto,
            CryptoRng.Generic oCryptoRng )
        {
            return phalMful_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero, ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCrypto != null ) ? oCrypto.m_pDataParams : IntPtr.Zero, ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE Ultralight and MIFARE Ultralight NANO products.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init( ushort wDataParamSize, palMifare.Generic oPalMifare)
        {
            return phalMful_Sw_Init( ref m_DataParamsInt[0], wDataParamSize, oPalMifare.m_pDataParams, IntPtr.Zero, IntPtr.Zero, IntPtr.Zero);
        }

        /// <summary>
        /// Initialization API for MIFARE Ultralight C and MIFARE Ultralight EV1 products.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        /// <param name="oKeyStore">Pointer to phKeystore parameter structure.</param>
        /// <param name="oCrypto">Pointer to phCrypto data parameters structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init( ushort wDataParamSize, palMifare.Generic oPalMifare, KeyStore.Generic oKeyStore, CryptoSym.Generic oCrypto, CryptoRng.Generic oCryptoRng)
        {
            return phalMful_Sw_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero, ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCrypto != null ) ? oCrypto.m_pDataParams : IntPtr.Zero, ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Software

    #region Sam NonX
    #region Sam AV2
    /// <summary>
    /// Class for Sam AV2 Non X (S Mode) layer initialization interface and data params.
    /// </summary>
    public class SamAV2 : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE Ultralight / NTAG Sam AV2 Non X (S Mode) layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to halHw_SamAV2 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to palMifare parameter structure. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Specific error codes for MFUL generic errors or To get the response length of some commands. </summary>
            public byte bAdditionalInfo;
        };
        #endregion Data Structure

        #region DLL Import
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SamAV2_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
        #endregion DLL Import

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV2 in S Mode.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( Hal.SamAV2 oHal, palMifare.Generic oPalMifare )
        {
            return phalMful_SamAV2_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV2 in S Mode.
        /// </summary>
        ///
        /// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init( ushort dwDataParamsSize, Hal.SamAV2 oHal, palMifare.Generic oPalMifare )
        {
            return phalMful_SamAV2_Init ( ref m_DataParamsInt[0], dwDataParamsSize,
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV2 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion Sam AV2

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam AV3
    /// <summary>
    /// Class for Sam AV3 Non X (S Mode) layer initialization interface and data params.
    /// </summary>
    public class SamAV3_NonX : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE Ultralight / NTAG Sam AV2 Non X (S Mode) layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to halHw_SamAV2 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to palMifare parameter structure. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Get/Set Config to include CMAC for required Commands. </summary>
            public byte bCMACReq;

            /// <summary> Command count within transaction. </summary>
            public ushort wCmdctr;

            /// <summary> Authentication PWD [0x1B] or AES [0x1A] </summary>
            public byte   bAuthMode;

            /// <summary> Specific error codes for MFUL generic errors or To get the response length of some commands. </summary>
            public byte bAdditionalInfo;

            /// <summary> Ram Key Number </summary>
            public byte   bRamKeyNo;

            /// <summary> Ram Key Version</summary>
            public byte   bRamKeyVer;
        };
        #endregion Data Structure

        #region DLL Import
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
        #endregion DLL Import

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV3 in S Mode.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( Hal.SamAV3 oHal, palMifare.Generic oPalMifare )
        {
            return phalMful_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV3 in S Mode.
        /// </summary>
        ///
        /// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( ushort dwDataParamsSize, Hal.SamAV3 oHal, palMifare.Generic oPalMifare )
        {
            return phalMful_SamAV3_NonX_Init ( ref m_DataParamsInt[0], dwDataParamsSize,
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Sam AV3
#endif
    #endregion  Sam NonX

    #region Sam X
    #region Sam AV2
    /// <summary>
    /// Class for Sam AV2 X layer initialization interface and data params.
    /// </summary>
    public class SamAV2_X : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE Ultralight / NTAG Sam AV2 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to halHw_SamAV2 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to palMifare parameter structure. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Specific error codes for MFUL generic errors or To get the response length of some commands. </summary>
            public byte bAdditionalInfo;
        };
        #endregion Data Structure

        #region DLL Import
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
        #endregion DLL Import

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV2 in X Mode.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( Hal.SamAV2 oHal, palMifare.SamAV2_X oPalMifare )
        {
            return phalMful_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV2 in X Mode.
        /// </summary>
        ///
        /// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHal">Pointer to Hal SamAV2 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init( ushort dwDataParamsSize, Hal.SamAV2 oHal, palMifare.SamAV2_X oPalMifare )
        {
            return phalMful_SamAV2_X_Init ( ref m_DataParamsInt[0], dwDataParamsSize,
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Sam AV2

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam AV3
    /// <summary>
    /// Class for Sam AV3 X layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE Ultralight / NTAG Sam AV3 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to halHw_SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to palMifare parameter structure. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Specific error codes for MFUL generic errors or To get the response length of some commands. </summary>
            public byte bAdditionalInfo;
        };
        #endregion Data Structure

        #region DLL Import

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMful_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV3 in X Mode.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init ( Hal.SamAV3 oHal, palMifare.SamAV3_X oPalMifare )
        {
            return phalMful_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE Ultralight / NTAG to communicate with SAM AV3 in X Mode.
        /// </summary>
        ///
        /// <param name="dwDataParamsSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHal">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to palMifare parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_DATA_PARAMS"/>
        ///         If the input size doest not match the DataParams size of this component.
        ///         If any of the DataParams are null.
        /// </returns>
        public Status_t Init (ushort dwDataParamsSize, Hal.SamAV3 oHal, palMifare.SamAV3_X oPalMifare )
        {
            return phalMful_SamAV3_X_Init( ref m_DataParamsInt[0], dwDataParamsSize,
                ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero, ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero );
        }
#endif

        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Sam AV3
#endif
    #endregion Sam X
}
