/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alNtagXDna
{
    #region Enumerations
    #region Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary>
        /// Field to indicate Out Of Memory error.
        /// Insufficient NV-Memory to complete command.
        /// </summary>
        OUT_OF_EEPROM_ERROR = CustomCodes.ERROR_BEGIN + 1,

        /// <summary>
        /// Field to indicate No Suck Key error.
        /// Invalid key number specified.
        /// </summary>
        NO_SUCH_KEY = CustomCodes.ERROR_BEGIN + 2,

        /// <summary>
        /// Field to indicate Permission Denied error.
        /// Current configuration / Status does not allow the requested command.
        /// </summary>
        PERMISSION_DENIED = CustomCodes.ERROR_BEGIN + 3,

        /// <summary>
        /// Field to indicate Boundary error.
        /// Current configuration / Status does not allow the requested command.
        /// </summary>
        BOUNDARY_ERROR = CustomCodes.ERROR_BEGIN + 4,

        /// <summary>
        /// Field to indicate Command Aborted error.
        /// Previous Command was not fully completed. Not all Frames were requested or provided by the PCD.
        /// </summary>
        COMMAND_ABORTED = CustomCodes.ERROR_BEGIN + 5,

        /// <summary>
        /// Field to indicate Duplicate error.
        /// Creation of file/application failed because file/application with same number already exists.
        /// </summary>
        DUPLICATE = CustomCodes.ERROR_BEGIN + 6,

        /// <summary>
        /// Field to indicate File Not Found error.
        /// Specified file number does not exist.
        /// </summary>
        FILE_NOT_FOUND = CustomCodes.ERROR_BEGIN + 7,

        /// <summary>
        /// Field to indicate Integrity error.
        /// CRC or MAC does not match data. Padding bytes not valid.
        /// </summary>
        PICC_CRYPTO = CustomCodes.ERROR_BEGIN + 8,

        /// <summary>
        /// Field to indicate Parameter error.
        /// Value of the parameter(s) invalid.
        /// </summary>
        PARAMETER_ERROR = CustomCodes.ERROR_BEGIN + 9,

        /// <summary>
        /// Field to indicate tag's General error. Call <see cref="Generic.GetConfig"/>
        /// interface with option as <see cref="Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///     <item><description>Memory Error (0xEE).</description></item>
        /// </list>
        /// </summary>
        DF_GEN_ERROR = CustomCodes.ERROR_BEGIN + 10,

        /// <summary>
        /// Field to indicate tag's ISO7816 error(s). Call <see cref="Generic.GetConfig"/>
        /// interface with option as <see cref="Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///     <item><description>Correct Operation with maximum supported protocol version (0x9Fxx).</description></item>
        ///     <item><description>Memory Failure (0x6581).</description></item>
        ///     <item><description>Wrong Length (0x6700).</description></item>
        ///     <item><description>Wrong Le (0x6C00).</description></item>
        ///     <item><description>Incorrect parameters in the command data field (0x6A80).</description></item>
        ///     <item><description>File not found (0x6A82).</description></item>
        ///     <item><description>Wrong Params (0x6A86).</description></item>
        ///     <item><description>Wrong LC (0x6A87).</description></item>
        ///     <item><description>No Precise Diagnostics (0x6F00).</description></item>
        ///     <item><description>EOF Reached (0x6282).</description></item>
        ///     <item><description>File Access (0x6982).</description></item>
        ///     <item><description>File Empty (0x6985).</description></item>
        ///     <item><description>Incorrect Params (0x6B00).</description></item>
        ///     <item><description>Wrong CLA (0x6E00).</description></item>
        ///     <item><description>Unsupported INS (0x6D00).</description></item>
        ///     <item><description>Limited Functionality INS (0x6283).</description></item>
        /// </list>
        /// </summary>
        DF_7816_GEN_ERROR = CustomCodes.ERROR_BEGIN + 11,

        /// <summary>
        /// Field to indicate Certificate error.
        /// Reader certificate or CertAccessRights related error.
        /// </summary>
        CERTIFICATE_ERROR = CustomCodes.ERROR_BEGIN + 12,

        /// <summary>
        /// Field to indicate Weak Field error.
        /// Field strength not sufficient to enable power harvesting for the targeted current/voltage level.
        /// </summary>
        WEAK_FIELD_ERROR = CustomCodes.ERROR_BEGIN + 13,

        /// <summary>
        /// Field to indicate Pad Voltage Unreliable error.
        /// Field strength not sufficient to enable power harvesting for the targeted current/voltage level.
        /// </summary>
        PAD_VOLTAGE_UNRELIABLE_ERROR = CustomCodes.ERROR_BEGIN + 14,

        /// <summary> Field to indicate Device Signature Verification failure. </summary>
        DEVICE_SIGNATURE_ERROR = CustomCodes.ERROR_BEGIN + 15,

        /// <summary> Field to indicate Device EndLeaf -Certificate Hash Verification failure. </summary>
        DEVICE_LEAF_CERTIFICATE_HASH_ERROR = CustomCodes.ERROR_BEGIN + 16,

        /// <summary> Field to indicate Signature Verification failure. </summary>
        SIGNATURE_VERIFICATION_FAILURE = CustomCodes.ERROR_BEGIN + 17
    }
    #endregion

    #region Secure Messaging Options
    #region Certificate Depth
    /// <summary>
    /// Option to indicate the Certificate Depth to be used with <see cref="Generic.ISOGeneralAuthenticate"/>.
    /// </summary>
    public enum Certificate_Depth : byte
    {
        /// <summary>
        /// Option to indicate Certificate depth as End-Leaf only.
        /// Also called as Depth Zero.
        /// </summary>
        END_LEAF_ONLY = 0x00,

        /// <summary>
        /// Option to indicate Certificate depth as End-Leaf and Parent.
        /// Also called as Depth One.
        /// </summary>
        END_LEAF_PARENT = 0x01,

        /// <summary>
        /// Option to indicate Certificate depth as End-Leaf, Parent and Grand-Parent.
        /// Also called as Depth Two.
        /// </summary>
        END_LEAF_PARENT_GRAND_PARENT = 0x02
    }
    #endregion

    #region Certificate Compression Type
    /// <summary>
    /// Option to indicate the Certificate Compression Type to be used with <see cref="Generic.ISOGeneralAuthenticate"/>.
    /// </summary>
    public enum Certificate_CompressionType : byte
    {
        /// <summary> Option to indicate Certificate compression type as UnCompressed. </summary>
        UN_COMPRESSED = 0x00,

        /// <summary> Option to indicate Certificate compression type as Compressed. </summary>
        COMPRESSED = 0x01
    }
    #endregion

    #region Session KeySize
    /// <summary>
    /// Option to indicate the Session KeySize to be used with <see cref="Generic.ISOGeneralAuthenticate"/>.
    /// </summary>
    public enum Session_KeySize : byte
    {
        /// <summary> Option to indicate Session Key Size as AES128. </summary>
        AES128 = 0x01,

        /// <summary> Option to indicate Session Key Size as AES256. </summary>
        AES256 = 0x02
    }
    #endregion

    #region Auth Types
    /// <summary>
    /// Type of ISOGeneralAuthentication or EV2 Authentication to perform. To be used with below mentioned interfaces.
    /// <see cref="Generic.AuthenticateEv2"/>
    /// </summary>
    public enum AuthType : byte
    {
        /// <summary> Authentication type as EV2 NonFirst. </summary>
        EV2_NON_FIRST = 0x00,

        /// <summary> Authentication type as EV2 First. </summary>
        EV2_FIRST = 0x01,

        /// <summary> Authentication type as Not Authenticated. </summary>
        NOT_AUTHENTICATED = 0xFF
    }
    #endregion Auth Types

    #region Diversification Options
    /// <summary>
    /// Diversification method to be used for key diversification. To be used with <see cref="Generic.AuthenticateEv2"/>
    /// interface.
    /// </summary>
    public enum DivOption_Auth : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary>
        /// Encryption based method of diversification.
        /// Also known as DESFire mode of diversification.
        /// </summary>
        ENCR = 0x0000,

        /// <summary>
        /// CMAC based method of diversification.
        /// Also known as PLUS mode of diversification.
        /// </summary>
        CMAC = 0x0001
    }
    #endregion Auth Types
    #endregion

    #region Communication Options
    /// <summary>
    /// The communication mode to be used for Command / Response.
    /// </summary>
    public enum CommOption : byte
    {
        /// <summary> Plain (0x00) mode of communication. The Command / Response will be is plain format.</summary>
        PLAIN = 0x00,

        /// <summary> Plain (0x20) mode of communication. The Command / Response will be is plain format.</summary>
        PLAIN_20 = 0x20,

        /// <summary> MAC mode of communication. The Command / Response will have MAC appended at last. </summary>
        MAC = 0x10,

        /// <summary>
        /// Enciphered mode of communication. The Command Data / Response Data will be encrypted
        /// and will have MAC appended at last.
        /// </summary>
        FULL = 0x30,
    }
    #endregion

    #region TargetCurve
    /// <summary>
    /// Target Curve IS to be used with below mentioned interface.
    /// - <see cref="Generic.ManageKeyPair"/>
    /// - <see cref="Generic.ManageCARootKey"/>
    /// - <see cref="Generic.CryptoRequest_ECCVerify"/>
    /// </summary>
    public enum TargetCurve
    {
        /// <summary> Option for NIST P-256 Curve ID. </summary>
        NIST_P256 = 0x0C,

        /// <summary> Option for BrainPool P-256 R1 Curve ID. </summary>
        BRAINPOOL_P256R1 = 0x0D
    }
    #endregion

    #region Memory and Configuration
    #region SetConfiguration
    /// <summary>
    /// Options for <see cref="Generic.SetConfiguration"/> interface.
    /// </summary>
    public enum SetConfig : byte
    {
        /// <summary> Option for updating the tag Configuration. </summary>
        PICC_CONFIGURATION = 0x00,

        /// <summary> Option for updating the ATS. </summary>
        ATS_UPDATE = 0x02,

        /// <summary> Option for updating the SAK. </summary>
        SAK_UPDATE = 0x03,

        /// <summary> Option for updating the Secure Messaging. </summary>
        SM_CONFIGURATION = 0x04,

        /// <summary> Option for updating the Capability Data. </summary>
        CAPABILITY_DATA = 0x05,

        /// <summary> Option for updating the ATQA Information. </summary>
        ATQA_UPDATE = 0x0C,

        /// <summary> Option for updating the Silent Mode configuration. </summary>
        SILENT_MODE_CONFIGURATION = 0x0D,

        /// <summary> Option for updating the Enhanced Privacy configuration. </summary>
        ENHANCED_PRIVACY_CONFIGURATION = 0x0E,

        /// <summary> Option for updating the NFC Management. </summary>
        NFC_MANAGEMENT = 0x0F,

        /// <summary> Option for updating the I2C Management. </summary>
        I2C_MANAGEMENT = 0x10,

        /// <summary> Option for updating the GPIO Management. </summary>
        GPIO_MANAGEMENT = 0x11,

        /// <summary> Option for updating the ECC Key Management. </summary>
        ECC_KEY_MANAGEMENT = 0x12,

        /// <summary> Option for updating the Certificate Management. </summary>
        CERTIFICATE_MANAGEMENT = 0x13,

        /// <summary> Option for updating the Watchdog Timer Management. </summary>
        WATCHDOG_TIMER_MANAGEMENT = 0x14,

        /// <summary> Option for updating the Crypto API Management. </summary>
        CRYPTO_API_MANAGEMENT = 0x15,

        /// <summary> Option for updating the Authentication Counter and Limit Configuration. </summary>
        AUTH_COUNTR_LIMIT_CONFIGURATION = 0x16,

        /// <summary> Option for updating the HALT and Wake-Up Configuration. </summary>
        HALT_WAKEUP_CONFIGURATION = 0x17,

        /// <summary> Option for updating the Deferred Configurations. </summary>
        DEFERRED_CONFIGURATION = 0xFE,

        /// <summary> Option for updating the Lock Configuration. </summary>
        LOCK_CONFIGURATION = 0xFF
    }
    #endregion

    #region GetVersion
    /// <summary>
    /// Options for <see cref="Generic.GetVersion"/> interface.
    /// </summary>
    public enum FabID : byte
    {
        /// <summary>
        /// Option for not exchanging the Option information in <see cref="Generic.GetVersion">GetVersion</see>
        /// command. If used, the FabID will not be available in the response.
        /// </summary>
        DO_NOT_RETURN = 0x00,

        /// <summary>
        /// Option for exchanging the Option information in <see cref="Generic.GetVersion">GetVersion</see>
        /// command to retrieve the FabID information. If used, the FabID will be available in the response.
        /// </summary>
        RETURN = 0x01
    }
    #endregion
    #endregion

    #region Symmetric Key Management
    #region Diversification Options
    /// <summary>
    /// Diversification options to be used with below mentioned interfaces.
    /// <see cref="Generic.ChangeKey"/>
    /// </summary>
    public enum DivOption_ChangeKey : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> Indicating diversification of new key required. </summary>
        NEW_KEY = 0x0002,

        /// <summary> Indicating diversification of old key required. </summary>
        OLD_KEY = 0x0004,

        /// <summary>
        /// Encryption based method of diversification.
        /// Also known as DESFire mode of diversification.
        /// </summary>
        ENCR = 0x0000,

        /// <summary>
        /// CMAC based method of diversification.
        /// Also known as PLUS mode of diversification.
        /// </summary>
        CMAC = 0x0001
    }
    #endregion

    #region KeySetting
    /// <summary>
    /// Option to be used for <see cref="Generic.GetKeySettings"/> interface.
    /// </summary>
    public enum KeySetting : byte
    {
        /// <summary> Option for Unknown Key settings option. </summary>
        UNKNOWN = 0xFF,

        /// <summary> Option for not exchanging option information. </summary>
        EXCLUDE_OPTION = 0xFE,

        /// <summary> Option for retrieval of CryptoRequest Keys meta-data </summary>
        CRYPTOREQUEST_KEY_METADATA = 0x00,

        /// <summary> Option for retrieval of ECC Private Key MetaData. </summary>
        ECC_PRIVATE_KEY_METADATA = 0x01,

        /// <summary> Option for retrieval of CA Root Key MetaData. </summary>
        CA_ROOT_KEY_METADATA = 0x02
    }
    #endregion

    #region KeyPolicy
    /// <summary>
    /// KeyPolicy Options to be used with <see cref="Generic.ChangeKey"/> interface.
    /// </summary>
    public enum KeyPolicy_Sym
    {
        /// <summary> Option for Key Policy as disabled. </summary>
        DISABLED = 0x0000,

        /// <summary> Option for Key Policy as CMAC verification. </summary>
        MAC_VERIFY = 0x0001,

        /// <summary> Option for Key Policy as CMAC generation. </summary>
        MAC_SIGN = 0x0002,

        /// <summary> Option for Key Policy as ECB / CBC Decryption. </summary>
        ECB_CBC_DECRYPT = 0x0004,

        /// <summary> Option for Key Policy as ECB / CBC Encryption. </summary>
        ECB_CBC_ENCRYPT = 0x0008,

        /// <summary> Option for Key Policy as GCM, CCM Decryption or Verification. </summary>
        CCM_GCM_DECRYPT_VERIFY = 0x0010,

        /// <summary> Option for Key Policy as GCM, CCM Encryption or Signing. </summary>
        CCM_GCM_ENCRYPT_SIGN = 0x0020,

        /// <summary> Option for Key Policy as GCM, CCM Encryption or Signing with internal Nonce. </summary>
        CCM_GCM_ENCRYPT_SIGN_INTERNAL_NONCE = 0x0040,

        /// <summary> Option for Key Policy as HMAC. </summary>
        HMAC = 0x0080,

        /// <summary> Option for Key Policy as HKDF. </summary>
        HKDF = 0x0100
    }
    #endregion
    #endregion

    #region ASymmetric Key Management
    #region TargetAction
    /// <summary>
    /// Target Action options to be used with <see cref="Generic.ManageKeyPair"/> interface.
    /// </summary>
    public enum TargetAction
    {
        /// <summary> Option for Key Pair generation. </summary>
        GENERATE_KEY_PAIR = 0x00,

        /// <summary> Option for Private Key Import. </summary>
        IMPORT_PRIVATE_KEY = 0x01,

        /// <summary> Option for Meta-Data update. </summary>
        UPDATE_META_DATA = 0x02
    }
    #endregion

    #region KeyPolicy
    /// <summary>
    /// KeyPolicy Options to be used with <see cref="Generic.ManageKeyPair"/> interface.
    /// </summary>
    public enum KeyPolicy_ASym
    {
        /// <summary> Option for Key Policy as disabled. </summary>
        DISABLED = 0x0000,

        /// <summary> Option for Key Policy as Freeze Key Usage Counter Limit. </summary>
        FREEZE_KUC_LIMIT = 0x8000,

        /// <summary> Option for Key Policy as ECC Based Card-Unilateral with ISOInternalAuthenticate. </summary>
        ECC_CARD_UNILATERAL_AUTHENTICATION = 0x0100,

        /// <summary> Option for Key Policy as ECC Based Secure Dynamic Messaging. </summary>
        ECC_SECURE_DYNAMINC_MESSAGING = 0x0020,

        /// <summary> Option for Key Policy as ECC Based CryptoRequest ECC Sign. </summary>
        CRYPTO_REQUEST_ECC_SIGN = 0x0010,

        /// <summary> Option for Key Policy as ECC Based CryptoRequest ECC DH. </summary>
        CRYPTO_REQUEST_ECC_DH = 0x0008,

        /// <summary> Option for Key Policy as ECC Based SIGMA-I Mutual Authentication. </summary>
        ECC_SIGMA_I_MUTUAL_AUTH = 0x0004
    }
    #endregion
    #endregion

    #region Certificate Management
    #region Action
    /// <summary>
    /// Action options to be used with <see cref="Generic.ManageCertRepo"/> interface.
    /// </summary>
    public enum Action : byte
    {
        /// <summary>Option for Certificate Repository creation.</summary>
        CREATE_CERTIFICATE_REPOSITORY = 0x00,

        /// <summary>Option for Loading Certificate.</summary>
        LOAD_CERTIFICATE = 0x01,

        /// <summary>Option for Loading Certificate Mapping information.</summary>
        LOAD_CERTIFICATE_MAPPING_INFO = 0x03,

        /// <summary>Option for Activating Repository.</summary>
        ACTIVATE_REPOSITORY = 0x04,

        /// <summary>Option for Resetting Certificate Repository.</summary>
        RESET_CERTIFICATE_REPOSITORY = 0x05
    }
    #endregion

    #region Action
    /// <summary>
    /// Data Item options to be used with <see cref="Generic.ReadCertRepo"/> interface.
    /// </summary>
    public enum DataItem : byte
    {
        /// <summary>Option for Data Item as End Leaf.</summary>
        END_LEAF = 0x00,

        /// <summary>Option for Data Item as Parent.</summary>
        PARENT = 0x01,

        /// <summary>Option for Data Item as Grand Parent.</summary>
        GRAND_PARENT = 0x02,

        /// <summary>Option for Repository Meta-data.</summary>
        REPOSITORY_METADATA = 0xFF,
    }
    #endregion
    #endregion

    #region File Management
    #region ISOFileInfo
    /// <summary>
    /// Options for application / file creation interfaces.
    /// </summary>
    public enum ISOFileInfo : byte
    {
        /// <summary> Option to indicate no ISO File ID or ISODFName are present. </summary>
        NOT_AVAILABLE = 0x00,

        /// <summary> Option to indicate the presence of ISO FileID. </summary>
        ISO_FILE_ID_AVAILABLE = 0x01
    }
    #endregion

    #region File Options
    /// <summary>
    /// The File Options to be used for all the File management commands.
    /// </summary>
    public enum FileOption : byte
    {
        /// <summary> Option for File communication mode as Plain.</summary>
        PLAIN = 0x00,

        /// <summary> Option for File communication mode as Plain. </summary>
        PLAIN_1 = 0x02,

        /// <summary> Option for File communication mode as Mac. </summary>
        MAC = 0x01,

        /// <summary> Option for File communication mode as Full. </summary>
        FULL = 0x03,

        /// <summary> Option to Enable Secure Dynamic Messaging and Mirroring support. </summary>
        SDM_MIRRORING_ENABLED = 0x40,

        /// <summary> Option to Enable Deferred Configuration support. </summary>
        DEFERRED_CONFIGURATION_ENABLED = 0x08
    }
    #endregion
    #endregion

    #region CryptoAPI
    #region Operation
    /// <summary>
    /// The Options for representing Crypto Operation. To be used with
    /// <see cref="Generic.CryptoRequest_SHA"/>
    /// <see cref="Generic.CryptoRequest_ECCSign"/>
    /// <see cref="Generic.CryptoRequest_ECCVerify"/>
    /// <see cref="Generic.CryptoRequest_AES"/>
    /// <see cref="Generic.CryptoRequest_AESCMAC"/>
    /// <see cref="Generic.CryptoRequest_HMAC"/>
    /// </summary>
    public enum CryptoOperation : byte
    {
        /// <summary> Option to indicate initialization of Crypto Operation. </summary>
        INIT = 0x01,

        /// <summary> Option to indicate update of Crypto Operation. </summary>
        UPDATE = 0x02,

        /// <summary> Option to indicate finalization of Crypto Operation. </summary>
        FINALIZE = 0x03,

        /// <summary> Option to indicate Crypto Operation in one shot. </summary>
        ONE_SHOT = 0x04,

        /// <summary> Option to indicate Crypto Operation in one shot with Hash precomputed on datas. </summary>
        ONE_SHOT_PRE_COMPUTED_HASH = 0x05
    }

    /// <summary>
    /// The Options for representing Crypto Operation. To be used with
    /// <see cref="Generic.CryptoRequest_ECCDH"/>
    /// </summary>
    public enum CryptoOperation_DH : byte
    {
        /// <summary> Option to indicate ECC DH Crypto Operation in One Step. </summary>
        ONE_STEP = 0x01,

        /// <summary> Option to indicate ECC DH Crypto Operation in Two Step first step. </summary>
        TWO_STEP_FIRST = 0x02,

        /// <summary> Option to indicate ECC DH Crypto Operation in Two Step final step. </summary>
        TWO_STEP_FINAL = 0x03
    }

    /// <summary>
    /// The Options for representing Crypto Operation. To be used with
    /// <see cref="Generic.CryptoRequest_HKDF"/>
    /// </summary>
    public enum CryptoOperation_HKDF : byte
    {
        /// <summary> Option to indicate ECC DH Crypto Operation in One Step. </summary>
        EXTRACT_EXPAND = 0x00,

        /// <summary> Option to indicate ECC DH Crypto Operation in Two Step first step. </summary>
        EXPAND = 0x01
    }
    #endregion

    #region Algorithm
    /// <summary>
    /// The Options for representing Crypto Algorithm's. To be used with
    /// <see cref="Generic.CryptoRequest_SHA"/>
    /// <see cref="Generic.CryptoRequest_HMAC"/>
    /// </summary>
    public enum CryptoAlgorithm : byte
    {
        /// <summary> Option to indicate Crypto Algorithm as SHA-256. </summary>
        SHA_256 = 0x01,

        /// <summary> Option to indicate Crypto Algorithm as SHA-384. </summary>
        SHA_384 = 0x02
    }
    #endregion

    #region AES Primitives
    /// <summary>
    /// The Options for representing AES Primitive. To be used with
    /// <see cref="Generic.CryptoRequest_AES"/>
    /// <see cref="Generic.CryptoRequest_AESCMAC"/>
    /// <see cref="Generic.CryptoRequest_HMAC"/>
    /// </summary>
    public enum Primitive : byte
    {
        /// <summary> Option to indicate Primitive as Sign (AES-CMAC or HMAC Generation). </summary>
        SIGN = 0x01,

        /// <summary> Option to indicate Primitive as Verify (AES-CMAC or HMAC Verification). </summary>
        VERIFY = 0x02,

        /// <summary> Option to indicate AES Primitive as CBC Encryption. </summary>
        CBC_ENCRYPT = 0x03,

        /// <summary> Option to indicate AES Primitive as CBC Decryption. </summary>
        CBC_DECRYPT = 0x04,

        /// <summary> Option to indicate AES Primitive as ECB Encryption. </summary>
        ECB_ENCRYPT = 0x05,

        /// <summary> Option to indicate AES Primitive as ECB Decryption. </summary>
        ECB_DECRYPT = 0x06,

        /// <summary> Option to indicate AES Primitive as CCM Encryption / Signing </summary>
        CCM_ENCRYPT_SIGN = 0x07,

        /// <summary>
        /// Option to indicate AES Primitive as CCM Encryption / Signing with
        /// internally generate nonce.
        /// </summary>
        CCM_ENCRYPT_SIGN_INTERNAL_NONCE = 0x08,

        /// <summary> Option to indicate AES Primitive as CCM Decryption / Verification </summary>
        CCM_DECRYPT_VERIFY = 0x09,

        /// <summary> Option to indicate AES Primitive as GCM Encryption / Signing </summary>
        GCM_ENCRYPT_SIGN = 0x0A,

        /// <summary>
        /// Option to indicate AES Primitive as GCM Encryption / Signing with internally
        /// generate nonce.
        /// </summary>
        GCM_ENCRYPT_SIGN_INTERNAL_NONCE = 0x0B,

        /// <summary> Option to indicate AES Primitive as GCM Decryption / Verification. </summary>
        GCM_DECRYPT_VERIFY = 0x0C
    }
    #endregion
    #endregion

    #region GPIO Management
    #region GPIO Number
    /// <summary>
    /// The Options for representing GPIO number. To be used with <see cref="ManageGPIO"/> interface.
    /// </summary>
    public enum GPIONo
    {
        /// <summary> Option to indicate GPIO number as 1. </summary>
        GPIO_1 = 0x00,

        /// <summary> Option to indicate GPIO number as 2. </summary>
        GPIO_2 = 0x01
    }
    #endregion

    #region Operation
    /// <summary>
    /// The Options for representing operation to perform for the respective GPIO.
    /// To be used with <see cref="ManageGPIO"/> interface.
    /// </summary>
    public static class GPIO_Operation
    {
        #region GPIOxMode Output
        /// <summary> The Options for representing operation to perform for the GPIO configured as output. </summary>
        public enum Output
        {
            /// <summary>
            /// Option to indicate GPIO Control as CLEAR.
            /// Clear the GPIO state to LOW(not driven) or stop power harvesting depending on the mode.
            /// </summary>
            GPIO_CONTROL_CLEAR = 0x00,

            /// <summary>
            /// Option to indicate GPIO Control as SET.
            /// Set the GPIO State to HIGH (driven) or start power harvesting depending on the mode.
            /// </summary>
            GPIO_CONTROL_SET = 0x01,

            /// <summary>
            /// Option to indicate GPIO Control as TOGGLE.
            /// Toggle the GPIO State or power harvesting state depending on the mode.
            /// </summary>
            GPIO_CONTROL_TOGGLE = 0x02,

            /// <summary>
            /// Combined option
            ///     - Option to indicate Pause NFC (only accepted on NFC).
            ///     - Option to indicate Release NFC Pause (only accepted on I2C)
            /// </summary>
            PAUSE_RELEASE_NFC = 0x80
        }
        #endregion

        #region GPIOxMode Down-Stream Power Out
        /// <summary> The Options for representing operation to perform for the GPIO configured as Down-Stream Power Out. </summary>
        public static class DownStreamPowerOut
        {
            /// <summary> Option to indicate GPIO Control to Enable Power Harvesting. </summary>
            public const byte ENABLE_POWER_HARVEST = 0x01;

            /// <summary> Option to indicate GPIO Measurement Control to Execute Measurement. </summary>
            public const byte EXECUTE_MEASUREMENT = 0x02;

            #region 1.8V
            /// <summary> Option to indicate Power downstream voltage of 1.8V. </summary>
            public enum Voltage_1_8
            {
                /// <summary> Option to indicate Power downstream current of 100uA. </summary>
                CURRENT_100uA = 0x04,

                /// <summary> Option to indicate Power downstream current of 300uA. </summary>
                CURRENT_300uA = 0x08,

                /// <summary> Option to indicate Power downstream current of 500uA. </summary>
                CURRENT_500uA = 0x0C,

                /// <summary> Option to indicate Power downstream current of 1mA. </summary>
                CURRENT_1mA = 0x10,

                /// <summary> Option to indicate Power downstream current of 2mA. </summary>
                CURRENT_2mA = 0x14,

                /// <summary> Option to indicate Power downstream current of 3mA. </summary>
                CURRENT_3mA = 0x18,

                /// <summary> Option to indicate Power downstream current of 5mA. </summary>
                CURRENT_5mA = 0x1C,

                /// <summary> Option to indicate Power downstream current of 7mA. </summary>
                CURRENT_7mA = 0x20,

                /// <summary> Option to indicate Power downstream current of 10mA. </summary>
                CURRENT_10mA = 0x24,

                /// <summary> Option to indicate Power downstream voltage of 1.8V and maximum available current. </summary>
                CURRENT_MAX_AVAILABLE = 0x3C,
            }
            #endregion

            #region 2V
            /// <summary> Option to indicate Power downstream voltage of 1.8V. </summary>
            public enum Voltage_2
            {
                /// <summary> Option to indicate Power downstream current of 100uA. </summary>
                CURRENT_100uA = 0x44,

                /// <summary> Option to indicate Power downstream current of 300uA. </summary>
                CURRENT_300uA = 0x48,

                /// <summary> Option to indicate Power downstream current of 500uA. </summary>
                CURRENT_500uA = 0x4C,

                /// <summary> Option to indicate Power downstream current of 1mA. </summary>
                CURRENT_1mA = 0x50,

                /// <summary> Option to indicate Power downstream current of 2mA. </summary>
                CURRENT_2mA = 0x54,

                /// <summary> Option to indicate Power downstream current of 3mA. </summary>
                CURRENT_3mA = 0x58,

                /// <summary> Option to indicate Power downstream current of 5mA. </summary>
                CURRENT_5mA = 0x5C,

                /// <summary> Option to indicate Power downstream current of 7mA. </summary>
                CURRENT_7mA = 0x60,

                /// <summary> Option to indicate Power downstream current of 10mA. </summary>
                CURRENT_10mA = 0x64,

                /// <summary> Option to indicate Power downstream voltage of 1.8V and maximum available current. </summary>
                CURRENT_MAX_AVAILABLE = 0x7C,
            }
            #endregion
        }
        #endregion
    }
    #endregion
    #endregion

    #region ISO7816 - 4
    #region ISOSelect Options
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 FCI modes.
    /// To be used for <see cref="Generic.IsoSelectFile"/> interface.
    /// </summary>
    public enum FCI : byte
    {
        /// <summary> Option to indicate the return of FCI. </summary>
        RETURNED = 0x00,

        /// <summary> Option to indicate the no return of FCI. </summary>
        NOT_RETURNED = 0x0C
    }

    /// <summary>
    /// Definitions for ISO/IEC 7816-4 Selection Controls.
    /// To be used for <see cref="Generic.IsoSelectFile"/> interface.
    /// </summary>
    public enum Selector : byte
    {
        /// <summary> Option to indicate Selection by 2 byte file Id. </summary>
        SELECT_MF_DF_EF_FILE_IDENTIFIER = 0x00,

        /// <summary> Option to indicate Selection by child DF. </summary>
        SELECT_CHILD_DF = 0x01,

        /// <summary> Option to indicate Select EF under current DF. FID = EF id. </summary>
        SELECT_EF_CURRENT_DF = 0x02,

        /// <summary> Option to indicate Select parent DF of the current DF. </summary>
        SELECT_PARENT_DF_CURRENT_DF = 0x03,

        /// <summary> Option to indicate Selection by DF Name. DFName and length is then valid. </summary>
        SELECT_BY_DF_NAME = 0x04
    }
    #endregion

    #region APDU Format
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 APDU Format.
    /// To be used all ISO7816 - 4 interface.
    /// </summary>
    public enum ApduFormat : byte
    {
        /// <summary>
        /// Option to indicate ISO7816-4 APDU format is Short Length
        /// format where LC and LE are of 1 byte.
        /// </summary>
        SHORT_LEN = 0x00,

        /// <summary>
        /// Option to indicate ISO7816-4 APDU format is Extended Length
        /// format where LC is 3 bytes and LE is either 2 or 3 bytes.
        /// </summary>
        EXTENDED_LEN = 0x01
    }
    #endregion

    #region Encoding
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 P1-P2 Encoding of ShortFile identifier or Offset.
    /// To be used with below mentioned interfaces.
    ///     <see cref="Generic.IsoReadBinary"/>
    ///     <see cref="Generic.IsoUpdateBinary"/>
    /// </summary>
    public enum SFID : byte
    {
        /// <summary> Field to indicate Encoding as offset </summary>
        DISABLED = 0x00,

        /// <summary> Field to indicate Encoding as Short File Identifier </summary>
        ENABLED = 0x80
    }
    #endregion
    #endregion

    #region Utility Options
    #region Configuration
    /// <summary>
    /// Option to configure some special operations. To be used with the below mentioned interfaces.
    /// <see cref="Generic.GetConfig"/>
    /// <see cref="Generic.SetConfig"/>
    /// </summary>
    public enum Config : ushort
    {
        /// <summary>
        /// Option for GetConfig/SetConfig to get/set additional info of a generic error or some length exposed by interfaces.
        /// </summary>
        ADDITIONAL_INFO = 0x00A1,

        /// <summary>
        /// Option for GetConfig/SetConfig to get/set current Status of Short Length APDU wrapping in ISO 7816-4 APDUs.
        /// 1: The commands will follow ISO7816 wrapped format with LC and LE as 1 byte.
        /// 0: The commands will follow ISO7816 wrapped format with LC as 3 bytes and LE as 2 or 3 bytes.
        /// </summary>
        SHORT_LENGTH_APDU = 0x00A2,
    };
    #endregion

    #region SDM Options
    /// <summary>
    /// Macro Definitions for below mentioned utility interfaces.
    /// <see cref="Generic.CalculateMACSDM"/>
    /// <see cref="Generic.DecryptSDMENCFileData"/>
    /// </summary>
    public enum SDMOption : ushort
    {
        /// <summary>
        /// Both VCUID and SDM Read Counter is considered for SDM Session key calculation.
        /// </summary>
        VCUID_RDCTR_PRESENT = 0xC0,

        /// <summary>
        /// Only VCUID is considered for SDM Session key calculation.
        /// </summary>
        VCUID_PRESENT = 0x80,

        /// <summary>
        /// Only RDCTR  is considered for SDM Session key calculation.
        /// </summary>
        RDCTR_PRESENT = 0x40
    };
    #endregion

    #region HashAlgorithm
    /// <summary>
    /// Hashing Algorithm options to be used with <see cref="Generic.ISOGeneralAuthenticate_Verify"/> interface.
    /// </summary>
    public enum HashAlgo : byte
    {
        /// <summary> None. Hashing will not be performed for the message. </summary>
        NO_HASH = 0x00,

        /// <summary> The SHA-256 hashing algorithm will be applied on Message. </summary>
        SHA256 = 0x04
    }
    #endregion HashAlgorithm
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    ///
    /// Note:
    ///     Refer below interface for initialization.
    ///         <see cref="Sw.Init">Software Initialization</see>
    /// </summary>
    [NxpRdLibAttribute
        (
            NTAG_Products.NTAG_X_DNA,
            NTAG_Products.NTAG_424_DNA,
            NTAG_Products.NTAG_424_DNA_TT,
            Message = "Products that support EV2 Authentication, EV2 SecureMessaging and EV2 Secure Dynamic Messaging. " +
                      "Products that support ECC Authentication, ECC SecureMessaging and ECC Secure Dynamic Messaging.",
            OtherInfo = "Backward compatible to Previous version of NTAG X DNA Products that supports EV2 Secure Messaging."
        )
    ]
    public abstract class Generic
    {
        #region DLL Imports
        #region Authentication and Secure Messaging
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ISOGeneralAuthenticate ( IntPtr pDataParams, byte bDeviceCert_Cached,
            byte bIsHostResponder, byte bCertRepoID, byte bCert_Depth, byte bKeySize, ushort wPriv_KeyNo,
            ushort wPriv_KeyPos, ushort wPub_KeyNo, ushort wPub_KeyPos, byte[] pExpRspLen, byte bExpRspLen,
            Certificate stHostCert, ref Certificate pDeviceCert );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ISOInternalAuthenticate ( IntPtr pDataParams, byte bPrivKeyNo, byte bCurveID,
            byte[] pPubBKey, ushort wPubBKeyLen, byte[] pOptsA, byte bOptsALen, byte[] pExpRspLen, byte bExpRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_AuthenticateEv2 ( IntPtr pDataParams, byte bFirstAuth, ushort wOption,
            ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput, byte bDivLen, byte[] pPcdCapsIn,
            byte bLenPcdCapsIn, byte[] pPcdCapsOut, byte[] pPdCapsOut );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ProcessSMApply ( IntPtr pDataParams, byte bCommMode, byte bOffset, byte bCmdCtrIncr,
            byte[] pData, byte bDataLen, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ProcessSMRemove ( IntPtr pDataParams, byte bCommMode, byte[] pData, byte bDataLen,
            ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

        #region Memory and Configuration Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_FreeMem ( IntPtr pDataParams, ref IntPtr ppMemInfo, ref ushort pMemInfoLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_SetConfiguration ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetConfiguration ( IntPtr pDataParams, byte bOption, byte bExchangeOption,
            ref IntPtr ppData, ref ushort pDataLen );
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ActivateConfiguration ( IntPtr pDataParams, byte bConfCount, byte[] pConfigList,
            byte bConfigList_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetVersion ( IntPtr pDataParams, byte bOption, ref IntPtr ppVersion,
            ref ushort pVerLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetCardUID ( IntPtr pDataParams, ref IntPtr ppUid, ref ushort pUidLen );
        #endregion

        #region Symmetric Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ChangeKey ( IntPtr pDataParams, ushort wOption, ushort wCurrKeyNo,
            ushort wCurrKeyVer, ushort wNewKeyNo, ushort wNewKeyVer, byte bKeyNoCard, byte[] pKeyPolicy,
            byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetKeySettings ( IntPtr pDataParams, byte bOption, ref IntPtr ppResponse,
            ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetKeyVersion ( IntPtr pDataParams, byte bKeyNo, ref IntPtr ppResponse,
            ref ushort pRespLen );
        #endregion

        #region ASymmetric Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ManageKeyPair ( IntPtr pDataParams, byte bComOption, byte bKeyNo,
            byte bOption, byte bCurveID, byte[] pKeyPolicy, byte bWriteAccess, uint dwKUCLimit, ushort wPrivKey_No,
            ushort wPrivKey_Pos, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ManageCARootKey ( IntPtr pDataParams, byte bComOption, byte bKeyNo,
            byte bCurveID, byte[] pAccessRights, byte bWriteAccess, ushort wPubKey_No, ushort wPubKey_Pos, byte[] pIssuer,
            byte bIssuerLen );
        #endregion

        #region Certificate Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ManageCertRepo ( IntPtr pDataParams, byte bComOption, byte bAction,
            byte bRepoID, byte[] pData, ushort wDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ReadCertRepo ( IntPtr pDataParams, byte bOption, byte bRepoID,
            byte bDataItem, ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CreateStdDataFile ( IntPtr pDataParams, byte bOption, byte bFileNo,
            byte[] pISOFileId, byte bFileOption, byte[] pAccessRights, byte[] pFileSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CreateCounterFile ( IntPtr pDataParams, byte bFileNo, byte bFileOption,
            byte[] pAccessRights, uint dwValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetFileIDs ( IntPtr pDataParams, ref IntPtr ppFileId, ref ushort pFileIdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetISOFileIDs ( IntPtr pDataParams, ref IntPtr ppISOFileId, ref ushort pISOFileIdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetFileSettings ( IntPtr pDataParams, byte bFileNo, ref IntPtr ppFSBuffer,
            ref ushort pFSBufLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetFileCounters ( IntPtr pDataParams, byte bOption, byte bFileNo, ref IntPtr ppFileCounters,
            ref ushort pFileCounterLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ChangeFileSettings ( IntPtr pDataParams, byte bOption, byte bFileNo, byte bFileOption,
            byte[] pAccessRights, byte[] pAddInfo, byte bAddInfoLen );
        #endregion

        #region Data Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ReadData ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pOffset,
            byte[] pLength, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_WriteData ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pOffset,
            byte[] pData, byte[] pLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_IncrementCounterFile ( IntPtr pDataParams, byte bOption, byte bFileNo,
            uint dwIncrValue );
        #endregion

        #region CryptoAPI
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestSHA ( IntPtr pDataParams, byte bOption, byte bOperation, byte bAlgorithm,
            byte bInputDataSource, byte bResultDst, byte[] pInputData, byte bInputDataLen, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestRNG ( IntPtr pDataParams, byte bOption, byte bNumBytes,
            byte bResultDst, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestECCSign ( IntPtr pDataParams, byte bOption, byte bOperation,
            byte bPrivateKeyID, byte bInputDataSource, byte[] pInputData, byte bInputDataLen, ref IntPtr ppResponse,
            ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestECCVerify ( IntPtr pDataParams, byte bOption, byte bOperation,
            byte bCurveID, byte[] pHostPubKey, byte bHostPubKeyLen, byte[] pSignature, byte bSigLen, byte bInputDataSource,
            byte[] pInputData, byte bInputDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestECCDH ( IntPtr pDataParams, byte bOption, byte bOperation,
            byte bKeyPairID, byte bSSDestination, byte[] pHostPubKey, byte bHostPubKeyLen, ref IntPtr ppResponse,
            ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestAES ( IntPtr pDataParams, byte bOption, byte bOperation,
            byte bPrimitive, byte bKeyID, byte bKeyLen, byte bICVSource, byte[] pICV, byte bICVLen, byte bInputDataSource,
            byte bResultDst, byte[] pInputData, byte bInputDataLen, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestAESCMAC ( IntPtr pDataParams, byte bOption, byte bOperation, byte bPrimitive,
            byte bKeyID, byte bKeyLen, byte[] pCMACSignature, byte bCMACSignLen, byte bInputDataSource, byte[] pInputData, byte bInputDataLen,
            ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestAESAEAD ( IntPtr pDataParams, byte bOption, byte bOperation, byte bPrimitive,
            byte bKeyID, byte bKeyLen, byte bNonceSource, byte[] pNonce, byte bNonceLen, ushort wTotal_AAdLen, ushort wTotal_InputDataLen,
            byte bAADSource, byte[] pAAD, byte bAADLen, byte bInputDataSource, byte[] pInputData, byte bInputDataLen, byte[] pTagData,
            byte bTagDataLen, byte bResultDst, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestWriteInternalBuffer ( IntPtr pDataParams, byte bOption, byte bDestination, byte[] pData,
            byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestHMAC ( IntPtr pDataParams, byte bOption, byte bOperation, byte bPrimitive, byte bDigestAlgo,
            byte bKeyID, byte bKeyLen, byte[] pHMac, byte bHMacLen, byte bInputDataSource, byte[] pInputData, byte bInputDataLen,
            byte bResultDst, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestHKDF ( IntPtr pDataParams, byte bOption, byte bOperation, byte bDigestAlgo, byte bKeyID,
            byte bKeyLen, byte bSaltSource, byte[] pSaltData, byte bSaltDataLen, byte bInfoSource, byte[] pInfoData,
            byte bInfoDataLen, byte bResultDst, byte bResultLen, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CryptoRequestECHO ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataLen,
            ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

        #region GPIO Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ManageGPIO ( IntPtr pDataParams, ushort wOption, byte bGPIONo,
            byte bOperation, byte[] pNFCPauseRspData, ushort wNFCPauseRspDataLen, ref IntPtr ppResponse,
            ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ReadGPIO ( IntPtr pDataParams, ushort wOption, ref IntPtr ppResponse,
            ref ushort pRspLen );
        #endregion

        #region ISO7816 - 4 Support
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_IsoSelectFile ( IntPtr pDataParams, byte bOption, byte bSelector,
            byte[] pFid, byte[] pDFname, byte bDFnameLen, byte bExtendedLenApdu, ref IntPtr ppFCI, ref ushort pFCILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_IsoReadBinary ( IntPtr pDataParams, ushort wOption, byte bOffset,
            byte bSfid, uint dwBytesToRead, byte bExtendedLenApdu, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_IsoUpdateBinary ( IntPtr pDataParams, byte bOffset, byte bSfid,
            byte bExtendedLenApdu, byte[] pData, ushort wDataLen );
        #endregion

        #region Utility
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_ResetAuthentication ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_CalculateMACSDM ( IntPtr pDataParams, byte bSdmOption, ushort wKeyNo_SDMMac,
            ushort wKeyVer_SDMMac, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pInData, ushort wInDataLen,
            ref IntPtr ppSDMMAC, ref ushort pSDMMACLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_VerifySDMSignature ( IntPtr pDataParams, ushort wPubKeyNo_SDMSig,
            ushort wPubKeyPos_SDMSig, byte[] pInData, ushort wInDataLen, byte[] pSignature, ushort pSigLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_DecryptSDMENCFileData ( IntPtr pDataParams, byte bSdmOption,
            ushort wKeyNo_SDMEnc, ushort wKeyVer_SDMEnc, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pEncData,
            ushort wEncDataLen, ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_DecryptSDMPICCData ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVer,
            byte[] pEncData, ushort wEncDataLen, ref IntPtr ppPlainData, ref ushort pPlainDataLen );
        #endregion
        #endregion

        #region Properties
        /// <summary>
        /// Get, Sets the Application Identifier.
        /// </summary>
        public abstract byte[] Aid { get; set; }

        /// <summary>
        /// Get, Sets the Session Encryption Key.
        /// </summary>
        public abstract byte[] SesAuthENCKey { get; set; }

        /// <summary>
        /// Get, Sets the Session MAC Key.
        /// </summary>
        public abstract byte[] SesAuthMACKey { get; set; }

        /// <summary>
        /// Get, Sets the Transaction Identifier.
        /// </summary>
        public abstract byte[] TI { get; set; }

        /// <summary>
        /// Get, Sets the Command Counter.
        /// </summary>
        public abstract ushort CmdCtr { get; set; }
        #endregion

        #region Wrapper Functions
        #region Authentication and Secure Messaging
        /// <summary>
        /// Performs ASymmetric Mutual SIGMA-I authentication. This interfaces performs Cmd.ISOGeneralAuthenticate with
        /// Host acts as Initiator or Responder mode based on the information provided \b bIsHostResponder parameter.
        ///     - During ISOGeneralAuthenticate, Ephemeral Key Pair is generated by the Library.
        ///     - Device End-Leaf Certificate hash and Signature are verified by the Library internally.
        ///     - Exchange of Abort Tag is not supported by the library.
        ///     - Interface provides received Device Certificates.The certificate validation is not carried by Library
        ///       and shall be taken up by the user.
        ///       - If \b bDeviceCert_Cached == <see cref="Value.ON"/>, means user has to provide Device End-Leaf certificate only.
        ///         This will be used for Certificate Hash and Signature Verification received from Device
        /// </summary>
        ///
        /// <param name="bDeviceCert_Cached">Whether or not the device certificates are cached.
        ///                                  If cached, user need to provide End-Leaf certificate for Certificate Hash
        ///                                  and Signature verification.
        ///                                     - <see cref="Value.OFF">Device Certificate(s) are not cached</see>
        ///                                     - <see cref="Value.OFF">Device Certificate(s) are cached</see>
        /// </param>
        /// <param name="bIsHostResponder">Flag to indicate whether Host is Initiator or responder. Will be one of the following values
        ///                                 - <see cref="Value.OFF">Host as Initiator</see>
        ///                                 - <see cref="Value.ON">Host as Responder</see>
        /// </param>
        /// <param name="bCertRepoID">Certificate repository Id to use to execute the protocol. Supported values are 0x00 - 0x07</param>
        /// <param name="bCert_Depth">Certificate Depth to exchange. Will be one of the following
        ///                             - <see cref="Certificate_Depth.END_LEAF">End-Leaf</see>
        ///                             - <see cref="Certificate_Depth.PARENT">Parent</see>
        ///                             - <see cref="Certificate_Depth.GRAND_PARENT">Grand-Parent</see>
        ///
        ///                             - <see cref="Certificate_Depth.END_LEAF">End-Leaf</see> | <see cref="Certificate_Depth.PARENT">Parent</see>
        ///                             - <see cref="Certificate_Depth.END_LEAF">End-Leaf</see> | <see cref="Certificate_Depth.PARENT">Parent</see> |
        ///                               <see cref="Certificate_Depth.GRAND_PARENT">Grand-Parent</see>
        /// </param>
        /// <param name="bCert_CompressionType">Type of Certificate compression. Will be one of the following
        ///                                         - <see cref="Certificate_CompressionType.UN_COMPRESSED">UnCompressed Certificate</see>
        ///                                         - <see cref="Certificate_CompressionType.COMPRESSED">Compressed Certificate</see>
        /// </param>
        /// <param name="bKeySize">Type of Session Key Size to use. Will be one of the following
        ///                         - <see cref="Session_KeySize.AES128">Session KeySize as AES128</see>
        ///                         - <see cref="Session_KeySize.AES256">Session KeySize as AES256</see>
        /// </param>
        /// <param name="wPriv_KeyNo">Key number in KeyStore to use, known as Initiator's / Responder's Private Key.
        ///                           This will be used to sign the message that will be exchanged, basically sk_init if Host is
        ///                           Initiator or sk_resp if Host is responder.
        ///                           Will be used for one of the following ECC Signature generation
        ///                             - Init_ECC_Sig = ECDSA-Sign ( sk_init , 0x02 || keySize_init || keySize_resp || yP ||
        ///                               xP || AES_CMAC(K_m1, 0x02|| leaf_cert_hash))
        ///                             - Resp_ECC_Sig = ECDSA-Sign ( sk_resp, 0x01 || keySize_init || keySize_resp || xP || yP ||
        ///                               AES-CMAC(K_m1, 0x01 || leaf_cert _hash ))
        /// </param>
        /// <param name="wPriv_KeyPos">Key position in KeyStore to use. The position where \b wPriv_KeyNo is stored.</param>
        /// <param name="wPub_KeyNo">Key number in KeyStore to use, known as Initiator's / Responder's Public Key.
        ///                          This will be used to verify the signature that was received, basically Init_ECC_Sig if
        ///                          Host is Responder or Resp_ECC_Sig if Host is Initiator.
        /// </param>
        /// <param name="wPub_KeyPos">Key position in KeyStore to use. The position where \b wPub_KeyNo is stored.</param>
        /// <param name="aExpRspLen">Length of expected response from Device.
        ///                             - This parameter is for exchanging the LE information.
        ///                             - If NULL is provided, then the expected Response length will be taken
        ///                               as 0x00 (1 byte) by default.
        ///                             - Possible values are NULL, Array consisting of 1 byte, 2 bytes or 3 bytes.
        /// </param>
        /// <param name="aHostCert_EndLeaf">The DER formated compressed or uncompressed Host End-Leaf Certificate. Should not include
        ///                                 7F 21 (UnCompressed) or 7F 22 (Compressed) notation mentioned in DataSheet and the Certificate
        ///                                 Length in BER-TLV format. The Compression and Certificate Length will be appended by this
        ///                                 method.
        /// </param>
        /// <param name="aHostCert_Parent">The DER formated compressed or uncompressed Host Parent Certificate. Should not include
        ///                                7F 21 (UnCompressed) or 7F 22 (Compressed) notation mentioned in DataSheet and the Certificate
        ///                                Length in BER-TLV format. The Compression and Certificate Length will be appended by this
        ///                                method.
        /// </param>
        /// <param name="aHostCert_GrandParent">The DER formated compressed or uncompressed Host Grand-Parent Certificate. Should not
        ///                                     include 7F 21 (UnCompressed) or 7F 22 (Compressed) notation mentioned in DataSheet and
        ///                                     the Certificate Length in BER-TLV format. The Compression and Certificate Length will
        ///                                     be appended by this method.
        /// </param>
        /// <param name="aDeviceCert_EndLeaf">The DER formated compressed or uncompressed Device End-Leaf Certificate. Will contain
        ///                                   only the Certificate data without 7F 21 (UnCompressed) or 7F 22 (Compressed) notation
        ///                                   mentioned in DataSheet and the Certificate length in BER-TLV.
        ///                                   Based on \b bDeviceCert_Cached, One of the following should be followed,
        ///                                     - [In] Provide End-Leaf Certificate if \b bDeviceCert_Cached == <see cref="Value.ON"/>
        ///                                     - [Out] End-Leaf Certificate will be returned if \b bDeviceCert_Cached == <see cref="Value.OFF"/>
        /// </param>
        /// <param name="aDeviceCert_Parent">The DER formated compressed or uncompressed Device Parent Certificate. Will contain
        ///                                  only the Certificate data without 7F 21 (UnCompressed) or 7F 22 (Compressed) notation
        ///                                  mentioned in DataSheet and the Certificate length in BER-TLV.
        /// </param>
        /// <param name="aDeviceCert_GrandParent">The DER formated compressed or uncompressed Device Grand-Parent Certificate.
        ///                                       Will contain only the Certificate data without 7F 21 (UnCompressed) or
        ///                                       7F 22 (Compressed) notation mentioned in DataSheet and the Certificate
        ///                                       length in BER-TLV.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///         Parameter Values not supported for Authentication.
        ///         If \b bDeviceCert_Cached == <see cref="Value.ON"/> and Device End-Leaf Certificate is not
        ///         available. i.e aDeviceCert_EndLeaf = null
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If information received is not proper.
        ///     Returns <see cref="Error_Comm.ABORTED"/>
        ///         If Device aborts the authentication.
        ///     Returns <see cref="Error.DEVICE_SIGNATURE_ERROR"/>
        ///         Device Signature Verification failure.
        ///     Returns <see cref="Error.DEVICE_LEAF_CERTIFICATE_HASH_ERROR"/>
        ///         Device End-Leaf Certificate Hash Verification failure.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Performs ASymmetric Mutual SIGMA-I authentication.",
                OtherInfo = "Performs both modes, Host as initiator or Host as Responder"
            )
        ]
        public Status_t ISOGeneralAuthenticate ( byte bDeviceCert_Cached, byte bIsHostResponder, byte bCertRepoID, byte bCert_Depth,
            byte bCert_CompressionType, byte bKeySize, ushort wPriv_KeyNo, ushort wPriv_KeyPos, ushort wPub_KeyNo, ushort wPub_KeyPos,
            byte[] aExpRspLen, byte[] aHostCert_EndLeaf, byte[] aHostCert_Parent, byte[] aHostCert_GrandParent,
            ref byte[] aDeviceCert_EndLeaf, out byte[] aDeviceCert_Parent, out byte[] aDeviceCert_GrandParent )
        {
            Certificate stHostCert = new Certificate ();
            Certificate stDeviceCert = new Certificate ();

            if ( aHostCert_EndLeaf != null )
            {
                aHostCert_EndLeaf = EncodeCertificate ( bCert_CompressionType, aHostCert_EndLeaf );
                stHostCert.pEndLeaf = GCHandle.Alloc ( aHostCert_EndLeaf, GCHandleType.Pinned ).AddrOfPinnedObject ();
                stHostCert.wEndLeaf_Len = ( ushort ) aHostCert_EndLeaf.Length;
            }

            if ( aHostCert_Parent != null )
            {
                aHostCert_Parent = EncodeCertificate ( bCert_CompressionType, aHostCert_Parent );
                stHostCert.pParent = GCHandle.Alloc ( aHostCert_Parent, GCHandleType.Pinned ).AddrOfPinnedObject ();
                stHostCert.wParent_Len = ( ushort ) aHostCert_Parent.Length;
            }

            if ( aHostCert_GrandParent != null )
            {
                aHostCert_GrandParent = EncodeCertificate ( bCert_CompressionType, aHostCert_GrandParent );
                stHostCert.pGrandParent = GCHandle.Alloc ( aHostCert_GrandParent, GCHandleType.Pinned ).AddrOfPinnedObject ();
                stHostCert.wGrandParent_Len = ( ushort ) aHostCert_GrandParent.Length;
            }

            byte[] aDeviceCert_EndLeaf_Tmp = ( ( bDeviceCert_Cached == 0x00 ) || ( aDeviceCert_EndLeaf == null ) ) ?
                new byte[2048] : ( byte[] ) aDeviceCert_EndLeaf.Clone ();
            stDeviceCert.pEndLeaf = GCHandle.Alloc ( aDeviceCert_EndLeaf_Tmp, GCHandleType.Pinned ).AddrOfPinnedObject ();
            stDeviceCert.wEndLeaf_Len = ( ushort ) ( ( ( bDeviceCert_Cached == 0x00 ) || ( aDeviceCert_EndLeaf == null ) ) ?
                0 : aDeviceCert_EndLeaf_Tmp.Length );

            aDeviceCert_Parent = null;
            byte[] aDeviceCert_Parent_Tmp = new byte[2048];
            stDeviceCert.pParent = GCHandle.Alloc ( aDeviceCert_Parent_Tmp, GCHandleType.Pinned ).AddrOfPinnedObject ();
            stDeviceCert.wParent_Len = 0;

            aDeviceCert_GrandParent = null;
            byte[] aDeviceCert_GrandParent_Tmp = new byte[2048];
            stDeviceCert.pGrandParent = GCHandle.Alloc ( aDeviceCert_GrandParent_Tmp, GCHandleType.Pinned ).AddrOfPinnedObject ();
            stDeviceCert.wGrandParent_Len = 0;

            Status_t oStatus = phalNtagXDna_ISOGeneralAuthenticate ( m_pDataParams, bDeviceCert_Cached, bIsHostResponder, bCertRepoID,
                bCert_Depth, bKeySize, wPriv_KeyNo, wPriv_KeyPos, wPub_KeyNo, wPub_KeyPos, aExpRspLen,
                ( byte ) ( ( aExpRspLen == null ) ? 0 : aExpRspLen.Length ), stHostCert, ref stDeviceCert );

            aDeviceCert_EndLeaf = MarshalCopy ( oStatus, stDeviceCert.pEndLeaf, stDeviceCert.wEndLeaf_Len );
            aDeviceCert_Parent = MarshalCopy ( oStatus, stDeviceCert.pParent, stDeviceCert.wParent_Len );
            aDeviceCert_GrandParent = MarshalCopy ( oStatus, stDeviceCert.pGrandParent, stDeviceCert.wGrandParent_Len );

            /* Free Memory of Host Certificates */
            if ( stHostCert.pEndLeaf != IntPtr.Zero )
            {
                stHostCert.pEndLeaf = IntPtr.Zero;
                stHostCert.wEndLeaf_Len = 0;
            }
            if ( stHostCert.pParent != IntPtr.Zero )
            {
                stHostCert.pEndLeaf = IntPtr.Zero;
                stHostCert.wEndLeaf_Len = 0;
            }
            if ( stHostCert.pEndLeaf != IntPtr.Zero )
            {
                stHostCert.pEndLeaf = IntPtr.Zero;
                stHostCert.wEndLeaf_Len = 0;
            }

            /* Free Memory of Device Certificates */
            if ( stDeviceCert.pEndLeaf != IntPtr.Zero )
            {
                stDeviceCert.pEndLeaf = IntPtr.Zero;
                stDeviceCert.wEndLeaf_Len = 0;
            }
            if ( stDeviceCert.pParent != IntPtr.Zero )
            {
                stDeviceCert.pParent = IntPtr.Zero;
                stDeviceCert.wParent_Len = 0;
            }
            if ( stDeviceCert.pGrandParent != IntPtr.Zero )
            {
                stDeviceCert.pGrandParent = IntPtr.Zero;
                stDeviceCert.wGrandParent_Len = 0;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs Asymmetric Card-Unilateral Authentication.
        /// </summary>
        ///
        /// <param name="bPrivKeyNo">Private Key number for signing the response.
        ///                             At PICC level, two keys are supported.
        ///                             At application level, up to five keys are supported.
        /// </param>
        /// <param name="bCurveID">The targeted curve for the public key provided in \b aPubBKey parameter.
        ///                        Should be one of the below values.
        ///                         - <see cref="TargetCurve.NIST_P256"/>
        ///                         - <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aPubBKey">Public Key (Pub.B) to be used for verification.</param>
        /// <param name="aOptsA">Complete PCD Options in TLV format.</param>
        /// <param name="aExpRspLen">Length of expected response from Device.
        ///                             - This parameter is for exchanging the LE information.
        ///                             - If NULL is provided, then the expected Response length will be taken
        ///                               as 0x00 (1 byte) by default.
        ///                             - Possible values are NULL, Array consisting of 1 byte, 2 bytes or 3 bytes.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If Tag information is not proper for AuthDOHdr, RndB and Signature.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/>
        ///         If Verification of Message / Signature combination failed.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Performs Asymmetric Card-Unilateral Authentication."
            )
        ]
        public Status_t ISOInternalAuthenticate ( byte bPrivKeyNo, byte bCurveID, byte[] aPubBKey, byte[] aOptsA,
            byte[] aExpRspLen )
        {
            return phalNtagXDna_ISOInternalAuthenticate ( m_pDataParams, bPrivKeyNo, bCurveID, aPubBKey,
                ( byte ) ( ( aPubBKey == null ) ? 0 : aPubBKey.Length ), aOptsA, ( byte ) ( ( aOptsA == null ) ? 0 :
                aOptsA.Length ), aExpRspLen, ( byte ) ( ( aExpRspLen == null ) ? 0 : aExpRspLen.Length ) );
        }

        /// <summary>
        /// Performs an EV2 First or Non First Authentication depending upon bFirstAuth Parameter. This will
        /// be using the AES128 keys and will generate and verify the contents based on generic AES algorithm.
        /// </summary>
        ///
        /// <param name="bFirstAuth">One of the below options.
        ///                          <list type="bullet">
        ///                             <item><see cref="AuthType.EV2_FIRST"/></item>
        ///                             <item><see cref="AuthType.EV2_NON_FIRST"/></item>
        ///                          </list>
        /// </param>
        /// <param name="wOption">One of the below options.
        ///                         <see cref="DivOption_Auth.NO_DIVERSIFICATION"/>
        ///                         <see cref="DivOption_Auth.ENCR"/>
        ///                         <see cref="DivOption_Auth.CMAC"/>
        /// </param>
        /// <param name="wKeyNo">Key number in KeyStore of software.</param>
        /// <param name="wKeyVer">Key version in the key store of software.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="TargetApp.SECONDARY"/> to indicate
        ///                          secondary application indicator.</param>
        /// <param name="aDivInput">Diversification input. Can be NULL.</param>
        /// <param name="aPcdCapsIn">PCD Capabilities. Upto 6 bytes.</param>
        /// <param name="aPcdCapsOut">PCD Capabilities.
        ///                           <list type="bullet">
        ///                             <item>If First_Auth    : Information will be available and buffer length will be 6 bytes long.</item>
        ///                             <item>If Non_First_Auth: Information will not be available and buffer will be null.</item>
        ///                           </list>
        /// </param>
        /// <param name="aPdCapsOut">PD Capabilities.
        ///                          <list type="bullet">
        ///                             <item>If First_Auth    : Information will be available and buffer length will be 6 bytes long.</item>
        ///                             <item>If Non_First_Auth: Information will not be available and buffer will be null.</item>
        ///                          </list>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///         If Invalid AuthType (bFirstAuth), Invalid Diversification Options (wOption).
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> If KeyType is not of AES128.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> Status If Response length is not of AES128 Block Size (Single or multiple).
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If Received RndA do not matches.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> If Tag information is not proper for AuthDOHdr and E.Pub.B.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "AES128 and AES256 key should be used for Authentication. After Authentication, EV2 Secure Messaging is used.",
                Message = @"The Secure Messaging depends on FirstAuth parameter.\n
                          EV2_FIRST: The Authentication is intended to be first in transaction.\n
                          EV2_NON_FIRST: The Authentication is intended for any subsequent authentication after Authenticate First."
            )
        ]
        public Status_t AuthenticateEv2 ( byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard,
            byte[] aDivInput, byte[] aPcdCapsIn, out byte[] aPcdCapsOut, out byte[] aPdCapsOut )
        {
            aPcdCapsOut = new byte[6];
            aPdCapsOut = new byte[6];

            Status_t oStatus = phalNtagXDna_AuthenticateEv2 ( m_pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPcdCapsIn,
                ( byte ) ( ( aPcdCapsIn == null ) ? 0 : aPcdCapsIn.Length ),
                aPcdCapsOut, aPdCapsOut );

            if ( !oStatus.Equals ( new Status_t () ) || bFirstAuth.Equals ( ( byte ) AuthType.EV2_NON_FIRST ) )
            {
                aPcdCapsOut = null;
                aPdCapsOut = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs ProcessSM Apply.
        /// The ProcessSM_Apply is used for applying secure messaging to a command before sending it to the target
        /// device.
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the protection mode to be applied.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.PLAIN_20"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOffset">Index of the first byte of CmdData in Data field.
        ///                       Only applicable when \b bCommMode is <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bCmdCtrIncr">Command counter increment value. Only applicable when \b bCommMode is
        ///                             - <see cref="CommOption.PLAIN"/>
        ///                             - <see cref="CommOption.PLAIN_20"/>
        /// </param>
        /// <param name="aData">Plain data to protect. One of the following
        ///                         - NULL if \b bCommMode is
        ///                             - <see cref="CommOption.PLAIN"/>
        ///                             - <see cref="CommOption.PLAIN_20"/>
        ///                         - The plain data to be protected if \b bCommMode is
        ///                             - <see cref="CommOption.MAC"/>
        ///                             - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aResponse">Buffer containing the following,
        ///                             - NULL if \b bCommMode is <see cref="CommOption.PLAIN"/> or <see cref="CommOption.PLAIN_20"/>
        ///                             - 8 bytes MAC if \b bCommMode is <see cref="CommOption.MAC"/>
        ///                             - Encrypted data + MAC if \b bCommMode is <see cref="CommOption.FULL"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform application of Protection mode to Plain data",
                OtherInfo = "The ProcessSM_Apply is used for applying secure messaging to a command before sending " +
                            "it to the target device."
            )
        ]
        public Status_t ProcessSM_Apply ( byte bCommMode, byte bOffset, byte bCmdCtrIncr, byte[] aData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ProcessSMApply ( m_pDataParams, bCommMode, bOffset, bCmdCtrIncr,
                aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Performs ProcessSM Remove.
        /// The ProcessSM_Remove is used for removing and validating secure messaging from a response received from
        /// a target device.
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the protection mode to be applied.
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aData">Data from which protection to be removed. One of the following
        ///                         - If \b bCommMode is \ref PHAL_NTAGXDNA_COMMUNICATION_MAC "MAC"
        ///                           then, input data will be RC[|| RespData] || MAC
        ///                         - If \b bCommMode is \ref PHAL_NTAGXDNA_COMMUNICATION_FULL "FULL"
        ///                           then, input data will be RC[|| Encrypted RespData] || MAC
        /// </param>
        /// <param name="aResponse">Buffer containing the following,
        ///                             - If \b bCommMode is \ref PHAL_NTAGXDNA_COMMUNICATION_MAC "MAC"
        ///                               then, there will be no response.
        ///                             - If \b bCommMode is \ref PHAL_NTAGXDNA_COMMUNICATION_FULL "FULL"
        ///                               then, Plain data will be returned.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform verification and removal of Protection from data",
                OtherInfo = "The ProcessSM_Remove is used for removing and validating secure messaging from a response " +
                            "received from a target device."
            )
        ]
        public Status_t ProcessSM_Remove ( byte bCommMode, byte[] aData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ProcessSMRemove ( m_pDataParams, bCommMode, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region Memory and Configuration Management
        /// <summary>
        /// Returns free memory available on the tag
        /// </summary>
        ///
        /// <param name="aMemInfo">Current free memory available. Response Will be of 3 bytes with LSB first.
        ///                        If the free memory available is 7592 bytes, then 7592 in Hex will be 0x001F10
        ///                        aMemInfo will contain 10 1F 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = @"Response Will be of 3 bytes with LSB first. Ex: If the free memory available is 7592 bytes, then
                              7592 in Hex will be 0x001F10. The output buffer will have 10 1F 00.",
                Message = "Returns the free memory available on the card."
            )
        ]
        public Status_t FreeMem ( out byte[] aMemInfo )
        {
            Status_t oStatus;
            IntPtr ppMemInfo = IntPtr.Zero;
            ushort wMemInfoLen = 0;

            oStatus = phalNtagXDna_FreeMem ( m_pDataParams, ref ppMemInfo, ref wMemInfoLen );
            aMemInfo = MarshalCopy ( oStatus, ppMemInfo, wMemInfoLen );

            return oStatus;
        }

        /// <summary>
        /// Performs SetConfiguration to configures several aspects of the card or the application.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option to configure. One of the below values.
        ///                         <see cref="SetConfig.PICC_CONFIGURATION"/>
        ///                         <see cref="SetConfig.ATS_UPDATE"/>
        ///                         <see cref="SetConfig.SAK_UPDATE"/>
        ///                         <see cref="SetConfig.SM_CONFIGURATION"/>
        ///                         <see cref="SetConfig.CAPABILITY_DATA"/>
        ///                         <see cref="SetConfig.ATQA_UPDATE"/>
        ///                         <see cref="SetConfig.SILENT_MODE_CONFIGURATION"/>
        ///                         <see cref="SetConfig.ENHANCED_PRIVACY_CONFIGURATION"/>
        ///                         <see cref="SetConfig.NFC_MANAGEMENT"/>
        ///                         <see cref="SetConfig.I2C_MANAGEMENT"/>
        ///                         <see cref="SetConfig.GPIO_MANAGEMENT"/>
        ///                         <see cref="SetConfig.ECC_KEY_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CERTIFICATE_MANAGEMENT"/>
        ///                         <see cref="SetConfig.WATCHDOG_TIMER_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CRYPTO_API_MANAGEMENT"/>
        ///                         <see cref="SetConfig.AUTH_COUNTR_LIMIT_CONFIGURATION"/>
        ///                         <see cref="SetConfig.HALT_WAKEUP_CONFIGURATION"/>
        ///                         <see cref="SetConfig.DEFERRED_CONFIGURATION"/>
        ///                         <see cref="SetConfig.LOCK_CONFIGURATION"/>
        /// </param>
        /// <param name="aData">Data for the option specified.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Performs SetConfiguration to configures several aspects of the card or the application."
            )
        ]
        public Status_t SetConfiguration ( byte bOption, byte[] aData )
        {
            byte[] aDataTmp = new byte[aData.Length];
            aData.CopyTo ( aDataTmp, 0 );

            return phalNtagXDna_SetConfiguration ( m_pDataParams, bOption, aDataTmp, ( byte ) aDataTmp.Length );
        }

        /// <summary>
        /// Performs GetConfiguration to retrieve configuration aspects of the card or the application.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option to configure. One of the below values.
        ///                         <see cref="SetConfig.PICC_CONFIGURATION"/>
        ///                         <see cref="SetConfig.ATS_UPDATE"/>
        ///                         <see cref="SetConfig.SAK_UPDATE"/>
        ///                         <see cref="SetConfig.SM_CONFIGURATION"/>
        ///                         <see cref="SetConfig.CAPABILITY_DATA"/>
        ///                         <see cref="SetConfig.ATQA_UPDATE"/>
        ///                         <see cref="SetConfig.SILENT_MODE_CONFIGURATION"/>
        ///                         <see cref="SetConfig.ENHANCED_PRIVACY_CONFIGURATION"/>
        ///                         <see cref="SetConfig.NFC_MANAGEMENT"/>
        ///                         <see cref="SetConfig.I2C_MANAGEMENT"/>
        ///                         <see cref="SetConfig.GPIO_MANAGEMENT"/>
        ///                         <see cref="SetConfig.ECC_KEY_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CERTIFICATE_MANAGEMENT"/>
        ///                         <see cref="SetConfig.WATCHDOG_TIMER_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CRYPTO_API_MANAGEMENT"/>
        ///                         <see cref="SetConfig.AUTH_COUNTR_LIMIT_CONFIGURATION"/>
        ///                         <see cref="SetConfig.HALT_WAKEUP_CONFIGURATION"/>
        ///                         <see cref="SetConfig.DEFERRED_CONFIGURATION"/>
        ///                         <see cref="SetConfig.LOCK_CONFIGURATION"/>
        /// </param>
        /// <param name="bExchangeOption">Whether or not to exchange Option (\b bOption) information to tag.
        ///                                 <see cref="Value.OFF"/>: Do not exchange Option information to tag.
        ///                                 <see cref="Value.ON"/> : Exchange Option information to tag.
        /// </param>
        /// <param name="aData">One of the following information.
        ///                         If bExchangeOption = <see cref="Value.OFF"/> then, Manufacturer Data. will be returned.
        ///                         If bExchangeOption = <see cref="Value.ON"/> then, information based on the Option data will be returned.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For Invalid ExchangeOption (bExchangeOption) value.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Performs GetConfiguration to retrieve configuration aspects of the card or the application."
            )
        ]
        public Status_t GetConfiguration ( byte bOption, byte bExchangeOption, out byte[] aData )
        {
            Status_t oStatus;
            IntPtr ppData = IntPtr.Zero;
            ushort wDataLen = 0;

            oStatus = phalNtagXDna_GetConfiguration ( m_pDataParams, bOption, bExchangeOption, ref ppData, ref wDataLen );
            aData = MarshalCopy ( oStatus, ppData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// Performs ActivateConfiguration to abolish a deferred configuration.
        /// ActivateConfiguration provides the list of deferred configurations for which the deferral can be ended.
        /// </summary>
        ///
        /// <param name="bConfCount">Number Of Configurations to be Activated (N)</param>
        /// <param name="aConfigList">List of configurations to be activated (with size N * 2).
        ///                           List must hold one or more of following values.
        ///                             - Activate SetConfiguration 0x01 (RandomID)
        ///                             - Activate SetConfiguration 0x0D (Silent Mode)
        ///                             - Activate SetConfiguration 0x11 (TagTamper boot measurements)
        ///                             - Activate ChangeFileSettings SDM encryptions
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>if the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Performs ActivateConfiguration to abolish a deferred configuration."
            )
        ]
        public Status_t ActivateConfiguration ( byte bConfCount, byte[] aConfigList )
        {
            byte[] aConfigList_Tmp = new byte[aConfigList.Length];
            aConfigList.CopyTo ( aConfigList_Tmp, 0 );

            return phalNtagXDna_ActivateConfiguration ( m_pDataParams, bConfCount, aConfigList_Tmp,
                ( byte ) aConfigList_Tmp.Length );
        }

        /// <summary>
        /// Returns manufacturing related data of the tag.
        /// </summary>
        ///
        /// <param name="aVerInfo">Returns the complete version information of the tag. The information includes
        ///                         - Hardware Information
        ///                         - Software Information
        ///                         - Production Related Information
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = @"Response Information includes Hardware, Software and Production Related Information.",
                Message = "Returns manufacturing related data of the tag."
            )
        ]
        public Status_t GetVersion ( out byte[] aVerInfo )
        {
            Status_t oStatus;
            IntPtr ppVersion = IntPtr.Zero;
            ushort wVerLen = 0;

            oStatus = phalNtagXDna_GetVersion ( m_pDataParams, ( byte ) FabID.DO_NOT_RETURN, ref ppVersion, ref wVerLen );
            aVerInfo = MarshalCopy ( oStatus, ppVersion, wVerLen );

            return oStatus;
        }

        /// <summary>
        /// Returns manufacturing related data of the tag. Also returns the FabID based on the information provided
        /// to bOption parameter.
        /// </summary>
        ///
        /// <param name="bOption">Option information to be exchanged. Will be one of the following,
        ///                         - <see cref="FabID.DO_NOT_RETURN"/>
        ///                         - <see cref="FabID.RETURN"/>
        /// </param>
        /// <param name="aVerInfo">Returns the complete version information of the tag. The information includes
        ///                         - Hardware Information
        ///                         - Software Information
        ///                         - Production Related Information
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = "Response Information includes Hardware, Software and Production Related Information.\n" +
                             "Response includes FabID based on bOption information",
                Message = "Returns manufacturing related data of the tag."
            )
        ]
        public Status_t GetVersion ( byte bOption, out byte[] aVerInfo )
        {
            Status_t oStatus;
            IntPtr ppVersion = IntPtr.Zero;
            ushort wVerLen = 0;

            oStatus = phalNtagXDna_GetVersion ( m_pDataParams, bOption, ref ppVersion, ref wVerLen );
            aVerInfo = MarshalCopy ( oStatus, ppVersion, wVerLen );

            return oStatus;
        }

        /// <summary>
        /// Returns the Unique Identifier of the tag.
        /// </summary>
        ///
        /// <param name="aUID">Returns the complete UID information of the tag.</param>
        /// <param name="boOnlyUID">When true, return only UID else return complete information received from tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Returns the Unique Identifier of the tag."
            )
        ]
        public Status_t GetCardUID ( out byte[] aUID, bool boOnlyUID = true )
        {
            Status_t oStatus;
            IntPtr ppUid = IntPtr.Zero;
            ushort wUidLen = 0;

            oStatus = phalNtagXDna_GetCardUID ( m_pDataParams, ref ppUid, ref wUidLen );
            aUID = MarshalCopy ( oStatus, ppUid, wUidLen );

            /* Extract UID */
            if ( boOnlyUID && ( aUID != null ) )
            {
                /* Remove UID Header */
                if ( aUID.Length != 0x07 )
                {
                    Array.Copy ( aUID, 2, aUID, 0, aUID.Length - 2 );
                    Array.Resize ( ref aUID, aUID.Length - 2 );
                }
            }

            return oStatus;
        }
        #endregion

        #region Symmetric Key Management
        /// <summary>
        /// Depending on the currently selected AID, this command update a key of the tag or
        /// of an application AKS ( Active Key Set).
        /// </summary>
        ///
        /// <param name="wOption">Option to be used for diversifying the Current and New key.
        ///                         - <see cref="DivOption_ChangeKey.NO_DIVERSIFICATION"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        /// </param>
        /// <param name="wCurrKeyNo">Current key number in KeyStore of software.</param>
        /// <param name="wCurrKeyVer">Current key version in KeyStore of software.</param>
        /// <param name="wNewKeyNo">New key number in KeyStore of software.</param>
        /// <param name="wNewKeyVer">New key version in KeyStore of software.</param>
        /// <param name="bKeyNoCard">Key number of the key to be changed. To be ORed with
        ///                             - Bit[7 - 6]:
        ///                                 - With the type of the key if targeting AppMasterKey or CryptoRequestKey,
        ///                                     - 10     : KeyType.AES128
        ///                                     - 11     : KeyType.AES256
        ///                                 - RFU otherwise
        ///                             - Bit[5 - 0]: Application Master or CryptoRequest Key.
        ///                                 - 0x00 - 0x04: Application Keys
        ///                                 - 0x10 - 0x17: CryptoRequest Keys
        /// </param>
        /// <param name="aDivInput">Diversification input to be used for diversifying the key. Can be NULL.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For invalid card key numbers (bKeyNoCard) are tag or Application level
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If command is called without prior authentication.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Depending on the currently selected AID, this command updates a key of the application AKS (Active Key Set) or CryptoRequest Keys."
            )
        ]
        public Status_t ChangeKey ( ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] aDivInput )
        {
            return phalNtagXDna_ChangeKey ( m_pDataParams, wOption, wCurrKeyNo, wCurrKeyVer, wNewKeyNo, wNewKeyVer,
                bKeyNoCard, null, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Depending on the currently selected AID, this command update a key of the tag or
        /// of an application AKS ( Active Key Set).
        /// </summary>
        ///
        /// <param name="wOption">Option to be used for diversifying the Current and New key.
        ///                         - <see cref="DivOption_ChangeKey.NO_DIVERSIFICATION"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        /// </param>
        /// <param name="wCurrKeyNo">Current key number in KeyStore of software.</param>
        /// <param name="wCurrKeyVer">Current key version in KeyStore of software.</param>
        /// <param name="wNewKeyNo">New key number in KeyStore of software.</param>
        /// <param name="wNewKeyVer">New key version in KeyStore of software.</param>
        /// <param name="bKeyNoCard">Key number of the key to be changed. To be ORed with
        ///                             - Bit[7 - 6]:
        ///                                 - With the type of the key if targeting AppMasterKey or CryptoRequestKey,
        ///                                     - 10     : KeyType.AES128
        ///                                     - 11     : KeyType.AES256
        ///                                 - RFU otherwise
        ///                             - Bit[5 - 0]: Application Master or CryptoRequest Key.
        ///                                 - 0x00 - 0x04: Application Keys
        ///                                 - 0x10 - 0x17: CryptoRequest Keys
        /// </param>
        /// <param name="aKeyPolicy">Defines the allowed crypto operations with the targeted key.
        ///                          Optional present if targeting CryptoRequest Key.
        ///                          <see cref="KeyPolicy_Sym.DISABLED"/>
        ///                          <see cref="KeyPolicy_Sym.MAC_SIGN"/>
        ///                          <see cref="KeyPolicy_Sym.MAC_VERIFY"/>
        ///                          <see cref="KeyPolicy_Sym.ECB_CBC_ENCRYPT"/>
        ///                          <see cref="KeyPolicy_Sym.ECB_CBC_DECRYPT"/>
        ///                          <see cref="KeyPolicy_Sym.CCM_GCM_ENCRYPT_SIGN"/>
        ///                          <see cref="KeyPolicy_Sym.CCM_GCM_ENCRYPT_SIGN_INTERNAL_NONCE"/>
        ///                          <see cref="KeyPolicy_Sym.CCM_GCM_DECRYPT_VERIFY"/>
        ///                          <see cref="KeyPolicy_Sym.HMAC"/>
        ///                          <see cref="KeyPolicy_Sym.HKDF"/>
        /// </param>
        /// <param name="aDivInput">Diversification input to be used for diversifying the key. Can be NULL.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For invalid card key numbers (bKeyNoCard) are tag or Application level
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If command is called without prior authentication.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Depending on the currently selected AID, this command updates a key of the application AKS (Active Key Set) or CryptoRequest Keys."
            )
        ]
        public Status_t ChangeKey ( ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] aKeyPolicy, byte[] aDivInput )
        {
            return phalNtagXDna_ChangeKey ( m_pDataParams, wOption, wCurrKeyNo, wCurrKeyVer, wNewKeyNo, wNewKeyVer,
                bKeyNoCard, aKeyPolicy, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Gets Application Key Setting for the application. In addition it returns the number of keys which are
        /// configured for the selected application and if applicable the AppKeySetSettings.
        ///
        /// Note:
        ///     The Option information will not be exchanged to tag in case of <see cref="KeySetting.PICC_APPLICATION"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option to be used for information retrieval. One of the below values
        ///                         <see cref="KeySetting.UNKNOWN"/>
        ///                         <see cref="KeySetting.EXCLUDE_OPTION"/>
        ///                         <see cref="KeySetting.CRYPTOREQUEST_KEY_METADATA"/>
        ///                         <see cref="KeySetting.ECC_PRIVATE_KEY_METADATA"/>
        ///                         <see cref="KeySetting.CA_ROOT_KEY_METADATA"/>
        /// </param>
        /// <param name="aResponse">Returns the key settings. Can be 2 or 3 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Gets Application Key Setting for the application. In addition it returns the number of keys which are " +
                          "configured for the selected application and if applicable the AppKeySetSettings."
            )
        ]
        public Status_t GetKeySettings ( byte bOption, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_GetKeySettings ( m_pDataParams, bOption, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Reads out the current key version of any key stored on the tag
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number of the targeted key.</param>
        /// <param name="aResponse">Key set versions of the selected application ordered by ascending
        ///                         key set number, i.e. starting with the AKS.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = @"Depending on the currently selected AID and given key number parameter, return key version of the key targeted or
                            return all key set versions of the selected application."
            )
        ]
        public Status_t GetKeyVersion ( byte bKeyNo, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_GetKeyVersion ( m_pDataParams, bKeyNo, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region ASymmetric Key Management
        /// <summary>
        /// Creates or updates a private key entry by generating a key pair or importing a private key.
        ///
        /// Note:
        ///     bComOption will be
        ///         Communication mode of the targeted key or
        ///         If targeting not yet existing key, default CommMode defined by Cmd.SetConfiguration for
        ///         option ECC Key Management should be used. Default is <see cref="CommOption.FULL"/>.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bKeyNo">Key number of the key to be managed.
        ///                         At PICC level, two keys are supported.
        ///                         At application level, up to five keys are supported.
        /// </param>
        /// <param name="bOption">Target action to perform. Should be one of the below values.
        ///                         <see cref="TargetAction.GENERATE_KEY_PAIR"/>
        ///                         <see cref="TargetAction.IMPORT_PRIVATE_KEY"/>
        ///                         <see cref="TargetAction.UPDATE_META_DATA"/>
        /// </param>
        /// <param name="bCurveID">The targeted curve. Should be one of the below values.
        ///                         <see cref="TargetCurve.NIST_P256"/>
        ///                         <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aKeyPolicy">Defines the allowed crypto operations with the targeted key.
        ///                             Should be two bytes as follows,
        ///                                 Byte 0 => Bit 7 - 0
        ///                                 Byte 1 => Bit 15 - 8
        ///
        ///                             Supported values are, should be ORed
        ///                                 <see cref="KeyPolicy_ASym.DISABLED"/>
        ///                                 <see cref="KeyPolicy_ASym.FREEZE_KUC_LIMIT"/>
        ///                                 <see cref="KeyPolicy_ASym.ECC_CARD_UNILATERAL_AUTHENTICATION"/>
        ///                                 <see cref="KeyPolicy_ASym.ECC_SECURE_DYNAMINC_MESSAGING"/>
        ///                                 <see cref="KeyPolicy_ASym.CRYPTO_REQUEST_ECC_SIGN"/>
        ///                                 <see cref="KeyPolicy_ASym.CRYPTO_REQUEST_ECC_DH"/>
        ///                                 <see cref="KeyPolicy_ASym.ECC_SIGMA_I_MUTUAL_AUTH"/>
        /// </param>
        /// <param name="bWriteAccess">Defines the CommMode and access right required to update the key with
        ///                            Cmd.ManageKeyPair. Should contain below information.
        ///                                 Bits[7 - 6]: RFU
        ///                                 Bits[5 - 4]: Communication Modes, One of the below values.
        ///                                     <see cref="CommOption.PLAIN"/>
        ///                                     <see cref="CommOption.MAC"/>
        ///                                     <see cref="CommOption.FULL"/>
        ///                                 Bits[3 - 0]: Access Rights, One of the below values.
        ///                                     - 0x00 - 0x0B: Authentication Required
        ///                                     - 0x0C       : Free Access over NFC, Authentication required over I2C
        ///                                     - 0x0D       : Free Access over I2C, Authentication required over NFC
        ///                                     - 0x0E       : Free Access
        ///                                     - 0x0F       : No Access or RFU
        /// </param>
        /// <param name="dwKUCLimit">Defines the key usage limit of the targeted key.
        ///                             0x00000000: Key Usage Counter Limit is disabled
        ///                             Any other value: Key Usage Counter Limit enabled with the given value ( LSB first ).
        /// </param>
        /// <param name="wPrivKey_No">Key number in KeyStore of Private Key.</param>
        /// <param name="wPrivKey_Pos">Key position in KeyStore of Private Key.</param>
        /// <param name="aResponse">The Public Key in uncompressed point representation format.
        ///                             Present if bOption = <see cref="TargetAction.GENERATE_KEY_PAIR"/>
        ///                             NULL otherwise
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         The values provided in bComOption is not supported.
        ///     Returns <see cref="Error_Param.KEY"/>
        ///         If Key type is not ECC.
        ///         The Key format is not Binary (Uncompressed Point Representation).
        ///         The Key pair is not Private type.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = @"CommMode of targeted key, or if targeting not yet existing key, default CommMode of the
                              command as defined by Cmd.SetConfiguration.",
                Message = @"Creates or updates a private key entry by generating a key pair or importing a private key."
            )
        ]
        public Status_t ManageKeyPair ( byte bComOption, byte bKeyNo, byte bOption, byte bCurveID, byte[] aKeyPolicy, byte bWriteAccess,
            uint dwKUCLimit, ushort wPrivKey_No, ushort wPrivKey_Pos, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ManageKeyPair ( m_pDataParams, bComOption, bKeyNo, bOption, bCurveID, aKeyPolicy, bWriteAccess,
                dwKUCLimit, wPrivKey_No, wPrivKey_Pos, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Creates or updates a public key entry for storing a CARootKey.
        ///
        /// Note:
        ///     bComOption will be
        ///         Communication mode of the targeted key or
        ///         If targeting not yet existing key, default CommMode defined by Cmd.SetConfiguration for
        ///         option ECC Key Management should be used. Default is <see cref="CommOption.FULL"/>.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bKeyNo">Key number of the key to be managed.
        ///                         At PICC level, two keys are supported.
        ///                         At application level, up to five keys are supported.
        /// </param>
        /// <param name="bCurveID">The targeted curve. Should be one of the below values.
        ///                         <see cref="TargetCurve.NIST_P256"/>
        ///                         <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aAccessRights">Access rights associated with the KeyID.CARootKey. Should be 2 byte.</param>
        /// <param name="bWriteAccess">Defines the CommMode and access right required to update the key with
        ///                            Cmd.ManageKeyPair. Should contain below information.
        ///                                 Bits[7 - 6]: RFU
        ///                                 Bits[5 - 4]: Communication Modes, One of the below values.
        ///                                     <see cref="CommOption.PLAIN"/>
        ///                                     <see cref="CommOption.MAC"/>
        ///                                     <see cref="CommOption.FULL"/>
        ///                                 Bits[3 - 0]: Access Rights, One of the below values.
        ///                                     - 0x00 - 0x0B: Authentication Required
        ///                                     - 0x0C       : Free Access over NFC, Authentication required over I2C
        ///                                     - 0x0D       : Free Access over I2C, Authentication required over NFC
        ///                                     - 0x0E       : Free Access
        ///                                     - 0x0F       : No Access or RFU
        /// </param>
        /// <param name="wPubKey_No">Key number in KeyStore of Public Key.</param>
        /// <param name="wPubKey_Pos">Key position in KeyStore of Public Key.</param>
        /// <param name="aIssuer">The Trusted issuer name. Should be one of the following.
        ///                         NULL in case if No trusted issuer name check required.
        ///                         The Trusted issuer information otherwise. Ranging from 1 - 255 bytes
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         The values provided in bComOption is not supported.
        ///     Returns <see cref="Error_Param.KEY"/>
        ///         If Key type is not ECC.
        ///         The Key format is not Binary (Uncompressed Point Representation).
        ///         The Key pair is not Public type.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = @"CommMode of targeted key, or if targeting not yet existing key, default CommMode of the
                              command as defined by Cmd.SetConfiguration.",
                Message = @"Creates or updates a public key entry for storing a CARootKey."
            )
        ]
        public Status_t ManageCARootKey ( byte bComOption, byte bKeyNo, byte bCurveID, byte[] aAccessRights, byte bWriteAccess,
            ushort wPubKey_No, ushort wPubKey_Pos, byte[] aIssuer )
        {
            return phalNtagXDna_ManageCARootKey ( m_pDataParams, bComOption, bKeyNo, bCurveID, aAccessRights, bWriteAccess,
                wPubKey_No, wPubKey_Pos, aIssuer, ( byte ) ( ( aIssuer == null ) ? 0 : aIssuer.Length ) );
        }
        #endregion

        #region Certificate Management
        /// <summary>
        /// Creates or updates a public key entry for storing a CARootKey.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param><param name="bAction">Target action to perform. Should be one of the below values.
        ///                                 - <see cref="Action.CREATE_CERTIFICATE_REPOSITORY"/>
        ///                                 - <see cref="Action.LOAD_CERTIFICATE"/>
        ///                                 - <see cref="Action.LOAD_CERTIFICATE_MAPPING_INFO"/>
        ///                                 - <see cref="Action.ACTIVATE_REPOSITORY"/>
        ///                                 - <see cref="Action.RESET_CERTIFICATE_REPOSITORY"/>
        /// </param>
        /// <param name="bRepoID">ID used to identify certificate repository for algorithm execution and
        ///                       repository modification.Note - the certificate Id shall be used to
        ///                       reference a private key/certificate chain when performing SIGMA-I
        /// </param>
        /// <param name="aData">Data to be exchanged to tag based on \b bAction information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         The values provided in bComOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = @"CommMode of command as defined by Cmd.SetConfiguration.",
                Message = @"Manages Certificate Repositories."
            )
        ]
        public Status_t ManageCertRepo ( byte bComOption, byte bAction, byte bRepoID, byte[] aData )
        {
            return phalNtagXDna_ManageCertRepo ( m_pDataParams, bComOption, bAction, bRepoID, aData,
                ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Perform Certificate Repository Read.
        /// Returns information related to certificate repositories
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and reading of data.
        ///                         Reading data.
        ///                             <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the Data.
        ///                             <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Data.
        ///
        ///                         Communication modes. To be ORed with above values
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bRepoID">ID used to identify certificate repository</param>
        /// <param name="bDataItem">Data Iitem to be used for receiving the information.
        ///                         Should be one of the below values.
        ///                             - <see cref="DataItem.END_LEAF"/>
        ///                             - <see cref="DataItem.PARENT"/>
        ///                             - <see cref="DataItem.GRAND_PARENT"/>
        ///                             - <see cref="DataItem.REPOSITORY_METADATA"/>
        /// </param>
        /// <param name="aResponse"></param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         The values provided in bOption is not supported.
        ///         For Invalid Exchange option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = "Returns information related to certificate repositories",
                Message = @"Perform Certificate Repository Read."
            )
        ]
        public Status_t ReadCertRepo ( byte bOption, byte bRepoID, byte bDataItem, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ReadCertRepo ( m_pDataParams, bOption, bRepoID, bDataItem, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region File Management
        /// <summary>
        /// Creates files for the storage of plain unformatted user data within an existing application
        /// on the tag.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the presence of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created.</param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.SDM_MIRRORING_ENABLED"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     - 0x00 - 0x0B: Authentication Required
        ///                                     - 0x0C       : Free Access over NFC, Authentication required over I2C
        ///                                     - 0x0D       : Free Access over I2C, Authentication required over NFC
        ///                                     - 0x0E       : Free Access
        ///                                     - 0x0F       : No Access or RFU
        /// </param>
        /// <param name="aFileSize">The size of the file. Will be of 3 bytes with LSB first.
        ///                         If size 0x10 need to be created, then the FileSize will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Creates files for the storage of plain unformatted user data within an existing application on the tag."
            )
        ]
        public Status_t CreateStdDataFile ( byte bOption, byte bFileNo, byte[] aISOFileId, byte bFileOption, byte[] aAccessRights,
            byte[] aFileSize )
        {
            return phalNtagXDna_CreateStdDataFile ( m_pDataParams, bOption, bFileNo, aISOFileId, bFileOption, aAccessRights, aFileSize );
        }

        /// <summary>
        /// Creates files for the storage of plain unformatted Counter information.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.SDM_MIRRORING_ENABLED"/>
        ///                                 <see cref="FileOption.DEFERRED_CONFIGURATION_ENABLED"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     - 0x00 - 0x0B: Authentication Required
        ///                                     - 0x0C       : Free Access over NFC, Authentication required over I2C
        ///                                     - 0x0D       : Free Access over I2C, Authentication required over NFC
        ///                                     - 0x0E       : Free Access
        ///                                     - 0x0F       : No Access or RFU
        /// </param>
        /// <param name="dwValue">Counter Value to be stored.
        ///                       Provide the counter value in regular format, like if 0x10 needs to be updated
        ///                       then \b dwValue = 0x00000010
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Creates files for the storage of plain unformatted Counter information."
            )
        ]
        public Status_t CreateCounterFile ( byte bFileNo, byte bFileOption, byte[] aAccessRights, uint dwValue )
        {
            return phalNtagXDna_CreateCounterFile ( m_pDataParams, bFileNo, bFileOption, aAccessRights, dwValue );
        }

        /// <summary>
        /// Returns the file IDs of all active files within the currently selected application.
        /// </summary>
        ///
        /// <param name="aFileId">The buffer containing the available File ID(s).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Returns the File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetFileIDs ( out byte[] aFileId )
        {
            Status_t oStatus;
            IntPtr ppFileId = IntPtr.Zero;
            ushort wFileIdLen = 0;

            oStatus = phalNtagXDna_GetFileIDs ( m_pDataParams, ref ppFileId, ref wFileIdLen );
            aFileId = MarshalCopy ( oStatus, ppFileId, wFileIdLen );

            return oStatus;
        }

        /// <summary>
        /// Get the ISO File IDs.
        /// </summary>
        ///
        /// <param name="aISOFileId">The buffer containing the available ISO File ID(s).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Returns the ISO File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetISOFileIDs ( out byte[] aISOFileId )
        {
            Status_t oStatus;
            IntPtr ppISOFileId = IntPtr.Zero;
            ushort wISOFileIdLen = 0;

            oStatus = phalNtagXDna_GetISOFileIDs ( m_pDataParams, ref ppISOFileId, ref wISOFileIdLen );
            aISOFileId = MarshalCopy ( oStatus, ppISOFileId, wISOFileIdLen );

            return oStatus;
        }

        /// <summary>
        /// Get information on the properties of a specific file
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aFSBuffer">The buffer containing the settings.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Get information on the properties of a specific file."
            )
        ]
        public Status_t GetFileSettings ( byte bFileNo, out byte[] aFSBuffer )
        {
            Status_t oStatus;
            IntPtr ppFSBuffer = IntPtr.Zero;
            ushort wFSBufLen = 0;

            oStatus = phalNtagXDna_GetFileSettings ( m_pDataParams, bFileNo, ref ppFSBuffer, ref wFSBufLen );
            aFSBuffer = MarshalCopy ( oStatus, ppFSBuffer, wFSBufLen );

            return oStatus;
        }

        /// <summary>
        /// Get file related counters used for Secure Dynamic Messaging.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                         <see cref="CommOption.PLAIN"/>
        ///                         <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aFileCounters">The SDMReadCounter information returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Get file related counters used for Secure Dynamic Messaging."
            )
        ]
        public Status_t GetFileCounters ( byte bOption, byte bFileNo, out byte[] aFileCounters )
        {
            Status_t oStatus;
            IntPtr ppFileCounters = IntPtr.Zero;
            ushort wFileCounterLen = 0;

            oStatus = phalNtagXDna_GetFileCounters ( m_pDataParams, bOption, bFileNo, ref ppFileCounters, ref wFileCounterLen );
            aFileCounters = MarshalCopy ( oStatus, ppFileCounters, wFileCounterLen );

            return oStatus;
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                         <see cref="CommOption.PLAIN"/>
        ///                         <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.SDM_MIRRORING_ENABLED"/>: If Standard File is targeted.
        ///                                 <see cref="FileOption.DEFERRED_CONFIGURATION_ENABLED"/>: If FileNo: 0x02 is targeted.
        /// </param>
        /// <param name="aAccessRights">Set of access conditions for the 1st set in the file. Should be 2 byte.</param>
        /// <param name="aAddInfo">Buffer should contain the following information.
        ///                         [NrAddARs] || [Additional access rights] || [SDMOption || SDM AccessRights || VCUIDOffset ||
        ///                         SDMReadCtrOffset || PICCDataOffset || GPIOStatusOffset || SDMMACInputOffset || SDMENCOffset ||
        ///                         SDMENCLength || SDMMACOffset || SDMReadCtrLimit] || [TMIExclFileMap] || [TMCLimit] || [CRLOptions ||
        ///                         CNSSize] || CRLSigKey]
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File numbers (bFileNo).
        ///          For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                Message = "Changes the access parameters of an existing file."
            )
        ]
        public Status_t ChangeFileSettings ( byte bOption, byte bFileNo, byte bFileOption, byte[] aAccessRights, byte[] aAddInfo )
        {
            return phalNtagXDna_ChangeFileSettings ( m_pDataParams, bOption, bFileNo, bFileOption, aAccessRights, aAddInfo,
                ( byte ) ( ( aAddInfo == null ) ? 0 : aAddInfo.Length ) );
        }
        #endregion

        #region Data Management
        /// <summary>
        /// Reads data from Standard data files.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing ofSecure Messaging and reading of data.
        ///                         Reading data.
        ///                             <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the Data.
        ///                             <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Data.
        ///
        ///                         Communication modes. To be ORed with above values
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="aOffset">The offset from where the data should be read. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        ///                         0 to ( FixeSize - 1): Starting position of Read operation.
        ///                         0xFFFFFFFF          : Return CRLFile meta-data
        /// </param>
        /// <param name="aLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///                         If 0x10 bytes need to be read, then it will be 10 00 00.
        ///                         If complete file need to be read, then it will be 00 00 00.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///          For Invalid Exchange option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Only ISO/IEC 14443-4 Chaining is supported.",
                Message = "Reads data from FileType.StandardData."
            )
        ]
        public Status_t ReadData ( byte bOption, byte bFileNo, byte[] aOffset, byte[] aLength, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ReadData ( m_pDataParams, bOption, bFileNo, aOffset, aLength, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Writes data to standard data files.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and writing of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be written.</param>
        /// <param name="aOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        /// </param>
        /// <param name="aData">The data to be written to the PICC.</param>
        /// <param name="aLength">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 bytes need to be written, then it will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Only ISO/IEC 14443-4 Chaining is supported.",
                Message = "Writes data to standard data files."
            )
        ]
        public Status_t WriteData ( byte bOption, byte bFileNo, byte[] aOffset, byte[] aData, byte[] aLength )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalNtagXDna_WriteData ( m_pDataParams, bOption, bFileNo, aOffset, aDataTmp, aLength );
        }

        /// <summary>
        /// Increments the counter file.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and writing of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be written.</param>
        /// <param name="dwIncrValue">The value to be incremented.
        ///                           Provide the counter value in regular format, like if 0x10 needs to be incremented
        ///                           then \b dwIncrValue = 0x00000010
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Increments the counter file."
            )
        ]
        public Status_t IncrementCounterFile ( byte bOption, byte bFileNo, uint dwIncrValue )
        {
            return phalNtagXDna_IncrementCounterFile ( m_pDataParams, bOption, bFileNo, dwIncrValue );
        }
        #endregion

        #region CryptoAPI
        /// <summary>
        /// Perform CryptoRequest SHA.
        /// It is possible to execute an SHA calculation using a single command or as a
        /// series of commands.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        /// </param>
        /// <param name="bAlgorithm">Indicates the Algorithm to be used. One of the following,
        ///                             - <see cref="CryptoAlgorithm.SHA_256"/>
        ///                             - <see cref="CryptoAlgorithm.SHA_384"/>
        /// </param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Input data to be hashed. One of the following
        ///                         - NULL if Input data source is not Command Buffer
        ///                         - Actual data to be hashed if Input data source
        ///                           is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="aResponse">Buffer containing the Hashed information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform CryptoRequest SHA",
                OtherInfo = "Supports SHA256 and SHA384 SHA Algorithm."
            )
        ]
        public Status_t CryptoRequest_SHA ( byte bOption, byte bOperation, byte bAlgorithm, byte bInputDataSource,
            byte bResultDst, byte[] aInputData, byte bInputDataLen, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestSHA ( m_pDataParams, bOption, bOperation, bAlgorithm,
                bInputDataSource, bResultDst, aInputData, bInputDataLen, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest RNG.
        /// It is possible to generate random data, which is compliant with NIST SP800-90B using a 256-bit key.
        /// The Maximum number of generated bytes is 128.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bNumBytes">The number of bytes to generate random bytes </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aResponse">Buffer containing the Random data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
                (
                    NTAG_Products.NTAG_X_DNA,
                    Message = @"Perform CryptoRequest RNG",
                    OtherInfo = "Generates Random number upto maximum of 128 bytes using NIST SP800-90B with AES256 key."
                )
            ]
        public Status_t CryptoRequest_RNG ( byte bOption, byte bNumBytes, byte bResultDst, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestRNG ( m_pDataParams, bOption, bNumBytes, bResultDst,
                ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest ECC Signature generation.
        /// The ECC signature generation API supports signing of a data stream or a pre-computed hash.
        ///
        /// Note:
        ///     Before using this interface, a PrivateKey should be generated or Imported using
        ///     <see cref="Generic.ManageKeyPair"/> interface with Key-Policy set to
        ///     <see cref="KeyPolicy_ASym.CRYPTO_REQUEST_ECC_SIGN"/>.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT_PRE_COMPUTED_HASH"/>
        /// </param>
        /// <param name="bPrivateKeyID">Id of the ECC key pair containing the private key to use.
        ///                             Note a key pair must be marked as Crypto API Signature
        /// </param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Input data to be Signed. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be Signed if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="aResponse">Buffer containing the Signature information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest ECC Signature generation. " +
                           "The ECC signature generation API supports signing of a data stream or a pre-computed hash",
                OtherInfo = "Before using this interface, a PrivateKey should be generated or Imported using " +
                            "ManageKeyPair interface with Key-Policy set to CryptoRequest ECC Sign (Action 0x03)"
            )
        ]
        public Status_t CryptoRequest_ECCSign ( byte bOption, byte bOperation, byte bPrivateKeyID, byte bInputDataSource,
            byte[] aInputData, byte bInputDataLen, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestECCSign ( m_pDataParams, bOption, bOperation, bPrivateKeyID,
                bInputDataSource, aInputData, bInputDataLen, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest ECC Signature verification.
        /// The ECC signature verification API supports verification of a data stream or data, which has
        /// already been hashed.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT_PRE_COMPUTED_HASH"/>
        /// </param>
        /// <param name="bCurveID">The targeted curve. Should be one of the below values.
        ///                         <see cref="TargetCurve.NIST_P256"/>
        ///                         <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aHostPubKey">The public key to use for signature verification provided in
        ///                           uncompressed format.i.e.leading 0x04 byte
        /// </param>
        /// <param name="aSignature">Signature to verify.</param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Input data to be verified. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be verified if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     Returns <see cref="Error.SIGNATURE_VERIFICATION_FAILURE"/> Verification of Message / Signature combination failed
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest ECC Signature verification.",
                OtherInfo = "The ECC signature verification API supports verification of a data stream or data, which has" +
                            " already been hashed"
            )
        ]
        public Status_t CryptoRequest_ECCVerify ( byte bOption, byte bOperation, byte bCurveID, byte[] aHostPubKey,
            byte[] aSignature, byte bInputDataSource, byte[] aInputData, byte bInputDataLen )
        {
            return phalNtagXDna_CryptoRequestECCVerify ( m_pDataParams, bOption, bOperation, bCurveID, aHostPubKey,
                ( byte ) ( ( aHostPubKey == null ) ? 0 : aHostPubKey.Length ), aSignature,
                ( byte ) ( ( aSignature == null ) ? 0 : aSignature.Length ), bInputDataSource,
                aInputData, bInputDataLen );
        }

        /// <summary>
        /// Perform CryptoRequest ECC Diffie-Hellman.
        /// The ECC Diffie-Hellman API supports the use of static keys or ephemeral keys.In addition,
        /// it allows the shared secret to be generated using a single or two-step approach.
        ///
        /// Note:
        ///     Before using this interface, a PrivateKey should be generated or Imported using
        ///     <see cref="Generic.ManageKeyPair"/> interface with Key-Policy set to
        ///     <see cref="KeyPolicy_ASym.CRYPTO_REQUEST_ECC_DH"/>.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation_DH.ONE_STEP"/>
        ///                             - <see cref="CryptoOperation_DH.TWO_STEP_FIRST"/>
        ///                             - <see cref="CryptoOperation_DH.TWO_STEP_FINAL"/>
        /// </param>
        /// <param name="bKeyPairID">Should be one of the below values.
        ///                             - 0x00 - 0x04: Static key pair - the key pair must be marked
        ///                               as Crypto API ECDH
        ///                             - 0xFE       : Use NIST 256 ephemeral key pair
        ///                             - 0xFF       : Use Brainpool 256 ephemeral key pair
        /// </param>
        /// <param name="bSSDestination">Crypto API Data Destination Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aHostPubKey">The host's public key to use for shared secret generation,
        ///                           provided in uncompressed format I.e leading 0x04 byte
        /// </param>
        /// <param name="aResponse">Buffer containing,
        ///                             - [Card's ephemeral publickey]: If key pair Id indicates an
        ///                               ephemeral key and single step or two-step step 1
        ///                             - [Shared Secret]: If single step or two-step step 2 and output
        ///                               destination is the command buffer
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest ECC Diffie-Hellman. " +
                           "The ECC Diffie-Hellman API supports the use of static keys or ephemeral keys. In addition," +
                           " it allows the shared secret to be generated using a single or two-step approach",
                OtherInfo = "Before using this interface, a PrivateKey should be generated or Imported using " +
                            "ManageKeyPair interface with Key-Policy set to CryptoRequest ECC DH (Action 0x05)"
            )
        ]
        public Status_t CryptoRequest_ECCDH ( byte bOption, byte bOperation, byte bKeyPairID, byte bSSDestination,
            byte[] aHostPubKey, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestECCDH ( m_pDataParams, bOption, bOperation, bKeyPairID,
                bSSDestination, aHostPubKey, ( byte ) ( ( aHostPubKey == null ) ? 0 : aHostPubKey.Length ),
                ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest AES ECB / CBC Encryption or Decryption.
        /// Note: The AES primitives supported by a static key are defined by the KeyPolicy set via the ChangeKey command
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        /// </param>
        /// <param name="bPrimitive">One of the following Primitive.
        ///                             - <see cref="Primitive.ECB_ENCRYPT"/>
        ///                             - <see cref="Primitive.ECB_DECRYPT"/>
        ///                             - <see cref="Primitive.CBC_ENCRYPT"/>
        ///                             - <see cref="Primitive.CBC_DECRYPT"/>
        /// </param>
        /// <param name="bKeyID">Id of the AES key.
        ///                         - 0x10 - 0x17: Id of AES Key ( must be in crypto API range: 10  17),
        ///                           the key length from the static key.
        ///                         - 0x80 - 0x87: Transient buffer slot number containing the AES key, the
        ///                           key length shall be in the following field
        ///                         - 0xC0 - 0xCF: Static buffer slot number containing the AES key, the key
        ///                           length shall be in the following field
        /// </param>
        /// <param name="bKeyLen">Length of AES key, only present when the key source is an internal buffer. </param>
        /// <param name="bICVSource">Crypto API ICV Source Selection. Only present for CBC operations.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aICV">Only present for CBC operations and the ICV is in the command buffer.</param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Data to be Encrypted or Decrypted. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be encrypted or decrypted if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="aResponse">Buffer containing the Encrypted or Decrypted data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest AES ECB / CBC Encryption or Decryption.",
                OtherInfo = "The AES primitives supported by a static key are defined by the KeyPolicy set via the ChangeKey command."
            )
        ]
        public Status_t CryptoRequest_AES ( byte bOption, byte bOperation, byte bPrimitive, byte bKeyID, byte bKeyLen,
            byte bICVSource, byte[] aICV, byte bInputDataSource, byte bResultDst, byte[] aInputData, byte bInputDataLen,
            out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestAES ( m_pDataParams, bOption, bOperation, bPrimitive,
                bKeyID, bKeyLen, bICVSource, aICV, ( byte ) ( ( aICV == null ) ? 0 : aICV.Length ),
                bInputDataSource, bResultDst, aInputData, bInputDataLen, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest AES CMAC Signing or Verification.
        /// \note: The AES API supports the use of static crypto API keys or keys stored in an internal buffer.
        /// The AES primitives supported by a static key are defined by the KeyPolicy set via the ChangeKey command.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        /// </param>
        /// <param name="bPrimitive">One of the following Primitive.
        ///                             - <see cref="Primitive.SIGN"/>
        ///                             - <see cref="Primitive.VERIFY"/>
        /// </param>
        /// <param name="bKeyID">Id of the AES key.
        ///                         - 0x10 - 0x17: Id of AES Key ( must be in crypto API range: 10  17),
        ///                           the key length from the static key.
        ///                         - 0x80 - 0x87: Transient buffer slot number containing the AES key, the
        ///                           key length shall be in the following field
        ///                         - 0xC0 - 0xCF: Static buffer slot number containing the AES key, the key
        ///                           length shall be in the following field
        /// </param>
        /// <param name="bKeyLen">Length of AES key, only present when the key source is an internal buffer. </param>
        /// <param name="aCMACSignature">CMAC Signature to be verified.
        ///                                 - NULL is Primitive is <see cref="Primitive.SIGN"/>
        ///                                 - 8 byte or 16 byte CMAC Signature if Primitive is <see cref="Primitive.VERIFY"/>
        /// </param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Data to be Signed or Verified. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be Signed or Verified if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="aResponse">Buffer containing the following,
        ///                             - 16 byte CMAC Signature if Primitive is <see cref="Primitive.VERIFY"/>
        ///                             - NULL if Primitive is <see cref="Primitive.SIGN"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     Returns <see cref="Error.SIGNATURE_VERIFICATION_FAILURE"/> Verification of Message / Signature combination failed
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest AES CMAC Signing or Verification.",
                OtherInfo = "The AES API supports the use of static crypto API keys or keys stored in an internal buffer. The AES primitives" +
                            " supported by a static key are defined by the KeyPolicy set via the ChangeKey command."
            )
        ]
        public Status_t CryptoRequest_AESCMAC ( byte bOption, byte bOperation, byte bPrimitive, byte bKeyID, byte bKeyLen,
            byte[] aCMACSignature, byte bInputDataSource, byte[] aInputData, byte bInputDataLen, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestAESCMAC ( m_pDataParams, bOption, bOperation, bPrimitive, bKeyID, bKeyLen,
                aCMACSignature, ( byte ) ( ( aCMACSignature == null ) ? 0 : aCMACSignature.Length ), bInputDataSource,
                aInputData, bInputDataLen, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest AES AEAD Encryption / Signing or Decryption / Verification.
        ///     - The AES API supports the use of static crypto API keys or keys stored in an internal buffer. The AES primitives
        ///       supported by a static key are defined by the KeyPolicy set via the ChangeKey command.
        ///     - The output destination for multi-part AEAD shall always be the command buffer. For a one-shot operation, the
        ///       result destination can be an internal static or transient buffer
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        /// </param>
        /// <param name="bPrimitive">One of the following Primitive.
        ///                             - <see cref="Primitive.CCM_ENCRYPT_SIGN"/>
        ///                             - <see cref="Primitive.CCM_ENCRYPT_SIGN_INTERNAL_NONCE"/>
        ///                             - <see cref="Primitive.CCM_DECRYPT_VERIFY"/>
        ///                             - <see cref="Primitive.GCM_ENCRYPT_SIGN"/>
        ///                             - <see cref="Primitive.GCM_ENCRYPT_SIGN_INTERNAL_NONCE"/>
        ///                             - <see cref="Primitive.GCM_DECRYPT_VERIFY"/>
        /// </param>
        /// <param name="bKeyID">Id of the AES key.
        ///                         - 0x10 - 0x17: Id of AES Key ( must be in crypto API range: 10  17),
        ///                           the key length from the static key.
        ///                         - 0x80 - 0x87: Transient buffer slot number containing the AES key, the
        ///                           key length shall be in the following field
        ///                         - 0xC0 - 0xCF: Static buffer slot number containing the AES key, the key
        ///                           length shall be in the following field
        /// </param>
        /// <param name="bKeyLen">Length of AES key, only present when the key source is an internal buffer. </param>
        /// <param name="bNonceSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aNonce">Nonce to be used. One of the following,
        ///                         - Present for \b bPrimitive = Internally generated
        ///                         - NULL Otherwise
        /// </param>
        /// <param name="bNonceLen">Length of Nonce.One of the following,
        ///                             - 0x0D       : AES CCM
        ///                             - 0x0C - 0x3C: AES GCM
        /// </param>
        /// <param name="wTotal_AAdLen">Total length of AAD information that will be exchanged. This will be application
        ///                             for below primitives,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        /// </param>
        /// <param name="wTotal_InputDataLen">Total length of Input data that will be exchanged. This will be application
        ///                                   for below primitives,
        ///                                     - <see cref="CryptoOperation.INIT"/>
        ///                                     - <see cref="CryptoOperation.UPDATE"/>
        ///                                     - <see cref="CryptoOperation.FINALIZE"/>
        /// </param>
        /// <param name="bAADSource">Crypto API AAD Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aAAD">AAD data. One of the following
        ///                     - NULL if Input data source is not Command Buffer
        ///                     - Actual data to be encrypted or decrypted if Input data source
        ///                       is Command Buffer
        /// </param>
        /// <param name="bAADLen">Length AAD as mentioned below
        ///                         - Actual data length if Input data source is not Command Buffer
        ///                         - Length of bytes available in \b pAAD buffer.
        /// </param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Data to be Encrypted / Signed or Decrypted / Verified. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be Encrypted / Signed or Decrypted / Verified if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="aTagData">Tag Data to be used. One of the following
        ///                         - Tag information to be used. Only present when \b bPrimitive is Decrypt or Verify.
        ///                         - NULL otherwise.
        /// </param>
        /// <param name="bTagDataLen">Tag length as mentioned below
        ///                             - 0x08 or 0x10: CCM
        ///                             - 0x0C or 0x10: GCM
        /// </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aResponse">Buffer containing the following
        ///                             - Nonce if
        ///                                 - \b bOperation is <see cref="CryptoOperation.ONE_SHOT"/> or <see cref="CryptoOperation.INIT"/>
        ///                                 - \b bPrimitive is Encrypt / Sign with internally generated Nonce.
        ///                             - Encrypted / Decrypted data for any of the operation.
        ///                             - Tag Data \b bPrimitive is Encryption / Sign and \b bOperation is <see cref="CryptoOperation.ONE_SHOT"/>
        ///                               or <see cref="CryptoOperation.INIT"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     Returns <see cref="Error.SIGNATURE_VERIFICATION_FAILURE"/> Verification of Message / Signature combination failed
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest AES AEAD Encryption / Signing or Decryption / Verification."
            )
        ]
        public Status_t CryptoRequest_AESAEAD ( byte bOption, byte bOperation, byte bPrimitive, byte bKeyID, byte bKeyLen,
            byte bNonceSource, byte[] aNonce, byte bNonceLen, ushort wTotal_AAdLen, ushort wTotal_InputDataLen, byte bAADSource,
            byte[] aAAD, byte bAADLen, byte bInputDataSource, byte[] aInputData, byte bInputDataLen, byte[] aTagData,
            byte bTagDataLen, byte bResultDst, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestAESAEAD ( m_pDataParams, bOption, bOperation, bPrimitive, bKeyID, bKeyLen,
                bNonceSource, aNonce, bNonceLen, wTotal_AAdLen, wTotal_InputDataLen, bAADSource, aAAD, bAADLen, bInputDataSource,
                aInputData, bInputDataLen, aTagData, bTagDataLen, bResultDst, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest Write Internal Buffer.
        /// It is possible to write a specific value to an internal buffer using this command option. This allows data to be
        /// loaded for use within other crypto API operations
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bDestination">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aData">Data to write to the internal buffer.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform CryptoRequest Write Internal Buffer",
                OtherInfo = "It is possible to write a specific value to an internal buffer using this command option. " +
                            "This allows data to be loaded for use within other crypto API operations."
            )
        ]
        public Status_t CryptoRequest_WriteInternalBuffer ( byte bOption, byte bDestination, byte[] aData )
        {
            return phalNtagXDna_CryptoRequestWriteInternalBuffer ( m_pDataParams, bOption, bDestination, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Perform CryptoRequest HMAC (Hash-Based Message Authentication Code) Signature generation and verification.
        /// Note: It is possible to execute an HMAC calculation using a single command or as a series of commands.
        /// Using multiple steps allows the input data to be taken from different sources.The API uses a secure SHA
        /// implementation.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation.INIT"/>
        ///                             - <see cref="CryptoOperation.UPDATE"/>
        ///                             - <see cref="CryptoOperation.FINALIZE"/>
        ///                             - <see cref="CryptoOperation.ONE_SHOT"/>
        /// </param>
        /// <param name="bPrimitive">One of the following Primitive.
        ///                             - <see cref="Primitive.SIGN"/>
        ///                             - <see cref="Primitive.VERIFY"/>
        /// </param>
        /// <param name="bDigestAlgo">Indicates the Algorithm to be used. One of the following,
        ///                             - <see cref="CryptoAlgorithm.SHA_256"/>
        ///                             - <see cref="CryptoAlgorithm.SHA_384"/>
        /// </param>
        /// <param name="bKeyID">Id of the HMAC key.
        ///                         - 0x10 - 0x17: Id of AES Key ( must be in crypto API range: 10  17),
        ///                           the key length from the static key.
        ///                         - 0x80 - 0x87: Transient buffer slot number containing the AES key, the
        ///                           key length shall be in the following field
        ///                         - 0xC0 - 0xCF: Static buffer slot number containing the AES key, the key
        ///                           length shall be in the following field
        /// </param>
        /// <param name="bKeyLen">Length of HMAC key, only present when the key source is an internal buffer. </param>
        /// <param name="aHMac">Hash MAC Signature to be verified.
        ///                         - 32 byte or 48 byte HMac signature if \b bOperation are <see cref="CryptoOperation.FINALIZE"/> or
        ///                           <see cref="CryptoOperation.ONE_SHOT"/> and \b bPrimitive <see cref="Primitive.VERIFY"/>
        ///                         - NULL otherwise
        /// </param>
        /// <param name="bInputDataSource">Crypto API Data Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInputData">Data to be Signed or Verified. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be Signed or Verified if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bInputDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aResponse">Buffer containing the following,
        ///                             - 32 byte or 48 byte HMac signature if \b bPrimitive is <see cref="Primitive.SIGN"/>
        ///                             - NULL otherwise
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     Returns <see cref="Error.SIGNATURE_VERIFICATION_FAILURE"/> Verification of Message / Signature combination failed
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest HMAC (Hash-Based Message Authentication Code) Signature generation and verification.",
                OtherInfo =  "It is possible to execute an HMAC calculation using a single command or as a series of commands." +
                             " Using multiple steps allows the input data to be taken from different sources.The API uses a secure SHA" +
                             "implementation."
            )
        ]
        public Status_t CryptoRequest_HMAC ( byte bOption, byte bOperation, byte bPrimitive, byte bDigestAlgo, byte bKeyID, byte bKeyLen,
            byte[] aHMac, byte bInputDataSource, byte[] aInputData, byte bInputDataLen, byte bResultDst, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestHMAC ( m_pDataParams, bOption, bOperation, bPrimitive, bDigestAlgo, bKeyID,
                bKeyLen, aHMac, ( byte ) ( ( aHMac == null ) ? 0 : aHMac.Length ), bInputDataSource, aInputData,
                bInputDataLen, bResultDst,  ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest HKDF (HMAC Key Derivation function) Extract and Expand.
        /// HKDF, as defined in RFC5869, requires execution of the extract operation followed by the expand operation.
        /// The API uses a secure SHA implementation
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Indicates the Operation to be used. One of the following,
        ///                             - <see cref="CryptoOperation_HKDF.EXTRACT_EXPAND"/>
        ///                             - <see cref="CryptoOperation_HKDF.EXPAND"/>
        /// </param>
        /// <param name="bDigestAlgo">Indicates the Algorithm to be used. One of the following,
        ///                             - <see cref="CryptoAlgorithm.SHA_256"/>
        ///                             - <see cref="CryptoAlgorithm.SHA_384"/>
        /// </param>
        /// <param name="bKeyID">Id of the HMAC key.
        ///                         - 0x10 - 0x17: Id of AES Key ( must be in crypto API range: 10  17),
        ///                           the key length from the static key.
        ///                         - 0x80 - 0x87: Transient buffer slot number containing the AES key, the
        ///                           key length shall be in the following field
        ///                         - 0xC0 - 0xCF: Static buffer slot number containing the AES key, the key
        ///                           length shall be in the following field
        /// </param>
        /// <param name="bKeyLen">Length of HMAC key, only present when the key source is an internal buffer. </param>
        /// <param name="bSaltSource">Crypto API Salt Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aSaltData">Salt data. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual data to be Signed or Verified if Input data source
        ///                               is Command Buffer
        /// </param>
        /// <param name="bSaltDataLen">Length of Salt dataas mentioned below
        ///                                 - Actual data length if Salt data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInfoData buffer.
        /// </param>
        /// <param name="bInfoSource">Crypto API Info Source Selection.
        ///                                 - 0x00       : Command Buffer
        ///                                 - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                                 - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="aInfoData">Context data. One of the following
        ///                             - NULL if Input data source is not Command Buffer
        ///                             - Actual Context data if Input data source is Command Buffer
        /// </param>
        /// <param name="bInfoDataLen">Length Input Data as mentioned below
        ///                                 - Actual data length if Input data source is not Command Buffer
        ///                                 - Length of bytes available in \b pInputData buffer.
        /// </param>
        /// <param name="bResultDst">Crypto API Data Destination Selection.
        ///                             - 0x00       : Command Buffer
        ///                             - 0x80 - 0x87: Transient buffer slot number (0 to 7)
        ///                             - 0xC0 - 0xCF: Static buffer slot number (0 to 15)
        /// </param>
        /// <param name="bResultLen">Number of bytes to output</param>
        /// <param name="aResponse">Buffer containing HKDF result. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = "Perform CryptoRequest HKDF (HMAC Key Derivation function) Extract and Expand.",
                OtherInfo = "HKDF, as defined in RFC5869, requires execution of the extract operation followed by the expand operation." +
                            "The API uses a secure SHA implementation"
            )
        ]
        public Status_t CryptoRequest_HKDF ( byte bOption, byte bOperation, byte bDigestAlgo, byte bKeyID, byte bKeyLen, byte bSaltSource,
            byte[] aSaltData, byte bSaltDataLen, byte bInfoSource, byte[] aInfoData, byte bInfoDataLen, byte bResultDst, byte bResultLen,
            out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestHKDF ( m_pDataParams, bOption, bOperation, bDigestAlgo, bKeyID, bKeyLen,
                bSaltSource, aSaltData, bSaltDataLen, bInfoSource, aInfoData, bInfoDataLen, bResultDst, bResultLen,
                ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform CryptoRequest ECHO.
        /// It is possible to have the device echo the command data provided to it.
        /// This may be useful to verify system setup.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                         - <see cref="CommOption.PLAIN"/>
        ///                         - <see cref="CommOption.MAC"/>
        ///                         - <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aData">Data to be Echoed back.</param>
        /// <param name="aResponse">Buffer containing the transmitted data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform CryptoRequest ECHO",
                OtherInfo = "It is possible to have the device echo the command data provided to it. " +
                            "This may be useful to verify system setup."
            )
        ]
        public Status_t CryptoRequest_ECHO ( byte bOption, byte[] aData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_CryptoRequestECHO ( m_pDataParams, bOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref ppResponse,
                ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region GPIO Management
        /// <summary>
        /// Perform GPIO Management
        /// </summary>
        ///
        /// <param name="wOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bGPIONo">GPIO Number to use for management. One of the below values.
        ///                         <see cref="GPIONo.GPIO_1"/>
        ///                         <see cref="GPIONo.GPIO_2"/>
        /// </param>
        /// <param name="bOperation">Targeted Operation perform on the selected GPIO. One of the below values.
        ///                             - GPIOxMode as Output
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_CLEAR"/>
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_SET"/>
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_TOGGLE"/>
        ///                                 - <see cref="GPIO_Operation.Output.PAUSE_RELEASE_NFC"/>
        ///
        ///                             - GPIOxMode as Down-Stream Power Out
        ///                                 - GPIO Control
        ///                                     -<see cref="GPIO_Operation.DownStreamPowerOut.ENABLE_POWER_HARVEST"/>
        ///
        ///                                 - GPIO Measurement Control
        ///                                     -<see cref="GPIO_Operation.DownStreamPowerOut.EXECUTE_MEASUREMENT"/>
        ///
        ///                                 - Target Voltage / Current Level
        ///                                     - Power downstream voltage of 1.8V
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_100uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_300uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_500uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_1mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_2mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_3mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_5mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_7mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_10mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_MAX_AVAILABLE"/>
        ///                                     - Power downstream voltage of 2V
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_100uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_300uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_500uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_1mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_2mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_3mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_5mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_7mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_10mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_MAX_AVAILABLE"/>
        /// </param>
        /// <param name="aNFCPauseRspData">NFC Pause Response Data: data to be returned to NFC host in the case of
        ///                               Release NFC Pause
        /// </param>
        /// <param name="aResponse">Response from PICC as follows.
        ///                             - If \b bOperation = #PHAL_MFECC_OPERATION_TOGGLE_PAUSE_NFC then,
        ///                               NFC Pause Response Data: data received from the I2C interface
        ///                             - NULL for others.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         If the InputType is not supported (bInputType).
        ///         The values provided in wOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform GPIO Management"
            )
        ]
        public Status_t ManageGPIO ( ushort wOption, byte bGPIONo, byte bOperation, byte[] aNFCPauseRspData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ManageGPIO ( m_pDataParams, wOption, bGPIONo, bOperation, aNFCPauseRspData,
                ( ushort ) ( ( aNFCPauseRspData == null ) ? 0 : aNFCPauseRspData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform GPIO read.
        /// </summary>
        ///
        /// <param name="wOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aResponse"></param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         If the InputType is not supported (bInputType).
        ///         The values provided in wOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                Message = @"Perform GPIO read."
            )
        ]
        public Status_t ReadGPIO ( ushort wOption, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalNtagXDna_ReadGPIO ( m_pDataParams, wOption, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region ISO7816 - 4 Support
        /// <summary>
        /// Perform File or Application selection. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///                             <see cref="FCI.NOT_RETURNED"/>
        ///                             <see cref="FCI.RETURNED"/>
        /// </param>
        /// <param name="bSelector">The selector to be used.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_PARENT_DF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_BY_DF_NAME"/>
        /// </param>
        /// <param name="aFid">The ISO File number to be selected.
        ///                     Valid only if bSelector is one of the following.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        /// </param>
        /// <param name="aDFname">The ISO DFName to be selected.
        ///                         Valid only when bSelector = <see cref="Selector.SELECT_BY_DF_NAME"/>.
        ///                         NULL for other bSelector options.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="aFCI">The FCI information returned by the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         DFName Length is greater than 16 (aDFname).
        ///         Invalid FCI ( File Control Identifier) (bOption).
        ///         Invalid Selector option (bSelector).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] aFid, byte[] aDFname, byte bExtendedLenApdu,
            out byte[] aFCI )
        {
            Status_t oStatus;
            IntPtr ppFCI = IntPtr.Zero;
            ushort wFCILen = 0;

            oStatus = phalNtagXDna_IsoSelectFile ( m_pDataParams, bOption, bSelector, aFid, aDFname,
                ( byte ) ( ( aDFname == null ) ? 0 : aDFname.Length ), bExtendedLenApdu,
                ref ppFCI, ref wFCILen );

            aFCI = MarshalCopy ( oStatus, ppFCI, wFCILen );

            return oStatus;
        }

        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///                             <see cref="FCI.NOT_RETURNED"/>
        ///                             <see cref="FCI.RETURNED"/>
        /// </param>
        /// <param name="bSelector">The selector to be used.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_PARENT_DF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_BY_DF_NAME"/>
        /// </param>
        /// <param name="aFid">The ISO File number to be selected.
        ///                     Valid only if bSelector is one of the following.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        /// </param>
        /// <param name="aDFname">The ISO DFName to be selected.
        ///                         Valid only when bSelector = <see cref="Selector.SELECT_BY_DF_NAME"/>.
        ///                         NULL for other bSelector options.
        /// </param>
        /// <param name="pFCI">The FCI information returned by the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         DFName Length is greater than 16 (aDFname).
        ///         Invalid FCI ( File Control Identifier) (bOption).
        ///         Invalid Selector option (bSelector).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is not supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] aFid, byte[] aDFname, out byte[] aFCI )
        {
            return IsoSelectFile ( bOption, bSelector, aFid, aDFname, 0x00, out aFCI );
        }

        /// <summary>
        /// Perform ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the tag and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from tag.
        /// </param>
        /// <param name="bOffset">The offset from where the data should be read.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file data is returned by tag.
        ///                                 - If non-zero is provided, then data starting from offset is returned.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="aResponse">The data returned by the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, uint dwBytesToRead, byte bExtendedLenApdu,
            out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalNtagXDna_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, dwBytesToRead, bExtendedLenApdu,
                ref ppResponse, ref wRspLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the tag and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from tag.
        /// </param>
        /// <param name="bOffset">The offset from where the data should be read.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file data is returned by tag.
        ///                                 - If non-zero is provided, then data starting from offset is returned.
        /// </param>
        /// <param name="aResponse">The data returned by the tag.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is not supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, uint dwBytesToRead, out byte[] aResponse )
        {
            return IsoReadBinary ( wOption, bOffset, bSfid, dwBytesToRead, 0, out aResponse );
        }

        /// <summary>
        /// Perform ISO Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] aData )
        {
            return phalNtagXDna_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, aData,
                 ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Perform ISO Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by tag call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 tag errors.
        ///     XXXX
        ///         Depending on status codes return by tag.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] aData )
        {
            return IsoUpdateBinary ( bOffset, bSfid, 0x00, aData );
        }
        #endregion

        #region Utility
        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///                       Refer <see cref="Config"/> for supported configuration items.</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Get the configuration like Force usage of ShortLengthAPDU and to retrieve additional information."
            )
        ]
        public Status_t GetConfig ( ushort wConfig, ref ushort wValue )
        {
            return phalNtagXDna_GetConfig ( m_pDataParams, wConfig, ref wValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///                       Refer <see cref="Config"/> for supported configuration items.</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Set the configuration like Force usage of ShortLengthAPDU and to update additional information."
            )
        ]
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalNtagXDna_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_413,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Reset the authentication state. Clears existing session keys and set the state to not authenticated."
            )
        ]
        public Status_t ResetAuthentication ()
        {
            return phalNtagXDna_ResetAuthentication ( m_pDataParams );
        }

        /// <summary>
        /// Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMMac">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer_SDMMac">Key version to be used from software KeyStore.</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input. Should be 4 bytes in Length.</param>
        /// <param name="aInData">Data read out from PICC that is between SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="aSDMMAC">Secure Dynamic MAC information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type."
            )
        ]
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wKeyNo_SDMMac, ushort wKeyVer_SDMMac, byte[] aUid,
            byte[] aSDMReadCtr, byte[] aInData, out byte[] aSDMMAC )
        {
            Status_t oStatus;
            IntPtr ppSDMMACPrev = IntPtr.Zero;
            ushort wSDMMACPrevLen = 0;

            oStatus = phalNtagXDna_CalculateMACSDM ( m_pDataParams, bSdmOption, wKeyNo_SDMMac, wKeyVer_SDMMac, aUid,
                ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aInData, ( ushort ) ( ( aInData == null ) ?
                0 : aInData.Length ), ref ppSDMMACPrev, ref wSDMMACPrevLen );
            aSDMMAC = MarshalCopy ( oStatus, ppSDMMACPrev, wSDMMACPrevLen );

            return oStatus;
        }

        /// <summary>
        /// Performs computation of SDM Signature information.
        /// </summary>
        ///
        /// <param name="wPubKeyNo_SDMSig">Key number to be used from software KeyStore.</param>
        /// <param name="wPubKeyPos_SDMSig">Key position to be used from software KeyStore.</param>
        /// <param name="aInData">Data read out from PICC that is between SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="aSignature">Secure Dynamic Signature information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/> Verification of Message / Signature combination failed.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs verification of SDM Signature information."
            )
        ]
        public Status_t VerifySDMSignature ( ushort wPubKeyNo_SDMSig, ushort wPubKeyPos_SDMSig, byte[] aInData, byte[] aSignature )
        {
            return phalNtagXDna_VerifySDMSignature ( m_pDataParams, wPubKeyNo_SDMSig, wPubKeyPos_SDMSig, aInData,
                ( ushort ) ( ( aInData == null ) ? 0 : aInData.Length ), aSignature,
                ( ushort ) ( ( aSignature != null ) ? aSignature.Length : 0 ) );
        }

        /// <summary>
        /// Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMEnc">Key number to be used from software KeyStore</param>
        /// <param name="wKeyVer_SDMEnc">Key version to be used from software KeyStore</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="aEncData">Encrypted NonASCII SDM Encrypted File data.</param>
        /// <param name="aPlainData">The decrypted SDMFile data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptSDMENCFileData ( byte bSdmOption, ushort wKeyNo_SDMEnc, ushort wKeyVer_SDMEnc, byte[] aUid,
            byte[] aSDMReadCtr, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t oStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            oStatus = phalNtagXDna_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wKeyNo_SDMEnc, wKeyVer_SDMEnc, aUid,
                ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aEncData,
                ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ),
                ref ppPlainData, ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Performs decryption of SDM PICC Data. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer">Key version to be used from software KeyStore.</param>
        /// <param name="aEncData">Encrypted NonASCII SDM PICC data. </param>
        /// <param name="aPlainData">The decrypted SDMPICC data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                NTAG_Products.NTAG_X_DNA,
                NTAG_Products.NTAG_424_DNA,
                NTAG_Products.NTAG_424_DNA_TT,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs decryption of SDM PICC Data. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptSDMPICCData ( ushort wKeyNo, ushort wKeyVer, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t oStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            oStatus = phalNtagXDna_DecryptSDMPICCData ( m_pDataParams, wKeyNo, wKeyVer, aEncData,
                ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref ppPlainData,
                ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );

            return oStatus;
        }
        #endregion
        #endregion

        #region Memory Maping
        /// <summary>
        /// Handle to store the specific implementation context.
        /// </summary>
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion

        #region Certificate Structure
        /// <summary>
        /// Structure for NTAG X DNA Host / Device Certificates to be used with the below mentioned interfaces.
        /// <see cref="ISOGeneralAuthenticate"/>
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct Certificate
        {
            /// <summary>
            /// Pointer to global buffer for Host / Device End-Leaf certificate.
            /// Buffer should be pre-allocated by user and the size should be larger
            /// or equal to the certificate size. Should include 7F 21 (UnCompressed)
            /// or 7F 22 (Compressed) notation mentioned in DataSheet followed by
            /// Length of Certificate in BER-TLV notation and the actual certificate
            /// information.
            /// </summary>
            public IntPtr pEndLeaf;

            /// <summary> Length of bytes available in \b aEndLeaf buffer.</summary>
            public ushort wEndLeaf_Len;

            /// <summary>
            /// Pointer to global buffer for Host / Device Parent certificate.
            /// Buffer should be pre-allocated by user and the size should be larger
            /// or equal to the certificate size. Should include 7F 21 (UnCompressed)
            /// or 7F 22 (Compressed) notation mentioned in DataSheet followed by
            /// Length of Certificate in BER-TLV notation and the actual certificate
            /// information.
            /// </summary>
            public IntPtr pParent;

            /// <summary> Length of bytes available in \b aParent buffer.</summary>
            public ushort wParent_Len;

            /// <summary>
            /// Pointer to global buffer for Host / Device Grand-Parent certificate.
            /// Buffer should be pre-allocated by user and the size should be larger
            /// or equal to the certificate size. Should include 7F 21 (UnCompressed)
            /// or 7F 22 (Compressed) notation mentioned in DataSheet followed by
            /// Length of Certificate in BER-TLV notation and the actual certificate
            /// information.
            /// </summary>
            public IntPtr pGrandParent;

            /// <summary> Length of bytes available in \b aGrandParent buffer.</summary>
            public ushort wGrandParent_Len;
        }
        #endregion

        #region Private Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error_Gen.SUCCESS_CHAINING ) ||
                oStatus.Error.Equals ( Error.DF_7816_GEN_ERROR ) ) && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( ( pBuffer != IntPtr.Zero ) && ( dwLength != 0 ) )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }

        private byte[] EncodeCertificate ( byte bCompression, byte[] aCertificate )
        {
            byte[] aEncodedCert = new byte[2 + 3 + aCertificate.Length];
            int dwEncodedCert_Offset = 0;

            byte[] aCompression = ( bCompression == ( byte ) Certificate_CompressionType.UN_COMPRESSED ) ? new byte[] { 0x7F, 0x21 }:
                new byte[] { 0x7F, 0x22 };

            Array.Copy ( aCompression, 0, aEncodedCert, 0, aCompression.Length );
            dwEncodedCert_Offset = aCompression.Length;

            ushort wCertLen = ( ushort ) aCertificate.Length;
            Array.Copy ( new byte[] { 0x82 }, 0, aEncodedCert, dwEncodedCert_Offset, 1 );
            dwEncodedCert_Offset += 1;

            Array.Copy ( new byte[] { ( byte ) ( ( wCertLen & 0xFF00 ) >> 8 ) }, 0, aEncodedCert, dwEncodedCert_Offset, 1 );
            dwEncodedCert_Offset += 1;

            Array.Copy ( new byte[] { ( byte ) ( wCertLen & 0x00FF ) }, 0, aEncodedCert, dwEncodedCert_Offset, 1 );
            dwEncodedCert_Offset += 1;

            Array.Copy ( aCertificate, 0, aEncodedCert, dwEncodedCert_Offset, aCertificate.Length );
            dwEncodedCert_Offset += aCertificate.Length;

            return aEncodedCert;
        }
        #endregion
    }
    #endregion

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    ///
    /// Note:
    ///     - The AL component uses two internal buffers for processing. One is named as <see cref="DataParams_t.pCmdBuf">Command Buffer</see>
    ///       and another as <see cref="DataParams_t.pProcessBuf">Processing Buffer"</see>.
    ///     - Memory for these internal buffers needs to be passed by the user during initialization.
    ///     - The internal buffer size should not be less than minimum.Minimum sizes are,
    ///         <see cref="DataParams_t.pCmdBuf">Command Buffer</see> should be >= <see cref="MIN_CMD_BUFFER_SIZE"/>
    ///         <see cref="DataParams_t.pProcessBuf">Processing Buffer</see> should be >= <see cref="MIN_PRS_BUFFER_SIZE"/>
    ///         If the sizes are less than minimum, <see cref="Error_Param.PARAMETER_SIZE">Parameter Size</see> error will be returned.
    ///     - Processing buffer size should not be less than Command buffer size else <see cref="Error_Param.PARAMETER_SIZE">Parameter Size</see>
    ///       error will be returned.
    ///     - After completion of the application, call below interfaces to clear all the internal buffers, its sizes, dependent components
    ///       like CryptoSym and CryptASym etc...
    ///         <see cref="DeInit"/>
    ///     - Its must to Initialize the component again after calling De-Initialization.
    ///     - During Initialization of HAL component, make sure the transmit and response buffer size are not less than tag frame size.
    ///     - Call <see cref="Generic.GetConfig">GetConfig</see> with <see cref="Config.ADDITIONAL_INFO">Additional Information</see> as
    ///       Configuration identifier when any of the interface returns <see cref="Error.DF_GEN_ERROR">General Failure</see>
    ///       <see cref="Error.DF_7816_GEN_ERROR">ISO7816 General Failure</see>.
    /// </summary>
    public class Sw : Generic
    {
        #region Constants
        /// <summary> Minimum size for allocating the command buffer during initializing. </summary>
        public const ushort MIN_CMD_BUFFER_SIZE = 256;

        /// <summary> Minimum size for allocating the response / SM processing buffer during initializing. </summary>
        public const ushort MIN_PRS_BUFFER_SIZE = 512;

        private const byte AID_LENGTH = 3;

        private const byte NONCE_LENGTH = 13;
        private const byte TI_LENGTH = 4;
        private const byte SESSION_KEY_LENGTH = 32;
        #endregion

        #region Variables
        private byte[] aCmdBuf = null;
        private byte[] aProcessBuf = null;
        private GCHandle m_pCmdBuf;
        private GCHandle m_pProcessBuf;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for NTAG X DNA Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the parameter structure of the ASymmetric Crypto component. </summary>
            public IntPtr pCryptoDataParamsASym;

            /// <summary> Pointer to the parameter structure of the Symmetric Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary> Pointer to the parameter structure of the Symmetric Crypto layer for MACing. </summary>
            public IntPtr pCryptoDataParamsMac;

            /// <summary> Pointer to the parameter structure of the Crypto layer for Random number generation. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to global buffer for processing the command. </summary>
            public IntPtr pCmdBuf;

            /// <summary> Size of global command buffer.</summary>
            public ushort wCmdBufSize;

            /// <summary> Length of bytes available in command buffer (pCmdBuf) for processing.</summary>
            public ushort wCmdBufLen;

            /// <summary> Command Buffer offset while performing crypto or exchange operations.</summary>
            public ushort wCmdBufOffset;

            /// <summary> Pointer to global buffer for processing the response / secure messaging information. </summary>
            public IntPtr pProcessBuf;

            /// <summary> Size of global response / secure messaging information buffer.</summary>
            public ushort wPrsBufSize;

            /// <summary> Length of bytes available in response / secure messaging information buffer (pProcessBuf) for processing.</summary>
            public ushort wPrsBufLen;

            /// <summary> Processing Buffer offset while performing crypto or exchange operations.</summary>
            public ushort wPrsBufOffset;

            /// <summary> Aid of the currently selected application. </summary>
            public fixed byte pAid[AID_LENGTH];

            /// <summary> Authentication Encryption key for the session. </summary>
            public fixed byte aSesAuthENCKey[SESSION_KEY_LENGTH];

            /// <summary> Authentication MAC key for the session. </summary>
            public fixed byte aSesAuthMACKey[SESSION_KEY_LENGTH];

            /// <summary> Authentication MAC key for the session. </summary>
            public fixed byte aSesNonce[NONCE_LENGTH];

            /// <summary> Transaction Identifier. </summary>
            public fixed byte aTI[TI_LENGTH];

            /// <summary> Command count within transaction. </summary>
            public ushort wCmdCtr;

            /// <summary> Specific error codes for NTAG X DNA generic errors or To get the response length of some commands. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Parameter for force set Short Length APDU in case of BIG ISO read. </summary>
            public byte bShortLenApdu;

            /// <summary> Authenticate Command used. Can be one of the following,
            ///     0x01: EV2_Authenticated
            ///     0x02: ECC_Authenticated
            ///     0xFF: NOT_Authenticated
            /// </summary>
            public byte bAuthState;

            /// <summary> Key number against which this authentication is done. </summary>
            public byte bKeyNo;

            /// <summary> Key Type being used for Authentication. </summary>
            public ushort wKeyType;

            /// <summary>
            /// Flag to Indicate if tag data Status. Indicates the following.
            ///     tag Data is complete but there is still more data that needs to be provided to user.
            ///     tag Data is complete and there is no data to be given to user, but last encrypted chunk needs to be verified.
            /// </summary>
            public byte bPICCDataComplete;

            /// <summary> Flag to indicate if MAC needs to be verified. </summary>
            public byte bMACVerified;

            /// <summary> Flag to indicate if Encryption of Data is complete or still pending. </summary>
            public byte bIsENCPending;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_Sw_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams,
            IntPtr pKeyStoreDataParams, IntPtr pCryptoDataParamsASym, IntPtr pCryptoDataParamsEnc, IntPtr pCryptoDataParamsMac,
            IntPtr pCryptoRngDataParams, IntPtr pCmdBuf, ushort wCmdBufSize, IntPtr pPrsBuf, ushort wPrsBufSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalNtagXDna_Sw_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for NTAG X DNA software component.
        /// </summary>
        ///
        /// <param name="oPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="oKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="oCryptoASym">Pointer to the parameter structure of the ASymmetric Crypto component.</param>
        /// <param name="oCryptoEnc">Pointer to Symmetric Crypto component context for encryption.</param>
        /// <param name="oCryptoMAC">Pointer to Symmetric Crypto component context for MACing.</param>
        /// <param name="oCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="oTMIUtils">Pointer to a TMI component.</param>
        /// <param name="oAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="wCmdBufSize">Size of global command buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alNtagXDna.Sw.CMD_BUFFER_SIZE).</param>
        /// <param name="wPrsBufSize">Size of global response / secure messaging information buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alNtagXDna.Sw.PRS_BUFFER_SIZE).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( palMifare.Generic oPalMifare, KeyStore.Generic oKeyStore, CryptoASym.Generic oCryptoASym, CryptoSym.Generic oCryptoEnc,
            CryptoSym.Generic oCryptoMAC, CryptoRng.Generic oCryptoRng, ushort wCmdBufSize = MIN_CMD_BUFFER_SIZE, ushort wPrsBufSize = MIN_PRS_BUFFER_SIZE )
        {
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            aCmdBuf = new byte[wCmdBufSize];
            m_pCmdBuf = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            aProcessBuf = new byte[wPrsBufSize];
            m_pProcessBuf = GCHandle.Alloc ( aProcessBuf, GCHandleType.Pinned );

            return phalNtagXDna_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero,
                ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCryptoASym != null ) ? oCryptoASym.m_pDataParams : IntPtr.Zero,
                ( oCryptoEnc != null ) ? oCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( oCryptoMAC != null ) ? oCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero,
                aCmdBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pCmdBuf.AddrOfPinnedObject (),
                wCmdBufSize,
                aProcessBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pProcessBuf.AddrOfPinnedObject (),
                wPrsBufSize );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for NTAG X DNA software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="oCryptoASym">Pointer to the parameter structure of the ASymmetric Crypto component.</param>
        /// <param name="oCryptoEnc">Pointer to Symmetric Crypto component context for encryption.</param>
        /// <param name="oCryptoMAC">Pointer to Symmetric Crypto component context for MACing.</param>
        /// <param name="oCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="oTMIUtils">Pointer to a TMI component.</param>
        /// <param name="oAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="wCmdBufSize">Size of global command buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alNtagXDna.Sw.CMD_BUFFER_SIZE).</param>
        /// <param name="wPrsBufSize">Size of global response / secure messaging information buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alNtagXDna.Sw.PRS_BUFFER_SIZE).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoASym.Generic oCryptoASym,
            CryptoSym.Generic oCryptoEnc, CryptoSym.Generic oCryptoMAC, CryptoRng.Generic oCryptoRng, ushort wCmdBufSize = MIN_CMD_BUFFER_SIZE,
            ushort wPrsBufSize = MIN_PRS_BUFFER_SIZE )
        {
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            aCmdBuf = new byte[wCmdBufSize];
            m_pCmdBuf = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            aProcessBuf = new byte[wPrsBufSize];
            m_pProcessBuf = GCHandle.Alloc ( aProcessBuf, GCHandleType.Pinned );

            return phalNtagXDna_Sw_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCryptoASym != null ) ? oCryptoASym.m_pDataParams : IntPtr.Zero,
                ( oCryptoEnc != null ) ? oCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( oCryptoMAC != null ) ? oCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero,
                aCmdBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pCmdBuf.AddrOfPinnedObject (),
                wCmdBufSize,
                aProcessBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pProcessBuf.AddrOfPinnedObject (),
                wPrsBufSize );
        }
#endif
        #endregion

        #region De-Initialization
        /// <summary>
        /// De Initialized NTAG X DNA Software component.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DeInit ()
        {
            /* Clear all memory. */
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            return phalNtagXDna_Sw_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter and Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public override byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }

        public override byte[] SesAuthENCKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aSesAuthENCKey[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > SESSION_KEY_LENGTH )
                    throw new ArgumentException ( string.Format ( "Expected Length is {0} but received {1}.", SESSION_KEY_LENGTH, value.Length ) );

                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aSesAuthENCKey[i] = value[i];
                        }
                    }
                }
            }
        }

        public override byte[] SesAuthMACKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aSesAuthMACKey[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > SESSION_KEY_LENGTH )
                    throw new ArgumentException ( string.Format ( "Expected Length is {0} but received {1}.", SESSION_KEY_LENGTH, value.Length ) );
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aSesAuthMACKey[i] = value[i];
                        }
                    }
                }
            }
        }

        public override byte[] TI
        {
            get
            {
                byte[] bValue = new byte[4];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 4; i++ )
                        {
                            bValue[i] = pDataParams->aTI[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 4 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aTI[i] = value[i];
                        }
                    }
                }
            }
        }

        public override ushort CmdCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        return pDataParams->wCmdCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        pDataParams->wCmdCtr = value;
                    }
                }
            }
        }
        #endregion
    }
    #endregion
}
