/*
 * Copyright 2013 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alVca
{
    #region Enumerations
    #region Error
    /// <summary>
    /// Custom Error Codes
    /// </summary>
    public enum Error : byte
    {
        /// <summary> VCA Invalid Command Error. </summary>
        CMD_INVALID = ( CustomCodes.ERROR_BEGIN ),

        /// <summary> VCA Format Error. </summary>
        FORMAT,

        /// <summary> VCA AUTH Error. </summary>
        AUTH,

        /// <summary> VCA GEN Error. </summary>
        GEN,

        /// <summary> VCA OVERFLOW Error. </summary>
        OVERFLOW,

        /// <summary> VCA Command Abort Error. </summary>
        CMD_ABORT,

        /// <summary> VCA ISO7816 Generic Error. </summary>
        ISO7816_GEN_ERROR
    }
    #endregion

    #region ISOSelect
    /// <summary>
    /// Option to be used for ISOSelect command when caled for SAM X or S configuration.
    /// </summary>
    public enum ISOSelect : byte
    {
        /// <summary>
        /// Option mask to indicate the selection control as DFName.
        /// </summary>
        SELECTION_DF_NAME = 0x04,
        SELECTION_INVLAID = 0xFF,

        /// <summary>
        /// Option mask to indicate the return of FCI information.
        /// </summary>
        FCI_RETURNED = 0x00,
        FCI_INVLAID = 0xFF,

        /// <summary> Option mask for complete VC selection in 1 part. </summary>
        VARIANT_PART1 = 0x00,

        /// <summary> Option mask for complete VC selection in 2 part. </summary>
        VARIANT_PART2 = 0x08,

        /// <summary> Default option mask to disable the diversification of VcSelect MAC and ENC key. </summary>
        DIV_DISABLED = 0x00,

        /// <summary> Option mask to perform diversification of VC SelectENC key using the diversification input provided. </summary>
        ENC_KEY_DIV_INPUT = 0x01,

        /// <summary> Option mask to perform diversification of VC SelectMAC key using the diversification input provided. </summary>
        MAC_KEY_DIV_INPUT = 0x02,

        /// <summary> Option mask to perform diversification of VC SelectMAC key using Virtual Card Identifier. </summary>
        MAC_KEY_DIV_VCUID = 0x04,

        /// <summary>
        /// Option mask to perform diversification of VC SelectMAC key using Virtual Card Identifier
        /// and Diversification input provided.
        /// </summary>
        MAC_KEY_DIV_INPUT_VCUID = 0x06
    }
    #endregion

    #region ProximityCheck
    /// <summary>
    /// Option to be used for ProximityCheckNew command.
    /// </summary>
    public enum ProximityCheck : byte
    {
        /// <summary> Option mask to indicate the Random number C is given by the user. </summary>
        RNDC_PROVIDED = 0x00,

        /// <summary> Option mask to indicate the Random number C to be generated by the library. </summary>
        RNDC_GENERATE = 0x01,

        /// <summary>
        /// Option mask to indicate VerifyProcessing should be Normal.
        /// </summary>
        NORMAL_PROCESSING = 0x00,

        /// <summary> Option mask to indicate VerifyProcessing should be Random. </summary>
        RANDOM_PROCESSING = 0x02,
    }
    #endregion

    #region Config
    /// <summary>
    /// Get/ Set configuration flags.
    /// </summary>
    public enum Config : ushort
    {
        /// <summary> Option for GetConfig / SetConfig to get / set additional info of a generic error. </summary>
        ADDITIONAL_INFO = 0x0001,

        /// <summary> Option for GetConfig / SetConfig to get / set Wrapped mode. </summary>
        WRAPPED_MODE = 0x0002,

        /// <summary> Option for GetConfig / SetConfig to get / set Timing measurements ON/OFF. </summary>
        TIMING_MODE = 0x0003,

        /// <summary> Option for GetConfig / SetConfig to Enable / Diable lower boundary(-20%) on threshold time. </summary>
        LOWER_THRESHOLD = 0x0004,

        /// <summary> Option for GetConfig / SetConfig to get / set current status of extended wrapping in ISO 7816-4 APDUs. </summary>
        EXTENDED_APDU = 0x0005
    }
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    public abstract class Generic
    {
        #region DLL Imports
        #region Virtual Card
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_StartCardSelection ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_FinalizeCardSelection ( IntPtr pDataParams, ref ushort pNumValidIids );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SelectVc ( IntPtr pDataParams, ushort wValidIidIndex, ushort wKeyNumber, ushort wKeyVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_DeselectVc ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_VcSupport ( IntPtr pDataParams, byte[] pIid, ushort wKeyEncNumber, ushort wKeyEncVersion, ushort wKeyMacNumber,
            ushort wKeyMacVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_VcSupportLast ( IntPtr pDataParams, byte[] pIid, byte bLenCap, byte[] pPcdCapabilities, ushort wKeyEncNumber,
            ushort wKeyEncVersion, ushort wKeyMacNumber, ushort wKeyMacVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_GetIidInfo ( IntPtr pDataParams, ushort wValidIidIndex, ref ushort pIidIndex, ref byte pVcUidSize, byte[] pVcUid,
            ref byte pInfo, byte[] pPdCapabilities );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_IsoSelect ( IntPtr pDataParams, byte bSelectionControl, byte bOption, byte bDFnameLen, byte[] pDFname, byte[] pDivInput,
            byte bDivInputLen, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer, byte[] pResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_IsoExternalAuthenticate ( IntPtr pDataParams, byte[] pInData, ushort wKeyNumber, ushort wKeyVersion );
        #endregion Virtual Card

        #region ProximityCheck
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_ProximityCheck ( IntPtr pDataParams, byte bGenerateRndC, byte[] pRndC, byte bPps1, byte bNumSteps,
            ushort wKeyNumber, ushort wKeyVersion, byte[] pUsedRndC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_ProximityCheckNew ( IntPtr pDataParams, byte bGenerateRndC, byte[] pPrndC, byte bNumSteps, ushort wKeyNumber,
            ushort wKeyVersion, byte[] pDivInput, byte bDivInputLen, ref byte pOption, byte[] pPubRespTime, byte[] pResponse, ref ushort pRespLen,
            byte[] pCumRndRC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_PrepareProximityCheckNew ( IntPtr pDataParams, ref byte pOption, byte[] pPubRespTime, byte[] pResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_ExecuteProximityCheckNew ( IntPtr pDataParams, byte bGenerateRndC, byte[] pPrndC, byte bNumSteps, byte[] pPubRespTime,
            byte[] pCumRndRC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_VerifyProximityCheckNew ( IntPtr pDataParams, byte bOption, byte[] pPubRespTime, byte[] aResponse, ushort wRespLen,
            ushort wKeyNumber, ushort wKeyVersion, byte[] pRndCmdResp );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_VerifyProximityCheckUtility ( IntPtr pDataParams, byte[] pCmdMac, byte[] pCmdResp );
        #endregion ProximityCheck

        #region Miscellaneous
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SetApplicationType ( IntPtr pDataParams, IntPtr pAlDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_DecryptResponse ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVersion, byte[] pInData, byte[] pRandChal, byte[] pVCData );
        #endregion Miscellaneous
        #endregion

        #region Wrapper Functions
        #region Virtual Card
        /// <summary>
        /// Start Card Selection.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t StartCardSelection ()
        {
            return phalVca_StartCardSelection ( m_pDataParams );
        }

        /// <summary>
        /// Finalize Card Selection.
        /// </summary>
        ///
        /// <param name="pNumValidIids">Number of valid IIDs detected during VCS/VCSL sequence execution.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t FinalizeCardSelection ( out int pNumValidIids )
        {
            ushort pNumValidIids_Int = 0;

            Status_t oStatus = phalVca_FinalizeCardSelection ( m_pDataParams, ref pNumValidIids_Int );

            pNumValidIids = ( int ) pNumValidIids_Int;

            return oStatus;
        }

        /// <summary>
        /// Performs a Select Virtual Card command (MIFARE Plus X only).
        /// </summary>
        ///
        /// <param name="wValidIidIndex">Index of valid IID, needs to be < pNumValidIids in FinalizeCardSelection call.</param>
        /// <param name="wKeyNumber">Key number of the MAC key used in SVC command.</param>
        /// <param name="wKeyVersion">Key version of the MAC key used in SVC command.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SelectVc ( int wValidIidIndex, int wKeyNumber, int wKeyVersion )
        {
            return phalVca_SelectVc ( m_pDataParams, ( ushort ) wValidIidIndex, ( ushort ) wKeyNumber, ( ushort ) wKeyVersion );
        }

        /// <summary>
        /// Performs a Deselect Virtual Card command.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DeselectVc ()
        {
            return phalVca_DeselectVc ( m_pDataParams );
        }

        /// <summary>
        /// Performs a Virtual Card Support command (MIFARE Plus X only).
        /// </summary>
        ///
        /// <param name="pIid">Installation Identifier (16 bytes).</param>
        /// <param name="wKeyEncNumber">Key number of the ENC key associated to the Iid.</param>
        /// <param name="wKeyEncVersion">Key version of the ENC key associated to the Iid.</param>
        /// <param name="wKeyMacNumber">Key number of the MAC key associated to the Iid.</param>
        /// <param name="wKeyMacVersion">Key version of the MAC key associated to the Iid.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t VcSupport ( byte[] pIid, int wKeyEncNumber, int wKeyEncVersion, int wKeyMacNumber, int wKeyMacVersion )
        {
            return phalVca_VcSupport ( m_pDataParams, pIid, ( ushort ) wKeyEncNumber, ( ushort ) wKeyEncVersion, ( ushort ) wKeyMacNumber, ( ushort ) wKeyMacVersion );
        }

        /// <summary>
        /// Perform a Virtual Card Support (Last) command. This cimmand performs the Virtual card support command in
        /// ISO14443 Layer 3 activated state. This comand is supported by Mifare Plus Ev1 product.
        /// </summary>
        ///
        /// <param name="pIid">Installation Identifier (16 bytes).</param>
        /// <param name="pPcdCapabilities">PCD Capabilities (bLenCap bytes), ignored if bLenCap == 0.</param>
        /// <param name="wKeyEncNumber">Key number of the ENC key associated to the Iid.</param>
        /// <param name="wKeyEncVersion">Key version of the ENC key associated to the Iid.</param>
        /// <param name="wKeyMacNumber">Key number of the MAC key associated to the Iid.</param>
        /// <param name="wKeyMacVersion">Key version of the MAC key associated to the Iid.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t VcSupportLast ( byte[] pIid, byte[] pPcdCapabilities, int wKeyEncNumber, int wKeyEncVersion, int wKeyMacNumber, int wKeyMacVersion )
        {
            return phalVca_VcSupportLast ( m_pDataParams, pIid, ( byte ) ( ( pPcdCapabilities == null ) ? 0 : pPcdCapabilities.Length ), pPcdCapabilities, ( ushort ) wKeyEncNumber, ( ushort ) wKeyEncVersion, ( ushort ) wKeyMacNumber, ( ushort ) wKeyMacVersion );
        }

        /// <summary>
        /// Retrieve card information.
        /// </summary>
        ///
        /// <param name="wValidIidIndex">Index of valid IID, needs to be < pNumValidIids in FinalizeCardSelection call.</param>
        /// <param name="pIidIndex">Corresponding IID to the key pair where the MAC was matching in VCSL command call</param>
        /// <param name="pVcUid">VC UID (pVcUidSize bytes).</param>
        /// <param name="pInfo">Info byte. </param>
        /// <param name="pPdCapabilities">PD Capabilities (2 bytes).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetIidInfo ( int wValidIidIndex, out int pIidIndex, out byte[] pVcUid, out byte pInfo, out byte[] pPdCapabilities )
        {
            byte vcUidSize = 0;
            ushort pIidIndex_Int = 0;
            pVcUid = new byte[7];
            pPdCapabilities = new byte[2];
            pIidIndex = 0;
            pInfo = 0x00;

            Status_t oStatus = phalVca_GetIidInfo ( m_pDataParams, ( ushort ) wValidIidIndex, ref pIidIndex_Int, ref vcUidSize, pVcUid, ref pInfo, pPdCapabilities );
            pIidIndex = ( int ) pIidIndex_Int;

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                System.Array.Resize<byte> ( ref pVcUid, vcUidSize );
            }

            return oStatus;
        }

        /// <summary>
        /// PCD explicitly indicates which Virtual Card it wants to target by issuing this command.
        /// Both PCD and the PD agree on which VC to use (if any) and they negotiate the capabilities
        /// that they will use.
        ///
        /// Note: This interface is for Software component only.
        /// </summary>
        ///
        /// <param name="bSelectionControl">Value equals NxpRdLibNet.alVca.ISOSelect.SELECTION_DF_NAME which indicates VC Selection by DF name.
        ///									Any other value results in INVALID_PARAMETER.</param>
        /// <param name="bOption">Value equals NxpRdLibNet.alVca.ISOSelect.FCI_RETURNED which indicates FCI template is returned.
        ///						  Any other value results in INVALID_PARAMETER
        /// <param name="pDFname">This is the IID which is DFName string upto 16 Bytes. Valid only when bSelectionControl = 0x04.</param>
        /// <param name="pFCI">File control information of 36 bytes and response code of 2 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelect ( byte bSelectionControl, byte bOption, byte[] pDFname, out byte[] pFCI )
        {
            ushort wRespLen = 0;

            pFCI = new byte[64];
            Status_t oStatus = phalVca_IsoSelect ( m_pDataParams, bSelectionControl, bOption, ( byte ) ( ( pDFname == null ) ? 0 : pDFname.Length ), pDFname,
                null, 0x00, 0xFF, /* Used in SAM, Not in SW */ 0xFF, /* Used in SAM, Not in SW */ 0xFF, /* Used in SAM, Not in SW */ 0xFF, /* Used in SAM, Not in SW */
				pFCI, ref wRespLen );

            /* Resizing of array pFCI is done to update the size as per the value of pwFCILen since pwFCILen can be different for each case of IsoSelect.
			 * It is 32 for response data case I and II and 36 for case III as per ref arch v17 */
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) ) && ( wRespLen != 0x00 ) )
            {
                System.Array.Resize<byte> ( ref pFCI, wRespLen );
            }
            else
            {
                /* Set the FCI buffer to NULL */
                pFCI = null;
            }

            return oStatus;
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// PCD explicitly indicates which Virtual Card it wants to target by issuing this command.
        /// Both PCD and the PD agree on which VC to use (if any) and they negotiate the capabilities
        /// that they will use.
        ///
        /// Note: This interface is for SAM component only.
        /// </summary>
        ///
        /// <param name="bSelectionControl">Value equals NxpRdLibNet.alVca.ISOSelect.SELECTION_DF_NAME which indicates VC Selection by DF name.
        ///									Any other value results in INVALID_PARAMETER.
        ///
        ///									Note: This parameter will be used for Software mode only.
        ///									For Sam X mode it can be neglected.</param>
        /// <param name="bOption">Value equals NxpRdLibNet.alVca.ISOSelect.FCI_RETURNED which indicates FCI template is returned.
        ///						  Any other value results in INVALID_PARAMETER.
        ///						  Note: This value will be used for Software mode only.
        ///
        ///						  Sam S or X communication, The macros can be combined by oring.
        ///							NxpRdLibNet.alVca.ISOSelect.DIV_DISABLED
        ///							NxpRdLibNet.alVca.ISOSelect.ENC_KEY_DIV_INPUT
        ///							NxpRdLibNet.alVca.ISOSelect.MAC_KEY_DIV_INPUT
        ///							NxpRdLibNet.alVca.ISOSelect.MAC_KEY_DIV_VCUID
        ///							NxpRdLibNet.alVca.ISOSelect.MAC_KEY_DIV_INPUT_VCUID
        ///
        ///							The below onse are valid for Sam X only. To be used with the bove ones.
        ///							NxpRdLibNet.alVca.ISOSelect.VARIANT_PART1
        ///							NxpRdLibNet.alVca.ISOSelect.VARIANT_PART2
        /// <param name="pDFname">This is the IID which is DFName string upto 16 Bytes. Valid only when bSelectionControl = 0x04.</param>
        /// <param name="pDivInput">Key diversification input. This is used for SAM Key diversification</param>
        /// <param name="bEncKeyNo">ENC Key number in Software or SAM keystore.</param>
        /// <param name="bEncKeyVer">ENC Key version in Software or SAM keystore.</param>
        /// <param name="bMacKeyNo">MAC Key number in Software or SAM keystore.</param>
        /// <param name="bMacKeyVer">MAC Key version in Software or SAM keystore.</param>
        /// <param name="pFCI">File control information of 36 bytes and response code of 2 bytes.</param>
        /// <param name="pResponse">Response as mentioned below.
        ///								Software: Response received from PICC.
        ///								SAM S   : 8 bytes MAC which should be used for ISOExternalAuthenticate +
        ///										  Decrypted Virtual Card Data (or) FileIdentifier.
        ///								SAM X   : Virtual Card Data or FCI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelect ( byte bSelectionControl, byte bOption, byte[] aDFname, byte[] pDivInput, byte bEncKeyNo, byte bEncKeyVer,
            byte bMacKeyNo, byte bMacKeyVer, out byte[] aResponse )
        {
            ushort wRespLen = 0x00;
            aResponse = new byte[64];

            Status_t oStatus = phalVca_IsoSelect ( m_pDataParams, bSelectionControl, bOption, ( byte ) ( ( aDFname == null ) ? 0 : aDFname.Length ),
                aDFname, pDivInput, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer,
                aResponse, ref wRespLen );

            /* Resizing of Response array is done to update the size as per the RespLen value since wRespLen can be
			 * different for each case of IsoSelect.
			 */
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) ) && ( wRespLen != 0x00 ) )
            {
                System.Array.Resize<byte> ( ref aResponse, wRespLen );
            }
            else
            {
                /* Set the respone buffer to NULL */
                aResponse = null;
            }

            return oStatus;
        }
#endif

        /// <summary>
        /// Depending on the VC configuration, the PCD might be required to subsequently (after IsoSelect execution)
        /// authenticate itself via this command before the targeted VC becomes selected.
        /// </summary>
        ///
        /// <param name="pInData">Input buffer to be passed for authentication.
        ///							Software: Response data ( RndChl||VCData) of IsoSelect command.
        ///							SAM S   : Response data (MAC + Decrypted VCData) of IsoSelect command.
        ///							SAM X   : Not Applicable.This command is not supported for X mode.</param>
        /// <param name="wKeyNumber">Key number of the VCSelect MAC key. Only applicable for Software.</param>
        /// <param name="wKeyVersion">Key vesion of the VCSelect MAC key. Only applicable for Software.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoExternalAuthenticate ( byte[] pInData, ushort wKeyNumber, ushort wKeyVersion )
        {
            return phalVca_IsoExternalAuthenticate ( m_pDataParams, pInData, wKeyNumber, wKeyVersion );
        }
        #endregion Virtual Card

        #region ProximityCheck
        /// <summary>
        /// Performs the whole Proximity Check command chain which calls all the three commands(PreparePC, ExecutePC and VerifyPC) together.
        /// </summary>
        ///
        /// <param name="bGenerateRndC">0: RndC provided; 1: generate RndC;</param>
        /// <param name="pRndC">Provided RndC (7 bytes), ignored if bGenerateRndC == 1.</param>
        /// <param name="bPps1">Speed(PPS1). This indicates communication data rate between PD and PCD.</param>
        /// <param name="bNumSteps">Number of ProximityCheck cycles; RFU, must be 1.</param>
        /// <param name="wKeyNumber">Proximity Check MAC Key number.</param>
        /// <param name="wKeyVersion">Proximity Check MAC Key version.</param>
        /// <param name="pUsedRndRC">Used RndC (7 bytes), can be NULL.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ProximityCheck ( byte bGenerateRndC, byte[] pRndC, byte bPps1, byte bNumSteps, int wKeyNumber, int wKeyVersion, out byte[] pUsedRndRC )
        {
            pUsedRndRC = new byte[14];
            return phalVca_ProximityCheck ( m_pDataParams, bGenerateRndC, pRndC, bPps1, bNumSteps, ( ushort ) wKeyNumber, ( ushort ) wKeyVersion, pUsedRndRC );
        }

        /// <summary>
        /// Performs the whole Proximity Check New command chain which calls all the three commands(PreparePC, ExecutePC and VerifyPC)
        /// together. This command is supported by Desfire EV2 and Mifare Plus Ev1 product only.
        /// </summary>
        ///
        /// <param name="bGenerateRndC">One of the below Option values.
        ///								For Software and SAM S mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_PROVIDED
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE
        ///
        ///								For Sam X mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.NORMAL_PROCESSING
        ///									NxpRdLibNet.alVca.ProximityCheck.RANDOM_PROCESSING</param>
        /// <param name="pPrndC">One of the below contents.
        ///							For Software mode and SAM S, if NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE the buffer can be null
        ///							else the random number should be provided by the user.
        ///
        ///							For Sam X mode, the buffer will not be utilized as the random number
        ///							will be generated by SAM.</param>
        /// <param name="bNumSteps">Number of ProximityCheck cycles. The value should be from 1 - 8.</param>
        /// <param name="wKeyNumber">Proximity Check MAC Key number in Software keystore.</param>
        /// <param name="wKeyVersion">Proximity Check MAC Key version in Software keystore.</param>
        /// <param name="pOption">Option field defining subsequent response content.
        ///							bit[0] of bOption denotes presence of PPS.</param>
        /// <param name="pPubRespTime">Published response time: time in microseconds. The PD will transmit the
        ///							   Cmd.ProximityCheck response as close as possible to this time.</param>
        /// <param name="pPps1">Speed(PPS1). This indicates communication data rate between PD and PCD.</param>
        /// <param name="pCumRndRC">Combination of both pRndCmd (cumulative of 8 bytes of
        ///							Random data sent from PCD) and pRndResp(cumulative of 8 bytes response
        ///							data received from PD).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ProximityCheckNew ( byte bGenerateRndC, byte[] pPrndC, byte bNumSteps, int wKeyNumber, int wKeyVersion, ref byte pOption,
            out byte[] pPubRespTime, ref byte pPps1, out byte[] pCumRndRC )
        {
            pCumRndRC = new byte[16];
            pPubRespTime = new byte[2];
            byte[] aResponse = new byte[1];
            ushort wRespLen = 0;

            Status_t oStatus =  phalVca_ProximityCheckNew ( m_pDataParams, bGenerateRndC, pPrndC, bNumSteps, ( ushort ) wKeyNumber,
                ( ushort ) wKeyVersion, null, 0x00, ref pOption, pPubRespTime, aResponse, ref wRespLen, pCumRndRC );

            pOption = ( byte ) ( pOption & 0x01 );
            if ( oStatus.Equals ( new Status_t () ) && !wRespLen.Equals ( 0 ) )
            {
                pPps1 = aResponse[0];
                pOption = ( byte ) ( pOption & 0x01 );
            }
            else
            {
                pPubRespTime = null;
                pCumRndRC = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs the whole Proximity Check New command chain which calls all the three commands(PreparePC, ExecutePC and VerifyPC) together.
        /// This command is supported by Desfire EV2 and Mifare Plus Ev1 product only.
        /// </summary>
        ///
        /// <param name="bGenerateRndC">One of the below Option values.
        ///								For Software and SAM S mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_PROVIDED
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE
        ///
        ///								For Sam X mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.NORMAL_PROCESSING
        ///									NxpRdLibNet.alVca.ProximityCheck.RANDOM_PROCESSING</param>
        /// <param name="pPrndC">One of the below contents.
        ///							For Software mode and SAM S, if NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE the buffer can be null
        ///							else the random number should be provided by the user.
        ///
        ///							For Sam X mode, the buffer will not be utilized as the random number
        ///							will be generated by SAM.</param>
        /// <param name="bNumSteps">Number of ProximityCheck cycles. The value should be from 1 - 8.</param>
        /// <param name="wKeyNumber">Proximity Check MAC Key number in Software keystore.</param>
        /// <param name="wKeyVersion">Proximity Check MAC Key version in Software keystore.</param>
        /// <param name="pOption">Option field defining subsequent response content.
        ///							bit[0] of bOption denotes presence of PPS.
        ///							bit[1] of bOption denotes presence of ActBitRate.</param>
        /// <param name="pPubRespTime">Published response time: time in microseconds. The PD will transmit the Cmd.ProximityCheck response as close as possible to this time.</param>
        /// <param name="aResponse">The response received by the PICC. Can be
        ///								Speed(PPS1). This indicates communication data rate between PD and PCD. OR
        ///								ActBitRate information.</param>
        /// <param name="pCumRndRC">Combination of both pRndCmd(cumulative of 8 bytes of Random data sent from PCD) and pRndResp(cumulative of 8 bytes response data received from PD).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ProximityCheckNew ( byte bGenerateRndC, byte[] pPrndC, byte bNumSteps, int wKeyNumber, int wKeyVersion, ref byte pOption,
            out byte[] pPubRespTime, out byte[] aResponse, out byte[] pCumRndRC )
        {
            pCumRndRC = new byte[16];
            pPubRespTime = new byte[2];
            ushort wRespLen = 0;

            aResponse = new byte[20];
            Status_t oStatus =  phalVca_ProximityCheckNew ( m_pDataParams, bGenerateRndC, pPrndC, bNumSteps, ( ushort ) wKeyNumber, ( ushort ) wKeyVersion, null, 0x00, ref pOption,
                pPubRespTime, aResponse, ref wRespLen, pCumRndRC );

            if ( oStatus.Equals ( new Status_t () ) && !wRespLen.Equals ( 0 ) )
            {
                Array.Resize ( ref aResponse, wRespLen );
            }
            else
            {
                pCumRndRC = null;
                pPubRespTime = null;
                aResponse = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs the whole Proximity Check New command chain which calls all the three commands(PreparePC, ExecutePC and VerifyPC) together.
        /// This command is supported by Desfire EV2 and Mifare Plus Ev1 product only. This method is for SAM S and X mode only.
        /// </summary>
        ///
        /// <param name="bOption">For Software and SAM S mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_PROVIDED
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE
        ///
        ///								For Sam X mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.NORMAL_PROCESSING
        ///									NxpRdLibNet.alVca.ProximityCheck.RANDOM_PROCESSING</param>
        /// <param name="pRndC">One of the below contents.
        ///							For SAM S mode, if NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE the buffer can be null
        ///							else the random number should be provided by the user.
        ///
        ///							For Sam X mode, the buffer will not be utilized as the random number
        ///							will be generated by SAM.</param>
        /// <param name="bNumSteps">Number of ProximityCheck cycles. The value should be from 1 - 8.</param>
        /// <param name="wKeyNo">Proximity Check MAC Key number in Software keystore</param>
        /// <param name="wKeyVer">Proximity Check MAC Key version in Software keystore</param>
        /// <param name="aDivInput">Diversification input in case of SAM. For software it should be null.</param>
        /// <param name="bOption_PICC">Option field defining subsequent response content.
        ///								bit[0] of bOption denotes presence of pPPS.</param>
        /// <param name="aPubRespTime">Published response time: time in microseconds. The PD will transmit the
        ///							   Cmd.ProximityCheck response as close as possible to this time.</param>
        /// <param name="bPps1">Speed(PPS1). This indicates communication data rate between PD and PCD.</param>
        /// <param name="aCumRndRC">One of the below response variants.
        ///								For Sam S Mode: Combination of both pRndCmd (cumulative of 8 bytes of
        ///								Random data sent from PCD) and pRndResp(cumulative of 8 bytes response
        ///								data received from PD).
        ///
        ///								For Sam X: NULL.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ProximityCheckNew ( byte bOption, byte[] aPrndC, byte bNumSteps, ushort wKeyNo, ushort wKeyVer, byte[] aDivInput,
            ref byte bOption_PICC, out byte[] aPubRespTime, ref byte bPps1, out byte[] aCumRndRC )
        {
            aCumRndRC = new byte[16];
            aPubRespTime = new byte[2];
            byte[] aResponse = new byte[1];
            ushort wRespLen = 0;

            Status_t oStatus =  phalVca_ProximityCheckNew ( m_pDataParams, bOption, aPrndC, bNumSteps, ( ushort ) wKeyNo, ( ushort ) wKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bOption_PICC, aPubRespTime, aResponse, ref wRespLen, aCumRndRC );

            if ( oStatus.Equals ( new Status_t () ) && !wRespLen.Equals ( 0 ) )
            {
                bOption_PICC = ( byte ) ( bOption_PICC & 0x01 );
                bPps1 = aResponse[0];
            }
            else
            {
                aPubRespTime = null;
                aCumRndRC = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs the whole Proximity Check New command chain which calls all the three commands(PreparePC, ExecutePC and VerifyPC) together.
        /// This command is supported by Desfire EV2 and Mifare Plus Ev1 product only. This method is for SAM S and X mode only.
        /// </summary>
        ///
        /// <param name="bOption">For Software and SAM S mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_PROVIDED
        ///									NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE
        ///
        ///								For Sam X mode.
        ///									NxpRdLibNet.alVca.ProximityCheck.NORMAL_PROCESSING
        ///									NxpRdLibNet.alVca.ProximityCheck.RANDOM_PROCESSING</param>
        /// <param name="pRndC">One of the below contents.
        ///							For SAM S mode, if NxpRdLibNet.alVca.ProximityCheck.RNDC_GENERATE the buffer can be null
        ///							else the random number should be provided by the user.
        ///
        ///							For Sam X mode, the buffer will not be utilized as the random number
        ///							will be generated by SAM.</param>
        /// <param name="bNumSteps">Number of ProximityCheck cycles. The value should be from 1 - 8.</param>
        /// <param name="wKeyNo">Proximity Check MAC Key number in Software keystore</param>
        /// <param name="wKeyVer">Proximity Check MAC Key version in Software keystore</param>
        /// <param name="aDivInput">Diversification input in case of SAM. For software it should be null.</param>
        /// <param name="bOption_PICC">Option field defining subsequent response content.
        ///							bit[0] of bOption denotes presence of PPS.
        ///							bit[1] of bOption denotes presence of ActBitRate.</param>
        /// <param name="aPubRespTime">Published response time: time in microseconds. The PD will transmit the Cmd.ProximityCheck response as close as possible to this time.</param>
        /// <param name="aResponse">The response received by the PICC. Can be
        ///								Speed(PPS1). This indicates communication data rate between PD and PCD. OR
        ///								ActBitRate information.</param>
        /// <param name="aCumRndRC">One of the below response variants.
        ///								For Sam S Mode: Combination of both pRndCmd (cumulative of 8 bytes of
        ///								Random data sent from PCD) and pRndResp(cumulative of 8 bytes response
        ///								data received from PD).
        ///
        ///								For Sam X: NULL.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ProximityCheckNew ( byte bOption, byte[] aPrndC, byte bNumSteps, int wKeyNo, int wKeyVer, byte[] aDivInput,
            ref byte bOption_PICC, out byte[] aPubRespTime, out byte[] aResponse, out byte[] aCumRndRC )
        {
            aCumRndRC = new byte[16];
            aPubRespTime = new byte[2];
            aResponse = new byte[20];

            ushort wRespLen = 0;
            Status_t oStatus =  phalVca_ProximityCheckNew ( m_pDataParams, bOption, aPrndC, bNumSteps, ( ushort ) wKeyNo, ( ushort ) wKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bOption_PICC, aPubRespTime, aResponse, ref wRespLen, aCumRndRC );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                Array.Resize ( ref aResponse, wRespLen );
            }
            else
            {
                aCumRndRC = null;
                aPubRespTime = null;
                aResponse = null;
            }

            return oStatus;
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="pOption"></param>
        /// <param name="pPubRespTime"></param>
        /// <param name="pPPS"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t PrepareProximityCheckNew ( ref byte pOption, out byte[] pPubRespTime, ref byte pPPS )
        {
            pPubRespTime = new byte[2];
            byte[] aResponse = new byte[1];
            ushort wRespLen = 0;

            pPPS = 0;
            Status_t oStatus =  phalVca_PrepareProximityCheckNew ( m_pDataParams, ref pOption, pPubRespTime, aResponse, ref wRespLen );

            if ( oStatus.Equals ( new Status_t () ) && !wRespLen.Equals ( 0 ) )
            {
                pPPS = aResponse[0];
                pOption = ( byte ) ( pOption & 0x01 );
            }

            return oStatus;
        }

        /// <summary>
        /// "PrepareProximityCheckNew" command shall prepare the card by generating the Option, publish response time and optional PPS.
        /// This command is supported by Desfire EV2 and Mifare Plus Ev1 product only.
        /// </summary>
        ///
        /// <param name="bOption">Option field defining subsequent response content
        ///							bit[0] of bOption denotes presence of PPS
        ///							bit[1] of bOption denotes presence of ActBitRate</param>
        /// <param name="aPubRespTime">Published response time: time in microseconds. The PD will transmit the
        ///							   Cmd.ProximityCheck response as close as possible to this time.</param>
        /// <param name="aResponse">Speed(PPS1). This indicates communication data rate between PD and PCD. OR
        ///							ActBitRate information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t PrepareProximityCheckNew ( ref byte bOption, out byte[] aPubRespTime, out byte[] aResponse )
        {
            aPubRespTime = new byte[2];
            aResponse = new byte[20];
            ushort wRespLen = 0;

            Status_t oStatus =  phalVca_PrepareProximityCheckNew ( m_pDataParams, ref bOption, aPubRespTime, aResponse, ref wRespLen );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                Array.Resize ( ref aResponse, wRespLen );
            }
            else
            {
                aPubRespTime = null;
                aResponse = null;
            }
            return oStatus;
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="bGenerateRndC"></param>
        /// <param name="pPrndC"></param>
        /// <param name="bNumSteps"></param>
        /// <param name="pPubRespTime"></param>
        /// <param name="pCumRndRC"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ExecuteProximityCheckNew ( byte bGenerateRndC, byte[] pPrndC, byte bNumSteps, byte[] pPubRespTime, out byte[] pCumRndRC )
        {
            pCumRndRC = new byte[16];
            return phalVca_ExecuteProximityCheckNew ( m_pDataParams, bGenerateRndC, pPrndC, bNumSteps, pPubRespTime, pCumRndRC );
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="bOption"></param>
        /// <param name="pPubRespTime"></param>
        /// <param name="bPps1"></param>
        /// <param name="wKeyNumber"></param>
        /// <param name="wKeyVersion"></param>
        /// <param name="pRndCmdResp"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t VerifyProximityCheckNew ( byte bOption, byte[] pPubRespTime, byte bPps1, ushort wKeyNumber, ushort wKeyVersion, byte[] pRndCmdResp )
        {
            return phalVca_VerifyProximityCheckNew ( m_pDataParams, bOption, pPubRespTime, new byte[] { bPps1 }, 1, wKeyNumber, wKeyVersion, pRndCmdResp );
        }

        /// <summary>
        /// "VerifyProximityCheckNew" command verifies all random numbers received after execution of command "ExecuteProximityCheckNew" using cryptographic methods.
        /// This command is supported by Desfire EV2 and Mifare Plus Ev1 product only.
        /// </summary>
        ///
        /// <param name="pOption">Option field defining subsequent response content
        ///							bit[0] of bOption denotes presence of PPS
        ///							bit[1] of bOption denotes presence of ActBitRate</param>
        /// <param name="aPubRespTime">Published response time: time in microseconds. The PD will transmit the Cmd.ProximityCheck response as close as
        ///							   possible to this time.</param>
        /// <param name="aResponse">Speed(PPS1). This indicates communication data rate between PD and PCD. OR
        ///							ActBitRate information.</param>
        /// <param name="wKeyNo">Proximity Check MAC Key number.</param>
        /// <param name="wKeyVer">Proximity Check MAC Key version.</param>
        /// <param name="aRndCmdResp">Combination of both pRndCmd(cumulative of 8 bytes of Random data sent from PCD) and pRndResp(cumulative of 8 bytes
        ///							  response data received from PD).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t VerifyProximityCheckNew ( byte bOption, byte[] aPubRespTime, byte[] aResponse, ushort wKeyNo, ushort wKeyVer, byte[] aRndCmdResp )
        {
            return phalVca_VerifyProximityCheckNew ( m_pDataParams, bOption, aPubRespTime, aResponse, ( ushort ) ( ( aResponse == null ) ? 0 : aResponse.Length ),
                wKeyNo, wKeyVer, aRndCmdResp );
        }

        /// <summary>
        ///
        /// This utility function is designed to meet the REC requirements
        /// </summary>
        ///
        /// <param name="bRndCLen"></param>
        /// <param name="pPrndC"></param>
        /// <param name="pPubRespTime"></param>
        /// <param name="pCumRndRC"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ExecuteProximityCheckUtility ( byte bRndCLen, byte[] pPrndC, byte[] pPubRespTime, out byte[] pCumRndRC )
        {
            pCumRndRC = new byte[16];
            byte bGenerateRndC = 0x00;
            byte bNumSteps = bRndCLen;
            return phalVca_ExecuteProximityCheckNew ( m_pDataParams, bGenerateRndC, pPrndC, bNumSteps, pPubRespTime, pCumRndRC );
        }

        /// <summary>
        ///
        /// This utility function is designed to meet the REC requirements
        /// </summary>
        ///
        /// <param name="pCmdMac"></param>
        /// <param name="pCmdResp"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t VerifyProximityCheckUtility ( byte[] pCmdMac, out byte[] pCmdResp )
        {
            pCmdResp = new byte[8];
            return phalVca_VerifyProximityCheckUtility ( m_pDataParams, pCmdMac, pCmdResp );
        }
        #endregion ProximityCheck

        #region Miscellaneous
        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="wConfig"></param>
        /// <param name="wValue"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( Config wConfig, ushort wValue )
        {
            return phalVca_SetConfig ( m_pDataParams, ( ushort ) wConfig, wValue );
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="wConfig"></param>
        /// <param name="pValue"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( Config wConfig, ref ushort pValue )
        {
            return phalVca_GetConfig ( m_pDataParams, ( ushort ) wConfig, ref pValue );
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="pAlDataParams"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetApplicationType ( IntPtr pAlDataParams )
        {
            return phalVca_SetApplicationType ( m_pDataParams, pAlDataParams );
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="wKeyNo"></param>
        /// <param name="wKeyVersion"></param>
        /// <param name="pInData"></param>
        /// <param name="pRandChal"></param>
        /// <param name="pVCData"></param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptResponse ( ushort wKeyNo, ushort wKeyVersion, byte[] pInData, out byte[] pRandChal, out byte[] pVCData )
        {
            pRandChal = new byte[16];
            pVCData = new byte[16];

            return phalVca_DecryptResponse ( m_pDataParams, wKeyNo, wKeyVersion, pInData, pRandChal, pVCData );
        }
        #endregion Miscellaneous
        #endregion

        #region Memory Maping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }

    #endregion

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    public class Sw : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for IID table.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct IidTableEntry_t
        {
            /// <summary> Index of the associated IID. </summary>
            public ushort wIidIndex;

            /// <summary> Key number for Encryption key. </summary>
            public ushort wKeyEncNumber;

            /// <summary> Key version for Encryption key. </summary>
            public ushort wKeyEncVersion;

            /// <summary> Key number for MAC key. </summary>
            public ushort wKeyMacNumber;

            /// <summary> Key version for MAC key. </summary>
            public ushort wKeyMacVersion;
        };

        /// <summary>
        /// Data structure for Card table.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct CardTableEntry_t
        {
            /// <summary> Index of the associated IID. </summary>
            public ushort wIidIndex;

            /// <summary> Indicates if an entry is valid or not. </summary>
            public byte bValid;

            /// <summary> Card Data received from the Card. </summary>
            public fixed byte bCardData[16];
        };

        /// <summary>
        /// Enum defining VC states.
        /// </summary>
        public enum phalVca_VirtualCardState : int
        {
            /// <summary> State to indicate no VC selection. </summary>
            VC_NOT_SELECTED = 0x00,

            /// <summary> State to indicate Proximity Check Failure. </summary>
            VC_PROXIMITYFAILED,

            /// <summary> State to indicate Proximity Check. </summary>
            VC_PROXIMITYCHECK,

            /// <summary> State to indicate DESFire PICC in not authenticaed. </summary>
            VC_DF_NOT_AUTH,

            /// <summary> State to indicate DESFire PICC in D40 authenticated. </summary>
            VC_DF_AUTH_D40,

            /// <summary> State to indicate DESFire PICC in ISO authenticated. </summary>
            VC_DF_AUTH_ISO,

            /// <summary> State to indicate DESFire PICC in EV2 Authenticated. </summary>
            VC_DF_AUTH_EV2,

            /// <summary> State to indicate DESFire PICC in AES Authenticated. </summary>
            VC_DF_AUTH_AES,

            /// <summary> State to indicate MIFARE Plus PICC authenticated in Security Level 1. </summary>
            VC_MFP_AUTH_AES_SL1,

            /// <summary> State to indicate MIFARE Plus PICC authenticated in Security Level 3. </summary>
            VC_MFP_AUTH_AES_SL3
        };

        /// <summary>
        /// Enum defining PC states.
        /// </summary>
        public enum phalVca_ProximityCheckState : int
        {
            /// <summary> State to indicate there is no Proximity check in progress. </summary>
            PC_NO_PCHK_IN_PROGRESS = 0x00,

            /// <summary> State to indicate Proximity check preparation is in progress. </summary>
            PC_PPC_IN_PROGRESS,

            /// <summary> State to indicate Proximity check is prepared. </summary>
            PC_PCHK_PREPARED,

            /// <summary> State to indicate Proximity check is in progress. </summary>
            PC_PCHK_IN_PROGRESS,

            /// <summary> State to indicate waiting of Proximity check verification. </summary>
            PC_WAITING_PC_VERIFICATION,

            /// <summary> State to indicate Proximity check verification is in progress. </summary>
            PC_VPC_IN_PROGRESS
        };

        /// <summary>
        /// Data structure for VCA / PC Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the parameter structure of the Crypto layer. </summary>
            public IntPtr pCryptoDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to the Card Table of the layer. </summary>
            public IntPtr pCardTable;

            /// <summary> Pointer to the Iid Table of the layer. </summary>
            public IntPtr pIidTable;

            /// <summary> Position of current entry in the CardTable. </summary>
            public ushort wCurrentCardTablePos;

            /// <summary> Number of Card table entries in the table. </summary>
            public ushort wNumCardTableEntries;

            /// <summary> Number of Iid table entries in the table. </summary>
            public ushort wNumIidTableEntries;

            /// <summary> Current index of the Iid sequence. </summary>
            public ushort wCurrentIidIndex;

            /// <summary> Position of current entry in the Iid Table. </summary>
            public ushort wCurrentIidTablePos;

            /// <summary> Specific error codes for VC generic errors. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Session Keys for Authentication of MAC. </summary>
            public fixed byte bSessionAuthMACKey[16];

            /// <summary> Enum variable holding Virtual Card state. </summary>
            phalVca_VirtualCardState eVCState;

            /// <summary> Enum variable holding Proximity Check state. </summary>
            phalVca_ProximityCheckState ePCState;

            /// <summary> Pointer to the parameter structure of the Desfire Ev2 component. </summary>
            public IntPtr pAlDesEv2DataParams;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. </summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format. </summary>
            public byte bExtendedLenApdu;

            /// <summary> bOption parameter for Timing measurements ON/OFF. </summary>
            public byte bOption;

            /// <summary> Parameter for enable/disable the LowerBound threshold time calculation(-20%). </summary>
            public byte bLowerBoundThreshold;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams, IntPtr pKeyStoreDataParams,
            IntPtr pCryptoDataParams, IntPtr pCryptoRngDataParams, ref IidTableEntry_t pIidTableStorage, ushort wNumIidTableStorageEntries, ref CardTableEntry_t pCardTableStorage,
            ushort wNumCardTableStorageEntries );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sw_Cmd_PrepareProximityCheck ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sw_Cmd_ProximityCheck ( IntPtr pDataParams, byte bGenerateRndC, byte[] pRndC, byte bNumSteps, byte[] pUsedRndRC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sw_Cmd_VerifyProximityCheck ( IntPtr pDataParams, byte[] pRndRC, byte bPps1, ushort wKeyNumber, ushort wKeyVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sw_SetApplicationType ( ref DataParams_t m_pDataParams, IntPtr pAlDataParams );
        #endregion

        #region Initialization
        private IidTableEntry_t[] m_IidTable;
        private GCHandle m_pIidTable;

        private CardTableEntry_t[] m_CardTable;
        private GCHandle m_pCardTable;

        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with direct reader and PICC.
        /// </summary>
        ///
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        /// <param name="pKeyStore">Pointer to the parameter structure of the keyStore layer.</param>
        /// <param name="pCrypto">Pointer to the parameter structure of the Crypto layer.</param>
        /// <param name="pCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="bNumIidEntries">Number of possible Iid table entries in the storage.</param>
        /// <param name="bNumCardTableEntries">Number of possible Card table entries in the storage.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( palMifare.Generic pMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCrypto, CryptoRng.Generic pCryptoRng,
            byte bNumIidEntries, byte bNumCardTableEntries )
        {
            // Free Buffers
            if ( this.m_pIidTable.IsAllocated )
            {
                this.m_pIidTable.Free ();
            }
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }

            this.m_IidTable = new IidTableEntry_t[bNumIidEntries];
            this.m_pIidTable = GCHandle.Alloc ( m_IidTable, GCHandleType.Pinned );
            this.m_CardTable = new CardTableEntry_t[bNumCardTableEntries];
            this.m_pCardTable = GCHandle.Alloc ( m_CardTable, GCHandleType.Pinned );

            return phalVca_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pCrypto.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ref m_IidTable[0], bNumIidEntries, ref m_CardTable[0], bNumCardTableEntries );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with direct reader and PICC.
        /// </summary>
        ///
        /// <param name="wSizeOfDataParams">Specifies the size of the data parameter structure.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        /// <param name="pKeyStore">Pointer to the parameter structure of the keyStore layer.</param>
        /// <param name="pCrypto">Pointer to the parameter structure of the Crypto layer.</param>
        /// <param name="pCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="bNumIidEntries">Number of possible Iid table entries in the storage.</param>
        /// <param name="bNumCardTableEntries">Number of possible Card table entries in the storage.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wSizeOfDataParams, palMifare.Generic pMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCrypto,
            CryptoRng.Generic pCryptoRng, byte bNumIidEntries, byte bNumCardTableEntries )
        {
            // Free Buffers
            if ( this.m_pIidTable.IsAllocated )
            {
                this.m_pIidTable.Free ();
            }
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }

            this.m_IidTable = new IidTableEntry_t[bNumIidEntries];
            this.m_pIidTable = GCHandle.Alloc ( m_IidTable, GCHandleType.Pinned );
            this.m_CardTable = new CardTableEntry_t[bNumCardTableEntries];
            this.m_pCardTable = GCHandle.Alloc ( m_CardTable, GCHandleType.Pinned );

            return phalVca_Sw_Init ( ref m_DataParamsInt[0], wSizeOfDataParams,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pCrypto.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pCryptoRng.m_pDataParams : IntPtr.Zero,
                ref m_IidTable[0], bNumIidEntries, ref m_CardTable[0], bNumCardTableEntries );
        }
#endif
        #endregion

        #region Wrapper Functions
        public Status_t Cmd_PrepareProximityCheck ()
        {
            return phalVca_Sw_Cmd_PrepareProximityCheck ( m_pDataParams );
        }

        public Status_t Cmd_ProximityCheck ( byte bGenerateRndC, byte[] pRndC, byte bNumSteps, out byte[] pUsedRndRC )
        {
            pUsedRndRC = new byte[14];
            return phalVca_Sw_Cmd_ProximityCheck ( m_pDataParams, bGenerateRndC, pRndC, bNumSteps, pUsedRndRC );
        }

        public Status_t Cmd_VerifyProximityCheck ( byte[] pRndRC, byte bPps1, ushort wKeyNumber, ushort wKeyVersion )
        {
            return phalVca_Sw_Cmd_VerifyProximityCheck ( m_pDataParams, pRndRC, bPps1, wKeyNumber, wKeyVersion );
        }
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free Buffers
            if ( this.m_pIidTable.IsAllocated )
            {
                this.m_pIidTable.Free ();
            }
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }

            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region Sam NonX
    #region SamAV2
    /// <summary>
    /// Class for Sam AV2 Non X (S Mode) layer initialization interface and data params.
    /// </summary>
    public class SamAV2 : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Dataparam Structure for IID table
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct CardTableEntry_t
        {
            /// <summary> Index of the corresponding iid. </summary>
            public ushort wIidIndex;

            /// <summary> Amount of key duos provided in that VCSL sequence. </summary>
            public byte bNumKeyDuos;

            /// <summary> Card Data received from the Card. </summary>
            public fixed byte bCardData[10];
        };

        /// <summary>
        /// Dataparam Structure for KeyDuo Storage
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct KeyDuos_t
        {
            /// <summary> Number of encryption Key. </summary>
            public byte bKeyEncNumber;

            /// <summary> Version of encryption Key. </summary>
            public byte bKeyEncVersion;

            /// <summary> Number of MAC Key. </summary>
            public byte bKeyMacNumber;

            /// <summary> Version of MAC Key. </summary>
            public byte bKeyMacVersion;
        };

        /// <summary>
        /// Data structure for VCA / PC SAM AV2 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Pointer to the Card Table of the layer. </summary>
            public IntPtr pCardTable;

            /// <summary> Position of current entry in the CardTable. </summary>
            public ushort wCurrentCardTablePos;

            /// <summary> Number of Card table entries in the table. </summary>
            public ushort wNumCardTableEntries;

            /// <summary> The memory storage is used to store RNDQ - Size: bNumCardTableEntries*12. </summary>
            public IntPtr pRndq;

            /// <summary> The memory storage is used to store Key duos. - Size: bNumKeyDuos * 4. </summary>
            public IntPtr pKeyDuos;

            /// <summary> Indicates the size of the key duos array. </summary>
            public byte bNumKeyDuos;

            /// <summary> LenPcdCaps || PCD Caps. </summary>
            public fixed byte pPcdCaps[4];
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SamAV2_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal, IntPtr pIMifareDataParams,
            IntPtr pRndq, IntPtr pKeyDuos, byte bNumKeyDuos, IntPtr pCardTableStorage, ushort wNumCardTableStorageEntries );
        #endregion

        #region Initialization
        private CardTableEntry_t[] m_CardTable;
        private GCHandle m_pCardTable;
        private byte[] m_Rndq;
        private GCHandle m_pRndq;
        private KeyDuos_t[] m_KeyDuos;
        private GCHandle m_pKeyDuos;

        public Status_t Init ( Hal.SamAV2 pSamHal, palMifare.Generic pMifare, byte bNumKeyDuos, ushort wNumCardTableEntries )
        {
            // Free Buffer
            if ( this.m_pRndq.IsAllocated )
            {
                this.m_pRndq.Free ();
            }
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }
            if ( this.m_pKeyDuos.IsAllocated )
            {
                this.m_pKeyDuos.Free ();
            }

            m_Rndq = new byte[12 * wNumCardTableEntries];
            m_pRndq = GCHandle.Alloc ( m_Rndq, GCHandleType.Pinned );
            m_CardTable = new CardTableEntry_t[wNumCardTableEntries];
            m_pCardTable = GCHandle.Alloc ( m_CardTable, GCHandleType.Pinned );
            m_KeyDuos = new KeyDuos_t[bNumKeyDuos];
            m_pKeyDuos = GCHandle.Alloc ( m_KeyDuos, GCHandleType.Pinned );

            return phalVca_SamAV2_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero,
                m_pRndq.AddrOfPinnedObject (), m_pKeyDuos.AddrOfPinnedObject (), bNumKeyDuos, m_pCardTable.AddrOfPinnedObject (), wNumCardTableEntries );
        }
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV2 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }

            // Free allocated pointer to data params
            if ( this.m_pRndq.IsAllocated )
            {
                this.m_pRndq.Free ();
            }

            // Free allocated pointer to key duos params
            if ( this.m_pKeyDuos.IsAllocated )
            {
                this.m_pKeyDuos.Free ();
            }

            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 Non X (S Mode) layer initialization interface and data params.
    /// </summary>
    public class SamAV3_NonX : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for VCA / PC SAM AV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs.</summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format.</summary>
            public byte bExtendedLenApdu;

            /// <summary> Option parameter for Timing measurements ON/OFF. </summary>
            public byte bOption;

            /// <summary> Specific error codes for VC generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal, IntPtr pIMifareDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in S Mode.
        /// </summary>
        ///
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pSamHal, palMifare.Generic pMifare )
        {
            return phalVca_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero, ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in S Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pSamHal, palMifare.Generic pMifare )
        {
            return phalVca_SamAV3_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero, ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region SAM AV4 and Future SAM's
    public class Sam_NonX : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for VCA / PC SAM AV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs.</summary>
            public byte bWrappedMode;

            /// <summary> Extended length APDU. If set the native commands should be wrapped in extended format.</summary>
            public byte bExtendedLenApdu;

            /// <summary> Option parameter for Timing measurements ON/OFF. </summary>
            public byte bOption;

            /// <summary> Specific error codes for VC generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sam_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal, IntPtr pIMifareDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in S Mode.
        /// </summary>
        ///
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam pSamHal, palMifare.Generic pMifare )
        {
            return phalVca_Sam_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero, ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in S Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.Sam pSamHal, palMifare.Generic pMifare )
        {
            return phalVca_Sam_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero, ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif
#endif
    #endregion

    #region Sam X
    #region SamAV2_X
    /// <summary>
    /// Class for Sam AV2 X layer initialization interface and data params.
    /// </summary>
    public class SamAV2_X : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for Card table.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct CardTableEntry_t
        {
            /// <summary> index of the corresponding iid.</summary>
            public ushort wIidIndex;

            /// <summary> Amount of key duos provided in that VCSL sequence. </summary>
            public byte bNumKeyDuos;

            /// <summary> Card Data received from the Card. </summary>
            public fixed byte bCardData[10];
        };

        /// <summary>
        /// Data structure for IID table.
        /// IMPORTANT: Structure is reused to store RNDQ!
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct IidTable_t
        {
            /// <summary> Iid sent to the card. </summary>
            public fixed byte bIid[16];
        };

        /// <summary>
        /// Data structure for key entries.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct KeysTable_t
        {
            /// <summary> Keys sent to the card. </summary>
            public fixed byte bKeys[4];
        };

        /// <summary>
        /// Data structure for VCA / PC SAM AV2 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Pointer to the Card Table of the layer. </summary>
            public IntPtr pCardTable;

            /// <summary> Pointer to the IidTable of the layer. </summary>
            public IntPtr pIidTable;

            /// <summary> Pointer to the IidTable of the layer. </summary>
            public IntPtr pKeysTable;

            /// <summary> Position of current entry in the CardTable. </summary>
            public ushort wCurrentCardTablePos;

            /// <summary> Number of Card table entries in the table. </summary>
            public ushort wNumCardTableEntries;

            /// <summary> Current position in IID array. </summary>
            public ushort wCurrentIidKeysTablePos;

            /// <summary> Indicates the size of the key table. </summary>
            public ushort wNumIidKeysTableEntries;

            /// <summary> LenPcdCaps || PCD Caps. </summary>
            public fixed byte bPcdCaps[4];
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal, IntPtr pIMifareDataParams,
            IntPtr pIidTable, IntPtr pKeysTable, ushort wNumIidKeysTableEntries, IntPtr pCardTableStorage, ushort wNumCardTableStorageEntries );
        #endregion

        #region Initialization
        private CardTableEntry_t[] m_CardTable;
        private GCHandle m_pCardTable;
        private KeysTable_t[] m_KeysTable;
        private GCHandle m_pKeysTable;
        private IidTable_t[] m_IidTable;
        private GCHandle m_pIidTable;

        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV2 in X Mode.
        /// </summary>
        ///
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        /// <param name="wNumIidKeysTableEntries">Indicates the size of the key/Iid table.</param>
        /// <param name="wNumCardTableEntries">Number of possible Card table entries in the storage.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV2 pSamHal, palMifare.SamAV2_X pMifare, ushort wNumIidKeysTableEntries, ushort wNumCardTableEntries )
        {
            // Free allocated pointer to data params
            if ( this.m_pCardTable.IsAllocated )
                this.m_pCardTable.Free ();

            // Free allocated pointer to keys table params
            if ( this.m_pKeysTable.IsAllocated )
                this.m_pKeysTable.Free ();

            // Free allocated pointer to Iid params
            if ( this.m_pIidTable.IsAllocated )
                this.m_pIidTable.Free ();

            m_CardTable = new CardTableEntry_t[wNumCardTableEntries];
            m_pCardTable = GCHandle.Alloc ( m_CardTable, GCHandleType.Pinned );
            m_KeysTable = new KeysTable_t[wNumIidKeysTableEntries];
            m_pKeysTable = GCHandle.Alloc ( m_KeysTable, GCHandleType.Pinned );
            m_IidTable = new IidTable_t[wNumIidKeysTableEntries];
            m_pIidTable = GCHandle.Alloc ( m_IidTable, GCHandleType.Pinned );

            return phalVca_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero,
                m_pIidTable.AddrOfPinnedObject (), m_pKeysTable.AddrOfPinnedObject (), wNumIidKeysTableEntries,
                m_pCardTable.AddrOfPinnedObject (), wNumCardTableEntries );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV2 in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        /// <param name="wNumIidKeysTableEntries">Indicates the size of the key/Iid table.</param>
        /// <param name="wNumCardTableEntries">Number of possible Card table entries in the storage.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV2 pSamHal, palMifare.SamAV2_X pMifare, ushort wNumIidKeysTableEntries, ushort wNumCardTableEntries )
        {
            // Free allocated pointer to data params
            if ( this.m_pCardTable.IsAllocated )
                this.m_pCardTable.Free ();

            // Free allocated pointer to keys table params
            if ( this.m_pKeysTable.IsAllocated )
                this.m_pKeysTable.Free ();

            // Free allocated pointer to Iid params
            if ( this.m_pIidTable.IsAllocated )
                this.m_pIidTable.Free ();

            m_CardTable = new CardTableEntry_t[wNumCardTableEntries];
            m_pCardTable = GCHandle.Alloc ( m_CardTable, GCHandleType.Pinned );
            m_KeysTable = new KeysTable_t[wNumIidKeysTableEntries];
            m_pKeysTable = GCHandle.Alloc ( m_KeysTable, GCHandleType.Pinned );
            m_IidTable = new IidTable_t[wNumIidKeysTableEntries];
            m_pIidTable = GCHandle.Alloc ( m_IidTable, GCHandleType.Pinned );

            return phalVca_SamAV2_X_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero,
                m_pIidTable.AddrOfPinnedObject (), m_pKeysTable.AddrOfPinnedObject (), wNumIidKeysTableEntries,
                m_pCardTable.AddrOfPinnedObject (), wNumCardTableEntries );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Desctructor
        /// </summary>
        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pCardTable.IsAllocated )
            {
                this.m_pCardTable.Free ();
            }

            // Free allocated pointer to keys table params
            if ( this.m_pKeysTable.IsAllocated )
            {
                this.m_pKeysTable.Free ();
            }

            // Free allocated pointer to Iid params
            if ( this.m_pIidTable.IsAllocated )
            {
                this.m_pIidTable.Free ();
            }

            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion SamAV2_X

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X
    /// <summary>
    /// Class for Sam AV3 X layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for VCA / PC SAM AV3 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs.</summary>
            public byte bWrappedMode;

            /// <summary> Specific error codes for VC generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal,
            IntPtr pIMifareDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pSamHal, palMifare.SamAV3_X pMifare )
        {
            return phalVca_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pSamHal, palMifare.SamAV3_X pMifare )
        {
            return phalVca_SamAV3_X_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Desctructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion SamAV3_X

#if PACKAGE_INTERNAL
    #region Sam_X
    /// <summary>
    /// Class for Sam (AV4, Future Sam) X layer initialization interface and data params.
    /// </summary>
    public class Sam_X : alVca.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for VCA / PC SAM X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the Sam HAL. </summary>
            public IntPtr pSamHal;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs.</summary>
            public byte bWrappedMode;

            /// <summary> Specific error codes for VC generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalVca_Sam_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pSamHal,
            IntPtr pIMifareDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with Sam in X Mode.
        /// </summary>
        ///
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam pSamHal, palMifare.Sam_X pMifare )
        {
            return phalVca_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL VCA / PC to communicate with Sam in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pSamHal">Pointer to the parameter structure of the Sam HAL.</param>
        /// <param name="pMifare">Pointer to the parameter structure of the palMifare layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.Sam pSamHal, palMifare.Sam_X pMifare )
        {
            return phalVca_Sam_X_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pSamHal != null ) ? pSamHal.m_pDataParams : IntPtr.Zero,
                ( pMifare != null ) ? pMifare.m_pDataParams : IntPtr.Zero );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Desctructor
        /// </summary>
        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion
#endif
#endif
    #endregion
}
