/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.CidManager
{
    public abstract class Generic
    {
        #region DEFINES

        public const byte LAST_CID = 14;

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phCidManager_GetFreeCid(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure. */
            ref byte pCid       /**< [Out] Card Identifier */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phCidManager_FreeCid(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure. */
            byte bCid           /**< [In] Card Identifier */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t GetFreeCid(
            out byte pCid   /**< [Out] Card Identifier */
            )
        {
            pCid = 0;
            return phCidManager_GetFreeCid(m_pDataParams, ref pCid);
        }

        public Status_t FreeCid(
            byte bCid   /**< [In] Card Identifier */
            )
        {
            return phCidManager_FreeCid(m_pDataParams, bCid);
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #region SW

    public class Sw : CidManager.Generic
    {
        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this HAL component, NEVER MODIFY! */
            public fixed byte bCidList[LAST_CID];   /**< Contains a list of Cids, where a value
                                                     * of /c 0 means free and a value of /c 1 means occupied. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phCidManager_Sw_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        public Status_t Init()
        {
            return phCidManager_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

#if DEBUG
        public Status_t Init(int wDataParamSize)
        {
            return phCidManager_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

	#region Rd710

	public class Rd710 : CidManager.Generic
	{
		#region DATA_STRUCTURE

		/// <summary>
		/// Private data storage definition of underlying C Object.
		/// </summary>
		[StructLayout(LayoutKind.Sequential, Pack = 1)]
		public unsafe struct DataParams_t
		{
			public ushort wId;                      /**< Layer ID for this HAL component, NEVER MODIFY! */
			IntPtr pHalDataParams;                  /**< pointer to the Hal parameter structure */
		};

		#endregion

		#region DLLIMPORTS

		[DllImport(Common.IMPORT_LIBRARY_NAME)]
		private static extern ushort phCidManager_Rd710_Init(
									ref DataParams_t pDataParams,		/**< [In] Pointer to this layers parameter structure. */
									ushort wSizeOfDataParams,			/**< [In] Specifies the size of the data parameter structure */
									IntPtr pHalDataParams				/**< pointer to the HAL parameter structure */
									);

		#endregion

		#region INIT

		public Status_t Init(Hal.Rd710 pHal)
		{
			return phCidManager_Rd710_Init(	ref m_DataParamsInt[0],
											(ushort)Marshal.SizeOf(typeof(DataParams_t)),
											pHal.m_pDataParams
											);
		}
#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
		{
			return phCidManager_Rd710_Init(	ref m_DataParamsInt[0],
											(ushort)wDataParamSize,
											pHal.m_pDataParams
											);
		}
#endif

		#endregion

		#region MEMORY_MAPPING

		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Allocate unmanaged memory for underlying C-Object
		/// </summary>
		public Rd710()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
		}

		/// <summary>
		/// Free allocated unmanaged memory.
		/// </summary>
		~Rd710()
		{
			// Free allocated pointer to data params
			if (this.m_pDataParamsInt.IsAllocated)
			{
				this.m_pDataParamsInt.Free();
			}
		}

		// Setter & Getter for DataParams structure
		private DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion
	}

	#endregion
}
