/*
 * Copyright 2021 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Text;

namespace NxpRdLibNet.CryptoASym
{
    #region Enumerations
    #region Error
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary>Field to indicate Curve by group ID not supported. </summary>
        UNSUPPORTED_CURVE_ID = CustomCodes.ERROR_BEGIN + 0,

        /// <summary>Field to indicate the key pair to be loaded / exported is not supported. </summary>
        UNSUPPORTED_KEY_PAIR_TYPE,

        /// <summary>Field to indicate Hashing Algorithm not supported. </summary>
        UNSUPPORTED_HASH_ALGO,

        /// <summary>Field to indicate Verification of Message / Signature combination failed. </summary>
        VERIFICATION_FAILED,
    }
    #endregion

    #region Keytype
    /// <summary>
    /// Supported Key Types to be used in key loading functionality.
    /// </summary>
    public enum KeyType : int
    {
        /// <summary> Field to indicate Key type as ECC (Elliptical Curve Cryptography). </summary>
        ECC = 0x0200,

        /// <summary> Field to indicate Invalid Key type. </summary>
        INAVLID = 0xFFFF
    }
    #endregion

    #region Load / Export
    /// <summary>
    /// Supported key for loading / exporting.
    /// </summary>
    public enum Key
    {
        /// <summary> Field to indicate Private Key load / export. </summary>
        PRIVATE_KEY = 0x1000,

        /// <summary> Field to indicate Public Key load / export. </summary>
        PUBLIC_KEY = 0x2000,

        /// <summary> Key-pair not supported or not applicable. </summary>
        INVALID = 0xF000
    }
    #endregion

    #region Hash
    /// <summary>
    /// Supported Hash algorithms.
    /// Warning:
    /// MD2, MD4, MD5 and SHA-1 are considered weak message digests and
    /// their use constitutes a security risk. We recommend considering
    /// stronger message digests instead.
    /// </summary>
    public enum HashAlgo
    {
        /// <summary>
        /// Field to represent Hashing Applicable as Not Applicable.
        /// This can be used when Buffering Options are used.
        /// </summary>
        NOT_APPLICABLE = 0x00,

        /// <summary> Field to represent Hashing algorithm as SHA-224. </summary>
        SHA224 = 0x03,

        /// <summary> Field to represent Hashing algorithm as SHA-256. </summary>
        SHA256 = 0x04,

        /// <summary> Field to represent Hashing algorithm as SHA-384. </summary>
        SHA384 = 0x05,

        /// <summary> Field to represent Hashing algorithm as SHA-512. </summary>
        SHA512 = 0x06
    }
    #endregion

    #region ECC
    #region CurveID
    /// <summary>
    /// Supported ECC Curves Group ID's
    /// The ID values matched with the underlying crypto library being used.
    /// </summary>
    public enum CurveID : byte
    {
        /// <summary> Field to indicate the curve group id as none. </summary>
        NONE = 0x00,

        /// <summary> Field to indicate the curve group id as secp192r1. </summary>
        SECP_192R1 = 0x01,

        /// <summary> Field to indicate the curve group id as secp224r1. </summary>
        SECP_224R1 = 0x02,

        /// <summary> Field to indicate the curve group id as secp256r1. </summary>
        SECP_256R1 = 0x03,

        /// <summary> Field to indicate the curve group id as secp384r1. </summary>
        SECP_384R1 = 0x04,

        /// <summary> Field to indicate the curve group id as brainpoolp256r1. </summary>
        BRAINPOOL_256R1 = 0x06,

        /// <summary> Field to indicate the curve group id as brainpoolp384r1. </summary>
        BRAINPOOL_384R1 = 0x07,
    }
    #endregion
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Generic ASymmetric Cryptography Component Wrapper of the Reader Library Framework.
    ///
    /// This is only a wrapper layer to abstract the different Crypto ASymmetric implementations.
    /// With this wrapper it is possible to support more than one ASymmetric implementation
    /// in parallel, by adapting this wrapper.
    ///
    /// Important hints for users of this component:
    ///     - Before use of any function, the dedicated crypto implementation has to be initialized using either of the interfaces mentioned below.
    ///         - <see cref="mBedTLS.Init"/>
    ///         - <see cref="DUT.Init"/>
    ///         - <see cref="Stub.Init"/>
    ///     - Functions using a KeyStore (<see cref="Generic.LoadKey"/>) are only available if KeyStore component has been passed during
    ///       component initialization.
    ///     - If return value is <see cref="Error_Comm.INTERNAL_ERROR"/>, means error has occured in the dependant ASymmetric crypto library.
    ///       Call <see cref="Generic.GetLastStatus"/> to know about the error details.
    ///     - Its must to set the CurveID if key to be loaded is of <see cref="KeyType.ECC"/> keytype and format is
    ///       <see cref="Format.BINARY"/>. To set the CurveID refer <see cref="Generic.SetConfig"/> interface.
    ///     - Public Key can alone be loaded separately. But if Private Key is loaded, then its must to load PublicKey.
    ///       If public key is not loaded after PrivateKey, then while Exporting Private key, error will be observed.
    ///     - While performing <see cref="Generic.LoadKey"/> or <see cref="Generic.LoadKeyDirect"/> commands with Key format as
    ///       <see cref="Format.PEM"/>, the key buffer should contain a terminating Null character and the length should include
    ///       the null character.
    ///     - While computing SharedSecret, its must to have a key-pair available using one of the below mentioned interfaces,
    ///         - <see cref="Generic.GenerateECC_KeyPair"/>
    ///         - <see cref="Generic.LoadKey"/>
    ///         - <see cref="Generic.LoadKeyDirect"/>
    /// </summary>
    public abstract class Generic
    {
        #region Private Variables
        private const int BUFFER_SIZE = 512;
        #endregion

        #region DLL Imports
        #region Hash
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ComputeHash ( IntPtr pDataParams, ushort wOption, byte bHashAlgo, byte[] pMessage,
            ushort wMsgLen, byte[] pHash, ref ushort pHashLen );
        #endregion

        #region ECC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_GenerateKeyPair ( IntPtr pDataParams, byte bCurveID );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_ExportKey ( IntPtr pDataParams, ushort wOption, ushort wKeyBuffSize, ref byte pcurveID,
            byte[] pKey, ref ushort pKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_LoadKey ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wPos );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_LoadKeyDirect ( IntPtr pDataParams, ushort wOption, byte[] pKey, ushort wKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_Sign ( IntPtr pDataParams, ushort wOption, byte bHashAlgo, byte[] pMessage, ushort wMsgLen, byte[] pSign,
            ref ushort pSignLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_Verify ( IntPtr pDataParams, ushort wOption, byte bHashAlgo, byte[] pMessage, ushort wMsgLen, byte[] pSign,
            ushort wSignLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_ECC_SharedSecret ( IntPtr pDataParams, ushort wOption, byte[] pPublicKey, ushort wPublicKeyLen,
            byte[] pSharedSecret, ref ushort pSharedSecretLen );
        #endregion

        #region Utility
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_InvalidateKey ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_GetLastStatus ( IntPtr pDataParams, ushort dwStatusMsgLen, byte[] pStatusMsg, ref int pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_GetKeySize ( ushort wKeyType, ushort wKeyPair, byte bCurveID );
        #endregion
        #endregion

        #region Wrapper Functions
        #region Hash
        /// <summary>
        /// Computes Hash for the given message.
        /// Note: Start, Update and Finish operation are supported and can be exercised by using \b wOption parameter.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options for Hashing the message. These flags can be used for intermediate Hash operation
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>     : Computes the Hash for the Message and returns the Hash.
        ///                                                                       This performs Start, Update and Finish in one operation.
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>: Computes the Hash and saves for completion. This is
        ///                                                                       equivalent to Start and here the Hash is not provided.
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/> : Computes the Hash along with the previous hash data
        ///                                                                       and saves for completion.This is equivalent to Update
        ///                                                                       and here the Hash is not provided.
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/> : Computes the Hash along with the previous hash data.
        ///                                                                       This is equivalent to Finish and here the Hash is provided.
        /// </param>
        /// <param name="bHashAlgo">Hashing Algorithm to use. Will be one of the following
        ///                         - <see cref="HashAlgo.SHA224">SHA-224</see>
        ///                         - <see cref="HashAlgo.SHA256">SHA-256</see>
        ///                         - <see cref="HashAlgo.SHA384">SHA-384</see>
        ///                         - <see cref="HashAlgo.SHA512">SHA-512</see>
        /// </param>
        /// <param name="aMessage">Input message to be Hashed.</param>
        /// <param name="aHash">The Hashed information for the message based on hashing algorithm.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error.UNSUPPORTED_HASH_ALGO"/> Hash Algorithm not supported. Refer \b bHashAlgo
        ///     parameter description.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If Hashing algorithm is <see cref="HashAlgo.NOT_APPLICABLE">No-Hashing</see> and buffering options is
        ///           not <see cref="ExchangeOptions.DEFAULT"/>.
        ///         - If the buffers are null
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ComputeHash ( ushort wOption, byte bHashAlgo, byte[] aMessage, out byte[] aHash )
        {
            byte[] aHashTmp = new byte[BUFFER_SIZE];
            ushort wHashLen = 0;

            aHash = null;
            Status_t oStatus = phCryptoASym_ComputeHash ( m_pDataParams, wOption, bHashAlgo, aMessage,
                ( ushort ) ( ( aMessage == null ) ? 0 : aMessage.Length), aHashTmp, ref wHashLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                if ( ( wOption == ExchangeOptions.DEFAULT ) || ( wOption == ExchangeOptions.BUFFER_LAST ) )
                {
                    aHash = new byte[wHashLen];
                    Array.Copy ( aHashTmp, 0, aHash, 0, wHashLen );
                }
            }

            return oStatus;
        }
        #endregion

        #region ECC
        /// <summary>
        /// Generates a ECC Private and Public Key pair based on the ECC Curve name specified.
        /// </summary>
        ///
        /// <param name="bCurveID">The <see cref="CurveID"/>'s to be used for key generation.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error.UNSUPPORTED_CURVE_ID"/> if CurveID is not supported. Refer bCurve parameter description.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GenerateKeyPair ( byte bCurveID )
        {
            return phCryptoASym_ECC_GenerateKeyPair ( m_pDataParams, bCurveID );
        }

        /// <summary>
        /// Computes the shared secret between current side private key and other side public key.
        /// The computation used ECDH algorithm.
        ///
        /// Note:
        ///     - This interface needs to be called post using one of the below mentioned interfaces,
        ///         - <see cref="GenerateKeyPair_ECC"/>
        ///         - <see cref="LoadKey"/>
        ///         - <see cref="LoadKeyDirect"/>
        /// </summary>
        ///
        /// <param name="wOption">The CurveID of the key thats available in aPublicKey buffer
        ///                       combined with internal key context to use for shared secret computation.
        ///                         - Refer <see cref="CurveID"/> for the Key to be loaded.
        /// </param>
        /// <param name="aPublicKey">The other side's Public key</param>
        /// <param name="aSharedSecret">Shared secret between private key and other side's public key</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> if KeyPair is not generated or loaded. Refer description notes for more information.
        ///     Returns <see cref="Error.UNSUPPORTED_CURVE_ID"/> if CurveID is not supported. Refer bCurve parameter description.
        ///     supported. Refer wOption parameter description.
        /// </returns>
        public Status_t SharedSecret ( ushort wOption, byte[] aPublicKey, out byte[] aSharedSecret )
        {
            byte[] aSecret = new byte[BUFFER_SIZE];
            ushort wLen = ( ushort ) aSecret.Length;

            aSharedSecret = null;
            Status_t oStatus = phCryptoASym_ECC_SharedSecret ( m_pDataParams, wOption, aPublicKey, ( ushort ) ( ( aPublicKey == null ) ? 0 :
                aPublicKey.Length ), aSecret, ref wLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aSharedSecret = new byte[wLen];
                Array.Copy ( aSecret, 0, aSharedSecret, 0, wLen );
            }

            return oStatus;
        }
        #endregion

        #region Common
        /// <summary>
        /// Exports the private or public key based on the option provided.
        ///
        /// Note:
        ///      This interface needs to be called post using one of the below mentioned interfaces.
        ///         <see cref="GenerateKeyPair_ECC"/>
        ///         <see cref="LoadKey"/>
        ///         <see cref="LoadKeyDirect"/>
        /// </summary>
        ///
        /// <param name="wOption">Combined values for Key to export, the format to be used and
        ///                       internal key context to be used while exporting the keys.
        ///                         - Supported Keys to export are
        ///                             - <see cref="Key.PRIVATE_KEY"/>
        ///                             - <see cref="Key.PUBLIC_KEY"/>
        /// </param>
        /// <param name="bCurveID"><see cref="CurveID"/> of the key being exported. </param>
        /// <param name="aKey">The generated key based on the option provided. The keys will be exported from either of the below interfaces,
        ///                         - <see cref="GenerateKeyPair_ECC"/>
        ///                         - <see cref="LoadKey"/>
        ///                         - <see cref="LoadKeyDirect"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> if KeyPair is not generated or loaded. Refer description notes for more information.
        ///     Returns <see cref="Error.UNSUPPORTED_KEY_EXPORT_TYPE"/> if export key type is not supported. Refer wOption parameter description.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ExportKey ( ushort wOption, out byte bCurveID, out byte[] aKey )
        {
            byte[] aKeyTmp = new byte[BUFFER_SIZE];
            ushort wKeyLen = 0;
            ushort wKeyBuffSize = ( ushort ) aKeyTmp.Length;

            bCurveID = 0;
            aKey = null;

            Status_t oStatus = phCryptoASym_ECC_ExportKey ( m_pDataParams, wOption, wKeyBuffSize, ref bCurveID, aKeyTmp, ref wKeyLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aKey = new byte[wKeyLen];
                Array.Copy ( aKeyTmp, 0, aKey, 0, wKeyLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Load Key
        /// This function uses the key storage provided at component initialization to retrieve the key identified by wKeyNo.
        /// After retrieving the key is loaded into the internal key storage array to be prepared for subsequent crypto operations.
        ///
        /// Note:
        ///     - Before calling this interface make sure the Private, Public Keys are loaded to KeyStore.
        /// </summary>
        ///
        /// <param name="wOption">Internal key context and Key-Pair to be used while loading the keys.
        ///                       One of the below values.
        ///                         - Refer <see cref="Key"/> Pair to load.
        /// </param>
        /// <param name="wKeyNo">Key number in KeyStore to be loaded.</param>
        /// <param name="wKeyNo">Key Position in KeyStore to be loaded.</param>
        /// <param name="wKeyType">Type of Key.
        ///                         - <see cref="KeyType.ECC"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported. Refer wKeyType parameter description.
        ///     Returns <see cref="Error.UNSUPPORTED_KEY_LOAD_TYPE"/> if load key type is not supported. Refer wOption parameter description.
        ///     Returns <see cref="Error.UNSUPPORTED_CURVE_ID"/> if Curve ID not supported. Refer \b wOption parameter description.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadKey ( ushort wOption, ushort wKeyNo, ushort wPos, ushort wKeyType )
        {
            Status_t oSatus = new Status_t ();

            switch ( wKeyType )
            {
                case ( ushort ) KeyType.ECC:
                    oSatus = phCryptoASym_ECC_LoadKey ( m_pDataParams, wOption, wKeyNo, wPos );
                    break;

                default:
                    oSatus = new Status_t ( Error_CompCode.CRYPTOASYM, Error_Param.KEY );
                    break;
            }

            return oSatus;
        }

        /// <summary>
        /// Direct Loads a private and Public Key
        /// The key provided in the aKey parameter is loaded directly into the internal key context and will
        /// be utilized for subsequent crypto operations.
        ///
        /// Note:
        ///     - This interface should be called more than ones to load Private or Public Key.
        ///     - This interface supports loading of Private or Public Key separately.
        ///     - This interface supports loading of Private or Public Key alone.
        /// </summary>
        ///
        /// <param name="wOption">Combined values for Key to load, the format to be used and
        ///                       internal context to be used while importing the keys.
        ///                         - Refer <see cref="CurveID"/> of the Key to be loaded.
        ///                         - Refer <see cref="Key"/> Pair to load.
        /// </param>
        /// <param name="wKeyType">Type of Key.
        ///                         - <see cref="KeyType.ECC"/>
        /// </param>
        /// <param name="aKey">The Private / Public key to be loaded. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported. Refer wKeyType parameter description.
        ///     Returns <see cref="Error.UNSUPPORTED_KEY_LOAD_TYPE"/> if load key type is not supported. Refer wOption parameter description.
        ///     Returns <see cref="Error.UNSUPPORTED_KEY_FORMAT"/> if load key format is not supported. Refer wOption parameter description.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadKeyDirect ( ushort wOption, ushort wKeyType, byte[] aKey )
        {
            Status_t oSatus = new Status_t ();

            switch ( wKeyType )
            {
                case ( ushort ) KeyType.ECC:
                    oSatus = phCryptoASym_ECC_LoadKeyDirect ( m_pDataParams, wOption, aKey, ( byte ) ( ( aKey == null ) ? 0 : aKey.Length ) );
                    break;

                default:
                    oSatus = new Status_t ( Error_CompCode.CRYPTOASYM, Error_Param.KEY );
                    break;
            }

            return oSatus;
        }

        /// <summary>
        /// Signs the message.
        ///
        /// Note:
        ///     - This interface needs to be called post using one of the below mentioned interfaces,
        ///         - <see cref="GenerateKeyPair_ECC"/>
        ///         - <see cref="LoadKey"/>
        ///         - <see cref="LoadKeyDirect"/>
        ///     - The aSign buffer allocation should be more than size of curve length to avoid memory corruption.
        ///     - If the Message is small or needs to be computed in one call, then <see cref="ExchangeOptions.DEFAULT"/> should be used for
        ///       wOption parameter.
        ///     - If the message is long enough and requires to be buffered, use <see cref="ExchangeOptions.BUFFER_FIRST"/>,
        ///       <see cref="ExchangeOptions.BUFFER_CONT"/> and <see cref="ExchangeOptions.BUFFER_LAST"/> options.
        ///       The Signature will be provided when <see cref="ExchangeOptions.BUFFER_LAST"/> is passed in wOption parameter.
        ///     - If Hashing algorithm is <see cref="HashAlgo.NOT_APPLICABLE"/>, then buffering options needs to be
        ///       <see cref="ExchangeOptions.DEFAULT"/> only. Other buffering options are not supported.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options combined with internal key context to use for signing.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>         : Computes the Hash for the Message and provides the Signature.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>    : Computes the Hash and saves for completion.
        ///                                                                           Here the Signature is not provided.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>     : Computes the Hash along with the previous hash data and
        ///                                                                           saves for completion. Here the Signature is not provided.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>     : Computes the Hash along with the previous hash data.
        ///                                                                           Here the Signature is provided.
        /// </param>
        /// <param name="bHashAlgo">The Hashing algorithm to be applied. Refer <see cref="HashAlgo"/> for list for supported hashing algorithms. </param>
        /// <param name="aMessage">Message to be signed.</param>
        /// <param name="aSign">The signature of the message. The Signature will be in R and S integer format.pSign = R data followed by S data.
        ///                     Here R and S length should be based on the curve length.
        ///                     Ex: If curve length is 256 bit then R and S length will be 32 bytes each.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> if KeyPair is not loaded or generated. Refer description notes for more information.
        ///     Returns <see cref="Error.UNSUPPORTED_HASH_ALGO"/> for Hashing algorithm not supported. Refer bHashAlgo parameter description.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Hashing algorithm is <see cref="HashAlgo.NOT_APPLICABLE"/> and buffering
        ///     option is not <see cref="ExchangeOptions.DEFAULT"/>.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Sign ( int wOption, byte bHashAlgo, byte[] aMessage, out byte[] aSign )
        {
            ushort wSignLen = 0;
            byte[] aSignTmp = new byte[BUFFER_SIZE];

            aSign = null;

            Status_t oStatus = phCryptoASym_ECC_Sign( m_pDataParams, ( ushort ) wOption, bHashAlgo , aMessage,
                ( ushort ) ( ( aMessage == null ) ? 0 : aMessage.Length ), aSignTmp, ref wSignLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aSign = new byte[wSignLen];
                Array.Copy ( aSignTmp, 0, aSign, 0, wSignLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Verifies the signature.
        ///
        /// Note:
        ///     - This interface needs to be called post using one of the below mentioned interfaces,
        ///         - <see cref="GenerateKeyPair_ECC"/>
        ///         - <see cref="LoadKey"/>
        ///         - <see cref="LoadKeyDirect"/>
        ///     - If the Message is small or needs to be verified in one call, then <see cref="ExchangeOptions.DEFAULT"/> should be used for
        ///       wOption parameter.
        ///     - If the message is long enough and requires to be buffered, use <see cref="ExchangeOptions.BUFFER_FIRST"/>,
        ///       <see cref="ExchangeOptions.BUFFER_CONT"/> and <see cref="ExchangeOptions.BUFFER_LAST"/> options.
        ///       The Signature will be utilized only when <see cref="ExchangeOptions.BUFFER_LAST"/> is passed in wOption parameter.
        ///     - If Hashing algorithm is <see cref="HashAlgo.NOT_APPLICABLE"/>, then buffering options needs to be
        ///       <see cref="ExchangeOptions.DEFAULT"/> only. Other buffering options are not supported.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options combined with internal key context to use for verification.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>         : Computes the Hash for the Message and provides the
        ///                                                                           verification based on the Signature.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>    : Computes the Hash and saves for completion.
        ///                                                                           Here the Signature is not taken for verification.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>     : Computes the Hash along with the previous hash data
        ///                                                                           and saves for completion. Here the Signature is not
        ///                                                                           taken for verification.
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>     : Computes the Hash along with the previous hash data.
        ///                                                                           Here the Signature is taken for verification.
        /// </param>
        /// <param name="bHashAlgo">The Hashing algorithm to be applied. Refer <see cref="HashAlgo"/> for list for supported hashing algorithms. </param>
        /// <param name="aMessage">Message to be signed.</param>
        /// <param name="aSign">The signature to be used for verification.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> if KeyPair is not loaded or generated. Refer description notes for more information.
        ///     Returns <see cref="Error.UNSUPPORTED_HASH_ALGO"/> for Hashing algorithm not supported. Refer bHashAlgo parameter description.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Hashing algorithm is <see cref="HashAlgo.NOT_APPLICABLE"/> and buffering
        ///     option is not <see cref="ExchangeOptions.DEFAULT"/>.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Verify ( int wOption, byte bHashAlgo, byte[] aMessage, byte[] aSign )
        {
            return phCryptoASym_ECC_Verify ( m_pDataParams, ( ushort ) wOption, bHashAlgo, aMessage,
                ( ushort ) ( ( aMessage == null ) ? 0 : aMessage.Length ), aSign,
                ( ushort ) ( ( aSign == null ) ? 0 : aSign.Length ) );
        }
        #endregion

        #region Utility
        /// <summary>
        /// Invalidate the currently loaded key. Resets the key
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t InvalidateKey ()
        {
            return phCryptoASym_InvalidateKey ( m_pDataParams );
        }

        /// <summary>
        /// Returns the status code and corresponding message.
        /// </summary>
        ///
        /// <param name="dwStatusCode">The status code returned by the underlying Crypto library. </param>
        /// <param name="sStatusMsg">The equivalent status message for the information available in dwStatusCode.</param>
        ///
        /// <returns>Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation. </returns>
        public Status_t GetLastStatus ( out string sStatusMsg, out int dwStatusCode )
        {
            byte[] aStatusMsg = new byte[500];

            dwStatusCode = 0;
            sStatusMsg = string.Empty;
            Status_t oStatus = phCryptoASym_GetLastStatus ( m_pDataParams, ( ushort ) aStatusMsg.Length, aStatusMsg, ref dwStatusCode );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                int dwNullChar_Index = 0;
                while ( aStatusMsg[dwNullChar_Index] != 0 )
                    dwNullChar_Index++;

                if ( dwNullChar_Index > 0 )
                {
                    Array.Resize ( ref aStatusMsg, dwNullChar_Index );
                    sStatusMsg = Encoding.ASCII.GetString ( aStatusMsg );
                }
            }

            return oStatus;
        }

        /// <summary>
        ///
        /// </summary>
        ///
        /// <param name="wKeyType">Refer <see cref="KeyType"/> for supported values.</param>
        /// <param name="wKeyPair">Refer <see cref="Key"/> for supported values.</param>
        /// <param name="bCurveID">Refer <see cref="CurveID"/> for supported values.</param>
        ///
        /// <returns>
        /// Return 0, if any of the parameter value is not supported.
        /// Return KeySize, for the key provided as parameter information.
        /// </returns>
        public ushort GetKeySize( ushort wKeyType, ushort wKeyPair, byte bCurveID )
        {
            return phCryptoASym_GetKeySize ( wKeyType, wKeyPair, bCurveID );
        }
        #endregion
        #endregion

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion

        #region Private Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) ) && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( ( pBuffer != IntPtr.Zero ) && ( dwLength != 0 ) )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }
        #endregion Support Methods
    }
    #endregion

    #region mBedTLS
    /// <summary>
    /// mBedTLS Sub-Component ASymmetric Cryptography library.
    /// </summary>
    public class mBedTLS : Generic
    {
        #region Private Variables
        /// <summary> Minimum size for allocating the Internal Processing buffer during initializing. </summary>
        public const int INTERNAL_BUFFER_SIZE = 512;

        private byte[] m_Buffer;
        private GCHandle m_pBuffer;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for ASymmetric Crypto mBedTLS layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to underlying ASymmetric Crypto context for storing KeyPair information. </summary>
            public IntPtr pCtx;

            /// <summary> Key Type. </summary>
            public ushort wKeyType;

            /// <summary> ECC Curve Id. </summary>
            public byte bCurveID;

            /// <summary> Hash Algorithm ID. </summary>
            public byte bHashAlgo;

            /// <summary> Error code returned by mbedTLS layer. </summary>
            public int dwErrorCode;

            /// <summary> Internal Buffer for processing. </summary>
            public IntPtr pBuffer;

            /// <summary> Size of bytes allocated for pBuffer member. </summary>
            public ushort wBufferSize;
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_mBedTLS_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pKeyStoreDataParams,
            IntPtr pBuffer, ushort wBufferSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_mBedTLS_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoASym with mBedTLS as sub-component.
        /// </summary>
        ///
        /// <param name="oKeyStore">Pointer to a KeyStore data parameter structure.</param>
        /// <param name="dwInternalBufSize">Size of global buffer. This buffer is for processing information internally.
        ///                                 Should not be less than the default specified one <see cref="INTERNAL_BUFFER_SIZE"/>.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ( KeyStore.Generic oKeyStore, int dwInternalBufSize = INTERNAL_BUFFER_SIZE )
        {
            /* Free the buffer. */
            if ( m_pBuffer.IsAllocated ) this.m_pBuffer.Free ();

            m_Buffer = new byte[dwInternalBufSize];
            m_pBuffer = GCHandle.Alloc ( m_Buffer, GCHandleType.Pinned );

            return phCryptoASym_mBedTLS_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero, m_pBuffer.AddrOfPinnedObject (),
                ( ushort ) dwInternalBufSize );
        }

#if DEBUG
        /// <summary>
        /// Initialize the CryptoASym with mBedTLS as sub-component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oKeyStore">Pointer to a KeyStore data parameter structure.</param>
        /// <param name="dwInternalBufSize">Size of global buffer. This buffer is for processing information internally.
        ///                                 Should not be less than the defaut specified one <see cref="INTERNAL_BUFFER_SIZE"/>.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ( ushort wDataParamSize, KeyStore.Generic oKeyStore, int dwInternalBufSize = INTERNAL_BUFFER_SIZE)
        {
            /* Free the buffer. */
            if ( m_pBuffer.IsAllocated ) this.m_pBuffer.Free ();

            m_Buffer = new byte[dwInternalBufSize];
            m_pBuffer = GCHandle.Alloc ( m_Buffer, GCHandleType.Pinned );

            return phCryptoASym_mBedTLS_Init ( ref m_DataParamsInt[0], wDataParamSize, ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero,
                m_pBuffer.AddrOfPinnedObject (), ( ushort ) dwInternalBufSize );
        }
#endif
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the CryptoASym with mBedTLS as sub-component.
        ///
        /// NOTE:
        ///     Its must to call this interface to de-initialize any used global context from other libraries.
        ///     If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DeInit ()
        {
            return phCryptoASym_mBedTLS_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public mBedTLS ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~mBedTLS ()
        {
            /* Free the buffer. */
            if ( m_pBuffer.IsAllocated ) this.m_pBuffer.Free ();

            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated ) m_pDataParamsInt.Free ();
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region Stub
    /// <summary>
    /// Stub Sub-Component ASymmetric Cryptography library.
    /// </summary>
    public class Stub : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for ASymmetric Crypto Stub layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoASym_Stub_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoASym with Stub as sub-component.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ()
        {
            return phCryptoASym_Stub_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ) );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Stub ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Stub ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
}
