/*
 * Copyright 2013, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Text;

namespace NxpRdLibNet.CryptoRng
{
    #region Generic
    /// <summary>
    /// Generic Random Number Component Wrapper of the Reader Library Framework.
    ///
    /// This is only a wrapper layer to abstract the different crypto random number generator implementations.
    /// With this wrapper it is possible to support more than one crypto random number generator implementation
    /// in parallel, by adapting this wrapper.
    ///
    /// Important hints for users of this component:
    ///	- Before use of any function, the dedicated RNG implementation has to be initialized (e.g. #phCryptoRng_Sw_Init)
    ///	- Before random numbers can be generated, the #phCryptoRng_Seed function has to be called
    /// </summary>
    public abstract class Generic
    {
        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_Seed ( IntPtr pDataParams, byte[] bSeed, byte bSeedLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_Rnd ( IntPtr pDataParams, ushort wNoOfRndBytes, byte[] pRnd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_GetLastStatus ( IntPtr pDataParams, ushort dwStatusMsgLen, byte[] pStatusMsg, ref int pStatusCode );
        #endregion

        #region Wrapper Functions
        /// <summary>
        /// Seeds the random number generator with the given seed.
        ///
        /// The seed length is depended on the underlaying implementation. Further details can be found in the detailed description
        /// of the underlying layers.
        /// </summary>
        ///
        /// <param name="bSeed">Seed information.</param>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Seed ( byte[] bSeed )
        {
            return phCryptoRng_Seed ( m_pDataParams, bSeed, ( byte ) ( ( bSeed == null ) ? 0 : bSeed.Length ) );
        }

        /// <summary>
        /// Obtains random bytes from the random source.
        /// </summary>
        ///
        /// <param name="wNoOfRndBytes">Number of random bytes to generate</param>
        /// <param name="pRnd">Generated bytes; uint8_t[wNoOfRndBytes]</param>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Rnd ( ushort wNoOfRndBytes, out byte[] pRnd )
        {
            pRnd = new byte[wNoOfRndBytes];
            return phCryptoRng_Rnd ( m_pDataParams, wNoOfRndBytes, pRnd );
        }

        /// <summary>
        /// Returns the status code and corresponding message.
        /// </summary>
        ///
        /// <param name="dwStatusCode">The status code returned by the underlying Crypto library. </param>
        /// <param name="sStatusMsg">The equivalent status message for the information available in dwStatusCode.</param>
        ///
        /// <returns>Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation. </returns>
        public Status_t GetLastStatus ( out string sStatusMsg, out int dwStatusCode )
        {
            byte[] aStatusMsg = new byte[500];

            dwStatusCode = 0;
            sStatusMsg = string.Empty;
            Status_t oStatus = phCryptoRng_GetLastStatus ( m_pDataParams, ( ushort ) aStatusMsg.Length, aStatusMsg, ref dwStatusCode );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                int dwNullChar_Index = 0;
                while ( aStatusMsg[dwNullChar_Index] != 0 )
                    dwNullChar_Index++;

                if ( dwNullChar_Index > 0 )
                {
                    Array.Resize ( ref aStatusMsg, dwNullChar_Index );
                    sStatusMsg = Encoding.ASCII.GetString ( aStatusMsg );
                }
            }

            return oStatus;
        }
        #endregion

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }

    #endregion

    #region Software
    /// <summary>
    /// Software implementation of the Random Number interface
    /// </summary>
    public class Sw : Generic
    {
        #region Data Structure
        /// <summary>
        /// Software implementation of the Random Number interface
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Data parameter structure for the AES engine. </summary>
            public IntPtr pCryptoDataParams;
            public fixed byte V[16];

            /// <summary>
            /// Counts the amount of requests between two seeding procedures.
            ///
            /// Note: according to NIST SP800-90 for AES this is 2^48, for storage
            /// reasons the limit is set to 2^32 in this particular implementation.
            /// </summary>
            public uint dwRequestCounter;
            public byte bState;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_Sw_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams,
            IntPtr pCryptoDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoRng with software as sub-component.
        /// </summary>
        ///
        /// <param name="pCrypto">Pointer to the parameter structure of the symmetric crypto layer.</param>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Init ( CryptoSym.Generic pCrypto )
        {
            return phCryptoRng_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pCrypto.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }

    #endregion

    #region Stub
    /// <summary>
    /// Stub implementation of the Random Number interface
    /// </summary>
    public class Stub : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for Random Number's Stub layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this component, NEVER MODIFY!.
            /// </summary>
            public ushort wId;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_Stub_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoRng with Stub as sub-component.
        /// </summary>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Init ()
        {
            return phCryptoRng_Stub_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ) );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Stub ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Stub ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion

    #region mBedTLS
    /// <summary>
    /// mBedTLS implementation of the Random Number interface
    /// </summary>
    public class mBedTLS : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for Random Number's mBedTLS layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this component, NEVER MODIFY!.
            /// </summary>
            public ushort wId;

            /// <summary> Error code returned by mbedTLS layer. </summary>
            public int dwErrorCode;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_mBedTLS_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_mBedTLS_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoRng with mBedTLS as sub-component.
        /// </summary>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Init ()
        {
            return phCryptoRng_mBedTLS_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ) );
        }
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the CryptoRng with mBedTLS as sub-component.
        ///
        /// NOTE:
        ///		Its must to call this interface to De-Initialize any used global context from other libraries.
        ///		If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t DeInit ()
        {
            return phCryptoRng_mBedTLS_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public mBedTLS ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~mBedTLS ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region DUT
    /// <summary>
    /// DUT implementation of the Random Number interface
    /// </summary>
    public class DUT : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for Random Number's DUT layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this component, NEVER MODIFY!.
            /// </summary>
            public ushort wId;

            /// <summary> Pointer to HAL parameter structure. </summary>
            public IntPtr pHalDataParams;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_DUT_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoRng_DUT_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoRng with DUT as sub-component.
        /// </summary>
        ///
        /// <param name="oHal">Object of DUT HAL component.</param>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t Init ( Hal.Generic oHal )
        {
            return phCryptoRng_DUT_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero ) );
        }
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the CryptoRng with DUT as sub-component.
        ///
        /// NOTE:
        ///		Its must to call this interface to De-Initialize any used global context from other libraries.
        ///		If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <returns>
        ///		Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///		Other Depending on implementation and underlying component.
        ///	</returns>
        public Status_t DeInit ()
        {
            return phCryptoRng_DUT_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public DUT ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~DUT ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
}
