/*
 * Copyright 2013, 2016, 2018 - 2019, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#undef DISABLE_DES
#undef DISABLE_AES
#define ENABLE_ONLINE_KEYSCHEDULING
#define ENABLE_ONLINE_CMAC_SUBKEY_CALCULATION

using System;
using System.Text;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.CryptoSym
{
    #region Enumeration
    #region Buffering Options
    /// <summary>
    /// Enumeration for all variants of buffering flags.
    /// </summary>
    public enum OptionFlags : int
    {
        /// <summary> Default mode. Computes the Crypto and returned the information. </summary>
        DEFAULT = 0x0000,

        /// <summary>
        /// The given data is the first part of a stream. (Helper definition for buffering).
        /// </summary>
        BUFFER_FIRST = 0x8000,

        /// <summary>
        /// The given data is a subsequent part of a stream. (Helper definition for buffering).
        /// </summary>
        BUFFER_CONT = 0xC000,

        /// <summary>
        /// The given data is the last part of a stream. (Helper definition for buffering).
        /// When this flag is used the data is exchanged between PICC and reader.
        /// </summary>
        BUFFER_LAST = 0x4000,

        /// <summary>
        /// Authentication tag information is not part of Output buffer
        /// for <see cref="Generic.Encrypt">Encrypt</see> and not part of input
        /// buffer for <see cref="Generic.Decrypt">Decrypt</see> operations.
        /// </summary>
        AUTH_TAG_NOT_AVAILABLE = 0x0000,

        /// <summary>
        /// Authentication tag information is part of Output buffer
        /// for <see cref="Generic.Encrypt">Encrypt</see> and part of input
        /// buffer for <see cref="Generic.Decrypt">Decrypt</see> operations.
        /// </summary>
        AUTH_TAG_AVAILABLE = 0x0080
    }
    #endregion

    #region KeyType
    /// <summary>
    ///
    /// </summary>
    public enum KeyType : int
    {
        /// <summary> AES 128 Key [16 Bytes]. </summary>
        AES128 = 0x0000,

        /// <summary> AES 192 Key [24 Bytes]. </summary>
        AES192,

        /// <summary> AES 256 Key [32 Bytes]. </summary>
        AES256,

        /// <summary> DES Single Key [8 Bytes]. </summary>
        DES,

        /// <summary> 2 Key Triple Des [16 Bytes]. </summary>
        DES2K3,

        /// <summary> 3 Key Triple Des [24 Bytes]. </summary>
        DES3K3,

        /// <summary> MIFARE (R) Key. </summary>
        MIFARE,

        /// <summary> Invalid Key. </summary>
        INAVLID = 0xFFFF
    }
    #endregion

    #region Cipher Modes
    /// <summary>
    /// Supported Cipher Modes
    /// </summary>
    public enum CipherMode : int
    {
        /// <summary> ECB Mode. </summary>
        ECB = 0x0000,

        /// <summary> CBC Mode. </summary>
        CBC = 0x0001,

        /// <summary> CBC DF4 Mode. </summary>
        CBC_DF4 = 0x0002,

        /// <summary> CCM Mode. </summary>
        CCM = 0x0004,

        /// <summary> CCM* Mode. </summary>
        CCM_STAR = 0x0005,

        /// <summary> . </summary>
        INVALID = 0xFFFF
    }
    #endregion

    #region MAC Modes
    /// <summary>
    /// Supported Mac Modes.
    /// </summary>
    public enum MacMode : int
    {
        /// <summary> CMAC Mode. </summary>
        CMAC = 0x0000,

        /// <summary> CBCMAC Mode. </summary>
        CBCMAC,

        /// <summary> . </summary>
        INVALID
    }
    #endregion

    #region Diversification Modes
    public enum DiversificationOption : int
    {
        /// <summary> Invalid Diversification option. </summary>
        INVALID = 0xFFFF,

        /// <summary> DESFire Key Diversification. </summary>
        DESFIRE = 0x0000,

        /// <summary> MIFARE Plus Key Diversification. </summary>
        MIFARE_PLUS = 0x0001,

        /// <summary> MIFARE Ultralight Key Diversification. </summary>
        MIFARE_ULTRALIGHT = 0x0002,

        /// <summary> Option for 2K3DES full-key diversification (only with #PH_CRYPTOSYM_DIV_MODE_DESFIRE).  </summary>
        DESFIRE_2K3DES_FULL = 0x0000,

        /// <summary> Option for 2K3DES half-key diversification (only with #PH_CRYPTOSYM_DIV_MODE_DESFIRE). </summary>
        DESFIRE_2K3DES_HALF = 0x8000
    }
    #endregion

    #region Padding
    /// <summary>
    /// Supported Padding Modes.
    /// </summary>
    public enum PaddingMode : byte
    {
        /// <summary> Pad with all zeros. </summary>
        MODE_1 = 0x00,

        /// <summary> Pad with a one followed by all zeros. </summary>
        MODE_2
    }
    #endregion

    #region Configuration
    /// <summary>
    /// CryptoSym Layer Configuration types.
    /// </summary>
    public enum Config : int
    {
        /// <summary> Key Type. Read-only. Possible Values are:
        ///     - <see cref="KeyType.INAVLID"/>
        ///     - <see cref="KeyType.AES128"/>
        ///     - <see cref="KeyType.AES192"/>
        ///     - <see cref="KeyType.AES256"/>
        ///     - <see cref="KeyType.DES"/>
        ///     - <see cref="KeyType.DES2K3"/>
        ///     - <see cref="KeyType.DES3K3"/>
        /// </summary>
        KEY_TYPE = 0x0000,

        /// <summary> Key Size of currently loaded key. Read-only. </summary>
        KEY_SIZE,

        /// <summary> Block Size of currently loaded key. Read-only. </summary>
        BLOCK_SIZE,

        /// <summary>
        /// Keep init vector. Either #PH_CRYPTOSYM_VALUE_KEEP_IV_OFF or #PH_CRYPTOSYM_VALUE_KEEP_IV_ON.
        /// This flag has to be used in combination with the option flag in the <see cref="Generic.Encrypt"/>,
        /// <see cref="Generic.Decrypt"/> or <see cref="Generic.CalcualteMac"/> function: If either the option in the
        /// function or this flag is set, the IV will be updated before returning of the function. R/W access possible.
        /// </summary>
        KEEP_IV,

        /// <summary>
        /// Additional information to be provided like diversified key length
        /// </summary>
        ADDITIONAL_INFO = 0x0006,

        /// <summary>
        /// Authentication Tag. To be used when CCM or CCM* cipher modes are used.
        /// Supported values are,
        ///     - 4, 6, 8, 10, 12, 14 or 16 in case of CCM
        ///     - 0, 4, 6, 8, 10, 12, 14 or 16 in case of CCM*
        /// </summary>
        CCM_TAG_LENGTH
    }

    /// <summary>
    /// Supported IV Updated Behavior Modes.
    /// </summary>
    public enum KeepIv : int
    {
        /// <summary> Switch off Keep-IV behavior. </summary>
        OFF = 0x0000,

        /// <summary> Switch on Keep-IV behavior. </summary>
        ON
    }
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Generic Symmetric Cryptography Component Wrapper of the Reader Library Framework.
    ///
    /// This is only a wrapper layer to abstract the different CryptoSym implementations.
    /// With this wrapper it is possible to support more than one CryptoSym implementation
    /// in parallel, by adapting this wrapper.
    ///
    /// Important hints for users of this component:
    /// - Before use of any function, the dedicated crypto implementation has to be initialized using either of the interfaces mentioned below.
    ///     - <see cref="Sw.Init"/>
    ///     - <see cref="Stub.Init"/>
    ///     - <see cref="mBedTLS.Init"/>
    ///     - <see cref="DUT.Init"/>
    /// - Functions using a key store (<see cref="LoadKey"/> and <see cref="DiversifyKey"/>) are only available if a key store has been passed
    ///   during component initialization.
    /// - Before any cipher operation or MAC operation (<see cref="Encrypt"/>, <see cref="Decrypt"/>, <see cref="CalculateMac"/>) can be used, a key
    ///   has to be loaded using either <see cref="LoadKey"/> or <see cref="LoadKeyDirect"/>.
    /// - Before any cipher operation or MAC operation (<see cref="Encrypt"/>, <see cref="Decrypt"/>, <see cref="CalculateMac"/>) can be used, an
    ///   appropriate IV has to be loaded by calling <see cref="LoadIv"/>.
    /// - Using <see cref="GetConfig"/>, the block sizes and key lengths for the currently loaded key can be retrieved.
    /// - Cipher mode CCM or CCM* is supported in mBedTLS component only or based on DUT feature support.
    /// - Before any cipher operation ( <see cref="Encrypt"/> and <see cref="Decrypt"/>) with <see cref="CipherMode.CCM"/> or
    ///   <see cref="CipherMode.CCM_STAR"/> as cipher mode,
    ///    - Nonce has to be loaded by calling <see cref="LoadNonce"/>. Prior to this Key should be loaded.
    ///    - Additional Data can be set using <see cref="LoadAdditionalData"/> interface. This is optional
    ///    - Tag Length can be configured using <see cref="GetConfig"/> with <see cref="Config.CCM_TAG_LENGTH"/>
    ///      as configuration identifier.
    ///    - Tag information can be configured using <see cref="SetAuthenticationTag">Set Authentication Tag</see> for <see cref="Decrypt"/>
    ///      operation. Refer <see cref="Decrypt"/> interface for more information.
    ///    - Tag information can be retrieved using \ref phCryptoSym_GetAuthenticationTag "Get Authentication Tag" for <see cref="Encrypt"/>
    ///      operation. Refer <see cref="Encrypt"/> interface for more details.
    ///
    /// Note: The following are applicable when CryptoSym is initialized to use <see cref="mBedTLS.Init"/> as underlying layer.
    ///     - CMAC implementation of mBedTLS library is not utilized due to below mentioned reason(s)
    ///        - When using <see cref="alMfdfEVx"/> AL component, CMAC computation leaving the first call requires
    ///          IV of the last subsequent calls. Here the IV is only zero for the first call and non zero for the rest of the
    ///          calls.This behavior is required for EV1 Secure messaging of MIFARE DESFire product.
    ///        - Its not possible to update the IV for intermediate / final calls provide by mBedTLS.
    ///        - To over come this, CMAC is implemented directly in this component using cipher interfaces of mBedTLS.
    ///        - The above limitation is valid only for <see cref="CalculateMac"/> interface.
    ///    - CMAC implementation of mBedTLS library is not utilized due to below mentioned reason (s)
    ///        - CMAC implementation provided by mBedTLS library do not support 3DES-2Key key diversification.
    ///        - Based on<a href="https://www.nxp.com/docs/en/application-note/AN10922.pdf"><b>AN10922</b></a> CryptoSym should
    ///          support diversification of 3DES-2Key but mBedTLS CMAC library do not support this key type.
    ///        - To over come this, CMAC is implemented directly in this component using cipher interfaces of mBedTLS.
    ///        - The above limitation is valid only for below mentioned interfaces.
    ///            - <see cref="DiversifyKey()">Diversify the key available in KeyStore</see>
    ///            - <see cref="DiversifyDirectKey()">Diversify the provided key as input</see>
    /// </summary>
    public abstract class Generic
    {
        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_InvalidateKey ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_Encrypt ( IntPtr pDataParams, ushort wOption, byte[] pPlainBuffer, ushort nBufferLength,
            byte[] pEncryptedBuffer );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_Decrypt ( IntPtr pDataParams, ushort wOption, byte[] pEncryptedBuffer,
            ushort wEncryptedBufferLength, byte[] pPlainBuffer );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_CalculateMac ( IntPtr pDataParams, ushort wOption, byte[] pData, ushort wDataLength,
            byte[] pMac, ref byte pMacLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_LoadIv ( IntPtr pDataParams, byte[] pIV, byte bIVLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_LoadNonce ( IntPtr pDataParams, byte bCipher, byte[] pNonce, byte bNonceLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_LoadAdditionalData ( IntPtr pDataParams, byte[] pAddData, ushort wAddData_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_LoadKey ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVersion, ushort wKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_LoadKeyDirect ( IntPtr pDataParams, byte[] pKey, ushort wKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_DiversifyKey ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVersion,
            byte[] pDivInput, byte bLenDivInput, byte[] pDiversifiedKey );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_DiversifyDirectKey ( IntPtr pDataParams, ushort wOption, byte[] pKey, ushort wKeyType,
            byte[] pDivInput, byte bLenDivInput, byte[] pDiversifiedKey );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_ApplyPadding ( byte bOption, byte[] pDataIn, ushort wDataInLength, byte bBlockSize,
            ushort wDataOutBufSize, byte[] pDataOut, ref ushort pDataOutLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_RemovePadding ( byte bOption, byte[] pDataIn, ushort wDataInLength, byte bBlockSize,
            ushort wDataOutBufSize, byte[] pDataOut, ref ushort pDataOutLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_SetAuthenticationTag ( IntPtr pDataParams, byte[] pTag, byte bTag_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_GetAuthenticationTag ( IntPtr pDataParams, byte[] pTag, ref byte pTag_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_GetLastStatus ( IntPtr pDataParams, ushort dwStatusMsgLen, byte[] pStatusMsg, ref int pStatusCode );
        #endregion

        #region Properties
        /// <summary> Returns the Key from CryptoSym context. </summary>
        public abstract byte[] pKey
        {
            get;
        }

        /// <summary> Returns the IV from CryptoSym context. </summary>
        public abstract byte[] pIV
        {
            get;
        }

        /// <summary> Returns the additional from CryptoSym context. </summary>
        public abstract byte[] Additional_Data
        {
            get;
        }
        #endregion

        #region Wrapper Functions
        /// <summary>
        /// Invalidate the currently loaded key.
        /// Resets the key, the IV, the keep IV flag and the key Type.
        /// </summary>
        ///
        /// <returns>Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation. </returns>
        public Status_t InvalidateKey ()
        {
            return phCryptoSym_InvalidateKey ( m_pDataParams );
        }

        /// <summary>
        /// Perform Encryption with one of the supported crypto modes
        /// </summary>
        ///
        /// <param name="CipherMode">Option byte specifying the cipher mode and the update behavior of the IV
        ///                             <see cref="CipherMode.ECB"/>
        ///                             <see cref="CipherMode.CBC"/>
        ///                             <see cref="CipherMode.CBC_DF4"/>
        ///                             <see cref="CipherMode.CCM"/>
        ///                             <see cref="CipherMode.CCM_STAR"/>
        /// </param>
        /// <param name="Flags">One of the below flag values.
        ///                         - To be ORed with Authentication Tag for CCM and CCM* operation modes
        ///                         - <see cref="OptionFlags.AUTH_TAG_NOT_AVAILABLE">Authentication Tag not available</see>:
        ///                           \b pEncryptedBuffer will have Encrypted data only and not have Authentication tag data.
        ///                         - <see cref="OptionFlags.AUTH_TAG_AVAILABLE">Authentication Tag Available</see>:
        ///                           \b pEncryptedBuffer will have Encrypted data followed by Authentication tag data.
        ///
        ///                         - To be ORed with Buffering Flags
        ///                             - <see cref="OptionFlags.DEFAULT"/>
        ///                             - The below ones are not supported by <see cref="CipherMode.CCM"/> and <see cref="CipherMode.CCM_STAR"/>
        ///                               cipher modes. These are supported by other cipher modes.
        ///                                 - <see cref="OptionFlags.BUFFER_FIRST"/>
        ///                                 - <see cref="OptionFlags.BUFFER_CONT"/>
        ///                                 - <see cref="OptionFlags.BUFFER_LAST"/>
        /// </param>
        /// <param name="pPlainBuffer">Plain data buffer.
        ///                         - Should always be in multiple of current block size.
        ///                         - If not of current block size then <see cref="ApplyPadding"/>
        ///                           needs to be used to make it upto current block size.
        ///                         - For CCM or CCM* cipher modes multiple of block size is not applicable.
        /// </param>
        /// <param name="pEncryptedBuffer">Encrypted data buffer.
        ///                                 - Allocation should be Should be equal to \b pPlainBuffer.
        ///                                 - For CCM or CCM* cipher modes,
        ///                                     - \b wOption = \ref PH_CRYPTOSYM_AUTH_TAG_OFF "Authentication Tag not available":
        ///                                       Should be equal to \b pPlainBuffer.
        ///                                     - \b wOption = \ref PH_CRYPTOSYM_AUTH_TAG_ON "Authentication Tag Available":
        ///                                       Should be equal to \b pPlainBuffer + Authentication Tag length to store
        ///                                       Encrypted data + Authentication Tag.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> An unsupported key is loaded (or no key is loaded) or \b pPlainBuffer is not
        ///     a multiple of the current block size.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Encrypt ( CipherMode CipherMode, OptionFlags Flags, byte[] pPlainBuffer, out byte[] pEncryptedBuffer )
        {
            ushort wOption = ( ushort ) ( ( int ) Flags | ( int ) CipherMode );
            ushort wBuffLen = ( ushort ) ( ( pPlainBuffer == null ) ? 0 : pPlainBuffer.Length );

            int dwTagLen = 0;
            pEncryptedBuffer = null;
            if ( ( ( CipherMode == CipherMode.CCM ) || ( CipherMode == CipherMode.CCM_STAR ) ) &&
                ( ( ushort ) Flags & ( ushort ) OptionFlags.AUTH_TAG_AVAILABLE )
                .Equals ( ( ushort ) OptionFlags.AUTH_TAG_AVAILABLE ) )
            {
                Status_t oStatus = GetConfig ( ( int ) Config.CCM_TAG_LENGTH, out dwTagLen );
                if ( !oStatus.Error.Equals ( Error_Gen.SUCCESS ) )
                    return oStatus;
            }

            pEncryptedBuffer = ( pPlainBuffer == null ) ? null : new byte[pPlainBuffer.Length + dwTagLen];

            return phCryptoSym_Encrypt ( m_pDataParams, wOption, pPlainBuffer, wBuffLen, pEncryptedBuffer );
        }

        /// <summary>
        /// Perform Decryption with one of the supported crypto modes
        /// </summary>
        ///
        /// <param name="CipherMode">Option byte specifying the cipher mode and the update behavior of the IV
        ///                             <see cref="CipherMode.ECB"/>
        ///                             <see cref="CipherMode.CBC"/>
        ///                             <see cref="CipherMode.CBC_DF4"/>
        /// </param>
        /// <param name="Flags">One of the below flag values.
        ///                         - To be ORed with Authentication Tag for CCM and CCM* operation modes
        ///                         - <see cref="OptionFlags.AUTH_TAG_NOT_AVAILABLE">Authentication Tag not available</see>:
        ///                           \b pEncryptedBuffer will have Encrypted data only and not have Authentication tag data.
        ///                         - <see cref="OptionFlags.AUTH_TAG_AVAILABLE">Authentication Tag Available</see>:
        ///                           \b pEncryptedBuffer will have Encrypted data followed by Authentication tag data.
        ///
        ///                         - To be ORed with Buffering Flags
        ///                             - <see cref="OptionFlags.DEFAULT"/>
        ///                             - The below ones are not supported by <see cref="CipherMode.CCM"/> and <see cref="CipherMode.CCM_STAR"/>
        ///                               cipher modes. These are supported by other cipher modes.
        ///                                 - <see cref="OptionFlags.BUFFER_FIRST"/>
        ///                                 - <see cref="OptionFlags.BUFFER_CONT"/>
        ///                                 - <see cref="OptionFlags.BUFFER_LAST"/>
        /// </param>
        /// <param name="pEncryptedBuffer">Encrypted data buffer.
        ///                                 - Should always be in multiple of current block size.
        ///                                 - If not of current block size then \ref phCryptoSym_ApplyPadding
        ///                                   "Apply Padding" needs to be used to make it upto current block size.
        ///                                 - For CCM or CCM* cipher modes multiple of block size is not applicable.
        /// </param>
        /// <param name="pPlainBuffer">Plain data buffer.
        ///                             - Will always be in multiple of current block size. Plain data may be
        ///                               padded with zeros if not current block size and needs to be removed
        ///                               using <see cref="RemovePadding"/> interface.
        ///                             - For CCM or CCM* cipher modes,
        ///                                 - \b wOption = \ref PH_CRYPTOSYM_AUTH_TAG_OFF "Authentication Tag not available":
        ///                                   Should be equal to \b pPlainBuffer.
        ///                                 - \b wOption = \ref PH_CRYPTOSYM_AUTH_TAG_ON "Authentication Tag Available":
        ///                                   Will contain the decrypted data.User has to remove Tag Length from the input
        ///                                   length specified
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> An unsupported key is loaded (or no key is loaded) or
        ///     \b pPlainBuffer is not a multiple of the current block size.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Decrypt ( CipherMode CipherMode, OptionFlags Flags, byte[] pEncryptedBuffer, out byte[] pPlainBuffer )
        {
            Status_t oStatus;
            ushort wOption = ( ushort ) ( ( int ) Flags | ( int ) CipherMode );
            ushort wBuffLen = ( ushort ) ( ( pEncryptedBuffer == null ) ? 0 : pEncryptedBuffer.Length );

            int dwTagLen = 0;
            pPlainBuffer = null;
            if ( ( ( CipherMode == CipherMode.CCM ) || ( CipherMode == CipherMode.CCM_STAR ) ) &&
                ( ( ushort ) Flags & ( ushort ) OptionFlags.AUTH_TAG_AVAILABLE )
                .Equals ( ( ushort ) OptionFlags.AUTH_TAG_AVAILABLE ) )
            {
                oStatus = GetConfig ( ( int ) Config.CCM_TAG_LENGTH, out dwTagLen );
                if ( !oStatus.Error.Equals ( Error_Gen.SUCCESS ) )
                    return oStatus;
            }

            pPlainBuffer = ( pEncryptedBuffer == null ) ? null : new byte[pEncryptedBuffer.Length];

            oStatus = phCryptoSym_Decrypt ( m_pDataParams, wOption, pEncryptedBuffer, wBuffLen, pPlainBuffer );
            if ( oStatus.Error.Equals ( Error_Gen.SUCCESS ) )
                Array.Resize ( ref pPlainBuffer, ( wBuffLen - dwTagLen ) );

            return oStatus;
        }

        /// <summary>
        /// Calculate MAC with one of the supported MAC modes
        ///
        /// Note: If <see cref="OptionFlags.BUFFER_FIRST"/> or <see cref="OptionFlags.BUFFER_CONT"/>, the input length needs to be a
        /// multiple of the block length!
        /// </summary>
        ///
        /// <param name="MacMode">Option byte specifying the MAC mode and the update behavior of the IV and the completion flag.
        ///                             <see cref="MacMode.CBCMAC"/>
        ///                             <see cref="MacMode.CMAC"/>
        /// </param>
        /// <param name="Flags">One of the below flag values.
        ///                         - <see cref="OptionFlags.DEFAULT"/>
        ///                         - <see cref="OptionFlags.BUFFER_FIRST"/>
        ///                         - <see cref="OptionFlags.BUFFER_CONT"/>
        ///                         - <see cref="OptionFlags.BUFFER_LAST"/>
        /// </param>
        /// <param name="pData">Input data on which the MAC needs to be conmputed.
        ///                     Input will be always be in multiple of current block size if Flags is
        ///                         - <see cref="OptionFlags.BUFFER_FIRST"/>
        ///                         - <see cref="OptionFlags.BUFFER_CONT"/>
        /// </param>
        /// <param name="pMac">Output MAC block; 16 bytes</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> An unsupported key is loaded (or no key is loaded) or \b pPlainBuffer is not
        ///     a multiple of the current block size.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t CalculateMac ( MacMode MacMode, OptionFlags Flags, byte[] pData, out byte[] pMac )
        {
            ushort wOption = ( ushort ) ( ( int ) Flags | ( int ) MacMode );
            pMac = new byte[16];
            byte pMacLength = 0;
            Status_t status = phCryptoSym_CalculateMac(m_pDataParams, wOption, pData, (ushort) ( (pData == null) ? 0 : pData.Length),
                pMac, ref pMacLength);
            Array.Resize<byte> ( ref pMac, pMacLength );
            return status;
        }

        /// <summary>
        /// Load IV
        /// </summary>
        ///
        /// <param name="pIV">Initialization vector to use. Should of current block size.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> pIV does not match the current block size.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadIv ( byte[] pIV )
        {
            return phCryptoSym_LoadIv ( m_pDataParams, pIV, ( byte ) ( ( pIV == null ) ? 0 : pIV.Length ) );
        }

        /// <summary>
        /// Load Initialization vector (Nonce).
        /// Can be used for below cipher modes
        ///     - <see cref="CipherMode.CCM">Counter with Cipher Block Chaining-Message Authentication Code</see>
        ///     - <see cref="CipherMode.CCM_STAR">Counter with Cipher Block Chaining-Message Authentication Code optional</see>
        ///
        /// Note: Support is available for mBedTLS component only or based on DUT feature support.
        /// </summary>
        ///
        /// <param name="bCipher">One of the cipher modes mentioned in the description</param>
        /// <param name="aNonce">Initialization vector to use. Should based on cipher mode.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Key Type not supported.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadNonce ( byte bCipher, byte[] aNonce )
        {
            return phCryptoSym_LoadNonce ( m_pDataParams, bCipher, aNonce, ( byte ) ( ( aNonce == null ) ? 0 : aNonce.Length ) );
        }

        /// <summary>
        /// Load Additional Data Field.
        /// This will be used by <see cref="Encrypt"/> and <see cref="Decrypt"/>
        /// when Cipher Mode is <see cref="CipherMode.CCM"/> or <see cref="CipherMode.CCM_STAR"/>
        /// </summary>
        ///
        /// <param name="aAddData">Additional Data to use for CCM cipher mode.
        ///                        Maximum size should be less than 2^16 - 2^8 = 65280.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if aAddData is Null
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/> if aAddData is higher than the
        ///     maximum allowed one.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadAdditionalData ( byte[] aAddData )
        {
            return phCryptoSym_LoadAdditionalData ( m_pDataParams, aAddData, ( ushort ) ( ( aAddData == null ) ? 0 : aAddData.Length ) );
        }

        /// <summary>
        /// Load Key
        ///
        /// This function uses the key storage provided at component initialization to retrieve the key identified by wKeyNo and wKeyVersion.
        /// After retrieving the key is loaded into the internal key storage array to be prepared for subsequent cipher operations.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number in KeyStore to be loaded.</param>
        /// <param name="wKeyVersion">Key Version in KeyStore to be loaded.</param>
        /// <param name="wKeyType">Type of Key to be loaded. Supported ones are
        ///                             - <see cref="KeyType.AES128"/>
        ///                             - <see cref="KeyType.AES192"/>
        ///                             - <see cref="KeyType.AES256"/>
        ///                             - <see cref="KeyType.DES"/>
        ///                             - <see cref="KeyType.DES2K3"/>
        ///                             - <see cref="KeyType.DES3K3"/>
        ///                             - <see cref="KeyType.MIFARE"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Key Type not supported.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadKey ( int wKeyNo, int wKeyVersion, KeyType wKeyType )
        {
            return phCryptoSym_LoadKey ( m_pDataParams, ( ushort ) wKeyNo, ( ushort ) wKeyVersion, ( ushort ) wKeyType );
        }

        /// <summary>
        /// Direct Load Key
        ///
        /// The key provided in the pKey parameter is loaded into the internal key storage array to be prepared for subsequent cipher operations.
        /// </summary>
        ///
        /// <param name="pKey">Key to be loaded. Number of bytes should be based on the key type mentioned in \b wKeyType parameter.
        /// </param>
        /// <param name="wKeyType">Type of Key to be loaded. Supported ones are
        ///                             - <see cref="KeyType.AES128"/>
        ///                             - <see cref="KeyType.AES192"/>
        ///                             - <see cref="KeyType.AES256"/>
        ///                             - <see cref="KeyType.DES"/>
        ///                             - <see cref="KeyType.DES2K3"/>
        ///                             - <see cref="KeyType.DES3K3"/>
        ///                             - <see cref="KeyType.MIFARE"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Key Type not supported.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t LoadKeyDirect ( byte[] pKey, KeyType wKeyType )
        {
            return phCryptoSym_LoadKeyDirect ( m_pDataParams, pKey, ( ushort ) wKeyType );
        }

        /// <summary>
        /// Diversify Key - Note: This function invalidates the currently loaded key.
        ///
        /// Using the key stored in the KeyStore passed at initialization of the component and identified by wKeyNo and wKeyVersion
        /// this function calculates a diversified key according to the wOption specified that can be used in different applications.
        /// </summary>
        ///
        /// <param name="wOption">Option to specify the diversification method.
        ///                       One of the below mentioned information.
        ///                         - <see cref="DiversificationOption.DESFIRE"/>
        ///                         - <see cref="DiversificationOption.MIFARE_PLUS"/>
        ///                         - <see cref="DiversificationOption.MIFARE_ULTRALIGHT"/>
        ///                         - <see cref="DiversificationOption.DESFIRE_2K3DES_FULL"/>
        ///                         - <see cref="DiversificationOption.DESFIRE_2K3DES_HALF"/>
        /// </param>
        /// <param name="wKeyNo">Key number in KeyStore to be loaded</param>
        /// <param name="wKeyVersion">Key Version in KeyStore to be loaded</param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="pDiversifiedKey">Diversified key. Will be of current block size.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Key Type not supported (for key diversification).
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/> Length of diversification input is wrong.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DiversifyKey ( DiversificationOption wOption, int wKeyNo, int wKeyVersion, byte[] pDivInput, out byte[] pDiversifiedKey )
        {
            pDiversifiedKey = new byte[32];

            Status_t oStatus = phCryptoSym_DiversifyKey ( m_pDataParams, ( ushort ) wOption, ( ushort ) wKeyNo, ( ushort ) wKeyVersion, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDiversifiedKey );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                ushort wValue = 0;
                Status_t oStatus1 = phCryptoSym_GetConfig ( m_pDataParams, ( ushort ) Config.ADDITIONAL_INFO, ref wValue );
                if ( oStatus1.Equals ( new Status_t () ) )
                    Array.Resize ( ref pDiversifiedKey, wValue );
            }

            return oStatus;
        }

        /// <summary>
        /// Diversify Direct Key - Note: This function invalidates the currently loaded key.
        ///
        /// Using the key passed in the pKey parameter this function calculates a diversified key according to the wOption
        /// specified that can be used in different applications.
        /// </summary>
        ///
        /// <param name="wOption">Option to specify the diversification method.
        ///                       One of the below mentioned information.
        ///                         - <see cref="DiversificationOption.DESFIRE"/>
        ///                         - <see cref="DiversificationOption.MIFARE_PLUS"/>
        ///                         - <see cref="DiversificationOption.MIFARE_ULTRALIGHT"/>
        ///                         - <see cref="DiversificationOption.DESFIRE_2K3DES_FULL"/>
        ///                         - <see cref="DiversificationOption.DESFIRE_2K3DES_HALF"/>
        /// </param>
        /// <param name="pKey">Key to be loaded. Number of bytes should be based on the key type mentioned in \b wKeyType parameter.</param>
        /// <param name="wKeyType">Type of Key to be loaded. Supported ones are
        ///                             - <see cref="KeyType.AES128"/>
        ///                             - <see cref="KeyType.AES192"/>
        ///                             - <see cref="KeyType.AES256"/>
        ///                             - <see cref="KeyType.DES"/>
        ///                             - <see cref="KeyType.DES2K3"/>
        ///                             - <see cref="KeyType.DES3K3"/>
        ///                             - <see cref="KeyType.MIFARE"/>
        /// </param>
        /// <param name="pDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="pDiversifiedKey">Diversified key. Will be of current block size.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Key Type not supported (for key diversification).
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/> Length of diversification input is wrong.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DiversifyDirectKey ( DiversificationOption wOption, byte[] pKey, KeyType wKeyType, byte[] pDivInput, out byte[] pDiversifiedKey )
        {
            pDiversifiedKey = new byte[32];
            Status_t oStatus = phCryptoSym_DiversifyDirectKey ( m_pDataParams, ( ushort ) wOption, pKey, ( ushort ) wKeyType, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pDiversifiedKey );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                ushort wValue = 0;
                Status_t oStatus1 = phCryptoSym_GetConfig ( m_pDataParams, ( ushort ) Config.ADDITIONAL_INFO, ref wValue );
                if ( oStatus1.Equals ( new Status_t () ) )
                    Array.Resize ( ref pDiversifiedKey, wValue );
            }

            return oStatus;
        }

        /// <summary>
        /// Apply Padding to a given data buffer.
        /// </summary>
        ///
        /// <param name="bOption">Specifies padding mode
        ///                         - <see cref="PaddingMode.MODE_1"/>
        ///                         - <see cref="PaddingMode.MODE_2"/>
        /// </param>
        /// <param name="pDataIn">Input data for which padding is required.</param>
        /// <param name="bBlockSize">Block size to be used for padding.</param>
        /// <param name="wDataOutBufSize">Size of output data buffer.</param>
        /// <param name="pDataOut">Output data containing the information with padded bytes added.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> Unsupported bOption.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> \b wDataOutBufSize is too small.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ApplyPadding ( PaddingMode bOption, byte[] pDataIn, byte bBlockSize, int wDataOutBufSize, out byte[] pDataOut )
        {
            ushort pDataOutLength = 0;
            pDataOut = new byte[wDataOutBufSize];
            Status_t status = phCryptoSym_ApplyPadding((byte)bOption, pDataIn, (ushort)pDataIn.Length, bBlockSize, (ushort)wDataOutBufSize, pDataOut, ref pDataOutLength);
            System.Array.Resize<byte> ( ref pDataOut, pDataOutLength );
            return status;

        }

        /// <summary>
        /// Remove Padding to a given data buffer.
        /// </summary>
        ///
        /// <param name="bOption">Specifies padding mode
        ///                         - <see cref="PaddingMode.MODE_1"/>
        ///                         - <see cref="PaddingMode.MODE_2"/>
        /// </param>
        /// <param name="pDataIn">Input data from which padding should be removed.</param>
        /// <param name="bBlockSize">Block size to be used for padding</param>
        /// <param name="wDataOutBufSize">Size of output data buffer</param>
        /// <param name="pDataOut">Output data containing the information with padded bytes removed.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> Unsupported \b bOption or \b wDataInLength is not a multiple of the bBlockSize parameter.
        ///     Returns <see cref="Error_Comm.FRAMING_ERROR"/> Padding byte wrong. Expected 80h as the first padding byte if \b bOption = <see cref="PaddingMode.MODE_2"/>.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t RemovePadding ( PaddingMode bOption, byte[] pDataIn, byte bBlockSize, int wDataOutBufSize, out byte[] pDataOut )
        {
            ushort pDataOutLength = 0;
            pDataOut = new byte[wDataOutBufSize];
            Status_t status = phCryptoSym_RemovePadding((byte)bOption, pDataIn, (ushort)pDataIn.Length, bBlockSize, (ushort)wDataOutBufSize, pDataOut, ref pDataOutLength);
            System.Array.Resize<byte> ( ref pDataOut, pDataOutLength );
            return status;
        }

        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier. One of the below mentioned ones,
        ///                         - <see cref="Config.KEEP_IV"/>
        ///                         -<see cref="Config.ADDITIONAL_INFO"/>
        ///                         -<see cref="Config.CCM_TAG_LENGTH"/>
        /// </param>
        /// <param name="wValue">Configuration Value for the provided configuration identifier.
        ///                         - <see cref="KeepIv.OFF"/>
        ///                         - <see cref="KeepIv.ON"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> Valid wConfig but invalid wValue for that config.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Invalid (Unsupported) wConfig.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetConfig ( int wConfig, int wValue )
        {
            return phCryptoSym_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        /// <summary>
        /// Load Authentication Tag information.
        /// This will be used by <see cref="Decrypt"/> when Cipher Mode is <see cref="CipherMode.CCM"/> and <see cref="CipherMode.CCM_STAR"/>.
        /// Refer <see cref="Decrypt"/> for more information.
        /// </summary>
        ///
        /// <param name="aTag">Authentication tag to use for CCM or CCM* decryption</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aTag is null.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b bTag_Len is higher than the maximum allowed one which is 16 bytes.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetAuthenticationTag ( byte[] aTag )
        {
            return phCryptoSym_SetAuthenticationTag ( m_pDataParams, aTag, ( byte ) ( ( aTag == null ) ? 0 : aTag.Length ) );
        }

        /// <summary>
        /// Get Authentication Tag information.
        /// This will be used by <see cref="Encrypt"/> when Cipher Mode is <see cref="CipherMode.CCM"/> and <see cref="CipherMode.CCM_STAR"/>.
        /// Refer <see cref="Encrypt"/> for more information.
        /// </summary>
        ///
        /// <param name="aTag">Authentication tag generated for CCM or CCM* encryption</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b pTag and \b pTag_Len is Null
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetAuthenticationTag ( out byte[] aTag )
        {
            byte bTagLen = 0;
            aTag = new byte[16];

            Status_t oStatus = phCryptoSym_GetAuthenticationTag ( m_pDataParams, aTag, ref bTagLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                Array.Resize ( ref aTag, bTagLen );
            }
            else
                aTag = null;

            return oStatus;
        }

        /// <summary>
        /// Get configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier. One of the below mentioned ones,
        ///                         - <see cref="Config.KEY_TYPE"/>
        ///                         - <see cref="Config.KEY_SIZE"/>
        ///                         - <see cref="Config.BLOCK_SIZE"/>
        ///                         - <see cref="Config.KEEP_IV"/>
        ///                         -<see cref="Config.ADDITIONAL_INFO"/>
        ///                         -<see cref="Config.CCM_TAG_LENGTH"/>
        /// </param>
        /// <param name="pValue">Configuration Value for the provided configuration identifier.
        ///                         - Refer <see cref="KeyType"/> for <see cref="Config.KEY_TYPE"/> configuration identifier.
        ///                         - Refer <see cref="KeepIv"/> for <see cref="Config.KEEP_IV"/> configuration identifier.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> Value behind wConfig not valid at the moment.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> Invalid (Unsupported) wConfig.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetConfig ( int wConfig, out int pValue )
        {
            Status_t status;
            ushort wValue = 0;
            status = phCryptoSym_GetConfig ( m_pDataParams, ( ushort ) wConfig, ref wValue );
            pValue = ( int ) wValue;
            return status;
        }

        /// <summary>
        /// Returns the status code and corresponding message.
        /// </summary>
        ///
        /// <param name="dwStatusCode">The status code returned by the underlying Crypto library. </param>
        /// <param name="sStatusMsg">The equivalent status message for the information available in dwStatusCode.</param>
        ///
        /// <returns>Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation. </returns>
        public Status_t GetLastStatus ( out string sStatusMsg, out int dwStatusCode )
        {
            byte[] aStatusMsg = new byte[500];

            dwStatusCode = 0;
            sStatusMsg = string.Empty;
            Status_t oStatus = phCryptoSym_GetLastStatus ( m_pDataParams, ( ushort ) aStatusMsg.Length, aStatusMsg, ref dwStatusCode );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                int dwNullChar_Index = 0;
                while ( aStatusMsg[dwNullChar_Index] != 0 )
                    dwNullChar_Index++;

                if ( dwNullChar_Index > 0 )
                {
                    Array.Resize ( ref aStatusMsg, dwNullChar_Index );
                    sStatusMsg = Encoding.ASCII.GetString ( aStatusMsg );
                }
            }

            return oStatus;
        }
        #endregion

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion

    #region Software
    /// <summary>
    /// Software implementation of the Symmetric Cryptography interface.
    ///
    /// This implementation was designed to optimize the footprint of crypto libraries used in embedded systems.
    /// The following standards are implemented:
    /// - Federal Information Processing Standards Publication 197: AES 128, 192 and 256
    /// - Federal Information Processing Standards Publication 46-3: DES
    /// - NIST Special Publication 800-67 Recommendation for the Triple Data Encryption Algorithm (TDEA) Block Cipher
    /// - NIST Special Publication 800-38B: CMAC
    /// - NIST Special Publication 800-38A: CBC and ECB mode
    /// - NIST Special Publication 800-38A: CMC-MAC
    ///
    /// Hints for compiling the library:
    /// - Carefully read the section on compile switches in order to find the optimum balance between speed and memory utilization.
    /// - Using the appropriate compile switches either AES or DES can be removed from the built completely.
    ///
    /// Architecture of the phCryptoSym_Sw Component:
    /// - The DES algorithm is implemented in the phCryptoSym_Sw_Des block
    /// - The AES algorithm is implemented in the phCryptoSym_Sw_Aes block
    /// - The phCryptoSym_Int block implements generic encrypt and decrypt functions.This offers the possibility to implement modes
    ///   of operations without consideration of currently selected key type or key size.
    ///   - The phCryptoSym_Int block in addition implements helper functions for CMAC calculations.
    /// </summary>
    public class Sw : Generic
    {
        #region Private Variables
        /// <summary> Key size in DES algorithm for 56 bit key. </summary>
        public const byte DES_KEY_SIZE = 8;

        /// <summary> Block size in DES algorithm. </summary>
        public const byte DES_BLOCK_SIZE = 8;

        /// <summary> Block size in AES algorithm. </summary>
        public const byte AES_BLOCK_SIZE = 16;
#if DISABLE_AES
#if DISABLE_DES
#error PLEASE SPECIFY AT LEAST ONE CIPHER
#else
        /// <summary> Maximum Block Size of the currently supported ciphers </summary>
        public const byte MAX_BLOCK_SIZE = DES_BLOCK_SIZE;
#endif
#else
        /// <summary> Maximum Block Size of the currently supported ciphers </summary>
        public const byte MAX_BLOCK_SIZE = AES_BLOCK_SIZE;
#endif
#if ENABLE_ONLINE_KEYSCHEDULING
        /// <summary> Key buffer size is calculated as follows:
        ///     - DES offline key scheduling: 3 #numKeys * 16 #numRounds * 8 #KeySize = 384 Bytes
        ///     - DES online key scheduling: 3 #numKeys * 2 #temporaryKey+originalKey * 8 #KeySize + 8 #intermediate result = 56 Bytes
        ///     - AES offline key scheduling: (13 + 2) (#numRounds + #original) * 16 #KeySize = 240 Bytes
        ///     - AES online key scheduling: (1 + 1) (#temporary + #original) * 32 #KeySize = 64 Bytes
        /// </summary>
        public const ushort PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE = 32;
#else
#if DISABLE_DES
#if DISABLE_AES
#error PLEASE SPECIFY AT LEAST ONE CIPHER
#else
        public const ushort PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE = 240;
#endif
#else
        public const ushort PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE = 384;
#endif
#endif
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for Symmetric Crypto Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY! </summary>
            public ushort wId;

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
            /// <summary> This specifies whether LRP Mode is enabled or not. </summary>
            public byte bLrpEnabled;

            /// <summary> This specifies the Number of times the Key to be updated while generating the SPT.UpdatedKey. </summary>
            public byte bNumofKeystoUpdate;

            /// <summary> This specifies the counter maintained which to increment for every Encryption/Decryption. </summary>
            public uint dwEncCtr;
#endif

            /// <summary> Pointer to Key Store object - can be NULL. </summary>
            public IntPtr pKeyStoreDataParams;

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
            /// <summary> Internal key storage array. </summary>
            public fixed byte pKey[PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 16];
#else
            /// <summary> . </summary>
            public fixed byte pKey[PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE];
#endif

            /// <summary> Internal IV storage array. </summary>
            public fixed byte pIV[MAX_BLOCK_SIZE];

#if ENABLE_ONLINE_CMAC_SUBKEY_CALCULATION

#else
            /// <summary> Internal Key1 storage for MAC calculation. </summary>
            public fixed byte pCMACSubKey1[MAX_BLOCK_SIZE];

            /// <summary> Internal Key2 storage for MAC calculation. </summary>
            public fixed byte pCMACSubKey2[MAX_BLOCK_SIZE];

            /// <summary> Indicates whether the subkeys have been calculated. </summary>
            public byte bCMACSubKeysInitialized;
#endif
            /// <summary> Key Type. </summary>
            public ushort wKeyType;

            /// <summary>
            /// Indicates if the init vector of a previous crypto operation shall be used for the next operation.
            /// </summary>
            public ushort wKeepIV;

            /// <summary>
            /// Additional information like diversified key length, etc.
            /// </summary>
            public ushort wAddInfo;
        };
        #endregion

        #region Properties
        /// <summary>
        /// Override aKey from abstract base class
        /// </summary>
        public override byte[] pKey
        {
            get
            {
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
                byte[] pKey = new byte[PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 16];
#else
                byte[] pKey = new byte[PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE];
#endif
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < pKey.Length; i++ )
                        {
                            pKey[i] = pDataParams->pKey[i];
                        }
                    }
                }
                return pKey;
            }
        }

        /// <summary> Override IV from abstract base class </summary>
        public override byte[] pIV
        {
            get
            {
                byte[] pIV = new byte[MAX_BLOCK_SIZE];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < MAX_BLOCK_SIZE; i++ )
                        {
                            pIV[i] = pDataParams->pIV[i];
                        }
                    }
                }
                return pIV;
            }
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary> Override Encryption Counter from abstract base class </summary>
        public uint dwEncCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->dwEncCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->dwEncCtr = value;
                    }
                }
            }
        }
#endif
        /// <summary> Override CCM or CCM* Additional Data from abstract base class </summary>
        public override byte[] Additional_Data
        {
            get
            {
                throw new Exception ( "Not Implemented" );
            }
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_Sw_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pKeyStoreDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoSym with Software as sub-component.
        /// </summary>
        ///
        /// <param name="pKeyStore">Pointer to a key store structure (can be null).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        /// </returns>
        public Status_t Init ( KeyStore.Generic pKeyStore )
        {
            IntPtr pKeyStorePtr;
            if ( pKeyStore != null )
            {
                pKeyStorePtr = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStorePtr = IntPtr.Zero;
            }

            return phCryptoSym_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pKeyStorePtr );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region Stub
    /// <summary>
    /// Initialize the CryptoSym with Stub as sub-component.
    /// </summary>
    public class Stub : Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for Symmetric Crypto Stub layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY! </summary>
            public ushort wId;
        };
        #endregion

        #region Properties
        /// <summary>
        /// Override Key property from abstract base class
        /// </summary>
        public override byte[] pKey
        {
            get
            {
                throw new Exception ( "Not Implemented" );
            }
        }

        /// <summary>
        /// Override IV property from abstract base class
        /// </summary>
        public override byte[] pIV
        {
            get
            {
                throw new Exception ( "Not Implemented" );
            }
        }

        /// <summary> Override CCM or CCM* Additional Data from abstract base class </summary>
        public override byte[] Additional_Data
        {
            get
            {
                throw new Exception ( "Not Implemented" );
            }
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_Stub_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoSym with Stub as sub-component.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successfull operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ()
        {
            return phCryptoSym_Stub_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ) );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Stub ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Stub ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region mBedTLS
    /// <summary>
    /// Initialize the CryptoSym with mBedTLS as sub-component.
    /// </summary>
    public class mBedTLS : Generic
    {
        #region Constants
        /// <summary> Key size in DES algorithm for 56 bit key. </summary>
        public const byte DES_KEY_SIZE = 8;

        /// <summary> Block size in DES algorithm. </summary>
        public const byte DES_BLOCK_SIZE = 8;

        /// <summary> Block size in AES algorithm. </summary>
        public const byte AES_BLOCK_SIZE = 16;
#if DISABLE_AES
#if DISABLE_DES
#error PLEASE SPECIFY AT LEAST ONE CIPHER
#else
        /// <summary> Maximum Block Size of the currently supported ciphers </summary>
        public const byte MAX_BLOCK_SIZE = DES_BLOCK_SIZE;
#endif
#else
        /// <summary> Maximum Block Size of the currently supported ciphers </summary>
        public const byte MAX_BLOCK_SIZE = AES_BLOCK_SIZE;

        public const ushort MAX_ADD_DATA_SIZE = 65280;
#endif
        #endregion

        #region Variables
        private byte[] aAddData = null;
        private GCHandle m_pAddData;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for Symmetric Crypto mBedTLS layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to Key Store object - can be NULL. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to underlying Symmetric Crypto context for AES or DES operations. </summary>
            public IntPtr pCtx_Crypto;

            /// <summary> Internal key storage array. </summary>
            public fixed byte aKey[AES_BLOCK_SIZE * 2];

            /// <summary> Internal IV storage array. </summary>
            public fixed byte aIV[MAX_BLOCK_SIZE];

            /// <summary> Length of bytes available in \b aIV buffer. This is required for CCM or CCM* Cipher mode</summary>
            public byte bIV_Len;

            /// <summary>Internal Additional Data storage array. This is required for CCM or CCM* Cipher mode.</summary>
            public IntPtr pAddData;

            /// <summary>Maximum size allocated for \b pAddData buffer. This is required for CCM or CCM* Cipher mode.</summary>
            public ushort wAddData_Size;

            /// <summary>Length of bytes available in \b pAddData buffer. This is required for CCM or CCM* Cipher mode.</summary>
            public ushort wAddData_Len;

            /// <summary>
            /// To store Authentication Tag information that will be generated during Encryption or used while Decryption.
            /// This is required for CCM or CCM* Cipher mode.
            /// </summary>
            public fixed byte aTag[AES_BLOCK_SIZE];

            /// <summary>
            /// Authentication Tag. To be used when CCM or CCM* cipher modes are used.
            /// Supported values are,
            ///     - 4, 6, 8, 10, 12, 14 or 16 in case of CCM
            ///     - 0, 4, 6, 8, 10, 12, 14 or 16 in case of CCM*
            /// </summary>
            public byte bTagLen;

            /// <summary> Error code returned by mbedTLS layer. </summary>
            public int dwErrorCode;

            /// <summary> Specific Key Type. </summary>
            public ushort wKeyType;

            /// <summary> Internal Key Storage number. Will be utilized for Alternate implementation. </summary>
            public ushort wKeyNo;

            /// <summary> Indicates if the init vector of a previous crypto operation shall be used for the next operation. </summary>
            public ushort wKeepIV;

            /// <summary> Length of Key in terms of Bits. </summary>
            public ushort wKey_Bit;

            /// <summary> Additional information like diversified key length, etc. </summary>
            public ushort wAddInfo;

            /// <summary>
            /// Specify if the Key to be used is directly loaded or taken from KeyStore.
            ///     - <see cref="Value.OFF"/>: If the Key is loaded directly.
            ///     - <see cref="Value.ON"/>: If the Key is loaded from KeyStore.
            /// </summary>
            public byte bIsDirectKey;
        };
        #endregion

        #region Properties
        /// <summary>
        /// Override Key property from abstract base class
        /// </summary>
        public override byte[] pKey
        {
            get
            {
                byte[] pKey = new byte[MAX_BLOCK_SIZE * 2];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < pKey.Length; i++ )
                        {
                            pKey[i] = pDataParams->aKey[i];
                        }
                    }
                }
                return pKey;
            }
        }

        /// <summary>
        /// Override IV property from abstract base class
        /// </summary>
        public override byte[] pIV
        {
            get
            {
                byte[] pIV = new byte[MAX_BLOCK_SIZE];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < MAX_BLOCK_SIZE; i++ )
                        {
                            pIV[i] = pDataParams->aIV[i];
                        }
                    }
                }
                return pIV;
            }
        }

        /// <summary> Override CCM or CCM* Additional Data from abstract base class </summary>
        public override byte[] Additional_Data
        {
            get
            {
                byte[] aAddData = null;
                if ( ( m_DataParamsInt[0].pAddData != IntPtr.Zero ) && ( m_DataParamsInt[0].wAddData_Len != 0 ) )
                {
                    aAddData = new byte[m_DataParamsInt[0].wAddData_Len];
                    Marshal.Copy ( m_DataParamsInt[0].pAddData, aAddData, 0, aAddData.Length );
                }

                return aAddData;
            }
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_mBedTLS_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pKeyStoreDataParams,
            IntPtr pAddData_Buffer, ushort wAddData_Size );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_mBedTLS_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoSym with mBedTLS as sub-component.
        /// </summary>
        ///
        /// <param name="oKeyStore">Pointer to a KeyStore data parameter structure.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ( KeyStore.Generic oKeyStore )
        {
            return phCryptoSym_mBedTLS_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero, IntPtr.Zero, 0 );
        }

        /// <summary>
        /// Initialize the CryptoSym with mBedTLS as sub-component.
        /// </summary>
        ///
        /// <param name="oKeyStore">Pointer to a KeyStore data parameter structure.</param>
        /// <param name="wAddDataSize">Size to be allocated for Additional Data required for CCM or CCM* Cipher mode</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> oStatus for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ( KeyStore.Generic oKeyStore, ushort wAddDataSize )
        {
            if ( m_pAddData.IsAllocated ) { m_pAddData.Free (); aAddData = null; }

            aAddData = new byte[wAddDataSize];
            m_pAddData = GCHandle.Alloc ( aAddData, GCHandleType.Pinned );

            return phCryptoSym_mBedTLS_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero, aAddData.Length.Equals ( 0 ) ?
                IntPtr.Zero : m_pAddData.AddrOfPinnedObject (), wAddDataSize );
        }
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the CryptoRng with mBedTLS as sub-component.
        ///
        /// NOTE:
        ///     Its must to call this interface to de-initialize any used global context from other libraries.
        ///     If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DeInit ()
        {
            return phCryptoSym_mBedTLS_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public mBedTLS ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~mBedTLS ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion

    #region DUT
    /// <summary>
    /// DUT implementation of the Symmetric Crypto interface
    /// </summary>
    public class DUT : Generic
    {
        #region Private Variables
        /// <summary> Minimum size for allocating the Internal Processing buffer during initializing. </summary>
        public const int INTERNAL_BUFFER_SIZE = 256;

        private byte[] aTxBuffer;
        private byte[] aRxBuffer;

        private GCHandle stTxBuffer;
        private GCHandle stRxBuffer;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for Symmetric Crypto's DUT layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this component, NEVER MODIFY!.
            /// </summary>
            public ushort wId;

            /// <summary> Pointer to HAL parameter structure. </summary>
            public IntPtr pHalDataParams;

            /// <summary> Pointer to internal transmit buffer. This buffer can be used to overwrite actual HAL buffers. </summary>
            public IntPtr pTxBuffer;

            /// <summary> Length of bytes allocated for \b pTxBuffer buffer. </summary>
            public ushort wTxBuffSize;

            /// <summary> Pointer to internal receive buffer. This buffer can be used to overwrite actual HAL buffers. </summary>
            public IntPtr pRxBuffer;

            /// <summary> Length of bytes allocated for \b pRxBuffer buffer. </summary>
            public ushort wRxBuffSize;

            /// <summary> Internal Key storage array </summary>
            public fixed  byte aKey[32];

            /// <summary> Internal IV storage array </summary>
            public fixed  byte aIV[16];

            /// <summary> Internal Key storage array. </summary>
            public ushort wKeyType;

            /// <summary>
            /// Indicates if the init vector of a previous crypto operation shall be
            /// used for the next operation.
            /// </summary>
            public ushort wKeepIV;

            /// <summary>
            /// Specify if the Key to be used is directly loaded or taken from KeyStore.
            ///     <see cref="Value.ON"/> : If the Key is loaded directly.
            ///     <see cref="Value.OFF"/>: If the Key is loaded from KeyStore.
            /// </summary>
            public byte bIsDirectKey;
        };
        #endregion

        #region Properties
        /// <summary>
        /// Override Key property from abstract base class
        /// </summary>
        public override byte[] pKey
        {
            get { return null; }
        }

        /// <summary>
        /// Override IV property from abstract base class
        /// </summary>
        public override byte[] pIV
        {
            get { return null; }
        }

        /// <summary> Override CCM or CCM* Additional Data from abstract base class </summary>
        public override byte[] Additional_Data
        {
            get
            {
                throw new Exception ( "Not Implemented" );
            }
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_DUT_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams,
            IntPtr pTxBuffer, ushort wTxBuffSize, IntPtr pRxBuffer, ushort wRxBuffSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phCryptoSym_DUT_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize the CryptoSym with DUT as sub-component.
        /// </summary>
        ///
        /// <param name="oHal">Object of DUT HAL component.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successfull operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Init ( Hal.Generic oHal, ushort wTxBufferSize = INTERNAL_BUFFER_SIZE, ushort wRxBufferSize = INTERNAL_BUFFER_SIZE )
        {
            /* Free the buffer. */
            if ( stTxBuffer.IsAllocated ) stTxBuffer.Free ();
            if ( stRxBuffer.IsAllocated ) stRxBuffer.Free ();

            aTxBuffer = new byte[wTxBufferSize];
            stTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );

            aRxBuffer = new byte[wRxBufferSize];
            stRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );

            return phCryptoSym_DUT_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( ( oHal != null ) ? oHal.m_pDataParams : IntPtr.Zero ), stTxBuffer.AddrOfPinnedObject (), wTxBufferSize,
                stRxBuffer.AddrOfPinnedObject (), wRxBufferSize );
        }
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the CryptoSym with DUT as sub-component.
        ///
        /// NOTE:
        ///     Its must to call this interface to de-initialize any used global context from other libraries.
        ///     If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successfull operation.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DeInit ()
        {
            /* Free the buffer. */
            if ( stTxBuffer.IsAllocated ) stTxBuffer.Free ();
            if ( stRxBuffer.IsAllocated ) stRxBuffer.Free ();

            return phCryptoSym_DUT_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public DUT ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~DUT ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }

            /* Free the buffer. */
            if ( stTxBuffer.IsAllocated ) stTxBuffer.Free ();
            if ( stRxBuffer.IsAllocated ) stRxBuffer.Free ();
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
}
