/*
 * Copyright 2013 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;

namespace NxpRdLibNet
{
    public class Status_t
    {
        private const ushort COMP_MASK = 0xFF00;
        private const ushort ERR_MASK = 0x00FF;

        private Error_CompCode m_CompCode;
        private byte m_Error;

        private string sSamType = null;

        /// <summary>
        /// Create status
        /// </summary>
        ///
        /// <param name="CompCode"></param>
        ///
        /// <param name="Error"></param>
        public Status_t ( Error_CompCode CompCode, Enum Error )
        {
            m_CompCode = CompCode;

            m_Error = ( byte ) ( Error_Gen ) Error;
            sSamType = Error.GetType ().FullName;
            foreach ( string sSplit in sSamType.Split ( new char[] { '.', '-' },
                StringSplitOptions.RemoveEmptyEntries ) )
            {
                if ( sSplit == "Sam" )
                {
                    sSamType = "Sam";
                    break;
                }

                if ( sSplit == "SamAV3" )
                {
                    sSamType = "SamAV3";
                    break;
                }

                if ( sSplit == "SamAV2" )
                {
                    sSamType = "SamAV2";
                    break;
                }
            }
        }

        /// <summary>
        /// Create status
        /// </summary>
        /// <param name="status"></param>
        public Status_t ( int status )
        {
            m_CompCode = ( Error_CompCode ) ( ( status & COMP_MASK ) >> 8 );
            // Cast does not matter
            m_Error = ( byte ) ( status & 0xFF );
        }

        /// <summary>
        /// Create SUCCESS Status
        /// </summary>
        public Status_t ()
        {
            m_CompCode = Error_CompCode.GENERIC;
            m_Error = ( byte ) Error_Gen.SUCCESS;
        }

        public override bool Equals ( object status )
        {
            if ( status.GetType () != typeof ( Status_t ) )
            {
                return false;
            }
            else
            {
                return Equals ( ( Status_t ) status );
            }
        }

        public bool Equals ( Status_t status )
        {
            if ( ( m_CompCode == status.m_CompCode ) &&
            ( m_Error == status.m_Error ) )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public bool Equals ( Error_CompCode CompCode, Enum Error )
        {
            if ( ( m_CompCode == CompCode ) &&
            ( m_Error == ( byte ) ( Error_Gen ) Error ) )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public bool Equals ( Enum Error )
        {
            if ( m_Error == ( byte ) ( Error_Gen ) Error )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Convert raw status from C-Object to managed status object
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        public static implicit operator Status_t ( ushort status )
        {
            Error_CompCode CompCode = (Error_CompCode)((status & COMP_MASK) >> 8);
            byte bError = (byte)(status & 0x00FF);

            // Cast does not matter
            return new Status_t ( CompCode, ( Error_Gen ) bError );
        }

        /// <summary>
        /// Convert managed status object to raw status for C-Object (ushort)
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        public static implicit operator ushort ( Status_t status )
        {
            ushort statusOut;
            statusOut = ( ushort ) ( ( ( ushort ) status.m_CompCode ) << 8 );
            statusOut |= ( byte ) ( status.m_Error );

            return statusOut;
        }

        /// <summary>
        /// Convert managed status object to raw status for C-Object (int)
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        public static implicit operator int ( Status_t status )
        {
            return ( int ) ( ( ushort ) status );
        }

        /// <summary>
        /// Overridden: Convert managed status object to string
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        public override string ToString ()
        {
            string statusOut = "";
            if ( m_Error != ( byte ) Error_Gen.SUCCESS )
            {
                statusOut = CompCode.ToString ();
                statusOut += ",";
            }
            statusOut += Error.ToString ();

            return statusOut;
        }

        public static bool operator ==( Status_t a, Status_t b )
        {
            if ( a.Equals ( b ) )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public static bool operator !=( Status_t a, Status_t b )
        {
            return !( a == b );
        }

        public override int GetHashCode ()
        {
            return ( int ) this;
        }

        /// <summary>
        /// Status Component Code
        /// </summary>
        public Error_CompCode CompCode
        {
            get
            {
                return m_CompCode;
            }
        }

        /// <summary>
        /// Status Error
        /// </summary>
        public Enum Error
        {
            get
            {
                if ( m_Error >= ( byte ) CustomCodes.ERROR_BEGIN )
                {
                    switch ( m_CompCode )
                    {
                        case Error_CompCode.BAL:
                            return ( Bal.Error ) m_Error;

                        /*
                        * All the error code for Sam are with respect to Sam AV3. If in future if next version is available, change this
                        * to the future versions and align the future version error codes with respect to Sam AV2 or Sam AV3.
                        */
                        case Error_CompCode.HAL:
                            if ( Enum.IsDefined ( typeof ( Hal.Error ), m_Error ) )
                            {
                                return ( Hal.Error ) m_Error;
                            }
                            else
                            {
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
                                if ( sSamType == "SamAV3" )
                                    return ( Hal.SamAV3.Error ) m_Error;

                                else
                                    return ( Hal.SamAV2.Error ) m_Error;

#elif PACKAGE_INTERNAL
                                else
                                    return ( Hal.Sam.Error ) m_Error;
                                else
                                    return ( Hal.SamAV2.Error ) m_Error;
#else
                                return ( Hal.SamAV2.Error ) m_Error;
#endif
                            }

                        case Error_CompCode.PAL_ISO14443P4:
                            return ( palI14443p4.Error ) m_Error;
                        case Error_CompCode.PAL_MIFARE:
                            return ( palMifare.Error ) m_Error;
                        case Error_CompCode.PAL_SLI15693:
                            return ( palSli15693.Error ) m_Error;
                        case Error_CompCode.PAL_I18092MPI:
                            return ( palI18092mPI.Error ) m_Error;
                        case Error_CompCode.AL_FELICA:
                            return ( alFelica.Error ) m_Error;
                        case Error_CompCode.AL_MFP:
                            return ( alMfp.Error ) m_Error;
                        case Error_CompCode.AL_MFPEVX:
                            return ( alMfpEVx.Error ) m_Error;
                        case Error_CompCode.AL_MFDF:
                            return ( alMfdf.Error ) m_Error;
                        case Error_CompCode.AL_MFDFEVX:
                            return ( alMfdfEVx.Error ) m_Error;
                        case Error_CompCode.AL_MFDFLIGHT:
                            return ( alMfdfLight.Error ) m_Error;
                        case Error_CompCode.AL_MIFAREPRIMENTAG:
                            return ( alMifarePrimeNtag.Error ) m_Error;
                        case Error_CompCode.AL_MFDUOX:
                            return ( alMfDuoX.Error ) m_Error;
                        case Error_CompCode.AL_NTAGXDNA:
                            return ( alNtagXDna.Error ) m_Error;

#if PACKAGE_INTERNAL
                        case Error_CompCode.AL_MFIDENTITY:
                            return ( alMfIdentity.Error ) m_Error;
#endif

                        case Error_CompCode.AL_MFNTAG42XDNA:
                            return ( alMfNtag42XDna.Error ) m_Error;
                        case Error_CompCode.AL_VCA:
                            return ( alVca.Error ) m_Error;
                        case Error_CompCode.AL_I18000P3M3:
                            return ( alI18000p3m3.Error ) m_Error;
                        case Error_CompCode.AL_ICODE:
                            return ( alICode.Error ) m_Error;
#if PACKAGE_INTERNAL
                        case Error_CompCode.DL_THSTRM:
                            return ( dlThstrm.Error ) m_Error;
                        case Error_CompCode.DL_OSCI:
                            return ( dlOsci.Error ) m_Error;
                        case Error_CompCode.DL_MSTAMPOSC:
                            return ( dlMstAmpOsc.Error ) m_Error;
                        case Error_CompCode.DL_RDFPGA:
                            return ( dlRdFpga.Error ) m_Error;
                        case Error_CompCode.DL_STEPPER:
                            return ( dlStepper.Error ) m_Error;
#endif

                        case Error_CompCode.CRYPTOASYM:
                            return ( CryptoASym.Error ) m_Error;

                        default:
                            return ( Error_Gen ) m_Error;
                    }
                }
                else if ( Enum.IsDefined ( typeof ( Error_Comm ), m_Error ) )
                {
                    return ( Error_Comm ) m_Error;
                }
                else if ( Enum.IsDefined ( typeof ( Error_Param ), m_Error ) )
                {
                    return ( Error_Param ) m_Error;
                }
                else
                {
                    //if (Enum.IsDefined(typeof(Error_Gen), m_Error))
                    return ( Error_Gen ) m_Error;
                }
            }
        }
    }

    /// <summary>
    /// Enumeration for different variant of Success responses.
    /// </summary>
    public enum Error_Gen : byte
    {
        /// <summary> Returned in case of no error. </summary>
        SUCCESS = 0x00,

        /// <summary> Rx chaining is not complete, further action needed. </summary>
        SUCCESS_CHAINING = 0x71,

        /// <summary> An incomplete byte was received. </summary>
        SUCCESS_INCOMPLETE_BYTE = 0x72
    }

    /// <summary>
    /// Enumeration for communication errors.
    /// </summary>
    public enum Error_Comm : byte
    {
        /// <summary> No reply received, e.g. PICC removal. </summary>
        IO_TIMEOUT = 0x01,

        /// <summary> Wrong CRC or parity detected. </summary>
        INTEGRITY_ERROR,

        /// <summary> A collision occurred. </summary>
        COLLISION_ERROR,

        /// <summary> Attempt to write beyond buffer size. </summary>
        BUFFER_OVERFLOW,

        /// <summary> Invalid frame format. </summary>
        FRAMING_ERROR,

        /// <summary> Received response violates protocol. </summary>
        PROTOCOL_ERROR,

        /// <summary> MIFARE Classic authentication error. </summary>
        AUTH_ERROR,

        /// <summary> A Read or Write error occurred in RAM/ROM or Flash. </summary>
        READ_WRITE_ERROR,

        /// <summary> The RC sensors signal overheating. </summary>
        TEMPERATURE_ERROR,

        /// <summary> Error on RF-Interface. </summary>
        RF_ERROR,

        /// <summary> An error occurred in RC communication </summary>
        INTERFACE_ERROR,

        /// <summary> A length error occurred. </summary>
        LENGTH_ERROR,

        /// <summary> Error due to External RF. </summary>
        EXT_RF_ERROR = 0x10,

        /// <summary> EMVCo EMD Noise Error. </summary>
        NOISE_ERROR = 0x11,

        /// <summary> Used when HAL ShutDown is called. </summary>
        ABORTED = 0x12,

        /// <summary> Authentication method / type is not supported. </summary>
        AUTH_NOT_SUPPORTED = 0x14,

        /// <summary> Authentication Delay. </summary>
        AUTH_DELAY = 0xAD,

        /// <summary> An resource error. </summary>
        RESOURCE_ERROR = 0x0D,

        /// <summary> An internal error occurred </summary>
        INTERNAL_ERROR = 0x7F,

        /// <summary> Invalid parameter supplied. </summary>
        INVALID_PARAMETER = 0x21,

        /// <summary> Condition of use not satisfied. </summary>
        USE_CONDITION = 0x25,

        /// <summary> Authentication Delay </summary>
        AUTHENTICATION_DELAY = 0xAD,

        /// <summary> An incomplete byte was received </summary>
        INCOMPLETE_BYTE = 0x72,
    }

    /// <summary>
    /// Enumeration for parameter errors.
    /// </summary>
    public enum Error_Param : byte
    {
        /// <summary> Invalid data parameters supplied (layer id check failed) </summary>
        INVALID_DATA_PARAMS = 0x20,

        /// <summary> Invalid parameter supplied </summary>
        INVALID_PARAMETER,

        /// <summary> Reading/Writing a parameter would produce an overflow </summary>
        PARAMETER_OVERFLOW,

        /// <summary> Parameter not supported </summary>
        UNSUPPORTED_PARAMETER,

        /// <summary> Command not supported </summary>
        UNSUPPORTED_COMMAND,

        /// <summary> Condition of use not satisfied. </summary>
        USE_CONDITION,

        /// <summary> A key error occurred. </summary>
        KEY,

        /// <summary> An error occurred when loading the Licensee HAL. </summary>
        LICENSEE_HAL,

        /// <summary> An error occurred when a operation was canceled with a Cancellation Token. </summary>
        CANCELED,

        /// <summary> An error occurred because the size of the buffer is not met. </summary>
        PARAMETER_SIZE = 0x31,
    }

    /// <summary>
    /// Enumeration for Component codes.
    /// </summary>
    public enum Error_CompCode : byte
    {
        /// <summary> Generic Component Code </summary>
        GENERIC = 0x00,

        /// <summary> BAL Component Code. </summary>
        BAL = 0x01,

        /// <summary> HAL Component Code. </summary>
        HAL = 0x02,

        /// <summary> ISO14443-3A PAL-Component Code. </summary>
        PAL_ISO14443P3A = 0x03,

        /// <summary> ISO14443-3B PAL-Component Code. </summary>
        PAL_ISO14443P3B = 0x04,

        /// <summary> ISO14443-4A PAL-Component Code. </summary>
        PAL_ISO14443P4A = 0x05,

        /// <summary> ISO14443-4 PAL-Component Code. </summary>
        PAL_ISO14443P4 = 0x06,

        /// <summary> MIFARE(R) PAL-Component Code. </summary>
        PAL_MIFARE = 0x07,

        /// <summary> Open FeliCa PAL-Component Code. </summary>
        PAL_FELICA = 0x08,

        /// <summary> ICode EPC/UID PAL-Component Code. </summary>
        PAL_EPCUID = 0x09,

        /// <summary> ICODE SLI/ISO15693 PAL-Component Code. </summary>
        PAL_SLI15693 = 0x0A,

        /// <summary> ISO18000-3 Mode3 PAL-Component Code. </summary>
        PAL_I18000P3M3 = 0x0B,

        /// <summary> ISO18000-3 Mode3 PAL-Component Code. </summary>
        PAL_I18092MPI = 0x0C,

        /// <summary> MIFARE(R) Classic AL-Component Code. </summary>
        AL_MFC = 0x10,

        /// <summary> MIFARE(R) Ultralight AL-Component Code. </summary>
        AL_MFUL = 0x11,

        /// <summary> MIFARE(R) Plus AL-Component Code. </summary>
        AL_MFP = 0x12,

        /// <summary> Virtual Card Architecture AL-Component Code. </summary>
        AL_VCA = 0x13,

        /// <summary> Open FeliCa AL-Component Code. </summary>
        AL_FELICA = 0x14,

        /// <summary> ICODE AL-Component Code. </summary>
        AL_ICODE = 0x15,

        /// <summary> ISO18000-3 Mode3 AL-Component Code. </summary>
        AL_I18000P3M3 = 0x18,

        /// <summary> MIFARE DESFIRE EV1 AL Component Code. </summary>
        AL_MFDF = 0x19,

        /// <summary> P40CmdPriv AL Component Code. </summary>
        AL_P40CMDPRIV = 0x1A,

        /// <summary> MIFARE DESFIRE EVx AL Component Code. </summary>
        AL_MFDFEVX = 0x20,

        /// <summary> MIFARE Plus EVx AL-Component Code. </summary>
        AL_MFPEVX = 0x21,

        /// <summary> MIFARE MFDFLIGHT AL-Component Code. </summary>
        AL_MFDFLIGHT = 0x23,

        /// <summary> MIFARE PRIME NTAG 413 AL-Component Code. </summary>
        AL_MIFAREPRIMENTAG = 0x24,

        /// <summary> MIFARE PRIME IDENTITY AL-Component Code. </summary>
        AL_MFIDENTITY = 0x25,

        /// <summary> MFNTAG42XDNA AL-Component Code. </summary>
        AL_MFNTAG42XDNA = 0x26,

        /// <summary> MIFARE DUOX AL Component Code. </summary>
        AL_MFDUOX = 0x27,

        /// <summary> NTAG X DNA AL Component Code. </summary>
        AL_NTAGXDNA = 0x28,


#if PACKAGE_INTERNAL
        /// <summary> Amplifier DL-Component Code. </summary>,
        DL_AMP = 0x30,

        /// <summary> ThermoStream DL-Component Code. </summary>,
        DL_THSTRM = 0x31,

        /// <summary> Osci DL-Component Code. </summary>,
        DL_OSCI = 0x32,

        /// <summary> Reader FPGA Box DL-Component Code. </summary>,
        DL_RDFPGA = 0x33,

        /// <summary> Master Amplifier Osci DL-Component Code. </summary>,
        DL_MSTAMPOSC = 0x34,

        /// <summary> Stepper DL-Component Code. </summary>,
        DL_STEPPER = 0x35,

        /// <summary> Power Supply DL-Component Code. </summary>,
        DL_POWERSUPPLY = 0x36,

        /// <summary> Spa DL-Component Code. </summary>,
        DL_SPA = 0x37,

        /// <summary> ISO3 DL-Component Code. </summary>,
        DL_ISO3 = 0x38,
#endif

        /// <summary> Cid Manager Component Code. </summary>
        CIDMANAGER = 0xE0,

        /// <summary> Crypto Component Code. </summary>
        CRYPTOSYM = 0xE1,

        /// <summary> KeyStore Component Code. </summary>
        KEYSTORE = 0xE2,

        /// <summary> Tools Component Code. </summary>
        TOOLS = 0xE3,

        /// <summary> CryptoRng Component Code. </summary>
        CRYPTORNG = 0xE4,

        /// <summary> TMIUtils Component Code. </summary>
        TMIUTILS = 0xE5,

        /// <summary> CryptoASym Component Code. </summary>
        CRYPTOASYM = 0xE6,

        /// <summary> Log Component Code. </summary>
        LOG = 0xEF,
    }

    /// <summary>
    /// Enumeration for Generic usage.
    /// </summary>
    public enum Value : int
    {
        /// <summary> Enabled </summary>
        ON = 0x0001,

        /// <summary> Disabled </summary>
        OFF = 0x0000,

        /// <summary> Supported </summary>
        SUPPORTED = 0x0001,

        /// <summary> Not Supported </summary>
        UNSUPPORTED = 0x0000
    }

    /// <summary>
    /// Enumeration for Error and Configuration start values.
    /// </summary>
    public enum CustomCodes : int
    {
        /// <summary> Enumeration field for start of Error code. </summary>
        ERROR_BEGIN = 0x0080,

        /// <summary> Enumeration field for start of configuration code. </summary>
        CONFIG_BEGIN = 0xE000
    }

    /// <summary>
    /// Class for all variants of buffering flags.
    /// </summary>
    public static class ExchangeOptions
    {
        /// <summary>
        /// Default exchange mode. Data exchange between PICC and reader.
        ///
        /// ISO14443-4 specific:
        /// Performs TX/RX chaining with the card.
        /// Returns #PH_ERR_SUCCESS_CHAINING when RxBuffer is full and does not ACK the last received block.
        /// </summary>
        public const int DEFAULT = 0x0000;

        /// <summary>
        /// Chains given data to card. Does not receive any data.
        /// </summary>
        public const int TXCHAINING = 0x0001;

        /// <summary>
        /// Returns <seealso cref="Error_Gen.SUCCESS_CHAINING"/> when RxBuffer is full and does not ACK the last received block.
        ///
        /// ISO14443-4 specific:
        /// Starts transmission with and R(ACK) block and performs Rx chaining with the card.
        /// </summary>
        public const int RXCHAINING = 0x0002;

        /// <summary>
        /// ISO14443-4 specific:
        /// Starts transmission with and R(ACK) block and performs Rx chaining with the card.
        /// Completes Rx chaining with the card if RxBuffer is full.
        /// </summary>
        public const int RXCHAINING_BUFSIZE = 0x0003;

        /// <summary>
        /// Number of Bytes to Read information is available while performing exchanges via peripheral(s).
        /// </summary>
        public const int BYTES_TO_READ = 0x0004;

        /// <summary>
        /// Mask for the above modes. All other bits are reserved.
        /// </summary>
        public const int MODE_MASK = 0x000F;

        /// <summary>
        /// Does not clear the internal buffer before operation. If this bit is set and data is transmitted,
        /// the contents of the internal buffer are sent first.
        /// </summary>
        public const int LEAVE_BUFFER_BIT = 0x4000;

        /// <summary>
        /// Buffers TX-Data into internal buffer instead of transmitting it.
        /// </summary>
        public const int BUFFERED_BIT = 0x8000;

        /// <summary>
        /// The given data is the first part of a stream. (Helper definition for buffering).
        /// </summary>
        public const int BUFFER_FIRST = (DEFAULT | BUFFERED_BIT);

        /// <summary>
        /// The given data is a subsequent part of a stream. (Helper definition for buffering).
        /// </summary>
        public const int BUFFER_CONT = (DEFAULT | BUFFERED_BIT | LEAVE_BUFFER_BIT);

        /// <summary>
        /// The given data is the last part of a stream. (Helper definition for buffering).
        /// When this flag is used the data is exchanged between PICC and reader.
        /// </summary>
        public const int BUFFER_LAST = (DEFAULT | LEAVE_BUFFER_BIT);
    }
}
