﻿/*
 * Copyright 2013 - 2015, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet
{
    /// <summary>
    /// Class foir TMI Colloction.
    /// </summary>
    public class TMIUtils
    {
        #region Constants
        /// <summary>
        /// Configuration field for Deactivating the TmiCollection.
        /// This flag will update TMIStatus to OFF, set TMIBufLen and OffsetInTMI to zero.
        /// After this the object that is used for TMICollection should be reinitialized.
        /// </summary>
        public const byte PH_TMIUTILS_DEACTIVATE_TMI = 0x00;

        /// <summary>
        /// Configuration field for Activating the TmiCollection.
        /// This flag will update TMIStatus to ON.
        /// </summary>
        public const byte PH_TMIUTILS_ACTIVATE_TMI = 0x01;

        /// <summary>
        /// Configuration field for pausing the TmiCollection.
        /// This flag will update TMIStatus to OFF.
        /// </summary>
        public const byte PH_TMIUTILS_PAUSE_TMI = 0x02;

        /// <summary>
        /// Configuration field for resuming the TmiCollection.
        /// This flag will update TMIStatus to ON.
        /// </summary>
        public const byte PH_TMIUTILS_RESUME_TMI = 0x03;

        /// <summary>
        /// Configuration field for resetting the TmiCollection.
        /// This flag will set TMIBufLen and OffsetInTMI to zero.
        /// </summary>
        public const byte PH_TMIUTILS_RESET_TMI = 0x04;

        /// <summary>
        /// Configuration field for reading the status of TMICollection.
        /// This flag will return PH_ON or PH_OFF.
        /// </summary>
        public const byte PH_TMIUTILS_TMI_STATUS = 0x01;

        /// <summary>
        /// Configuration field for updating / reading the offest length of TMICollection buffer.
        /// </summary>
        public const byte PH_TMIUTILS_TMI_OFFSET_LENGTH = 0x02;

        /// <summary>
        /// Configuration field for reading the current index of TMICollection buffer.
        /// </summary>
        public const byte PH_TMIUTILS_TMI_BUFFER_INDEX    =0x04;

        /// <summary>
        /// Configuration field for reading the current offset of TMICollection buffer.
        /// </summary>
        public const byte PH_TMIUTILS_TMI_OFFSET_VALUE    =0x08;

        /// <summary>
        /// Flag indicating that padding is not required.
        /// </summary>
        public const byte PH_TMIUTILS_NO_PADDING          =0x00;

        /// <summary>
        /// Flag indicating that instruction is Read Operation.
        /// </summary>
        public const byte PH_TMIUTILS_READ_INS           = 0x01;

        /// <summary>
        /// Flag to indicate zero padding after Cmd buffer
        /// </summary>
        public const byte PH_TMIUTILS_ZEROPAD_CMDBUFF    = 0x02;

        /// <summary>
        /// Flag to indicate zero padding after Data buffer.
        /// </summary>
        public const byte PH_TMIUTILS_ZEROPAD_DATABUFF   = 0x04;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for TMIUtils layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Pointer to buffer for collecting the TMI. </summary>
            public unsafe byte* pTMIBuffer;

            /// <summary> Length of bytes available in pTMIBuffer. </summary>
            public uint dwTMIBufLen;

            /// <summary> Indicates the TMIBuffer fill index. </summary>
            public uint dwTMIbufIndex;

            /// <summary> Indicates whether TMI collection is PH_ON or PH_OFF. </summary>
            public byte bTMIStatus;

            /// <summary> Indicates the offset in TMI buffer where the Length field is stored in case of unspecified length read. </summary>
            public uint dwOffsetInTMI;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_Init ( IntPtr pDataParams, byte[] pTMIBuffer, uint dwBufLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_ActivateTMICollection ( IntPtr pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_GetTMI ( IntPtr pDataParams, ref IntPtr ppTMIBuffer, ref uint dwTMILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_CollectTMI ( IntPtr pDataParams, byte bOption, byte[] pCmdBuffer, ushort wCmdBufLen,
            byte[] pData, uint dwDataLen, ushort wBlockSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_SetConfig ( IntPtr pDataParams, ushort wConfig, uint dwValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phTMIUtils_GetConfig ( IntPtr pDataParams, ushort wConfig, ref uint pValue );
        #endregion

        #region Wrapper Functions
        /// <summary>
        /// Initialize TMI Utils component.
        /// </summary>
        ///
        /// <param name="pTMIBuffer">Input buffer for collecting the command / data information.</param>
        /// <param name="dwBufLen">Size of input buffer. </param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( byte[] pTMIBuffer, uint dwBufLen )
        {
            return phTMIUtils_Init ( m_pDataParams, pTMIBuffer, dwBufLen );
        }

        /// <summary>
        /// Perfomrs Activation, Deactivation, Resume, Reset and Pause of TmiCollection.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options should be used.
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_DEACTIVATE_TMI
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_ACTIVATE_TMI
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_PAUSE_TMI
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_RESUME_TMI
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_RESET_TMI
        ///							</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ActivateTMICollection ( byte bOption )
        {
            return phTMIUtils_ActivateTMICollection ( m_pDataParams, bOption );
        }

        /// <summary>
        /// Get TMI buffer
        /// </summary>
        ///
        /// <param name="pTMI">Get the collected command / data information. </param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetTMI ( out byte[] pTMI )
        {
            Status_t status;
            IntPtr ppRxBuffer = IntPtr.Zero;
            uint dwTMIBufLen = 0;

            pTMI = null;
            status = phTMIUtils_GetTMI ( m_pDataParams, ref ppRxBuffer, ref dwTMIBufLen );
            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwTMIBufLen != 0 ) )
            {
                pTMI = new byte[dwTMIBufLen];
                Marshal.Copy ( ppRxBuffer, pTMI, 0, ( int ) dwTMIBufLen );
            }

            return status;
        }

        /// <summary>
        /// Performs collection of command / data information.
        /// </summary>
        ///
        /// <param name="bOption">One of the below values.
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_NO_PADDING
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_READ_INS
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_ZEROPAD_CMDBUFF
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_ZEROPAD_DATABUFF</param>
        /// <param name="pCmdBuffer">Buffer containing the command information. </param>
        /// <param name="pData">Buffer containing the data information.</param>
        /// <param name="wBlockSize">Block size for appending padding bytes if PH_TMIUTILS_NO_PADDING is not used. </param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CollectTMI ( byte bOption, byte[] pCmdBuffer, byte[] pData, ushort wBlockSize )
        {
            return phTMIUtils_CollectTMI (
                m_pDataParams,
                bOption,
                pCmdBuffer,
                ( ushort ) ( ( pCmdBuffer == null ) ? 0 : pCmdBuffer.Length ),
                pData,
                ( uint ) ( ( pData == null ) ? 0 : pData.Length ),
                wBlockSize
                );
        }

        /// <summary>
        /// Performs SetConfig command.
        /// </summary>
        /// <param name="wConfig">One of the Below Configurations.
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_TMI_OFFSET_LENGTH
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_TMI_OFFSET_VALUE</param></param>
        /// <param name="dwValue">The value for the given configuration.</param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( ushort wConfig, uint dwValue )
        {
            return phTMIUtils_SetConfig ( m_pDataParams, wConfig, dwValue );
        }

        /// <summary>
        /// Performs GetConfig command.
        /// </summary>
        /// <param name="wConfig">One of the Below Configurations.
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_TMI_STATUS
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_TMI_OFFSET_LENGTH
        ///							NxpRdLibNet.TMIUtils.PH_TMIUTILS_TMI_BUFFER_INDEX</param>
        /// <param name="pValue">The value for the given configuration. </param>
        ///
        /// <returns>Returns Success oStatus for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( ushort wConfig, ref uint pValue )
        {
            return phTMIUtils_GetConfig ( m_pDataParams, wConfig, ref pValue );
        }
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        /// <summary>
        /// Create the class instance.
        /// </summary>
        public TMIUtils ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~TMIUtils ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
}
