/*
 * Copyright 2013, 2015, 2017, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System.Runtime.InteropServices;

namespace NxpRdLibNet
{
    public class Tools
    {
        #region DEFINES

        public enum ParityOption : byte
        {
            EVEN = 0x00,
            ODD = 0x01
        }

        public static class CrcPreset
        {
            public const ushort CRC16_PRESET_ISO14443A = 0x6363;
            public const ushort CRC16_PRESET_ISO14443B = 0xFFFF;
            public const uint CRC32_PRESET_DF8 = 0xFFFFFFFF;
            public const byte CRC8_PRESET_EPC = 0xFF;
            public const byte CRC8_PRESET_UID = 0xFD;
            public const ushort CRC16_PRESET_EPCUID = 0xFFFF;
            public const byte CRC5_PRESET_I18000P3 = 0x09;
            public const uint CRC16_PRESET_I18000P3 = 0xFFFF;
        }

        public static class CrcPolynom
        {
            public const ushort CRC16_POLY_ISO14443 = 0x8408;
            public const uint CRC32_POLY_DF8 = 0xEDB88320;
            public const byte CRC8_POLY_EPCUID = 0x1D;
            public const ushort CRC16_POLY_EPCUID = 0x1021;
            public const byte CRC5_POLY_I18000P3 = 0x09;
            public const uint CRC16_POLY_I18000P3 = 0x8408;
        }

        public static class CrcOption
        {
            public const byte DEFAULT = 0x00;
            public const byte OUPUT_INVERTED = 0x01;
            public const byte MSB_FIRST = 0x02;
            public const byte BITWISE = 0x04;
        }

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_EncodeParity(
            byte bOption,                   /**< [In] Parity option; e.g. #PH_TOOLS_PARITY_OPTION_EVEN. */
            byte[] pInBuffer,               /**< [In] Array to input data.*/
            ushort wInBufferLength,         /**< [In] Length of input data in bytes.*/
            byte bInBufferBits,             /**< [In] Number of valid bits in last byte of pInBuffer. */
            ushort wOutBufferSize,          /**< [In] Size of the output buffer. */
            byte[] pOutBuffer,              /**< [Out] Output buffer. */
            ref ushort pOutBufferLength,    /**< [Out] Number of valid bytes in pOutBuffer. */
            ref byte pOutBufferBits         /**< [Out] Number of valid bits in last byte of pOutBuffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_DecodeParity(
            byte bOption,                   /**< [In] Parity option; e.g. #PH_TOOLS_PARITY_OPTION_EVEN. */
            byte[] pInBuffer,               /**< [In] Array to input data.*/
            ushort wInBufferLength,         /**< [In] Length of input data in bytes.*/
            byte bInBufferBits,             /**< [In] Number of valid bits in last byte of pInBuffer. */
            ushort wOutBufferSize,          /**< [In] Size of the output buffer. */
            byte[] pOutBuffer,              /**< [Out] Output buffer. */
            ref ushort pOutBufferLength,    /**< [Out] Number of valid bytes in pOutBuffer. */
            ref byte pOutBufferBits         /**< [Out] Number of valid bits in last byte of pOutBuffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_CalculateCrc5(
            byte bOption,       /**< [In] Specifies whether the output should be inverted or not */
            byte bPreset,       /**< [In] Preset used for CRC calculation, e.g. #PH_TOOLS_CRC5_PRESET_I18000P3. */
            byte bPolynom,      /**< [In] Polynom used for reduction, e.g. #PH_TOOLS_CRC5_POLY_I18000P3. */
            byte[] pData,       /**< [In] Array to input data.*/
            ushort wDataLength, /**< [In] Length of input data.*/
            ref byte pCrc       /**< [Out] Resulting CRC.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_CalculateCrc8(
            byte bOption,       /**< [In] Specifies whether the output should be inverted or not */
            byte bPreset,       /**< [In] Preset used for CRC calculation, e.g. PH_TOOLS_CRC8_INIT_ISO14443.*/
            byte bPolynom,      /**< [In] Polynom used for reduction, e.g. PH_TOOLS_CRC8_POLY_ISO14443.*/
            byte[] pData,       /**< [In] Array to input data.*/
            ushort wDataLength, /**< [In] Length of input data.*/
            ref byte pCrc       /**< [Out] Resulting CRC.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_CalculateCrc16(
            byte bOption,       /**< [In] Specifies whether the output should be inverted or not. */
            ushort wPreset,     /**< [In] Preset used for CRC calculation, e.g. PH_TOOLS_CRC8_INIT_ISO14443. */
            ushort wPolynom,    /**< [In] Polynom used for reduction, e.g. PH_TOOLS_CRC8_POLY_ISO14443. */
            byte[] pData,       /**< [In] Array to input data. */
            ushort wDataLength, /**< [In] Length of input data. */
            ref ushort pCrc     /**< [Out] Resulting CRC. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_CalculateCrc32(
            byte bOption,       /**< [In] Specifies whether the output should be inverted or not */
            uint dwPreset,      /**< [In] Preset used for CRC calculation, e.g. PH_TOOLS_CRC8_INIT_ISO14443.*/
            uint dwPolynom,     /**< [In] Polynom used for reduction, e.g. PH_TOOLS_CRC8_POLY_ISO14443.*/
            byte[] pData,       /**< [In] Array to input data.*/
            uint dwDataLength, /**< [In] Length of input data.*/
            ref uint pCrc       /**< [Out] Resulting CRC.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phTools_CalculateLrc(
            byte[] pData,       /**< [In] Array to input data.*/
            ushort wDataLength, /**< [In] Length of input data.*/
            ref byte pLrc       /**< [Out] Resulting LRC.*/
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t EncodeParity(
            ParityOption bMode,
            byte[] pInBuffer,
            byte bInBufferBits,
            ushort wOutBufferSize,
            out byte[] pOutBuffer,
            out byte pOutBufferBits
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            pOutBufferBits = 0;
            pOutBuffer = new byte[wOutBufferSize];

            status = phTools_EncodeParity(
                (byte)bMode,
                pInBuffer,
                (ushort)pInBuffer.Length,
                bInBufferBits,
                wOutBufferSize,
                pOutBuffer,
                ref wRxLength,
                ref pOutBufferBits);

            System.Array.Resize<byte>(ref pOutBuffer, wRxLength);
            return status;
        }

        public Status_t DecodeParity(
            ParityOption bMode,
            byte[] pInBuffer,
            byte bInBufferBits,
            ushort wOutBufferSize,
            out byte[] pOutBuffer,
            out byte pOutBufferBits
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            pOutBufferBits = 0;
            pOutBuffer = new byte[wOutBufferSize];

            status = phTools_DecodeParity(
                (byte)bMode,
                pInBuffer,
                (ushort)pInBuffer.Length,
                bInBufferBits,
                wOutBufferSize,
                pOutBuffer,
                ref wRxLength,
                ref pOutBufferBits);

            System.Array.Resize<byte>(ref pOutBuffer, wRxLength);
            return status;
        }

        public Status_t CalculateCrc5(
                byte bOption,
                byte bPreset,
                byte bPolynom,
                byte[] pData,
                ushort wDataLength,
                out byte pCrc
                )
        {
            pCrc = 0;
            return phTools_CalculateCrc5(bOption, bPreset, bPolynom, pData, wDataLength, ref pCrc);
        }

        public Status_t CalculateCrc8(
                byte bOption,
                byte bPreset,
                byte bPolynom,
                byte[] pData,
                ushort wDataLength,
                out byte pCrc
                )
        {
            pCrc = 0;
            return phTools_CalculateCrc8(bOption, bPreset, bPolynom, pData, wDataLength, ref pCrc);
        }

        public Status_t CalculateCrc16(
                byte bOption,
                ushort wPreset,
                ushort wPolynom,
                byte[] pData,
                ushort wDataLength,
                out ushort pCrc
                )
        {
            pCrc = 0;
            return phTools_CalculateCrc16(bOption, wPreset, wPolynom, pData, wDataLength, ref pCrc);
        }

        public Status_t CalculateCrc32(
                byte bOption,
                uint dwPreset,
                uint dwPolynom,
                byte[] pData,
                uint dwDataLength,
                out uint pCrc
                )
        {

            pCrc = 0;
            return phTools_CalculateCrc32(bOption, dwPreset, dwPolynom, pData, dwDataLength, ref pCrc);
        }

        public Status_t CalculateLrc(
                byte[] pData,
                ushort wDataLength,
                out byte pLrc
                )
        {

            pLrc = 0;
            return phTools_CalculateLrc(pData, wDataLength, ref pLrc);
        }

        #endregion

        #region MEMORY_MAPPING

        /// <summary>
        /// Create the class instance.
        /// </summary>
        public Tools()
        {

        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Tools()
        {

        }

        #endregion
    }
}
