/*
 * Copyright 2013, 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlAmp
{
    #region BASE

    #region DEFINES
    /* Osci Config */
    public enum Config : ushort
    {
        WAIT = 0,   /**<Defines if the function waits until all changes are applied in the amplififer*/
        MIN_GAIN,   /**<Defines the minimum gain setting of the amplifier (read only) */
        MAX_GAIN    /**<Defines the maximum gain setting of the amplifier (read only) */
    };
    #endregion

    public abstract class Generic
    {

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_SetPower(
            IntPtr pDataParams,          /**< [In] Pointer to this layers parameter structure. */
            byte bPowerOn
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_SetGain(
            IntPtr pDataParams,          /**< [In] Pointer to this layers parameter structure. */
            ushort wGainValue		    /**< [In] Specifies the gain value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_SetConfig(
                             IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                             ushort wIdentifier,		/**< [In] Specifies parameter identifier. */
                             ushort wValue				/**< [In] Specifies parameter value. */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_GetConfig(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wIdentifier,		/**< [In] Specifies parameter identifier. */
                                     out ushort pwValue			/**< [Out] Return parameter value. */
                                     );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS


        public Status_t SetPower(byte bPowerOn
        )
        {
            return phdlAmp_SetPower(m_pDataParams, bPowerOn);
        }

        public Status_t SetGain(ushort wGainValue
        )
        {
            return phdlAmp_SetGain(m_pDataParams, wGainValue);
        }

        public Status_t SetConfig(
                             ushort wIdentifier,
                             ushort wValue
                             )
        {
            return phdlAmp_SetConfig(m_pDataParams, wIdentifier, wValue);
        }

        public Status_t GetConfig(
                                     ushort wIdentifier,
                                     out ushort pwValue
                                     )
        {
            return phdlAmp_GetConfig(m_pDataParams, wIdentifier, out pwValue);
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region AR75A250

    public class AR75A250 : dlAmp.Generic
    {
        #region DEFINES

        public enum Power : byte
        {
            OFF = 0,
            ON = 1
        };

        public enum Gain : ushort
        {
            MIN = 0,
            MAX = 4095
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            ushort wCurrentGain;            /**< Current gain value. */
            byte bWait;                     /**< Defines whether the function only returns after the transients of setting the gain are finished. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_AR75A250_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlAmp_AR75A250_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }


        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public AR75A250()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~AR75A250()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region BBA100

    public class BBA100 : dlAmp.Generic
    {
        #region DEFINES

        public enum Power : byte
        {
            OFF = 0,
            ON = 1
        };

        public enum Gain : ushort
        {
            MIN = 0,
            MAX = 42
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            ushort wCurrentGain;            /**< Current gain value. */
            byte bWait;                     /**< Defines whether the function only returns after the transients of setting the gain are finished. */
            byte bSelectedPath;             /**< Output path that is used with this config */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_BBA100_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlAmp_BBA100_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }


        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public BBA100()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~BBA100()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region BBA150

    public class BBA150 : dlAmp.Generic
    {
        #region DEFINES

        public enum Power : byte
        {
            OFF = 0,
            ON = 1
        };

        public enum Gain : ushort
        {
            MIN = 0,
            MAX = 4000
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            ushort wCurrentGain;            /**< Current gain value. */
            byte bWait;                     /**< Defines whether the function only returns after the transients of setting the gain are finished. */
            byte bSelectedPath;             /**< Output path that is used with this config */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_BBA150_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlAmp_BBA150_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }


        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public BBA150()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~BBA150()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region MPRFA

    public class MPRFA : dlAmp.Generic
    {
        #region DEFINES

        public enum Power : byte
        {
            OFF = 0,
            ON = 1
        };

        private const int VERSION_SIZE = 20;
        private const int SERIAL_SIZE  = 13;

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            public ushort wCurrentGain;            /**< Current gain value. */
            public ushort wMinGain;                /**< The minumum gain that should be used for the amplfier */
            public byte bWait;                     /**< Defines whether the function only returns after the transients of setting the gain are finished. */
            public byte bOn;                       /**< us set to PH_ON if the amplifier is activated */
            public fixed byte bVersionString[VERSION_SIZE];  /**< This value stores the version string of the amplifier */
            public fixed byte bSerialString[SERIAL_SIZE];    /**< This value stores the serial string of the amplifier */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlAmp_MPRFA_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlAmp_MPRFA_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }


        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public MPRFA()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~MPRFA()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        public string Version
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bVersionString, VERSION_SIZE);
                    }
                }
            }
        }

        public string Serial
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bSerialString, SERIAL_SIZE);
                    }
                }
            }
        }
        #endregion

        unsafe private string ByteArrayToString(byte* arr, int maxLength)
        {
            char[] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }
    }

    #endregion
}
#endif