/*
 * Copyright 2013, 2017, 2019, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlOsci
{
    #region BASE

    #region DEFINES
    public enum Error : byte
    {
        ERR_COULD_NOT_CONVERT_DIVISOR_INTO_CMD = (CustomCodes.ERROR_BEGIN),
        ERR_CORR_RANGE_FAIL,
        ERR_AMPLITUDE_VAL,

        /* Command Errors*/
        ERR_CMD_UNKNOWN_COMMAND = (CustomCodes.ERROR_BEGIN + 3),
        ERR_CMD_ILLEGAL_HEADER_PATH,
        ERR_CMD_ILLEGAL_NUMBER,
        ERR_CMD_ILLEGAL_NUMBER_SUFFIX,
        ERR_CMD_UNKNOWN_KEYWORD,
        ERR_CMD_STRING_ERROR,
        ERR_CMD_GET_EMBEDDED_IN_ANOTHER_MESSAGE,
        ERR_CMD_ARBITRARY_DATA_BLOCK_EXPECTED,
        ERR_CMD_NONDIGIT_CHAR_IN_BYTE_COUNT_FIELD,
        ERR_CMD_EOI_DETECTED,
        ERR_CMD_EXTRA_BYTES_DETECTED,

        /*Execution Errors*/
        ERR_EXE_PERMISSION_ERROR = (CustomCodes.ERROR_BEGIN + 14),
        ERR_EXE_ENVIRONMENT_ERROR,
        ERR_EXE_OPTION_ERROR,
        ERR_EXE_UNRESOLVED_PARSING,
        ERR_EXE_PARAMETER_ERROR,
        ERR_EXE_CMD_NOT_IMPLEMENTED,
        ERR_EXE_PARAM_MISSING,
        ERR_EXE_HEX_DATA_ERROR,
        ERR_EXE_WAVEFORM_ERROR,
        ERR_EXE_WAVEFORM_DESCRIPTION_ERROR,
        ERR_EXE_WAVEFORM_TEXT_ERROR,
        ERR_EXE_WAVEFORM_TIME_ERROR,
        ERR_EXE_WAVEFORM_DATA_ERROR,
        ERR_EXE_PANEL_SETUP_ERROR,
        ERR_EXE_NO_MASS_STORAGE_PRESENT,
        ERR_EXE_MASS_STORAGE_NOT_FORMATTED,
        ERR_EXE_MASS_STORAGE_WAS_WRITE_PROTECTED,
        ERR_EXE_BAD_MASS_STORAGE_DETECTED,
        ERR_EXE_MASS_STOR_ROOT_DIR_FULL,
        ERR_EXE_MASS_STOR_FULL,
        ERR_EXE_MASS_STOR_FILE_SEQ_NUM_EXHAUSTED,
        ERR_EXE_MASS_STOR_FILE_NOT_FOUND,
        ERR_EXE_REQ_DIR_NOT_FOUND,
        ERR_EXE_FILENAME_ILLEGAL,
        ERR_EXE_MASS_STORAGE_FILENAME_ALREADY_EXISTS,

        /*other errors*/
        ERR_READ_WAVEFORM_BUFFER_TOO_SMALL = (CustomCodes.ERROR_BEGIN + 39),
        ERR_READ_WAVEFORM_HEADER_PARSE_INVALID,         /**< Could NOT parse header of waveform data correctly. */
        ERR_PICOSCOPE_DLL,                              /**< PicoScope DLL driver threw an non standard error read wAddionalInfo to get error code */
        ERR_TRIGGER_NOT_FIRED,                          /**< PicoScope Data Capturing not successfull */
        ERR_COULD_NOT_DETERMINE_MAX_POINTS              /**< Number of Maximum Points coult not been evaluated */
    }

    /* Osci Config */
    public enum Config : ushort
    {
        TRIGGER = 0,                    /**< Select the trigger mode */
        MODE,                           /**< Select the configuration mode, eg. FDT_106 */
        FIELD_MULTIPLIER,               /**< Select the Field Multiplier value, eg. RMS = FieldStrength * 0.318(318/1000) of small antennas 0.118 (118/1000) */
        AVERAGE_FACT,                   /**< Select the averaging factor for the measurements */
        RANGE_MIN,                      /**< Set the minimum range for SetCorrRange [Div/10] */
        RANGE_MAX,                      /**< Set the minimum range for SetCorrRange [Div/10] */
        //TRIGGER_DELAYUS,                  /**< Set the Trigger Delay */
        TIMEBASE_RANGE_US = 7,          /**< Set the Trigger Delay */
        CHANNEL_ACTIVE = 8,             /**< To which channel the settings for channel should be applied. PHDL_OSCI_CHANNEL_1 .. PHDL_OSCI_CHANNEL_4 */
        CHANNEL_BW_LIMIT,               /**< Is the BW Limit enabled? PH_ON or PH_OFF */
        CHANNEL_COUPLING,               /**< Coupling PHDL_OSCI_COUPLING_AC or PHDL_OSCI_COUPLING_DC */
        CHANNEL_DISPLAY,                /**< Is the selected channel displayed? PH_ON or PH_OFF*/
        CHANNEL_IMPEDANCE,              /**< The impedance of the selected channel. PHDL_OSCI_IMPEDANCE_FIFTY or PHDL_OSCI_IMPEDANCE_ONE_MEGA */
        CHANNEL_INVERT,                 /**< Is the channel inverted. PH_ON or PH_OFF*/
        CHANNEL_PROBE      = 0x000F,    /**< The atenuation of the attached probe. PHDL_OSCI_PROBE_ATTENUATION_1, PHDL_OSCI_PROBE_ATTENUATION_10, PHDL_OSCI_PROBE_ATTENUATION_100 */
        CHANNEL_UNITS      = 0x0012,    /**< The unit of the measured value. PHDL_OSCI_UNIT_VOLT, PHDL_OSCI_UNIT_AMPERE */
        CHANNEL_VERNIER    = 0x0013,    /**< Vernier enabled. PH_ON or PH_OFF */
        TRIGGER_TYPE       = 0x0014,    /**< Which trigger mode is used. At the moment only PHDL_OSCI_TRIGGER_TYPE_EDGE is supported */
        TRIGGER_SOURCE     = 0x0015,    /**< The source that is used for the current trigger type. PHDL_OSCI_CHANNEL_1 .. PHDL_OSCI_CHANNEL_4, PHDL_OSCI_CHANNEL_EXTERNAL, PHDL_OSCI_CHANNEL_LINE, PHDL_OSCI_CHANNEL_DIGITAL1 .. PHDL_OSCI_CHANNEL_DIGITAL15 */
        TRIGGER_EDGE_SLOPE = 0x0017,    /**< Which edge should be used. PHDL_OSCI_EDGE_SLOPE_POSITIVE, PHDL_OSCI_EDGE_SLOPE_NEGATIVE, PHDL_OSCI_EDGE_SLOPE_EITHER, PHDL_OSCI_EDGE_SLOPE_ALTERNATE*/
        TIMEBASE_MODE      = 0x0018,    /**< Mode of the timebase. PHDL_OSCI_TIMEBASE_MODE_MAIN, PHDL_OSCI_TIMEBASE_MODE_WINDOW, PHDL_OSCI_TIMEBASE_MODE_XY, PHDL_OSCI_TIMEBASE_MODE_ROLL */
        TIMEBASE_VERNIER   = 0x001C,    /**< Vernier enabled. PH_ON or PH_OFF */
        NUMBER_CHANNELS    = 0x001F,    /**< Getter for the number of data channels of the current osci */
        MEASUREMENT_OPTIONS= 0x0020     /**< Configure additional options when perform measurement. PHDL_OSCI_MEASUREMENT_ADJUST_TIME, PHDL_OSCI_MEASUREMENT_ADJUST_RANGE, PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER, PHDL_OSCI_MEASUREMENT_WAIT_STOP */
    };

    public enum Config64 : ushort
    {
        CHANNEL_OFFSET_MV    = 0x000E,  /**< The offset of the channel in mV */
        CHANNEL_SCALE_MV     = 0x0010,  /**< Scale in mV/div */
        CHANNEL_RANGE_MV     = 0x0011,  /**< Scale in mV/div */
        TRIGGER_LEVEL_MV     = 0x0016,  /**< The level that is used for the trigger in mV */
        TIMEBASE_POSITION_NS = 0x0019,  /**< Delay of the trigger point in ns */
        TIMEBASE_SCALE_NS    = 0x001A,  /**< Scale of the timebase in ns/div */
        TIMEBASE_RANGE_NS    = 0x001B,  /**< Range of the full scale timebase in ns */
        TIMEBASE_MAX_SAMPLES = 0x001D,  /**< Fixed Memory Max Sample Points for adjusting the Sampling Rate, the real used points may depend on the scope and sampling time/rate */
        CURRENT_SAMPLE_RATE  = 0x001E   /**< Getter for the current sample rate in Hz */
    }

    public enum Coupling : ushort
    {
        AC = 0,                         /**< Coupling AC */
        DC                              /**< Coupling DC */
    }

    public enum Impedance : ushort
    {
        FIFTY = 0,                      /**< 50 Ohm imput */
        ONE_MEGA                        /**< 1 Mega Ohm imput */
    }

    public enum ProbeAttenuation : ushort
    {
        ATTENUATION_1 = 1,              /**< 1:1 probe */
        ATTENUATION_10 = 10,            /**< 1:10 probe */
        ATTENUATION_100 = 100,          /**< 1:100 probe */
    }

    public enum Unit : ushort
    {
        VOLT = 0,                       /**< The unit of the channel is Volt */
        AMPERE                          /**< The unit of the channel is Ampere */
    }

    public enum TriggerType : ushort
    {
        EDGE = 0,                       /**< A edge is used for trigger */
    }

    public enum Channel : ushort
    {
        CHANNEL_1 = 1,                  /**< Channel 1 */
        CHANNEL_2,                      /**< Channel 2 */
        CHANNEL_3,                      /**< Channel 3 */
        CHANNEL_4,                      /**< Channel 4 */
        EXTERNAL,                       /**< External in */
        LINE,                           /**< Line in */
        DIGITAL0,                       /**< Digital input 0 */
        DIGITAL1,                       /**< Digital input 1 */
        DIGITAL2,                       /**< Digital input 2 */
        DIGITAL3,                       /**< Digital input 3 */
        DIGITAL4,                       /**< Digital input 4 */
        DIGITAL5,                       /**< Digital input 5 */
        DIGITAL6,                       /**< Digital input 6 */
        DIGITAL7,                       /**< Digital input 7 */
        DIGITAL8,                       /**< Digital input 8 */
        DIGITAL9,                       /**< Digital input 9 */
        DIGITAL10,                      /**< Digital input 10 */
        DIGITAL11,                      /**< Digital input 11 */
        DIGITAL12,                      /**< Digital input 12 */
        DIGITAL13,                      /**< Digital input 13 */
        DIGITAL14,                      /**< Digital input 14 */
        DIGITAL15                       /**< Digital input 15 */
    }

    public enum TriggerEdgeSlope : ushort
    {
        POSITIVE = 0,                   /**< Use a positive slope for trigger */
        NEGATIVE,                       /**< Use a negative slope for trigger */
        EITHER,                         /**< Use a either slope for trigger */
        ALTERNATE                       /**< Use a alternate slope for trigger */
    }

    public enum TimebaseMode : ushort
    {
        MAIN = 0,                       /**< The normal time base mode is the main time base. */
        WINDOW,                         /**< In the WINDow (zoomed or delayed) time base mode, measurements are made in the zoomed time base if possible */
        XY,                             /**< In the XY mode */
        ROLL                            /**< In the ROLL mode, data moves continuously across the display from left to right. The oscilloscope runs continuously and is untriggered. */
    }

    /*Osci divisors*/
    public enum Divisor : ushort
    {
        FieldStrength = 1000,       /**< Divisor for resolution of field strength (1mV resolution == 1000). */
        Voltage = 1000,             /**< Divisor for resolution of amplitude (1/1000 = 1mV). */
        FieldMultiplier = 1000,     /**< Divisor for resolution of field multiplier (1/1000 resolution == 1000). */
        Range = 10,                 /**< Divisor for range factor. */
    };

    /*Osci TriggerModes*/
    public enum Trigger : byte
    {
        AUTO = 0,                   /**<Trigger setting Auto */
        NORMAL,                     /**<Trigger setting Normal */
        SINGLE,                     /**<Trigger setting Single */
        STOP,                       /**<Trigger setting Stop */
    };

    /*Osci Measurement Options*/
    [Flags]
    public enum MeasurementOptions : byte
    {
        ADJUST_TIME = 0x0001,                   /**< Auto adjust timescale for measurements */
        ADJUST_RANGE = 0x0002,                  /**< Auto adjust range for measurements */
        SUPPRESS_FORCE_TRIGGER = 0x0004,        /**< Suppress the force trigger command (only supported with LeCroy) */
        WAIT_STOP = 0x0008                      /**< Wait until aquisition is in stopped (only supported with LeCroy) */
    };

    /*Osci configs*/
    public enum Mode : byte
    {
        FDT_PCD = 0,                /**<Frame Delay Time PCD setting */
        FDT_PICC_106_L3,            /**<Frame Delay Time PICC 106 Layer 3 setting */
        FDT_PICC_106_L4,            /**<Frame Delay Time PICC 106 Layer 4 setting */
        FDT_PICC_212_L4,            /**<Frame Delay Time PICC 212 Layer 4 setting */
        FDT_PICC_424_L4,            /**<Frame Delay Time PICC 424 Layer 4 setting */
        FDT_PICC_848_L4,            /**<Frame Delay Time PICC 848 Layer 4 setting */
        LIVE,                       /**<Live setting */
        SET,                        /**<SET setting */
        GLOBAL,                     /**<GLOBAL setting */
        VHD = 0x10,                 /**< VHD setting */
        EMD                         /**< EMD setting */
    };

    #endregion
    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetFieldStrength(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,                  /**< [In] Specifies the channel. */
            ref ushort pdwFieldStrength     /**< [Out] Field strength value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_InitOsci(
            IntPtr pDataParams          /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_SetConfig(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ushort pwValue	            /**< [In] Specifies parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetConfig(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ref ushort pwValue	        /**< [Out] Return parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_SetConfig64(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            long qwValue	            /**< [In] Specifies parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetConfig64(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ref long pqwValue	        /**< [Out] Return parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetAmpl(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            ref ushort pwAmplitude      /**< [Out] Amplitude value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetPeakPeak(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            ref double pdfPkPk          /**< [Out] Peak Peak value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetRMS(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            ref ushort pwRMS            /**< [Out] RMS value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetRMSDouble(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            ref double pdfRMS           /**< [Out] RMS value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_SetCorrRange(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel               /**< [In] Specifies the channel. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_GetWaveForm(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,                  /**< [In] Specifies the channel. */
            UInt32 dwWaveFormBufferSize,    /**< [In] Specifies the buffer size. */
            ref UInt32 pdwWaveFormLength,   /**< [Out] Specifies the waveform size. */
            ref byte pWaveFormBuffer,       /**< [Out] Waveform buffer  */
            ref byte pbHeaderOffset         /**< [Out] Header offset from Osci*/
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t GetFieldStrength(
            byte bChannel,
            out ushort pwFieldStrength
            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlOsci_GetFieldStrength(m_pDataParams, bChannel, ref wTmp);
            pwFieldStrength = wTmp;
            return status;
        }

        public Status_t GetAmpl(
            byte bChannel,
            out ushort pwAmplitude
            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlOsci_GetAmpl(m_pDataParams, bChannel, ref wTmp);
            pwAmplitude = wTmp;
            return status;
        }

        public Status_t GetPeakPeak(byte bChannel, out double pdfPkPk)
        {
            Status_t status;
            double wTmp = 0;
            status = phdlOsci_GetPeakPeak(m_pDataParams, bChannel, ref wTmp);
            pdfPkPk = wTmp;
            return status;
        }

        public Status_t GetRMS(byte bChannel, out ushort pwRMS)
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlOsci_GetRMS(m_pDataParams, bChannel, ref wTmp);
            pwRMS = wTmp;
            return status;
        }

        public Status_t GetRMSDouble(byte bChannel, out double pdfRMS)
        {
            Status_t status;
            double wTmp = 0;
            status = phdlOsci_GetRMSDouble(m_pDataParams, bChannel, ref wTmp);
            pdfRMS = wTmp;
            return status;
        }

        public Status_t SetConfig(
           ushort wIdentifier,
           ushort wValue
            )
        {
            return phdlOsci_SetConfig(m_pDataParams, wIdentifier, wValue);
        }

        public Status_t GetWaveForm(
             byte bChannel,
            UInt32 dwWaveFormBufferSize,
            ref UInt32 pdwWaveFormLength,
            ref byte pWaveFormBuffer,
            ref byte pbHeaderOffset
            )
        {
            Status_t status;
            status = phdlOsci_GetWaveForm(m_pDataParams, bChannel, dwWaveFormBufferSize, ref pdwWaveFormLength, ref pWaveFormBuffer, ref pbHeaderOffset);
            return status;
        }

        public Status_t InitOsci()
        {
            return phdlOsci_InitOsci(m_pDataParams);
        }

        public Status_t SetCorrRange(
            byte bChannel
            )
        {
            return phdlOsci_SetCorrRange(m_pDataParams, bChannel);
        }

        public Status_t GetConfig(
            ushort wIdentifier,
            out ushort pwValue
            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlOsci_GetConfig(m_pDataParams, wIdentifier, ref wTmp);
            pwValue = wTmp;
            return status;
        }

        public Status_t SetConfig64(
           ushort wIdentifier,
           long qwValue
            )
        {
            return phdlOsci_SetConfig64(m_pDataParams, wIdentifier, qwValue);
        }

        public Status_t GetConfig64(
            ushort wIdentifier,
            out long pqwValue
            )
        {
            Status_t status;
            long qTmp = 0;
            status = phdlOsci_GetConfig64(m_pDataParams, wIdentifier, ref qTmp);
            pqwValue = qTmp;
            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region LW64Xi

    public class LW64Xi : dlOsci.Generic
    {
        #region DEFINES
        public enum SelChannel : byte
        {
            F = 0x80                /**< Mask to select function channel. */
        };

        /* Osci Config */
        public enum Config : ushort
        {
            AUTOADJUSTCHANNELS = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,              /**< Sets Flag in DataParams to auto adjust channels when initializing the osci */
            NOISE_FILTER                                                            /**< Noise Filter Option */
        };

        /* Osci Config64 */
        public enum Config64 : ushort
        {
            WAVEFORM_SAMPLE_SIZE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 2         /**< Number samples that should be transfered when get waveform */
        };

        /*Osci Noise Filter Options */
        public enum NoiseFilter : ushort
        {
            NONE = 0,                   /**< Noise Filter None */
            HALFBIT,                    /**< Noise Filter 0.5 Bit */
            ONEBIT,                     /**< Noise Filter 1 Bit */
            ONEANDHALFBIT,              /**< Noise Filter 1.5 Bit */
            TWOBIT,                     /**< Noise Filter 2 Bit */
            TWOANDHALFBIT,              /**< Noise Filter 2.5 Bit */
            THREEBIT                    /**< Noise Filter 3 Bit */
        };
        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;                         /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		    /**< Pointer to the parameter structure of the balReg component. */
            public ushort wFieldStrengthMultiplier;    /**< Defines the conversion coefficient for the field Strengh. */
            public ushort wAverageFact;                /**< Defines the number of averages for the measurements. */
            public ushort wRangeMin;                   /**< Defines the minimum range for SetCorrRange. */
            public ushort wRangeMax;                   /**< Defines the maximum range for SetCorrRange. */
            public byte bMeasurementOptions;           /**< Store additional options for performing measurement. */
            public byte bSelectedChannel;              /**< Defines the channel that is selected to apply settings */
            public byte bAutoAdjustChannels;           /**< Auto Adjusts Channels on InitOsci if set */
            public UInt32 dwMaxSamplesMinValue;        /**< The minimal value that can be set for MaxSamples Config */
            public UInt32 dwMaxSamplesMaxValue;        /**< The maximum value that can be set for MaxSamples Config */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_LW64Xi_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                    /**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_LW64Xi_Cmd_SaveSetupSlotNumber(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_LW64Xi_Cmd_RecallSetupSlotNumber(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_LW64Xi_Cmd_SaveSetupFileName(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_LW64Xi_Cmd_RecallSetupFileName(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlOsci_LW64Xi_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_SaveSetupSlotNumber(
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_LW64Xi_Cmd_SaveSetupSlotNumber(m_pDataParams, bSlotNumber);
        }

        public Status_t Cmd_RecallSetupSlotNumber(
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_LW64Xi_Cmd_RecallSetupSlotNumber(m_pDataParams, bSlotNumber);
        }

        public Status_t Cmd_SaveSetupFileName(
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_LW64Xi_Cmd_SaveSetupFileName(m_pDataParams, pFileName);
        }

        public Status_t Cmd_RecallSetupFileName(
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_LW64Xi_Cmd_RecallSetupFileName(m_pDataParams, pFileName);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public LW64Xi()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~LW64Xi()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region DSO7052A

    public class DSO7052A : dlOsci.Generic
    {
        #region DEFINES
        public enum SelChannel : byte
        {
            F = 0x80                /**< Mask to select function channel. */
        };
        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;                         /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		    /**< Pointer to the parameter structure of the balReg component. */
            public ushort wFieldStrengthMultiplier;    /**< Defines the conversion coefficient for the field Strengh. */
            public ushort wCurrentTriggerMode;         /**< Represents the current Trigger setting in use. */
            public ushort wAverageFact;                /**< Defines the number of averages for the measurements. */
            public ushort wRangeMin;                   /**< Defines the minimum range for SetCorrRange. */
            public ushort wRangeMax;                   /**< Defines the maximum range for SetCorrRange. */
            public byte bMeasurementOptions;           /**< Store additional options for performing measurement. */
            public byte bSelectedChannel;              /**< Defines the channel that is selected for apply settings */
            public byte bAcOrDcMeasurementCh1;         /**< Defines if the following measurements are on AC or DC on Channel 1. */
            public byte bAcOrDcMeasurementCh2;         /**< Defines if the following measurements are on AC or DC on Channel 2. */
            public byte bAcOrDcMeasurementCh3;         /**< Defines if the following measurements are on AC or DC on Channel 3. */
            public byte bAcOrDcMeasurementCh4;         /**< Defines if the following measurements are on AC or DC on Channel 4. */
            public byte bNumberOfChannels;             /**< The number of channels available depending on Model. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                    /**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Cmd_SaveSetupSlotNumber(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Cmd_RecallSetupSlotNumber(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Cmd_SaveSetupFileName(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Cmd_RecallSetupFileName(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_DSO7052A_Cmd_GetAverage(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            ref ushort pwAverage        /**< [Out] Amplitude value. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlOsci_DSO7052A_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_SaveSetupSlotNumber(
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_DSO7052A_Cmd_SaveSetupSlotNumber(m_pDataParams, bSlotNumber);
        }

        public Status_t Cmd_RecallSetupSlotNumber(
            byte bSlotNumber        /**< [In] SlotNumber you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_DSO7052A_Cmd_RecallSetupSlotNumber(m_pDataParams, bSlotNumber);
        }

        public Status_t Cmd_SaveSetupFileName(
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_DSO7052A_Cmd_SaveSetupFileName(m_pDataParams, pFileName);
        }

        public Status_t Cmd_RecallSetupFileName(
            byte[] pFileName        /**< [In] FileName you want to save the current Osci Setup to */
            )
        {
            return phdlOsci_DSO7052A_Cmd_RecallSetupFileName(m_pDataParams, pFileName);
        }

        public Status_t Cmd_GetAverage(
            byte bChannel,
            out ushort pwAverage
            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlOsci_DSO7052A_Cmd_GetAverage(m_pDataParams, bChannel, ref wTmp);
            pwAverage = wTmp;
            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public DSO7052A()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~DSO7052A()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region PICOSCOPE6000

    public class PicoScope6000 : dlOsci.Generic
    {
        #region DEFINES
        public enum SelChannel : byte
        {
            F = 0x80                /**< Mask to select function channel. */
        };
        private const int MAX_CHANNELS = 5;

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                     /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;                       /**< Pointer to the parameter structure of the balReg component. */
            public ushort wAdditionalInfo;                         /**< Storage for additional error information. */
            public ushort wFieldStrengthMultiplier;                /**< Defines the conversion coefficient for the field Strengh. */
            public ushort wCurrentTriggerMode;                     /**< Represents the current Trigger setting in use. */
            public ushort wTriggerAutoTriggerMilliseconds;         /**< The time in milliseconds after which the trigger fires when trigger condition not reached */
            public ushort wAverageFact;                            /**< Defines the number of averages for the measurements. */
            public ushort wRangeMin;                               /**< Defines the minimum range for SetCorrRange. */
            public ushort wRangeMax;                               /**< Defines the maximum range for SetCorrRange. */
            public byte bMeasurementOptions;                       /**< Store additional options for performing measurement. */
            public uint dwLastSamplingTimebase;                    /**< Timebase used for the last sampling progress */
            /* Osci derivate parameter */
            public byte bNumberOfChannels;                         /**< The number of channels available depending on Model. */
            public ushort wBandwidthMHz;                           /**< Bandwidth of the device in MHz. */
            public ushort wMaxMemoryMS;                            /**< Available memory size of the device in Mega Samples. */
            public ushort wSamplingRateMSs;                        /**< Max Sampling rate of the device in Mega Samples per second. */
            public byte bAWG;                                      /**< Is a AWG available for the current device. */
            /* Channel setting */
            public byte bSelectedChannel;                          /**< Defines the channel that is selected for apply settings */
            public fixed byte bChannelEnabled[MAX_CHANNELS];       /**< Defines if a Channel is enabled */
            public fixed byte bChannelCoupling[MAX_CHANNELS];      /**< Defines if the channel coupling is AC or DC. */
            public fixed byte bChannelImpedance[MAX_CHANNELS];     /**< Defines the impedance of a channel. */
            public fixed byte bChannelBWLimit[MAX_CHANNELS];       /**< Defines if the BW Limit is enabled for the channel. */
            public fixed byte bChannelInvert[MAX_CHANNELS];        /**< Defines if the channel is inverted */
            public fixed ushort wChannelProbeGain[MAX_CHANNELS];   /**< Gain of the probe connected to this Channel */
            public fixed byte bChannelRangeIndex[MAX_CHANNELS];    /**< Range index of the Channel */
            /* Time Setting */
            public Int64 qwTimebaseRangeNs;                        /**< The full scale timebase that should be used */
            public Int64 qwTimebasePositionNs;                     /**< The offset of the timebase that should be used */
            public Int64 qwSamplePointsTarget;                     /**< The target number of points, the real number of points could be also
                                                                    *   higher if the target could not be reached with the sampling rate and downsampling or
                                                                    *   lower if the maximum sampling rate is already used and the time is to short */
            public byte bTriggerType;                              /**< current trigger type */
            public byte bTriggerSource;                            /**< current trigger source */
            public byte bTriggerEdgeSlope;                         /**< current trigger slope for edge type */
            public Int64 qwTriggerLevelmV;                         /**< current trigger level */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_PicoScope6000_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                    /**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlOsci_PicoScope6000_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public PicoScope6000()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~PicoScope6000()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region ISO3SOFTSCOPE

    public class ISO3SoftScope : dlOsci.Generic
    {
        #region DEFINES

        public enum Config : ushort
        {
            DATA_SRC         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,            /**< Set/Get current data source */
            TRIGGER_1_COND_RISE,                                                /**< Event condition for trigger 1 rise */
            TRIGGER_1_COND_FALL,                                                /**< Event condition for trigger 1 fall */
            TRIGGER_2_COND_RISE,                                                /**< Event condition for trigger 2 rise */
            TRIGGER_2_COND_FALL,                                                /**< Event condition for trigger 2 fall */
            TRIGGER_1_EXTERN_ENABLE,                                            /**< Output trigger 1 to external oscilloscope */
            TRIGGER_2_EXTERN_ENABLE,                                            /**< Output trigger 2 to external oscilloscope */
            BIG_BUFFER_TRIGGER_SRC,                                             /**< Specify the event which should be written to the big soft scope buffer */
            BIG_BUFFER_SAMPLE_SIZE,                                             /**< Soft scope buffer size in samples, default 0x50000 samples = 655kb */
            RING_BUFFER_TRG1_SAMPLE_SIZE,                                       /**< Ring buffer size in samples, read only */
            RING_BUFFER_TRG2_SAMPLE_SIZE,                                       /**< Ring buffer size in samples, read only */
            TRIGGER_RISE_BIT_NUMB,                                              /**< Rise on bit number, set Trigger rise condition on Bit Number */
            TRIGGER_FALL_BIT_NUMB,                                              /**< Fall on bit number, set Trigger rise condition on Bit Number */
            WAVEFORM_HEADER_ENABLED,                                            /**< Enable or disable wave form header of GetWaveform function */
            WAVEFORM_SAMPLE_SIZE,                                               /**< Softscope Waveform Size in samples */
            FDT_MEASUREMENT,                                                    /**< Enables/disables FDT measurement */
            TRIG1_SIGNAL_ID,
            TRIG2_SIGNAL_ID
        }

        public const int SAMPLE_SIZE = sizeof(short);                           /**< Size of a single wave shape sample */
        private const int RING_BUFFER_COUNT = 0x800;                            /**< Space of internal buffer for storing ring buffer samples */

        public enum Channel : byte
        {
            BIG_BUFFER = 1,                                                     /**< Read the big buffer of the FPGA starting from a keyword */
            RING_BUFFER_TRIGGER1 = 2,                                           /**< Read ring buffer used for Trigger 1 */
            RING_BUFFER_TRIGGER2 = 3,                                           /**< Read ring buffer used for Trigger 2 */
            BIG_BUFFER_TRG1_RISE = 4,                                           /**< Read big buffer area starting from trig1 rise condition */
            BIG_BUFFER_TRG2_RISE = 5,                                           /**< Read big buffer area starting from trig2 rise condition */
            BIG_BUFFER_TRG1_FALL = 6,                                           /**< Read big buffer area before trig1 fall condition */
            BIG_BUFFER_TRG2_FALL = 7,                                           /**< Read big buffer area before trig2 fall condition */
        }

        public enum TriggerCond : uint
        {
            TX_FIELD_OFF = 0x0U,                                                /**< Trigger event when switching off carrier */
            TX_IDLE = 0x1U,                                                     /**< Trigger event when transmitter enters IDLE STATE */
            TX_SOC = 0x2U,                                                      /**< Trigger event when transmitter modulates SOC */
            TX_LAST_BIT = 0x3U,                                                 /**< Trigger event when transmitter modulates last bit of a frame */
            TX_RX_AFTER_LAST_TX_BIT = 0x4U,                                     /**< Trigger event when transmitter activates receiver after last TX bit */
            TX_SOC_RX = 0x5U,                                                   /**< Trigger event when SOC was detected by receiver */
            TX_LMA_RX_EVENT = 0x6U,                                             /**< Trigger event when LMA event (8 subcarrier modulations) was detected */
            RX_FIRST_STATE_AFTER_IDLE = 0x9U,                                   /**< Trigger event when receiver changes state after IDLE */
            RX_FIFTH_BIT = 0xAU,                                                /**< Trigger event when receiver receives fifth bit. This Trigger is immune to EMD */
            RX_1ST_MOD = 0xCU,                                                  /**< Trigger event when receiver detected first modulation */
            RX_SOC = 0xDU,                                                      /**< Trigger event when receiver detected SOC */
            RX_BIT_NUM = 0xEU,                                                  /**< Trigger event when received a certain bit (see TRIGGER_CTRL1 ) */
            RX_EOC = 0xBU,                                                      /**< Trigger event when recieved EOC */
            /* RX_EOC = 0xFU, */                                                /**< Trigger event when recieved EOC Obsolete */
        };

        public enum TriggerSrc : ushort
        {
            MLS = 0x1,                                                          /**< Capture when MLS enabled and detected */
            FORCE = 0x4,                                                        /**< Force to capture */
            TRIGGER_1 = 0x8,                                                    /**< Trigger 1 will be logged into the soft scope buffer */
            TRIGGER_2 = 0x10,                                                   /**< Trigger 2 will be logged into the soft scope buffer */
        }

        public enum DataSrc : ushort
        {
            RX = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,                          /**< RX from ADC_IN (FPGA input from ADC) */
            CAL,                                                                /**< CAL from ADC_IN (FPGA input from ADC) */
            COS,                                                                /**< internal cos wave */
            SIN,                                                                /**< internal sinus wave */
            AGC,                                                                /**< AGC gain output */
            IQ,                                                                 /**< IQ output from modulator, before gain control */
            DAC                                                                 /**< Final input to DAC from FPGA */
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                                  /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pHalISO3DataParams;                                   /**< Pointer to the parameter structure of the underlaying component. */
            public uint dwLogCtrlReg;                                           /**< Register to configure soft scope signals and mls trigger */
            public uint dwLogSizeReg;                                           /**< Register to configure the size (number of 16-bit samples) of the soft scope buffer */
            public uint dwTriggerCtrl0Reg;                                      /**< Control register to set trigger events */
            public uint dwTriggerCtrl1Reg;                                      /**< Register to read number of received bits */
            public fixed uint pdwRingBuffer[RING_BUFFER_COUNT];                 /**< Temporary buffer to store samples of ring buffer for Trigger 1 and 2 */
            public uint dwLastExchangeCounter;                                  /**< ID of last exchange where FDT_DATA has been queried */
            public byte bWaveFormHeaderEnabled;                                 /**< Indicates whether waveform header is included by GetWaveForm function or not */
            public uint dwWaveFormSampleSize;                                   /**< Softscope Waveform Size in Samples */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Init(
            ref DataParams_t m_pDataParams,                                     /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                                           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalISO3DataParams                                           /**< [In] Pointer to the parameter structure of the hal ISO3 layer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Cmd_GetTriggerCounter(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte trigger,           /**< [In] SlotNumber you want to save the current Osci Setup to */
            ref uint riseCounter,   /**< [OUT] rise counter */
            ref uint fallCounter    /**< [OUT] fall counter */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Cmd_Reset(
            IntPtr pDataParams     /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Cmd_ForceTrigger(
            IntPtr pDataParams     /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Cmd_IsSignalInRange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            UInt32 signalSrc,       /**< Signal source to check range */
            out byte overflow       /**< Overflow flga: 0 means signal within range, 1 means signal out of range */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlOsci_ISO3SoftScope_Cmd_GetWaveFormBlocking(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,                  /**< [In] Specifies the channel. */
            UInt32 dwWaveFormBufferSize,    /**< [In] Specifies the buffer size. */
            ref UInt32 pdwWaveFormLength,   /**< [Out] Specifies the waveform size. */
            ref byte pWaveFormBuffer,       /**< [Out] Waveform buffer  */
            ref byte pbHeaderOffset         /**< [Out] Header offset from Osci*/
            );

        #endregion

        #region INIT

        public Status_t Init(
            Hal.ISO3 pHalISO3
            )
        {
            return phdlOsci_ISO3SoftScope_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHalISO3.m_pDataParams);
        }

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_GetTriggerCounter(
            byte channel,           /**< [In]  ISO3Softscope channel for trigger 1 or trigger 2 */
            out uint riseCounter,   /**< [OUT] Rise counter */
            out uint fallCounter    /**< [OUT] Fall counter */
            )
        {
            ushort result;
            uint rc = 0;
            uint fc = 0;
            result = phdlOsci_ISO3SoftScope_Cmd_GetTriggerCounter(m_pDataParams, channel, ref rc, ref fc);

            riseCounter = rc;
            fallCounter = fc;

            return result;
        }

        public Status_t Cmd_Reset()
        {
            return phdlOsci_ISO3SoftScope_Cmd_Reset(m_pDataParams);
        }

        public Status_t Cmd_ForceTrigger()
        {
            return phdlOsci_ISO3SoftScope_Cmd_ForceTrigger(m_pDataParams);
        }

        public Status_t Cmd_IsSignalInRange(UInt32 dataSrc, out byte overflow)
        {
            return phdlOsci_ISO3SoftScope_Cmd_IsSignalInRange(m_pDataParams, dataSrc, out overflow);
        }

        public Status_t Cmd_GetWaveFormBlocking(
            byte bChannel,
            UInt32 dwWaveFormBufferSize,
            ref UInt32 pdwWaveFormLength,
            ref byte pWaveFormBuffer,
            ref byte pbHeaderOffset
            )
        {
            Status_t status;
            status = phdlOsci_ISO3SoftScope_Cmd_GetWaveFormBlocking(m_pDataParams, bChannel, dwWaveFormBufferSize, ref pdwWaveFormLength, ref pWaveFormBuffer, ref pbHeaderOffset);
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public ISO3SoftScope()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~ISO3SoftScope()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
#endif
