/*
 * Copyright 2013, 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlRdFpga
{
    #region BASE

    #region DEFINES
    public enum Error : byte
    {
        ERR_DLL = (CustomCodes.ERROR_BEGIN)    /**< phCDllReader.dll threw an non standard error read wAddionalInfo to get error code */
    }

    public enum Config : int
    {
        ADDITIONAL_INFO = 0,    /**< Set / Get additional information . */
        MIN_POTI,		        /**< get minimum poti value. */
        MAX_POTI,               /**< get maximum poti value. */
        SNR_LIMIT_POTI,         /**< get the limit of the poti value for acceptable SNR. */
    }

      public enum phParamSetMode : uint
     {
         PH_RDL_PARAM_SETMODE_NORMAL = 0x0001,   /*!< \brief 0x0001 Set the value directly */
         PH_RDL_PARAM_SETMODE_DEFAULT = 0x0002,  /*!< \brief 0x0002 Set the value as default value */
         PH_RDL_PARAM_SETMODE_ALL = 0x0003       /*!< \brief 0x0003 Set the value directly and as default value */
     };

      public enum phFPGAParams : uint
      {
          PH_RDL_FPGA_PARAMETERS_BEGIN = 3 << 16,  /*!< \brief Indicates the Begin of the FPGA Parameters */
          PH_RDL_FPGA_ACTIVATION_STATUS = PH_RDL_FPGA_PARAMETERS_BEGIN,   /*!< \brief Indicates that id for the parameter to activate or deactivate the FPGA (see #phFPGAValues). */
          PH_RDL_FPGA_SEQUENCE_ON_TRIGGER,         /*!< \brief Indicates that id for the selection of the data sequence to play on trigger (see #phFPGAValues). */
          PH_RDL_FPGA_SELECTED_SEQUENCE,           /*!< \brief Indicates that id for the selection of the data sequence to to set some parameter (see #phFPGAValues). */
          PH_RDL_FPGA_RESET_CYCLES_OFF,            /*!< \brief Indicates that id for the setting of the number of OFF cycles of the rf pause. */
          PH_RDL_FPGA_RESET_CYCLES_ON,             /*!< \brief Indicates that id for the setting of the number of ON cycles after the rf pause. */
          PH_RDL_FPGA_DEFAULT_SEQUENCE_RELOAD_MODE,/*!< \brief Indicates that id for the setting of the reload mode of the patterns (see #phFPGAValues). */
          PH_RDL_FPGA_TIMEOUT_BEHAVIOUR,           /*!< \brief Indicates that id for the setting of the timeout behavior of the FPGA (see #phFPGAValues) */
          PH_RDL_FPGA_SEQUENCE_FDT_OFFSET_PCD,     /*!< \brief Indicates that id for the setting of the FDT Offset PCD (the pattern must be selected previous with #PH_RDL_FPGA_SELECTED_SEQUENCE) */
          PH_RDL_FPGA_AUTO_ADJUST_SEQUENCE,        /*!< \brief Indicates that id for the setting of the Auto Ajust Sequence mode (see #phFPGAValues). */
          PH_RDL_FPGA_FILTER_MODE,                 /*!< \brief Indicates that id for the setting of the Filter Mode (see #phFPGAValues). */
          PH_RDL_FPGA_DEFAULT_ENVELOPED_VALUE,     /*!< \brief Indicates that id for the setting of the default enveloped value (0..4095). */
          PH_RDL_FPGA_DEBUG_OUTPUT,                /*!< \brief Indicates that id for the setting of the Debug Output Mode (see #phFPGAValues). */
          PH_RDL_FPGA_FAST_EXCHANGE_MODE,          /*!< \brief Indicates that id for the setting of the Fast Exchange Mode (valid values are 0 (is deactivated) and 1 (is activated) ). */
          PH_RDL_FPGA_ANTENNA_TYPE,                /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */

          //vhbr psk
          PH_RDL_FPGA_SOC_MODE,                    /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_EOC_MODE,                    /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_CHANNEL_MODE,                /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_BITS_PER_SYMBOL,             /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_ETU,                         /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_SOC_SEQ,                     /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */

          //vhbr ask
          PH_RDL_FPGA_BIT_RATES,                   /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_START_BIT_MODE,              /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_STOP_BIT_MODE,               /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_EGT,                         /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_SOF_LOW_TIME,                /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_SOF_HIGH_TIME,               /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_EOF_LOW_TIME,                /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_EOF_HIGH_TIME,               /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_BIT_BOUNDARIES_HIGH,         /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_BIT_BOUNDARIES_LOW,          /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_VHBR_ASK_MODE,               /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */
          PH_RDL_FPGA_VHBR_PSK_MODE,               /*!< \brief Indicates that id for the setting of the antenna type (see #phFPGAValues) */

          PH_RDL_FPGA_PARAMETERS_END,              /*!< \brief 0x00060020 Indicates that the id is not a id of #phFPGAParams */
          PH_RDL_FPGA_PARAMETERS_LENGTH = (PH_RDL_FPGA_PARAMETERS_END - PH_RDL_FPGA_PARAMETERS_BEGIN)
      };

      /* FPGA String Params */
      enum phFPGAStringParams : uint
      {
          PH_RDL_FPGA_STRING_PARAMETERS_BEGIN = phFPGAParams.PH_RDL_FPGA_PARAMETERS_END,
          PH_RDL_FPGA_STRING_SEQUENCE_FILE = PH_RDL_FPGA_STRING_PARAMETERS_BEGIN,
          PH_RDL_FPGA_STRING_LOG_FILE_NAME,        /*!< \brief Indicates that id for the setting of the Debug Log File Name (Note: max file name len is 256)*/
          PH_RDL_FPGA_STRING_PARAMETERS_END,
          PH_RDL_FPGA_STRING_PARAMETERS_LENGTH = (PH_RDL_FPGA_STRING_PARAMETERS_END - PH_RDL_FPGA_STRING_PARAMETERS_BEGIN)
      };
    #endregion

    public abstract class Generic
    {

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetPotiPos(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wPos					/**< [In] Poti Position. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetDefaultPotiPos(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wPos					/**< [In] Poti Position. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetPotiPos(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref ushort pwPos				/**< [Out] Poti Position. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_IncPotiPos(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_DecPotiPos(
                                     IntPtr pDataParams			/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetTimeOutVal(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwTimeOutVal			/**< [In] Timeout value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetTimeOutVal(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref UInt32 pdwTimeOutVal		/**< [Out] Timeout value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LoadJoinerIniFile(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref byte pFilename			/**< [In] Pointer to filename. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LoadPattern(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref byte pFilename,			/**< [In] Pointer to filename. */
                                     byte bId					/**< [In] Pattern ID. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_PlayPattern(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     byte bId					/**< [In] Pattern ID. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_PlaySeqA(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_PlaySeqB(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_PlaySinus(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SelectTrigger(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     byte bTriggerSource			/**< [In] Trigger Source. */
                                     );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LabelReset(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wRisingEdgeCnt,		/**< [In] Rising edge count. */
                                     UInt32 dwSinus,				/**< [In] Sinus value. */
                                     UInt32 dwHfPause				/**< [In] HF pause value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LabelResetCounterReset(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wRisingEdgeCnt,		/**< [In] Rising edge count. */
                                     UInt32 dwSinus,				/**< [In] Sinus value. */
                                     UInt32 dwHfPause				/**< [In] HF pause value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LabelResetCounterResetSingleShot(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wRisingEdgeCnt,		/**< [In] Rising edge count. */
                                     UInt32 dwSinus,				/**< [In] Sinus value. */
                                     UInt32 dwHfPause 			/**< [In] HF pause value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LabelResetSingleShot(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wRisingEdgeCnt,		/**< [In] Rising edge count. */
                                     UInt32 dwSinus,				/**< [In] Sinus value. */
                                     UInt32 dwHfPause 			/**< [In] HF pause value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LabelResetWithCommand(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwSinus				/**< [In] Sinus value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateFDTMeasurement(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateFastExchange(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetFDT(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref UInt32 pdwFdt				/**< [Out] Frame Delay Time. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetFDTFast(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref UInt32 pdwFdt				/**< [Out] Frame Delay Time. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateAntiTearingMode(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwSinus,				/**< [In] Sinus value. */
                                     UInt32 dwHfPause				/**< [In] HF pause value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ResFreqTest(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwHfPause,			    /**< [In] HF pause value. */
                                     byte bRepeatFreqId			    /**< [In] Repeat frequency ID. */
                                     );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateFDTMeasurementPICCtoPCD(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwTime,				    /**< [In] Time value. */
                                     byte bCheckResp,			    /**< [In] Check response value. */
                                     ushort wRxByteCnt,			    /**< [In] Return byte count value. */
                                     byte bRxBitCnt,				/**< [In] Return bit count value. */
                                     ref byte pRxExpected			/**< [In] Pointer to expected response. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateLMAMeasurement(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateSequencerMode(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_DeactivateModes(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetJoinerRxGain(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     byte bValue					/**< [In] RX Gain value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetJoinerRxGain(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ref byte bValue				/**< [Out] RX Gain value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetFDTPCDforActiveDS(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwValue				    /**< [In] Delay value. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetFDTPCDforDS(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     UInt32 dwValue,				/**< [In] Delay value. */
                                     byte bId					    /**< [In] Pattern ID. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetConfig(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ushort wIdentifier,			/**< [In] Specifies parameter identifier. */
                                    ref ushort pwValue		    /**< [Out] Return parameter value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetVersion(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pwValue			/**< [Out] FPGA version value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetRevision(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pwValue			/**< [Out] FPGA revision value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetErrorCodes(
                                    IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pbErrors,          /**< [Out] Error values. */
                                    byte bMaxErrorCnt,          /**< [In] Specifies the maximum number of errors. */
                                    ref byte pbErrorCnt         /**< [Out] Number of errors received. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetClockSource(
                                    IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                    ushort wValue               /**< [In] Clock source value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetIQTimeoutValue(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ref UInt32 pdwValue         /**< [Out] IQ timeout value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetIQTimeoutValue(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    UInt32 pdwValue             /**< [In] IQ timeout value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetFirstResponse(
                                    IntPtr pDataParams,	        /**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pbRxBuffer,        /**< [In] Pointer to the receive buffer. */ /* :TODO simple pointer or double pointer? */
                                    ushort wRxBufferSize,       /**< [In] Receive buffer size. */
                                    ref ushort pwRxBytes,       /**< [Out] Number of bytes received. */
                                    ref byte pbNumValidRxBits,  /**< [Out] Number of bits received. */
                                    ref ushort pwError          /**< [Out] Protocol errors. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetResponse(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pbRxBuffer,        /**< [In] Pointer to the receive buffer. */ /* :TODO simple pointer or double pointer? */
                                    ushort wRxBufferSize,       /**< [In] Receive buffer size. */
                                    ref ushort pwRxBytes,       /**< [Out] Number of bytes received. */
                                    ref byte pbNumValidRxBits,  /**< [Out] Number of bits received. */
                                    ref ushort pwError          /**< [Out] Protocol errors. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetTerminalId(
                                    IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                    ref byte pbValue            /**< [Out] Terminal ID value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ResetJoiner(
                                    IntPtr pDataParams			/**< [In] Pointer to this layer's parameter structure. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_SetConfig32(
                                    IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                    UInt32 dwIdentifier,        /**< [In] Parameter Identifier. */
                                    UInt32 dwValue,              /**< [IN] Parameter Value. */
                                    byte bParamSetMode
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_GetConfig32(
                                    IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                    UInt32 dwIdentifier,        /**< [In] Parameter Identifier. */
                                    ref UInt32 dwValue          /**< [Out] Parameter Value. */
                                    );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_LoadFilterFile(
                                     IntPtr pDataParams,		/**< [In] Pointer to this layer's parameter structure. */
                                     ref byte pFilename			/**< [In] Pointer to filename. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ExchangeFPGACommand(
                                     IntPtr pDataParams,		/**< [In] Pointer to this layer's parameter structure. */
                                     ref byte txData,           /**< [In] Pointer to transmitted Data. */
                                     UInt32 txLen,              /**< [In] Parameter transmitted Data Length. */
                                     ref byte rxData,           /**< [Out] Pointer to recieved Data. */
                                     UInt32 maxRxLen,           /**< [In] Parameter maximal recieved Data Length. */
                                     ref UInt32 rxLen           /**< [Out] Pointer to recieved Data Length. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_ActivateTypeBMode(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS
        public Status_t SetPotiPos(ushort wPos)
        {
            return phdlRdFpga_SetPotiPos(m_pDataParams, wPos);
        }

        public Status_t SetDefaultPotiPos(ushort wPos)
        {
            return phdlRdFpga_SetDefaultPotiPos(m_pDataParams, wPos);
        }

        public Status_t GetPotiPos(out ushort pwPos)
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlRdFpga_GetPotiPos(m_pDataParams, ref wTmp);
            pwPos = wTmp;
            return status;
        }

        public Status_t IncPotiPos()
        {
            return phdlRdFpga_IncPotiPos(m_pDataParams);
        }

        public Status_t DecPotiPos()
        {
            return phdlRdFpga_DecPotiPos(m_pDataParams);
        }

        public Status_t SetTimeOutVal(UInt32 dwTimeOutVal)
        {
            return phdlRdFpga_SetTimeOutVal(m_pDataParams, dwTimeOutVal);
        }

        public Status_t GetTimeOutVal(out UInt32 pdwTimeOutVal)
        {
            Status_t status;
            UInt32 dwTmp = 0;
            status = phdlRdFpga_GetTimeOutVal(m_pDataParams, ref dwTmp);
            pdwTimeOutVal = dwTmp;
            return status;
        }

        public Status_t LoadJoinerIniFile(ref byte pFilename)
        {
            return phdlRdFpga_LoadJoinerIniFile(m_pDataParams, ref pFilename);
        }

        public Status_t LoadFilterFile(ref byte pFilename)
        {
            return phdlRdFpga_LoadFilterFile(m_pDataParams, ref pFilename);
        }

        public Status_t ExchangeFPGACommand(ref byte txData, UInt32 txLen, out byte rxData, UInt32 maxRxLen, out UInt32 rxLen)
        {
            Status_t status;
            byte rxDataTemp = 0;
            UInt32 rxLenTemp = 0;
            status = phdlRdFpga_ExchangeFPGACommand(m_pDataParams, ref txData, txLen, ref rxDataTemp, maxRxLen, ref rxLenTemp);
            rxData = rxDataTemp;
            rxLen = rxLenTemp;
            return status;
        }

        public Status_t LoadPattern(ref byte pFilename, byte bId)
        {
            return phdlRdFpga_LoadPattern(m_pDataParams, ref pFilename, bId);
        }

        public Status_t PlayPattern(byte bId)
        {
            return phdlRdFpga_PlayPattern(m_pDataParams, bId);
        }

        public Status_t PlaySeqA()
        {
            return phdlRdFpga_PlaySeqA(m_pDataParams);
        }

        public Status_t PlaySeqB()
        {
            return phdlRdFpga_PlaySeqB(m_pDataParams);
        }

        public Status_t PlaySinus()
        {
            return phdlRdFpga_PlaySinus(m_pDataParams);
        }

        public Status_t SelectTrigger(byte bTriggerSource)
        {
            return phdlRdFpga_SelectTrigger(m_pDataParams, bTriggerSource);
        }

        public Status_t LabelReset(ushort wRisingEdgeCnt, UInt32 dwSinus, UInt32 dwHfPause)
        {
            return phdlRdFpga_LabelReset(m_pDataParams, wRisingEdgeCnt, dwSinus, dwHfPause);
        }

        public Status_t LabelResetCounterReset(ushort wRisingEdgeCnt, UInt32 dwSinus, UInt32 dwHfPause)
        {
            return phdlRdFpga_LabelResetCounterReset(m_pDataParams, wRisingEdgeCnt, dwSinus, dwHfPause);
        }

        public Status_t LabelResetCounterResetSingleShot(ushort wRisingEdgeCnt, UInt32 dwSinus, UInt32 dwHfPause)
        {
            return phdlRdFpga_LabelResetCounterResetSingleShot(m_pDataParams, wRisingEdgeCnt, dwSinus, dwHfPause);
        }

        public Status_t LabelResetSingleShot(ushort wRisingEdgeCnt, UInt32 dwSinus, UInt32 dwHfPause)
        {
            return phdlRdFpga_LabelResetSingleShot(m_pDataParams, wRisingEdgeCnt, dwSinus, dwHfPause);
        }

        public Status_t LabelResetWithCommand(UInt32 dwSinus)
        {
            return phdlRdFpga_LabelResetWithCommand(m_pDataParams, dwSinus);
        }

        public Status_t ActivateFDTMeasurement()
        {
            return phdlRdFpga_ActivateFDTMeasurement(m_pDataParams);
        }

        public Status_t ActivateFastExchange()
        {
            return phdlRdFpga_ActivateFastExchange(m_pDataParams);
        }

        public Status_t GetFDT(out UInt32 pdwFdt)
        {
            Status_t status;
            UInt32 dwTmp = 0;
            status = phdlRdFpga_GetFDT(m_pDataParams, ref dwTmp);
            pdwFdt = dwTmp;
            return status;
        }

        public Status_t GetFDTFast(out UInt32 pdwFdt)
        {
            Status_t status;
            UInt32 dwTmp = 0;
            status = phdlRdFpga_GetFDTFast(m_pDataParams, ref dwTmp);
            pdwFdt = dwTmp;
            return status;
        }

        public Status_t ActivateAntiTearingMode(UInt32 dwSinus, UInt32 dwHfPause)
        {
            return phdlRdFpga_ActivateAntiTearingMode(m_pDataParams, dwSinus, dwHfPause);
        }

        public Status_t ResFreqTest(UInt32 dwSinus, byte bRepeatFreqId)
        {
            return phdlRdFpga_ActivateAntiTearingMode(m_pDataParams, dwSinus, bRepeatFreqId);
        }

        public Status_t ActivateFDTMeasurementPICCtoPCD(
                                                         UInt32 dwTime,
                                                         byte bCheckResp,
                                                         ushort wRxByteCnt,
                                                         byte bRxBitCnt,
                                                         ref byte pRxExpected
                                                        )
        {
            return phdlRdFpga_ActivateFDTMeasurementPICCtoPCD(
                                                m_pDataParams,
                                                dwTime,
                                                bCheckResp,
                                                wRxByteCnt,
                                                bRxBitCnt,
                                                ref pRxExpected
                                                );
        }

        public Status_t ActivateLMAMeasurement()
        {
            return phdlRdFpga_ActivateLMAMeasurement(m_pDataParams);
        }

        public Status_t ActivateSequencerMode()
        {
            return phdlRdFpga_ActivateSequencerMode(m_pDataParams);
        }

        public Status_t DeactivateModes()
        {
            return phdlRdFpga_DeactivateModes(m_pDataParams);
        }

        public Status_t SetJoinerRxGain(byte bValue)
        {
            return phdlRdFpga_SetJoinerRxGain(m_pDataParams, bValue);
        }

        public Status_t GetJoinerRxGain(out byte bValue)
        {
            Status_t status;
            byte bTmp = 0;
            status = phdlRdFpga_GetJoinerRxGain(m_pDataParams, ref bTmp);
            bValue = bTmp;
            return status;
        }

        public Status_t SetFDTPCDforActiveDS(UInt32 dwValue)
        {
            return phdlRdFpga_SetFDTPCDforActiveDS(m_pDataParams, dwValue);
        }

        public Status_t SetFDTPCDforDS(UInt32 dwValue, byte bId)
        {
            return phdlRdFpga_SetFDTPCDforDS(m_pDataParams, dwValue, bId);
        }

        public Status_t GetConfig(ushort wIdentifier, out ushort pwValue)
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlRdFpga_GetConfig(m_pDataParams, wIdentifier, ref wTmp);
            pwValue = wTmp;
            return status;
        }

        public Status_t GetVersion(out byte bValue)
        {
            Status_t status;
            byte bTmp = 0;
            status = phdlRdFpga_GetVersion(m_pDataParams, ref bTmp);
            bValue = bTmp;
            return status;
        }

        public Status_t GetRevision(out byte bValue)
        {
            Status_t status;
            byte bTmp = 0;
            status = phdlRdFpga_GetRevision(m_pDataParams, ref bTmp);
            bValue = bTmp;
            return status;
        }

        public Status_t GetErrorCodes(ref byte pbErrors, byte bMaxErrorCnt, ref byte pbErrorCnt)
        {
            return phdlRdFpga_GetErrorCodes(m_pDataParams, ref pbErrors, bMaxErrorCnt, ref pbErrorCnt);
        }

        public Status_t SetClockSource(byte bValue)
        {
            return phdlRdFpga_SetClockSource(m_pDataParams, bValue);
        }

        public Status_t GetFirstResponse(ref byte pbRxBuffer, ushort wRxBufferSize, ref ushort pwRxBytes, ref byte pbNumValidRxBits, ref ushort pwError)
        {
            return phdlRdFpga_GetFirstResponse(m_pDataParams, ref pbRxBuffer, wRxBufferSize, ref pwRxBytes, ref pbNumValidRxBits, ref pwError);
        }

        public Status_t GetResponse(ref byte pbRxBuffer, ushort wRxBufferSize, ref ushort pwRxBytes, ref byte pbNumValidRxBits, ref ushort pwError)
        {
            return phdlRdFpga_GetResponse(m_pDataParams, ref pbRxBuffer, wRxBufferSize, ref pwRxBytes, ref pbNumValidRxBits, ref pwError);
        }

        public Status_t SetIQTimeoutValue(UInt32 dwValue)
        {
            return phdlRdFpga_SetIQTimeoutValue(m_pDataParams, dwValue);
        }

        public Status_t GetIQTimeoutValue(out UInt32 bValue)
        {
            Status_t status;
            UInt32 bTmp = 0;
            status = phdlRdFpga_GetIQTimeoutValue(m_pDataParams, ref bTmp);
            bValue = bTmp;
            return status;
        }

        public Status_t GetTerminalId(out byte bValue)
        {
            Status_t status;
            byte bTmp = 0;
            status = phdlRdFpga_GetTerminalId(m_pDataParams, ref bTmp);
            bValue = bTmp;
            return status;
        }

        public Status_t ResetJoiner()
        {
            return phdlRdFpga_ResetJoiner(m_pDataParams);
        }

        public Status_t SetConfig32(UInt32 dwIdentifier, UInt32 dwValue, byte bParamSetMode)
        {
            return phdlRdFpga_SetConfig32(m_pDataParams, dwIdentifier,dwValue, bParamSetMode);
        }

        public Status_t GetConfig32(UInt32 dwIdentifier, out UInt32 dwValue)
        {
            Status_t status;
            UInt32 dwTmp = 0;
            status = phdlRdFpga_GetConfig32(m_pDataParams, dwIdentifier, ref dwTmp);
            dwValue = dwTmp;
            return status;
        }

        public Status_t ActivateTypeBMode()
        {
            return phdlRdFpga_ActivateTypeBMode(m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }
    #endregion

    #region V6

    public class V6 : dlRdFpga.Generic
    {
        #region DEFINES
        public enum Trigger : byte
        {
            INTERN = 1,  /*!< 1 Intern FPGA trigger. */
            EXTERN = 2   /*!< 2 Extern FPGA trigger. */
        }

        public enum ResFreqRepRate : byte
        {
            RATE_0_5 = 0x01,	/*!< 0x01 0.5Hz / 2000ms */
            RATE_1 = 0x02,	/*!< 0x02 1Hz / 1000ms */
            RATE_2 = 0x03,	/*!< 0x03 2Hz / 500ms */
            RATE_4 = 0x04,	/*!< 0x04 4Hz / 250ms */
            RATE_8 = 0x05,	/*!< 0x05 8Hz / 125ms */
            RATE_16_13 = 0x06,	/*!< 0x06 16.13Hz / 62ms */
            RATE_32_25 = 0x07,	/*!< 0x07 32.25Hz / 31ms */
            RATE_62_5 = 0x08,	/*!< 0x08 62.5Hz / 16ms */
            RATE_125 = 0x09,	/*!< 0x09 125Hz / 8ms */
            RATE_250 = 0x0A	/*!< 0x0A 250Hz / 4ms */
        }

        public enum ClockSource : byte
        {
            INTERN = 0,         /*!< 0x00 Internal clock source */
            EXTERN = 1,         /*!< 0x01 External clock source */
            EXTERN_DOUBLE = 2   /*!< 0x02 External clock*2 as clock source */
        }

        #endregion
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;                         /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		    /**< Pointer to the parameter structure of the balReg component. */
            public ushort wAdditionalInfo;		        /**< Storage for additional error information. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlRdFpga_V6_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlRdFpga_V6_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        public DataParams_t[] m_DataParamsInt;

        public V6()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~V6()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

}
#endif
