/*
 * Copyright 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlSPA
{
    #region BASE

    #region DEFINES

    public enum ConfigFSV : ushort
    {
        /*FSV Functions*/
        IQ_MEASUREMENT = (CustomCodes.CONFIG_BEGIN),        /**< Configures the IQ Measuremnet*/
        ACTIVE_TRACE,                                       /**< Configures the Active Trace*/
        ACTIVE_PROBE_CONFIGURATION                          /**< Configures the Active Probe*/
    }

    public enum Error : byte
    {
        /* Custom errors*/
        ERR_SELF_TEST_FAIL = (CustomCodes.ERROR_BEGIN),     /**<Self test failed*/
        ERR_CALIBRATION_FAIL,                               /**<Calibartion failed*/
        ACTIVE_PROBE_NOT_CONNECTED                          /**<Calibartion failed*/
    };

    /* SPA Config */
    public enum Config64 : ushort
    {
        TRIGGER_OFFSET = 0x0B,              /**< Select the externelTrigger offset*/
        CENTER_FREQUENCY = 0x0F,            /**< Select the frequenzy configuration  */
        FREQUENCY_SPAN,                     /**< Select the frequenzy span configuration  */
        FREQUENCY_OFFSET,                   /**< Select the frequenzy offset configuration  */
        FREQUENCY_START,                    /**< Select the frequenzy start configuration  */
        FREQUENCY_STOP,                     /**< Select the frequenzy stop configuration  */
        BANDWIDTH_RESOLUTION = 0x1F,        /**< Select the bandwidth resultion configuration */
        BANDWIDTH_RESOLUTION_VIDEO = 0x21,  /**< Select the bandwidth resultion configuration */
        SWEEP_TIME = 0x27,                  /**< Select the Sweep Time*/
        REFERENCE_LEVEL = 0x25,             /**< Select the reference level */
        REFERENCE_LEVEL_OFFSET = 0x2B       /**< Select the reference level offset*/
    };

    public enum Config : ushort
    {
        TRIGGER_SOURCE = 0x00,          /**< Select the trigger source S902*/
        TRIGGER_SLOPE,                  /**< Select the trigger slope S902*/
        TRIGGER_LEVEL_IFPOWER,          /**< Select the IF power trigger level S900*/
        TRIGGER_LEVEL_AMABS,            /**< Select the AM power trigger level S900*/
        TRIGGER_LEVEL_AMREL,            /**< Select the AM relativ trigger level S900*/
        TRIGGER_LEVEL_EXTERNAL,         /**< Select the external trigger level S900*/
        TRIGGER_LEVEL_FMPOWER,          /**< Select the FM power trigger level S900*/
        TRIGGER_LEVEL_PMPOWER,          /**< Select the PM power trigger level S900*/
        TRIGGER_LEVEL_RFPOWER,          /**< Select the RF power trigger level S900*/
        TRIGGER_LEVEL_VIDEO,            /**< Select the video trigger level S900*/
        TRIGGER_CONDITION_MASKCOND,     /**< Select the mask condition for mask trigger S900*/

        MODE = 0x0C,                    /**< Select the instrument mode */
        OUTPUT_FORMAT,                  /**< Select the output format ASCII or Binary */

        FREQUENCY_MODE = 0x0E,          /**< Select the display trace configuration */

        SWEEP_MODE = 0x14,              /**< Select the sweep mode configuration  */
        SWEEP_TYPE,                     /**< Select the sweep type configuration  */
        SWEEP_COUNT,                    /**< Select the sweep  count configuration  */
        SWEEP_POINTS,                   /**< Select the sweep points configuration  */

        AVERAGE_STATE = 0x18,           /**< Set averaging to on or off mode */
        AVERAGE_COUNT,                  /**< Set the count for the averaging mode */
        AVERAGE_TYPE,                   /**< Set averaging type */

        INPUT_COUPLING = 0x1B,          /**< Select the input coupling configurations */
        INPUT_IMPEDANCE,                /**< Select the input impedanceconfigurations */
        INPUT_ATTENUATION_VALUE,        /**< Select the input attenuation value configurations */
        INPUT_ATTENUATION_AUTO,         /**< Select the input attenuation auto configurations */

        BANDWIDTH_FILTER_TYPE = 0x20,   /**< Select the bandwidth filter configuration */

        TRACE_DETECTOR = 0x22,          /**< Select the trace detector configuration */
        TRACE_MODE,                     /**< Select the trace mode */
        POWER_UNIT,                     /**< Select the unit for the power */
        RUN_MODE = 0x26,                /**< Select the Run mode*/

        CONTINUE_MEAS =0x28,            /**< Continue measurement*/
        OPERATION_COMPLETED_STATE,      /**< Get Operation completed state*/
        PRESET_SPA,                     /**< Resets the instrument*/
        MEASURE_CONFIGURATION = 0x2C   /**< Configure the measuremnt source*/
    };

    /* Unit for the power settings (possible values for config POWER_UNIT) */
    public enum PowerUnit : ushort
    {
        DBM = 0,                    /**< Define the unit for the power to dBm S695*/
        DBMV,                       /**< Define the unit for the power to dBmV S695*/
        DBUV,                       /**< Define the unit for the power to dBuV S695*/
        DBUA,                       /**< Define the unit for the power to dBuA S695*/
        DBPW,                       /**< Define the unit for the power to dBpW S695*/
        VOLT,                       /**< Define the unit for the power to Volt S695*/
        AMPERE,                     /**< Define the unit for the power to Ampere S695*/
        WATT                        /**< Define the unit for the power to Watt S695*/
    };

    /*SPA TrigggerSources (possible values for config TRIGGER_SOURCE)*/
    public enum TriggerSource : ushort
    {
        EXTERNAL = 0,               /**< Define the External trigger */
        IFPOWER,                    /**< Define the IF power trigger */
        FREERUN,                    /**< Define the Free run trigger */
        MASK,                       /**< Define the Mask trigger */
        TIMEDOMAIN,                 /**< Define the Timedomain trigger */
        TIME,                       /**< Define the Time trigger */
        VIDEO,                      /**< Define the Video trigger */
        AFPOWER,                    /**< Define the AF power trigger */
        FMPOWER,                    /**< Define the FM power trigger */
        AMPOWER,                    /**< Define the AM power trigger */
        AMRELATIV,                  /**< Define the AM relativ trigger */
        PMPOWER                     /**< Define the PM power trigger */
    };

    /*SPA TriggerSlope (possible values for config TRIGGER_SLOPE)*/
    public enum TriggerSlope : ushort
    {
        POSITIVE = 0,               /**< Define the Positive trigger slope */
        NEGATIVE                    /**< Define the Negativ trigger slope */
    };

    /*SPA TriggerMaskCond (possible values for config TRIGGER_CONDITION_MASKCOND)*/
    public enum TriggerMaskCond : ushort
    {
        ENTER = 0,                  /**< Define the On mask enter */
        LEAVE,                      /**< Define the On masl leave */
        IN,                         /**< Define the Inside mask */
        OUT                         /**< Define the Outside mask */
    };

    /*SPA Mode (possible values for config MODE)*/
    public enum Mode : ushort
    {
        SPECTRUM = 0,               /**< Define the Spectrum mode */
        REALTIME,                   /**< Define the Realtime mode */
        ANALOGDEMOD                 /**< Define the Analog demodulation mode */
    };

    /*SPA OutputFormat (possible values for config OUTPUT_FORMAT)*/
    public enum OutputFormat : ushort
    {
        ASCII = 0,                  /**< Define the Output file is ASCII */
        REAL,                       /**< Define the Output file is binary */
        REAL_HEADER                 /**< Define the Output file is binary and add a header at the beginning */
    };

    /*SPA FrequncyMode (possible values for config FREQUENCY_MODE)*/
    public enum FrequencyMode : ushort
    {
        CW = 0,                     /**< Define the Zero Span mode */
        FIXED,                      /**< Define the Zero span mode */
        SWEEP                       /**< Define the Sweep mode */
    };

    /*SPA SweepMode (possible values for config SWEEP_MODE)*/
    public enum SweepMode : ushort
    {
        AUTO = 0,                   /**< Define the Automatic mode*/
        SPECTRUM_EMMISION_MASK,     /**< Define the Spectrum emmision mode*/
        SPURIOS_EMMISION            /**< Define the Spurios emmision mode*/
    };

    /*SPA SweepType (possible values for config SWEEP_TYPE)*/
    public enum SweepType : ushort
    {
        SWEEPLIST = 0,              /**< Define the Sweeplist type*/
        AUTO,                       /**< Define the Automatic type*/
        FFT                         /**< Define the FFT type*/
    };

    /*SPA AveragingState (possible values for config AVERAGE_Type)*/
    public enum AveragingType : ushort
    {
        VIDEO = 0,                  /**< Define the Video averaging state*/
        LINEAR,                     /**< Define the Linear averaging state*/
        POWER                       /**< Define the Power averaging state*/
    };

    /*SPA InputCoupling (possible values for config INPUT_COUPLING)*/
    public enum InputCoupling : ushort
    {
        AC = 0,                     /**< Define the AC coupling*/
        DC                          /**< Define the DC coupling*/
    };

    /*SPA BandwidthFilter (possible values for config BANDWIDTH_FILTER_TYPE)*/
    public enum BandwidthFilter : ushort
    {
        NORMAL = 0,                 /**< Define the Normal Bandwidth filter*/
        CHANNEL,                    /**< Define the Channel Bandwidth filter*/
        RRC,                        /**< Define the RRC Bandwidth filter*/
        P5                          /**< Define the P5 Bandwith filter*/
    };

    /*SPA TraceDetector (possible values for config TRACE_DETECTOR)*/
    public enum TraceDetector : ushort
    {
        APEAK = 0,                  /**< Define the detector type to APeak*/
        NEGATIVE,                   /**< Define the detector type to Negative*/
        POSITIVE,                   /**< Define the detector type to Positive*/
        SAMPLE,                     /**< Define the detector type to Sample*/
        RMS,                        /**< Define the detector type to RMS*/
        AVERAGE,                    /**< Define the detector type to Average*/
        QPEAK,                      /**< Define the detector type to QPeak*/
        CAVERAGE,                   /**< Define the detector type to CAverage*/
        CRMS                        /**< Define the detector type to CRMS*/
    };

    /*SPA TraceMode (possible values for config TRACE_MODE)*/
    public enum TraceMode : ushort
    {
        WRITE = 0,                  /**< Define the trace type to Write*/
        VIEW,                       /**< Define the trace type to View*/
        AVERAGE,                    /**< Define the trace type to Average*/
        MAXHOLD,                    /**< Define the trace type to Maxhold*/
        MINHOLD,                    /**< Define the trace type to Minhold*/
        BLANK                       /**< Define the trace type to Blank*/
    };

    /*SPA RunMode (possible values for config RUN_MODE)*/
    public enum RunMode : ushort
    {
        CONTINOUSE = 0,             /**< Define the run mode contiouse */
        SINGLE,                     /**< Define the run mode single */
        ABORT                       /**< Cancels all current measurements*/
    };

    /*SPA RunMode (possible values for config MEASURE_CONFIGURATION)*/
    public enum MeasureConfiguration : ushort
    {
        PEAK = 0,                   /**< Set measure type to Peak*/
        MEAN,                       /**< Set measure type to Mean*/
        RMS,                        /**< Set measure type to RMS*/
        SDEVIATION                  /**< Set measure type to standard deviation*/
    };

    /*SPA ActiveProbeConfig (possible values for config ACTIVE_PROBE_CONFIGURATION)*/
    public enum ActiveProbeConfiguration : ushort
    {
        DEACTIVATED = 0,/**< Activate Active Probe*/
        ACTIVATED,      /**< Deactivate Active Probe*/
        NOT_CONNECTED,  /**< Active Probe is not connected*/
        NOT_QUERIED     /**< The state was not queried*/
    };

    #endregion
    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_InitSPA(
            IntPtr pDataParams          /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_SetConfig(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ushort wValue               /**< [In] Specifies parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_GetConfig(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ref ushort pwValue          /**< [Out] Return parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_SetConfig64(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            long qwValue                /**< [In] Specifies parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_GetConfig64(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,         /**< [In] Specifies parameter identifier. */
            ref long pqwValue           /**< [Out] Return parameter value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_GetTrace(
            IntPtr pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            byte bChannel,              /**< [In] Specifies the channel. */
            UInt32 dwTraceBufferSize,   /**< [In] Specifies the buffer size. */
            ref UInt32 pdwTraceLength,  /**< [Out] Specifies the trace size. */
            ref byte pTraceBuffer,      /**< [Out] Trace buffer  */
            ref byte pbHeaderOffset     /**< [Out] Header offset from SPA*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_GetMeasurement(
            IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
            ushort wMeasureSourceType,  /**< [In] Specifies measured source type. */
            ref double fValue            /**< [OUT] Pointer to measured value. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS
        public Status_t InitSPA()
        {
            return phdlSPA_InitSPA(m_pDataParams);
        }

        public Status_t SetConfig(
           ushort wIdentifier,
           ushort wValue
            )
        {
            return phdlSPA_SetConfig(m_pDataParams, wIdentifier, wValue);
        }

        public Status_t GetConfig(
            ushort wIdentifier,
            out ushort pwValue
            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlSPA_GetConfig(m_pDataParams, wIdentifier, ref wTmp);
            pwValue = wTmp;
            return status;
        }

        public Status_t SetConfig64(
            ushort wIdentifier,
            long qwValue
            )
        {
            return phdlSPA_SetConfig64(m_pDataParams, wIdentifier, qwValue);
        }

        public Status_t GetConfig64(
            ushort wIdentifier,
            out long pqwValue
            )
        {
            Status_t status;
            long qwTmp = 0;
            status = phdlSPA_GetConfig64(m_pDataParams, wIdentifier, ref qwTmp);
            pqwValue = qwTmp;
            return status;
        }

        public Status_t GetTrace(
            byte bChannel,
            UInt32 dwTraceBufferSize,
            ref UInt32 pdwTraceLength,
            ref byte pTraceBuffer,
            ref byte pbHeaderOffset
            )
        {
            Status_t status;
            status = phdlSPA_GetTrace(m_pDataParams, bChannel, dwTraceBufferSize, ref pdwTraceLength, ref pTraceBuffer, ref pbHeaderOffset);
            return status;
        }

        public Status_t GetMeasurement(
            ushort wMeasureSourceType,
            ref double fValue
            )
        {
            Status_t status;
            status = phdlSPA_GetMeasurement(m_pDataParams, wMeasureSourceType, ref fValue);
            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region FSV

    public class FSV : dlSPA.Generic
    {
        #region DEFINES
        private const int SERIAL_NUMBER_SIZE = 20;          /**< The size of the seriel number*/
        private const int SOFTWARE_VERSION_SIZE = 8;        /**< The size of the software number*/
        private const int DEVICE_ID_STRING_SIZE = 10;       /**< The size of the id string*/
        private const int ACTIVE_PROBE_NAME_SIZE = 10;      /**< The size of the Active Probe name string*/
        private const int ACTIVE_PROBE_SERIAL_ID_SIZE = 30; /**< The size of the Active Probe seriel id string*/
        private const int ACTIVE_PROBE_PART_ID_SIZE = 30;   /**< The size of the Active Probe part id string*/

        public enum SelChannel : byte
        {
            F = 0x80                                        /**< Mask to select function channel. */
        };
        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                                 /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;                                          /**< Pointer to the parameter structure of the balReg component. */
            byte bMaxFreq;                                                     /**< Holds the maximum frequenzy value of the instrument*/
            public long bMaxVideoBandwidth;                                    /**< Holds the maximum vidoe Bandwith value*/
            public byte bUseHeaderForTrace;                                    /**< Holds the value if a header should be added to the trace file */
            public fixed byte bDevId[SERIAL_NUMBER_SIZE];                      /**< Holds the device ID value of the instrument*/
            public fixed byte bSwVersion[SOFTWARE_VERSION_SIZE];               /**< Holds the software version value of the instrument*/
            public fixed byte bDevIdStr[DEVICE_ID_STRING_SIZE];                /**< Holds the device id string*/
            public ushort wActiveProbeState;                                   /**< Holds the Active Probe state*/
            public fixed byte bActiveProbeName[ACTIVE_PROBE_NAME_SIZE];        /**< Holds the name of the Active Probe if connected*/
            public fixed byte bActiveProbeSerNum[ACTIVE_PROBE_SERIAL_ID_SIZE]; /**< Holds the seriel number of the Active Probe if connected*/
            public fixed byte bActiveProbePartId[ACTIVE_PROBE_PART_ID_SIZE];   /**< Holds the seriel number of the Active Probe if connected*/
            public ushort wMaxAttValue;                                        /**< Holds the max attenuation value*/
            public long qMaxRefLevel;                                          /**< Holds the max reference level value*/
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlSPA_FSV_Init(
            ref DataParams_t m_pDataParams,                 /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                        /**< [In] Pointer to the parameter structure of the balReg component. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Visa pBal
            )
        {
            return phdlSPA_FSV_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public FSV()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~FSV()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Setter & Getter for DataParams structure
        unsafe  private string ByteArrayToString(byte* arr, int maxLength)
        {
            char [] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }

        public string InstrumentIdStr
        {
            get
            {
                unsafe
                {
                    fixed(DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = ByteArrayToString(pDataParams->bDevIdStr, DEVICE_ID_STRING_SIZE);
                        return value;
                    }
                }
            }
        }

        #endregion
    }

    #endregion
}
#endif
