#if PACKAGE_INTERNAL
/*
 * Copyright 2013, 2017, 2019, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlStepper
{
    #region BASE

    #region DEFINES
    public enum Error : byte
    {
       UPPER_LIMIT_INDICATOR = CustomCodes.ERROR_BEGIN,	/**< Stepper touch the upper limit indicator. */
       LOWER_LIMIT_INDICATOR,		                        /**< Stepper touch the lower limit indicator. */
       POS_OUT_OF_LIMITS,					                /**< If the requested position is out of reach. */
       DRIVE_FAILURE						                /**< Stepper Hardware does not work. */
    }
    public enum Dir : ushort
    {
        /* direction of stepper */
        DOWN = 0,
        UP = 1
    }
    public enum Config : ushort
    {
        RAMP_DELAY = 0,         /* number of milliseconds one single speed is kept during the ramp */
        RAMP_ON_OFF,            /* indicates if an acceleration ramp is used */
        DIR,                    /* indicates the direction the stepper will move */
        START_SPEED,            /* indicates the speed the carriage will begin to move */
        MAX_SPEED,              /* speed of the carriage after the acceleration ramp */
        STEPS_WAY_RATIO,        /* ratio between the steps of the motor and the moved distance (in mm) */
        DISTANCE,               /* distance in mm the stepper move in DRIVE_DISTANCE mode */
        MAX_CURRENT,            /* The maximal current indicate the maximal current in % */
        STDBY_CURRENT,          /* Standby current indicates the standby motor hold current in % */
        STDBY_CURRENT_FLAG,     /* The standby current flag indicate if the motor current is switched on while the motor is not driving */
    }
    public enum Config32 : ushort
    {
        POSITION_COUNTER = 0x10,/* indicates the numbers of steps the carriage is away from a defined zero point */
        MAX_POSITION            /* indicates the maximum position of the stepper */
    }
    #endregion

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public abstract class Generic
    {
        #region DLLIMPORTS


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_SetConfig(
                             IntPtr pDataParams,		        /**< [In] Pointer to this layer's parameter structure. */
                             ushort wConfig,			        /**< [In] Configuration Identifier */
                             ushort wValue			            /**< [In] Configuration Value */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_SetConfig32(
                             IntPtr pDataParams,		        /**< [In] Pointer to this layer's parameter structure. */
                             ushort wConfig,			        /**< [In] Configuration Identifier */
                             UInt32 dwValue			            /**< [In] Configuration Value */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_GetConfig(
                            IntPtr pDataParams,                 /**< [In] Pointer to this layers parameter structure. */
                            ushort wConfig,			            /**< [In] Configuration Identifier */
                            ref ushort pValue			        /**< [Out] Return parameter value. */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_GetConfig32(
                            IntPtr pDataParams,                 /**< [In] Pointer to this layers parameter structure. */
                            ushort wConfig,			            /**< [In] Configuration Identifier */
                            ref UInt32 pValue			        /**< [Out] Return parameter value. */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_MoveSteps(
                            IntPtr pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
                            UInt16 wSpeed,                      /**< [In] Moving speed. */
                            byte bDirection,                    /**< [In] Moving direction. */
                            UInt32 wSteps,                      /**< [In] Moving steps. */
                            byte bBlocking                      /**< [In] Indicates (if = 1) if the funktion waits till the movement is complited before it returns. */
                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_MoveDistance(
                            IntPtr pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
                            UInt16 wSpeed,		                /**< [In] Moving speed. */
                            byte bDirection,	                /**< [In] Moving direction. */
                            UInt32 wDistance,	                /**< [In] Moving distance. */
                            byte bBlocking                      /**< [In] Indicates (if = 1) if the funktion waits till the movement is complited before it returns. */
                            );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_GoToPosition(
                            IntPtr pDataParams,			        /**< [In] Pointer to this layer's parameter structure. */
                            UInt16 wSpeed,			            /**< [In] Moving speed. */
                            UInt32 wPosition,	                /**< [In] Desired position. */
                            byte bBlocking                      /**< [In] Indicates if the funktion waits till the movement is complited before it returns. */
                            );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_Initialize(
                             IntPtr pDataParams		            /**< [In] Pointer to this layer's parameter structure. */
                             );


        #endregion

        #region DLL_WRAPPED_FUNCTIONS


        public Status_t MoveSteps(UInt16 wSpeed, byte bDirection, UInt32 wSteps, byte bBlocking)
        {
            return phdlStepper_MoveSteps(m_pDataParams, wSpeed, bDirection, wSteps, bBlocking);
        }

        public Status_t MoveDistance(UInt16 wSpeed, byte bDirection, UInt32 wDistance, byte bBlocking)
        {
            return phdlStepper_MoveDistance(m_pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        }

        public Status_t GoToPosition(UInt16 wSpeed, UInt32 wPosition, byte bBlocking)
        {
            return phdlStepper_GoToPosition(m_pDataParams, wSpeed, wPosition, bBlocking);
        }

        public Status_t SetConfig(ushort wConfig, ushort wValue)
        {
            return phdlStepper_SetConfig(m_pDataParams, wConfig, wValue);
        }

        public Status_t SetConfig32(ushort wConfig, UInt32 dwValue)
        {
            return phdlStepper_SetConfig32(m_pDataParams, wConfig, dwValue);
        }

        public Status_t GetConfig(ushort wConfig, out ushort pValue)
        {
            pValue = 0;
            return phdlStepper_GetConfig(m_pDataParams, wConfig, ref pValue);
        }

        public Status_t GetConfig32(ushort wConfig, out UInt32 pValue)
        {
            pValue = 0;
            return phdlStepper_GetConfig32(m_pDataParams, wConfig, ref pValue);
        }

        public Status_t Initialize()
        {
            return phdlStepper_Initialize(m_pDataParams);
        }

        public abstract IntPtr m_pDataParams
        {
            get;
        }
        #endregion
    }
    #endregion

    #region V1

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class V1 : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            public byte bInit;						/**< Indicates the state of the connection */
            public byte bState;					    /**< Current state parameter */
            public ushort wRamp_delay;				/**< Indicates the number of ms, one single speed is kept during the ramp. */
            public byte bRamp_on_of;				/**< Indicates, if an acceleration ramp is used (0 = off, 1 = start & stor ramp). */
            public byte bDirection;				    /**< Indicates the direction of the stepper (0 = down, 1 = up). */
            public ushort wStart_Speed;			    /**< Indicates the direction of the stepper (0 = down, 1 = up) */
            public ushort wMax_Speed;				/**< Maximal speed of the ramp. */
            public ushort wSteps_Way_Ratio;		    /**< Ratio between steps of the motor and moved mm of the stepper. */
            public ushort wDistance;				/**< Distance in mm the stepper move in DRIVE_DISTANCE mode. */
            public byte bMaxCurrent;				/**< Indicates the maximal current in %. */
            public byte bStdby_Current;			    /**< Indicates the standby motor hold current in %. */
            public byte bStdby_Current_Flag;		/**< Indicates if motor current is on during non-drive. */
            public uint dwMax_Position;			    /**< Indicates the maximal reachable position of the stepper. */
        };
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_V1_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlStepper_V1_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        public V1()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~V1()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region Robot

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class Robot : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            public byte bInit;						/**< Indicates the state of the connection */
            public ushort wMaxSpeed;				/**< Maximal speed of the ramp. */
            public ushort wStepsWayRatio;		    /**< Ratio between steps of the motor and moved mm of the stepper. */
            public uint dwMaxPosition;			    /**< Indicates the maximal reachable position of the stepper. */
            public ushort wMaxGetPositionRetrys;    /**< How often the get Position loop is called if an error occours */
            public ushort wWaitPositionTimeout;     /**< Number of loops used if position limit not is used */
        };
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_Robot_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlStepper_Robot_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        public Robot()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Robot()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region Microbot

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class Microbot : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
            public byte bInit;						/**< Indicates the state of the connection */
            public ushort wStepsWayRatio;		    /**< Ratio between steps of the motor and moved mm of the stepper. */
            public byte bPositionMode;              /**< How the position is set */
            public byte bActiveAxis;                /**< Which Axis is used with the set Position functions */
            public int dwPositionX;                 /**< Current X Position. */
            public int dwPositionY;                 /**< Current Y Position. */
            public int dwPositionZ;                 /**< Current Z Position. */
            public short wAngle;                    /**< Current Angle Position. */
        };
        #endregion

        #region DEFINES
        public enum Config : ushort
        {
            POSITION_MODE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,               /**<  */
            ACTIVE_AXIS   = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+1              /**<  */
        }
        public enum Config32 : ushort
        {
            POSITION_X = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+2,               /**<  */
            POSITION_Y = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+3,               /**<  */
            POSITION_Z = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+4                /**<  */
        }

        public enum PositionMode : int
        {
            SET_CURRENT_AS_ZERO = 0x01, /**< Sets the current positon as Zero Position */
            REMOTE = 0x02,              /**< The Position is done with remote command */
            LOCAL = 0x03,               /**< Position is done with buttons on Mircobot */
            COLLECT = 0x04,             /**< Position set with SetConfig32 are only stored but is not send to the robot */
            START = 0x05,               /**< The robot moves to the actual stored position and the Mode is set to REMOTE */
            START_NONBLOCKING = 0x06    /**< The robot moves to the actual stored position and the Mode is set to REMOTE with nonblocking flag*/
        }

        public enum ActiveAxis : ushort
        {
            X = 0x01,   /**< Axis X is used as default */
            Y = 0x02,   /**< Axis Y is used as default */
            Z = 0x03    /**< Axis Z is used as default */
        }
        #endregion


        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_Microbot_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlStepper_Microbot_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        public Microbot()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Microbot()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region HighZ

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class HighZ : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                          /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;		    /**< Pointer to the parameter structure of the balReg component. */
            public byte bInit;						    /**< Indicates the state of the connection */
            public byte bPowerIsOn;                     /**< Indicates if the engines are powered */
            public fixed ushort wMicrometerPerTurn[3];   /**< Ratio between steps of the motor and moved um of the stepper. */
            public fixed ushort wStepsPerTurn[3];        /**< Ratio between steps of the motor and moved um of the stepper. */
            public uint dwMaxRangeX;           /**< Max value in um for X axis. */
            public uint dwMaxRangeY;           /**< Max value in um for Y axis. */
            public uint dwMaxRangeZ;           /**< Max value in um for Z axis. */

            public ushort wRampDelayMin;         /**< Min delay for movement. */
            public ushort wRampDelayMax;         /**< Max delay for movement. */
            public ushort wRampSpeed;            /**< Speed of the acceleration ramp. */
            public byte bRampEnabled;           /**< Flag if acceleration is enabled */

            public int dwPaddingX;             /**< ep_switches_offset for X axis */
            public int dwPaddingY;             /**< ep_switches_offset for Y axis */
            public int dwPaddingZ;             /**< ep_switches_offset for Z axis */

            public uint dwDriveTimeout;        /**< maximum time that the drives is allowed to be done, depends on delay and #steps */

            public byte bPositionMode;              /**< How the position is set */
            public byte bActiveAxis;                /**< Which Axis is used with the set Position functions */
            public int dwPositionX;                 /**< Current X Position. */
            public int dwPositionY;                 /**< Current Y Position. */
            public int dwPositionZ;                 /**< Current Z Position. */
        };
        #endregion

        #region DEFINES
        public enum Config : ushort
        {
            POSITION_MODE       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,    /**< mode used for set multiple axis. */
            ACTIVE_AXIS         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN +1, /**< Axis used in Set Commands. */
            RAMP_DELAY_MIN      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+11, /**< Delay between steps for full speed. */
            RAMP_DELAY_MAX      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+12, /**< Delay between steps for min speed. */
            ROBOT_SPEED         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+13, /**< Speed of the robot, value is then calculated to delay, not all values are possible. */
            RAMP_SPEED          = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+14, /**< Speed of the acceleration ramp. */
            DISTANCE_PER_TURN_X = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+15, /**< Distance in um that is performed per full turn of the robot for x axis. */
            DISTANCE_PER_TURN_Y = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+16, /**< Distance in um that is performed per full turn of the robot for y axis. */
            DISTANCE_PER_TURN_Z = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+17, /**< Distance in um that is performed per full turn of the robot for z axis. */
            STEPS_PER_TURN_X    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+18, /**< Steps of the engine to perform a full turn of the x axis. */
            STEPS_PER_TURN_Y    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+19, /**< Steps of the engine to perform a full turn of the y axis. */
            STEPS_PER_TURN_Z    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+20  /**< Steps of the engine to perform a full turn of the z axis. */

        }
        public enum Config32 : ushort
        {
            POSITION_X = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 2, /**< indicates the position in um the carriage is away from a defined zero point. */
            POSITION_Y = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 3, /**< indicates the position in um the carriage is away from a defined zero point. */
            POSITION_Z = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 4, /**< indicates the position in um the carriage is away from a defined zero point. */
            MAX_X      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 5, /**< max position for the x axis. */
            MAX_Y      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 6, /**< max position for the y axis. */
            MAX_Z      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 7, /**< max position for the z axis. */
            PADDING_X  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 8, /**< padding for the x axis. */
            PADDING_Y  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 9, /**< padding for the y axis. */
            PADDING_Z  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 10 /**< padding for the z axis. */
        }

        public enum PositionMode : int
        {
            SET_CURRENT_AS_ZERO = 0x01, /**< Sets the current positon as Zero Position */
            CLEAR_CURRENT_ZERO = 0x07,  /**< Clears the current Zero Position */
            REMOTE = 0x02,              /**< The Position is done with remote command */
            COLLECT = 0x04,             /**< Position set with SetConfig32 are only stored but is not send to the robot */
            START = 0x05                /**< The robot moves to the actual stored position and the Mode is set to REMOTE */
        }

        public enum ActiveAxis : ushort
        {
            X = 0x01,   /**< Axis X is used as default */
            Y = 0x02,   /**< Axis Y is used as default */
            Z = 0x03    /**< Axis Z is used as default */
        }
        #endregion


        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_HighZ_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlStepper_HighZ_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        public HighZ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~HighZ()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region DensoVS60

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class DensoVS60 : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                      /**< Layer ID for this component, NEVER MODIFY! */
            public byte bInit;					                    /**< Indicates the state of the connection */
            public ushort wStepsWayRatio;		                    /**< Ratio between steps of the motor and moved mm of the stepper. */
            public byte bPositionMode;                              /**< How the position is set */
            public byte bPositionModeCorrect;                       /**< How the position is set */
            public byte bActiveAxis;                                /**< Which Axis is used with the set Position functions */
            public byte bActiveAxisType;                            /**< Which Axis Type is used as active for collect  mode */
            public byte bArmTaken;                                  /**< Arm control flag: 1 - taken, 0 -free */
            public byte bActiveRotation;                            /**< Active rotation control flag: 1 - taken, 0 -free */
            public ushort wRobotSpeed;                              /**< Speed of the robot arm from 1 to 1000 */
            public byte bRobotAcceeration;                          /**< Acceeration of the robot arm */
            public fixed byte pPortName[MAX_IP_LENGTH];             /**< Ip adresse of the robot arm */
            public IntPtr pEngine;                                  /**< Void pointer to the ICaoEngine object of the Denso Robot. */
            public IntPtr pController;                              /**< Void pointer to the ICaoContoler object of the Denso Robot. */
            public IntPtr pWorkspaces;                              /**< Void pointer to the ICaoWorkspaces object of the Denso Robot. */
            public IntPtr pWorkspace;                               /**< Void pointer to the ICaoWorkspaces object of the Denso Robot. */
            public IntPtr pArm;                                     /**< Void pointer to the ICaoRobot object of the Denso Robot. */
            public fixed UInt64 pDevicesArray[IO_DEVICE_MAX_NUMBER];/**< Array of pointers to the IO devices of the robot arm. */
            public DataParams_t_XY_coord_t structTargetXy;          /**< Structure that holds coordinates of the X-Y axis */
            public DataParams_t_Joint_coord_t structTargetJoint;    /**< Structure that holds coordinates of the Joint axis */
        };


        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t_XY_coord_t
        {
            public int dwPositionX;          /**< Current X Position. */
            public int dwPositionY;          /**< Current Y Position. */
            public int dwPositionZ;          /**< Current Z Position. */
            public int dwPositionRx;         /**< Current Rx Position. */
            public int dwPositionRy;         /**< Current Ry Position. */
            public int dwPositionRz;         /**< Current Rz Position. */
            public short wFigure;            /**< Current Figure Position. */
            public DataParams_t_Offset_XY_coord_t offsetCoords;    /**< Array with offset values */
        };

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t_Joint_coord_t
        {
            public int dwPositionJ1;          /**< Current J1 Position. */
            public int dwPositionJ2;          /**< Current J2 Position. */
            public int dwPositionJ3;          /**< Current J3 Position. */
            public int dwPositionJ4;          /**< Current J4 Position. */
            public int dwPositionJ5;          /**< Current J5 Position. */
            public int dwPositionJ6;          /**< Current J6 Position. */
            public int dwPositionJ7;          /**< Current J7 Position. */
            public int dwPositionJ8;          /**< Current J8 Position. */
            public DataParams_t_Offset_Joint_coord_t offsetCoords;     /**< Array with offset values */
        };

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t_Offset_XY_coord_t
        {
            public double dwPositionX;          /**< Current X Position. */
            public double dwPositionY;          /**< Current Y Position. */
            public double dwPositionZ;          /**< Current Z Position. */
            public double dwPositionRx;         /**< Current Rx Position. */
            public double dwPositionRy;         /**< Current Ry Position. */
            public double dwPositionRz;         /**< Current Rz Position. */
            public double wFigure;            /**< Current Figure Position. */
        };

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t_Offset_Joint_coord_t
        {
            public double dwPositionJ1;          /**< Current J1 Position. */
            public double dwPositionJ2;          /**< Current J2 Position. */
            public double dwPositionJ3;          /**< Current J3 Position. */
            public double dwPositionJ4;          /**< Current J4 Position. */
            public double dwPositionJ5;          /**< Current J5 Position. */
            public double dwPositionJ6;          /**< Current J6 Position. */
            public double dwPositionJ7;          /**< Current J7 Position. */
            public double dwPositionJ8;          /**< Current J8 Position. */
        };
        #endregion

        #region DEFINES
        private const int MAX_IP_LENGTH = 16;           /**< Maximum number of chars in the ip adresse string */
        private const int IO_DEVICE_MAX_NUMBER = 40;    /**< The maximum number of possible IO devices ports */

        public enum Config : ushort
        {
            POSITION_MODE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,                 /**<  */
            ACTIVE_AXIS = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 1,               /**<  */
            ACTIVATE_LASER = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 2,            /**<  */
            ACTIVATE_PUMP = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 3,             /**<  */
            CHECK_HEAD_SENSOR = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 4,         /**<  */
            ACTIVE_ROTATION = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 20,          /**<  */
            ROBOT_SPEED = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 21,              /**<  */
            ROBOT_ACCELERATION = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 22,       /**<  */
            CORRECT_COLLECT = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 23           /**<  */
        }

        public enum Config32 : ushort
        {
            POSITION_X  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 5,               /**<  */
            POSITION_Y  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 6,               /**<  */
            POSITION_Z  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 7,               /**<  */
            POSITION_RX = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 8,               /**<  */
            POSITION_RY = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 9,               /**<  */
            POSITION_RZ = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 10,              /**<  */
            POSITION_FIG = NxpRdLibNet.CustomCodes.CONFIG_BEGIN +11,              /**<  */

            POSITION_J1  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 12,             /**<  */
            POSITION_J2  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 13,             /**<  */
            POSITION_J3  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 14,             /**<  */
            POSITION_J4 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 15,              /**<  */
            POSITION_J5 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 16,              /**<  */
            POSITION_J6 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 17,              /**<  */
            POSITION_J7 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 18,              /**<  */
            POSITION_J8 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 19,               /**<  */

            OFFSET_X  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 20,                 /**<  */
            OFFSET_Y  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 21,                 /**<  */
            OFFSET_Z  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 22,                 /**<  */
            OFFSET_RX = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 23,                 /**<  */
            OFFSET_RY = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 24,                 /**<  */
            OFFSET_RZ = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 25,                /**<  */
            OFFSET_FIG = NxpRdLibNet.CustomCodes.CONFIG_BEGIN +26,                /**<  */

            OFFSET_J1  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 27,             /**<  */
            OFFSET_J2  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 28,             /**<  */
            OFFSET_J3  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 29,             /**<  */
            OFFSET_J4 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 30,              /**<  */
            OFFSET_J5 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 31,              /**<  */
            OFFSET_J6 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 32,              /**<  */
            OFFSET_J7 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 33,              /**<  */
            OFFSET_J8 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 34               /**<  */
        }

        public enum PositionMode : int
        {
            SET_CURRENT_AS_ZERO = 0x01,     /**< Sets the current positon as Zero Position */
            SET_CLEAR_ZERO_OFFSET = 0x02,   /**< Clears the current offset and sets it back  */
            REMOTE = 0x03,                  /**< Remote mode via contoller  */
            LOCAL = 0x04,                   /**< Local mode, directly over the software  */
            COLLECT = 0x05,                 /**< Position set with SetConfig32 are only stored but is not send to the robot */
            START = 0x06,                   /**< The robot moves to the actual stored position and the Mode is set to REMOTE */
            START_NONBLOCKING = 0x07,       /**< The robot moves to the actual stored position and the Mode is set to REMOTE with nonblocking flag*/
            HALT_MOVE = 0x08                /**< Stops the move of the robot when executed  */
        }

        public enum ActiveAxis : ushort
        {
            X =  0x01,   /**< Axis X is used as default */
            Y =  0x02,   /**< Axis Y is used as default */
            Z =  0x03,   /**< Axis Z is used as default */
            RX = 0x04,   /**< Axis Rx is used as default */
            RY = 0x05,   /**< Axis Ry is used as default */
            RZ = 0x06,   /**< Axis Rz is used as default */
            FIG = 0x07,  /**< Axis Fig is used as default */
            J1 = 0x08,   /**< Axis X is used as default */
            J2 = 0x09,   /**< Axis Y is used as default */
            J3 = 0x10,   /**< Axis Z is used as default */
            J4 = 0x11,   /**< Axis Z is used as default */
            J5 = 0x12,   /**< Axis Z is used as default */
            J6 = 0x13,   /**< Axis Z is used as default */
            J7 = 0x14,   /**< Axis Z is used as default */
            J8 = 0x15    /**< Axis Z is used as default */
        }
        #endregion


        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams                    /**< [In] Specifies the size of the data parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_GoDownUntilContact(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            Int32 dwMinZPoint,                           /**< [In] Z-axis value to which the robor should lower to */
            UInt16 dwRobotSpeed                           /**< [In] Robot speed */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SplineExecuteMove(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            byte bPathIndex,                            /**< [In] Index of the choosen path for the spline movement */
            byte bBlocking                              /**< [In] Blocking flag 1 - Block, 0 - Non blocking */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SplineClearPath(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            byte   bPathIndex                           /**< [In] Index of the choosen path for the spline movement */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SplineExecuteMoveInverse(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            byte   bPathIndex,                          /**< [In] Index of the choosen path for the spline movement */
            byte   bBlocking                            /**< [In] Blocking flag 1 - Block, 0 - Non blocking */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SplineAddPathPoint(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            byte   bPathIndex,                          /**< [In] Index of the choosen path for the spline movement */
            Int32  dwPositionX,                         /**< [In] Position of the X axis in the micrometers */
            Int32  dwPositionY,                         /**< [In] Position of the Y axis in the micrometers */
            Int32  dwPositionZ                          /**< [In] Position of the Z axis in the micrometers */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_WaitUntilMoveFinished(
            IntPtr pDataParams                         /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SetPort(
            IntPtr pDataParams,                         /**< [In] Pointer to this layers parameter structure. */
            byte[] PortName                             /**< [In] Port Name as String. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_SetToolDef(
            IntPtr pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
            double x,                                   /**< [In] Coordinate to set the new ToolDef. */
            double y,                                   /**< [In] Coordinate to set the new ToolDef. */
            double z,                                   /**< [In] Coordinate to set the new ToolDef. */
            double rx,                                  /**< [In] Coordinate to set the new ToolDef. */
            double ry,                                  /**< [In] Coordinate to set the new ToolDef. */
            double rz,                                  /**< [In] Coordinate to set the new ToolDef. */
            ushort wtoolNumber                          /**< [In] ToolNumber to be set. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_PickupCard(
            IntPtr pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_DensoVS60_Cmd_DropCard(
            IntPtr pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
            );

        #endregion

        #region INIT
        public Status_t Init()
        {
            return phdlStepper_DensoVS60_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }
        #endregion

        #region CUSTOM_FUNCTIONS

        /// <summary>
        /// Move the robot arm down until it touches the surface
        /// </summary>
        /// <param name="dwMinZPoint"></param>
        /// <param name="dwRobotSpeed"></param>
        /// <returns></returns>
        public Status_t Cmd_GoDownUntilContact(
            Int32 dwMinZPoint,
            UInt16 dwRobotSpeed
            )
        {
            return phdlStepper_DensoVS60_Cmd_GoDownUntilContact(m_pDataParams, dwMinZPoint, dwRobotSpeed);
        }

        /// <summary>
        /// Execute spline movement with the robot arm
        /// </summary>
        /// <param name="bPathIndex"></param>
        /// <param name="bBlocking"></param>
        /// <returns></returns>
        public Status_t Cmd_SplineExecuteMove(
               byte bPathIndex,
               byte bBlocking
               )
        {
            return phdlStepper_DensoVS60_Cmd_SplineExecuteMove(m_pDataParams, bPathIndex, bBlocking);
        }

        /// <summary>
        /// Clear the saved spline path from the memory
        /// </summary>
        /// <param name="bPathIndex"></param>
        /// <returns></returns>
        public Status_t Cmd_SplineClearPath(
               byte bPathIndex
               )
        {
            return phdlStepper_DensoVS60_Cmd_SplineClearPath(m_pDataParams, bPathIndex);
        }

        /// <summary>
        /// Execute the inverse spline movement with the robot arm
        /// </summary>
        /// <param name="bPathIndex"></param>
        /// <param name="bBlocking"></param>
        /// <returns></returns>
        public Status_t Cmd_SplineExecuteMoveInverse(
               byte bPathIndex,
               byte bBlocking
               )
        {
            return phdlStepper_DensoVS60_Cmd_SplineExecuteMoveInverse(m_pDataParams, bPathIndex, bBlocking);
        }

        /// <summary>
        /// Add point cooridnates to the selected spline path
        /// </summary>
        /// <param name="bPathIndex"></param>
        /// <param name="dwPositionX"></param>
        /// <param name="dwPositionY"></param>
        /// <param name="dwPositionZ"></param>
        /// <returns></returns>
        public Status_t Cmd_SplineAddPathPoint(
               byte bPathIndex,
               Int32 dwPositionX,
               Int32 dwPositionY,
               Int32 dwPositionZ
               )
        {
            return phdlStepper_DensoVS60_Cmd_SplineAddPathPoint(m_pDataParams, bPathIndex, dwPositionX, dwPositionY, dwPositionZ);
        }

        /// <summary>
        /// Wait with the next command until the movment is finished
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_WaitUntilMoveFinished(
            )
        {
            return phdlStepper_DensoVS60_Cmd_WaitUntilMoveFinished(m_pDataParams);
        }

        /// <summary>
        /// Set communication port.
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_SetPort(
            string PortName
            )
        {
            // Allocate internal data parameters and pointer to them
            byte[] m_PortNameInt = new byte[PortName.Length + 1];
            for (int i = 0; i < PortName.Length; i++)
            {
                m_PortNameInt[i] = (byte)PortName[i];
            }
            m_PortNameInt[PortName.Length] = 0;
            GCHandle m_pPortNameInt = GCHandle.Alloc(m_PortNameInt, GCHandleType.Pinned);
            return phdlStepper_DensoVS60_Cmd_SetPort(m_pDataParams, m_PortNameInt);
        }

        /// <summary>
        /// Set a new ToolDef and the corresponding ToolDefNumber.
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_SetToolDef(
            double x,
            double y,
            double z,
            double rx,
            double ry,
            double rz,
            ushort wtoolNumber
            )
        {
            return phdlStepper_DensoVS60_Cmd_SetToolDef(m_pDataParams, x, y, z, rx, ry, rz, wtoolNumber);
        }

        /// <summary>
        /// Moves the robot down, activates the pump and picks up a card.
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_PickupCard()
        {
            return phdlStepper_DensoVS60_Cmd_PickupCard(m_pDataParams);
        }

        /// <summary>
        /// Moves the robot down, deactivates the pump and drops the card.
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_DropCard()
        {
            return phdlStepper_DensoVS60_Cmd_DropCard(m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        /// <summary>
        /// Default constructor
        /// </summary>
        public DensoVS60()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Default destructor
        /// </summary>
        ~DensoVS60()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region Wachler

    [ClassInterface(ClassInterfaceType.AutoDual)]
    public class Wachler : dlStepper.Generic
    {

        #region DATA_STRUCTURE
        private const int IDN_SIZE = 128;                       /**< Max Number of IDN response stored. */

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalRegDataParams;  /**< Pointer to the parameter structure of the balReg component. */
            public byte bInit;			      /**< Indicates the state of the connection */
            float fMaxPositionDegree;         /**< Indicates the maximal reachable position of the wachler. */
            float fMinPositionDegree;         /**< Indicates the minimum reachable position of the wachler. */
            UInt32 actPosition;               /**< Actual position */
            byte bNumSteps;                   /**< Number of steps supported by the wachler */
            float fModeUpDownLevel;           /**< Level used in Up/Down */
            float fModePosLevel;              /**< Level used in Pos */
            byte bPwmSpeed;                   /**< current Speed set with PWM */
            public fixed byte bIdn[IDN_SIZE]; /**< This value stores the idn string of the wachler */
        };
        #endregion

        #region DEFINES
        public enum Config : ushort
        {
            MODE            = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 0,  /**< Operation mode of the wachler. */
            REMOTE          = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 1,  /**< Enable/Disable remote mode. */
            NUMBER_STEPS    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 2,  /**< Number of steps supported by the robot. */
            UP_DOWN_LEVEL   = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 3,  /**< Limit that is used on up/down mode. (valueDegree = value/100) */
            POS_LEVEL       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 4,  /**< Limit that is used on pos mode. (valueDegree = value/100) */
            DELAY           = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 5,  /**< Delay used on AUTO and BSTOP Mode */
            POS_MIN         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 6,  /**< Get the minimum position in 100*degree. */
            POS_MAX         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 7,  /**< Get the maximum position in 100*degree. */
            COUNTER         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 8,  /**< If this config is set to 0 the counter is reseted. Any other value is not supported. */
            POSITION_Z      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 9,  /**< Set/Get the current Z point. */
            PWM             = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 10, /**< Enable or disable PWM Mode */
            PWM_SPEED       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 11, /**< Set PWM speed */
            PWM_CALIBRATION = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 12  /**< Set PWM calibration value for minimum speed. The value must be between 0 and 100 */
        }

        public enum PositionMode : uint
        {
            RUN = 0x00U,     /**< Moves all the time */
            STOP = 0x01U,    /**< Stop until a new position is set */
            CAL = 0x02U,     /**< Calibrate the min and max values */
            PWMCAL = 0x03U,  /**< Calibrate the min and max values */
            AUTO = 0x04U,    /**< Move up and down and stop at both positions */
            BSTOP = 0x05U,   /**< Move up and down and stop at bottom positions */
            POS = 0x06U      /**< position was set manual */
        }
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_Wachler_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams    				/**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlStepper_Wachler_Cmd_WaitUntilMoveFinished(
            IntPtr pDataParams                         /**< [In] Pointer to this layers parameter structure. */
            );
        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlStepper_Wachler_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }
        #endregion

        #region CUSTOM_FUNCTIONS

        /// <summary>
        /// Wait with the next command until the movment is finished
        /// </summary>
        /// <returns></returns>
        public Status_t Cmd_WaitUntilMoveFinished(
            )
        {
            return phdlStepper_Wachler_Cmd_WaitUntilMoveFinished(m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private GCHandle m_pDataParamsInt;

        public Wachler()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Wachler()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        public String Manufacturer
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        var idn = ByteArrayToString(pDataParams->bIdn, IDN_SIZE);
                        var items = idn.Split(',');
                        if (items.Length < 4)
                            return null;
                        return items[0];
                    }
                }
            }
        }

        public String Type
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        var idn = ByteArrayToString(pDataParams->bIdn, IDN_SIZE);
                        var items = idn.Split(',');
                        if (items.Length < 4)
                            return null;
                        return items[1];
                    }
                }
            }
        }

        public String Version
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        var idn = ByteArrayToString(pDataParams->bIdn, IDN_SIZE);
                        var items = idn.Split(',');
                        if (items.Length < 4)
                            return null;
                        return items[2];
                    }
                }
            }
        }

        public String Serial
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        var idn = ByteArrayToString(pDataParams->bIdn, IDN_SIZE);
                        var items = idn.Split(',');
                        if (items.Length < 4)
                            return null;
                        return items[3].Trim();
                    }
                }
            }
        }

        unsafe private string ByteArrayToString(byte* arr, int maxLength)
        {
            char[] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }
        #endregion
    }

    #endregion
}
#endif
