/*
 * Copyright 2013, 2017, 2019, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlThstrm
{
    #region BASE

    #region DEFINES
        public enum Error : byte
        {
            /// <summary> Thermostream Not Ready error </summary>
            NOT_READY = (CustomCodes.ERROR_BEGIN),
            /// <summary> Thermostream Query error </summary>
            QUERY_ERROR,
            /// <summary> Thermostream execution error </summary>
            EXECUTION_ERROR,
            /// <summary> Thermostream command error </summary>
            COMMAND_ERROR,
            /// <summary> Thermostream overheat error </summary>
            DDE_OVERHEAT,
            /// <summary> Thermostream air open loop error </summary>
            DDE_AIR_OPEN_LOOP,
            /// <summary> Thermostream setpoint out of range error </summary>
            DDE_SETP_OUT_OF_RANGE,
            /// <summary> Thermostream air flow too low error </summary>
            DDE_LOW_FLOW,
            /// <summary> Thermostream air pressure too low error </summary>
            DDE_LOW_AIR_PRESSURE,
            /// <summary> Thermostream no purge flow error </summary>
            DDE_NO_PURGE_FLOW,
            /// <summary> Thermostream open purge sensor error </summary>
            DDE_OPEN_PURGE_TEMP_SENSOR,
            /// <summary> Thermostream internal error </summary>
            DDE_INTERNAL_ERROR,
            /// <summary> Thermostream DUT open loop error </summary>
            DDE_DUT_OPEN_LOOP,
            /// <summary> Thermostream flow sensor hardware error </summary>
            DDE_FLOW_SENSOR_HARDWARE_ERROR,
            /// <summary> Thermostream purge heat failure </summary>
            DDE_PURGE_HEAT_FAILURE,
            /// <summary> Thermostream improper software version error </summary>
            DDE_IMPROPER_SOFTWARE_VERSION,
            /// <summary> Thermostream no DUT sensor selected error </summary>
            DDE_NO_DUT_SENSOR_SELECTED,
            /// <summary> Thermostream wrong temperature error </summary>
            SP_WRONG_TEMP,
            /// <summary> Thermostream head is still up error </summary>
            HEAD_IS_STILL_UP,
            /// <summary> Thermostream invalid setpoint error </summary>
            INVALID_SETPOINT,
            /// <summary> Thermostream invalid soaktime error </summary>
            SP_INVALID_SOAK_TIME,
            /// <summary> Thermostream invalid temperature window error </summary>
            INVALID_WINDOW,
            /// <summary> Thermostream invalid flow rate error </summary>
            INVALID_FLOWRATE,
            /// <summary> Thermostream invalid temperature format error </summary>
            TEMP_FORMAT,
        }

        public enum Config: ushort
        {
            SELECT_SETP = 0,                /**< Select the setpoint. */
            SETP_TEMP,                      /**< Select the setpoint temperature. */
            SETP_SOAKTIME,                  /**< Select the setpoint soaktime. */
            SETP_WINDOW,                    /**< Select the setpoint temperature window. */
            HEAD,                           /**< Select the head position. */
            FLOW,                           /**< Select the flow state. */
            FLOWRATE,                       /**< Select the flow rate. */
            SETP_TEMP_REACHED,              /**< Select the temperature reached flag (read only). */
            CURR_TEMP,                      /**< Select the current temperature (read only). */
            COMPRESSOR,                     /**< Select the compressor state (read only). */
            HEADLOCKED,                     /**< Select if head is locked */
            WAITTEMP_TIMEOUT_SEC,           /**< Select the timeout to wait for temp to be reached */
            CANCELLATION_TOKEN,             /**< configs the cancellation token */
            EDIT_MODE,                      /**< configs the edit mode for the setpoints */
            TEMPERATURE_PERFORMANCE_MODE    /**< configs the Temperature Performance Mode */
    }

    public enum SetPoint : byte
        {
            HOT = 0,                        /**< Setpoint Hot. */
            AMBIENT = 1,                    /**< Setpoint Ambient. */
            COLD = 2                        /**< Setpoint Cold. */
        }
        public enum Head : byte
        {
            UP = 0,                         /**< Head up*/
            DOWN = 1                        /**< Head down*/
        }
        #endregion

    public abstract class Generic
    {

        #region DLLIMPORTS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_InitThstrm(
            IntPtr pDataParams          /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_SetConfig(
            IntPtr m_pDataParams,              /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,
            ushort wValue
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_GetConfig(
            IntPtr m_pDataParams,              /**< [In] Pointer to this layers parameter structure. */
            ushort wIdentifier,
            ref ushort wValue
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_PowerOff(
            IntPtr pDataParams          /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_WaitTemp(
            IntPtr m_pDataParams               /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_SoftReset(
            IntPtr pDataParams          /**< [In] Pointer to this layers parameter structure. */
            );


        #endregion

        #region DLL_WRAPPED_FUNCTIONS
        public Status_t InitThstrm(
)
        {
            return phdlThstrm_InitThstrm(m_pDataParams);
        }

        public Status_t SetConfig(
                            ushort wIdentifier,
                            ushort wValue
                            )
        {
            return phdlThstrm_SetConfig(m_pDataParams, wIdentifier, wValue);
        }

        public Status_t GetConfig(
                            ushort wIdentifier,
                            out ushort wValue
                            )
        {
            Status_t status;
            ushort wTmp = 0;
            status = phdlThstrm_GetConfig(m_pDataParams, wIdentifier, ref wTmp);
            wValue = wTmp;
            return status;
        }

        public Status_t PowerOff(
        )
        {
            return phdlThstrm_PowerOff(m_pDataParams);

        }
        public Status_t WaitTemp(
        )
        {
            return phdlThstrm_WaitTemp(m_pDataParams);
        }
        public Status_t SoftReset(
)
        {
            return phdlThstrm_SoftReset(m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region TP04030

    public class TP04030 : dlThstrm.Generic
    {

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;       /**< Pointer to the parameter structure of the balReg component. */
            UInt16 wWaitTempTimeoutSec;     /**< Timeout used while waiting for the Temperature to be reached. */
            byte bCancellationToken;        /**< if set to one the waiting for temp reached should be canceled. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_TP04030_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                    /**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlThstrm_TP04030_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public TP04030()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~TP04030()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region ATS545

    public class ATS545 : dlThstrm.Generic
    {

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pBalRegDataParams;       /**< Pointer to the parameter structure of the balReg component. */
            UInt16 wWaitTempTimeoutSec;     /**< Timeout used while waiting for the Temperature to be reached. */
            byte bCancellationToken;        /**< if set to one the waiting for temp reached should be canceled. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlThstrm_ATS545_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalRegDataParams                    /**< [In] Pointer to the parameter structure of the balReg layer. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            Bal.Generic pBal
            )
        {
            return phdlThstrm_ATS545_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pBal.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public ATS545()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~ATS545()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
#endif
