/*
 * Copyright 2013, 2015, 2017 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    #region Enumeration
    #region Configuration
    public enum Config : int
    {
        /// <summary> Enable or Disable Parity. </summary>
        PARITY = 0x0000,

        /// <summary> Enable or Disable TxCrc. </summary>
        TXCRC,

        /// <summary> Enable or Disable RxCrc. </summary>
        RXCRC,

        /// <summary> Set number of valid bits of last Tx-Byte. </summary>
        TXLASTBITS,

        /// <summary> Get number of valid bits of last Rx-Byte. </summary>
        RXLASTBITS,

        /// <summary> Set Rx-Aligned Bits. </summary>
        RXALIGN,

        /// <summary> Configure Receiver Deaf-Time in ETUs. </summary>
        RXDEAFBITS,

        /// <summary> Set TxWait (= time between last RxIrq and Tx of succeeding frame) in microseconds. </summary>
        TXWAIT_US,

        /// <summary> Enable or Disable clearing of bits after coll. </summary>
        CLEARBITSAFTERCOLL,

        /// <summary> Configure Data-Rate for Transmission. </summary>
        TXDATARATE,

        /// <summary>  </summary>
        RXDATARATE,

        /// <summary>  </summary>
        MODINDEX,

        /// <summary>  </summary>
        ASK100,

        /// <summary>  </summary>
        TIMEOUT_VALUE_US,

        /// <summary>  </summary>
        TIMEOUT_VALUE_MS,

        /// <summary>  </summary>
        SUBCARRIER,

        /// <summary>  </summary>
        TIMING_MODE,

        /// <summary>  </summary>
        TIMING_US,

        /// <summary>  </summary>
        TIMING_MS,

        /// <summary>  </summary>
        FIELD_OFF_TIME,

        /// <summary>  </summary>
        FIELD_RECOVERY_TIME,

        /// <summary>  </summary>
        SYMBOL_START,

        /// <summary>  </summary>
        SYMBOL_END,

        /// <summary>  </summary>
        DISABLE_MF_CRYPTO1 = 0x002E,

        /// <summary>  </summary>
        ADDITIONAL_INFO = 0x002F,

        /// <summary>  </summary>
        RXBUFFER_STARTPOS = 0x0030,

        /// <summary>  </summary>
        RXBUFFER_BUFSIZE,

        /// <summary>  </summary>
        TXBUFFER_BUFSIZE,

        /// <summary>  </summary>
        TXBUFFER_LENGTH,

        /// <summary>  </summary>
        TXBUFFER,

        /// <summary>  </summary>
        MAX_PRECACHED_BYTES,

        /// <summary>  </summary>
        BAL_CONNECTION = 0x0040,

        /// <summary>  </summary>
        SERIAL_BITRATE,

        /// <summary>  </summary>
        RFRESET_ON_TIMEOUT = 0x0050,

        /// <summary>  </summary>
        SETMINFDT = 0x0051,

        /// <summary>  </summary>
        FRAME_SYNC = 0x0017,

        /// <summary>  </summary>
        TXDATANUM = 0x0018,

        /// <summary>  </summary>
        CARD_TYPE = 0x0019,

        /// <summary>  </summary>
        READER_IC = 0x0020,

        /// <summary>
        /// Configure communication type.
        ///     - <see cref="CommunicationType.TDA"/>
        ///     - <see cref="CommunicationType.I2C"/>
        /// </summary>
        SAM_COMMUNICATION_TYPE = 0x0060,

        /// <summary> Config the I2C Slave address to be used. </summary>
        I2C_SLAVE_ADDRESS = 0x0061,

        /// <summary>
        /// Config the Data-Rate for I2C communication. Supported values are
        ///     - <see cref="I2C_BitRate.STANDARD_MODE"/>
        ///     - <see cref="I2C_BitRate.FAST_MODE"/>
        ///     - <see cref="I2C_BitRate.FAST_PLUS_MODE"/>
        ///     - <see cref="I2C_BitRate.HIGH_SPEED_MODE"/>
        /// </summary>
        I2C_BITRATE = 0x0062,

        /// <summary> Configure I2C polling Timeout in terms of millisecond. </summary>
        I2C_TIMEOUT_MS = 0x0063,

        /// <summary>
        /// Configure the Timer deviation percentage. This configuration should
        /// be used while performing ProximityCheck command.
        /// </summary>
        TIMER_DEVIATION_PROXIMITY_CHECK = 0x8013,

        /// <summary>
        /// Configure the Handling of ISO15693 In-Reply process dual frames reception
        /// for an authentication command TAM1.2 / TAM1.3
        /// </summary>
        IN_REPLY_PROCESS_CMD = 0x8014
    }
    #endregion

    #region Error
    /// <summary> Custom Error Codes returned by the library </summary>
    public enum Error : byte
    {
        /// <summary> Non standard error. Read wAddionalInfo to get error code. </summary>
        ISO3_ERR_ISO3 = ( CustomCodes.ERROR_BEGIN + 48 ),

        /// <summary> IIR filter overflow error. </summary>
        ISO3_ERR_IIR_FILTER_OVERFLOW = ( CustomCodes.ERROR_BEGIN + 49 ),

        /// <summary> Incompatible server version for current ISO3 driver implementation </summary>
        ISO3_ERR_INCOMPATIBLE_SERVER_VERSION = ( CustomCodes.ERROR_BEGIN + 50 ),

        /// <summary> Incompatible FPGA version for current ISO3 driver implementation </summary>
        ISO3_ERR_INCOMPATIBLE_FPGA_VERSION = ( CustomCodes.ERROR_BEGIN + 51 ),

        /// <summary> phCDllReader.dll threw an non standard error read wAddionalInfo to get error code. </summary>
        RDFPGAV6_ERR_DLL = ( CustomCodes.ERROR_BEGIN + 52 ),

        /// <summary> Non standard error. Read wAddionalInfo to get error code. </summary>
        MP300_ERR_MP300 = ( CustomCodes.ERROR_BEGIN + 53 ),

        /// <summary> Adjust rx Channel failed. </summary>
        MP300_ERR_ADJUST_RX_CHANNEL_2 = ( CustomCodes.ERROR_BEGIN + 54 ),

        /// <summary> FDT Measurement methode 2 is not supported with the current response. </summary>
        MP300_ERR_FDT_METHODE2_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 55 )
    }
    #endregion

    #region CardType
    /// <summary> Card Configs </summary>
    public enum CardType : byte
    {
        /// <summary> Re-Apply settings for current Communication Mode. </summary>
        CURRENT = 0x00,

        /// <summary> ISO/IEC 14443A Mode. </summary>
        ISO14443A,

        /// <summary> ISO/IEC 14443B Mode. </summary>
        ISO14443B,

        /// <summary> Felica (JIS X 6319) Mode. </summary>
        FELICA,

        /// <summary> ISO/IEC 15693 Mode. </summary>
        ISO15693,

        /// <summary> NXP I-Code EPC/UID Mode. </summary>
        ICODEEPCUID,

        /// <summary> ISO/IEC 18000-3 Mode3. </summary>
        I18000P3M3,

        /// <summary> ISO/IEC ISO18092 (NFC) Passive Initiator Mode. </summary>
        I18092MPI,

        /// <summary> ISO/IEC ISO18092 (NFC) Passive Target Mode. </summary>
        I18092MPT
    }
    #endregion

    #region RfDataRate
    public enum RfDataRate : int
    {
        /// <summary> Data rate setting for 106 kBit/s. </summary>
        DATARATE_106 = 0x0000,

        /// <summary> Data rate setting for 212 kBit/s. </summary>
        DATARATE_212,

        /// <summary> Data rate setting for 424 kBit/s. </summary>
        DATARATE_424,

        /// <summary> Data rate setting for 848 kBit/s. </summary>
        DATARATE_848,

        /// <summary> Data rate setting for 1695 kBit/s. </summary>
        DATARATE_1695,

        /// <summary> Data rate setting for 3390 kBit/s. </summary>
        DATARATE_3390,

        /// <summary> Data rate setting for 6780 kBit/s. </summary>
        DATARATE_6780,

        /// <summary> Data rate setting for ISO15693 '1 out of 256' mode. 1.66kb/s </summary>
        TX_1_OUT_OF_256 = 0x000A,

        /// <summary> Data rate setting for ISO15693 '1 out of 4' mode. 26 kb/s </summary>
        TX_1_OUT_OF_4,

        /// <summary> Data rate setting for ISO15693 low datarate. </summary>
        RX_DATARATE_LOW,

        /// <summary> Data rate setting for ISO15693 high datarate. </summary>
        RX_DATARATE_HIGH,

        /// <summary> Data rate setting for ISO15693 fast mode combined with #PHHAL_HW_RF_RX_DATARATE_LOW (FAST INVENTORY (PAGE) READ). </summary>
        RX_DATARATE_FAST_LOW,

        /// <summary> Data rate setting for ISO15693 fast mode combined with #PHHAL_HW_RF_RX_DATARATE_HIGH (FAST INVENTORY (PAGE) READ). </summary>
        RX_DATARATE_FAST_HIGH,

        /// <summary> Data rate setting for EPC/UID. </summary>
        DATARATE_EPCUID,

        /// <summary> Tx Data rate setting for ISO18000-3 Mode3. </summary>
        TX_DATARATE_I18000P3M3,

        /// <summary> Tx data rate setting for ISO18000-3 mode3. TARI Value = 18.88 us </summary>
        TX_DATARATE_I18000P3M3_TARI1866,

        /// <summary> ICode ISO18000-3 Mode3 424 kBit/s (M=2). </summary>
        RX_DATARATE_I18000P3M3_FL_423_MAN2,

        /// <summary> ICode ISO18000-3 Mode3 847 kBit/s (M=2). </summary>
        RX_DATARATE_I18000P3M3_FL_847_MAN2,

        /// <summary> ICode ISO18000-3 Mode3 424 kBit/s (M=4). </summary>
        RX_DATARATE_I18000P3M3_FL_423_MAN4,

        /// <summary> ICode ISO18000-3 Mode3 847 kBit/s (M=4). </summary>
        RX_DATARATE_I18000P3M3_FL_847_MAN4,

        /// <summary> Data rate setting for ISO15693 '1 out of 4' mode. 53 kb/s </summary>
        TX_1_OUT_OF_4_53 = 0x0017,

        /// <summary> Data rate setting for ISO15693 '1 out of 4' mode. 106 kb/s </summary>
        TX_1_OUT_OF_4_106,

        /// <summary> Data rate setting for ISO15693 '1 out of 4' mode. 212 kb/s </summary>
        TX_1_OUT_OF_4_212
    }
    #endregion

    #region RFSub Carrier
    public enum RfSubcarrier : int
    {
        /// <summary> Sub carrier setting for ISO15693 single subcarrier. </summary>
        SINGLE = 0x0000,

        /// <summary> Sub carrier setting for ISO15693 dual subcarrier. </summary>
        DUAL,

        /// <summary> Sub carrier setting for EPC V2 quad subcarrier. </summary>
        QUAD
    }
    #endregion

    #region TimeUnit
    /// <summary> Time Units </summary>
    public enum TimeUnit : int
    {
        /// <summary> Time in microseconds (us). </summary>
        MICROSECONDS = 0x0000,

        /// <summary> Time in milliseconds (ms). </summary>
        MILLISECONDS
    }
    #endregion

    #region Timing Mode
    /// <summary> Timing Mode </summary>
    public enum TimingMode : int
    {
        /// <summary> Time Measurement disabled. </summary>
        OFF = 0x0000,

        /// <summary> Measures time from Tx-End until Rx-Begin. </summary>
        FDT = 0x0001,

        /// <summary>
        /// Measures time from Tx-End until Rx-Begin and
        /// Automatically resets the current timing value prior to any communication
        /// </summary>
        FDT_AUTOCLEAR = 0x0101,

        /// <summary> Measures the duration of the complete communication. </summary>
        COMM = 0x0002,

        /// <summary>
        /// Measures the duration of the complete communication
        /// Automatically resets the current timing value prior to any communication.
        /// </summary>
        COMM_AUTOCLEAR = 0x0102
    }
    #endregion

    #region Frame Symbols
    /// <summary>
    /// Frame Symbols (use #PH_OFF to disable)
    /// </summary>
    public enum FrameSymbols : int
    {
        /// <summary> ICode EPC/UID Short Start Of Frame </summary>
        ICODEEPCUID_SSOF = 0x0001,

        /// <summary> ICode EPC/UID Long Start Of Frame </summary>
        ICODEEPCUID_LSOF = 0x0002,

        /// <summary> ICode EPC/UID Common End Of Frame </summary>
        ICODEEPCUID_CEOF = 0x0003,

        /// <summary> ICode EPC/UID Close Slot </summary>
        ICODEEPCUID_CS = 0x0004,

        /// <summary> ISO15693 SOF (aumatically choosen between 1/256 and 1/4). </summary>
        I15693_SOF = 0x0005,

        /// <summary> ISO18000-3 mode3 preamble. </summary>
        I18000P3M3_PREAMBLE = 0x0006,

        /// <summary> ISO18000-3 mode3 frame-sync. </summary>
        I18000P3M3_FSYNC = 0x0007,
    }
    #endregion

    #region Key Type
    /// <summary> MIFARE Classic Authenticate Options </summary>
    public enum KeyType : byte
    {
        /// <summary> MIFARE Classic Key Type A. </summary>
        KEYA = 0x0A,

        /// <summary> MIFARE Classic Key Type B. </summary>
        KEYB = 0x0B
    }
    #endregion

    #region Bal Connection Type
    /// <summary> BAL Connection types </summary>
    public enum BalConnectionType : int
    {
        /// <summary> ReaderIC is connected via RS232. </summary>
        RS232 = 0x0000,

        /// <summary> ReaderIC is connected via SPI. </summary>
        SPI,

        /// <summary> ReaderIC is connected via I2C. </summary>
        I2C
    }
    #endregion

    #region Rs232 BitRate
    /// <summary> RS232 Bitrate Values </summary>
    public enum Rs232BitRate : int
    {
        /// <summary> 7.200 kBit/s. </summary>
        VALUE_7200 = 0x0000,

        /// <summary> 9.600 kBit/s. </summary>
        VALUE_9600,

        /// <summary> 14.400 kBit/s. </summary>
        VALUE_14400,

        /// <summary> 19.200 kBit/s. </summary>
        VALUE_19200,

        /// <summary> 38.400 kBit/s. </summary>
        VALUE_38400,

        /// <summary> 57.500 kBit/s. </summary>
        VALUE_57600,

        /// <summary> 115.200 kBit/s. </summary>
        VALUE_115200,

        /// <summary> 128.000 kBit/s. </summary>
        VALUE_128000,

        /// <summary> 230.400 kBit/s.</summary>
        VALUE_230400,

        /// <summary> 460.800 kBit/s. </summary>
        VALUE_460800,

        /// <summary> 921.600 kBit/s. </summary>
        VALUE_921600,

        /// <summary> 1.228.800 kBit/s. </summary>
        VALUE_1228800
    }
    #endregion

    #region FDT Values
    /// <summary> Min FDT values </summary>
    public enum FDTValue : byte
    {
        /// <summary> (n * 128 + 84) / 13.56; n = 9. </summary>
        MINFDT_106_US = 91 + 10,

        /// <summary> (n * 128 + 148) / 13.56; n = 8. </summary>
        MINFDT_212_US = 87 + 40,

        /// <summary> (n * 128 + 116) / 13.56; n = 8. </summary>
        MINFDT_424_US = 84 + 35,

        /// <summary> (n * 128 + 100) / 13.56; n = 8. </summary>
        MINFDT_848_US = 83 + 45
    }
    #endregion

    #region TSProcessing
    /// <summary> Parameter values for ISO/IEC 18000p3m3 HAL APIs </summary>
    public enum TSProcessing : byte
    {
        /// <summary> Gets responses from more than one slots limited by the RX buffer size and number of slots </summary>
        GET_MAX_RESPS,

        /// <summary> Gets response for only one time slot </summary>
        ONE_TS_ONLY,

        /// <summary> Gets response for only one time slot. Also sends a ReqRN to get the tag handle for this slot </summary>
        GET_TAG_HANDLE
    }
    #endregion

    #region Communication Type
    /// <summary>
    /// Type of communication being enabled for SAM.
    /// </summary>
    public enum CommunicationType : ushort
    {
        /// <summary>
        /// SAM Communication type as TDA. SAM APDU's will be exchanged via TDA interface.
        /// </summary>
        TDA = 0x01,

        /// <summary>
        /// SAM Communication type as I2C. SAM APDU's will be exchanged via I2C interface.
        /// </summary>
        I2C = 0x02
    }
    #endregion

    #region I2C BitRate


    /// <summary>
    /// Type of I2C Bit Rates to be used.
    /// </summary>
    public enum I2C_BitRate : ushort
    {
        /// <summary>
        /// I2C bit rate configuration value for 100 KBits/s.
        /// </summary>
        STANDARD_MODE = 0x00,

        /// <summary>
        /// I2C bit rate configuration value for 400 KBits/s.
        /// </summary>
        FAST_MODE = 0x01,

        /// <summary>
        /// I2C bit rate configuration value for 1 MBits/s.
        /// </summary>
        FAST_PLUS_MODE = 0x02,

        /// <summary>
        /// I2C bit rate configuration value for 3.4 MBits/s.
        /// </summary>
        HIGH_SPEED_MODE = 0x03
    }
    #endregion
    #endregion

    #region Interface
    /// <summary> Declaration of interface methods a HAL object needs to implement. </summary>
    public interface IHal
    {
        /// <summary>
        /// Perform Data Exchange with Picc.
        ///
        /// \b wOption can be one of:
        /// \li <see cref="ExchangeOptions.DEFAULT"/>
        ///
        /// \b wOption can be combined with:
        /// \li <see cref="ExchangeOptions.BUFFERED_BIT"/>
        /// \li <see cref="ExchangeOptions.LEAVE_BUFFER_BIT"/>
        /// </summary>
        ///
        /// <param name="wOption">Option parameter.</param>
        /// <param name="pTxBuffer">Data to transmit.</param>
        /// <param name="pRxBuffer">Pointer to received data.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t Exchange ( int wOption, byte[] pTxBuffer, out byte[] pRxBuffer );

        /// <summary>
        /// Write value to register address.
        /// </summary>
        /// <param name="bAddress">Register Address</param>
        /// <param name="bValue">Register Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t WriteRegister ( byte bAddress, byte bValue );

        /// <summary>
        /// Read value from register address.
        /// </summary>
        ///
        /// <param name="bAddress">Register Address</param>
        /// <param name="pValue">Register Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t ReadRegister ( byte bAddress, out byte pValue );

        /// <summary>
        /// Apply protocol settings.
        /// </summary>
        ///
        /// <param name="bCardType">Type of card for which the hal should be configured for</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t ApplyProtocolSettings ( CardType bCardType );

        /// <summary>
        /// Perform MIFARE Classic Authentication using a key number.
        /// </summary>
        ///
        /// <param name="bBlockNo">Blocknumber on card to authenticate to</param>
        /// <param name="bKeyType">Either <see cref="KeyType.KEYA"/> or <see cref="KeyType.KEYB"/></param>
        /// <param name="wKeyNo">Key number to be used in authentication</param>
        /// <param name="wKeyVersion">Key version to be used in authentication</param>
        /// <param name="pUid">Serial number of current cascade level; uint8_t[4].</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t MfcAuthenticateKeyNo ( byte bBlockNo, KeyType bKeyType, ushort wKeyNo, ushort wKeyVersion,
            byte[] pUid );

        /// <summary>
        /// Perform MIFARE Classic Authentication using a key.
        /// </summary>
        ///
        /// <param name="bBlockNo">Blocknumber on card to authenticate to</param>
        /// <param name="bKeyType">Either <see cref="KeyType.KEYA"/> or <see cref="KeyType.KEYB"/></param>
        /// <param name="pKey">Key to be used in authentication</param>
        /// <param name="pUid">Serial number of current cascade level; uint8_t[4].</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t MfcAuthenticate ( byte bBlockNo, KeyType bKeyType, byte[] pKey, byte[] pUid );

        /// <summary>
        /// Set configuration parameter.
        ///
        /// Sam <em>Special Behaviour:</em>
        /// In NonX-Mode every setting ( except PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and CONFIG_HOSTMODE )
        /// is passed on to the linked Reader-HAL.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="wValue">Configuration Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t SetConfig ( Config wConfig, int wValue );

        /// <summary>
        /// Get configuration parameter.
        ///
        /// Sam <em>Special Behaviour:</em>
        /// In NonX-Mode every setting ( except PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and CONFIG_HOSTMODE )
        /// is passed on to the linked Reader-HAL.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="wValue">Configuration Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t GetConfig ( Config wConfig, out int pValue );

        /// <summary>
        /// Switch on the RF-field.
        /// </summary>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t FieldOn ();

        /// <summary>
        /// Switch off the RF-field.
        /// </summary>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t FieldOff ();

        /// <summary>
        /// Perform RF-Reset.
        /// </summary>
        /// <param name="wFieldOffTime"></param>
        /// <param name="wFieldRecoveryTime"></param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t FieldReset ();

        /// <summary>
        /// Blocks until the time given by 	<see cref="wTimeout"/> elapsed.
        /// </summary>
        ///
        /// <param name="bUnit">Unit of given timeout value (either <see cref="TimeUnit.MICROSECONDS"/> or
        /// <see cref="TimeUnit.MILLISECONDS"/>).
        /// </param>
        /// <param name="wTimeout">Timeout value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t Wait ( byte bUnit, int wTimeout );

        /// <summary>
        /// Perform 18000p3m3 Inventory based on BeginRound and Select Command
        ///
        /// Note: This function does
        ///         ISO18000p3m3 BeginRound
        ///         Sends an Ack
        ///         Optionally sends a ReqRN
        ///         Stores the information related to tag reply in the HAL RX Buffer
        ///         Continues with NextSlot or else pauses and returns based on the option specified
        ///         Can return multi card information based on the configuration option and the size of
        ///         HAL RX Buffer.
        ///
        /// The bTSprocessing ( Time slot processing behavior ) can take the following values
        ///     \li <see cref="TSProcessing.GET_MAX_RESPS"/> - Gets responses from more than one
        ///     slot limited by the Rx Buffer size and number of slots specified in begin round.
        ///
        ///     \li <see cref="TSProcessing.ONE_TS_ONLY"/> - Gets response for only one time slot.
        ///     Here the number of time slots should be zero in the begin round command.
        ///
        ///     \li <see cref="TSProcessing.GET_TAG_HANDLE"/> - Gets response for only one time slot.
        ///     Also send a ReqRN to get the tag handle for this slot. Here the number of time slots
        ///     should be zero in the begin round command.
        ///
        /// The response is a Pointer to a buffer containing the result of the inventory.
        /// The buffer may contain the result of one or more timeslots.Each timeslot result has the
        /// following structure:
        ///     1 Byte time slot status
        ///         0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field' are present.
        ///         1 ... Tag response available.In addition to three field above, 'Tag handle field' also present.
        ///         2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field', 'Tag Handle' are not present.
        ///         3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag reply field not present.
        ///     1 Byte 'Tag reply' Length (1-66)
        ///     1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
        ///     0-66 Bytes Tag reply
        ///     0 or 2 Bytes Handle of the tag, in case the field
        ///
        /// A typical sequence in which this API is called is given below
        ///     \li BAL initialization (where applicable) - Example - <see cref="Bal.SerialWin.Init"/>
        ///     \li HAL initialization - Example <see cref="DUT.Init()"/>
        ///     \li Load ISO 1800p3m3 protocol - <see cref="Generic.ApplyProtocolSettings"/>with <see cref="CardType.I18000P3M3"/>
        ///     \li Turn ON the field - <see cref="Generic.FieldOn"/>
        ///     \li Wait for initial guard time - <see cref="Generic.Wait"/> with wait tie as 5100us
        ///     \li Start inventory - <see cref="Generic.I18000p3m3Inventory"/>
        ///     \li Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate with the tag.
        ///     \li Turn OFF the field - <see cref="Generic.FieldOff"/>
        ///     \li Termination of the application flow.
        /// </summary>
        ///
        /// <param name="pSelCmd">ISO 18000p3m3 Select command frame.</param>
        /// <param name="bSelCmdLen">Select command length in bytes- 1 to 39 bytes.</param>
        /// <param name="bNumValidBitsinLastByte">Number of valid bits in last byte of pSelCmd.</param>
        /// <param name="pBeginRndCmd">ISO 18000p3m3 BeginRound command frame. This is 17bits i.e.,
        /// 3 bytes are expected. CRC5 should not be provided.
        /// </param>
        /// <param name="bTSprocessing">TimeSlot processing behavior.</param>
        /// <param name="ppRxBuffer">Pointer to HAL Rx Buffer containing response from single/multiple tags.</param>
        /// <param name="wRxBufferLen">Length of response in hal Rx Buffer.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t I18000p3m3Inventory ( byte[] pSelCmd, byte bSelCmdLen, byte bNumValidBitsinLastByte, byte[] pBeginRndCmd,
            byte bTSprocessing, out byte[] ppRxBuffer, out ushort wRxBufferLen );

        /// <summary>
        /// ISO 18000p3m3 resume inventory command.
        ///
        /// Note: This function is to be called after <see cref="Generic.I18000p3m3Inventory"/>
        /// This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
        /// remaining slots.
        ///
        /// Resume inventory also returns the response similar to phhalHw_I18000p3m3Inventory.
        ///
        /// A typical sequence in which this API is called is given below
        ///     \li BAL initialization (where applicable) - Example - <see cref="Bal.SerialWin.Init"/>
        ///     \li HAL initialization - Example <see cref="DUT.Init()"/>
        ///     \li Load ISO 1800p3m3 protocol - <see cref="Generic.ApplyProtocolSettings"/>with <see cref="CardType.I18000P3M3"/>
        ///     \li Turn ON the field - <see cref="Generic.FieldOn"/>
        ///     \li Wait for initial guard time - <see cref="Generic.Wait"/> with wait tie as 5100us
        ///     \li Start inventory - <see cref="Generic.I18000p3m3Inventory"/>
        ///     \li Store the responses received for the tags.
        ///     \li If inventory has to be continued then issue the ResumeInventory command - <see cref="Generic.I18000p3m3ResumeInventory"/>
        ///     \li Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate with the tag.
        ///     \li Turn OFF the field - <see cref="Generic.FieldOff"/>
        ///     \li Termination of the application flow.
        /// </summary>
        ///
        /// <param name="ppRxBuffer">Pointer to HAL Rx Buffer containing response from single/multiple tags.</param>
        /// <param name="wRxBufferLen">Length of response in hal Rx Buffer.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t I18000p3m3ResumeInventory ( out byte[] ppRxBuffer, out ushort wRxBufferLen );

        /// <summary>
        /// Reads the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Reading.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchagned to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified
        ///                           information to this paramter will be exchagned to Reader.
        ///                           These values are specific to Pegoda - 3 Reader.
        ///                             - 0x0010: Read information from User area
        ///                             - 0x0011: Read information from Secure Lib area
        ///                             - 0x0012: Read information from IC Configuration area
        ///                             - 0x0020: Read information from RF Configuration area
        ///                         - Values other that above specified ones.
        /// </param>
        /// <param name="dwAddr_ProtIndex">The EEPROM address or Protocol Index from where the
        ///                                information should be read.One of the following,
        ///                                 - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                                   EEPROM address.This is supported by Pegoda - 3 Reader
        ///                                   only.
        ///                                 - If \b wOption = 0x0020, then 1 byte Protocol
        ///                                   index of the RF Configuration.This is supported
        ///                                   by Pegoda - 3 Reader only.
        ///                                 - If \b wOption = 0x0000, then 4 byte EEPROM address.
        ///                                   Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="dwNoOfBytesToRead">Number of bytes to be read from EEPROM starting from address specified.
        ///                                 If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
        ///                                 provided.
        /// </param>
        /// <param name="aResponse">EEPROM information from the mentioned start address (\b dwStartAddr)
        ///                         until the requested length.Make sure the buffer is allocated for the
        ///                         requested number of bytes (\b dwLength ).
        /// </param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t ReadEEPROM ( ushort wOption, uint dwAddr_ProtIndex, uint dwNoOfBytesToRead, out byte[] aResponse );

        /// <summary>
        /// Writes the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Writing.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchagned to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified
        ///                           information will be exchagned to Reader. These values are
        ///                           specific to Pegoda - 3 Reader.
        ///                             - 0x0010: Read information from User area
        ///                             - 0x0011: Read information from Secure Lib area
        ///                             - 0x0012: Read information from IC Configuration area
        ///                             - 0x0020: Read information from RF Configuration area
        /// </param>
        /// <param name="dwStartAddr">The EEPROM address information to be written.
        ///                           One of the following,
        ///                             - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                               EEPROM address.This is supported by Pegoda - 3 Reader
        ///                               only.
        ///                             - If \b wOption = 0x0020, zero should be provided.
        ///                             - If \b wOption = 0x0000, then 4 byte EEPROM address.
        ///                               Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="aData">Information to be written to EERPOM.
        ///                     One of the following,
        ///                         - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
        ///                           EEPROM value should be provided.
        ///                         - If \b wOption = 0x0020, then 1 byte Protocol index followed
        ///                           1 byte Register address followed by 4 byte Register Value.
        ///                           This is supported by Pegoda - 3 Reader only.
        ///                           Protocol Index1 ( 1 Byte) ||
        ///                           Reg Addr1 ( 1 Byte) || Reg Value1 ( 4 Bytes) ||
        ///                           Reg Addr2 ( 1 Byte) || Reg Value2 ( 4 Bytes) ||
        ///                           ...
        ///                           Reg AddrN ( 1 Byte) || Reg ValueN ( 4 Bytes)
        /// </param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        Status_t WriteEEPROM ( ushort wOption, uint dwStartAddr, byte[] aData );
    }
    #endregion

    #region Generic
    /// <summary>
    /// Abstract Base-Class for every HAL object.
    /// (Uses only the Generic HAL C-Object)
    /// </summary>
    public abstract class Generic : IHal
    {
        #region Variables
        /// <summary> Default Field Off time</summary>
        public const int FIELD_OFF_DEFAULT = 0x05;

        /// <summary> Default Field recovery time</summary>
        public const int FIELD_RECOVERY_DEFAULT = 0x05;

        /// <summary> MIFARE Classic Key Length</summary>
        public const byte MFC_KEY_LENGTH = 0x06;
        #endregion

        #region Dll Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Exchange ( IntPtr pDataParams, ushort wOption, byte[] pTxBuffer, ushort wTxLength,
            ref IntPtr ppRxBuffer, ref ushort pRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_WriteRegister ( IntPtr pDataParams, byte bAddress, byte bValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_ReadRegister ( IntPtr pDataParams, byte bAddress, ref byte pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_ApplyProtocolSettings ( IntPtr pDataParams, byte bCardType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_MfcAuthenticateKeyNo ( IntPtr pDataParams, byte bBlockNo, byte bKeyType,
            ushort wKeyNo, ushort wKeyVersion, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_MfcAuthenticate ( IntPtr pDataParams, byte bBlockNo, byte bKeyType,
            byte[] pKey, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_FieldOn ( IntPtr m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_FieldOff ( IntPtr m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_FieldReset ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Wait ( IntPtr pDataParams, byte bUnit, ushort wTimeout );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_I18000p3m3Inventory ( IntPtr pDataParams, byte[] pSelCmd, byte bSelCmdLen,
            byte bNumValidBitsinLastByte, byte[] pBeginRndCmd, byte bTSprocessing, ref IntPtr ppRxBuffer,
            ref ushort wRxBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_I18000p3m3ResumeInventory ( IntPtr pDataParams, ref IntPtr ppRxBuffer,
            ref ushort wRxBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_ExecuteCmd ( IntPtr pDataParams, byte bCmd, ushort wOption, byte bIrq0WaitFor,
            byte bIrq1WaitFor, byte[] pTxBuffer, ushort wTxLength, ushort wRxBufferSize, byte[] pRxBuffer,
            ref ushort pRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_ReadEEPROM ( IntPtr pDataParams, ushort wOption, uint dwAddr_ProtIndex,
            uint dwNoOfBytesToRead, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_WriteEEPROM ( IntPtr pDataParams, ushort wOption, uint dwStartAddr,
            byte[] pData, ushort wDataLen );
        #endregion

        #region Wrapper Functions
        /// <summary>
        /// Perform Data Exchange with Picc.
        ///
        /// \b wOption can be one of:
        /// \li <see cref="ExchangeOptions.DEFAULT"/>
        ///
        /// \b wOption can be combined with:
        /// \li <see cref="ExchangeOptions.BUFFERED_BIT"/>
        /// \li <see cref="ExchangeOptions.LEAVE_BUFFER_BIT"/>
        /// </summary>
        ///
        /// <param name="wOption">Option parameter.</param>
        /// <param name="pTxBuffer">Data to transmit.</param>
        /// <param name="pRxBuffer">Pointer to received data.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Exchange ( int wOption, byte[] pTxBuffer, out byte[] pRxBuffer )
        {
            Status_t oStatus;

            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            oStatus = phhalHw_Exchange ( m_pDataParams, ( ushort ) wOption, pTxBuffer,
                ( ushort ) pTxBuffer.Length, ref pRxBuff, ref wRxLen );

            if ( ( pRxBuff != IntPtr.Zero ) && ( wRxLen > 0 ) )
            {
                pRxBuffer = new byte[wRxLen];
                Marshal.Copy ( pRxBuff, pRxBuffer, 0, wRxLen );
            }
            else
            {
                pRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Write value to register address.
        /// </summary>
        /// <param name="bAddress">Register Address</param>
        /// <param name="bValue">Register Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteRegister ( byte bAddress, byte bValue )
        {
            return phhalHw_WriteRegister ( m_pDataParams, bAddress, bValue );
        }

        /// <summary>
        /// Read value from register address.
        /// </summary>
        ///
        /// <param name="bAddress">Register Address</param>
        /// <param name="pValue">Register Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadRegister ( byte bAddress, out byte pValue )
        {
            pValue = 0;
            return phhalHw_ReadRegister ( m_pDataParams, bAddress, ref pValue );
        }

        /// <summary>
        /// Apply protocol settings.
        /// </summary>
        ///
        /// <param name="bCardType">Type of card for which the hal should be configured for</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ApplyProtocolSettings ( CardType bCardType )
        {
            return phhalHw_ApplyProtocolSettings ( m_pDataParams, ( byte ) bCardType );
        }

        /// <summary>
        /// Perform MIFARE Classic Authentication using a key number.
        /// </summary>
        ///
        /// <param name="bBlockNo">Blocknumber on card to authenticate to</param>
        /// <param name="bKeyType">Either <see cref="KeyType.KEYA"/> or <see cref="KeyType.KEYB"/></param>
        /// <param name="wKeyNo">Key number to be used in authentication</param>
        /// <param name="wKeyVersion">Key version to be used in authentication</param>
        /// <param name="pUid">Serial number of current cascade level; uint8_t[4].</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t MfcAuthenticateKeyNo ( byte bBlockNo, KeyType bKeyType, ushort wKeyNo, ushort wKeyVersion,
            byte[] pUid )
        {
            return phhalHw_MfcAuthenticateKeyNo ( m_pDataParams, bBlockNo, ( byte ) bKeyType, wKeyNo,
                wKeyVersion, pUid );
        }

        /// <summary>
        /// Perform MIFARE Classic Authentication using a key.
        /// </summary>
        ///
        /// <param name="bBlockNo">Blocknumber on card to authenticate to</param>
        /// <param name="bKeyType">Either <see cref="KeyType.KEYA"/> or <see cref="KeyType.KEYB"/></param>
        /// <param name="pKey">Key to be used in authentication</param>
        /// <param name="pUid">Serial number of current cascade level; uint8_t[4].</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t MfcAuthenticate ( byte bBlockNo, KeyType bKeyType, byte[] pKey, byte[] pUid )
        {
            return phhalHw_MfcAuthenticate ( m_pDataParams, bBlockNo, ( byte ) bKeyType, pKey,
                pUid );
        }

        /// <summary>
        /// Set configuration parameter.
        ///
        /// Sam <em>Special Behaviour:</em>
        /// In NonX-Mode every setting ( except PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and CONFIG_HOSTMODE )
        /// is passed on to the linked Reader-HAL.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="wValue">Configuration Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t SetConfig ( Config wConfig, int wValue )
        {
            return phhalHw_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        /// <summary>
        /// Get configuration parameter.
        ///
        /// Sam <em>Special Behaviour:</em>
        /// In NonX-Mode every setting ( except PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and CONFIG_HOSTMODE )
        /// is passed on to the linked Reader-HAL.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="wValue">Configuration Value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t GetConfig ( Config wConfig, out int pValue )
        {
            ushort wValue = 0;

            Status_t oStatus = phhalHw_GetConfig ( m_pDataParams, ( ushort ) wConfig, ref wValue );
            pValue = wValue;

            return oStatus;
        }

        /// <summary>
        /// Set & Get configuration setting
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        ///
        /// <returns>Configuration Value</returns>
        public int this[Config wConfig]
        {
            get
            {
                int wValue;
                Status_t oStatus = GetConfig(wConfig, out wValue);
                if ( ( Error_Gen ) oStatus.Error != Error_Gen.SUCCESS )
                {
                    throw new Exception ( oStatus.ToString () );
                }

                return wValue;
            }
            set
            {
                Status_t oStatus = SetConfig(wConfig, value);
                if ( ( Error_Gen ) oStatus.Error != Error_Gen.SUCCESS )
                {
                    throw new Exception ( oStatus.ToString () );
                }
            }
        }

        /// <summary>
        /// Switch on the RF-field.
        /// </summary>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FieldOn ()
        {
            return phhalHw_FieldOn ( m_pDataParams );
        }

        /// <summary>
        /// Switch off the RF-field.
        /// </summary>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t FieldOff ()
        {
            return phhalHw_FieldOff ( m_pDataParams );
        }

        /// <summary>
        /// Perform RF-Reset.
        /// </summary>
        /// <param name="wFieldOffTime"></param>
        /// <param name="wFieldRecoveryTime"></param>
        /// <returns></returns>
        public Status_t FieldReset ()
        {
            return phhalHw_FieldReset ( m_pDataParams );
        }

        /// <summary>
        /// Blocks until the time given by 	<see cref="wTimeout"/> elapsed.
        /// </summary>
        ///
        /// <param name="bUnit">Unit of given timeout value (either <see cref="TimeUnit.MICROSECONDS"/> or
        /// <see cref="TimeUnit.MILLISECONDS"/>).
        /// </param>
        /// <param name="wTimeout">Timeout value</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t Wait ( byte bUnit, int wTimeout )
        {
            return phhalHw_Wait ( m_pDataParams, bUnit, ( ushort ) wTimeout );
        }

        /// <summary>
        /// Perform 18000p3m3 Inventory based on BeginRound and Select Command
        ///
        /// Note: This function does
        ///         ISO18000p3m3 BeginRound
        ///         Sends an Ack
        ///         Optionally sends a ReqRN
        ///         Stores the information related to tag reply in the HAL RX Buffer
        ///         Continues with NextSlot or else pauses and returns based on the option specified
        ///         Can return multi card information based on the configuration option and the size of
        ///         HAL RX Buffer.
        ///
        /// The bTSprocessing ( Time slot processing behavior ) can take the following values
        ///     \li <see cref="TSProcessing.GET_MAX_RESPS"/> - Gets responses from more than one
        ///     slot limited by the Rx Buffer size and number of slots specified in begin round.
        ///
        ///     \li <see cref="TSProcessing.ONE_TS_ONLY"/> - Gets response for only one time slot.
        ///     Here the number of time slots should be zero in the begin round command.
        ///
        ///     \li <see cref="TSProcessing.GET_TAG_HANDLE"/> - Gets response for only one time slot.
        ///     Also send a ReqRN to get the tag handle for this slot. Here the number of time slots
        ///     should be zero in the begin round command.
        ///
        /// The response is a Pointer to a buffer containing the result of the inventory.
        /// The buffer may contain the result of one or more timeslots.Each timeslot result has the
        /// following structure:
        ///     1 Byte time slot status
        ///         0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field' are present.
        ///         1 ... Tag response available.In addition to three field above, 'Tag handle field' also present.
        ///         2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field', 'Tag Handle' are not present.
        ///         3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag reply field not present.
        ///     1 Byte 'Tag reply' Length (1-66)
        ///     1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
        ///     0-66 Bytes Tag reply
        ///     0 or 2 Bytes Handle of the tag, in case the field
        ///
        /// A typical sequence in which this API is called is given below
        ///     \li BAL initialization (where applicable) - Example - <see cref="Bal.SerialWin.Init"/>
        ///     \li HAL initialization - Example <see cref="DUT.Init()"/>
        ///     \li Load ISO 1800p3m3 protocol - <see cref="Generic.ApplyProtocolSettings"/>with <see cref="CardType.I18000P3M3"/>
        ///     \li Turn ON the field - <see cref="Generic.FieldOn"/>
        ///     \li Wait for initial guard time - <see cref="Generic.Wait"/> with wait tie as 5100us
        ///     \li Start inventory - <see cref="Generic.I18000p3m3Inventory"/>
        ///     \li Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate with the tag.
        ///     \li Turn OFF the field - <see cref="Generic.FieldOff"/>
        ///     \li Termination of the application flow.
        /// </summary>
        ///
        /// <param name="pSelCmd">ISO 18000p3m3 Select command frame.</param>
        /// <param name="bSelCmdLen">Select command length in bytes- 1 to 39 bytes.</param>
        /// <param name="bNumValidBitsinLastByte">Number of valid bits in last byte of pSelCmd.</param>
        /// <param name="pBeginRndCmd">ISO 18000p3m3 BeginRound command frame. This is 17bits i.e.,
        /// 3 bytes are expected. CRC5 should not be provided.
        /// </param>
        /// <param name="bTSprocessing">TimeSlot processing behavior.</param>
        /// <param name="ppRxBuffer">Pointer to HAL Rx Buffer containing response from single/multiple tags.</param>
        /// <param name="wRxBufferLen">Length of response in hal Rx Buffer.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t I18000p3m3Inventory ( byte[] pSelCmd, byte bSelCmdLen, byte bNumValidBitsinLastByte, byte[] pBeginRndCmd,
            byte bTSprocessing, out byte[] ppRxBuffer, out ushort wRxBufferLen )
        {
            Status_t oStatus;
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxBuffLen = 0;

            wRxBufferLen = 0;

            oStatus = phhalHw_I18000p3m3Inventory ( m_pDataParams, pSelCmd, bSelCmdLen, bNumValidBitsinLastByte,
                pBeginRndCmd, bTSprocessing, ref pRxBuff, ref wRxBuffLen );

            wRxBufferLen = wRxBuffLen;

            if ( ( pRxBuff != IntPtr.Zero ) && ( wRxBufferLen > 0 ) )
            {
                ppRxBuffer = new byte[wRxBufferLen];
                Marshal.Copy ( pRxBuff, ppRxBuffer, 0, wRxBufferLen );
            }
            else
            {
                ppRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// ISO 18000p3m3 resume inventory command.
        ///
        /// Note: This function is to be called after <see cref="Generic.I18000p3m3Inventory"/>
        /// This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
        /// remaining slots.
        ///
        /// Resume inventory also returns the response similar to phhalHw_I18000p3m3Inventory.
        ///
        /// A typical sequence in which this API is called is given below
        ///     \li BAL initialization (where applicable) - Example - <see cref="Bal.SerialWin.Init"/>
        ///     \li HAL initialization - Example <see cref="DUT.Init()"/>
        ///     \li Load ISO 1800p3m3 protocol - <see cref="Generic.ApplyProtocolSettings"/>with <see cref="CardType.I18000P3M3"/>
        ///     \li Turn ON the field - <see cref="Generic.FieldOn"/>
        ///     \li Wait for initial guard time - <see cref="Generic.Wait"/> with wait tie as 5100us
        ///     \li Start inventory - <see cref="Generic.I18000p3m3Inventory"/>
        ///     \li Store the responses received for the tags.
        ///     \li If inventory has to be continued then issue the ResumeInventory command - <see cref="Generic.I18000p3m3ResumeInventory"/>
        ///     \li Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate with the tag.
        ///     \li Turn OFF the field - <see cref="Generic.FieldOff"/>
        ///     \li Termination of the application flow.
        /// </summary>
        ///
        /// <param name="ppRxBuffer">Pointer to HAL Rx Buffer containing response from single/multiple tags.</param>
        /// <param name="wRxBufferLen">Length of response in hal Rx Buffer.</param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t I18000p3m3ResumeInventory ( out byte[] ppRxBuffer, out ushort wRxBufferLen )
        {
            Status_t oStatus;

            IntPtr pRxBuf = IntPtr.Zero;
            ushort wRxBuffLen = 0;

            wRxBufferLen = 0;
            oStatus = phhalHw_I18000p3m3ResumeInventory ( m_pDataParams, ref pRxBuf, ref wRxBuffLen );

            wRxBufferLen = wRxBuffLen;

            if ( ( pRxBuf != IntPtr.Zero ) && ( wRxBufferLen > 0 ) )
            {
                ppRxBuffer = new byte[wRxBufferLen];
                Marshal.Copy ( pRxBuf, ppRxBuffer, 0, wRxBufferLen );
            }
            else
            {
                ppRxBuffer = null;
            }

            return oStatus;
        }

        public Status_t ExecuteCmd ( byte bCmd, ushort wOption, byte bIrq0WaitFor, byte bIrq1WaitFor, byte[] pTxBuffer,
            ushort wTxLength, ushort wRxBufferSize, out byte[] pRxBuffer )
        {
            pRxBuffer = new byte[wRxBufferSize];
            ushort pRxLength = 0;
            Status_t status;

            status = phhalHw_ExecuteCmd ( m_pDataParams, bCmd, wOption, bIrq0WaitFor, bIrq1WaitFor, pTxBuffer,
                wTxLength, wRxBufferSize, pRxBuffer, ref pRxLength );

            Array.Resize<byte> ( ref pRxBuffer, pRxLength );

            return status;
        }

        /// <summary>
        /// Reads the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Reading.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchagned to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified
        ///                           information to this paramter will be exchagned to Reader.
        ///                           These values are specific to Pegoda - 3 Reader.
        ///                             - 0x0010: Read information from User area
        ///                             - 0x0011: Read information from Secure Lib area
        ///                             - 0x0012: Read information from IC Configuration area
        ///                             - 0x0020: Read information from RF Configuration area
        ///                         - Values other that above specified ones.
        /// </param>
        /// <param name="dwAddr_ProtIndex">The EEPROM address or Protocol Index from where the
        ///                                information should be read.One of the following,
        ///                                 - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                                   EEPROM address.This is supported by Pegoda - 3 Reader
        ///                                   only.
        ///                                 - If \b wOption = 0x0020, then 1 byte Protocol
        ///                                   index of the RF Configuration.This is supported
        ///                                   by Pegoda - 3 Reader only.
        ///                                 - If \b wOption = 0x0000, then 4 byte EEPROM address.
        ///                                   Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="dwNoOfBytesToRead">Number of bytes to be read from EEPROM starting from address specified.
        ///                                 If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
        ///                                 provided.
        /// </param>
        /// <param name="aResponse">EEPROM information from the mentioned start address (\b dwStartAddr)
        ///                         until the requested length.Make sure the buffer is allocated for the
        ///                         requested number of bytes (\b dwLength ).
        /// </param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t ReadEEPROM ( ushort wOption, uint dwAddr_ProtIndex, uint dwNoOfBytesToRead, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_ReadEEPROM ( m_pDataParams, wOption, dwAddr_ProtIndex, dwNoOfBytesToRead,
                ref pResponse, ref wRespLen );

            if ( ( pResponse != IntPtr.Zero ) && ( wRespLen > 0 ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( pResponse, aResponse, 0, wRespLen );
            }
            else
            {
                aResponse = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Writes the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Writing.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchagned to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified
        ///                           information will be exchagned to Reader. These values are
        ///                           specific to Pegoda - 3 Reader.
        ///                             - 0x0010: Read information from User area
        ///                             - 0x0011: Read information from Secure Lib area
        ///                             - 0x0012: Read information from IC Configuration area
        ///                             - 0x0020: Read information from RF Configuration area
        /// </param>
        /// <param name="dwStartAddr">The EEPROM address information to be written.
        ///                           One of the following,
        ///                             - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                               EEPROM address.This is supported by Pegoda - 3 Reader
        ///                               only.
        ///                             - If \b wOption = 0x0020, zero should be provided.
        ///                             - If \b wOption = 0x0000, then 4 byte EEPROM address.
        ///                               Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="aData">Information to be written to EERPOM.
        ///                     One of the following,
        ///                         - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
        ///                           EEPROM value should be provided.
        ///                         - If \b wOption = 0x0020, then 1 byte Protocol index followed
        ///                           1 byte Register address followed by 4 byte Register Value.
        ///                           This is supported by Pegoda - 3 Reader only.
        ///                           Protocol Index1 ( 1 Byte) ||
        ///                           Reg Addr1 ( 1 Byte) || Reg Value1 ( 4 Bytes) ||
        ///                           Reg Addr2 ( 1 Byte) || Reg Value2 ( 4 Bytes) ||
        ///                           ...
        ///                           Reg AddrN ( 1 Byte) || Reg ValueN ( 4 Bytes)
        /// </param>
        ///
        /// <returns>
        ///     <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t WriteEEPROM ( ushort wOption, uint dwStartAddr, byte[] aData )
        {
            return phhalHw_WriteEEPROM ( m_pDataParams, wOption, dwStartAddr, aData,
                ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion
}
