/*
 * Copyright 2018 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Class for DUT layer initialization interface and data params.
    /// </summary>
    public class DUT : Generic
    {
        #region Private Variables
        private byte[] aTxBuffer;
        private GCHandle pTxBuffer;

        private byte[] aRxBuffer;
        private GCHandle pRxBuffer;

        /// <summary> Amount of needed and reserved memory for the protocol overhead. </summary>
        private const int RESERVED_BUFFER_LEN = 11;
        #endregion

        #region Enumerations
        #region Initialization Types
        /// <summary>
        /// Enumeration for Reader Initialization.
        /// </summary>
        public enum Init_Reader
        {
            /// <summary> Initialize the reader to perform Contactless operation. </summary>
            [Obsolete ( "the current flag is obsolete and will be removed. Use CONTACTLESS flag." )]
            DUT_HAL_INIT = 0xB001,

            /// <summary> Initialize the reader to perform Contactless operation. </summary>
            CONTACTLESS = 0xB001,

            /// <summary> Initialize the reader to perform Contact Card (SAM) operation via TDA interface. </summary>
            CONTACT_CARD = 0xB010,

            /// <summary> Initialize the reader to perform I2C operation. </summary>
            I2C = 0x6001,
        }
        #endregion

        #region ReaderTypes
        /// <summary>
        /// Enumeration for FrontEnd (Reader IC) or NFC Controller IC for which the code is compiled.
        /// </summary>
        public enum ReaderType
        {
            /// <summary> Reader Type as Unknown. </summary>
            UNKNOWN = 0x00,

            /// <summary> Connected DUT is PN7462AU. </summary>
            PN7462AU = 0x01,

            /// <summary> Connected DUT is PN7640. </summary>
            PN7640 = 0x02,

            /// <summary> Connected DUT is PN7642. </summary>
            PN7642 = 0x03,

            /// <summary> Connected DUT is RC663. </summary>
            RC663 = 0x80,

            /// <summary> Connected DUT is PN5180. </summary>
            PN5180 = 0x81,

            /// <summary> Connected DUT is PN5190. </summary>
            PN5190 = 0x82
        }
        #endregion

        #region TSProcessing
        /// <summary>
        /// TSprocessing ( Time slot processing behavior )
        /// </summary>
        public enum TsProcesing
        {
            /// <summary>
            /// Gets responses from more than one slot limited by the
            /// Rx Buffer size and number of slots specified in begin round.
            /// </summary>
            GET_MAX_RESPS,

            /// <summary>
            /// Gets response for only one time slot. Here the number of
            /// time slots should be zero in the begin round command.
            /// </summary>
            ONE_TS_ONLY,

            /// <summary>
            /// Gets response for only one time slot. Also send a ReqRN
            /// to get the tag handle for this slot. Here the number of
            /// time slots should be zero in the begin round command.
            /// </summary>
            GET_TAG_HANDLE
        }
        #endregion

        #region EEPROM Tags
        /// <summary>
        /// Tag information to be used while Reading or Writing to EEPROM.
        /// </summary>
        public enum EEPROM_Tag : ushort
        {
            /// <summary> Field to read / Write from User Area </summary>
            USER_AREA = 0x0010,

            /// <summary> Field to read / Write from Secure Lib Area </summary>
            SECURE_LIB_AREA = 0x0011,

            /// <summary> Field to read / Write from IC Configuration Area </summary>
            IC_CONFIGURATION_AREA = 0x0012,

            /// <summary> Field to read / Write from RF (Protocol) Configuration Area </summary>
            RF_CONFIGURATION_AREA = 0x0020
        }
        #endregion
        #endregion

        #region Data Structure
        /// <summary>
        /// Data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the lower layers parameter structure. </summary>
            public IntPtr pBalDataParams;

            /// <summary> Slot number. </summary>
            public byte bSlotNumber;

            /// <summary> TxLastbits for ExchangeRaw function. </summary>
            public byte bTxLastBits;

            /// <summary> Pointer to global transmit buffer used by the Exchange() function. </summary>
            public IntPtr pTxBuffer;

            /// <summary> Size of the global transmit buffer. </summary>
            public ushort wTxBufSize;

            /// <summary> Number of valid bytes within the transmit buffer. </summary>
            public ushort wTxBufLen;

            /// <summary> Start position in case of buffering within the transmit buffer. </summary>
            public ushort wTxBufStart;

            /// <summary> Pointer to global receive buffer used by the Exchange() function. </summary>
            public IntPtr pRxBuffer;

            /// <summary> Size of the global receive buffer. </summary>
            public ushort wRxBufSize;

            /// <summary> Number of valid bytes within the receive buffer. </summary>
            public ushort wRxBufLen;

            /// <summary> Starting position within the global receive buffer. </summary>
            public ushort wRxBufStartPos;

            /// <summary> Starting position within the global transmit buffer (used if TxBuffer = RxBuffer).</summary>
            public ushort wTxBufStartPos;

            /// <summary> Type of card for which the Hal is configured for. </summary>
            public byte bCardType;

            /// <summary> Current timing value backup for PC. </summary>
            public ushort wFdtPc;
        };
        #endregion

        #region Dll Imports
        #region Initialization
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Init ( IntPtr pDataParams, ushort wSizeOfDataParams,
            IntPtr pBalDataParams, byte bSlotNumber, IntPtr pTxBuffer, ushort wTxBufSize, IntPtr pRxBuffer,
            ushort wRxBufSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_InitReader ( IntPtr pDataParams, ushort Cmd );
        #endregion

        #region Version
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_GetVersion ( IntPtr pDataParams, ref byte pProductVer,
            ref ushort pMajor, ref byte pMinor, ref ushort pPatch_Dev, IntPtr pVersionString,
            ref ushort pVersionLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_GetVersion_NFC ( IntPtr pDataParams, ref byte pProductVer,
            ref ushort pMajor, ref byte pMinor, ref ushort pPatch_Dev, IntPtr pVersionString,
            ref ushort pVersionLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_GetVersion_CompiledInfo ( IntPtr pDataParams,
            IntPtr pResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_GetVersion_ReaderIcType ( IntPtr pDataParams,
            IntPtr pResponse, ref ushort pRspLen );
        #endregion

        #region Reader Operation
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_GetReaderStatus ( IntPtr pDataParams,
            ref IntPtr ppReaderStatus, ref ushort pReaderStatusLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_Read_Register ( IntPtr pDataParams,
            uint dwRegAddr, ref uint pRegData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_Write_Register ( IntPtr pDataParams,
            uint dwRegAddr, uint dwRegData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_FieldOn ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_FieldOff ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_FieldReset ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_Read_EEPROM ( IntPtr pDataParams, ushort wOption,
            uint dwAddr_ProtIndex, uint dwNoOfBytesToRead, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_RdOps_Write_EEPROM ( IntPtr pDataParams, ushort wOption,
            uint dwStartAddr, byte[] pData, ushort wDataLen );
        #endregion

        #region HAL Operation
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_Exchange ( IntPtr pDataParams, ushort wOption,
            byte[] pTxBuff, ushort wTxLen,
            ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_SetConfig ( IntPtr pDataParams, ushort wConfig,
            ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_GetConfig ( IntPtr pDataParams, ushort wConfig,
            ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_ApplyProtocolSettings ( IntPtr pDataParams,
            byte bCardType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_Wait ( IntPtr pDataParams, byte bUnit,
            ushort wTimeout );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_MfcAuthenticate ( IntPtr pDataParams, byte bBlockNo,
            byte bKeyType, byte[] pKey, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_Execute ( IntPtr pDataParams, byte bCmd, ushort wOption,
            byte[] pTxBuff, ushort wTxLen, ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_MfcAuthenticateKeyNo ( IntPtr pDataParams, byte bBlockNo,
            byte bKeyType, ushort wKeyNo, ushort wKeyVer, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_Transmit ( IntPtr pDataParams, ushort wOption,
            byte[] pTxBuff, ushort wTxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_Receive ( IntPtr pDataParams, ushort wOption,
            ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_I18000p3m3Inventory ( IntPtr pDataParams, byte[] pSelCmd,
            byte bSelCmdLen, byte bNumValidBitsinLastByte, byte[] pBeginRndCmd, byte bTSprocessing,
            ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_HAL_I18000p3m3ResumeInventory ( IntPtr pDataParams,
            ref IntPtr ppRxBuff, ref ushort pRxLen );
        #endregion

        #region Contact Card
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_Activate ( IntPtr pDataParams, IntPtr pAtr,
            ref ushort usAtrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_ColdReset ( IntPtr pDataParams, IntPtr pAtr,
            ref ushort usAtrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_WarmReset ( IntPtr pDataParams, IntPtr pAtr,
            ref ushort usAtrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_StopClock ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_StartClock ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_Deactivate ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_PresenceCheck ( IntPtr pDataParams, ref byte pCardInserted );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_TransmitData ( IntPtr pDataParams, byte[] pTxBuff,
            ushort wTxLen, ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_CC_SendPPS ( IntPtr pDataParams, byte bOption, byte[] pPPS,
            IntPtr pRxBuff, ref ushort pRxLen );
        #endregion

        #region I2C Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_Exchange ( IntPtr pDataParams, ushort wOption, ushort wSlaveAddr,
            byte[] pTxBuff, ushort wTxLen, ushort wBytesToRead, ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_Transmit ( IntPtr pDataParams, ushort wOption, ushort wSlaveAddr,
            byte[] pTxBuff, ushort wTxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_Receive ( IntPtr pDataParams, ushort wOption, ushort wSlaveAddr,
            ushort wBytesToRead,
            ref IntPtr ppRxBuff, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_Start ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_Stop ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_DUT_Cmd_I2C_RepeatedStart ( IntPtr pDataParams );
        #endregion
        #endregion

        #region Wrapper Functions
        #region Initialization
        /// <summary>
        /// Initialize the HAL component.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="bSlotNumber">Slot number.</param>
        /// <param name="wTxBufferSize">Pointer to global transmit buffer.</param>
        /// <param name="wRxBufferSize">Pointer to global receive buffer</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( Bal.Generic oBal, byte bSlotNumber, int wTxBufferSize, int wRxBufferSize )
        {
            /* Adjust TxBuffer length to fit command header. */
            if ( ( wTxBufferSize + RESERVED_BUFFER_LEN ) > 0xFFFF )
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            /* Adjust RxBuffer length to fit command header. */
            if ( ( wRxBufferSize + RESERVED_BUFFER_LEN ) > 0xFFFF )
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_LEN;
            }

            /* Free the previously allocated Buffers */
            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
            }
            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
            }

            /* Allocate buffers */
            aTxBuffer = new byte[wTxBufferSize];
            aRxBuffer = new byte[wRxBufferSize];
            pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );
            pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );

            // Call init function
            return phhalHw_DUT_Init ( m_pDataParams, ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oBal.m_pDataParams, bSlotNumber, pTxBuffer.AddrOfPinnedObject (), ( ushort ) wTxBufferSize,
                pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize );
        }

#if DEBUG
        /// <summary>
        /// Initialize the HAL component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Size of HAL DUT's DataParams structure.</param>
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="bSlotNumber">Slot number.</param>
        /// <param name="wTxBufferSize">Pointer to global transmit buffer.</param>
        /// <param name="wRxBufferSize">Pointer to global receive buffer</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( int wDataParamSize, Bal.Generic oBal, byte bSlotNumber, int wTxBufferSize,
            byte[] aTxBuffer, int wRxBufferSize, byte[] aRxBuffer )
        {
            /* Free the previously allocated Buffers */
            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
            }

            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
            }

            // Link given buffers
            pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );
            pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );

            // Call init function
            return phhalHw_DUT_Init ( m_pDataParams, ( ushort ) wDataParamSize, oBal.m_pDataParams, bSlotNumber,
                pTxBuffer.AddrOfPinnedObject (), ( ushort ) wTxBufferSize, pRxBuffer.AddrOfPinnedObject (),
                ( ushort ) wRxBufferSize );
        }
#endif

        /// <summary>
        /// The function performs the initialization of different DataParams.
        ///
        /// To initialize the communication interface of the reader below information should be
        /// used for \b wCmd parameter.
        /// -\ref PHHAL_HW_DUT_INIT_CONTACTLESS "Contactless via RF Interface"
        /// -\ref PHHAL_HW_DUT_INIT_CONTACT_CARD_TDA "ContactCard via TDA Interface"
        /// -\ref PHHAL_HW_DUT_INIT_I2C "I2C Interface"
        /// </summary>
        ///
        /// <param name="eInitCmd">Command to initialize the Reader. Refer description for values
        ///                        and combination to use.
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t DUT_InitReader ( Init_Reader eInitCmd )
        {
            return phhalHw_DUT_InitReader ( m_pDataParams, ( ushort ) eInitCmd );
        }
        #endregion

        #region Version
        /// <summary>
        /// The function gets the Version of the Firmware of DUT.
        /// </summary>
        ///
        /// <param name="bProductVer">Product Version.</param>
        /// <param name="wMajor">Major Version.</param>
        /// <param name="bMinor">Minor Version.</param>
        /// <param name="wPatch_Dev">Development/Patch Version.</param>
        /// <param name="aVersionString">Complete version information in form of ASCII bytes.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        [Obsolete ( "The current method is obsolete and will be removed. Use GetVersion method." )]
        public Status_t DUT_GetVersion ( out byte bProductVer, out ushort wMajor, out byte bMinor,
            out ushort wPatch_Dev, out byte[] aVersionString )
        {
            ushort wVerStringLen = 255;

            bProductVer = 0;
            wMajor = 0;
            bMinor = 0;
            wPatch_Dev = 0;

            aVersionString = new byte[wVerStringLen];
            GCHandle oVersionString = GCHandle.Alloc ( aVersionString, GCHandleType.Pinned );
            IntPtr pVersionString = oVersionString.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_GetVersion ( m_pDataParams, ref bProductVer, ref wMajor,
                ref bMinor, ref wPatch_Dev, pVersionString, ref wVerStringLen );
            aVersionString = MarshalCopy ( oStatus, pVersionString, wVerStringLen );

            pVersionString = IntPtr.Zero;
            oVersionString.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function gets the Version of the Firmware of DUT.
        /// </summary>
        ///
        /// <param name="bProductVer">Product Version.</param>
        /// <param name="wMajor">Major Version.</param>
        /// <param name="bMinor">Minor Version.</param>
        /// <param name="wPatch_Dev">Development/Patch Version.</param>
        /// <param name="aVersionString">Complete version information in form of ASCII bytes.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetVersion ( out byte bProductVer, out ushort wMajor, out byte bMinor,
            out ushort wPatch_Dev, out byte[] aVersionString )
        {
            ushort wVerStringLen = 255;

            bProductVer = 0;
            wMajor = 0;
            bMinor = 0;
            wPatch_Dev = 0;

            aVersionString = new byte[wVerStringLen];
            GCHandle oVersionString = GCHandle.Alloc ( aVersionString, GCHandleType.Pinned );
            IntPtr pVersionString = oVersionString.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_GetVersion ( m_pDataParams, ref bProductVer, ref wMajor,
                ref bMinor, ref wPatch_Dev, pVersionString, ref wVerStringLen );
            aVersionString = MarshalCopy ( oStatus, pVersionString, wVerStringLen );

            pVersionString = IntPtr.Zero;
            oVersionString.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function gets the Version of the NFC Controller connected to the DUT.
        /// </summary>
        ///
        /// <param name="bProductVer">Product Version.</param>
        /// <param name="wMajor">Major Version.</param>
        /// <param name="bMinor">Minor Version.</param>
        /// <param name="wPatch_Dev">Development/Patch Version.</param>
        /// <param name="aVersionString">Complete version information in form of ASCII bytes.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetVersion_NFC ( out byte bProductVer, out ushort wMajor, out byte bMinor,
            out ushort wPatch_Dev, out byte[] aVersionString )
        {
            ushort wVerStringLen = 255;

            bProductVer = 0;
            wMajor = 0;
            bMinor = 0;
            wPatch_Dev = 0;

            aVersionString = new byte[wVerStringLen];
            GCHandle oVersionString = GCHandle.Alloc ( aVersionString, GCHandleType.Pinned );
            IntPtr pVersionString = oVersionString.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_GetVersion_NFC ( m_pDataParams, ref bProductVer, ref wMajor,
                ref bMinor, ref wPatch_Dev, pVersionString, ref wVerStringLen );
            aVersionString = MarshalCopy ( oStatus, pVersionString, wVerStringLen );

            pVersionString = IntPtr.Zero;
            oVersionString.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function gets the date and time at which the firmware code is compiled.
        /// </summary>
        ///
        /// <param name="aResponse">Complete version information in form of ASCII bytes.
        ///                         Should be minimum 128 bytes long.
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetVersion_CompiledInfo ( out byte[] aResponse )
        {
            ushort wRspLen = 255;

            aResponse = new byte[wRspLen];
            GCHandle oResponse = GCHandle.Alloc ( aResponse, GCHandleType.Pinned );
            IntPtr pResponse = oResponse.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_GetVersion_CompiledInfo ( m_pDataParams, pResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRspLen );

            pResponse = IntPtr.Zero;
            oResponse.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function gets the type of NFC Reader IC being used.
        /// </summary>
        ///
        /// <param name="aReaderType">NFCController / FrontEnd type.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetVersion_ReaderIcType ( out byte[] aReaderType )
        {
            ushort wRspLen = 0xFF;

            aReaderType = new byte[wRspLen];
            GCHandle oResponse = GCHandle.Alloc ( aReaderType, GCHandleType.Pinned );
            IntPtr pResponse = oResponse.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_GetVersion_ReaderIcType ( m_pDataParams, pResponse, ref wRspLen );
            aReaderType = MarshalCopy ( oStatus, pResponse, wRspLen );

            pResponse = IntPtr.Zero;
            oResponse.Free ();

            return oStatus;
        }
        #endregion

        #region Reader Operation
        /// <summary>
        /// Gets the Status and Firmware Version of the Reader.
        /// </summary>
        ///
        /// <param name="aReaderStatus">The Status and Firmware Version information returned by the reader.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_GetReaderStatus ( out byte[] aReaderStatus )
        {
            ushort wRspLen = 0;

            aReaderStatus = new byte[20];
            GCHandle oResponse = GCHandle.Alloc ( aReaderStatus, GCHandleType.Pinned );
            IntPtr pResponse = oResponse.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_RdOps_GetReaderStatus ( m_pDataParams, ref pResponse, ref wRspLen );
            aReaderStatus = MarshalCopy ( oStatus, pResponse, wRspLen );

            pResponse = IntPtr.Zero;
            oResponse.Free ();

            return oStatus;
        }

        /// <summary>
        /// Reads the Register information.
        /// </summary>
        ///
        /// <param name="dwRegAddr">The Register address from where the information should be read.</param>
        /// <param name="dwRegData">The Register data for the provided Address.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_Read_Register ( uint dwRegAddr, out uint dwRegData )
        {
            dwRegData = 0;
            Status_t oStatus = phhalHw_DUT_Cmd_RdOps_Read_Register ( m_pDataParams, dwRegAddr,
                ref dwRegData );

            return oStatus;
        }

        /// <summary>
        /// Reads the Register information.
        /// </summary>
        ///
        /// <param name="dwRegAddr">The Register address to which the information should be written.</param>
        /// <param name="dwRegData">The Register data for the provided Address.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_Write_Register ( uint dwRegAddr, uint dwRegData )
        {
            return phhalHw_DUT_Cmd_RdOps_Write_Register ( m_pDataParams,
                dwRegAddr, dwRegData );
        }

        /// <summary>
        /// Switches ON the RF Field.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_FieldOn ()
        {
            return phhalHw_DUT_Cmd_RdOps_FieldOn ( m_pDataParams );
        }

        /// <summary>
        /// Switches OFF the RF Field.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        ///
        public Status_t RdOps_FieldOff ()
        {
            return phhalHw_DUT_Cmd_RdOps_FieldOff ( m_pDataParams );
        }

        /// <summary>
        /// Resets the RF Field
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_FieldReset ()
        {
            return phhalHw_DUT_Cmd_RdOps_FieldReset ( m_pDataParams );
        }

        /// <summary>
        /// Reads the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Reading.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchanged to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified information
        ///                           will be exchanged to Reader. These values are specific to Pegoda - 3
        ///                           Reader.
        ///                             - <see cref="EEPROM_Tag.USER_AREA"/>: Read information from User area
        ///                             - <see cref="EEPROM_Tag.SECURE_LIB_AREA"/>: Read information from Secure Lib area
        ///                             - <see cref="EEPROM_Tag.IC_CONFIGURATION_AREA"/>: Read information from IC Configuration area
        ///                             - <see cref="EEPROM_Tag.RF_CONFIGURATION_AREA"/>: Read information from RF Configuration area
        /// </param>
        /// <param name="dwAddr_ProtIndex">The EEPROM address or Protocol Index from where the information
        ///                           should be read. One of the following,
        ///                             - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                               EEPROM address.This is supported by Pegoda - 3 Reader
        ///                               only.
        ///                             - If \b wOption = 0x0020, then 1 byte Protocol
        ///                               index of the RF Configuration.This is supported
        ///                               by Pegoda - 3 Reader only.
        ///                             - If \b wOption = 0x0000, then 2 byte EEPROM address.
        ///                               Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="dwNoOfBytesToRead">Number of bytes to be read from EEPROM starting from address specified.
        ///                        If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
        ///                        provided
        /// </param>
        /// <param name="aResponse">EEPROM information from the mentioned start address (\b dwStartAddr)
        ///                         until the requested length.
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_Read_EEPROM ( ushort wOption, uint dwAddr_ProtIndex, uint dwNoOfBytesToRead, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_RdOps_Read_EEPROM ( m_pDataParams, wOption, dwAddr_ProtIndex,
                dwNoOfBytesToRead, ref pResponse,ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Writes the EEPROM information.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for EEPROM Writing.
        ///                         - 0x0000: Only \b dwStartAddr and \b dwLength information
        ///                           will be exchanged to Reader.
        ///                         - \b dwStartAddr and \b dwLength along with the specified information
        ///                           will be exchanged to Reader. These values are specific to Pegoda - 3
        ///                           Reader.
        ///                             - <see cref="EEPROM_Tag.USER_AREA"/>: Read information from User area
        ///                             - <see cref="EEPROM_Tag.SECURE_LIB_AREA"/>: Read information from Secure Lib area
        ///                             - <see cref="EEPROM_Tag.IC_CONFIGURATION_AREA"/>: Read information from IC Configuration area
        ///                             - <see cref="EEPROM_Tag.RF_CONFIGURATION_AREA"/>: Read information from RF Configuration area
        /// </param>
        /// <param name="dwStartAddr">The EEPROM address information to be written.
        ///                           One of the following,
        ///                             - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
        ///                               EEPROM address.This is supported by Pegoda - 3 Reader
        ///                               only.
        ///                             - If \b wOption = 0x0020, zero should be provided.
        ///                             - If \b wOption = 0x0000, then 4 byte EEPROM address.
        ///                               Reader other than Pegoda 3 reader.
        /// </param>
        /// <param name="aData">Information to be written to EERPOM.
        ///                     One of the following,
        ///                         - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
        ///                           EEPROM value should be provided.
        ///                         - If \b wOption = 0x0020, then 1 byte Protocol index followed
        ///                           1 byte Register address followed by 4 byte Register Value.
        ///                           This is supported by Pegoda - 3 Reader only.
        ///                           Protocol Index1 || Reg Addr1 || Reg Value1 ||
        ///                           Protocol Index2 || Reg Addr2 || Reg Value2 ||
        ///                           ...
        ///                           Protocol IndexN || Reg AddrN || Reg ValueN
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/> for \b aData length greater than 64K Bytes.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t RdOps_Write_EEPROM ( ushort wOption, uint dwStartAddr, byte[] aData )
        {
            if ( aData.Length > 65536 )
                return new Status_t ( Error_CompCode.HAL, Error_Param.PARAMETER_SIZE );

            return phhalHw_DUT_Cmd_RdOps_Write_EEPROM ( m_pDataParams, wOption, dwStartAddr, aData,
                ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion

        #region HAL Operation
        /// <summary>
        /// Perform Data Exchange with a PICC. This command is used to exchange information to PICC and at the same
        /// time receives the payload from PICC if available including the Status.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to PICC and also
        ///           receives the payload from PICC if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command
        ///           header of PICC. Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the
        ///           Data part for PICC (if any). Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part
        ///           for PICC (if any). Here the complete buffered information is exchanged to PICC
        ///           and also receives the information from PICC if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.TXCHAINING"/>
        ///             - To send any pending payload to DUT.
        ///             - If DUT requires more data, then DUT will respond with
        ///               <see cref="Error_Gen.SUCCESS_CHAINING">Require More Data</see> as status for
        ///               the current exchange call. The interface should be called for this flag until
        ///               DUT returns <see cref="Error_Gen.SUCCESS">Complete</see> as status.
        ///
        ///         - <see cref="ExchangeOptions.RXCHAINING"/>
        ///             - To receive any pending payload from DUT.
        ///             - To start using this flag, DUT will return <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               More Data Available</see> as status during previous reception. Payload should be
        ///               received until <see cref="Error_Gen.SUCCESS">Complete</see> as status is returned
        ///               by DUT.
        ///
        /// NOTE
        ///     Before calling this interface the setting of Firmware / PICC must be configured
        ///     using <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange.
        ///                       Refer description for more details.
        /// </param>
        /// <param name="aTxBuff">Payload information to be exchanged / buffers based on
        ///                       \b wOption information.
        /// </param>
        /// <param name="aRxBuff">Information provided by PICC.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_Exchange ( ushort wOption, byte[] aTxBuff, out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_HAL_Exchange ( m_pDataParams, wOption, aTxBuff,
                ( ushort ) ( ( aTxBuff == null ) ? 0 : aTxBuff.Length ), ref pRxBuff,
                ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier.</param>
        /// <param name="wValue">Configuration Value.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> if Configuration is not supported
        ///                                                             or invalid.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the
        ///                                                      allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data
        ///                                                     length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_SetConfig ( ushort wConfig, ushort wValue )
        {
            return phhalHw_DUT_Cmd_HAL_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Get configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier.</param>
        /// <param name="wValue">Configuration Value.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> if Configuration is not supported
        ///                                                             or invalid.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the
        ///                                                      allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data
        ///                                                     length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_GetConfig ( ushort wConfig, out ushort wValue )
        {
            wValue = 0;
            return phhalHw_DUT_Cmd_HAL_GetConfig ( m_pDataParams, wConfig, ref wValue );
        }

        /// <summary>
        /// Apply protocol settings.
        /// To keep the current card type, use <see cref="CardType.CURRENT"/> for \b bCardType.
        /// </summary>
        ///
        /// <param name="bCardType">Type of card for which the HAL should be configured for.
        ///                         One for the below values.
        ///                             <see cref="CardType.CURRENT">Current Configuration</see>
        ///                             <see cref="CardType.ISO14443A">ISO14443 Type-A Configuration</see>
        ///                             <see cref="CardType.ISO14443B">ISO14443 Type-B Configuration</see>
        ///                             <see cref="CardType.ISO15693">ISO15693 Configuration</see>
        ///                             <see cref="CardType.I18000P3M3">ISO18000 p3m3 Configuration</see>
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_ApplyProtocolSettings ( byte bCardType )
        {
            return phhalHw_DUT_Cmd_HAL_ApplyProtocolSettings ( m_pDataParams, bCardType );
        }

        /// <summary>
        /// Blocks until the time given by \b wTimeout elapsed.
        /// </summary>
        /// <param name="bUnit">Unit of given timeout value. Supported values are
        ///                         - <see cref="Config.TIMEOUT_VALUE_US"/>
        ///                         - <see cref="Config.TIMEOUT_VALUE_MS"/>
        /// </param>
        /// <param name="wTimeout">Timeout value based on \b bUnit information.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_Wait ( byte bUnit, ushort wTimeout )
        {
            return phhalHw_DUT_Cmd_HAL_Wait ( m_pDataParams, bUnit, wTimeout );
        }

        /// <summary>
        /// Perform MIFARE(R) Classic Authentication using a key.
        /// </summary>
        /// <param name="bBlockNo">Block Number on card to Authenticate to.</param>
        /// <param name="bKeyType">Key Type to use for Authentication.
        ///                         - <see cref="KeyType.KEYA"/>
        ///                         - <see cref="KeyType.KEYB"/>
        /// </param>
        /// <param name="aKey">Key to be used in authentication.</param>
        /// <param name="aUid">4 byte Serial number of current cascade level.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.IO_TIMEOUT"/> for Authentication failure.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for Authentication failure.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_MfcAuthenticate ( byte bBlockNo, byte bKeyType, byte[] aKey, byte[] aUid )
        {
            return phhalHw_DUT_Cmd_HAL_MfcAuthenticate ( m_pDataParams, bBlockNo, bKeyType, aKey, aUid );
        }

        /// <summary>
        /// Perform Command Execution with a PICC. This command is used to exchange information to PICC and at the same
        /// time receives the payload from PICC if available including the Status.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to PICC and also
        ///           receives the payload from PICC if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command
        ///           header of PICC. Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the
        ///           Data part for PICC (if any). Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part
        ///           for PICC (if any). Here the complete buffered information is exchanged to PICC
        ///           and also receives the information from PICC if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.TXCHAINING"/>
        ///             - To send any pending payload to DUT.
        ///             - If DUT requires more data, then DUT will respond with <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               Require More Data</see> as status for the current exchange call. The interface should be called
        ///               for this flag until DUT returns <see cref="Error_Gen.SUCCESS">Complete</see> as status.
        ///
        ///         - <see cref="ExchangeOptions.RXCHAINING"/>
        ///             - To receive any pending payload from DUT.
        ///             - To start using this flag, DUT will return <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               More Data Available</see> as status during previous reception. Payload should be received until
        ///               <see cref="Error_Gen.SUCCESS">Complete</see> as status is returned by DUT.
        ///
        /// NOTE
        ///     Before calling this interface the setting of Firmware / PICC must be configured using
        ///     <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="bCmd">Command to be executed on the Reader IC.</param>
        /// <param name="wOption">Buffering options to perform exchange.
        ///                       Refer description for more details.
        /// </param>
        /// <param name="aTxBuff">Payload information to be exchanged / buffers based
        ///                       on \b wOption information.
        /// </param>
        /// <param name="aRxBuff">Information provided by PICC.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data
        ///                                                     length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_Execute ( byte bCmd, ushort wOption, byte[] aTxBuff, out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_HAL_Execute ( m_pDataParams, bCmd, wOption, aTxBuff,
                ( ushort ) ( ( aTxBuff == null ) ? 0 : aTxBuff.Length ), ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// Perform MIFARE(R) Classic Authentication using a key number.
        /// </summary>
        ///
        /// <param name="bBlockNo">Block Number on card to Authenticate to.</param>
        /// <param name="bKeyType">Key Type to use for Authentication.
        ///                         - <see cref="KeyType.KEYA"/>
        ///                         - <see cref="KeyType.KEYB"/>
        /// </param>
        /// <param name="wKeyNo">Key number to be used in authentication.</param>
        /// <param name="wKeyVer">Key version to be used in authentication.</param>
        /// <param name="aUid">4 byte Serial number of current cascade level.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> for Communication error.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if Parameter value is invalid.
        ///     Returns <see cref="Error_Comm.IO_TIMEOUT"/> for Authentication failure.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> for Authentication failure.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_MfcAuthenticateKeyNo ( byte bBlockNo, byte bKeyType, ushort wKeyNo,
            ushort wKeyVer, byte[] aUid )
        {
            return phhalHw_DUT_Cmd_HAL_MfcAuthenticateKeyNo ( m_pDataParams, bBlockNo, bKeyType,
                wKeyNo, wKeyVer, aUid );
        }

        /// <summary>
        /// Perform Transmission of data with a PICC. This command is used to exchange information to PICC and at the same
        /// time receives the payload from PICC if available including the Status.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to PICC and also
        ///           receives the payload from PICC if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command
        ///           header of PICC. Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the
        ///           Data part for PICC (if any). Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part
        ///           for PICC (if any). Here the complete buffered information is exchanged to PICC
        ///           and also receives the information from PICC if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.TXCHAINING"/>
        ///             - To send any pending payload to DUT.
        ///             - If DUT requires more data, then DUT will respond with <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               Require More Data</see> as status for the current exchange call. The interface should be called
        ///               for this flag until DUT returns <see cref="Error_Gen.SUCCESS">Complete</see> as status.
        ///
        /// NOTE
        ///     Before calling this interface the setting of Firmware / PICC must be configured using
        ///     <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange.
        ///                       Refer description for more details.
        /// </param>
        /// <param name="aTxBuff">Payload information to be exchanged / buffers based
        ///                       on \b wOption information.
        /// </param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_Transmit ( ushort wOption, byte[] aTxBuff, ushort wTxLen )
        {
            return phhalHw_DUT_Cmd_HAL_Transmit ( m_pDataParams, wOption, aTxBuff,
                ( ushort ) ( ( aTxBuff == null ) ? 0 : aTxBuff.Length ) );
        }

        /// <summary>
        /// Perform Reception of Data from a PICC. This command is used to exchange information to PICC and at the same
        /// time receives the payload from PICC if available including the Status.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to PICC and also
        ///           receives the payload from PICC if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command
        ///           header of PICC. Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the
        ///           Data part for PICC (if any). Here the information is not exchanged with PICC.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part
        ///           for PICC (if any). Here the complete buffered information is exchanged to
        ///           PICC and also receives the information from PICC if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.RXCHAINING"/>
        ///             - To receive any pending payload from DUT.
        ///             - To start using this flag, DUT will return <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               More Data Available</see> as status during previous reception. Payload should be
        ///               received until <see cref="Error_Gen.SUCCESS">Complete</see> as status is returned
        ///               by DUT.
        ///
        /// NOTE
        ///     Before calling this interface the setting of Firmware / PICC must be configured using
        ///     <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange.
        ///                       Refer description for more details.
        /// </param>
        /// <param name="aRxBuff">Information provided by PICC.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_Recieve ( ushort wOption, out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_HAL_Receive ( m_pDataParams, wOption, ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// ISO 18000p3m3 Inventory command.
        ///
        /// Note: This function does
        ///     ISO18000p3m3 BeginRound
        ///     Sends an Acknowledgment
        ///     Optionally sends a ReqRN
        ///     Stores the information related to tag reply in the HAL RX Buffer
        ///     Continues with NextSlot or else pauses and returns based on the option specified
        ///     Can return multi card information based on the configuration option and the size of HAL RX Buffer.
        ///     The bTSprocessing ( Time slot processing behavior ) can take the following values
        ///         - <see cref="TsProcesing.GET_MAX_RESPS"/> - Gets responses from more than one slot limited by the
        ///           Rx Buffer size and number of slots specified in begin round.
        ///         - <see cref="TsProcesing.ONE_TS_ONLY"/> - Gets response for only one time slot. Here the number of
        ///           time slots should be zero in the begin round command.
        ///         - <see cref="TsProcesing.GET_TAG_HANDLE"/> - Gets response for only one time slot. Also send a ReqRN
        ///           to get the tag handle for this slot. Here the number of time slots should be zero in the begin
        ///           round command.
        ///
        /// The response is a Pointer to a buffer containing the result of the inventory.
        /// The buffer may contain the result of one or more timeslots. Each timeslot result has the following structure:
        /// 1 Byte time slot status
        /// (
        ///     0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field'
        ///           are present.
        ///     1 ... Tag response available.In addition to three field above, 'Tag handle field' also present.
        ///     2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field',
        ///           'Tag Handle' are not present.
        ///     3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag
        ///           reply field not present.
        /// )
        /// 1 Byte 'Tag reply' Length (1-66)
        /// 1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
        /// 0-66 Bytes Tag reply
        /// 0 or 2 Bytes Handle of the tag, in case the field
        ///
        /// A typical sequence in which this API is called is given below
        ///     - <see cref="Bal.SerialWin.Init">BAL initialization</see>
        ///     - <see cref="DUT.Init">HAL initialization</see>
        ///     - <see cref="HAL_ApplyProtocolSettings">Load ISO 1800p3m3 protocol</see> with Card Type as
        ///       <see cref="CardType.I18000P3M3"/>
        ///     - <see cref="RdOps_FieldOn">Turn ON the field</see>
        ///     - <see cref="HAL_Wait">Wait for initial 5000us guard time</see>
        ///     - <see cref="HAL_I18000p3m3Inventory">Start inventory</see>
        ///     - Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate
        ///       with the tag.
        ///     - <see cref="RdOps_FieldOff">Turn OFF the field</see>
        ///     - Termination of the application flow.
        /// </summary>
        ///
        /// <param name="aSelCmd">ISO 18000p3m3 Select command frame.</param>
        /// <param name="bSelCmdLen">Select command length in bytes- 1 to 39 bytes.</param>
        /// <param name="bNumValidBitsinLastByte">Number of valid bits in last byte of aSelCmd.</param>
        /// <param name="aBeginRndCmd">ISO 18000p3m3 BeginRound command frame. This is 17bits
        ///                            i.e., 3 bytes are expected. CRC5 should not be provided.
        /// </param>
        /// <param name="bTSprocessing">TimeSlot processing behavior.</param>
        /// <param name="aRxBuff">Information provided by ISO18000 tag.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the
        ///                                                      allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data
        ///                                                     length or command code mismatch.
        ///     Returns <see cref="Error_Comm.IO_TIMEOUT"/> when there is no tag in the field.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If wrong value specified for bNumValidBits
        ///                                                         or pBeginRound.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_I18000p3m3Inventory ( byte[] aSelCmd, byte bSelCmdLen, byte bNumValidBitsinLastByte,
            byte[] aBeginRndCmd, byte bTSprocessing, out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_HAL_I18000p3m3Inventory ( m_pDataParams, aSelCmd, bSelCmdLen,
                bNumValidBitsinLastByte, aBeginRndCmd, bTSprocessing, ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// ISO 18000p3m3 resume inventory command.
        ///
        /// Note: This function is to be called after \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
        /// This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
        /// remaining slots.
        ///
        /// Resume inventory also returns the response similar to \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
        ///
        /// A typical sequence in which this API is called is given below
        ///     - <see cref="Bal.SerialWin.Init">BAL initialization</see>
        ///     - <see cref="DUT.Init">HAL initialization</see>
        ///     - <see cref="HAL_ApplyProtocolSettings">Load ISO 1800p3m3 protocol</see> with Card Type as
        ///       <see cref="CardType.I18000P3M3"/>
        ///     - <see cref="RdOps_FieldOn">Turn ON the field</see>
        ///     - <see cref="HAL_Wait">Wait for initial 5000us guard time</see>
        ///     - <see cref="HAL_I18000p3m3Inventory">Start inventory</see>
        ///     - Store the responses received for the tags.
        ///     - If inventory has to be continued then issue the <see cref="HAL_I18000p3m3ResumeInventory"/> command
        ///     - Use <see cref="palI18000p3m3"/> and <see cref="alI18000p3m3"/> APIs to further communicate with the tag.
        ///     - <see cref="RdOps_FieldOff">Turn OFF the field</see>
        ///     - Termination of the application flow.
        /// </summary>
        ///
        /// <param name="aRxBuff">Information provided by ISO18000 tag.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the
        ///                                                      allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data
        ///                                                     length or command code mismatch.
        ///     Returns <see cref="Error_Comm.IO_TIMEOUT"/> when there is no tag in the field.
        ///     Returns <see cref="Error_Param.USE_CONDITION"/> If called when the phhalHw_I18000p3m3 was not
        ///                                                     called previously with bTSprocessing =
        ///                                                     <see cref="TsProcesing.GET_MAX_RESPS"/>.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t HAL_I18000p3m3ResumeInventory ( out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_HAL_I18000p3m3ResumeInventory ( m_pDataParams,
                ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }
        #endregion

        #region Contact Card
        /// <summary>
        /// The function performs contact card activation procedure according to ISO7816-3 and
        /// when successful outputs the returned ATR.
        /// </summary>
        ///
        /// <param name="aAtr">Buffer holding the returned ATR string from the contact card.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_Activate ( out byte[] aAtr )
        {
            ushort wAtrLen = 0;

            aAtr = new byte[128];
            GCHandle oAtr = GCHandle.Alloc ( aAtr, GCHandleType.Pinned );
            IntPtr pAtr = oAtr.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_CC_Activate ( m_pDataParams, pAtr, ref wAtrLen );
            aAtr = MarshalCopy ( oStatus, pAtr, wAtrLen );

            pAtr = IntPtr.Zero;
            oAtr.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function performs contact card cold reset.
        /// </summary>
        ///
        /// <param name="aAtr">Buffer holding the returned ATR string from the contact card.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_ColdReset ( out byte[] aAtr )
        {
            ushort wAtrLen = 128;

            aAtr = new byte[wAtrLen];
            GCHandle oAtr = GCHandle.Alloc ( aAtr, GCHandleType.Pinned );
            IntPtr pAtr = oAtr.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_CC_ColdReset ( m_pDataParams, pAtr, ref wAtrLen );
            aAtr = MarshalCopy ( oStatus, pAtr, wAtrLen );

            pAtr = IntPtr.Zero;
            oAtr.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function performs contact card warm reset.
        /// </summary>
        ///
        /// <param name="aAtr">Buffer holding the returned ATR string from the contact card.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_WarmReset ( out byte[] aAtr )
        {
            ushort wAtrLen = 128;

            aAtr = new byte[wAtrLen];
            GCHandle oAtr = GCHandle.Alloc ( aAtr, GCHandleType.Pinned );
            IntPtr pAtr = oAtr.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_CC_WarmReset ( m_pDataParams, pAtr, ref wAtrLen );
            aAtr = MarshalCopy ( oStatus, pAtr, wAtrLen );

            pAtr = IntPtr.Zero;
            oAtr.Free ();

            return oStatus;
        }

        /// <summary>
        /// The function stops the clock of the contact card interface.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_StopClock ()
        {
            return phhalHw_DUT_Cmd_CC_StopClock ( m_pDataParams );
        }

        /// <summary>
        /// The function starts the clock pulse of the contact card interface.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_StartClock ()
        {
            return phhalHw_DUT_Cmd_CC_StartClock ( m_pDataParams );
        }

        /// <summary>
        /// The function deactivates the card connected to the contact interface.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_Deactivate ()
        {
            return phhalHw_DUT_Cmd_CC_Deactivate ( m_pDataParams );
        }

        /// <summary>
        /// The function checks if a card is present at the contact interface.
        /// </summary>
        ///
        /// <param name="bCardInserted"></param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_PresenceCheck ( out byte bCardInserted )
        {
            bCardInserted = 0;
            return phhalHw_DUT_Cmd_CC_PresenceCheck ( m_pDataParams, ref bCardInserted );
        }

        /// <summary>
        /// The function transmits data to a Contact Card.
        /// </summary>
        ///
        /// <param name="aTxBuffer">Buffer containing the data to be transmitted to Contact Card.</param>
        /// <param name="aRxBuffer">Buffer containing the received data from the Contact Card.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_TransmitData ( byte[] aTxBuffer, out byte[] aRxBuffer )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_CC_TransmitData ( m_pDataParams, aTxBuffer,
                (ushort) aTxBuffer.Length, ref pRxBuff, ref wRxLen);
            aRxBuffer = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// The function performs protocol and parameter exchange(PPS) with a contact card according to ISO7816-3.
        /// Supported are the following values:
        ///     - 0x12,  etu=186
        ///     0x13,  etu=93
        ///     0x18,  etu=31  speed 145.161 bit/s
        ///     0x95,  etu=32
        ///     0x96,  etu=16 - speed 281.250 bit/s
        ///
        ///
        /// The PPSS identifier byte, having value 0xff is not included in the<em> pPPS</em> buffer.
        ///     The first byte in the pPPS buffer,
        ///     i.e. <em>pPPS[0]</em> is PPS0.In case of set exchange speed <em>bOption= 0x01 </em>,
        ///     the first byte in the pPPS buffer,
        ///     i.e. <em>pPPS[0]</em> indicates the transmission factors Fi[b7 - b4], Di[b3 - b0].
        /// </summary>
        /// <param name="bOption">0x00 = Perform PPS, 0x01 = Set only the exchange speed.</param>
        /// <param name="aPPS">Buffer holding valid PPS exchange data. The presence of the optional bytes,
        ///                    respectively the length the PPS message is indicated by pPPS[0].
        /// </param>
        /// <param name="aResponse">Buffer where the received data from the Contact Card shall be stored.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t CC_SendPPS ( byte bOption, byte[] aPPS, out byte[] aResponse )
        {
            ushort wRspLen = 128;

            aResponse = new byte[wRspLen];
            GCHandle oResponse = GCHandle.Alloc ( aResponse, GCHandleType.Pinned );
            IntPtr pResponse = oResponse.AddrOfPinnedObject ();

            Status_t oStatus = phhalHw_DUT_Cmd_CC_SendPPS( m_pDataParams, bOption, aPPS, pResponse,
                ref wRspLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRspLen );

            pResponse = IntPtr.Zero;
            oResponse.Free ();
            return oStatus;
        }
        #endregion

#if PACKAGE_INTERNAL
        #region I2C Commands
        /// <summary>
        /// Perform Data Exchange with I2C slave device. This command is used to exchange information to I2C Slave
        /// device and at the same time receives the payload from I2C slave device if available including the Status.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to I2C Slave device and also
        ///           receives the payload from I2C Slave device if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command header of
        ///           I2C Slave device. Here the information is not exchanged with I2C Slave device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the Data part
        ///           for I2C Slave device (if any). Here the information is not exchanged with I2C Slave
        ///           device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part for I2C
        ///           Slave device (if any). Here the complete buffered information is exchanged to I2C Slave
        ///           device and also receives the information from I2C Slave device if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.TXCHAINING"/>
        ///             - To send any pending payload to DUT.
        ///             - If DUT requires more data, then DUT will respond with <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               Require More Data</see> as status for the current exchange call. The interface should be called
        ///               for this flag until DUT returns <see cref="Error_Gen.SUCCESS">Complete</see> as status.
        ///
        ///         - <see cref="ExchangeOptions.RXCHAINING"/>
        ///             - To receive any pending payload from DUT.
        ///             - To start using this flag, DUT will return <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               More Data Available</see> as status during previous reception. Payload should be
        ///               received until <see cref="Error_Gen.SUCCESS">Complete</see> as status is returned
        ///               by DUT.
        ///
        ///     - wBytesToRead
        ///         - To receive a known amount of bytes use <see cref="ExchangeOptions.BYTES_TO_READ"/> available flag.
        ///         - Leave <see cref="ExchangeOptions.BYTES_TO_READ"/> if the amount of bytes to read is not known.
        ///         - <see cref="ExchangeOptions.BYTES_TO_READ"/> flag can be ORed with one of the options mentioned in
        ///           <b>Normal Exchange</b> and <b>Chained Exchange</b>
        ///
        /// NOTE
        ///     Before calling this interface the setting of I2C must be configured using
        ///     <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange. Refer description for more details.</param>
        /// <param name="wSlaveAddr">Slave address of the Device to perform the Exchange.
        ///                             - 7Bit Addressing: Will have one byte address with LSB bit representing the
        ///                               Read or Write operation. Ex: WriteAddress: 0x92, ReadAddress: 0x93
        /// </param>
        /// <param name="aTxBuff">Payload information to be exchanged / buffers based on \b wOption information.</param>
        /// <param name="wBytesToRead"> Number of Bytes to read from I2C Device.</param>
        /// <param name="aRxBuff">Information provided by I2C Slave device.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer
        ///                                                      than the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_Exchange ( ushort wOption, ushort wSlaveAddr, byte[] aTxBuff, ushort wBytesToRead,
            out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_I2C_Exchange ( m_pDataParams, wOption, wSlaveAddr, aTxBuff,
                ( ushort ) ( ( aTxBuff == null ) ? 0 : aTxBuff.Length ), wBytesToRead, ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// Perform Transmission of data to I2C slave device. This command is used to exchange information
        /// to I2C Slave device.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to I2C Slave device and also
        ///           receives the payload from I2C Slave device if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command header of I2C
        ///           Slave device. Here the information is not exchanged with I2C Slave device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the Data part for
        ///           I2C Slave device (if any). Here the information is not exchanged with I2C Slave device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part for I2C Slave
        ///           device (if any). Here the complete buffered information is exchanged to I2C Slave device and
        ///           also receives the information from I2C Slave device if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.TXCHAINING"/>
        ///             - To transmit any pending payload to DUT.
        ///             - If DUT requires more data, then DUT will respond with <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               Require More Data</see> as status for the current exchange call. The interface should be called
        ///               for this flag until DUT returns <see cref="Error_Gen.SUCCESS">Complete</see> as status.
        ///
        /// NOTE
        ///     Before calling this interface the setting of I2C must be configured using <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange. Refer description for more details.</param>
        /// <param name="wSlaveAddr">Slave address of the Device to perform the Exchange.
        ///                             - 7Bit Addressing: Will have one byte address with LSB bit representing the Read
        ///                               or Write operation. Ex: WriteAddress: 0x92, ReadAddress: 0x93
        /// </param>
        /// <param name="aTxBuff">Payload information to be exchanged / buffers based on \b wOption information.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_Transmit ( ushort wOption, ushort wSlaveAddr, byte[] aTxBuff )
        {
            return phhalHw_DUT_Cmd_I2C_Transmit ( m_pDataParams, wOption, wSlaveAddr, aTxBuff,
                ( ushort ) ( ( aTxBuff == null ) ? 0 : aTxBuff.Length ) );
        }

        /// <summary>
        /// Perform Reception of Data from I2C Slave device. This command is used to receive information to I2C Slave device.
        ///
        /// wOption can be one of:
        ///     - Normal Exchange
        ///         - <see cref="ExchangeOptions.DEFAULT"/>
        ///           To frame and exchange the complete payload information to I2C Slave device and also
        ///           receives the payload from I2C Slave device if available.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///           To buffer initial payload information. This payload can have the Command header of I2C
        ///           Slave device. Here the information is not exchanged with I2C Slave device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///           To buffer intermediate payload information.This payload contains only the Data part for
        ///           I2C Slave device (if any). Here the information is not exchanged with I2C Slave device.
        ///
        ///         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///           To buffer final payload information.This payload contains only the Data part for I2C Slave
        ///           device (if any). Here the complete buffered information is exchanged to I2C Slave device and
        ///           also receives the information from I2C Slave device if available.
        ///
        ///     - Chained Exchange
        ///         - <see cref="ExchangeOptions.RXCHAINING"/>
        ///             - To receive any pending payload from DUT.
        ///             - To start using this flag, DUT will return <see cref="Error_Gen.SUCCESS_CHAINING">
        ///               More Data Available</see> as status during previous reception. Payload should be
        ///               received until <see cref="Error_Gen.SUCCESS">Complete</see> as status is returned
        ///               by DUT.
        ///
        ///     - wBytesToRead
        ///         - To receive a known amount of bytes use <see cref="ExchangeOptions.BYTES_TO_READ"/> available flag.
        ///         - Leave <see cref="ExchangeOptions.BYTES_TO_READ"/> if the amount of bytes to read is not known.
        ///         - <see cref="ExchangeOptions.BYTES_TO_READ"/> flag can be ORed with one of the options
        ///           mentioned in <b>Normal Exchange</b> and <b>Chained Exchange</b>
        ///
        /// NOTE
        ///     Before calling this interface the setting of I2C must be configured using
        ///     <see cref="SetConfig"/> command.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to perform exchange. Refer description for more details.</param>
        /// <param name="wSlaveAddr">Slave address of the Device to perform the Exchange.
        ///                             - 7Bit Addressing: Will have one byte address with LSB bit representing
        ///                               the Read or Write operation. Ex: WriteAddress: 0x92, ReadAddress: 0x93
        /// </param>
        /// <param name="wBytesToRead">Number of Bytes to read from I2C Device.</param>
        /// <param name="aRxBuff">Information provided by I2C Slave device.</param>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_Receive ( ushort wOption, ushort wSlaveAddr, ushort wBytesToRead, out byte[] aRxBuff )
        {
            IntPtr pRxBuff = IntPtr.Zero;
            ushort wRxLen = 0;

            Status_t oStatus = phhalHw_DUT_Cmd_I2C_Receive ( m_pDataParams, wOption, wSlaveAddr, wBytesToRead,
                ref pRxBuff, ref wRxLen );
            aRxBuff = MarshalCopy ( oStatus, pRxBuff, wRxLen );

            return oStatus;
        }

        /// <summary>
        /// Perform I2C Start Sequence.
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_Start ()
        {
            return phhalHw_DUT_Cmd_I2C_Start ( m_pDataParams );
        }

        /// <summary>
        /// Perform I2C Stop Sequence.
        ///
        /// Note:
        ///     - Post this interface call any of the below interfaces to perform exchanges.
        ///         - <see cref="I2C_Exchange"/>
        ///         - <see cref="I2C_Transmit"/>
        ///         - <see cref="I2C_Receive"/>
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_Stop ()
        {
            return phhalHw_DUT_Cmd_I2C_Stop ( m_pDataParams );
        }

        /// <summary>
        /// Perform I2C Repeated Start Sequence.
        ///
        /// Note:
        ///     - Post this interface call any of the below interfaces to perform exchanges.
        ///         - <see cref="I2C_Exchange"/>
        ///         - <see cref="I2C_Transmit"/>
        ///         - <see cref="I2C_Receive"/>
        /// </summary>
        ///
        /// <returns>
        ///     Return <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.INTERFACE_ERROR"/> Communication error.
        ///     Returns <see cref="Error_Comm.BUFFER_OVERFLOW"/> if Transmit data length is longer than
        ///                                                      the allocated transmit buffer size.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> for Invalid frame or unexpected receive
        ///                                                     data length or command code mismatch.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t I2C_RepeatedStart ()
        {
            return phhalHw_DUT_Cmd_I2C_RepeatedStart ( m_pDataParams );
        }
        #endregion
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public DUT ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~DUT ()
        {
            // Free Buffers
            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
            }
            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
            }
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        #endregion

        #region Private Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error_Gen.SUCCESS_CHAINING ) )
                && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( pBuffer != IntPtr.Zero )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }
        #endregion
    }
}
