/*
 * Copyright 2013 - 2020, 2024, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Mp300 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Mp300 : Hal.Generic
    {
        public const int DEFAULT_TIMEOUT_MS = 5;            /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0F;              /**< Number of shadowed configurations. */
        private const int RESERVED_BUFFER_LEN = 32;         /**< Amount of needed and reserved memory for the protocol overhead. */
        private const int RESERVED_BUFFER_CRC_LEN = 2;      /**< Amount of needed and reserved memory for the crc overhead. */
        private const int SERIAL_NUMBER_SIZE = 16;          /**< The size of the serial number. */
        private const int VERSION_STR_SIZE = 8;             /**< The size of the version strings. */
        private const int COUPLER_NAME_SIZE = 7;            /**< The size of the coupler name (longest = SWPSPY). */
        private const int COUPLER_SERIAL_NUMBER_SIZE = 9;   /**< The size of the coupler serial number. */
        private const int COUPLER_REVISION_SIZE = 5;        /**< The size of the coupler revision. */
        private const int COUPLER_CALC_DATE_SIZE = 4;       /**< The size used to store the calibration date + checksum */
        private const int MAX_SPY_FILENAME_SIZE = 256;      /**< Max size of the file name for spy functionality */

        public enum Config : int
        {
            FIELDSTRENGTH = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,                   /**< Set Fieldstrength in percent. */
            CARRIER_FREQUENCY_SHIFT = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+1,       /**< Configure Carrier frequency. */
            RX_CHANNEL = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+2,                    /**< Configure RX channel. */
            RELAY = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+3,                         /**< Configure Relay port. */
            FIELD_RISE_TIME = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+4,               /**< Configure Field rise time(only after Fieldstrength 0%). */
            FALL_TIME = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+5,                     /**< Configure Fall time of modulation. */
            RISE_TIME = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+6,                     /**< Configure Rise time of modulation. */
            PAUSE_WIDTH = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+7,                   /**< Configure Pausewidth of modulation. */
            COUPLER_NUMBER = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+8,                /**< Configure of Coupler. */
            RX_GAIN = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+9,                       /**< Configure of the gain of the receiver. */
            TYPE_B_SOF_LOW_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+10,        /**< Configure of the type B SOF low timing. */
            TYPE_B_SOF_HIGH_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+11,       /**< Configure of the type B SOF high timing. */
            TYPE_B_EGT_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+12,            /**< Configure of the type B EGT timing. */
            TYPE_B_EOF_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+13,            /**< Configure of the type B EOF timing. */
            ANTI_EMD = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 14,                   /**< Configure of Anti EMD. */
            FIELDSTRENGTH_THOUSAND = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 15,     /**< Configure Fieldstrength in Permil. */
            WAVEFORM_GENERATION_MODE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 16,   /**< Configure the generation mode of the waveform. */
            RX_GAIN_TARGET = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 17,             /**< Target value that should be reached with Adjust Rx (Default 750 but with card with low LMA and at high fieldstrength a higher value should be used */
            TYPE_B_START_BIT_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 18,    /**< Configure of the type B Start Bit timing. */
            TYPE_B_BIT_0_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 19,        /**< Configure of the type B Bit 0 timing. */
            TYPE_B_BIT_1_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 20,        /**< Configure of the type B Bit 1 timing. */
            TYPE_B_BIT_2_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 21,        /**< Configure of the type B Bit 2 timing. */
            TYPE_B_BIT_3_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 22,        /**< Configure of the type B Bit 3 timing. */
            TYPE_B_BIT_4_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 23,        /**< Configure of the type B Bit 4 timing. */
            TYPE_B_BIT_5_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 24,        /**< Configure of the type B Bit 5 timing. */
            TYPE_B_BIT_6_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 25,        /**< Configure of the type B Bit 6 timing. */
            TYPE_B_BIT_7_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 26,        /**< Configure of the type B Bit 7 timing. */
            TYPE_B_STOP_BIT_CYCLES = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 27,     /**< Configure of the type B Stop Bit timing. */
            SOURCE_OF_CARRIER = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 28,          /**< Configure of the carrier source. */
            MAX_FIELDSTRENGTH_THOUSAND = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 29, /**< Get the maximum possible Fieldstrength. */
            FORCE_FDT_METHOD_1 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 30,         /**< Configure that always method one of FDT is used. If PH_OFF then for type a the method 2 is used for type A. */
            RFAFTERTEARING = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 31,             /**< Configure of RF after Tearing. */
            TEARING_EXCHANGE_COUNTER = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 32    /**< Configure the exchange index for tearing functionality. */
        }

        public enum TriggerMode : int
        {
            TRIG_FORCE =          0x01,               /**< Force Triggerchannel */
            TRIG_TX_ON =          0x02,               /**< The trigger out will be set to a logic 0 during the transmission of a frame to the card. */
            TRIG_TX_OUT =         0x03,               /**< The trigger will represent the signal that is used to modulate the carrier. */
            TRIG_RX_ON =          0x06,               /**< The trigger out will be set to a logic 0 during the reception of a frame coming from the card. */
            TRIG_DELAY_AFTER_TX = 0x07,               /**< The Trigger is set to logic 0 after the transmission of the PCD and after a time specified with the Value Parameter it is set to logical 1 again. */
            TRIG_CARRIER =        0x08,               /**< Trigger will represent the logic representation of the carrier. */
            TRIG_EMD =            0x13                /**< The trigger out will be set to a logic 0 during the reception of a frame coming from the card. Do supress EMD spikes it starts after receiving a few symbols
                                                       * type A after 10 symbols
                                                       * type B after 9 bits
                                                       * type A HD after 8 bits
                                                       * Felica after 8 bits (at least FPGA 33 has to be used
                                                       */
        };

        public enum CouplerResourceId : int
        {
            TCL1 = 0x78,                           /**< Id of a MP300 TCL1 */
            TCL2 = 0x7A,                           /**< Id of a MP300 TCL2 */
            TCL3 = 0xF6                            /**< Id of a MP500 TCL3 */
        };

        public enum SourceOfCarrier : byte
        {
            INTERNAL = 0x01,                            /**< Internal programmable carrier (default). */
            EXTERNAL = 0x02,                            /**< Take carrier from logical signal from trigger. */
            INTERNAL_FIXED = 0x03                       /**<  Fixed 13.56 MHz. */
        };

        public enum TriggerPort : byte
        {
            TRIG_PORT_1 = 0x01,                       /**< Trigger Port 1 */
            TRIG_PORT_2 = 0x02,                       /**< Trigger Port 2 */
            TRIG_PORT_3 = 0x03,                       /**< Trigger Port 3 */
            TRIG_PORT_4 = 0x04                        /**< Trigger Port 4 */
        };

        public enum WaveformGenerationMode : byte
        {
            LINEAR_MODULATION = 0x01,           /**< Waveform generation linear */
            CUSTOMIZED_MODULATION = 0x02,       /**< Waveform generation with custom shape */
            CUSTOMIZED_MODULATION_CLK = 0x03,   /**< Waveform generation with custom shape in clk step size */
            CUSTOMIZED_MODULATION_10CLK = 0x04  /**< Waveform generation with custom shape in 1/10 clk step size */
        };

        public enum RxChannel : byte
        {
            RXCHANNEL_NORMAL = 0x01,                  /**< Use Normal Rx Channel (Tx and Rx are not seperated) */
            RXCHANNEL_EXTERNAL = 0x02                 /**< Use seperate Rx Channel */
        }

        public enum Relay : byte
        {
            ON  = 0x01,                               /**< Output +5 V to the Relay Port */
            OFF = 0x00                                /**< Disable +5 V Output to the Relay Port */
        }

        public enum License : byte
        {
            LICENSE_ISO             = 1,              /**< ISO */
            LICENSE_SIMULATOR       = 6,              /**< Card Emulation */
            LICENSE_VHBR            = 2,              /**< VHBR */
            LICENSE_125K            = 18,             /**< 125 kHz*/
            LICENSE_QI              = 19,             /**< Qi */
            LICENSE_ADVANCED_MEAS   = 21,             /**< Advanced Measurements */
            LICENSE_BIT_BOUNDARY    = 23              /**< Bit boundary */
        }

        public enum DisturbanceOperation : byte
        {
            SIGNAL_CARRIER_ADD = 0,              /**< add the carrier with the disturbing signal */
            SIGNAL_CARRIER_MUL = 1               /**< multiply the carrier with the disturbing signal */
        }

        public enum DisturbanceType : byte
        {
            RAMP = 0,              /**< Disturbing signal is a continuous signal with a single ramp */
            SQUARE = 1,            /**< Disturbing signal is a periodic square signal */
            SINE = 2,              /**< Disturbing signal is a periodic sinusoidal signal */
            GLITCH = 3,            /**< Disturbing signal is a continuous signal with a single glitch */
            SINE2 = 4              /**< Disturbing signal is a sinusoidal signal */
        }

        public enum DisturbanceTrigger : byte
        {
            TRIG_EVT_TRIG_FORCE = 100,              /**< immediately */
            TRIG_EVT_RF_FIELD_ON = 101,             /**< upon RF field activation */
            TRIG_EVT_RF_FIELD_OFF = 102,            /**< upon RF field deactivation */
            TRIG_EVT_TX_ON = 103,                   /**< upon start of transmission */
            TRIG_EVT_TX_OFF = 104,                  /**< upon end of transmission */
            TRIG_EVT_TX_ON_NFC = 105,               /**< upon start of reception */
            TRIG_EVT_FS_DETECT_ON = 106,            /**< upon start of subcarrier detection */
            TRIG_EVT_FS_DETECT_OFF = 107,           /**< upon end of subcarrier detection */
            TRIG_EVT_TRIGGER_IN = 108               /**< upon a rising edge on the trigger In (trigger 4) */
        }

        public enum TearingBehaviour : byte
        {
            FIELD_OFF               = 1,            /**< Field is always off after exchange (either tearing occured or field is switched of after exchange */
            FIELD_ON                = 2,            /**< Field is always in after exchange (either tearing occured and field is switched on again of a HF is performed after exchange */
            FIELD_OFF_AFTER_TEARING = 3,            /**< Field stay if after tearing event, no change in the field when no tearing occure */
            FIELD_ON_AFTER_TEARING  = 4,            /**< Field is switched on after tearing event, no change in the field when no tearing occure */
        }

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                      /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;                           /**< pointer to the lower layers parameter structure */
            public IntPtr pTxBuffer;                                /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                               /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                                /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                                /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                               /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                                /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                           /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                           /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public byte bCardType;                                  /**< Type of card for which the hal is configured for */
            public fixed ushort wCfgShadow[SHADOW_COUNT];           /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                              /**< Current timing measurement mode. */
            public byte bTimeoutUnit;                               /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public ulong qwTiming_ns;                               /**< Current timing value. */
            public ulong qwCommunicationTimePCDtoPICCCyclesAll;     /**< Sum of time for communication between PCD and PICC. */
            public ulong qwCommunicationTimePICCtoPCDCyclesAll;     /**< Sum of time for communication between PICC and PCD. */
            public ulong qwCommunicationTimePCDtoPICCCyclesCurrent; /**< Time for communication between PCD and PICC of the last exchange. */
            public ulong qwCommunicationTimePICCtoPCDCyclesCurrent; /**< Time for communication between PICC and PCD of the last exchange. */
            public byte bLastBit;                                   /**< Last bit exchanged (0 if last bit was Zero 1 if last bit was One. */
            public ushort wAdditionalInfo;                          /**< storage for additional error information */
            public ushort wFieldOffTime;                            /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;                       /**< Field-Recovery-Time in milliseconds. */
            public byte bSymbolStart;                               /**< Preamble of a frame. */
            public byte bSymbolEnd;                                 /**< Trailer symbol of a frame. */
            public ushort wCurrentFieldStrength;                    /**< Current Fieldstrength in Permil. */
            public byte bRfResetAfterTo;                            /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public byte bRfAfterTearing;                            /**< Storage for additional #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            ushort wTearingExchangeCounter;                         /**< Counter decreasing at each exchange until desired exchange is reached for tearing. */
            IntPtr pIntBuffer;                                      /**< Internal buffer used as a scratch buffer. Size must be wIntBufferLen*wIntBufferNum */
            ushort wIntBufferSize;                                  /**< Total length of internal buffer used as a scratch buffer */
            ushort wIntBufferLen;                                   /**< Flag to show the status of buffer allocation. */
            public ushort wIntBufferReadPos;                        /**< Read position within \c pIntBuffer for protocol. */
            public int dwCarrierFrequency;                          /**< Current Carrier frequency in use. */
            public byte bRxGain;                                    /**< Storage for Rx Gain setting. */
            public byte bRXChannel;                                 /**< Storage for the current RX Channel Setting. */
            public byte bCouplerNumber;                             /**< Select the Coupler used for communication */
            public ushort wFieldRiseTime;                           /**< Storage for the field rise time. */
            public byte bRelay;                                     /**< Storage for Relay PH_ON or PH_OFF.*/
            public byte bModulationMode;                            /**< Lower nibble stores the modulation mode upper nibble stores if the custom shape is loaded */
            public ushort wFallTime;                                /**< Storage of Fall time of modulation.*/
            public ushort wRiseTime;                                /**< Storage of Rise time of modulation.*/
            public ushort wPauseWidth;                              /**< Storage of Pause width of modulation(Type A only).*/
            public byte bSourceOfCarrier;                           /**< Storage for the source of the carrier.*/
            public byte bSubCarrierMode;                            /**< Storage for the current subcarrier mode for 15693.*/
            public fixed ushort wTypeBTimingParameter[14];          /**< Storage of the Type B modulation timing parameter.*/
            public ushort wCouplerResourceId;                       /**< Storage of the resource ID of the Coupler to know the type of Reader (TCL1 or TCL2 supported).*/
            public byte bAntiEMD;                                   /**< Storage for Anti EMD PH_ON or PH_OFF.*/
            public byte bForceFdtMethod1;                           /**< Storage for Force Fdt Method 1 PH_ON or PH_OFF.*/
            public ushort wTargetRxGainLevel;                       /**< Value that should be reached with Adjust Rx .. Default is 750 but if PICC has low LMA a higher value should be used in high field */
            public fixed byte bSerialNumber[SERIAL_NUMBER_SIZE];    /**< This value stores the serial number of the reader */
            public fixed byte bDeviceFamily[VERSION_STR_SIZE];                     /**< This value stores the DeviceFamily of the reader */
            public fixed byte bDeviceReferenceWithinFamily[VERSION_STR_SIZE];      /**< This value stores the DeviceReferenceWithinFamily of the reader */
            public fixed byte bMotherboardSystemVersion[VERSION_STR_SIZE];         /**< This value stores the MotherboardSystemVersion of the reader */
            public fixed byte bMotherboardBootVersion[VERSION_STR_SIZE];           /**< This value stores the MotherboardBootVersion of the reader */
            public fixed byte bCouplerName[COUPLER_NAME_SIZE];                     /**< This value stores the CouplerName of the reader */
            public fixed byte bCouplerSerialNumber[COUPLER_SERIAL_NUMBER_SIZE];    /**< This value stores the CouplerSerialNumber of the reader */
            public fixed byte bCouplerRevision[COUPLER_REVISION_SIZE];             /**< This value stores the CouplerRevision of the reader */
            public fixed byte bCouplerDriverVersion[VERSION_STR_SIZE];             /**< This value stores the CouplerDriverVersion of the reader */
            public ushort wCouplerFPGAVersion;                                     /**< This value stores the CouplerFPGAVersion of the reader */
            public fixed byte bCouplerCalibrationDate[COUPLER_CALC_DATE_SIZE];     /**< Date where the till the coupler is calibrated */
            public uint dwScenarioID;                                              /**< ID of the current in use scenario */
            public uint dwScenarioPcdId;                                           /**< ID of the current in use scenario for card emulation */
            public IntPtr pBalTimeoutHandling;                                     /**< Pointer to the handler function. */
            public IntPtr pBalTimeoutHandlingContext;                              /**< Context that can be used with handler function. */
            public fixed byte bSpyFilename[MAX_SPY_FILENAME_SIZE];                 /**< Filename of the spied Tracefile */
            public IntPtr SpyThreadHandle;                                         /**< Handle of the SpyThread */
            public IntPtr GEvtFinished;                                            /**< Event to check if SpyThread is ready */
            public IntPtr GEvtReady;                                               /**< Event to check if SpyThread has finished */
            public fixed ushort wTriggerExchangeCounters[4];                       /**< Array of counters used to count the exchanges before trigger is set. */
            public fixed int bTriggerConfigs[4];                                   /**< Array of configs for each channel. */
            public fixed int bTriggerValues[4];                                    /**< Array of values for trigger configs. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize,              /**< [In] Size of the global receive buffer. */
            IntPtr pInternalBuffer,         /**< [In] Internal buffer used as a scratch buffer */
            ushort wInternalBufferLen       /**< [In] Length of innternal buffer used as a scratch buffer */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_TriggerOut(
            IntPtr pDataParams,
            byte bTriggerPort,
            uint dwConfig,
            uint dwValue
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_AdjustRXChannel(
            IntPtr pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_SetModulationShape(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwNbPointsFallingEdge,       /**< [In] Number of points of falling edge. */
            ushort[] pPointsFallingEdge,      /**< [In] points of falling edge. In a range of 0..1000 */
            uint dwNbPointsRisingEdge,        /**< [In] Number of points of rising edge. */
            ushort[] pPointsRisingEdge        /**< [In] points of rising edge. In a range of 0..1000 */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_OpenScenario(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_ExecuteScenario(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_CloseScenario(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_AddToScenario(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwActionID,                  /**< [In] ID of the action that should be added */
            char[] pParameter,                /**< [In] string with parameter for the action. */
            ushort wParameterLength           /**< [In] Length of the parameter string. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_SetupCardEmulationMode(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwAdjustTimeout,             /**< [In] Timeout used to adjust the input channel */
            uint dwMaxRetryCnt                /**< [In] number of retries performed if adjust failed */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_CloseCardEmulationMode(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_OpenScenarioPcd(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_ExecuteScenarioPcd(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwTimeout                    /**< [In] Timeout for execution of the scenario in ms. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_CloseScenarioPcd(
            IntPtr pDataParams                /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_AddToScenarioPcd(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwActionId,                  /**< [In] ID of the action that should be added */
            char[] pParameter,                /**< [In] string with parameter for the action. */
            ushort wParameterLength           /**< [In] Length of the parameter string. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_CouplerCheckLicense(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwLicenseId,                 /**< [In] Specifies the license. */
            ref int pValidLicense             /**< [Out] license status. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_SetBalTimeoutHandler(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            IntPtr pBalTimeoutHandling,       /**< [In] Pointer to the callback function */
            IntPtr pContext                   /**< [In] Pointer to the context of the callback function */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_EnableSpy(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            uint dwEventMask,               /**< [In] EventMask to chose Events to spy. */
            char[] pFileName,               /**< [In] Filename for .mplog File. */
            ushort wDateFormat              /**< [In] Date Format of Trace. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_DisableSpy(
            IntPtr pDataParams              /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_GenerateDisturbance(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bDisturbanceOperation,
            byte bDisturbanceType,
            int dwAmplitude,
            short wOffset,
            uint dwDuration,
            uint dwMaintainLevel
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_ResetDisturbance(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bDisturbanceOperation
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_SetDisturbanceTrigger(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bDisturbanceOperation,
            uint dwBehavior,
            uint dwDelay,
            ushort wLoop
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Mp300_Cmd_SetAdvancedTrigger(
            IntPtr pDataParams,
            byte bTriggerPort,
            uint dwConfig,
            uint dwValue,
            ushort wExchangeCount
            );
        #endregion

        #region DELEGATES
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate Status_t BalTimeoutDelegate(
            IntPtr pContext,        /**< [Out] User Defined Context */
            byte bCouplerNumber     /**< Number of the used coupler */
            );
        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;
        private byte[] m_bIntBuffer;
        private GCHandle m_pIntBuffer;
        private GCHandle contextHandle;
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_CRC_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_CRC_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_bIntBuffer = new byte[wIntBufferSize];

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(m_bIntBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Mp300_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (ushort)wIntBufferSize);
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_CRC_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_CRC_LEN;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_bIntBuffer = new byte[wIntBufferSize];

            return Init(wDataParamSize, pBal, wTxBufferSize, m_bTxBuffer, wRxBufferSize, m_bRxBuffer, wIntBufferSize, m_bIntBuffer);
        }
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            m_bIntBuffer = new byte[wIntBufferSize];

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(m_bIntBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Mp300_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (ushort)wIntBufferSize);
        }
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer, int wIntBufferSize, byte[] bIntBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(bIntBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Mp300_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (ushort)wIntBufferSize);
        }
#endif
        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_TriggerOut(
            TriggerPort bTriggerPort,
            TriggerMode dwConfig,
            uint dwValue
            )
        {
            return phhalHw_Mp300_Cmd_TriggerOut(m_pDataParams, (byte)bTriggerPort, (uint)dwConfig, dwValue);
        }

        public Status_t Cmd_AdjustRXChannel()
        {
            return phhalHw_Mp300_Cmd_AdjustRXChannel(m_pDataParams);
        }

        public Status_t Cmd_SetModulationShape(
            ushort[] pPointsFallingEdge,
            ushort[] pPointsRisingEdge
            )
        {
            return phhalHw_Mp300_Cmd_SetModulationShape(m_pDataParams,
                (uint)pPointsFallingEdge.Length, pPointsFallingEdge,
                (uint)pPointsRisingEdge.Length, pPointsRisingEdge);
        }

        public Status_t Cmd_OpenScenario()
        {
            return phhalHw_Mp300_Cmd_OpenScenario(m_pDataParams);
        }

        public Status_t Cmd_ExecuteScenario()
        {
            return phhalHw_Mp300_Cmd_ExecuteScenario(m_pDataParams);
        }

        public Status_t Cmd_CloseScenario()
        {
            return phhalHw_Mp300_Cmd_CloseScenario(m_pDataParams);
        }

        public Status_t Cmd_AddToScenario(
            uint dwActionID,
            string pParameter
            )
        {
            char[] pParameterArray = pParameter.ToCharArray();
            return phhalHw_Mp300_Cmd_AddToScenario(m_pDataParams,
                dwActionID,
                pParameterArray,
                (ushort)pParameterArray.Length);
        }

        public Status_t Cmd_SetupCardEmulationMode(
            uint dwAdjustTimeout, /**< [In] Timeout used to adjust the input channel */
            uint dwMaxRetryCnt /**< [In] number of retries performed if adjust failed */
            )
        {
            return phhalHw_Mp300_Cmd_SetupCardEmulationMode(m_pDataParams,
                dwAdjustTimeout,
                dwMaxRetryCnt);
        }

        public Status_t Cmd_CloseCardEmulationMode()
        {
            return phhalHw_Mp300_Cmd_CloseCardEmulationMode(m_pDataParams);
        }

        public Status_t Cmd_OpenScenarioPcd()
        {
            return phhalHw_Mp300_Cmd_OpenScenarioPcd(m_pDataParams);
        }

        public Status_t Cmd_ExecuteScenarioPcd(
            uint dwTimeout                           /**< [In] Timeout for execution of the scenario in ms. */
            )
        {
            return phhalHw_Mp300_Cmd_ExecuteScenarioPcd(m_pDataParams, dwTimeout);
        }

        public Status_t Cmd_CloseScenarioPcd(
            )
        {
            return phhalHw_Mp300_Cmd_CloseScenarioPcd(m_pDataParams);
        }

        public Status_t Cmd_AddToScenarioPcd(
            uint dwActionId,
            string pParameter
            )
        {
            char[] pParameterArray = pParameter.ToCharArray();
            return phhalHw_Mp300_Cmd_AddToScenarioPcd(m_pDataParams,
                dwActionId,
                pParameterArray,
                (ushort)pParameterArray.Length);
        }

        public Status_t Cmd_CouplerCheckLicense(
            License licenseId,      /**< [In] Specifies the license. */
            out int pValidLicense   /**< [Out] license status. */
            )
        {
            pValidLicense = 0;

            int validLicense = 0;
            var status = phhalHw_Mp300_Cmd_CouplerCheckLicense(m_pDataParams, (uint)licenseId, ref validLicense);
            if (status.Equals(new Status_t()))
                pValidLicense = validLicense;
            return status;
        }

        public Status_t Cmd_SetBalTimeoutHandler(BalTimeoutDelegate pBalTimeoutHandling, object context)
        {
            if (contextHandle.IsAllocated)
            {
                contextHandle.Free();
            }
            contextHandle = GCHandle.Alloc(context);
            return phhalHw_Mp300_Cmd_SetBalTimeoutHandler(m_pDataParams,
                Marshal.GetFunctionPointerForDelegate(pBalTimeoutHandling), (IntPtr)contextHandle);
        }

        public Status_t Cmd_EnableSpy(
            uint dwEventMask,
            string pFilename,
            ushort wDateFormat
            )
        {
            pFilename += "\0";
            char[] pFileNameArray = pFilename.ToCharArray();

            return phhalHw_Mp300_Cmd_EnableSpy(m_pDataParams,
                dwEventMask,
                pFileNameArray,
                wDateFormat);
        }

        public Status_t Cmd_DisableSpy()
        {
            return phhalHw_Mp300_Cmd_DisableSpy(m_pDataParams);
        }

        public Status_t Cmd_GenerateDisturbance(
            DisturbanceOperation bDisturbanceOperation,
            DisturbanceType bDisturbanceType,
            int dwAmplitude,
            short wOffset,
            uint dwDuration,
            bool bMaintainLevel
            )
        {
            return phhalHw_Mp300_Cmd_GenerateDisturbance(m_pDataParams,
                (byte)bDisturbanceOperation,
                (byte)bDisturbanceType,
                dwAmplitude,
                wOffset,
                dwDuration,
                bMaintainLevel ? (uint)1 : (uint)0);
        }

        public Status_t Cmd_ResetDisturbance(
            DisturbanceOperation bDisturbanceOperation
            )
        {
            return phhalHw_Mp300_Cmd_ResetDisturbance(m_pDataParams,
                (byte)bDisturbanceOperation);
        }

        public Status_t Cmd_SetDisturbanceTrigger(
            DisturbanceOperation bDisturbanceOperation,
            DisturbanceTrigger dwBehavior,
            uint dwDelay,
            ushort wLoop
            )
        {
            return phhalHw_Mp300_Cmd_SetDisturbanceTrigger(m_pDataParams,
                (byte)bDisturbanceOperation,
                (uint)dwBehavior,
                dwDelay,
                wLoop);
        }

        public Status_t Cmd_SetAdvancedTrigger(
            TriggerPort bTriggerPort,
            TriggerMode dwConfig,
            uint dwValue,
            ushort wExchangeCount
            )
        {
            return phhalHw_Mp300_Cmd_SetAdvancedTrigger(m_pDataParams, (byte)bTriggerPort, (uint)dwConfig, dwValue, wExchangeCount);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Mp300()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Mp300()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (contextHandle.IsAllocated)
            {
                contextHandle.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Setter & Getter for DataParams structure
        public string Serial
        {
            get
            {
                char[] serial = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < SERIAL_NUMBER_SIZE; i++)
                        {
                            if (pDataParams->bSerialNumber[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        serial = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            serial[i] = (char)pDataParams->bSerialNumber[i];
                        }
                    }
                }
                string serialString = new string(serial);
                if (serialString.StartsWith("MP3") && serialString.Length == 9)
                {
                    return "MP3." + serialString.Substring(3,2) + "." +serialString.Substring(5,2) + "." +serialString.Substring(7,2);
                }
                return serialString;
            }
        }
        public DateTime CalibrationDate
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        if (pDataParams->bCouplerCalibrationDate[0] == 0xFF &&
                            pDataParams->bCouplerCalibrationDate[1] == 0xFF &&
                            pDataParams->bCouplerCalibrationDate[2] == 0xFF)
                            return DateTime.MinValue;

                        DateTime date = new DateTime(2000 + pDataParams->bCouplerCalibrationDate[2],
                            pDataParams->bCouplerCalibrationDate[1],
                            pDataParams->bCouplerCalibrationDate[0]);
                        return date;
                    }
                }
            }
        }

        unsafe  private string ByteArrayToString(byte* arr, int maxLength)
        {
            char [] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }

        public string ReaderDescription
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = Serial;
                        value += " MotherboardSystem:" + ByteArrayToString(pDataParams->bMotherboardSystemVersion, VERSION_STR_SIZE);
                        value += " MotherboardBootVersion:" + ByteArrayToString(pDataParams->bMotherboardBootVersion, VERSION_STR_SIZE);
                        value += " Coupler#:" + pDataParams->bCouplerNumber.ToString();
                        value += " CouplerName:" + ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                        value += " CouplerSerialNumber:" + ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                        value += " CouplerRevision:" + ByteArrayToString(pDataParams->bCouplerRevision, COUPLER_REVISION_SIZE);
                        value += " CouplerDriverVersion:" + ByteArrayToString(pDataParams->bCouplerDriverVersion, VERSION_STR_SIZE);
                        value += " CouplerFPGAVersion:" + pDataParams->wCouplerFPGAVersion.ToString();
                        if( CalibrationDate != DateTime.MinValue)
                            value += " CouplerCalibrationDate:" + CalibrationDate.ToString("yyyy-MM-dd");
                        return value;
                    }
                }
            }
        }

        public string ReaderVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = "";// Serial;
                        value += " MotherboardSystem:" + ByteArrayToString(pDataParams->bMotherboardSystemVersion, VERSION_STR_SIZE);
                        value += " MotherboardBootVersion:" + ByteArrayToString(pDataParams->bMotherboardBootVersion, VERSION_STR_SIZE);
                        value += " Coupler#:" + pDataParams->bCouplerNumber.ToString();
                        value += " CouplerName:" + ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                        //value += " CouplerSerialNumber:" + ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                        value += " CouplerRevision:" + ByteArrayToString(pDataParams->bCouplerRevision, COUPLER_REVISION_SIZE);
                        value += " CouplerDriverVersion:" + ByteArrayToString(pDataParams->bCouplerDriverVersion, VERSION_STR_SIZE);
                        value += " CouplerFPGAVersion:" + pDataParams->wCouplerFPGAVersion.ToString();
                        //if (CalibrationDate != DateTime.MinValue)
                        //    value += " CouplerCalibrationDate:" + CalibrationDate.ToString("yyyy-MM-dd");
                        return value;
                    }
                }
            }
        }

        public string CouplerName
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                    }
                }
            }
        }

        public string DeviceFamily
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bDeviceFamily, VERSION_STR_SIZE) +
                            ByteArrayToString(pDataParams->bDeviceReferenceWithinFamily, VERSION_STR_SIZE);
                    }
                }
            }
        }

        public string CouplerSerial
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                    }
                }
            }
        }

        #endregion
    }
}
