/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// RC523 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Rc523 : Hal.Generic
    {
        #region DEFINES

        public const byte FIFOSIZE = 64;        /**< Size of RC internal FiFo buffer    */
        public const int DEFAULT_TIMEOUT = 150; /**< Default timeout in microseconds    */
        public const int SHADOW_COUNT = 0x0F;   /**< Number of shadowed configurations. */

        public enum RxTxOption : byte
        {
            DEFAULT = 0x00,
            TIMER_NOSTART = 0x00,
            TIMER_START = 0x10
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                              /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pBalDataParams;                   /**< pointer to the lower layers parameter structure. */
            public IntPtr pKeyStoreDataParams;              /**< Pointer to the parameter structure of the keyStore layer. */
            public IntPtr pTxBuffer;                        /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                       /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                        /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                        /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                       /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                        /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                   /**< Starting position within the global receive buffer. */
            public ushort wMaxPrecachedBytes;               /**< Holds the max. number of bytes which are precached prior to command execution. */
            public byte bCardType;                          /**< Type of card for which the hal is configured for. */
            public fixed ushort wCfgShadow[SHADOW_COUNT];   /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                      /**< Current timing measurement mode. */
            public byte bTimeoutUnit;                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public uint dwTimingUs;                         /**< Current timing value. */
            public ushort wFieldOffTime;                    /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;               /**< Field-Recovery-Time in milliseconds. */
            public ushort wAdditionalInfo;                  /**< storage for additional error information. */
            public byte bBalConnectionType;                 /**< Type of the underlying BAL connection. Refer to #PHHAL_HW_CONFIG_BALCONNECTION. */
            public byte bRfResetAfterTo;                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public ushort wFdtPc;                           /**< Current timing value backup for PC*/
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pKeyStoreDataParams,     /**< [In] Pointer to the parameter structure of the keyStore layer. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize               /**< [In] Size of the global receive buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_Mem(
            IntPtr pDataParams,
            byte[] pDataIn,
            byte bDataLength,
            byte[] pDataOut
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_Config(
            IntPtr pDataParams,
            byte[] pSensRes,
            byte[] pNfcId1,
            byte SelRes,
            byte[] pPollingResp,
            byte bNfcId3
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_GenerateRandomId(IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_CalcCrc(
            IntPtr pDataParams,
            byte[] pData,
            ushort wDataLength,
            ref ushort pCrcResult
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_Transmit(
            IntPtr pDataParams,
            ushort wOption,
            byte[] pTxBuffer,
            ushort wTxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_Receive(
            IntPtr pDataParams,
            ushort wOption,
            ref IntPtr ppRxBuffer,
            ref ushort pRxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_Autocoll(
            IntPtr pDataParams,
            ref IntPtr ppRxBuffer,
            ref ushort pRxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc523_Cmd_SoftReset(IntPtr pDataParams);

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, KeyStore.Generic pKeyStore, int wTxBufferSize, int wRxBufferSize)
        {
            IntPtr pKeyStoreDataParams;
            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            // Add 1 SPI reserved byte
            if (wTxBufferSize < 0xFFFF)
            {
                ++wTxBufferSize;
            }
            if (wRxBufferSize < 0xFFFF)
            {
                ++wRxBufferSize;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rc523_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, KeyStore.Generic pKeyStore, int wTxBufferSize, byte[] m_bTxBuffer, int wRxBufferSize, byte[] m_bRxBuffer)
        {
            IntPtr pKeyStoreDataParams;
            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rc523_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_Mem(
            byte[] pDataIn,
            byte bDataLength,
            out byte[] pDataOut)
        {
            Status_t status;
            byte[] pData = new byte[25];

            status = phhalHw_Rc523_Cmd_Mem(m_pDataParams, pDataIn, bDataLength, pData);
            pDataOut = pData;
            return status;
        }

        public Status_t Cmd_Config(
            byte[] pSensRes,
            byte[] pNfcId1,
            byte SelRes,
            byte[] pPollingResp,
            byte bNfcId3
            )
        {
            return phhalHw_Rc523_Cmd_Config(m_pDataParams, pSensRes, pNfcId1, SelRes, pPollingResp, bNfcId3);
        }

        public Status_t Cmd_GenerateRandomId()
        {
            return phhalHw_Rc523_Cmd_GenerateRandomId(m_pDataParams);
        }

        public Status_t Cmd_CalcCrc(
            byte[] pData,
            ushort wDataLength,
            ref ushort pCrcResult
)
        {
            return phhalHw_Rc523_Cmd_CalcCrc(
                m_pDataParams,
                pData,
                wDataLength,
                ref pCrcResult);
        }

        public Status_t Cmd_Transmit(
            int wOption,
            byte[] pTxBuffer
            )
        {
            ushort wTxLength = 0;

            if (pTxBuffer != null)
            {
                wTxLength = (ushort)pTxBuffer.Length;
            }
            return phhalHw_Rc523_Cmd_Transmit(m_pDataParams, (ushort)wOption, pTxBuffer, wTxLength);
        }

        public Status_t Cmd_Receive(
            int wOption,
            out byte[] pRxBuffer
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHw_Rc523_Cmd_Receive(m_pDataParams, (ushort)wOption, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Cmd_Autocoll(
            out byte[] pRxBuffer
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHw_Rc523_Cmd_Autocoll(m_pDataParams, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Cmd_SoftReset()
        {
            return phhalHw_Rc523_Cmd_SoftReset(m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rc523()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rc523()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
