/*
 * Copyright 2013, 2015 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// SamAV2 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class RdCardSim : Hal.Generic
    {
        public enum Config : int
        {
          COMMUNICATION_CONFIG = CustomCodes.CONFIG_BEGIN,  /**< Change the setting for the Socket communication (only for RdCardSim in combination with testpad communication). */
          TIME_T1 = CustomCodes.CONFIG_BEGIN + 1,               /**< Change the additional timeout T1 for the Socket communication (only for RdCardSim in combination with testpad communication). */
          TIME_T2 = CustomCodes.CONFIG_BEGIN + 2,               /**< Change the additional timeout T2 for the Socket communication (only for RdCardSim in combination with testpad communication). */
          TIME_T3 = CustomCodes.CONFIG_BEGIN + 3,               /**< Change the additional timeout T3 for the Socket communication (only for RdCardSim in combination with testpad communication). */
          TIME_T4 = CustomCodes.CONFIG_BEGIN + 4,               /**< Change the additional timeout T4 for the Socket communication (only for RdCardSim in combination with testpad communication). */
          SYNC_OPTION = CustomCodes.CONFIG_BEGIN + 5,            /**< Change the optional Sync option for the Socket communication (only for RdCardSim in combination with testpad communication). */
          ENV_ENABLE = CustomCodes.CONFIG_BEGIN + 6,
          UT_FI         = CustomCodes.CONFIG_BEGIN + 7,
          UT_DI         = CustomCodes.CONFIG_BEGIN + 8,
          UT_ETU_CLK    = CustomCodes.CONFIG_BEGIN + 9,
          UT_STOPBITS   = CustomCodes.CONFIG_BEGIN + 10,
          UT_CWI        = CustomCodes.CONFIG_BEGIN + 11,
          UT_BWI        = CustomCodes.CONFIG_BEGIN + 12,
          UT_FWI        = CustomCodes.CONFIG_BEGIN + 13,
          UT_WI         = CustomCodes.CONFIG_BEGIN + 14,
          UT_T0         = CustomCodes.CONFIG_BEGIN + 15,
          UT_T1         = CustomCodes.CONFIG_BEGIN + 16,
          UT_RSTTYPE    = CustomCodes.CONFIG_BEGIN + 17,
          UT_CRC_LRC    = CustomCodes.CONFIG_BEGIN + 18,
          UT_N          = CustomCodes.CONFIG_BEGIN + 19,
          UT_PARITY     = CustomCodes.CONFIG_BEGIN + 20,
          UT_INV        = CustomCodes.CONFIG_BEGIN + 21,
          UT_PAROFF     = CustomCodes.CONFIG_BEGIN + 22,
          UT_PARERRCNT = CustomCodes.CONFIG_BEGIN + 23,
          UT_PARERR     = CustomCodes.CONFIG_BEGIN + 24,
          UT_SFRAME     = CustomCodes.CONFIG_BEGIN + 25,
          UT_CLKSTOP    = CustomCodes.CONFIG_BEGIN + 26,
          UT_CLASS      = CustomCodes.CONFIG_BEGIN + 27,
          UT_TB_CLK     = CustomCodes.CONFIG_BEGIN + 28,
          UT_RBSYNC     = CustomCodes.CONFIG_BEGIN + 29,
          RX2TXWAIT     = CustomCodes.CONFIG_BEGIN + 30,   /**< Gets or Set the timing between the Rx and Tx frame. This configuration will be used by RdCardSim component. */
          ENV_C40_MODE  = CustomCodes.CONFIG_BEGIN + 31    /**< Enables the C40 mode communication */
        }

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;       /**< Pointer to the lower layers parameter structure. */
            public IntPtr pKeyStoreDataParams;  /**< Pointer to the parameter structure of the keyStore layer. */
            public IntPtr pTxBuffer;            /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;           /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;            /**< Number of valid bytes within the transmit buffer. */
            public byte bTxLastBits;            /**< Number of valid bits of the last byte in the transmit buffer. */
            public IntPtr pRxBuffer;            /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;           /**< Size of the global receive buffer. */
            public ushort wRxBufLen;            /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;       /**< Starting position within the global receive buffer. */
            public byte bRxAlign;               /**< Desired Bit-Alignment for reception. */
            public ushort wAdditionalInfo;      /**< Storage for additional error information. */
            public byte bCardType;              /**< Type of card for which the hal is configured for. */
            public ushort wRx2TxWait;           /**< Wait time in us for the reader model to wait between the Rx and TX frames*/
            public byte bTxBAUD;                /**< Storage for Tx baud rate setting*/
            public byte bRxBAUD;                /**< Storage for Rx baud rate setting. */
            public byte bTxCrc;                 /**< Storage for Tx CRC settings  */
            public byte bRxCrc;                 /**< Storage for Rx CRC Settings . */
            public byte bParity;                /**< Storage for Parity setttings. */
            public uint dwResetWaitTime;        /**< Storage for reset wait time. */
            public uint dwResetRecoveryTime;    /**< Storage for reset recovery time. */
            public IntPtr pIntBuffer;           /**< Internal buffer used as a scratch buffer. Size must be wIntBufferLen*wIntBufferNum */
            public ushort wIntBufferLen;        /**< Total length of internal buffer used as a scratch buffer */
            public ushort wIntBufferAlloc;      /**< Flag to show the status of buffer allocation. */
            public ushort wTimingMode;          /**< Current timing measurement mode. */
            public byte bTimeoutUnit;           /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public uint dwTimingUs;             /**< Current timing value. */
            public byte bTPactive;              /**< Is Testpad communication active or not. */
            public ushort wTimeT1;              /**< Optinal timeout T1 for Testpad communication. */
            public ushort wTimeT2;              /**< Optinal timeout T2 for Testpad communication. */
            public ushort wTimeT3;              /**< Optinal timeout T3 for Testpad communication. */
            public ushort wTimeT4;              /**< Optinal timeout T4 for Testpad communication. */
            public ushort wSyncOption;          /**< Syncronisation option for Testpad communication. */
            public byte  bEnvEnable;
            public byte bC40ModeEnabled;        /**< Indicates if the C40 mode communication is enabled */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_RdCardSim_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure */
            IntPtr pKeyStoreDataParams,     /**< [In] Pointer to the parameter structure of the keyStore layer. */
            IntPtr pTxBuffer,               /**< [In]Pointer to internal transmit buffer.*/
            ushort wTxBufSize,              /**< [In]internal transmit buffer size.*/
            IntPtr pRxBuffer,               /**< [In]Pointer to internal receive buffer (can be same as transmit buffer).*/
            ushort wRxBufSize,              /**< [In] internal receive buffer size.*/
            IntPtr pInternalBuffer,         /**< [In] Internal buffer used as a scratch buffer */
            ushort wInternalBufferLen       /**< [In] Length of innternal buffer used as a scratch buffer */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_RdCardSim_Cmd_DownloadImage(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            Int32 dAddress,                          /**< [In] Specifies the address to start loading data to */
            byte bMemType,                        /**< [In] Specifies the enumerated memory type to use */
            ref  ushort pImageLen,               /**< [In]Pointer to image length size word.*/
            byte[] pImage                              /**< [In] Pointer to the image data. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_RdCardSim_Cmd_WaferTestExchange(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_RdCardSim_Cmd_SetTestPadValue(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            byte bValue
                                                );

        #endregion



        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(
            Bal.Generic pBal,
            KeyStore.Generic pKeyStore,
            ushort wTxBufferLength,
            ushort wRxBufferLength,
            ushort wTmpBufferLength)
        {
            IntPtr pKeyStoreDataParams;

            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            m_bTxBuffer = new byte[wTxBufferLength];
            m_bRxBuffer = new byte[wRxBufferLength];
            m_bHalTmpBuffer = new byte[wTmpBufferLength];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);
            m_pHalTmpBuffer = GCHandle.Alloc(m_bHalTmpBuffer, GCHandleType.Pinned);

            return phhalHw_RdCardSim_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                wTxBufferLength,
                m_pRxBuffer.AddrOfPinnedObject(),
                wRxBufferLength,
                m_pHalTmpBuffer.AddrOfPinnedObject(),
                wTmpBufferLength);
        }
#if DEBUG
        public Status_t Init(
            ushort wDataParamSize,
            Bal.Generic pBal,
            KeyStore.Generic pKeyStore,
            int wTxBufferSize,
            byte[] bTxBuffer,
            int wRxBufferSize,
            byte[] bRxBuffer,
            int wTmpBufferSize,
            byte[] bHalTmpBuffer
            )
        {
            IntPtr pKeyStoreDataParams;

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);
            m_pHalTmpBuffer = GCHandle.Alloc(bHalTmpBuffer, GCHandleType.Pinned);

            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            return phhalHw_RdCardSim_Init(
                ref m_DataParamsInt[0],
                wDataParamSize,
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pHalTmpBuffer.AddrOfPinnedObject(),
                (ushort)wTmpBufferSize);
        }


#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private byte[] m_bTxBuffer;
        private byte[] m_bRxBuffer;
        private byte[] m_bHalTmpBuffer;
        private GCHandle m_pHalTmpBuffer;
        private GCHandle m_pTxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdCardSim()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdCardSim()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pHalTmpBuffer.IsAllocated)
            {
                this.m_pHalTmpBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// DownLoad memory image.
        /// </summary>
        /// <param name="dAddr"></param>
        /// <param name="bMemTypel"></param>
        /// <param name="pImage"></param>
        /// <returns></returns>
        public Status_t DownloadImage(
            int Addr,
            byte MemTyp,
            ushort imgLen,
            byte[] image)
        {
            return phhalHw_RdCardSim_Cmd_DownloadImage(
                ref m_DataParamsInt[0],
                Addr,
                MemTyp,
                ref imgLen,
                image);
//            return status;
        }

        public Status_t WaferTestExchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHw_RdCardSim_Cmd_WaferTestExchange(ref m_DataParamsInt[0], (ushort)wOption, pTxBuffer, (ushort)pTxBuffer.Length, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t SetTestPadValue(
            byte bValue
            )
        {
            return phhalHw_RdCardSim_Cmd_SetTestPadValue(ref m_DataParamsInt[0], bValue);
        }



        #endregion
    }
}
