/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// SAM (AV4 and future SAM's) specific HAL-Component of Basic Function Library Framework.
    ///     - Supports New features of SAM AV4.
    ///     - Supports AV3 backwards compatibility mode.
    ///     - Post <see cref="Sam.Init">Initialization</see>, it's must to call <see cref="DetectMode">Detect Mode</see>
    ///       interface to know the current state of SAM. Whether it's is in one of the following modes
    ///       - "AV3 Activated or Un-Activated" mode
    ///       - "AV4 Activated or Un-Activated" mode
    ///     - SAM supports 4 logical channels.
    ///     - Each Logical channel can have it's own authentication and session keys, and e.g. can have
    ///       multiple PICC authentications at a time, on separate logical channels, basically running 2 independent
    ///       secure messaging channels.
    ///     - Other example would be that one logical channel for PICC communication and authentication, second one for an Offline
    ///       Crypto key, that is used to additionally encrypt data that is written to a PICC with a separate key.
    ///     - This component returns mapped error code for the actual error codes received from SAM. To get the actual error code
    ///     returned by SAM call <see cref="GetConfig"/> as <see cref="Config.GET_SAM_ERROR_CODE"/>
    ///     identifier.
    /// </summary>
    public class Sam : Generic
    {
        #region Macros
        private const ushort MAX_BUFFER_SIZE = 261;

        /// <summary>
        /// Default timeout in microseconds
        /// </summary>
        public const int DEFAULT_TIMEOUT = 150;

        /// <summary>
        /// Number of shadowed configurations.
        /// </summary>
        public const int SHADOW_COUNT = 0x000F;

        /// <summary>
        /// Amount of needed and reserved memory for the protocol overhead.
        /// </summary>
        public const int RESERVED_TX_BUFFER_LEN = 6;

        /// <summary>
        /// Amount of needed and reserved memory for the protocol overhead.
        /// </summary>
        public const int RESERVED_RX_BUFFER_LEN = 2;

        public const int MFP_AUTHENTICATE_PART2_CAPBILITIY_SIZE = 6;
        #endregion

        #region Enumerations
        #region Error Codes
        /// <summary>
        /// SAM (AV4 and future SAM's) NXP Reader Library mapped Error Codes for the respective status codes
        /// returned by SAM.
        /// </summary>
        public enum Error : byte
        {
            /// <summary>
            /// Custom error code - EEProm failure.
            /// This error represents SAM's RESP.ISO6400
            /// </summary>
            HW_EEPROM = ( CustomCodes.ERROR_BEGIN + 0 ),

            /// <summary>
            /// Custom error code - Reader IC interface error.
            /// This error represents SAM's RESP.ISO6401
            /// </summary>
            HW_RC5XX = ( CustomCodes.ERROR_BEGIN + 1 ),

            /// <summary>
            /// Custom error code - Key creation failure.
            /// This error represents SAM's RESP.ISO6501
            /// </summary>
            KEY_CREATE_FAILED = ( CustomCodes.ERROR_BEGIN + 2 ),

            /// <summary>
            /// Custom error code - Key reference number invalid.
            /// This error represents SAM's RESP.ISO6502
            /// </summary>
            KEY_REF_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 3 ),

            /// <summary>
            /// Custom error code - Key usage counter number invalid.
            /// This error represents SAM's RESP.SO6503
            /// </summary>
            KEY_KUC_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 4 ),

            /// <summary>
            /// Custom error code - EEPROM high voltage failure; SAM permanently disabled.
            /// This error represents SAM's RESP.ISO6581
            /// </summary>
            HW_EE_HIGH_VOLTAGE = ( CustomCodes.ERROR_BEGIN + 5 ),

            /// <summary>
            /// Custom error code - Wrong length of the APDU or wrong LC byte.
            /// This error represents SAM's RESP.ISO6700
            /// </summary>
            ISO7816_WRONG_LENGTH_LC = ( CustomCodes.ERROR_BEGIN + 6 ),

            /// <summary>
            /// Custom error code - Final chained command expected; running command aborted,
            /// new command ignored
            /// This error represents SAM's RESP.ISO6883
            /// </summary>
            INCOMPLETE_CHAINING = ( CustomCodes.ERROR_BEGIN + 7 ),

            /// <summary>
            /// Custom error code - [If AuthenticateFirst] Un-matching PCD/PD capabilities
            /// This error represents SAM's RESP.ISO6982
            /// </summary>
            UNMATCHING_PCD_PD = ( CustomCodes.ERROR_BEGIN + 8 ),

            /// <summary>
            /// SAM Custom error code - One of the following
            ///     - CRC mismatch; wrong key handling buffer size
            ///     - HostRespData received from programmable logic is not
            ///       multiple of 16 bytes, while chaining applying HostMode.Full
            ///       is ongoing
            /// This error represents SAM's RESP.ISO6984
            /// </summary>
            KEY_INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 9 ),

            /// <summary>
            /// SAM Custom error code - Conditions of use not satisfied. Multiple meaning for
            /// this status, refer the command for actual information.
            /// This error represents SAM's RESP.ISO6985
            /// </summary>
            COND_USE_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 10 ),

            /// <summary>
            /// Custom error code - Command not allowed. Multiple meaning for
            /// this status, refer the command for actual information.
            /// This error represents SAM's RESP.ISO6986
            /// </summary>
            ISO7816_COMMAND_NOT_ALLOWED = ( CustomCodes.ERROR_BEGIN + 11 ),

            /// <summary>
            /// Custom error code - Unexpected RFU bit(s) value(s) in data field parameter.
            /// Multiple meaning for this status, refer the command for actual information.
            /// This error represents SAM's RESP.ISO6A80
            /// </summary>
            ISO7816_WRONG_PARAMS_FOR_INS = ( CustomCodes.ERROR_BEGIN + 12 ),

            /// <summary>
            /// Custom error code - No valid key version found.
            /// This error represents SAM's RESP.ISO6A82
            /// </summary>
            KEY_VERSION_INVALID = ( CustomCodes.ERROR_BEGIN + 13 ),

            /// <summary>
            /// Custom error code - SAM Host protection error.
            /// This error represents SAM's RESP.ISO6A84
            /// </summary>
            HOST_PROTECTION = ( CustomCodes.ERROR_BEGIN + 14 ),

            /// <summary>
            /// Custom error code - Incorrect value for parameter P1 and/or P2.
            /// This error represents SAM's RESP.ISO6A86
            /// </summary>
            ISO7816_WRONG_P1P2 = ( CustomCodes.ERROR_BEGIN + 15 ),

            /// <summary>
            /// Custom error code - Incorrect LE value.
            /// This error represents SAM's RESP.ISO6C00
            /// </summary>
            ISO7816_WRONG_LE = ( CustomCodes.ERROR_BEGIN + 16 ),

            /// <summary>
            /// Custom error code - Instruction code not supported or invalid or not available
            /// in the current state of the SAM.
            /// This error represents SAM's RESP.ISO6D00
            /// </summary>
            ISO7816_UNKNOWN_INS = ( CustomCodes.ERROR_BEGIN + 17 ),

            /// <summary>
            /// Custom error code - Class not supported.
            /// This error represents SAM's RESP.ISO6E00
            /// </summary>
            ISO7816_UNKNOWN_CLASS = ( CustomCodes.ERROR_BEGIN + 18 ),

            /// <summary>
            /// Custom error code - MAC verification failed. Multiple meaning for
            /// this status, refer the command for actual information.
            /// This error represents SAM's RESP.ISO901E
            /// </summary>
            CRYPTO = ( CustomCodes.ERROR_BEGIN + 19 ),

            /// <summary>
            /// Custom error code - MIFARE PICC sent incorrect amount of data (for example
            /// wrong length of MAC)
            /// This error represents SAM's RESP.ISO90BE
            /// </summary>
            MIFARE_PLUS_INCORRECT_DATA = ( CustomCodes.ERROR_BEGIN + 20 ),

            /// <summary>
            /// Custom error code - Correct SAM execution but MFP response is an error return code.
            /// This error represents SAM's RESP.ISO90BF
            /// </summary>
            MIFARE_PLUS_GEN = ( CustomCodes.ERROR_BEGIN + 21 ),

            /// <summary>
            /// Custom error code - Correct execution - UID not complete (selection sequence
            /// has to be continued)
            /// This error represents SAM's RESP.ISO90C0
            /// </summary>
            ISO_UID_INCOMPLETE = ( CustomCodes.ERROR_BEGIN + 22 ),

            /// <summary>
            /// Custom error code - Correct execution, DESFire returned error or incorrect length
            /// This error represents SAM's RESP.ISO90DF
            /// </summary>
            DESFIRE_GEN = ( CustomCodes.ERROR_BEGIN + 23 ),

            /// <summary>
            /// Custom error code - Incorrect block number
            /// This error represents SAM's RESP.ISO90EB
            /// </summary>
            ISO_WRONG_BNR = ( CustomCodes.ERROR_BEGIN + 24 ),

            /// <summary>
            /// Custom error code - Status - One of the following.
            ///     - ATS length byte does not match actual length of received ATS
            ///       or PPS response invalid ( incorrect length or PPSS byte)
            ///     - Block length or PCB invalid
            ///     - Incorrect ( no R-block or I-block) answer received
            /// This error represents SAM's RESP.ISO90EC
            /// </summary>
            ISO_INVALID_FORMAT = ( CustomCodes.ERROR_BEGIN + 25 ),

            /// <summary>
            /// SAM Custom error code - One of the following.
            ///     - User buffer length below ISO14443-4 minimum
            ///     - Invalid UID bit count (too many bits received)
            ///     - CID, NAD either not expected, missing or incorrect
            /// This error represents SAM's RESP.ISO90ED
            /// </summary>
            ISO_INVALID_PARAMETER = ( CustomCodes.ERROR_BEGIN + 26 ),

            /// <summary>
            /// Custom error code - Correct execution but PICC response is an error return code.
            /// This error represents SAM's RESP.ISO90EF
            /// </summary>
            MIFARE_GEN = ( CustomCodes.ERROR_BEGIN + 27 ),

            /// <summary>
            /// Custom error code - Correct execution - more data expected
            /// This error represents SAM's RESP.ISO90AF
            /// </summary>
            OK_CHAINING_ACTIVE = ( CustomCodes.ERROR_BEGIN + 28 ),

            /// <summary>
            /// Custom error code - [If MFP command] Command-chaining not supported
            /// This error represents SAM's RESP.ISO6884
            /// </summary>
            COMMAND_CHAINING_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 29 ),

            /// <summary>
            /// Custom error code - Unknown Public Key Index when trying to update
            /// This error represents SAM's RESP.ISO6A83
            /// </summary>
            UNKNOWN_PUBLIC_KEY_INDEX = ( CustomCodes.ERROR_BEGIN + 30 ),

            /// <summary>
            /// Custom error code - No precise diagnosis (physical address invalid, EEPROM error,
            /// no correct key loaded for cryptographic operation, improper length for cryptographic
            /// operation, internal limit exceeded...)
            /// This error represents SAM's RESP.ISO6F00
            /// </summary>
            NO_PRECISE_DIAGNOSIS = ( CustomCodes.ERROR_BEGIN + 31 ),

            /// <summary>
            /// Custom error code - Programmable Logic error. Error code returned by
            /// <see cref="Cmd_SAM_PLExec"/> command.
            /// This error represents SAM's RESP.ISOXXXX
            /// </summary>
            PROGRAMMABLE_LOGIC = ( CustomCodes.ERROR_BEGIN + 32 ),

            /// <summary>
            /// Custom error code - Correct execution - Correct execution - more data expected
            /// (SAM-Host secure messaging to be applied on each command exchange)
            /// This error represents SAM's RESP.ISO90AE
            /// </summary>
            OK_CHAINING_ACTIVE_EXT = ( CustomCodes.ERROR_BEGIN + 33 ),

            /// <summary>
            /// Custom error code - UID check byte (BCC) incorrect.
            /// This error represents SAM's RESP.ISO90EA
            /// </summary>
            ISO_UID_BCC_INCORRECT = ( CustomCodes.ERROR_BEGIN + 34 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 0 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK0 = ( CustomCodes.ERROR_BEGIN + 35 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 1 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK1 = ( CustomCodes.ERROR_BEGIN + 36 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 4 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK4 = ( CustomCodes.ERROR_BEGIN + 37 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 5 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK5 = ( CustomCodes.ERROR_BEGIN + 38 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 6 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK6 = ( CustomCodes.ERROR_BEGIN + 39 ),

            /// <summary>
            /// Custom error code - NACK received from PICC. MIFARE NACK 7 received.
            /// This error represents SAM's RESP.ISO90Fx
            /// </summary>
            MIFARE_NAK7 = ( CustomCodes.ERROR_BEGIN + 40 ),

            /// <summary>
            /// Custom error code - Correct Execution with Cert.A - more data remaining
            /// This error represents SAM's RESP.ISO90AD
            /// </summary>
            OK_CHAINING_ACTIVE_DUOX = ( CustomCodes.ERROR_BEGIN + 41 )
        }
        #endregion

        #region Generic Enumerations
        #region Mode
        /// <summary>
        /// Macro Definitions used for Host communication mode
        /// </summary>
        public enum HcMode : byte
        {
            /// <summary> Sam hardware is used with Sam in
            /// Un-Activated or Activated state
            /// </summary>
            AV3 = 0x03,

            /// <summary>
            /// SAM AV4 hardware is used with Sam in Un-Activated
            /// or Activated state
            /// </summary>
            AV4 = 0x03,
        }
        #endregion

        #region Operational Modes
        /// <summary>
        /// Operation mode of SAM.
        /// </summary>
        public enum OpMode : byte
        {
            /// <summary> Option to initialize SAM communication in Non X ( S ) mode. </summary>
            NON_X = 0x00,

            /// <summary> Option to initialize SAM communication in X mode with RC523 reader. </summary>
            X_RC523 = 0x01,

            /// <summary> Option to initialize SAM communication in X mode with RC663 reader. </summary>
            X_RC663 = 0x02
        }
        #endregion

        #region Logical Channel
        /// <summary>
        /// The logical channel to be used for communicating with SAM. This will internally update the Class byte. These logical channel
        /// can be used to communicate with 4 different PICC's. Each channel maintains its own states.
        /// </summary>
        public enum LogicalChannel : byte
        {
            /// <summary> Logical Channel 0.</summary>
            LC0 = 0x00,

            /// <summary> Logical Channel 1.</summary>
            LC1 = 0x01,

            /// <summary> Logical Channel 2.</summary>
            LC2 = 0x02,

            /// <summary> Logical Channel 3.</summary>
            LC3 = 0x03
        }
        #endregion

        #region LFI
        /// <summary>
        /// Enumeration filed to indicate the Last Frame Indicator.
        /// </summary>
        public enum LFI : byte
        {
            /// <summary> Indication of last frame. </summary>
            LAST_FRAME = 0x00,

            /// <summary> Indication of intermediate frame. More data will be sent to SAM. </summary>
            CHAINED_FRAME = 0xAF
        }
        #endregion

        #region Custom Exchange bits
        /// <summary>
        /// Custom exchange bit used for custom exchanging.
        /// </summary>
        public enum CustomExchangeBit : int
        {
            NO_ENCIPHERING_BIT = 0x0010,
            NO_DECIPHERING_BIT = 0x0020
        }
        #endregion Custom Exchange bits

        #region Configuration
        /// <summary>
        /// Gets or Sets the configuration parameter.
        /// </summary>
        public enum Config : int
        {
            /// <summary>Get / Set the HostMode; (e.g. #PHHAL_HW_SAMAV2_HC_AV1_MODE).</summary>
            HOSTMODE = CustomCodes.CONFIG_BEGIN,

            /// <summary>
            /// Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
            /// Default is #PH_OFF; Only applicable in NonX-Mode.
            ///</summary>
            DISABLE_NONX_CFG_MAPPING,

            /// <summary> Get the Actual Error Code returned by Sam. </summary>
            GET_SAM_ERROR_CODE,

            /// <summary>
            /// Configure communication type. Refer <see cref="CommunicationType"/> for supported values.
            /// </summary>
            COMMUNICATION_TYPE = 0x0060,

            /// <summary>
            /// Configure I2C Slave Address.
            /// </summary>
            I2C_SLAVE_ADDRESS = 0x0061,

            /// <summary>
            /// Configure I2C BitRate. Refer <see cref="I2C_BitRate"/> for supported values.
            /// </summary>
            I2C_BITRATE = 0x0062,

            /// <summary>
            /// Configure I2C Timeout.
            /// </summary>
            I2C_TIMEOUT_MS = 0x0063
        }
        #endregion

        #region Communication Mode
        /// <summary>
        /// Definitions for Communication Modes to be used for Data Processing / PICC commands.
        /// To be used for below mentioned interfaces
        ///     - <see cref="Cmd_SAM_ApplySM"/>
        ///     - <see cref="Cmd_SAM_RemoveSM"/>
        ///     - <see cref="Cmd_DESFire_WriteX"/>
        ///     - <see cref="Cmd_DESFire_ReadX"/>
        ///     - <see cref="Cmd_SAM_BindCertificate_Part1"/>
        /// </summary>
        public enum CommMode : byte
        {
            /// <summary> Option mask for communication mode as plain. </summary>
            PLAIN = 0x00,

            /// <summary> Option mask for communication mode as MAC. </summary>
            MAC = 0x10,

            /// <summary> Option mask for communication mode as FULL. </summary>
            FULL = 0x30
        }
        #endregion

        #region ISOMode
        /// <summary>
        /// Definitions for ISO mode selection.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_CreateTMFilePICC"/>
        ///     - <see cref="Cmd_DUOX_BindCertificate"/>
        /// </summary>
        public enum ISOMode
        {
            /// <summary> SAM ISO mode selection for Native command set. </summary>
            NATIVE = 0x00,

            /// <summary> SAM ISO mode selection for ISO 7816-4 command set. </summary>
            ISO7816 = 0x40,

            /// <summary> SAM ISO mode selection for ISO complaint Authentication. </summary>
            ISO_AUTHENTICATION = 0x80
        }
        #endregion
        #endregion

        #region Command specific enumerations
        #region Host Communication
        #region Cmd.SAM_LockUnlock
        /// <summary>
        /// Definitions of Lock types. To be used with <see cref="Cmd_SAM_LockUnlock"/> interface.
        /// </summary>
        public enum LockUnlock : byte
        {
            /// <summary> Option mask for Sub-command type as unlock. </summary>
            UNLOCK = 0x00,

            /// <summary> Option mask for Sub-command type as lock without specifying unlock key. </summary>
            LOCK_NO_KEY = 0x01,

            /// <summary> Option mask for Sub-command type as lock with specifying unlock key. </summary>
            LOCK_KEY = 0x02,

            /// <summary> Option mask for Sub-command type as activate MIFARE SAM to AV3. </summary>
            ACTIVATE_SAM = 0x03,

            /// <summary> Option mask for Sub-command type as unlock PL. </summary>
            UNLOCK_PL = 0x04
        }
        #endregion

        #region Cmd.SAM_AuthenticateHost
        /// <summary>
        /// Definitions of Host Protection modes.
        /// To be used with <see cref="Cmd_SAM_AuthenticateHost"/>" interface.
        /// </summary>
        public enum HostMode : byte
        {
            /// <summary> Option mask for protection mode as plain. </summary>
            PLAIN = 0x00,

            /// <summary> Option mask for protection mode as MAC protection. </summary>
            MAC = 0x01,

            /// <summary> Option mask for protection mode as Full protection. </summary>
            FULL = 0x02
        }
        #endregion
        #endregion

        #region Security and Configuration
        #region Cmd.SAM_DisableCrypto
        /// <summary>
        /// Definitions for disabling Crypto functionalities. To be used with <see cref="Cmd_SAM_DisableCrypto"/>
        /// interface.
        /// </summary>
        public enum DisableCrypto : ushort
        {
            /// <summary> Option mask for Disable Crypto with no change for programming mask bit. </summary>
            NO_CHANGE = 0x0000,

            /// <summary> Option mask for Disable Crypto to disable DESFire Key change. </summary>
            DES_PICC_CHANGE_KEY = 0x0800,

            /// <summary> Option mask for Disable Crypto to disable the decryption of data. </summary>
            DECRYPTION = 0x1000,

            /// <summary> Option mask for Disable Crypto to disable encryption of data. </summary>
            ENCRYPTION = 0x2000,

            /// <summary> Option mask for Disable Crypto to disable verification of MAC. </summary>
            MAC_VERIFICATION = 0x4000,

            /// <summary> Option mask for Disable Crypto to disable generation of MAC. </summary>
            MAC_GENERATION = 0x8000
        }
        #endregion

        #region Cmd.SAM_ActivateOfflineKey
        /// <summary>
        /// Definitions for LRP update keys. To be used with <see cref="Cmd_SAM_ActivateOfflineKey"/>
        /// interface.
        /// method.
        /// </summary>
        public enum LRPUpdate
        {
            /// <summary>
            /// Option mask for ActivateOffline with LRP Update keys to generate One
            /// updated key (KeyID.LRPUpdate).
            /// </summary>
            ONE = 0,

            /// <summary>
            /// Option mask for ActivateOffline with LRP Update keys to generate Two
            /// updated key (KeyID.LRPMACUpdate and KeyID.LRPENCUpdate).
            /// </summary>
            TWO = 2
        }
        #endregion

        #region Cmd.SAM_LoadInitVector
        /// <summary>
        /// Definitions for Load Initialization Vector. To be used with <see cref="Cmd_SAM_LoadInitVector"/>
        /// interface.
        /// </summary>
        public enum LoadIV : byte
        {
            /// <summary> Option mask for Load Init Vector to set the IV. </summary>
            SET_IV = 0x00,

            /// <summary> Option mask for Load Init Vector to set the LRP_EncCtr. </summary>
            SET_LRP_ENC_CTR = 0x01
        }
        #endregion

        #region Cmd.SAM_KillAuthentication
        /// <summary>
        /// Definitions to kill authentication. To be used with <see cref="Cmd_SAM_KillAuthentication"/>
        /// interface.
        /// </summary>
        public enum KillAuth : byte
        {
            /// <summary> Option mask for killing any authentication on corresponding LC. </summary>
            FULL,

            /// <summary>
            /// Option mask for killing PICC or offline key activation but preserving any
            /// Host Authentication.
            /// </summary>
            PARTIAL
        }
        #endregion

        #region Cmd.SAM_SetConfiguration
        /// <summary>
        /// Definitions to update SAM configuration settings. To be used with <see cref="Cmd_SAM_SetConfiguration"/>
        /// interface. To be used with <see cref="Cmd_SAM_SetConfiguration"/>, \b bOption parameter.
        /// </summary>
        public enum SetConfiguration : byte
        {
            /// <summary> Option mask for exchanging the historical bytes. </summary>
            HISTORICAL_BYTES = 0x00,

            /// <summary> Option mask for exchanging the reader IC configuration. </summary>
            READER_IC_CONFIG = 0x01,

            /// <summary> Option mask for exchanging the I2C processing clock speed configuration. </summary>
            I2C_CLOCK_SPEED = 0x02,

            /// <summary> Option mask for exchanging the Full ATR configuration. </summary>
            FULL_ATR = 0x04,
        }

        /// <summary>
        /// To be used with <see cref="Cmd_SAM_SetConfiguration"/> method,
        /// \b pData parameter for \b bOption = <see cref="SetConfiguration.READER_IC_CONFIG"/>
        /// </summary>
        public enum ReaderIC : byte
        {
            /// <summary> Option mask for exchanging the reader IC configuration as RC512. </summary>
            RC512 = 0x01,

            /// <summary> Option mask for exchanging the reader IC configuration as RC523. </summary>
            RC523 = 0x02,

            /// <summary> Option mask for exchanging the reader IC configuration as RC663. </summary>
            RC663 = 0x03,
        }

        /// <summary>
        /// To be used with <see cref="Cmd_SAM_SetConfiguration"/> method,
        /// \b pData parameter for \b bOption = <see cref="SetConfiguration.I2C_CLOCK_SPEED"/>
        /// </summary>
        public enum I2CClock
        {
            /// <summary> Option mask I2C processing clock speed as 0.5 MHz. </summary>
            SPEED_0_5_MHZ = 0x04,

            /// <summary> Option mask I2C processing clock speed as 1 MHz. </summary>
            SPEED_1_MHZ = 0x05,

            /// <summary> Option mask I2C processing clock speed as 2 MHz. </summary>
            SPEED_2_MHZ = 0x06,

            /// <summary> Option mask I2C processing clock speed as 3 MHz. </summary>
            SPEED_3_MHZ = 0x07,

            /// <summary> Option mask I2C processing clock speed as 4 MHz. </summary>
            SPEED_4_MHZ = 0x08,

            /// <summary> Option mask I2C processing clock speed as 6 MHz. </summary>
            SPEED_6_MHZ = 0x09,

            /// <summary> Option mask I2C processing clock speed as 8 MHz. </summary>
            SPEED_8_MHZ = 0x0A,

            /// <summary> Option mask I2C processing clock speed as 12 MHz (Default). </summary>
            SPEED_12_MHZ = 0x0B,

            /// <summary> Option mask I2C processing clock speed as 16 MHz. </summary>
            SPEED_16_MHZ = 0x0C,

            /// <summary> Option mask I2C processing clock speed as free running mode. </summary>
            SPEED_FREE_RUN = 0x0F
        }
        #endregion
        #endregion

        #region Key Management
        #region Cmd.SAM_ChangeKeyEntry
        /// <summary>
        /// Definitions for Non-volatile programming mask. To be used with <see cref="Cmd_SAM_ChangeKeyEntry"/> and
        /// <see cref="Cmd_SAM_ChangeKeyEntryOffline"/> interface.
        /// method.
        /// </summary>
        public enum ProMas : byte
        {
            /// <summary>Option mask for updating key number and version A.</summary>
            UPDATE_KEY_VA = 0x80,

            /// <summary>Option mask for updating key number and version B.</summary>
            UPDATE_KEY_VB = 0x40,

            /// <summary>Option mask for updating key number and version C.</summary>
            UPDATE_KEY_VC = 0x20,

            /// <summary>Option mask for updating DESFire application identifier.</summary>
            UPDATE_DF_AID = 0x10,

            /// <summary>Option mask for updating key number and version of change entry key (CEK).</summary>
            UPDATE_KEY_CEK = 0x08,

            /// <summary>Option mask for updating reference key usage counter.</summary>
            UPDATE_REF_NO_KUC = 0x04,

            /// <summary>Option mask for updating SET and Extended SET.</summary>
            UPDATE_SET_EXTSET = 0x02,

            /// <summary>Option mask for specifying the versions in the data field after SET and ExtSET.</summary>
            INCLUDE_VERSION = 0x01
        }

        #region Cmd.SAM_ChangeKUCEntry
        /// <summary>
        /// Definitions for Non-volatile programming mask. To be used with <see cref="Cmd_SAM_ChangeKUCEntry"/>
        /// and <see cref="Cmd_SAM_ChangeKUCEntryOffline"/> interface.
        /// </summary>
        public enum KUC_ProMas : byte
        {
            /// <summary>Option mask for updating the limit.</summary>
            UPDATE_LIMIT = 0x80,

            /// <summary>Option mask for updating CKUC key number.</summary>
            UPDATE_KEY_NO_CKUC = 0x40,

            /// <summary>Option mask for updating the VCKUC.</summary>
            UPDATE_VCKUC = 0x20,
        }
        #endregion
        #endregion

        #region Cmd.SAM_EncipherKeyEntry
        /// <summary>
        /// Definitions for Key diversification programming mask. To be used with
        /// <see cref="Cmd_SAM_EncipherKeyEntry"/> interface.
        /// </summary>
        public enum EncipherKeyEntry : byte
        {
            /// <summary> Option mask to include the diversification input in command frame. </summary>
            DIV_ON = 0x01,

            /// <summary> Option mask to include the SAM UID in command frame. </summary>
            SAM_UID_ON = 0x02
        }
        #endregion

        #region Cmd.SAM_DumpSessionKey, Cmd.SAM_DumpSecretKey
        /// <summary>
        /// Definitions for Key diversification programming mask. To be used with <see cref="Cmd_SAM_DumpSessionKey"/>
        /// and <see cref="Cmd_SAM_DumpSecretKey"/> interface.
        /// </summary>
        public enum DumpMode : byte
        {
            /// <summary>Option mask for dumping session or secret keys in plain.</summary>
            PLAIN = 0x00,

            /// <summary>Option mask for dumping session or secret keys in enciphered.</summary>
            ENCIPHERED = 0x01,

            /// <summary>Option mask for enabling the diversification of the provided key.</summary>
            DIVERSIFICATION_ON = 0x02
        }
        #endregion
        #endregion

        #region File Management
        /// <summary> Definitions for File Type. To be used with <see cref="Cmd_SAM_CreateFile"/>
        /// interface.
        /// </summary>
        public enum FileType
        {
            /// <summary> Option mask for file type as Certificate File (pure binary data file) </summary>
            CERTIFICATE = 0x00,

            /// <summary> Option mask for file type as CRL File. </summary>
            CRL = 0x01
        }
        #endregion

        #region Data Processing
        #region Cmd.SAM_ApplySM
        /// <summary>
        /// Definitions for exchanging Offset information. To be used with <see cref="Cmd_SAM_ApplySM"/>
        /// interface.
        /// </summary>
        public enum Offset : byte
        {
            /// <summary> Option mask to include Offset. </summary>
            EXCLUDE = 0x00,

            /// <summary> Option mask to include Offset. </summary>
            INCLUDE = 0x80
        }
        #endregion

        #region Cmd.SAM_VerifyMAC, Cmd.SAM_GenerateMAC
        /// <summary>
        /// Definitions for Protection Mode. To be used with <see cref="Cmd_SAM_GenerateMAC"/>
        /// and <see cref="Cmd_SAM_VerifyMAC"/> interface.
        /// </summary>
        public enum TruncationMode : byte
        {
            /// <summary> Option mask for truncation mode as standard truncation. </summary>
            STANDARD = 0x00,

            /// <summary> Option mask for truncation mode as MFP truncation. </summary>
            MFP = 0x80
        }
        #endregion

        #region Cmd.SAM_GenerateMAC
        /// <summary>
        /// Option mask for inclusion of LC in the command frame.
        /// To be used with<see cref= "Cmd_SAM_GenerateMAC" />
        /// </summary>
        public enum IncludeLC : byte
        {
            /// <summary> Option mask for exclusion of LC in the command frame. </summary>
            EXCLUDE = 0x00,

            /// <summary> Option mask for inclusion of LC in the command frame. </summary>
            INCLUDE = 0x80
        }
        #endregion

        #region Cmd.SAM_DecipherData
        /// <summary>
        /// Definitions for exchanging Length information to SAM. To be used with
        /// <see cref="Cmd_SAM_DecipherData"/> interface.
        /// </summary>
        public enum Length : byte
        {
            /// <summary> Option mask for excluding the length information in
            /// the command frame.
            /// </summary>
            EXCLUDE = 0x00,

            /// <summary> Option mask for including the length information in
            /// the command frame.
            /// </summary>
            INCLUDE = 0x80
        }
        #endregion
        #endregion

        #region Public Key Infrastructure
        #region Update Mode
        /// <summary>
        /// Definitions for Update settings and value or settings only.
        /// To be used with below mentioned interface(s)
        ///     - <see cref="Cmd_PKI_ImportKey"/>
        ///     - <see cref="Cmd_PKI_ImportEccKey"/>
        ///     - <see cref="Cmd_PKI_ImportEccCurve"/>
        /// </summary>
        public enum PKI_UpdateMode
        {
            /// <summary> Option mask for updating settings and values. </summary>
            SETTINGS_VALUE,

            /// <summary> Option mask for updating settings only. </summary>
            SETTINGS_ONLY
        }
        #endregion

        #region FirstFrame
        /// <summary>
        /// Definitions for indicating the interface call for first frame and add the
        /// required information to command buffer for exchange. For the next do not add the
        /// previously exchanged information.
        /// To be used with below mentioned interface(s)
        ///     - <see cref="Cmd_PKI_GenerateHash"/>
        ///     - <see cref="Cmd_PKI_DecipherData"/>
        /// </summary>
        public enum PKI_LFI
        {
            /// <summary> Option mask for a framing the first frame of Decipher Data command. </summary>
            FIRST_FRAME = 0x80
        }
        #endregion

        #region Hash Algorithm
        /// <summary>
        /// Definitions for Hashing algorithm. To be used with below mentioned interface(s)
        ///     - <see cref="Cmd_PKI_UpdateKeyEntries"/>
        ///     - <see cref="Cmd_PKI_EncipherKeyEntries"/>
        ///     - <see cref="Cmd_PKI_GenerateHash"/>
        ///     - <see cref="Cmd_PKI_GenerateSignature"/>
        ///     - <see cref="Cmd_PKI_VerifySignature"/>
        ///     - <see cref="Cmd_PKI_EncipherData"/>
        ///     - <see cref="Cmd_PKI_DecipherData"/>
        ///     - <see cref="Cmd_PKI_GenerateEccSignature"/>
        /// </summary>
        public enum PKI_HashAlgo
        {
            /// <summary> Option mask for SHA 1 hashing algorithm to be used. </summary>
            SHA_1 = 0x00,

            /// <summary> Option mask for SHA 224 hashing algorithm to be used. </summary>
            SHA_224 = 0x01,

            /// <summary> Option mask for SHA 256 hashing algorithm to be used. </summary>
            SHA_256 = 0x03,

            /// <summary> Option mask for SHA 384 hashing algorithm to be used. </summary>
            SHA_384 = 0x04,

            /// <summary> Option mask for SHA 512 hashing algorithm to be used. </summary>
            SHA_512 = 0x05
        }
        #endregion

        #region RSA
        #region GenerationMode
        /// <summary>
        /// Definitions for Random number generation mode.
        /// To be used with <see cref="Cmd_PKI_GenerateKeyPair"/> interface.
        /// </summary>
        public enum PKI_GenerationMode
        {
            /// <summary> Option mask for a key generation with a randomly selected exponent e. </summary>
            RANDOM_E = 0x00,

            /// <summary> Option mask for a key generation with a given exponent e. </summary>
            HOST_E = 0x01
        }
        #endregion

        #region AccesEntryKey
        /// <summary>
        /// Definitions for Access Entry Key exchange to SAM and retrieval from SAM.
        /// To be used with below mentioned interface(s)
        ///     - To exchange to SAM
        ///         - <see cref="Cmd_PKI_GenerateKeyPair"/>
        ///         - <see cref="Cmd_PKI_ImportKey"/>
        ///
        ///     - To retrieve from SAM
        ///         - <see cref="Cmd_PKI_ExportPrivateKey"/>
        ///         - <see cref="Cmd_PKI_ExportPublicKey"/>
        /// </summary>
        public enum PKI_AEK
        {
            /// <summary>
            /// Option mask for exchanging Access Entry Key information to SAM.
            /// To be used with below mentioned interface(s)
            ///     - <see cref="Cmd_PKI_GenerateKeyPair"/>
            ///     - <see cref="Cmd_PKI_ImportKey"/>
            /// </summary>
            INCLUDE_AEK = 0x02,

            /// <summary>
            /// Option mask for receiving Access Entry Key information from SAM.
            /// To be used with below mentioned interface(s)
            ///     - <see cref="Cmd_PKI_ExportPrivateKey"/>
            ///     - <see cref="Cmd_PKI_ExportPublicKey"/>
            /// </summary>
            RECEIVE_AEK = 0x80
        }
        #endregion

        #region Acknowledge
        /// <summary>
        /// Definitions to include Le byte and Acknowledge key number to command frame for receiving
        /// the UpdateAck data from SAM. To be used with <see cref="Cmd_PKI_UpdateKeyEntries"/> interface.
        /// </summary>
        public enum PKI_ACK
        {
            /// <summary>
            /// Option mask for not receiving Update Acknowledgment information from SAM.
            /// </summary>
            DO_NOT_RECEIVE_ACK = 0x00,

            /// <summary>
            /// Option mask for receiving Update Acknowledgment information from SAM.
            /// </summary>
            RECEIVE_ACK = 0x80
        }
        #endregion

        #region Diversification
        /// <summary>
        /// Definitions for SAM Public Key Infrastructure key diversification.To be used
        /// with <see cref="Cmd_PKI_EncipherKeyEntries"/> interface.
        /// </summary>
        public enum PKI_Div
        {
            /// <summary> Option mask disabling the key diversification. </summary>
            OFF = 0x00,

            /// <summary> Option mask enabling the key diversification. </summary>
            ON = 0x10
        }
        #endregion
        #endregion

        #region ECC
        #region Certificate Format
        /// <summary>
        /// Definitions for SAM Public Key Infrastructure Certificate Format. To be used
        /// * with <see cref="Generic.Cmd_PKI_ValidateEccCert"/> interface.
        /// </summary>
        public enum CertificateFormat
        {
            /// <summary> Option mask for certificate format as X.509 certificate </summary>
            X509 = 0x00,

            /// <summary> Option mask for certificate format as EV charging GP certificate </summary>
            EV_CHARGING = 0x01
        }
        #endregion
        #endregion
        #endregion

        #region Virtual Card
        #region SelectVC
        /// <summary>
        /// Definitions for Key diversification selection. To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_SelectVC"/>
        ///     - <see cref="Cmd_VCA_Select"/>
        /// </summary>
        public enum VCA_Select : byte
        {
            /// <summary>
            /// Default option mask to disable the diversification of
            /// VcSelect MAC and ENC key.
            /// </summary>
            DEFAULT = 0x00,

            /// <summary>
            /// Option mask to perform diversification of VC SelectENC key using the
            /// diversification input provided.
            /// </summary>
            ENC_KEY_DIV_INPUT = 0x01,

            /// <summary>
            /// Option mask to perform diversification of VC SelectMAC key using the
            /// diversification input provided.
            /// </summary>
            MAC_KEY_DIV_INPUT = 0x02,

            /// <summary>
            /// Option mask to perform diversification of VC SelectMAC key using
            /// Virtual Card Identifier.
            /// </summary>
            MAC_KEY_DIV_VCUID = 0x04,

            /// <summary>
            /// Option mask to perform diversification of VC SelectMAC key using Virtual
            /// Card Identifier and Diversification input provided.
            /// </summary>
            MAC_KEY_DIV_INPUT_VCUID = 0x06,

            /// <summary> Option mask for complete VC selection in 1 part. </summary>
            VARIANT_PART1 = 0x00,

            /// <summary> Option mask for complete VC selection in 2 part. </summary>
            VARIANT_PART2 = 0x08
        }
        #endregion

        #region Proximity Check
        /// <summary>
        /// Definitions for Key diversification selection. To be used with below mentioned interfaces.
        ///     - <see cref= "Cmd_SAM_ProximityCheck_Part1" />
        ///     - <see cref= "Cmd_VCA_ProximityCheck" />
        /// </summary>
        public enum VCA_ProximityCheck : byte
        {
            /// <summary>
            /// Option mask to perform diversification of ProximityCheck key using the
            /// diversification input provided.
            /// </summary>
            DIVERSIFICATION_ON = 0x01,

            /// <summary> Option mask to perform Normal Cmd.VerifyPC processing. </summary>
            NORMAL_PROCESSING = 0x00,

            /// <summary> Option mask to perform Random Cmd.VerifyPC processing. </summary>
            RANDOM_PROCESSING = 0x02,

            /// <summary> Option mask to perform Proximity Check in native format. </summary>
            NATIVE_FORMAT = 0x00,

            /// <summary> Option mask to perform Proximity Check in ISO7816-4 wrapped format. </summary>
            WRAPPED_FORMAT = 0x04
        }
        #endregion
        #endregion

        #region MIFARE DESFire
        #region Authentication
        #region Authentication Type
        /// <summary>
        /// Definitions for Authentication mode selection.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public enum MFD_AuthType : byte
        {
            /// <summary> SAM DESFire Authentication mode as D40 and EV1. </summary>
            D40_EV1 = 0x00,

            /// <summary> SAM DESFire Authentication mode as EV2 First. </summary>
            EV2_FIRST_AUTH = 0x80,

            /// <summary> SAM DESFire Authentication mode as EV2 Non First. </summary>
            EV2_NON_FIRST_AUTH = 0xC0
        }
        #endregion

        #region Secure Messaging
        /// <summary>
        /// Definitions for Secure Messaging.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public enum MFD_AuthSM
        {
            /// <summary> SAM DESFire Allow Secure messaging. </summary>
            ALLOW = 0x00,

            /// <summary> SAM DESFire Suppress Secure messaging. </summary>
            SUPPRESS = 0x20
        }
        #endregion

        #region Key Diversification Mode
        /// <summary>
        /// Definitions for Key diversification mode selection.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_SAM_IsoAuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public enum MFD_AuthDivMode : byte
        {
            /// <summary>
            /// SAM DESFire key derivation type as AV1 and
            /// SAM DESFire key derivation type as AV1 double encryption round.
            /// </summary>
            AV1_DOUBLE_ENCRYPTION = 0x00,

            /// <summary>
            /// SAM DESFire key derivation type as AV1 single encryption
            /// round.
            /// </summary>
            AV1_SINGLE_ENCRYPTION = 0x08,

            /// <summary> SAM DESFire key derivation type as AV2. </summary>
            AV2 = 0x10
        }
        #endregion

        #region Key Selection
        /// <summary>
        ///
        /// Definitions for Key diversification mode selection.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_SAM_IsoAuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public enum MFD_AuthKeySel : byte
        {
            /// <summary> SAM DESFire key selection by key entry number. </summary>
            KEY_ENTRY_NUMBER = 0x00,

            /// <summary> SAM DESFire key selection by DESFIRE key number. </summary>
            DESFIRE_KEY_NUMBER = 0x02
        }
        #endregion

        #region Use Diversification
        /// <summary>
        /// Definitions for diversification usage.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_SAM_IsoAuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public enum MFD_AuthDiv : byte
        {
            /// <summary>SAM DESFire key diversification disabled.</summary>
            OFF = 0x00,

            /// <summary>SAM DESFire key diversification enabled.</summary>
            ON = 0x01,
        }
        #endregion
        #endregion

        #region Change Key
        #region Key compilation method
        #region Diversification Method
        /// <summary>
        /// Definitions for Key diversification method.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKDivMethod
        {
            /// <summary> SAM DESFire diversification method as AV1. </summary>
            AV1 = 0x00,

            /// <summary> SAM DESFire diversification method as AV2. </summary>
            AV2 = 0x20,
        }
        #endregion

        #region Diversification Method (CurrentKey)
        /// <summary>
        /// Definitions for Key diversification method for Current Key.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKCurrKey
        {
            /// <summary>
            /// SAM DESFire diversification method of current
            /// key for AV1 as double encryption.
            /// </summary>
            DOUBLE_ENCRYPTION = 0x00,

            /// <summary>
            /// SAM DESFire diversification method of current
            /// key for AV1 as single encryption.
            /// </summary>
            SINGLE_ENCRYPTION = 0x10,

            /// <summary>
            /// SAM DESFire diversification usage for current
            /// key is disabled.
            /// </summary>
            DIV_OFF = 0x00,

            /// <summary>
            /// SAM DESFire diversification usage for current
            /// key is enabled.
            /// </summary>
            DIV_ON = 0x04,
        }
        #endregion

        #region Diversification Method (NewKey)
        /// <summary>
        /// Definitions for Key diversification method for New Key.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKNewKey
        {
            /// <summary>
            /// SAM DESFire diversification method of new
            /// key for AV1 as double encryption.
            /// </summary>
            DOUBLE_ENCRYPTION = 0x00,

            /// <summary>
            /// SAM DESFire diversification method of new
            /// key for AV1 as single encryption.
            /// </summary>
            SINGLE_ENCRYPTION = 0x10,

            /// <summary>
            /// SAM DESFire diversification usage for new
            /// key is disabled.
            /// </summary>
            DIV_OFF = 0x00,

            /// <summary>
            /// SAM DESFire diversification usage for new
            /// key is enabled.
            /// </summary>
            DIV_ON = 0x04,
        }
        #endregion

        #region Cryptogram mode
        /// <summary>
        /// Definitions for Cryptogram computation mode.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKCryptoMode
        {
            /// <summary>
            /// SAM DESFire crypto computation mode are different
            /// for target and auth keys.
            /// </summary>
            DIFFERENT_KEY = 0x00,

            /// <summary>
            /// SAM DESFire crypto computation mode are same for
            /// target and auth keys. The parameters CurrKeyNo and
            /// CurrKeyV are ignored
            /// </summary>
            SAME_KEY = 0x01
        }
        #endregion
        #endregion

        #region Configuration
        #region Command Type
        /// <summary>
        /// Definitions for PICC Change Key command.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKCmdType
        {
            /// <summary> SAM DESFire Change Key command type as ChangeKey. </summary>
            CHANGE_KEY = 0x00,

            /// <summary> SAM DESFire Change Key command type as ChangeKeyEV2. </summary>
            CHANGE_KEY_EV2 = 0x20,
        }
        #endregion

        #region Master Key Update
        /// <summary>
        /// Definitions common for MIFARE DESFire S and X mode communication for
        /// ChangeKey interface. To be used for updating P2 information byte.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public enum MFD_CKMasterUpdate
        {
            /// <summary> SAM DESFire PICC Master key update to exclude key type in cryptogram. </summary>
            EXCLUDE_KEYTYPE = 0x00,

            /// <summary>
            /// SAM DESFire PICC Master key update to include key
            /// type in cryptogram.
            /// </summary>
            INCLUDE_KEYTYPE = 0x10,
        }
        #endregion
        #endregion
        #endregion

        #region Create TransactionMac File
        #region Diversification
        /// <summary>
        /// Definitions for diversification usage.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_CreateTMFilePICC"/>
        ///     - <see cref="Cmd_DESFire_CreateTMFilePICC"/>
        /// </summary>
        public enum MFD_TMDiv : byte
        {
            /// <summary> SAM DESFire key diversification disabled. </summary>
            OFF = 0x00,

            /// <summary> SAM DESFire key diversification enabled. </summary>
            ON = 0x01
        }
        #endregion

        #region FileOption
        /// <summary>
        /// Definitions for File Options.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_CreateTMFilePICC"/>
        ///     - <see cref="Cmd_DESFire_CreateTMFilePICC"/>
        /// </summary>
        public enum MFD_TMFileOption : byte
        {
            /// <summary> SAM communication mode as Plain. </summary>
            PLAIN = 0x00,

            /// <summary> SAM communication mode as MAC. </summary>
            MAC = 0x01,

            /// <summary> SAM communication mode as FULL. </summary>
            FULL = 0x03,

            /// <summary> SAM File Option as TMI Exclusion File Map. </summary>
            TMI_EXCLUSION_FILEMAP = 0x40
        }
        #endregion FileOption

        #region TMKeyOption
        /// <summary>
        /// Definitions for TMKey Options.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_CreateTMFilePICC"/>
        ///     - <see cref="Cmd_DESFire_CreateTMFilePICC"/>
        /// </summary>
        public enum MFD_TMKeyOption : byte
        {
            /// <summary> SAM TMKeyOption KeyType as AES128. </summary>
            KEYTYPE_AES128 = 0x02,

            /// <summary> SAM TMKeyOption KeyType as AES256. </summary>
            KEYTYPE_AES256 = 0x03,

            /// <summary> SAM TMKeyOption mode as Transaction MAC. </summary>
            MODE_TMAC = 0x00,

            /// <summary> SAM TMKeyOption mode as Transaction Signature. </summary>
            MODE_TSIG = 0x80
        }
        #endregion TMKeyOption
        #endregion

        #region S_Mode
        #region Auth Mode
        /// <summary>
        /// Definitions for PICC authentication mode.
        /// To be used with <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        /// </summary>
        public enum MFD_AuthMode : byte
        {
            /// <summary> SAM DESFire Auth mode as EV2. </summary>
            EV2 = 0x00,

            /// <summary> SAM DESFire Auth mode as LRP. </summary>
            LRP = 0x01
        }
        #endregion Auth Mode
        #endregion

        #region X_Mode
        #region Extended Offset
        /// <summary>
        /// Definitions for Extended Offset.
        /// To be used with <see cref="Cmd_DESFire_WriteX"/> interface
        /// </summary>
        public enum MFD_Offset
        {
            /// <summary> SAM DESFire DESFire EV2 mode as legacy mode (SAM AV2). </summary>
            LEGACY_MODE = 0x00,

            /// <summary> SAM DESFire DESFire EV2 mode as EV2 mode. </summary>
            EXTENDED_OFFSET = 0x80
        }
        #endregion

        #region Write Chaining
        /// <summary>
        /// Definitions for chaining configuration.
        /// To be used with <see cref="Cmd_DESFire_WriteX"/> interface
        /// </summary>
        public enum MFD_Chaining
        {
            /// <summary> SAM DESFire chaining mode as DESFire application chaining. </summary>
            DESFIRE_CHAINING = 0x00,

            /// <summary> SAM DESFire chaining mode as ISO/IEC 14443-4 chaining. </summary>
            ISO_CHAINING = 0x40,
        }
        #endregion
        #endregion
        #endregion

        #region MIFARE DUOX
        #region CertificateB Option
        /// <summary>
        /// Definitions for MIFARE DUOX CertificateB Processing Option.
        /// To be used with below mentioned interfaces,
        ///     - <see cref="Cmd_SAM_MutualAuthEcc_Part1"/>
        ///     - <see cref="Cmd_DUOX_MutualAuthEcc"/>
        /// </summary>
        public enum DUOX_CertBOption
        {
            /// <summary>
            /// SAM DUOX Certificate B processing option as No Msg.B processing by SAM - Return
            /// full Msg.B.Plain
            /// </summary>
            MSB_B_PLAIN = 0x00,

            /// <summary>
            /// SAM DUOX Certificate B processing option as Sig.B and (if present) Cert.B certificate
            /// (chain) validated by SAM - No data returned
            /// </summary>
            NO_DATA = 0x01,

            /// <summary>
            /// SAM DUOX Certificate B processing option as Sig.B and (if present) Cert.B certificate
            /// (chain) validated by SAM - Return TBSCert.B
            /// </summary>
            TBS_CERTB = 0x01
        }
        #endregion

        #region Protocol Option
        /// <summary>
        /// Definitions for MIFARE DUOX Protocol Options. To be used with
        /// <see cref="Cmd_SAM_UnilatAuthEcc_Part1"/>
        /// </summary>
        public enum DUOX_ProtocolOption
        {
            /// <summary> SAM DUOX protocol option as no ISOInternal Authentication. </summary>
            ISO_INTERNAL_AUTH = 0x00,

            /// <summary> SAM DUOX protocol option as VDE ECDSA Signing. </summary>
            VDE_ECDSA_SIGN = 0x01
        }
        #endregion

        #region S_Mode
        /// <summary>
        /// Definitions for MIFARE DUOX Authentication methods. To be used with
        /// <see cref="Cmd_SAM_MutualAuthEcc_Part1"/>
        /// </summary>
        public enum DUOX_AuthMethod
        {
            /// <summary>
            /// SAM DUOX Authentication method as ASymmetric Mutual Authentication
            /// with Certificate.A included
            /// </summary>
            MUTUAL_AUTH_CERT = 0x80,

            /// <summary>
            /// SAM DUOX Authentication method as ASymmetric Mutual Authentication
            /// with Certificate.A not included
            /// </summary>
            MUTUAL_AUTH_NO_CERT = 0xA0,

            /// <summary>
            /// SAM DUOX Authentication method as ASymmetric Reader-Unilateral
            /// Authentication with Certificate.A included
            /// </summary>
            READER_UNILATERAL_AUTH_CERT = 0x40,

            /// <summary>
            /// SAM DUOX Authentication method as ASymmetric Reader-Unilateral
            /// Authentication with Certificate.A not included
            /// </summary>
            READER_UNILATERAL_AUTH_NO_CERT = 0x60
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// Definitions for MIFARE DUOX Flow Option. To be used with
        /// <see cref="Cmd_DUOX_MutualAuthEcc"/>
        /// </summary>
        public enum DUOX_FlowOption
        {
            /// <summary> SAM DUOX flow option as no ISOGeneral Authentication Part 1 and Part2. </summary>
            ISO_GENERAL_AUTH = 0x00,

            /// <summary> SAM DUOX flow option as ISOSelectFile and ISOGeneralAuthenticate Final. </summary>
            ISO_GENERAL_AUTH_FINAL = 0x01
        }
        #endregion
        #endregion

        #region MIFARE Plus
        #region Authenticate command
        #region Authenticate
        /// <summary>
        /// Definitions common for MIFARE Plus S and X mode communication for
        /// Authentication interfaces.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_Authenticate"/>
        /// </summary>
        public enum MFP_Authenticate
        {
            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFY_OFF = MFP_AuthDiv.DIVERSIFY_OFF,

            /// <summary> Option to enable the key diversification.</summary>
            DIVERSIFY_ON = MFP_AuthDiv.DIVERSIFY_ON,

            /// <summary> Option to perform First authentication.</summary>
            FIRST_AUTH = MFP_AuthMode.AUTH_FIRST,

            /// <summary> Option to perform NonFirst (following) authentication.</summary>
            NONFIRST_AUTH = MFP_AuthMode.AUTH_NONFIRST,

            /// <summary> Option to set the key derivation info for SL0 or SL1 layer.</summary>
            SL1_NO_KDF = MFP_AuthSLKDF.SL1_NO_KDF,

            /// <summary> Option to set the key derivation info for SL3 layer.</summary>
            SL3_KDF = MFP_AuthSLKDF.SL3_KDF
        }
        #endregion

        #region Diversification
        /// <summary>
        /// Definitions for Key Diversification Mode.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///     - <see cref="Cmd_Cmd_MFP_Authenticate"/>
        /// </summary>
        public enum MFP_AuthDiv : byte
        {
            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFY_OFF = 0x00,

            /// <summary> Option to enable the key diversification.</summary>
            DIVERSIFY_ON = 0x01,
        }
        #endregion

        #region Authentication Mode
        /// <summary>
        /// Definitions for Authentication Mode.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///     - <see cref="Cmd_Cmd_MFP_Authenticate"/>
        /// </summary>
        public enum MFP_AuthMode : byte
        {
            /// <summary> Option to perform First authentication.</summary>
            AUTH_FIRST = 0x00,

            /// <summary> Option to perform NonFirst (following) authentication.</summary>
            AUTH_NONFIRST = 0x02
        }
        #endregion

        #region SL_KDF
        /// <summary>
        /// Definitions for Security Level and Key derivation information.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///     - <see cref="Cmd_Cmd_MFP_Authenticate"/>
        /// </summary>
        public enum MFP_AuthSLKDF : byte
        {
            /// <summary> Option to set the key derivation info for SL0 or SL1 layer.</summary>
            SL1_NO_KDF = 0x00,

            /// <summary> Option to set the key derivation info for SL3 layer.</summary>
            SL3_KDF = 0x0C
        }
        #endregion
        #endregion

        #region Sector Switch Authenticate command
        #region Sector Switch Authentication
        /// <summary>
        /// Definitions common for MIFARE Plus S and X mode communication for
        /// SectorSwitch Authentication interfaces.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public enum MFP_SectorSwitchAuth : ushort
        {
            /// <summary> Option to disable the Sector Switch key diversification.</summary>
            SECTOR_SWITCH_DIV_OFF = MFP_SSKeyDiv.DIV_OFF,

            /// <summary> Option to enable the Sector Switch key diversification.</summary>
            SECTOR_SWITCH_DIV_ON = MFP_SSKeyDiv.DIV_ON,

            /// <summary> Option to disable the Sector key diversification.</summary>
            SECTOR_DIV_OFF = MFP_SKeyDiv.DIV_OFF,

            /// <summary> Option to enable the Sector key diversification.</summary>
            SECTOR_DIV_ON = MFP_SKeyDiv.DIV_ON,

            /// <summary> Option to disable the Master Sector key diversification.</summary>
            MASTER_SECTOR_DIV_OFF = MFP_MKeyDiv.DIV_OFF,

            /// <summary> Option to enable the Master Sector key diversification with given sector number.</summary>
            MASTER_SECTOR_DIV_ON = MFP_MKeyDiv.DIV_ON,

            /// <summary> Option to buffer the KeyBlocks information.</summary>
            BUFFER_KEY_BLOCKS = MFP_SSBuffer.KEY_BLOCKS,

            /// <summary> Option to buffer the Diversification input information.</summary>
            BUFFER_DIV_INPUT = MFP_SSBuffer.DIV_INPUT
        }
        #endregion

        #region Sector Switch Key Diversification
        /// <summary>
        /// Definitions for Diversification for Sector Switch Key.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public enum MFP_SSKeyDiv : ushort
        {
            /// <summary> Option to disable the Sector Switch key diversification.</summary>
            DIV_OFF = 0x00,

            /// <summary> Option to enable the Sector Switch key diversification.</summary>
            DIV_ON = 0x01,
        }
        #endregion

        #region Sector Key Diversification
        /// <summary>
        /// Definitions for Diversification for Sector Key.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public enum MFP_SKeyDiv : ushort
        {
            /// <summary> Option to disable the Sector key diversification.</summary>
            DIV_OFF = 0x00,

            /// <summary> Option to enable the Sector key diversification.</summary>
            DIV_ON = 0x02,
        }
        #endregion

        #region Master Key Diversification
        /// <summary>
        /// Definitions for Diversification for Master Key.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public enum MFP_MKeyDiv : ushort
        {
            /// <summary> Option to disable the Master Sector key diversification.</summary>
            DIV_OFF = 0x00,

            /// <summary> Option to enable the Master Sector key diversification with given sector number.</summary>
            DIV_ON = 0x04,
        }
        #endregion
        #endregion

        #region PDC Authenticate command
        /// <summary>
        /// Definitions common for MIFARE Plus S and X mode communication for
        /// Post Delivery Configuration Authentication interfaces.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePDC_Part1"/>
        ///     - <see cref="Cmd_PDC_Authenticate"/>
        /// </summary>
        public enum MFP_PDCAuth : byte
        {
            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFICATION_OFF = 0x00,

            /// <summary> Option to indicate the UpgradeKey derivation form ICUpgradeKey given UpgradeInfo.</summary>
            DERIVE_UPGRADE_KEY = 0x02,

            /// <summary> Option to indicate the diversification of YearUpgradeKey with the given DivInput
            /// and then derive the actual UpgradeKey with UpgradeInfo.
            /// </summary>
            DIVERSIFY_YEAR_UPGRADE_KEY = 0x03
        }
        #endregion

        #region ChangeKey command
        /// <summary>
        /// Definitions common for MIFARE Plus S and X mode communication for ChagneKey
        /// interfaces for diversification.
        /// To be used with below mentioned interfaces .
        ///     - <see cref="Cmd_SAM_ChangeKeyMFP"/>
        ///     - <see cref="Cmd_MFP_ChangeKey"/>
        /// </summary>
        public enum MFP_ChangeKeyDiv : byte
        {
            /// <summary> Option to disable the key diversification. </summary>
            DIVERSIFY_OFF = 0x00,

            /// <summary>Option to enable the key diversification. </summary>
            DIVERSIFY_ON = 0x02,
        }
        #endregion

        #region S_Mode
        #region Payload Type
        /// <summary>
        /// Definitions for Payload type. To be used with below mentioned interfaces
        ///     - <see cref="Cmd_SAM_CombinedReadMFP"/>
        ///     - <see cref="Cmd_SAM_CombinedWriteMFP"/>
        ///     - <see cref="Cmd_SAM_ChangeKeyMFP"/>
        /// </summary>
        public enum MFP_PayloadType : byte
        {
            /// <summary> Option value for Combined Write - Command. </summary>
            COMMAND = 0x00,

            /// <summary> Option value for Combined Write - Command. </summary>
            RESPONSE = 0x01,

            /// <summary> Option value for Combined Read - Both Command and Response. </summary>
            BOTH = 0x02
        }
        #endregion

        #region CombinedWrite
        /// <summary>
        /// Definitions for Combined Write Plain data in response.
        /// To be used with <see cref= "Cmd_SAM_CombinedWriteMFP" /> interface
        /// </summary>
        public enum MFP_CWMode : byte
        {
            /// <summary> Option macro to return the plain data in response. </summary>
            RETURN_PLAIN = 0x00,

            /// <summary> Option macro to skip the plain data in response. </summary>
            SKIP_PLAIN = 0x02
        }
        #endregion
        #endregion

        #region X_Mode
        #region Sector Switch Authenticate
        /// <summary>
        /// Definitions common for MIFARE Plus X mode communication for SectorSwitch
        /// Authentication interface. To be used with <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public enum MFP_SSBuffer : ushort
        {
            /// <summary> Option to buffer the KeyBlocks information. </summary>
            KEY_BLOCKS =     0x10,

            /// <summary> Option to buffer the Diversification input information. </summary>
            DIV_INPUT = 0x20
        }
        #endregion
        #endregion
        #endregion

        #region MIFARE Ultralight
        /// <summary>
        /// Definitions for Key Diversification and LE Exchange.
        /// To be used with <see cref="Cmd_UL_AuthenticatePICC"/>interface.
        /// </summary>
        public enum MFUL_Option : byte
        {
            /// <summary> Option mask to enable the exchange of diversification input </summary>
            DIV_ON = 0x01,

            /// <summary> Option mask to include LE in exchange buffer </summary>
            INCLUDE_LE = 0x02
        }

        /// <summary>
        /// Definitions for Data to be written.
        /// To be used with <see cref="Cmd_MF_Write"/>interface.
        /// </summary>
        public enum MFUL_Write : byte
        {
            /// <summary>
            /// Option mask for writing data to MIFARE Ultralight in
            /// Plain compatible write (16 byte data blocks).
            /// </summary>
            PLAIN = 0x00,

            /// <summary>
            /// Option mask for writing data to MIFARE Ultralight product
            /// (4 byte data block).
            /// </summary>
            ULTRALIGHT = 0x01
        }

        /// <summary> Options to be used for RndA usage. </summary>
        public enum MFUL_Challenge : byte
        {
            /// <summary> Option to generate the random number internally. </summary>
            GENERATE_INTERNAL = 0x00,

            /// <summary> Option to use the random number provided by user. </summary>
            USER_PROVIDED = 0x01
        }
        #endregion

        #region MIFARE Common
        #region ISOMode
        /// <summary>
        /// Definitions for ISO Mode selection.
        /// To be used with <see cref="Cmd_TMRI_CommitReaderID"/> interface
        /// </summary>
        public enum TMRI_IsoMode : byte
        {
            /// <summary> SAM CommitReaderID ISO mode selection for Native command set. </summary>
            NATIVE = 0x00,

            /// <summary> SAM CommitReaderID ISO mode selection for ISO 7816-4 command set. </summary>
            ISO7816_4 = 0x40
        }
        #endregion

        #region State
        /// <summary>
        /// Definitions to identify the PICC for the command is
        /// exchanged to SAM. To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_CommitReaderID_Part1"/>
        ///     - <see cref="Cmd_TMRI_CommitReaderID"/>
        /// </summary>
        public enum TMRI_State : byte
        {
            /// <summary> Option to indicate the PICC state as MIFARE Plus EV1. </summary>
            PLUS = 0x00,

            /// <summary> Option to indicate the PICC state as MIFARE DESFire EV2. </summary>
            DESFIRE = 0x01
        }
        #endregion
        #endregion

        #region ICODE UCODE
        #region Cmd.SAM_AuthenticateTAM
        /// <summary>
        /// Definitions to Get IChallange or Exchange TResponse.
        /// To be used with below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateTAM1"/>
        ///     - <see cref="Cmd_SAM_AuthenticateTAM2"/>
        /// </summary>
        public enum Tam : byte
        {
            /// <summary> Option for exchanging Key and Div input and receive the IChallange as response. </summary>
            GET_RND = 0x00,

            /// <summary> Option for exchanging TResponse and validating it. </summary>
            PROCESS_TRESPONE = 0x01
        }
        #endregion

        #region Cmd.SAM_AuthenticateMAM
        /// <summary>
        /// Definitions for updating the PurposeMAM2.
        /// To be used with <see cref="Cmd_SAM_AuthenticateMAM1"/> interface.
        /// </summary>
        public enum Mam : byte
        {
            /// <summary> Option to indicate PurposeMAM2 value as None. Here only authentication will be performed.</summary>
            NONE = 0x00,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy disable until HF reset. </summary>
            DISABLE_PRIVACY_HF_RESET = 0x08,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy enable. </summary>
            ENABLE_PRIVACY = 0x09,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy disable. </summary>
            DISABLE_PRIVACY = 0x0A,

            /// <summary> Option to indicate PurposeMAM2 value as Destroy. </summary>
            DESTROY = 0x0B
        }
        #endregion
        #endregion

        #region ISO14443-3
        #region Cmd.ISO14443-3_Request_Wakeup
        /// <summary>
        /// Definitions for ISO14443 - 3 L3 command code.
        /// To be used with <see cref="Cmd_X_ISO14443_3_RequestA_Wakeup"/> interface
        /// </summary>
        public enum CmdCode : byte
        {
            /// <summary> Request command code. </summary>
            REQA = 0x26,

            /// <summary> Wakeup command code. </summary>
            WUPA = 0x52
        }
        #endregion

        #region Cmd.ISO14443-3_AnticollisionSelect
        /// <summary>
        /// Definitions for ISO14443 - 3 L3 Select (Cascade Level) codes.
        /// To be used with <see cref="Cmd_X_ISO14443_3_AnticollisionSelect"/> interface
        /// </summary>
        public enum CascadeLevel : byte
        {
            /// <summary> Option flag for Cascade Level 1 </summary>
            LEVEL_1 = 0x93,

            /// <summary> Option flag for Cascade Level 2 </summary>
            LEVEL_2 = 0x95,

            /// <summary> Option flag for Cascade Level 3 </summary>
            LEVEL_3 = 0x97
        }
        #endregion

        #region Cmd.ISO14443-3_ActivateIdle
        /// <summary>
        /// Definitions for ISO14443 - 3 L3 Activate Idle.
        /// To be used with <see cref="Cmd_X_ISO14443_3_ActivateIdle"/> interface
        /// </summary>
        public enum ActivateIdle : byte
        {
            /// <summary> Default option mask for ActivateIdle. </summary>
            DEFAULT = 0x00,

            /// <summary> Option flag for applying the ATQA filter. </summary>
            APPLY_ATQA = 0x01,

            /// <summary> Option flag for applying the SAK filter. </summary>
            APPLY_SAK = 0x02,

            /// <summary> Option flag for setting the time to wait. </summary>
            APPLY_TIME = 0x04
        }
        #endregion
        #endregion

        #region ISO14443-4
        /// <summary>
        /// Definitions for ISO14443 - 4 DeSelect.
        /// To be used with <see cref="Cmd_X_ISO14443_4_Deselect"/> interface
        /// </summary>
        public enum Deselect : byte
        {
            /// <summary> Option to not free CID is DeSelect fails. </summary>
            DO_NOT_FREE_CID = 0x00,

            /// <summary> Option to forcefully free CID in any case. </summary>
            FORCE_FREE_CID = 0x01
        }
        #endregion
        #endregion
        #endregion

        #region Data Structure
        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the lower layers parameter structure. </summary>
            public IntPtr pBalDataParams;

            /// <summary> Pointer to the HAL data params of a reader. NULL in case of X-Mode. </summary>
            public IntPtr pReaderHalDataParams;

            /// <summary> Pointer to the KeyStore used for Host Authentication. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the ENC crypto layers parameter structure. </summary>
            public IntPtr pENCCryptoDataParams;

            /// <summary> Pointer to the MAC crypto layers parameter structure. </summary>
            public IntPtr pMACCryptoDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary>
            /// Pointer to the ENC crypto layers parameter structure.
            /// This will be used for ProgrammableLogic feature only.
            /// </summary>
            public IntPtr pPLUpload_ENCCryptoDataParams;

            /// <summary>
            /// Pointer to the MAC crypto layers parameter structure.
            /// This will be used for ProgrammableLogic feature only.
            /// </summary>
            public IntPtr pPLUpload_MACCryptoDataParams;

            /// <summary> Command counter for Secure Messaging. </summary>
            public uint wCmd_Ctr;

            /// <summary>
            /// Indicated the one of the below configurations.
            ///     - <see cref="HcMode.AV3"/>
            ///     - <see cref="HcMode.AV4"/>
            /// </summary>
            public byte bHostMode;

            /// <summary> The current Authentication Type used for SM. </summary>
            public byte bAuthType;

            /// <summary> Store the current authentication key. </summary>
            public byte bKeyNo;

            /// <summary> Command Data pending for encryption. </summary>
            public fixed byte bPendingEncCmdData[16];

            /// <summary> Length of pending command data to encrypt. </summary>
            public byte bPendingEncCmdDataLength;

            /// <summary> InitVector for pending CMD-Data encryption. </summary>
            public fixed byte bPendingCmdIv[16];

            /// <summary> Command Data pending for MACing. </summary>
            public fixed byte bPendingMacCmdData[16];

            /// <summary> Length of pending command data to MAC. </summary>
            public byte bPendingMacCmdDataLength;

            /// <summary> InitVector for pending CMD-Data MACing. </summary>
            public fixed byte bPendingCmdMac[16];

            /// <summary> Response Data pending for MACing. </summary>
            public fixed byte bPendingMacRespData[16];

            /// <summary> Length of pending receive data to MAC. </summary>
            public byte bPendingMacRespDataLength;

            /// <summary> Intermediate response MAC. </summary>
            public fixed byte bPendingRespMac[16];

            /// <summary> InitVector for pending receive-Data decryption. </summary>
            public fixed byte bPendingRespIv[16];

            /// <summary> Pending (unreturned) response data. </summary>
            public fixed byte bPendingRespData[16];

            /// <summary> Length of pending response data. </summary>
            public byte bPendingRespDataLength;

            /// <summary> Type of secure messaging for current command. </summary>
            public byte bCmdSM;

            /// <summary> Type of secure messaging for current response. </summary>
            public byte bRespSM;

            /// <summary> Whether command chaining is active or not. </summary>
            public byte bCommandChaining;

            /// <summary> Whether response chaining is active or not. </summary>
            public byte bResponseChaining;

            /// <summary> SAM UID. </summary>
            public fixed byte bUid[7];

            /// <summary> Operation mode; One of the below values.
            ///     - <see cref="OpMode.NON_X"/>
            ///     - <see cref="OpMode.X_RC523"/>
            ///     - <see cref="OpMode.X_RC663"/>
            public byte bOpMode;

            /// <summary>
            /// Logical channel number to select for Authentication
            ///     - <see cref="LogicalChannel.LC0"/>
            ///     - <see cref="LogicalChannel.LC1"/>
            ///     - <see cref="LogicalChannel.LC2"/>
            ///     - <see cref="LogicalChannel.LC3"/>
            /// </summary>
            public byte bLogicalChannel;

            /// <summary> Pointer to global transmit buffer used by the Exchange() function. </summary>
            public IntPtr pTxBuffer;

            /// <summary> Size of the global transmit buffer. </summary>
            public ushort wTxBufSize;

            /// <summary> Number of valid bytes for exchange within the transmit buffer. </summary>
            public ushort wTxBufLen;

            /// <summary> Number of valid bytes for other commands within the transmit buffer. </summary>
            public ushort wTxBufLen_Cmd;

            /// <summary> Pointer to global receive buffer used by the Exchange() function. </summary>
            public IntPtr pRxBuffer;

            /// <summary> Size of the global receive buffer. </summary>
            public ushort wRxBufSize;

            /// <summary> Number of valid bytes within the receive buffer. </summary>
            public ushort wRxBufLen;

            /// <summary> Starting position within the global receive buffer. </summary>
            public ushort wRxBufStartPos;

            /// <summary> Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). </summary>
            public ushort wTxBufStartPos;

            /// <summary> Configuration shadow; Stores configuration for current CardType. </summary>
            public fixed ushort wCfgShadow[SHADOW_COUNT];

            /// <summary> Type of card for which the HAL is configured for. </summary>
            public byte bCardType;

            /// <summary>
            /// Unit of current timeout value
            ///     - <see cref="TimeUnit.MICROSECONDS"/>
            ///     - <see cref="TimeUnit.MILLISECONDS"/>
            /// </summary>
            public byte bTimeoutUnit;

            /// <summary> Field-Off-Time in milliseconds. </summary>
            public ushort wFieldOffTime;

            /// <summary> Field-Recovery-Time in milliseconds. </summary>
            public ushort wFieldRecoveryTime;

            /// <summary> Storage for additional error information. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Actual Error Code received from SAM. </summary>
            public ushort wErrorCode;

            /// <summary> Disables/Enables time measurement. </summary>
            public ushort wTimingMode;

            /// <summary> Current timing value. </summary>
            public uint dwTimingUs;

            /// <summary> Storage for NxpRdLibNet.Hal.Config.RFRESET_ON_TIMEOUT setting. </summary>
            public byte bRfResetAfterTo;

            /// <summary> Storage for NxpRdLibNet.Hal.SAM (AV4 and future SAM's).Config.DISABLE_NONX_CFG_MAPPING setting. </summary>
            public byte bDisableNonXCfgMapping;

            /// <summary> Current timing value backup for PC. </summary>
            public ushort wFdtPc;

            /// <summary>
            /// Buffer to store the complete segment information of PLUpload command.
            /// The buffer should have a memory size equivalent to the complete PLUpload Segment(s) size.
            /// </summary>
            public IntPtr pPLUploadBuff;

            /// <summary> The size of bytes available in PLUploadBuff buffer. </summary>
            public ushort wPLUploadBuffLen;

            /// <summary> Programmable Logic Initial session key for macing the data. </summary>
            public fixed byte aPLUploadSessMAC0[32];

            /// <summary> The current key type used for crypto operations. </summary>
            public byte bPLUploadKeyType;

            /// <summary>
            /// Pointer to global command buffer. This will be used to frame SAM Command.
            /// </summary>
            public IntPtr pCmdBuff;

            /// <summary>
            /// Size of the global command buffer.
            /// </summary>
            public ushort wCmdBuffSize;
        };
        #endregion Data Structure

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sam ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sam ()
        {
            // Free Buffers
            if ( m_pTxBuffer.IsAllocated )
            {
                m_pTxBuffer.Free ();
            }
            if ( m_pRxBuffer.IsAllocated )
            {
                m_pRxBuffer.Free ();
            }
            if ( m_pPLUploadBuffer.IsAllocated )
            {
                m_pPLUploadBuffer.Free ();
            }
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping

        #region DLL Import
        #region Initialization
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pBalDataParams,
            IntPtr pReaderHalDataParams, IntPtr pKeyStoreDataParams, IntPtr pCryptoENCDataParams, IntPtr pCryptoMACDataParams,
            IntPtr pCryptoRngDataParams, IntPtr pPLUpload_CryptoENCDataParams, IntPtr pPLUpload_CryptoMACDataParams,
            byte bOpMode, byte bLogicalChannel, IntPtr pTxBuffer, ushort wTxBufSize, IntPtr pRxBuffer,
            ushort wRxBufSize, IntPtr pPLUploadBuf, IntPtr pCmdBuff, ushort wCmdBuffSize );
        #endregion

        #region Utilities
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_DetectMode ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_7816Exchange ( IntPtr pDataParams, ushort wOption, byte[] pData,
            ushort wDataLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Utils_UpdateLc ( IntPtr pDataParams );
        #endregion

        #region Host Communication
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_LockUnlock ( ref DataParams_t pDataParams, byte bLockType, ushort wRdKeyNo,
            ushort wRdKeyVer, byte bSamKeyNo, byte bSamKeyVer, byte bUnlockKeyNo, byte bUnlockKeyVer, uint dwMaxChainBlocks );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateHost ( ref DataParams_t pDataParams, byte bHostMode, ushort wRdKeyNo,
            ushort wRdKeyV, byte bSamKeyNo, byte bSamKeyVer );
        #endregion

        #region Security and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GetVersion ( ref DataParams_t pDataParams, byte[] pVersion, ref byte pVersionLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DisableCrypto ( ref DataParams_t pDataParams, ushort wProMas );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ActivateOfflineKey ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pDivInput, byte bDivInputLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_LoadInitVector ( ref DataParams_t pDataParams, byte bOption, byte[] pData,
            byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_KillAuthentication ( ref DataParams_t pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_SelectApplication ( ref DataParams_t pDataParams, byte[] pDF_Aid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GetRandom ( ref DataParams_t pDataParams, byte bExpLen, byte[] pRnd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_Sleep ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_SetConfiguration ( ref DataParams_t m_pDataParams, byte bOption, byte[] pData,
            byte bDataLen );
        #endregion

        #region Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo, byte bProMas,
            byte[] bKeyData, byte bKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKeyEntryOffline ( ref DataParams_t pDataParams, byte bKeyNo, byte bProMas,
            ushort wChangeCtr, byte[] pOfflineCrypto, byte bOfflineCryptoLen, byte bEnableOfflineAck, ref IntPtr ppOfflineAck,
            ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKUCEntry ( ref DataParams_t pDataParams, byte bKucNo, byte bProMas,
            byte[] pKucData, byte bKucDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKUCEntryOffline ( ref DataParams_t pDataParams, byte bKucNo, byte bProMas,
            ushort wChangeCtr, byte[] pOfflineCrypto, byte bOfflineCryptoLen, byte bEnableOfflineAck, ref IntPtr ppOfflineAck,
            ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DisableKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DisableKeyEntryOffline ( ref DataParams_t pDataParams, byte bKeyNo,
            ushort wChangeCtr, byte[] pOfflineCrypto, byte bOfflineCryptoLen, byte bEnableOfflineAck, ref IntPtr ppOfflineAck,
            ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_EncipherKeyEntry ( ref DataParams_t pDataParams, byte bPersoKeyNo, byte bKeyNo,
            byte bOption, ushort wPersoCtr, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppOfflineCrypto, ref ushort pOfflineCryptoLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GetKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo, byte bMode,
            ref IntPtr ppKeyEntry, ref ushort pKeyEntryLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GetKUCEntry ( ref DataParams_t pDataParams, byte bKucNo, ref IntPtr ppKucEntry,
            ref ushort pKucEntryLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DumpSessionKey ( ref DataParams_t pDataParams, byte bDumpMode, ref IntPtr ppSessionKey,
            ref ushort pSessionKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DumpSecretKey ( ref DataParams_t pDataParams, byte bCrypto, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppSecretKey, ref ushort pSecretKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DeriveKey ( ref DataParams_t pDataParams, byte bSrcKeyNo, byte bSrcKeyVer,
            byte bDstKeyNo, byte[] pDeriveIn, byte bDeriveInLen );
        #endregion

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CreateFile ( ref DataParams_t pDataParams, byte bFileNo, byte bFileType,
            byte bKeyNoAEK_Read, byte bKeyVAEK_Read, byte bKeyNoAEK_Write, byte bKeyVAEK_Write, byte[] pFileSize,
            byte bCRLOptions, byte bCSNSize, byte bCSNSigKey );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ReadFile ( ref DataParams_t pDataParams, byte bOption, byte bFileNo,
            byte[] pOffset, byte[] pLength, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_WriteFile ( ref DataParams_t pDataParams, byte bOption, byte bFileNo,
            ushort wCRLVer, byte[] pOffset, byte[] pData, byte[] pLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_WriteFileOffline ( ref DataParams_t pDataParams, byte bFileNo, ushort wChangeCtr,
            byte[] pOfflineCrypto, uint dwOfflineCryptoLen, byte bEnableOfflineAck, ref IntPtr ppOfflineAck, ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GetFileSettings ( ref DataParams_t pDataParams, byte bFileNo,
            ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_Format ( ref DataParams_t pDataParams );
        #endregion

        #region Data Processing
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ApplySM ( ref DataParams_t pDataParams, ushort wOption, byte bCommMode,
            byte bOffset, byte bCmdCtrIncr, byte[] pData, byte bDataLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_RemoveSM ( ref DataParams_t pDataParams, ushort wOption, byte bCommMode,
            byte[] pData, byte bDataLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_VerifyMAC ( ref DataParams_t pDataParams, ushort wOption, byte bNum,
            byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_GenerateMAC ( ref DataParams_t pDataParams, ushort wOption, byte bNum,
            byte[] pData, byte bDataLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DecipherData ( ref DataParams_t pDataParams, ushort wOption,
            byte[] pEncData, ushort bEncDataLen, byte[] pLength, ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_EncipherData ( ref DataParams_t pDataParams, ushort wOption,
            byte[] pPlainData, byte bPlainDataLen, byte bOffset, ref IntPtr ppEncData, ref ushort pEncDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_DecipherOfflineData ( ref DataParams_t pDataParams, ushort wOption,
            byte[] pEncData, ushort bEncDataLen, ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_EncipherOfflineData ( ref DataParams_t pDataParams, ushort wOption,
            byte[] pPlainData, byte bPlainDataLen, ref IntPtr ppEncData, ref ushort pEncDataLen );
        #endregion

        #region Public Key Infrastructure
        #region RSA
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_GenerateKeyPair ( ref DataParams_t pDataParams, byte bOption, byte bPKI_KeyNo,
            ushort wPKI_Set, byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK,
            ushort wPKI_NLen, ushort wPKI_eLen, byte[] pPKI_e );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ImportKey ( ref DataParams_t pDataParams, byte bOption, byte bPKI_KeyNo,
            ushort wPKI_Set, byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK,
            ushort wPKI_NLen, ushort wPKI_eLen, ushort wPKI_PLen, ushort wPKI_QLen, byte[] pPKI_N, byte[] pPKI_e, byte[] pPKI_p,
            byte[] pPKI_q, byte[] pPKI_dP, byte[] pPKI_dQ, byte[] pPKI_ipq );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ExportPrivateKey ( ref DataParams_t pDataParams, ushort wOption, byte bPKI_KeyNo,
            ref IntPtr ppKeyData, ref ushort pKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ExportPublicKey ( ref DataParams_t pDataParams, ushort wOption, byte bPKI_KeyNo,
            ref IntPtr ppKeyData, ref ushort pKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_UpdateKeyEntries ( ref DataParams_t pDataParams, byte bOption, byte bNoOfKeyEntries,
            byte bHashingAlg, byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Ack, byte[] pKeyFrame, ushort wKeyFrameLen,
            ref IntPtr ppUpdateACK, ref ushort wUpdateACKLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_EncipherKeyEntries ( ref DataParams_t pDataParams, ushort wOption, byte bNoOfKeyEntries,
            byte bHashingAlg, byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Dec, byte bPKI_KeyNo_Verif, ushort wPerso_Ctr,
            byte[] pKeyEntries, byte bKeyEntriesLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppEncKeyFrame_Sign,
            ref ushort pEncKeyFrame_Sign_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_GenerateHash ( ref DataParams_t pDataParams, ushort wOption, byte bHashingAlg,
            uint dwMLen, byte[] pMessage, ushort wMsgLen, ref IntPtr ppHash, ref ushort pHashLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_GenerateSignature ( ref DataParams_t pDataParams, byte bHashingAlg,
            byte bPKI_KeyNo_Sign, byte[] pHash, byte bHashLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_SendSignature ( ref DataParams_t pDataParams, ref IntPtr ppSignature,
            ref ushort pSignatureLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_VerifySignature ( ref DataParams_t pDataParams, byte bPKI_KeyNo_Verif,
            byte bHashingAlg, byte[] pHash, byte bHashLen, byte[] pSignature, ushort wSignatureLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_EncipherData ( ref DataParams_t pDataParams, byte bHashingAlg,
            byte bPKI_KeyNo_Enc, byte[] pPlainData, ushort wPlainDataLen, ref IntPtr ppEncData, ref ushort pEncDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_DecipherData ( ref DataParams_t pDataParams, ushort wOption,
            byte bHashingAlg, byte bPKI_KeyNo_Dec, byte[] pEncData, ushort wEncDataLen, ref IntPtr ppPlainData,
            ref ushort pPlainDataLen );
        #endregion

        #region ECC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_GenerateEccKey ( ref DataParams_t pDataParams, byte bECC_KeyNo,
            ushort wECC_Set, byte bECC_KeyNoCEK, byte bECC_KeyVCEK, byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK,
            byte bECC_RefNoCurve, ref IntPtr ppECC_xy, ref ushort pECC_xyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ImportEccKey ( ref DataParams_t pDataParams, byte bOption, byte bECC_KeyNo,
            ushort wECC_Set, byte bECC_KeyNoCEK, byte bECC_KeyVCEK, byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK,
            ushort wECC_Len, byte[] pECC_KeyValue, ushort wECC_KeyValueLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ImportEccCurve ( ref DataParams_t pDataParams, byte bOption, byte bECC_CurveNo,
            byte bECC_KeyNoCCK, byte bECC_KeyVCCK, byte bECC_N, byte bECC_M, byte[] pECC_Prime, byte[] pECC_ParamA, byte[] pECC_ParamB,
            byte[] pECC_Px, byte[] pECC_Py, byte[] pECC_Order );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ExportEccPrivateKey ( ref DataParams_t pDataParams, byte bECC_KeyNo,
            ref ushort pECC_Set, ref byte pECC_KeyNoCEK, ref byte pECC_KeyVCEK, ref byte pECC_RefNoKUC, ref byte pECC_KeyNoAEK,
            ref byte pECC_KeyVAEK, ref ushort pECC_Len, ref byte pECC_RefNoCurve, ref IntPtr ppECC_Priv, ref byte pECC_PrivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ExportEccPublicKey ( ref DataParams_t pDataParams, byte bECC_KeyNo,
            ref ushort pECC_Set, ref byte pECC_KeyNoCEK, ref byte pECC_KeyVCEK, ref byte pECC_RefNoKUC, ref byte pECC_KeyNoAEK,
            ref byte pECC_KeyVAEK, ref ushort pECC_Len, ref IntPtr ppECC_xy, ref byte pECC_xyLen, ref byte pCRLFile );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_GenerateEccSignature ( ref DataParams_t pDataParams, byte bHashingAlg,
            byte bECC_KeyNo_Sign, byte[] pHash, byte bHashLen, ref IntPtr ppSignature, ref ushort pSigLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_VerifyEccSignature ( ref DataParams_t pDataParams, byte bECC_KeyNo,
            byte bECC_CurveNo, byte bLen, byte[] pMessage, byte[] pSignature, ushort wSignatureLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PKI_ValidateEccCert ( ref DataParams_t pDataParams, byte bCertFormat,
            byte bECC_KeyNo, byte bECC_CurveNo, byte bECC_Target, byte[] pCertificate, ushort wCertLen );
        #endregion
        #endregion

        #region Virtual Card
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_SelectVC ( ref DataParams_t pDataParams, byte bOption, byte bEncKeyNo, byte bEncKeyVer,
            byte bMacKeyNo, byte bMacKeyVer, byte[] pData, byte bDataLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse,
            ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pPPCData, byte bPPCDataLen, byte[] pPCData, byte bPCDataLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppMac,
            ref ushort pMacLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2 ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen,
            ref byte pPiccRetCode );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_VCA_Select ( ref DataParams_t pDataParams, byte bOption, byte bEncKeyNo, byte bEncKeyVer,
            byte bMacKeyNo, byte bMacKeyVer, byte[] pIID, byte bIIDLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse,
            ref ushort pRespLen, ref ushort pPiccRetCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_VCA_ProximityCheck ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte bNumOfRand, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse, ref ushort pRespLen );
        #endregion
        #endregion

        #region MIFARE DESFire
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte bAuthMode, byte[] pDivInput, byte bDivInputLen, byte[] pCardResponse, byte bCardRespLen,
            ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2 ( ref DataParams_t pDataParams, byte bPiccErrorCode,
            byte[] pCardResponse, byte bCardRespLen, byte[] pPDcap2, byte[] pPCDcap2, ref byte pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pCardResponse, byte bCardRespLen, ref IntPtr ppSamResponse,
            ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2 ( ref DataParams_t pDataParams, byte[] pCardResponse,
            byte bCardRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKeyPICC ( ref DataParams_t pDataParams, byte bCryptoMethod, byte bConfig,
            byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo, byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] pDivInput,
            byte bDivInputLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CreateTMFilePICC ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte[] pTMIExclFileMap, byte bTMKeyOptions,
            byte bTSIGKeyNo, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DESFire_AuthenticatePICC ( ref DataParams_t pDataParams, byte bOption, byte bISOMode,
            byte bDFKeyNo, byte bKeyNo, byte bKeyVer, byte bPCDcap2InLen, byte[] pPCDcap2In, byte[] pDivInput, byte bDivInputLen,
            byte[] pPDcap2, byte[] pPCDcap2, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC ( ref DataParams_t pDataParams, byte bCryptoMethod, byte bConfig,
            byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo, byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] pDivInput,
            byte bDivInputLen, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DESFire_WriteX ( ref DataParams_t pDataParams, ushort wOption, byte bCrypto, byte[] pData,
            byte bDataLen, byte[] pPiccReturnCode, ref byte pErrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DESFire_ReadX ( ref DataParams_t pDataParams, ushort wOption, byte bCrypto, byte[] pAppData,
            byte bAppDataLen, ref IntPtr ppResponse, ref ushort pRespLen, byte[] pPiccReturnCode, ref byte pErrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC ( ref DataParams_t pDataParams, byte bOption, byte bISOMode,
            byte bKeyNo, byte bKeyVer, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte[] pTMIExclFileMap, byte bTMKeyOptions,
            byte bTSIGKeyNo, byte[] pDivInput, byte bDivInputLen, byte[] pPiccReturnCode );
        #endregion
        #endregion

        #region MIFARE DUOX
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1 ( ref DataParams_t pDataParams, byte bOption,
            byte bECCKeyNo_Priv, byte bCertA_FileNo, byte bCertB_Options, byte bECCKeyNo_CA, ref IntPtr ppSamResponse,
            ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2 ( ref DataParams_t pDataParams, byte bOption,
            byte[] pPiccErrorCode, byte[] pCardResponse, byte bCardRespLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen,
            ref byte pPiccRetCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3 ( ref DataParams_t pDataParams, byte bOption,
            byte[] pPiccErrorCode, byte[] pCardResponse, byte bCardRespLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen,
            ref byte pPiccRetCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1 ( ref DataParams_t pDataParams, byte bOption,
            byte bECCKeyNo_Priv, byte bECC_CurveNo, ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2 ( ref DataParams_t pDataParams, byte[] pPiccErrorCode,
            byte[] pCardResponse, byte bCardRespLen, ref byte pPiccRetCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_BindCertificate_Part1 ( ref DataParams_t pDataParams, byte bOption,
            byte[] pMKPParams, byte bMKPParamsLen, byte bECCKeyNo_Priv, ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_BindCertificate_Part2 ( ref DataParams_t pDataParams, byte bOption,
            byte[] pData, ushort wDataLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ImportEccKeyDUOX ( ref DataParams_t pDataParams, byte bECCKeyNo_Priv,
            byte[] pMKPParams, byte bMKPParamsLen, ref IntPtr ppMKPCrypto, ref ushort pMKPCryptoLen );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DUOX_MutualAuthEcc ( ref DataParams_t pDataParams, byte bOption, byte[] pOptsA,
            byte bOptsALen, byte bDUOX_P2, byte bECCKeyNo_Priv, byte bCertA_FileNo, byte bCertB_Options, byte bECCKeyNo_CA,
            ref IntPtr ppSamResponse, ref ushort pSamRespLen, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DUOX_UnilatAuthEcc ( ref DataParams_t pDataParams, byte bOption, byte bECC_KeyNo,
            byte bECC_CurveNo, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DUOX_BindCertificate ( ref DataParams_t pDataParams, byte bOption, byte[] pMKPParams,
            byte bMKPParamsLen, byte bECCKeyNo_Priv, byte[] pTBSCertificate, ushort wTBSCertLen, ref IntPtr ppSamResponse,
            ref ushort pSamRespLen, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_DUOX_ImportEccKey ( ref DataParams_t pDataParams, byte bOption, byte[] pMKPParams,
            byte bMKPParamsLen, byte bECCKeyNo_Priv, byte[] pPiccReturnCode );
        #endregion
        #endregion

        #region MIFARE Plus
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pPDChal, byte bPDChalLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPCDChalResp,
            ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode,
            byte[] pPDResp, byte bPDRespLen, ref IntPtr ppPDCap2, ref IntPtr ppPCDCap2, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 ( ref DataParams_t pDataParams, byte bOption,
            byte[] pPDChal, byte bPDChalLen, ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer, byte bMSKeyNo, byte bMSKeyVer,
            byte bSectorCount, byte[] pKeyBlocks, byte bKeyBlocksLen, byte[] pDivInput, byte bDivInputLen,
            ref IntPtr ppPCDChalResp, ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode,
            byte[] pPDResp, byte bPDRespLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pPDChal, byte bPDChalLen, byte[] pUpgradeInfo, byte bLen, byte[] pDivInput, byte bDivInputLen,
            ref IntPtr ppPCDChalResp, ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode,
            byte[] pPDResp, byte bPDRespLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CombinedReadMFP ( ref DataParams_t pDataParams, byte bLFI, ushort wOption,
            byte[] pData, byte bDataLen, ref IntPtr ppOutput, ref ushort pOutputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CombinedWriteMFP ( ref DataParams_t pDataParams, ushort wOption, byte[] pData,
            byte bDataLen, ref IntPtr ppOutput, ref ushort pOutputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ChangeKeyMFP ( ref DataParams_t pDataParams, byte bOption, byte[] pData,
            byte bDataLen, ref IntPtr ppProtectedData, ref ushort pProtectedDataLen, ref byte pPiccReturnCode );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_Authenticate ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            ushort wBlockNo, byte[] pPcdCapsIn, byte bPcdCapsInLen, byte[] pDivInput, byte bDivInputLen, byte[] pPcdCapsOut, byte[] pPdCaps,
            ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_AuthSectorSwitch ( ref DataParams_t pDataParams, ushort wOption, ushort wSSKeyBNr,
            byte bSSKeyNo, byte bSSKeyVer, byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount, byte[] pKeyBlocks, byte bKeyBlocksLen,
            byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_PDC_Authenticate ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            ushort wUpgradeKey, byte[] pUpgradeInfo, byte bUpgradeInfoLen, byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_CombinedRead ( ref DataParams_t pDataParams, ushort wOption, byte[] pReadCmd,
            byte bReadCmdLen, ref IntPtr ppData, ref ushort pDataLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_CombinedWrite ( ref DataParams_t pDataParams, ushort wOption, byte[] pData,
            byte bDataLen, byte[] aTMC, byte[] aTMV, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_ChangeKey ( ref DataParams_t pDataParams, byte bOption, byte bCmdCode, ushort wBlockNo,
            byte bKeyNo, byte bKeyVer, byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MFP_WritePerso ( ref DataParams_t pDataParams, ushort wOption, byte[] pBlocks,
            byte bBlockLen, ref byte pPiccReturnCode );
        #endregion
        #endregion

        #region MIFARE Ultralight
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part1 ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPwd, ref ushort pPwdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part2 ( ref DataParams_t pDataParams, ushort wPack );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_UL_PwdAuthPICC ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, ref byte pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_UL_AuthenticatePICC ( ref DataParams_t pDataParams, byte bOption,
            byte bKeyNo, byte bKeyVer, byte[] pDivInput, byte bDivInputLen, ref byte pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MF_Read ( ref DataParams_t pDataParams, byte[] pBlocks, byte bBlocksLen,
            ref IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_MF_Write ( ref DataParams_t pDataParams, byte bOption, byte[] pData,
            byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_UL_SMExchange ( ref DataParams_t pDataParams, byte[] pData,
            byte bDataLen, ref IntPtr ppResponse, ref ushort pRespLen );
        #endregion
        #endregion

        #region MIFARE Common
        #region S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1 ( ref DataParams_t pDataParams, byte bState, ushort wBlockNr,
            ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode,
            byte[] pData, byte bDataLen, ref byte pPiccReturnCode );
        #endregion

        #region X_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_TMRI_CommitReaderID ( ref DataParams_t pDataParams, byte bISOMode, byte bState,
            ushort wBlockNr, ref IntPtr ppEncTMRI, ref ushort pEncTMRILen, ref byte pStatusCode );
        #endregion
        #endregion

        #region ICODE and UCODE S_Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateTAM1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pData, byte bDataLen, ref IntPtr ppIChallange, ref ushort pIChallangeLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateTAM2 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo,
            byte bKeyVer, byte[] pData, byte bDataLen, byte bBlockSize, byte bBlockCount, byte bProtMode, ref IntPtr ppResponse,
            ref ushort pResponseLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateMAM1 ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer,
            byte[] pData, byte bDataLen, byte bPurposeMAM2, ref IntPtr ppIChallange, ref ushort pIChallangeLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_AuthenticateMAM2 ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen,
            ref IntPtr ppIResponse, ref ushort pIResponseLen );
        #endregion

        #region Programmable Logic
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_PLExec ( ref DataParams_t pDataParams, byte bLFI, byte[] pPLData, byte bPLDataLen,
            ref IntPtr ppPLResp, ref ushort pPLResp );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_PLUpload ( ref DataParams_t pDataParams, byte bIsFirstFrame, byte bIsFinalFrame,
            ushort wUploadCtr, byte bKeyNo, byte bKeyVer, byte[] pPLCode, ushort wPLCodeLen, byte[] aPLReKey, byte bPLReKeyLen );
        #endregion

        #region Reader Chips
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_RC_ReadRegister ( ref DataParams_t pDataParams, byte[] pRegAddr, byte bRegAddrLen,
            byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_RC_WriteRegister ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_RC_RFControl ( ref DataParams_t pDataParams, ushort wTime );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_RC_Init ( ref DataParams_t pDataParams, byte bLoadReg );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_RC_LoadRegisterValueSet ( ref DataParams_t pDataParams, byte bStoreReg,
            byte[] pData, byte bDataLen );
        #endregion

        #region ISO14443-3
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup ( ref DataParams_t pDataParams, byte bCmdCode,
            byte[] pAtqa );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_AnticollisionSelect ( ref DataParams_t pDataParams, byte[] pSelCodes,
            byte bSelCodesLen, ref byte pSak, byte[] pUid, ref byte pUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle ( ref DataParams_t pDataParams, byte bOption,
            byte bNumCards, ushort wTime, byte[] pAtqaIn, byte[] pSakIn, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_ActivateWakeUp ( ref DataParams_t pDataParams, byte[] pUid,
            byte bUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_HaltA ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_3_TransparentExchange ( ref DataParams_t pDataParams, byte[] pTxBuf,
            byte bTxLen, byte bTxBitLen, ref IntPtr ppRxBuf, ref ushort pRxLen, ref byte pRxBitLen );
        #endregion

        #region ISO14443-4
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS ( ref DataParams_t pDataParams, byte bCidIn, byte bDsiIn,
            byte bDriIn, ref byte pCidOut, ref byte pDsiOut, ref byte pDriOut, byte[] pAts );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_Init ( ref DataParams_t pDataParams, byte bCid, byte bDri, byte bDsi,
            byte bFwi, byte bFsci );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_Exchange ( ref DataParams_t pDataParams, ushort wOption, byte[] pAppDataIn,
            byte bLenAppData, ref IntPtr ppAppDataOut, ref ushort pAppDataOutLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_PresenceCheck ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_Deselect ( ref DataParams_t pDataParams, byte bFreeCID );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_X_ISO14443_4_FreeCid ( ref DataParams_t pDataParams, byte[] pCid, byte bCidLen );
        #endregion

        #region Originality Check
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_Sam_Cmd_SAM_ISOInternalAuthenticate ( ref DataParams_t pDataParams, byte bOption,
            byte[] pOptsA, uint bOptsALen, ref byte[] pRndA, ref byte pRndALen, ref IntPtr ppResponse, ref ushort pRespLen );
        #endregion
        #endregion

        #region Wrapper Functions
        #region Initialization
        private byte[] m_bHalBuffer;
        private byte[] aCmdBuffer;
        private GCHandle m_pTxBuffer;
        private GCHandle m_pRxBuffer;
        private GCHandle m_pPLUploadBuffer;
        private GCHandle m_CmdBuffer;

        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="oPLUpload_ENCCrypto">Pointer to the ENC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="oPLUpload_MACCrypto">Pointer to the MAC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        /// <param name="aTxBuffer">Buffer for SAM command transmission. </param>
        /// <param name="aRxBuffer">Buffer for SAM command Reception. </param>
        /// <param name="aPLUploadBuf">Buffer for Programmable Logic operations.</param>
        /// <param name="aCmdBuf">Buffer for framing SAM command. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           oPLUpload_ENCCrypto, oPLUpload_MACCrypto are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, CryptoSym.Generic oPLUpload_ENCCrypto,
            CryptoSym.Generic oPLUpload_MACCrypto, OpMode eMode, LogicalChannel eLc, byte[] aTxBuffer,
            byte[] aRxBuffer, byte[] aPLUploadBuf, byte[] aCmdBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;
            IntPtr pCmdBuffer = IntPtr.Zero;

            if ( oBal != null ) pBalPtr = oBal.m_pDataParams;
            if ( oReaderHal != null ) pReaderHalPtr = oReaderHal.m_pDataParams;
            if ( oKeyStore != null ) pKeyStorePtr = oKeyStore.m_pDataParams;
            if ( oENCCrypto != null ) pENCCryptoPtr = oENCCrypto.m_pDataParams;
            if ( oMACCrypto != null ) pMACCryptoPtr = oMACCrypto.m_pDataParams;
            if ( oCryptoRng != null ) pCryptoRngPtr = oCryptoRng.m_pDataParams;
            if ( oPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = oPLUpload_ENCCrypto.m_pDataParams;
            if ( oPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = oPLUpload_MACCrypto.m_pDataParams;
            if ( m_CmdBuffer.IsAllocated ) m_CmdBuffer.Free ();

            // Free Buffer
            if ( m_pTxBuffer.IsAllocated ) m_pTxBuffer.Free ();
            if ( m_pRxBuffer.IsAllocated ) m_pRxBuffer.Free ();
            if ( m_pPLUploadBuffer.IsAllocated ) m_pPLUploadBuffer.Free ();

            // Use only one Buffer with fixed length
            m_pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );
            m_pPLUploadBuffer = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );
            m_CmdBuffer = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            return phhalHw_Sam_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pBalPtr,
                pReaderHalPtr, pKeyStorePtr, pENCCryptoPtr, pMACCryptoPtr, pCryptoRngPtr, pPLUpload_ENCCryptoPtr,
                pPLUpload_MACCryptoPtr, ( byte ) eMode, ( byte ) eLc, m_pTxBuffer.AddrOfPinnedObject (),
                ( ushort ) aTxBuffer.Length, m_pRxBuffer.AddrOfPinnedObject (), ( ushort ) aRxBuffer.Length,
                m_pPLUploadBuffer.AddrOfPinnedObject (), m_CmdBuffer.AddrOfPinnedObject (),
                ( ushort ) aCmdBuf.Length );
        }

        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="oPLUpload_ENCCrypto">Pointer to the ENC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="oPLUpload_MACCrypto">Pointer to the MAC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        /// <param name="aPLUploadBuf">Buffer for Programmable Logic operations.</param>
        /// <param name="aCmdBuf">Buffer for framing SAM command. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           oPLUpload_ENCCrypto, oPLUpload_MACCrypto are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, CryptoSym.Generic oPLUpload_ENCCrypto,
            CryptoSym.Generic oPLUpload_MACCrypto, OpMode eMode, LogicalChannel eLc, byte[] aPLUploadBuf,
            byte[] aCmdBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;
            IntPtr pCmdBuffer = IntPtr.Zero;

            if ( oBal != null ) pBalPtr = oBal.m_pDataParams;
            if ( oReaderHal != null ) pReaderHalPtr = oReaderHal.m_pDataParams;
            if ( oKeyStore != null ) pKeyStorePtr = oKeyStore.m_pDataParams;
            if ( oENCCrypto != null ) pENCCryptoPtr = oENCCrypto.m_pDataParams;
            if ( oMACCrypto != null ) pMACCryptoPtr = oMACCrypto.m_pDataParams;
            if ( oCryptoRng != null ) pCryptoRngPtr = oCryptoRng.m_pDataParams;
            if ( oPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = oPLUpload_ENCCrypto.m_pDataParams;
            if ( oPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = oPLUpload_MACCrypto.m_pDataParams;

            // Free Buffer
            if ( m_pTxBuffer.IsAllocated ) m_pTxBuffer.Free ();
            if ( m_pRxBuffer.IsAllocated ) m_pRxBuffer.Free ();
            if ( m_pPLUploadBuffer.IsAllocated ) m_pPLUploadBuffer.Free ();
            if ( m_CmdBuffer.IsAllocated ) m_CmdBuffer.Free ();

            // Use only one Buffer with fixed length
            m_bHalBuffer = new byte[MAX_BUFFER_SIZE];
            m_pTxBuffer = GCHandle.Alloc ( m_bHalBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( m_bHalBuffer, GCHandleType.Pinned );
            m_pPLUploadBuffer = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );
            m_CmdBuffer = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            return phhalHw_Sam_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pBalPtr,
                pReaderHalPtr, pKeyStorePtr, pENCCryptoPtr, pMACCryptoPtr, pCryptoRngPtr, pPLUpload_ENCCryptoPtr,
                pPLUpload_MACCryptoPtr, ( byte ) eMode, ( byte ) eLc, m_pTxBuffer.AddrOfPinnedObject (),
                MAX_BUFFER_SIZE, m_pRxBuffer.AddrOfPinnedObject (), MAX_BUFFER_SIZE,
                m_pPLUploadBuffer.AddrOfPinnedObject (),
                m_CmdBuffer.AddrOfPinnedObject (),
                ( ushort ) aCmdBuf.Length );
        }

        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        /// <param name="aTxBuffer">Buffer for SAM command transmission. </param>
        /// <param name="aRxBuffer">Buffer for SAM command Reception. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, OpMode eMode, LogicalChannel eLc, byte[] aTxBuffer,
            byte[] aRxBuffer )
        {
            aCmdBuffer = new byte[MAX_BUFFER_SIZE];
            return Init ( oBal, oReaderHal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng, null, null, eMode, eLc,
                aTxBuffer, aRxBuffer, new byte[0], aCmdBuffer);
        }

        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, OpMode eMode, LogicalChannel eLc )
        {
            aCmdBuffer = new byte[MAX_BUFFER_SIZE];
            return Init ( oBal, oReaderHal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng, null, null, eMode, eLc,
                new byte[0], aCmdBuffer );
        }

#if DEBUG
        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        ///  <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="oPLUpload_ENCCrypto">Pointer to the ENC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="oPLUpload_MACCrypto">Pointer to the MAC crypto layers parameter structure.
        ///                                   This will be used for ProgrammableLogic feature only.
        /// </param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        /// <param name="aTxBuffer">Buffer for SAM command transmission. </param>
        /// <param name="aRxBuffer">Buffer for SAM command Reception. </param>
        /// <param name="aPLUploadBuf">Buffer for Programmable Logic operations.</param>
        /// <param name="aCmdBuf">Buffer for framing SAM command. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           oPLUpload_ENCCrypto, oPLUpload_MACCrypto are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init( int wDataParamSize, Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, CryptoSym.Generic oPLUpload_ENCCrypto, CryptoSym.Generic oPLUpload_MACCrypto,
            OpMode eMode, LogicalChannel eLc, int wTxBufferSize, byte[] aTxBuffer, int wRxBufferSize, byte[] aRxBuffer, byte[] aPLUploadBuf,
            byte[] aCmdBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;
            IntPtr pCmdBuffer = IntPtr.Zero;

            if (oBal != null) pBalPtr = oBal.m_pDataParams;
            if (oReaderHal != null) pReaderHalPtr = oReaderHal.m_pDataParams;
            if (oKeyStore != null) pKeyStorePtr = oKeyStore.m_pDataParams;
            if (oENCCrypto != null) pENCCryptoPtr = oENCCrypto.m_pDataParams;
            if (oMACCrypto != null) pMACCryptoPtr = oMACCrypto.m_pDataParams;
            if (oCryptoRng != null) pCryptoRngPtr = oCryptoRng.m_pDataParams;
            if ( oPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = oPLUpload_ENCCrypto.m_pDataParams;
            if ( oPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = oPLUpload_MACCrypto.m_pDataParams;

            // Free Buffers
            if ( m_pTxBuffer.IsAllocated ) m_pTxBuffer.Free ();
            if ( m_pRxBuffer.IsAllocated ) m_pRxBuffer.Free ();
            if ( m_pPLUploadBuffer.IsAllocated ) m_pPLUploadBuffer.Free ();
            if ( m_CmdBuffer.IsAllocated ) m_CmdBuffer.Free ();

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(aTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(aRxBuffer, GCHandleType.Pinned);
            m_pPLUploadBuffer = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );
            m_CmdBuffer = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            return phhalHw_Sam_Init ( ref m_DataParamsInt[0], (ushort) wDataParamSize, pBalPtr, pReaderHalPtr, pKeyStorePtr,
                pENCCryptoPtr, pMACCryptoPtr, pCryptoRngPtr, pPLUpload_ENCCryptoPtr, pPLUpload_MACCryptoPtr, (byte) eMode,
                (byte) eLc, m_pTxBuffer.AddrOfPinnedObject (), (ushort) wTxBufferSize, m_pRxBuffer.AddrOfPinnedObject (),
                (ushort) wRxBufferSize, m_pPLUploadBuffer.AddrOfPinnedObject (), m_CmdBuffer.AddrOfPinnedObject (),
                ( ushort ) aCmdBuf.Length );
        }

        /// <summary>
        /// Interface to initialize the SAM (AV4 and future SAM's) component.
        ///     - \b pTxBuffer should be Size + Reserved Tx Buffer Len. If \b pTxBuffer is 256, then it should be 256 + 5.
        ///     - \b pRxBuffer should be Size + Reserved Rx Buffer Len. If \b pRxBuffer is 256, then it should be 256 + 2.
        ///     - The operation mode used is only dependent on the \b pReaderHalDataParams parameter.
        ///         - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed,
        ///           otherwise the HAL operates in S-Mode ( Non-X) Mode.
        ///         - <b>Non-X Mode Specific:</b> It is \b not necessary to do any ( non-specific) calls to
        ///           the linked Reader-HAL except an \b Init (), the SAM HAL will take over complete control
        ///           of the linked Reader.
        ///     - Specific initialization like <see cref="Hal.Config.BAL_CONNECTION"/> or <see cref="Hal.Config.SERIAL_BITRATE"/>
        ///       etc, though may be necessary depending on the used HAL.
        ///     - In case of S-Mode communication,
        ///     - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///       component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked to one of the following,
        ///     - <see cref="Bal.PcscWin"/> for PCSC interface.
        ///     - <see cref="Bal.SerialWin"/> for Serial ( COMPORT) interface.
        ///     - Pegoda - 2 reader ( MFEV710)
        ///         - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with <see cref="Bal.SAM"/>
        ///           component and <see cref="Bal.SAM"/> \b pLowerBalDataParams parameter should be linked to <see cref="Bal.PcscWin"/>.
        ///         - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
        ///         <see cref="Bal.PcscWin"/> component.
        ///     - In case of X-Mode communication for Pegoda - 2 reader ( MFEV710, SAM inserted in Reader slot ), \b pBalDataParams parameter
        ///     should be initialized with <see cref="Bal.PcscWin"/> component.
        /// </summary>
        ///
        ///  <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication interfaces.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="eMode">The desired operation mode. One of the below values
        ///                     - <see cref="OpMode.NON_X"/>
        ///                     - <see cref="OpMode.X_RC523"/>
        ///                     - <see cref="OpMode.X_RC663"/>
        /// </param>
        /// <param name="eLc">The desired logical channel for this HAL.
        ///                     - <see cref="LogicalChannel.LC0"/>
        ///                     - <see cref="LogicalChannel.LC1"/>
        ///                     - <see cref="LogicalChannel.LC2"/>
        ///                     - <see cref="LogicalChannel.LC3"/>
        /// <param name="aTxBuffer">Buffer for SAM command transmission. </param>
        /// <param name="aRxBuffer">Buffer for SAM command Reception. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the objects oBal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng,
        ///           are NULL.
        ///         - If the pointers are NULL.
        ///         - If \b wTxBufSize &#60;= Reserved Transmit Buffer Len.
        ///         - If \b wRxBufSize &#60;= Reserved Response Buffer Length
        ///         - If \b bOpMode is not one of the following
        ///             - <see cref="OpMode.NON_X"/>
        ///             - <see cref="OpMode.X_RC523"/>
        ///             - <see cref="OpMode.X_RC663"/>
        ///         - If \b bOpMode = <see cref="OpMode.NON_X"/> and \b oReaderHal is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Init ( int wDataParamSize, Bal.Generic oBal, Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto,
            CryptoSym.Generic oMACCrypto, CryptoRng.Generic oCryptoRng, OpMode eMode, LogicalChannel eLc, int wTxBufferSize, byte[] aTxBuffer,
            int wRxBufferSize, byte[] aRxBuffer )
        {
            aCmdBuffer = new byte[MAX_BUFFER_SIZE];
            return Init ( wDataParamSize, oBal, oReaderHal, oKeyStore, oENCCrypto, oMACCrypto, oCryptoRng, null, null, eMode, eLc,
                wTxBufferSize, aTxBuffer, wRxBufferSize, aRxBuffer, new byte[0], aCmdBuffer );
        }
#endif
        #endregion

        #region Utilities
        /// <summary>
        /// Saves UID and mode of SAM (AV3, AV4, future SAM's). This function should be called prior
        /// to all other functions even before Host Authentication.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t DetectMode ()
        {
            return phhalHw_Sam_DetectMode ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// <para>Exchange commands with the SAM.</para>
        ///
        /// \note If this interface is called with one of the below mentioned buffering
        /// options and if LC needs to be updated before the final call with \b wOption =
        /// <see cref="ExchangeOptions.BUFFER_LAST"/>: Buffer last information,
        /// then <see cref="Sam_Utils_UpdateLc"/> interface can be used.
        /// <see cref="Sam_Utils_UpdateLc"/> will update the LC byte which
        /// the data that is loaded via \b pData parameter.
        ///     - \b wOption = <see cref="ExchangeOptions.BUFFER_FIRST"/> Buffer first set
        ///       of information. Exchange is not performed
        ///     - \b wOption = <see cref="ExchangeOptions.BUFFER_CONT"/> Buffer intermediate
        ///       information. Exchange is not performed
        ///     - \b wOption = <see cref="ExchangeOptions.BUFFER_LAST"/> Buffer last set of
        ///       information. Exchange and received information from SAM
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange and Receive information from SAM
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information. Exchange is
        ///                           not performed
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///                           Buffer intermediate information. Exchange is
        ///                           not performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///                           Buffer last information, exchange and received
        ///                           information from SAM
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                           Exchange chained data
        ///                         - <see cref="ExchangeOptions.RXCHAINING"/>:
        ///                           Receive chained data
        /// </param>
        /// <param name="aData">The command to be sent to Sam hardware.</param>
        /// <param name="aResponse">The data received form SAM hardware.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_7816Exchange ( int wOption, byte[] aData, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_7816Exchange ( m_pDataParams, (ushort) wOption, aData,
                (ushort) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Update LC byte according to data already available in Buffer
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Sam_Utils_UpdateLc ()
        {
            return phhalHw_Sam_Utils_UpdateLc ( m_pDataParams );
        }

        /// <summary>
        /// Set the configuration.
        /// </summary>
        ///
        /// <param name="eConfig">Configuration to set. One of the following
        ///                         - <see cref="Config.HOSTMODE"/>
        ///                         - <see cref="Config.DISABLE_NONX_CFG_MAPPING"/>
        ///                         - <see cref="Hal.Config"/> for rest of the configuration.
        /// </param>
        /// <param name="wValue">The value to be set for the mentioned configuration.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t SetConfig ( Config eConfig, int wValue )
        {
            return SetConfig ( ( Hal.Config ) eConfig, wValue );
        }

        /// <summary>
        /// Get the configuration.
        /// </summary>
        ///
        /// <param name="eConfig">Configuration to set. One of the following
        ///                         - <see cref="Config.HOSTMODE"/>
        ///                         - <see cref="Config.DISABLE_NONX_CFG_MAPPING"/>
        ///                         - <see cref="Hal.Config"/> for rest of the configuration.
        /// </param>
        /// <param name="wValue">The value returned for the mentioned configuration.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetConfig ( Config eConfig, out int wValue )
        {
            return GetConfig ( ( Hal.Config ) eConfig, out wValue );
        }
        #endregion

        #region Host Communication
        /// <summary>
        /// Activate, Lock or Unlock the SAM.
        ///     - The SAM_LockUnlock command is used to modify the SAM Global State for all active LC's, and to activate
        ///       the SAM.
        ///     - It is implemented by mean of a mutual authentication between the SAM and host system.
        ///     - Such authentication proves that both the SAM and the host contain the same secret, namely the AES
        ///       authentication key Kx.
        ///     - Only authorized Host systems are allowed to change the SAM Global State.
        ///     - Starting SAM_LockUnlock command invalidates any other existing authentication (incl.Offline Key Activation)
        ///       on any of the logical channels.
        ///     - If key entry (SAMKey) is not of AES_128, AES_192 or AES_256 key, no authentication is possible with this
        ///       key entry.
        ///     - If key entry (SAMKey != MasterKey) is not a LockKey, no authentication is possible with this key entry.
        ///     - During the first part of the authentication, the host indicates whether the SAM shall execute the SAM Activation,
        ///       the lock or the unlock operation.
        ///     - The activation is used to protect the SAM against tampering before delivery to and to provide a way to configure\
        ///       some settings of the SAM.
        ///
        ///     \note
        ///     - It's important to perform Host Authentication after successful LockUnlock operation to set the new session keys.
        ///     - Host Authentication is required because, LockUnlock interface will utilize the same CryptoParams which was utilized
        ///       by Host Authentication interface.
        ///     - Since the CryptoParams are common, the keys will be updated by LockUnlock interface and the existing Host
        ///       Authentication session keys will not be available.
        /// </summary>
        ///
        /// <param name="bLockType">Sub Command type.
        ///                             - <see cref="LockUnlock.UNLOCK"/>
        ///                             - <see cref="LockUnlock.LOCK_NO_KEY"/>
        ///                             - <see cref="LockUnlock.LOCK_KEY"/>
        ///                             - <see cref="LockUnlock.ACTIVATE_SAM"/>
        ///                             - <see cref="LockUnlock.UNLOCK_PL"/>
        /// <param name="wRdKeyNo">Key reference number to be used in Software KeyStore.</param>
        /// <param name="wRdKeyVer">Key version to be used in Software KeyStore..</param>
        /// <param name="bSamKeyNo">Key reference number in SAM.  One of the following,
        ///                             - NVM Key: 0x00 - 0x07
        ///                             - PL Unlock Key: 0xF0
        /// </param>
        /// <param name="bSamKeyVer">Key version to be used in SAM.</param>
        /// <param name="bUnlockKeyNo">Unlock Key Number to be used in SAM (only used
        ///                            when \b bLockType = <see cref="LockUnlock.LOCK_NO_KEY"/>.
        /// </param>
        /// <param name="bUnlockKeyVer">Unlock Key Version to be used in SAM (only used
        ///                             when \b bLockType = <see cref="LockUnlock.LOCK_NO_KEY"/>.
        /// </param>
        /// <param name="dwMaxChainBlocks">Maximal message size under command chaining in MAC or Full Protection (only used
        ///                                when \b bLockType = <see cref="LockUnlock.ACTIVATE_SAM"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> If \b wRdKeyNo is not one of AES128, AES192 or AES256
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/> If response length received is not as expected
        ///     for part1, part2 and part3.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> The response for RndA is not equal to sent RndA
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_LockUnlock ( byte bLockType, ushort wRdKeyNo, ushort wRdKeyVer, byte bSamKeyNo, byte bSamKeyVer,
            byte bUnlockKeyNo, byte bUnlockKeyVer, uint dwMaxChainBlocks )
        {
            return phhalHw_Sam_Cmd_SAM_LockUnlock ( ref m_DataParamsInt[0], bLockType, wRdKeyNo, wRdKeyVer, bSamKeyNo, bSamKeyVer,
                bUnlockKeyNo, bUnlockKeyVer, dwMaxChainBlocks );
        }

        /// <summary>
        /// Mutual 3-pass authentication between Host and SAM.
        ///     - The SAM_AuthenticateHost will perform a mutual authentication for a specific LC between the SAM and host.
        ///       system.Such an authentication proves that both the SAM and the host contain the same secret, namely the
        ///       AES authentication key Kx.
        ///     - Starting SAM_AuthenticateHost command invalidates any other existing authentication (incl.Offline Key
        ///       Activation) in the LC.
        ///     - If the authentication key entry (SAMKey) is not of AES_128, AES_192 or AES_256, no authentication is
        ///       possible with this key entry.
        ///     - If the authentication key entry (SAMKey != MasterKey) is not a AuthKey, no authentication is possible
        ///       with this key entry.
        ///     - SAM_AuthenticateHost is executed in three parts.
        ///         - During the first part of the authentication, the host indicates the AES authentication key (Kx) and
        ///           the selected protection mode ( HostMode) to be used by the established SAC: Plain, MAC or Full mode.
        ///         - If an invalid key entry and/or key version are referenced, or if the key to be used is not an AuthKey
        ///           (or an AuthLockKey in the SAM Locked State ), an error is returned.
        ///         - If the provided HostMode is not one of the expected values, an error is returned and the protocol is
        ///           aborted.
        /// </summary>
        ///
        /// <param name="bHostMode">Type of Protection mode to be applied.
        ///                             - <see cref="HostMode.PLAIN"/>
        ///                             - <see cref="HostMode.MAC"/>
        ///                             - <see cref="HostMode.FULL"/>
        /// </param>
        /// <param name="wRdKeyNo">Key reference number to be used in Software KeyStore.</param>
        /// <param name="wRdKeyVer">Key version to be used in Software KeyStore.</param>
        /// <param name="bSamKeyNo">Key reference number in SAM.</param>
        /// <param name="bSamKeyVer">Key version to be used in SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> If \b wRdKeyNo is not one of AES128, AES192 or AES256
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/> If response length received is not as expected
        ///     for part1, part2 and part3.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> The response for RndA is not equal to sent RndA
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateHost ( byte bHostMode, ushort wRdKeyNo, ushort wRdKeyVer, byte bSamKeyNo, byte bSamKeyVer )
        {
            return phhalHw_Sam_Cmd_SAM_AuthenticateHost ( ref m_DataParamsInt[0], bHostMode, wRdKeyNo, wRdKeyVer, bSamKeyNo,
                bSamKeyVer );
        }
        #endregion

        #region Security and Configuration
        /// <summary>
        /// Get version information from the SAM. Returns
        ///     - <b>Bytes[0 - 6]: Hardware version information</b>
        ///         - Byte[0]      : Vendor ID ( set to 0x04 for NXP)
        ///         - Byte[1]      : Type
        ///         - Byte[2]      : Subtype
        ///         - Byte[3]      : Major version number
        ///         - Byte[4]      : Minor version number
        ///         - Byte[5]      : Storage size
        ///         - Byte[6]      : Communication protocol type ( set to 0x01, meaning T = 1 )
        ///
        ///     - <b>Bytes[7 - 13]: Software version information</b>
        ///         - Byte[7]      : Vendor ID ( set to 0x04 for NXP)
        ///         - Byte[8]      : Type
        ///         - Byte[9]      : Subtype
        ///         - Byte[10]     : Major version number
        ///         - Byte[11]     : Minor version number
        ///         - Byte[12]     : Storage size
        ///         - Byte[13]     : Communication protocol type ( set to 0x01, meaning T = 1 )
        ///
        ///     - <b>Bytes[14 - 30]: Manufacturer data</b>
        ///         - Byte[14 - 20]: Code the unique serial number
        ///         - Byte[11 - 25]: Code the production batch number
        ///         - Byte[26]     : Code the day of production[0x00..0x1F]
        ///         - Byte[27]     : Code the month of production[0x00..0x0C]
        ///         - Byte[28]     : Code the year of production (year is 2000 + code)
        ///         - Byte[29]     : Global crypto settings; most significant byte of CryptoSecRow
        ///         - Byte[30]     : Code the SAM version and activated state ( set to 0x0X for the unactivated MIFARE SAM AVx,
        ///                          set to 0xAX after SAM Activation ). For Ex: 0x04 will be for SAM AV4 Unactivated and 0xA4 will
        ///                          be for SAM AV4 after SAM Activation. For future SAM version the number will increment from
        ///                          lower nibble.
        /// </summary>
        ///
        /// <param name="aVersion">Buffer containing the read version.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GetVersion ( out byte[] aVersion )
        {
            aVersion = new byte[31];
            byte bVerLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_GetVersion ( ref m_DataParamsInt[0], aVersion, ref bVerLen );
            Array.Resize ( ref aVersion, bVerLen );

            return oStatus;
        }

        /// <summary>
        /// Disable Crypto-related features of the SAM permanently and is irreversible.
        ///     - The SAM shall maintain the status of the cryptographic functionality in the Global Crypto
        ///       Security Row (CryptoSecRow).
        ///     - The command shall apply the ProMas bits to the CryptoSecRow and shall not be able to unset
        ///       the bit ( cannot re-enable the functionality).
        ///     - The initial value of the CryptoSecRow are set to 0x0000, therefore full cryptographic functionality
        ///       is available.
        ///     - The configuration settings ( SET) of all KST key entries shall reflect the changes after the execution
        ///       of SAM_DisableCrypto.
        /// </summary>
        ///
        /// <param name="wProMas">Two byte mask to specify the desired settings for
        ///                       cryptography-related features.
        ///                         - <see cref="DisableCrypto.NO_CHANGE"/>
        ///                         - <see cref="DisableCrypto.DES_PICC_CHANGE_KEY"/>
        ///                         - <see cref="DisableCrypto.DECRYPTION"/>
        ///                         - <see cref="DisableCrypto.ENCRYPTION"/>
        ///                         - <see cref="DisableCrypto.MAC_VERIFICATION"/>
        ///                         - <see cref="DisableCrypto.MAC_GENERATION"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DisableCrypto ( ushort wProMas )
        {
            return phhalHw_Sam_Cmd_SAM_DisableCrypto ( ref m_DataParamsInt[0], wProMas );
        }

        /// <summary>
        /// Activation of an OfflineCrypto or an OfflineChange Key.
        /// SAM_ActivateOfflineKey shall activate Offline keys for further offline data processing.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information if Keytype is AES_128LRP.
        ///                         - <see cref="LRPUpdate.ONE"/>
        ///                         - <see cref="LRPUpdate.TWO"/>
        /// </param>
        /// <param name="bKeyNo">Key reference number in hardware KeyStore.</param>
        /// <param name="bKeyVer">Key version to be used in hardware key store.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ActivateOfflineKey ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput )
        {
            return phhalHw_Sam_Cmd_SAM_ActivateOfflineKey ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );

        }

        /// <summary>
        /// Load an Init Vector for the next cryptographic operation into the SAM.
        /// </summary>
        ///
        /// <param name="bOption">One of the below option.
        ///                         - <see cref="LoadIV.SET_IV"/>
        ///                         - <see cref="LoadIV.SET_LRP_ENC_CTR"/>
        /// </param>
        /// <param name="aData">Data based on the option selected.
        ///                         - If <see cref="LoadIV.SET_IV"/>: 8 bytes of initialization
        ///                           vector for DES or 16 bytes of initialization vector for AES.
        ///                         - If <see cref="LoadIV.SET_LRP_ENC_CTR"/>: 1 - 16 bytes
        ///                           Encryption counter.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_LoadInitVector ( byte bOption, byte[] aData )
        {
            return phhalHw_Sam_Cmd_SAM_LoadInitVector ( ref m_DataParamsInt[0], bOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Kill all active authentications in this logical channel.
        /// SAM_KillAuthentication invalidates either any authentication or all but the Host Authentication
        /// on the LC on which the command is applied
        /// </summary>
        ///
        /// <param name="bOption">The type of authentication to be killed.
        ///                         - <see cref="KillAuth.PARTIAL"/>
        ///                         - <see cref="KillAuth.FULL"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_KillAuthentication ( byte bOption )
        {
            return phhalHw_Sam_Cmd_SAM_KillAuthentication ( ref m_DataParamsInt[0], bOption );
        }

        /// <summary>
        /// Select an application by the DF_AID
        ///     - SAM_SelectApplication is the equivalent of the SelectApplication command of DESFire.
        ///     - The SAM generates a list of available keys linked to the specified Application ID as defined
        ///       in the key entry property DF_AID.
        ///     - For every key number, up to 6 key versions can be stored in the list (so it can read the keys
        ///       from maximum two key entries per DESFire AID and DESFire key number ).
        ///     - This list is filled starting with key entry zero.If the KST contains more than 6 key versions
        ///       per DESFire AID and DESFire key number, only the first 6 versions will be listed.
        /// </summary>
        ///
        /// <param name="aDF_Aid">DESFire application identifier (3 bytes).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aDF_Aid is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_SelectApplication ( byte[] aDF_Aid )
        {
            return phhalHw_Sam_Cmd_SAM_SelectApplication ( ref m_DataParamsInt[0], aDF_Aid );
        }

        /// <summary>
        /// Gets a random number from SAM.
        /// </summary>
        ///
        /// <param name="bExpLen">The length of random bytes expected form SAM.</param>
        /// <param name="aRnd">The random number returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/> If Number of returned bytes
        ///     are not equal expected number.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GetRandom ( byte bExpLen, out byte[] aRnd )
        {
            aRnd = new byte[bExpLen];
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_GetRandom ( ref m_DataParamsInt[0], bExpLen, aRnd );
            if ( !oStatus.Equals ( new Status_t () ) )
                aRnd = null;

            return oStatus;
        }

        /// <summary>
        /// Set the SAM into power down mode.
        ///     - SAM_Sleep will force the SAM to put a connected reader chip into sleep mode and itself
        ///       into idle mode to reduce power consumption
        ///     - The SAM will answer the command and afterwards switch to idle mode.
        ///     - The SAM will automatically return to normal operation after receiving the first character
        ///       of the next command.
        ///     - The reader chip will stay in sleep mode until a command is issued which utilizes the reader IC.
        ///     - Then the SAM automatically carries out the wake-up sequence before starting the requested operation.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_Sleep ()
        {
            return phhalHw_Sam_Cmd_SAM_Sleep ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Used to update SAM configuration settings.
        /// A successful Host Authentication with one of the three SAM master keys ( KeyNo set to 0x00) is
        /// required to perform the command
        /// </summary>
        ///
        /// <param name="bOption">Configuration setting ID. Define length and content of the Data parameter.
        ///                         <see cref="SetConfiguration.HISTORICAL_BYTES"/>
        ///                         <see cref="SetConfiguration.READER_IC_CONFIG"/>
        ///                         <see cref="SetConfiguration.I2C_CLOCK_SPEED"/>
        ///                         <see cref="SetConfiguration.FULL_ATR"/>
        /// </param>
        /// <param name="aData">Configuration setting data.
        ///                         - If <see cref="SetConfiguration.HISTORICAL_BYTES"/>:
        ///                           the historical bytes should be exchanged.
        ///
        ///                         - If <see cref="SetConfiguration.READER_IC_CONFIG"/>:
        ///                           one of the below values,
        ///                             - <see cref="ReaderIC.RC512"/>
        ///                             - <see cref="ReaderIC.RC523"/>
        ///                             - <see cref="ReaderIC.RC663"/>
        ///
        ///                         - If <see cref="SetConfiguration.I2C_CLOCK_SPEED"/>:
        ///                           one of the below values,
        ///                             - <see cref="I2CClock.SPEED_0_5_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_1_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_2_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_3_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_4_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_6_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_8_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_12_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_16_MHZ"/>
        ///                             - <see cref="I2CClock.SPEED_FREE_RUN"/>
        ///
        ///                         - If <see cref="SetConfiguration.FULL_ATR"/>:
        ///                           Full ATR update
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_SetConfiguration ( byte bOption, byte[] aData )
        {
            return phhalHw_Sam_Cmd_SAM_SetConfiguration ( ref m_DataParamsInt[0], bOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion

        #region Key Management
        /// <summary>
        /// Change a symmetric key entry in the key table of the SAM.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be changed.
        ///                      One of the following,
        ///                         - NVM Keys: 0x00 - 0x7F
        ///                         - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///                       option can be combined by using bitwise OR operator. Also the byte can
        ///                       be framed by using a helper class <seealso cref="ChangeKeyEntry_ProMas"/>
        ///                         - <see cref="ProMas.UPDATE_KEY_VA"/>:
        ///                           Update KeyVa
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_VB"/>:
        ///                           Update KeyVb (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_VC"/>:
        ///                           Update KeyVc (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_DF_AID"/>:
        ///                           Update DF_AID, DF_KeyNo (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_CEK"/>:
        ///                           Update KeyNoCEK, KeyVCEK
        ///
        ///                         - <see cref="ProMas.UPDATE_REF_NO_KUC"/>:
        ///                           Update RefNoKUC
        ///
        ///                         - <see cref="ProMas.UPDATE_SET_EXTSET"/>:
        ///                           Update SET, ExtSET and, if present, KeyNoAEK, KeyVAEK"
        ///
        ///                         - <see cref="ProMas.INCLUDE_VERSION"/>:
        ///                           Key versions specified separately"
        /// </param>
        /// <param name="aKeyData">Buffer containing the key data information to be updated.
        ///                         - Key Entry buffer can be framed using a helper class
        ///                           <seealso cref="KeyEntry"/>. Here SET, ExtSET, Key Entry
        ///                           properties (like Versions, Keys, etc...) can be configured
        ///                           and complete Key Entry frame as bytes can be retrieved.
        ///                         - <seealso cref="KeyEntry"/> can also be used to
        ///                           extract SET, ExtSET, Key Entry properties (like Versions,
        ///                           DF_AID, etc...)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKeyEntry ( byte bKeyNo, byte bProMas, byte[] aKeyData )
        {
            return phhalHw_Sam_Cmd_SAM_ChangeKeyEntry ( ref m_DataParamsInt[0], bKeyNo, bProMas, aKeyData,
                ( byte ) ( ( aKeyData == null ) ? 0 : aKeyData.Length ) );
        }

        /// <summary>
        /// Change a Offline symmetric key entry in the key entry table of the SAM.
        ///     - Selected by a change key ( KeyNoCEK) of OfflineChange Key Class is restricting
        ///       the key update to an active Offline Key Activation.
        ///     - The command and its payload are protected by the Offline change protection.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be changed.
        ///                      One of the following,
        ///                         - NVM Keys: 0x00 - 0x7F
        ///                         - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///                       option can be combined by using bitwise OR operator. Also the byte can
        ///                       be framed by using a helper class <seealso cref="ChangeKeyEntry_ProMas"/>
        ///                         - <see cref="ProMas.UPDATE_KEY_VA"/>:
        ///                           Update KeyVa
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_VB"/>:
        ///                           Update KeyVb (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_VC"/>:
        ///                           Update KeyVc (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_DF_AID"/>:
        ///                           Update DF_AID, DF_KeyNo (shall be '0' if RAMKey, P1 >= 0xE0)
        ///
        ///                         - <see cref="ProMas.UPDATE_KEY_CEK"/>:
        ///                           Update KeyNoCEK, KeyVCEK
        ///
        ///                         - <see cref="ProMas.UPDATE_REF_NO_KUC"/>:
        ///                           Update RefNoKUC
        ///
        ///                         - <see cref="ProMas.UPDATE_SET_EXTSET"/>:
        ///                           Update SET, ExtSET and, if present, KeyNoAEK, KeyVAEK"
        ///
        ///                         - <see cref="ProMas.INCLUDE_VERSION"/>:
        ///                           Key versions specified separately"
        /// </param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCrypto">Offline Cryptogram exchanged to SAM (EncKeyEntry + OfflineMAC)
        ///                                 - Encrypted key entry as E ( Kce, KeyEntry[|| SAMUID])
        ///                                 - Integrity MAC protection as MACt ( Kcm, CLA || INS || P1 ||
        ///                                   P2 || LC || Change_Ctr || E(Kce, KeyEntry[|| SAMUID]))
        /// </param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///                                     - <see cref="Value.OFF"/>: Disable reception of Offline ACK
        ///                                       (LE will not be exchanged)
        ///                                     - <see cref="Value.ON"/>: Enable reception of Offline ACK
        ///                                       (LE will be exchanged)
        /// </param>
        /// <param name="aOfflineAck">Offline Acknowledge information received from SAM. Will be based on
        ///                             - If \b bEnableOfflineAck = <see cref="Value.OFF"/>, Disable reception of Offline ACK,
        ///                               buffer will contain NULL.
        ///                             - If \b bEnableOfflineAck = <see cref="Value.ON"/>, Enable reception of Offline ACK,
        ///                               buffer will contain Offline change acknowledge as MACt (Kcm, 0x90 || 0x00
        ///                               || INS || Change_Ctr || KeyNo || SAMUID).
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKeyEntryOffline ( byte bKeyNo, byte bProMas, ushort wChangeCtr, byte[] aOfflineCrypto,
            byte bEnableOfflineAck, out byte[] aOfflineAck )
        {
            IntPtr pOfflineAck = IntPtr.Zero;
            ushort wOfflineAckLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ChangeKeyEntryOffline ( ref m_DataParamsInt[0], bKeyNo, bProMas, wChangeCtr,
                aOfflineCrypto, (byte) ( ( aOfflineCrypto == null ) ? 0 : aOfflineCrypto.Length ), bEnableOfflineAck,
                ref pOfflineAck, ref wOfflineAckLen );
            aOfflineAck = MarshalCopy ( oStatus, pOfflineAck, wOfflineAckLen );

            if ( bEnableOfflineAck.Equals ( 0 ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Change the key usage counter (KUC). Selection is done by its reference number.
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be updated (00h to 0Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///                       option can be combined by using bitwise OR operator. Also the byte can
        ///                       be framed by using a helper class <seealso cref="ChangeKUCEntry_ProMas"/>
        ///                         - <see cref="KUC_ProMas.UPDATE_LIMIT"/>:
        ///                           Update limit
        ///
        ///                         - <see cref="KUC_ProMas.UPDATE_KEY_NO_CKUC"/>:
        ///                           Update KeyNoCKUC
        ///
        ///                         - <see cref="KUC_ProMas.UPDATE_VCKUC"/>:
        ///                           Update KeyVCKUC
        /// </param>
        /// <param name="aKucData">Buffer containing the Key Usage Counter data.
        ///                         - Key Entry buffer can be framed using a helper class
        ///                           <seealso cref="KucEntry"/>. Here properties (like Limit, KeyNoCKUC, etc...)
        ///                           can be configured and complete KUC Entry frame as bytes can be retrieved.
        ///                         - <seealso cref="KucEntry"/> can also be used to extract properties (like
        ///                           Limit, KeyNoCKUC, etc...)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKUCEntry ( byte bKucNo, byte bProMas, byte[] aKucData )
        {
            return phhalHw_Sam_Cmd_SAM_ChangeKUCEntry ( ref m_DataParamsInt[0], bKucNo, bProMas, aKucData,
                ( byte ) ( ( aKucData == null ) ? 0 : aKucData.Length ) );
        }

        /// <summary>
        /// Change a Offline KUC entry in the key entry table of the SAM.
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be updated (00h to 0Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///                       option can be combined by using bitwise OR operator. Also the byte can
        ///                       be framed by using a helper class <seealso cref="ChangeKUCEntry_ProMas"/>
        ///                         - <see cref="KUC_ProMas.UPDATE_LIMIT"/>:
        ///                           Update limit
        ///
        ///                         - <see cref="KUC_ProMas.UPDATE_KEY_NO_CKUC"/>:
        ///                           Update KeyNoCKUC
        ///
        ///                         - <see cref="KUC_ProMas.UPDATE_VCKUC"/>:
        ///                           Update KeyVCKUC
        /// </param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCrypto">Offline Cryptogram exchanged to SAM (EncKUCEntry + OfflineMAC)
        ///                                 - Encrypted key entry as E ( Kce, KUCEntry[|| SAMUID])
        ///                                 - Integrity MAC protection as MACt ( Kcm, CLA || INS || P1 ||
        ///                                   P2 || LC || Change_Ctr || E(Kce, KUCEntry[|| SAMUID]))
        /// </param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///                                     - <see cref="Value.OFF"/>, Disable reception of Offline ACK
        ///                                       (LE will not be exchanged)
        ///                                     - <see cref="Value.ON"/>, Enable reception of Offline ACK
        ///                                       (LE will be exchanged)
        /// </param>
        /// <param name="aOfflineAck">Offline Acknowledge information received from SAM. Will be based on
        ///                             - If \b bEnableOfflineAck = <see cref="Value.OFF"/>, Disable reception of Offline ACK,
        ///                               buffer will contain NULL.
        ///                             - If \b bEnableOfflineAck = <see cref="Value.ON"/>, Enable reception of Offline ACK,
        ///                               buffer will contain Offline change acknowledge as MACt (Kcm, 0x90 || 0x00
        ///                               || INS || Change_Ctr || RefNo KUC || SAMUID).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKUCEntryOffline ( byte bKucNo, byte bProMas, ushort wChangeCtr, byte[] aOfflineCrypto,
            byte bEnableOfflineAck, out byte[] aOfflineAck )
        {
            IntPtr pOfflineAck = IntPtr.Zero;
            ushort wOfflineAckLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ChangeKUCEntryOffline ( ref m_DataParamsInt[0], bKucNo, bProMas, wChangeCtr,
                aOfflineCrypto, ( byte ) ( ( aOfflineCrypto == null ) ? 0 : aOfflineCrypto.Length ), bEnableOfflineAck,
                ref pOfflineAck, ref wOfflineAckLen );
            aOfflineAck = MarshalCopy ( oStatus, pOfflineAck, wOfflineAckLen );

            if ( bEnableOfflineAck.Equals ( 0 ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Disable a key entry.
        ///     - The SAM_DisableKeyEntry is used to disable any key entry of the KST, hence enabling SETBit9. The plain data
        ///       field is not present.
        ///     - If there is an active authentication with a key of the targeted key entry, be it a Host Authentication (Logical
        ///       Channel Global State ), or any PICC Authentication or key activation ( XMode State ), this will be reset on any LC.
        ///     - Note that even if the Host Authentication on the current LC is reset, the current command response will still be
        ///       processed with secure messaging.
        ///     - After executing this command, the corresponding disable flag in the key entry is set and the key entry cannot be
        ///       used anymore for authentication and key change procedures.
        ///     - The key entry can still be read by a SAM_GetKeyEntry command.
        ///     - To reactivate the entry, a SAM_ChangeKeyEntry command has to be issued.
        ///     - All fields in the key entry can still be changed by this command even if the entry has been disabled.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key reference number of the key entry to disable.
        ///                      One of the following,
        ///                         - NVM Keys: 0x00 - 0x7F
        ///                         - RAM Keys: 0xE0 - 0xE3
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DisableKeyEntry ( byte bKeyNo )
        {
            return phhalHw_Sam_Cmd_SAM_DisableKeyEntry ( ref m_DataParamsInt[0], bKeyNo );
        }

        /// <summary>
        /// Disable a key entry using offline cryptogram.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key reference number of the key entry to disable.
        ///                      One of the following,
        ///                         - NVM Keys: 0x00 - 0x7F
        ///                         - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCrypto">Offline Cryptogram exchanged to SAM (EncKUCEntry + OfflineMAC)
        ///                                 - Encrypted GoldField as E ( Kce, SAMUID)
        ///                                 - Integrity MAC protection as MACt ( Kcm, CLA || INS || P1 ||
        ///                                   P2 || LC || Change_Ctr[|| E ( Kce, SAMUID )])
        /// </param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///                                     - <see cref="Value.OFF"/>: Disable reception of Offline ACK
        ///                                       (LE will not be exchanged)
        ///                                     - <see cref="Value.ON"/>: Enable reception of Offline ACK
        ///                                       (LE will be exchanged)
        /// </param>
        /// <param name="aOfflineAck">Offline Acknowledge information received from SAM. Will be based on
        ///                             - If \b bEnableOfflineAck = <see cref="Value.OFF"/>, Disable reception of Offline ACK,
        ///                               buffer will contain NULL.
        ///                             - If \b bEnableOfflineAck = <see cref="Value.ON"/>, Enable reception of Offline ACK,
        ///                               buffer will contain Offline disable acknowledge as MACt (Kcm, 0x90 || 0x00
        ///                               || INS || Change_Ctr || KeyNo || SAMUID).
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DisableKeyEntryOffline ( byte bKeyNo, ushort wChangeCtr, byte[] aOfflineCrypto,
            byte bEnableOfflineAck, out byte[] aOfflineAck )
        {
            IntPtr pOfflineAck = IntPtr.Zero;
            ushort wOfflineAckLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_DisableKeyEntryOffline ( ref m_DataParamsInt[0], bKeyNo, wChangeCtr,
                aOfflineCrypto, ( byte ) ( ( aOfflineCrypto == null ) ? 0 : aOfflineCrypto.Length ), bEnableOfflineAck,
                ref pOfflineAck, ref wOfflineAckLen );
            aOfflineAck = MarshalCopy ( oStatus, pOfflineAck, wOfflineAckLen );

            if ( bEnableOfflineAck.Equals ( 0 ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// The SAM_EncipherKeyEntry is a Personalization SAM command used to prepare a cryptogram (according
        /// to Offline change protection) for the OfflineChange key on a target SAM. The target Offline change counter
        /// (Perso_Ctr) are included in the generated cryptogram.It is only relevant for the target SAM and it shall
        /// not be compared to the ( local) Change_Ctr.
        /// </summary>
        ///
        /// <param name="bPersoKeyNo">Key reference number of the KST Key Entry to include
        ///                           in the cryptogram.One of the following,
        ///                             - NVM Keys: 0x00 - 0x7F
        ///                             - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyNo">Key reference number of the KSTKeyEntry in the target SAM (00h to 7Fh).</param>
        /// <param name="bOption">Option to include Key diversification and SAM UID in command.
        ///                         - <see cref="EncipherKeyEntry.DIV_ON"/>:
        ///                           Diversify the injected key with the given DivInput
        ///
        ///                         - <see cref="EncipherKeyEntry.SAM_UID_ON"/>:
        ///                           Include target GoldField (SAM UID)
        /// </param>
        /// <param name="wPersoCtr">Change Counter to avoid replay attacks</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aOfflineCrypto">Offline crypto information received from SAM.
        ///                                     - EncKeyEntry: Encrypted key entry as E ( K_{ pe} , NewEntry[|| SAMUID])
        ///                                     - OfflineMAC: Integrity MAC protection as
        ///                                       MACt ( Kpm , Channel || 0xC1 || KeyNo || ProMas || 2 + 80 + 8 || Perso_Ctr
        ///                                       || EncKeyEntry)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_EncipherKeyEntry ( byte bPersoKeyNo, byte bKeyNo, byte bOption, ushort wPersoCtr, byte[] aDivInput,
            out byte[] aOfflineCrypto )
        {
            IntPtr pOfflineCrypto = IntPtr.Zero;
            ushort wOfflineCryptoLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_EncipherKeyEntry ( ref m_DataParamsInt[0], bPersoKeyNo, bKeyNo, bOption, wPersoCtr,
                aDivInput, (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref pOfflineCrypto, ref wOfflineCryptoLen );
            aOfflineCrypto = MarshalCopy ( oStatus, pOfflineCrypto, wOfflineCryptoLen );

            return oStatus;
        }

        /// <summary>
        /// Get information about a key entry.
        ///     - SAM_GetKeyEntry allows reading the contents of the KST Key Entry specified in \b bKeyNo parameter.
        ///     - Instead of the full keys on positions a, b and c, only their key version will be returned,
        ///       each packed in one byte.
        ///     - This command can be issued without valid ( host) authentication.
        ///     - Supports retrieval of NVM and RAM Keys.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key reference number of the key entry to get.
        ///                      One of the following,
        ///                         - NVM Keys    : 0x00 - 0x7F
        ///                         - PLUnlock Key: 0xF0
        ///                         - RAM Key     : 0xE0 - 0xE3
        /// </param>
        /// <param name="bMode">Key entry format to be used. Not applicable for RAM Keys
        ///                         - <see cref="Value.OFF"/>:
        ///                           Old Key Entry Format (up to 1 - byte ExtSet)
        ///
        ///                         - <see cref="Value.ON"/>: New Key Entry Format
        /// <param name="aKeyEntry">Buffer containing the information about the key entry.
        ///                         <seealso cref="KeyEntry"/> can also be used to extract
        ///                         SET, ExtSET, Key Entry properties (like Versions, DF_AID,
        ///                         etc...)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GetKeyEntry ( byte bKeyNo, byte bMode, out byte[] aKeyEntry )
        {
            IntPtr pKeyEntry = IntPtr.Zero;
            ushort wKeyEntryLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_GetKeyEntry ( ref m_DataParamsInt[0], bKeyNo, bMode,
                ref pKeyEntry, ref wKeyEntryLen );
            aKeyEntry = MarshalCopy ( oStatus, pKeyEntry, wKeyEntryLen );

            return oStatus;
        }

        /// <summary>
        /// Get information about a key usage counter (KUC).
        ///     - SAM_GetKUCEntry allows reading the contents of the Key usage counter specified in \b bKucNo parameter.
        ///     - This command can be issued without valid ( host) authentication.
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be returned (00h to 0Fh).</param>
        /// <param name="aKucEntry">Buffer containing the KUC entry. <seealso cref="KucEntry"/> can also be
        ///                         used to extract properties (like Limit, KeyNoCKUC, etc...)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GetKUCEntry ( byte bKucNo, out byte[] aKucEntry )
        {
            IntPtr pKucEntry = IntPtr.Zero;
            ushort wKucEntryLen = 0;

            Status_t oStatus =  phhalHw_Sam_Cmd_SAM_GetKUCEntry ( ref m_DataParamsInt[0], bKucNo, ref pKucEntry,
                ref wKucEntryLen );
            aKucEntry = MarshalCopy ( oStatus, pKucEntry, wKucEntryLen );

            return oStatus;
        }

        /// <summary>
        /// Dump the current session key.
        ///     - Retrieve the session key of an established authentication with a DESFire PICC or a
        ///       MIFARE Plus PICC. In this case, an active PICC authentication (for these card types) is required.
        ///     - Retrieve the derived LRP key material. In this case, an KeyClass.OfflineCrypto activation of
        ///       KeyType.AES_128LRP is required, that is the SAM must be in PICCState.OfflineCrypto.
        /// </summary>
        ///
        /// <param name="bDumpMode">Dump Mode.
        ///                             - <see cref="DumpMode.PLAIN"/>: Plain Dump
        ///                             - <see cref="DumpMode.ENCIPHERED"/>: Enciphered dump
        /// </param>
        /// <param name="aSessionKey">Buffer containing Current session key information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DumpSessionKey ( byte bDumpMode, out byte[] aSessionKey )
        {
            IntPtr pSessionKey = IntPtr.Zero;
            ushort wSessionKeyLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_DumpSessionKey ( ref m_DataParamsInt[0], bDumpMode,
                ref pSessionKey, ref wSessionKeyLen );
            aSessionKey = MarshalCopy ( oStatus, pSessionKey, wSessionKeyLen );

            return oStatus;

        }

        /// <summary>
        /// Retrieve a PICC or OfflineCrypto key stored in the key table.
        /// </summary>
        ///
        /// <param name="bDumpMode">Dump Mode.
        ///                             - <see cref="DumpMode.PLAIN"/>: Plain Dump
        ///                             - <see cref="DumpMode.ENCIPHERED"/>: Enciphered dump
        ///                             - <see cref="DumpMode.DIVERSIFICATION_ON"/>: To be ored with above
        ///                               2 options
        /// </param>
        /// <param name="bKeyNo">Reference number of the key entry to be dumped. One of the following
        ///                         - NVM Keys: 0x00 - 0x7F
        ///                         - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Reference version of the key entry to be dumped.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aSecretKey">Buffer containing the plain secret key.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DumpSecretKey ( byte bDumpMode, byte bKeyNo, byte bKeyVer, byte[] aDivInput, out byte[] aSecretKey )
        {
            IntPtr pSecretKey = IntPtr.Zero;
            ushort wSecretKeyLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_DumpSecretKey ( ref m_DataParamsInt[0], bDumpMode, bKeyNo, bKeyVer, aDivInput,
                (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref pSecretKey, ref wSecretKeyLen );
            aSecretKey = MarshalCopy ( oStatus, pSecretKey, wSecretKeyLen );

            return oStatus;
        }

        /// <summary>
        /// Is used to derive a key from a source key (in other contexts often called master key) based
        /// on a CMAC operation.In a MIFARE context, this command can be used to support session key generations
        /// for the Transaction MAC and Secure Dynamic Messaging features, for back-end (and / or reader )
        /// interpretation and validation of the cryptograms created by the PICC.
        /// </summary>
        ///
        /// <param name="bSrcKeyNo">Key number of the source key. One of the following,
        ///                             - NVM Keys: 0x00 - 0x7F
        ///                             - RAM Keys: 0xE0 - 0xE3
        /// </param>
        /// <param name="bSrcKeyVer">Key version of the source key (00h to FFh).</param>
        /// <param name="bDstKeyNo">Key number of the destination key entry, only RAM Keys: 0xE0 - 0xE3</param>
        /// <param name="aDeriveIn">The derivation input for deriving the key.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DeriveKey ( byte bSrcKeyNo, byte bSrcKeyVer, byte bDstKeyNo, byte[] aDeriveIn )
        {
            return phhalHw_Sam_Cmd_SAM_DeriveKey ( ref m_DataParamsInt[0], bSrcKeyNo, bSrcKeyVer, bDstKeyNo, aDeriveIn,
                ( byte ) ( ( aDeriveIn == null ) ? 0 : aDeriveIn.Length ) );
        }
        #endregion

        #region File Management
        /// <summary>
        /// Performs creation of file. The FileType defines if the created file is a regular binary
        /// data file ( typically used to store certificates), or a CRLFile. A CRLFile is used for certificate
        /// revocation.If a CRLFile is targeted, the following additional parameters are present: CRLOptions,
        /// CSNSize and , depending on CRLOptions value CSNSigKey.The CRLVersion is initialized to zero.
        /// </summary>
        ///
        /// <param name="bFileNo">File number to be created. Supported numbers are 00 - 0F</param>
        /// <param name="bFileType">Type of file to create. One of the following,
        ///                             - <see cref="FileType.CERTIFICATE"/>
        ///                             - <see cref="FileType.CRL"/>
        /// </param>
        /// <param name="bKeyNoAEK_Read">Key reference number of KST access entry key for reading.
        ///                                 - 0xFE       : No access restriction
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bKeyVAEK_Read">Key version of KST access entry key for reading.</param>
        /// <param name="bKeyNoAEK_Write">Key reference number of KST access entry key for writing.
        ///                                 - 0xFE       : No access restriction
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bKeyVAEK_Write">Key version of KST access entry key for writing.</param>
        /// <param name="aFileSize">File size in bytes for the file to be created. Will be of 3 bytes
        ///                         with LSB first.
        ///                             - If size 0x10 need to be created, then the FileSize will be 10 00 00.
        ///                             - Supported size ranges from 0x000001 - 0x008000
        /// </param>
        /// <param name="bCRLOptions">CRLVersion required to be incremented by 1. Will be exchange if
        ///                           \b bFileType = <see cref="FileType.CRL"/>
        /// </param>
        /// <param name="bCSNSize">CSN size to use. Supported values are 0x04, 0x07, 0x0A. Will
        ///                        be exchange if \b bFileType = <see cref="FileType.CRL"/>
        /// </param>
        /// <param name="bCSNSigKey">Targeted ECC Key Entry for CRLSignature validation. Supported
        ///                          values are 0x00 - 0x07. Will be exchange if \b bFileType =
        ///                          <see cref="FileType.CRL"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CreateFile ( byte bFileNo, byte bFileType, byte bKeyNoAEK_Read, byte bKeyVAEK_Read,
            byte bKeyNoAEK_Write, byte bKeyVAEK_Write, byte[] aFileSize, byte bCRLOptions, byte bCSNSize, byte bCSNSigKey )
        {
            return phhalHw_Sam_Cmd_SAM_CreateFile ( ref m_DataParamsInt[0], bFileNo, bFileType, bKeyNoAEK_Read, bKeyVAEK_Read,
                bKeyNoAEK_Write, bKeyVAEK_Write, aFileSize, bCRLOptions, bCSNSize, bCSNSigKey );
        }

        /// <summary>
        /// Performs reading of data.
        ///     - The data to be read is defined by the file number of the targeted file, the offset
        ///       in the data file where to start the reading and its size in bytes. The file number
        ///       specifying the file where to read the data from is given by FileNo encoded in P1.
        ///     - The position byte-wise in the data file where to start to read data is given by Offset.
        ///       Its valid range is from 0x000000 to FileSize - 1. The data size to be read is given by
        ///       Length specifying the number of bytes.If Length is equal to 0x000000 then the entire
        ///       data file has to be read starting from the position specified by the Offset value.
        ///       Length valid range is 0x000000 to FileSize - Offset.
        ///     - If the number of bytes to read does not fit into one single APDU response,
        ///       <see cref="Error_Gen.SUCCESS_CHAINING"/> is provided to user and user should call
        ///       this interface again with \b bOption set to <see cref="ExchangeOptions.RXCHAINING"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option of reception of data from SAM. One of the following,
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>
        ///                           Exchange command and receive the response. This
        ///                           option means there is no further data to be read from SAM.
        ///
        ///                         - <see cref="ExchangeOptions.RXCHAINING"/> Receive Remaining Data
        ///                           from SAM. Should be used only when interface returns
        ///                           <see cref="Error_Gen.SUCCESS_CHAINING"/>. Here \b bFileNo,
        ///                           \b pOffset and \b pLength information will not be exchanged
        ///                           to SAM.
        /// </param>
        /// <param name="bFileNo">File number of the file to be read.
        ///                         - 00 - 0F: User File
        ///                         - F0     : Originality Certificate File
        /// </param>
        /// <param name="aOffset">The offset from where the data should be read. Will be of 3 bytes.
        ///                       with LSB first.If 0x10 need to be offset, then it will be 10 00 00.
        ///                         - 0 to ( FileSize - 1): Starting position of Read operation
        ///                         - 0xFFFFFFFF         : Return CRLFile Meta-Data
        /// </param>
        /// <param name="aLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///                         - If 0x10 bytes need to be read, then it will be 10 00 00.
        ///                         - If complete file need to be read, then it will be 00 00 00.
        /// </param>
        /// <param name="aResponse">The contents of the File returned by SAM. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> for successful operation with
        ///             more response data remaining
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ReadFile ( byte bOption, byte bFileNo, byte[] aOffset, byte[] aLength, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ReadFile ( ref m_DataParamsInt[0], bOption, bFileNo, aOffset, aLength,
                ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Performs writing of data.
        ///     - The location of data to be written is defined by the file number of the targeted file,
        ///       the offset in the data file where to start the writing and its size in bytes. The file
        ///       number specifying the file where to write to is given by FileNo encoded in P1.
        ///     - The position byte-wise in the data file where to start to write data is given by Offset.
        ///       Its valid range is from 0x000000 to FileSize - 1. The data size to be written is given
        ///       by Length specifying the number of bytes.
        ///     - If the number of bytes to send does not fit into one single APDU response, chaining is
        ///       applied internally.
        /// </summary>
        ///
        /// <param name="bOption">Option to exchange CRL Version, One of the following,
        ///                         - <see cref="Value.OFF"/> Do Not Exchange CRL Version
        ///                         - <see cref="Value.ON"/> Exchange CRL Version
        /// </param>
        /// <param name="bFileNo">File number of the file to be read.
        ///                         - Bit[7]  : Secondary Application Indicator
        ///                         - Bit[4 - 0]: File Number
        /// </param>
        /// <param name="wCRLVer">CRLVersion is a 16-bit value encoding the current version
        ///                       of the CRLFile.Valid if targeting CRL File.
        /// </param>
        /// <param name="aOffset">The offset from where the data should be written. Will be
        ///                       of 3 bytes. with LSB first.If 0x10 need to be offset, then
        ///                       it will be 10 00 00.
        ///                         - 0 to ( FileSize - 1): Starting position of Write operation
        /// </param>
        /// <param name="aData">The data to be written to the PICC.
        ///                         - Complete Data to be provided if not targeting CRL File
        ///                         - Complete Data including CRLSignature should be provided if
        ///                           targeting CRF File.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_WriteFile ( byte bOption, byte bFileNo, ushort wCRLVer, byte[] aOffset, byte[] aData )
        {
            byte[] aLength = BitConverter.GetBytes ( ( aData == null ) ? 0 : aData.Length );
            Array.Resize ( ref aLength, 3 );

            return phhalHw_Sam_Cmd_SAM_WriteFile ( ref m_DataParamsInt[0], bOption, bFileNo, wCRLVer, aOffset,
                aData, aLength );
        }

        /// <summary>
        /// Performs writing of data offline.
        ///     - User should complete the Offline cryptogram and provide the computed information to
        ///       this interface.
        ///     - If the number of bytes to send does not fit into one single APDU response, chaining is
        ///       applied internally.
        /// </summary>
        ///
        /// <param name="bFileNo">File number of the file to be read.
        ///                         - Bit[7]  : Secondary Application Indicator
        ///                         - Bit[4 - 0]: File Number
        /// </param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCrypto">Offline Cryptogram exchanged to SAM (EncKeyEntry + OfflineMAC)
        ///                                 - Encrypted Write File data ( Kce, WriteFileData[|| SAMUID])
        ///                                   where WriteFileData is the command data field parameters,
        ///                                   i.e. CRLVersion until CRLSignature (if applicable) in plain.
        ///                                 - Integrity MAC protection as MACt ( Kcm, CLA || INS || P1 ||
        ///                                   P2 || LC || Change_Ctr || E(Kce, WriteFileData[|| SAMUID]))
        /// </param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///                                     - <see cref="Value.OFF"/>: Disable reception of Offline ACK
        ///                                       (LE will not be exchanged)
        ///                                     - <see cref="Value.ON"/>: Enable reception of Offline ACK
        ///                                       (LE will be exchanged)
        /// </param>
        /// <param name="aOfflineAck">Offline Acknowledge information received from SAM. Will be based on
        ///                             - If \b bEnableOfflineAck = <see cref="Value.OFF"/>, Disable reception of Offline ACK,
        ///                               buffer will contain NULL.
        ///                             - If \b bEnableOfflineAck = <see cref="Value.ON"/>, Enable reception of Offline ACK,
        ///                               buffer will contain Offline disable acknowledge as MACt (Kcm, 0x90 || 0x00
        ///                               || INS || Change_Ctr || FileNo || SAMUID).
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_WriteFileOffline ( byte bFileNo, ushort wChangeCtr, byte[] aOfflineCrypto,
            byte bEnableOfflineAck, out byte[] aOfflineAck )
        {
            IntPtr pOfflineAck = IntPtr.Zero;
            ushort wOfflineAckLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_WriteFileOffline ( ref m_DataParamsInt[0], bFileNo, wChangeCtr,
                aOfflineCrypto, ( byte ) ( ( aOfflineCrypto == null ) ? 0 : aOfflineCrypto.Length ),
                bEnableOfflineAck, ref pOfflineAck, ref wOfflineAckLen );
            aOfflineAck = MarshalCopy ( oStatus, pOfflineAck, wOfflineAckLen );

            if ( bEnableOfflineAck.Equals ( 0 ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Performs reading the meta-data of a file.
        ///     - FileType, Access Rights and File Size of the targeted file encoded in P1 will be returned.
        ///     - Additionally, if a CRLFile is targeted the CRLOptions, CSNSize and, if applicable,
        ///       CSNSigKey will be returned.
        ///     - If P1 is set to 0xFE, will return the remaining fee user memory that is available for file
        ///       creation.
        ///     - If P1 is set to 0xFF, a list of the already created file identifiers will be returned.
        /// </summary>
        ///
        /// <param name="bFileNo">File number of the file to be read.
        ///                         - 00 - 0F: User File
        ///                         - F0     : Originality Certificate File
        ///                         - FE     : Free Memory
        ///                         - FF     : List of file IDs
        /// </param>
        /// <param name="apResponse">The settings of the File returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GetFileSettings ( byte bFileNo, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_GetFileSettings ( ref m_DataParamsInt[0], bFileNo,
                ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Performs deletion of all files. After successful execution the full user memory
        /// for files is again available for file creation.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_Format ()
        {
            return phhalHw_Sam_Cmd_SAM_Format ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Data Processing
        /// <summary>
        /// Apply the DESFire EVx (EV2 or higher) Secure Messaging in S-mode on the provided DESFire EVx (EV2 or higher)
        /// command according to the required mode and the currently activated session keys.The required protection mode is
        /// selected via the command parameter \b bCommMode.
        /// </summary>
        ///
        /// <param name="wOption">Refer below for Option to be used.
        ///                         - <see cref="Offset.INCLUDE"/>:
        ///                           Offset (\b bOffset is exchanged to SAM)
        ///
        ///                         -Buffering options. Ored with above options.
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                               Buffer first set of information.
        ///                               No exchange is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                               Buffer intermediate set of information.
        ///                               No exchange is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                               Buffer last" set of information.
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                              Should be used to exchange chunks of data and receive
        ///                              response
        /// <param name="bCommMode">Communication mode to be used.
        ///                             - <see cref="CommMode.PLAIN"/>:
        ///                               Protection Mode as PLAIN
        ///
        ///                             - <see cref="CommMode.MAC"/>:
        ///                               Protection Mode as MAC
        ///
        ///                             - <see cref="CommMode.FULL"/>:
        ///                               Protection Mode as FULL
        /// </param>
        /// <param name="bOffset">Command offset. Index of the first byte in data field of the PICC command data.</param>
        /// <param name="bCmdCtrIncr">Command counter increment value. Value by which to increase the CmdCtr.</param>
        /// <param name="aData">Plain data to be protected according to the communication mode specified.</param>
        /// <param name="aResponse">The protected data returned by SAM according to communication mode specified.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ApplySM ( ushort wOption, byte bCommMode, byte bOffset, byte bCmdCtrIncr, byte[] aData,
            out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ApplySM ( ref m_DataParamsInt[0], wOption, bCommMode, bOffset, bCmdCtrIncr, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Removes the DESFire EVx (EV2 or higher) Secure Messaging in S-mode from the provided PICC response
        /// payload according to the required mode and the currently activated session keys.The required protection
        /// mode is selected via the command parameter \b bCommMode.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                          Should be used to exchange chunks of data and receive
        ///                          response
        /// </param>
        /// <param name="bCommMode">Communication mode to be used.
        ///                             - <see cref="CommMode.MAC"/>:
        ///                               Protection Mode as MAC
        ///
        ///                             - <see cref="CommMode.FULL"/>:
        ///                               Protection Mode as FULL
        /// </param>
        /// <param name="aData">The complete data received form the PICC including the status code.</param>
        /// <param name="aResponse">The plain data returned by SAM according to communication mode specified.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_RemoveSM ( ushort wOption, byte bCommMode, byte[] aData, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_RemoveSM ( ref m_DataParamsInt[0], wOption, bCommMode, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Verifies the MAC which was sent by the PICC or any other system based on the given MACed plain text data
        /// and the currently valid cryptographic key.
        ///     - The valid key has been activated using a valid PICC authentication ( SAM_AuthenticatePICC ,
        ///       SAM_IsoAuthenticatePICC) in case of a PICC key or using a valid key activation ( SAM_ActivateOfflineKey)
        ///       in case of an OfflineCrypto Key
        ///     - The applied MAC algorithm depends on the key type. The command can be used for verifying only a part, up
        ///       to the full MAC. The number of MAC bytes to be verified is defined by parameter \b bNum. By default the
        ///       standard truncation is expected. For AES key types (including KeyType.AES_128LRP), also the truncation
        ///       as used by a.o. the MIFARE Plus Secure Messaging can be applied.This option for MFP Truncation is not
        ///       allowed with a key that has the option 'Keep IV' (SET bit 2) set.
        ///     - In OfflineCrypto PICC State, if the targeted key is of KeyType.AES_128LRP, SAM_Verify_MAC shall apply
        ///       KeyID.LRPUpdate or KeyID.LRPMACUpdate, if respectively one or two updated keys were generated with
        ///       SAM_ActivateOfflineKey.
        ///     - In OfflineCrypto PICC State, if activated key from KST is referencing a KUC, SAM_Verify_MAC increments
        ///       the KUC associated to the targeted key before any crypto processing.
        ///     - Note that SAM_Verify_MAC supports the verification of a MAC over zero-byte input length messages. In that
        ///       case the Data field of the APDU will just contain the MAC.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options to exchange data to SAM and receive data from SAM.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                          Should be used to exchange chunks of data and receive
        ///                          response
        /// </param>
        /// <param name="bNum">Type of truncation mode to be applied if AES key type is used. Also the byte can
        ///                    be framed by using a helper class <seealso cref="VerifyGenerateMAC_Num"/>
        ///                         - <see cref="TruncationMode.STANDARD"/>: Standard truncation
        ///                         - <see cref="TruncationMode.MFP"/>: MFP truncation
        ///
        ///                         - Number of MAC bytes (\b bNum) to be verified.
        ///                           Should be combined with one of the above option.
        ///                             - 0x01 - 0x08: (3)DES key types
        ///                             - 0x01 - 0x10: AES key types
        ///                             - 0x00: Default number of bytes ( depending on key type )
        /// </param>
        /// <param name="aData">Plain data including the MAC to be checked.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_VerifyMAC ( ushort wOption, byte bNum, byte[] aData )
        {
            return phhalHw_Sam_Cmd_SAM_VerifyMAC ( ref m_DataParamsInt[0], wOption, bNum, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Generates a MAC which is meant to be sent to the PICC or any other system based on the given plain text data
        /// and the currently valid cryptographic key.
        ///     - The valid key has been activated using a valid PICC authentication ( SAM_AuthenticatePICC ,
        ///       SAM_IsoAuthenticatePICC) in case of a PICC key or using a valid key activation ( SAM_ActivateOfflineKey)
        ///       in case of an OfflineCrypto Key.
        ///     - The applied MAC algorithm depends on the key type.The command can be used for creating only a part, up to
        ///       the full MAC. The number of MAC bytes to be generated is defined by parameter \b bNum. By default the standard
        ///       truncation is applied. For AES key types (including KeyType.AES_128LRP), also the truncation as used by a.o.
        ///       the MIFARE Plus Secure Messaging can be applied. This option for MFP Truncation is not allowed with a key that
        ///       has the option 'Keep IV' (SET bit 2) set.
        ///     - In OfflineCrypto PICC State, if the targeted key is of KeyType.AES_128LRP, SAM_Generate_MAC shall apply
        ///       KeyID.LRPUpdate or KeyID.LRPMACUpdate, if respectively one or two updated keys were generated with
        ///       SAM_ActivateOfflineKey.
        ///     - In OfflineCrypto PICC State, if activated key from KST is referencing a KUC, SAM_Generate_MAC increments the
        ///       KUC associated to the targeted key before any crypto processing.
        ///     - Note that SAM_Generate_MAC supports the generation of a MAC over zero-byte input length messages. In that case
        ///       the LC and Data fields of the command APDU will be absent.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                           Should be used to exchange chunks of data and receive
        ///                           response
        ///
        ///                          - <see cref="IncludeLC.INCLUDE"/> Include LC" should be used
        ///                            for inclusion of LC when <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///                            First Data Chunk is used and \b bDataLen is not available but \b bDataLen
        ///                            is part of <see cref="ExchangeOptions.BUFFER_CONT"/> Intermediate Data Chunk
        ///                            or <see cref="ExchangeOptions.BUFFER_LAST"/> Last Data Chunk.
        /// </param>
        /// <param name="bNum">Type of truncation mode to be applied if AES key type is used. Also the byte can
        ///                    be framed by using a helper class <seealso cref="VerifyGenerateMAC_Num"/>
        ///                         - <see cref="TruncationMode.STANDARD"/>: Standard truncation
        ///                         - <see cref="TruncationMode.MFP"/>: MFP truncation
        ///
        ///                         - Number of MAC bytes (\b bNum) to be processed.
        ///                           Should be combined with one of the above option.
        ///                             - 0x01 - 0x08: (3)DES key types
        ///                             - 0x01 - 0x10: AES key types
        ///                             - 0x00: Default number of bytes ( depending on key type )
        /// </param>
        /// <param name="aData">Plain data to be maced. Can be null if there is no data. </param>
        /// <param name="aResponse">The generated MAC returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_GenerateMAC ( ushort wOption, byte bNum, byte[] aData, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_GenerateMAC ( ref m_DataParamsInt[0], wOption, bNum, aData,
                (byte) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Deciphers data packages sent by a PICC, any other system or a MIFARE card based on the currently valid
        /// cryptographic key and returns plain data to the host.
        /// </summary>
        ///
        /// <param name="wOption">Option for including the length information in command frame.
        ///                             - <see cref="Length.INCLUDE"/>:
        ///                               Length information (\b pLength) will not be exchanged
        ///
        ///                             - <see cref="Length.INCLUDE"/>:
        ///                               Length information (\b pLength) will be exchanged
        ///
        ///                         - Buffering options. Ored with above options.
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                               Buffer first set of information.
        ///                               No exchange is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                               Buffer intermediate set of information.
        ///                               No exchange is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                               Buffer last" set of information.
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                              Should be used to exchange chunks of data and receive
        ///                              response
        /// </param>
        /// <param name="aEncData">Encrypted data to be deciphered.</param>
        /// <param name="dwLength">Overall length of encrypted input data. This 3 byte value is only used if
        ///                        \b wOption has <see cref="Length.INCLUDE"/> Length information
        ///                        (\b pLength) will be exchanged.
        /// </param>
        /// <param name="aPlainData">Deciphered data returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DecipherData ( ushort wOption, byte[] aEncData, int dwLength, out byte[] aPlainData )
        {
            IntPtr pPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            byte[] aLength = new byte[3];
            aLength[0] = ( byte ) dwLength;
            aLength[1] = ( byte ) ( dwLength >> 8 );
            aLength[2] = ( byte ) ( dwLength >> 16 );

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_DecipherData ( ref m_DataParamsInt[0], wOption, aEncData,
                ( byte ) ( ( aEncData == null ) ? 0 : aEncData.Length ), aLength, ref pPlainData,
                ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, pPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Enciphers data packages which are meant to be sent to a PICC or any other system based on the given
        /// plain text data and the currently valid cryptographic key.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                           Should be used to exchange chunks of data and receive
        ///                           response
        /// </param>
        /// <param name="aPlainData">Plain data to be enciphered.</param>
        /// <param name="bOffset">Offset into the input data indicating the first data byte
        ///                       to be enciphered.
        /// </param>
        /// <param name="aEncData">Enciphered data returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_EncipherData ( ushort wOption, byte[] aPlainData, byte bOffset, out byte[] aEncData )
        {
            IntPtr pPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_EncipherData ( ref m_DataParamsInt[0], wOption, aPlainData,
                (byte) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), bOffset, ref pPlainData, ref wPlainDataLen );
            aEncData = MarshalCopy ( oStatus, pPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Decrypts data received from any other system based on the given cipher text data and the currently valid cryptographic
        /// OfflineCrypto Key. The valid key has been activated using a valid key activation (SAM_ActivateOfflineKey).
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                           Should be used to exchange chunks of data and receive
        ///                           response
        /// </param>
        /// <param name="aEncData">Encrypted data to be deciphered.</param>
        /// <param name="aPlainData">Deciphered data returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_DecipherOfflineData ( ushort wOption, byte[] aEncData, out byte[] aPlainData )
        {
            IntPtr pPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_DecipherOfflineData ( ref m_DataParamsInt[0], wOption, aEncData,
                (byte) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref pPlainData, ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, pPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Encrypts data received from any other system based on the given cipher text data and the currently valid cryptographic
        /// OfflineCrypto Key. The valid key has been activated using a valid key activation (SAM_ActivateOfflineKey).
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.
        ///                           No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information.
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>:
        ///                           Should be used to exchange chunks of data and receive
        ///                           response
        /// </param>
        /// <param name="aPlainData">Plain data to be enciphered.</param>
        /// <param name="aEncData">Enciphered data returned by SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_EncipherOfflineData ( ushort wOption, byte[] aPlainData, out byte[] aEncData )
        {
            IntPtr pEncData = IntPtr.Zero;
            ushort wEncDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_EncipherOfflineData ( ref m_DataParamsInt[0], wOption, aPlainData,
                (byte) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), ref pEncData, ref wEncDataLen );
            aEncData = MarshalCopy ( oStatus, pEncData, wEncDataLen );

            return oStatus;
        }
        #endregion

        #region Public Key Infrastructure
        #region RSA
        /// <summary>
        /// Create an RSA key pair. This command is available for AV2 and AV3 version(s).
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte. Also the byte can be framed
        ///                       by using a helper class <seealso cref="PKI_RSA_GenerateKeyPair_Option"/>
        ///                         - <see cref="PKI_GenerationMode.RANDOM_E"/>:
        ///                           PKI_e is randomly generated
        ///
        ///                         - <see cref="PKI_GenerationMode.HOST_E"/>:
        ///                           PKI_e is provided by the host
        ///
        ///                         - <see cref="PKI_AEK.INCLUDE_AEK"/>: Access Entry Key
        ///                           (PKI_KeyNoAEK, PKI_KeyVAEK) exchanged to SAM
        /// </param>
        /// <param name="bPKI_KeyNo">Key reference number of the PKI Key Entry (00h to 01h).</param>
        /// <param name="wPKI_Set">Configuration settings of the created key entry</param>
        /// <param name="bPKI_KeyNoCEK">Key reference number of KST change entry key.
        ///                                 - 0xFE       : No change restriction
        ///                                 - 0xFF       : Entry locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bPKI_KeyVCEK">Version of the change key of the created key entry.</param>
        /// <param name="bPKI_RefNoKUC">Reference number of key usage counter (00h - 0Fh, FFh)</param>
        /// <param name="bPKI_KeyNoAEK">Key reference number of KST access entry key.
        ///                                 - 0xFE       : No Access Restrictions
        ///                                 - 0xFF       : Entry Disabled
        ///                                 - 0x00 - 0x7F: Access key entry number
        /// </param>
        /// <param name="bPKI_KeyVAEK">Key version of KST access entry key.</param>
        /// <param name="wPKI_NLen">RSA key length size in bytes: between 64 and 256 bytes, a multiple
        ///                         of 8 byte.
        /// </param>
        /// <param name="wPKI_eLen">Public exponent length size in bytes between 4 and 256 bytes, a
        ///                         multiple of 4 byte, and shall not be greater than PKI_NLen:
        ///                         between 4 and 256 bytes, a multiple of 4 byte
        /// </param>
        /// <param name="aPKI_e">Public exponent e where eLen is PKI_eLen. It must be an odd integer.
        ///                         - Exchanged if \b bOption = <see cref="PKI_GenerationMode.HOST_E"/>:
        ///                           PKI_e is provided by the host".
        ///
        ///                         - NULL if \b bOption = <see cref="PKI_GenerationMode.HOST_E"/>:
        ///                           PKI_e is randomly generated".
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aPKI_e is NULL.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_GenerateKeyPair ( byte bOption, byte bPKI_KeyNo, ushort wPKI_Set, byte bPKI_KeyNoCEK,
            byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen,
            ushort wPKI_eLen, byte[] aPKI_e )
        {
            return phhalHw_Sam_Cmd_PKI_GenerateKeyPair ( ref m_DataParamsInt[0], bOption, bPKI_KeyNo, wPKI_Set,
                bPKI_KeyNoCEK, bPKI_KeyVCEK, bPKI_RefNoKUC, bPKI_KeyNoAEK, bPKI_KeyVAEK, wPKI_NLen, wPKI_eLen,
                aPKI_e );
        }

        /// <summary>
        /// Import a public or private RSA key.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.  Also the byte can be framed
        ///                       by using a helper class <seealso cref="PKI_RSA_ImportKey_Option"/>
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_VALUE"/>:
        ///                           Update key settings and key value
        ///
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_ONLY"/>:
        ///                           Update only key settings" (PKI_KeyNo, PKI_SET,
        ///                           PKI_KeyNoCEK, PKI_KeyVCEK, PKI_RefNoKUC, and, if
        ///                           present, PKI_KeyNoAEK and PKI_KeyVAEK)
        ///
        ///                         - <see cref="PKI_AEK.INCLUDE_AEK"/>Access Entry Key
        ///                           (PKI_KeyNoAEK, PKI_KeyVAEK) exchanged to SAM
        /// </param>
        /// <param name="bPKI_KeyNo">Key reference number of the PKI Key Entry to update.
        ///                             - 0x00 - 0x02: Without private key
        ///                             - 0x00 - 0x01: With private key included
        /// </param>
        /// <param name="wPKI_Set">Configuration settings of the imported key entry. It indicates
        ///                        whether a private of public key shall be imported.
        /// </param>
        /// <param name="bPKI_KeyNoCEK">Key reference number of KST change entry key.
        ///                                 - 0xFE       : No change restriction
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bPKI_KeyVCEK">Key version of KST change entry key.</param>
        /// <param name="bPKI_RefNoKUC">Reference number of key usage counter (00h - 0Fh, FFh).</param>
        /// <param name="bPKI_KeyNoAEK">Key reference number of KST access entry key
        ///                                 - 0xFE         : No Access Restrictions
        ///                                 - 0xFF         : Entry Disabled
        ///                                 - 0x00 - 0x7F  : Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bPKI_KeyVAEK">Key version of KST access entry key.</param>
        /// <param name="wPKI_NLen">RSA key length size in bytes: between 64 and 256 bytes, a multiple
        ///                         of 8 byte.
        /// </param>
        /// <param name="wPKI_eLen">Public exponent length size in bytes between 4 and 256 bytes, a
        ///                         multiple of 4 byte, and shall not be greater than PKI_NLen:
        ///                         between 4 and 256 bytes, a multiple of 4 byte.
        /// </param>
        /// <param name="wPKI_PLen">If private key included, the prime p length size in bytes
        ///                         Ceil ( PKI_pLen / 4) + 2 <= ceil ( PKI_NLen / 4).
        /// </param>
        /// <param name="wPKI_QLen">If private key included, the prime q length size in bytes
        ///                         Ceil ( PKI_qLen/4) + 2 <= ceil ( PKI_NLen/4).
        /// </param>
        /// <param name="aPKI_N">If private key included, Modulus N where NLen is PKI_NLen. The most
        ///                      significant 32-bit word of N shall not be equal to zero.
        /// </param>
        /// <param name="aPKI_e">If private key included, Public exponent e where eLen is PKI_eLen. It
        ///                      must be an odd integer.
        /// </param>
        /// <param name="aPKI_p">If private key included, Prime p where pLen is PKI_pLen. The MSB of p
        ///                      shall not be equal to zero.
        /// </param>
        /// <param name="aPKI_q">If private key included, Prime q where pLen is PKI_qLen. The MSB of q
        ///                      shall not be equal to zero.
        /// </param>
        /// <param name="aPKI_dP">If private key included, Private exponent d_p where qLen is PKI_pLen.</param>
        /// <param name="aPKI_dQ">If private key included, Private exponent d_q where qLen is PKI_qLen.</param>
        /// <param name="aPKI_ipq">If private key included, Inverse p-1mod(q) where qLen is PKI_qLen.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - \b aPKI_N and \b aPKI_e are NULL if \b bOption = <see cref="PKI_UpdateMode.SETTINGS_ONLY">
        ///           Update only key settings</see>
        ///         - \b aPKI_p, \b aPKI_q, \b aPKI_dP, \b aPKI_dQ and \b aPKI_ipq are NULL if
        ///           \b bOption = <see cref="PKI_UpdateMode.SETTINGS_ONLY">Update only key settings and
        ///           Private Key is included.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ImportKey ( byte bOption, byte bPKI_KeyNo, ushort wPKI_Set, byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK,
            byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen, ushort wPKI_eLen, ushort wPKI_PLen,
            ushort wPKI_QLen, byte[] aPKI_N, byte[] aPKI_e, byte[] aPKI_p, byte[] aPKI_q, byte[] aPKI_dP, byte[] aPKI_dQ,
            byte[] aPKI_ipq )
        {
            return phhalHw_Sam_Cmd_PKI_ImportKey ( ref m_DataParamsInt[0], bOption, bPKI_KeyNo, wPKI_Set, bPKI_KeyNoCEK,
                bPKI_KeyVCEK, bPKI_RefNoKUC, bPKI_KeyNoAEK, bPKI_KeyVAEK, wPKI_NLen, wPKI_eLen, wPKI_PLen, wPKI_QLen,
                aPKI_N, aPKI_e, aPKI_p, aPKI_q, aPKI_dP, aPKI_dQ, aPKI_ipq );
        }

        /// <summary>
        /// Export a full RSA key entry (i.e including the private key if present).
        /// A successful host authentication in the LC using SAM_AuthenticateHost with a Host Key is required to
        /// execute the PKI_ExportPrivateKey command. This needs to be done with the Host Key referenced by the
        /// PKI_KeyNoCEK and PKI_KeyVCEK of the targeted key entry.
        /// </summary>
        ///
        /// <param name="wOption">Option for AEK selection and differentiating between first part
        ///                       and last part of data.
        ///                         - <see cref="PKI_AEK.RECEIVE_AEK"/>: Access Entry Key
        ///                           (PKI_KeyNoAEK, PKI_KeyVAEK) received from SAM
        ///
        ///                         - Should be combined with the above option
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>: Receiving the first part of data
        ///                             - <see cref="ExchangeOptions.RXCHAINING"/>: Receiving the intermediate or
        ///                               final part of data"
        /// </param>
        /// <param name="bPKI_KeyNo">Key reference number of the PKI Key Entry (00h to 01h).</param>
        /// <param name="aKeyEntry">Received Private Key information from SAM.
        ///                         - Key Entry buffer can be framed using a helper class
        ///                           <seealso cref="PKI_RSA_KeyEntry"/>. Here SET, Private Key
        ///                           Information, Public Key Information, Key properties can be
        ///                           configured and complete Key Entry frame as bytes can be
        ///                           retrieved.
        ///                         - <seealso cref="PKI_RSA_KeyEntry"/> can also be used to
        ///                           extract SET, Public Key Information and Key properties
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful.
        ///                                                         More data available to receive from SAM.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ExportPrivateKey ( ushort wOption, byte bPKI_KeyNo, out byte[] aKeyEntry )
        {
            IntPtr pKeyData = IntPtr.Zero;
            ushort wKeyDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_ExportPrivateKey ( ref m_DataParamsInt[0], wOption, bPKI_KeyNo,
                ref pKeyData, ref wKeyDataLen );
            aKeyEntry = MarshalCopy ( oStatus, pKeyData, wKeyDataLen );

            return oStatus;
        }

        /// <summary>
        /// Export the public part of an RSA key pair.
        /// </summary>
        ///
        /// <param name="wOption">Option for AEK selection and differentiating between first part
        ///                       and last part of data.
        ///                         - <see cref="PKI_AEK.RECEIVE_AEK"/>: Access Entry Key
        ///                           (PKI_KeyNoAEK, PKI_KeyVAEK) received from SAM
        ///
        ///                         - Should be combined with the above option
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>: Receiving the first part of data
        ///                             - <see cref="ExchangeOptions.RXCHAINING"/>: Receiving the intermediate or
        ///                               final part of data"
        /// </param>
        /// <param name="bPKI_KeyNo">Key reference number of the PKI Key Entry (00h to 02h).</param>
        /// <param name="aKeyEntry">Received Public Key information from SAM.
        ///                         - Key Entry buffer can be framed using a helper class
        ///                           <seealso cref="PKI_RSA_KeyEntry"/>. Here SET, Private Key
        ///                           Information, Public Key Information, Key properties can be
        ///                           configured and complete Key Entry frame as bytes can be
        ///                           retrieved.
        ///                         - <seealso cref="PKI_RSA_KeyEntry"/> can also be used to
        ///                           extract SET, Public Key Information and Key properties
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful.
        ///                                                         More data available to receive from SAM.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ExportPublicKey ( ushort wOption, byte bPKI_KeyNo, out byte[] aKeyEntry )
        {
            IntPtr pKeyData = IntPtr.Zero;
            ushort wKeyDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_ExportPublicKey ( ref m_DataParamsInt[0], wOption, bPKI_KeyNo,
                ref pKeyData, ref wKeyDataLen );
            aKeyEntry = MarshalCopy ( oStatus, pKeyData, wKeyDataLen );

            return oStatus;
        }

        /// <summary>
        /// The PKI_UpdateKeyEntries can be used to change key entries of the symmetric key storage (KST).
        ///     - Executing this command does not use the protection based on the Key Access Control, hence
        ///       the addressed KST Key Entrys change entry key (KeyNoCEK) is ignored. Instead the commands
        ///       execution is protected by asymmetric techniques using the PKI support of the SAM.
        ///     - The SAM supports the update of up to three key entries with PKI_UpdateKeyEntries. The SAM
        ///       will limit the number of key entries according to the key size and the hashing function
        ///       used for OAEP padding.
        /// </summary>
        ///
        /// <param name="bOption">Option for AEK selection and differentiating between
        ///                       first part and last part of data.
        ///                         - <see cref="PKI_ACK.RECEIVE_ACK"/>: Acknowledge
        ///                           key number. Option to include Le byte and
        ///                           Acknowledge key number to command frame for
        ///                           receiving the UpdateAck data from SAM hardware.
        ///
        ///                         - Should be combined with the above option
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Exchange and receive information from SAM
        ///
        ///                             - <see cref="ExchangeOptions.RXCHAINING"/>:
        ///                               To be used if <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///                               Chaining status is provided by Library
        /// </param>
        /// <param name="bNoOfKeyEntries">Number of key entries to include in the cryptogram
        ///                                 - 0x00: RFU
        ///                                 - 0x01: 1 Key Entry
        ///                                 - 0x02: 2 Key Entry
        ///                                 - 0x03: 3 Key Entry
        /// </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature ).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="bPKI_KeyNo_Enc">Key reference number of the PKI Key Entry to be used
        ///                              for decryption (00h to 01h).
        /// </param>
        /// <param name="bPKI_KeyNo_Sign">Key reference number of the PKI Key Entry to be used
        ///                               for signature verification (00h to 02h).
        /// </param>
        /// <param name="bPKI_KeyNo_Ack">Key reference number of the PKI Key Entry to be used
        ///                              for acknowledge signature generation (00h to 01h).
        /// </param>
        /// <param name="aKeyFrame">Buffer containing the RSA encrypted key entries and the
        ///                         signature ( EncKeyFrame || Signature).
        ///                         - EncKeyFrame: RSA encrypted key frame as
        ///                           RSA_E ( KPKI_KeyNo_Enc , Change_Ctr || KeyNo1 || ProMas1
        ///                           || NewEntry1[|| KeyNo2 || ProMas2 || NewEntry2[|| KeyNo3
        ///                           || ProMas3 || New Entry3]]) where LenE is the PKI_NLen of
        ///                           the PKI_KeyNo_Enc key entry.
        ///                         - Signature: RSA digital signature as
        ///                           RSA_S ( KPKI_KeyNo_Sign, PKI_KeyNo_Enc || PKI_KeyNo_Sign ||
        ///                           EncKeyFrame) where LenS is the PKI_NLen of the PKI_KeyNo_Sign
        ///                           key entry
        /// </param>
        /// <param name="aUpdateACK">Buffer containing the RSA encrypted Acknowledge signature.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_UpdateKeyEntries ( byte bOption, byte bNoOfKeyEntries, byte bHashingAlg,
            byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Ack, byte[] aKeyFrame,
            out byte[] aUpdateACK )
        {
            IntPtr pUpdateACK = IntPtr.Zero;
            ushort wUpdateACKLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_UpdateKeyEntries ( ref m_DataParamsInt[0], bOption, bNoOfKeyEntries, bHashingAlg,
                bPKI_KeyNo_Enc, bPKI_KeyNo_Sign, bPKI_KeyNo_Ack, aKeyFrame, ( ushort ) ( ( aKeyFrame == null ) ? 0 : aKeyFrame.Length ),
                ref pUpdateACK, ref wUpdateACKLen );
            aUpdateACK = MarshalCopy ( oStatus, pUpdateACK, wUpdateACKLen );

            return oStatus;
        }

        /// <summary>
        /// Prepare a cryptogram (according to Asymmetric Offline Change Cryptogram) for the PKI offline update of
        /// KST key entries on a target SAM.
        /// </summary>
        ///
        /// <param name="wOption">Option to differentiate between the command frame to be exchanged.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Receive" first part of
        ///                           information from SAM
        ///                         - <see cref="ExchangeOptions.RXCHAINING"/>: Receive intermediate
        ///                           or final part of information from SAM
        ///
        ///                         - For enabling or disabling of key diversification. Should be
        ///                           combined with the above options
        ///                             - <see cref="PKI_Div.ON"/>: Diversification enabled
        /// </param>
        /// <param name="bNoOfKeyEntries">Number of key entries to include in the cryptogram
        ///                                 - 0x00: RFU
        ///                                 - 0x01: 1 Key Entry
        ///                                 - 0x02: 2 Key Entry
        ///                                 - 0x03: 3 Key Entry
        /// </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="bPKI_KeyNo_Enc">Key reference number of the PKI Key Entry to be used
        ///                              for the cryptogram encryption (00h to 02h).
        /// </param>
        /// <param name="bPKI_KeyNo_Sign">Key reference number of the PKI Key Entry to be used
        ///                               for the cryptogram signature generation (00h to 01h).
        /// </param>
        /// <param name="bPKI_KeyNo_Dec">Key reference number of the PKI Key Entry that will be
        ///                              used for decryption in the target SAM (00h to 01h).
        /// </param>
        /// <param name="bPKI_KeyNo_Verif">Key reference number of the PKI Key Entry that will be
        ///                                used for signature verification in the target SAM (00h to 02h).
        /// </param>
        /// <param name="wPerso_Ctr">Targeted offline change counter data.</param>
        /// <param name="aKeyEntries">Key entry descriptions
        ///                             - PersoKeyNo: Key reference number of the KST Key Entry
        ///                               to include in the cryptogram
        ///                                 - NVM Key: 0x00 - 0x7F
        ///                                 - RAM Key: 0x#0 - 0xE3
        ///                                 - KeyNo: Key reference number of the KST Key Entry in the
        ///                                   target SAM (0x00 - 0x7F)
        /// </param>
        /// <param name="aDivInput">Diversification input for key diversification.
        ///                         (1 to 31 byte(s) input).
        /// </param>
        /// <param name="aEncKeyFrame_Sign">The Encrypted Key frame and Signature as returned by
        ///                                 SAM hardware.
        ///                                     - EncKeyFrame: RSA encrypted key frame as RSA_E ( KPKI_KeyNo_Enc,
        ///                                       Perso_Ctr || KeyNo1 || ProMas1 || NewEntry1[|| KeyNo2 || ProMas2
        ///                                       || NewEntry2[|| KeyNo3 || ProMas3 || New Entry3]]) where LenE is
        ///                                       the PKI_NLen of the PKI_KeyNo_Enc key entry
        ///                                     - Signature: RSA digital signature as RSA_S ( KPKI_KeyNo_Sign,
        ///                                       PKI_KeyNo_Dec || PKI_KeyNo_Verif || EncKeyFrame) where LenS is
        ///                                       the PKI_NLen of the PKI_KeyNo_Sign key entry
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_EncipherKeyEntries ( ushort wOption, byte bNoOfKeyEntries, byte bHashingAlg, byte bPKI_KeyNo_Enc,
            byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Dec, byte bPKI_KeyNo_Verif, ushort wPerso_Ctr, byte[] aKeyEntries,
            byte[] aDivInput, out byte[] aEncKeyFrame_Sign )
        {
            IntPtr pEncKeyFrame_Sign = IntPtr.Zero;
            ushort wEncKeyFrame_Sign_Len = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_EncipherKeyEntries ( ref m_DataParamsInt[0], wOption, bNoOfKeyEntries,
                bHashingAlg, bPKI_KeyNo_Enc, bPKI_KeyNo_Sign, bPKI_KeyNo_Dec, bPKI_KeyNo_Verif, wPerso_Ctr, aKeyEntries,
                ( byte ) ( ( aKeyEntries != null ) ? aKeyEntries.Length : 0 ), aDivInput,
                ( byte ) ( ( aDivInput != null ) ? aDivInput.Length : 0 ),
                ref pEncKeyFrame_Sign, ref wEncKeyFrame_Sign_Len );
            aEncKeyFrame_Sign = MarshalCopy ( oStatus, pEncKeyFrame_Sign, wEncKeyFrame_Sign_Len );

            return oStatus;
        }

        /// <summary>
        /// Generate Hash DataFrame from Data.
        /// </summary>
        ///
        /// <param name="wOption">Buffering Options:
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Exchange Message and Receive
        ///                           Hash from SAM
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Buffer Fist part of message.
        ///                           Information is not exchanged to SAM
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Buffer intermediate message.
        ///                           Information is not exchanged to SAM.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Buffer last message".
        ///                           Information is exchanged to SAM and Hash is received.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: The LFI is set to AFh if the
        ///                           flag
        ///
        ///                         - On the first frame of the command chain the flag
        ///                           <see cref="PKI_LFI.FIRST_FRAME"/>: Exchange overall length
        ///                           has to be set to exchange HashAlgo and length of the overall
        ///                           message.
        /// </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="dwMLen">Overall message length (4 byte).</param>
        /// <param name="aMessage">Message chunk to be hashed.</param>
        /// <param name="aHash">Buffer containing the hash after sending the last message chunk.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>    Operation successful, chaining ongoing.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_GenerateHash ( ushort wOption, byte bHashingAlg, uint dwMLen, byte[] aMessage, out byte[] aHash )
        {
            IntPtr pHash = IntPtr.Zero;
            ushort wHashLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_GenerateHash ( ref m_DataParamsInt[0], wOption, bHashingAlg, dwMLen, aMessage,
                ( ushort ) ( ( aMessage == null ) ? 0 : aMessage.Length ), ref pHash, ref wHashLen );
            aHash = MarshalCopy ( oStatus, pHash, wHashLen );

            return oStatus;
        }

        /// <summary>
        /// Generate a signature with a given RSA key entry.
        /// </summary>
        ///
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="bPKI_KeyNo_Sign">Key reference number of the PKI Key Entry to be used
        ///                               for the cryptogram signature generation (00h to 01h).
        /// </param>
        /// <param name="aHash">Hash message to be signed.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>             Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_GenerateSignature ( byte bHashingAlg, byte bPKI_KeyNo_Sign, byte[] aHash )
        {
            return phhalHw_Sam_Cmd_PKI_GenerateSignature ( ref m_DataParamsInt[0], bHashingAlg, bPKI_KeyNo_Sign,
                aHash, ( byte ) ( ( aHash != null ) ? aHash.Length : 0 ) );
        }

        /// <summary>
        /// Get a previously generated signature.
        /// </summary>
        ///
        /// <param name="aSignature">The signature received from SAM hardware.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_SendSignature ( out byte[] aSignature )
        {
            IntPtr pSignature = IntPtr.Zero;
            ushort wSignatureLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_SendSignature ( ref m_DataParamsInt[0], ref pSignature,
                ref wSignatureLen );
            aSignature = MarshalCopy ( oStatus, pSignature, wSignatureLen );

            return oStatus;
        }

        /// <summary>
        /// Verify a hash / signature pair with a given RSA key.
        /// </summary>
        ///
        /// <param name="bPKI_KeyNo_Verif">Key reference number of the PKI Key Entry to be used
        ///                                for the cryptogram signature verification (00h to 02h)
        /// </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="aHashData">Hash message to be signed.</param>
        /// <param name="aSignature">RSA digital signature where NLen is the PKI_NLen of
        ///                          the PKI_KeyNo key entry
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_VerifySignature ( byte bPKI_KeyNo_Verif, byte bHashingAlg, byte[] aHashData,
            byte[] aSignature )
        {
            return phhalHw_Sam_Cmd_PKI_VerifySignature ( ref m_DataParamsInt[0], bPKI_KeyNo_Verif, bHashingAlg,
                aHashData, ( byte ) ( ( aHashData == null ) ? 0 : aHashData.Length ), aSignature,
                ( ushort ) ( ( aSignature == null ) ? 0 : aSignature.Length ) );
        }

        /// <summary>
        /// Performs the offline encryption of plain RSA data.
        /// </summary>
        ///
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="bPKI_KeyNo_Enc">Key reference number of the PKI Key Entry to be used
        ///                              for encryption (00h to 02h).
        /// </param>
        /// <param name="aPlainData">RSA Plain Data to be encrypted where mLen (\b wPlainDataLen)
        ///                          is restricted by the PKI_NLen of the PKI_KeyNo_Enc key entry
        ///                          and the output length of the hash function
        /// </param>
        /// <param name="aEncData">RSA encrypted data as
        ///                        EncData = RSA_E ( KPKI_KeyNo_Enc , PlainData)
        ///                        where LenE is the PKI_NLen of the PKI_KeyNo_Enc
        ///                        key entry.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_EncipherData ( byte bHashingAlg, byte bPKI_KeyNo_Enc, byte[] aPlainData,
            out byte[] aEncData )
        {
            IntPtr pEncData = IntPtr.Zero;
            ushort wEncDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_EncipherData ( ref m_DataParamsInt[0], bHashingAlg, bPKI_KeyNo_Enc,
                aPlainData, ( ushort ) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), ref pEncData,
                ref wEncDataLen );
            aEncData = MarshalCopy ( oStatus, pEncData, wEncDataLen );

            return oStatus;
        }

        /// <summary>
        /// Performs the offline decryption of encrypted RSA data.
        /// </summary>
        ///
        /// <param name="wOption">Option parameter:
        ///                         - Local buffering is implemented by the flags
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>: Exchange information to SAM
        ///                             - <see cref="ExchangeOptions.TXCHAINING"/>: The LFI is set to AFh if the flag
        ///                             - Usage:
        ///                                 - For chained data
        ///                                     - For First Frame: Use flags <see cref="PKI_LFI.FIRST_FRAME"/>:
        ///                                       First Frame | <see cref="ExchangeOptions.TXCHAINING"/>Chained
        ///                                       Frame
        ///                                     - For Next N Frame ( s): Use flag <see cref="ExchangeOptions.TXCHAINING"/>
        ///                                       Chained Frame
        ///                                     - For Last Frame: Use flags <see cref="ExchangeOptions.DEFAULT"/>:
        ///                                       Last Frame
        ///
        ///                                 - For non chained data i.e. only single frame use
        ///                                   <see cref="PKI_LFI.FIRST_FRAME"/>: First Frame |
        ///                                   <see cref="ExchangeOptions.DEFAULT"/>: Exchange
        ///                                   information to SAM
        /// </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        /// </param>
        /// <param name="bPKI_KeyNo_Dec">Key reference number of the PKI Key Entry to be used
        ///                              for encryption (00h to 02h).
        /// </param>
        /// <param name="aEncData">The RSA encrypted data to be decrypted.
        /// </param>
        /// <param name="aPlainData">RSA Plain Data returned by SAM hardware.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_DecipherData ( ushort wOption, byte bHashingAlg, byte bPKI_KeyNo_Dec, byte[] aEncData,
            out byte[] aPlainData )
        {
            IntPtr pPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_DecipherData ( ref m_DataParamsInt[0], wOption, bHashingAlg, bPKI_KeyNo_Dec,
                aEncData, ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref pPlainData, ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, pPlainData, wPlainDataLen );

            return oStatus;
        }
        #endregion

        #region ECC
        /// <summary>
        /// PKI_GenerateEccKey creates a pair of a public and a private ECC key. To make the key usable,
        /// Bit 0 of ECC_SET must be unset ( that is disable key entry is false). ECC_SET Bit 1 must be set,
        /// indicating a private key entry.
        /// </summary>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be update.
        ///                             - NVM key: 0x00 - 0x0F
        ///                             - RAM key: 0xE0 - 0xE3
        /// </param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="bECC_RefNoCurve">Reference of ECC Curve</param>
        /// <param name="aECC_xy">Public key point coordinate</param>
        ///
        /// <returns></returns>
        public Status_t Cmd_PKI_GenerateEccKey ( byte bECC_KeyNo, ushort wECC_Set, byte bECC_KeyNoCEK, byte bECC_KeyVCEK,
            byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK, byte bECC_RefNoCurve, out byte[] aECC_xy )
        {
            IntPtr pECC_xy = IntPtr.Zero;
            ushort wECC_xyLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_PKI_GenerateEccKey ( ref m_DataParamsInt[0], bECC_KeyNo, wECC_Set,
                bECC_KeyNoCEK, bECC_KeyVCEK, bECC_RefNoKUC, bECC_KeyNoAEK, bECC_KeyVAEK, bECC_RefNoCurve,
                ref pECC_xy, ref wECC_xyLen );
            aECC_xy = MarshalCopy ( oStatus, pECC_xy, wECC_xyLen );

            return oStatus;
        }

        /// <summary>
        /// The PKI_ImportEccKey imports an ECC private or public key in the Key Storage. The command can also be
        /// used to only update the key settings ( that is ECC_SET, ECC_KeyNoCEK, ECC_KeyVCEK, ECC_RefNoKUC,
        /// ECC_KeyNoAEK and ECC_KeyVAEK. This can be done for example to temporarily disable a key entry.
        /// This is indicated by P1.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_VALUE"/>:
        ///                           Update key settings and key value
        ///
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_ONLY"/>:
        ///                           Update only key settings" (PKI_KeyNo, PKI_SET,
        ///                           PKI_KeyNoCEK, PKI_KeyVCEK, PKI_RefNoKUC, and, if
        ///                           present, PKI_KeyNoAEK and PKI_KeyVAEK)
        /// </param>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be update.
        ///                             - NVM key: 0x00 - 0x0F
        ///                             - RAM key: 0xE0 - 0xE3
        /// </param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="aECC_KeyValue">Public key point coordinate. Ranges from 33 - 65 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ImportEccKey ( byte bOption, byte bECC_KeyNo, ushort wECC_Set, byte bECC_KeyNoCEK, byte bECC_KeyVCEK,
            byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK, ushort wECC_Len, byte[] aECC_KeyValue )
        {
            return phhalHw_Sam_Cmd_PKI_ImportEccKey ( ref m_DataParamsInt[0], bOption, bECC_KeyNo, wECC_Set, bECC_KeyNoCEK,
                bECC_KeyVCEK, bECC_RefNoKUC, bECC_KeyNoAEK, bECC_KeyVAEK, wECC_Len, aECC_KeyValue,
                ( byte ) ( aECC_KeyValue != null ? aECC_KeyValue.Length : 0 ) );
        }

        /// <summary>
        /// The PKI_ImportEccCurve imports a full ECC Curve description in the ECC Curve Storage Table.
        /// The command can also be used to only update the curve settings ( that is ECC_KeyNoCCK and
        /// ECC_KeyVCCK). This is indicated by P1.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_VALUE"/>:
        ///                           Update key settings and key value
        ///
        ///                         - <see cref="PKI_UpdateMode.SETTINGS_ONLY"/>:
        ///                           Update only key settings" (PKI_KeyNo, PKI_SET,
        ///                           PKI_KeyNoCEK, PKI_KeyVCEK, PKI_RefNoKUC, and, if
        ///                           present, PKI_KeyNoAEK and PKI_KeyVAEK)
        /// </param>
        /// <param name="bECC_CurveNo">Curve reference number of the ECC curve entry to
        ///                            update (00h to 03h).
        /// </param>
        /// <param name="bECC_KeyNoCCK">Key reference number to change the curve entry.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCCK">Key version to change curve entry.</param>
        /// <param name="bECC_N">Size of the field in bytes. Ranges from 16 to 32 bytes.</param>
        /// <param name="bECC_M">Size of the order in bytes. Ranges from 16 to 32 bytes.</param>
        /// <param name="aECC_Prime">Prime, field definition: ECC_N bytes.</param>
        /// <param name="aECC_ParamA">Curve parameter (a): ECC_N bytes.</param>
        /// <param name="aECC_ParamB">Curve parameter (b): ECC_N bytes.</param>
        /// <param name="aECC_Px">x-coordinate of base-point: ECC_N bytes.</param>
        /// <param name="aECC_Py">y-coordinate of base-point: ECC_N bytes.</param>
        /// <param name="aECC_Order">Order of base-point: ECC_M bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///     \b aECC_Prime, \b aECC_ParamA, \b aECC_ParamB,
        ///     \b aECC_Px, \b aECC_Py and \b aECC_Order are NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ImportEccCurve ( byte bOption, byte bECC_CurveNo, byte bECC_KeyNoCCK, byte bECC_KeyVCCK, byte bECC_N,
            byte bECC_M, byte[] aECC_Prime, byte[] aECC_ParamA, byte[] aECC_ParamB, byte[] aECC_Px, byte[] aECC_Py, byte[] aECC_Order )
        {
            return phhalHw_Sam_Cmd_PKI_ImportEccCurve ( ref m_DataParamsInt[0], bOption, bECC_CurveNo, bECC_KeyNoCCK, bECC_KeyVCCK,
                bECC_N, bECC_M, aECC_Prime, aECC_ParamA, aECC_ParamB, aECC_Px, aECC_Py, aECC_Order );
        }

        /// <summary>
        /// The PKI_ExportEccPrivateKey exports a private ECC key entry, including the private key.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be
        ///                          update (00h to 07h).
        /// </param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0x00 - 0x7F: Access key entry number
        /// </param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="bECC_RefNoCurve">Reference of ECC Curve.</param>
        /// <param name="aECC_Priv">Private key scalar. Ranges from 16 - 80 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ExportEccPrivateKey ( byte bECC_KeyNo, out ushort wECC_Set, out byte bECC_KeyNoCEK, out byte bECC_KeyVCEK,
            out byte bECC_RefNoKUC, out byte bECC_KeyNoAEK, out byte bECC_KeyVAEK, out ushort wECC_Len, byte bECC_RefNoCurve,
            out byte[] aECC_Priv )
        {
            IntPtr pECC_xy = IntPtr.Zero;
            byte bECC_PrivLen = 0;

            wECC_Set = 0;
            bECC_KeyNoCEK = 0;
            bECC_KeyVCEK = 0;
            bECC_RefNoKUC = 0;
            bECC_KeyNoAEK = 0;
            bECC_KeyVAEK = 0;
            wECC_Len = 0;
            bECC_RefNoCurve = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_PKI_ExportEccPrivateKey ( ref m_DataParamsInt[0], bECC_KeyNo, ref wECC_Set, ref bECC_KeyNoCEK,
                ref bECC_KeyVCEK, ref bECC_RefNoKUC, ref bECC_KeyNoAEK, ref bECC_KeyVAEK, ref wECC_Len, ref bECC_RefNoCurve, ref pECC_xy,
                ref bECC_PrivLen );
            aECC_Priv = MarshalCopy ( oStatus, pECC_xy, bECC_PrivLen );

            return oStatus;
        }

        /// <summary>
        /// Exports the ECC public key from Key Storage. This is only supported by Sam version.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be
        ///                          update (00h to 07h).
        /// </param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0x00 - 0x7F: Access key entry number
        /// </param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="aECC_xy">Public key point coordinate. Ranges from 33 - 65 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ExportEccPublicKey ( byte bECC_KeyNo, out ushort wECC_Set, out byte bECC_KeyNoCEK, out byte bECC_KeyVCEK,
            out byte bECC_RefNoKUC, out byte bECC_KeyNoAEK, out byte bECC_KeyVAEK, out ushort wECC_Len, out byte[] aECC_xy )
        {
            IntPtr pECC_xy = IntPtr.Zero;
            byte bECC_xyLen = 0;
            byte bCRLFIle = 0;

            wECC_Set = 0;
            bECC_KeyNoCEK = 0;
            bECC_KeyVCEK = 0;
            bECC_RefNoKUC = 0;
            bECC_KeyNoAEK = 0;
            bECC_KeyVAEK = 0;
            wECC_Len = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_PKI_ExportEccPublicKey ( ref m_DataParamsInt[0], bECC_KeyNo, ref wECC_Set, ref bECC_KeyNoCEK,
                ref bECC_KeyVCEK, ref bECC_RefNoKUC, ref bECC_KeyNoAEK, ref bECC_KeyVAEK, ref wECC_Len, ref pECC_xy, ref bECC_xyLen,
                ref bCRLFIle);
            aECC_xy = MarshalCopy ( oStatus, pECC_xy, bECC_xyLen );

            return oStatus;
        }

        /// <summary>
        /// Exports the ECC public key from Key Storage. This is only supported by Sam version.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be
        ///                          update (00h to 07h).
        /// </param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0xFF       : Entry Locked
        ///                                 - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
        /// </param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///                                 - 0xFE       : No Restrictions
        ///                                 - 0x00 - 0x7F: Access key entry number
        /// </param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="bCRLFIle">CRL File.
        ///                         - Exchanged if ECC_SET Bit 12 is set and value will be from 00 - 0F
        ///                         - Will be 0xFF otherwise
        /// </param>
        /// <param name="aECC_xy">Public key point coordinate. Ranges from 33 - 65 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ExportEccPublicKey ( byte bECC_KeyNo, out ushort wECC_Set, out byte bECC_KeyNoCEK, out byte bECC_KeyVCEK,
            out byte bECC_RefNoKUC, out byte bECC_KeyNoAEK, out byte bECC_KeyVAEK, out ushort wECC_Len, out byte[] aECC_xy,
            out byte bCRLFIle )
        {
            IntPtr pECC_xy = IntPtr.Zero;
            byte bECC_xyLen = 0;

            wECC_Set = 0;
            bECC_KeyNoCEK = 0;
            bECC_KeyVCEK = 0;
            bECC_RefNoKUC = 0;
            bECC_KeyNoAEK = 0;
            bECC_KeyVAEK = 0;
            wECC_Len = 0;
            bCRLFIle = 0xFF;
            Status_t oStatus = phhalHw_Sam_Cmd_PKI_ExportEccPublicKey ( ref m_DataParamsInt[0], bECC_KeyNo, ref wECC_Set, ref bECC_KeyNoCEK,
                ref bECC_KeyVCEK, ref bECC_RefNoKUC, ref bECC_KeyNoAEK, ref bECC_KeyVAEK, ref wECC_Len, ref pECC_xy, ref bECC_xyLen,
                ref bCRLFIle );
            aECC_xy = MarshalCopy ( oStatus, pECC_xy, bECC_xyLen );

            return oStatus;
        }

        /// <summary>
        /// Generate ECC Signature generates a signature on a hash given as input using a
        /// private key stored in the ECC Key Entry.The signature is immediately returned in the
        /// response as the concatenation of r and s. This response has a length of 2 * ECC_Len of
        /// the targeted ECC Key Entry, i.e.twice the curve size.
        /// </summary>
        ///
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and
        ///                           digital signature ).
        ///                           - <see cref="PKI_HashAlgo.SHA_1"/>: SHA-1
        ///                           - <see cref="PKI_HashAlgo.SHA_224"/>: SHA-224
        ///                           - <see cref="PKI_HashAlgo.SHA_256"/>: SHA-256
        ///                           - <see cref="PKI_HashAlgo.SHA_384"/>: SHA-384
        ///                           - <see cref="PKI_HashAlgo.SHA_512"/>: SHA-512
        /// </param>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be used for
        ///                          signature generation (00h to 07h).
        /// </param>
        /// <param name="aHash">Hash message to be signed.</param>
        /// <param name="aSignature">Generated signature (r, s) with an actual length of
        ///                          2 * ECC_Len of the targeted ECC key entry.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>  \b aMessage is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_GenerateEccSignature ( byte bHashingAlg, byte bECC_KeyNo, byte[] aHash,
            out byte[] aSignature )
        {
            IntPtr pSignature = IntPtr.Zero;
            ushort wSigLen = 0;

             Status_t oStatus = phhalHw_Sam_Cmd_PKI_GenerateEccSignature ( ref m_DataParamsInt[0], bHashingAlg, bECC_KeyNo,
                aHash, ( byte ) ( ( aHash != null ) ? aHash.Length : 0 ), ref pSignature, ref wSigLen );
            aSignature = MarshalCopy ( oStatus, pSignature, wSigLen );

            return oStatus;
        }

        /// <summary>
        /// The PKI_VerifyEccSignature is used to support the originality check architecture. The originality
        /// check allows verification of the genuineness of NXP chips after manufacturing. The command verifies
        /// the correctness of an ECC signature (for example: NXPOriginalitySignature) obtained from the product
        /// to verify. The signature is computed according to Elliptic Curve DSA (ECDSA). In case of originality
        /// checking, the PKI_VerifyEccSignature cryptographic parameters, such as the input message ( M), the
        /// selected ECC curve and the required public key depend on the type of product for which the verification
        /// is performed.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be used for
        ///                          signature verification (00h to 07h).
        /// </param>
        /// <param name="bECC_CurveNo">The curve reference number of the ECC curve entry to be used
        ///                            for signature verification (00h to 03h).
        /// </param>
        /// <param name="bLen">Length in bytes of the message to verify</param>
        /// <param name="aMessage">Signed input data.</param>
        /// <param name="aSignature">The ECC digital signature where N is 2 * ECC_Len of ECC_KeyNo
        ///                          key entry.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>  \b aMessage is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_VerifyEccSignature ( byte bECC_KeyNo, byte bECC_CurveNo, byte bLen, byte[] aMessage,
            byte[] aSignature )
        {
            return phhalHw_Sam_Cmd_PKI_VerifyEccSignature ( ref m_DataParamsInt[0], bECC_KeyNo, bECC_CurveNo, bLen,
                aMessage, aSignature, ( ushort ) ( ( aSignature != null ) ? aSignature.Length : 0 ) );
        }

        /// <summary>
        /// The PKI_ValidateEccCert command supports certificate validation in the context of the following
        /// non-exhaustive list of use cases
        ///     - ECC-based Card-Unilateral Authentication supporting the MIFARE DUOX ISOInternalAuthenticate
        ///     - EV charging protocol supporting the MIFARE DUOX VDE_ECDSA_Sign
        ///     - Secure Dynamic Messaging, i.e. the SUN-feature, with ECDSA signature as supported by MIFARE DUOX
        ///     - Transaction Signature with ECDSA signature as supported by MIFARE DUOX
        /// </summary>
        ///
        /// <param name="bCertFormat">Format of the Certificate. This will be used for P1 parameter information
        ///                             - <see cref="CertificateFormat.X509"/>
        ///                             - <see cref="CertificateFormat.EV_CHARGING"/>
        /// </param>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be used for
        ///                          signature verification.
        ///                             - NVM key: 0x00 - 0x0F
        ///                             - RAM key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bECC_CurveNo">Curve reference number of the ECC curve entry to
        ///                            update (00h to 03h).
        /// </param>
        /// <param name="bECC_Target">ECC key entry targeted to store the extracted public key.
        ///                             - NVM key: 0x00 - 0x0F
        ///                             - RAM key: 0xE0 - 0xE3
        /// </param>
        /// <param name="aCertificate">Certificate to be validated</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>  Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PKI_ValidateEccCert ( byte bCertFormat, byte bECC_KeyNo, byte bECC_CurveNo, byte bECC_Target,
            byte[] aCertificate )
        {
            return phhalHw_Sam_Cmd_PKI_ValidateEccCert ( ref m_DataParamsInt[0], bCertFormat, bECC_KeyNo, bECC_CurveNo,
                bECC_Target, aCertificate, ( ushort ) ( ( aCertificate != null ) ? aCertificate.Length : 0 ) );
        }
        #endregion
        #endregion

        #region Virtual Card
        #region S_Mode
        /// <summary>
        /// Performs Virtual card selection based on the response of ISOSelect command.
        ///     - The SAM_SelectVC command is designed to support in S-mode the VC Selection with
        ///       PCD authentication, that is if the VC is configured for AuthVCMandatory.
        ///     - The AuthVCMandatory configuration is deducted from the presence of a valid FCI
        ///       ( that is: expected 4-bytes TLV headers and payload size ) in the response of the
        ///       PICC ISOSelect command.
        ///     - If the PICC ISOSelect command does not return a FCI or returns an invalid FCI,
        ///       there is no need to execute SAM_SelectVC
        ///     - The command supports key diversification.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined
        ///                       by bitwise OR. Also the byte can be framed by using a
        ///                       helper class <seealso cref="VCA_Select_Option"/>
        ///                         - <see cref="VCA_Select.ENC_KEY_DIV_INPUT"/>:
        ///                           Diversification of VCSelectENCKey" using
        ///                           Diversification Input
        ///
        ///                         - Option for diversification of VCSelectMACKey.
        ///                             - <see cref="VCA_Select.MAC_KEY_DIV_INPUT"/>:
        ///                               Diversify the used key with the given DivInput
        ///                             - <see cref="VCA_Select.MAC_KEY_DIV_VCUID"/>:
        ///                               Diversify with the Virtual Card Unique Identifier
        ///                               (VCUID)
        ///                             -<see cref="VCA_Select.MAC_KEY_DIV_INPUT_VCUID"/>:
        ///                              Diversify with the VCUID and the given DivInput
        ///
        ///                         - <see cref="VCA_Select.DEFAULT"/>: Option to disable
        ///                           the diversification for VcSelect MAC and ENC keys.
        /// </param>
        /// <param name="bEncKeyNo">KeyID.VCSelectENCKey SAM Key Number.
        ///                             - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                             - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bEncKeyVer">KeyID.VCSelectENCKey SAM Key Version.</param>
        /// <param name="bMacKeyNo">KeyID.VCSelectMACKey SAM Key Number.
        ///                             - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                             - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bMacKeyVer">KeyID.VCSelectMACKey SAM Key Version.</param>
        /// <param name="aData">ISOSelect response payload (32 Bytes) without TLV headers,
        ///                     including the VC related data.
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response received from SAM (MAC || VCData)
        ///                             - MAC: Response to Challenge as MACt ( DivKey(KVCSelMAC)),
        ///                               RndChal || VCData)
        ///                             - VCData: Decrypted Virtual Card ( VC) Data
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_SelectVC ( byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer,
            byte[] aData, byte[] aDivInput, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_SelectVC ( ref m_DataParamsInt[0], bOption, bEncKeyNo,
                bEncKeyVer, bMacKeyNo, bMacKeyVer, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ),
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref pResponse,
                ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>Performs Proximity Check Part 1 command execution.</para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_ProximityCheck_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///       <see cref="Cmd_SAM_ProximityCheck_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption"><see cref="VCA_ProximityCheck.DIVERSIFICATION_ON"/>:
        ///                       Use Key Diversification. Instructs SAM to
        ///                       Diversify the Key.
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM.
        ///                         - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                         - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="aPPCData">PICC_PreparePC response data: Option || Pub
        ///                        RespTime[|| PPS1][|| ActBitRate]
        /// </param>
        /// <param name="aPCData">Response and challenge bytes exchanged during the
        ///                       proximity check protocol as (pRndR1 || pRndC1) ||
        ///                       ... || (pRndR8 || pRndC8)
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aMac">The MAC to be exchanged to the PICC.
        ///                    MACt ( KPC, code || Option || PubRespTime[|| PPS1] || (pRndR1
        ///                    || pRndC1) ||...|| (pRndR8 || pRndC8))
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ProximityCheck_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPPCData, byte[] aPCData,
            byte[] aDivInput, out byte[] aMac )
        {
            IntPtr pMac = IntPtr.Zero;
            ushort wMacLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                aPPCData, ( byte ) ( ( aPPCData == null ) ? 0 : aPPCData.Length ), aPCData,
                ( byte ) ( ( aPCData == null ) ? 0 : aPCData.Length ), aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                ref pMac, ref wMacLen );
            aMac = MarshalCopy ( oStatus, pMac, wMacLen );

            return oStatus;
        }

        /// <summary>
        /// <para>Performs Proximity Check Part 2 command execution.</para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_ProximityCheck_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="aData">The data to be exchanged to SAM hardware.
        ///                         - For PICC Error  : The PICC error code should be
        ///                           passed and \b bDataLen should be 1.
        ///                         - For PICC Success: The MAC received from PICC
        ///                           should be passed and \b bDataLen should be 8.
        /// </param>
        /// <param name="bPiccRetCode">The PICC response code echoed back by the SAM.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ProximityCheck_Part2 ( byte[] aData, out byte bPiccRetCode )
        {
            bPiccRetCode = 0;
            return phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2 ( ref m_DataParamsInt[0], aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref bPiccRetCode );
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// The VCA_Select command handles in X-mode a complete VC Selection with or without PCD
        /// authentication depending on the AuthVCMandatory configuration of the VC. The SAM shall deduct
        /// the AuthVCMandatory configuration from the presence of a valid FCI ( that is: expected 4-bytes
        /// TLV headers and payload size ) in the response of the PICC ISOSelect.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined
        ///                       by bitwise OR. Also the byte can be framed by using a
        ///                       helper class <seealso cref="VCA_Select_Option"/>
        ///                         - <see cref="VCA_Select.ENC_KEY_DIV_INPUT"/>:
        ///                           Diversification of VCSelectENCKey using Diversification Input
        ///
        ///                         - Option for diversification of VCSelectMACKey.
        ///                             - <see cref="VCA_Select.MAC_KEY_DIV_INPUT"/>:
        ///                               Diversify the used key with the given DivInput
        ///                             - <see cref="VCA_Select.MAC_KEY_DIV_VCUID"/>:
        ///                               Diversify with the Virtual Card Unique Identifier
        ///                               (VCUID)
        ///                             - <see cref="VCA_Select.MAC_KEY_DIV_INPUT_VCUID"/>:
        ///                               Diversify with the VCUID and the given DivInput
        ///
        ///                         - <see cref="VCA_Select.DEFAULT"/>: Option to disable
        ///                           the diversification for VcSelect MAC and ENC keys.
        ///
        ///                         - VC selection can be done in 1-part of 2-parts.
        ///                             - <see cref="VCA_Select.VARIANT_PART1"/>:
        ///                               The complete VC Selection is performed in 1-part
        ///                             - <see cref="VCA_Select.VARIANT_PART2"/>:
        ///                               The complete VC Selection is performed in 2-parts,
        ///                               Postpone Cmd.PICC_ISO_ExternalAuthenticate to second part
        /// </param>
        /// <param name="bEncKeyNo">KeyID.VCSelectENCKey SAM Key Number.
        ///                             - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                             - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bEncKeyVer">KeyID.VCSelectENCKey SAM Key Version.</param>
        /// <param name="bMacKeyNo">KeyID.VCSelectMACKey SAM Key Number.
        ///                             - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                             - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bMacKeyVer">KeyID.VCSelectMACKey SAM Key Version.</param>
        /// <param name="aIID">The Installation Identifier (IID) to be selected.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response received from SAM based on the variants and
        ///                         AuthVCMandatory settings
        ///                             - AuthVC Not Mandatory & Variant 1 : Conf0 (1byte), FCI ( N bytes )
        ///                             - AuthVC Mandatory & Variant 1     : Conf1 (1byte), Decrypted VC
        ///                               Data (16 bytes)
        ///                             - AuthVC Mandatory & Variant 2     : Decrypted VC Data (16 bytes)
        /// </param>
        /// <param name="wPiccRetCode">The status code returned from the PICC. This will be
        ///                            applicable for both the variants.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_VCA_Select ( byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer,
            byte[] aIID, byte[] aDivInput, out byte[] aResponse, out ushort wPiccRetCode )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            wPiccRetCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_VCA_Select ( ref m_DataParamsInt[0], bOption, bEncKeyNo, bEncKeyVer,
                bMacKeyNo, bMacKeyVer, aIID, ( byte ) ( ( aIID == null ) ? 0 : aIID.Length ), aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref pResponse, ref wRespLen,
                ref wPiccRetCode );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// The VCA_ProximityCheck command performs the complete PICC proximity check between the SAM
        /// and the PICC. It performs the PICC PreparePC command, the one or more PICC ProxmityChecks and the
        /// PICC VerifyPC commands.
        /// </summary>
        ///
        /// <param name="bOption">Option to indicate diversification, processing and format options.
        ///                       Options can be ORed.  Also the byte can be framed by using a
        ///                       helper class <seealso cref="VCA_ProximityCheck_Option"/>
        ///                         - <see cref="VCA_ProximityCheck.DIVERSIFICATION_ON"/>
        ///                           Use Key Diversification. Instructs SAM to Diversify the Key.
        ///
        ///                         - Option to indicate VerifyPC processing.
        ///                             - <see cref="VCA_ProximityCheck.NORMAL_PROCESSING"/>
        ///                               Normal Cmd.PICC_VerifyPC processing
        ///                             - <see cref="VCA_ProximityCheck.RANDOM_PROCESSING"/>
        ///                               Random Cmd.PICC_VerifyPC processing
        ///
        ///                         - The command format (ISO7816-4 mode) to be used to
        ///                           communicated to PICC.
        ///                             - <see cref="VCA_ProximityCheck.NATIVE_FORMAT"/>
        ///                               Use native commands
        ///                             - <see cref="VCA_ProximityCheck.WRAPPED_FORMAT"/>
        ///                               Wrap commands in ISO7816-4 compliant format
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM.
        ///                         - 0x00 - 0x7F: Standard ( NVRam) Key
        ///                         - 0xE0 - 0xE3: RAM Key
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="bNumOfRand">Maximum number of random bytes sent in one command.
        ///                             PICC_ProximityCheck
        ///                                 - M == 8: Only one iteration is made, SAM send all 8
        ///                                   random bytes in one Cmd.PICC_ProximityCheck
        ///                                 - M == 7: The SAM sends the first 7 random bytes in one
        ///                                   Cmd.PICC_ProximityCheck and the remaining random byte in
        ///                                   another one
        ///                                 - And so on as
        ///                                     - M = 6 + 2
        ///                                     - M = 5 + 3
        ///                                     - M = 4 + 4
        ///                                     - M = 3 + 3 + 2
        ///                                     - M = 2 + 2 + 2 + 2
        ///                                     - Finally M == 1
        ///                                     - The SAM sends 8 Cmd.PICC_ProximityCheck with one random
        ///                                       byte.
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response received from SAM.
        ///                             - For PICC Error  : The PICC error code will be returned.
        ///                             - For PICC Success: The PPCDataLen and PPCData will be returned.
        ///                                 - PPCDataLen: Length of Cmd.PICC_PreparePC response data
        ///                                 - PPCData: Cmd.PICC_PreparePC response data: Option || PubRespTime
        ///                                   [|| PPS1][|| ActBitRate]
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_VCA_ProximityCheck ( byte bOption, byte bKeyNo, byte bKeyVer, byte bNumOfRand, byte[] aDivInput,
            out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_VCA_ProximityCheck ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                bNumOfRand, aDivInput, (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref pResponse,
                ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }
        #endregion
        #endregion

        #region MIFARE DESFire
        #region S_Mode
        /// <summary>
        /// <para>Performs first part of encryption and decryption of data received from PICC and to be sent to PICC.
        /// Here the Encrypted RndB data will be sent to SAM.SAM will Decrypt the data and Encrypt
        /// RndA with RndB'. This encrypted RndA and RndB' will be returned to the caller for further transmission to
        /// the PICC.
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_AuthenticatePICC_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///       <see cref="Cmd_SAM_AuthenticatePICC_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by
        ///                       bitwise OR. Also the byte can be framed by using a
        ///                        helper class <seealso cref="DESFire_AuthenticateOption"/>
        ///                         - Option for Authentication mode and Authentication type
        ///                             - <see cref="MFD_AuthType.D40_EV1"/>: D40 & EV1 authentication type
        ///                             - <see cref="MFD_AuthType.D40_EV1"/>: EVx Authenticate first
        ///                             - <see cref="MFD_AuthType.EV2_NON_FIRST_AUTH"/>: EVx Authenticate non first
        ///
        ///                         - Option for Suppressing secure messaging
        ///                             - <see cref="MFD_AuthSM.ALLOW"/>: Allow secure messaging
        ///                             - <see cref="MFD_AuthSM.SUPPRESS"/>: Suppress secure messaging
        ///                               (for example for originality keys)
        ///
        ///                         - Option for Key Derivation Information and Key Diversification Types.
        ///                             - <see cref="MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION"/>:
        ///                                 - AV1 compatibility mode  key diversification methods, 3TDEA, AES key
        ///                                 - AV1 compatibility mode" key diversification methods, TDEA Key,
        ///                                   diversified using one encryption round
        ///                             - <see cref="MFD_AuthDivMode.AV1_DOUBLE_ENCRYPTION"/>: AV1 compatibility mode
        ///                               key diversification methods, TDEA Key, diversified using two encryption rounds
        ///                             - <see cref="MFD_AuthDivMode.AV2"/>: AV2 mode key diversification methods
        ///
        ///                         - Option for Key Selection
        ///                             - <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>: Key selection by key entry number
        ///                             - <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>: Key selection by DESFire key number
        ///
        ///                         - Option for Key Diversification
        ///                             - <see cref="MFD_AuthDiv.OFF"/>: No diversification
        ///                             - <see cref="MFD_AuthDiv.ON"/>: Diversify the used key with
        ///                               the given DivInput
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM or DESFire Key number.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>,
        ///                           0x00 - 0x0D should be used.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>,
        ///                           then below one should be used
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="bAuthMode">The type of key to be used for EV2 authentication.
        ///                             - <see cref="MFD_AuthMode.EV2"/>: EV2 authentication
        ///                             - <see cref="MFD_AuthMode.LRP"/>: LRP authentication
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If <see cref="MFD_AuthDiv.ON"/>: Diversification
        ///                               is Enabled, then
        ///                                 - 8 bytes (if AV1 key diversification with DES)
        ///                                 - 16 bytes (if AV1 key diversification with AES)
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                   diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aCardResponse">Encrypted Challenge (RndB) data returned by the PICC.
        ///                                 - If targeting D40 & EV1 authentication with 2TDEA
        ///                                   Key: 8 bytes as E ( Kx, RndB)
        ///                                 - If targeting D40 & EV1 authentication with 3TDEA, AES
        ///                                   Key or if targeting EV2 authentication: 16 bytes as
        ///                                   E ( Kx, RndB)
        ///                                 - If targeting LRP authentication: 16 bytes as RndB</param>
        /// <param name="aSamResponse">Encrypted PCD Challenge (RndA and RndB') returned by
        ///                            the SAM.One of the following will be available,
        ///                                 - If targeting D40 & EV1 authentication with 2TDEA Key:
        ///                                   16 bytes as E(Kx, RndA || RndB')
        ///                                 - If targeting D40 & EV1 authentication with 3TDEA, AES
        ///                                   Key or if targeting EV2 authentication: 32 bytes as
        ///                                   E(Kx, RndA || RndB')
        ///                                 - If targeting LRP authentication: 32 bytes as RndA ||
        ///                                   MAC(Km, RndA||RndB)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aDivInput is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticatePICC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte bAuthMode, byte[] aDivInput,
            byte bDivInputLen, byte[] aCardResponse, out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                bAuthMode, aDivInput, bDivInputLen, aCardResponse, ( byte) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ),
                ref pSamResponse, ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>
        /// Performs second part of decryption of data received from PICC. Here the Encrypted RndA'
        /// data will be sent to SAM.SAM will Decrypt the data and extract PCD and PD Capabilities for EV2
        /// First Auth and null in case of rest of Authentication modes.This PCD and PD information will be
        /// returned to the caller.Also the status code of PICC will be returned to the caller in case of
        /// error.
        /// </para>
        ///
        /// \note
        /// This interface should be called only if <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bPiccErrorCode">Status code returned by the PICC in case of failure else
        ///                              00h for success.
        /// </param>
        /// <param name="aCardResponse">Encrypted RndA' returned by the SAM. One of the following,
        ///                                 - If targeting EV2 first authentication: 32 bytes as
        ///                                   E ( Kx, TI||RndA'||PDcap2||PCDcap2)
        ///                                 - If targeting LRP first authentication: 32 bytes as
        ///                                   E(Ke, TI||PDCap2||PCDCap2) || MAC ( Km, RndB || RndA
        ///                                   E(Ke, TI||PDCap2||PCDCap2))
        ///                                 - If targeting LRP non-first authentication: 16 bytes
        ///                                   as MAC ( Km, RndB || RndA)
        ///                                 - If targeting D40 & EV1 authentication with 2TDEA
        ///                                   Key: 8 or 16 bytes as E ( Kx, RndA')
        /// </param>
        /// <param name="aPCDCap2">Buffer containing the output PD capabilities. This will
        ///                        contain 6 bytes of PD information, if targeting EVx
        ///                        authentication type and first authentication.
        /// </param>
        /// <param name="aPDCap2">Buffer containing the output PCD capabilities. This will
        ///                       contain 6 bytes of PCD information, if targeting EVx
        ///                       authentication type and first authentication.
        /// </param>
        /// <param name="bStatusCode">Status code from MIFARE DESFire PICC if available else zero.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticatePICC_Part2 ( byte bPiccErrorCode, byte[] aCardResponse, out byte[] aPDcap2,
            out byte[] aPCDcap2, out byte bStatusCode )
        {
            bStatusCode = 0;
            aPCDcap2 = new byte[6];
            aPDcap2 = new byte[6];

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2 ( ref m_DataParamsInt[0], bPiccErrorCode,
                aCardResponse, ( byte ) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ), aPDcap2, aPCDcap2,
                ref bStatusCode );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                if ( aCardResponse != null )
                {
                    if ( !aCardResponse.Length.Equals ( 32 ) )
                    {
                        aPCDcap2 = null;
                        aPDcap2 = null;
                    }
                }
            }

            return oStatus;
        }

        /// <summary>
        /// <para>Performs first part of encryption and decryption of data received from PICC and to be sent to PICC.
        /// Here the Encrypted RndB data will be sent to SAM.SAM will Decrypt the data and Encrypt RndA with RndB'.
        /// This encrypted RndA and RndB' will be returned to the caller for further transmission to the PICC.
        /// </para>
        ///
        /// \note
        /// - Post calling this interface it's must to call <see cref="Cmd_SAM_IsoAuthenticatePICC_Part2"/>
        ///   to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///   returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        /// - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///   <see cref="Cmd_SAM_IsoAuthenticatePICC_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by
        ///                       bitwise OR. Also the byte can be framed by using a
        ///                        helper class <seealso cref="DESFire_AuthenticateOption"/>
        ///                         - Option for Key Derivation Information and Key Diversification Types.
        ///                             - <see cref="MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION"/>:
        ///                                 - AV1 compatibility mode  key diversification methods, 3TDEA, AES key
        ///                                 - AV1 compatibility mode" key diversification methods, TDEA Key,
        ///                                   diversified using one encryption round
        ///                             - <see cref="MFD_AuthDivMode.AV1_DOUBLE_ENCRYPTION"/>: AV1 compatibility mode
        ///                               key diversification methods, TDEA Key, diversified using two encryption rounds
        ///                             - <see cref="MFD_AuthDivMode.AV2"/>: AV2 mode key diversification methods
        ///
        ///                         - Option for Key Selection
        ///                             - <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>: Key selection by key entry number
        ///                             - <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>: Key selection by DESFire key number
        ///
        ///                         - Option for Key Diversification
        ///                             - <see cref="MFD_AuthDiv.OFF"/>: No diversification
        ///                             - <see cref="MFD_AuthDiv.ON"/>: Diversify the used key with
        ///                               the given DivInput
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM or DESFire Key number.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>,
        ///                           0x00 - 0x0D should be used.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>,
        ///                           then below one should be used
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If <see cref="MFD_AuthDiv.ON"/>: Diversification
        ///                               is Enabled, then
        ///                                 - 8 bytes (if AV1 key diversification with DES)
        ///                                 - 16 bytes (if AV1 key diversification with AES)
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                   diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aCardResponse">PICC challenge in clear as received from
        ///                             PICC's Cmd.ISOGetChallenge.
        ///                                 - If targeting TDEA Key: 8 bytes encrypted Challenge
        ///                                 - If targeting 3TDEA or AES Key: 16 bytes encrypted
        ///                                   Challenge
        /// </param>
        /// <param name="aSamResponse">Encrypted PCD Challenge (RndA and RndB') and response
        ///                            returned by the SAM.One of the following will be available.
        ///                                 - If targeting TDEA Key: 24 bytes as E(Kx, RPCD1 ||
        ///                                   RPICC1 ) || RPCD2
        ///                                 - If targeting 3TDEA or AES Key: 48 bytes as E ( Kx,
        ///                                   RPCD1 || RPICC1 ) || RPCD2
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aDivInput is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput,
            byte bDivInputLen, byte[] aCardResponse, out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            aSamResponse = null;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 ( ref m_DataParamsInt[0], bOption,
                bKeyNo, bKeyVer, aDivInput, bDivInputLen, aCardResponse,
                ( byte) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ),
                ref pSamResponse, ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>Performs second part of decryption of data received from PICC. Here the Encrypted RndA' data
        /// will be sent to SAM. SAM will Decrypt the data and extract PCD and PD Capabilities for EV2 First
        /// Auth and null in case of rest of Authentication modes. This PCD and PD information will be returned
        /// to the caller. Also the status code of PICC will be returned to the caller in case of error.
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_IsoAuthenticatePICC_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="aCardResponse">PICC encrypted response as received from
        ///                             Cmd.ISOInternalAuthenticate. One of the following,
        ///                                 - If targeting TDEA Key: 16 bytes as E ( Kx, RPICC2
        ///                                   || RRPCD2 )
        ///                                 - If targeting 3TDEA or AES Key: 32 bytes as
        ///                                   E ( Kx, RPICC2 || RRPCD2 )
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part2 ( byte[] aCardResponse )
        {
            return phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2 ( ref m_DataParamsInt[0], aCardResponse,
                ( byte ) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ) );
        }

        /// <summary>
        /// Performs key change for the specified current key to a new key. The crypto operation of the key to be
        /// changed will be calculated by SAM hardware. This crypto data will then sent to card to perform Change
        /// Key operations.
        /// </summary>
        ///
        /// <param name="bCryptoMethod">Key compilation method. Options for P1 information byte.
        ///                             The options can be combined by bitwise OR. Also the
        ///                             byte can also be framed by using a helper class
        ///                             <seealso cref="DESFire_ChangeKeyCompilationOption"/>
        ///                                 - Key diversification method
        ///                                     - <see cref="MFD_CKDivMethod.AV1"/>:
        ///                                       SAM AV1 compatibility diversification method
        ///                                     - <see cref="MFD_CKDivMethod.AV2"/>:
        ///                                       SAM AV2 diversification method
        ///
        ///                                 - SAM AV1 and SAM AV2 Key diversification method
        ///                                     - Current Key
        ///                                         - <see cref="MFD_CKCurrKey.DOUBLE_ENCRYPTION"/>:
        ///                                           Diversify using two encryption rounds
        ///                                         - <see cref="MFD_CKCurrKey.SINGLE_ENCRYPTION"/>:
        ///                                           Diversify using one encryption round
        ///                                         - <see cref="MFD_CKCurrKey.DIV_OFF"/>:
        ///                                           Do not diversified the current key
        ///                                         - <see cref="MFD_CKCurrKey.DIV_ON"/>:
        ///                                           Diversified the current key
        ///                                     - New Key
        ///                                         - <see cref="MFD_CKNewKey.DOUBLE_ENCRYPTION"/>:
        ///                                           Diversify using two encryption rounds
        ///                                         - <see cref="MFD_CKNewKey.SINGLE_ENCRYPTION"/>:
        ///                                           Diversify using one encryption round
        ///                                         - <see cref="MFD_CKNewKey.DIV_OFF"/>:
        ///                                           Do not diversified the current key
        ///                                         - <see cref="MFD_CKNewKey.DIV_ON"/>:
        ///                                           Diversified the current key
        ///
        ///                                     - Cryptogram computation mode
        ///                                         - <see cref="MFD_CKCryptoMode.DIFFERENT_KEY"/>:
        ///                                           PICC targeted key different" from the PICC authenticated
        ///                                           key
        ///                                         - <see cref="MFD_CKCryptoMode.SAME_KEY"/>:
        ///                                           PICC targeted key equal" to PICC authenticated key.
        ///                                           The parameters CurrKeyNo and CurrKeyV are ignored"
        /// </param>
        /// <param name="bConfig">Options for P2 information byte. The options can be
        ///                       combined by bitwise OR. Also the byte can
        ///                       be framed by using a helper class
        ///                       <seealso cref="DESFire_ChangeKeyConfigurationOption"/>
        ///                         - PICC Change Key command
        ///                             - <see cref="MFD_CKCmdType.CHANGE_KEY"/>:
        ///                               Cmd.ChangeKey
        ///                             - <see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>:
        ///                               Cmd.ChangeKeyEV2
        ///
        ///                         - PICC master key update
        ///                             - <see cref="MFD_CKMasterUpdate.EXCLUDE_KEYTYPE"/>:
        ///                               Do not include the key type in the cryptogram"
        ///                             - <see cref="MFD_CKMasterUpdate.INCLUDE_KEYTYPE"/>:
        ///                               Include the key type in the cryptogram
        ///
        ///                         - Number of PICC key to be changed.Will be taken from \b bDFKeyNo
        ///                           parameter if \b bConfig has <see cref="MFD_CKCmdType.CHANGE_KEY"/>
        /// </param>
        /// <param name="bKeySetNo">If \b bConfig = <see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>,
        ///                         Key set number to which the key to be changed belongs to.
        /// </param>
        /// <param name="bDFKeyNo">Block number of the key available in the PICC. This will be used while
        ///                        exchanging the command with PICC.
        ///                         - The lower nibble will be used for P2 information byte if \b bConfig
        ///                           has <see cref="MFD_CKCmdType.CHANGE_KEY"/>.
        ///                         - Complete byte will be used if \b bConfig has
        ///                           <see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>
        /// </param>
        /// <param name="bCurrKeyNo">Number of key entry holding the current key in SAM.
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bCurrKeyVer">Key version of the current key in SAM.</param>
        /// <param name="bNewKeyNo">Number of key entry holding the new key in SAM.
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bNewKeyVer">Key version of new key in SAM. </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If any of diversification option is set in
        ///                               \b bCryptoMethod parameter, then
        ///                                 - 8 bytes (if AV1 key diversification with DES)
        ///                                 - 16 bytes (if AV1 key diversification with AES)
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                   diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer. </param>
        /// <param name="aSamResponse">Cryptogram holding key data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aDivInput is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKeyPICC ( byte bCryptoMethod, byte bConfig, byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo,
            byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] aDivInput, byte bDivInputLen, out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ChangeKeyPICC ( ref m_DataParamsInt[0], bCryptoMethod, bConfig,
                bKeySetNo, bDFKeyNo, bCurrKeyNo, bCurrKeyVer, bNewKeyNo, bNewKeyVer, aDivInput, bDivInputLen,
                ref pSamResponse, ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// The SAM_CreateTMFilePICC command supports the procedure to create a Transaction MAC
        /// File in the PICC in S-mode from a key stored in the SAM.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification selection. Options for P1
        ///                       information byte.
        ///                         - <see cref="MFD_TMDiv.OFF"/>: No diversification
        ///                         - <see cref="MFD_TMDiv.ON"/>: Diversify the used key
        ///                           with the given DivInput
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM. One of the following
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bFileOption">Options for the targeted file.
        ///                             - Communication Mode
        ///                             - <see cref="MFD_TMFileOption.PLAIN"/>
        ///                             - <see cref="MFD_TMFileOption.MAC"/>
        ///                             - <see cref="MFD_TMFileOption.FULL"/>
        ///
        ///                             Can bit bitwise OR with above option
        ///                             - <see cref="MFD_TMFileOption.TMI_EXCLUSION_FILEMAP"/>
        /// </param>
        /// <param name="aAccessRights">Access conditions to be applied for the file. Refer
        ///                             respective product DataSheet for access rights
        ///                             information. This should be two bytes long.
        /// </param>
        /// <param name="aTMIExclFileMap">TMI exclusion FileMap. Should be 4 byte.</param>
        /// <param name="bTMKeyOptions">Option for the TransactionMAC Key.
        ///                                 - TMI exclusion FileMap. Should be 4 byte.
        ///                                     - <see cref="MFD_TMKeyOption.KEYTYPE_AES128"/>
        ///                                     - <see cref="MFD_TMKeyOption.KEYTYPE_AES256"/>
        ///
        ///                                 - Mode. Can bit bitwise OR with above options
        ///                                     <see cref="MFD_TMKeyOption.MODE_TMAC"/>
        ///                                     <see cref="MFD_TMKeyOption.MODE_TSIG"/>
        /// </param>
        /// <param name="bTSIGKeyNo">Key Number pointing to AppTransactionSIGKey.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aSamResponse">Cryptogram holding Transaction MAC Key data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aDivInput is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CreateTMFilePICC ( byte bOption, byte bKeyNo, byte bKeyVer, byte bFileNo, byte bFileOption,
            byte[] aAccessRights, byte[] aTMIExclFileMap, byte bTMKeyOptions, byte bTSIGKeyNo, byte[] aDivInput,
            byte bDivInputLen, out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_CreateTMFilePICC ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                bFileNo, bFileOption, aAccessRights, aTMIExclFileMap, bTMKeyOptions, bTSIGKeyNo, aDivInput,
                bDivInputLen, ref pSamResponse, ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// Perform an authentication procedure between SAM AV3 and MIFARE DESFire.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by
        ///                       bitwise OR. Also the byte can be framed by using a
        ///                        helper class <seealso cref="DESFire_AuthenticateOption"/>
        ///                         - Option for Authentication mode and Authentication type
        ///                             - <see cref="MFD_AuthType.D40_EV1"/>: D40 & EV1 authentication type
        ///                             - <see cref="MFD_AuthType.D40_EV1"/>: EVx Authenticate first
        ///                             - <see cref="MFD_AuthType.EV2_NON_FIRST_AUTH"/>: EVx Authenticate non first
        ///
        ///                         - Option for Suppressing secure messaging
        ///                             - <see cref="MFD_AuthSM.ALLOW"/>: Allow secure messaging
        ///                             - <see cref="MFD_AuthSM.SUPPRESS"/>: Suppress secure messaging
        ///                               (for example for originality keys)
        ///
        ///                         - Option for Key Derivation Information and Key Diversification Types.
        ///                             - <see cref="MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION"/>:
        ///                                 - AV1 compatibility mode  key diversification methods, 3TDEA, AES key
        ///                                 - AV1 compatibility mode" key diversification methods, TDEA Key,
        ///                                   diversified using one encryption round
        ///                             - <see cref="MFD_AuthDivMode.AV1_DOUBLE_ENCRYPTION"/>: AV1 compatibility mode
        ///                               key diversification methods, TDEA Key, diversified using two encryption rounds
        ///                             - <see cref="MFD_AuthDivMode.AV2"/>: AV2 mode key diversification methods
        ///
        ///                         - Option for Key Selection
        ///                             - <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>: Key selection by key entry number
        ///                             - <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>: Key selection by DESFire key number
        ///
        ///                         - Option for Key Diversification
        ///                             - <see cref="MFD_AuthDiv.OFF"/>: No diversification
        ///                             - <see cref="MFD_AuthDiv.ON"/>: Diversify the used key with
        ///                               the given DivInput
        /// </param>
        /// <param name="bISOMode">ISO mode to be used.
        ///                         - <see cref="ISOMode.NATIVE"/>:
        ///                           Command will be sent to PICC using native command set
        ///                         - <see cref="ISOMode.ISO7816_4"/>:
        ///                           Command will be sent to PICC using ISO/IEC 7816-4APDU
        ///                         - <see cref="ISOMode.ISO_AUTHENTICATION"/>:
        ///                           ISO compliant authentication (Cmd.ISOGetChallenge,
        ///                           Cmd.ISOExternalAuthenticate and Cmd.ISOInternalAuthenticate)
        ///                           will be performed
        /// </param>
        /// <param name="bDFKeyNo">DESFire Key Number to be used for authentication.</param>
        /// <param name="bKeyNo">Key number to be used in SAM or DESFire Key number.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.DESFIRE_KEY_NUMBER"/>,
        ///                           0x00 - 0x0D should be used.
        ///                         - If Key selection by <see cref="MFD_AuthKeySel.KEY_ENTRY_NUMBER"/>,
        ///                           then below one should be used
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="aPCDcap2In">Input PCD capabilities to be exchanged.
        ///                             - NonFirstAuth                             : Should be null.
        ///                             - FirstAuth with no Input PCD capabilities : Should be null.
        ///                             - FirstAuth with PCDcap2InLen = 0xFF       : Should be null.
        ///                             - FirstAuth with PCDcap2InLen = 0x00 - 0x06: Should not be null.
        ///                               The PCD input capabilities should be passed.
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If <see cref="MFD_AuthDiv.ON"/>, then
        ///                                 - 8 bytes (if AV1 key diversification with DES)
        ///                                 - 16 bytes (if AV1 key diversification with AES)
        ///                                 - 1 to 31 bytes (if AV2 key diversification) diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="aPcdCapOut">Buffer containing the output PD capabilities. This will
        ///                          contain 6 bytes of PD information, if targeting EVx
        ///                          authentication type and first authentication.
        /// </param>
        /// <param name="aPDcap2">Buffer containing the output PCD capabilities. This will
        ///                       contain 6 bytes of PCD information, if targeting EVx
        ///                       authentication type and first authentication.
        /// </param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. Will be
        ///                                 - 1 byte if \b bISOMode = <see cref="ISOMode.NATIVE"/>
        ///                                 - 2 bytes if \b bISOMode = <see cref="ISOMode.ISO7816_4"/> or
        ///                                   <see cref="MFD_ISOMode.ISO_AUTHENTICATION"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DESFire_AuthenticatePICC ( byte bOption, byte bISOMode, byte bDFKeyNo, byte bKeyNo, byte bKeyVer,
            byte[] aPCDcap2In, byte[] aDivInput, out byte[] aPDcap2, out byte[] aPCDcap2, out byte[] aPiccErrorCode )
        {
            aPCDcap2 = new byte[6];
            aPDcap2 = new byte[6];
            aPiccErrorCode = new byte[2];
            byte bPCDcap2InLen = 0;
            byte[] aPCDcap2In_Tmp = null;

            /* Frame the Input PCD capabilities length and the buffer. */
            /* For FirstAuth with no input capabilities or NonFirstAuth. */
            if ( aPCDcap2In == null )
            {
                bPCDcap2InLen = 0;
                aPCDcap2In_Tmp = null;
            }
            else
            {
                /* For FirstAuth and Default PCD capabilities. */
                if ( aPCDcap2In.Length.Equals ( 1 ) && aPCDcap2In[0].Equals ( 0xFF ) )
                {
                    bPCDcap2InLen = 0xFF;
                    aPCDcap2In_Tmp = null;
                }
                /* Reset of the PCD capabilities information. */
                else
                {
                    bPCDcap2InLen = ( byte ) aPCDcap2In.Length;
                    aPCDcap2In_Tmp = ( byte[] ) aPCDcap2In.Clone ();
                }
            }

            Status_t oStatus = phhalHw_Sam_Cmd_DESFire_AuthenticatePICC ( ref m_DataParamsInt[0], bOption, bISOMode,
                bDFKeyNo, bKeyNo, bKeyVer, bPCDcap2InLen, aPCDcap2In_Tmp, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPDcap2,
                aPCDcap2, aPiccErrorCode );

            if ( !oStatus.Equals ( new Status_t () ) )
            {
                aPCDcap2 = null;
                aPDcap2 = null;
            }

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bISOMode.Equals ( ( byte ) ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            if ( !( bOption & ( byte ) MFD_AuthType.EV2_FIRST_AUTH ).Equals ( ( byte ) MFD_AuthType.EV2_FIRST_AUTH ) )
            {
                aPCDcap2 = null;
                aPDcap2 = null;
            }

            return oStatus;
        }

        /// <summary>
        /// The DESFire_ChangeKeyPICC command supports the procedure to change a key
        /// stored in the PICC in XMode
        /// </summary>
        ///
        /// <param name="bCryptoMethod">Key compilation method. Options for P1 information byte.
        ///                             The options can be combined by bitwise OR. Also the
        ///                             byte can also be framed by using a helper class
        ///                             <seealso cref="DESFire_ChangeKeyCompilationOption"/>
        ///                                 - Key diversification method
        ///                                     - <see cref="MFD_CKDivMethod.AV1"/>:
        ///                                       SAM AV1 compatibility diversification method
        ///                                     - <see cref="MFD_CKDivMethod.AV2"/>:
        ///                                       SAM AV2 diversification method
        ///
        ///                                 - SAM AV1 and SAM AV2 Key diversification method
        ///                                     - Current Key
        ///                                         - <see cref="MFD_CKCurrKey.DOUBLE_ENCRYPTION"/>:
        ///                                           Diversify using two encryption rounds
        ///                                         - <see cref="MFD_CKCurrKey.SINGLE_ENCRYPTION"/>:
        ///                                           Diversify using one encryption round
        ///                                         - <see cref="MFD_CKCurrKey.DIV_OFF"/>:
        ///                                           Do not diversified the current key
        ///                                         - <see cref="MFD_CKCurrKey.DIV_ON"/>:
        ///                                           Diversified the current key
        ///                                     - New Key
        ///                                         - <see cref="MFD_CKNewKey.DOUBLE_ENCRYPTION"/>:
        ///                                           Diversify using two encryption rounds
        ///                                         - <see cref="MFD_CKNewKey.SINGLE_ENCRYPTION"/>:
        ///                                           Diversify using one encryption round
        ///                                         - <see cref="MFD_CKNewKey.DIV_OFF"/>:
        ///                                           Do not diversified the current key
        ///                                         - <see cref="MFD_CKNewKey.DIV_ON"/>:
        ///                                           Diversified the current key
        ///
        ///                                     - Cryptogram computation mode
        ///                                         - <see cref="MFD_CKCryptoMode.DIFFERENT_KEY"/>:
        ///                                           PICC targeted key different" from the PICC authenticated
        ///                                           key
        ///                                         - <see cref="MFD_CKCryptoMode.SAME_KEY"/>:
        ///                                           PICC targeted key equal" to PICC authenticated key.
        ///                                           The parameters CurrKeyNo and CurrKeyV are ignored"
        /// </param>
        /// <param name="bConfig">Options for P2 information byte. The options can be
        ///                       combined by bitwise OR. Also the byte can
        ///                       be framed by using a helper class
        ///                       <seealso cref="DESFire_ChangeKeyConfigurationOption"/>
        ///                         - ISO mode to be used.
        ///                             - <see cref="ISOMode.NATIVE"/>: Command will be sent
        ///                               to PICC using native command set
        ///                             - <see cref="ISOMode.ISO7816_4"/>: Command will be sent
        ///                               to PICC using ISO/IEC 7816-4APDU
        ///
        ///                         - PICC Change Key command
        ///                             - <see cref="MFD_CKCmdType.CHANGE_KEY"/>:
        ///                               Cmd.ChangeKey
        ///                             - <see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>:
        ///                               Cmd.ChangeKeyEV2
        ///
        ///                         - PICC master key update
        ///                             - <see cref="MFD_CKMasterUpdate.EXCLUDE_KEYTYPE"/>:
        ///                               Do not include the key type in the cryptogram"
        ///                             - <see cref="MFD_CKMasterUpdate.INCLUDE_KEYTYPE"/>:
        ///                               Include the key type in the cryptogram
        ///
        ///                         - Number of PICC key to be changed.Will be taken from \b bDFKeyNo
        ///                           parameter if \b bConfig has <see cref="MFD_CKCmdType.CHANGE_KEY"/>
        /// </param>
        /// <param name="bKeySetNo">If \b bConfig = <see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>,
        ///                         Key set number to which the key to be changed belongs to.
        /// </param>
        /// <param name="bDFKeyNo">Number of DESFire PICC key to be changed. This should be present only if
        ///                        \b bConfig =<see cref="MFD_CKCmdType.CHANGE_KEY_EV2"/>.
        /// </param>
        /// <param name="bCurrKeyNo">Number of key entry holding the current key in SAM.
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bCurrKeyVer">Key version of the current key in SAM.</param>
        /// <param name="bNewKeyNo">Number of key entry holding the new key in SAM.
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bNewKeyVer">Key version of new key in SAM. </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If any of diversification option is set in
        ///                               \b bCryptoMethod parameter, then
        ///                                 - 8 bytes (if AV1 key diversification with DES)
        ///                                 - 16 bytes (if AV1 key diversification with AES)
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                   diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. Will be
        ///                                 - 1 byte if \b bISOMode = <see cref="MFD_ISOMode.NATIVE"/>
        ///                                 - 2 bytes if \b bISOMode = <see cref="MFD_ISOMode.ISO7816_4"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DESFire_ChangeKeyPICC ( byte bCryptoMethod, byte bConfig, byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo,
            byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] aDivInput, out byte[] aPiccErrorCode )
        {
            aPiccErrorCode = new byte[2];
            Status_t oStatus = phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC ( ref m_DataParamsInt[0], bCryptoMethod, bConfig, bKeySetNo,
                bDFKeyNo, bCurrKeyNo, bCurrKeyVer, bNewKeyNo, bNewKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                aPiccErrorCode );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bConfig.Equals ( ( byte ) ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            return oStatus;
        }

        /// <summary>
        /// The DESFire_WriteX command is used to apply the DESFire Secure Messaging in X-mode
        /// on PICC commands writing data to the DESFire PICC.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options for exchanging information to SAM.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first" set of information. No exchange is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate" set of information. No exchange is
        ///                           performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last" set of information. Exchange information
        ///                           to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/>: Should be used
        ///                           to exchange chunks of data and receive response.To be
        ///                           ORed with above option and to set LFI flag of P1 byte.
        /// </param>
        /// <param name="bCryptoConfig">Option to set the P2 information byte. CAn be combined by
        ///                             using bitwise OR operator. Also the byte can be framed
        ///                             by using a helper class <seealso cref="DESFire_WriteCryptoConfiguration"/>
        ///                                 - Extended offset
        ///                                     - <see cref="MFD_Offset.LEGACY_MODE"/>: Legacy mode
        ///                                       (MIFARE SAM AV2)
        ///                                     - <see cref="MFD_Offset.EXTENDED_OFFSET"/>: Extended offset
        ///
        ///                                 - Chaining configuration
        ///                                     - <see cref="MFD_Chaining.DESFIRE_CHAINING"/>
        ///                                       DESFire application chaining (0xAF)
        ///                                     - <see cref="MFD_Chaining.ISO_CHAINING"/>
        ///                                       ISO/IEC 14443-4 chaining
        ///
        ///                                 - Communication Mode
        ///                                     - <see cref="CommMode.PLAIN"/>: CommMode.Plain
        ///                                     - <see cref="CommMode.MAC"/>: CommMode.MAC
        ///                                     - <see cref="CommMode.FULL"/>: CommMode.Full
        ///
        ///                                 - Index of byte in data field where crypto operation shall
        ///                                   start
        ///                                     - If Legacy mode: Full Range (0x00 - 0x0F)
        ///                                     - If Extended offset: RFU (0x00)
        /// </param>
        /// <param name="aData">The data to be written to the DESFire PICC.
        ///                         - If <see cref="MFD_Offset.EXTENDED_OFFSET"/>, Communication mode is
        ///                           <see cref="CommMode.FULL"/> and first frame, the buffer should
        ///                           contain offset followed by the PICC related data.
        ///                         - For any other options, the buffer should contain only the PICC related
        ///                           data.
        /// </param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. Will be
        ///                                 - 1 byte if \b bISOMode = <see cref="ISOMode.NATIVE"/>
        ///                                 - 2 bytes if \b bISOMode = <see cref="ISOMode.ISO7816_4"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DESFire_WriteX ( ushort wOption, byte bCryptoConfig, byte[] aData, out byte[] aPiccErrorCode )
        {
            byte bErrLen = 0;
            aPiccErrorCode = new byte[2];

            Status_t oStatus = phhalHw_Sam_Cmd_DESFire_WriteX ( ref m_DataParamsInt[0], wOption, bCryptoConfig, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), aPiccErrorCode, ref bErrLen );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
                Array.Resize ( ref aPiccErrorCode, bErrLen );
            else
                aPiccErrorCode = null;

            return oStatus;
        }

        /// <summary>
        /// The DESFire_ReadX command is used to remove the DESFire Secure Messaging in X-mode
        /// on PICC commands reading data from the DESFire PICC.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: To buffer
        ///                           the initial command information. Use this flag
        ///                           to buffer the Length information also in case
        ///                           of FULL communication mode and more data is
        ///                           expected.
        ///
        ///                         - To receive data in Native Chaining mode use
        ///                           <see cref="ExchangeOptions.RXCHAINING"/>
        ///
        ///                         - To receive data in ISO chaining mode, use
        ///                           <see cref="ExchangeOptions.RXCHAINING"/> |
        ///                           <see cref="MFD_Chaining.ISO_CHAINING"/>
        /// </param>
        /// <param name="bCrypto">Crypto configuration. Option to set the P2 information
        ///                       byte.
        ///                         - <see cref="CommMode.PLAIN"/>: CommMode.Plain
        ///                         - <see cref="CommMode.MAC"/>: CommMode.MAC
        ///                         - <see cref="CommMode.FULL"/>: CommMode.Full
        /// </param>
        /// <param name="aAppData">The following information should be passed.
        ///                         - 3 bytes length information in case if Communication
        ///                           mode is <see cref="CommMode.FULL"/> and more data
        ///                           is expected.
        ///                         - Complete PICC command header and data to be sent to PICC
        ///                           for initial exchange.
        ///                         - DESFire Chaining command code (0xAF) in case more data is
        ///                           expected.
        /// </param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. Will be
        ///                                 - 1 byte if \b bISOMode = <see cref="ISOMode.NATIVE"/>
        ///                                 - 2 bytes if \b bISOMode = <see cref="ISOMode.ISO7816_4"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DESFire_ReadX ( ushort wOption, byte bCrypto, byte[] aAppData, out byte[] aResponse,
            out byte[] aPiccErrorCode )
        {
            byte bErrLen = 0;
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            aPiccErrorCode = new byte[2];
            Status_t oStatus = phhalHw_Sam_Cmd_DESFire_ReadX ( ref m_DataParamsInt[0], wOption, bCrypto,
                aAppData, ( byte ) ( ( aAppData == null ) ? 0 : aAppData.Length ), ref pResponse,
                ref wRespLen, aPiccErrorCode, ref bErrLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
                Array.Resize ( ref aPiccErrorCode, bErrLen );
            else
                aPiccErrorCode = null;

            return oStatus;
        }

        /// <summary>
        /// The DESFire_CreateTMFilePICC command supports the procedure to create a Transaction
        /// MAC File in the PICC in X-mode from a key stored in the SAM.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification selection. Options for P1
        ///                       information byte.
        ///                         - <see cref="MFD_TMDiv.OFF"/>: No diversification
        ///                         - <see cref="MFD_TMDiv.ON"/>: Diversify the used key
        ///                           with the given DivInput
        /// </param>
        /// <param name="bISOMode">Options for P2 information byte.
        ///                         - <see cref="ISOMode.NATIVE"/>
        ///                           "Command will be sent to PICC using native command set"
        ///                         - <see cref="ISOMode.ISO7816_4"/>
        ///                           "Command will be sent to PICC using ISO/IEC 7816-4APDU"
        /// </param>
        /// <param name="bKeyNo">Key number to be used in SAM. One of the following
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used in SAM.</param>
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bFileOption">Options for the targeted file.
        ///                             - Communication Mode
        ///                             - <see cref="MFD_TMFileOption.PLAIN"/>
        ///                             - <see cref="MFD_TMFileOption.MAC"/>
        ///                             - <see cref="MFD_TMFileOption.FULL"/>
        ///
        ///                             Can bit bitwise OR with above option
        ///                             - <see cref="MFD_TMFileOption.TMI_EXCLUSION_FILEMAP"/>
        /// </param>
        /// <param name="aAccessRights">Access conditions to be applied for the file. Refer
        ///                             respective product DataSheet for access rights
        ///                             information. This should be two bytes long.
        /// </param>
        /// <param name="aTMIExclFileMap">TMI exclusion FileMap. Should be 4 byte.</param>
        /// <param name="bTMKeyOptions">Option for the TransactionMAC Key.
        ///                                 - TMI exclusion FileMap. Should be 4 byte.
        ///                                     - <see cref="MFD_TMKeyOption.KEYTYPE_AES128"/>
        ///                                     - <see cref="MFD_TMKeyOption.KEYTYPE_AES256"/>
        ///
        ///                                 - Mode. Can bit bitwise OR with above options
        ///                                     <see cref="MFD_TMKeyOption.MODE_TMAC"/>
        ///                                     <see cref="MFD_TMKeyOption.MODE_TSIG"/>
        /// </param>
        /// <param name="bTSIGKeyNo">Key Number pointing to AppTransactionSIGKey.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. Will be
        ///                                 - 1 byte if \b bISOMode = <see cref="ISOMode.NATIVE"/>
        ///                                 - 2 bytes if \b bISOMode = <see cref="ISOMode.ISO7816_4"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DESFire_CreateTMFilePICC ( byte bOption, byte bISOMode, byte bKeyNo, byte bKeyVer, byte bFileNo,
            byte bFileOption, byte[] aAccessRights, byte[] aTMIExclFileMap, byte bTMKeyOptions, byte bTSIGKeyNo,
            byte[] aDivInput, out byte[] aPiccErrorCode )
        {
            aPiccErrorCode = new byte[2];
            Status_t oStatus = phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC ( ref m_DataParamsInt[0], bOption, bISOMode, bKeyNo,
                bKeyVer, bFileNo, bFileOption, aAccessRights, aTMIExclFileMap, bTMKeyOptions, bTSIGKeyNo, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPiccErrorCode );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bISOMode.Equals ( ( byte ) ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            return oStatus;
        }
        #endregion
        #endregion

        #region MIFARE DUOX
        #region S_Mode
        /// <summary>
        /// The SAM_MutualAuthEcc supports in S-mode the ECC-based Mutual and Reader-Unilateral
        /// Authentication as implemented by MIFARE DUOX with the ISOGeneralAuthenticate command.
        /// Generates its ephemeral key pair for the ECDH-based key agreement and returns the public key
        ///
        /// Note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_MutualAuthEcc_Part2"/>
        ///       and <see cref="Cmd_SAM_MutualAuthEcc_Part3"/> to place SAM into proper completion state
        ///       if <see cref="Error_Gen.SUCCESS_CHAINING"/> is returned from Library. Otherwise error
        ///       will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///         - <see cref="Cmd_SAM_MutualAuthEcc_Part2"/> interface.
        ///         - <see cref="Cmd_SAM_MutualAuthEcc_Part3"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information.
        ///                         - <see cref="DUOX_AuthMethod.MUTUAL_AUTH_CERT"/>
        ///                         - <see cref="DUOX_AuthMethod.MUTUAL_AUTH_NO_CERT"/>
        ///                         - <see cref="DUOX_AuthMethod.READER_UNILATERAL_AUTH_CERT"/>
        ///                         - <see cref="DUOX_AuthMethod.READER_UNILATERAL_AUTH_NO_CERT"/>
        /// </param>
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be used for signing.
        ///                              Will be 0x00 - 0x0F
        /// </param>
        /// <param name="bCertA_FileNo">The file number of the file holding the Cert.A certificate
        ///                             (chain). Present in command frame only if \b bOption has one
        ///                             of the following,
        ///                                 - <see cref="DUOX_AuthMethod.MUTUAL_AUTH_CERT"/>
        ///                                 - <see cref="DUOX_AuthMethod.READER_UNILATERAL_AUTH_CERT"/>
        /// </param>
        /// <param name="bCertB_Options">Cert.B processing options. One of the following
        ///                                 - <see cref="DUOX_CertBOption.MSB_B_PLAIN"/>
        ///                                 - <see cref="DUOX_CertBOption.NO_DATA"/>
        ///                                 - <see cref="DUOX_CertBOption.TBS_CERTB"/>
        /// </param>
        /// <param name="bECCKeyNo_CA">ECC key entry holding the public key of the targeted CA Root Key
        ///                            for Cert.B validation. Will be 0x00 - 0x0F. Present in command
        ///                            frame only if \b bOption has one of the following,
        ///                             - <see cref="DUOX_AuthMethod.MUTUAL_AUTH_CERT"/>
        ///                             - <see cref="DUOX_AuthMethod.READER_UNILATERAL_AUTH_CERT"/>
        /// </param>
        /// <param name="aSamResponse">Ephemeral public key in uncompressed point representation
        ///                            ( 0x04 || E.Pub.A.x || E.Pub.A.y )
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_MutualAuthEcc_Part1 ( byte bOption, byte bECCKeyNo_Priv, byte bCertA_FileNo, byte bCertB_Options,
            byte bECCKeyNo_CA, out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1(ref m_DataParamsInt[0], bOption, bECCKeyNo_Priv,
                bCertA_FileNo, bCertB_Options, bECCKeyNo_CA, ref pSamResponse, ref wSamRespLen);
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// Performs second part, SAM receives the ephemeral public key of the MIFARE DUOX,
        /// executes the ephemeral key agreement and returns an encrypted message containing its
        /// certificate chain and signature. Also the status code of PICC will be returned to the
        /// caller in case of error.
        ///
        /// Note: This interface should be called only if <see cref="Cmd_SAM_MutualAuthEcc_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: To buffer
        ///                           the initial command information.Use this flag
        ///                           exchange Ephemeral public key ( E.Pub.B) or PICC
        ///                           Status code and start reception of encrypted
        ///                           message ( Msg.A.enc).
        ///
        ///                         - To receive remaining <see cref="ExchangeOptions.RXCHAINING"/>
        ///                           for reception of Encrypted Message (Msg.A.enc)
        /// </param>
        /// <param name="aPiccErrorCode">Status code returned by the PICC in case of failure.
        ///                              Should be 2 byte in length with LSB first.
        /// </param>
        /// <param name="aCardResponse">Ephemeral public key in uncompressed point representation
        ///                             ( 0x04 || E.Pub.B.x || E.Pub.B.y )
        /// </param>
        /// <param name="aSamResponse">Enciphered Signature and Certificate (Chain, Options)
        ///                             - If AuthMethod = 0x80 or 0x40, Correct execution with
        ///                               Certificate.A.
        ///                               Msg.A.enc = Msg.A.enc.0 || Msg.A.enc.1 || .. ||
        ///                               Msg.A.enc.n = E ( KSesAuthENC, 0xE0E0 || Cert.A ||
        ///                               Sig.A )
        ///                             -If AuthMethod = 0xA0 or 0x60, Correct execution without
        ///                             Certificate.A.
        ///                             Msg.A.enc = E ( KSesAuthENC, 0xE0E0 || Sig.A )
        /// </param>
        /// <param name="bPiccRetCode">Status code from MIFARE DUOX PICC if available else zero.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Returns <see cref="Error.OK_CHAINING_ACTIVE_DUOX"/> Correct Execution with Cert.A - final frame.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failed: public key
        ///             validation for E.Pub.B failed.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_MutualAuthEcc_Part2 ( byte bOption, byte[] aPiccErrorCode, byte[] aCardResponse,
            out byte[] aSamResponse, out byte bPiccRetCode )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            bPiccRetCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2 ( ref m_DataParamsInt[0], bOption, aPiccErrorCode,
                aCardResponse, ( byte ) ( ( aCardResponse == null ) ? 0 : aCardResponse.Length ), ref pSamResponse,
                ref wSamRespLen, ref bPiccRetCode);
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// Performs second part, the SAM receives the encrypted message from the MIFARE DUOX
        /// and, depending on the protocol choice, returns the decrypted OptsB ( Reader-Unilateral
        /// Authentication ) or additionally validates and/or returns the decrypted certificate
        /// chain and signature of the MIFARE DUOX. Also the status code of PICC will be returned
        /// to the caller in case of error.
        ///
        /// Note: This interface should be called only if <see cref="Cmd_SAM_MutualAuthEcc_Part1"/>
        /// <see cref="Cmd_SAM_MutualAuthEcc_Part2"/> returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bOption">Buffering options.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: To buffer the
        ///                           initial command information.Use this flag exchange
        ///                           Enciphered response from PICC (Msg.B.enc) and start
        ///                           reception of response.
        ///
        ///                         - To receive remaining <see cref="ExchangeOptions.RXCHAINING"/>
        ///                           Mutual Authentication response
        /// </param>
        /// <param name="aPiccErrorCode">Status code returned by the PICC in case of failure.
        ///                              Should be 2 byte in length with LSB first.
        /// </param>
        /// <param name="aCardResponse">Response from PICC. One of the following
        ///                                 - If no PICC error and AuthMethod = 0x40 or 0x40,
        ///                                   Reader-unilateral authentication Enciphered response
        ///                                   from PICC ( length X = 16 ):
        ///                                   Msg.B.enc = E(K_{ SesAuthEN }, 0xE1E1 || OptsB)
        ///
        ///                                 - If no PICC error and AuthMethod = 0x80 or 0xA0 and
        ///                                   CertificateB Options == 0x00, Mutual Authentication
        ///                                   with no SAM validation.Enciphered response from PICC
        ///                                   (no limitation on accumulated length Sum(X))
        ///                                   Msg.B.enc = E(K_{ SesAuthEN }, 0xE1E1 || OptsB ||
        ///                                   Cert.B||Sig.B)
        ///
        ///                                 - If no PICC error and AuthMethod = 0x80 or 0xA0 and
        ///                                   CertificateB Options != 0x00, Mutual Authentication
        ///                                   with SAM validation. Enciphered response from PICC
        ///                                   ( Sum ( X) le 960 or 1024 or... )
        ///                                   Msg.B.enc = E(KSesAuth ENC, 0xE1E1 || OptsB || Cert.B
        ///                                   || Sig.B)
        /// </param>
        /// <param name="aSamResponse">One of the following,
        ///                             - If AuthMethod = 0x80 or 0xA0 and CertificateB Options 0x00,
        ///                               Mutual Authentication with no SAM validation.
        ///                               Opts.B || Cert.B || Sig.B
        ///                             - If AuthMethod = 0x80 or 0xA0 and CertificateB Options
        ///                               == 0x02, Mutual authentication - SAM validation - more
        ///                               data available.
        ///                               Opts.B || TBSCert.B.parent || TBSCert.B.leaf
        ///                             - If AuthMethod = 0x80 or 0xA0 and CertificateB Options
        ///                               == 0x01, Mutual authentication - SAM validation - no data
        ///                               returned.
        ///                               Opts.B
        ///                             - If AuthMethod = 0x40 or 0x60, Reader-unilateral
        ///                               authentication.
        ///                               Opts.B
        /// </param>
        /// <param name="bPiccRetCode">Status code from MIFARE DUOX PICC if available else zero.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> Authentication failed
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_MutualAuthEcc_Part3 ( byte bOption, byte[] aPiccErrorCode, byte[] aCardResponse,
            out byte[] aSamResponse, out byte bPiccRetCode )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            bPiccRetCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3 ( ref m_DataParamsInt[0], bOption, aPiccErrorCode,
                aCardResponse, ( byte ) ( ( aCardResponse == null ) ? 0 : aCardResponse.Length ), ref pSamResponse,
                ref wSamRespLen, ref bPiccRetCode);
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// The SAM_UnilatAuthEcc supports in S-mode the ECC-based Card-Unilateral Authentication
        /// as implemented by MIFARE DUOX with the ISOInternalAuthenticate or VDE_ECDSASign command.
        /// In the first part the SAM generates and returns a random challenge. The actual protocol
        /// choice is provided to the SAM via the P1 (\b bOpton ) parameter
        ///
        /// Note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_UnilatAuthEcc_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///       is returned from Library. Otherwise error will be returned by SAM for any subsequent
        ///       commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to
        ///       call <see cref="Cmd_SAM_UnilatAuthEcc_Part2"/> interface.
        /// </summary>
        /// <param name="bOption">Protocol Option to update the P1 information.
        ///                         - <see cref="DUOX_ProtocolOption.ISO_INTERNAL_AUTH"/>
        ///                         - <see cref="DUOX_ProtocolOption."/>
        /// </param>
        /// <param name="bECCKeyNo_Priv">The key reference number of the ECC key entry to be
        ///                              used for signature verification.
        ///                                 - NVM Key: 0x00 - 0x7F
        ///                                 - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bECC_CurveNo">The curve reference number of the ECC curve entry to
        ///                            be used for signature verification.Will be 0x00 - 0x03
        /// </param>
        /// <param name="aSamResponse">Random challenge from SAM</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_UnilatAuthEcc_Part1 ( byte bOption, byte bECCKeyNo_Priv, byte bECC_CurveNo,
            out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1 ( ref m_DataParamsInt[0], bOption, bECCKeyNo_Priv,
                bECC_CurveNo, ref pSamResponse, ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// Performs second part, the SAM receives the signature of the MIFARE DUOX, and executes
        /// the signature validation. Also the status code of PICC will be returned to the caller
        /// in case of error.
        ///
        /// Note: This interface should be called only if <see cref="Cmd_SAM_UnilatAuthEcc_Part2"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/> .
        /// </summary>
        ///
        /// <param name="aPiccErrorCode">Status code returned by the PICC in case of failure.
        ///                              Should be 2 byte in length with LSB first.
        /// </param>
        /// <param name="aCardResponse">Response from PICC. One of the following
        ///                                 - If no PICC error and ProtocolOption = 0x00,
        ///                                   16 byte Random challenge from PICC followed
        ///                                   by 64 byte Signature.
        ///                                   RndA || Sig.B = ECDSA_Sign ( Priv.B, 0xF0F0
        ///                                   [|| OptsA] || RndB || RndA )
        ///
        ///                                 -If no PICC error and ProtocolOption = 0x01,
        ///                                  16 byte Random challenge from PICC followed
        ///                                  by 64 byte Signature.
        ///                                  RndA || Sig.B = ECDSA_Sign ( Priv.B, RA )
        /// </param>
        /// <param name="bPiccRetCode">Status code from MIFARE DUOX PICC if available else zero.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_UnilatAuthEcc_Part2 ( byte[] aPiccErrorCode, byte[] aCardResponse,
            out byte bPiccRetCode )
        {
            bPiccRetCode = 0;
            return phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2 ( ref m_DataParamsInt[0],
                aPiccErrorCode, aCardResponse, ( byte) ( ( aCardResponse == null ) ? 0 : aCardResponse.Length ),
                ref bPiccRetCode );
        }

        /// <summary>
        /// The SAM_BindCertificate supports in S-mode a key pair generation on MIFARE DUOX with ManageKeyPair
        /// command, followed by a certificate signing by the SAM.SAM enables the key pair generation by computing
        /// the ManageKeyPair secure messaging.This is equivalent to a SAM_Apply_SM execution.MIFARE DUOX may
        /// return an error code during the execution of the protocol, the SAM will accept this also in Part2
        /// and Part3. In that case, the certificate binding is interrupted and the PICC error will be echoed
        /// with Resp.ISO90DF
        ///
        /// Note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_BindCertificate_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///       is returned from Library. Otherwise error will be returned by SAM for any subsequent
        ///       commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to
        ///       call <see cref="Cmd_SAM_BindCertificate_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Protection mode for ManageKeyPair command towards MIFARE DUOX.
        ///                         - <see cref="CommMode.PLAIN"/>
        ///                         - <see cref="CommMode.MAC"/>
        ///                         - <see cref="CommMode.FULL"/>
        /// </param>
        /// <param name="aMKPParams">MIFARE DUOX ManageKeyPair Parameters. One of the following
        ///                             - KeyNo: Key number of the key to be managed on MIFARE DUOX
        ///                             - Option: Targeted action: Generate Key Pair
        ///                             - CurveID: Targeted curve on MIFARE DUOX
        ///                             - KeyPolicy: Defines allowed crypto operations on MIFARE DUOX
        ///                             - WriteAccess: Access right and CommMode for further updates on
        ///                               MIFARE DUOX
        ///                             - KUCLimit: Defines the key usage limit of the targeted key on
        ///                               MIFARE DUOX
        /// </param>
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be used for
        ///                              certificate signing in Part 2. Will be 0x00 - 0x0F
        /// </param>
        /// <param name="aSamResponse">MKPMAC: 8 bytes as MACt(KsesAuthMAC, Cmd || MKPParams)</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_BindCertificate_Part1 ( byte bOption, byte[] aMKPParams, byte bECCKeyNo_Priv,
            out byte[] aSamResponse )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_BindCertificate_Part1 ( ref m_DataParamsInt[0], bOption, aMKPParams,
                ( byte) ( ( aMKPParams == null ) ? 0 : aMKPParams.Length ), bECCKeyNo_Priv, ref pSamResponse,
                ref wSamRespLen );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen );

            return oStatus;
        }

        /// <summary>
        /// The SAM processed the ManageKeyPair response coming from MIFARE DUOX, extracts
        /// the public key, injects in the to-be-signed certificate.Then it signs the certificate
        /// and returns the signed certificate.
        /// </summary>
        ///
        /// <param name="bOption">Buffering Options:
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: To buffer
        ///                           the initial command information.Use this flag
        ///                           to exchange MIFARE DUOX ManageKeyPair Parameters,
        ///                           Certificate to be signed and start reception of
        ///                           response.
        ///
        ///                         - <see cref="ExchangeOptions.RXCHAINING"/> The LFI
        ///                           is set to AFh for reception of remaining data from
        ///                           SAM
        /// </param>
        /// <param name="aData">Information to be exchanged to SAM.
        ///                         - In case of no PICC error, MIFARE DUOX ManageKeyPair Parameters
        ///                             - Comm.MAC : RC || PublicKey || MACt (KSesAuthMAC, RC ||
        ///                               CmdCtr || TI || PublicKey)
        ///                             - Comm.FULL: RC || E ( KSesAuthENC, PublicKey) ||
        ///                               MACt ( RC || CmdCtr || TI || E (KSesAuthENC, PublicKey))
        ///                             - Certificate to be signed
        ///                             - In case of PICC error: 2 byte PICC error
        /// </param>
        /// <param name="aResponse">Buffer containing the information received from SAM.
        ///                             - Certificate signed by SAM
        ///                             - Status code of PICC
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_BindCertificate_Part2 ( byte bOption, byte[] aData, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_BindCertificate_Part2 ( ref m_DataParamsInt[0], bOption, aData,
                ( ushort) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// The SAM_ImportEccKeyDUOX command supports the procedure to import a private ECC key
        /// stored on the SAM into the PICC in S-mode. The command generates and returns the cryptogram
        /// to be sent to the PICC. For the ManageKeyPair response received from the PICC, regular
        /// secure messaging processing with SAM_Remove_SM can be applied
        /// </summary>
        ///
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be imported in
        ///                              MIFARE DUOX.Will be 0x00 - 0x0F
        /// </param>
        /// <param name="aMKPParams">MIFARE DUOX ManageKeyPair Parameters. One of the following
        ///                             - KeyNo: Key number of the key to be managed on MIFARE DUOX
        ///                             - Option: Targeted action: Import Key Pair
        ///                             - CurveID: Targeted curve on MIFARE DUOX
        ///                             - KeyPolicy: Defines allowed crypto operations on MIFARE DUOX
        ///                             - WriteAccess: Access right and CommMode for further updates on
        ///                               MIFARE DUOX
        ///                             - KUCLimit: Defines the key usage limit of the targeted key on
        ///                               MIFARE DUOX
        /// </param>
        /// <param name="aMKPCrypto">Cryptogram holding key data: E (KSesAuthENC, Private
        ///                          Key) || MACt ( Cmd || CmdCtr || TI || MKPParams ||
        ///                          E (KSesAuthENC, PrivateKey))
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ImportEccKeyDUOX ( byte bECCKeyNo_Priv, byte[] aMKPParams, out byte[] aMKPCrypto )
        {
            IntPtr pMKPCrypto = IntPtr.Zero;
            ushort wMKPCryptoLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ImportEccKeyDUOX ( ref m_DataParamsInt[0], bECCKeyNo_Priv,
                aMKPParams, ( byte ) ( ( aMKPParams == null ) ? 0 : aMKPParams.Length ), ref pMKPCrypto,
                ref wMKPCryptoLen );
            aMKPCrypto = MarshalCopy ( oStatus, pMKPCrypto, wMKPCryptoLen );

            return oStatus;
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// The DUOX_MutualAuthEcc supports the ECC-based Mutual and Reader-Unilateral Authentication
        /// in X-Mode. The command supports both the flows triggered by ISOGeneralAuthentication and
        /// ISOSelectFile as implemented by MIFARE DUOX.
        /// </summary>
        ///
        /// <param name="bOption">Flow Option to update the P1 information and receive remaining data
        ///                       from SAM.
        ///                         - <see cref="DUOX_FlowOption.ISO_GENERAL_AUTH"/>
        ///                           ISOGeneralAuthenticate + ISOGeneralAuthenticate Part2
        ///
        ///                         - <see cref="DUOX_FlowOption.ISO_GENERAL_AUTH_FINAL"/>
        ///                           ISOSelectFile + ISOGeneralAuthenticateFinal
        ///
        ///                         - Use <see cref="ExchangeOptions.RXCHAINING"/> Receive Remaining Data
        ///                           from SAM. Should not be merged with above option
        /// </param>
        /// <param name="aOptsA">PCD Option (TLV). One of the following
        ///                         - Tag            : 0x80
        ///                         - Length         : Length of Value field.Length is either 0x02 or 0x04
        ///                         - Value          : One of the following,
        ///                             - Auth Method
        ///                                 - 0x80: Asymmetric mutual authentication with Cert.A
        ///                                 - 0xA0: Asymmetric mutual authentication without Cert.A
        ///                                 - 0x40: Asymmetric reader-unilateral authentication with Cert.A
        ///                                 - 0x60: Asymmetric reader-unilateral authentication without Cert.A
        ///                             - ProtocolVersion: One of the following
        ///                                 - 0x00 : Protocol version as supported by SAM
        ///                                 - Other: Other protocol version (sent to PICC but ignored by SAM)
        ///                             - CertFileNo: Certificate File Number. Present only if AuthMethod has
        ///                               0x80 or 0xA0 as values.
        ///                             - PrivKeyNo: Private Key Number. Present only if AuthMethod has 0x80
        ///                               or 0xA0 as values.
        /// </param>
        /// <param name="bDUOX_P2">Mapped to ISOGeneralAuthenticate or ISOGeneral AuthenticateFinal P2.
        ///                        Refer DataSheet for more information
        /// </param>
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be used for signing.
        ///                              Will be 0x00 - 0x0F
        /// </param>
        /// <param name="bCertA_FileNo">The file number of the file holding the Cert.A certificate
        ///                             (chain). Present in command frame only if \b pOptsA has AuthOptions
        ///                             as 0x80 or 0x40
        /// </param>
        /// <param name="bCertB_Options">Cert.B processing options. One of the following
        ///                                 - <see cref="DUOX_CertBOption.MSB_B_PLAIN"/>
        ///                                   No Msg.B processing by SAM. Return full Msg.B.Plain
        ///                                 - <see cref="DUOX_CertBOption.NO_DATA"/>
        ///                                   Sig.B and (if present) Cert.B certificate (chain) validated
        ///                                   by SAM but there is no response from SAM.
        ///                                 - <see cref="DUOX_CertBOption.TBS_CERTB"/>
        ///                                   Sig.B and (if present) Cert.B certificate (chain) validated
        ///                                   by SAM and there no response from SAM.
        /// </param>
        /// <param name="bECCKeyNo_CA">ECC key entry holding the public key of the targeted CA Root Key
        ///                            for Cert.B validation. Will be 0x00 - 0x0F. Present in command
        ///                            frame only if \b bOption has one of the following,
        ///                             - <see cref="DUOX_AuthMethod.MUTUAL_AUTH_CERT"/>
        ///                             - <see cref="DUOX_AuthMethod.READER_UNILATERAL_AUTH_CERT"/>
        /// </param>
        /// <param name="aSamResponse">Buffer containing the information received from SAM.
        ///                             - If AuthMethod = 0x80, 0xA0
        ///                                 - CertB_Options = <see cref="DUOX_CertBOption.MSB_B_PLAIN"/>
        ///                                   No Msg.B processing by SAM". Return full Msg.B.Plain, Mutual
        ///                                   authentication. OptB || Cert.B || Sig.B
        ///                                 - CertB_Options = <see cref="DUOX_CertBOption.TBS_CERTB"/>
        ///                                   Sig.B and (if present) Cert.B certificate (chain) validated by
        ///                                   SAM and there no response from SAM, Mutual authentication - SAM
        ///                                   validation. OptB || TBSCert.B.parent || TBSCert.B.leaf
        ///                                 - CertB_Options = <see cref="DUOX_CertBOption.NO_DATA"/>
        ///                                   Sig.B and (if present) Cert.B certificate (chain) validated by
        ///                                   SAM but there is no response from SAM, Mutual authentication -
        ///                                   SAM validation - no data returned. OptB
        ///                                 - If AuthMethod = 0x80, 0xA0, Reader-unilateral authentication.
        ///                                   OptB
        /// </param>
        /// <param name="wPiccReturnCode">Status code from PICC, One of the following
        ///                                 - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte)
        ///                                 - No response data, indicating unexpected length returned by PICC
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DUOX_MutualAuthEcc ( byte bOption, byte[] aOptsA, byte bDUOX_P2, byte bECCKeyNo_Priv, byte bCertA_FileNo,
            byte bCertB_Options, byte bECCKeyNo_CA, out byte[] aSamResponse, out ushort wPiccReturnCode )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            byte[] aPiccErrCode = new byte[2];

            wPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_DUOX_MutualAuthEcc ( ref m_DataParamsInt[0], bOption, aOptsA,
                ( byte ) ( ( aOptsA == null ) ? 0 : aOptsA.Length ), bECCKeyNo_Priv, bDUOX_P2, bCertA_FileNo,
                bCertB_Options, bECCKeyNo_CA, ref pSamResponse, ref wSamRespLen , aPiccErrCode );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen, aPiccErrCode, out wPiccReturnCode );

            return oStatus;
        }

        /// <summary>
        /// The DUOX_MutualAuthEcc supports in X-mode the ECC-based Card-Unilateral Authentication
        /// as implemented by MIFARE DUOX with the ISOInternalAuthenticate or VDE_ECDSASign command.The
        /// actual protocol choice is provided to the SAM via the P1 parameter.
        /// </summary>
        ///
        /// <param name="bOption">Protocol Option to update the P1 information.
        ///                         - <see cref="DUOX_ProtocolOption.ISO_INTERNAL_AUTH"/>
        ///                         - <see cref="DUOX_ProtocolOption.VDE_ECDSA_SIGN"/>
        /// </param>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be used for
        ///                          signature verification.
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bECC_CurveNo">The curve reference number of the ECC curve entry to
        ///                            be used for signature verification.Will be 0x00 - 0x03
        /// </param>
        /// <param name="wPiccReturnCode">Status code from PICC, One of the following
        ///                                 - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte)
        ///                                 - No response data, indicating unexpected length returned by PICC
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DUOX_UnilatAuthEcc ( byte bOption, byte bECC_KeyNo, byte bECC_CurveNo, out ushort wPiccReturnCode )
        {
            byte[] aPICCErrCode = new byte[2];

            wPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_DUOX_UnilatAuthEcc ( ref m_DataParamsInt[0], bOption, bECC_KeyNo,
                bECC_CurveNo, aPICCErrCode );

            wPiccReturnCode = BitConverter.ToUInt16 ( aPICCErrCode, 0 );

            return oStatus;
        }

        /// <summary>
        /// The DUOX_BindCertificate supports in X-mode a key pair generation on MIFARE DUOX with
        /// ManageKeyPair command, followed by a certificate signing by the SAM
        /// </summary>
        ///
        /// <param name="bOption">Configuration for ManageKeyPair command towards MIFARE DUOX.
        ///                         - ISO mode selection
        ///                             - <see cref="ISOMode.NATIVE"/>
        ///                               Command will be sent to PICC using native command set
        ///                             - <see cref="ISOMode.ISO7816"/>
        ///                               Command will be sent to PICC using ISO/IEC 7816-4APDU
        ///
        ///                         - Communication mode. To be ORed with above options.
        ///                             - <see cref="CommMode.MAC"/>
        ///                             - <see cref="CommMode.PLAIN"/>
        ///
        ///                             - <see cref="ExchangeOptions.RXCHAINING"/>
        ///                               Receive Remaining Data from SAM. Should not be merged with
        ///                               above option
        /// </param>
        /// <param name="aMKPParams">MIFARE DUOX ManageKeyPair Parameters. One of the following
        ///                             - KeyNo: Key number of the key to be managed on MIFARE DUOX
        ///                             - Option: Targeted action: Generate Key Pair
        ///                             - CurveID: Targeted curve on MIFARE DUOX
        ///                             - KeyPolicy: Defines allowed crypto operations on MIFARE DUOX
        ///                             - WriteAccess: Access right and CommMode for further updates on
        ///                               MIFARE DUOX
        ///                             - KUCLimit: Defines the key usage limit of the targeted key on
        ///                               MIFARE DUOX
        /// </param>
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be used for
        ///                              certificate signing in Part 2. Will be 0x00 - 0x0F
        /// </param>
        /// <param name="aTBSCertificate">Certificate to be signed </param>
        /// <param name="aSamResponse">Certificate signed by SAM.</param>
        /// <param name="wPiccReturnCode">Status code from PICC, One of the following
        ///                                 - Native return code (1 byte) if \b bOption has <see cref="ISOMode.NATIVE"/>
        ///                                 - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte) if <see cref="ISOMode.ISO7816"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with Chaining response.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DUOX_BindCertificate ( byte bOption, byte[] aMKPParams, byte bECCKeyNo_Priv,
            byte[] aTBSCertificate, out byte[] aSamResponse, out ushort wPiccReturnCode )
        {
            IntPtr pSamResponse = IntPtr.Zero;
            ushort wSamRespLen = 0;

            byte[] aPiccErrCode = new byte[2];

            wPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_DUOX_BindCertificate ( ref m_DataParamsInt[0], bOption, aMKPParams,
                ( byte ) ( ( aMKPParams == null ) ? 0 : aMKPParams.Length ), bECCKeyNo_Priv, aTBSCertificate,
                ( ushort ) ( ( aTBSCertificate == null ) ? 0 : aTBSCertificate.Length ), ref pSamResponse,
                ref wSamRespLen , aPiccErrCode );
            aSamResponse = MarshalCopy ( oStatus, pSamResponse, wSamRespLen, aPiccErrCode, out wPiccReturnCode );

            return oStatus;
        }

        /// <summary>
        /// The SAM_ImportEccKeyDUOX command supports the procedure to import a private ECC key stored
        /// on the SAM into the PICC in S-mode. The command generates and returns the cryptogram to be
        /// sent to the PICC. For the ManageKeyPair response received from the PICC, regular secure
        /// messaging processing with SAM_Remove_SM can be applied
        /// </summary>
        ///
        /// <param name="bOption">Configuration for ManageKeyPair command towards MIFARE DUOX.ISO mode
        ///                       selection,
        ///                         - <see cref="ISOMode.NATIVE"/>
        ///                           Command will be sent to PICC using native command set
        ///                         - <see cref="ISOMode.ISO7816"/>
        ///                           Command will be sent to PICC using ISO/IEC 7816-4APDU
        /// </param>
        /// <param name="aMKPParams">MIFARE DUOX ManageKeyPair Parameters. One of the following
        ///                             - KeyNo: Key number of the key to be managed on MIFARE DUOX
        ///                             - Option: Targeted action: Generate Key Pair
        ///                             - CurveID: Targeted curve on MIFARE DUOX
        ///                             - KeyPolicy: Defines allowed crypto operations on MIFARE DUOX
        ///                             - WriteAccess: Access right and CommMode for further updates on
        ///                               MIFARE DUOX
        ///                             - KUCLimit: Defines the key usage limit of the targeted key on
        ///                               MIFARE DUOX
        /// </param>
        /// <param name="bECCKeyNo_Priv">ECC key entry holding the private key to be imported in
        ///                              MIFARE DUOX.Will be of 0x00 - 0x0F
        /// </param>
        /// <param name="wPiccReturnCode">Status code from PICC, One of the following
        ///                                 - Native return code (1 byte) if \b bOption has <see cref="ISOMode.NATIVE"/>
        ///                                 - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte) if <see cref="ISOMode.ISO7816"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_DUOX_ImportEccKey ( byte bOption, byte[] aMKPParams, byte bECCKeyNo_Priv,
            out ushort wPiccReturnCode )
        {
            byte[] aPICCErrCode = new byte[2];

            wPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_DUOX_ImportEccKey ( ref m_DataParamsInt[0], bOption, aMKPParams,
                ( byte ) ( ( aMKPParams == null ) ? 0 : aMKPParams.Length ), bECCKeyNo_Priv, aPICCErrCode );

            wPiccReturnCode = BitConverter.ToUInt16 ( aPICCErrCode, 0 );

            return oStatus;
        }
        #endregion
        #endregion

        #region MIFARE Plus
        #region S_Mode
        /// <summary>
        /// <para>Perform a MFP Authenticate command part1. This command will generate a
        /// 16 byte random number with the one received from PICC and return an 32 byte
        /// encrypted data which will be sent to PICC.
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_AuthenticateMFP_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///       <see cref="Cmd_SAM_AuthenticateMFP_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification & authentication mode selection. Option to update
        ///                       the P1 information. The options can be combined by bitwise OR. Also
        ///                       the byte can also be framed by using a helper class
        ///                       <seealso cref="Plus_AuthenticateOption"/>
        ///                         - SL and Key derivation info
        ///                             - <see cref="MFP_Authenticate.SL1_NO_KDF"/>:
        ///                               No Key Derivation information (SL1,
        ///                               originality keys...)
        ///                             - <see cref="MFP_Authenticate.SL3_KDF"/>:
        ///                               Security Level 3 Key Derivation information
        ///
        ///                         - Select Authentication Mode
        ///                             - <see cref="MFP_Authenticate.FIRST_AUTH"/>:
        ///                               Authenticate First
        ///                             - <see cref="MFP_Authenticate.NONFIRST_AUTH"/>:
        ///                               Authenticate NonFirst
        ///
        ///                         - Use Key Diversification
        ///                             - <see cref="MFP_Authenticate.DIVERSIFY_OFF"/>:
        ///                               No diversification
        ///                             - <see cref="MFP_Authenticate.DIVERSIFY_ON"/>:
        ///                               Diversify the used key with the given DivInput
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aPDChal">Buffer containing the challenge generated by the PICC.
        ///                       E ( Kx, RndB) (16 bytes).
        /// </param>
        /// <param name="bPDChalLen">Length of bytes available in \b aPDChal buffer.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If any of diversification option is set in
        ///                               \b bOption parameter, then
        ///                                 - 8 (if AV1 key diversification with DES)
        ///                                 - 16 (if AV1 key diversification with AES) bytes
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                 diversification input
        ///                             - NULL otherwise
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be
        ///                            sent to PICC. Challenge response as E ( Kx, RndA || RndB')
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPDChal and \b aDivInput are NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateMFP_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPDChal,
            byte bPDChalLen, byte[] aDivInput, byte bDivInputLen, out byte[] aPCDChalResp )
        {
            IntPtr pPCDChalResp = IntPtr.Zero;
            ushort wPCDChalRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo,
                bKeyVer, aPDChal, bPDChalLen, aDivInput, bDivInputLen, ref pPCDChalResp, ref wPCDChalRespLen );
            aPCDChalResp = MarshalCopy ( oStatus, pPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>Perform a MFP Authenticate command part2. This command will decrypt the response
        /// received from PICC and will return the PC capabilities and PCD capabilities.
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        ///     - This interface should be called only if <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///       returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        ///     - This interface also support Cmd.SAM_AuthenticateMFPError internally. For this
        ///       \b bPiccErrCode should utilized and \b bPiccReturnCode will have the PICC error
        ///       code that is exchanged.
        /// </summary>
        /// <param name="bPiccErrCode">Error Code sent by the MIFARE Plus PICC. For success it
        ///                            should be 0x90.
        /// </param>
        /// <param name="aPDResp">Buffer containing the input received from PICC.
        ///                         - If no PICC error: PICC authentication response.
        ///                             - If <see cref="MFP_AuthMode.AUTH_FIRST"/>: Authenticate First,
        ///                               E ( Kx, TI || RndA' || adjPDcap2 || adjPCDcap2)
        ///                             - If <see cref="MFP_AuthMode.AUTH_NONFIRST"/>: Authenticate NonFirst,
        ///                               E ( Kx, RndA') (16 bytes)
        /// </param>
        /// <param name="bPDRespLen">Length of bytes available in \b aPDResp buffer.</param>
        /// <param name="aPDCap2">Buffer containing the Output PCD capabilities.
        ///                       This will be of 6 bytes.
        /// </param>
        /// <param name="aPCDCap2">Buffer containing the Output PD capabilities.
        ///                        This will be of 6 bytes.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will of 1
        ///                               byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPDResp is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateMFP_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen,
            out byte[] aPDCap2, out byte[] aPCDCap2, out byte bPiccReturnCode )
        {
            IntPtr pPDCap2 = IntPtr.Zero;
            IntPtr pPCDCap2 = IntPtr.Zero;

            bPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2 ( ref m_DataParamsInt[0],
                bPiccErrCode, aPDResp, bPDRespLen, ref pPDCap2, ref pPCDCap2, ref bPiccReturnCode );
            aPDCap2 = MarshalCopy ( oStatus, pPDCap2, 6 );
            aPCDCap2 = MarshalCopy ( oStatus, pPCDCap2, 6 );

            return oStatus;
        }

        /// <summary>
        /// <para>Perform a MFP Sector Switch Authenticate command part1. This command will
        /// generate a 16 byte random number with the one received from PICC and return an
        /// 32 byte encrypted data which will be sent to PICC.
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///       <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification selection. Option to update the
        ///                       P1 information.The options can be combined by bitwise
        ///                       OR. Also the byte can be framed by using a
        ///                       helper class <seealso cref="Plus_AuthSectorSwitchOption"/>
        ///                         - Apply sector number key diversification
        ///                             - <see cref="MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_OFF"/>:
        ///                               No diversification
        ///                             - <see cref="MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON"/>:
        ///                               Diversify the master Sector Key with the given sector number
        ///
        ///                         - Use Key Diversification for Sector Keys
        ///                             - <see cref="MFP_SectorSwitchAuth.SECTOR_DIV_OFF"/>:
        ///                               No diversification
        ///                             - <see cref="MFP_SectorSwitchAuth.SECTOR_DIV_ON"/>:
        ///                               Diversify the used keys with the given DivInput
        ///
        ///                         - Use Key Diversification for Sector Switch Key
        ///                             - <see cref="MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_OFF"/>:
        ///                               No diversification
        ///                             - <see cref="MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON"/>:
        ///                               Diversify the used key with the given DivInput
        /// </param>
        /// <param name="aPDChal">Buffer containing the challenge generated by the PICC.
        ///                       PICC challenge as E ( Kn , ..., E ( K_1 , E ( Kss , RndB))...)
        /// </param>
        /// <param name="bPDChalLen">Length of bytes available in \b aPDChal buffer.</param>
        /// <param name="wSSKeyBNr">PICC Sector Switch key block number to be used for
        ///                         authentication.
        /// </param>
        /// <param name="bSSKeyNo">Sector Switch Key number to be used from SAM (that is: Kss selection).
        ///                        One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bSSKeyVer">Sector Switch Key version to be used from SAM (that
        ///                         is: Kss selection)
        /// </param>
        /// <param name="bMSKeyNo">Master Key number to be used from SAM One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bMSKeyVer">Master Key version to be used from SAM.</param>
        /// <param name="bSectorCount">Number of sectors to be switched inside the PICC.
        ///                             - 0x01 - 0x2A: If <see cref="MFP_MKeyDiv.DIV_OFF"/>
        ///                               No sector number key diversification
        ///                             - 0x01 - 0x37: If <see cref="MFP_MKeyDiv.DIV_ON"/>
        ///                               Sector number key diversification
        /// </param>
        /// <param name="aKeyBlocks">Buffer containing the PICC KeyB block number and reference key number
        ///                          and version to be used form SAM. Should contain the following,
        ///                             - KeyBxBNr: 2 byte Key Block sector number on the PICC ( encoded MSB first)
        ///                             - KeyNo: If <see cref="MFP_MKeyDiv.DIV_OFF"/> No sector number
        ///                               key diversification", SAM Key Number of the block sector key
        ///                               ( that is: Kn selection ).
        ///                                 - NVM Key: 0x00 - 0x7F
        ///                                 - RAM Key: 0xE0 - 0xE3
        ///                             - KeyVer: If <see cref="MFP_MKeyDiv.DIV_OFF"/> No sector number
        ///                               key diversification, SAM Key Version of the block sector key
        ///                               ( that is: Kn selection ).
        ///
        ///                             - The format of the buffer should be,
        ///                               KeyBxBNr_1 || [KeyNo_1] || [KeyVer_1] || KeyBxBNr_2 || [KeyNo_2]
        ///                               || [KeyVer_2] || ... || KeyBxBNr_N || [KeyNo_N] || [KeyVer_N],
        ///                               Where N equal to the \b bSectorCount information.
        /// </param>
        /// <param name="bKeyBlocksLen">Length of bytes available in \b aKeyBlocks buffer. It should
        ///                             be equal to ( KeyBxBNr + [KeyNo] + [KeyVer]) * \b bSectorCount
        /// </param>
        /// <param name="aDivInput">Buffer containing the diversification inputs to be used for diversifying the key.
        ///                             - SSKeyDivLen: Optional, present if <see cref="MFP_SSKeyDiv.DIV_ON"/>
        ///                               Sector Switch Diversification is enabled", Length of the Sector Switch
        ///                               Key DivInput (and of Sector Keys DivInput if using same)
        ///                             - SSKeyDivInput: Optional, present if <see cref="MFP_SSKeyDiv.DIV_ON"/>
        ///                               Sector Switch Diversification is enabled", 1 to 31 bytes diversification
        ///                               input for Sector Switch Key ( and for Sector Keys if using same)
        ///                             - KeyBxDivLen: Optional, present if <see cref="MFP_SKeyDiv.DIV_ON"/>
        ///                               Sector Diversification is enabled". One of the following,
        ///                                 - Using same DivInput for Sector Switch Key and Sector Keys: 0x00
        ///                                 - Using different DivInput for Sector keys: 0x01 - 0x1F
        ///                             - KeyBxDivInput: Optional, present if <see cref="MFP_SKeyDiv.DIV_ON"/>
        ///                               Sector Diversification is enabled and KeyBxDivLen != 0x00.
        ///                               1 to 31 bytes diversification input for Sector Keys
        ///
        ///                             - The format of the buffer should be,
        ///                               [[SSKeyDivLen] || [SSKeyDivInput]] || [[KeyBxDivLen] || [KeyBxDivInput]]
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be sent to PICC.
        ///                            Challenge response as E ( K_n , ..., E ( K_1 , E ( Kss, RndA || RndB'))...)
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPDChal, \b aKeyBlocks and \b aDivInput are NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthSectorSwitchMFP_Part1 ( byte bOption, byte[] aPDChal, byte bPDChalLen,
            ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer, byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount,
            byte[] aKeyBlocks, byte bKeyBlocksLen,  byte[] aDivInput, byte bDivInputLen, out byte[] aPCDChalResp )
        {
            IntPtr pPCDChalResp = IntPtr.Zero;
            ushort wPCDChalRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 ( ref m_DataParamsInt[0], bOption,
                aPDChal, bPDChalLen, wSSKeyBNr, bSSKeyNo, bSSKeyVer, bMSKeyNo, bMSKeyVer, bSectorCount,
                aKeyBlocks, bKeyBlocksLen, aDivInput, bDivInputLen, ref pPCDChalResp,
                ref wPCDChalRespLen );
            aPCDChalResp = MarshalCopy ( oStatus, pPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>
        /// Perform a MFP Sector Switch Authenticate command part2. This command will decrypt
        /// the response received from PICC and will return success status if the challenges
        /// matches.
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by the MFP PICC. For success it
        ///                            should be 0x90.
        /// </param>
        /// <param name="aPDResp">Buffer containing the input received from PICC.
        ///                       PICC challenge response as E ( Kn, ..., E(K1,
        ///                       E(Kss, RndA'))...)
        /// </param>
        /// <param name="bPDRespLen">Length of bytes available in \b aPDResp buffer.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPDResp is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthSectorSwitchMFP_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen,
            out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2 ( ref m_DataParamsInt[0], bPiccErrCode,
                aPDResp, bPDRespLen, ref bPiccReturnCode );
        }

        /// <summary>
        /// <para>
        /// Perform a Post delivery configuration Authenticate command part1. This command will
        /// generate a 16 byte random number with the one received from PICC and return an 32
        /// byte encrypted data which will be sent to PICC.
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_AuthenticatePDC_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///       <see cref="Cmd_SAM_AuthenticatePDC_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification selection. Option to set the P1
        ///                       information byte.
        ///                         - <see cref="MFP_PDCAuth.DIVERSIFICATION_OFF"/>: No derivation
        ///                         - <see cref="MFP_PDCAuth.DERIVE_UPGRADE_KEY"/>: Derive UpgradeKey
        ///                           from targeted ICUpgradeKey given UpgradeInfo
        ///                         - <see cref="MFP_PDCAuth.DIVERSIFY_YEAR_UPGRADE_KEY"/>: Diversify
        ///                           the targeted YearUpgradeKey with the given DivInput and then
        ///                           derive the actual UpgradeKey with UpgradeInfo
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aPDChal">Buffer containing the challenge generated by the PICC.
        ///                       Authentication message from the PICC as E ( DivKey (Kx), RndB)
        /// </param>
        /// <param name="bPDchalLen">Length of bytes available in \b aPDChal buffer.</param>
        /// <param name="aUpgradeInfo">Upgrade information of the target product state.</param>
        /// <param name="bLen">Length of bytes available in \b pUpgradeInfo buffer.
        ///                     - 0x00: No key derivation
        ///                     - 0x01 - 0x08: Key derivation involving UpgradeInfo
        /// </param>
        /// <param name="aDivInput">Diversification input for generating KeyID.ICUpgrade
        ///                         Key , ie.: VCUID.
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be sent to
        ///                            PICC. Challenge response as E ( DivKey (Kx), RndA || RndB')
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPDChal, \b aUpgradeInfo and \b aDivInput are NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticatePDC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPDChal,
            byte bPDchalLen, byte[] aUpgradeInfo, byte bLen, byte[] aDivInput, byte bDivInputLen,
            out byte[] aPCDChalResp )
        {
            IntPtr pPCDChalResp = IntPtr.Zero;
            ushort wPCDChalRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1 ( ref m_DataParamsInt[0], bOption,
                bKeyNo, bKeyVer, aPDChal, bPDchalLen, aUpgradeInfo, bLen, aDivInput, bDivInputLen,
                ref pPCDChalResp, ref wPCDChalRespLen );
            aPCDChalResp = MarshalCopy ( oStatus, pPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        /// <para>
        /// Perform a Post delivery configuration Authenticate command part2. This command
        /// will decrypt the response received from PICC and will return success status if the
        /// challenges matches.
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_AuthenticatePDC_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by the MIFARE Plus PICC.
        ///                            For success it should be 0x90.
        /// </param>
        /// <param name="aPDResp">Buffer containing the input received from PICC.
        ///                       PICC authentication response as E ( DivKey (Kx), RndA')
        /// </param>
        /// <param name="bPDRespLen">Length of bytes available in \b aPDResp buffer.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        /// of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aPDResp is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticatePDC_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen,
            out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aPDResp,
                bPDRespLen, ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Read command. This command is used to perform GetVersion,
        /// ReadSignature and all Read related operations.
        /// </summary>
        ///
        /// <param name="bLFI">Last Frame Indication (LFI). Option for updating the
        ///                    P2 information of SAM frame.
        ///                     - <see cref="LFI.LAST_FRAME"/>
        ///                     - <see cref="LFI.CHAINED_FRAME"/>
        /// </param>
        /// <param name="wOption">Buffering options and Payload Type, for exchanging information
        ///                       to SAM. The options can be combined by bitwise OR.  Also
        ///                       the byte can also be framed by using a helper class
        ///                       <seealso cref="Plus_CombinedReadOption"/>
        ///                         - Buffering Options
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                               Buffer first set of information. No exchange is
        ///                               performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                               Buffer intermediate set of information. No exchange
        ///                               is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                               Buffer last set of information. Exchange information
        ///                               to SAM and Receive the response
        ///
        ///                         - The below flags should be used to switch between command and response.
        ///                             - <see cref="MFP_PayloadType.COMMAND"/>: MFP Command
        ///                             - <see cref="MFP_PayloadType.RESPONSE"/>: MFP Response
        ///                             - <see cref="MFP_PayloadType.BOTH"/>: MFP Command + Response
        ///                               (only for unMAC'ed commands)
        /// </param>
        /// <param name="aData">The data to be sent to SAM. Should be one of the following.
        ///                         - For <see cref="MFP_PayloadType.COMMAND"/>:
        ///                             - For READxyz  : Read Cmd (1byte) || BNR (2byte) || No.Of Blocks (1byte)
        ///                             - For GetV     : GetVersion command (1byte)
        ///                             - For Read_Sign: Read_Sign Cmd (1byte) || Address (1btye)
        ///
        ///                         - For <see cref="MFP_PayloadType.RESPONSE"/>:
        ///                             - 1 - 256 bytes received response from PICC ( Maced / Encrypted data )
        ///                             - Error Code
        ///
        ///                         - For <see cref="MFP_PayloadType.BOTH"/>: MFP command + response
        ///                             - READxyU: Read Cmd (1byte) || BNR (2byte) || No.Of Blocks (1byte) ||
        ///                               RC || Data ( N bytes ) [|| MAC ( 8 bytes )]
        ///                             - RC: Response Code
        /// </param>
        /// <param name="bDataLen">Length of bytes available in \b aData buffer.</param>
        /// <param name="aOutput">The complete information received from SAM. This will be information that
        ///                       will be exchanged with PICC. One of the following.
        ///                         - If <see cref="MFP_PayloadType.COMMAND"/>: MAC of command
        ///                         - If not <see cref="MFP_PayloadType.COMMAND"/>:
        ///                           [Optional, present if READEyz] Decrypted data
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aData is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CombinedReadMFP ( byte bLFI, ushort wOption, byte[] aData, byte bDataLen,
            out byte[] aOutput, out byte bPiccReturnCode )
        {
            ushort wOutputLen = 0;
            IntPtr pOutput = IntPtr.Zero;

            bPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_CombinedReadMFP ( ref m_DataParamsInt[0], bLFI, wOption,
                aData, bDataLen, ref pOutput, ref wOutputLen, ref bPiccReturnCode );
            aOutput = MarshalCopy ( oStatus, pOutput, wOutputLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a MFP CombinedWrite command. This command is common for Write, WriteValue,
        /// Increment, Decrement, IncrementTransfer, DecrementTransfer, Transfer and Restore.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options and Payload Type, for exchanging information
        ///                       to SAM.The options can be combined by bitwise OR. Also
        ///                       the byte can also be framed by using a helper class
        ///                       <seealso cref="Plus_CombinedWriteOption"/>
        ///                         - Buffering Options
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Exchange information to SAM and Receive the response
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                               Buffer first" set of information.No exchange is performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                               Buffer intermediate" set of information.No exchange is
        ///                               performed
        ///
        ///                             - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                               Buffer last" set of information.Exchange information
        ///                               to SAM and Receive the response
        ///
        ///                             - The below flags should be used to switch between command
        ///                               and response.
        ///                                 - <see cref="MFP_PayloadType.COMMAND"/>
        ///                                 - <see cref="MFP_PayloadType.RESPONSE"/>
        ///
        ///                             - The below flags should be used to update the Plain data
        ///                               in response.
        ///                                 - <see cref="MFP_CWMode.RETURN_PLAIN"/>
        ///                                     - If <see cref="MFP_PayloadType.RESPONSE"/> or
        ///                                       <see cref="MFP_PayloadType.COMMAND"/> with CmdCode
        ///                                       different from Cmd.WRITEPy: RFU
        ///                                     - <see cref="MFP_PayloadType.COMMAND"/> with CmdCode
        ///                                       set to Cmd.WRITEPy: Return plain data in response
        ///                                 - <see cref="MFP_CWMode.SKIP_PLAIN"/>:
        ///                                   If <see cref="MFP_PayloadType.COMMAND"/> with CmdCode
        ///                                   set to Cmd.WRITEPy
        /// </param>
        /// <param name="aData">The data to be sent to SAM. Should be one of the following,
        ///                         - For <see cref="MFP_PayloadType.COMMAND"/>:
        ///                             - For WRITExy : Write Cmd (1byte) || BNR (2byte) || PlainData ( N* 16 bytes)
        ///                             - For INCx    : Inc Cmd (1byte) || BNR (2byte) || PlainData (4bytes)
        ///                             - For DECx    : Dec Cmd (1byte) || BNR (2byte) || PlainData (4bytes)
        ///                             - For INCTx   : Inc Transfer Cmd (1byte) || BNR (4byte) || PlainData (4bytes)
        ///                             - For DECTx   : Dec Transfer Cmd (1byte) || BNR (4byte) || PlainData (4bytes)
        ///                             - For TRFx    : Transfer Cmd (1byte) || BNR (2byte)
        ///                             - For RESx    : Restore Cmd (1byte) || BNR (2byte)
        ///
        ///                         - For <see cref="MFP_PayloadType.RESPONSE"/>:
        ///                             - RC (1byte)
        ///                             - RC (1byte) || MAC (8byte)
        ///                             - RC (1byte) || TMC (4byte) || TMV (8byte)
        ///                             - RC (1byte) || TMC (4byte) || TMV (8byte) || MAC (8byte)
        /// </param>
        /// <param name="bDataLen">Length of bytes available in \b aData buffer.</param>
        /// <param name="aOutput">The complete information received from SAM. This
        ///                       will be information that will be exchanged with
        ///                       PICC. One of the following.
        ///                         - <see cref="MFP_PayloadType.COMMAND"/>
        ///                             - MAC: 8 bytes
        ///                             - Data:
        ///                                 - EncValue (16 byte) || MAC (8 byte)
        ///                                 - EncData ( N* 16 + 8 byte (with N = 1..15)) || MAC (8 byte
        ///                         - NULL for <see cref="MFP_PayloadType.RESPONSE"/>
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aData is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CombinedWriteMFP ( ushort wOption, byte[] aData, byte bDataLen, out byte[] aOutput,
            out byte bPiccReturnCode )
        {
            IntPtr pOutput = IntPtr.Zero;
            ushort wOutputLen = 0;

            bPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_CombinedWriteMFP ( ref m_DataParamsInt[0], wOption, aData,
                bDataLen, ref pOutput, ref wOutputLen, ref bPiccReturnCode );
            aOutput = MarshalCopy ( oStatus, pOutput, wOutputLen );

            return oStatus;
        }

        /// <summary>
        /// Prepare the MFP Change Key command. This command will return the protected data
        /// to be written to PICC.
        /// </summary>
        ///
        /// <param name="bOption">The below flags should be for updating the P1 information byte.
        ///                       Also the byte can be framed by using a helper class
        ///                       <seealso cref="Plus_ChangeKeyOption"/>
        ///                         - Use Key Diversification
        ///                             - <see cref="MFP_ChangeKeyDiv.DIVERSIFY_OFF"/>: No diversification
        ///                             - <see cref="MFP_ChangeKeyDiv.DIVERSIFY_ON"/>:
        ///                               Diversify the used key with the given DivInput
        ///
        ///                         - Payload type
        ///                             - <see cref="MFP_PayloadType.COMMAND"/>: MFP command
        ///                             - <see cref="MFP_PayloadType.RESPONSE"/>: MFP response
        /// </param>
        /// <param name="aData">The information to be exchanged to SAM.
        ///                         - For <see cref="MFP_PayloadType.COMMAND"/>:
        ///                           MIFARE Plus command data. The buffer should contain,
        ///                             - CmdCode (1 byte): MIFARE Plus command code.Cmd.WRITEEy (0xA0, 0xA1)
        ///                             - KeyBNo (2 byte): MIFARE Plus Key Block number
        ///                                 - 0x4000 ... 0x4FFF, 0x8000 ... 0xAFFF, 0xCXX0, 0xCXX1
        ///                                 - With X any value in 0x0..0xF. Therefore the latter two
        ///                                   values cover: 0xC000, 0xC001, 0xC010, 0xC011, 0xC020, 0xC021,
        ///                                   ... till 0xCFF0, 0xCFF1
        ///                             - KeyNo (1 byte): Key number to be used from SAM. One of the following,
        ///                                 - NVM Key: 0x00 - 0x7F
        ///                                 - RAM Key: 0xE0 - 0xE3
        ///                             - KeyVer (1 byte): Key version to be used from SAM.
        ///                             - DivInput (1 - 31 bytes): Diversification Input.Present if \b bOption
        ///                               has <see cref="MFP_ChangeKeyDiv.DIVERSIFY_ON"/>: Diversification Enabled.
        ///
        ///                             - For <see cref="MFP_PayloadType.RESPONSE"/>:
        ///                               MIFARE Plus response data. The buffer should contain,
        ///                                 - PICCReturnCode (1 byte): PICC response code
        ///                                 - MAC (8 byte): PICC MAC response
        /// </param>
        /// <param name="bDataLen">Length of bytes available in \b aData buffer.</param>
        /// <param name="aProtectedData">The complete information received from SAM. This will be
        ///                              information that will be exchanged with PICC. One of the
        ///                              following.
        ///                                 - <see cref="MFP_PayloadType.COMMAND"/>
        ///                                     - Encrypted and MACed data
        ///                                         - EncValue (16 byte) || MAC (8 byte)
        ///                                         - EncData ( N* 16 + 8 byte (with N = 1..15)) || MAC (8 byte)
        ///                                 - NULL for <see cref="MFP_PayloadType.RESPONSE"/>
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> \b aData is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ChangeKeyMFP ( byte bOption, byte[] aData, byte bDataLen, out byte[] aProtectedData,
            out byte bPiccReturnCode )
        {
            IntPtr pProtectedData = IntPtr.Zero;
            ushort wProtDataLen = 0;

            bPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ChangeKeyMFP ( ref m_DataParamsInt[0], bOption, aData, bDataLen,
                ref pProtectedData, ref wProtDataLen, ref bPiccReturnCode );
            aProtectedData = MarshalCopy ( oStatus, pProtectedData, wProtDataLen );

            return oStatus;
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// The MFP_Authenticate supports the PICC authentication in X-mode. It performs an
        /// authentication with the PICC and upon success generates the session keys.
        /// </summary>
        ///
        /// <param name="bOption">Key diversification & authentication mode selection. Option
        ///                       to update the P1 information.The options can be combined
        ///                       by bitwise OR. Also the byte can be framed by
        ///                       using a helper class <seealso cref="Plus_AuthenticateOption"/>
        ///                       - SL and Key derivation info
        ///                         - <see cref="MFP_AuthSLKDF.SL1_NO_KDF"/>:
        ///                           No Key Derivation information (SL1, originality keys...)
        ///                         - <see cref="MFP_AuthSLKDF.SL3_KDF"/>:
        ///                           Security Level 3 Key Derivation information
        ///
        ///                     - Select Authentication Mode
        ///                         - <see cref="MFP_AuthMode.AUTH_FIRST"/>:
        ///                           Authenticate First
        ///                         - <see cref="MFP_AuthMode.AUTH_NONFIRST"/>:
        ///                           Authenticate NonFirst
        ///
        ///                     - Use Key Diversification
        ///                         - <see cref="MFP_AuthDiv.DIVERSIFY_OFF"/>:
        ///                           No diversification
        ///                         - <see cref="MFP_AuthDiv.DIVERSIFY_ON"/>:
        ///                           Diversify the used key with the given DivInput
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="wBlockNo">MIFARE Plus block number of key to be used for authentication.</param>
        /// <param name="aPcdCapsIn">Buffer containing the Input PCD Capabilities.
        ///                          One of the following,
        ///                             - If <see cref="MFP_AuthMode.AUTH_FIRST"/>: 0x00..0x06
        ///                             - If <see cref="MFP_AuthMode.AUTH_NONFIRST"/>: 0x00
        /// </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.
        ///                             - If any of diversification option is set in
        ///                               \b bOption parameter, then
        ///                                 - 8 (if AV1 key diversification with DES)
        ///                                 - 16 (if AV1 key diversification with AES) bytes
        ///                                 - 1 to 31 bytes (if AV2 key diversification)
        ///                                   diversification input
        ///                             - NULL otherwise
        /// .</param>
        /// <param name="aPcdCapsOut">Buffer containing the Output PCD capabilities.
        ///                           This will be of 6 bytes.
        /// </param>
        /// <param name="aPdCaps">Buffer containing the Output PD capabilities.
        ///                       This will be of 6 bytes.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_Authenticate ( byte bOption, byte bKeyNo, byte bKeyVer, ushort wBlockNo, byte[] aPcdCapsIn,
            byte[] aDivInput, out byte[] aPcdCapsOut, out byte[] aPdCaps, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            aPcdCapsOut = new byte[6];
            aPdCaps = new byte[6];
            Status_t oStatus = phhalHw_Sam_Cmd_MFP_Authenticate ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, wBlockNo,
                aPcdCapsIn, ( byte ) ( ( aPcdCapsIn == null ) ? 0 : aPcdCapsIn.Length ), aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPcdCapsOut, aPdCaps,
                ref bPiccReturnCode );
            if ( !oStatus.Equals ( new Status_t () ) )
            {
                aPcdCapsOut = null;
                aPdCaps = null;
            }

            return oStatus;
        }

        /// <summary>
        /// The MFP_AuthSectorSwitch supports the X-mode procedure to switch the security level
        /// of MIFARE Plus sectors on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options and Key diversification selection, for
        ///                       exchanging information to SAM.The options can be combined
        ///                       by bitwise OR. Also the byte can be framed by using a
        ///                       helper class <seealso cref="Plus_AuthSectorSwitchOption"/>
        ///                         - Buffering Options:
        ///                             - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                               Framing the single command which has all the information.
        ///                             - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                               For framing ISO7816 command header, SSKeyBNr, SSKeyNo,
        ///                               SSKeyVer, MsKeyNo, MsKeyVer, SectorCount, KeyBlocks.
        ///                               Information is not exchanged to SAM.
        ///                             - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                               For buffering Intermediate / Final Key Blocks or First /
        ///                               Intermediate DivInput. Information is not exchanged to SAM.
        ///                             - <see cref="ExchangeOptions.BUFFER_LAST"/>: For buffering
        ///                               Final DivInput and exchanging the information is to SAM.
        ///                               Also receives information from SAM.
        ///
        ///                         - Key diversification selection.Option to set the P1 information
        ///                           bytes.
        ///                             - Apply sector number key diversification
        ///                                 - <see cref="MFP_MKeyDiv.DIV_OFF"/>: No diversification
        ///                                 - <see cref="MFP_MKeyDiv.DIV_ON"/>: Diversify the master
        ///                                   Sector Key with the given sector number
        ///
        ///                             - Use Key Diversification for Sector Keys
        ///                                 - <see cref="MFP_SKeyDiv.DIV_OFF"/>: No diversification
        ///                                 - <see cref="MFP_SKeyDiv.DIV_ON"/>: Diversify the used keys
        ///                                   with the given DivInput
        ///
        ///                             - Use Key Diversification for Sector Switch Key
        ///                                 - <see cref="MFP_SSKeyDiv.DIV_OFF"/>: No diversification
        ///                                 - <see cref="MFP_SSKeyDiv.DIV_ON"/>: Diversify the used
        ///                                   key with the given DivInput
        ///
        ///                         - Additional option. Will not be exchanged to SAM. It's for
        ///                           internal purpose.
        ///                             - <see cref="MFP_SSBuffer.KEY_BLOCKS"/>: Buffer the KeyBlocks
        ///                               information
        ///                             - <see cref="MFP_SSBuffer.DIV_INPUT"/>: Buffer the Diversification
        ///                               input information
        /// </param>
        /// <param name="wSSKeyBNr">PICC Sector Switch key block number to be used for
        ///                         authentication.
        /// </param>
        /// <param name="bSSKeyNo">Sector Switch Key number to be used from SAM (that is: Kss selection).
        ///                        One of the following,
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bSSKeyVer">Sector Switch Key version to be used from SAM (that
        ///                         is: Kss selection)
        /// .</param>
        /// <param name="bMSKeyNo">Master Key number to be used from SAM One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bMSKeyVer">Master Key version to be used from SAM. </param>
        /// <param name="bSectorCount">Number of sectors to be switched inside the PICC.
        ///                             - 0x01 - 0x2A: If <see cref="MFP_MKeyDiv.DIV_OFF"/>,
        ///                               No sector number key diversification
        ///                             - 0x01 - 0x37: If <see cref="MFP_MKeyDiv.DIV_ON"/>,
        ///                               Sector number key diversification
        /// </param>
        /// <param name="aKeyBlocks">Buffer containing the PICC KeyB block number and reference key number
        ///                          and version to be used form SAM. Should contain the following,
        ///                             - KeyBxBNr: 2 byte Key Block sector number on the PICC ( encoded MSB first)
        ///                             - KeyNo: If <see cref="MFP_MKeyDiv.DIV_OFF"/>, SAM Key Number of the block
        ///                               sector key ( that is: Kn selection ).
        ///                                 - NVM Key: 0x00 - 0x7F
        ///                                 - RAM Key: 0xE0 - 0xE3
        ///                             - KeyVer: If <see cref="MFP_MKeyDiv.DIV_OFF"/>, SAM Key Version of the block
        ///                               sector key ( that is: Kn selection ).
        ///
        ///                             - The format of the buffer should be,
        ///                               KeyBxBNr_1 || [KeyNo_1] || [KeyVer_1] || KeyBxBNr_2 || [KeyNo_2] ||
        ///                               [KeyVer_2] || ... || KeyBxBNr_N || [KeyNo_N] || [KeyVer_N], Where N
        ///                               equal to the \b bSectorCount information.
        /// </param>
        /// <param name="aDivInput">Buffer containing the diversification inputs to be used for diversifying the key.
        ///                             - SSKeyDivLen: Optional, present if <see cref="MFP_SSKeyDiv.DIV_ON"/>,
        ///                               Length of the Sector Switch Key DivInput (and of Sector Keys DivInput
        ///                               if using same)
        ///                             - SSKeyDivInput: Optional, present if <see cref="MFP_SSKeyDiv.DIV_ON"/>,
        ///                               1 to 31 bytes diversification input for Sector Switch Key ( and for
        ///                               Sector Keys if using same)
        ///                             - KeyBxDivLen: Optional, present if <see cref="MFP_SKeyDiv.DIV_ON"/>,
        ///                               One of the following,
        ///                                 - Using same DivInput for Sector Switch Key and Sector Keys: 0x00
        ///                                 - Using different DivInput for Sector keys: 0x01 - 0x1F
        ///                             - KeyBxDivInput: Optional, present if <see cref="MFP_SKeyDiv.DIV_ON"/>
        ///                               and KeyBxDivLen != 0x00. 1 to 31 bytes diversification input for Sector
        ///                               Keys
        ///
        ///                             - The format of the buffer should be,
        ///                               [[SSKeyDivLen] || [SSKeyDivInput]] || [[KeyBxDivLen] || [KeyBxDivInput]]
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_AuthSectorSwitch ( ushort wOption, ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer,
            byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount, byte[] aKeyBlocks, byte[] aDivInput,
            out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_MFP_AuthSectorSwitch ( ref m_DataParamsInt[0], wOption, wSSKeyBNr, bSSKeyNo, bSSKeyVer,
                bMSKeyNo, bMSKeyVer, bSectorCount, aKeyBlocks, ( byte ) ( ( aKeyBlocks == null ) ? 0 : aKeyBlocks.Length ),
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bPiccReturnCode );
        }

        /// <summary>
        /// The PDC_Authenticate supports the X-mode procedure to perform the
        /// Post-Delivery configuration on the PICC
        /// </summary>
        ///
        /// <param name="bOption">Key diversification selection. Option to set the P1
        ///                       information byte.
        ///                         - <see cref="MFP_PDCAuth.DIVERSIFICATION_OFF"/>:
        ///                           No derivation
        ///                         - <see cref="MFP_PDCAuth.DERIVE_UPGRADE_KEY"/>:
        ///                           Derive UpgradeKey from targeted ICUpgradeKey given
        ///                           UpgradeInfo
        ///                         - <see cref="MFP_PDCAuth.DIVERSIFY_YEAR_UPGRADE_KEY"/>:
        ///                           Diversify the targeted YearUpgradeKey with the given
        ///                           DivInput and then derive the actual UpgradeKey with
        ///                           UpgradeInfo
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="wUpgradeKey">PICC UpgradeKey to be used for authentication.</param>
        /// <param name="aUpgradeInfo">Upgrade information of the target product state.
        ///                             - NULL: No key derivation
        ///                             - 0x01 - 0x08: Key derivation involving UpgradeInfo
        /// </param>
        /// <param name="aDivInput">Diversification input for generating KeyID.ICUpgrade
        ///                         Key, ie.: VCUID.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_PDC_Authenticate ( byte bOption, byte bKeyNo, byte bKeyVer, ushort wUpgradeKey,
            byte[] aUpgradeInfo, byte[] aDivInput, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_PDC_Authenticate ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                wUpgradeKey, aUpgradeInfo, ( byte ) ( ( aUpgradeInfo == null ) ? 0 : aUpgradeInfo.Length ),
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Read command in X mode. This command is used to
        /// perform GetVersion, ReadSignature and all Read related operations.
        /// </summary>
        ///
        /// <param name="wOption">Option for receiving the next data chunk if
        ///                       previous status from SAM was chaining.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>: Exchange Information
        ///                           to MIFARE Plus PICC and receive response.
        ///                         - <see cref="ExchangeOptions.RXCHAINING"/> Receive remaining
        ///                           information from MIFARE Plus PICC.
        /// </param>
        /// <param name="aReadCmd">The different types of command to be sent to
        ///                        MIFARE Plus PICC and received the response.
        ///                         - GetVersion: GetVersion cmd (1byte)
        ///                         - ReadSig   : Read Signature cmd (1byte) + Address (1byte)
        ///                         - Read      : Read cmd (1byte) + BlockNr (2byte) + NoBlocks (1byte)
        /// </param>
        /// <param name="aData">The information returned by SAM for the mentioned
        ///                     command in \b aReadCmd buffer.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Successful Chaining operation.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_CombinedRead ( ushort wOption, byte[] aReadCmd, out byte[] aData, out byte bPiccReturnCode )
        {
            IntPtr pData = IntPtr.Zero;
            ushort wDataLen = 0;

            bPiccReturnCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_MFP_CombinedRead ( ref m_DataParamsInt[0], ( ushort ) wOption, aReadCmd,
                ( byte ) ( ( aReadCmd == null ) ? 0 : aReadCmd.Length ), ref pData, ref wDataLen,
                ref bPiccReturnCode );
            aData = MarshalCopy ( oStatus, pData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Write command in X mode. This command performs
        /// Write, Increment, Decrement, Transfer, Restore, IncrementTransfer and
        /// DecrementTransfer commands of the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>
        ///                           "Exchange information to SAM and Receive the response"
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///                           "Buffer first" set of information. No exchange is
        ///                           performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///                           "Buffer intermediate" set of information. No exchange
        ///                           is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>
        ///                           "Buffer last" set of information. Exchange information
        ///                           to SAM and Receive the response
        /// </param>
        /// <param name="aData">The different types of command to be sent to
        ///                     MIFARE Plus PICC and received the response,
        ///                         - For WRITExy : Write Cmd (1byte) || BNR (2byte) ||
        ///                         PlainData ( N* 16 bytes)
        ///                         - For INCx    : Inc Cmd (1byte) || BNR (2byte) ||
        ///                           PlainData (4bytes)
        ///                         - For DECx    : Dec Cmd (1byte) || BNR (2byte) ||
        ///                           PlainData (4bytes)
        ///                         - For INCTx   : Inc Transfer Cmd (1byte) || BNR (4byte)
        ///                           || PlainData (4bytes)
        ///                         - For DECTx   : Dec Transfer Cmd (1byte) || BNR (4byte)
        ///                           || PlainData (4bytes)
        ///                         - For TRFx    : Transfer Cmd (1byte) || BNR (2byte)
        ///                         - For RESx    : Restore Cmd (1byte) || BNR (2byte)
        /// </param>
        /// <param name="aTMC">Only available is the block is a TMProtected block.
        ///                    The buffer will have 4 bytes of Transaction MAC counter
        ///                    information.
        /// </param>
        /// <param name="aTMV">Only available is the block is a TMProtected block.
        ///                    The buffer will have 8 bytes of Transaction MAC value.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_CombinedWrite ( ushort wOption, byte[] aData, out byte[] aTMC, out byte[] aTMV,
            out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            aTMC = new byte[4];
            aTMV = new byte[8];
            Status_t oStatus = phhalHw_Sam_Cmd_MFP_CombinedWrite ( ref m_DataParamsInt[0], wOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), aTMC, aTMV,
                ref bPiccReturnCode );

            if ( !oStatus.Equals ( new Status_t () ) )
            {
                aTMC = null;
                aTMV = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Plus ChangeKey command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">The below flags should be for updating the P1
        ///                       information byte.
        ///                         - <see cref="MFP_ChangeKeyDiv.DIVERSIFY_OFF"/>:
        ///                           No diversification
        ///
        ///                         - <see cref="MFP_ChangeKeyDiv.DIVERSIFY_ON"/>:
        ///                           Diversify the used key with the given DivInput
        /// </param>
        /// <param name="bCmdCode">The MIFARE Plus Write (Cmd.WRITEEy) command code
        ///                        to be used for writing the key. (0xA0 or 0xA1)
        /// </param>
        /// <param name="wBlockNo">MIFARE Plus Key Block number
        ///                         - 0x4000 ... 0x4FFF, 0x8000 ... 0xAFFF, 0xCXX0, 0xCXX1
        ///                         - With X any value in 0x0..0xF. Therefore the latter two
        ///                           values cover: 0xC000, 0xC001, 0xC010, 0xC011, 0xC020, 0xC021,
        ///                           ... till 0xCFF0, 0xCFF1
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3.
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_ChangeKey ( byte bOption, byte bCmdCode, ushort wBlockNo, byte bKeyNo, byte bKeyVer,
            byte[] aDivInput, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_MFP_ChangeKey ( ref m_DataParamsInt[0], bOption, bCmdCode, wBlockNo, bKeyNo,
                bKeyVer, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                ref bPiccReturnCode );
        }

        /// <summary>
        /// MFP_WritePerso is a multi-block write command. It efficiently performs in X-mode up
        /// to 13 MIFARE Plus PICC WritePerso commands targeting one block each. If more than 13
        /// blocks are to be updated, several MFP_WritePerso are to be issued.The blocks are
        /// transmitted to the SAM in one MFP_WritePerso, then the SAM shall execute the many
        /// required PICC WritePerso commands.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/>:
        ///                           Exchange information to SAM and Receive the response
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_FIRST"/>:
        ///                           Buffer first set of information.No exchange is
        ///                           performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_CONT"/>:
        ///                           Buffer intermediate set of information.No exchange
        ///                           is performed
        ///
        ///                         - <see cref="ExchangeOptions.BUFFER_LAST"/>:
        ///                           Buffer last set of information.Exchange information
        ///                           to SAM and Receive the response
        /// </param>
        /// <param name="aBlocks">Buffer containing the Block and Data pair to be written
        ///                       to MIFAER Plus PICC by SAM hardware.Should be holding an
        ///                       array of block number and data like,
        ///                         - Block Addr: MIFARE Plus Block address (2 bytes)
        ///                         - BlockData: MIFARE Plus Block data to be written at
        ///                         the address (16 bytes)
        ///                         - Ex: BNR_1 || Data || BNR_2 || Data || BNR_3 || Data ||
        ///                         ... || BNR_N || Data BNR_x should be 2 bytes and Data
        ///                         should 16 bytes.
        /// </param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE Plus PICC. This will
        ///                               of 1 byte in length.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_MFP_WritePerso ( ushort wOption, byte[] aBlocks, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_Sam_Cmd_MFP_WritePerso ( ref m_DataParamsInt[0], wOption, aBlocks,
                ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ), ref bPiccReturnCode );
        }
        #endregion
        #endregion

        #region MIFARE Ultralight
        #region S_Mode
        /// <summary>
        /// <para>Performs PwdAuthUL command execution part 1 in S mode. The first part includes the header,
        /// Key number, Key Version, Diversification input and returns Message
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_PwdAuthUL_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///       is returned from Library. Otherwise error will be returned by SAM for any subsequent
        ///       commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to
        ///     call <see cref="Cmd_SAM_PwdAuthUL_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                             - NVM Key: 0x00 - 0x7F
        ///                             - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aDivInput">Diversification input for key diversification.
        ///                         (1 to 31 byte(s) input).
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="aPwd">Password received from SAM (4 byte).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_PwdAuthUL_Part1 ( byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen,
            out byte[] aPwd )
        {
            IntPtr pPwd = IntPtr.Zero;
            ushort wPwdLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part1 ( ref m_DataParamsInt[0], bKeyNo, bKeyVer,
                aDivInput, bDivInputLen, ref pPwd, ref wPwdLen );
            aPwd = MarshalCopy ( oStatus, pPwd, wPwdLen );

            return oStatus;
        }

        /// <summary>
        /// <para>
        /// Performs PwdAuthUL command execution part 2 in S mode. The Last part includes
        /// the header and Password authentication acknowledge
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_PwdAuthUL_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="wPack">Password Authentication Acknowledge.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_PwdAuthUL_Part2 ( ushort wPack )
        {
            return phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part2 ( ref m_DataParamsInt[0], wPack );
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// The UL_PwdAuthPICC supports the X-mode procedure to perform the Password Authentication on the
        /// MIFARE Ultralight EV1 PICC. The command generates the password and password acknowledgment bytes and
        /// performs the password authentication with the MIFARE Ultralight EV1 PICC.Finally, the command
        /// compares the generated password acknowledgment bytes with the PACK bytes returned from the MIFARE
        /// Ultralight EV1 PICC.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aDivInput">Diversification input for key diversification.
        ///                         (1 to 31 byte(s) input).
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        /// <param name="bStatusCode">Status code returned by PICC</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_UL_PwdAuthPICC ( byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen,
            out byte bStatusCode )
        {
            bStatusCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_UL_PwdAuthPICC ( ref m_DataParamsInt[0], bKeyNo, bKeyVer,
                aDivInput, bDivInputLen, ref bStatusCode );

            return oStatus;
        }

        /// <summary>
        /// Performs AuthenticatePICC command execution in X mode. Includes the header, Key number,
        /// Key Version, Diversification input based on \b bOption parameter.
        /// </summary>
        ///
        /// <param name="bOption">Options for Key diversification selection and Exchanging LE
        ///                       to SAM. The options can be combined by bitwise OR. Also
        ///                       the byte can also be framed by using a helper class
        ///                       <seealso cref="MFUL_AuthenticateOption"/>
        ///                         - Use Key Diversification.Option for P1
        ///                           information byte.
        ///                             - <see cref="MFUL_Option.DIV_ON"/>:
        ///                               Diversify the used key with the given DivInput
        ///
        ///                         - Option for exchanging LE to SAM. Used internally by
        ///                           Library.
        ///                             - <see cref="MFUL_Option.INCLUDE_LE"/>:
        ///                               LE is exchanged to SAM
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aDivInput">Diversification input for key diversification.
        ///                         (1 to 15 byte(s) input)
        /// </param>
        /// <param name="bDivInputLen">Length of bytes available in \b aDivInput buffer.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_UL_AuthenticatePICC ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput,
            byte bDivInputLen, out byte bStatusCode )
        {
            bStatusCode = 0;
            return phhalHw_Sam_Cmd_UL_AuthenticatePICC ( ref m_DataParamsInt[0], bOption,
                bKeyNo, bKeyVer, aDivInput, bDivInputLen, ref bStatusCode );
        }

        /// <summary>
        /// Perform a MIFARE Ultralight Read command in X mode.
        ///     - In X-mode, MF_Read reads blocks of a MIFARE Ultralight or MIFARE Ultralight C card in
        ///       plain and return the data.The MIFARE SAM AV4 constructs the READ command (Cmd 0x30)
        ///       and sends it to the card.
        ///     - If more than one block address is issued, the MIFARE SAM AV4 accesses the blocks in the
        ///     same order as addresses listed in the command data field. The order of the returned data
        ///     is the same as the order of addresses in the data field.
        /// </summary>
        ///
        /// <param name="aBlocks">The block numbers from where the data should be read.
        ///                       Should be holding an array of block numbers like
        ///                       BNR_1, BNR_2, BNR_3, ..., BNR_N</param>
        /// <param name="aData">Data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Read ( byte[] aBlocks, out byte[] aData )
        {
            IntPtr pData = IntPtr.Zero;
            ushort wDataLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_MF_Read ( ref m_DataParamsInt[0], aBlocks, ( byte ) ( ( aBlocks == null )
                ? 0 : aBlocks.Length ), ref pData, ref wDataLen );

            aData = MarshalCopy ( oStatus, pData, wDataLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Ultralight Write command in X mode.
        ///     - In X-mode, MF_Write writes one or several blocks a MIFARE Ultralight or MIFARE
        ///       Ultralight C card.The command supports writing 16 bytes or 4 bytes plain. The
        ///       length can be selected by bit 0 of parameter byte P2. If 4 byte data blocks is
        ///       chosen, the MIFARE SAM AV4 constructs the WRITE command (Cmd 0xA2). If 16 byte
        ///       data blocks, the COMPATIBILITY_WRITE ( Cmd 0xA0) is used.
        ///
        ///     - If more than one block is written, the MIFARE SAM AV4 accesses the blocks in the
        ///       same order as addresses listed in the command data field.
        /// </summary>
        ///
        /// <param name="bOption">Specifying the product type targeted for writing the data.
        ///                         <see cref="MFUL_Write.PLAIN"/>
        ///                         <see cref="MFUL_Write.ULTRALIGHT"/>
        /// </param>
        /// <param name="aData">Buffer containing the Block and Data pair to be written.
        ///                     to card by Sam hardware. Should be holding an array of
        ///                     block number and data like.
        ///                     BNR_1 + Data, BNR_2 + Data, BNR_3 + Data, ..., BNR_N + Data
        ///                     Data could be 4 bytes or 16 bytes depending on the option specified.
        /// </param>
        ///
        /// <returns>Returns Success for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Write ( byte bOption, byte[] aData )
        {
            return phhalHw_Sam_Cmd_MF_Write ( ref m_DataParamsInt[0], bOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// The UL_SMExchange command is used to apply the MIFARE Ultralight AES CMAC-based
        /// secure messaging in X-mode on PICC commands.
        ///     - The user is responsible for providing the plain command data including the
        ///       PICC command code, and if applicable additional parameters and data.
        ///     - Supports the following command or MIFARE Ultralight AES product.
        ///         - GET_VERSION
        ///         - READ
        ///         - FAST_READ
        ///         - WRITE
        ///         - READ_CNT
        ///         - INCR_CNT
        ///         - READ_SIG
        ///         - WRITE_SIG
        ///         - LOCK_SIG
        /// </summary>
        ///
        /// <param name="aData">Plain command data of PICC. SAM will apply SM and exchange to PICC.
        /// </param>
        /// <param name="aResponse">Plain Data returned by Sam. </param>
        ///
        /// <returns>Returns Success for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_UL_SMExchange ( byte[] aData, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus =  phhalHw_Sam_Cmd_UL_SMExchange ( ref m_DataParamsInt[0], aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }
        #endregion
        #endregion

        #region MIFARE Common
        #region S_Mode
        /// <summary>
        /// <para>
        /// Performs CommitReaderID command execution part 1 in S mode. The first part
        /// includes the header, block number if its MifarePlus state or only header if its DESFire state.
        /// </para>
        ///
        /// \note
        ///     - Post calling this interface it's must to call <see cref="Cmd_SAM_CommitReaderID_Part2"/>
        ///       to place SAM into proper completion state if <see cref="Error_Gen.SUCCESS_CHAINING"/> is
        ///       returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
        ///     - If <see cref="Error_Gen.SUCCESS_CHAINING"/> is not returned from Library, no need to call
        ///     <see cref="Cmd_SAM_CommitReaderID_Part2"/> interface.
        /// </summary>
        ///
        /// <param name="bState">Options for framing the command. Below options
        ///                      should be used.
        ///                         - <see cref="TMRI_State.PLUS"/>: MIFARE Plus
        ///                         - <see cref="TMRI_State.PLUS"/>: MIFARE DESFire
        /// </param>
        /// <param name="wBlockNr">Number of the Transaction MAC Configuration Block. Will
        ///                        be used if \b bState is <see cref="TMRI_State.PLUS"/>
        /// </param>
        /// <param name="aResponse">Transaction MAC Reader ID and MAC to be sent to MIFARE Plus
        ///                         or MIFARE DESFire PICC.
        ///                             - TMRI: Transaction MAC Reader ID (16 byte)
        ///                                 - UID: SAM UID (7 byte)
        ///                                 - Padding: 0x80 || zero padding
        ///                             - MAC: 4 or 8 bytes MAC according to active Secure Messaging
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CommitReaderID_Part1 ( byte bState, ushort wBlockNr, out byte[] aResponse )
        {
            IntPtr pTMRI = IntPtr.Zero;
            ushort wTMRILen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1 ( ref m_DataParamsInt[0], bState, wBlockNr,
                ref pTMRI, ref wTMRILen );
            aResponse = MarshalCopy ( oStatus, pTMRI, wTMRILen );

            return oStatus;
        }

        /// <summary>
        /// <para>
        /// Performs CommitReaderID command execution part 2 in S mode. The last part includes
        /// the header and the response received from MIFARE Plus or DESFire PICC.
        /// </para>
        ///
        /// \note This interface should be called only if <see cref="Cmd_SAM_CommitReaderID_Part1"/>
        /// returns <see cref="Error_Gen.SUCCESS_CHAINING"/>.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by MIFARE Plus PICC. For success
        ///                            it should will
        ///                             - 0x90: MIFARE Plus
        ///                             - 0x00: MIFARE DESFire
        /// </param>
        /// <param name="aData">The complete data (EncTMRI and MAC) received from
        ///                     MIFARE Plus or MIFARE DESFire PICC.
        ///                         - EncTMRI (16 bytes): Encrypted Transaction MAC ReaderID of
        ///                           the latest successful transaction returned by the PICC.
        ///                           Present if \b bPiccErrCode = 0x00 or 0x90
        ///
        ///                         - MAC (8byte): Response MAC protection.Present if
        ///                           \b bPiccErrCode = 0x00 or 0x90 and not applying
        ///                           D40 Secure Messaging
        ///
        ///                         - Null otherwise
        /// </param>
        /// <param name="bDataLen">Length of bytes available in \b aData buffer. </param>
        /// <param name="bPiccRetCode">Error code returned by MIFARE PICC. This will of 1 byte
        ///                            in length.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_CommitReaderID_Part2 ( byte bPiccErrCode, byte[] aData, byte bDataLen,
            out byte bPiccRetCode )
        {
            bPiccRetCode = 0;
            return phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aData, bDataLen,
                ref bPiccRetCode );
        }
        #endregion

        #region X_Mode
        /// <summary>
        /// Performs CommitReaderID command execution in X mode. If success is returned the PICC return code will
        /// have 0x00 as the value else the actual error code.
        /// </summary>
        ///
        /// <param name="bISOMode">Options for P2 information byte.
        ///                         - <see cref="TMRI_IsoMode.NATIVE"/>:
        ///                           Command will be sent to PICC using native command set
        ///
        ///                         - <see cref="TMRI_IsoMode.ISO7816_4"/>:
        ///                           Command will be sent to PICC using ISO/IEC 7816-4APDU
        /// </param>
        /// <param name="bState">Options for framing the command. Below options
        ///                      should be used.
        ///                         - <see cref="TMRI_State.PLUS"/>: MIFARE Plus
        ///                         - <see cref="TMRI_State.DESFIRE"/>: MIFARE DESFire
        /// </param>
        /// <param name="wBlockNr">Number of the Transaction MAC Configuration Block.
        ///                        Will be used if \b bState is <see cref="TMRI_State.PLUS"/>
        /// </param>
        /// <param name="aEncTMRI">Encrypted Transaction MAC ReaderID of the latest
        ///                        successful transaction returned by the PICC.
        /// </param>
        /// <param name="bPiccRetCode">Status code returned by PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_TMRI_CommitReaderID ( byte bISOMode, byte bState, ushort wBlockNr, out byte[] aEncTMRI,
            out byte bPiccRetCode )
        {
            IntPtr pEncTMRI = IntPtr.Zero;
            ushort wEncTMRILen = 0;

            bPiccRetCode = 0;
            Status_t oStatus = phhalHw_Sam_Cmd_TMRI_CommitReaderID ( ref m_DataParamsInt[0], bISOMode, bState,
                wBlockNr, ref pEncTMRI, ref wEncTMRILen, ref bPiccRetCode );
            aEncTMRI = MarshalCopy ( oStatus, pEncTMRI, wEncTMRILen );

            return oStatus;
        }
        #endregion
        #endregion

        #region ICODE UCODE S_Mode
        /// <summary>
        /// Generates 10 bytes of random challenge to be given to Tag. Also Decrypts the TResponse
        /// received from Tag and verifies the decrypted data for authenticity.
        /// </summary>
        ///
        /// <param name="bOption">The command to be framed for SAM.
        ///                         - <see cref="Tam.GET_RND"/>: Get IChallange
        ///                         - <see cref="Tam.PROCESS_TRESPONE"/>: Exchange TResponse
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aData">Diversification Input or TResponse received from Tag.</param>
        /// <param name="aIChallange">The IChallange to be sent to tag. This will contain
        ///                           10 bytes of random challenge data.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>
        ///         Operation successful when \b bOption is
        ///         <see cref="Tam.PROCESS_TRESPONE"/>: Exchange TResponse
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///         Operation successful with chaining status when \b bOption is
        ///         <see cref="Tam.GET_RND"/>: Get IChallange
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateTAM1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aData, out byte[] aIChallange )
        {
            IntPtr pIChallange = IntPtr.Zero;
            ushort wIChallangeLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateTAM1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pIChallange, ref wIChallangeLen );
            aIChallange = MarshalCopy ( oStatus, pIChallange, wIChallangeLen );

            return oStatus;
        }

        /// <summary>
        /// Generates 10 bytes of random challenge to be given to Tag. Also Decrypts the TResponse received from
        /// Tag and verifies the decrypted data for authenticity and provides the custom data received form Tag.
        /// </summary>
        /// <param name="bOption">The command to be framed for SAM.
        ///                         - <see cref="Tam.GET_RND"/>: Get IChallange
        ///                         - <see cref="Tam.PROCESS_TRESPONE"/>: Exchange TResponse
        /// </param>
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aData">Diversification Input or TResponse received from Tag.</param>
        /// <param name="bBlockSize">To select the size of custom data block to be used.
        ///                          The value should either be 0x00 for 16 bit block size or
        ///                          0x01 for 64 bit block size. As per ISO 29167
        /// </param>
        /// <param name="bBlockCount">To select the custom data block to be used from the offset
        ///                           specified. The BlockCount range is from 1 - 16.
        /// </param>
        /// <param name="bProtMode">To specify the mode of operation to be used for encryption
        ///                         / decryption. The ProtMode ranges form 0 - 3. As per ISO 29167
        /// </param>
        /// <param name="aResponse">The IChallange to be sent to Tag. This will contain 10 bytes
        ///                         of random challenge data.Or the Custom Data received from Tag.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>
        ///         Operation successful when \b bOption is
        ///         <see cref="Tam.PROCESS_TRESPONE"/>: Exchange TResponse
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///         Operation successful with chaining status when \b bOption is
        ///         <see cref="Tam.GET_RND"/>: Get IChallange
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateTAM2 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aData, byte bBlockSize,
            byte bBlockCount, byte bProtMode, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateTAM2 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer,
                aData, ( byte ) aData.Length, bBlockSize, bBlockCount, bProtMode, ref pResponse, ref wRespLen );
            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            return oStatus;
        }

        /// <summary>
        /// Generates 10 bytes of random challenge to be given to TAg.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number to be used from SAM. One of the following,
        ///                         - NVM Key: 0x00 - 0x7F
        ///                         - RAM Key: 0xE0 - 0xE3
        /// </param>
        /// <param name="bKeyVer">Key version to be used from SAM.</param>
        /// <param name="aData">Diversification Input.</param>
        /// <param name="bPurposeMAM2">Purpose MAM 2 data. A 4 bit value.
        ///                                 - <see cref="Mam.NONE"/>: None
        ///                                 - <see cref="Mam.NONE"/>: Privacy disable until HF reset
        ///                                 - <see cref="Mam.NONE"/>: Privacy enable
        ///                                 - <see cref="Mam.NONE"/>: Privacy disable
        ///                                 - <see cref="Mam.NONE"/>: Destroy
        /// </param>
        /// <param name="aIChallange">The IChallange to be sent to Tag. This will contain
        ///                           10 bytes of random challenge data.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>
        ///         Operation successful when \b bOption is
        ///         <see cref="Tam.PROCESS_TRESPONE"/>: Exchange TResponse
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///         Operation successful with chaining status when \b bOption is
        ///         <see cref="Tam.GET_RND"/>: Get IChallange
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateMAM1 ( byte bKeyNo, byte bKeyVer, byte[] aData, byte bPurposeMAM2, out byte[] aIChallange )
        {
            IntPtr pIChallange = IntPtr.Zero;
            ushort wIChallangeLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateMAM1 ( ref m_DataParamsInt[0], bKeyNo, bKeyVer, aData,
                bPurposeMAM2, ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pIChallange, ref wIChallangeLen );
            aIChallange = MarshalCopy ( oStatus, pIChallange, wIChallangeLen );

            return oStatus;
        }

        /// <summary>
        /// Decrypts the TResponse received from Tag and verifies the decrypted data for authenticity. Also
        /// frames the IResponse with will be sent to the card.
        /// </summary>
        ///
        /// <param name="aData">TResponse received from card.</param>
        /// <param name="aIResponse">The IResponse generated by SAM that will be sent
        ///                          to Tag.This will contain 16 bytes of data.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_AuthenticateMAM2 ( byte[] aData, out byte[] aIResponse )
        {
            IntPtr pIResponse = IntPtr.Zero;
            ushort wIResponseLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_AuthenticateMAM2 ( ref m_DataParamsInt[0], aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pIResponse, ref wIResponseLen );
            aIResponse = MarshalCopy ( oStatus, pIResponse, wIResponseLen );

            return oStatus;
        }
        #endregion

        #region Programmable Logic
        /// <summary>
        /// The command SAM_PLExec is used to trigger the execution of the programmable logic.
        /// The command payload PLData is passed to the PL native code for processing. The SAM shall
        /// not interpret the command payload.
        /// </summary>
        ///
        /// <param name="bLFI">Option for updating the P1 information of SAM frame.
        ///                     - <see cref="LFI.LAST_FRAME"/>: Default or Last Frame
        ///                     - <see cref="LFI.CHAINED_FRAME"/>:Chained Frame
        /// </param>
        /// <param name="aPLData">Programmable Logic command data.</param>
        /// <param name="bPLDataLen">Length of bytes available in \b aPLData buffer.</param>
        /// <param name="aPLResp">Buffer holding the Programmable Logic response data.
        ///                         - Actual data received from SAM
        ///                         - Actual error data received from SAM.The response
        ///                           will be of 2 bytes.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///         Operation successful with chaining status.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPLData is NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_PLExec ( byte bLFI, byte[] aPLData, byte bPLDataLen, out byte[] aPLResp )
        {
            IntPtr pPLResp = IntPtr.Zero;
            ushort wPLRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_PLExec ( ref m_DataParamsInt[0], bLFI, aPLData,
                bPLDataLen, ref pPLResp, ref wPLRespLen );
            aPLResp = MarshalCopy ( oStatus, pPLResp, wPLRespLen );

            return oStatus;
        }

        /// <summary>
        /// The command SAM_PLUpload is used to update the code of the programmable logic.
        /// </summary>
        ///
        /// <param name="bIsFirstFrame">Option to represent the first frame where the UploadCtr
        ///                             will be exchanged.
        ///                                 - <see cref="Value.OFF"/>: Not First Frame
        ///                                 - <see cref="Value.OFF"/>: First Frame
        /// </param>
        /// <param name="bIsFinalFrame">Option to represent the last frame. If set, the LE byte
        ///                             will be exchanged and PLUploadACK will be received from
        ///                             SAM and will be validated internally.
        ///                                 - <see cref="Value.OFF"/>: Not Final Frame
        ///                                 - <see cref="Value.OFF"/>: Final Frame
        /// </param>
        /// <param name="wUploadCtr">The upload counter value. This should be greater than
        ///                          the one available in SAM.
        /// </param>
        /// <param name="bKeyNo">Key number of Upload key (Ku) in Software KeyStore.</param>
        /// <param name="bKeyVer">Key version of Upload key (Ku) in software KeyStore.</param>
        /// <param name="aPLCode">Plain Programmable Logic code.</param>
        /// <param name="wPLCodeLen">Length of bytes available in \b aPLCode buffer.</param>
        /// <param name="aPLReKey">The ReKey to be used for next crypto segment. This should
        ///                        have the next SessionENC key followed by the SessionMAC key.
        /// </param>
        /// <param name="bPLReKeyLen">Length of bytes available in \b aPLReKey buffer. The length
        ///                           should be equal to double AES key size.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/>
        ///         Operation successful with chaining status.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         \b aPLCode and \b aPLReKey are NULL
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_PLUpload ( byte bIsFirstFrame, byte bIsFinalFrame, ushort wUploadCtr,
            byte bKeyNo, byte bKeyVer, byte[] aPLCode, ushort wPLCodeLen, byte[] aPLReKey,
            byte bPLReKeyLen )
        {
            byte[] aPlCodeTmp = new byte[wPLCodeLen];
            aPLCode.CopyTo ( aPlCodeTmp, 0 );

            return phhalHw_Sam_Cmd_SAM_PLUpload ( ref m_DataParamsInt[0], bIsFirstFrame, bIsFinalFrame,
                wUploadCtr, bKeyNo, bKeyVer, aPlCodeTmp, wPLCodeLen, aPLReKey, bPLReKeyLen );
        }
        #endregion

        #region Reader Chip
        /// <summary>
        /// Read the content of one or more register(s) of the connected reader chip.
        /// </summary>
        ///
        /// <param name="bRegAddr">Address for the registers to be read.</param>
        /// <param name="aValue">Register(s) content in the same order as the
        ///                      command data field address(es).
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_RC_ReadRegister ( byte[] bRegAddr, out byte[] aValue )
        {
            aValue = new byte[( ( bRegAddr == null ) ? 0 : bRegAddr.Length )];
            Status_t oStatus = phhalHw_Sam_Cmd_RC_ReadRegister ( ref m_DataParamsInt[0], bRegAddr,
                ( byte ) ( ( bRegAddr == null ) ? 0 : bRegAddr.Length ), aValue );
            if ( !oStatus.Equals ( new Status_t () ) )
                aValue = null;

            return oStatus;
        }

        /// <summary>
        /// Write the content of one or more register(s) of the connected reader chip.
        /// </summary>
        ///
        /// <param name="aData">The registers to be updated. This buffer should
        ///                     contain register address followed by value.
        ///                     The format should be as mentioned below.
        ///                     RegAdd || Value || RegAdd || Value || ... ||
        ///                     RegAdd || Value
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_RC_WriteRegister ( byte[] aData )
        {
            return phhalHw_Sam_Cmd_RC_WriteRegister ( ref m_DataParamsInt[0], aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Turn the radio frequency field On or Off
        /// </summary>
        ///
        /// <param name="dwTime">Shut-down time for the RF field,
        ///                      zero turns the field off.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_RC_RFControl ( int dwTime )
        {
            return phhalHw_Sam_Cmd_RC_RFControl ( ref m_DataParamsInt[0], ( ushort ) dwTime );
        }

        /// <summary>
        /// Establishes the serial connection between MIFARE SAM and Reader IC
        /// and initializes the reader chip with the register values stored in the
        /// selected register value set.
        /// </summary>
        ///
        /// <param name="bLoadReg">The Load register information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_RC_Init ( byte bLoadReg )
        {
            return phhalHw_Sam_Cmd_RC_Init ( ref m_DataParamsInt[0], bLoadReg );
        }

        /// <summary>
        /// Stores a customer defined register value set for the PN512/RC52x/RC663 in the
        /// non-volatile memory of the MIFARE SAM
        /// </summary>
        ///
        /// <param name="bStoreReg">Number of register value set to be used for storing the values.</param>
        /// <param name="aData">List of Register descriptions. This buffer should be as
        ///                     mentioned below.
        ///                         - If RC663: SpeedID || ItemCount || RegAddr || RegValue...
        ///                           [|| RegAddr || RegValue]
        ///                         - If RC5XX: RegAddr || RegValue... [|| RegAddr || RegValue]
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_RC_LoadRegisterValueSet ( byte bStoreReg, byte[] aData )
        {
            return phhalHw_Sam_Cmd_RC_LoadRegisterValueSet ( ref m_DataParamsInt[0],
                bStoreReg, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion

        #region ISO14443-3
        /// <summary>
        /// Perform a request or wake-up command (type A).
        /// </summary>
        ///
        /// <param name="bCmdCode">Command code.
        ///                         - <see cref="CmdCode.REQA"/>
        ///                         - <see cref="CmdCode.WUPA"/>
        /// </param>
        /// <param name="aAtqa">AtqA returned by the card. The buffer has to be
        ///                     2 bytes long.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_RequestA_Wakeup ( byte bCmdCode, out byte[] aAtqa )
        {
            aAtqa = new byte[2];
            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup ( ref m_DataParamsInt[0],
                bCmdCode, aAtqa );

            if ( !oStatus.Equals ( new Status_t () ) )
                aAtqa = null;

            return oStatus;
        }

        /// <summary>
        /// Perform bit-wise AntiCollision and select. The AntiCollision and the following
        /// select are performed according to the select code in the data field.
        ///     - The selection can be carried out for a variable amount of cascade levels.The select codes
        ///       have to be listed in the data field subsequently. The MIFARE SAM AV4 will take the parameters
        ///       exactly and use them as select code. Therefore to fully select a card with a triple UID, the
        ///       data field has to be of three bytes length indicating 93h, 95h and 97h whereas the data field
        ///       has to be of one byte length indicating 93h if a single size UID card are selected.
        ///     - If the select code indicates a cascade level of 93h and 95h, and the UID of the card consists
        ///       only of four bytes, the MIFARE SAM AV4 will exit the command and return the SAK and the UID of
        ///       the card.
        ///     - If the select code indicates a cascade level of 93h, and the UID consists of more than four bytes
        ///       the MIFARE SAM will also exit the command and return the SAK and the first three bytes of the UID
        ///       but indicates with a special return code the incompleteness of the UID separately.The caller has
        ///       then to take care about continuing the procedure on his own by calling the command once more with
        ///       a higher select code.The already returned UID bytes will not be returned a second time. The same
        ///       applies for a select code of 95h if the UID is of ten bytes length (suggest that a selection with
        ///       code 93h is carried out previously).
        /// </summary>
        ///
        /// <param name="aSelCodes">Buffer containing the SEL sequence bytes.
        ///                             <see cref="CascadeLevel.LEVEL_1"/>
        ///                             <see cref="CascadeLevel.LEVEL_2"/>
        ///                             <see cref="CascadeLevel.LEVEL_3"/>
        /// </param>
        /// <param name="bSak">Buffer to the 1 byte Select Acknowledge received from card.</param>
        /// <param name="aUid">Buffer containing the received UID. This buffer has to be
        ///                    10 bytes long.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_AnticollisionSelect ( byte[] aSelCodes, out byte bSak, out byte[] aUid )
        {
            bSak = 0;

            aUid = new byte[10];
            byte bUidLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_3_AnticollisionSelect ( ref m_DataParamsInt[0], aSelCodes,
                ( byte ) ( ( aSelCodes == null ) ? 0 : aSelCodes.Length ), ref bSak, aUid, ref bUidLen );

            /* Resize pUid buffer to actual bVersion size */
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error.ISO_UID_INCOMPLETE ) )
                Array.Resize ( ref aUid, bUidLen );

            return oStatus;
        }

        /// <summary>
        /// Perform one or several Request - AntiCollision - select sequences and returns the
        /// SAK and the UID of the selected PICC ( s).
        ///     - The ATQA is returned for every request issued, this means for every newly activated card.
        ///       Due to the fact that the resulting ATQA is the OR-function of all ATQAs, the value may
        ///       change frequently.
        ///     - The command offers several different possibilities for activating cards:
        ///         - Wait for a definable number of cards and return if this number of cards has been activated
        ///         - Either try only one activation sequence and return if no PICC is detected, wait for a
        ///           defined amount of time, or wait until the expected number of PICCs was activated
        ///         - Apply a filter for the ATQA response of the PICC(s)
        ///         - Apply a filter for the SAK response of the PICC ( s)
        ///     - If more than one PICC are activated, the MIFARE SAM puts all activated PICC ( s) into the Halt
        ///       state.
        /// </summary>
        ///
        /// <param name="bOption">Option to be used for P2 and internal to library.
        ///                       Can be combined by using bitwise OR operator.
        ///                         - Option to be used for P2 byte information.
        ///                             - <see cref="ActivateIdle.DEFAULT"/>: Do not apply SAK and ATQA filter
        ///                             - <see cref="ActivateIdle.APPLY_SAK"/>: Apply SAK filter
        ///                             - <see cref="ActivateIdle.APPLY_ATQA"/>: Apply ATQA filter
        ///
        ///                         - Option used by Library internally.
        ///                             - <see cref="ActivateIdle.APPLY_TIME"/>:
        ///                               Exchange Short Time to wait information to SAM.
        /// </param>
        /// <param name="bNumCards">Holds the number of cards to activate.</param>
        /// <param name="dwTime">Time to wait for a card.</param>
        /// <param name="aAtqaIn">Buffer containing the AtqA filter. Should contain the following,
        ///                       Mask (1 byte) || Value (1 byte) || Mask2 (1 byte) || Value2 (1 byte)
        /// </param>
        /// <param name="aSakIn">Buffer containing the SAK filter. Should contain the following,
        ///                      Mask (1 byte) || Value (1 byte)
        /// </param>
        /// <param name="stCardInfo">Buffer containing the received data.
        ///                             - ATQA ( Answer To re-Quest type A): 2 bytes
        ///                             - SAK ( Select AcKnowledge )       : 1 byte
        ///                             - Len ( Length of card UID )       : 4, 7, 10 bytes
        ///                             - UID ( Unique Identified )        : Card UID
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_ActivateIdle ( byte bOption, byte bNumCards, int dwTime, byte[] aAtqaIn,
            byte[] aSakIn, out ActivateIdleParams[] stCardInfo )
        {
            byte[] aResponse;
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle ( ref m_DataParamsInt[0], bOption, bNumCards,
                ( ushort ) dwTime, aAtqaIn, aSakIn, ref pResponse, ref wRespLen );

            /* Copy the response to the Structure. */
            if ( ( wRespLen > 0 ) && oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aResponse = new byte[wRespLen];
                Marshal.Copy ( pResponse, aResponse, 0, wRespLen );

                /* Copy the response to the Structure. If bNumCards == 1 */
                stCardInfo = new ActivateIdleParams[( bNumCards > 1 ) ? aResponse[0] : bNumCards];
                if ( bNumCards == 1 )
                {
                    /* Copy ATQA */
                    stCardInfo[0].aAtqa = new byte[2];
                    Array.Copy ( aResponse, stCardInfo[0].aAtqa, 2 );

                    /* Copy SAK */
                    stCardInfo[0].bSak = aResponse[2];

                    /* Copy UID */
                    stCardInfo[0].aUid = new byte[aResponse[3]];
                    Array.Copy ( aResponse, 4, stCardInfo[0].aUid, 0, aResponse[3] );
                }

                /* Copy the response to the Structure. If bNumCards > 1 */
                else
                {
                    int dwInfoOffset = 1;
                    for ( int dwIndex = 0; dwIndex < aResponse[0]; dwIndex++ )
                    {
                        stCardInfo[dwIndex].aAtqa = new byte[2];
                        Array.Copy ( aResponse, dwInfoOffset, stCardInfo[dwIndex].aAtqa, 0, 2 );

                        stCardInfo[dwIndex].bSak = aResponse[dwInfoOffset + 2];
                        stCardInfo[dwIndex].aUid = new byte[aResponse[dwInfoOffset + 3]];

                        Array.Copy ( aResponse, dwInfoOffset + 4, stCardInfo[dwIndex].aUid, 0, aResponse[dwInfoOffset + 3] );
                        dwInfoOffset = dwInfoOffset + 4 + stCardInfo[dwIndex].aUid.Length;
                    }
                }
            }
            else
                stCardInfo = null;

            return oStatus;
        }

        /// <summary>
        /// Perform reactivates and selects a PICC that has previously been set to Halt state.
        /// </summary>
        ///
        /// <param name="aUid">UID of the card to reactivate.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_ActivateWakeUp ( byte[] aUid )
        {
            return phhalHw_Sam_Cmd_X_ISO14443_3_ActivateWakeUp ( ref m_DataParamsInt[0], aUid,
                ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ) );
        }

        /// <summary>
        /// Puts a selected card into Halt state
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_HaltA ()
        {
            return phhalHw_Sam_Cmd_X_ISO14443_3_HaltA ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Perform exchange bytes/bits transparently with a PICC. The MIFARE SAM will take the user data
        /// and send it without changing, inserting or appending any content to the contactless card.
        /// Appending of a CRC, time-out settings, etc.have to be configured by directly writing the
        /// RC52x/RC663 registers. Take into account that switching settings of the Reader IC influence all
        /// subsequent MIFARE SAM commands proposing the correct Reader IC settings, that is ISO14443-
        /// 4_Exchange.
        ///     - If bits are exchanged, parameter P1 indicates the number of bits valid in the last input byte.
        ///       A value of 00h indicates that the full last byte are sent to the PICC. Consequently, for using
        ///       the function as pure byte exchange, P1 has to be set to zero.In case that only a few bits of
        ///       the last byte are sent, the MIFARE SAM will take the indicated number of bits beginning with
        ///       the LSBit of the last input byte.
        ///     - Take into account that this function cannot be used for bit wise AntiCollision since the MIFARE
        ///       SAM will react on a detected collision with an error and not return any bytes (bits).
        ///     - If the card responds with single bits, the MIFARE SAM will indicate the number of valid bits
        ///       within the last returned byte in the lower nibble of SW2.SW2 = 00h indicates that the full
        ///       byte was received from the PICC.The last returned data byte contains the valid bits according
        ///       to the indicated amount starting with the LSBit.
        /// </summary>
        ///
        /// <param name="aTxBuffer">Buffer containing the data to be sent.</param>
        /// <param name="bTxBitLen">Number valid bits (0x00 - 0x07) in the last byte
        ///                         of the \b aTxBuffer buffer. If set to 00h all bits
        ///                         are valid.
        /// </param>
        /// <param name="aRxBuffer">Buffer containing the received data from PICC.</param>
        /// <param name="bRxBitLen">Amount of valid bits in the last byte in case of an incomplete byte.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_3_TransparentExchange ( byte[] aTxBuffer, byte bTxBitLen, out byte[] aRxBuffer,
            out byte bRxBitLen )
        {
            ushort wRxLen = 0;
            IntPtr pRxBuf = IntPtr.Zero;
            bRxBitLen = 0;

            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_3_TransparentExchange ( ref m_DataParamsInt[0], aTxBuffer,
                ( byte ) ( ( aTxBuffer == null ) ? 0 : aTxBuffer.Length ), bTxBitLen, ref pRxBuf, ref wRxLen,
                ref bRxBitLen );
            aRxBuffer = MarshalCopy ( oStatus, pRxBuf, wRxLen );

            return oStatus;
        }
        #endregion

        #region ISO14443-4
        /// <summary>
        /// Perform a combined RATS and PPS sequence to prepare a PICC for T = CL data exchange.
        ///     - The CID assigned to the PICC will be assigned to the current logical channel.This means,
        ///       every further ISO14443-4 command issued in this logical channel will be executed using
        ///       this CID automatically
        ///     - If a CID value of FFh is passed to the MIFARE SAM, the MIFARE SAM will automatically assign a
        ///       CID to the PICC.Therefore it takes already assigned CIDs of other logical channels into account.
        ///       If a PICC is activated in any other logical channel with an assigned CID of 00h, or not supporting
        ///       a CID, the PICC will not be activated and the command returns an error. Also all user-defined CIDs
        ///       will be checked against the already activated PICCs and rejected if the PICC cannot be activated
        ///       following the ISO/IEC 14443-4 standard.
        ///     - For bit rate selection, the values in the data field define the maximum bit rates. If a PICC does
        ///       not support these bit rates, the MIFARE SAM will automatically decide the bit rate to apply.
        ///     - The MIFARE SAM returns the actual CID, DRI, DSI and the ATS. If a PICC was activated, which does not
        ///       support a CID, a CID value of 0 is returned.
        ///     - The MIFARE SAM automatically configures the frame sizes to apply for T= CL data exchanges. As for a
        ///       bit rate of 848 kbit/s FSD is limited to 96 byte, it is recommended to not indicate a DSI value of
        ///       3 in case the PICC is not able to support it, if already known in advance.The activation procedure
        ///       will succeed since the MIFARE SAM automatically chooses the next applicable bit rate, but since the
        ///       FSDI value has already to be sent to the PICC in the RATS command, the MIFARE SAM will indicate an
        ///       FSD of 96 byte supposing that 848 kbit/s will be used for communication.The actually supported bit
        ///       rate of the PICC is not known to the MIFARE SAM at this time (returned in ATS afterwards) so also
        ///       if a lower bit rate is selected later on, FSD will be at the non-optimum value of 96 byte.
        ///     - It could happen that the command indicates an error although a PICC has been activated correctly by
        ///       the RATS command. This is the case if the subsequently sent PPS failed.
        /// </summary>
        ///
        /// <param name="bCidIn">CID to be sent.</param>
        /// <param name="bDsiIn">DSI to be sent.
        ///                         - <see cref="RfDataRate.DATARATE_106"/>
        ///                         - <see cref="RfDataRate.DATARATE_212"/>
        ///                         - <see cref="RfDataRate.DATARATE_424"/>
        ///                         - <see cref="RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="bDriIn">DRI to be sent.
        ///                         - <see cref="RfDataRate.DATARATE_106"/>
        ///                         - <see cref="RfDataRate.DATARATE_212"/>
        ///                         - <see cref="RfDataRate.DATARATE_424"/>
        ///                         - <see cref="RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="bCidOut">CID used (1 byte).</param>
        /// <param name="bDsiOut">DSI used (1 byte).
        ///                         - <see cref=".RfDataRate.DATARATE_106"/>
        ///                         - <see cref=".RfDataRate.DATARATE_212"/>
        ///                         - <see cref=".RfDataRate.DATARATE_424"/>
        ///                         - <see cref=".RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="bDriOut">DRI used (1 byte).
        ///                         - <see cref="RfDataRate.DATARATE_106"/>
        ///                         - <see cref="RfDataRate.DATARATE_212"/>
        ///                         - <see cref="RfDataRate.DATARATE_424"/>
        ///                         - <see cref="RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="aAts">Buffer containing the received ATS (Answer to Select).
        ///                    The length of the ATS can be found on the first position.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_RATS_PPS ( byte bCidIn, byte bDsiIn, byte bDriIn, out byte bCidOut, out byte bDsiOut,
            out byte bDriOut, out byte[] aAts )
        {
            bCidOut = 0;
            bDsiOut = 0;
            bDriOut = 0;
            aAts = new byte[256];

            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS ( ref m_DataParamsInt[0], bCidIn, bDsiIn, bDriIn, ref bCidOut,
                ref bDsiOut, ref bDriOut, aAts );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
                Array.Resize ( ref aAts, aAts[0] );
            else
                aAts = null;

            return oStatus;
        }

        /// <summary>
        /// Perform a init of ISO-14443-4 layer (init T = CL protocol). The intent of this command
        /// is to configure the protocol for data exchanges.This is necessary if a PICC was already activated
        /// and configured for doing data exchanges without using the <see cref="Cmd_X_ISO14443_4_RATS_PPS"/>
        /// command.
        ///     - The CID assigned to the PICC will be assigned to the current logical channel.This means, every
        ///       further ISO14443-4 command issued in this logical channel, will be executed using this CID
        ///       automatically. If in any other logical channel a PICC is activated with an assigned CID of 00h or
        ///       not supporting a CID, the PICC will not be activated and the command returns an error. Also, all
        ///       user-defined CIDs will be checked against the already activated PICCs and rejected if the PICC
        ///       cannot be activated following the ISO/IEC 14443-4 standard.If a PICC does not support a CID, this
        ///       has to be indicated by a CID value of FFh.
        ///     - The MIFARE SAM will automatically adjust FSD according to the DSI input value.The user driving the
        ///       activation sequence is responsible for indicating the correct FSDI value to the PICC. For details on
        ///       applied frame sizes pleas refer to ISO14443-4_Exchange
        /// </summary>
        ///
        /// <param name="bCid">CID to apply.One of the following
        ///                     - CID to apply: 0x00 - 0x0E
        ///                     - No CID to apply: 0xFF
        /// </param>
        /// <param name="bDri">Bit rate PCD -> PICC.
        ///                         - <see cref="RfDataRate.DATARATE_106"/>
        ///                         - <see cref="RfDataRate.DATARATE_212"/>
        ///                         - <see cref="RfDataRate.DATARATE_424"/>
        ///                         - <see cref="RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="bDsi">Bit rate PICC -> PCD.
        ///                         - <see cref="RfDataRate.DATARATE_106"/>
        ///                         - <see cref="RfDataRate.DATARATE_212"/>
        ///                         - <see cref="RfDataRate.DATARATE_424"/>
        ///                         - <see cref="RfDataRate.DATARATE_848"/>
        /// </param>
        /// <param name="bFwi">Frame waiting time indicator (0x00 - 0x0F).</param>
        /// <param name="bFsci">Frame size card indicator (0x00 - 0x08).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_Init ( byte bCid, byte bDri, byte bDsi, byte bFwi, byte bFsci )
        {
            return phhalHw_Sam_Cmd_X_ISO14443_4_Init ( ref m_DataParamsInt[0], bCid, bDri, bDsi, bFwi,
                bFsci );
        }

        /// <summary>
        /// Perform an exchange bytes with a PICC according ISO14443-4 T = CL protocol.
        /// The MIFARE SAM offers the possibility to send more data to a PICC than the maximum data field size.
        /// In this case the last frame indicator shall indicate AFh. The MIFARE SAM will then set the chaining
        /// bit of the last sent I-block indicating to the PICC that more data will follow and to continue the
        /// chaining process after receiving the next command from the host. If more than 256 bytes are to be
        /// received from the PICC, the MIFARE SAM will not acknowledge the last received chained I-block from
        /// the PICC, returns the data to the host and waits for the next command to continue the chaining process.
        /// The following exchange command from the host shall not contain any data, since more data has to be
        /// received from the PICC.However, if there is a data field content in the command, the MIFARE SAM will
        /// ignore it and continue reception.
        /// </summary>
        ///
        /// <param name="wOption">Buffering Option.
        ///                         - <see cref="ExchangeOptions.DEFAULT"/> To send and receive complete data
        ///
        ///                         - <see cref="ExchangeOptions.TXCHAINING"/> To exchange chained data. Here
        ///                           <see cref="LFI.CHAINED_FRAME"/> flag will be set by the library internally
        ///                           to P1 information byte.
        /// </param>
        /// <param name="aAppDataOut">Pointer to buffer containing the data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/>          Operation successful.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Operation successful with chaining status.
        ///     Returns <see cref="Error_Comm.LENGTH_ERROR"/>
        ///         If received response length is greater than 0xF8
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_Exchange ( ushort wOption, byte[] aAppDataIn, out byte[] aAppDataOut )
        {
            ushort wAppDataOutLen = 0;
            IntPtr pAppDataOut = IntPtr.Zero;

            Status_t oStatus = phhalHw_Sam_Cmd_X_ISO14443_4_Exchange ( ref m_DataParamsInt[0], wOption, aAppDataIn,
                (byte) ( ( aAppDataIn == null ) ? 0 : aAppDataIn.Length ), ref pAppDataOut, ref wAppDataOutLen );
            aAppDataOut = MarshalCopy ( oStatus, pAppDataOut, wAppDataOutLen );

            return oStatus;
        }

        /// <summary>
        /// Check if an activated card is still in the field.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_PresenceCheck ()
        {
            return phhalHw_Sam_Cmd_X_ISO14443_4_PresenceCheck ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Perform De-selection of an activated PICC. The CID is freed by this command. If the DeSelect fails,
        /// the CID will not be freed and cannot be used for activating another PICC. This behavior might be overridden
        /// by setting a flag in the P1 byte. CIDs can also be freed using the <see cref="Cmd_X_ISO14443_4_FreeCid"/>
        /// command.
        /// </summary>
        ///
        /// <param name="bFreeCid">Bit mask for DeSelect option.
        ///                         - <see cref="Deselect.DO_NOT_FREE_CID"/>:
        ///                           CID will not be freed if DeSelect fails
        ///
        ///                         - <see cref="Deselect.FORCE_FREE_CID"/>:
        ///                           Force CID to be freed in any case
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_Deselect ( byte bFreeCid )
        {
            return phhalHw_Sam_Cmd_X_ISO14443_4_Deselect ( ref m_DataParamsInt[0], bFreeCid );
        }

        /// <summary>
        /// Free up one, more, or all currently assigned CIDs.
        ///     - This command might be necessary if several DeSelect commands failed and the CIDs were not forced to be
        ///       freed but the PICC is deactivated or no longer available in the field.
        ///     - The command takes at least one CID value in the data field.A CID value of FFh indicates that all CIDs have to
        ///       be freed.Sending the same CID several times will not cause an error.The command is not related to a logical
        ///       channel since CIDs are managed in a global pool.
        ///     - Be careful when calling this command.It may free CIDs in logical channels in which PICCs are still activated.
        ///       The activated PICCs can then no longer be accessed. Freeing the CID additionally causes the MIFARE SAM to reset
        ///       the state of the logical channel related to the CID, thus the MIFARE SAM clears the indication for an activated
        ///       PICC.
        ///     - If an incorrect CID (value higher than 14) is detected, the command is aborted. Nevertheless all CIDs which were
        ///       contained in the data field previous to the incorrect one are freed.The value FFh will cause all CIDs to be
        ///       freed independent from its occurring position in the data field.
        /// </summary>
        ///
        /// <param name="aCid">Buffer containing all CIDs to be freed.
        ///                     - CID: 0x00 - 0x0E
        ///                     - Free all CIDs: 0xFF
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_X_ISO14443_4_FreeCid ( byte[] aCid )
        {
            return phhalHw_Sam_Cmd_X_ISO14443_4_FreeCid ( ref m_DataParamsInt[0], aCid,
                ( byte ) ( ( aCid == null ) ? 0 : aCid.Length ) );
        }
        #endregion

        #region Originality Check
        /// <summary>
        /// Performs Originality Check of SAM . The authentication is initiated by ISOInternalAuthenticate.
        /// </summary>
        ///
        /// <param name="bOption">Challenge usage (RndA)
        ///                         - <see cref="MFUL_Challenge.GENERATE_INTERNAL"/>
        ///                         - <see cref="MFUL_Challenge.USER_PROVIDED"/>
        /// </param>
        /// <param name="aOptsA">PCD Option.
        ///                         - Will be exchanged if \b bOptsALen has a value
        ///                           other than zero
        ///                         - User should provide Value field only.Tag and
        ///                           Length will be included internally.
        /// </param>
        /// <param name="aRndA">Authentication Data Object: random challenge from PCD.
        ///                         - Empty buffer if <see cref="MFUL_Challenge.GENERATE_INTERNAL"/>
        ///                           is provided in \b bOption parameter. Will be updated with Random
        ///                           Challenge generated internally. The buffer size should be equal to
        ///                           the RndA size that is expected.
        ///
        ///                         - Random Challenge if <see cref="MFUL_Challenge.USER_PROVIDED"/>
        ///                           is provided in \b bOption parameter. User should provide Value
        ///                           field only. AuthDOHdr (Tag and Length) will be included internally.
        /// </param>
        /// <param name="aResponse"></param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> Operation successful.
        ///     Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Cmd_SAM_ISOInternalAuthenticate ( byte bOption, byte[] aOptsA, ref byte[] aRndA, out byte[] aResponse )
        {
            IntPtr pResponse = IntPtr.Zero;
            ushort wRespLen = 0;
            byte bRndA = ( byte ) aRndA.Length;

            Status_t oStatus = phhalHw_Sam_Cmd_SAM_ISOInternalAuthenticate ( ref m_DataParamsInt[0], bOption, aOptsA,
                ( byte ) ( ( aOptsA == null ) ? 0 : aOptsA.Length ), ref aRndA, ref bRndA, ref pResponse, ref wRespLen);

            aResponse = MarshalCopy ( oStatus, pResponse, wRespLen );

            if ( !oStatus.Equals ( new Status_t () ) &&
                ( bOption == ( byte ) MFUL_Challenge.GENERATE_INTERNAL ) )
            {
                Buffer.SetByte ( aRndA, 0, 0 );
                bRndA = 0;
            }

            return oStatus;
        }
        #endregion
        #endregion

        #region Support classes
        #region Key Management
        #region Cmd.SAM_ChangeKeyEntry
        #region Program Mask bits
        /// <summary>
        /// Helper Class containing the Options (Non-volatile programming mask) to be
        /// framed and utilized by <see cref="Cmd_SAM_ChangeKeyEntry"/> and
        /// <see cref="Cmd_SAM_ChangeKeyEntryOffline"/> interfaces.
        /// </summary>
        public class ChangeKeyEntry_ProMas
        {
            private int dwProMask;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator ChangeKeyEntry_ProMas ( byte bProMas )
            {
                ChangeKeyEntry_ProMas oObject = new ChangeKeyEntry_ProMas ();
                oObject.dwProMask = bProMas;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( ChangeKeyEntry_ProMas oProMas )
            {
                return ( byte ) oProMas.dwProMask;
            }

            /// <summary>
            /// Update KeyVa.
            /// This set Bit[7] of P2 byte
            /// </summary>
            public bool UpdateKeyVa
            {
                set
                {
                    if ( value ) dwProMask |= 0x80;
                    else dwProMask &= ~0x80;
                }
                get
                {
                    return ( ( dwProMask & 0x80 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update KeyVb (shall be '0' if RAMKey, P1 >= 0xE0).
            /// This set Bit[6] of P2 byte
            /// </summary>
            public bool UpdateKeyVb
            {
                set
                {
                    if ( value ) dwProMask |= 0x40;
                    else dwProMask &= ~0x40;
                }
                get
                {
                    return ( ( dwProMask & 0x40 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update KeyVc (shall be '0' if RAMKey, P1 >= 0xE0).
            /// This set Bit[5] of P2 byte
            /// </summary>
            public bool UpdateKeyVc
            {
                set
                {
                    if ( value ) dwProMask |= 0x20;
                    else dwProMask &= ~0x20;
                }
                get
                {
                    return ( ( dwProMask & 0x20 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update DF_AID; DF_KeyNo (shall be '0' if RAMKey, P1 >= 0xE0).
            /// This set Bit[4] of P2 byte
            /// </summary>
            public bool UpdateDFAid_DFKeyNo
            {
                set
                {
                    if ( value ) dwProMask |= 0x10;
                    else dwProMask &= ~0x10;
                }
                get
                {
                    return ( ( dwProMask & 0x10 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update KeyNoCEK and KeyVCEK
            /// This set Bit[3] of P2 byte
            /// </summary>
            public bool UpdateKeyNoCEK_KeyVCEK
            {
                set
                {
                    if ( value ) dwProMask |= 0x08;
                    else dwProMask &= ~0x08;
                }
                get
                {
                    return ( ( dwProMask & 0x08 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update RefNoKUC
            /// This set Bit[2] of P2 byte
            /// </summary>
            public bool UpdateRefNoKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x04;
                    else dwProMask &= ~0x04;
                }
                get
                {
                    return ( ( dwProMask & 0x04 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update SET, ExtSET and, if present, KeyNoAEK and KeyVAEK
            /// This set Bit[1] of P2 byte
            /// </summary>
            public bool UpdateSET_ExtSET
            {
                set
                {
                    if ( value ) dwProMask |= 0x02;
                    else dwProMask &= ~0x02;
                }
                get
                {
                    return ( ( dwProMask & 0x02 ) != 0x00 );
                }
            }

            /// <summary>
            /// Key versions specified separately.
            /// If set, Key versions are specified in the data field ( KeyEntry )
            /// after configuration settings ( SET)
            /// This set Bit[1] of P2 byte
            /// </summary>
            public bool KeyVersionsSentSeparately
            {
                set
                {
                    if ( value ) dwProMask |= 0x01;
                    else dwProMask &= ~0x01;
                }
                get
                {
                    return ( ( dwProMask & 0x01 ) != 0x00 );
                }
            }
        }
        #endregion

        #region KeyEntry
        /// <summary>
        /// Helper Class to frame Key Entry buffer as bytes and also extract various Key
        /// information from byte array. To be used by <see cref="Cmd_SAM_ChangeKeyEntry"/>
        /// interface.
        /// </summary>
        public class KeyEntry
        {
            #region KeyTypes
            /// <summary>
            /// Enumeration for Key Types mentioned in SAM document
            /// </summary>
            public enum KeyType : int
            {
                /// <summary> TDEA DESFire4 (DES or 2TDEA key) </summary>
                DESFIRE4 = 0x0000,

                /// <summary> TDEA ISO 10116 (16-bits CRC, 4-bytes MAC) (DES or 2TDEA key) </summary>
                ISO_10116 = 0x0008,

                /// <summary> 3TDEA ISO 10116 (3TDEA key) </summary>
                TDEA3 = 0x0018,

                /// <summary> AES 128-bits key (AES_128 key) </summary>
                AES_128 = 0x0020,

                /// <summary> AES 192-bits key (AES_192 key) </summary>
                AES_192 = 0x0028,

                /// <summary> TDEA ISO 10116 (32-bits CRC, 8-bytes MAC) (DES or 2TDEA key) </summary>
                ISO_CRC32 = 0x0030,

                /// <summary> AES 256-bits key (AES_256 key) </summary>
                AES_256 = 0x0038,

                /// <summary> AES 128-bits key (AES_128) restricted for LRP usage. </summary>
                AES128_LRP = 0x0040
            }
            #endregion

            #region KeyClass
            /// <summary>
            /// Enumeration for Key Class mentioned in SAM document
            /// </summary>
            public enum KeyClass : int
            {
                /// <summary>
                /// Host keys: used for protecting the SAM - Host communication
                /// These keys are restricted to AES and are not supported by RAM
                /// keys.
                /// </summary>
                HOST = 0x00,

                /// <summary>
                /// PICC Keys: used for protecting the SAM-PICC communication;
                /// depending on the key types they can be used for authenticating and
                /// communicating with a MIFARE Plus, MIFARE DESFire, MIFARE
                /// Ultralight and/or MIFARE Ultralight C card.
                /// </summary>
                PICC = 0x01,

                /// <summary>
                /// OfflineChange Keys: used for some key management commands,
                /// to allow offline preparation of the cryptograms for these commands
                /// (compared to when the key management is done with Host Keys).
                /// These keys are restricted to AES and are not supported by RAM
                /// keys.
                /// </summary>
                OFFLINE_CHANGE = 0x02,

                /// <summary>
                /// OfflineCrypto Keys: used for offline crypto operations: for example
                /// for communication with the back-end or for writing encrypted data on
                /// a MIFARE Plus or MIFARE Ultralight C card
                /// </summary>
                OFFLINE_CRYPTO = 0x04,

                /// <summary>
                /// OfflineUpload Keys: used for the upload of the programmable logic
                /// code. These keys are restricted to AES and are not supported by RAM
                /// keys.
                /// </summary>
                OFFLINE_UPLOAD = 0x05,

                /// <summary>
                /// OfflinePerso Keys: used for offline encipher operations: for
                /// example for personalization SAM. These keys are restricted to
                /// AES.
                /// </summary>
                OFFLINE_PERSO = 0x06,
            }
            #endregion

            #region SET Configuration
            /// <summary> Helper Class to frame SET configuration information. </summary>
            public class Set
            {
                private int dwSet;

                public Set ()
                {

                }

                public Set ( int dwSet )
                {
                    this.dwSet = dwSet;
                }

                /// <summary>
                /// Extracts the Options and returns as an object of the class.
                /// </summary>
                ///
                /// <param name="dwSet">The options that are configured.</param>
                public static implicit operator Set ( int dwSet )
                {
                    Set oObject = new Set ();
                    oObject.dwSet = dwSet;
                    return oObject;
                }

                /// <summary>
                /// Combines the values of the properties and returns the option information
                /// as one single integer.
                /// </summary>
                /// <param name="oSet">Object of the class having the properties configured.</param>
                public static implicit operator int ( Set oSet )
                {
                    return oSet.dwSet;
                }

                /// <summary>
                /// Allow dump session key / MIFARE key (from standard host)
                /// This set Bit[0] of SET Configuration settings. The commands
                /// <see cref="Cmd_SAM_DumpSessionKey"/>, for plain key dump, is allowed for
                /// session keys based on this key entry.
                /// </summary>
                public bool AllowDumpSessionKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0001;
                        else dwSet &= ~0x0001;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0001 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Keep IV
                /// This set Bit[2] of SET Configuration settings. After finishing a cryptographic
                /// command the init vector (IV) will be kept and not reset.
                /// </summary>
                public bool KeepIV
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0004;
                        else dwSet &= ~0x0004;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0004 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Keep Type
                /// This set Bit[3 - 6] of SET Configuration settings.
                /// </summary>
                public KeyType KeyType
                {
                    set
                    {
                        dwSet &= 0xFF87;
                        dwSet |= ( int ) value;
                    }
                    get
                    {
                        return ( KeyType ) ( dwSet & 0x0078 );
                    }
                }

                /// <summary>
                /// Programmable Logic Key
                /// This set Bit[7] of SET Configuration settings. When this bit is set in SAM Master Key,
                /// it indicates for HOST Keys whether this key entry can be used to get access to
                /// <see cref="Cmd_SAM_PLExec"/> command.
                /// </summary>
                public bool PLKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0080;
                        else dwSet &= ~0x0080;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0080 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Authentication Key
                /// This set Bit[8] of SET Configuration settings. Indicates for Host Keys whether this key
                /// entry can be used to execute the <see cref="Cmd_SAM_AuthenticateHost"/> Host command.
                /// </summary>
                public bool AuthKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0100;
                        else dwSet &= ~0x0100;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0100 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable key entry
                /// This set Bit[9] of SET Configuration settings. The key entry is disabled and can be
                /// reactivated by a <see cref="Cmd_SAM_ChangeKeyEntry"/> command (default configuration).
                /// </summary>
                public bool DisableKeyEntry
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0200;
                        else dwSet &= ~0x0200;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0200 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Lock Key
                /// This set Bit[10] of SET Configuration settings. Indicates for Host Keys whether this key
                /// entry can be used to execute the <see cref="Cmd_SAM_LockUnlock"/> Host command.
                /// </summary>
                public bool LockKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0400;
                        else dwSet &= ~0x0400;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0400 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable writing the key to a PICC
                /// This set Bit[11] of SET Configuration settings. Indicates if the key stored in the key
                /// entry can be downloaded into a PICC
                /// </summary>
                public bool DisableChangeKeyPICC
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0800;
                        else dwSet &= ~0x0800;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x0800 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable decryption
                /// This set Bit[12] of SET Configuration settings. Indicates if the key entry can be used to
                /// decrypt data with <see cref="Cmd_SAM_DecipherData"/> and
                /// <see cref="Cmd_SAM_DecipherOfflineData"/> commands
                /// </summary>
                public bool DisableDecryption
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x1000;
                        else dwSet &= ~( int ) 0x1000;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x1000 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable encryption
                /// This set Bit[13] of SET Configuration settings. Indicates if the key entry can be used to
                /// encrypt data with <see cref="Cmd_SAM_EncipherData"/> and
                /// <see cref="Cmd_SAM_EncipherOfflineData"/> commands
                /// </summary>
                public bool DisableEncryption
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x2000;
                        else dwSet &= ~( int ) 0x2000;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x2000 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable verify MAC
                /// This set Bit[14] of SET Configuration settings. Indicates if the key entry can be used to
                /// verify a MAC with <see cref="Cmd_SAM_VerifyMAC"/> and <see cref="Cmd_SAM_RemoveSM"/>
                /// commands
                /// </summary>
                public bool DisableVerifyMac
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x4000;
                        else dwSet &= ~( int ) 0x4000;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x4000 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable generate MAC
                /// This set Bit[15] of SET Configuration settings. Indicates if the key entry can be used to
                /// verify a MAC with <see cref="Cmd_SAM_GenerateMAC"/> and <see cref="Cmd_SAM_ApplySM"/>
                /// commands
                /// </summary>
                public bool DisableGenerateMac
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x8000;
                        else dwSet &= ~( int ) 0x8000;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x8000 ) != 0x0000 );
                    }
                }
            }
            #endregion

            #region ExtSET Configuration
            /// <summary> Helper Class to frame Extended SET configuration information. </summary>
            public class ExtSet
            {
                int dwExtSet;

                /// <summary>
                /// Extracts the Options and returns as an object of the class.
                /// </summary>
                ///
                /// <param name="dwExtSet">The options that are configured.</param>
                public static implicit operator ExtSet ( int dwExtSet )
                {
                    ExtSet oObject = new ExtSet ();
                    oObject.dwExtSet = dwExtSet;
                    return oObject;
                }

                /// <summary>
                /// Combines the values of the properties and returns the option information
                /// as integer.
                /// </summary>
                /// <param name="oExtSet">Object of the class having the properties configured.</param>
                public static implicit operator int ( ExtSet oExtSet )
                {
                    return oExtSet.dwExtSet;
                }

                /// <summary>
                /// Key class
                /// This set Bit[0 - 2] of ExtSET Configuration settings.
                /// </summary>
                public KeyClass KeyClass
                {
                    set
                    {
                        dwExtSet &= 0x00F8;
                        dwExtSet |= ( int ) value;
                    }
                    get
                    {
                        return ( KeyClass ) ( dwExtSet & 0x0007 );
                    }
                }

                /// <summary>
                /// Allow dump secret key (from standard host)
                /// This set Bit[3] of ExtSET Configuration settings. The command
                /// <see cref="Cmd_SAM_DumpSecretKey"/> is allowed for this key entry.
                /// </summary>
                public bool AllowDumpSecret
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0008;
                        else dwExtSet &= ~0x0008;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0008 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Diversification restriction
                /// This set Bit[4] of ExtSET Configuration settings. Diversification when using
                /// this key entry is mandatory. The stored value of this key will only be used
                /// directly to compute a diversified key.Use for other cryptographic operations
                /// is impossible.
                /// </summary>
                public bool DiversifiedRestriction
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0010;
                        else dwExtSet &= ~0x0010;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0010 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Reserved for personalization SAM
                /// This set Bit[5] of ExtSET Configuration settings. The key entry is reserved for
                /// personalization and is disabled for any other operations. The key can only be
                /// used for inclusion by <see cref="Cmd_SAM_EncipherKeyEntry"/> and
                /// <see cref="Cmd_PKI_EncipherKeyEntries"/>. The values of the change entry key
                /// ( KeyNoCEK and KeyVerCEK) are ignored
                /// </summary>
                public bool PersonalizeSAM
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0020;
                        else dwExtSet &= ~0x0020;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0020 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Force key usage by Internal Host
                /// This set Bit[8] of ExtSET Configuration settings. Indicates whether the usage of
                /// this key entry are restricted to the Internal Host. When enabled, the key entry
                /// are considered disabled when accessed from the standard host.
                /// </summary>
                public bool ForceKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0100;
                        else dwExtSet &= ~0x0100;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0100 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Force key change by Internal Host
                /// This set Bit[9] of ExtSET Configuration settings. Indicates whether the change of
                /// this key entry are restricted to the Internal Host. When enabled, the key entry
                /// cannot be changed from the standard host.
                /// </summary>
                public bool ForceKeyChangeInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0200;
                        else dwExtSet &= ~0x0200;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0200 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Force session usage by Internal Host
                /// This set Bit[10] of ExtSET Configuration settings. Indicates whether the PICC session
                /// key derived from the usage of this key entry are restricted to the Internal Host. When
                /// enabled, the PICC session key entry is not usable by the standard host. In other words,
                /// the PICC secure messaging is controlled by the Programmable Logic.
                /// </summary>
                public bool ForceSessionKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0400;
                        else dwExtSet &= ~0x0400;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0400 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Allow dump secret key by Internal Host
                /// This set Bit[11] of ExtSET Configuration settings. When executed by the Internal Host,
                /// <see cref="Cmd_SAM_DumpSecretKey"/> command is allowed for this key entry
                /// </summary>
                public bool AllowDumpSecretKeyInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0800;
                        else dwExtSet &= ~0x0800;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x0800 ) != 0x00 );
                    }
                }

                /// <summary>
                /// Allow dump session key by Internal Host
                /// This set Bit[12] of ExtSET Configuration settings. When executed by the Internal Host,
                /// <see cref="Cmd_SAM_DumpSessionKey/> command is allowed for this key entry.
                /// </summary>
                public bool AllowDumpSessionKeyInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x1000;
                        else dwExtSet &= ~0x1000;
                    }
                    get
                    {
                        return ( ( dwExtSet & 0x1000 ) != 0x00 );
                    }
                }
            }
            #endregion

            #region Key Entries
            /// <summary>
            /// Key (version a). The property KeyVa stores
            ///     - in 16-bytes field a DES, a 2TDEA or an AES_128 key format
            ///     - in 24-bytes field a 3TDEA or an AES_192 key format
            ///     - in 32-bytes field an AES_256 key format
            /// The property KeyVa of the KST RAM Key Entry is not be preserved on a SAM reset and is
            /// only stored in volatile RAM.
            /// </summary>
            public byte[] KeyA;

            /// <summary>
            /// Key (version b). The property KeyVb stores
            ///     - in 16-bytes field a DES, a 2TDEA or an AES_128 key format.Key ( version b ) starts after
            ///       the 16-bytes of KeyVa.
            ///     - in 24-bytes field a 3TDEA or an AES_192 key format.Key ( version b ) starts after the 24-bytes
            ///       of KeyVa.
            ///     - in 32-bytes field Key ( version b ) is not present, all key value storage is reserved for one
            ///       32-bytes key.
            /// The property KeyVb is not present in KST RAM Key Entry.
            /// </summary>
            public byte[] KeyB;

            /// <summary>
            /// Key (version c). The property KeyVc stores
            ///     - in 16-bytes field a DES, a 2TDEA or an AES_128 key format.Key ( version c ) starts after the
            ///       16-bytes of KeyVb.
            ///     - in 24-bytes field Key ( version c ) is not present, all key value storage is reserved for two
            ///       24-bytes keys.
            ///     - in 32-bytes field Key (version c ) is not present, all key value storage is reserved for one
            ///       32-bytes key.
            /// The property KeyVc is not present in KST RAM Key Entry
            /// </summary>
            public byte[] KeyC;

            /// <summary>
            /// Corresponding DESFire AID
            /// The 24-bit unsigned integer holds the DESFire AID the key entry belongs to. The DF_AID field is used
            /// by the <see cref="Cmd_SAM_SelectApplication"/>  command to pre-select keys for 3-pass authentication.
            /// The property DF_AID is not present in KST RAM Key Entry.
            /// </summary>
            public byte[] DF_AID;

            /// <summary>
            /// Corresponding DESFire key number
            /// According to the MIFARE DESFire data sheets, the DF_KeyNo is defined in the range 00h to 0Dh.
            /// The DF_KeyNo field is used by the command SAM_SelectApplication to build a table of valid keys
            /// for the selected DF_AID. The property DF_AID is not present in KST RAM Key Entry.
            /// </summary>
            public byte DF_KeyNo;

            /// <summary>
            /// Key reference of change key.
            /// The two 1-byte field KeyNoCEK specify the symmetric key entry required to change
            /// the key entry. The referenced change entry key is used to control the key entry update with
            /// <see cref="Cmd_SAM_ChangeKeyEntry"/> and <see cref="Cmd_SAM_DisableKeyEntry"/> commands, both
            /// with support for OfflineChange.
            ///     - The default KeyNoCEK value is set to 0x00 (MasterKey).
            ///     - The KeyNoCEK value FEh disables the need for authentication for key load.
            ///     - The KeyNoCEK value FFh irreversibly locks the entire key entry.
            /// Note that for key entries to be used for personalizing other SAMs (that is if ExtSETBit is set),
            /// the change key entry will need to be aligned between target SAM and personalization SAM.
            /// </summary>
            public byte KeyNoCEK;

            /// <summary>
            /// Key version of change key.
            /// The two 1-byte field KeyVerCEK specify the symmetric key entry required to change
            /// the key entry. The referenced change entry key is used to control the key entry update with
            /// <see cref="Cmd_SAM_ChangeKeyEntry"/> and <see cref="Cmd_SAM_DisableKeyEntry"/> commands, both
            /// with support for OfflineChange.
            ///     - The default KeyVerCEK value is set to 0x00 (MasterKey).
            /// Note that for key entries to be used for personalizing other SAMs (that is if ExtSETBit is set),
            /// the change key entry will need to be aligned between target SAM and personalization SAM.
            /// </summary>
            public byte KeyVCEK;

            /// <summary>
            /// Reference number of key usage counter
            /// The 1-byte field RefNoKUC holds the reference number of the key usage counter entry controlling
            /// the key usage. The referenced key usage counter entry is automatically incremented each time a
            /// defined key entry is used for most authentication or for some Offline operations.
            /// </summary>
            public byte RefNoKUC;

            /// <summary>
            /// Version of key a
            /// This 1-byte field (Va) are used to store the versions of Key version a (KeyVa).
            /// When issuing a <see cref="Cmd_SAM_ChangeKeyEntry"/> command, this field is optional
            /// if a TDEA key is to be updated in the MIFARE SAM.
            /// </summary>
            public byte VersionA;

            /// <summary>
            /// Version of key b
            /// This 1-byte field (Vb) are used to store the versions of Key version b (KeyVb).
            /// When issuing a <see cref="Cmd_SAM_ChangeKeyEntry"/> command, this field is optional
            /// if a TDEA key is to be updated in the MIFARE SAM.
            /// </summary>
            public byte VersionB;

            /// <summary>
            /// Version of key c
            /// This 1-byte field (Vc) are used to store the versions of Key version c (KeyVc).
            /// When issuing a <see cref="Cmd_SAM_ChangeKeyEntry"/> command, this field is optional
            /// if a TDEA key is to be updated in the MIFARE SAM.
            /// </summary>
            public byte VersionC;

            /// <summary>
            /// Key reference of access key.
            /// The two 1-byte field KeyNoAEK specify the symmetric key entry required to access the
            /// key. The referenced access entry key is used to lock the key entry usage to specific host.
            /// The default KeyNoAEK value is set to 0xFE. For HOST Key Class, it is not be possible to set
            /// a value other than 0xFE.
            /// </summary>
            public byte KeyNoAEK;

            /// <summary>
            ///
            /// Key version of access key.
            /// The two 1-byte field KeyVerAEK specify the symmetric key entry required to access the
            /// key entry.
            /// </summary>
            public byte KeyVAEK;

            /// <summary>
            /// SET configuration settings.
            /// </summary>
            public Set SET;

            /// <summary>
            /// Extended SET (ExtSET) configuration settings.
            /// </summary>
            public ExtSet ExtSET;
            #endregion

            /// <summary>
            /// Constructor
            ///     - SET, ExtSET, DF_AID are configured to default zero.
            ///     - RefKUCNo is configured to 0xFF
            ///     - KeyNoCEK to 0xFE
            ///     - KeyNoAEK to 0xFF
            /// </summary>
            public KeyEntry ()
            {
                SET = new Set ();
                ExtSET = new ExtSet ();

                DF_AID = new byte[3];

                RefNoKUC = 0xFF;
                KeyNoCEK = 0xFE;
                KeyNoAEK = 0xFE;
            }

            /// <summary>
            /// Constructor
            ///     - SET KeyType is configured to user provided one and reset are configured to zero.
            ///     - ExtSET KeyClass is configured to user provided one and reset are configured to zero.
            ///     - Key buffers (KeyA, KeyB and KeyC) are initialized based on Key Type.
            ///     - VersionA is configured to 0,
            ///     - VersionB is configured to 1 if not RAM Key else zero based on Key Type and if RAM Key.
            ///     - VersionC is configured to 2 if not Ram Key else zero based on Key Type and if RAM Key.
            ///     - RefKUCNo is configured to 0xFF
            ///     - KeyNoCEK to 0xFE
            ///     - KeyNoAEK to 0xFF
            /// </summary>
            ///
            /// <param name="eKeyType">Key Type to use for framing KeyEntry. </param>
            /// <param name="eKeyClass">Key Class to use for framing KeyEntry. </param>
            /// <param name="boIsRamKey">Is the Key a NVM or RAM Key
            ///                             - NVM Key: False
            ///                             - Ram Key: True
            /// </param>
            public KeyEntry ( KeyType eKeyType, KeyClass eKeyClass, bool boIsRamKey = false )
            {
                SET = new Set ();
                SET.KeyType = eKeyType;

                ExtSET = new ExtSet ();
                ExtSET.KeyClass = eKeyClass;

                if ( eKeyClass.Equals ( KeyClass.HOST ) )
                    SET.AuthKey = true;

                VersionA = 0x00;
                VersionB = ( byte ) ( boIsRamKey ? 0x00 : 0x01 );
                VersionC = ( byte ) ( boIsRamKey ? 0x00 : 0x02 );

                if ( eKeyType.Equals ( KeyType.AES_256 ) )
                {
                    KeyA = new byte[48];
                    VersionB = 0x00;
                    VersionC = 0x00;
                }
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.TDEA3 ) )
                {
                    KeyA = new byte[24];
                    KeyB = new byte[24];
                    VersionC = 0x00;
                }
                else
                {
                    KeyA = new byte[16];
                    KeyB = new byte[16];
                    KeyC = new byte[16];
                }

                DF_AID = new byte[3];

                RefNoKUC = 0xFF;
                KeyNoCEK = 0xFE;
                KeyNoAEK = 0xFE;
            }

            /// <summary>
            /// Constructor
            /// Extracts information from \b aKeyInfoBuffer and updates the properties of this
            /// class accordingly.
            /// </summary>
            ///
            /// <param name="aKeyEntry">Buffer containing the Key information.</param>
            /// <param name="boIsNewFormat">If the Key Entry AV2 or AV3 (Future SAM's) format.</param>
            /// <param name="boIsRamKey">Is the Key a NVM or RAM Key
            ///                             - NVM Key: False
            ///                             - Ram Key: True
            /// </param>
            public KeyEntry ( byte[] aKeyEntry, bool boIsRamKey = false )
            {
                byte bOffset = 0;

                SET = new Set ();
                bOffset = ( byte ) ( aKeyEntry.Length - 6 );
                SET = ( aKeyEntry[bOffset + 1] << 8 ) | aKeyEntry[bOffset];

                ExtSET = new ExtSet ();
                bOffset = ( byte ) ( aKeyEntry.Length - 4 );
                ExtSET = ( aKeyEntry[bOffset + 1] << 8 ) | aKeyEntry[bOffset];

                DF_AID = new byte[3];

                if ( SET.KeyType.Equals ( KeyType.AES_256 ) )
                    KeyA = new byte[48];
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.TDEA3 ) )
                {
                    KeyA = new byte[24];
                    KeyB = new byte[24];
                }
                else
                {
                    KeyA = new byte[16];
                    KeyB = new byte[16];
                    KeyC = new byte[16];
                }

                if ( !boIsRamKey )
                {
                    bOffset = 0;
                    VersionA = aKeyEntry[bOffset++];
                    VersionB = aKeyEntry[bOffset++];

                    if ( !SET.KeyType.Equals ( KeyType.AES_256 ) && !SET.KeyType.Equals ( KeyType.AES_192 ) &&
                        !SET.KeyType.Equals ( KeyType.TDEA3 ) )
                        VersionC = aKeyEntry[bOffset++];

                    Array.Copy ( aKeyEntry, bOffset, DF_AID, 0, 3 );
                    bOffset += 3;

                    DF_KeyNo = aKeyEntry[bOffset++];
                }

                if ( boIsRamKey )
                    bOffset = 0;

                KeyNoCEK = aKeyEntry[bOffset++];
                KeyVCEK = aKeyEntry[bOffset++];
                RefNoKUC = aKeyEntry[bOffset++];

                bOffset += 4;
                KeyNoAEK = aKeyEntry[bOffset++];
                KeyVAEK = aKeyEntry[bOffset++];
            }

            /// <summary>
            /// Converts the Key Entry information available as object to bytes.
            /// </summary>
            ///
            /// <returns>Key Entry information available as object to bytes</returns>
            public byte[] ToByteArray ()
            {
                byte bOffset = 0;
                byte[] aKeyInfoBuffer = new byte[64];

                if ( SET.KeyType.Equals ( KeyType.AES_256 ) )
                    Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 48 );
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.TDEA3 ) )
                {
                    Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 24 );
                    Array.Copy ( KeyB, 0, aKeyInfoBuffer, 24, 24 );
                }
                else
                {
                    Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 16 );
                    Array.Copy ( KeyB, 0, aKeyInfoBuffer, 16, 16 );
                    Array.Copy ( KeyC, 0, aKeyInfoBuffer, 32, 16 );
                }
                bOffset = 48;

                Array.Copy ( DF_AID, 0, aKeyInfoBuffer, bOffset, 3 );
                bOffset += 3;

                aKeyInfoBuffer[bOffset++] = DF_KeyNo;
                aKeyInfoBuffer[bOffset++] = KeyNoCEK;
                aKeyInfoBuffer[bOffset++] = KeyVCEK;
                aKeyInfoBuffer[bOffset++] = RefNoKUC;
                aKeyInfoBuffer[bOffset++] = ( byte ) ( SET );
                aKeyInfoBuffer[bOffset++] = ( byte ) ( SET >> 8 );
                aKeyInfoBuffer[bOffset++] = VersionA;
                aKeyInfoBuffer[bOffset++] = VersionB;
                aKeyInfoBuffer[bOffset++] = VersionC;

                aKeyInfoBuffer[bOffset++] = ( byte ) ExtSET;
                aKeyInfoBuffer[bOffset++] = ( byte ) ( ExtSET >> 8 );
                aKeyInfoBuffer[bOffset++] = KeyNoAEK;
                aKeyInfoBuffer[bOffset++] = KeyVAEK;

                return aKeyInfoBuffer;
            }
        }
        #endregion
        #endregion

        #region Cmd.SAM_ChangeKUCEntry
        #region Program Mask bits
        /// <summary>
        /// Helper Class containing the Options (Non-volatile programming mask) to be
        /// framed and utilized by <see cref="Cmd_SAM_ChangeKUCEntry"/> and
        /// <see cref="Cmd_SAM_ChangeKUCEntryOffline"/> interfaces.
        /// </summary>
        public class ChangeKUCEntry_ProMas
        {
            private int dwProMask;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator ChangeKUCEntry_ProMas ( byte bProMask )
            {
                ChangeKUCEntry_ProMas oObject = new ChangeKUCEntry_ProMas ();
                oObject.dwProMask = bProMask;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( ChangeKUCEntry_ProMas oProMask )
            {
                return ( byte ) oProMask.dwProMask;
            }

            /// <summary>
            /// Update limit.
            /// This set Bit[7] of P2 byte
            /// </summary>
            public bool UpdateLimit
            {
                set
                {
                    if ( value ) dwProMask |= 0x80;
                    else dwProMask &= ~0x80;
                }
                get
                {
                    return ( ( dwProMask & 0x80 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update KeyNoCKUC.
            /// This set Bit[6] of P2 byte
            /// </summary>
            public bool UpdateKeyNoCKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x40;
                    else dwProMask &= ~0x40;
                }
                get
                {
                    return ( ( dwProMask & 0x40 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update KeyVCKUC.
            /// This set Bit[5] of P2 byte
            /// </summary>
            public bool UpdateKeyVCKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x20;
                    else dwProMask &= ~0x20;
                }
                get
                {
                    return ( ( dwProMask & 0x20 ) != 0x00 );
                }
            }
        }
        #endregion

        #region KUCEntry
        /// <summary>
        /// Helper Class to frame Key Usage Counter Entry buffer as bytes and also extract various Key
        /// information from byte array. To be used by <see cref="Cmd_SAM_ChangeKUCEntry"/>
        /// interface.
        /// </summary>
        public class KucEntry
        {
            /// <summary>
            /// The Limit field stores the current limit for this key usage counter. It is only possible
            /// to use a key that is linked to this counter for authentication if the \b CurrentValue is
            /// smaller than the current limit. As soon as the \b CurrentValue is equal to, or higher than,
            /// the current limit, the usage of all key entries linked to this counter is prohibited.
            /// If the limit is changed to a value lower than the current value, the usage of all key
            /// entries linked to this counter is prohibited.
            /// Initial limit is set to 0xFFFFFFFF.
            /// </summary>
            public uint Limit;

            /// <summary>
            /// Key reference number of change key.
            /// The two 1-byte field KeyNoCKUC specify the symmetric key entry required to change the KUC entry.
            /// The access control mechanism to update an KUC entry is identical to the one for the update of a
            /// key entry. The referenced change usage counter entry key is used to control the KUC entry update
            /// with <see cref="Cmd_SAM_ChangeKUCEntry"/> command and with support for OfflineChange.
            /// </summary>
            public byte KeyNoCKUC;

            /// <summary>
            /// Key reference version of change key.
            /// The two 1-byte fields KeyVerCKUC specify the symmetric key entry required to change the KUC entry.
            /// The access control mechanism to update an KUC entry is identical to the one for the update of a
            /// key entry. The referenced change usage counter entry key is used to control the KUC entry update
            /// with <see cref="Cmd_SAM_ChangeKUCEntry"/> command and with support for OfflineChange.
            /// </summary>
            public byte KeyVCKUC;

            /// <summary>
            /// The CurrentValue field stores the current value of this key usage counter. It is possible to use all
            /// keys referring to this counter for authentication only if the current value is smaller than the
            /// current limit. As soon as the current value is equal to, or higher than, the LimVal, the usage of all
            /// key entries linked to this counter is prohibited.
            /// </summary>
            public uint CurrentValue;

            /// <summary>
            /// Constructor
            ///     - Limit is configure to 0xFFFFFFFF.
            ///     - KeyNoCKUC to 0xFE
            ///     - KeyVCKUC to 0xFF
            /// </summary>
            public KucEntry ()
            {
                Limit = 0xFFFFFFFF;
                KeyNoCKUC = 0xFE;
                KeyVCKUC = 0x00;
            }

            /// <summary>
            /// Constructor
            /// Extracts information from \b aKuc and updates the properties of this
            /// class accordingly.
            /// </summary>
            ///
            /// <param name="aKuc">Buffer containing Key Usage Counter information.</param>
            public KucEntry ( byte[] aKuc )
            {
                if ( aKuc == null )
                    return;

                if ( aKuc.Length < 10 )
                    return;

                Limit = ( uint ) ( ( aKuc[3] << 24 ) | ( aKuc[2] << 16 ) | ( aKuc[1] << 8 ) | aKuc[0] );
                KeyNoCKUC = aKuc[4];
                KeyVCKUC = aKuc[5];
                CurrentValue = ( uint ) ( ( aKuc[9] << 24 ) | ( aKuc[8] << 16 ) | ( aKuc[7] << 8 ) | aKuc[6] );
            }

            /// <summary>
            /// Converts the Key Usage Counter Entry information available as object to bytes.
            /// </summary>
            ///
            /// <returns>Key Usage Counter Entry information available as object to bytes</returns>
            public byte[] ToByteArray ()
            {
                byte[] aKUCEntryBuffer = new byte[6];
                aKUCEntryBuffer[0] = ( byte ) ( Limit );
                aKUCEntryBuffer[1] = ( byte ) ( Limit >> 8 );
                aKUCEntryBuffer[2] = ( byte ) ( Limit >> 16 );
                aKUCEntryBuffer[3] = ( byte ) ( Limit >> 24 );
                aKUCEntryBuffer[4] = KeyNoCKUC;
                aKUCEntryBuffer[5] = KeyVCKUC;

                return aKUCEntryBuffer;
            }
        }
        #endregion KUCEntry
        #endregion

        #region Cmd.SAM_DumpSecretKey
        public class DumpSecretKey_Crypto
        {
            private int dwCrypto;

            public static implicit operator DumpSecretKey_Crypto ( byte bCrypto )
            {
                DumpSecretKey_Crypto MyInstance = new DumpSecretKey_Crypto ();
                MyInstance.dwCrypto = bCrypto;
                return MyInstance;
            }

            public static implicit operator byte ( DumpSecretKey_Crypto oCrypto )
            {
                return ( byte ) oCrypto.dwCrypto;
            }

            public bool EncryptedDump
            {
                set
                {
                    if ( value ) dwCrypto |= 0x01;
                    else dwCrypto &= ~0x01;
                }
                get
                {
                    if ( ( dwCrypto & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool DiversifiedDump
            {
                set
                {
                    if ( value ) dwCrypto |= 0x02;
                    else dwCrypto &= ~0x02;
                }
                get
                {
                    if ( ( dwCrypto & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion Cmd.SAM_DumpSecretKey

        #endregion Key Management

        #region Data Processing
        /// <summary>
        /// Helper Class containing the Number of MAC bytes to be processed and Truncation Mode
        /// to be applied and utilized by <see cref="Cmd_SAM_GenerateMAC"/> and
        /// <see cref="Cmd_SAM_VerifyMAC"/> interfaces.
        /// </summary>
        public class VerifyGenerateMAC_Num
        {
            private int dwNum;

            /// <summary>
            /// Extracts the Number of MAC bytes to be processed and
            /// returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bNum">The Number of MAC bytes to be processed.</param>
            public static implicit operator VerifyGenerateMAC_Num ( byte bNum )
            {
                VerifyGenerateMAC_Num oObject = new VerifyGenerateMAC_Num ();
                oObject.dwNum = bNum;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the Number of MAC
            /// bytes to be processed information as one single byte.
            /// </summary>
            /// <param name="oNum">Object of the class having the properties configured.</param>
            public static implicit operator byte ( VerifyGenerateMAC_Num oNum )
            {
                return ( byte ) oNum.dwNum;
            }

            /// <summary>
            /// Truncation mode.
            /// This set Bit[7] of P2 byte
            /// </summary>
            public bool UseMFPTruncation
            {
                set
                {
                    if ( value ) { dwNum = ( int ) TruncationMode.MFP; }
                    else { dwNum &= ( int ) ~TruncationMode.MFP; }
                }
                get
                {
                    return ( ( dwNum & ( int ) TruncationMode.MFP ) != ( int ) TruncationMode.MFP );
                }
            }

            /// <summary>
            /// Number of MAC bytes to be processed.
            /// This set Bit[4 - 0] of P2 byte
            /// </summary>
            public byte NumOfMacBytes
            {
                set
                {
                    dwNum &= 0xE0;
                    dwNum |= ( byte ) ( value & 0x1F );
                }
                get
                {
                    return ( byte ) ( ( dwNum & 0x1F ) );
                }
            }
        }
        #endregion

        #region Public Key Infrastructure
        #region KeyEntry
        /// <summary>
        /// Helper Class to frame Key Entry buffer as bytes and also extract various Key
        /// information from byte array. To be used by
        ///     - <see cref="Cmd_PKI_ExportPrivateKey"/>
        ///     - <see cref="Cmd_PKI_ExportPublicKey"/>
        /// interface.
        /// </summary>
        public class PKI_RSA_KeyEntry
        {
            #region SET Configuration
            /// <summary> Helper Class to frame SET configuration information. </summary>
            public class Set
            {
                private int dwSet;

                /// <summary>
                /// Extracts the Options and returns as an object of the class.
                /// </summary>
                ///
                /// <param name="dwSet">The options that are configured.</param>
                public static implicit operator Set ( ushort dwSet )
                {
                    Set oObject = new Set ();
                    oObject.dwSet = dwSet;
                    return oObject;
                }

                /// <summary>
                /// Combines the values of the properties and returns the option information
                /// as one single integer.
                /// </summary>
                /// <param name="oSet">Object of the class having the properties configured.</param>
                public static implicit operator ushort ( Set oSet )
                {
                    return ( ushort ) oSet.dwSet;
                }

                /// <summary>
                /// Private key included in key entry
                /// This set Bit[0] of SET Configuration settings. The key entry contains both the
                /// public and the private key. This is a possible setting for entries 0x00 - 0x01.
                /// </summary>
                public bool IncludePrivateKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0001;
                        else dwSet &= ~0x0001;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0001 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Allow private key export
                /// This set Bit[1] of SET Configuration settings. The command <see cref="Cmd_PKI_ExportPrivateKey"/>
                /// if used for exporting the private key will succeed on this key entry if a valid host authentication
                /// with the change key of this entry is active. Note that this is different compared to the dumping of
                /// secret symmetric keys, as private key export should only be allowed for back-up reasons after the key
                /// was internally created by the SAM itself.
                /// </summary>
                public bool AllowPrivateKeyExport
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0002;
                        else dwSet &= ~0x0002;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0002 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable key entry
                /// This set Bit[2] of SET Configuration settings. The key entry is disabled and can only be reactivated
                /// by a <see cref="Cmd_PKI_ImportKey"/> or <see cref="Cmd_PKI_GenerateKeyPair"/> command (default
                /// configuration).
                /// </summary>
                public bool DisableKeyEntry
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0004;
                        else dwSet &= ~0x0004;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0004 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable encryption handling
                /// This set Bit[3] of SET Configuration settings. This key entry cannot be used for encryption and
                /// decryption operations with the following commands: <see cref="Cmd_PKI_UpdateKeyEntries"/>,
                /// <see cref="Cmd_PKI_EncipherKeyEntries"/>, <see cref="Cmd_PKI_EncipherData"/> and
                /// <see cref="Cmd_PKI_DecipherData"/>.
                /// </summary>
                public bool DisableEncryptionHandling
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0008;
                        else dwSet &= ~0x0008;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0008 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Disable signature handling
                /// This set Bit[4] of SET Configuration settings. This key entry cannot be used for signature
                /// verification and generation operation with the following commands: <see cref="Cmd_PKI_UpdateKeyEntries"/>,
                /// <see cref="Cmd_PKI_EncipherKeyEntries"/>, <see cref="Cmd_PKI_GenerateSignature"/> and
                /// <see cref="Cmd_PKI_VerifySignature"/>.
                /// </summary>
                public bool DisableSignatureHandling
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0010;
                        else dwSet &= ~0x0010;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0010 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Enable PKI_UpdateKeyEntries
                /// This set Bit[5] of SET Configuration settings. This key can be used for the protection of the
                /// <see cref="Cmd_PKI_UpdateKeyEntries"/>, but cannot be used for PKI offline crypto functions using
                /// the private key: <see cref="Cmd_PKI_DecipherData"/> and <see cref="Cmd_PKI_GenerateSignature"/>
                /// commands. PKI offline crypto functions are not allowed to prevent generation of fake UpdateAck and
                /// decryption of the new key entries cryptogram ( EncKeyFrame). Note that the presence of a private key
                /// in the entry and the setting of the other configuration options define whether the key can be used
                /// for decryption or for signature verification.
                /// </summary>
                public bool EnableUpdateKeyEntries
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0020;
                        else dwSet &= ~0x0020;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0020 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Private key representation (only relevant if bit 0 is set to 1)
                /// This set Bit[6] of SET Configuration settings. The private key is represented by CRT representation.
                /// </summary>
                public bool PrivateKeyRepresentation
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0040;
                        else dwSet &= ~0x0040;
                    }
                    get
                    {
                        return ( ( dwSet & 0x0040 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Enable PKI_EncipherKeyEntries
                /// This set Bit[7] of SET Configuration settings. This key can be used for the protection of the
                /// <see cref="Cmd_PKI_EncipherKeyEntries"/>, but cannot be used for PKI offline crypto functions using the
                /// private key: <see cref="Cmd_PKI_DecipherData"/> and <see cref="Cmd_PKI_GenerateSignature"/>. PKI offline
                /// crypto functions are not allowed to prevent generation of fake Signature and decryption of the new key
                /// entries cryptogram ( EncKeyFrame). Note that the presence of a private key in the entry and the setting
                /// of the other configuration options define whether the key can be used for encryption or for signature
                /// generation.
                /// </summary>
                public bool EnableEncipherKeyEntries
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0080;
                        else dwSet &= ~0x0080;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x0080 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Force key usage by Internal Host
                /// This set Bit[8] of SET Configuration settings. Indicates whether the usage of this key entry are restricted
                /// to the Internal Host. When enabled, the key entry are considered disabled when accessed from the standard
                /// host.
                /// </summary>
                public bool ForceKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x0100;
                        else dwSet &= ~( int ) 0x0100;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x0100 ) != 0x0000 );
                    }
                }

                /// <summary>
                /// Force key change by Internal Host
                /// This set Bit[9] of SET Configuration settings. Indicates whether the change of this key entry are restricted
                /// to the Internal Host. When enabled, the key entry cannot be changed from the standard host.
                /// </summary>
                public bool ForceKeyChangeInternalHost
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x0200;
                        else dwSet &= ~( int ) 0x0200;
                    }
                    get
                    {
                        return ( ( dwSet & ( int ) 0x0200 ) != 0x0000 );
                    }
                }
            }
            #endregion

            #region Key Entries
            #region Public Key
            /// <summary>
            /// The RSA key length is restricted to be between 512 and 2048 bit. This means PKI_NLen are
            /// between 64 and 256 (bytes). Note however that the actual operations put extra requirements
            /// on the minimal acceptable length for the key to be usable
            /// </summary>
            public ushort NLen;

            /// <summary>
            /// the length of the public exponent. It are between 4 and 256 bytes and shall not be greater
            /// than PKI_NLen bytes.
            /// </summary>
            public ushort eLen;

            /// <summary>
            /// The modulus N is the product of two distinct primes p and q. It are of length PKI_NLen.
            /// </summary>
            public byte[] N;

            /// <summary>
            /// The public exponent e. It are PKI_eLen bytes. e are odd. To have a low execution time,
            /// it is very common to use a short prime for e, but due to security aspects it shall not
            /// be too small. The value 65537 for e has proven to be a good choice.
            /// </summary>
            public byte[] e;
            #endregion

            #region Private Key
            /// <summary>
            ///
            /// </summary>
            public ushort pLen;

            /// <summary>
            ///
            /// </summary>
            public ushort qLen;

            /// <summary>
            /// The prime p. It are PKI_NLen/2 bytes, if created with <see cref="Cmd_PKI_GenerateKeyPair"/>;
            /// if imported with <see cref="Cmd_PKI_ImportKey"/>, the length is given by the user.
            /// </summary>
            public byte[] p;

            /// <summary>
            /// The prime q. It are PKI_NLen/2 bytes, if created with <see cref="Cmd_PKI_GenerateKeyPair"/>;
            /// if imported with <see cref="Cmd_PKI_ImportKey"/>, the length is given by the user.
            /// </summary>
            public byte[] q;

            /// <summary>
            /// The private exponent dp. It are PKI_NLen/2 bytes, if created with <see cref="Cmd_PKI_GenerateKeyPair"/>;
            /// if imported with <see cref="Cmd_PKI_ImportKey"/>, the length is given by the user.
            /// </summary>
            public byte[] dP;

            /// <summary>
            /// The private exponent dQ. It are PKI_NLen/2 bytes, if created with <see cref="Cmd_PKI_GenerateKeyPair"/>;
            /// if imported with <see cref="Cmd_PKI_ImportKey"/>, the length is given by the user.
            /// </summary>
            public byte[] dQ;

            /// <summary>
            /// The inverse (p^-1) mod q. It are PKI_NLen/2 bytes, if created with <see cref="Cmd_PKI_GenerateKeyPair"/>;
            /// if imported with <see cref="Cmd_PKI_ImportKey"/>, the length is given by the user.
            /// </summary>
            public byte[] ipq;
            #endregion

            /// <summary>
            /// PKI Key reference of change key.
            /// The two 1-byte field PKI_KeyNoCEK hold the values of KeyNoCEK specifying the symmetric key entry
            /// required to change the PKI key entry. The referenced change entry key is used to control the
            /// PKI key entry update with <see cref="Cmd_PKI_GenerateKeyPair"/>, <see cref="Cmd_PKI_ImportKey"/>
            /// and <see cref="Cmd_PKI_ExportPrivateKey"/> commands, but without support for OfflineChange.
            /// </summary>
            public byte KeyNoCEK;

            /// <summary>
            /// PKI Key version of change key.
            /// The two 1-byte field PKI_KeyVCEK hold the values of KeyVerCEK specifying the symmetric key entry
            /// required to change the PKI key entry. The referenced change /// entry key is used to control the
            /// PKI key entry update with <see cref="Cmd_PKI_GenerateKeyPair"/>, <see cref="Cmd_PKI_ImportKey"/>
            /// and <see cref="Cmd_PKI_ExportPrivateKey"/> commands, but without support for OfflineChange.
            /// </summary>
            public byte KeyVCEK;

            /// <summary>
            /// PKI Reference number of key usage counter.
            /// The 1-byte field PKI_RefNoKUC holds the value of RefNoKUC controlling the PKI key entry usage.
            /// The referenced key usage counter entry is automatically incremented each time the PKI key entry
            /// is used for an operation ( both private and public key operations).
            /// </summary>
            public byte RefNoKUC;

            /// <summary>
            /// PKI SET configuration settings for key entry
            /// </summary>
            public Set SET;

            /// <summary>
            /// PKI Key reference of access key.
            /// The two 1-byte fields PKI_KeyNoAEK hold the values of KeyNoAEK specifying the symmetric key entry
            /// required to access the PKI key entry. The referenced access entry key is used to lock the key entry
            /// usage to specific host.
            /// </summary>
            public byte KeyNoAEK;

            /// <summary>
            /// PKI Key version of access key.
            /// The two 1-byte fields PKI_KeyVAEK hold the values of KeyVerAEK specifying the symmetric key entry
            /// required to access the PKI key entry. The referenced access entry key is used to lock the key entry
            /// usage to specific host.
            /// </summary>
            public byte KeyVAEK;
            #endregion

            /// <summary>
            /// Constructor
            ///     - SET is configured to default zero.
            ///     - RefKUCNo is configured to 0xFF
            ///     - KeyNoCEK to 0x00
            ///     - KeyNoAEK to 0x00
            /// </summary>
            public PKI_RSA_KeyEntry ()
            {
                SET = new Set ();

                RefNoKUC = 0xFF;
                KeyNoCEK = 0x00;
                KeyNoAEK = 0x00;
            }

            /// <summary>
            /// Constructor
            /// Extracts information from \b aKeyEntry and updates the properties of this
            /// class accordingly.
            /// </summary>
            ///
            /// <param name="aKeyEntry">Buffer containing the Key information.</param>
            /// <param name="boAEK_Available">If Access Key is available.</param>
            public PKI_RSA_KeyEntry ( byte[] aKeyEntry, bool boAEK_Available = false )
            {
                ushort wOffset = 0;

                SET = new Set ();
                SET |= ( ushort ) ( aKeyEntry[wOffset++] << 8 );
                SET = aKeyEntry[wOffset++];

                KeyNoCEK = aKeyEntry[wOffset++];
                KeyVCEK = aKeyEntry[wOffset++];
                RefNoKUC = aKeyEntry[wOffset++];

                if ( boAEK_Available )
                {
                    KeyNoAEK = aKeyEntry[wOffset++];
                    KeyVAEK = aKeyEntry[wOffset++];
                }

                NLen = ( ushort ) ( aKeyEntry[wOffset++] << 8 );
                NLen |= aKeyEntry[wOffset++];

                eLen = ( ushort ) ( aKeyEntry[wOffset++] << 8 );
                eLen |= aKeyEntry[wOffset++];

                if ( SET.IncludePrivateKey )
                {
                    pLen = ( ushort ) ( aKeyEntry[wOffset++] << 8 );
                    pLen |= aKeyEntry[wOffset++];

                    qLen = ( ushort ) ( aKeyEntry[wOffset++] << 8 );
                    qLen |= aKeyEntry[wOffset++];
                }

                N = new byte[NLen];
                Array.Copy ( aKeyEntry, wOffset, N, 0, NLen );
                wOffset += NLen;

                e = new byte[eLen];
                Array.Copy ( aKeyEntry, wOffset, e, 0, eLen );
                wOffset += eLen;

                if ( SET.IncludePrivateKey )
                {
                    p = new byte[pLen];
                    Array.Copy ( aKeyEntry, wOffset, p, 0, pLen );
                    wOffset += pLen;

                    q = new byte[qLen];
                    Array.Copy ( aKeyEntry, wOffset, q, 0, qLen );
                    wOffset += qLen;

                    dP = new byte[pLen];
                    Array.Copy ( aKeyEntry, wOffset, dP, 0, pLen );
                    wOffset += pLen;

                    dQ = new byte[qLen];
                    Array.Copy ( aKeyEntry, wOffset, dQ, 0, qLen );
                    wOffset += qLen;

                    ipq = new byte[qLen];
                    Array.Copy ( aKeyEntry, wOffset, ipq, 0, qLen );

                    RemovePaddingBE ( ref dP );
                    RemovePaddingBE ( ref dQ );
                    RemovePaddingBE ( ref ipq );
                }
            }

            /// <summary>
            /// Converts the Key Entry information available as object to bytes.
            /// </summary>
            ///
            /// <param name="boAEK_Available">If Access Key is available.</param>
            ///
            /// <returns>Key Entry information available as object to bytes</returns>
            public byte[] ToByteArray ( bool boAEK_Available = false )
            {
                List<byte> aData = new List<byte> ();

                aData.Add ( ( byte ) ( SET >> 8 ) );
                aData.Add ( ( byte ) SET );

                aData.Add ( KeyNoCEK );
                aData.Add ( KeyVCEK );

                if ( boAEK_Available )
                {
                    aData.Add ( KeyNoAEK );
                    aData.Add ( KeyVAEK );
                }

                aData.Add ( ( byte ) ( NLen >> 8 ) );
                aData.Add ( ( byte ) NLen );

                aData.Add ( ( byte ) ( eLen >> 8 ) );
                aData.Add ( ( byte ) eLen );

                if ( SET.IncludePrivateKey )
                {
                    aData.Add ( ( byte ) ( pLen >> 8 ) );
                    aData.Add ( ( byte ) pLen );

                    aData.Add ( ( byte ) ( qLen >> 8 ) );
                    aData.Add ( ( byte ) qLen );
                }

                aData.AddRange ( N );
                aData.AddRange ( e );

                if ( SET.IncludePrivateKey )
                {
                    aData.AddRange ( p );
                    aData.AddRange ( q );
                    aData.AddRange ( Pad2LengthBE ( dP, pLen ) );
                    aData.AddRange ( Pad2LengthBE ( dQ, qLen ) );
                    aData.AddRange ( Pad2LengthBE ( ipq, qLen ) );
                }

                return ( aData.Count.Equals ( 0 ) ? null : aData.ToArray () );
            }

            private static void RemovePaddingBE ( ref byte[] aData )
            {
                for ( int iOffset = 0; iOffset < aData.Length; iOffset++ )
                {
                    if ( aData[iOffset] != 0 )
                    {
                        Array.Copy ( aData, iOffset, aData, 0, aData.Length - iOffset );
                        Array.Resize ( ref aData, aData.Length - iOffset );
                        break;
                    }
                }
            }

            private static byte[] Pad2LengthBE ( byte[] aData, int dwLength )
            {
                if ( aData.Length >= dwLength )
                    return aData;
                else
                {
                    byte[] buffer = new byte[dwLength];
                    Array.Copy ( aData, 0, buffer, dwLength - aData.Length, aData.Length );
                    return buffer;
                }
            }
        }
        #endregion

        #region Cmd.PKI_GenerateKeyPair
        /// <summary>
        /// Helper Class containing the Options (Generation mode) to be framed and
        /// utilized by <see cref="Cmd_PKI_GenerateKeyPair"/> interface.
        /// </summary>
        public class PKI_RSA_GenerateKeyPair_Option
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator PKI_RSA_GenerateKeyPair_Option ( byte bOption )
            {
                PKI_RSA_GenerateKeyPair_Option oInstance = new PKI_RSA_GenerateKeyPair_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( PKI_RSA_GenerateKeyPair_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Include Access Entry Key.
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool IncludeAEK
            {
                set
                {
                    if ( value ) dwOption = 0x02;
                    else dwOption &= ~0x02;
                }
                get
                {
                    return ( ( dwOption & 0x02 ) != 0x00 );
                }
            }

            /// <summary>
            /// Public exponent selection.
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool ExponentProvidedByHost
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    return ( ( dwOption & 0x01 ) != 0x00 );
                }
            }
        }
        #endregion

        #region Cmd.PKI_ImportKey
        /// <summary>
        /// Helper Class containing the Options (Update mode) to be framed and
        /// utilized by <see cref="Cmd_PKI_ImportKey"/> interface.
        /// </summary>
        public class PKI_RSA_ImportKey_Option
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator PKI_RSA_ImportKey_Option ( byte bOption )
            {
                PKI_RSA_ImportKey_Option oInstance = new PKI_RSA_ImportKey_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( PKI_RSA_ImportKey_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Include Access Entry Key.
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool IncludeAEK
            {
                set
                {
                    if ( value ) dwOption = 0x02;
                    else dwOption &= ~0x02;
                }
                get
                {
                    return ( ( dwOption & 0x02 ) != 0x00 );
                }
            }

            /// <summary>
            /// Update selection.
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool UpdateKeySettingsOnly
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    return ( ( dwOption & 0x01 ) != 0x00 );
                }
            }
        }
        #endregion
        #endregion

        #region Virtual Card
        #region Select
        /// <summary>
        /// Helper Class containing the Options (Key diversification selection and
        /// 2-parts variant support) to be framed and utilized by <see cref="Cmd_SAM_SelectVC"/>
        /// and <see cref="Cmd_VCA_Select"/> interfaces.
        /// </summary>
        public class VCA_Select_Option
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator VCA_Select_Option ( byte bOption )
            {
                VCA_Select_Option oMyInstance = new VCA_Select_Option ();
                oMyInstance.dwOption = bOption;
                return oMyInstance;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( VCA_Select_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Use Key Diversification for KeyID.VCSelectENCKey.
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool ENCKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.ENC_KEY_DIV_INPUT;
                    else
                        dwOption &= ~( int ) VCA_Select.ENC_KEY_DIV_INPUT;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_Select.ENC_KEY_DIV_INPUT ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification for the KeyID.VCSelectMACKey (Diversify the used
            /// key with the given DivInput). This set Bit[2 - 1] of P1 byte
            /// </summary>
            public bool MACKeyDiversification_DivInput
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_INPUT;
                    else
                        dwOption &= ~( int ) VCA_Select.MAC_KEY_DIV_INPUT;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_INPUT ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification for the KeyID.VCSelectMACKey (Diversify with the
            /// Virtual Card Unique Identifier (VCUID)). This set Bit[2 - 1] of P1 byte
            /// </summary>
            public bool MACKeyDiversification_VCUID
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_VCUID;
                    else
                        dwOption &= ~( int ) VCA_Select.MAC_KEY_DIV_VCUID;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_VCUID ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification for the KeyID.VCSelectMACKey (Diversify with the
            /// VCUID and the given DivInput as VCUID (DivInput limited to 31 bytes)).
            /// This set Bit[2 - 1] of P1 byte
            /// </summary>
            public bool MACKeyDiversification_VCUID_DivInput
            {
                set
                {
                    if ( value )
                    {
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_INPUT_VCUID;
                    }
                    else
                    {
                        if ( value ) dwOption &= 0xFD;
                        else dwOption &= 0xF9;
                    }
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_INPUT_VCUID ) != 0x00 );
                }
            }

            /// <summary>
            /// 2-parts variant support (Applicable for <see cref="Cmd_VCA_Select"/>).
            /// The complete VC Selection is performed in 2-parts, Postpone
            /// Cmd.ISOExternalAuthenticate to second part.
            /// This set Bit[3] of P1 byte
            /// </summary>
            public bool Use2PartVariantSupport
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.VARIANT_PART2;
                    else
                        dwOption &= ~( int ) VCA_Select.VARIANT_PART2;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_Select.VARIANT_PART2 ) != 0x00 );
                }
            }
        }
        #endregion

        #region ProximityCheck
        /// <summary>
        /// Helper Class containing the Options (Key diversification and operation
        /// mode) to be framed and utilized by <see cref="Cmd_VCA_ProximityCheck"/>
        /// interface.
        /// </summary>
        public class VCA_ProximityCheck_Option
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator VCA_ProximityCheck_Option ( byte bOption )
            {
                VCA_ProximityCheck_Option oMyInstance = new VCA_ProximityCheck_Option ();
                oMyInstance.dwOption = bOption;
                return oMyInstance;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( VCA_ProximityCheck_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Use Key Diversification.
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool UseKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.DIVERSIFICATION_ON;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.DIVERSIFICATION_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_ProximityCheck.DIVERSIFICATION_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Use random Cmd.VerifyPC processing.
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool UseRandomProcessing
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.RANDOM_PROCESSING;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.RANDOM_PROCESSING;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_ProximityCheck.RANDOM_PROCESSING ) != 0x00 );
                }
            }

            /// <summary>
            /// ISO/IEC 7816-4 mode.
            /// This set Bit[2] of P1 byte
            /// </summary>
            public bool UseISO7816WrappedFormat
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.WRAPPED_FORMAT;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.WRAPPED_FORMAT;
                }
                get
                {
                    return ( ( dwOption & ( int ) VCA_ProximityCheck.WRAPPED_FORMAT ) != 0x00 );
                }
            }
        }
        #endregion
        #endregion

        #region MIFARE DESFire
        #region Authentication
        /// <summary>
        /// Helper Class containing the Options to be framed and utilized by below mentioned
        /// interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_SAM_IsoAuthenticatePICC_Part1"/>
        ///     - <see cref="Cmd_DESFire_AuthenticatePICC"/>
        /// </summary>
        public class DESFire_AuthenticateOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator DESFire_AuthenticateOption ( byte bOption )
            {
                DESFire_AuthenticateOption oObject = new DESFire_AuthenticateOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( DESFire_AuthenticateOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Enable Diversification.
            /// This set Bit[0] of P1 byte (Diversify the used key with the
            /// given DivInput)
            /// </summary>
            public bool UseDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthDiv.ON;
                    else dwOption &= ~( int ) MFD_AuthDiv.ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthDiv.ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Key selection
            /// This set Bit[1] of P1 byte (Key selection by DESFire key number)
            /// </summary>
            public bool DESFireKeyNumber
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthKeySel.DESFIRE_KEY_NUMBER;
                    else dwOption &= ~( int ) MFD_AuthKeySel.DESFIRE_KEY_NUMBER;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthKeySel.DESFIRE_KEY_NUMBER ) != 0x00 );
                }
            }

            /// <summary>
            /// Key Diversification Mode
            /// This set Bit[4 - 3] of P1 byte (AV1 compatibility mode key diversification
            /// methods, TDEA Key, diversified using one encryption round)
            /// </summary>
            public bool UseKeyDiversification_OneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION ) != 0x00 );
                }
            }

            /// <summary>
            /// Key Diversification Mode
            /// This set Bit[4 - 3] of P1 byte (AV2 mode key diversification methods)
            /// </summary>
            public bool UseKeyDiversification_AV2
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) MFD_AuthDivMode.AV2;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthDivMode.AV1_SINGLE_ENCRYPTION ) != 0x00 );
                }
            }

            /// <summary>
            /// Allow secure messaging
            /// This set Bit[5] of P1 byte (Suppress secure messaging (for
            /// example for originality keys))
            /// </summary>
            public bool SupressSecureMessaging
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthSM.SUPPRESS;
                    else dwOption &= ~( int ) MFD_AuthSM.SUPPRESS;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthSM.SUPPRESS ) != 0x00 );
                }
            }

            /// <summary>
            /// Authentication type and EVx authentication mode
            /// This set Bit[7 - 6] of P1 byte (Authenticate Type as EV2 and
            /// Authentication Mode as First)
            /// </summary>
            public bool AuthenticateEV2_First
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthType.EV2_FIRST_AUTH;
                    else dwOption &= ~( int ) MFD_AuthType.EV2_FIRST_AUTH;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthType.EV2_FIRST_AUTH ) != 0x00 );
                }
            }

            /// <summary>
            /// Authentication type and EVx authentication mode
            /// This set Bit[7 - 6] of P1 byte (Authenticate Type as EV2 and
            /// Authentication Mode as NonFirst)
            public bool AuthenticateEV2_NonFirst
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_AuthType.EV2_NON_FIRST_AUTH;
                    else dwOption &= ~( ( int ) MFD_AuthType.EV2_NON_FIRST_AUTH & 0x7F );
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_AuthType.EV2_NON_FIRST_AUTH ) != 0x00 );
                }
            }
        }
        #endregion

        #region ChangeKey
        #region Key Compilation Method
        /// <summary>
        /// Helper Class containing the Options to be framed and utilized by below mentioned
        /// interfaces. This should be used to update Key diversification method, P1 byte
        /// of SAM Command frame
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public class DESFire_ChangeKeyCompilationOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator DESFire_ChangeKeyCompilationOption ( byte bOption )
            {
                DESFire_ChangeKeyCompilationOption oObject = new DESFire_ChangeKeyCompilationOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( DESFire_ChangeKeyCompilationOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Cryptogram computation mode
            /// This set Bit[0] of P1 byte (PICC targeted key equal to PICC authenticated key
            /// The parameters CurrKeyNo and CurrKeyV are ignored.)
            /// </summary>
            public bool PICCTargetAndAuthKeySame
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKCryptoMode.SAME_KEY;
                    else dwOption &= ~( int ) MFD_CKCryptoMode.SAME_KEY;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKCryptoMode.SAME_KEY ) != 0x00 );
                }
            }

            /// <summary>
            /// Use of key diversification for the new key
            /// This set Bit[1] of P1 byte (Diversified the new key)
            /// </summary>
            public bool DiversifyNewKey
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKNewKey.DIV_ON;
                    else dwOption &= ~( int ) MFD_CKNewKey.DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKNewKey.DIV_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Use of key diversification for the current key
            /// This set Bit[2] of P1 byte (Diversified the current key)
            /// </summary>
            public bool DiversifyCurrentKey
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKCurrKey.DIV_ON;
                    else dwOption &= ~( int ) MFD_CKCurrKey.DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKCurrKey.DIV_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Key diversification method for the new key
            /// This set Bit[3] of P1 byte (Diversify using one encryption round)
            /// </summary>
            public bool UseNewKeyDiversificationOneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKNewKey.SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) MFD_CKNewKey.SINGLE_ENCRYPTION;
                }
                get
                {
                    if ( ( dwOption & ( int ) MFD_CKNewKey.SINGLE_ENCRYPTION ) != 0x00 ) return true;
                    else return false;
                }
            }

            /// <summary>
            /// Key diversification method for the current key
            /// This set Bit[4] of P1 byte (Diversify using one encryption round)
            /// </summary>
            public bool UseCurrentKeyDiversificationOneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKCurrKey.SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) MFD_CKCurrKey.SINGLE_ENCRYPTION;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKCurrKey.SINGLE_ENCRYPTION ) != 0x00 );
                }
            }

            /// <summary>
            /// Key diversification method
            /// This set Bit[5] of P1 byte (SAM AV2 diversification method)
            /// </summary>
            public bool UseSamAV2Diversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKDivMethod.AV2;
                    else dwOption &= ~( int ) MFD_CKDivMethod.AV2;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKDivMethod.AV2 ) != 0x00 );
                }
            }
        }
        #endregion

        #region Configuration
        /// <summary>
        /// Helper Class containing the Options to be framed and utilized by below mentioned
        /// interfaces. This should be used to update Configuration, P2 byte of
        /// SAM Command frame
        ///     - <see cref="Cmd_SAM_ChangeKeyPICC"/>
        ///     - <see cref="Cmd_DESFire_ChangeKeyPICC"/>
        /// </summary>
        public class DESFire_ChangeKeyConfigurationOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator DESFire_ChangeKeyConfigurationOption ( byte bOption )
            {
                DESFire_ChangeKeyConfigurationOption oObject = new DESFire_ChangeKeyConfigurationOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( DESFire_ChangeKeyConfigurationOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// PICC master key update
            /// This set Bit[3 - 0] of P2 byte (Number of PICC key to be changed) in case of ChangeKey
            /// user DEKeyNo (Number of PICC key to be changed) in case of ChangeKeyEV2.
            /// </summary>
            public byte DESFireKeyNo
            {
                set
                {
                    dwOption &= ChangeKeyEV2 ? 0xFF : 0xF0;
                    dwOption |= ( byte ) ( value & ( ChangeKeyEV2 ? 0xFF : 0xF0 ) );
                }
                get
                {
                    return ( byte ) ( dwOption & ( ChangeKeyEV2 ? 0xFF : 0xF0 ) );
                }
            }

            /// <summary>
            /// PICC master key update
            /// This set Bit[4] of P2 byte (Include the key type in the cryptogram)
            /// </summary>
            public bool IncludeKeyType
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKMasterUpdate.INCLUDE_KEYTYPE;
                    else dwOption &= ~( int ) MFD_CKMasterUpdate.INCLUDE_KEYTYPE;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKMasterUpdate.INCLUDE_KEYTYPE ) != 0x00 );
                }
            }

            /// <summary>
            /// PICC Change Key command
            /// This set Bit[5] of P2 byte (Cmd.ChangeKeyEV2)
            /// </summary>
            public bool ChangeKeyEV2
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_CKCmdType.CHANGE_KEY_EV2;
                    else dwOption &= ~( int ) MFD_CKCmdType.CHANGE_KEY_EV2;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_CKCmdType.CHANGE_KEY_EV2 ) != 0x00 );
                }
            }

            /// <summary>
            /// ISO mode selection
            /// This set Bit[7 - 6] of P2 byte (Command will be sent to PICC using
            /// ISO/IEC 7816-4APDU)
            /// </summary>
            public bool UseISO7816_4_APDU
            {
                set
                {
                    if ( value ) dwOption |= ( int ) ISOMode.ISO7816;
                    else dwOption &= ~( int ) ISOMode.ISO7816;
                }
                get
                {
                    return ( ( dwOption & ( int ) ISOMode.ISO7816 ) != 0x00 );
                }
            }
        }
        #endregion
        #endregion

        #region WriteX
        /// <summary>
        /// Helper Class containing the Options to be framed and utilized <see cref="Cmd_DESFire_WriteX"/>
        /// interfaces. This should be used to update Crypto configuration, P2 byte
        /// of SAM Command frame.
        /// </summary>
        public class DESFire_WriteCryptoConfiguration
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator DESFire_WriteCryptoConfiguration ( byte bOption )
            {
                DESFire_WriteCryptoConfiguration oObject = new DESFire_WriteCryptoConfiguration ();
                oObject.dwOption = bOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( DESFire_WriteCryptoConfiguration oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Extended offset
            /// This set Bit[7] of P2 byte
            /// </summary>
            public bool ExtendedOffset
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_Offset.EXTENDED_OFFSET;
                    else dwOption &= ~( int ) MFD_Offset.EXTENDED_OFFSET;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_Offset.EXTENDED_OFFSET ) != 0x00 );
                }
            }

            /// <summary>
            /// Chaining configuration
            /// This set Bit[6] of P2 byte (ISO/IEC 14443-4 chaining)
            /// </summary>
            public bool ISOChaining
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFD_Chaining.ISO_CHAINING;
                    else dwOption &= ~( int ) MFD_Chaining.ISO_CHAINING;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFD_Chaining.ISO_CHAINING ) != 0x00 );
                }
            }

            /// <summary>
            /// Communication mode
            /// This set Bit[5 - 4] of P2 byte (ISO/IEC 14443-4 chaining)
            /// </summary>
            public CommMode CommMode
            {
                set
                {
                    dwOption &= 0xCF;
                    dwOption |= ( byte ) value;
                }
                get
                {
                    return ( CommMode ) ( dwOption & 0x30 );
                }
            }

            /// <summary>
            /// Command offset.
            /// Present if <see cref="MFD_Offset.LEGACY_MODE"/> and <see cref="MFD_CommMode.FULL"/>
            /// and first frame
            /// </summary>
            public byte Offset
            {
                set
                {
                    dwOption &= 0xF0;
                    dwOption |= ( byte ) ( value & 0x0F );
                }
                get
                {
                    return ( byte ) ( dwOption & 0x0F );
                }
            }
        }
        #endregion
        #endregion

        #region MIFARE Plus
        #region Authenticate
        /// <summary>
        /// Helper Class containing the Options (Key diversification & authentication
        /// mode selection) to be framed and utilized by below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthenticateMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_Authenticate"/>
        /// </summary>
        public class Plus_AuthenticateOption
        {
            private int dwOption;

            #region Key Derivation
            /// <summary>
            /// Enumeration for SL and Key derivation info
            /// </summary>
            public enum KDF : byte
            {
                /// <summary> No Key derivation (SL1, originality keys...) </summary>
                NO_KDF = 0x00,

                /// <summary> Security Level 3 Key derivation </summary>
                SL3_KDF = 0x03
            }
            #endregion

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator Plus_AuthenticateOption ( byte Option )
            {
                Plus_AuthenticateOption oObject = new Plus_AuthenticateOption ();
                oObject.dwOption = Option;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( Plus_AuthenticateOption Option )
            {
                return ( byte ) Option.dwOption;
            }

            /// <summary>
            /// SL and Key derivation info.
            /// This set Bit[3 - 2] of P1 byte
            /// </summary>
            public KDF KeyDerivation
            {
                set
                {
                    dwOption &= 0xF3;
                    dwOption |= ( ( byte ) value << 2 );
                }
                get
                {
                    return ( KDF ) ( ( dwOption & 0x0C ) >> 2 );
                }
            }

            /// <summary>
            /// Select Authentication Mode.
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool Authenticate_First
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_AuthMode.AUTH_NONFIRST;
                    else dwOption &= ( int ) ~MFP_AuthMode.AUTH_NONFIRST;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_AuthMode.AUTH_NONFIRST ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification.
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool UseDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_AuthDiv.DIVERSIFY_ON;
                    else dwOption &= ( int ) ~MFP_AuthDiv.DIVERSIFY_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_AuthDiv.DIVERSIFY_ON ) != 0x00 );
                }
            }
        }
        #endregion

        #region Authenticate SectorSwitch
        /// <summary>
        /// Helper Class containing the Options (Key diversification selection) to be
        /// framed and utilized by below mentioned interfaces.
        ///     - <see cref="Cmd_SAM_AuthSectorSwitchMFP_Part1"/>
        ///     - <see cref="Cmd_MFP_AuthSectorSwitch"/>
        /// </summary>
        public class Plus_AuthSectorSwitchOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator Plus_AuthSectorSwitchOption ( byte bOption )
            {
                Plus_AuthSectorSwitchOption oObject = new Plus_AuthSectorSwitchOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( Plus_AuthSectorSwitchOption Option )
            {
                return ( byte ) Option.dwOption;
            }

            /// <summary>
            /// Apply sector number key diversification
            /// This set Bit[2] of P1 byte
            /// </summary>
            public bool ApplySectorNumberKeyDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification for Sector Keys
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool EnableSectorKeysDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Use Key Diversification for Sector Switch Key
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool EnableSectorSwitchKeysDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON ) != 0x00 );
                }
            }
        }
        #endregion

        #region Combined Read
        /// <summary>
        /// Helper Class containing the Options (Command case) to be framed
        /// and utilized by <see cref="Cmd_SAM_CombinedReadMFP"/> interface
        /// </summary>
        public class Plus_CombinedReadOption
        {
            private int dwOption;
            private bool boFirst, boCont, boLast;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator Plus_CombinedReadOption ( int dwOption )
            {
                Plus_CombinedReadOption oObject = new Plus_CombinedReadOption ();
                oObject.dwOption = dwOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator int ( Plus_CombinedReadOption oOption )
            {
                oOption.dwOption &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                oOption.dwOption &= ~ExchangeOptions.BUFFERED_BIT;

                if ( oOption.boFirst )
                    oOption.dwOption |= ExchangeOptions.BUFFER_FIRST;
                else if ( oOption.boCont )
                    oOption.dwOption |= ExchangeOptions.BUFFER_CONT;
                else if ( oOption.boLast )
                    oOption.dwOption |= ExchangeOptions.BUFFER_LAST;

                return oOption.dwOption;
            }

            /// <summary> Buffer first set of information. No exchange is performed </summary>
            public bool SetBufferFirst
            {
                set
                {
                    boFirst = value;
                    boCont = boLast = !value;
                }
                get
                {
                    return boFirst;
                }
            }

            /// <summary> Buffer intermediate set of information. No exchange is performed </summary>
            public bool SetBufferCont
            {
                set
                {
                    boCont = value;
                    boFirst = boLast = !value;
                }
                get
                {
                    return boCont;
                }
            }

            /// <summary> Buffer intermediate set of information. No exchange is performed </summary>
            public bool SetBufferLast
            {
                set
                {
                    boLast = value;
                    boFirst = boCont = !value;
                }
                get
                {
                    return boLast;
                }
            }

            /// <summary>
            /// Payload type
            /// This set Bit[1 - 0] of P1 byte
            /// </summary>
            public MFP_PayloadType CommandType
            {
                set
                {
                    dwOption &= 0xFFFC;
                    dwOption |= ( int ) value;
                }
                get
                {
                    return ( MFP_PayloadType ) ( dwOption & 0x0003 );
                }
            }
        }
        #endregion

        #region Combined Write
        /// <summary>
        /// Helper Class containing the Options (Command case) to be framed and
        /// utilized by <see cref="Cmd_SAM_CombinedWriteMFP"/> interface.
        /// </summary>
        public class Plus_CombinedWriteOption
        {
            private int dwOption;
            private bool boFirst, boCont, boLast;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator Plus_CombinedWriteOption ( int dwOption )
            {
                Plus_CombinedWriteOption oObject = new Plus_CombinedWriteOption ();
                oObject.dwOption = dwOption;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator int ( Plus_CombinedWriteOption oOption )
            {
                oOption.dwOption &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                oOption.dwOption &= ~ExchangeOptions.BUFFERED_BIT;

                if ( oOption.boFirst )
                    oOption.dwOption |= ExchangeOptions.BUFFER_FIRST;
                else if ( oOption.boCont )
                    oOption.dwOption |= ExchangeOptions.BUFFER_CONT;
                else if ( oOption.boLast )
                    oOption.dwOption |= ExchangeOptions.BUFFER_LAST;

                return oOption.dwOption;
            }

            /// <summary> Buffer first set of information. No exchange is performed </summary>
            public bool SetBufferFirst
            {
                set
                {
                    boFirst = value;
                    boCont = boLast = !value;
                }
                get
                {
                    return boFirst;
                }
            }

            /// <summary> Buffer intermediate set of information. No exchange is performed </summary>
            public bool SetBufferCont
            {
                set
                {
                    boCont = value;
                    boFirst = boLast = !value;
                }
                get
                {
                    return boCont;
                }
            }

            /// <summary> Buffer intermediate set of information. No exchange is performed </summary>
            public bool SetBufferLast
            {
                set
                {
                    boLast = value;
                    boFirst = boCont = !value;
                }
                get
                {
                    return boLast;
                }
            }

            /// <summary>
            /// Payload type
            /// This set Bit[0] of P1 byte
            /// </summary>
            public MFP_PayloadType CommandType
            {
                set
                {
                    dwOption &= 0xFFF1;
                    dwOption |= ( int ) value;
                }
                get
                {
                    return ( MFP_PayloadType ) ( dwOption & 0x0001 );
                }
            }
        }
        #endregion

        #region ChangeKey
        /// <summary>
        /// Helper Class containing the Options (Command config) to be framed and
        /// utilized by <see cref="Cmd_SAM_ChangeKeyPICC"/> interface.
        /// </summary>
        public class Plus_ChangeKeyOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator Plus_ChangeKeyOption ( byte Option )
            {
                Plus_ChangeKeyOption oObject = new Plus_ChangeKeyOption ();
                oObject.dwOption = Option;
                return oObject;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( Plus_ChangeKeyOption Option )
            {
                return ( byte ) Option.dwOption;
            }

            /// <summary>
            /// Use Key Diversification.
            /// This set Bit[1] of P1 byte
            /// </summary>
            public bool UseDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_ChangeKeyDiv.DIVERSIFY_ON;
                    else dwOption &= ( int ) ~MFP_ChangeKeyDiv.DIVERSIFY_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFP_ChangeKeyDiv.DIVERSIFY_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Payload type
            /// This set Bit[0] of P1 byte
            /// </summary>
            public MFP_PayloadType CommandType
            {
                set
                {
                    dwOption &= 0xFFF1;
                    dwOption |= ( int ) value;
                }
                get
                {
                    return ( MFP_PayloadType ) ( dwOption & 0x0001 );
                }
            }
        }
        #endregion
        #endregion

        #region MIFARE Ultralight
        /// <summary>
        /// Helper Class containing the Options (Key diversification selection) to be
        /// framed and utilized by <see cref="Cmd_UL_AuthenticatePICC"/> interface.
        /// </summary>
        public class MFUL_AuthenticateOption
        {
            private int dwOption;

            /// <summary>
            /// Extracts the Options and returns as an object of the class.
            /// </summary>
            ///
            /// <param name="bOption">The options that are configured.</param>
            public static implicit operator MFUL_AuthenticateOption ( byte bOption )
            {
                MFUL_AuthenticateOption oInstance = new MFUL_AuthenticateOption ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            /// <summary>
            /// Combines the values of the properties and returns the option information
            /// as one single byte.
            /// </summary>
            /// <param name="oOption">Object of the class having the properties configured.</param>
            public static implicit operator byte ( MFUL_AuthenticateOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            /// <summary>
            /// Use Key Diversification
            /// This set Bit[0] of P1 byte
            /// </summary>
            public bool UseKeyDiversification
            {
                set
                {
                    if ( value ) dwOption = ( int ) MFUL_Option.DIV_ON;
                    else dwOption &= ( int ) ~MFUL_Option.DIV_ON;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFUL_Option.DIV_ON ) != 0x00 );
                }
            }

            /// <summary>
            /// Exchange LE to SAM.
            /// [Optional for AV2 backwards compatibility reasons]
            /// Bytes expected in the response data field
            /// </summary>
            public bool IncludeLE
            {
                set
                {
                    if ( value ) dwOption = ( int ) MFUL_Option.INCLUDE_LE;
                    else dwOption &= ( int ) ~MFUL_Option.INCLUDE_LE;
                }
                get
                {
                    return ( ( dwOption & ( int ) MFUL_Option.INCLUDE_LE ) != 0x00 );
                }
            }
        }
        #endregion

        #region ISO14443-3
        /// <summary>
        /// Structure containing the information received by <see cref="Cmd_X_ISO14443_3_ActivateIdle"/>
        /// interface.
        /// </summary>
        public struct ActivateIdleParams
        {
            /// <summary> ATQA (Answer To re-Quest type A): 2 bytes </summary>
            public byte[] aAtqa;

            /// <summary> SAK (Select AcKnowledge) 1 byte </summary>
            public byte bSak;

            /// <summary> UID (Unique Identified) : Card UID (4, 7, 10 bytes) </summary>
            public byte[] aUid;
        };
        #endregion
        #endregion

        #region Support Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength, byte[] aPICCErrCode,
            out ushort wPiccReturnCode )
        {
            wPiccReturnCode = ( ushort ) ( aPICCErrCode == null ? 0 : BitConverter.ToUInt16 ( aPICCErrCode, 0 ) );

            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error_Gen.SUCCESS_CHAINING ) ||
                oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.OK_CHAINING_ACTIVE ) ) ||
                oStatus.Equals ( Error_Gen.SUCCESS_INCOMPLETE_BYTE ) ) && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( pBuffer != IntPtr.Zero )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }

        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            ushort wPiccErrorCode = 0;
            return MarshalCopy ( oStatus, pBuffer, dwLength, null, out wPiccErrorCode );
        }
        #endregion

        #region DataParams Members
        public byte[] Uid
        {
            get
            {
                byte[] aUid = new byte[7];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 7; i++ )
                        {
                            aUid[i] = pDataParams->bUid[i];
                        }
                    }
                }
                return aUid;
            }
        }
        #endregion
    }
}
#endif
