/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.HalContact
{
    #region CALLBACK

    /// <summary>
    /// Callback HALContact-Component of Basic Function Library Framework.
    /// </summary>
    public class Callback : HalContact.Generic
    {
        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pExchangeCb;
            IntPtr pApplyProtocolSettingsCb;
            IntPtr pSetConfig32Cb;
            IntPtr pGetConfig32Cb;
            IntPtr pActivateCb;
            IntPtr pDeactivateCb;
            IntPtr pClockStopCb;
            IntPtr pWaitCb;
            IntPtr pInitContactReader;
            IntPtr pCbContext;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Callback_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pExchangeCb,
            IntPtr pApplyProtocolSettingsCb,
            IntPtr pSetConfig32Cb,
            IntPtr pGetConfig32Cb,
            IntPtr pActivateCb,
            IntPtr pDeactivateCb,
            IntPtr pClockStopCb,
            IntPtr pWaitCb,
            IntPtr pInitContactReader,
            IntPtr pCbContext
            );

        #endregion

        #region DELEGATES

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ExchangeDelegate(
            ushort wOption,         /**< [In] Option parameter. */
            IntPtr pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] received data */
            ref ushort pRxLength,   /**< [Out] number of received data bytes */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ApplyProtocolSettingsDelegate(
            IntPtr Atr,         /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */
            ushort wAtrLength,  /**< [In] Length of the given Atr Buffer. */
            byte bProtocolType, /**< [In] Type of protocol for which the HAL should be configured for. */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort SetConfig32Delegate(
            Config wConfig,     /**< [In] Configuration Identifier */
            uint wValue,        /**< [In] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
        );


        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort GetConfig32Delegate(
            Config wConfig,     /**< [In] Configuration Identifier */
            ref uint pValue,    /**< [Out] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ActivateDelegate(
            IntPtr pATRBuffer,      /**< [Out] Data to the received ATR. */
            ref short pwATRLenght,  /**< [Out] Length of the reiceived ATR. */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort DeactivateDelegate(
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ClockStopDelegate(
            ushort wTgClockCount,   /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
            ushort wThClockCount,   /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
            uint dwPinState,        /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
            IntPtr pContext         /**< [Out] User Defined Context */
            );


        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort WaitDelegate(
            byte bUnit,             /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            int wTimeout,           /**< [In] Timeout value */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort InitContactReaderDelegate(
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        #endregion

        #region CALLBACKS

        private ushort cbExchange(
                    ushort wOption,         /**< [In] Option parameter. */
                    IntPtr pTxBuffer,       /**< [In] data to transmit */
                    ushort wTxLength,       /**< [In] length of input data */
                    ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
                    ref ushort pRxLength,   /**< [Out] number of received data bytes */
                    IntPtr pContext         /**< [Out] User Defined Context */
                    )
        {
            Status_t status;
            byte[] pTxBufferTmp = new byte[wTxLength];
            byte[] pRxBufferTmp;

            // Prepare input buffer(s)
            for (int dIndex = 0; dIndex < wTxLength; dIndex++)
            {
                pTxBufferTmp[dIndex] = Marshal.ReadByte(pTxBuffer, dIndex);
            }

            // Reset Receive length
            pRxLength = 0;

            // Execute callback function
            status = m_ExtHal.Exchange((int)wOption, pTxBufferTmp, out pRxBufferTmp);

            // Return pointer to global buffer
            if (pRxBufferTmp != null)
            {
                if (pRxBufferTmp.Length != 0)
                {
                    // Buffer overflow check
                    if (pRxBufferTmp.Length > m_RxBuffer.Length)
                    {
                        return new Status_t(Error_CompCode.HAL, Error_Comm.BUFFER_OVERFLOW);
                    }

                    // Copy
                    System.Array.Copy(pRxBufferTmp, m_RxBuffer, pRxBufferTmp.Length);

                    // Set Output Buffer to persistent HAL buffer pointer
                    ppRxBuffer = m_pRxBuffer.AddrOfPinnedObject();

                    // Set number of received bytes
                    pRxLength = (ushort)pRxBufferTmp.Length;
                }
            }

            return status;
        }

        private ushort cbApplyProtocolSettings(
            IntPtr Atr,         /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */
            ushort wAtrLength,  /**< [In] Length of the given Atr Buffer. */
            byte bProtocolType, /**< [In] Type of protocol for which the HAL should be configured for. */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            byte[] pAtrBufferTmp = new byte[wAtrLength];

            // Prepare input buffer(s)
            for (int dIndex = 0; dIndex < wAtrLength; dIndex++)
            {
                pAtrBufferTmp[dIndex] = Marshal.ReadByte(Atr, dIndex);
            }

            //Execute callback function
            return m_ExtHal.ApplyProtocolSettings((Protocol)bProtocolType, pAtrBufferTmp);
        }

        private ushort cbSetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            uint wValue,        /**< [In] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
                                          )
        {
            //Execute callback function
            return m_ExtHal.SetConfig32(wConfig, (uint)wValue);
        }

        private ushort cbGetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            ref uint pValue,    /**< [Out] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            //Execute callback function
            return m_ExtHal.GetConfig32(wConfig, out pValue);
        }

        private ushort cbWait(
            byte bUnit,             /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            int wTimeout,           /**< [In] Timeout value */
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            //Execute callback function
            return m_ExtHal.Wait(bUnit, wTimeout);
        }

        private ushort cbActivate(
            IntPtr pATRBuffer,      /**< [Out] Data to the received ATR. */
            ref short pwATRLenght,  /**< [Out] Length of the reiceived ATR. */
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            Status_t status;
            byte[] ATRBuffer;

            //Execute callback function
            status = m_ExtHal.Activate(out ATRBuffer);

            if (ATRBuffer != null)
            {
                for (int dIndex = 0; dIndex < ATRBuffer.Length; dIndex++)
                {
                    Marshal.WriteByte(pATRBuffer, dIndex, ATRBuffer[dIndex]);
                }

                pwATRLenght = (short)ATRBuffer.Length;
            }
            else
            {
                pwATRLenght = 0;
            }

            return status;
        }

        private ushort cbDeactivate(
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            //Execute callback function
            return m_ExtHal.Deactivate();
        }

        private ushort cbClockStop(
            ushort wTgClockCount,   /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
            ushort wThClockCount,   /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
            uint dwPinState,        /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            //Execute callback function
            return m_ExtHal.ClockStop(wTgClockCount, wThClockCount, dwPinState);
        }

        private ushort cbInitContactReader(
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            //Execute callback function
            return m_ExtHal.InitContactReader();
        }

        #endregion

        #region INIT

        // Storage for internal callbacks
        private List<Delegate> m_Callbacks;
        private HalContact.IHalContact m_ExtHal;
        private byte[] m_RxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="extInterface"></param>
        /// <returns></returns>
        public Status_t Init(
            HalContact.IHalContact ExtHal,
            int RxBufferSize
            )
        {
            // Allocate internal callback delegates
            m_Callbacks = new List<Delegate>();
            m_Callbacks.Add(new ExchangeDelegate(cbExchange));
            m_Callbacks.Add(new ApplyProtocolSettingsDelegate(cbApplyProtocolSettings));
            m_Callbacks.Add(new SetConfig32Delegate(cbSetConfig32));
            m_Callbacks.Add(new GetConfig32Delegate(cbGetConfig32));
            m_Callbacks.Add(new ActivateDelegate(cbActivate));
            m_Callbacks.Add(new DeactivateDelegate(cbDeactivate));
            m_Callbacks.Add(new ClockStopDelegate(cbClockStop));
            m_Callbacks.Add(new WaitDelegate(cbWait));
            m_Callbacks.Add(new InitContactReaderDelegate(cbInitContactReader));

            // Store reference to external hal object
            m_ExtHal = ExtHal;

            // Free allocated RxBuffer if any
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Create RxBuffer
            m_RxBuffer = new byte[RxBufferSize];
            this.m_pRxBuffer = GCHandle.Alloc(this.m_RxBuffer, GCHandleType.Pinned);

            // Call init. function
            return phhalHwContact_Callback_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[0]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[1]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[2]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[3]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[4]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[5]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[6]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[7]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[8]),
                IntPtr.Zero);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Callback()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Callback()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }

            // Free allocated RxBuffer
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

	#endregion
}
#endif
