/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.HalContact
{
    #region Micropross

    /// <summary>
    /// Micropross specific HAL-Contact-Component of Basic Function Library Framework.
    /// </summary>
    public class Micropross : HalContact.Generic
    {
        public const int DEFAULT_TIMEOUT_MS = 5;            /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0068;            /**< Number of shadowed configurations. */
        private const int RESERVED_BUFFER_LEN = 32;         /**< Amount of needed and reserved memory for the protocol overhead. */
        private const int RESERVED_BUFFER_CRC_LEN = 2;      /**< Amount of needed and reserved memory for the crc overhead. */
        private const int SERIAL_NUMBER_SIZE = 16;          /**< The size of the serial number. */
        private const int VERSION_STR_SIZE = 8;             /**< The size of the version strings. */
        private const int COUPLER_NAME_SIZE = 7;            /**< The size of the coupler name (longest = SWPSPY). */
        private const int COUPLER_SERIAL_NUMBER_SIZE = 9;   /**< The size of the coupler serial number. */
        private const int COUPLER_REVISION_SIZE = 5;        /**< The size of the coupler revision. */
        private const int COUPLER_CALC_DATE_SIZE = 4;       /**< The size used to store the calibration date + checksum */
        private const int MAX_SPY_FILENAME_SIZE = 256;      /**< Max size of the file name for spy functionality */

        public enum Config : int
        {
            COUPLER_NUMBER              = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,   /**< Configure of Coupler. */
            I2C_PARAMETER_ADDRESS_WIDTH = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+1, /**< Config I2C address width. */
            I2C_PARAMETER_RX_TIMEOUT    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+2, /**< Config I2C rx timeout. */
            I2C_PARAMETER_TX_TIMEOUT    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+3, /**< Config I2C tx timeout. */
            I2C_PARAMETER_TSU_DAT       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+4, /**< Config I2C setup time data. */
            I2C_PARAMETER_THD_DAT       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+5, /**< Config I2C hold time data. */
            I2C_PARAMETER_TSU_STA       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+6, /**< Config I2C setup time start. */
            I2C_PARAMETER_THD_STA       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+7, /**< Config I2C hold time start. */
            I2C_PARAMETER_TSU_STO       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+8, /**< Config I2C setup time stop. */
            I2C_PARAMETER_THIGH_SCL     = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+9, /**< Config I2C clock high time. */
            I2C_PARAMETER_TLOW_SCL      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+10,/**< Config I2C clock low time. */
            I2C_PARAMETER_AUTO_NACK     = NxpRdLibNet.CustomCodes.CONFIG_BEGIN+11 /**< Config I2C auto NACK transmit. */
        }

        public enum DateFormat : int
        {
            EXTERNAL_CLOCK = 1,                                             /**< Date Format External Clock Mode */
            INTERNAL_CLOCK                                                  /**< Date Format Internal Clock Mode */
        }

        public enum I2CAddressWidth
        {
            WIDTH_7BIT  = 0, /**< 7bit I2C address */
            WIDTH_10BIT = 1  /**< 10bit I2C address */
        }

        public enum License : byte
        {
            LICENSE_EXT_SPY = 7,            /**< Use of the TC3 as an external spy */
            LICENSE_I2C = 8,                /**< I2C channel */
            LICENSE_IO_ANALOG = 17          /**< IO monitoring capabilities */
        }

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                                      /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;                                           /**< pointer to the lower layers parameter structure */
            public uint wATRLength;                                                 /**< Length of the ATR stored internally in data params. */
            public fixed byte pATR[MAX_ATR_SIZE];                                   /**< ATR of current card. */
            public IntPtr pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                                               /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                                               /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                                           /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public uint dwNumExpectedBytes;                                         /**< Number of expected bytes for Exchanges. */
            public byte bProtocolType;                                              /**< Type of protocol for which the hal is configured for */
            public uint dwI2CRxPollingTimeoutMs;                                    /**< Timeout Value in ms to be used by I2C exchange function. */
            public fixed uint wCfgShadow[SHADOW_COUNT];                             /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                                              /**< Current timing measurement mode. */
            public byte bTimeoutUnit;                                               /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public ulong qwTiming_ns;                                               /**< Current timing value. */
            public ushort wAdditionalInfo;                                          /**< storage for additional error information */
            public ushort wFieldOffTime;                                            /**< Field-Off-Time in milliseconds. */
            public byte bRfResetAfterTo;                                            /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            IntPtr pIntBuffer;                                                      /**< Internal buffer used as a scratch buffer. Size must be wIntBufferLen*wIntBufferNum */
            ushort wIntBufferSize;                                                  /**< Total length of internal buffer used as a scratch buffer */
            ushort wIntBufferLen;                                                   /**< Flag to show the status of buffer allocation. */
            public byte bCouplerNumber;                                             /**< Select the Coupler used for communication */
            public uint dwGuardTimeEtu;                                             /**< Specify the value of the guard time in etu, and an additional value express in clock cycles */
            public uint dwGuardTimeClk;                                             /**< Use the specify the value of the guard time in etu and addan additional value express in clock cycles */
            public ushort wCouplerResourceId;                                       /**< Storage of the resource ID of the Coupler to know the type of Reader (TCL1 or TCL2 supported).*/
            public fixed byte bSerialNumber[SERIAL_NUMBER_SIZE];                    /**< This value stores the serial number of the reader */
            public fixed byte bDeviceFamily[VERSION_STR_SIZE];                      /**< This value stores the DeviceFamily of the reader */
            public fixed byte bDeviceReferenceWithinFamily[VERSION_STR_SIZE];       /**< This value stores the DeviceReferenceWithinFamily of the reader */
            public fixed byte bMotherboardSystemVersion[VERSION_STR_SIZE];          /**< This value stores the MotherboardSystemVersion of the reader */
            public fixed byte bMotherboardBootVersion[VERSION_STR_SIZE];            /**< This value stores the MotherboardBootVersion of the reader */
            public fixed byte bCouplerName[COUPLER_NAME_SIZE];                      /**< This value stores the CouplerName of the reader */
            public fixed byte bCouplerSerialNumber[COUPLER_SERIAL_NUMBER_SIZE];     /**< This value stores the CouplerSerialNumber of the reader */
            public fixed byte bCouplerRevision[COUPLER_REVISION_SIZE];              /**< This value stores the CouplerRevision of the reader */
            public fixed byte bCouplerDriverVersion[VERSION_STR_SIZE];              /**< This value stores the CouplerDriverVersion of the reader */
            public ushort wCouplerFPGAVersion;                                      /**< This value stores the CouplerFPGAVersion of the reader */
            public fixed byte bCouplerCalibrationDate[COUPLER_CALC_DATE_SIZE];      /**< Date where the till the coupler is calibrated */
            public uint dwScenarioID;                                               /**< ID of the current in use scenario */
            public fixed byte bSpyFilename[MAX_SPY_FILENAME_SIZE];                  /**< Filename of the spied Tracefile */
            public IntPtr SpyThreadHandle;                                          /**< Handle of the SpyThread */
            public IntPtr GEvtFinished;                                             /**< Event to check if SpyThread is ready */
            public IntPtr GEvtReady;                                                /**< Event to check if SpyThread has finished */
            public byte bCommunicationChannel;                                      /**< Current Communication Channel - Standard or I2C. */
            public ushort wSlaveAddress;                                            /**< Address of the Slave for I2C Communication. */
            public uint dwTHighScl;                                                 /**< Duration of high level of I2C Clock. */
            public uint dwTLowScl;                                                  /**< Duration of low level of I2C Clock. */
            public uint dwTSuDat;                                                   /**< Duration of SDA Setup time for data. */
            public uint dwTHdDat;                                                   /**< Duration of SDA Hold time for data. */
            public IntPtr pBalTimeoutHandling;                                      /**< Pointer to the handler function. */
            public IntPtr pBalTimeoutHandlingContext;                               /**< Context that can be used with handler function. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize,              /**< [In] Size of the global receive buffer. */
            IntPtr pInternalBuffer,         /**< [In] Internal buffer used as a scratch buffer */
            ushort wInternalBufferLen       /**< [In] Length of innternal buffer used as a scratch buffer */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_EnableSpy(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            uint dwEventMask,               /**< [In] EventMask to chose Events to spy. */
            char[] pFileName,               /**< [In] Filename for .mplog File. */
            ushort wDateFormat              /**< [In] Date Format of Trace. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_DisableSpy(
            IntPtr pDataParams              /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_SimAtrString(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            char[] pAtrString               /**< [In] Hex string of the ATR */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_OpenScenario(
            IntPtr pDataParams              /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_ExecuteScenario(
            IntPtr pDataParams,             /**< [In] Pointer to an initialized HAL parameter structure. */
            UInt32 dwTimeout                /**< [In] timeout of command in seconds. 0 direct return, 0xFFFFFFFF infinite timeout */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_CloseScenario(
            IntPtr pDataParams              /**< [In] Pointer to an initialized HAL parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_AddToScenario(
            IntPtr pDataParams,             /**< [In] Pointer to an initialized HAL parameter structure. */
            char[] pAction,                 /**< [In] string with the action. */
            ushort wActionLength,           /**< [In] Length of the action string. */
            char[] pParameter,              /**< [In] string with parameter for the action. */
            ushort wParameterLength         /**< [In] Length of the parameter string. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_GetRxBuffer(
            IntPtr pDataParams,             /**< [In] Pointer to an initialized HAL parameter structure. */
            ref IntPtr ppRxBuffer,          /**< [Out] Pointer to received data */
            ref ushort pRxLength            /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_SetBalTimeoutHandler(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            IntPtr pBalTimeoutHandling,       /**< [In] Pointer to the callback function */
            IntPtr pContext                   /**< [In] Pointer to the context of the callback function */
        );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Micropross_Cmd_CouplerCheckLicense(
            IntPtr pDataParams,               /**< [In] Pointer to an initialized HAL parameter structure. */
            uint dwLicenseId,                 /**< [In] Specifies the license. */
            ref int pValidLicense             /**< [Out] license status. */
            );
        #endregion

        #region DELEGATES
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate Status_t BalTimeoutDelegate(
            IntPtr pContext,        /**< [Out] User Defined Context */
            byte bCouplerNumber     /**< Number of the used coupler */
        );
        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;
        private byte[] m_bIntBuffer;
        private GCHandle m_pIntBuffer;
        private GCHandle contextHandle;
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_CRC_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_CRC_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_bIntBuffer = new byte[wIntBufferSize];

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(m_bIntBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHwContact_Micropross_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (ushort)wIntBufferSize);
        }
        #endregion

        #region CUSTOM_FUNCTIONS
        public Status_t Cmd_EnableSpy(
            uint dwEventMask,
            string pFilename,
            ushort wDateFormat
            )
        {
            pFilename += "\0";
            char[] pFileNameArray = pFilename.ToCharArray();

            return phhalHwContact_Micropross_Cmd_EnableSpy(m_pDataParams,
                dwEventMask,
                pFileNameArray,
                wDateFormat);
        }

        public Status_t Cmd_DisableSpy()
        {
            return phhalHwContact_Micropross_Cmd_DisableSpy(m_pDataParams);
        }

        public Status_t Cmd_SimAtrString(
            string pAtrString
            )
        {
            pAtrString += "\0";
            char[] pAtrArray = pAtrString.ToCharArray();

            return phhalHwContact_Micropross_Cmd_SimAtrString(m_pDataParams,
                pAtrArray);
        }

        public Status_t Cmd_OpenScenario()
        {
            return phhalHwContact_Micropross_Cmd_OpenScenario(m_pDataParams);
        }

        public Status_t Cmd_ExecuteScenario(UInt32 dwTimeout)
        {
            return phhalHwContact_Micropross_Cmd_ExecuteScenario(m_pDataParams, dwTimeout);
        }

        public Status_t Cmd_CloseScenario()
        {
            return phhalHwContact_Micropross_Cmd_CloseScenario(m_pDataParams);
        }

        public Status_t Cmd_AddToScenario(
            string action,
            string pParameter
            )
        {
            char[] pActionArray = (action+"\0").ToCharArray();
            char[] pParameterArray = (pParameter+"\0").ToCharArray();
            return phhalHwContact_Micropross_Cmd_AddToScenario(m_pDataParams,
                pActionArray,
                (ushort)pActionArray.Length,
                pParameterArray,
                (ushort)pParameterArray.Length);
        }

        public Status_t Cmd_GetRxBuffer(
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHwContact_Micropross_Cmd_GetRxBuffer(m_pDataParams, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Cmd_SetBalTimeoutHandler(BalTimeoutDelegate pBalTimeoutHandling, object context)
        {
            if (contextHandle.IsAllocated)
            {
                contextHandle.Free();
            }
            contextHandle = GCHandle.Alloc(context);
            return phhalHwContact_Micropross_Cmd_SetBalTimeoutHandler(m_pDataParams,
                Marshal.GetFunctionPointerForDelegate(pBalTimeoutHandling), (IntPtr)contextHandle);
        }

        public Status_t Cmd_CouplerCheckLicense(
            License licenseId,      /**< [In] Specifies the license. */
            out int pValidLicense   /**< [Out] license status. */
            )
        {
            pValidLicense = 0;

            int validLicense = 0;
            var status = phhalHwContact_Micropross_Cmd_CouplerCheckLicense(m_pDataParams, (uint)licenseId, ref validLicense);
            if (status.Equals(new Status_t()))
                pValidLicense = validLicense;
            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Micropross()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Micropross()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (contextHandle.IsAllocated)
            {
                contextHandle.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Setter & Getter for DataParams structure
        public string Serial
        {
            get
            {
                char[] serial = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < SERIAL_NUMBER_SIZE; i++)
                        {
                            if (pDataParams->bSerialNumber[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        serial = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            serial[i] = (char)pDataParams->bSerialNumber[i];
                        }
                    }
                }
                string serialString = new string(serial);
                if (serialString.StartsWith("Micropross") && serialString.Length == 9)
                {
                    return "Micropross." + serialString.Substring(3, 2) + "." + serialString.Substring(5, 2) + "." + serialString.Substring(7, 2);
                }
                return serialString;
            }
        }
        public DateTime CalibrationDate
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        if (pDataParams->bCouplerCalibrationDate[0] == 0xFF &&
                            pDataParams->bCouplerCalibrationDate[1] == 0xFF &&
                            pDataParams->bCouplerCalibrationDate[2] == 0xFF)
                            return DateTime.MinValue;

                        DateTime date = new DateTime(2000 + pDataParams->bCouplerCalibrationDate[2],
                            pDataParams->bCouplerCalibrationDate[1],
                            pDataParams->bCouplerCalibrationDate[0]);
                        return date;
                    }
                }
            }
        }

        unsafe  private string ByteArrayToString(byte* arr, int maxLength)
        {
            char [] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }

        public string ReaderDescription
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = Serial;
                        value += " MotherboardSystem:" + ByteArrayToString(pDataParams->bMotherboardSystemVersion, VERSION_STR_SIZE);
                        value += " MotherboardBootVersion:" + ByteArrayToString(pDataParams->bMotherboardBootVersion, VERSION_STR_SIZE);
                        value += " Coupler#:" + pDataParams->bCouplerNumber.ToString();
                        value += " CouplerName:" + ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                        value += " CouplerSerialNumber:" + ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                        value += " CouplerRevision:" + ByteArrayToString(pDataParams->bCouplerRevision, COUPLER_REVISION_SIZE);
                        value += " CouplerDriverVersion:" + ByteArrayToString(pDataParams->bCouplerDriverVersion, VERSION_STR_SIZE);
                        value += " CouplerFPGAVersion:" + pDataParams->wCouplerFPGAVersion.ToString();
                        if( CalibrationDate != DateTime.MinValue)
                            value += " CouplerCalibrationDate:" + CalibrationDate.ToString("yyyy-MM-dd");
                        return value;
                    }
                }
            }
        }

        public string ReaderVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = "";// Serial;
                        value += " MotherboardSystem:" + ByteArrayToString(pDataParams->bMotherboardSystemVersion, VERSION_STR_SIZE);
                        value += " MotherboardBootVersion:" + ByteArrayToString(pDataParams->bMotherboardBootVersion, VERSION_STR_SIZE);
                        value += " Coupler#:" + pDataParams->bCouplerNumber.ToString();
                        value += " CouplerName:" + ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                        //value += " CouplerSerialNumber:" + ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                        value += " CouplerRevision:" + ByteArrayToString(pDataParams->bCouplerRevision, COUPLER_REVISION_SIZE);
                        value += " CouplerDriverVersion:" + ByteArrayToString(pDataParams->bCouplerDriverVersion, VERSION_STR_SIZE);
                        value += " CouplerFPGAVersion:" + pDataParams->wCouplerFPGAVersion.ToString();
                        //if (CalibrationDate != DateTime.MinValue)
                        //    value += " CouplerCalibrationDate:" + CalibrationDate.ToString("yyyy-MM-dd");
                        return value;
                    }
                }
            }
        }

        public string CouplerName
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bCouplerName, COUPLER_NAME_SIZE);
                    }
                }
            }
        }

        public string DeviceFamily
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bDeviceFamily, VERSION_STR_SIZE) +
                            ByteArrayToString(pDataParams->bDeviceReferenceWithinFamily, VERSION_STR_SIZE);
                    }
                }
            }
        }

        public string CouplerSerial
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->bCouplerSerialNumber, COUPLER_SERIAL_NUMBER_SIZE);
                    }
                }
            }
        }

        #endregion
    }

	#endregion
}
#endif
