/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palEpcUid
{
    #region DEFINES

    public enum LabelType : byte
    {
        /// <summary> EPC </summary>
        LABEL_EPC = 0x00,
        /// <summary> UID </summary>
        LABEL_UID = 0x01
    }

    public enum CloseSlotOption : byte
    {
        NON_LAST = 0x00,
        LAST = 0x01
    }

    public enum NumSlots : byte
    {
        VALUE_1 = 0x00,
        VALUE_4 = 0x01,
        VALUE_8 = 0x03,
        VALUE_16 = 0x07,
        VALUE_32 = 0x0F,
        VALUE_64 = 0x1F,
        VALUE_128 = 0x3F,
        VALUE_256 = 0x7F,
        VALUE_512 = 0xFF
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DEFINES

        public const byte EPC_LENGTH = 12;          /**< EPC length in bytes without CRC. */
        public const byte IDD_LENGTH = 19;          /**< IDD length in bytes without UID-CRC. */
        public const byte DESTROYCODE_LENGTH = 3;   /**< Length of the destroy code. */

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_ActivateCard(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            byte bTagType,                  /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            byte bNumSlots,                 /**< [In] Number of slots. */
            byte[] pMask,                   /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength,            /**< [In] Length of the mask in bits. */
            byte bHash,                     /**< [In] Used by EPC labels to generate random slot position. */
            byte[] pRxBuffer,               /**< [Out] Data returned by the tag which was detected first. */
            ref byte pRxLength,             /**< [Out] Received data length. */
            ref byte pMoreCardsAvailable    /**< [Out] Indicates if more than one tag was detected */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_BeginRound(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte bTagType,          /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            byte bNumSlots,         /**< [In] Number of slots. */
            byte[] pMask,           /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bHash,             /**< [In] Used by EPC labels to generate random slot position. */
            byte[] pRxBuffer,       /**< [Out] Data returned by the tag which was detected first. */
            ref byte pRxLength      /**< [Out] Received data length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_CloseSlot(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte bOption,           /**< [In] Option; Either #CLOSESLOT_NONLAST or #CLOSESLOT_LAST. */
            byte[] pRxBuffer,       /**< [Out] Data returned by the tag which answers during the next slot. */
            ref byte pRxLength      /**< [Out] Received data length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_FixSlot(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte bTagType,          /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            byte[] pMask,           /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength     /**< [In] Length of the mask in bits. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_Write(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte bTagType,		    /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            byte bBlockNo,          /**< [In] Destination block number. */
            byte bData              /**< [In] Data to write */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_Destroy(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte bTagType,          /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            byte[] pData,           /**< [In] Complete EPC/IDD data of the tag to be destroyed. */
            byte bDataLength,       /**< [In] Length of EPC/IDD data. */
            byte[] pDestroyCode     /**< [In] 3 bytes destroy code. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_GetSerialNo(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte[] pRxBuffer,       /**< [Out] Data returned by the tag which was detected first. */
            ref byte pRxLength      /**< [Out] Received data length. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t ActivateCard(
            LabelType bTagType,             /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            NumSlots bNumSlots,             /**< [In] Number of slots. */
            byte[] pMask,                   /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength,            /**< [In] Length of the mask in bits. */
            byte bHash,                     /**< [In] Used by EPC labels to generate random slot position. */
            out byte[] pRxBuffer,           /**< [Out] Data returned by the tag which was detected first. */
            out byte pMoreCardsAvailable    /**< [Out] Indicates if more than one tag was detected */
            )
        {
            Status_t status;
            byte bRxLength = 0;
            pRxBuffer = new byte[IDD_LENGTH];
            pMoreCardsAvailable = 0;

            status = phpalEpcUid_ActivateCard(
                m_pDataParams,
                (byte)bTagType,
                (byte)bNumSlots,
                pMask,
                bMaskBitLength,
                bHash,
                pRxBuffer,
                ref bRxLength,
                ref pMoreCardsAvailable
                );

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize(ref pRxBuffer, bRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t BeginRound(
            LabelType bTagType,     /**< [In] Tag type; Either #LABEL_EPC or #LABEL_UID. */
            NumSlots bNumSlots,     /**< [In] Number of slots. */
            byte[] pMask,           /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bHash,             /**< [In] Used by EPC labels to generate random slot position. */
            out byte[] pRxBuffer    /**< [Out] Data returned by the tag which was detected first. */
            )
        {
            Status_t status;
            byte bRxLength = 0;
            pRxBuffer = new byte[IDD_LENGTH];

            status = phpalEpcUid_BeginRound(
                m_pDataParams,
                (byte)bTagType,
                (byte)bNumSlots,
                pMask,
                bMaskBitLength,
                bHash,
                pRxBuffer,
                ref bRxLength
                );

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize(ref pRxBuffer, bRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t CloseSlot(
            CloseSlotOption bOption,    /**< [In] Option; Either #CLOSESLOT_NONLAST or #CLOSESLOT_LAST. */
            out byte[] pRxBuffer        /**< [Out] Data returned by the tag which was detected first. */
            )
        {
            Status_t status;
            byte bRxLength = 0;
            pRxBuffer = new byte[IDD_LENGTH];

            status = phpalEpcUid_CloseSlot(
                m_pDataParams,
                (byte)bOption,
                pRxBuffer,
                ref bRxLength
                );

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize(ref pRxBuffer, bRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t FixSlot(
            LabelType bTagType, /**< [In] Tag type: either LABEL_EPC or LABEL_UID. */
            byte[] pMask,       /**< [In] Fractional or complete EPC/IDD for selecting certain labels. */
            byte bMaskBitLength /**< [In] Length of the mask in bits. */
            )
        {
            return phpalEpcUid_FixSlot(m_pDataParams, (byte)bTagType, pMask, bMaskBitLength);
        }

        public Status_t Write(
            LabelType bTagType, /**< [In] Tag type: either LABEL_EPC or LABEL_UID. */
            byte bBlockNo,      /**< [In] Destination block number. */
            byte bData          /**< [In] Data to write */
            )
        {
            return phpalEpcUid_Write(m_pDataParams, (byte)bTagType, bBlockNo, bData);
        }

        public Status_t Destroy(
            LabelType bTagType,     /**< [In] Tag type: either LABEL_EPC or LABEL_UID. */
            byte[] pData,           /**< [In] Complete EPC/IDD data of the tag to be destroyed. */
            byte[] pDestroyCode     /**< [In] 3 bytes destroy code. */
            )
        {
            return phpalEpcUid_Destroy(m_pDataParams, (byte)bTagType, pData, (byte)((pData == null) ? 0 : pData.Length), pDestroyCode);
        }

        public Status_t GetSerialNo(
            out byte[] pRxBuffer    /**< [Out] Data returned by the tag which was detected first. */
            )
        {
            Status_t status;
            byte bRxLength = 0;
            pRxBuffer = new byte[IDD_LENGTH];

            status = phpalEpcUid_GetSerialNo(
                m_pDataParams,
                pRxBuffer,
                ref bRxLength
                );

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize(ref pRxBuffer, bRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palEpcUid.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                          /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;               /**< Pointer to the parameter structure of the underlying layer.*/
            public byte bUidValid;                      /**< Whether current UID is valid or not. */
            public byte bLabelType;                     /**< Type of selected label. */
            public byte bMaskBitLength;                 /**< Length of the mask. The Mask itself is stored in the LabelInfo union. */
            public fixed byte LabelInfo[IDD_LENGTH];    /**< Array holding the activated UID.*/
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalEpcUid_Sw_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams               /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            );

        #endregion

        #region INIT

        public Status_t Init(Hal.Generic pHal)
        {
            return phpalEpcUid_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalEpcUid_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
