/*
 * Copyright 2013, 2016 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI14443p3a
{
    #region Enumerations
    /// <summary>
    /// Type of cascade levels.
    /// </summary>
    public enum CascadeLevel : byte
    {
        /// <summary> Cascade Level 1 code for ISO14443-3A anticollision/select command. </summary>
        LEVEL_1 = 0x93,

        /// <summary> Cascade Level 2 code for ISO14443-3A anticollision/select command. </summary>
        LEVEL_2 = 0x95,

        /// <summary> Cascade Level 3 code for ISO14443-3A anticollision/select command. </summary>
        LEVEL_3 = 0x97
    };

    /// <summary>
    /// The configuration to be used for SetConfig / GetConfig.
    /// </summary>
    public enum Config
    {
        /// <summary>
        /// Option for Get/Set Config to enable/disable Silent mode activation
        /// (<see cref="Generic.ActivateCard(byte[], out byte[], out byte, out byte)"/>).
        ///		By default the configuration will be disabled.
        ///		<see cref="SlientMode.DISABLED"/>
        ///		<see cref="SlientMode.ENABLED"/>
        /// </summary>
        SILENT_MODE = 0x00A1
    }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    /// <summary>
    /// Values for SilentMode comfiguration.
    /// </summary>
    public enum SlientMode
    {
        /// <summary>
        /// Silent mode is disabled. ReqA / WupA command will be used. This is the default
        /// option for <see cref="Config.SILENT_MODE"/>
        /// </summary>
        DISABLED = 0x00,

        /// <summary>
        /// Silent mode is enabled. ReqS / WupS command will be used.
        /// </summary>
        ENABLED = 0x01
    }
#endif
    #endregion

    #region Interface

    /// <summary>
    /// Interface definition of interfaces a PAL object needs to implement.
    /// </summary>
    public interface IPalI14443p3a
    {
        /// <summary>
        /// Perform RequestA.
        /// </summary>
        /// <param name="pAtqa"></param>
        /// <returns></returns>
        Status_t RequestA (
            out byte[] pAtqa        /**< [Out] Response of RequestA command. The buffer cannot be null and size should be 2 bytes long. */
            );

        /// <summary>
        /// Perform RequestS.
        /// </summary>
        /// <param name="pAtqs"></param>
        /// <returns></returns>
        Status_t RequestS (
            out byte[] pAtqs        /**< [Out] Response of RequestS command. The buffer cannot be null and size should be 2 bytes long. */
            );

        /// <summary>
        /// Perform WakeUpA.
        /// </summary>
        /// <param name="pAtqa"></param>
        /// <returns></returns>
        Status_t WakeUpA (
            out byte[] pAtqa        /**< [Out] Response of WakeUpA command. The buffer cannot be null and size should be 2 bytes long. */
            );

        /// <summary>
        /// Perform WakeUpS.
        /// </summary>
        /// <param name="pAtqs"></param>
        /// <returns></returns>
        Status_t WakeUpS (
            out byte[] pAtqs        /**< [Out] Response of WakeUpS command. The buffer cannot be null and size should be 2 bytes long. */
            );

        /// <summary>
        /// Perform HaltA.
        /// </summary>
        /// <returns></returns>
        Status_t HaltA ();

        /// <summary>
        /// Perform Anticollision.
        /// </summary>
        /// <param name="bCascadeLevel"></param>
        /// <param name="pUidIn"></param>
        /// <param name="bNvbUidIn"></param>
        /// <param name="pUidOut"></param>
        /// <param name="pNvbUidOut"></param>
        /// <returns></returns>
        Status_t Anticollision (
            byte bCascadeLevel,                         /**< [In] Cascade level code.
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                         */
            byte[] pUidIn,                             /**< [In] Known Uid, can be NULL; uint8_t[0-4]. */
            byte bNvbUidIn,                             /**< [In] Number of valid bits of UidIn. MSB codes the valid bytes, LSB codes the valid bits. */
            out byte[] pUidOut,                         /**< [Out] Complete Uid; uint8_t[4]. */
            out byte pNvbUidOut                             /**< [Out] Number of valid bits of UidOut. MSB codes the valid bytes, LSB codes the valid bits. */
            );

        /// <summary>
        /// Perform Select.
        /// </summary>
        /// <param name="bCascadeLevel"></param>
        /// <param name="pUidIn"></param>
        /// <param name="pSak"></param>
        /// <returns></returns>
        Status_t Select (
            byte bCascadeLevel,                         /**< [In] Cascade level code.
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                         *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                         */
            byte[] pUidIn,                              /**< [In] Uid; uint8_t[4]. */
            out byte pSak                               /**< [Out] Select Acknowledge; uint8_t. */
            );

        /// <summary>
        /// Perform ActivateCard.
        /// </summary>
        /// <param name="pUidIn"></param>
        /// <param name="pUidOut"></param>
        /// <param name="pSak"></param>
        /// <param name="pMoreCardsAvailable"></param>
        /// <returns></returns>
        Status_t ActivateCard (
            byte[] pUidIn,                              /**< [In] Known Uid, can be NULL if bLenUidIn is 0; uint8_t[4/7/10]. */
            out byte[] pUidOut,                         /**< [Out] Complete Uid; uint8_t[10]. */
            out byte pSak,                              /**< [Out] Select Acknowledge; uint8_t. */
            out byte pMoreCardsAvailable                /**< [Out] Whether there are more cards in the field or not; uint8_t. */
            );

        /// <summary>
        /// Perform Exchange.
        /// </summary>
        /// <param name="wOption"></param>
        /// <param name="pTxBuffer"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        Status_t Exchange (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            );

        /// <summary>
        /// Perform GetSerialNo.
        /// </summary>
        /// <param name="pUidOut"></param>
        /// <returns></returns>
        Status_t GetSerialNo (
            out byte[] pUidOut      /**< [Out] Last Complete Uid; uint8_t[10]. */
            );

        /// <summary>
        /// Perform GetConfig.
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="pValue"></param>
        /// <returns></returns>
        Status_t GetConfig (
            Config wConfig,         /**< [In] Configuration to read. Will be one of the below values. \arg #PHPAL_I14443P3A_SILENT_MODE */
            out ushort pValue       /**< [Out] The value for the mentioned configuration. */
            );

        /// <summary>
        /// Perform SetConfig.
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="wValue"></param>
        /// <returns></returns>
        Status_t SetConfig (
            Config wConfig,         /**< [In] Configuration to set. Will be one of the below values. \arg #PHPAL_I14443P3A_SILENT_MODE */
            ushort wValue           /**< [In] The value for the mentioned configuration. */
            );
    }
    #endregion

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    public abstract class Generic
    {
        #region Constants
        public const int TIMEOUT_DEFAULT_MS = 10;
        public const int EXT_TIME_US = 60;
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_RequestA ( IntPtr pDataParams, byte[] pAtqA );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_RequestS ( IntPtr pDataParams, byte[] pAtqS );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_WakeUpA ( IntPtr pDataParams, byte[] pAtqA );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_WakeUpS ( IntPtr pDataParams, byte[] pAtqS );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_HaltA ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Anticollision ( IntPtr pDataParams, byte bCascadeLevel, byte[] pUidIn,
            byte bNvbUidIn, byte[] pUidOut, ref byte pNvbUidOut );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Select ( IntPtr pDataParams, byte bCascadeLevel, byte[] pUidIn, ref byte pSak );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_ActivateCard ( IntPtr pDataParams, byte[] pUidIn, byte bLenUidIn, byte[] pUidOut,
            ref byte pLenUidOut, ref byte pSak, ref byte pMoreCardsAvailable );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Exchange ( IntPtr pDataParams, ushort wOption, byte[] pTxBuffer, ushort wTxLength,
            ref IntPtr ppRxBuffer, ref ushort pRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_GetSerialNo ( IntPtr pDataParams, byte[] pRxBuffer, ref byte pRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );
        #endregion

        #region Wrapper Functions
        /// <summary>
        /// Perform a ISO14443-3A Request A command.
        ///
        /// Note: The "Request Guard Time" <em>(see 6.2.2, ISO/IEC 14443-3:2009(E))</em> is mandatory and is
        ///		  neither implemented here nor implemented in every \ref phhalHw layer.\n
        ///		  Make sure that either the used HAL or the used application does comply to this rule.
        /// </summary>
        ///
        /// <param name="pAtqA">Response of RequestA command.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t RequestA ( out byte[] pAtqA )
        {
            pAtqA = new byte[2];
            return phpalI14443p3a_RequestA ( m_pDataParams, pAtqA );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Perform a ISO14443-3A Request S command. This command is support for software mode only.
        ///
        /// Note: The "Request Guard Time" <em>(see 6.2.2, ISO/IEC 14443-3:2009(E))</em> is mandatory and is
        ///		  neither implemented here nor implemented in every \ref phhalHw layer.\n
        ///		  Make sure that either the used HAL or the used application does comply to this rule.
        /// </summary>
        ///
        /// <param name="aAtqS">Response of RequestS command.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t RequestS ( out byte[] aAtqS )
        {
            Status_t oStatus;

            aAtqS = new byte[2];
            oStatus = phpalI14443p3a_RequestS ( m_pDataParams, aAtqS );
            if ( !oStatus.Equals ( new Status_t () ) )
                aAtqS = null;

            return oStatus;
        }
#endif

        /// <summary>
        /// Perform a ISO14443-3A Wakeup A command.
        ///
        /// Note: The "Request Guard Time" <em>(see 6.2.2, ISO/IEC 14443-3:2009(E))</em> is mandatory and is
        ///		  neither implemented here nor implemented in every \ref phhalHw layer.\n
        ///		  Make sure that either the used HAL or the used application does comply to this rule.
        /// </summary>
        ///
        /// <param name="pAtqA">Response of WakeUpA command.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WakeUpA ( out byte[] pAtqA )
        {
            pAtqA = new byte[2];
            return phpalI14443p3a_WakeUpA ( m_pDataParams, pAtqA );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Perform a ISO14443-3A Wakeup S command. This command is support for software mode only.
        ///
        /// Note: The "Request Guard Time" <em>(see 6.2.2, ISO/IEC 14443-3:2009(E))</em> is mandatory and is
        ///		  neither implemented here nor implemented in every \ref phhalHw layer.\n
        ///		  Make sure that either the used HAL or the used application does comply to this rule.
        /// </summary>
        ///
        /// <param name="aAtqS">Response of WakeUpS command.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WakeUpS ( out byte[] aAtqS )
        {
            Status_t oStatus;

            aAtqS = new byte[2];
            oStatus = phpalI14443p3a_WakeUpS ( m_pDataParams, aAtqS );
            if ( !oStatus.Equals ( new Status_t () ) )
                aAtqS = null;

            return oStatus;
        }
#endif

        /// <summary>
        /// Perform a ISO14443-3A Halt command.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t HaltA ()
        {
            return phpalI14443p3a_HaltA ( m_pDataParams );
        }

        /// <summary>
        /// Perform a ISO14443-3A Anticollision or Select command.
        ///		bNvbUidIn != 0x40: Perform Anticollision command.
        ///		bNvbUidIn = 0x40: Perform Select command.
        /// </summary>
        ///
        /// <param name="bCascadeLevel">Cascade level code.
        ///									<see cref="CascadeLevel.LEVEL_1"/>
        ///									<see cref="CascadeLevel.LEVEL_2"/>
        ///									<see cref="CascadeLevel.LEVEL_3"/></param>
        /// <param name="pUidIn">Known Uid, can be NULL</param>
        /// <param name="bNvbUidIn">Number of valid bits of UidIn. MSB codes the valid bytes,
        ///							LSB codes the valid bits.</param>
        /// <param name="pUidOut">Complete Uid</param>
        /// <param name="pNvbUidOut">Number of valid bits of UidOut. MSB codes the valid bytes,
        ///							 LSB codes the valid bits.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Anticollision ( byte bCascadeLevel, byte[] pUidIn, byte bNvbUidIn, out byte[] pUidOut, out byte pNvbUidOut )
        {
            if ( bNvbUidIn < 0x40 )
            {
                pUidOut = new byte[4];
            }
            else
            {
                pUidOut = new byte[1];
            }

            pNvbUidOut = 0;
            return phpalI14443p3a_Anticollision ( m_pDataParams, bCascadeLevel, pUidIn, bNvbUidIn, pUidOut, ref pNvbUidOut );
        }

        /// <summary>
        /// Perform a ISO14443-3A Select command.
        /// </summary>
        ///
        /// <param name="bCascadeLevel">Cascade level code.
        ///									<see cref="CascadeLevel.LEVEL_1"/>
        ///									<see cref="CascadeLevel.LEVEL_2"/>
        ///									<see cref="CascadeLevel.LEVEL_3"/></param>
        /// <param name="pUidIn">Uid</param>
        /// <param name="pSak">Select Acknowledge</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Select ( byte bCascadeLevel, byte[] pUidIn, out byte pSak )
        {
            pSak = 0;
            return phpalI14443p3a_Select ( m_pDataParams, bCascadeLevel, pUidIn, ref pSak );
        }

        /// <summary>
        /// Perform ISO14443-3A ReqA/WupA or ReqS/WupS and Anticollision/Select commands for all cascade levels.
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        ///
        /// If CardUid not given ( bLenUidIn is '0')
        ///		If SilentMode is disabled (<see cref="SlientMode.DISABLED"/>): ReqA is performed.
        ///		If SilentMode is enabled (<see cref="SlientMode.ENABLED"/>): ReqS is performed.
        ///
        /// If CardUid is given (bLenUidIn is not '0')
        ///		If SilentMode is disabled (<see cref="SlientMode.DISABLED"/>): WupA is performed.
        ///		If SilentMode is enabled (<see cref="SlientMode.ENABLED"/>): WupS is performed.
#endif
        /// </summary>
        ///
        /// <param name="pUidIn">Known Uid, can be NULL if bLenUidIn is 0; uint8_t[4/7/10].</param>
        /// <param name="pUidOut">Complete Uid; uint8_t[10].</param>
        /// <param name="pSak">Select Acknowledge</param>
        /// <param name="pMoreCardsAvailable">Whether there are more cards in the field or not</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ActivateCard ( byte[] pUidIn, out byte[] pUidOut, out byte pSak, out byte pMoreCardsAvailable )
        {
            Status_t status;
            pUidOut = new byte[10];
            pSak = 0;
            pMoreCardsAvailable = 0;
            byte bUidLength;

            if ( pUidIn != null )
            {
                bUidLength = ( byte ) pUidIn.Length;
            }
            else
            {
                bUidLength = 0;
            }

            status = phpalI14443p3a_ActivateCard ( m_pDataParams, pUidIn, bUidLength, pUidOut, ref bUidLength, ref pSak, ref pMoreCardsAvailable );
            Array.Resize<byte> ( ref pUidOut, bUidLength );

            return status;
        }

        /// <summary>
        /// Perform ISO14443-3A Data Exchange with Picc.
        /// </summary>
        ///
        /// <param name="wOption">Option parameter.
        ///							<see cref="ExchangeOptions.DEFAULT"/>
        ///							<see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///							<see cref="ExchangeOptions.BUFFER_CONT"/>
        ///							<see cref="ExchangeOptions.BUFFER_LAST"/>
        ///
        ///						  Alternatively, the following bits can be combined
        ///							<see cref="ExchangeOptions.BUFFERED_BIT"/>
        ///							<see cref="ExchangeOptions.LEAVE_BUFFER_BIT"/></param>
        /// <param name="pTxBuffer">Data to transmit.</param>
        /// <param name="pRxBuffer">Received data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Exchange ( int wOption, byte[] pTxBuffer, out byte[] pRxBuffer )
        {
            Status_t status;
            ushort wTxLength = 0;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            if ( pTxBuffer != null )
            {
                wTxLength = ( ushort ) pTxBuffer.Length;
            }

            status = phpalI14443p3a_Exchange (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                wTxLength,
                ref pRxBufferInt,
                ref wRxLength );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        /// <summary>
        /// Retrieve the serial number.
        /// </summary>
        ///
        /// <param name="pSnr">Last Complete Uid</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetSerialNumber ( out byte[] pSnr )
        {
            Status_t status;
            byte bRxLength = 0;

            pSnr = new byte[10];
            status = phpalI14443p3a_GetSerialNo ( m_pDataParams, pSnr, ref bRxLength );
            Array.Resize<byte> ( ref pSnr, bRxLength );

            return status;
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///							<see cref="Config.SILENT_MODE"/></param>
        /// <param name="pValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
        {
            return phpalI14443p3a_GetConfig ( m_pDataParams, wConfig, ref pValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///							<see cref="Config.SILENT_MODE"/></param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phpalI14443p3a_SetConfig ( m_pDataParams, wConfig, wValue );
        }
#endif
        #endregion

        #region Memory Maping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        #endregion
    }
    #endregion Generic

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    public class Sw : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// Software PAL-ISO14443P3A parameter structure.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            public IntPtr pHalDataParams;

            /// <summary> Array holding the activated UID. </summary>
            public fixed byte abUid[10];

            /// <summary> Length of UID stored in \c abUid. </summary>
            public byte bUidLength;

            /// <summary> Whether the UID is complete (\c 1) or not (\c 0). </summary>
            public byte bUidComplete;

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
            /// <summary>
            /// Whether the SilentMode is enabled (<see cref="SlientMode.ENABLED"/>)
            /// or disabled (<see cref="SlientMode.DISABLED"/>).
            /// </summary>
            public byte bSilentMode;
#endif
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p3a_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
		/// <summary>
		/// Initialise this layer.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHal">Pointer to HAL layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI14443p3a_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Software

    #region SamAV2_X
    /// <summary>
    /// Class for Sam AV2 X Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV2_X : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// ISO14443-3A parameter structure for SAM AV2 X.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;

            /// <summary> Array holding the activated UID. </summary>
            public fixed byte abUid[10];

            /// <summary> Length of UID stored in \c abUid. </summary>
            public byte bUidLength;

            /// <summary> Whether the UID is complete (\c 1) or not (\c 0). </summary>
            public byte bUidComplete;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV2 pHal )
        {
            return phpalI14443p3a_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion SamAV2_X

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X
    /// <summary>
    /// Class for Sam AV3 X Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// ISO14443-3A parameter structure for SAM AV3 X.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;

            /// <summary> Array holding the activated UID. </summary>
            public fixed byte abUid[10];

            /// <summary> Length of UID stored in \c abUid. </summary>
            public byte bUidLength;

            /// <summary> Whether the UID is complete (\c 1) or not (\c 0). </summary>
            public byte bUidComplete;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHal )
        {
            return phpalI14443p3a_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion SamAV3_X
#endif

    #region Rd70x
    /// <summary>
    /// Class for Rd70x layer initialization interface and data params.
    /// </summary>
    public class Rd70x : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P3A parameter structure for Rd70x reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;

            /// <summary> Array holding the activated UID. </summary>
            public fixed byte abUid[10];

            /// <summary> Length of UID stored in \c abUid. </summary>
            public byte bUidLength;

            /// <summary> Whether the UID is complete (\c 1) or not (\c 0). </summary>
            public byte bUidComplete;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Rd70x_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Rd70x pHal )
        {
            return phpalI14443p3a_Rd70x_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Rd70x ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Rd70x ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Rd70x

    #region Rd710
    /// <summary>
    /// Class for Rd710 layer initialization interface and data params.
    /// </summary>
    public class Rd710 : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P3A parameter structure for Rd710 reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Rd710_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Rd710 pHal )
        {
            return phpalI14443p3a_Rd710_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                pHal.m_pDataParams );
        }
#if DEBUG
		/// <summary>
		/// Initialise this layer.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHal">Pointer to HAL layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
		{
			return phpalI14443p3a_Rd710_Init(	ref m_DataParamsInt[0], (ushort)wDataParamSize,
				pHal.m_pDataParams );
		}
#endif
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Rd710 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Rd710 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Rd710

    #region PCSC
    /// <summary>
    /// Class for PCSC layer initialization interface and data params.
    /// </summary>
    public class PCSC : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P3A parameter structure for PCSC reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;

            /// <summary> Pointer to the parameter structure of the underlying pcsc bal. Is used to access the pcsc parameter </summary>
            public IntPtr pBalDataParams;

            // <summary> pointer to the parameter structure of the underlying pcsc hal. Is used to access the pcsc parameter </summary>
            public IntPtr pHalPcscDataParams;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Pcsc_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialise PCSC PAL ISO14443-3A layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p3a_Pcsc_Init (
                                                ref m_DataParamsInt[0],
                                                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                                pHal.m_pDataParams );
        }
#if DEBUG
		/// <summary>
		/// Initialise PCSC PAL ISO14443-3A layer.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHal">Pointer to HAL layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
        {
            return phpalI14443p3a_Pcsc_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public PCSC ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~PCSC ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion PCSC

#if PACKAGE_INTERNAL
    #region Callback
    /// <summary>
    /// Class for Callback layer initialization interface and data params.
    /// </summary>
    public class Callback : palI14443p3a.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P3A parameter structure for Callback reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pRequestACb;
            IntPtr pRequestSCb;
            IntPtr pWakeUpACb;
            IntPtr pWakeUpSCb;
            IntPtr pHaltACb;
            IntPtr pAnticollisionCb;
            IntPtr pSelectCb;
            IntPtr pActivateCardCb;
            IntPtr pExchangeCb;
            IntPtr pGetSerialNoCb;
            IntPtr pGetConfigCb;
            IntPtr pSetConfigCb;
            IntPtr pCbContext;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Callback_Init (
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pRequestACb,
            IntPtr pRequestSCb,
            IntPtr pWakeUpACb,
            IntPtr pWakeUpSCb,
            IntPtr pHaltACb,
            IntPtr pAnticollisionCb,
            IntPtr pSelectCb,
            IntPtr pActivateCardCb,
            IntPtr pExchangeCb,
            IntPtr pGetSerialNoCb,
            IntPtr pGetConfigCb,
            IntPtr pSetConfigCb,
            IntPtr pCbContext );
        #endregion DLL Imports

        #region DELEGATES

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort RequestADelegate (
            IntPtr pAtqa,                                                                           /**< [Out] Response of RequestA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort RequestSDelegate (
            IntPtr pAtqs,                                                                           /**< [Out] Response of RequestS command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort WakeUpADelegate (
            IntPtr pAtqa,                                                                           /**< [Out] Response of WakeUpA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort WakeUpSDelegate (
            IntPtr pAtqs,                                                                           /**< [Out] Response of WakeUpS command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort HaltADelegate (
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort AnticollisionDelegate (
            byte bCascadeLevel,                                                                     /**< [In] Cascade level code.
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                                                                     */
            IntPtr pUidIn,                                                                          /**< [In] Known Uid, can be NULL; uint8_t[0-4]. */
            byte bNvbUidIn,                                                                         /**< [In] Number of valid bits of UidIn. MSB codes the valid bytes,
                                                                                                     *        LSB codes the valid bits.
                                                                                                     */
            IntPtr pUidOut,                                                                         /**< [Out] Complete Uid; uint8_t[4]. */
            ref byte pNvbUidOut,                                                                    /**< [Out] Number of valid bits of UidOut. MSB codes the valid bytes,
                                                                                                     *         LSB codes the valid bits.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SelectDelegate (
            byte bCascadeLevel,                                                                     /**< [In] Cascade level code.
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                                                                     */
            IntPtr pUidIn,                                                                          /**< [In] Uid; uint8_t[4]. */
            ref byte pSak,                                                                          /**< [Out] Select Acknowledge; uint8_t. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort ActivateCardDelegate (
            IntPtr pUidIn,                                                                          /**< [In] Known Uid, can be NULL if bLenUidIn is 0; uint8_t[4/7/10]. */
            byte bLenUidIn,                                                                         /**< [In] Length of known Uid; 0/4/7/10. */
            IntPtr pUidOut,                                                                         /**< [Out] Complete Uid; uint8_t[10]. */
            ref byte pLenUidOut,                                                                    /**< [Out] Length of Uid; 4/7/10. */
            ref byte pSak,                                                                          /**< [Out] Select Acknowledge; uint8_t. */
            ref byte pMoreCardsAvailable,                                                           /**< [Out] Whether there are more cards in the field or not; uint8_t. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort ExchangeDelegate (
            ushort wOption,                                                                         /**< [In] Option parameter. */
            IntPtr pTxBuffer,                                                                       /**< [In] Data to transmit. */
            ushort wTxLength,                                                                       /**< [In] Length of data to transmit. */
            ref IntPtr ppRxBuffer,                                                                  /**< [Out] Pointer to received data. */
            ref ushort pRxLength,                                                                   /**< [Out] number of received data bytes. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort GetSerialNoDelegate (
            IntPtr pUidOut,                                                                         /**< [Out] Last Complete Uid; uint8_t[10]. */
            ref byte pLenUidOut,                                                                    /**< [Out] Length of Uid; 4/7/10. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort GetConfigDelegate (
            ushort wConfig,                                                                         /**< [In] Configuration to read. Will be one of the below values.
                                                                                                     *          \arg #PHPAL_I14443P3A_SILENT_MODE
                                                                                                     */
            ref ushort pValue,                                                                      /**< [Out] The value for the mentioned configuration. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SetConfigDelegate (
            ushort wConfig,                                                                         /**< [In] Configuration to set. Will be one of the below values.
                                                                                                     *          \arg #PHPAL_I14443P3A_SILENT_MODE
                                                                                                     */
            ushort wValue,                                                                          /**< [In] The value for the mentioned configuration. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            );

        #endregion DELEGATES

        private ushort cbRequestA (
            IntPtr pAtqa,                                                                           /**< [Out] Response of RequestA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtqaTmp;

            if ( pAtqa == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.RequestA ( out pAtqaTmp );

            if ( pAtqaTmp != null && pAtqaTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtqaTmp.Length > 2 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtqaTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAtqa, dIndex, pAtqaTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbRequestS (
            IntPtr pAtqs,                                                                           /**< [Out] Response of RequestA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtqsTmp;

            if ( pAtqs == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.RequestS ( out pAtqsTmp );

            if ( pAtqsTmp != null && pAtqsTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtqsTmp.Length > 2 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtqsTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAtqs, dIndex, pAtqsTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbWakeUpA (
            IntPtr pAtqa,                                                                           /**< [Out] Response of RequestA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtqaTmp;

            if ( pAtqa == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.WakeUpA ( out pAtqaTmp );

            if ( pAtqaTmp != null && pAtqaTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtqaTmp.Length > 2 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtqaTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAtqa, dIndex, pAtqaTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbWakeUpS (
            IntPtr pAtqs,                                                                           /**< [Out] Response of RequestA command. The buffer cannot be null and
                                                                                                     *         size should be 2 bytes long.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtqsTmp;

            if ( pAtqs == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.WakeUpS ( out pAtqsTmp );

            if ( pAtqsTmp != null && pAtqsTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtqsTmp.Length > 2 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtqsTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAtqs, dIndex, pAtqsTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbHaltA (
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            // Execute callback function
            return m_ExtPal.HaltA ();
        }

        private ushort cbAnticollision (
            byte bCascadeLevel,                                                                     /**< [In] Cascade level code.
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                                                                     */
            IntPtr pUidIn,                                                                          /**< [In] Known Uid, can be NULL; uint8_t[0-4]. */
            byte bNvbUidIn,                                                                         /**< [In] Number of valid bits of UidIn. MSB codes the valid bytes,
                                                                                                     *        LSB codes the valid bits.
                                                                                                     */
            IntPtr pUidOut,                                                                         /**< [Out] Complete Uid; uint8_t[4]. */
            ref byte pNvbUidOut,                                                                    /**< [Out] Number of valid bits of UidOut. MSB codes the valid bytes,
                                                                                                     *         LSB codes the valid bits.
                                                                                                     */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pUidInTmp = null;
            byte[] pUidOutTmp;

            // Prepare input buffer(s)
            if ( pUidIn != IntPtr.Zero )
            {
                pUidInTmp = new byte[4];
                for ( int dIndex = 0; dIndex < 4; dIndex++ )
                {
                    pUidInTmp[dIndex] = Marshal.ReadByte ( pUidIn, dIndex );
                }
            }

            // Execute callback function
            status = m_ExtPal.Anticollision ( bCascadeLevel, pUidInTmp, bNvbUidIn, out pUidOutTmp, out pNvbUidOut );

            if ( pUidOutTmp != null && pUidOutTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pUidOutTmp.Length > 4 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pUidOutTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pUidOut, dIndex, pUidOutTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbSelect (
            byte bCascadeLevel,                                                                     /**< [In] Cascade level code.
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_1
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_2
                                                                                                     *          \arg #PHPAL_I14443P3A_CASCADE_LEVEL_3
                                                                                                     */
            IntPtr pUidIn,                                                                          /**< [In] Uid; uint8_t[4]. */
            ref byte pSak,                                                                          /**< [Out] Select Acknowledge; uint8_t. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            byte[] pUidInTmp = null;

            // Prepare input buffer(s)
            if ( pUidIn != IntPtr.Zero )
            {
                pUidInTmp = new byte[4];
                for ( int dIndex = 0; dIndex < 4; dIndex++ )
                {
                    pUidInTmp[dIndex] = Marshal.ReadByte ( pUidIn, dIndex );
                }
            }

            // Execute callback function
            return m_ExtPal.Select ( bCascadeLevel, pUidInTmp, out pSak );
        }

        private ushort cbActivateCard (
            IntPtr pUidIn,                                                                          /**< [In] Known Uid, can be NULL if bLenUidIn is 0; uint8_t[4/7/10]. */
            byte bLenUidIn,                                                                         /**< [In] Length of known Uid; 0/4/7/10. */
            IntPtr pUidOut,                                                                         /**< [Out] Complete Uid; uint8_t[10]. */
            ref byte pLenUidOut,                                                                    /**< [Out] Length of Uid; 4/7/10. */
            ref byte pSak,                                                                          /**< [Out] Select Acknowledge; uint8_t. */
            ref byte pMoreCardsAvailable,                                                           /**< [Out] Whether there are more cards in the field or not; uint8_t. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pUidInTmp = null;
            byte[] pUidOutTmp;

            // Prepare input buffer(s)
            if ( pUidIn != IntPtr.Zero )
            {
                pUidInTmp = new byte[bLenUidIn];
                for ( int dIndex = 0; dIndex < bLenUidIn; dIndex++ )
                {
                    pUidInTmp[dIndex] = Marshal.ReadByte ( pUidIn, dIndex );
                }
            }

            // Execute callback function
            status = m_ExtPal.ActivateCard ( pUidInTmp, out pUidOutTmp, out pSak, out pMoreCardsAvailable );

            if ( pUidOutTmp != null && pUidOutTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pUidOutTmp.Length > 10 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                pLenUidOut = ( byte ) pUidOutTmp.Length;

                for ( int dIndex = 0; dIndex < pUidOutTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pUidOut, dIndex, pUidOutTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbExchange (
            ushort wOption,                                                                         /**< [In] Option parameter. */
            IntPtr pTxBuffer,                                                                       /**< [In] Data to transmit. */
            ushort wTxLength,                                                                       /**< [In] Length of data to transmit. */
            ref IntPtr ppRxBuffer,                                                                  /**< [Out] Pointer to received data. */
            ref ushort pRxLength,                                                                   /**< [Out] number of received data bytes. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pTxBufferTmp = new byte[wTxLength];
            byte[] pRxBufferTmp;
            int dwRxOffset;

            // Prepare input buffer(s)
            for ( int dIndex = 0; dIndex < wTxLength; dIndex++ )
            {
                pTxBufferTmp[dIndex] = Marshal.ReadByte ( pTxBuffer, dIndex );
            }

            // Reset Receive length
            pRxLength = 0;

            // Execute callback function
            status = m_ExtPal.Exchange ( ( int ) wOption, pTxBufferTmp, out pRxBufferTmp );

            // Return pointer to global buffer
            if ( pRxBufferTmp != null && pRxBufferTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pRxBufferTmp.Length > m_RxBuffer.Length )
                {
                    return new Status_t ( Error_CompCode.HAL, Error_Comm.BUFFER_OVERFLOW );
                }

                //TODO: Check if this is required
                // Get RxStartPos
                dwRxOffset = 0;
                //var getStatus = m_ExtHal.GetConfig(NxpRdLibNet.Hal.Config.RXBUFFER_STARTPOS, out dwRxOffset);
                //if (!getStatus.Equals(new Status_t()))
                //    return getStatus;

                // Copy only new Rx-Contents to persistent HAL buffer (to prevent override of already restored data)
                System.Array.Copy ( pRxBufferTmp, dwRxOffset, m_RxBuffer, dwRxOffset, pRxBufferTmp.Length - dwRxOffset );

                // Set Output Buffer to persistent HAL buffer pointer
                ppRxBuffer = m_pRxBuffer.AddrOfPinnedObject ();

                // Set number of received bytes
                pRxLength = ( ushort ) pRxBufferTmp.Length;
            }

            return status;
        }

        private ushort cbGetSerialNo (
            IntPtr pUidOut,                                                                         /**< [Out] Last Complete Uid; uint8_t[10]. */
            ref byte pLenUidOut,                                                                    /**< [Out] Length of Uid; 4/7/10. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pUidOutTmp;

            // Execute callback function
            status = m_ExtPal.GetSerialNo ( out pUidOutTmp );

            if ( pUidOutTmp != null && pUidOutTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pUidOutTmp.Length > 10 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P3A, Error_Comm.BUFFER_OVERFLOW );
                }

                pLenUidOut = ( byte ) pUidOutTmp.Length;

                for ( int dIndex = 0; dIndex < pUidOutTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pUidOut, dIndex, pUidOutTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbGetConfig (
            ushort wConfig,                                                                         /**< [In] Configuration to read. Will be one of the below values.
                                                                                                     *          \arg #PHPAL_I14443P3A_SILENT_MODE
                                                                                                     */
            ref ushort pValue,                                                                      /**< [Out] The value for the mentioned configuration. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.GetConfig ( ( Config ) wConfig, out pValue );
        }

        private ushort cbSetConfig (
            ushort wConfig,                                                                         /**< [In] Configuration to set. Will be one of the below values.
                                                                                                     *          \arg #PHPAL_I14443P3A_SILENT_MODE
                                                                                                     */
            ushort wValue,                                                                          /**< [In] The value for the mentioned configuration. */
            IntPtr pContext                                                                         /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.SetConfig ( ( Config ) wConfig, wValue );
        }

        #endregion

    #region CALLBACKS

        #region Initialization
        // Storage for internal callbacks
        private List<Delegate> m_Callbacks;
        private palI14443p3a.IPalI14443p3a m_ExtPal;
        private byte[] m_RxBuffer;
        private GCHandle m_pRxBuffer;


        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <param name="RxBufferSize"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p3a.IPalI14443p3a ExtPal,
            int RxBufferSize
            )
        {
            return Init ( ExtPal, RxBufferSize, IntPtr.Zero );
        }

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <param name="RxBufferSize"></param>
        /// <param name="cbContext"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p3a.IPalI14443p3a ExtPal,
            int RxBufferSize,
            IntPtr cbContext
            )
        {
            // Allocate internal callback delegates
            m_Callbacks = new List<Delegate> ();
            m_Callbacks.Add ( new RequestADelegate ( cbRequestA ) );
            m_Callbacks.Add ( new RequestSDelegate ( cbRequestS ) );
            m_Callbacks.Add ( new WakeUpADelegate ( cbWakeUpA ) );
            m_Callbacks.Add ( new WakeUpSDelegate ( cbWakeUpS ) );
            m_Callbacks.Add ( new HaltADelegate ( cbHaltA ) );
            m_Callbacks.Add ( new AnticollisionDelegate ( cbAnticollision ) );
            m_Callbacks.Add ( new SelectDelegate ( cbSelect ) );
            m_Callbacks.Add ( new ActivateCardDelegate ( cbActivateCard ) );
            m_Callbacks.Add ( new ExchangeDelegate ( cbExchange ) );
            m_Callbacks.Add ( new GetSerialNoDelegate ( cbGetSerialNo ) );
            m_Callbacks.Add ( new GetConfigDelegate ( cbGetConfig ) );
            m_Callbacks.Add ( new SetConfigDelegate ( cbSetConfig ) );

            // Store reference to external hal object
            m_ExtPal = ExtPal;

            // Free allocated RxBuffer if any
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Create RxBuffer
            m_RxBuffer = new byte[RxBufferSize];
            this.m_pRxBuffer = GCHandle.Alloc ( this.m_RxBuffer, GCHandleType.Pinned );

            // Call init. function
            return phpalI14443p3a_Callback_Init (
                ref m_DataParamsInt[0],
                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[0] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[1] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[2] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[3] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[4] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[5] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[6] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[7] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[8] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[9] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[10] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[11] ),
                cbContext );
        }

        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Callback ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Callback ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            // Free allocated RxBuffer
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Callback

    #region Sam_X
    /// <summary>
    /// Class for SAM (AV4 and future SAM's) X Mode layer initialization interface and data params.
    /// </summary>
    public class Sam_X : Generic
    {
        #region Data Structure
        /// <summary>
        /// ISO14443-3A parameter structure for SAM AV3 X.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL layer, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer </summary>
            public IntPtr pHalDataParams;

            /// <summary> Array holding the activated UID. </summary>
            public fixed byte abUid[10];

            /// <summary> Length of UID stored in \c abUid. </summary>
            public byte bUidLength;

            /// <summary> Whether the UID is complete (\c 1) or not (\c 0). </summary>
            public byte bUidComplete;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p3a_Sam_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialise this layer.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to HAL layer.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Sam oHal )
        {
            return phpalI14443p3a_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams );
        }
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif
}
