/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI14443p3b
{
    #region BASE

    public abstract class Generic
    {
        #region DEFINES

        public const int TIMEOUT_DEFAULT_MS = 10;
        public const int EXT_TIME_US = 60;

        public enum Config : int
        {
            ATTRIB_PARAM1 = 0x0000  /**< Set or get the AttribB param1 byte. */
        }

        #endregion

        #region MEMORY_MAPPING

        // buffers for HigherLayerInf and HigherLayerResp must be provided for the underlaying c implementation
        private byte[] m_HigherLayerInf;
        private GCHandle m_pHigherLayerInf;
        private byte[] m_HigherLayerResp;
        private GCHandle m_pHigherLayerResp;

        ~Generic()
        {
            // Free allocated pointers to HigherLayer Buffers
            if (this.m_pHigherLayerInf.IsAllocated)
            {
                this.m_pHigherLayerInf.Free();
            }
            if (this.m_pHigherLayerResp.IsAllocated)
            {
                this.m_pHigherLayerResp.Free();
            }
        }
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_SetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue       /**< [In] Configuration Value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_GetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue   /**< [Out] Configuration Value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_RequestB(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure. */
            byte bNumSlots,     /**< [In] Number of slots. */
            byte bAfi,          /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,      /**< [In] Enable Extended AtqB. */
            byte[] pAtqb,       /**< [Out] AtqB; uint8_t[13]. */
            ref byte pAtqbLen   /**< [Out] Length of received ATQB (12/13 bytes). */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_WakeUpB(
            IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            byte bNumSlots,       /**< [In] Number of slots. */
            byte bAfi,            /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,        /**< [In] Enable Extended AtqB. */
            byte[] pAtqb,         /**< [Out] AtqB; uint8_t[12/13]. */
            ref byte pAtqbLen     /**< [Out] Length of received ATQB (12/13 bytes). */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_SlotMarker(
            IntPtr pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bSlotNumber,      /**< [In] Slot Number. */
            byte[] pAtqb,          /**< [Out] AtqB; uint8_t[12/13]. */
            ref byte pAtqbLen      /**< [Out] Length of received ATQB (12/13 bytes). */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_HaltB(
            IntPtr pDataParams /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_Attrib(
            IntPtr pDataParams,    /**< [In] Pointer to this layers parameter structure */
            byte[] pAtqb,          /**< [In] Atqb from ReqB/WupB; uint8_t[12/13]. */
            byte bAtqbLen,         /**< [In] Length of Atqb. */
            byte bFsdi,            /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            byte bCid,             /**< [In] Card Identifier; 0-14. */
            byte bDri,             /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi,             /**< [In] Divisor Send (PICC to PCD) Integer; 0-3. */
            ref byte pMbli         /**< [Out] MBLI byte; uint8_t. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_ActivateCard(
            IntPtr pDataParams,           /**< [In] Pointer to this layers parameter structure */
            byte[] pPupi,                 /**< [In] Known PUPI, can be NULL; uint8_t[4]. */
            byte bPupiLength,             /**< [In] Length of given PUPI, only a value of 0 or 4 is allowed. */
            byte bNumSlots,               /**< [In] Number of slots. */
            byte bAfi,                    /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,                /**< [In] Enable Extended AtqB. */
            byte bFsdi,                   /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            byte bCid,                    /**< [In] Card Identifier; 0-14. */
            byte bDri,                    /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi,                    /**< [In] Divisor Send (PICC to PCD) Integer; 0-3. */
            byte[] pAtqb,                 /**< [Out] AtqB; uint8_t[12/13]. */
            ref byte pAtqbLen,            /**< [Out] Length of received ATQB (12/13 bytes). */
            ref byte pMbli,               /**< [Out] MBLI byte; uint8_t. */
            ref byte pMoreCardsAvailable  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_GetSerialNo(
                                   IntPtr pDataParams,  /**< [In] Pointer to this layers parameter structure.*/
                                   byte[] pPupi         /**< [Out] Most recent PUPI; uint8_t[4] */
                                   );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_GetProtocolParams(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ref byte pCidEnabled,   /**< [Out] Unequal '0' if Card Identifier is enabled. */
            ref byte pCid,          /**< [Out] Card Identifier. */
            ref byte pNadSupported, /**< [Out] Node Address Support; Unequal '0' if supported. */
            ref byte pFwi,          /**< [Out] Frame Waiting Integer. */
            ref byte pFsdi,         /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            ref byte pFsci          /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_SetHigherLayerInf(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            byte[] pRxBuffer,       /**< [In] Pointer to received data */
            ushort pRxLength        /**< [In] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_GetHigherLayerResp(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t SetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return phpalI14443p3b_SetConfig(m_pDataParams, (ushort)wConfig, (ushort)wValue);
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phpalI14443p3b_GetConfig(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = (int)wValue;
            return status;
        }

        public Status_t RequestB(
            byte bNumSlots,     /**< [In] Number of slots. */
            byte bAfi,          /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,      /**< [In] Enable Extended AtqB. */
            out byte[] pAtqB    /**< [Out] AtqB; uint8_t[12/13] */
            )
        {
            Status_t status;
            byte bAtqbLen = 0;
            pAtqB = new byte[13];
            status = phpalI14443p3b_RequestB(m_pDataParams, bNumSlots, bAfi, bExtAtqb, pAtqB, ref bAtqbLen);
            Array.Resize<byte>(ref pAtqB, bAtqbLen);
            return status;
        }

        public Status_t WakeUpB(
            byte bNumSlots,   /**< [In] Number of slots. */
            byte bAfi,        /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,    /**< [In] Enable Extended AtqB. */
            out byte[] pAtqB  /**< [Out] AtqB; uint8_t[12/13] */
            )
        {
            Status_t status;
            byte bAtqbLen = 0;
            pAtqB = new byte[13];
            status = phpalI14443p3b_WakeUpB(m_pDataParams, bNumSlots, bAfi, bExtAtqb, pAtqB, ref bAtqbLen);
            Array.Resize<byte>(ref pAtqB, bAtqbLen);
            return status;
        }

        public Status_t HaltB()
        {
            return phpalI14443p3b_HaltB(m_pDataParams);
        }

        public Status_t SlotMarker(
            byte bSlotNumber,   /**< [In] Slot Number. */
            out byte[] pAtqb    /**< [Out] AtqB; uint8_t[12/13]. */
            )
        {
            Status_t status;
            byte bAtqbLen = 0;
            pAtqb = new byte[13];
            status = phpalI14443p3b_SlotMarker(m_pDataParams, bSlotNumber, pAtqb, ref bAtqbLen);
            Array.Resize<byte>(ref pAtqb, bAtqbLen);
            return status;
        }

        public Status_t Attrib(
            byte[] pAtqb,       /**< [In] Atqb from ReqB/WupB; uint8_t[12/13]. */
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            byte bCid,          /**< [In] Card Identifier; 0-14. */
            byte bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3. */
            out byte pMbli      /**< [Out] MBLI byte; uint8_t. */
            )
        {
            pMbli = 0;
            return phpalI14443p3b_Attrib(m_pDataParams, pAtqb, (byte)pAtqb.Length, bFsdi, bCid, bDri, bDsi, ref pMbli);
        }

        public Status_t ActivateCard(
            byte[] pPupi,                 /**< [In] Known PUPI, can be NULL; uint8_t[4]. */
            byte bNumSlots,               /**< [In] Number of slots. */
            byte bAfi,                    /**< [In] AFI; Application Family Indentifier. */
            byte bExtAtqb,                /**< [In] Enable Extended AtqB. */
            byte bFsdi,                   /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            byte bCid,                    /**< [In] Card Identifier; 0-14. */
            byte bDri,                    /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi,                    /**< [In] Divisor Send (PICC to PCD) Integer; 0-3. */
            out byte[] pAtqb,             /**< [Out] AtqB; uint8_t[12/13]. */
            out byte pMbli,               /**< [Out] MBLI byte; uint8_t. */
            out byte pMoreCardsAvailable  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
                            )
        {
            Status_t status;
            byte bAtqbLen = 0;
            byte bPupiLen;
            pAtqb = new byte[13];
            pMbli = 0;
            pMoreCardsAvailable = 0;

            if (pPupi != null)
            {
                bPupiLen = (byte)pPupi.Length;
            }
            else
            {
                bPupiLen = 0;
            }

            status = phpalI14443p3b_ActivateCard(m_pDataParams, pPupi, bPupiLen, bNumSlots, bAfi, bExtAtqb, bFsdi, bCid, bDri, bDsi, pAtqb, ref bAtqbLen, ref pMbli, ref pMoreCardsAvailable);
            Array.Resize<byte>(ref pAtqb, bAtqbLen);
            return status;
        }

        /// <summary>
        /// Perform Data Exchange with Picc.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI14443p3b_Exchange(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t GetSerialNumber(
            out byte[] pPupi       /**< [Out] Most recent PUPI; uint8_t[4] */
            )
        {
            pPupi = new byte[4];
            return phpalI14443p3b_GetSerialNo(m_pDataParams, pPupi);
        }

        public Status_t GetProtocolParams(
            out byte pCidEnabled,   /**< [Out] Unequal '0' if Card Identifier is enabled. */
            out byte pCid,          /**< [Out] Card Identifier. */
            out byte pNadSupported, /**< [Out] Node Address Support; Unequal '0' if supported. */
            out byte pFwi,          /**< [Out] Frame Waiting Integer. */
            out byte pFsdi,         /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            out byte pFsci          /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            )
        {
            pCidEnabled = 0;
            pCid = 0;
            pNadSupported = 0;
            pFwi = 0;
            pFsdi = 0;
            pFsci = 0;

            return phpalI14443p3b_GetProtocolParams(
                m_pDataParams,
                ref pCidEnabled,
                ref pCid,
                ref pNadSupported,
                ref pFwi,
                ref pFsdi,
                ref pFsci);
        }
        public Status_t SetHigherLayerInf(
            byte[] pTxBuffer,       /**< [In] data to transmit as HigherLayer information */
            int wRxBufferSize       /**< [In] size of buffer to allocate for HigherLayer response */
            )
        {
            // Free allocated pointers to HigherLayer buffers
            if (this.m_pHigherLayerInf.IsAllocated)
            {
                this.m_pHigherLayerInf.Free();
            }
            if (this.m_pHigherLayerResp.IsAllocated)
            {
                this.m_pHigherLayerResp.Free();
            }

            // Allocate Buffers
            this.m_HigherLayerInf = pTxBuffer;
            this.m_pHigherLayerInf = GCHandle.Alloc(this.m_HigherLayerInf, GCHandleType.Pinned);
            this.m_HigherLayerResp = new byte[wRxBufferSize];
            this.m_pHigherLayerResp = GCHandle.Alloc(this.m_HigherLayerResp, GCHandleType.Pinned);

            // execute the command
            return phpalI14443p3b_SetHigherLayerInf(
                m_pDataParams,
                m_HigherLayerInf,
                (ushort)pTxBuffer.Length,
                m_HigherLayerResp,
                (ushort)m_HigherLayerResp.Length);
        }

        public Status_t GetHigherLayerResp(
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI14443p3b_GetHigherLayerResp(
                m_pDataParams,
                ref pRxBufferInt,
                ref wRxLength);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI14443p3b.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this HAL layer, NEVER MODIFY!                 */
            public IntPtr pHalDataParams;           /**< Pointer to the parameter structure of the underlying layer */
            public byte bExtAtqb;                   /**< Stores whether last request used extended ATQB. */
            public fixed byte bPupi[4];             /**< Array holding the activated PUPI. */
            public byte bPupiValid;                 /**< Whether the sored Pupi is valid (\c 1) or not (\c 0). */
            public byte bCidSupported;              /**< Cid Support indicator; Unequal '0' if supported. */
            public byte bNadSupported;              /**< Nad Support indicator; Unequal '0' if supported. */
            public byte bCid;                       /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
            public byte bFwi;                       /**< Frame Waiting Integer. */
            public byte bFsci;                      /**< PICC Frame Size Integer; 0-0xC; */
            public byte bFsdi;                      /**< (Current) PCD Frame Size Integer; 0-0xC; */
            public byte bDri;                       /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3; */
            public byte bDsi;                       /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3; */
            public byte bAttribParam1;              /**< Param1 parameter for attrib command. */
            public IntPtr pHigherLayerInf;          /**< Pointer to higher layer information buffer. */
            public ushort wHigherLayerInfLen;       /**< Length higher layer information. */
            public IntPtr pHigherLayerResp;         /**< Pointer to higher layer response buffer. */
            public ushort wHigherLayerRespSize;     /**< Size of higher layer response buffer. */
            public ushort wHigherLayerRespLen;      /**< Length of higher layer response. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI14443p3b_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init(Hal.Generic pHal)
        {
            return phpalI14443p3b_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI14443p3b_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
