/*
 * Copyright 2013, 2016 - 2020, 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.palI14443p4
{
    #region DEFINES

    public enum Error : byte
    {
        /// <summary> ISO14443-4 error recovery failed. </summary>
        ERR_RECOVERY_FAILED = ( CustomCodes.ERROR_BEGIN )
    }

    public enum Config : int
    {
        BLOCKNO = 0x0000,   /**< Set / Get Blocknumber. */
        CID,                /**< Set / Get Card Identifier. */
        NAD,                /**< Set / Get Node Address. */
        FWI,                /**< Set / Get Frame Waiting Integer. */
        FSI,                /**< Set / Get PCD & PICC Frame Size Integer. */
        RETRYCOUNT          /**< Set / Get Maximum Retry Count. */
    }

    public enum CidNad : int
    {
        ENABLE = 0x0100,
        DISABLE = 0x0000
    }

    public enum SParamMode : byte
    {
        CHECK = 0, /**< Default mode, sends a SParam request, checks if the requested parameter are supported from the card and return an error if the features are not supported. Otherwise send and activation of the required parameter */
        FORCE,     /**< Same as the default mode but did not check if the required parameters are supported just send an activation with the requied parameter */
        AUTO,      /**< Does not check the input of the SParam call, just request the supported data from the card and set the maximum possible datarate/best framing options */
        GET,       /**< Only send the sparam request and return the supported parameter */
        ACTIVATE   /**< Perform the sparam activation without any request or check */
    }

    [Flags]
    public enum SParamBitRate : ushort
    {
        BitRate_106 = 0x0100,
        BitRate_212 = 0x0200,
        BitRate_424 = 0x0400,
        BitRate_848 = 0x0800,
        BitRate_1695 = 0x1000,
        BitRate_3390 = 0x2000,
        BitRate_6780 = 0x4000,
        BitRate_PSK_3FC4 = 0x0001,
        BitRate_PSK_FC = 0x0002,
        BitRate_PSK_3FC2 = 0x0004,
        BitRate_PSK_2FC = 0x0008
    }

    [Flags]
    public enum SParamFrame : byte
    {
        STANDARD_FRAME = 0x01,
        FRAME_WITH_ERROR_CORRECTION = 0x02,
        SAME_FOR_BOTH_DIRECTION = 0x80
    }

    [Flags]
    public enum SParamFrameFormat : byte
    {
        START_STOP_BIT_SUPPRESSION = 0x01,
        SOF_EOF_SUPPRESSION = 0x02,
        SYNC_SUPPRESSION = 0x04,
        NO_SUPPRESSION = 0x00
    }
    #endregion

    #region INTERFACE

    /// <summary>
    /// Interface definition of interfaces a PAL object needs to implement.
    /// </summary>
    public interface IPalI14443p4
    {
        /// <summary>
        /// Perform SetProtocol.
        /// </summary>
        /// <param name="bCidEnable"></param>
        /// <param name="bCid"></param>
        /// <param name="bNadEnable"></param>
        /// <param name="bNad"></param>
        /// <param name="bFwi"></param>
        /// <param name="bFsdi"></param>
        /// <param name="bFsci"></param>
        /// <returns></returns>
        Status_t SetProtocol (
            byte bCidEnable,      /**< [In] Enable usage of Card Identifier; Unequal '0' if enabled. */
            byte bCid,            /**< [In] Card Identifier; Ignored if \c bCidEnable is equal '0'. */
            byte bNadEnable,      /**< [In] Enabler usage of Node Address; Unequal '0' if enabled. */
            byte bNad,            /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bFwi,            /**< [In] Frame Waiting Integer. */
            byte bFsdi,           /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bFsci            /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            );

        /// <summary>
        /// Perform ResetProtocol.
        /// </summary>
        /// <returns></returns>
        Status_t ResetProtocol ();

        /// <summary>
        /// Perform Deselect.
        /// </summary>
        /// <returns></returns>
        Status_t Deselect ();

        /// <summary>
        /// Perform PresCheck.
        /// </summary>
        /// <returns></returns>
        Status_t PresCheck ();

        /// <summary>
        /// Perform Exchange.
        /// </summary>
        /// <param name="wOption"></param>
        /// <param name="pTxBuffer"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        Status_t Exchange (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            );

        /// <summary>
        /// Perform SetConfig.
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="wValue"></param>
        /// <returns></returns>
        Status_t SetConfig (
            Config wConfig,         /**< [In] Configuration Identifier */
            ushort wValue           /**< [In] Configuration Value */
            );

        /// <summary>
        /// Perform GetConfig.
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="pValue"></param>
        /// <returns></returns>
        Status_t GetConfig (
            Config wConfig,         /**< [In] Configuration Identifier */
            out ushort pValue       /**< [Out] Configuration Value */
            );

        /// <summary>
        /// Perform SParamFrameFormat.
        /// </summary>
        /// <param name="bMode"></param>
        /// <param name="pbFramePcd2Picc"></param>
        /// <param name="pbFramePicc2Pcd"></param>
        /// <param name="pbFramingOptionsPcd2Picc"></param>
        /// <param name="pbFramingOptionsPicc2Pcd"></param>
        /// <returns></returns>
        Status_t SParamFrameFormat (
            byte bMode,                        /**< [In] Mode of the SParam Exchange */
            ref byte pbFramePcd2Picc,          /**< [In/Out] Frame from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramePicc2Pcd,          /**< [In/Out] Frame from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPcd2Picc, /**< [In/Out] Framing Options from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPicc2Pcd  /**< [In/Out] Framing Options from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            );

        /// <summary>
        /// Perform SParamBitRates.
        /// </summary>
        /// <param name="bMode"></param>
        /// <param name="pwBitRatePcd2Picc"></param>
        /// <param name="pwBitRatePicc2Pcd"></param>
        /// <param name="pbFramingOptions"></param>
        /// <returns></returns>
        Status_t SParamBitRates (
            byte bMode,                  /**< [In] Mode of the SParam Exchange */
            ref ushort pwBitRatePcd2Picc, /**< [In/Out] Datarate from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref ushort pwBitRatePicc2Pcd, /**< [In/Out] Datarate from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptions    /**< [In/Out] Framing Options, supplied value is the required, returned value is the values supported by PICC */
            );
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SetProtocol (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure. */
            byte bCidEnable,    /**< [In] Card Identifier Enabler; Unequal '0' if enabled. */
            byte bCid,          /**< [In] Card Identifier; Ignored if *pCidSupported is equal '0'. */
            byte bNadEnable,    /**< [In] Node Address Enabler; Unequal '0' if enabled. */
            byte bNad,          /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bFwi,          /**< [In] Frame Waiting Integer. */
            byte bFsdi,         /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bFsci          /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_ResetProtocol (
            IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Deselect (
            IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_PresCheck (
            IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Exchange (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SetConfig (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue       /**< [In] Configuration Value */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_GetConfig (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue   /**< [Out] Configuration Value */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SParamFrameFormat (
            IntPtr pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
            byte bMode,                         /**< [In] Mode of the SParam Exchange */
            IntPtr pbFramePcd2Picc,             /**< [In/Out] Frame from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pbFramePicc2Pcd,             /**< [In/Out] Frame from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pbFramingOptionsPcd2Picc,    /**< [In/Out] Framing Options from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pbFramingOptionsPicc2Pcd     /**< [In/Out] Framing Options from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SParamBitRates (
            IntPtr pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
            byte bMode,                         /**< [In] Mode of the SParam Exchange */
            IntPtr pwBitRatePcd2Picc,           /**< [In/Out] Datarate from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pwBitRatePicc2Pcd,           /**< [In/Out] Datarate from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pbFramingOptions             /**< [In/Out] Framing Options, supplied value is the required, returned value is the values supported by PICC */
            );
        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t SetProtocol (
            byte bCidEnable,    /**< [In] Card Identifier Enabler; Unequal '0' if enabled. */
            byte bCid,          /**< [In] Card Identifier; Ignored if *pCidSupported is equal '0'. */
            byte bNadEnable,    /**< [In] Node Address Enabler; Unequal '0' if enabled. */
            byte bNad,          /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bFwi,          /**< [In] Frame Waiting Integer. */
            byte bFsdi,         /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bFsci          /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            )
        {
            return phpalI14443p4_SetProtocol ( m_pDataParams, bCidEnable, bCid, bNadEnable, bNad, bFwi, bFsdi, bFsci );
        }

        public Status_t ResetProtocol ()
        {
            return phpalI14443p4_ResetProtocol ( m_pDataParams );
        }

        public Status_t Deselect ()
        {
            return phpalI14443p4_Deselect ( m_pDataParams );
        }

        public Status_t PresCheck ()
        {
            return phpalI14443p4_PresCheck ( m_pDataParams );
        }

        /// <summary>
        /// Perform Data Exchange with Picc.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Exchange (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI14443p4_Exchange (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                ( ushort ) pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t SetConfig (
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return phpalI14443p4_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        public Status_t GetConfig (
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phpalI14443p4_GetConfig ( m_pDataParams, ( ushort ) wConfig, ref wValue );
            pValue = ( int ) wValue;
            return status;
        }

        /// <summary>
        /// Set & Get configuration setting
        /// </summary>
        /// <param name="wConfig"></param>
        /// <returns></returns>
        public int this[Config wConfig]
        {
            get
            {
                int wValue;
                Status_t status = GetConfig(wConfig, out wValue);
                if ( ( Error_Gen ) status.Error != Error_Gen.SUCCESS )
                {
                    throw new Exception ( status.ToString () );
                }
                return wValue;
            }
            set
            {
                Status_t status = SetConfig(wConfig, value);
                if ( ( Error_Gen ) status.Error != Error_Gen.SUCCESS )
                {
                    throw new Exception ( status.ToString () );
                }
            }
        }

        public Status_t phpalI14443p4_SParamFrameFormat (
            byte bMode,
            ref byte? pbFramePcd2Picc,
            ref byte? pbFramePicc2Pcd,
            ref byte? pbFramingOptionsPcd2Picc,
            ref byte? pbFramingOptionsPicc2Pcd
            )
        {
            GCHandle pbFramePcd2PiccHandle = GCHandle.Alloc(pbFramePcd2Picc, GCHandleType.Pinned);
            IntPtr pbFramePcd2PiccPtr = pbFramePcd2PiccHandle.AddrOfPinnedObject();
            GCHandle pbFramePicc2PcdHandle = GCHandle.Alloc(pbFramePicc2Pcd, GCHandleType.Pinned);
            IntPtr pbFramePicc2PcdPtr = pbFramePicc2PcdHandle.AddrOfPinnedObject();
            GCHandle pbFramingOptionsPcd2PiccHandle = GCHandle.Alloc(pbFramingOptionsPcd2Picc, GCHandleType.Pinned);
            IntPtr pbFramingOptionsPcd2PiccPtr = pbFramingOptionsPcd2PiccHandle.AddrOfPinnedObject();
            GCHandle pbFramingOptionsPicc2PcdHandle = GCHandle.Alloc(pbFramingOptionsPicc2Pcd, GCHandleType.Pinned);
            IntPtr pbFramingOptionsPicc2PcdPtr = pbFramingOptionsPicc2PcdHandle.AddrOfPinnedObject();

            var status = phpalI14443p4_SParamFrameFormat(m_pDataParams, bMode, pbFramePcd2PiccPtr, pbFramePicc2PcdPtr, pbFramingOptionsPcd2PiccPtr, pbFramingOptionsPicc2PcdPtr);

            if ( pbFramePcd2PiccHandle.IsAllocated )
            {
                pbFramePcd2Picc = ( byte? ) pbFramePcd2PiccHandle.Target;
                pbFramePcd2PiccHandle.Free ();
            }
            if ( pbFramePicc2PcdHandle.IsAllocated )
            {
                pbFramePicc2Pcd = ( byte? ) pbFramePicc2PcdHandle.Target;
                pbFramePicc2PcdHandle.Free ();
            }
            if ( pbFramingOptionsPcd2PiccHandle.IsAllocated )
            {
                pbFramingOptionsPcd2Picc = ( byte? ) pbFramingOptionsPcd2PiccHandle.Target;
                pbFramingOptionsPcd2PiccHandle.Free ();
            }
            if ( pbFramingOptionsPicc2PcdHandle.IsAllocated )
            {
                pbFramingOptionsPicc2Pcd = ( byte? ) pbFramingOptionsPicc2PcdHandle.Target;
                pbFramingOptionsPicc2PcdHandle.Free ();
            }
            return status;
        }

        public Status_t phpalI14443p4_SParamBitRates (
            byte bMode,
            ref ushort? pwBitRatePcd2Picc,
            ref ushort? pwBitRatePicc2Pcd,
            ref byte? pbFramingOptions
            )
        {
            GCHandle pwBitRatePcd2PiccHandle = GCHandle.Alloc(pwBitRatePcd2Picc, GCHandleType.Pinned);
            IntPtr pwBitRatePcd2PiccPtr = pwBitRatePcd2PiccHandle.AddrOfPinnedObject();
            GCHandle pwBitRatePicc2PcdHandle = GCHandle.Alloc(pwBitRatePicc2Pcd, GCHandleType.Pinned);
            IntPtr pwBitRatePicc2PcdPtr = pwBitRatePicc2PcdHandle.AddrOfPinnedObject();
            GCHandle pbFramingOptionsHandle = GCHandle.Alloc(pbFramingOptions, GCHandleType.Pinned);
            IntPtr pbFramingOptionsPtr = pbFramingOptionsHandle.AddrOfPinnedObject();

            var status = phpalI14443p4_SParamBitRates(m_pDataParams, bMode, pwBitRatePcd2PiccPtr, pwBitRatePicc2PcdPtr, pbFramingOptionsPtr);

            if ( pwBitRatePcd2PiccHandle.IsAllocated )
            {
                pwBitRatePcd2Picc = ( ushort? ) pwBitRatePcd2PiccHandle.Target;
                pwBitRatePcd2PiccHandle.Free ();
            }
            if ( pwBitRatePicc2PcdHandle.IsAllocated )
            {
                pwBitRatePicc2Pcd = ( ushort? ) pwBitRatePicc2PcdHandle.Target;
                pwBitRatePicc2PcdHandle.Free ();
            }
            if ( pbFramingOptionsHandle.IsAllocated )
            {
                pbFramingOptions = ( byte? ) pbFramingOptionsHandle.Target;
                pbFramingOptionsHandle.Free ();
            }
            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI14443p4.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            MAX_WTX_LOOP_TIME = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,            /**< Set / Get Maximum Time used within a WTX loop (in seconds). */
            SUPPRESS_DESELECT = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 1,          /**< Enable/Diable the automatic send of a Deselect/Is the deselect disabled. */
            USED_RETRY_COUNT = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 2,          /**< Get the numbers of retransmits used. */
            FRAME_FORMAT_PCD_2_PICC = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 3,   /**< Set the frame format pcd 2 picc. */
            FRAME_FORMAT_PICC_2_PCD = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 4,   /**< Set the frame format picc 2 pcd. */
            FRAMING_OPTIONS_SYNC_SUPPRESSION_PCD_2_PICC = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 5,           /**< Set the frame options sync suppression pcd 2 picc. */
            FRAMING_OPTIONS_SYNC_SUPPRESSION_PICC_2_PCD = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 6,           /**< Set the frame options sync suppression  picc 2 pcd. */
            FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PCD_2_PICC = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 7,        /**< Set the frame options SOF/EOF suppression pcd 2 picc. */
            FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 8,        /**< Set the frame options SOF/EOF suppression picc 2 pcd. */
            FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PCD_2_PICC = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 9, /**< Set the frame options start stop bit suppression pcd 2 picc. */
            FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 10 /**< Set the frame options stop bit suppression picc 2 pcd. */
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            public ushort    wId;              /**< Layer ID for this HAL component, NEVER MODIFY!                     */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer     */
            public byte bStateNow;          /**< Current Exchange-State                                         */
            public byte bCidEnabled;        /**< Card Identifier Enabler; Unequal '0' if enabled.               */
            public byte bCid;               /**< Card Identifier; Ignored if *pCidSupported is equal '0'.       */
            public byte bNadEnabled;        /**< Node Address Enabler; Unequal '0' if enabled.                  */
            public byte bNad;               /**< Node Address; Ignored if bNadEnabled is equal '0'.             */
            public byte bFwi;               /**< Frame Waiting Integer.                                         */
            public byte bFsdi;              /**< PCD Frame Size Integer; 0-0xC                                    */
            public byte bFsci;              /**< PICC Frame Size Integer; 0-0xC                                   */
            public byte bPcbBlockNum;       /**< Current Block-Number; 0/1                                      */
            public byte bMaxRetryCount;     /**< Maximum Retry count for ISO/IEC 14443-4:2008(E) Rule 4 and 5   */
            public byte bUsedRetryCount;    /**< How much retransmits are used durring the last exchange */
            public byte bSuppressDeselect;  /**< If this is PH_ON the deselect is suppressed even if retry count != 0 */
            public uint dwWTXLoopTimeLimit; /**< Maximum Timing allowed to be on a WTX loop (if zero no limit).  */
            public fixed byte pbEnhancedBlockBufferTx[5000]; /**< Buffer used for storing the plain frame for FWEC. */
            public fixed byte pbEnhancedBlockBufferRx[5000]; /**< Buffer used for storing the plain frame for FWEC. */
            public fixed byte pbEnhancedBlockBufferTmp[5000]; /**< Buffer used for storing the plain frame for FWEC. */
            public ushort wTxBufferLength;      /**< Number of Data stored for transmission in pbEnhancedBlockBuffer */
            public ushort wRxBufferStartPos;    /**< Start position of the rx buffer in pbEnhancedBlockBuffer */
            public byte   bEnhancedBlockStateTx;  /**< Indicates if enhanced Block format is enabled or disabled. */
            public byte   bFwecSyncEnabledTx;     /**< Indicates if sync is enabled or disabled for Frame with error correction. */
            public byte   bEnhancedBlockStateRx;  /**< Indicates if enhanced Block format is enabled or disabled. */
            public byte   bFwecSyncEnabledRx;     /**< Indicates if sync is enabled or disabled for Frame with error correction. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Sw_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams               /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p4_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI14443p4_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region SamAV2_X

    public class SamAV2_X : palI14443p4.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;             /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;  /**< pointer to the parameter structure of the underlying layer. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SamAV2_X_Init (
                                        ref DataParams_t m_pDataParams,  /**< [In] Pointer to this layers parameter structure */
                                        ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
                                        IntPtr pHalDataParams            /**< [In] Pointer to the parameter structure of the underlying layer */
                                   );

        #endregion

        #region INIT

        public Status_t Init ( Hal.SamAV2 pHal )
        {
            return phpalI14443p4_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X

    public class SamAV3_X : palI14443p4.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;             /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;  /**< pointer to the parameter structure of the underlying layer. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_SamAV3_X_Init (
                                        ref DataParams_t m_pDataParams,  /**< [In] Pointer to this layers parameter structure */
                                        ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
                                        IntPtr pHalDataParams            /**< [In] Pointer to the parameter structure of the underlying layer */
                                   );

        #endregion

        #region INIT

        public Status_t Init ( Hal.SamAV3 pHal )
        {
            return phpalI14443p4_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
#endif

    #region Rd710

    public class Rd710 : palI14443p4.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!                     */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer     */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Rd710_Init (
                                ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
                                ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
                                IntPtr pHalDataParams               /**< [In] Pointer to the parameter structure of the underlying layer */
                                );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Rd710 pHal )
        {
            return phpalI14443p4_Rd710_Init (
                                            ref m_DataParamsInt[0],
                                            ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                            pHal.m_pDataParams );
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
        {
            return phpalI14443p4_Rd710_Init(
                                            ref m_DataParamsInt[0],
                                            (ushort)wDataParamSize,
                                            pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Rd710 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Rd710 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region Android

    public class Android : palI14443p4.Generic
    {
        private const int TECHNOLOGIE_CNT = 0x05;        /**< The size of the coupler revision. */
        private const int EXCHANGE_DATA_OFFSET = 14;     /**< offset in the exchange buffer where the data starts. */

        public enum Config : int
        {
            SYS_INFO_SIZE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,               /**< Get the actual sysinfo size. */
        }

        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!                 */
            public IntPtr pBalDataParams;   /**< Pointer to the parameter structure of the underlying layer     */
            byte bIsOpen;                   /**< Inticated if the connection is opened */
            public fixed byte abAvailableTechnologies[TECHNOLOGIE_CNT]; /**< Stores all available technologies for the current present Tag */
            byte bUsedTechnologies;         /**< Stores the current used technologie */
            public IntPtr pExchangeBuffer;  /**< Pointer to global Exchange buffer used by the Exchange() function. */
            public ushort wExchangeBufSize; /**< Size of the global Exchange buffer. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Android_Init (
                                ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
                                ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
                                IntPtr pHalDataParams,              /**< [In] Pointer to the parameter structure of the underlying layer */
                                IntPtr pExchangeBuffer,                   /**< [In] Pointer to global receive buffer used by the Exchange() function. */
                                ushort wExchangeBufSize                   /**< [In] Size of the global receive buffer. */
                                );

        #endregion

        #region INIT
        private byte[] m_bExchangeBuffer;
        private GCHandle m_pExchangeBuffer;

        public Status_t Init ( Bal.SocketWin pBal, int wExchangeBufferSize )
        {
            // Free Buffers
            if ( this.m_pExchangeBuffer.IsAllocated )
            {
                this.m_pExchangeBuffer.Free ();
            }

            // Allocate buffers
            m_bExchangeBuffer = new byte[wExchangeBufferSize + 3]; /* 3 bytes header */

            // Link given buffers
            m_pExchangeBuffer = GCHandle.Alloc ( m_bExchangeBuffer, GCHandleType.Pinned );

            return phpalI14443p4_Android_Init (
                                            ref m_DataParamsInt[0],
                                            ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                            pBal.m_pDataParams,
                                            m_pExchangeBuffer.AddrOfPinnedObject (),
                                            ( ushort ) ( wExchangeBufferSize + 3 )
                                            );
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.SocketWin pBal, int wExchangeBufferSize)
        {
            // Free Buffers
            if (this.m_pExchangeBuffer.IsAllocated)
            {
                this.m_pExchangeBuffer.Free();
            }

            // Allocate buffers
            m_bExchangeBuffer = new byte[wExchangeBufferSize + 3]; /* 3 bytes header */

            // Link given buffers
            m_pExchangeBuffer = GCHandle.Alloc(m_bExchangeBuffer, GCHandleType.Pinned);

            return phpalI14443p4_Android_Init(
                                            ref m_DataParamsInt[0],
                                            (ushort)wDataParamSize,
                                            pBal.m_pDataParams,
                                            m_pExchangeBuffer.AddrOfPinnedObject(),
                                            (ushort)(wExchangeBufferSize + 3));
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Android ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Android ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        public byte[] ExchangeBuffer
        {
            get
            {
                return this.m_bExchangeBuffer;
            }
        }

        public byte[] ExchangeBufferData
        {
            get
            {
                byte[] tmp = new byte[m_bExchangeBuffer.Length - EXCHANGE_DATA_OFFSET];
                System.Array.Copy ( m_bExchangeBuffer, EXCHANGE_DATA_OFFSET, tmp, 0, tmp.Length );
                return tmp;
            }
        }

        #endregion
    }

    #endregion

    #region PCSC

    public class PCSC : palI14443p4.Generic
    {
        #region DEFINES

        private const int RESERVED_BUFFER_LEN = 8;  /**< Amount of needed and reserved memory for the protocol overhead. */

        public enum PcscMode : int
        {
            DEFAULT = 0x0000,
            DIRECT,
        }

        public enum PcscExchangeMode : int
        {
            TRANSPARENT = 0x0000,
            NORMAL,
        }

        public enum PcscConfig : int
        {
            CUSTOM_BEGIN = 0xE000,
            EXCHANGE = CUSTOM_BEGIN,
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!                     */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer     */
            public IntPtr pBalDataParams;   /**< Pointer to the parameter structure of the underlying pcsc bal. Is used to access the pcsc parameter */
            public IntPtr pHalPcscDataParams;   /**< pointer to the parameter structure of the underlying pcsc hal. Is used to access the pcsc parameter */
            public byte bStateNow;          /**< Current Exchange-State                                         */
            public byte bCidEnabled;        /**< Card Identifier Enabler; Unequal '0' if enabled.               */
            public byte bCid;               /**< Card Identifier; Ignored if *pCidSupported is equal '0'.       */
            public byte bNadEnabled;        /**< Node Address Enabler; Unequal '0' if enabled.                  */
            public byte bNad;               /**< Node Address; Ignored if bNadEnabled is equal '0'.             */
            public byte bFwi;               /**< Frame Waiting Integer.                                         */
            public byte bFsdi;              /**< PCD Frame Size Integer; 0-8                                    */
            public byte bFsci;              /**< PICC Frame Size Integer; 0-8                                   */
            public byte bPcbBlockNum;       /**< Current Block-Number; 0/1                                      */
            public byte bMaxRetryCount;     /**< Maximum Retry count for ISO/IEC 14443-4:2008(E) Rule 4 and 5   */
            public byte bExchangeType;      /**< Type of exchange to use. #PHPAL_I14443P4_PCSC_NORMAL (default) or #PHPAL_I14443P4_PCSC_TRANSPARENT. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Pcsc_Init (
                                ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
                                ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
                                IntPtr pHalDataParams               /**< [In] Pointer to the parameter structure of the underlying layer */
                                );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p4_Pcsc_Init (
                ref m_DataParamsInt[0],
                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                            pHal.m_pDataParams );
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
        {
            return phpalI14443p4_Pcsc_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                                            pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public PCSC ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~PCSC ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL
    #region Callback
    /// <summary>
    /// Class for Callback layer initialization interface and data params.
    /// </summary>
    public class Callback : palI14443p4.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P4 parameter structure for Callback reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pSetProtocolCb;
            IntPtr pResetProtocolCb;
            IntPtr pDeselectCb;
            IntPtr pPresCheckCb;
            IntPtr pExchangeCb;
            IntPtr pSetConfigCb;
            IntPtr pGetConfigCb;
            IntPtr pSParamFrameFormatCb;
            IntPtr pSParamBitRatesCb;
            IntPtr pCbContext;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Callback_Init (
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pSetProtocolCb,
            IntPtr pResetProtocolCb,
            IntPtr pDeselectCb,
            IntPtr pPresCheckCb,
            IntPtr pExchangeCb,
            IntPtr pSetConfigCb,
            IntPtr pGetConfigCb,
            IntPtr pSParamFrameFormatCb,
            IntPtr pSParamBitRatesCb,
            IntPtr pCbContext );
        #endregion DLL Imports

        #region DELEGATES

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SetProtocolDelegate (
            byte bCidEnable,      /**< [In] Enable usage of Card Identifier; Unequal '0' if enabled. */
            byte bCid,            /**< [In] Card Identifier; Ignored if \c bCidEnable is equal '0'. */
            byte bNadEnable,      /**< [In] Enabler usage of Node Address; Unequal '0' if enabled. */
            byte bNad,            /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bFwi,            /**< [In] Frame Waiting Integer. */
            byte bFsdi,           /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bFsci,           /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            IntPtr pContext       /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort ResetProtocolDelegate (
            IntPtr pContext       /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort DeselectDelegate (
            IntPtr pContext       /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort PresCheckDelegate (
            IntPtr pContext       /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort ExchangeDelegate (
            ushort wOption,         /**< [In] Option parameter. */
            IntPtr pTxBuffer,       /**< [In] Data to transmit. */
            ushort wTxLength,       /**< [In] Length of data to transmit. */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data. */
            ref ushort pRxLength,   /**< [Out] number of received data bytes. */
            IntPtr pContext         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SetConfigDelegate (
            ushort wConfig,         /**< [In] Configuration Identifier */
            ushort wValue,          /**< [In] Configuration Value */
            IntPtr pContext         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort GetConfigDelegate (
            ushort wConfig,         /**< [In] Configuration Identifier */
            ref ushort pValue,      /**< [Out] Configuration Value */
            IntPtr pContext         /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SParamFrameFormatDelegate (
            byte bMode,                        /**< [In] Mode of the SParam Exchange */
            ref byte pbFramePcd2Picc,          /**< [In/Out] Frame from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramePicc2Pcd,          /**< [In/Out] Frame from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPcd2Picc, /**< [In/Out] Framing Options from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPicc2Pcd, /**< [In/Out] Framing Options from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pContext                    /**< [Out] User Defined Context. */
           );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort SParamBitRatesDelegate (
            byte bMode,                   /**< [In] Mode of the SParam Exchange */
            ref ushort pwBitRatePcd2Picc, /**< [In/Out] Datarate from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref ushort pwBitRatePicc2Pcd, /**< [In/Out] Datarate from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptions,    /**< [In/Out] Framing Options, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pContext               /**< [Out] User Defined Context. */
           );

        #endregion DELEGATES

        #region CALLBACKS

        private ushort cbSetProtocol (
            byte bCidEnable,      /**< [In] Enable usage of Card Identifier; Unequal '0' if enabled. */
            byte bCid,            /**< [In] Card Identifier; Ignored if \c bCidEnable is equal '0'. */
            byte bNadEnable,      /**< [In] Enabler usage of Node Address; Unequal '0' if enabled. */
            byte bNad,            /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bFwi,            /**< [In] Frame Waiting Integer. */
            byte bFsdi,           /**< [In] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bFsci,           /**< [In] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            IntPtr pContext       /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.SetProtocol ( bCidEnable, bCid, bNadEnable, bNad, bFwi, bFsdi, bFsci );
        }

        private ushort cbResetProtocol (
            IntPtr pContext       /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.ResetProtocol ();
        }

        private ushort cbDeselect (
            IntPtr pContext       /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.Deselect ();
        }

        private ushort cbPresCheck (
            IntPtr pContext       /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.PresCheck ();
        }

        private ushort cbExchange (
            ushort wOption,         /**< [In] Option parameter. */
            IntPtr pTxBuffer,       /**< [In] Data to transmit. */
            ushort wTxLength,       /**< [In] Length of data to transmit. */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data. */
            ref ushort pRxLength,   /**< [Out] number of received data bytes. */
            IntPtr pContext         /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pTxBufferTmp = new byte[wTxLength];
            byte[] pRxBufferTmp;
            int dwRxOffset;

            // Prepare input buffer(s)
            for ( int dIndex = 0; dIndex < wTxLength; dIndex++ )
            {
                pTxBufferTmp[dIndex] = Marshal.ReadByte ( pTxBuffer, dIndex );
            }

            // Reset Receive length
            pRxLength = 0;

            // Execute callback function
            status = m_ExtPal.Exchange ( ( int ) wOption, pTxBufferTmp, out pRxBufferTmp );

            // Return pointer to global buffer
            if ( pRxBufferTmp != null && pRxBufferTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pRxBufferTmp.Length > m_RxBuffer.Length )
                {
                    return new Status_t ( Error_CompCode.HAL, Error_Comm.BUFFER_OVERFLOW );
                }

                //TODO: Check if this is required
                // Get RxStartPos
                dwRxOffset = 0;
                //var getStatus = m_ExtHal.GetConfig(NxpRdLibNet.Hal.Config.RXBUFFER_STARTPOS, out dwRxOffset);
                //if (!getStatus.Equals(new Status_t()))
                //    return getStatus;

                // Copy only new Rx-Contents to persistent HAL buffer (to prevent override of already restored data)
                System.Array.Copy ( pRxBufferTmp, dwRxOffset, m_RxBuffer, dwRxOffset, pRxBufferTmp.Length - dwRxOffset );

                // Set Output Buffer to persistent HAL buffer pointer
                ppRxBuffer = m_pRxBuffer.AddrOfPinnedObject ();

                // Set number of received bytes
                pRxLength = ( ushort ) pRxBufferTmp.Length;
            }

            return status;
        }

        private ushort cbSetConfig (
            ushort wConfig,         /**< [In] Configuration Identifier */
            ushort wValue,          /**< [In] Configuration Value */
            IntPtr pContext         /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.SetConfig ( ( Config ) wConfig, wValue );
        }

        private ushort cbGetConfig (
            ushort wConfig,         /**< [In] Configuration Identifier */
            ref ushort pValue,      /**< [Out] Configuration Value */
            IntPtr pContext         /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.GetConfig ( ( Config ) wConfig, out pValue );
        }

        private ushort cbSParamFrameFormat (
            byte bMode,                        /**< [In] Mode of the SParam Exchange */
            ref byte pbFramePcd2Picc,          /**< [In/Out] Frame from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramePicc2Pcd,          /**< [In/Out] Frame from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPcd2Picc, /**< [In/Out] Framing Options from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptionsPicc2Pcd, /**< [In/Out] Framing Options from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pContext                    /**< [Out] User Defined Context. */
           )
        {
            return m_ExtPal.SParamFrameFormat ( bMode, ref pbFramePcd2Picc, ref pbFramePicc2Pcd, ref pbFramingOptionsPcd2Picc, ref pbFramingOptionsPicc2Pcd );
        }

        private ushort cbSParamBitRates (
            byte bMode,                   /**< [In] Mode of the SParam Exchange */
            ref ushort pwBitRatePcd2Picc, /**< [In/Out] Datarate from PCD 2 PICC, supplied value is the required, returned value is the values supported by PICC */
            ref ushort pwBitRatePicc2Pcd, /**< [In/Out] Datarate from PICC 2 PCD, supplied value is the required, returned value is the values supported by PICC */
            ref byte pbFramingOptions,    /**< [In/Out] Framing Options, supplied value is the required, returned value is the values supported by PICC */
            IntPtr pContext               /**< [Out] User Defined Context. */
           )
        {
            return m_ExtPal.SParamBitRates ( bMode, ref pwBitRatePcd2Picc, ref pwBitRatePicc2Pcd, ref pbFramingOptions );
        }

        #endregion

        #region Initialization
        // Storage for internal callbacks
        private List<Delegate> m_Callbacks;
        private palI14443p4.IPalI14443p4 m_ExtPal;
        private byte[] m_RxBuffer;
        private GCHandle m_pRxBuffer;


        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <param name="RxBufferSize"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p4.IPalI14443p4 ExtPal,
            int RxBufferSize
            )
        {
            return Init ( ExtPal, RxBufferSize, IntPtr.Zero );
        }

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <param name="RxBufferSize"></param>
        /// <param name="cbContext"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p4.IPalI14443p4 ExtPal,
            int RxBufferSize,
            IntPtr cbContext
            )
        {
            // Allocate internal callback delegates
            m_Callbacks = new List<Delegate> ();
            m_Callbacks.Add ( new SetProtocolDelegate ( cbSetProtocol ) );
            m_Callbacks.Add ( new ResetProtocolDelegate ( cbResetProtocol ) );
            m_Callbacks.Add ( new DeselectDelegate ( cbDeselect ) );
            m_Callbacks.Add ( new PresCheckDelegate ( cbPresCheck ) );
            m_Callbacks.Add ( new ExchangeDelegate ( cbExchange ) );
            m_Callbacks.Add ( new SetConfigDelegate ( cbSetConfig ) );
            m_Callbacks.Add ( new GetConfigDelegate ( cbGetConfig ) );
            m_Callbacks.Add ( new SParamFrameFormatDelegate ( cbSParamFrameFormat ) );
            m_Callbacks.Add ( new SParamBitRatesDelegate ( cbSParamBitRates ) );

            // Store reference to external hal object
            m_ExtPal = ExtPal;

            // Free allocated RxBuffer if any
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Create RxBuffer
            m_RxBuffer = new byte[RxBufferSize];
            this.m_pRxBuffer = GCHandle.Alloc ( this.m_RxBuffer, GCHandleType.Pinned );

            // Call init. function
            return phpalI14443p4_Callback_Init (
                ref m_DataParamsInt[0],
                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[0] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[1] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[2] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[3] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[4] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[5] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[6] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[7] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[8] ),
                cbContext );
        }

        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Callback ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Callback ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }

            // Free allocated RxBuffer
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Callback

    #region Sam_X
    public class Sam_X : Generic
    {
        #region Data Structure
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pHalDataParams;
        };
        #endregion

        #region Dll Import

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4_Sam_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion

        #region Initalization
        public Status_t Init ( Hal.Sam oHal )
        {
            return phpalI14443p4_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif
}
