/*
 * Copyright 2022, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI7816p4
{
    #region DEFINES

    public enum Config : int
    {
        IFSC = 0x0001,          /**< Set / Get PICC Frame Size. */
        NAD,                    /**< Set / Get Node Address. */
        AUTO_GET_RESPONSE,      /**< Set / Get Auto GetResponse. */
        PROTOCOL,               /**< Set / Get Protocol. */
        AUTO_6C_HANDLING,       /**< Set / Get Auto 6C Handling. */
        TX_RETRANSMIT_COUNT,    /**< Set / Get Tx Retransmit Count. */
        RX_RETRANSMIT_COUNT,    /**< Set / Get Rx Retransmit Count. */
        IFSD,                   /**< Set / Get PCD Frame Size. */
        I2C_PROTOCOL            /**< Set / Get protocol for I2C communication. */
    }

    public enum I2C_Protocol : int
    {
        T1overI2C_LenOne_Lrc = 0x00,
        T1overI2C_LenOne_Crc = 0x02,
        GlobalPlatform_LenOne_Crc = 0x0A,
        GlobalPlatform_LenTwo_Crc = 0x0B,
        GlobalPlatform_LenTwo_Crc_Swapped = 0x0F
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_SetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue       /**< [In] Configuration Value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_GetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue   /**< [Out] Configuration Value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_ResetProtocol(
            IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        /// <summary>
        /// Perform Data Exchange with Picc.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI7816p4_Exchange(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t SetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return phpalI7816p4_SetConfig(m_pDataParams, (ushort)wConfig, (ushort)wValue);
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phpalI7816p4_GetConfig(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = (int)wValue;
            return status;
        }

        public Status_t ResetProtocol()
        {
            return phpalI7816p4_ResetProtocol(m_pDataParams);
        }

        /// <summary>
        /// Set & Get configuration setting
        /// </summary>
        /// <param name="wConfig"></param>
        /// <returns></returns>
        public int this[Config wConfig]
        {
            get
            {
                int wValue;
                Status_t status = GetConfig(wConfig, out wValue);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
                return wValue;
            }
            set
            {
                Status_t status = SetConfig(wConfig, value);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
            }
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI7816p4.Generic
    {
        #region DEFINES

        private const int INT_BUFFER_SIZE = 4096;      /**< Size of the internally used buffer */
        private const int T0_HEADER_SIZE = 5;          /**< Size of the header for T=0 Protocol. */

        public enum I2C_SBlock : byte
        {
            Resynch = 0x00,
            Ifs,
            Abort,
            Wtx,
            Swr = 0x0F
        }

        public enum I2C_GP_SBlock : byte
        {
            Cip = 0x04,
            Release = 0x06
        }

        public enum I2C_T1overI2C_SBlock : byte
        {
            EndOfAPDUSession = 0x05,
            SEChipReset,
            GetATR
        }

        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;                       /**< Pointer to the parameter structure of the underlying layer */
            public byte bT1MainState;                           /**< Current Main Exchange-State for T=1 Protocol. */
            public byte bT1TxState;                             /**< Current TX Exchange-State for T=1 Protocol. */
            public byte bT1RxState;                             /**< Current RX Exchange-State for T=1 Protocol. */
            public byte bT1SeqNumReceive;                       /**< Sequence Number for received Block for T=1 Protocol. */
            public byte bT1SeqNumSend;                          /**< Sequence Number for sending Blocks for T=1 Protocol. */
            public byte bT1CardIfs;                             /**< Card sent IFS request containing the blocksize */
            public byte bT1TxRetransmitCount;                   /**< Number of retransmits for Sending Blocks */
            public byte bT1RxRetransmitCount;                   /**< Number of retransmits for receiving Block */
            public fixed byte pRxBuffer[INT_BUFFER_SIZE];       /**< Buffer internally used as rx buffer */
            public fixed byte pTxBuffer[INT_BUFFER_SIZE];       /**< Buffer internally used as tx buffer to collect buffered exchanges */
            public ushort wTxBufLen;                            /**< Number of valid bytes within the transmit buffer. */
            public ushort wRxBufferLen;                         /**< Length of the internally used buffer */
            public ushort wT0LengthExpected;                    /**< Length of the response expected, up to 256 for T=0 Protocol. */
            public byte bT0CurrentCase;                         /**< The currently used case for data exchange according to I7816. */
            public fixed byte pT0Header[T0_HEADER_SIZE];        /**< To save command header for T=0 Protocol. */
            public byte bNad;                                   /**< Node Address; Ignored if bNadEnabled is equal '0'. */
            public byte bFsci;                                  /**< PICC Frame Size Integer; 0-0xC */
            public byte bIfsd;                                  /**< PCD Frame Size */
            public byte bAutoGetResponse;                       /**< automatic GetResponse on case 4S.3 */
            public byte bAuto6cHandling;                        /**< Automatic 6C Handling. */
            public byte bProtocol;                              /**< protocol T=0 or T=1 */
            public fixed byte pIntBuffer[INT_BUFFER_SIZE];      /**< Buffer internally used as rx tx buffer */
            public ushort wIntBufferLen;                        /**< Length of the internally used buffer */
            public IntPtr pIntPayload;                          /**< Buffer internally used as payload buffer */
            public ushort wIntPayloadLen;                       /**< Length of the internally used payload buffer */
            public ushort wI2CProtocol;                         /**< Used Protocol for I2C communication */
            public byte bBwtMultiplier;                         /**< Multiplier for Bwt Timeout in case of WTX */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_Sw_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams               /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4_Sw_Cmd_SendI2CSBlock(
            IntPtr pDataParams,                 /**< [In] Pointer to an initialized HAL parameter structure. */
            byte bType,                         /**< [In] Type of S Block */
            byte[] pData,                       /**< [In] Pointer to data to send */
            ushort wDataLength,                 /**< [In] Length of the data to send */
            ref IntPtr ppResponse,              /**< [Out] Pointer to received data */
            ref ushort pResponseLength          /**< [Out] number of received data bytes */
            );

        #endregion

        #region INIT

        public Status_t Init(HalContact.Generic pHal)
        {
            return phpalI7816p4_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

        #endregion

        #region CUSTOM_FUNCTIONS
        public Status_t Cmd_SendI2CSBlock(
            byte bType,             /**< [In] type of s block */
            byte[] pData,           /**< [In] data to send */
            out byte[] pResponse    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wResponseLength = 0;
            IntPtr pResponseInt = IntPtr.Zero;

            status = phpalI7816p4_Sw_Cmd_SendI2CSBlock(
                m_pDataParams,
                bType,
                pData,
                (ushort)pData.Length,
                ref pResponseInt,
                ref wResponseLength);

            if ((pResponseInt != IntPtr.Zero) && (wResponseLength > 0))
            {
                pResponse = new byte[wResponseLength];
                Marshal.Copy(pResponseInt, pResponse, 0, wResponseLength);
            }
            else
            {
                pResponse = null;
            }

            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

	#endregion
}
#endif
