/*
 * Copyright 2022, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI7816p4a
{
    #region DEFINES

    public enum Config : int
    {
        IFSC = 0x0001,      /**< Set / Get PICC Frame Size. */
        IFSD                /**< Set / Get PCD Frame Size. */
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4a_IfsRequest(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bIfsd          /**< [In] Frame Size Integer */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4a_Pps(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bProtocol,     /**< [In] ProtocolType used after PPS */
            byte bFValue,       /**< [In] F Value for Datarate */
            byte bDValue        /**< [In] D Value for Datarate */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4a_ActivateCard(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bProtocol,         /**< [In] ProtocolType used after PPS */
            byte bIfsd,             /**< [In] Frame Size Integer */
            byte bFValue,           /**< [In] F Value for Datarate */
            byte bDValue,           /**< [In] D Value for Datarate */
            byte[] pAtr,            /**< [Out] ATR */
            ref ushort pwATRLength  /**< [Out] Length of the reiceived ATR. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4a_GetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue   /**< [Out] Configuration Value */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t IfsRequest(
            byte bIfsd          /**< [In] Frame Size Integer */
            )
        {
            return phpalI7816p4a_IfsRequest(m_pDataParams, bIfsd);
        }

        public Status_t Pps(
            NxpRdLibNet.HalContact.Protocol bProtocol,      /**< [In] ProtocolType used after PPS */
            byte bFValue,                                   /**< [In] F Value for Datarate */
            byte bDValue                                    /**< [In] D Value for Datarate */
            )
        {
            return phpalI7816p4a_Pps(m_pDataParams, (byte)bProtocol, bFValue, bDValue);
        }

        public Status_t ActivateCard(
            NxpRdLibNet.HalContact.Protocol bProtocol,      /**< [In] ProtocolType used after PPS */
            byte bFValue,                                   /**< [In] F Value for Datarate */
            byte bDValue,                                   /**< [In] D Value for Datarate */
            out byte[] pAtr,                                /**< [Out] ATR */
            byte bIfsd = 0                                  /**< [In] Frame Size Integer */
            )
        {
            Status_t status;
            pAtr = new byte[33];
            ushort wAtrSize = 0;
            status = phpalI7816p4a_ActivateCard(m_pDataParams, (byte)bProtocol, bIfsd, bFValue, bDValue, pAtr, ref wAtrSize);

            /* Resize Atr buffer to actual Atr size */
            if (status.Equals(Error_Gen.SUCCESS))
            {
                Array.Resize<byte>(ref pAtr, wAtrSize);
            }

            return status;
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phpalI7816p4a_GetConfig(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = (int)wValue;
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI7816p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer */
            public byte bCid;               /**< Card Identifier */
            public byte bIfsd;              /**< (Current) PCD Frame Size Integer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI7816p4a_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init(HalContact.Generic pHalContact)
        {
            return phpalI7816p4a_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHalContact.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, HalContact.Generic pHalContact)
        {
            return phpalI7816p4a_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHalContact.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

	#endregion
}
#endif
