/*
 * Copyright 2013 - 2019, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palMifare
{
    #region DEFINES

    public enum Error : byte
    {
        /// <summary> NAK 0 </summary>
        NAK0 = ( CustomCodes.ERROR_BEGIN ),
        /// <summary> NAK 1 </summary>
        NAK1,
        /// <summary> NAK 4 </summary>
        NAK4,
        /// <summary> NAK 5 </summary>
        NAK5,
        /// <summary> NAK 6 </summary>
        NAK6,
        /// <summary> NAK 7 </summary>
        NAK7,
        /// <summary> NAK 9 </summary>
        NAK9
    }

    public enum KeyType : byte
    {
        KEYA = 0x0A,
        KEYB = 0x0B
    }

    public enum Config : int
    {
        BLOCKNO = 0x0000,   /**< Set / Get Blocknumber. */
        CID,                /**< Set / Get Card Identifier. */
        NAD,                /**< Set / Get Node Address. */
        FWI,                /**< Set / Get Frame Waiting Integer. */
        FSI,                /**< Set / Get PCD & PICC Frame Size Integer. */
        RETRYCOUNT,         /**< Set / Get Maximum Retry Count. */
        ExchangeMode = CustomCodes.CONFIG_BEGIN
    }

    public enum PCSCExchangeType : byte
    {
        PCSC_TRANSPARENT = 0,
        PCSC_NORMAL = 1,
        UNKNOWN = 0xFF
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DEFINES

        public const byte KEY_LENGTH = 0x06;

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_ExchangeL3 (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_ExchangeL4 (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_ExchangePc (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_ExchangeRaw (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            byte bTxLastBits,       /**< [In] number of valid bits of last byte (Tx) */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength,   /**< [Out] number of received data bytes */
            ref byte pRxLastBits    /**< [Out] number of valid bits of last byte (Rx) */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_MfcAuthenticateKeyNo (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure.*/
            byte bBlockNo,          /**< [In] Blocknumber on card to authenticate to. */
            byte bKeyType,          /**< [In] Either #PHPAL_MIFARE_KEYA or #PHPAL_MIFARE_KEYB, ombineable with #PHPAL_MIFARE_USE_KEYMODIFIER. */
            ushort wKeyNo,          /**< [In] Key number to be used in authentication. */
            ushort wKeyVersion,     /**< [In] Key version to be used in authentication. */
            byte[] pUid             /**< [In] Serial number of current cascade level; uint8_t[4]. */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_MfcAuthenticate (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure.*/
            byte bBlockNo,          /**< [In] Blocknumber on card to authenticate to. */
            byte bKeyType,          /**< [In] Either #PHPAL_MIFARE_KEYA or #PHPAL_MIFARE_KEYB, ombineable with #PHPAL_MIFARE_USE_KEYMODIFIER. */
            byte[] pKey,            /**< [In] Key to be used in authentication. */
            byte[] pUid             /**< [In] Serial number of current cascade level; uint8_t[4]. */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_SetConfig (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue       /**< [In] Configuration Value */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t ExchangeL3 (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalMifare_ExchangeL3 (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                ( ushort ) pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t ExchangeL4 (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalMifare_ExchangeL4 (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                ( ushort ) pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t ExchangePc (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalMifare_ExchangePc (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                ( ushort ) pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t ExchangeRaw (
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            byte bTxLastBits,       /**< [In] number of valid bits of last byte (Tx) */
            out byte[] pRxBuffer,   /**< [Out] received data */
            out byte pRxLastBits    /**< [Out] number of valid bits of last byte (Rx) */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            pRxLastBits = 0;
            status = phpalMifare_ExchangeRaw (
                m_pDataParams,
                ( ushort ) wOption,
                pTxBuffer,
                ( ushort ) pTxBuffer.Length,
                bTxLastBits,
                ref pRxBufferInt,
                ref wRxLength,
                ref pRxLastBits );

            if ( wRxLength > 0 )
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, pRxBuffer, 0, wRxLength );
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        /// <summary>
        /// Mifare Classic Authenticate
        /// </summary>
        /// <param name="pDataParams"></param>
        /// <param name="bBlockNo"></param>
        /// <param name="bKeyType"></param>
        /// <param name="pKey"></param>
        /// <param name="pUid"></param>
        /// <returns></returns>
        public Status_t MfcAuthenticateKeyNo (
            byte bBlockNo,          /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,       /**< [In] Either #PHPAL_MIFARE_KEYA or #PHPAL_MIFARE_KEYB, combineable with #PHPAL_MIFARE_USE_KEYMODIFIER. */
            ushort wKeyNo,          /**< [In] Key number to be used in authentication. */
            ushort wKeyVersion,     /**< [In] Key version to be used in authentication. */
            byte[] pUid             /**< [In] Serial number of current cascade level; uint8_t[4]. */
            )
        {
            byte KeyType = (byte)bKeyType;
            return phpalMifare_MfcAuthenticateKeyNo ( m_pDataParams, bBlockNo, KeyType, wKeyNo, wKeyVersion, pUid );
        }

        public Status_t MfcAuthenticate (
            byte bBlockNo,      /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,   /**< [In] Either #PHPAL_MIFARE_KEYA or #PHPAL_MIFARE_KEYB, combineable with #PHPAL_MIFARE_USE_KEYMODIFIER. */
            byte[] pKey,        /**< [In] Key to be used in authentication. */
            byte[] pUid         /**< [In] Serial number of current cascade level; uint8_t[4]. */
            )
        {
            byte KeyType = (byte)bKeyType;
            return phpalMifare_MfcAuthenticate ( m_pDataParams, bBlockNo, KeyType, pKey, pUid );
        }

        public Status_t SetConfig (
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return phpalMifare_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;       /**< Pointer to the parameter structure of the underlying HAL layer */
            public IntPtr pI14443p4DataParams;  /**< Pointer to the parameter structure of the underlying ISO14443-4 layer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_Sw_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams,              /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pI14443p4DataParams          /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal, palI14443p4.Generic pI14443p4 )
        {
            IntPtr pI14443p4DataParams = IntPtr.Zero;

            if ( pI14443p4 != null )
            {
                pI14443p4DataParams = pI14443p4.m_pDataParams;
            }

            return phpalMifare_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams, pI14443p4DataParams );
        }

#if DEBUG
        public Status_t Init ( int wDataParamSize, Hal.Generic pHal, palI14443p4.Generic pI14443p4 )
        {
            IntPtr pI14443p4DataParams = IntPtr.Zero;

            if ( pI14443p4 != null )
            {
                pI14443p4DataParams = pI14443p4.m_pDataParams;
            }

            return phpalMifare_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, pHal.m_pDataParams, pI14443p4DataParams );
        }
#endif
        public void SetPalWId ( ushort val )
        {
            this.m_DataParamsInt[0].wId = val;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL
    #region SW_CONTACT

    public class SwContact : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalContactDataParams;/**< Pointer to the parameter structure of the underlying HALContact layer */
            public IntPtr pI7816p4DataParams;   /**< Pointer to the parameter structure of the underlying ISO7816-4 layer */
            public IntPtr pRxBuffer;            /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;           /**< Size of the global receive buffer. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_SwContact_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalContactDataParams,       /**< [In] Pointer to the parameter structure of the underlying HALContact layer */
            IntPtr pI7816p4DataParams,          /**< [In] Pointer to the parameter structure of the underlying ISO7816-4 layer */
            IntPtr pRxBuffer,                   /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize                  /**< [In] Size of the global receive buffer. */
            );

        #endregion

        #region INIT

        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        public Status_t Init ( HalContact.Generic pHalContact, palI7816p4.Generic pI7816p4, int wRxBufferSize = 0xFFFF )
        {
            IntPtr pI7816p4DataParams = IntPtr.Zero;

            // Adjust RxBuffer length
            if ( wRxBufferSize > 0xFFFF )
            {
                wRxBufferSize = 0xFFFF;
            }

            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Allocate buffers
            m_bRxBuffer = new byte[wRxBufferSize];

            // Link given buffers
            m_pRxBuffer = GCHandle.Alloc ( m_bRxBuffer, GCHandleType.Pinned );

            if ( pI7816p4 != null )
            {
                pI7816p4DataParams = pI7816p4.m_pDataParams;
            }

            return phpalMifare_SwContact_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHalContact.m_pDataParams, pI7816p4DataParams, m_pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize );
        }

#if DEBUG
        public Status_t Init ( int wDataParamSize, HalContact.Generic pHalContact, palI7816p4.Generic pI7816p4, int wRxBufferSize = 0xFFFF )
        {
            IntPtr pI7816p4DataParams = IntPtr.Zero;

            // Adjust RxBuffer length
            if ( wRxBufferSize > 0xFFFF )
            {
                wRxBufferSize = 0xFFFF;
            }

            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Allocate buffers
            m_bRxBuffer = new byte[wRxBufferSize];

            // Link given buffers
            m_pRxBuffer = GCHandle.Alloc ( m_bRxBuffer, GCHandleType.Pinned );

            if ( pI7816p4 != null )
            {
                pI7816p4DataParams = pI7816p4.m_pDataParams;
            }

            return phpalMifare_SwContact_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, pHalContact.m_pDataParams, pI7816p4DataParams, m_pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize );
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SwContact ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SwContact ()
        {
            // Free Buffers
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region Sam_X
    public class Sam_X : Generic
    {
        #region Data Structure
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pHalDataParams;
            IntPtr pI14443p4DataParams;
        };

        #endregion

        #region Dll Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_Sam_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams, IntPtr pI14443p4DataParams );
        #endregion

        #region Initialization
        public Status_t Init ( Hal.Sam oHal, palI14443p4.Generic oI14443p4 )
        {
            IntPtr pI14443p4DataParams = IntPtr.Zero;

            if ( oI14443p4 != null )
            {
                pI14443p4DataParams = oI14443p4.m_pDataParams;
            }

            return phpalMifare_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams, pI14443p4DataParams );
        }
        #endregion

        #region Memorey Mapping
        private DataParams_t[] m_DataParamsInt;

        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif

    #region SamAV2_X

    public class SamAV2_X : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;          /**< Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pI14443p4DataParams;     /**< Pointer to the parameter structure of the underlying ISO14443-4 layer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_SamAV2_X_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams,              /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pI14443p4DataParams          /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer */
            );

        #endregion

        #region INIT

        public Status_t Init (
            Hal.SamAV2 pHal,
            palI14443p4.Generic pI14443p4 )
        {
            IntPtr pI14443p4DataParams = IntPtr.Zero;

            if ( pI14443p4 != null )
            {
                pI14443p4DataParams = pI14443p4.m_pDataParams;
            }

            return phpalMifare_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams, pI14443p4DataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X

    public class SamAV3_X : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;                     /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;          /**< Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pI14443p4DataParams;     /**< Pointer to the parameter structure of the underlying ISO14443-4 layer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_SamAV3_X_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams,              /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pI14443p4DataParams          /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer */
            );

        #endregion

        #region INIT

        public Status_t Init (
            Hal.SamAV3 pHal,
            palI14443p4.Generic pI14443p4 )
        {
            IntPtr pI14443p4DataParams = IntPtr.Zero;

            if ( pI14443p4 != null )
            {
                pI14443p4DataParams = pI14443p4.m_pDataParams;
            }

            return phpalMifare_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams, pI14443p4DataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
#endif

    #region Rd710

    public class Rd710 : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;       /**< Pointer to the parameter structure of the underlying HAL layer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_Rd710_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams              /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Rd710 pHal )
        {
            return phpalMifare_Rd710_Init (
                                        ref m_DataParamsInt[0],
                                        ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                        pHal.m_pDataParams );
        }

#if DEBUG
        public Status_t Init ( int wDataParamSize, Hal.Rd710 pHal, palI14443p4.Generic pI14443p4 )
        {
            return phpalMifare_Rd710_Init (
                                        ref m_DataParamsInt[0],
                                        ( ushort ) wDataParamSize,
                                        pHal.m_pDataParams );
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Rd710 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Rd710 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region PCSC

    public class PCSC : palMifare.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;                   /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;        /**< Pointer to the parameter structure of the underlying HAL layer */
            public IntPtr pPalI14443p4DataParams;/**< Pointer to the parameter structure of the underlying PAL IS014443-4 layer. */
            public byte   bWaitForWriteData;     /**< Flag to indicate wait for write data for update binary command. */
            public byte bWaitForIncrementValue;  /**< Flag to indicate wait for increment value for increment command. */
            public byte bWaitForDecrementValue;  /**< Flag to indicate wait for decrement value for decrement command. */
            public byte bExchangeType;           /**< Type of exchange to use. #PHPAL_MIFARE_PCSC_NORMAL (default) or #PHPAL_MIFARE_PCSC_TRANSPARENT. */
            public ushort wWriteAddress;         /**< Address for update binary command. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalMifare_Pcsc_Init (
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams,              /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            IntPtr pPalI14443p4DataParams       /**< [In] Pointer to the parameter structure of the underlying HAL layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal, palI14443p4.Generic pI14443p4 )
        {
            return phpalMifare_Pcsc_Init (
                                        ref m_DataParamsInt[0],
                                        ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                        pHal.m_pDataParams,
                                        pI14443p4.m_pDataParams );
        }

#if DEBUG
        public Status_t Init ( int wDataParamSize, Hal.PCSC pHal, palI14443p4.Generic pI14443p4 )
        {
            return phpalMifare_Pcsc_Init (
                                        ref m_DataParamsInt[0],
                                        ( ushort ) wDataParamSize,
                                        pHal.m_pDataParams,
                                        pI14443p4.m_pDataParams );
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public PCSC ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~PCSC ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
