/*
 * Copyright 2013 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palSli15693
{
    #region Defines
    #region Custom Error Codes
    public enum Error : byte
    {
        /// <summary>
        /// ISO15693 error from VICC; ErrorCode can be accessed via <seealso cref="Config.ADD_INFO"/>
        /// </summary>
        ISO15693 = ( CustomCodes.ERROR_BEGIN ),

        /// <summary>
        /// ISO15693 error from VICC for In-Process reply Part 1 response;
        /// ErrorCode can be accessed via <seealso cref="Config.ADD_INFO"/>
        /// </summary>
        IN_PROCESS_REPLY_PART1_ERROR = ( CustomCodes.ERROR_BEGIN + 1 ),

        /// <summary>
        /// ISO15693 error from VICC for In-Process reply Part 2 response;
        /// ErrorCode can be accessed via <seealso cref="Config.ADD_INFO"/>
        /// </summary>
        IN_PROCESS_REPLY_PART2_ERROR = ( CustomCodes.ERROR_BEGIN + 2 )
    }
    #endregion

    #region Error codes of VICC
    /// <summary>
    /// Response ErrorCodes according to ISO15693. Codes ranging from A0 - DF are custom command error codes.
    /// </summary>
    public enum I15693Errors : byte
    {
        /// <summary>
        /// The command is not supported, i.e. the request code is not recognized.
        /// </summary>
        NOT_SUPPORTED = 0x01,

        /// <summary>
        /// The command is not recognized, for example: a format error occurred.
        /// </summary>
        FORMAT = 0x02,

        /// <summary>
        /// The command option is not supported.
        /// </summary>
        OPTION = 0x03,

        /// <summary>
        /// Error with no information given or a specific error code is not supported.
        /// </summary>
        GENERIC = 0x0F,

        /// <summary>
        /// The specified block is not available (doesn't exist).
        /// </summary>
        BLOCK_NA = 0x10,

        /// <summary>
        /// The specified block is already locked and thus cannot be locked again.
        /// </summary>
        BLOCK_LOCKED1 = 0x11,

        /// <summary>
        /// The specified block is locked and its content cannot be changed.
        /// </summary>
        BLOCK_LOCKED2 = 0x12,

        /// <summary>
        /// The specified block was not successfuly programmed.
        /// </summary>
        BLOCK_WRITE1 = 0x13,

        /// <summary>
        /// The specified block was not successfuly locked.
        /// </summary>
        BLOCK_WRITE2 = 0x14
    }
    #endregion

    #region Request Flags
    /// <summary>
    /// Request Flags 1 to 4 according to ISO15693. In a request, the field "flags" specifies the actions to be performed by the VICC.
    /// It consists of eight bits.
    /// </summary>
    public enum ReqFlags : byte
    {
        /// <summary>
        /// Sub-carrier_flag. If set two sub-carriers shall be used (VICC to VCD). Else a single sub-carrier
        /// frequency shall be used by the VICC.
        /// </summary>
        TWO_SUB_CARRIERS = 1,

        /// <summary>
        /// Data_rate_flag. If set high data rate shall be used (VICC to VCD). Else low data rate shall be used.
        /// </summary>
        DATA_RATE = 2,

        /// <summary>
        /// Inventory_flag. Shall be set for inventory request. If set flags 5 to 8 meaning is as per this
        /// <seealso cref="ReqFlagsInv"/>. If not set flags 5 to 8 meaning is defined by this
        /// <seealso cref="ReqFlagsNoInv"/>.
        /// </summary>
        INVENTORY = 4,

        /// <summary>
        /// Protocol_Extension_flag. If set protocol format is extended. Reserved for future use. No supported
        /// in current version.
        /// </summary>
        PROTOCOL_EXTENSION = 8,

        /// <summary>
        /// Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by the command.
        /// </summary>
        OPTION = 0x40,

        /// <summary>
        /// RFU as per ISO15693 but this flag should be utilized for PLUTUS product only. This flag is used to
        /// set the hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps)
        /// </summary>
        FAST_DATA_RATE = 0x80
    }
    #endregion

    #region Request Flags if Inventory Flag is Not Set
    /// <summary>
    /// Request Flags 5 to 7 according to ISO15693 when inventory flag is NOT set.
    /// Flag 8 is RFU and shall be set to zero.
    /// </summary>
    public enum ReqFlagsNoInv : byte
    {
        /// <summary>
        /// Select_flag. If set only VICC in selected state shall respond. The Address_flag <seealso cref="ADDRESSED"/>
        /// shall be set to 0 and the UID field will not be included in the request. If not set Request shall be executed by any VICC according to the setting
        /// of Address_flag <seealso cref="ADDRESSED"/>.
        /// </summary>
        SELECTED = 0x10,

        /// <summary>
        /// Address_flag. If set request is addressed. UID field is included in request. It shall be executed only by the
        /// VICC whose UID matches the UID specified in the request. If not set request is not addressed. UID field is not
        /// included in request. It shall be executed by any VICC.
        /// </summary>
        ADDRESSED = 0x20,

        /// <summary>
        /// Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by the command.
        /// </summary>
        OPTION = 0x40,

        /// <summary>
        /// RFU as per ISO15693 but this flag should be utilized for PLUTUS product only. This flag is used to
        /// set the hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps)
        /// </summary>
        FAST_DATA_RATE = 0x80
    }
    #endregion

    #region Request Flags if Inventory Flag is Set
    /// <summary>
    /// Request Flags 5 to 7 according to ISO15693 when inventory flag IS set.
    /// Flag 8 is RFU and shall be set to zero.
    /// </summary>
    public enum ReqFlagsInv : byte
    {
        /// <summary>
        /// AFI_flag. If set AFI field is present in request. If not set AFI field is not present in request.
        /// </summary>
        AFI = 0x10,

        /// <summary>
        /// Nb_slots_flag. If set Use 1 slot instead of 16 slots for inventory. If not set use 16 slots.
        /// </summary>
        NBSLOTS = 0x20,

        /// <summary>
        /// Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by the command.
        /// </summary>
        OPTION = 0x40,

        /// <summary>
        /// RFU as per ISO15693 but this flag should be utilized for PLUTUS product only. This flag is used to
        /// set the hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps)
        /// </summary>
        FAST_DATA_RATE = 0x80
    }
    #endregion

    #region Response Flags
    /// <summary>
    /// Response Flags 1 and 4 according to ISO15693. All others flags are RFU and shall not be interpreted.
    /// </summary>
    public enum RespFlags : byte
    {
        /// <summary>
        /// Error_flag. Error detected. Error code is in the "Error" field of response. When a response is received from the VICC with
        /// the error flag set, library returns the custom error code <seealso cref="Error.ISO15693"/>.  The
        /// configuration option <seealso cref="Config.ADD_INFO"/> can be used to get the actual
        /// <seealso cref="I15693Errors"/> returned by the VICC.
        /// </summary>
        ERROR = 0x01,

        /// <summary>
        /// Extension_flag. Protocol format is extended. Reserved for future use.
        /// </summary>
        EXTENSION = 0x08
    }
    #endregion

    #region Activate Options
    /// <summary>
    /// Activate Card command options. Options for <seealso cref="Generic.ActivateCard"/>.
    /// </summary>
    public enum OptionsActivate : int
    {
        /// <summary>
        /// Default option. Same as <seealso cref="OptionsActivate.ADDRESSED"/>.
        /// <summary>
        DEFAULT = 0,

        /// /// <summary>
        /// Card will be in addressed state (i.e. SELECT command is not send) after <seealso cref="Generic.ActivateCard"/>
        /// command. After activate card with this option, to bring VICC to SELECTED state, <seealso cref="Generic.Select"/>
        /// command can be used.
        /// </summary>
        ADDRESSED = DEFAULT,

        /// <summary>
        /// Card will be in selected state (i.e. SELECT command is send) after \<seealso cref="Generic.ActivateCard"/>
        /// command.
        /// </summary>
        SELECTED = 1
    }
    #endregion

    #region EOF Options
    /// <summary>
    /// EOF command options for <seealso cref="Generic.SendEof"/> command.
    /// </summary>
    public enum OptionsEof : int
    {
        /// <summary>
        /// Send an EOF with timing configuration for an <seealso cref="Generic.Inventory"/> round.
        /// </summary>
        NEXT_SLOT = 0,

        /// <summary>
        /// Send an EOF with timing configuration for
        /// <seealso cref="Generic.InventoryRead"/>
        /// <seealso cref="Generic.FastInventoryRead"/>,
        /// <seealso cref="Generic.InventoryPageRead"/> and
        /// <seealso cref="Generic.FastInventoryPageRead"/>.
        /// </summary>
        NEXT_SLOT_INV_READ = 1,

        /// <summary>
        /// Send an EOF with timing configuration for write alike commands without long waiting time
        /// <seealso cref="Config.TIMEOUT_US"/>
        /// </summary>
        WRITE_ALIKE = 2,

        /// <summary>
        /// Send an EOF with timing configuration for write alike commands with long waiting time
        /// <seealso cref="Config.TIMEOUT_US"/>
        /// </summary>
        WRITE_ALIKE_WITH_WAIT = 3,
    }
    #endregion

    #region Timeout Options
    /// <summary>
    /// Timeout values.
    /// </summary>
    public enum Timeout : ushort
    {
        /// <summary>
        /// Timeout T1 in microseconds as defined in ISO/IEC 15693, 9.1.4.
        /// </summary>
        SHORT_US = 324,

        /// <summary>
        /// Tolerance Timeout Delta in microseconds.
        /// </summary>
        DELTA_US = 50000,

        /// <summary>
        /// Timeout T1 in microseconds, 160.5 is taken as 162us Supported only by NTAG5.
        /// </summary>
        NTAG5_I2C_US = 162,

        /// <summary>
        /// Timeout T1 in microseconds, 80.2 is taken as 81us Supported only by NTAG5.
        /// </summary>
        NTAG5_I2C_81_US = 81,

        /// <summary>
        /// Extended timeout value in microseconds for long timeouts.
        /// </summary>
        LONG_US = 20000,

        /// <summary>
        /// Special frame EOF delay time in microseconds (FDTv, EOF).
        /// </summary>
        EOF_US = 16850,

        /// <summary>
        /// FDT time max2 in microseconds (FDTv, listen, max2).
        /// </summary>
        MAX2_US = 19950
    }
    #endregion

    #region DataRate Options
    /// <summary>
    /// The baud Rates supported by NTAG5 and mentioned in NTAG5 ICAD.
    /// </summary>
    public enum DataRate : byte
    {
        /// <summary>
        /// The default baud rate is 26kbps baud rate, is supported in both NFC and ISO mode
        /// This is just for representation Purpose, a Get Config after applying this baud rate
        /// will return PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4 for TX and PHHAL_HW_RF_RX_DATARATE_HIGH
        /// for RX.
        /// </summary>
        KBPS_26 = 0x1A,

        /// <summary>
        /// 53kbps baud rate, supported in TX as well as RX but only for ISO mode
        /// For RX this is just for representation Purpose, a Get Config after applying this baud rate
        /// will return PHPAL_SLI15693_53KBPS_DATARATE for TX and PHHAL_HW_RF_RX_DATARATE_FASTHIGH
        /// for RX.
        /// </summary>
        KBPS_53 = 0x1B,

        /// <summary>
        /// 106kbps baud rate, supported in TX as well as RX but only for ISO mode.
        /// </summary>
        KBPS_106 = 0x1C,

        /// <summary>
        /// 212kbps baud rate, supported only in TX and ISO mode.
        /// </summary>
        KBPS_212 = 0x1D
    }
    #endregion

    #region Extended flags
    /// <summary>
    /// Extended Inventory Read Options Flag.
    /// </summary>
    public enum ExtendedFlags : byte
    {
        /// <summary>
        /// Inventory Read Extension Flag
        ///     - If set the additional features of inventory mentioned along with the command will be performed by the VICC.
        ///     - This flags needs to be set as the MSB bit for Mask Len.
        /// </summary>
        INVENTORY_READ_EXTENSION = 0x80,

        /// <summary>
        /// Extended Inventory Read Options Flag. Option to indicate that the VICC with EAS enabled will respond
        /// </summary>
        EAS_MODE = 0x01,

        /// <summary>
        /// If set the complete UID is transmitted by the VICC to the reader irrespective of the mask length.
        /// </summary>
        UID_MODE = 0x02,

        /// <summary>
        /// If set the 16-bit CID will be transmitted in the command and only tags with the same CID will respond.
        /// </summary>
        CID_COMPARE = 0x04,

        /// <summary>
        /// If set the Custom ID(CID) is transmitted by the VICC to the reader.
        /// </summary>
        CID_RESPONSE = 0x08,

        /// <summary>
        /// If set no user memory data are requested from the tag. If not set the tag will add the user memory blocks
        /// in the response as requested in command
        /// </summary>
        SKIP_DATA = 0x10,

        /// <summary>
        /// If set the VICC will go to quiet state after receiving the command.
        /// </summary>
        GOTO_QUIET = 0x20,

        /// <summary>
        /// If set the VICC will go to quiet state after receiving the command.
        /// </summary>
        GOTO_PERSISTENT_QUIET = 0x40,

        RFU3 = 0x80
    }
    #endregion

    #region Config Options
    /// <summary>
    /// Configuration and Option definitions. These options shall be used along with <seealso cref="Generic.SetConfig"/> and
    /// <seealso cref="Generic.GetConfig"/> to configure PAL Sli15693 layer.
    /// </summary>
    public enum Config : int
    {
        /// <summary>
        /// Set or get the flags byte. This shall be used to set or get the value of <seealso cref="ReqFlags"/> "Request flag"
        /// field used in the commands sent by the library.
        /// </summary>
        FLAGS = 0,

        /// <summary>
        /// Get the additional info word. This shall be used to set or get the value of "Reponse Flag" for authentication commands or the error code
        /// field used in the response received by the library.
        /// </summary>
        ADD_INFO = 1,

        /// <summary>
        /// Set or get the current timeout in microseconds.
        /// </summary>
        TIMEOUT_US = 2,

        /// <summary>
        /// Set or get the current timeout in milliseconds.
        /// </summary>
        TIMEOUT_MS = 3,

        /// <summary>
        /// Set or get the current TX Data rate.
        /// </summary>
        TXDATARATE = 4,

        /// <summary>
        /// Set or get the current RX Data rate.
        /// </summary>
        RXDATARATE = 5,

        /// <summary>
        /// Set or get the current T1 timeout after Parameter Select.
        /// </summary>
        T1_PARAMETER = 6,

        /// <summary>
        /// Option to Get / Set the buffering of response data to be enabled or disabled.
        /// By default this flag will be disabled.
        /// </summary>
        ENABLE_BUFFERING = 7,

        /// <summary>
        /// Set / Get Maximum Retry Count.
        /// </summary>
        MAX_RETRYCOUNT = 8,

        /// <summary>
        /// Set / Get IC Manufacturer Code.
        /// </summary>
        IC_MFGCODE = 9,

        /// <summary>
        /// Set or get the Transparent mode.
        /// </summary>
        TRANSPARENT = 16,
    }
    #endregion Config  Options
    #endregion Defines

    #region Generic
    /// <summary>
    /// Generic Iso15693 PAL component of Reader Library .Net framework.
    /// </summary>
    public abstract class Generic
    {
        #region Public variables
        /// <summary>
        /// Variable to represent the UID length.
        /// </summary>
        public const byte UID_LENGTH = 0x08;

        /// <summary>
        /// Variable to represent the bytes available in one block.
        /// </summary>
        public const byte BLOCK_LENGTH = 0x04;

        /// <summary>
        /// Variable to represent the bytes available in one page. One page consists of 4
        /// blocks which is nothing but 16 bytes.</summary>
        public const byte PAGE_LENGTH = 0x10;
        #endregion Public variables

        #region DLL Imports
        #region Mandatory Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_Inventory ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, ref byte pDsfid, byte[] pUid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_StayQuiet ( IntPtr pDataParams );
        #endregion

        #region Optional Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_Select ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_ResetToReady ( IntPtr pDataParams );
        #endregion

        #region Custom Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_InventoryRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bBlockNo, ushort wNoOfBlocks, byte[] pUid, ref byte bUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_InventoryReadExtended ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bExtendedOptions, byte[] pCID, byte bBlockNo, ushort wNumOfBlocks, byte[] pCDIDOut, byte[] pUid,
            ref byte pUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_FastInventoryRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bBlockNo, ushort wNoOfBlocks, byte[] pUid, ref byte bUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_FastInventoryReadExtended ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bExtendedOptions, byte[] pCID, byte bBlockNo, ushort wNumOfBlocks, byte[] pCDIDOut, byte[] pUid,
            ref byte pUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_InventoryPageRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bPageNo, ushort wNoOfPages, byte[] pUid, ref byte bUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_FastInventoryPageRead ( IntPtr pDataParams, byte bFlags, byte bAfi, byte[] pMask,
            byte bMaskBitLen, byte bPageNo, ushort wNoOfPages, byte[] pUid, ref byte bUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_StayQuietPersistent ( IntPtr pDataParams );
        #endregion

        #region Common Commands
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_ActivateCard ( IntPtr pDataParams, byte bOptions, byte bFlags, byte bAfi,
            byte[] pMask, byte bMaskBitLen, ref byte pDsfid, byte[] pUid, ref byte pMoreCardsAvailable );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_SendEof ( IntPtr pDataParams, byte bOption, ref byte pDsfid, byte[] pUid,
            ref byte pUidLen, byte[] pData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_Exchange ( IntPtr pDataParams, ushort wOption, byte[] pTxBuffer, ushort wTxLen,
            ref IntPtr ppRxBuffer, ref ushort pRxLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_InProcessReplyASync ( IntPtr pDataParams, byte[] pTxBuffer, ushort wTxBuffLen,
            byte[] pResponse1, ref ushort pResp1_Len, byte[] pResponse2, ref ushort pResp2_Len );
        #endregion

        #region Utility
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_GetSerialNo ( IntPtr pDataParams, byte[] pUid, ref byte bUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_SetSerialNo ( IntPtr pDataParams, byte[] pUid, byte bUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort wValue );
        #endregion
        #endregion DLL Imports

        #region DLL Wrapper
        #region Mandatory Commands
        /// <summary>
        /// Perform a ISO15693 Inventory command.
        ///
        /// When receiving the Inventory request, the VICC shall perform the anti-collision sequence. This is the fist command
        /// in the activation/detection sequence of VICC.
        ///
        /// When issuing the inventory command, the reader shall set the Nb_slots_flag <seealso cref="ReqFlagsInv.NBSLOTS"/> to the desired
        /// setting to select 1 slot or 16 slots for the inventory process.
        ///
        /// The mask length indicates the number of significant bits of the mask value. It can have any value between 0 and 60 when
        /// 16 slots are used and any value between 0 and 64 when 1 slot is used. The mask value is contained in an integer number
        /// of bytes. If the mask length is not a multiple of 8 (bits), the mask value MSB shall be padded with the required number
        /// of null (set to 0) bits so that the mask value is contained in an integer number of bytes.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bDsfid">Data Storage Format Identifier.</param>
        /// <param name="aUid">Received UID.</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Inventory ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, out byte bDsfid, out byte[] aUid )
        {
            bDsfid = 0x00;
            aUid = new byte[8];

            Status_t oStatus = phpalSli15693_Inventory ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, ref bDsfid, aUid );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
                aUid = new byte[0];

            return oStatus;
        }

        /// <summary>
        /// Perform a ISO15693 StayQuiet command.
        ///
        /// When receiving the Stay quiet command, the VICC shall enter the quiet state and shall not send back a response. There is
        /// no response to the stay quiet command.
        ///
        /// In quiet state, the VICC shall not process any request where Inventory_flag is set and the VICC shall process any addressed
        /// request. The VICC shall exit the quiet state when reset (power off), receiving a <seealso cref="Generic.Select"/> (It shall
        /// then go to the selected state if supported or return an error if not supported), receiving a <seealso cref="Generic.ResetToReady"/>
        /// request" (It shall then go to the Ready state).
        /// </summary>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t StayQuiet ()
        {
            return phpalSli15693_StayQuiet ( m_pDataParams );
        }
        #endregion

        #region Optional Commands
        /// <summary>
        /// Perform a ISO15693 Select command.
        ///
        /// When receiving the Select command, if the UID is equal to its own UID, the VICC shall enter the selected state and
        /// shall send a response. If the UID is different to its own and in selected state, the VICC shall return to the Ready
        /// state and shall not send a response. If the UID is different to its own and not in selected state, the VICC shall
        /// remain in its state and shall not send a response.
        /// </summary>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Select ()
        {
            return phpalSli15693_Select ( m_pDataParams );
        }

        /// <summary>
        /// Perform a ISO15693 ResetToReady command.
        /// When receiving a Reset to ready command, the VICC shall return to the Ready state.
        /// </summary>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t ResetToReady ()
        {
            return phpalSli15693_ResetToReady ( m_pDataParams );
        }
        #endregion

        #region Custom Commands
        /// <summary>
        /// Perform ICODE SLI InventoryRead command.
        ///
        /// When receiving the INVENTORY READ request, the ICODE SLI tag performs the same as the anti-collision sequence, with the difference that
        /// instead of the UID and the DSFID, the requested memory content is re-transmitted from the ICODE SLI. If an error is detected, the ICODE
        /// SLI tag remains silent.
        ///
        /// If the Option_flag  is not set, n blocks of data are re-transmitted. If the Option_flag is set, n blocks of data and the part of the UID
        /// which is not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead of padding with
        /// zeros up to the next byte boundary, the corresponding bits of the UID are returned.
        ///
        /// The number of bits of the re-transmitted UID can be calculated as follows:
        ///     16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
        ///     1 slot: 64 bits - mask length rounded up to the next byte boundary.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bBlockNo">Block number of first block to read</param>
        /// <param name="wNumOfBlocks">Number of blocks to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t InventoryRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bBlockNo, ushort wNumOfBlocks,
            out byte[] aUid, out byte[] aData )
        {
            byte bUidLen = 0;
            ushort wDataLen = 0;
            aUid = new byte[UID_LENGTH];
            aData = new byte[256 * BLOCK_LENGTH];

            Status_t oStatus = phpalSli15693_InventoryRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bBlockNo,
                wNumOfBlocks, aUid, ref bUidLen, aData, ref wDataLen );
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize ( ref aUid, bUidLen );
                Array.Resize ( ref aData, wDataLen );
            }
            else
            {
                aUid = new byte[0];
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ICODE SLI InventoryRead command in extended mode.
        ///
        /// When receiving the INVENTORY READ EXTENDED request, the ICODE SLI tag performs the same as the anti-collision sequence, with the difference
        /// that instead of the UID and the DSFID, the requested memory content is re-transmitted from the ICODE SLI. If an error is detected, the ICODE
        /// SLI tag remains silent.
        ///
        /// If the Option_flag is not set, n blocks of data are re-transmitted. If the Option_flag is set, n blocks of data and the part of the UID which
        /// is not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead of padding with zeros up
        /// to the next byte boundary, the corresponding bits of the UID are returned.
        ///
        /// The number of bits of the re-transmitted UID can be calculated as follows:
        ///     16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
        ///     1 slot: 64 bits - mask length rounded up to the next byte boundary.
        ///
        /// Depending on different values of Extended Options the API behaves differently.
        /// | **Bit** | **Value** | **Description**                                                                                             |
        /// |:-------:|:---------:|:------------------------------------------------------------------------------------------------------------|
        /// |    b1   |     0     | Label responds independent from the EAS status.                                                             |
        /// |    ^    |     1     | Only labels will respond which have the EAS enabled.                                                        |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_EAS_MODE "EAS"**                           |
        /// |    b2   |     0     | UID will be transmitted as in regular mode.                                                                 |
        /// |    ^    |     1     | Complete UID will be transmitted.                                                                           |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE "UID"**                           |
        /// |    b3   |     0     | No CID is transmitted in the command.                                                                       |
        /// |    ^    |     1     | 16 bits CID will be transmitted in the command and only matching tags should reply.                         |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE "CID In Command"**             |
        /// |    b4   |     0     | CID will not be transmitted in the response from the tag.                                                   |
        /// |    ^    |     1     | CID will be transmitted in the response from the tag.                                                       |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE "CID In Response"**           |
        /// |    b5   |     0     | Tag will add the user memory block in the response as requested.                                            |
        /// |    ^    |     1     | No user memory data is requested from the tag.                                                              |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA "Skip Data"**                    |
        /// | b6 & b7 |     00    | Remain in current state.                                                                                    |
        /// |    ^    |     01    | Go to persistent quiet state after response.                                                                |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_PERSISTENT_QUIET "Persistent Quiet"** |
        /// |    ^    |     10    | Go to quiet state after response.                                                                           |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_QUIET "Quiet"**                       |
        /// |    ^    |     11    | Only tags in the PERSISTENT QUIET state will respond to the command.                                        |
        /// |    b8   |     0     | RFU                                                                                                         |
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bExtendedOptions">Extended options</param>
        /// <param name="aCID">Two byte CID -> if marked in extended options</param>
        /// <param name="bBlockNo">Block number of first block to read</param>
        /// <param name="wNumOfBlocks">Number of blocks to read</param>
        /// <param name="aCDIDOut">Received CID</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t InventoryReadExtended ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bExtendedOptions, byte[] aCID,
            byte bBlockNo, ushort wNumOfBlocks, out byte[] aCDIDOut, out byte[] aUid, out byte[] aData )
        {
            byte bUidLen = 0;
            ushort wDataLen = 0;
            aUid = new byte[UID_LENGTH];
            aCDIDOut = new byte[2];
            aData = new byte[256 * BLOCK_LENGTH];

            Status_t oStatus = phpalSli15693_InventoryReadExtended ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bExtendedOptions,
                aCID, bBlockNo, wNumOfBlocks, aCDIDOut, aUid, ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                if ( !bUidLen.Equals ( 0 ) )
                    Array.Resize ( ref aUid, bUidLen );
                else
                    aUid = null;

                if ( !wDataLen.Equals ( 0 ) )
                    Array.Resize ( ref aData, wDataLen );
                else
                    aData = null;

                if ( ( bExtendedOptions & ( byte ) ExtendedFlags.CID_RESPONSE ).Equals ( 0 ) )
                    aCDIDOut = null;
            }
            else
            {
                aUid = null;
                aData = null;
                aCDIDOut = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 FastInventoryRead command.
        ///
        /// When receiving the FAST INVENTORY READ command the ICODE SLI tag behaves the same as the <seealso cref="Generic.InventoryRead"/>
        /// command with the exceptions that the data rate in the direction ICODE SLI tag to the reader is twice than defined in ISO/IEC 15693-3
        /// depending on the Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bBlockNo">Block number of first block to read</param>
        /// <param name="wNumOfBlocks">Number of blocks to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t FastInventoryRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bBlockNo, ushort wNumOfBlocks,
            out byte[] aUid, out byte[] aData )
        {
            aUid = new byte[8];
            aData = new byte[512];
            byte bUidLen = 0;
            ushort wDataLen = 0;

            Status_t oStatus = phpalSli15693_FastInventoryRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bBlockNo,
                wNumOfBlocks, aUid, ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize ( ref aUid, bUidLen );
                Array.Resize ( ref aData, wDataLen );
            }
            else
            {
                aUid = new byte[0];
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ICODE SLI FastInventoryRead command in extended mode.
        ///
        /// When receiving the FAST INVENTORY READ command the ICODE SLIX2\DNA tag behaves the same as the <seealso cref="Generic.InventoryReadExtended"/>
        /// command with the exceptions that the data rate in the direction ICODE SLIX2\DNA tag to the reader is twice than defined in ISO/IEC 15693-3 depending on the
        /// Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
        ///
        /// If the Option_flag is not set, n blocks of data are re-transmitted. If the Option_flag is set, n blocks of data and the part of the UID which
        /// is not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead of padding with zeros up
        /// to the next byte boundary, the corresponding bits of the UID are returned.
        ///
        /// The number of bits of the re-transmitted UID can be calculated as follows:
        ///     16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
        ///     1 slot: 64 bits - mask length rounded up to the next byte boundary.
        /// | **Bit** | **Value** | **Description**                                                                                             |
        /// |:-------:|:---------:|:------------------------------------------------------------------------------------------------------------|
        /// |    b1   |     0     | Label responds independent from the EAS status.                                                             |
        /// |    ^    |     1     | Only labels will respond which have the EAS enabled.                                                        |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_EAS_MODE "EAS"**                           |
        /// |    b2   |     0     | UID will be transmitted as in regular mode.                                                                 |
        /// |    ^    |     1     | Complete UID will be transmitted.                                                                           |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE "UID"**                           |
        /// |    b3   |     0     | No CID is transmitted in the command.                                                                       |
        /// |    ^    |     1     | 16 bits CID will be transmitted in the command and only matching tags should reply.                         |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE "CID In Command"**             |
        /// |    b4   |     0     | CID will not be transmitted in the response from the tag.                                                   |
        /// |    ^    |     1     | CID will be transmitted in the response from the tag.                                                       |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE "CID In Response"**           |
        /// |    b5   |     0     | Tag will add the user memory block in the response as requested.                                            |
        /// |    ^    |     1     | No user memory data is requested from the tag.                                                              |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA "Skip Data"**                    |
        /// | b6 & b7 |     00    | Remain in current state.                                                                                    |
        /// |    ^    |     01    | Go to persistent quiet state after response.                                                                |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_PERSISTENT_QUIET "Persistent Quiet"** |
        /// |    ^    |     10    | Go to quiet state after response.                                                                           |
        /// |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_QUIET "Quiet"**                       |
        /// |    ^    |     11    | Only tags in the PERSISTENT QUIET state will respond to the command.                                        |
        /// |    b8   |     0     | RFU                                                                                                         |
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bExtendedOptions">Extended options</param>
        /// <param name="aCID">Two byte CID -> if marked in extended options</param>
        /// <param name="bBlockNo">Block number of first block to read</param>
        /// <param name="wNumOfBlocks">Number of blocks to read</param>
        /// <param name="aCDIDOut">Received CID</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t FastInventoryReadExtended ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bExtendedOptions,
            byte[] aCID, byte bBlockNo, ushort wNumOfBlocks, out byte[] aCDIDOut, out byte[] aUid, out byte[] aData )
        {
            aUid = new byte[8];
            aCDIDOut = new byte[2];
            aData = new byte[512];
            byte bUidLen = 0;
            ushort wDataLen = 0;

            Status_t oStatus = phpalSli15693_FastInventoryReadExtended ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen,
                bExtendedOptions, aCID, bBlockNo, wNumOfBlocks, aCDIDOut, aUid, ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                if ( 0 != bUidLen )
                {
                    Array.Resize ( ref aUid, bUidLen );
                }
                else
                {
                    aUid = null;
                }

                if ( 0 != wDataLen )
                {
                    Array.Resize ( ref aData, wDataLen );
                }
                else
                {
                    aData = null;
                }

                if ( 0 == ( bExtendedOptions & ( byte ) ExtendedFlags.CID_RESPONSE ) )
                {
                    aCDIDOut = null;
                }
            }
            else
            {
                aUid = null;
                aData = null;
                aCDIDOut = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 InventoryPageRead command.
        ///
        /// When receiving the INVENTORY PAGE READ request, the ICODE SLI tag performs the same as the anti-collision sequence, with the difference that
        /// instead of the UID and the DSFID, the requested memory content is re-transmitted from the ICODE SLI. If an error is detected, the ICODE
        /// SLI tag remains silent.
        ///
        /// If the Option_flag  is not set, n blocks of data are re-transmitted. If the Option_flag is set, n blocks of data and the part of the UID
        /// which is not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead of padding with
        /// zeros up to the next byte boundary, the corresponding bits of the UID are returned.
        ///
        /// The number of bits of the re-transmitted UID can be calculated as follows:
        ///     16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
        ///     1 slot: 64 bits - mask length rounded up to the next byte boundary.
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bPageNo">Block number of first page to read</param>
        /// <param name="wNumOfPages">Number of pages to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t InventoryPageRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bPageNo, ushort wNumOfPages,
            out byte[] aUid, out byte[] aData )
        {
            aUid = new byte[8];
            aData = new byte[512];
            byte bUidLen = 0;
            ushort wDataLen = 0;

            Status_t oStatus = phpalSli15693_InventoryPageRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bPageNo,
                wNumOfPages, aUid, ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize ( ref aUid, bUidLen );
                Array.Resize ( ref aData, wDataLen );
            }
            else
            {
                aUid = new byte[0];
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 FastInventoryPageRead command.
        ///
        /// When receiving the FAST INVENTORY PAGE READ command the ICODE SLI tag behaves the same as the <seealso cref="Generic.InventoryPageRead"/>
        /// command with the exceptions that the data rate in the direction ICODE SLI tag to the reader is twice than defined in ISO/IEC 15693-3
        /// depending on the Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
        /// </summary>
        ///
        /// <param name="bFlags">Request flags byte.</param>
        /// <param name="bAfi">Application Family Identifier.</param>
        /// <param name="aMask">UID mask, holding known UID bits.</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask.</param>
        /// <param name="bPageNo">Block number of first page to read</param>
        /// <param name="wNumOfPages">Number of pages to read</param>
        /// <param name="aUid">Received UID.</param>
        /// <param name="aData">Received data</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t FastInventoryPageRead ( byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, byte bPageNo, ushort wNumOfPages,
            out byte[] aUid, out byte[] aData )
        {
            aUid = new byte[8];
            aData = new byte[512];
            byte bUidLen = 0;
            ushort wDataLen = 0;

            Status_t oStatus = phpalSli15693_FastInventoryPageRead ( m_pDataParams, bFlags, bAfi, aMask, bMaskBitLen, bPageNo,
                wNumOfPages, aUid, ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize ( ref aUid, bUidLen );
                Array.Resize ( ref aData, wDataLen );
            }
            else
            {
                aUid = new byte[0];
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ICODE SLI Stay quite persistent command.
        ///
        /// When receiving the STAY QUIET PERSISTENT command, the tag enters the persistent quiet state and will not send back a response.
        ///
        /// The STAY QUIET PERSISTENT command provides the same behavior as the mandatory <seealso cref="Generic.StayQuiet"/> command
        /// with the only difference at a reset (power off), the tag will turn to the ready state, if the power off time is exceeding
        /// the persistent time. Refer the corresponding ICODE SLI tag specification for the value of persistent time.
        ///
        /// The tag will exit the persistent quiet state when:
        ///     Reset (power off) exceeding the persistent time.
        ///     Receiving a SELECT request. It shall then go to the Selected state.
        ///     Receiving a RESET TO READY request. It shall then go to the Ready state.
        /// </summary>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t StayQuietPersistent ()
        {
            return phpalSli15693_StayQuietPersistent ( m_pDataParams );
        }
        #endregion

        #region Common Commands
        /// <summary>
        /// Perform ISO15693 ActivateCard command.
        ///
        /// This function activates the first VICC detected during the collision resolution and optionally move it to selected state
        /// based on the wOption specified.
        ///
        /// wOption can be one of
        ///     <see cref="OptionsActivate.DEFAULT"/>
        ///     <see cref="OptionsActivate.ADDRESSED"/>
        ///     <see cref="OptionsActivate.SELECTED"/>
        /// </summary>
        ///
        /// <param name="bOptions">Selected or addressed</param>
        /// <param name="bFlags">Request flags byte</param>
        /// <param name="bAfi">Application Family Identifier</param>
        /// <param name="aMask">UID mask, holding known UID bits</param>
        /// <param name="bMaskBitLen">Number of UID bits within pMask</param>
        /// <param name="aDsfid">Data Storage Format Identifier</param>
        /// <param name="aUid">Received UID of first found card</param>
        /// <param name="bMoreCardsAvailable">If there are more cards available this value is different from zero</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t ActivateCard ( OptionsActivate bOptions, byte bFlags, byte bAfi, byte[] aMask, byte bMaskBitLen, out byte aDsfid,
            out byte[] aUid, out byte bMoreCardsAvailable )
        {
            aDsfid = 0x00;
            aUid = new byte[8];
            bMoreCardsAvailable = 0;

            Status_t oStatus = phpalSli15693_ActivateCard ( m_pDataParams, ( byte ) bOptions, bFlags, bAfi, aMask, bMaskBitLen,
                ref aDsfid, aUid, ref bMoreCardsAvailable );

            if ( !oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aUid = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Send a ISO15693 EOF (End of Frame).
        ///
        /// When 16 slots are used in the inventory process, to switch to the next slot, the reader shall send an EOF with option
        /// OptionsEof.NEXT_SLOT.
        ///
        /// To send EOF for write alike commands, #PHPAL_SLI15693_EOF_WRITE_ALIKE or #PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT shall
        /// be used. For ICODE SLI custom commands like Inventory Read, #PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ option shall be used.
        ///
        /// bOption can be one of:
        ///     OptionsEof.NEXT_SLOT
        ///     OptionsEof.NEXT_SLOT_INV_READ
        ///     OptionsEof.WRITE_ALIKE
        ///     OptionsEof.WRITE_ALIKE_WITH_WAIT
        /// </summary>
        ///
        /// <param name="bOption">Option parameter.</param>
        /// <param name="bDsfid">Data Storage Format Identifier.</param>
        /// <param name="aUid">Received UID</param>
        /// <param name="aData">Received data.</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t SendEof ( OptionsEof bOption, out byte bDsfid, out byte[] aUid, out byte[] aData )
        {
            bDsfid = 0x00;
            aUid = new byte[8];
            aData = new byte[512];
            byte bUidLen = 0;
            ushort wDataLen = 0;

            Status_t oStatus = phpalSli15693_SendEof ( m_pDataParams, (byte) bOption, ref bDsfid, aUid,
                ref bUidLen, aData, ref wDataLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize ( ref aUid, bUidLen );
                Array.Resize ( ref aData, wDataLen );
            }
            else
            {
                aUid = new byte[0];
                aData = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 Data Exchange with VICC.
        ///
        /// wOption can be one of:
        ///     <see cref="ExchangeOptions.DEFAULT"/>
        ///     <see cref="ExchangeOptions.BUFFER_FIRST"/>
        ///     <see cref="ExchangeOptions.BUFFER_CONT"/>
        ///     <see cref="ExchangeOptions.BUFFER_LAST"/>
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.</param>
        /// <param name="aTxBuffer">Data to transmit. Flags, IC MFC code and UID will be added automatically.</param>
        /// <param name="aRxBuffer">Pointer to received data.</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t Exchange ( int wOption, byte[] aTxBuffer, out byte[] aRxBuffer )
        {
            ushort wTxLen = 0;
            ushort wRxLen = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            if ( aTxBuffer != null )
            {
                wTxLen = ( ushort ) aTxBuffer.Length;
            }

            Status_t oStatus = phpalSli15693_Exchange ( m_pDataParams, (ushort) wOption, aTxBuffer, wTxLen,
                ref pRxBufferInt, ref wRxLen );

            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) ) && ( wRxLen > 0 ) )
            {
                aRxBuffer = new byte[wRxLen];
                Marshal.Copy ( pRxBufferInt, aRxBuffer, 0, wRxLen );
            }
            else
            {
                aRxBuffer = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISO15693 Data Exchange with VICC with dual response for a single command.
        ///
        /// \note:
        ///     - The implementation is based on ISO/IEC 15693-3:2019, VICC in-reply process  Asynchronous mode, section 9.8.3.3
        ///     - The response is given out as is received from the VICC removing the Flags in case of SUCCESS or FAILURE.
        ///     - The actual error code received from VICC can be retrieved using <see cref="SetConfig"/>
        ///       with <see cref="Config.ADD_INFO"/> as configuration identifier.
        ///     - Validation of the response is not carried out by this interface.
        /// </summary>
        ///
        /// <param name="aTxBuffer">Data to transmit. Flags, IC MFC code and UID will be added automatically.</param>
        /// <param name="aResponse1">First Information received from VICC.</param>
        /// <param name="aResponse2">Second Information received from VICC.</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> error for invalid response.
        /// Returns <see cref="Error.IN_PROCESS_REPLY_PART1_ERROR"/> if there is an error from PArt 1 response.
        /// Returns <see cref="Error.IN_PROCESS_REPLY_PART2_ERROR"/> if there is an error from PArt 2 response.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t InProcessReply_ASync ( byte[] aTxBuffer, out byte[] aResponse1, out byte[] aResponse2 )
        {
            ushort wReply1_Len = 0;
            ushort wReply2_Len = 0;

            aResponse1 = new byte[2048];
            aResponse2 = new byte[2048];

            Status_t oStatus =  phpalSli15693_InProcessReplyASync ( m_pDataParams, aTxBuffer,
                ( ushort ) ( ( aTxBuffer == null ) ? 0 : aTxBuffer.Length ), aResponse1,
                ref wReply1_Len, aResponse2, ref wReply2_Len );

            if ( wReply1_Len > 0 )
                Array.Resize ( ref aResponse1, wReply1_Len );
            else
                aResponse1 = null;

            if ( wReply2_Len > 0 )
                Array.Resize ( ref aResponse2, wReply2_Len );
            else
                aResponse2 = null;

            return oStatus;
        }
        #endregion

        #region Utility Commands
        /// <summary>
        /// Retrieve the serial number (UID).
        /// This API is used to get the UID of current VICC with which this layer is communicating.
        /// </summary>
        ///
        /// <param name="aUid">Buffer, holding the UID (8 bytes)</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Comm.USE_CONDITION"/> if no UID is available.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetSerialNo ( out byte[] aUid )
        {
            aUid = new byte[8];
            byte bUidLen = 0;

            Status_t oStatus = phpalSli15693_GetSerialNo ( m_pDataParams, aUid, ref bUidLen );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize<byte> ( ref aUid, bUidLen );
            }
            else
            {
                aUid = new byte[0];
            }

            return oStatus;
        }

        /// <summary>
        /// Perform the SetSerialNo command.
        /// </summary>
        ///
        /// <param name="aUid">Buffer, holding the UID (8 bytes)</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t SetSerialNo ( byte[] aUid )
        {
            return phpalSli15693_SetSerialNo ( m_pDataParams, aUid, ( byte ) ( ( aUid != null ) ? aUid.Length : 0 ) );
        }

        /// <summary>
        /// Set value of the specified Configuration Identifier.
        /// The SetConfig command is used to set values for the different configuration parameters of this layer.
        ///
        /// NOTE: Both the flags <seealso cref="Flags.DATA_RATE"/> and
        /// <seealso cref="ReqFlagsNoInv.FAST_DATA_RATE"/> or
        /// <seealso cref="ReqFlagsInv.FAST_DATA_RATE"/> should not be
        /// combined, it should be passed separately along with other flag.
        ///
        /// wConfig can be one of:
        ///     <see cref="Config.FLAGS"/>
        ///     <see cref="Config.ADD_INFO"/>
        ///     <see cref="Config.TIMEOUT_US"/>
        ///     <see cref="Config.TIMEOUT_MS"/>
        ///     <see cref="Config.TXDATARATE"/>
        ///     <see cref="Config.RXDATARATE"/>
        ///     <see cref="Config.T1_PARAMETER"/>
        ///     <see cref="Config.ENABLE_BUFFERING"/>
        ///     <see cref="Config.MAX_RETRYCOUNT"/>
        ///     <see cref="Config.IC_MFGCODE"/>
        ///     <see cref="Config.TRANSPARENT"/>
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to configure.</param>
        /// <param name="wValue">Value to set.</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> for invalid configuration identifier
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t SetConfig ( int wConfig, int wValue )
        {
            return phpalSli15693_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        /// <summary>
        /// Get value of the specified Configuration Identifier.
        /// The GetConfig command is used to get values for the different configuration parameters of this layer.
        ///
        /// wConfig can be one of:
        ///     <see cref="Config.FLAGS"/>
        ///     <see cref="Config.ADD_INFO"/>
        ///     <see cref="Config.TIMEOUT_US"/>
        ///     <see cref="Config.TIMEOUT_MS"/>
        ///     <see cref="Config.TXDATARATE"/>
        ///     <see cref="Config.RXDATARATE"/>
        ///     <see cref="Config.T1_PARAMETER"/>
        ///     <see cref="Config.ENABLE_BUFFERING"/>
        ///     <see cref="Config.MAX_RETRYCOUNT"/>
        ///     <see cref="Config.IC_MFGCODE"/>
        ///     <see cref="Config.TRANSPARENT"/>
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read.</param>
        /// <param name="wValue">Read value</param>
        ///
        /// <returns>
        /// Returns <see cref="Error_Gen.SUCCESS"/> status for successful operation.
        /// Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        /// Returns <see cref="Error_Param.UNSUPPORTED_PARAMETER"/> for invalid configuration identifier
        /// Other Depending on implementation and underlaying component.
        /// </returns>
        public Status_t GetConfig ( int wConfig, out int wValue )
        {
            ushort wValueTmp = 0;

            Status_t oStatus = phpalSli15693_GetConfig ( m_pDataParams, (ushort) wConfig, ref wValueTmp );
            wValue = wValueTmp;

            return oStatus;
        }
        #endregion
        #endregion DLL Wrapper

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion Memory Mapping
    }
    #endregion Generic

    #region Software
    /// <summary>
    /// Software Iso15693 PAL component of Reader Library .Net framework.
    /// </summary>
    public class Sw : palSli15693.Generic
    {
        #region Data Structure
        /// <summary>
        /// Software DataParams of PalSLI15693 component.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// ID for this component, not to be modified by user. This is used for validating the data parameter structure
            /// and to make sure that APIs are called after initialization.
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the HAL Rd710 data parameter. This pointer is initialized with the address of Rd710 HAL data
            /// parameter structure by <seealso cref="Sw.Init"/> .
            /// </summary>
            public IntPtr pHalDataParams;

            /// <summary>
            /// Holds the last error code sent by the VICC. But for Authentication Scenario contains the Flag Byte[non-error]
            /// as well. When a response is received from the VICC with the error flag <seealso cref="RespFlags.ERROR"/>
            /// set, the error code returned by VICC is stored here.
            /// </summary>
            public ushort wAdditionalInfo;

            /// <summary>
            /// This holds the value of <seealso cref="ReqFlags"/> field used in the commands sent by the library.
            /// </summary>
            public byte bFlags;

            /// <summary>
            /// This holds the value of <seealso cref="RespFlags"/> field used in the commands sent by the library
            /// </summary>
            public byte bResFlags;

            /// <summary>
            /// The VICC's Unique Identifier, which always is of 8 bytes.
            /// </summary>
            public fixed byte bUid[UID_LENGTH];

            /// <summary>
            /// The current length of the UID in bits.
            /// </summary>
            public byte bUidBitLen;

            /// <summary>
            /// Indicator of API to be in explicit Addressed mode
            ///     - This indicates that the API is explicitly asked to be used in addressed mode.
            ///     - This parameter differentiate the understanding of explicitly addressed mode from the APIs like
            ///       <see cref="Generic.Select"/> and <see cref="Generic.ResetToReady"/> which are implicitly addressed
            /// </summary>
            public byte bExplicitlyAddressed;

            /// <summary>
            /// Operation Mode. Will be one of the following
            ///     - \arg RD_LIB_MODE_NFC "NFC"
            ///     - \arg RD_LIB_MODE_EMVCO "EMVCo"
            ///     - \arg RD_LIB_MODE_ISO "ISO"
            /// </summary>
            public byte bOpeMode;

            /// <summary>
            /// Enable or disable the data buffering. Supported values are
            ///     - <seealso cref="Value.OFF"/>
            ///     - <seealso cref="Value.ON"/>
            /// </summary>
            public byte bBuffering;

            /// <summary> Set / Get Maximum Retry Count </summary>
            public byte bMaxRetryCount;

            /// <summary> Set / Get IC Manufacturer Code </summary>
            public byte bIcMfgCode;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
        /// This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "configuration options" for supported configurations
        /// and their default values.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to phhalHw_Rd710_DataParams_t parameter structure.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        ///          </returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalSli15693_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
        /// This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "configuration options" for supported configurations
        /// and their default values.
        /// </summary>
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to phhalHw_Rd710_DataParams_t parameter structure.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalSli15693_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        /// <summary>
        /// Layers structure instance.
        /// </summary>
        private DataParams_t[] m_DataParamsInt;

        #region Constructor
        /// <summary>
        /// Constructor of this class.
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }
        #endregion Constructor

        #region Destructor
        /// <summary>
        /// Destructor of this class.
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }
        #endregion Destructor

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Software

    #region Rd71x
    /// <summary>
    /// Rd71x Iso15693 PAL component of Reader Library .Net framework.
    /// </summary>
    public class Rd71x : palSli15693.Generic
    {
        #region Data Structure
        /// <summary>
        /// Rd710 DataParams of PalSLI15693 component.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// ID for this component, not to be modified by user. This is used for validating the data parameter structure
            /// and to make sure that APIs are called after initialization.
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the HAL Rd710 data parameter. This pointer is initialized with the address of Rd710 HAL data
            /// parameter structure by <seealso cref="Rd71x.Init"/> .
            /// </summary>
            public IntPtr pHalDataParams;

            /// <summary>
            /// Holds the last error code sent by the VICC. But for Authentication Scenario contains the Flag Byte[non-error]
            /// as well. When a response is received from the VICC with the error flag <seealso cref="RespFlags.ERROR"/>
            /// set, the error code returned by VICC is stored here.
            /// </summary>
            public ushort wAdditionalInfo;

            /// <summary>
            /// This holds the value of <seealso cref="ReqFlags"/> field used in the commands sent by the library.
            /// </summary>
            public byte bFlags;

            /// <summary>
            /// The VICC's Unique Identifier, which always is of 8 bytes.
            /// </summary>
            public fixed byte bUid[UID_LENGTH];

            /// <summary>
            /// The current length of the UID in bits.
            /// </summary>
            public byte bUidBitLen;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalSli15693_Rd710_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
        /// This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "configuration options" for supported configurations
        /// and their default values.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to phhalHw_Rd710_DataParams_t parameter structure.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        ///          </returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalSli15693_Rd710_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
        /// This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "configuration options" for supported configurations
        /// and their default values.
        /// </summary>
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to phhalHw_Rd710_DataParams_t parameter structure.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalSli15693_Rd710_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif
        #endregion Initialization

        #region Memory Mapping
        /// <summary>
        /// Layers structure instance.
        /// </summary>
        private DataParams_t[] m_DataParamsInt;

        #region Constructor
        /// <summary>
        /// Constructor of this class.
        /// </summary>
        public Rd71x ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }
        #endregion Constructor

        #region Destructor
        /// <summary>
        /// Destructor of this class.
        /// </summary>
        ~Rd71x ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }
        #endregion Destructor

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping
    }
    #endregion Rd71x
}
