/*
 * Copyright 2014, 2021-2022, 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 * NXP USBSIO Library: USB serial I/O test application
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <wchar.h>
#include <string.h>
#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#include <pthread.h>
#define sscanf_s sscanf
#endif

#include "lpcusbsio.h"

extern int RunI2CTest(LPC_HANDLE hSIOPort);
extern int RunSPITest(LPC_HANDLE hSIOPort);
extern int RunGPIOTest(LPC_HANDLE hSIOPort);
extern int RunI2CRWTest(LPC_HANDLE hSIOPort);
extern int RunI2CDataTest(LPC_HANDLE hSIOPort);

/*****************************************************************************
 * Private functions
 ****************************************************************************/

static void print_menu(void)
{
    printf("\n");
    printf("Press '1' to run I2C Transfer test \n");
    printf("Press '2' to run SPI Transfer test \n");
    printf("Press '3' to run GPIO test \n");
    printf("Press '4' to run I2C Read Write test \n");
    printf("Press '5' to run I2C Large data transfer test \n");
    printf("Press 'q' to exit \n");
}

/* use our own wchar_t puts routine to avoid mixing of char/wchar_t in stdout */
static void wputs(wchar_t* p)
{
    wchar_t c;

    if (p)
    {
        while ((c = *p++) != 0)
            putc(c >> 8 ? '?' : c, stdout);
    }
}

void usage()
{
    printf("LIBUSBSIO Test application.\n");
    printf("Copyright 2014, 2021-2025 NXP\n\n");
    printf("Usage: \n");
    printf("  testapp [INDEX]   ... open USBSIO device by index; default=0\n");
    printf("  testapp -p PATH   ... open USBSIO device by USB path\n");
    printf("  testapp -s SERIAL ... open USBSIO device by serial number\n");
    printf("  testapp -l        ... print USBSIO device list\n");
    printf("  testapp -h        ... print this help\n");
}

void device_list(int dev_count)
{
    HIDAPI_DEVICE_INFO_T info;
    int i;

    for (i = 0; i < dev_count; i++)
    {
        memset(&info, 0, sizeof(info));
        if (LPCUSBSIO_GetDeviceInfo(i, &info) == LPCUSBSIO_OK)
        {
            printf("Device #%d at path '%s':\n", i, info.path);
            printf(" -manufacturer: ");
            wputs(info.manufacturer_string);
            printf("\n -product string: ");
            wputs(info.product_string);
            printf("\n -serial number: ");
            wputs(info.serial_number);
            printf("\n\n");
        }
    }
}

static int enumerate_devices()
{
    int res;

#if 0
    if ((res = LPCUSBSIO_GetNumPorts(LPCUSBSIO_VID, 0)) > 0)
    {
        printf("Total MCULink/LPCLink2 devices: %d\n", res);
    }
#else
    if ((res = LPCUSBSIO_GetNumPorts(LPCUSBSIO_VID, LPCUSBSIO_PID)) > 0)
    {
        printf("Total LPCLink2 devices: %d\n", res);
    }
    else if ((res = LPCUSBSIO_GetNumPorts(LPCUSBSIO_VID, MCULINKSIO_PID)) > 0)
    {
        printf("Total MCULink devices: %d\n", res);
    }
#endif
    else
    {
        printf("Error: No free USBSIO devices found.\n");
        res = 0;
    }

    return res;
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

int main(int argc, char *argv[])
{
    LPC_HANDLE hSIOPort = NULL;
    int dev_count = 0;
    int res = 0;
    char ch;

    const char* arg = argc > 1 ? argv[1] : "0";
    char option = *arg == '-' ? arg[1] : 0;

    // enumerate all USBSIO devices
    if (option != 'h')
        dev_count = enumerate_devices();

    switch (option)
    {
    // no option, parse index
    case 0:
    {
        int index = 0;
        if (sscanf_s(arg, "%i", &index) != 1)
            index = 0;
        printf("Opening device at index #%d\n", index);
        if(index >= 0 && index < dev_count)
            hSIOPort = LPCUSBSIO_Open(index);
        break;
    }

    case 'p':
    {
        char* path = argc >= 2 ? argv[2] : "";
        printf("Opening device at path '%s'\n", path);
        hSIOPort = LPCUSBSIO_OpenByPath(path);
        break;
    }

    case 's':
    {
        char* serial = argc >= 2 ? argv[2] : "";
        int len = (int)strlen(serial);
        wchar_t* wserial = (wchar_t*)malloc((len + 1) * sizeof(wchar_t));
        if (wserial != NULL)
        {
            int i;
            for (i = 0; i < len; i++)
                wserial[i] = serial[i];
            wserial[len] = 0;
            printf("Opening device with serial number '%s'\n", serial);
            hSIOPort = LPCUSBSIO_OpenBySerial(wserial);
            free(wserial);
        }
        break;
    }

    case 'l':
        device_list(dev_count);
        return 0;

    case 'h':
    default:
        usage();
        return 0;
    }

    if (hSIOPort != NULL)
    {
        printf("Device version: %s \n ", LPCUSBSIO_GetVersion(hSIOPort));

        printf("\nTestApp options menu:  \n");
        print_menu();

        while ((ch = getchar()) != 'q')
        {
            switch (ch)
            {
            case '1':
                res = RunI2CTest(hSIOPort);
                break;
            case '2':
                res = RunSPITest(hSIOPort);
                break;
            case '3':
                res = RunGPIOTest(hSIOPort);
                break;
            case '4':
                res = RunI2CRWTest(hSIOPort);
                break;
            case '5':
                res = RunI2CDataTest(hSIOPort);
                break;
            default:
                continue;
            }
            if (res == LPCUSBSIO_ERR_HID_LIB)
            {
                printf("HID Library Error, exiting...\n");
                LPCUSBSIO_Close(hSIOPort);
                return 1;
            }

            print_menu();
        }
        printf("Exiting \n");
        LPCUSBSIO_Close(hSIOPort);
    }
    else
    {
        /* This couold be an issue of /dev/hidrawX access rights */
        printf("Could not open USBSIO device\n");
        return 1;
    }

    return 0;
}
