// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2014-2016 NXP Semiconductors
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// NXP Confidential
// -----------------------------------------------------------------------------
// FILE NAME:           bcsrs.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@nxp.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR                  DESCRIPTION
// 1.0      2018-0703   Gary Milliorn           Extracted from <ifc.v>
// -----------------------------------------------------------------------------
// KEYWORDS :           IFC BCSRS REGS CONFIG
// -----------------------------------------------------------------------------
// PURPOSE :            QixMin implementation of the BCSRs (registers) sufficient
//                      to boot u-boot/Linux.
// -----------------------------------------------------------------------------
// PARAMETERS
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:   asynchronous, active low
// Clock Domains:        clk:     system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module bcsrs( 

    // IFC or I2C interface
    //
    input   wire    [7:0]   reg_addr,
    input   wire            reg_cs_b,
    input   wire            reg_we_b,
    output  reg     [7:0]   reg_rdata,
    input   wire    [7:0]   reg_wdata,

    // Switch inputs
    //
    input   wire    [1:8]   sw1, sw2, sw3, sw4, sw5,
    output  wire            sw_auto_on,
    output  wire            sw_bypass_b,
    output  wire	[2:0]	sw_debug,

    // Reset monitoring.
    //
    input   wire            int_poreset_b,   
    input   wire            int_hreset_b,   
    input   wire            int_reset_req_b, 
    input   wire            int_asleep,
    input   wire    [7:0]   rst_reason,

	// DUT config outputs
	//
    input   wire            cfg_drv_b,
    output	wire    [15:0]  cfg_ifc_ad,
    output	wire            cfg_ifc_a5,
    output	wire            cfg_ifc_a1,
    output	wire            cfg_ifc_a0,

    output	wire            cfg_ifc_cle,
    output	wire            cfg_ifc_oe_b,
    output	wire            cfg_ifc_te,
    output	wire            cfg_ifc_we0_b,
    output	wire            cfg_ifc_wp_b,

	output	wire			cfg_rsp_dis,	

    // BCSR outputs
    //
    output  wire            cfg_cfg_wp,
    output  wire            cfg_ieee_clk,
    output  wire            cfg_mux_i2c2, 
    output  wire            cfg_mux_i2c3, 
    output  wire            cfg_mux_sd,
    output  wire            cfg_mux_spi,
	output	wire			cfg_pci1_wgig_en,
	output	wire			cfg_pci1_dis_b,
	output	wire			cfg_pci2_wgig_en,
	output	wire			cfg_pci2_dis_b,
    output  wire    [4:0]   cfg_qspi_map, 
    output  wire            cfg_rstreq_md,
    output  wire            cfg_rst_noddr,
    output  wire            cfg_sfp_txen_b,
    output  wire            cfg_spread,
    output  wire            cfg_svdd_sel,
    output  wire            cfg_tabb_sel,
    output  wire            cfg_testsel_b,

    output  wire            int_cfg_mux_sd,
    output  wire            cfg_evdd_force,
    output  wire            dut_tbscan_en_b,

    // Fault, Power. reporting
    //
    input   wire    [7:0]   alarm,
	output	reg		[7:0]	alarm_mask,
    output  wire            ctl_led,
    output  wire            ctl_xtest,
    output  wire            ctl_passfail,
    output  wire	[1:0]	ctl_pwrled,
	
    input   wire    [7:0]   irq_stat0,
    input   wire    [7:0]   irq_stat1,

    output  wire    [3:0]   leds_user,
    input   wire    [7:0]   pwr_reason,
    input   wire    [7:0]   pwr_mstat,
    input   wire    [7:0]   pwr_stat1,
    input   wire    [7:0]   pwr_stat2,
	output	reg 			pwr_ctl2_pwr,				// PWRCTL2[PWR]

    // Reset forcing
    //
    output  wire            rst_rst, 
    output  reg     [7:0]   rst_force1,
    output  reg     [7:0]   rst_force2,
    output  reg     [7:0]   rst_force3,
    output  reg     [7:0]   rst_mask1,
    output  reg     [7:0]   rst_mask2,
    output  reg     [7:0]   rst_mask3,

    // Config Controls
    //
    input   wire            rst_cfg_regs_b, 
    input   wire            rst_ctl_regs_b, 
    output  wire            rcfg_go,
    output  wire            rcfg_wden,
    output  reg     [7:0]   rcfg_watch,

    // Misc
    //
    input   wire            present_proc_b,
    input   wire            present_slot3_b,
    input   wire            present_tdm_b,
    input   wire    [1:0]   cpu_id,
	input	wire			brd_id,
    input   wire            los_retimer_b,
    input   wire            retimer_done_b,
    input   wire            sdslot_cd_b,
    input   wire            sdslot_wp,
    input   wire            sfp_tx_fault,
    input   wire            sfp_rx_los,
    input   wire    [2:0]   pcb_rev, 


    // General
    //
    input   wire            disable_io_b, 
    input   wire            rst_b, 
    input   wire            clk
);


//===========================================================================
// Include standardized definitions and values.
//
`include <qixis_defs.h>


//===========================================================================
//                          FORWARD DECLARE
//===========================================================================
//
	wire	[7:0]	brdcfg1;					// For CLKSPD1
    reg		[7:0]   brdcfg4;					// For SSCG config
    reg		[7:0]   cms_addr;					// For SW management
	wire			cms_cs_b,	cms_we_b;		// "
	reg		[7:0]	sws_ctl;					// "
	wire			sws_en;						// "


//===========================================================================
//                          QIXMIN TAGROM DATA
//===========================================================================

`include "../hdl/build.v"

	wire	[31:0]	build;
	assign			build	= BUILD_DATE;

	localparam		RELEASED	= 1;


//===========================================================================
//                          MONITORING
//===========================================================================
// Count register accesses (RAC) and writes (RWC) to the lower half (i.e. 
// not CMSA/CMSD).
// This is mostly to help track down errant I2C access from Uboot.
// All counters are reset on writes to RACC.

	reg		[7:0]	rac,		rwc;
	wire			rac_we_b,	stat_clr_b;

	assign	rac_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'hE0)) ? 1'b0 : 1'b1;
	assign	stat_clr_b	= (rst_b  &  rac_we_b);

	always @(negedge stat_clr_b  or  posedge reg_cs_b)
		if (!stat_clr_b)		rac	<= 8'd0;
		else if (!reg_addr[7])	rac	<= rac + 8'd1;

	always @(negedge stat_clr_b  or  posedge reg_we_b)
		if (!stat_clr_b)		rwc	<= 8'd0;
		else if (!reg_addr[7])	rwc	<= rwc + 8'd1;


//===========================================================================
//                          SWITCHES
//===========================================================================
// The CPLD has pullup options only, so when a SW is ON, it shorts a pullup to
// GND.  To keep the UP=ON=1 standard, invert them here. The value is
// continually updated in the register, unless disabled via SWSCTL.
//
    reg     [1:8]   isw1,		isw2,		isw3,		isw4,		isw5;
    wire			isw1_we_b,	isw2_we_b,	isw3_we_b,	isw4_we_b,	isw5_we_b;

	assign			isw1_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'd1)) ? 1'b0 : 1'b1;
	assign			isw2_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'd2)) ? 1'b0 : 1'b1;
	assign			isw3_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'd3)) ? 1'b0 : 1'b1;
	assign			isw4_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'd4)) ? 1'b0 : 1'b1;
	assign			isw5_we_b	= (!cms_cs_b  &&  !cms_we_b &&  (cms_addr == 8'd5)) ? 1'b0 : 1'b1;

	always @(posedge clk)
		if (sws_en)				isw1	<= sw1 ^ 8'hFF;
		else if (!isw1_we_b)	isw1	<= reg_wdata;

	always @(posedge clk)
		if (sws_en)				isw2	<= sw2 ^ 8'hFF;
		else if (!isw2_we_b)	isw2	<= reg_wdata;

	always @(posedge clk)
		if (sws_en)				isw3	<= sw3 ^ 8'hFF;
		else if (!isw3_we_b)	isw3	<= reg_wdata;

	always @(posedge clk)
		if (sws_en)				isw4	<= sw4 ^ 8'hFF;
		else if (!isw4_we_b)	isw4	<= reg_wdata;

	always @(posedge clk)
		if (sws_en)				isw5	<= sw5 ^ 8'hFF;
		else if (!isw5_we_b)	isw5	<= reg_wdata;

// Now map "isw" fields as needed.
//
    wire    [8:0]   sw_rcw_src;
    wire            sw_rst_mode,	sw_testsel_b, sw_tbscan_en;
    wire            sw_spread,		sw_cfg_wp,    sw_rspdis;
    wire            sw_svdd_sel;
    wire    [2:0]   sw_qspi_map;
    wire    [1:0]   sw_gpin,		sw_sdmode;
    wire    [0:1]   sw_svr;
    wire    [0:2]   sw_enguse;

    assign  sw_rcw_src      = { isw1, isw2[1] };

    assign  sw_rst_mode     =   isw2[  2];
	assign	sw_sdmode		=	isw2[3:4];
                            //         5 unused
    assign  sw_qspi_map     =   isw2[6:8];

    assign  sw_testsel_b    =   isw3[  1];
    assign  sw_svr[0:1]     =   isw3[2:3];
    assign  sw_svdd_sel     =   isw3[  4];
    assign  sw_spread       =   isw3[  5];
                            //	     6:8 unused

    assign  sw_bypass_b     =   isw4[  1];
    assign  sw_auto_on      =   isw4[  2];
                            //	       3 unused			(I2CRCW_WP but HW does
							//							not wire up seperate from CFG)
    assign  sw_cfg_wp       =   isw4[  4];
                            //       5:6 unused
                            //         7 unused
    assign  sw_tbscan_en    =   isw4[  8];

    assign  sw_enguse       =   isw5[1:3];
    assign  sw_rspdis       =   isw5[  4];
                            //       5:6 unused
    assign  sw_gpin         =   isw5[7:8];

	assign	sw_debug		=	{ isw5[6:8] };

    

// Useless in the CPLD, no need for SW programming, etc. - just drive it.
//
    assign  dut_tbscan_en_b = (!disable_io_b) ? 1'bZ 
                            :                   sw_tbscan_en;


//===========================================================================
// QGEN:    Register documentation is embeded before each register definition.
//          Documentation and reference material is generated with the QGEN
//          utility.
//: <qgen	class=RDB	endian=little	toc=0x0FF />

// Enable ONE of -PA or -PB; defaults is -PA
//
//: <if LS1088ARDB-PB>
//: <else>
//:   <define LS1088ARDB-PA>
//: </if>

//: <sect level=1  title="Programming Model"         template="sect_intro_rdb" >
//===========================================================================


//===========================================================================
//                          IDENTIFICATION SECTION
//===========================================================================
//: <sect level=2  title="Identification Registers"  template=sect_idregs>

//---------------------------------------------------------------------------
//: <reg name=ID  addr=0x000  access=RO  reset=none  title=Identification  template="reg_id">
//:   <bit name=ID  size=8  access=RO  reset=0x39>
//:     The board-specific identifier for the system.
//:     0x39= LS1088ARDB(-PA)
//:     0x4A= LS1088ARDB-PB
//:   </bit>
//: </reg>

    wire    [7:0]   id;

	assign id		= (brd_id == 1'b1)	?	8'h39				// LS1088ARDB
					:						8'h4A;				// LS1088ARDB-PB


//---------------------------------------------------------------------------
//: <reg name=VER  addr=0x001  access=RO  reset=none  title="Board Version"  template="reg_ver">
//: </reg>

//      HW BITS		REGBITS		DESCRIPTION
//        000		  000		PCB Rev A
//        001		  001		"       B
//        etc.

    wire    [7:0]   ver;

	assign	ver		= { (brd_id) ? 4'd2			// Architecture: V1
							     : 4'd1,
                        1'b0,
                        pcb_rev + 3'd0          // PCB Version: PCB-controlled
												// Note: older method, don't propagate!
                      };


//---------------------------------------------------------------------------
//: <reg name=QVER  addr=0x002  access=RO  reset=none  title="Qixis Version"  template="reg_qver_rdb">
//: </reg>

//      V1:  PCB Rev A version.
//		V2:	 -PB support version and later.
//		V3:  Fix ALT_HOT_CLK clamping; initial AUTO_ON override; import pwrseq
//			 improvements.
//		V4:	 Fix rapid power-cycling issue.
//		V5:	 -PB Rev B: auto-on mods; restore RST_FORCE2[0]

    wire    [7:0]   qver;

	assign qver		= 8'd5;                     // Version #


//---------------------------------------------------------------------------
//: <reg name=MODEL  addr=0x003  access=RO  reset=none  title="Programming Model"  template="reg_model"></reg>
//
    wire    [7:0]   model;

	assign model	= { 1'b1,                   // 1= minimal Qixis
                        7'd3
                      };


//---------------------------------------------------------------------------
//: <reg name=MINOR  addr=0x004  access=RW  reset=none  title="Minor Revision" template="reg_mintag">
//: </reg>

    reg     [7:0]   minor;
	wire	[7:0]	minor_val;
    reg		[4:0]   mreg;
    wire			mreg_we_b;

// Minor value:
//      V5:  RUN->ASLEEP
//      V6:  Rev B support
//      V7:  Dual IFC/I2C; TA_BB fixes; AUTO SD/EMMC
//      V8:  AUTO_ON always; ROT_ERR disable; RST_CTL bit incorrect; PRELOAD
//           on power-down.
//      V9:  Move SVDD per HW spec.
//      V10: Fix NOR reset release.
//      V11: Fix wrong polarity of FAIL LED.
//      V12: Fix pinswap of DUT_CD_B/DUT_WP.
//		VX.0:Switch to dot-less numbering, see QVER above.
//
    assign minor_val	= 8'd0;


// 5b index to select MINOR vs. BUILD.  Upper 4b must be zero, capture as
// 0 or 1.

	assign mreg_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_QTAG)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge int_poreset_b)
		if (!int_poreset_b)			mreg <= 4'd5;
		else if (!mreg_we_b)		mreg <= { (|reg_wdata[7:4]), reg_wdata[3:0] };

	always @*
		case (mreg)			    // synthesis parallel_case
        5'h00:      minor <= minor_val;						// TAG[0]   : not implemented
        5'h05:      minor <= minor_val;						// minor.lsb:     implemented (LE field)
        5'h06:      minor <= 8'h00;							// minor.msb:     implemented (LE field)
        5'h08:      minor <= build[31:24];					// date		:     implemented (BE field)
        5'h09:      minor <= build[23:16];					// "
        5'h0A:      minor <= build[15: 8];					// "
        5'h0B:      minor <= build[ 7: 0];					// "
        5'h0C:      minor <= (RELEASED) ? 8'h01 : 8'b00;	// RELEASE	:     implemented.
		5'h10:		minor <= 8'h00;							// NAME		:	  implemented as empty string.
        default:    minor <= 8'hFF;
		endcase

//: </sect end_id>


//================================================================================
//                                  CSR SECTION
//================================================================================
//: <sect level=2  title="Control and Status Registers"  template=sect_csrregs>

//---------------------------------------------------------------------------
//: <reg name=CTL   addr=0x005  access=RW  reset=CRST  title="General Control">
//:   <bit name=SWLED     size=2  access="RW"   reset=00>
//:     Controls front-panel power-switch LEDs: when CTL.LED is set
//:   </bit>
//:   <bit name=AMASK     size=1  access="RW"  reset=0>
//:     Alarm Mask Mode:
//:     0: ALARM register reads return alarm status (normal).
//:     1: ALARM register reads return the alarm mask.
//:   </bit>
//:   <bit name=XTEST     size=1  access="RW"   reset=0>
//:     This bit directly drives the XTEST signal, typically driving an SMA\
//:     connector. The function is user-defined.
//:   </bit>
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//:   <bit name=LED       size=1  access="RW"   reset=0>
//:     Software Diagnostic LED Enable:
//:     0= Diagnostic LEDs operate normally.
//:     1= Software can directly control the M3:M0 monitoring LEDs using the LED register.
//:   </bit>
//:   <bit name=FAIL      size=1  access="RW"   reset=0>
//:     Software Failure Diagnostic LED:
//:     0= FAIL LED is not asserted due to software (it might still be on due to hardware failures).
//:     1= FAIL LED is forced on. Generally, this indicates a software-diagnosed error.
//:   </bit>
//: </reg>
//
    reg [7:0]   ctl;
    wire        ctl_we_b;
	wire		ctl_amask;

	assign	    ctl_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_CTL)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)		ctl <= 8'h00;
		else if (!ctl_we_b)			ctl <= reg_wdata;

	assign  ctl_pwrled      = ctl[7:6];
    assign  ctl_amask       = ctl[ 5 ];
    assign  ctl_xtest       = ctl[ 4 ];
							//	  3:2
    assign  ctl_led         = ctl[ 1 ];
    assign  ctl_passfail    = ctl[ 0 ];


//---------------------------------------------------------------------------
//: <reg name=AUX   addr=0x006  access=RW  reset=ARST  title="Auxiliary"  template="reg_aux">
//: </reg>
//
    reg [7:0]   aux;
    wire        aux_we_b;

	assign	aux_we_b	 = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_AUX))  ? 1'b0 : 1'b1;

   	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)				aux <= 8'h00;
		else if (!aux_we_b)		aux <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=STAT_SYS  raddr=0x009  access="RO"  reset=NONE  title="System Status">
//:   The STAT_SYS register reports general system status.
//:   <bit name=-         size=4  access="RO"   reset=0000></bit>
//:   <bit name=ALTCPU    size=1  access="RO"   reset=0>
//:     Alternate CPU Installed:
//:     0= The standard DUT is installed.
//:     1= An alternate, pin-compatible, DUT has been installed.
//:   </bit>
//:   <bit name=ALTERED   size=1  access="RO"   reset=0>
//:     Reconfiguration Active:
//:     0= The system has been configured as normal.
//:     1= The system has been reconfigured by software.
//:   </bit>
//:   <bit name=ASLEEP    size=1  access="RO"   reset=0>
//:     ASLEEP Reporting:
//:     0= At least one core is actively operating.
//:     1= All cores are in sleep mode.
//:   </bit>
//:   <bit name=-         size=1  access="RO"   reset=0></bit>
//: </reg>
//
    wire [7:0]  stat_sys;

    assign  stat_sys    = {  4'b0000,
                             (cpu_id == 2'b11) ? 1'b0 : 1'b1,   // ALTCPU: set if not '11' (LS1088)
                             rcfg_go,
                             int_asleep,
                             1'b0,              // TEST: always off
                          };


//---------------------------------------------------------------------------
//: <reg name=ALARM     raddr=0x00A  access="RW1C"  reset=ARST  title="Alarm">
//:   The STAT_ALARM register detects and reports any alarms raised in the QIXIS system.
//:   Write 1 to a STAT_ALARM register bit to disable alarms handling.
//:   <bit name=-         size=2  access="RO"   reset=00></bit>
//:   <bit name=CTL       size=1  access="RO"   reset=0>
//:     Software Fault (CTL[FAIL] was set).
//:   </bit>
//:   <bit name=VTOFF     size=1  access="RO"   reset=0>
//:     VCC Power Supply Temp Off Fault
//:     0= The system is powered normally.
//:     1= The system powers off due to overtemperature of the supply monitored by VT (see bit 5).
//:   </bit>
//:   <bit name=ORIENT    size=1  access="RO"   reset=0></bit>
//:   <bit name=VCT       size=1  access="RO"   reset=0>
//:     VCC/Core Power Supply Temperature Warning:
//:     0= The VCC power supply temperature is within normal limits.
//:     1= The VCC power supply temperature has exceeded warning limits.
//:   </bit>
//:   <bit name=TWARN     size=1  access="RO"   reset=0>
//:     Temperature Fault:
//:     0= The temperature is within normal limits.
//:     1= The temperature has exceeded warning limits.
//:   </bit>
//:   <bit name=TALERT    size=1  access="RO"   reset=0>
//:     Temperature Alert:
//:     0= The temperature is within normal limits.
//:     1= The temperature has exceeded fault limits.
//:   </bit>
//: </reg>

    wire        alarm_we_b;

	assign	    alarm_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_ALARM)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)		alarm_mask <= { 8'b0000_0100 };
		else if (!alarm_we_b)		alarm_mask <= reg_wdata;


//:---------------------------------------------------------------------------
//: <reg name=STAT_PRES1  raddr=0x00B  access="RO"  reset=NONE  title="Presence Detect 1">
//:   The STAT_PRES1 detects the installation of various devices/cards.
//:   <bit name=DUT       size=1  access="RO"   reset=0>
//:     Processor Present:
//:     0= A processor is detected (soldered-in or socketed).
//:     1= No device detected.
//:   </bit>
//:   <bit name=-         size=1  access="RO"  reset=1></bit>
//:   <bit name=CPU_ID    size=2  access="RO"  reset=11  expand>
//:     Processor Installed:
//:     00= Unknown device is installed.
//:     01= LS1043A detected.
//:     10= LS1047A detected.
//:     11= LS1088A detected.
//:   </bit>
//:   <if LS1088ARDB-PA>
//:     <bit name=TDMR      size=1  access="RO"   reset=1>
//:       TDMRiser Card Present:
//:       0= A TDMRiser card is installed in the dedicated slot.
//:       1= No TDMRiser detected.
//:     </bit>
//:   <else>
//:     <bit name=-         size=1  access="RO"   reset=1></bit>
//:   </if>
//:   <bit name=-         size=1  access="RO"   reset=1></bit>
//:   <bit name=WP        size=1  access="RO"   reset=1>
//:     SDCard write-protect:
//:     0= A write-enabled card is installed in the slot.
//:     1= The card is write-protected, or no card is installed.
//:   </bit>
//:   <bit name=CD        size=1  access="RO"   reset=1>
//:     SDCard Present:
//:     0= A card is detected in the slot.
//:     1= No card detected.
//:   </bit>
//: </reg>
//
    wire [7:0]  stat_pres1;

    assign	stat_pres1  =  { present_proc_b, 
							 1'b1,                  // IFCCard
							 cpu_id,
							 present_tdm_b,
							 1'b1,
                             sdslot_wp,
                             sdslot_cd_b 
                           };


//---------------------------------------------------------------------------
//: <reg name=STAT_PRES2  raddr=0x00C  access="RO"  reset=NONE  title="Presence Detect 2">
//:   The STAT_PRES2 detects the installation of cards in various PCI Express or SGMII slots.
//:   <bit name=-         size=5  access="RO"  reset=11111></bit>
//:   <if LS1088ARDB-PA>
//:     <bit name=SLOT3     size=1  access="RO"  reset=1>
//:       0= a PCIe card is installed in slot #3.
//:       1= no PCIe card is installed in slot #3.
//:     </bit>
//:   <else>
//:     <bit name=-         size=1  access="RO"  reset=1></bit>
//:   </if>
//:   <bit name=-         size=2  access="RO"  reset=11></bit>
//: </reg>
//
    wire [7:0]  stat_pres2;

    assign	stat_pres2  =  { 5'b1111_1,
                             present_slot3_b, 
							 2'b11
                           };


//---------------------------------------------------------------------------
//: <reg name=LED       raddr=0x00E  access="RW"  reset=ARST  title="LED Control">
//:   The LED register can be used to directly control the monitoring LEDs\
//:   (M7-M0) for software debugging and messaging purposes. Direct control of\
//:   the LEDs is possible only when CTL[LED] is set to 1; otherwise Qixis uses\
//:   them to display general system activity.
//:   <bit name=LED       size=8  access="RW"   reset=00000000>
//:     LED Status Control:
//:     0= LED M[bitno] is off.
//:     1= LED M[bitno] is on.
//:   </bit>
//: </reg>
//
    reg [7:0]   leds;
    wire        led_we_b;

	assign led_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_LED)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)             leds <= 8'h00;
		else if (!led_we_b)		leds <= reg_wdata;

    assign  leds_user = leds[3:0];


//: </sect end_csr>


//================================================================================
//                                  RECONFIG SECTION
//================================================================================
//: <sect level=2  title="Reconfiguration Registers"  template="sect_reconfig">


//---------------------------------------------------------------------------
//: <reg name=RCFG      raddr=0x010  access="RW"  reset=CRST  title="Reconfiguration Control">
//:   The RCFG_CTL register is used to control the reconfiguration sequencer.
//:   <bit name=-         size=2  access="RO"   reset=00></bit>
//:   <bit name=LIVE      size=1  access="RO"   reset=1>
//:     Immediate changes for BRDCFG registers:
//:     1= BRDCFG registers outputs occur immediately.  For QixMin, LIVE is always 1.
//:   </bit>
//:   <bit name=-         size=1  access="RO"   reset=0></bit>
//:   <bit name=WDEN      size=1  access="RW"  reset=SW_BOOTBOX_B>
//:     Watchdog Enable:
//:     0= The watchdog is not enabled during reconfiguration.
//:     1= The watchdog is enabled during reconfiguration. If not disabled within\
//:        2^29 clock cycles (\> 8 minutes), the system is reset.
//:     NOTE: This is not a highly-secure watchdog; software can reset this bit at\ 
//:           any time and disable the watchdog.
//:   </bit>
//:   <bit name=-         size=2  access="RO"   reset=00></bit>
//:   <bit name=GO        size=1  access="RW"   reset=RRST=0>
//:     Reconfiguration Start:
//:     0= Reconfiguration sequencer is idle.
//:     1= On the 0-to-1 transition, the reconfiguration process begins.
//:   </bit>
//: </reg>
//
    reg     [7:0]   rcfg;
    wire            rcfg_we_b;

	assign rcfg_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RCFG)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)        rcfg <= { 1'b0,					// b7 = SAM  == 0, not supported
                                              1'b0,					// b6 = UPD  == 0, not supported
                                              1'b1,					// b5 = LIVE == 1, fixed on RDBs
                                              1'b0,					// b4 = LOCK == 0, not supported
                                              1'b0,					// b3 = WDEN	   enabled if 1
                                              2'b00,
                                              1'b0
                                             };
		else if (!rcfg_we_b)		rcfg <= reg_wdata;

    assign  rcfg_wden   = rcfg[3];
    assign  rcfg_go     = rcfg[0];


//---------------------------------------------------------------------------
//: <reg name=LOS_STAT  raddr=0x01D  access="RO"  reset=NONE  title="Loss Of Signal">
//:   The LOS_STAT reports loss of lock or fault status for various devices.
//:   <bit name=RT1LOL    size=1  access="RO"   reset=0>
//:     Retimer #1 (Lane D) Loss:
//:     0= Receive and transmit pathway(s) operating normally
//:     1= Receive or transmit pathway(s) pathway lost sync.
//:   </bit>
//:   <bit name=RTIMER    size=1  access="RO"   reset=0>
//:     Retimer Done Monitor:
//:     0= The retimer is in reset or loading configuration.
//:     1= The retimer has completed loading configuration data.
//:   </bit>
//:   <bit name=RXLOS     size=1  access="RO"   reset=0>
//:     SFP TX LOS:
//:     0= The SFP tranceiver is receiving normally.
//:     1= The SFP tranceiver is reporting a receive loss.
//:   </bit>
//:   <bit name=TXFLT     size=1  access="RO"   reset=0>
//:     SFP TX FAULT:
//:     0= The SFP tranceiver is transmitting normally.
//:     1= The SFP tranceiver is reporting a transmit fault.
//:   </bit>
//:   <bit name=-         size=4  access="RO"   reset=0000></bit>
//: </reg>
//
    wire    [7:0]   los_stat;

    assign  los_stat = { ~los_retimer_b, 
                        ~ retimer_done_b,
						  sfp_rx_los,
                          sfp_tx_fault,
						  4'b0000 };


//---------------------------------------------------------------------------
//: <reg name=WATCH     raddr=0x01F  access="RW"  reset=CRST  title="Watchdog"  template="reg_watch">
//: </reg>
//
    wire    rcfg_watch_we_b;

	assign  rcfg_watch_we_b       = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_WATCH)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)  	    rcfg_watch <= 8'h7F;
		else if (!rcfg_watch_we_b)  rcfg_watch <= reg_wdata;


//: </sect end_reconfig>


//===========================================================================
//                              POWER
//===========================================================================
//: <sect level=2  title="Power Control/Status Registers"  template="sect_power">


//---------------------------------------------------------------------------
//: <reg name=PWR_CTL2  raddr=0x021  access="RW"  reset=CRST  title="Power Control 2">
//:   The PWR_CTL2 register is used to control system power-on/power-off events.
//:   <bit name=PWR       size=1  access="RW"  reset=0>
//:     Toggle System Power
//:     0= No action.
//:     1= On the 0-to-1 transition, toggle the system power supply. The bit must\
//:        be reset to zero before additional power cycles can occur.
//:   </bit>
//:  <bit name=-         size=7  access="RO"  reset=0000000></bit>
//: </reg>

	wire	pwr_ctl2_we_b;

	assign pwr_ctl2_we_b    = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_PWR_CTL2)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)		pwr_ctl2_pwr	<= 1'b0;
		else if (!pwr_ctl2_we_b) 	pwr_ctl2_pwr	<= reg_wdata[7];


//---------------------------------------------------------------------------
//: <reg name=PWR_EVENT  raddr=0x022  access="RO"  reset=NONE  title="Power Event Trace"  template="reg_pwrevent">
//: </reg>


//---------------------------------------------------------------------------
//: <reg name=PWR_MSTAT  raddr=0x024  access="RO"  reset=NONE  title="Power Status 0"  template="reg_pwr_mstat">
//: </reg>


//---------------------------------------------------------------------------
//: <reg name=PWR_STAT1  raddr=0x025  access="RO"  reset=NONE  title="Power Status 1">
//:   The PWR_STATn registers are used to monitor the status of individual power\
//:   supplies. If a bit is set to '1', the respective power supply is operating\
//:   correctly.
//:   Note that unassigned bits default to one, allowing power failure detection\
//:   to be easily performed (if the value is not 0xFF, at least one supply is\
//:   not operating).
//:   Due to the high variability of the hardware devices used, PWR_STATn\
//:   register bits are not assigned any universally fixed values. For more\
//:   details, see the target platform documentation.
//:   <bit name=-         size=2  access="RO"   reset=11></bit>
//:   <bit name=XVDD      size=1  access="RO"   reset=1>
//:     XVDD (SerDes IO Voltage) Monitor.
//:   </bit>
//:   <bit name=TA_BB     size=1  access="RO"   reset=1>
//:     TA_BB Monitor
//:   </bit>
//:   <bit name=-         size=1  access="RO"   reset=1></bit>
//:   <bit name=V0.85     size=1  access="RO"   reset=1>
//:     VCC_0.85V Monitor.
//:   </bit>
//:   <bit name=VPP       size=1  access="RO"   reset=1>
//:     DDR4 Interface VPP Monitor
//:   </bit>
//:   <bit name=VDD       size=1  access="RO"   reset=1>
//:     Core Power Monitor
//:   </bit>
//: </reg>


//---------------------------------------------------------------------------
//: <reg name=PWR_STAT2  raddr=0x026  access="RO"  reset=NONE  title="Power Status 2">
//:   Monitors various power statuses; see PWR_STAT1 for details.
//:   <bit name=-         size=2  access="RO"   reset=11></bit>
//:   <bit name=VCC1p2    size=1  access="RO"   reset=1>
//:     VCC_1.2V Monitor
//:   </bit>
//:   <bit name=OVDD      size=1  access="RO"   reset=1>
//:     OVDD Monitor
//:   </bit>
//:   <bit name=-         size=1  access="RO"   reset=1></bit>
//:   <bit name=VCC_2V5   size=1  access="RO"   reset=1>
//:     2V5 Monitor
//:   </bit>
//:   <bit name=VTT1      size=1  access="RO"   reset=1>
//:     DDR termination power #1
//:   </bit>
//:   <bit name=GVDD      size=1  access="RO"   reset=1>
//:     DDR Power.
//:   </bit>
//: </reg>


//: </sect end_power>


//===========================================================================
//                              CLOCKS
//===========================================================================
//: <sect level=2  title="Clock Control Registers"  template="sect_clocks"> 


//---------------------------------------------------------------------------
//: <reg name=CLK_SPD1  raddr=0x030  access="RO"  reset=NONE  title="Clock Speed 1"  template="reg_clk_spd1">
//:   <bit name=DDRCLK    size=4  access="RO"  reset=0000>
//:     DDRCLK Rate Selection:
//:     `0001= 100.00 MHz (fixed)`
//:     `Other values are reserved.`
//:   </bit>
//:   <bit name=SYSCLK    size=4  access="RO"  reset=0000>
//:     SYSCLK Rate Selection:
//:     `0010= 100.00 MHz (fixed)`
//:     `Other values are reserved.`
//:   </bit>
//: </reg>

    wire    [7:0]   clk_spd1;

	assign	clk_spd1	= { 2'b00, brdcfg1[5:0] };


//---------------------------------------------------------------------------
//: <reg name=CLK_ID    raddr=0x033  access="RO"  reset=NONE  title="Clock ID/Status">
//:   The CLK_ID register is used to identify the arrangement of the clock\
//:   control registers. Software should check CLK_ID register before attempting\
//:   to interpret/control the clock control registers.
//:   <bit name=-         size=4  access="RO"   reset=0000></bit>
//:   <bit name=ID        size=4  access="RO"   reset=0100>
//:     System Clock ID = 0000 (NONE)
//:     CLK0= SYSCLK is fixed on this system.
//:   </bit>
//: </reg>

    wire    [7:0]   clk_id;

	assign	clk_id 	    = { 8'b00000000 };


//: </sect end_clocks>


//===========================================================================
//                              RESET
//===========================================================================
//: <sect level=2  title="Reset Control Registers"  template="sect_reset">


//---------------------------------------------------------------------------
//: <reg name=RST_CTL   raddr=0x040  access="RW"  reset=CRST  title="Reset Control">
//:   The RST_CTL register is used configure or trigger reset actions.
//:   <bit name=-         size=2  access="RW"  reset=00></bit>
//:   <bit name=REQMD     size=2  access="RW"  reset=SW_RSTMD>
//:     00= Disabled - do nothing.
//:     11= Normal   - assert PORESET_B to DUT to begin normal reset sequence.
//:   </bit>
//:   <bit name=DDRLK     size=1  access="RW"  reset=ARST=0>
//:     DDR Reset Lock:
//:     0= Reset is asserted to the DDR DIMMs/devices normally.
//:     1= Reset will not be asserted to the DDR DIMMs/devices. With proper DDR\
//:        controller setup and careful software setup DDR contents can survive\
//:        resets.
//:     This bit is not cleared with a general reset, but is preserved, as long as\
//:     power is available. It is expected that software that sets this bit is also\
//:     responsible for clearing it.
//:   </bit>
//:   <bit name=-         size=2  access="RW"  reset=00></bit>
//:   <bit name=RST       size=1  access="RW"  reset=0>
//:     Reset:
//:     0= Reset sequencer operates normally.
//:     1= Upon transition from 0 to 1, restart the reset sequence.
//:   </bit>
//: </reg>
//
    reg [7:0]   rst_ctl;
    wire        rst_ctl_we_b;

	assign	rst_ctl_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_CTL)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)			rst_ctl[7:4]    <= { 2'b00,
                                                             sw_rst_mode ? 2'b11 : 2'b00
                                                           };
		else if (!rst_ctl_we_b)			rst_ctl[7:4]    <= reg_wdata[7:4];

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)  	                rst_ctl[  3]    <= 1'b0;
		else if (!rst_ctl_we_b)  	    rst_ctl[  3]    <= reg_wdata[3];

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)  	        rst_ctl[2:0]    <= 3'b000;
		else if (!rst_ctl_we_b)  	    rst_ctl[2:0]    <= reg_wdata[2:0];


    assign cfg_rstreq_md    = (rst_ctl[5:4] == 2'b11) ? 1'b1 : 1'b0;
    assign cfg_rst_noddr    = rst_ctl[3];
    assign rst_rst          = rst_ctl[0];


//---------------------------------------------------------------------------
//: <reg name=RST_STAT  raddr=0x041  access="RO"  reset=NONE  title="Reset Status"  template="reg_rst_stat">
//: </reg>
//
    wire [7:0]  rst_stat;

    assign  rst_stat    = { 5'b0000_0, 
                            ~ int_hreset_b, 
                            ~ int_poreset_b, 
                            ~ int_reset_req_b 
                          };


//---------------------------------------------------------------------------
//: <reg name=RST_REASON  raddr=0x042  access="RO"  reset=NONE  title="Reset Event Trace">
//:   The RST_REASON register is used to report the cause of the most-recent reset cycle.
//:   <bit name=PREV    size=4  access="RO"  reset=1111  expand=@.REASON>
//:     Previous reset reason:
//:     See REASON field codes.
//:   </bit>
//:   <bit name=REASON    size=4  access="RO"   reset=0000  expand>
//:     Reset Reason:
//:     `0000= Power-on reset`
//:     `0001= JTAG_RST_B asserted`
//:     `0010= (reserved)`
//:     `0011= RST_CTL[RST] asserted`
//:     `0100= Reset switch (chassis or on-board) was pushed.`
//:     `0101= RCFG_CTL[GO] asserted (a reconfiguration reset).`
//:     `0110= RESET_REQ_B assertion (from processor) was asserted.`
//:     `1111= No event recorded yet.`
//:   </bit>
//: </reg>


//---------------------------------------------------------------------------
//: <reg name=RST_FORCE1  raddr=0x043  access="RW"  reset=GRST  title="Reset Force 1">
//:   The RST_FORCEn registers are used to force reset to a particular device,\
//:   independent of the general reset sequencer. As long as a bit is set to 1,\
//:   the reset signal to grouped devices will be asserted.
//:   Due to the high variability of hardware devices used, RST_FORCEn registers\
//:   do not have a universally-defined purpose. Consult the target platform\
//:   documentation for details; a table of customary assignments is shown below,\
//:   but keep in mind it is only a reference.
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//:   <bit name=QSPI      size=1  access="RW"   reset=0>
//:     1= Assert RST_QSPI_B.
//:   </bit>
//:   <bit name=I2C       size=1  access="RW"   reset=0>
//:     1= Assert RST_I2C_B.
//:   </bit>
//:   <bit name=EPHY3     size=1  access="RW"   reset=0>
//:     1= Assert RST_EPHY3_B
//:   </bit>
//:   <bit name=EPHY2     size=1  access="RW"   reset=0>
//:     1= Assert RST_EPHY2_B
//:   </bit>
//:   <bit name=EPHY1     size=1  access="RW"   reset=0>
//:     1= Assert RST_EPHY1_B
//:   </bit>
//:   <bit name=MEM       size=1  access="RW"   reset=0>
//:     Reset all DDR slots.
//:     1= Assert RST_MEM_B (RST_MEM_DRV_B translated to DDR voltage level).
//:   </bit>
//: </reg>
//
    wire        rst_force1_we_b;

	assign	rst_force1_we_b = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_FORCE1)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)			rst_force1 <= 8'b0000_0000;
		else if (!rst_force1_we_b)      rst_force1 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=RST_FORCE2  raddr=0x044  access="RW"  reset=GRST  title="Reset Force 2">
//:   Assert selected reset sources.  See RST_FORCE1 for details.
//:   <bit name=IEEE      size=1  access="RW"   reset=0>
//:     1= Assert RST_IEEE_B / RST_1588_B for IEEE-1588 test cards.
//:   </bit>
//:   <bit name=RTIME     size=1  access="RW"   reset=0>
//:     1= Assert RST_RETIMER_B
//:   </bit>
//:   <bit name=CLK       size=1  access="RW"   reset=0>
//:     1= Assert RST_CLK.
//:   </bit>
//:   <if LS1088ARDB-PA>
//:     <bit name=TDM       size=1  access="RW"   reset=0>
//:       1= Assert RST_TDMRISER_B.
//:     </bit>
//:   <else>
//:     <bit name=-         size=1  access="RW"   reset=0></bit>
//:	  </if>
//:   <bit name=EMMC      size=1  access="RW"   reset=0>
//:     1= Assert RST_EMMC_B.
//:   </bit>
//:   <bit name=TRST      size=1  access="RW"   reset=0>
//:     1= Assert DUT_TRST_B.
//:   </bit>
//:   <bit name=HRST      size=1  access="RW"   reset=0>
//:     1= Assert DUT_HRESET_B.
//:     NOTE: This bit only asserts the signal to the DUT; it is not intended to be\
//:     used as a general system reset.
//:   </bit>
//:   <bit name=PORST     size=1  access="RW"   reset=0>
//:     1= Assert DUT_PORESET_B.
//:     NOTE: This bit only asserts the signal to the DUT; it is not intended to be\
//:     used as a general system reset.
//:   </bit>
//: </reg>
//      
    wire    rst_force2_we_b;
	assign	rst_force2_we_b = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_FORCE2)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)			rst_force2 <= 8'b0000_0000;
		else if (!rst_force2_we_b)      rst_force2 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=RST_FORCE3  raddr=0x045  access="RW"  reset=GRST  title="Reset Force 3">
//:   Assert selected reset sources.  See RST_FORCE1 for details.
//:   <if LS1088ARDB-PA>
//:     <bit name=SLOT1     size=1  access="RW"   reset=0>
//:       1= Assert RST_SLOT1_B.
//:     </bit>
//:     <bit name=SLOT2     size=1  access="RW"   reset=0>
//:       1= Assert RST_SLOT2_B.
//:     </bit>
//:     <bit name=SLOT3     size=1  access="RW"   reset=0>
//:       1= Assert RST_SLOT3_B.
//:     </bit>
//:   <else>
//:     <bit name=-         size=3  access="RW"   reset=000></bit>
//:	  </if>
//:   <bit name=-         size=5  access="RW"   reset=00000></bit>
//: </reg>
//
    wire    rst_force3_we_b;

	assign	rst_force3_we_b = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_FORCE3)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)			rst_force3 <= 8'b0000_0000;
		else if (!rst_force3_we_b)      rst_force3 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=RST_MASK1  raddr=0x04B  access="RW"  reset=ARST  title="Reset Mask 1">
//:   The RST_MASKn registers are used to block reset to a particular device,\
//:   independent of the general reset sequencer. As long as a bit is set to 1,\
//:   the reset signal to that device or devices will be blocked.
//:   RST_MASKn bits have the same bit definition as their counterparts in\
//:   RST_FORCEn; refer to Table 5-53 for details.
//:   Note that RST_MASK bits are cleared on AUX reset, and so are usually only\
//:   cleared by software. This is very different from the RST_FORCE registers.
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//:   <bit name=QSPI      size=1  access="RW"   reset=0>
//:     1= Mast RST_QSPI_B.
//:   </bit>
//:   <bit name=I2C       size=1  access="RW"   reset=0>
//:     1= Mask RST_I2C_B.
//:   </bit>
//:   <bit name=EPHY3     size=1  access="RW"   reset=0>
//:     1= Mask RST_EPHY3_B
//:   </bit>
//:   <bit name=EPHY2     size=1  access="RW"   reset=0>
//:     1= Mask RST_EPHY3_B
//:   </bit>
//:   <bit name=EPHY1     size=1  access="RW"   reset=0>
//:     1= Mask RST_EPHY1_B
//:   </bit>
//:   <bit name=MEM       size=1  access="RW"   reset=0>
//:     1= Mask RST_MEM_B (DDR slots, etc.)
//:   </bit>
//: </reg>
//
    wire    rst_mask1_we_b;

	assign  rst_mask1_we_b   = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_MASK1)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)  	            rst_mask1 <= { 8'h00 };
		else if (!rst_mask1_we_b)	rst_mask1 <= reg_wdata;

//---------------------------------------------------------------------------
//: <reg name=RST_MASK2  raddr=0x04C  access="RW"  reset=ARST  title="Reset Mask 2">
//:   Mask selected reset sources.  See RST_FORCE1 for details.
//:   <bit name=IEEE      size=1  access="RW"   reset=0>
//:     1= Mask RST_IEEE_B.
//:   </bit>
//:   <bit name=RTIME     size=1  access="RW"   reset=0>
//:     1= Mask RST_RTIMER_B.
//:   </bit>
//:   <bit name=CLK       size=1  access="RW"   reset=0>
//:     1= Mask RST_CLK.
//:   </bit>
//:   <if LS1088ARDB-PA>
//:     <bit name=TDM       size=1  access="RW"   reset=0>
//:       1= Mask RST_TDMRISER_B.
//:     </bit>
//:   <else>
//:     <bit name=-         size=1  access="RW"   reset=0></bit>
//:	  </if>
//:   <bit name=EMMC      size=1  access="RW"   reset=0>
//:     1= Mask RST_EMMC_B.
//:   </bit>
//:   <bit name=-         size=1  access="RW"   reset=0></bit>
//:   <bit name=HRST      size=1  access="RW"   reset=0>
//:     1= Mask DUT_HRESET_B.
//:   </bit>
//:   <bit name=PORST     size=1  access="RW"   reset=0>
//:     1= Mask DUT_PORESET_B.
//:   </bit>
//: </reg>
    wire    rst_mask2_we_b;

	assign  rst_mask2_we_b   = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_MASK2)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)  	            rst_mask2 <= { 8'h00 };
		else if (!rst_mask2_we_b)	rst_mask2 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=RST_MASK3  raddr=0x04D  access="RW"  reset=ARST  title="Reset Mask 2">
//:   Mask selected reset sources.  See RST_FORCE1 for details.
//:   <if LS1088ARDB-PA>
//:     <bit name=SLOT1     size=1  access="RW"   reset=0>
//:       1= Mask RST_SLOT1_B.
//:     </bit>
//:     <bit name=SLOT2     size=1  access="RW"   reset=0>
//:       1= Mask RST_SLOT2_B.
//:     </bit>
//:     <bit name=SLOT3     size=1  access="RW"   reset=0>
//:       1= Mask RST_SLOT3_B.
//:     </bit>
//:   <else>
//:     <bit name=-         size=3  access="RW"   reset=000></bit>
//:	  </if>
//:   <bit name=-         size=5  access="RW"   reset=00000></bit>
//: </reg>

    wire    rst_mask3_we_b;

	assign  rst_mask3_we_b   = (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_RST_MASK3)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)  	            rst_mask3 <= { 8'h00 };
		else if (!rst_mask3_we_b)	rst_mask3 <= reg_wdata;


//: </sect end_reset>


//===========================================================================
//                              BRDCFG
//===========================================================================
//: <sect level=2  title="Board Configuration Registers"  template="sect_brdcfg">
//
// BRDCFG registers.  Only selected registers are available.


//---------------------------------------------------------------------------
//: <reg name=BRDCFG0   raddr=0x050  access="RW"  reset=RRST  title="Board Configuration 0">
//:   The BRDCFG0 register is commonly used to select IFC and QSPI boot devices.
//:   <bit name=QMAP      size=3  access="RW"   reset=SW_QMAP  expand>
//:     QMAP controls hows QSPI_A chip-selects are connected to various peripherals.
//:     `      QSPI_CSA_0   QSPI_CSA_1`
//:     `      ==========   ==========`
//:     `000=  DEV #0       DEV #1`
//:     `001=  DEV #1       DEV #0`
//:     `010=  EMU          DEV #0`
//:     `011=  EMU          DEV #1`
//:     `100=  DEV #0       EMU`
//:   </bit>
//:   <bit name=-         size=5  access="RW"   reset=00000></bit>
//: </reg>
//
    reg [7:0]   brdcfg0;
    wire        brdcfg0_we_b;

	assign	    brdcfg0_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_BRDCFG0)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			brdcfg0 <= { sw_qspi_map, 
                                                     5'd0
                                                   };
		else if (!brdcfg0_we_b)			brdcfg0 <= reg_wdata;


// QSPI is swappable among the two devices and the emulator, by proper control of the muxes,
// using the following table:
//
//  CFG_QSPI/|
//  SW_QSPI  |  CFG_QSPI_MAP[4:0]   |       CONNECTIONS                 | DESCRIPTION
//  =========|======================|===================================|==========
//    000    |   1 0 0 X 0          | CS0->DEV0   CS1->DEV1   EMU=off   | Default
//    001    |   1 0 0 X 1          | CS0->DEV1   CS1->DEV0   EMU=off   | Swap
//    010    |   0 1 0 1 1          | CS0->EMU    CS1->DEV0   DEV1=off  | Boot EMU/Prog DEV0
//    011    |   0 0 1 1 0          | CS0->EMU    CS1->DEV1   DEV0=off  | Boot EMU/Prog DEV1
//    100    |   0 1 0 0 0          | CS0->DEV0   CS1->EMU    DEV1=off  | General EMU access
//    101    |   0 1 0 0 1          | EMU_CS0->DEV0_CS0       CS1=off   | EMU in-system prog. [*]
//    110    |   0 1 0 1 0          | EMU_CS0->DEV0_CS1       CS0=off   | EMU in-system prog. [*]
//    111    |   1 1 1 X X          | CS0->off    CS1->off              | QSPI disabled.
//  =========|======================|===================================|==========
//  [*] DUT must have IFC tristated.
//	 

    assign  cfg_qspi_map    = (!disable_io_b)       ? 5'bZZZZZ
                            : (brdcfg0[7:5] == 3'b000)  ? 5'b10000
                            : (brdcfg0[7:5] == 3'b001)  ? 5'b10001
                            : (brdcfg0[7:5] == 3'b010)  ? 5'b01011
                            : (brdcfg0[7:5] == 3'b011)  ? 5'b00110
                            : (brdcfg0[7:5] == 3'b100)  ? 5'b01000
                            : (brdcfg0[7:5] == 3'b101)  ? 5'b01001
                            : (brdcfg0[7:5] == 3'b110)  ? 5'b01010
                            :							  5'b11111;     // Off is default for reserved


//---------------------------------------------------------------------------
//: <reg name=BRDCFG1   raddr=0x051  access="RO"  reset=RRST  title="Board Configuration 1">
//:   The BRDCFG1 register shows/controls SYSCLK and DDRCLK speeds.
//:   <bit name=-         size=2  access="RO"   reset=00></bit>
//:   <bit name=DDRCLK    size=2  access="RO"   reset=01>
//:     DDRCLK Frequency Selection:
//:     `01= 100.00 MHz (fixed)`
//:     `Other values are reserved.`
//:   </bit>
//:   <bit name=SYSCLK    size=4  access="RO"   reset=0010>
//:     SYSCLK Frequency Selection:
//:     `0010= 100.000000 MHz (fixed)`
//:   </bit>
//: </reg>
//
    assign  brdcfg1 = { 2'b00,
						2'b01,				// DDRCLK = 100 MHz
                        4'b0010,            // SYSCLK = 100 MHz
                      };


//---------------------------------------------------------------------------
//: <reg name=BRDCFG2   raddr=0x052  access="RO"  reset=RRST  title="Board Configuration 2">
//:   The BRDCFG2 register controls SerDes clock speeds for all SerDes blocks.
//:   <bit name=SD1CK1    size=2  access="RO"   reset=00  expand=@.SD2CK1>
//:     SerDes1 Clock #1 Rate:
//:     `00= 100.000 MHz`
//:   </bit>
//:   <bit name=SD1CK2    size=2  access="RO"   reset=10  expand>
//:     SerDes1 Clock #2 Rate:
//:     `00= (reserved)`
//:     `01= (reserved)`
//:     `10= 156.250 MHz  (fixed)`
//:     `11= (reserved)`
//:   </bit>
//:   <bit name=SD2CK1    size=2  access="RO"   reset=SW_SPREAD  expand>
//:     SerDes2 Clock #1 Rate:
//:     `00= 100.000 MHz  (fixed, SW_SPREAD=0)`
//:     `01= (reserved)`
//:     `10= (reserved)`
//:     `11= 100.000 MHz  (spread-spectrum enabled, SW_SPREAD=1)`
//:   </bit>
//:   <bit name=SD2CK2    size=2  access="RO"   reset=SW_SPREAD  expand=@.SD2CK1>
//:     SerDes2 Clock #2 Rate:
//:     `00= 100.000 MHz  (fixed, SW_SPREAD=0)`
//:     `11= 100.000 MHz  (spread-spectrum enabled, SW_SPREAD=1)`
//:   </bit>
//: </reg>
//
	wire [7:0]	brdcfg2;
    wire [1:0]  sd2_rate;

    assign  sd2_rate   = (brdcfg4[2]) ? 2'b11 : 2'b00;

    assign  brdcfg2 = { 2'b00, 
                        2'b10, 
                        sd2_rate, 
                        sd2_rate
                      };


//---------------------------------------------------------------------------
//: <reg name=BRDCFG4   raddr=0x054  access="RW"  reset=RRST  title="Board Configuration 4">
//:   BRDCFG4 controls EMI routing as well as other target-dependent resources.
//:   <bit name=-         size=3  access="RW"   reset=000></bit>
//:   <bit name=TABBVSEL  size=1  access="RW"   reset=1>
//:     TA_BB Voltage Selection:
//:     0= 0.9V
//:     1= 1.0V (default)
//:     NOTE: If VDD is changed via PMBus, TABBVSEL must be changed as well to track VDD as close\
//:     as possible.
//:     Valid for Rev B boards or later only.
//:   </bit>
//:   <bit name=SVDDVSEL  size=1  access="RW"   reset=1>
//:     SVDD Voltage Selection:
//:     0= 1.0V
//:     1= 0.9V
//:   </bit>
//:   <bit name=PEXSS     size=1  access="RW"   reset=SW_SPREAD_EN>
//:     PCI Express Spread-Spectrum Enable:
//:     Controls the net labelled CFG_PEXSS or CFG_SPREAD_EN.
//:     0= Disabled.
//:     1= Enabled for SERDES3 and SERDES4.
//:   </bit>
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//: </reg>
//
    wire        brdcfg4_we_b;

	assign	    brdcfg4_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_BRDCFG4)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			brdcfg4 <= { 4'b0001, 
                                                     sw_svdd_sel, 
                                                     sw_spread, 
                                                     2'd0 
                                                    };
		else if (!brdcfg4_we_b)			brdcfg4 <= reg_wdata;

	assign	cfg_tabb_sel    = (!disable_io_b) ? 1'bZ : brdcfg4[4];
	assign	cfg_svdd_sel    = (!disable_io_b) ? 1'bZ : brdcfg4[3];
	assign	cfg_spread      = (!disable_io_b) ? 1'bZ : brdcfg4[2];


//---------------------------------------------------------------------------
//: <reg name=BRDCFG5   raddr=0x055  access="RW"  reset=RRST  title="Board Configuration 5">
//:   BRDCFG5 controls target-dependent resources.
//:   <bit name=I2C2      size=1  access="RW"   reset=0>
//:     Controls I2C2 routing:
//:     0= I2C2 connect to CPLD for SDHC_CD_B/SDHC_WP signals.
//:   <if LS1088ARDB-PA>
//:     1= I2C2 connect to TDM Riser for CLK09/CLK10 signals.
//:   <else>
//:     1= reserved.
//:	  </if>
//:   </bit>
//:   <bit name=I2C3      size=1  access="RW"   reset=0>
//:     Controls I2C3 routing:
//:     0= I2C3 connect to USB2 for USB2_DRVBUS/USB2_PWRFAULT signals.
//:   <if LS1088ARDB-PA>
//:     1= reserved.
//:   <else>
//:	  </if>
//:   </bit>
//:   <bit name=-         size=1  access="RW"   reset=0></bit>
//:   <bit name=IEEECLK   size=1  access="RW"   reset=0>
//:     Controls where the processor receives EC clocks:
//:     0= On-board 125 MHz reference.
//:     1= Via IEEE-1588 header.
//:   </bit>
//:   <bit name=SPISDHC   size=2  access="RW"   reset=SW_SDMODE  expand>
//:     Controls the SPI and SDHC signal routing:
//:     `0X= Auto Mode`
//:     `    If SDHC_CS_B=0 (SDHC card installed):`
//:     `      Use SDHC mode described below.`
//:     `    Else SDHC_CS_B=1 (no SDHC card installed):`
//:     `      Use eMMC mode described below.`
//:     `10= Force SDHC Mode`
//:     `      SPI_PCS[0] is routed to CPLD for SDHC_VS use.`
//:     `      SPI_PCS[1] is unused.`
//:   <if LS1088ARDB-PA>
//:     `      SPI_PCS[2:3] are routed to the TDMRiser slot.`
//:	  </if>
//:     `11= Force eMMC Mode`
//:     `      SPI_PCS[0:3] are routed to the eMMC card as eMMC_DAT4-7.`
//:   </bit>
//:   <bit name=-         size=1  access="RW"   reset=0></bit>
//:   <bit name=EVDDF     size=1  access="RW"   reset=0>
//:     EVDD Force.
//:     0= EVDD operates normally: tracking SDCard vs. EMMC use.
//:     1= EVDD is forced to 1.8V always.
//:   </bit>
//: </reg>
//
    reg	  [7:0]	brdcfg5;
    wire        brdcfg5_we_b;
	wire  [1:0]	int_cfg_spisdhc;
    reg         sd_mode;

	assign	    brdcfg5_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_BRDCFG5)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			brdcfg5 <= { 4'b0000,
                                                     sw_sdmode,
                                                     1'b0,
                                                     1'b0
                                                    };
		else if (!brdcfg5_we_b)			brdcfg5 <= reg_wdata;

	assign	cfg_mux_i2c2	= (!disable_io_b) ? 1'bZ 	:  brdcfg5[7];
	assign	cfg_mux_i2c3	= (!disable_io_b) ? 1'bZ 	:  brdcfg5[6];
                                                        //         5 unused
	assign	cfg_ieee_clk	= (!disable_io_b) ? 1'bZ 	:  brdcfg5[4];
	assign	int_cfg_spisdhc	=                              brdcfg5[3:2];
                                                        //         1 unused
	assign	cfg_evdd_force 	=                              brdcfg5[0];      


// SPISDHC controls how SPI+SDHC muxes are routed.  Since eMMC cannot work in
// 4bit mode as originally thought, the two pins are always in tandem.
//
	always @*
		if (int_cfg_spisdhc == 2'b11)				// eMMC always
			sd_mode	<= 1'b1;

		else if (int_cfg_spisdhc == 2'b10)      	// SDHC always
			sd_mode	<= 1'b0;

		else										// Auto: if card (CD_B==0), SD mode.
			sd_mode	<= sdslot_cd_b;	

	assign	cfg_mux_spi  	= (!disable_io_b) ? 1'bZ 	:  sd_mode;
	assign	cfg_mux_sd  	= (!disable_io_b) ? 1'bZ 	:  sd_mode;
	assign	int_cfg_mux_sd 	=                              sd_mode;         // Internal copy for EVDD control.


//---------------------------------------------------------------------------
//: <if LS1088ARDB-PB>
//: <reg name=BRDCFG6   raddr=0x056  access="RW"  reset=RRST  title="Board Configuration 6">
//:   BRDCFG6 controls PCIe M.2 resources for the LS1088ARDB-PB.  It is
//:   ignored on the LS1088ARDB.
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//:   <bit name=P2WGE     size=1  access="RW"   reset=1>
//:     M.2 #2 WIFI Gbit Enable (net PCIE2_WGIG_EN):
//:     0= Gbit WIFI mode disabled.
//:     1= Gbit WIFI mode enabled.
//:   </bit>
//:   <bit name=P2DIS     size=1  access="RW"   reset=1>
//:     M.2 #2 WIFI Enable (net PCIE2_WDISABLE_B):
//:     0= WIFI disabled.
//:     1= WIFI enabled.
//:   </bit>
//:   <bit name=-         size=2  access="RW"   reset=00></bit>
//:   <bit name=P1WGE     size=1  access="RW"   reset=1>
//:     M.2 #1 WIFI Gbit Enable (net PCIE1_WGIG_EN):
//:     0= Gbit WIFI mode disabled.
//:     1= Gbit WIFI mode enabled.
//:   </bit>
//:   <bit name=P2DIS     size=1  access="RW"   reset=1>
//:     M.2 #1 WIFI Enable (net PCIE1_WDISABLE_B):
//:     0= WIFI disabled.
//:     1= WIFI enabled.
//:   </bit>
//: </reg>
//: </if>

    reg [7:0]   brdcfg6;
    wire        brdcfg6_we_b;

	assign	    brdcfg6_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_BRDCFG6)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			brdcfg6 <= { 2'b00,
                                                     1'b1,
                                                     1'b1,
                                                     2'b00,
                                                     1'b1,
                                                     1'b1
                                                    };
		else if (!brdcfg6_we_b)			brdcfg6 <= reg_wdata;

															//         7:6 unused
	assign	cfg_pci2_wgig_en	= (!disable_io_b) ? 1'bZ 	:  brdcfg6[ 5 ];
	assign	cfg_pci2_dis_b		= (!disable_io_b) ? 1'bZ 	:  brdcfg6[ 4 ];
															//         3:2 unused
	assign	cfg_pci1_wgig_en	= (!disable_io_b) ? 1'bZ 	:  brdcfg6[ 1 ];
	assign	cfg_pci1_dis_b		= (!disable_io_b) ? 1'bZ 	:  brdcfg6[ 0 ];


//---------------------------------------------------------------------------
//: <reg name=BRDCFG9   raddr=0x059  access="RW"  reset=RRST  title="Board Configuration 9">
//:   BRDCFG9 controls target-dependent resources.
//:   <bit name=-         size=3  access="RW"   reset=000></bit>
//:   <bit name=SFP_TX    size=1  access="RW"   reset=X>
//:     Enables transmit of an SFP module by controlling the TX_DISABLE pin (net nSFP_TX_EN).
//:     0= SFP transmit enabled.
//:     1= SFP transmit disabled.
//:     The default value is based on the serdes protocol selected.
//:   </bit>
//:   <bit name=-         size=4  access="RW"   reset=0000></bit>
//: </reg>
//
    reg [7:0]   brdcfg9;
    wire        brdcfg9_we_b;

	assign	    brdcfg9_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_BRDCFG9)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			brdcfg9 <= { 3'b0000,
                                                     1'b0,
                                                     4'b0000
                                                    };
		else if (!brdcfg9_we_b)			brdcfg9 <= reg_wdata;

                                                        //         7:5 unused
	assign	cfg_sfp_txen_b	= (!disable_io_b) ? 1'bZ 	:  brdcfg9[ 4 ];
                                                        //         3:0 unused


//: </sect end_brdcfg>


//===========================================================================
//                              DUTCFG
//===========================================================================
// Note that there is no "live" option with DUTCFG, as they can only be driven during
// PORESET_B by definition.  Therefore, there are no "shadow" registers.
//
//: <sect level=2  title="DUT Configuration Registers"  template="sect_dutcfg">


//---------------------------------------------------------------------------
//: <reg name=DUTCFG0   raddr=0x060  access="RW"  reset=RRST  title="DUT Configuration 0">
//:   The DUTCFG0 register is used to control fixed DUT configurations, in\
//:   particular the RCW location setting (cfg_rcw_src).
//:   <bit name=RCWSRC    size=8  access="RW"   reset=SW_RCRSRC>
//:     RCW Source Location:
//:     The 8 most-significant bits of the RCW location configuration (cfg_rcw_src[8:1]).
//:   </bit>
//: </reg>
//
    reg [7:0]   dutcfg0;
    wire        dutcfg0_we_b;

	assign	    dutcfg0_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG0)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg0 <= sw_rcw_src[8:1];
		else if (!dutcfg0_we_b)		    dutcfg0 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=DUTCFG1   raddr=0x061  access="RW"  reset=RRST  title="DUT Configuration 1">
//:   DUTCFG1 manages additional cfg_rcw_src bits and DRAM type (if supported).
//:   <bit name=DRAM      size=1  access="RW"   reset=SW_DRAM>
//:     DRAM Type:
//:     0= DDR4 (LS1043A only).
//:     1= DDR4 (LS1088A only).
//:     NOTE: Only DDR4 is supported, but the configuration value varies\
//:     depending on the device.
//:   </bit>
//:   <bit name=-         size=2  access="RW"   reset=11></bit>
//:   <bit name=TE        size=1  access="RW"   reset=1>
//:     IFC Transceiver Enable:
//:     Controls cfg_te.
//:   </bit>
//:   <bit name=-         size=3  access="RW"   reset=111></bit>
//:   <bit name=RSRC8     size=1  access="RW"   reset=1>
//:     RCW Source Location (additional):
//:     Controls cfg_rcw_src[0] (the LSB of the 9-bit value).
//:   </bit>
//: </reg>
//
    reg [7:0]   dutcfg1;
    wire        dutcfg1_we_b;
    wire        dutcfg1_dram;


	assign	    dutcfg1_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG1)) ? 1'b0 : 1'b1;

    assign      dutcfg1_dram    = (cpu_id == 2'b01) ? 1'b0      // LS1043:     DDR4= 1'b0
                                : (cpu_id == 2'b10) ? 1'b0      // LS1046:     DDR4= 1'b0
                                :                     1'b1;     // All others: DDR4 = 1'b1

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg1 <= { dutcfg1_dram,
                                                     2'b11,
                                                     1'b0,          // TE mode
                                                     3'b111,
                                                     sw_rcw_src[0]
                                                   };
		else if (!dutcfg1_we_b)		    dutcfg1 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=DUTCFG2   raddr=0x062  access="RW"  reset=RRST  title="DUT Configuration 2">
//:   DUTCFG2 manages device selection (SVR) and internal-only device test features.
//:   <bit name=-         size=5  access="RW"   reset=11111></bit>
//:   <bit name=SVR01     size=2  access="RW"   reset=SW_SVR>
//:     Controls cfg_svr[0:1] (note the bit order).
//:   </bit>
//:   <bit name=TEST      size=1  access="RW"   reset=1>
//:     Controls processor pin 'TESTSEL_B'.
//:     NOTE: Unlike all other DUTCFG bits, this one is static (always driven).
//:   </bit>
//: </reg>
//
    reg [7:0]   dutcfg2;
    wire        dutcfg2_we_b;

	assign	    dutcfg2_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG2)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg2 <= { 5'b1111_1,
                                                     sw_svr[0],
                                                     sw_svr[1],
                                                     sw_testsel_b
                                                   };
		else if (!dutcfg2_we_b)		    dutcfg2 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=DUTCFG6   raddr=0x066  access="RW"  reset=RRST  title="DUT Configuration 6">
//:   The DUTCFG6 register is used to sample device-specific configuration modes.
//:   <bit name=RSPDS     size=1  access="RW"   reset=1>
//:     Reset Sequencer Pause Disable:
//:     Configures cfg_rsp_dis:
//:     0= Reset Sequencer Pause
//:     1= Reset Sequencer Normal
//:   </bit>
//:   <bit name=-         size=6  access="RW"   reset=111111></bit>
//:   <bit name=SOC       size=1  access="RW"   reset=1>
//:     Controls cfg_soc_use:
//:     0= (system-dependent)
//:     1= (system-dependent)
//:   </bit>
//: </reg>
// DUTCFG6
//
    reg [7:0]   dutcfg6;
    wire        dutcfg6_we_b;

	assign	    dutcfg6_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG6)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg6 <= { sw_rspdis, 7'b111_1111 };
		else if (!dutcfg6_we_b)		    dutcfg6 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=DUTCFG11  raddr=0x06B  access="RW"  reset=RRST  title="DUT Configuration 11">
//:   The DUTCFG11 register is used to control the CFG_ENGUSE[7:0] signals.  The\
//:   function of these bits are defined by silicon designers for special use.
//:   <bit name=ENGUSE0   size=1  access="RW"   reset=SW_ENG0>
//:     Differential Clock Mode (cfg_enguse0):
//:     0= Processor uses differential SYSCLK_P/SYSCLK_N input.
//:     1= Processor uses single-ended SYSCLK input.
//:   </bit>
//:   <bit name=ENGUSE1   size=1  access="RW"   reset=SW_ENG1>
//:     Reserved (cfg_enguse1):
//:     1= Default value for unassigned pins.
//:   </bit>
//:   <bit name=ENGUSE2   size=1  access="RW"   reset=SW_ENG2>
//:     Reserved (cfg_enguse2):
//:     1= Default value for unassigned pins.
//:   </bit>
//:   <bit name=-         size=5  access="RW"   reset=11111></bit>
//: </reg>
//
    reg  [7:0]  dutcfg11;
    wire        dutcfg11_we_b;

	assign	    dutcfg11_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG11)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg11 <= { sw_enguse, 5'b1_1111 }; 
		else if (!dutcfg11_we_b)        dutcfg11 <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=DUTCFG12  raddr=0x06C  access="RW"  reset=RRST  title="DUT Configuration 12">
//:   The DUTCFG12 register is used to control the general-purpose GPCFG signals. 
//:   These configurations are used exclusively by end customers for their own purposes.
//:   <bit name=GCA       size=8  access="RW"   reset=11111111>
//:     General Purpose Configuration Controls:
//:     XX= GCA drives cfg_gpin[7:0]. 
//:     No hardware definition, software determined meanings.
//:   </bit>
//: </reg>
//
    reg [7:0]   dutcfg12;
    wire        dutcfg12_we_b;

	assign	    dutcfg12_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_DUTCFG12)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_cfg_regs_b)
		if (!rst_cfg_regs_b)			dutcfg12 <= { sw_gpin, 
                                                      sw_gpin, 
                                                      cpu_id,
                                                      2'b11
                                                    };
		else if (!dutcfg12_we_b)        dutcfg12 <= reg_wdata;


//: </sect end_dutcfg>


//===========================================================================
//                              CORE MANAGEMENT SYSTEM
//===========================================================================
//: <sect level=2  title="Core Management Space Registers"  template="sect_cms">
//
// CMS: On the RDB it is only used for switch accessing.


//---------------------------------------------------------------------------
//: <reg name=CMSA      raddr=0x0D8  access="RW"  reset=ARST  title="Core Management Address">
//:   The CMSA register selects one of the internal core management registers\
//:   within Qixis for subsequent read- or write-access via the CMSD register.
//:   For RDB systems, only the switch-access registers are defined.
//:   <bit name=ADDR      size=8  access="RW"  reset=00000000>
//:     Select internal CMS register for read/write via CMS_D, one of:
//:     `0x00-0x0F : SW#      : Number of switches the system has.`
//:     `0x01-0x0F : SW[1:n]  : Image of SW1..SWn`
//:     Ranges not listed are reserved.
//:   </bit>
//:   <note>
//:     A standard use of the CMSA/CMSD port is to read the state of\
//:     configuration switches, as follows:
//:     `Qixis_Set_Reg( CMS_A, 0x00 );`
//:     `nr = Qixis_Get_Reg( CMS_D );`
//:     `for (i = 1; i \<= nr; i++) {`
//:     `   Qixis_Set_Reg( CMS_A, i );`
//:     `   printf(\"SW%1d = %02X\\ n\", i, Qixis_Get_Reg( CMS_D ));`
//:     `}`
//:   </note>
//: </reg>
//
    reg [7:0]   cms_data;
    wire        cms_addr_we_b;

	assign	    cms_addr_we_b	= (!reg_cs_b  &&  !reg_we_b &&  (reg_addr == ADDR_CMSA)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_ctl_regs_b)
		if (!rst_ctl_regs_b)			cms_addr <= 8'h00;
		else if (!cms_addr_we_b)        cms_addr <= reg_wdata;


//---------------------------------------------------------------------------
//: <reg name=CMSD      raddr=0x0D9  access="RW"  reset=ARST  title="Core Management Data">
//:   CMSD contains the value of a CMS register selected by CMSA.  See CMSA for details.
//:   <bit name=DATA      size=8  access="RW"  reset=00000000>
//:     Read/write internal CMS registers selected with CMS_A.
//:   </bit>
//: </reg>

	assign	    cms_cs_b	= (!reg_cs_b  &&                (reg_addr == ADDR_CMSD)) ? 1'b0 : 1'b1;
	assign	    cms_we_b	= (!reg_cs_b  &&  !reg_we_b &&  (reg_addr == ADDR_CMSD)) ? 1'b0 : 1'b1;

	always @*
		case (cms_addr)			    // synthesis parallel_case
        8'd0:       cms_data <= 8'd5;
        8'd1:       cms_data <= isw1;
        8'd2:       cms_data <= isw2;
        8'd3:       cms_data <= isw3;
        8'd4:       cms_data <= isw4;
        8'd5:       cms_data <= isw5;

		// Register access tracking.
		//
		8'hE0:		cms_data <= rac;
		8'hE1:		cms_data <= rwc;

        default:    cms_data <= 8'd0;
        endcase


//: </sect end_cms>


//===========================================================================
//                              SWITCH CONTROL
//===========================================================================
//: <sect level=2  title="Switch Manager Registers"  template="sect_switch">
//: For the LX2160ARDB, a limited subset of the QDS switch management system\
//: is provided to allow software to override all switches.


//---------------------------------------------------------------------------
//: <reg name=SWS_CTL   raddr=0x0DC  access="RW"  reset=ARST  title="Switch Control">
//:   The SWS_CTL register manages the switch sampler.
//:   <bit name=-         size=1  access="RW"  reset=0></bit>
//:   <bit name=-         size=4  access="RW"  reset=0000></bit>
//:   <bit name=POLL      size=1  access="RW"  reset=1>
//:     Poll Switches:
//:     0= Polling is disabled.
//:     1= Polling is enabled; the switches are sampled continually.
//:  </bit>
//:  <bit name=-         size=1  access="RW"  reset=0></bit>
//:  <bit name=-         size=1  access="RW"  reset=1></bit>
//: </reg>

	wire	sws_ctl_we_b;

	assign	sws_ctl_we_b	= (!reg_cs_b  &&  !reg_we_b  &&  (reg_addr == ADDR_SWS_CTL)) ? 1'b0 : 1'b1;

	always @(posedge clk  or  negedge rst_b)
		if (!rst_b)					sws_ctl <= { 1'b0, 
                                                 4'b0000, 
                                                 1'b1,              // POLL=1
                                                 1'b0,              // NOW=0
                                                 1'b1               // IRQ=1
                                               };
		else if (!sws_ctl_we_b)  	sws_ctl <= reg_wdata;

	assign	sws_en	= |(sws_ctl[2:0]);


//---------------------------------------------------------------------------
//: <reg name=SWS_STAT  raddr=0x0DD  access="RO"  reset=NONE  title="Switch Sample Status">
//:   SWS_STAT reports on update activity from the serial switch sampler.
//:   <bit name=UPD       size=1  access="RW"  reset=1>
//:     Updated:
//:     0= (reserved)
//:     1= The switches were updated. 
//:   </bit>
//:   <bit name=-         size=6  access="RW"  reset=000000></bit>
//:   <bit name=-         size=1  access="RW"  reset=0></bit>
//: </reg>

	wire	[7:0]	sws_stat;

	assign	sws_stat	= { sws_en,
							6'b000000,
							1'b0
						  };


//: </sect end_switch>


//: </sect end_intro>


//===========================================================================
//===========================================================================


//---------------------------------------------------------------------------
// Read mux
// 
	always @*
		case (reg_addr)			    // synthesis parallel_case
		ADDR_ID						:	reg_rdata <= id;
		ADDR_VER					:	reg_rdata <= ver;
		ADDR_QVER					:	reg_rdata <= qver;
		ADDR_MODEL					:	reg_rdata <= model;
        ADDR_QTAG                   :   reg_rdata <= minor;

		ADDR_CTL				    :	reg_rdata <= ctl;
		ADDR_AUX 					:	reg_rdata <= aux;
        ADDR_STAT_SYS				:	reg_rdata <= stat_sys;
		ADDR_ALARM					:	reg_rdata <= (ctl_amask) ? alarm_mask : alarm;
        ADDR_STAT_PRES1				:	reg_rdata <= stat_pres1;
        ADDR_STAT_PRES2				:	reg_rdata <= stat_pres2;
		ADDR_LED    				:	reg_rdata <= leds;

        ADDR_RCFG   		        :	reg_rdata <= rcfg;
        ADDR_LOS_STAT  		        :	reg_rdata <= los_stat;
        ADDR_WATCH                  :   reg_rdata <= rcfg_watch;

		ADDR_PWR_CTL2				:	reg_rdata <= { pwr_ctl2_pwr, 7'd0 };
		ADDR_PWR_EVENT              :   reg_rdata <= pwr_reason;
        ADDR_PWR_MSTAT		        :	reg_rdata <= pwr_mstat;
        ADDR_PWR_STAT1		        :	reg_rdata <= pwr_stat1;
        ADDR_PWR_STAT2		        :	reg_rdata <= pwr_stat2;
        ADDR_PWR_STAT3		        :	reg_rdata <= 8'b1111_1111;

        ADDR_RST_CTL	            :	reg_rdata <= rst_ctl;
        ADDR_RST_STAT	            :	reg_rdata <= rst_stat;
        ADDR_RST_REASON             :   reg_rdata <= rst_reason;
        ADDR_RST_FORCE1	            :	reg_rdata <= rst_force1;
        ADDR_RST_FORCE2	            :	reg_rdata <= rst_force2;
        ADDR_RST_FORCE3	            :	reg_rdata <= rst_force3;
		ADDR_RST_MASK1              :   reg_rdata <= rst_mask1;
		ADDR_RST_MASK2              :   reg_rdata <= rst_mask2;
		ADDR_RST_MASK3              :   reg_rdata <= rst_mask3;

   		ADDR_CLK_SPD1				:	reg_rdata <= clk_spd1;
		ADDR_CLK_ID 				:	reg_rdata <= clk_id;

        ADDR_BRDCFG0				:	reg_rdata <= brdcfg0;
		ADDR_BRDCFG1				:	reg_rdata <= brdcfg1;
        ADDR_BRDCFG2				:	reg_rdata <= brdcfg2;
        ADDR_BRDCFG4				:	reg_rdata <= brdcfg4;
		ADDR_BRDCFG5				:	reg_rdata <= brdcfg5;
		ADDR_BRDCFG6				:	reg_rdata <= brdcfg6;
		ADDR_BRDCFG9				:	reg_rdata <= brdcfg9;

        ADDR_DUTCFG0				:	reg_rdata <= dutcfg0;
        ADDR_DUTCFG1				:	reg_rdata <= dutcfg1;
        ADDR_DUTCFG2				:	reg_rdata <= dutcfg2;
        ADDR_DUTCFG6				:	reg_rdata <= dutcfg6;
        ADDR_DUTCFG11				:	reg_rdata <= dutcfg11;
        ADDR_DUTCFG12				:	reg_rdata <= dutcfg12;

        ADDR_IRQSTAT0			    :	reg_rdata <= irq_stat0;
        ADDR_IRQSTAT1			    :	reg_rdata <= irq_stat1;

        ADDR_CMSA					:	reg_rdata <= cms_addr;
		ADDR_CMSD					:	reg_rdata <= cms_data;

		ADDR_SWS_CTL				:	reg_rdata <= sws_ctl;
		ADDR_SWS_STAT				:	reg_rdata <= sws_stat; 

		default						:	reg_rdata <= 8'H00;
		endcase



//===========================================================================
// CONFIG
//      Map register values to config data.
//
// IFC_AD contains GPCFG + RCWSRC bits.  The order of the bits varies by
// device; see the mux at the IFC_AD output driver level.
//
    assign  cfg_ifc_ad[15:8]    = dutcfg0;          // cfg_rcw_src[8:1]
    assign  cfg_ifc_ad[ 7:0]    = dutcfg12;         // cfg_gpcfg[7:0]

// IFC_A is not used except for "cfg_svr[0:1]".  Externally there are pullups on
// the unused pins for handling non-essential "cfg_*" pins.
//
	assign	cfg_ifc_a5	        = dutcfg1[ 7];      // cfg_dram_type
	assign	cfg_ifc_a1	        = dutcfg2[ 1];      // cfg_svr[1]
	assign	cfg_ifc_a0	        = dutcfg2[ 2];      // cfg_svr[0]

// These configs are common across devices.
//
	assign	cfg_ifc_cle         = dutcfg1[ 0];
	assign	cfg_ifc_we0_b       = dutcfg11[7];
	assign	cfg_ifc_oe_b        = dutcfg11[6];
	assign	cfg_ifc_te          = dutcfg1[ 4];
	assign	cfg_ifc_wp_b        = dutcfg11[5];

	assign	cfg_rsp_dis         = dutcfg6[ 7];
	assign	cfg_testsel_b       = dutcfg2[ 0];

    assign  cfg_cfg_wp          = sw_cfg_wp;

endmodule
