// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2014 Freescale Semiconductor, Inc
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           clock_si5341b.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR                DESCRIPTION
// 1.0      2011-10-14  Gary Milliorn         New
// -----------------------------------------------------------------------------
// KEYWORDS:            CLOCK SI5341B
// -----------------------------------------------------------------------------
// PURPOSE:             Sends up/down pulses to convert
// -----------------------------------------------------------------------------
// PARAMETERS
//                      HW_DEFAULT          default clock output value (HW based).
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:	 	<none>
// Clock Domains:		<none>
// Critical Timing:		<none>
// Test Features:		<none>
// Asynchronous I/F:	<none>
// Scan Methodology:	<none>
// Instantiations:		
// Synthesizable (y/n):	y
// Other:
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module clock_si5341b #( 
    parameter CLK_FREQ          = 66666,
    parameter [9:0] HW_DEFAULT  = 10'd100
) (

    // Reset controller interface.
    //
    input   wire        clk_go_b,
    output  reg         clk_ack_b,

    // Debug/Manual Control interface
    //
    input   wire        reg_go,
    input   wire  [9:0] reg_sysclk,
    input   wire        reg_fdec,
    input   wire        reg_finc,

    // SI5341B interface.
    //
    output  wire        clk_rst_b,
    output  wire        clk_fdec,
    output  wire        clk_finc, 

    // Config data.
    //
    input   wire        disable_io_b,
    input   wire        rst_b,
    input   wire        clk_10kHz_ena,
    input   wire        clk
);


//---------------------------------------------------------------------------
// Delay timer (post clock reset).
//
    reg  [15:0]  clkp_timer, next_clkp_timer;
    wire         clkp_timer_zero;

    localparam  [15:0]  CLKP_TIMER_INIT     = 16'd25000,          // 250 ms
                        CLKP_TIMER_LOCK     = 16'd15000;

// Update timer always.  Async block stops if =0000.
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             clkp_timer  <= 15'd0;
        else if (clk_10kHz_ena) clkp_timer  <= next_clkp_timer;

    assign clkp_timer_zero = (clkp_timer == 16'd0) ? 1'b1 : 1'b0;


//===========================================================================
// Clock changes are calculated as a +/- pulses from the known-good-reset
// value, which is supplied as a parameter.  Since we operate in 0.333MHz
// steps, multiply the default by 3 as well (compile time multiply).
//
    wire    [9:0]   hw_default_3x;
//	wire            is_default;

    assign	hw_default_3x   = (10'd3 * HW_DEFAULT);
//  assign  is_default      = (reg_sysclk == hw_default_3x);

	 
//===========================================================================
// FSM.
//
    localparam      IDLE     = 4'h00,
                    ACK      = 4'h01,
                    RESET    = 4'h02,
                    RESETX   = 4'h03,
                    RESETDLY = 4'h04,
                    COMPARE  = 4'h05,
                    UP       = 4'h06,
                    DOWN     = 4'h07;
                    
// Sync elements.
//
    reg [3:0]   state,      next_state;
    reg                     next_clk_ack_b;
    reg         cclk_rst_b, next_cclk_rst_b;
    reg         cclk_fdec,  next_cclk_fdec;
    reg         cclk_finc,  next_cclk_finc;
    reg [9:0]   clk_curr,   next_clk_curr;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             state       <= IDLE;
        else if (clk_10kHz_ena) state       <= next_state;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             clk_ack_b   <= 1'b1;
        else if (clk_10kHz_ena) clk_ack_b   <= next_clk_ack_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             cclk_rst_b   <= 1'b1;
        else if (clk_10kHz_ena) cclk_rst_b   <= next_cclk_rst_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             cclk_fdec    <= 1'b0;
        else if (clk_10kHz_ena) cclk_fdec    <= next_cclk_fdec;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             cclk_finc    <= 1'b0;
        else if (clk_10kHz_ena) cclk_finc    <= next_cclk_finc;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             clk_curr    <= hw_default_3x;
        else if (clk_10kHz_ena) clk_curr    <= next_clk_curr;

// Async process.
//
    always @* begin
        next_state          <= state;
        next_clk_ack_b      <= clk_ack_b;
        next_cclk_rst_b     <= cclk_rst_b;
        next_cclk_fdec      <= cclk_fdec;
        next_cclk_finc      <= cclk_finc;
        next_clk_curr       <= clk_curr;
        next_clkp_timer     <= (|clkp_timer) ? clkp_timer - 16'd1 : clkp_timer;

        case (state)                                        // pragma parallel_case   

//---------------------------------------------------------------------------
// DOWN:
//      Step down.
//
        DOWN: begin
            next_cclk_fdec  <= 1'b1;
            next_clk_curr   <= clk_curr - 10'd1;
            next_state      <= COMPARE;
        end

// UP:
//      Step up.
//
        UP: begin
            next_cclk_finc  <= 1'b1;
            next_clk_curr   <= clk_curr + 10'd1;
            next_state      <= COMPARE;
        end

//---------------------------------------------------------------------------
// COMPARE:
//      Compare the current frequency to the target.
//
        COMPARE: begin
            next_cclk_fdec  <= 1'b0;
            next_cclk_finc  <= 1'b0;
            next_cclk_rst_b <= 1'b1;

            if (clk_curr < reg_sysclk)
                next_state  <= UP;
            else if (clk_curr > reg_sysclk)
                next_state  <= DOWN;
            else
                next_state  <= ACK;
        end

//---------------------------------------------------------------------------
// RESET:
//      Assert reset to device.   While resetting, setup current frequency
//  register (=DEFAULT, since we reset it).
//
        RESET: begin
            next_cclk_rst_b <= 1'b0;
            next_clk_curr   <= hw_default_3x;
            next_state      <= RESETX;
        end
        RESETX: begin
            next_cclk_rst_b <= 1'b0;
            next_clk_curr   <= hw_default_3x;
            next_state      <= RESETDLY;
        end
        RESETDLY: begin
            next_cclk_rst_b <= 1'b1;
            next_state      <= (clkp_timer_zero) ? COMPARE : RESETDLY;
        end

//---------------------------------------------------------------------------
// ACK:
//      Done.  Assert "ack_b" to host until strobe removed.
//
        ACK: begin
            next_clk_ack_b  <= 1'b0;
            next_state      <= (!clk_go_b  ||  reg_go) ? ACK : IDLE;
        end

// IDLE:
//      Watch for a signal from the reset controller; start the operation when done.
//
        IDLE: begin
            next_clk_ack_b  <= 1'b1;
            next_cclk_rst_b <= 1'b1;
            next_cclk_fdec  <= 1'b0;
            next_cclk_finc  <= 1'b0;
            next_clk_curr   <= 1'b0;
            next_state      <= (!clk_go_b  ||  reg_go) ? RESET : IDLE;
        end

// (recovery)
//      Reset and go to idle.
//
        default: begin
            next_state  <= IDLE;
        end
        endcase
    end


//---------------------------------------------------------------------------
// Control SI5341 pins with FSM outputs, or user-managed register bits.  All
// must be off when powered down.
//
    assign  clk_rst_b   =                            cclk_rst_b;
    assign  clk_fdec    = (!disable_io_b)   ? 1'bZ : cclk_fdec  | reg_fdec;
    assign  clk_finc    = (!disable_io_b)   ? 1'bZ : cclk_finc  | reg_finc;

endmodule
