// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2016 NXP Semiconductors
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
// -----------------------------------------------------------------------------
// FILE NAME:       	debouncer.v
// DEPARTMENT:      	Austin Hardware Design
// AUTHOR:          	Gary Milliorn
// AUTHOR'S EMAIL:  	gary.milliorn@nxp.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE       	AUTHOR                DESCRIPTION
// 1.0      2013-10-17 	Gary Milliorn         New
// -----------------------------------------------------------------------------
// KEYWORDS:    		DEBOUNCE SWITCH
// -----------------------------------------------------------------------------
// PURPOSE:     		Simple digital filtering for noisy mechanical switches.
// -----------------------------------------------------------------------------
// PARAMETERS
//                      CLK_FREQ            core clock in kHz.
//                      SIM_MODE            set to 1 to accelerate simulation
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:      rst_b:   asynchronous, active low
// Clock Domains:       clk:     system main clock
// Critical Timing:     <none>
// Test Features:       <none>
// Asynchronous I/F:    <none>
// Scan Methodology:    <none>
// Instantiations:      <none>
// Synthesizable (y/n): Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module debouncer #(
   parameter CLK_FREQ = 33333,
   parameter SIM_MODE = 0
)
( 
   input   wire      clk, 
   input   wire      clk_db_ena, 
   input   wire      rst_b, 

   input   wire      sw_b, 
   output  reg       out_b
);

    reg [3:0]   sw_samples;


//---------------------------------------------------------------------------
// Sample the switch signal at the sampling rate.
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             sw_samples <= 4'b1111;
        else if (clk_db_ena)    sw_samples <= { sw_samples[2:0], sw_b };


//---------------------------------------------------------------------------
// Change the output when the samples are all 1's, or all 0's.
//
    wire    next_out_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             out_b <= 1'b1;
        else if (clk_db_ena)    out_b <= next_out_b;
   
    assign  next_out_b  = (sw_samples == 4'b1111)   ? 1'b1
                        : (sw_samples == 4'b0000)   ? 1'b0
                        :                             out_b;    

endmodule 
