// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2015 Freescale Semiconductor, Inc. All rights reserved
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           i2c_remote
// DEPARTMENT:          SD646
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// CREATED:             2014-0530
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR                DESCRIPTION
// 1.0      2015-08-12  Gary Milliorn
// -----------------------------------------------------------------------------
// KEYWORDS:            I2C RMT ACCESS IRMT
// -----------------------------------------------------------------------------
// PURPOSE:             Converts I2C accesses from an I2C slave IP block into internal 
//                      register accesses.  
// -----------------------------------------------------------------------------
// PARAMETERS
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:      active low async
// Clock Domains:        clk:        FPGA core clock.
// Critical Timing:     <none>
// Test Features:       <none>
// Asynchronous I/F:    <none>
// Scan Methodology:    <none>
// Instantiations:       
// Synthesizable (y/n): y
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module i2c_remote (
   input   wire [7:0]   i2c_addr, 
   input   wire         is_i2c_ctl, 
   input   wire         i2c_rd, 
   input   wire         i2c_wr, 
   output  wire [7:0]   i2c_rdata, 
   input   wire [7:0]   i2c_wdata, 

   output  wire [7:0]   reg_addr, 
   output  wire         reg_cs_b, 
   output  wire         reg_we_b, 
   input   wire [7:0]   reg_rdata, 
   output  wire [7:0]   reg_wdata, 

   output  wire         activity_b,
   input   wire         rst_b, 
   input   wire         clk
);


//---------------------------------------------------------------------------
// Capture addr on I2C RD/WR events.
//
    reg  [7:0]  addr_hold;

    always @(negedge rst_b  or  negedge clk)
    if (!rst_b)                 addr_hold <= 8'h00; 
    else if (i2c_rd | i2c_wr)   addr_hold <= i2c_addr;


//---------------------------------------------------------------------------
// FSM Controls.
//
    reg [3:0]   state,      next_state;         // pragma attribute state safe_fsm TRUE 
                                                // pragma attribute state fsm_state ONEHOT

    reg [7:0]   caddr,      next_caddr;         // Common addr for int. or ext. registers.
    reg [7:0]   rdata,      next_rdata;
    reg [7:0]   wdata,      next_wdata;
    reg         rcs_b,      next_rcs_b;
    reg         rwe_b,      next_rwe_b;         // Write BCSRs.
    reg         cwe_b,      next_cwe_b;         // Write our regs.

    wire [7:0]  ctl_rdata;                      // Internal registers

    localparam      IDLE        = 4'd0,
                    CREAD       = 4'd1,
                    QREAD       = 4'd2,
                    QRD         = 4'd3,
                    QRDCAP      = 4'd4,
                    QRDEND      = 4'd5,
                    DOWR        = 4'd6,
                    WRCAP       = 4'd7,
                    CWRITE      = 4'd8,
                    CWRITEX     = 4'd9,
                    QWRITE      = 4'd10,
                    QWR         = 4'd11,
                    QWEND       = 4'd12;

//---------------------------------------------------------------------------
// FSM sync block.
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     state <= IDLE;
        else            state <= next_state;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     caddr <= 'd0;
        else            caddr <= next_caddr;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     rdata <= 'd0;
        else            rdata <= next_rdata;
    assign i2c_rdata    = rdata;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     wdata <= 'd0;
        else            wdata <= next_wdata;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     rcs_b <= 1'b0;
        else            rcs_b <= next_rcs_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     rwe_b <= 1'b0;
        else            rwe_b <= next_rwe_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)     cwe_b <= 1'b0;
        else            cwe_b <= next_cwe_b;


//---------------------------------------------------------------------------
// FSM async block.
//
	always @* begin 
        next_state      <= state;
        next_caddr      <= caddr;
        next_rdata      <= rdata;
        next_wdata      <= wdata;
        next_rcs_b      <= 1'b1;
        next_rwe_b      <= 1'b1;
        next_cwe_b      <= 1'b1;

		case (state)										// pragma parallel_case
        
// CREAD: read one of our control registers.
//
        CREAD: begin
            next_caddr      <= addr_hold;
            next_rdata      <= ctl_rdata;
            next_state      <= IDLE;
        end

// 2|QREAD:     I2C read of targeted destination
//
		QREAD: begin
            next_caddr		<= addr_hold;
			next_rcs_b		<= 1'b0;
			next_rwe_b		<= 1'b1;
            next_rdata      <= reg_rdata;
            next_state      <= QRD;
			end

// 3|QRD:       stall until read is completed.
//
		QRD: begin
			next_rcs_b      <= 1'b0;
			next_rwe_b	    <= 1'b1;
			next_rdata		<= reg_rdata;
            next_state      <= QRDCAP;
		end

// 4|QRDCAP:    Capture read data.
//
		QRDCAP: begin
			next_rcs_b      <= 1'b0;
			next_rwe_b	    <= 1'b1;
			next_rdata		<= reg_rdata;
            next_state      <= QRDEND;
        end

// 5|QRDEND:    Complete read. Do not exit until I2C and register ports are idle.
//
		QRDEND: begin
			next_rcs_b      <= 1'b1;
			next_rwe_b	    <= 1'b1;
            next_state      <= (!i2c_rd) ? IDLE : QRDEND;
        end

// 6|DOWR:      Do a write cycle.  Capture data on the fall of "i2c_wr", so we cannot do
//              anything yet.
//
		DOWR: begin
            next_caddr		<= addr_hold;
            next_state      <= (!i2c_wr) ? WRCAP : DOWR;
		end

// 7|WRCAP:     Capture write data, then vector to control or BCSR writing.
//
		WRCAP: begin
            next_wdata		<= i2c_wdata;
            next_state      <= (!is_i2c_ctl) ? QWRITE : CWRITE;
		end

// 8|CWRITE:    Write to I2C control register.
//
		CWRITE: begin
            next_cwe_b		<= 1'b0;
            next_state      <= CWRITEX;
		end

// 9|CWRITEX:   Complete write to I2C control register.
//
		CWRITEX: begin
            next_cwe_b		<= 1'b1;
            next_state      <= IDLE;
		end

// 10|QWRITE:   Write to BCSR register.
//
		QWRITE: begin
            next_wdata		<= i2c_wdata;
            next_rcs_b		<= 1'b0;
            next_rwe_b		<= 1'b0;
            next_state      <= QWR;
		end

// 11|QWR:      Write to BCSR register until ACK received.
//
		QWR: begin
            next_rcs_b		<= 1'b0;
            next_rwe_b		<= 1'b0;
            next_state      <= QWEND;
		end

// 12|QWEND:    Complete BCSR write.
//
		QWEND: begin
            next_rcs_b		<= 1'b1;
            next_rwe_b		<= 1'b1;
            next_state      <= IDLE;
		end

// 13|IDLE:     Preserve addresses and wait for reads or writes.
//
		IDLE: begin
            next_state      <= IDLE;
            next_rcs_b      <= 1'b1;
            next_rwe_b      <= 1'b1;
            next_cwe_b      <= 1'b1;

			if      (i2c_rd  &&  !is_i2c_ctl)	next_state <= QREAD;        // Read BCSR
			else if (i2c_rd  &&   is_i2c_ctl)	next_state <= CREAD;        // Read I2CCTL space
			else if (i2c_wr)	                next_state <= DOWR;         // Write BCSR or I2CCTL.
		end

// X|(recovery):    Jump to IDLE.
//
        default: begin
            next_state      <= IDLE;
        end
        endcase
	end


//---------------------------------------------------------------------------
// Control-plane registers.  On RDB systems, 05..0F are RO.
// by the host.
//
// ====     ====    ====    ==========  =============================
// ADDR     NAME    TYPE    VALUE       DESCRIPTION
// ====     ====    ====    ==========  =============================
//  00      TAG0    RO      0x49 ('I')  Part of space tagging.
//  01      TAG1    RO      0x52 ('R')  "
//  02      TAG2    RO      0x50 ('P')  "
//  03      TAG3    RO      0x58 ('X')  "
//  04      CTL     RW      0x04        b7-3:   reserved.
//                                      b2-0:   DEST (default = BCSR).
//  05      SIZE    RW      0x00        Transfer size: 0=8b
//  06      n/a     RO      0x00        -reserved-
//  07      XDT1    RO      0x00        -reserved-
//  08      EA3     RO      0x00        -reserved-
//  09      EA2     RO      0x00        -reserved-
//  0A      EA1     RO      0x00        -reserved-
//  0B      n/a     RO      0x00        -reserved-
//  0C      n/a     RO      0x00        -reserved-
//  0D      n/a     RO      0x00        -reserved-
//  0E      XDT3    RO      0x00        -reserved-
//  0F      XDT2    RO      0x00        -reserved-
//
    reg [7:0]  ex_ctl;
    wire       ex_ctl_we_b;

    assign ex_ctl_we_b = (!cwe_b  &&  caddr[3:0] == 4'h4) ? 1'b0 : 1'b1;

    always @(negedge clk  or  negedge rst_b)
        if (!rst_b)              ex_ctl = 8'h04;             // default DEST=100=BCSRs
        else if (!ex_ctl_we_b)   ex_ctl = wdata;

    assign ctl_rdata    = (caddr[3:0] == 4'h0) ? 8'h49
                        : (caddr[3:0] == 4'h1) ? 8'h52
                        : (caddr[3:0] == 4'h2) ? 8'h50
                        : (caddr[3:0] == 4'h3) ? 8'h58
                        : (caddr[3:0] == 4'h4) ? ex_ctl
                        :                        8'h00;


//-----------------------------------------------------
// Connect signals to register port.
//
    assign  reg_addr    = caddr;
    assign  reg_wdata   = wdata;
    assign  reg_cs_b    = rcs_b;
    assign  reg_we_b    = (!rcs_b  &&  !rwe_b) ? 1'b0 : 1'b1;


//---------------------------------------------------------------------------
// Debug, unused.
//
    wire    unused;

    assign  activity_b  = (state != IDLE) ? 1'b0 : 1'b1;
    assign  unused      = (| reg_rdata[7:0]);

endmodule
