// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2014-2016 NXP Semiconductors
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// NXP Confidential
// -----------------------------------------------------------------------------
// FILE NAME:           ifc.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@nxp.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR                  DESCRIPTION
// 1.0      2014-10-14  Gary Milliorn           New
// 1.1      2016-05-25  Gary Milliorn           Support for Rev B
// -----------------------------------------------------------------------------
// KEYWORDS :           IFC BCSRS REGS CONFIG
// -----------------------------------------------------------------------------
// PURPOSE :            Manages connections between the IFC port and the 
//                      NOR/NAND devices, and contains a minimal
//                      implementation of the BCSRs (registers) sufficient
//                      to boot u-boot/Linux.
//                      Additionally, since so many pin-sampled configuration
//                      pins are on IFC signals, they are all handled here with
//                      the "xcfg_drv_b" signal (asserted by the reset module).
// -----------------------------------------------------------------------------
// PARAMETERS
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:   asynchronous, active low
// Clock Domains:        clk:     system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module ifc( 

    // IFC interface
    //
    inout   wire    [15:0]  ifc_ad,
    inout   wire            ifc_ale,
    input   wire            ifc_bctl,
    inout   wire            ifc_cle,
    input   wire            ifc_clk,
    input   wire    [2:0]   ifc_cs_b,
    inout   wire            ifc_oe_b,
    output  wire            ifc_rb1_b,
    inout   wire            ifc_te,
    inout   wire            ifc_we0_b,
    inout   wire            ifc_wp_b,

    output  wire            drv_ifc_a5,
    output  wire            drv_ifc_a1,
    output  wire            drv_ifc_a0,
    output  wire            nand_cs_b,

    // Register interface
    //
    output  wire    [7:0]   ifcreg_addr,
    output  wire            ifcreg_cs_b,
    output  wire            ifcreg_oe_b,
    output  wire            ifcreg_we_b,
    input   wire    [7:0]   ifcreg_rdata,
    output  wire    [7:0]   ifcreg_wdata,
	input	wire			ifcreg_drv_b,			// Assert on IFC register reads.

    // Config inputs
    //
    input   wire    [1:0]   cpu_id,

    // Reset handling.
    //
    inout   wire            dut_reset_req_b, 
    output  wire            int_reset_req_b, 
    inout   wire            dut_asleep, 
    output  wire            int_asleep,

    // Config Controls
    //
    input   wire            cfg_drv_b,

    input	wire    [15:0]  cfg_ifc_ad,
    input	wire            cfg_ifc_a5,
    input	wire            cfg_ifc_a1,
    input	wire            cfg_ifc_a0,

    input	wire            cfg_ifc_cle,
    input	wire            cfg_ifc_oe_b,
    input	wire            cfg_ifc_te,
    input	wire            cfg_ifc_we0_b,
    input	wire            cfg_ifc_wp_b,

	input	wire			cfg_rsp_dis,

	// Monitoring
	//
	output	wire			mon_boot_b,

    // General
    //
    input   wire            disable_io_b, 
    input   wire            rst_b, 
    input   wire            clk
);


// Internal Declarations
//
	reg     [15:0]  ifc_addr;  
    wire            ifc_bcsr_cs_b;
    wire    [7:0]   ifc_rdata;
    wire    [7:0]   ifc_wdata;


//---------------------------------------------------------------------------
// ALE
//   Drive with CFG_RSPDIS.
//
	assign  ifc_ale		= (!cfg_drv_b)		? cfg_rsp_dis
						:					  1'bz;


//---------------------------------------------------------------------------
// IFC_AD: 
//  Drive with GPIN & RCW_SRC[0:7].  Otherwise, bidirectional data and MSB of address.
//  Only drive with NAND or BCSR accesses; others are tri-stated.
//      IFC_AD[15:8]    = cfg_rcw_src[8:1]
//      IFC_AD[ 7:0]    = cfg_gpin[7:0]
//
    wire [15:0] ifc_ad_in,  ifc_ad_mux,  ifc_ad_map;
    wire        ifc_ad_oe;
    wire        ifc_endian;

    assign  ifc_ad_oe   = (!disable_io_b)   ? 1'b0								// No drive if off
                        : (!cfg_drv_b)      ? 1'b1								// Drive internal config
                        : (!ifcreg_drv_b)   ? 1'b1								// Drive BCSR read.
                        :                     1'b0;								// Other, no drive.

    assign  ifc_ad_mux  = (!cfg_drv_b)      ? { cfg_ifc_ad }					// Internal cfg drive
				        : (!ifcreg_drv_b)	? { ifcreg_rdata, ifcreg_rdata }	// BCSR/reg read
                        :                     16'd0;

// Note that the LS1043 has IFC_A[0:7] crossed with the LS1088 IFC_A[7:0], so these
// must be corrected for that one case (for both config and normal IO).
//
    assign  ifc_endian  = (cpu_id == 2'b01) ? 1'b1                      // LS1043
                        : (cpu_id == 2'b10) ? 1'b1                      // LS1046/LS1047
                        :                     1'b0;                     // LS1088

    assign  ifc_ad_map  = (ifc_endian)      ? { ifc_ad_mux[ 8],         // LS104x mode
                                                ifc_ad_mux[ 9],
                                                ifc_ad_mux[10],
                                                ifc_ad_mux[11],
                                                ifc_ad_mux[12],
                                                ifc_ad_mux[13],
                                                ifc_ad_mux[14],
                                                ifc_ad_mux[15],
                                                ifc_ad_mux[7:0]
                                              }
                        :                     ifc_ad_mux;

	assign  ifc_ad      = (ifc_ad_oe)       ? ifc_ad_map
                        :                     16'hzzzz;
    assign  ifc_ad_in   = (ifc_ad_oe)       ? 16'd0
                        :                     ifc_ad;

	assign  ifc_wdata   = (!disable_io_b)    ? 8'h00 
                        : (ifc_endian)       ? { ifc_ad_in[ 8],
                                                 ifc_ad_in[ 9],
                                                 ifc_ad_in[10],
                                                 ifc_ad_in[11],
                                                 ifc_ad_in[12],
                                                 ifc_ad_in[13],
                                                 ifc_ad_in[14],
                                                 ifc_ad_in[15]
                                               }
                        :                      ifc_ad_in[ 7:0];


//---------------------------------------------------------------------------
// IFC_A is not available in QSPI mode; it is still a config pin, though.
//
    assign  drv_ifc_a5   = (!cfg_drv_b) ? cfg_ifc_a5 : 1'bZ;
    assign  drv_ifc_a1   = (!cfg_drv_b) ? cfg_ifc_a1 : 1'bZ;
    assign  drv_ifc_a0   = (!cfg_drv_b) ? cfg_ifc_a0 : 1'bZ;


//---------------------------------------------------------------------------
// Latch the upper addresses from data during ALE=1.
// Note that IFC 2.0 does not output IFC_CLK all the time (only for NAND), 
// so there's no other synchronous way to do this.
//
	always @(negedge rst_b  or  negedge ifc_ale)
		if (!rst_b)     ifc_addr <= 16'h0000;
		else            ifc_addr <= ifc_ad;


//---------------------------------------------------------------------------
// OE - Output Enable
//      OE supports cfg_eng_use[1].
//
	assign ifc_oe_b	    = (!cfg_drv_b)		? cfg_ifc_oe_b
						:					  1'bz;

// TE - Termination Enable
//      TE supports cfg_ifc_te;
//
	assign ifc_te		= (!cfg_drv_b)		? cfg_ifc_te
						:				      1'bz;

// WE0_B - write enable 0 (MSB)
//		WE0_B supports cfg_eng_use[0]
//
	assign ifc_we0_b    = (!cfg_drv_b)		? cfg_ifc_we0_b
						:					  1'bz;

// CLE/WE1_B
//      CLE supports cfg_rcw_src[0] (LSB)
//
	assign ifc_cle      = (!cfg_drv_b)		? cfg_ifc_cle
					    :					  1'bz;

// WP_B - write protect
//		WP_B supports cfg_eng_use[2]
//
	assign ifc_wp_b 	= (!cfg_drv_b)		? cfg_ifc_wp_b
						:					  1'bz;


//---------------------------------------------------------------------------
// Note: the following two stragglers are not part of the IFC, but are part of
//       the hardware configuration, so the CFG drive is handled here.
//
// ASLEEP - must be 1.
//      Config is "cfg_soc_use".  No current definition, park at '1'.
//
    assign  dut_asleep      = (!disable_io_b)   ? 1'bz
                            : (!cfg_drv_b)      ? 1'b1
                            :                     1'bz;

    assign  int_asleep      = (!disable_io_b)   ? 1'b1
                            : (!cfg_drv_b)      ? 1'b1
                            :                     dut_asleep;


// RESET_REQ_B
//      Config is "cfg_test_port_mux_sel", which must always be 1.  There is an
//      external pullup, but enforce it here as well so "int_reset_req_b" stays
//      valid.
//
	assign  dut_reset_req_b = (!disable_io_b)   ? 1'bz
                            : (!cfg_drv_b)		? 1'b1
							:					  1'bz;

    assign  int_reset_req_b = (!disable_io_b)   ? 1'b1
                            : (!cfg_drv_b)		? 1'b1
                            :                     dut_reset_req_b;



//---------------------------------------------------------
// IFC mode is legacy-only, so the address map is completely fixed.
// CS0=NAND
// CS2=BCSRs.
//
    assign	nand_cs_b		= (!disable_io_b)   ? 1'bZ : ifc_cs_b[0];
    assign  ifc_bcsr_cs_b   = (!disable_io_b)   ? 1'b1 : ifc_cs_b[2];


//---------------------------------------------------------
// IFC-mode register interface.
//
	assign	ifcreg_addr		= ifc_addr[7:0];
	assign	ifcreg_cs_b		= ifc_bcsr_cs_b;
	assign	ifcreg_oe_b		= ifc_oe_b;
	assign	ifcreg_we_b		= ifc_we0_b;
	assign	ifcreg_wdata	= ifc_wdata;

//---------------------------------------------------------
// Activity monitor.
//
    assign  mon_boot_b		= ifc_cs_b[0];


//---------------------------------------------------------
// R/B - Ready/Busy.
//      RB1 is available, but unused currently.
//
	assign ifc_rb1_b    = (!disable_io_b)		? 1'bz
                        :                         1'b1;
		
endmodule
