// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2015 Freescale Semiconductor, Inc. 
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           monitor.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR               DESCRIPTION
// 1.0      2015-06-30  Gary Milliorn        New
// -----------------------------------------------------------------------------
// KEYWORDS:            MONITOR SYSTEM LED
// -----------------------------------------------------------------------------
// PURPOSE:             Monitors various signals and drives status LEDs.
// -----------------------------------------------------------------------------
// PARAMETERS		
//                      CLK_FREQ		    System clock frequency, divided down to 2Hz.
//                      SIM_MODE		    If set, accelerate for simulation.
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:             asynchronous, active low
// Clock Domains:        clk:               33-66 MHz system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module monitor #(
   parameter CLK_FREQ = 33333,
   parameter SIM_MODE = 0
)
( 
	input   wire    [7:0]	alarm_code, 
	input   wire			in_alarm_b, 
	input   wire			therm_fault_b,
	input   wire			therm_warn_b,

	input   wire			ctl_led,
	input   wire			ctl_passfail,
	input   wire    [1:0]	ctl_pwrled,
	input   wire    [3:0]	leds_user,

    // System status.
    //
    input   wire            pwr_is_off,
    input   wire    [7:0]   pwr_state,
    input   wire    [7:0]   pwr_tier_rpt,
    input   wire    [1:0]   sps,
	input	wire			ps_faulted,

    input   wire            rst_in_rst,
    input   wire    [7:0]   rst_state,

	input   wire			poreset_b,
	input   wire			reset_req_b,
	input   wire			asleep,

	input   wire			mon_act0_b,

	output  wire    [3:0]	leds,
	output  wire			led_pass,
	output  wire			led_fail,
	output  wire			led_asleep,
	output  wire			led_porst,
	output  wire			led_rstreq,
	output  wire			led_therm,
	output  wire			led_pwrsw_grn_b,
	output  wire			led_pwrsw_yel_b,

	input	wire	[2:0]	sw_debug,

	input   wire			rst_b,
	input   wire			clk_100Hz_ena,
	input   wire			clk_4Hz_ena,
	input   wire			clk
);


//===========================================================================
// Include standardized definitions and values.
//
`include <qixis_defs.h>


//---------------------------------------------------------------------------
// 2Hz and 100Hz non-critical clocks use for blinking.
//
    reg	pblink, fblink;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pblink <= 1'b0;
        else if (clk_4Hz_ena)   pblink <= ~pblink;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             fblink <= 1'b0;
        else if (clk_100Hz_ena) fblink <= ~fblink;


//---------------------------------------------------------------------------
// Generate the busy pattern, it is arbitrary but intended to allow
// distinguishing between a system completely up and ready vs. one
// still trying to come up.

    reg  [7:0]  idle_stat,      next_idle_stat;
    wire [3:0]  istat;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             idle_stat   <= 8'b1100_0000;
        else if (clk_4Hz_ena)   idle_stat   <= next_idle_stat;
        
    always @*
        next_idle_stat  <= { idle_stat[6:0], idle_stat[7] };


//---------------------------------------------------------------------------
// Startup timer:
//		Count how long the system is in startup.  If greater than 4 seconds,
// there's a problem.
//
	reg	[5:0]	startup_timer;
	wire		starterr;


    always @(posedge clk)
		if (clk_4Hz_ena) begin
			if (!rst_b  ||  (sps != SPS_POWERING))	// Init / No longer in power-up mode.
				startup_timer	<= 6'd0;
			else if (startup_timer[4])				// Timer overflow, stop
				;
			else
				startup_timer <= startup_timer + 6'd1;
		end

	assign	starterr	= startup_timer[4];


//---------------------------------------------------------------------------
// Drive status LEDS as needed.  If Fault, show the code, else show the
// user code.
//
	assign	leds    = (sw_debug == 3'b100)		? { pwr_state[7:4]			}
					: (sw_debug == 3'b101)		? { pwr_state[3:0]			}
					: (sw_debug == 3'b110)		? { pwr_tier_rpt[7:4]		}
					: (sw_debug == 3'b111)		? { pwr_tier_rpt[3:0]		}

					: (starterr    &&  !pblink)	? {		4'b0000				}
					: (starterr    &&   pblink)	? {	  pwr_state[7:4]		}

					: (ps_faulted  &&  !pblink)	? {		4'b1111				}
					: (ps_faulted  &&   pblink)	? {	  pwr_state[3:0]		}

					: (!in_alarm_b)				? { alarm_code[5:3], alarm_code[0] }

					: (sps == SPS_OFF)			? {		4'b0000				}

					: (sps[1])					? {		pwr_state[3:0]		} 
					: (rst_in_rst)				? {	pblink, rst_state[6:4]	} 

                    : (ctl_led)					? {		leds_user			}
					:							  { idle_stat[2],
													idle_stat[1],
													idle_stat[0],
													~mon_act0_b
												  };


//---------------------------------------------------------------------------
// Other status LEDs.
//
    assign  led_pass    = (pwr_is_off)      ? 1'b0
                        :					  ~ctl_passfail;

    assign  led_fail    = (ps_faulted)		? 1'b1			// Power fault
						: (!in_alarm_b)		? 1'b1			// On if fault even if power off
                        : (pwr_is_off)      ? 1'b0
                        :					  ctl_passfail;
                        
    assign  led_asleep  = (pwr_is_off)      ? 1'b0
                        :                     asleep;

    assign  led_porst   = (pwr_is_off)      ? 1'b0
                        :                     ~poreset_b;

    assign  led_rstreq  = (pwr_is_off)      ? 1'b0
                        :                     ~reset_req_b;

// THERM LED is on for FAULT, even if not used to shutdown the system.
// It is also on for WARN, which is only a warning.
//
    assign  led_therm   = (therm_fault_b == 1'b0)
						| (therm_warn_b  == 1'b0);


//---------------------------------------------------------------------------
// The Power and Sleep LEDs are a bidirectional led.  It is controlled with a
// FET "short-to-ground", meaning the outputs are not actually active-low.  
// The truth table is:
//
//      sw_led[0]        sw_led[1]
//      led_pwrsw_grn_b  led_pwrsw_yel_b    |   GREEN   YELLOW
//      ------------------------------------+--------------------------
//          0                   0           |   OFF     OFF
//          0                   1           |   OFF     ON
//          1                   0           |   ON      OFF
//          1                   1           |   OFF     OFF
//
// 1. During power-off, both should be high to disable both LEDs.
// 2. From power-on through power-stable, blink the power/green LED.
// 3. Post power-on until DUT initialized, the amber ASLEEP led should be on.
// 4. If the DUT is successfully initialized, set green solid.
//
    wire    [1:0]   sw_led;

// Both YEL and GRN cannot be on at the same time, so if requested, switch
// between the two rapidly.
//
	wire	[1:0]	map_pwrled;

	assign	map_pwrled	= (ctl_pwrled == 2'b11)	? { fblink, ~fblink }
						:						    ctl_pwrled;

// Manage dual-led power switch.
//													YEL   GRN
//													===	  ===
    assign  sw_led	= (ctl_led)					? {	map_pwrled   }		// SW control :	as specified
					: (sps == SPS_OFF)			? { 1'b0, 1'b0	 }		// power off  :	both off
					: (sps == SPS_POWERING)		? { 1'b0, pblink }		// powering   :	green blink
                    : (starterr | ps_faulted)	? { pblink, 1'b0 }		// startup err: yellow blink
                    : (asleep)					? { 1'b0, 1'b1   }		// asleep     : yellow on
                    :							  { 1'b1, 1'b0   };		// normal     : green on

    assign  led_pwrsw_grn_b = sw_led[0];
    assign  led_pwrsw_yel_b = sw_led[1];

endmodule
