// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// -----------------------------------------------------------------------------
// Copyright (c) 2017 NXP Semiconductors.
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// -----------------------------------------------------------------------------
// FILE NAME:       QX_I2CSlave.v
// DEPARTMENT:      Austin Hardware Design
// AUTHOR:          Gary Milliorn
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE		AUTHOR               DESCRIPTION
// 1.0      2014_0519	Gary Milliorn        New
// 1.1      2017_1012	Gary Milliorn        Work around Mentor Precision 2016
//												bug re: exported FSM state.
// -----------------------------------------------------------------------------
// KEYWORDS:  I2C SLAVE 
// -----------------------------------------------------------------------------
// PURPOSE:	Implements a compliant I2C slave interface, supporting extended
//          address range as well as a unique dual-address concept.
//
//          The QXI2C_Slave can accept any length of data, so it never asserts NAK.  
//          If the parent interface cannot accept data, it should assert "pause_b" 
//          to stall IO; or disable the controller with "enable".
//
//          Read timing: rd_b is asserted for 4 clocks (configurable).
//          Write timing: wr_b is asserted for the same.
//
// Note that rd_b and wr_b are asserted for an I2C bit time.  This can be very
// long, so external pulse limiting may be needed.
// -----------------------------------------------------------------------------
// PARAMETERS           CLK_FREQ            Clock frequency in kHz.
//                      EXT_ADDR            If 1, expect 2-byte addresses.
//                      DUAL_ADDR           If 1, respond to (slv_addr+1) also.
//                      SIM_MODE            If 1, speed up delays, etc.
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:             asynchronous, active low
// Clock Domains:        clk:               system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module QX_I2CSlave #(
   parameter CLK_FREQ  = 33333,
   parameter EXT_ADDR  = 0,
   parameter DUAL_ADDR = 0,
   parameter SIM_MODE  = 0
) (
    // Config. and control.
    input   wire            clk,
    input   wire            rst_b,
    input   wire            enable,         // 1: enable controller.
    input   wire    [ 6:0]  slv_addr,       // Slave address (or base if dual).
    output  wire    [ 7:0]  slv_stat,       // General status.

    // Controller interface.
    //
    output  reg             addr_sel,       // 1 if alternate address (DUAL_ADDR-only)
    output  wire    [15:8]  ext_addr,       // MSB of extended address.  0's if non-extended.
    output  wire    [ 7:0]  reg_addr,       // 8-bit (non-extended) address.
    input   wire            pause_b,        // Asserted to stall I2C bus.

    // Read interface.
    //
    input   wire    [ 7:0]  reg_rdata,      // 8-bit bus for reading registers
    output  reg             reg_rd_b,       // Asserted during start of read operation, data
                                            // must be valid on rising edge.  Controller may
                                            // assert "pause_b" if completion is not possible.

    // Write interface.
    //
    output  reg     [7:0]   reg_wdata,      // Write data output.
    output  reg             reg_we_b,       // Strobe, low during valid write data.

    // I2C I/O.
    //
    output  wire            sda_drv,        // 1: drive low; 0: float high.
    input   wire            sda_rcv,
    output  wire            scl_drv,        // 1: drive low; 0: float high.
    input   wire            scl_rcv
);


//---------------------------------------------------------------------------	
// I2C_Slave timing.  
//      The I2C_Slave I/P needs to be no more than 1/4 of coreclk.  For margin and
//      for debug, limit speed to around 5 MHz.
//      The I2C bus speed is set by the master.
//
    reg		    iclk_en; 
    reg  [ 4:0] iclk_div; 
    wire [ 4:0] ilimit; 

    generate 
        if (CLK_FREQ/4 > 0) begin
            assign ilimit = (CLK_FREQ/4 > 0);
        end
        // else synthesis error.
    endgenerate

    always @(negedge rst_b  or  posedge clk) begin
        if (!rst_b) begin
            iclk_en  <= 1'b0;
            iclk_div <= 5'd0;

        end else if (iclk_div == ilimit) begin
            iclk_en	 <= 1'b1;
            iclk_div <= 5'd0;

        end else begin
            iclk_en  <= 1'b0;
            iclk_div <= iclk_div + 5'd1;
        end
    end


//---------------------------------------------------------------------------
// Slave address selection.
//      For non-dual-mode instantiations, the addresses are set to the same
//      value.
//
	wire	[6:0]		slv_addr_A, slv_addr_B;

    generate
        if (DUAL_ADDR) begin    
            assign  slv_addr_A  = slv_addr;
            assign  slv_addr_B  = slv_addr + 7'd1;
        end else begin
            assign  slv_addr_A  = slv_addr;
            assign  slv_addr_B  = slv_addr;
        end
    endgenerate


//---------------------------------------------------------------------------
// I2C INPUT:
//      Filter noise out of SCL/SDA signals, in the ICLK domain.
//
    reg [3:0]   sda_samp, scl_samp;
    reg         sda_f,    scl_f;
    reg [1:0]   sda_s,    scl_s;
    wire        sda_rising, sda_falling;
    wire        scl_rising, scl_falling;

// Sample:
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         sda_samp <= 4'b1111;
        else if (iclk_en)   sda_samp <= {sda_samp[2:0], sda_rcv };

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         scl_samp <= 4'b1111;
        else if (iclk_en)   scl_samp <= {scl_samp[2:0], scl_rcv };

    //---------------------------------------------------------------------------
    // s_filter -- filter input signal using samples collected.
    //
    function [0:0] s_filter( input reg [3:0] sig_samp );
        s_filter = (sig_samp == 4'b1111) ? 1'b1
                 : (sig_samp == 4'b0000) ? 1'b0
                 :                         sig_samp[3];     // Oldest
    endfunction

// Filter, and provide edge detection signals.
//      XXX_s = 00 - signal is low
//      XXX_s = 01 - signal is rising
//      XXX_s = 11 - signal is high
//      XXX_s = 10 - signal is falling
//
	always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            sda_f <= 1'b1;
		    sda_s <= 2'b11;

        end else if (iclk_en) begin
            sda_f <= s_filter( sda_samp );
            sda_s <= { sda_s[0], s_filter( sda_samp ) };
        end

	always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            scl_f <= 1'b1;
		    scl_s <= 2'b11;

        end else if (iclk_en) begin
            scl_f <= s_filter( scl_samp );
            scl_s <= { scl_s[0], s_filter( scl_samp ) };
        end

    assign  sda_rising  = (sda_s == 2'b01);
    assign  sda_falling = (sda_s == 2'b10);
    assign  scl_rising  = (scl_s == 2'b01);
    assign  scl_falling = (scl_s == 2'b10);


//---------------------------------------------------------------------------
// Detect START and STOP events.
//  START   S = Falling edge of SDA with SCL high
//  STOP    P = Rising edge of SDA with SCL high
//
    reg     start, stop;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         start <= 1'b0;
        else if (iclk_en)   start <= (sda_falling  &&  scl_s == 2'b11);

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         stop  <= 1'b0;
        else if (iclk_en)   stop  <= (sda_rising   &&  scl_s == 2'b11);


//---------------------------------------------------------------------------
// Input receiver
//
    reg [7:0]   in_sr;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)                             in_sr <= 'd0;
        else if (iclk_en  &&  scl_s == 2'b01)   in_sr <= { in_sr[6:0], sda_f };



    localparam  IDLE        = 4'd0,
                APHASE      = 4'd1,
                AACK        = 4'd2,
                WPHASE      = 4'd3,
                WACK        = 4'd4,
                WACKWR      = 4'd5,
                RPHASELD    = 4'd7,
                RPHASEXF    = 4'd8,
                RPHASE      = 4'd9;

// FSM: sync.
//
    reg [ 3:0]  state,          next_state;                 // pragma attribute state safe_fsm TRUE 
                                                            // pragma attribute state fsm_state ONEHOT
    reg [ 3:0]  acnt,           next_acnt;
    reg [ 3:0]  bcnt,           next_bcnt;
    reg [ 7:0]  iaddr,          next_iaddr;
    reg         isda_drv,       next_isda_drv;
    reg         iscl_drv,       next_iscl_drv;
    reg         addressed,      next_addressed;
    reg                         next_addr_sel;
    reg [ 8:0]  out_sr,         next_out_sr;

    reg [15:0]  addr,           next_addr;
    reg [ 7:0]                  next_reg_wdata;
    reg                         next_reg_we_b;
    reg                         next_reg_rd_b;

    reg [ 1:0]  ainc;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         state       <= IDLE;
        else if (iclk_en)   state       <= next_state;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         iaddr       <= 'h00;
        else if (iclk_en)   iaddr       <= next_iaddr;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         acnt        <= 4'd0;
        else if (iclk_en)   acnt        <= next_acnt;

// "ainc" monitor R/W strobes; one clock after either expires the address will be
// incremented.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         ainc        <= 2'b11;
        else if (iclk_en)   ainc        <= { ainc[0], reg_rd_b&reg_we_b };

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         bcnt        <= 4'd0;
        else if (iclk_en)   bcnt        <= next_bcnt;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         out_sr      <= 'd0;
        else if (iclk_en)   out_sr      <= next_out_sr;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         isda_drv    <= 1'b1;
        else if (iclk_en)   isda_drv    <= next_isda_drv;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         iscl_drv    <= 1'b1;
        else if (iclk_en)   iscl_drv    <= next_iscl_drv;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         addressed   <= 1'b0;
        else if (iclk_en)   addressed   <= next_addressed;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         addr_sel    <= 1'b0;
        else if (iclk_en)   addr_sel    <= next_addr_sel;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         addr        <= 'h00;
        else if (iclk_en)   addr        <= next_addr;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         reg_wdata   <= 'h00;
        else if (iclk_en)   reg_wdata   <= next_reg_wdata;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         reg_we_b    <= 'b1;
        else if (iclk_en)   reg_we_b    <= next_reg_we_b;

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)         reg_rd_b    <= 'b1;
        else if (iclk_en)   reg_rd_b    <= next_reg_rd_b;

// FSM: async.
//
    always @* begin
        next_state      <= state;
        next_acnt       <= acnt;
        next_bcnt       <= bcnt;
        next_iaddr      <= iaddr;
        next_isda_drv   <= isda_drv;
        next_iscl_drv   <= iscl_drv;
        next_addressed  <= addressed;
        next_addr_sel   <= addr_sel;
        next_addr       <= addr;
        next_reg_wdata  <= reg_wdata;
        next_reg_we_b   <= reg_we_b;
        next_reg_rd_b   <= reg_rd_b;
        next_out_sr     <= out_sr;

		case (state)										// pragma parallel_case

// 7|RPHASELD   Read data for master.  Address was already setup, start a read cycle.
//              RPHASELD is entered only on a SCL falling state, so SCL=0.  If pause_b
//              is asserted, force SCL=0 for clock stretching).  Clock stretching is
//              only permissible while SCL=0, so this is safe.
//
//              Note that here is the I2C flaw: if SCL is forced low, no STOP can be
//              detected so the system locks up unless the slave co-operates.
//
        RPHASELD: begin
            next_reg_rd_b   <= 1'b0;
            next_isda_drv   <= 1'b1;
            next_iscl_drv   <= pause_b;
            next_addr       <= (ainc == 2'b01) ? addr + 16'd1 : addr;

            if (stop)
                next_state  <= IDLE;
            else if (|acnt == 4'd0)
                next_state  <= RPHASEXF;
            else if (pause_b)                               // Timing does not run during pause.
                next_acnt   <= acnt - 4'd1;
        end

// 8|RPHASEXF   Read completion, transfer into out_sr.
//
        RPHASEXF: begin
            next_reg_rd_b   <= 1'b1;
            next_iscl_drv   <= 1'b1;
            next_isda_drv   <= out_sr[8];
            next_out_sr     <= { reg_rdata, 1'b1 };         // ACK=1 (or rather, undriven).
            next_bcnt       <= 4'd0;
            next_state      <= RPHASE;
        end

// 9|RPHASE     Sending 8b + ACK=1 (master will let ACK float if this is the last byte.)
//              If NAK is received, we are done.  Otherwise, start the next address phase.
//              Note that "ainc" is guaranteed to be active only one cycle, so the address
//              increment is safe.
//
        RPHASE: begin
            next_isda_drv   <= out_sr[8];                   // Send data
            if (scl_falling) begin
                if (bcnt == 4'd9  &&  sda_f == 1'b1)        // Done + Master NAK, we're all done.
                    next_state  <= IDLE;
                else if (bcnt == 4'd9) begin                // Done + Master ACK, get next byte
                    next_acnt   <= 4'd4;                    //  (for read, ACNT = access time).
                    next_state  <= RPHASELD;
                end
                else
                    next_out_sr <= { out_sr[7:0], 1'b1 };
            end

            if (scl_rising) begin
                next_bcnt   <= bcnt + 4'd1;
            end

            next_addr       <= (ainc == 2'b01) ? addr + 16'd1 : addr;

            if (start) begin                            // Restart cycle
                next_bcnt       <= 4'd0;
                next_state      <= APHASE;
            end
            else if (stop)                              // Abort cycle.
                next_state      <= IDLE;
        end

// 5|WACKWR     Complete the write WE_B output pulse, which is (was) short and not gated on the
//              SCL clock speed.  Wait for the SCL phase to end WACKWR phase.
//
        WACKWR: begin
            next_reg_we_b   <= 1'b1;                        // Complete write, if any.
            if (scl_falling)  begin
                next_isda_drv   <= 1'b1;                    // Release ACK=0 
                next_state      <= WPHASE;
            end

            next_addr       <= (ainc == 2'b01) ? addr + 16'd1 : addr;

            if (stop)
                next_state      <= IDLE;
        end

// 4|WACK       Got 8 bits of write data, and asserted ACK(SDA=0) on entry.  On the rising
//              edge, clear it and get more but also do something with it now.
//
        WACK: begin
            if (scl_rising) begin
                if (acnt == 4'd2)
                    next_addr[15:8] <= in_sr;               // Load MSB
                else if (acnt == 4'd1)
                    next_addr[ 7:0] <= in_sr;               // Load LSB (or only)
                else begin
                    next_reg_we_b   <= 1'b0;
                    next_reg_wdata  <= in_sr;
                end

                next_acnt       <= (|acnt) ? acnt - 4'd1 : acnt;
                next_bcnt       <= 4'd0;
                next_state      <= WACKWR; 
            end

            if (start) begin                            // Restart cycle
                next_bcnt       <= 4'd0;
                next_state      <= APHASE;
            end
            else if (stop)                              // Abort cycle.
                next_state      <= IDLE;
        end


// 3|WPHASE     Collecting data from master.  If "acnt" is non-zero, we are collecting 1 or more
//              address bytes.
//              Since we can collect data forever, we never do not assert ACK. 
//
        WPHASE: begin
            if (scl_falling  &&  bcnt == 4'd8) begin    // Got data
                next_isda_drv   <= 1'b0;                // Assert ACK.
                next_state      <= WACK;
            end

            if (scl_rising)                             // Collecting device address bits.
                next_bcnt       <= bcnt + 4'd1;

            if (start) begin                            // Restart cycle
                next_bcnt       <= 4'd0;
                next_state      <= APHASE;
            end
            else if (stop)                              // Abort cycle.
                next_state      <= IDLE;
        end


// 2|AACK       Assert ACK=0 on entry, ACK must remain low while SCL is high.
//              If this is a write, start collecting 1..2 bytes of address data
//              followed by target data, if any.
//
        AACK: begin
            next_isda_drv   <= 1'b0;            // Drive ACK=0
            if (scl_falling) begin
                if (iaddr[0] == 1'b0) begin         // Write
                    next_acnt       <= (EXT_ADDR) ? 4'd2 : 4'd1;
                    next_bcnt       <= 4'd0;
                    next_state      <= WPHASE;
                    next_isda_drv   <= 1'b1;
                end else begin                      // Read
                    next_acnt       <= 4'd4;        // For read, ACNT = access time.
                    next_bcnt       <= 4'd0;
                    next_state      <= RPHASELD;
                end
            end

            if (start) begin                            // Restart cycle
                next_bcnt       <= 4'd0;
                next_state      <= APHASE;
            end
            else if (stop)                              // Abort cycle.
                next_state      <= IDLE;
        end


    
// 1|APHASE     Receiving address information.  Once 8 bits were received (7 address
//              + 1 r/w), if the address matches drive ACK low; else ignore everything.
//
        APHASE: begin
            if (scl_falling  &&  bcnt == 4'd8) begin     // Got address bits; if address match, drive ACK.
                next_state  <= IDLE;
                if (in_sr[7:1] == slv_addr_A  ||  in_sr[7:1] == slv_addr_B) begin
                    next_addressed  <=  'd1;
                    next_addr_sel   <= (in_sr[7:1] == slv_addr_B);
                    next_iaddr      <= in_sr;
                    next_isda_drv   <= 1'b0;            // Assert ACK.
                    next_state      <= AACK;
                end
            end

            if (scl_rising)                             // Collecting device address bits.
                next_bcnt <= bcnt + 4'd1;

            if (stop)                                   // Aborted cycle.
                next_state <= IDLE;
        end

// 0|IDLE       Wait for START signals.
//
        IDLE: begin
            next_reg_rd_b   <= 1'b1;
            next_reg_we_b   <= 1'b1;
            next_isda_drv   <= 1'b1;                    // SDA=1 (undriven)
            next_iscl_drv   <= 1'b1;                    // SCL=1 (undriven)
            next_acnt       <= 4'd0;
            next_bcnt       <= 4'd0;
            next_addressed  <=  'd0;
            next_state      <= (start) ? APHASE : IDLE;
        end

// X|(recovery) Jump to IDLE.
//
        default: begin
        end
        endcase
    end


//---------------------------------------------------------------------------
// Address output:  Split into two pieces for easier use by slave controllers.
//  
    assign  ext_addr    = addr[15: 8];
    assign  reg_addr    = addr[ 7: 0];


//---------------------------------------------------------------------------
// I2C OUTPUT.
//      Set "s**_drv" to 0 to force the signal low.
//
    assign  sda_drv     = (!enable)     ?   1'b0
                        : (!isda_drv)    ?  1'b1            // ISDA_DRV=0 = drive to 0.
                        :                   1'b0;

    assign  scl_drv     = (!enable)     ?   1'b0
                        : (!iscl_drv)   ?   1'b1
                        :                   1'b0;


//---------------------------------------------------------------------------
// Export status.
//   NOTE: state[3:0] is shown as separate bits as somehow that prevents
//		   Prec. Synth. from ignoring FSMs.  Do not change.
//
    assign  slv_stat    = { (state == RPHASELD),// b7  : in RPHASELD state.
                            ~pause_b,           // b6  : paused
                            addressed,          // b5  : was addressed
                            addr_sel,           // b4  : which address space
							state[3],			// b3  : current state
							state[2],			// b2  : "
							state[1],			// b1  : "
							state[0] 			// b0  : current state
                          };



//---------------------------------------------------------------------------
// "scl_f" is unused because we typically look at the edge detected version.
//
	wire	unused;

	assign	unused      = scl_f;

endmodule
